#!/usr/bin/env python3
# The above line is a "shebang". It tells the system to use the Python 3
# interpreter to run this script if it's executed directly.

# --- Shell Guard ---
# The following block of code will cause the script to exit with an error
# if it is accidentally run with a shell like bash or sh, instead of Python.
# In Python, this is just a multi-line string, which is safely ignored.
"""
echo "ERROR: This is a Python script, not a shell script." >&2
echo "Please run it with the 'python' command: python $0" >&2
exit 1
"""
# --- End Shell Guard ---

import argparse
from pathlib import Path
import pandas as pd
import geopandas as gpd
from shapely.geometry import Polygon, LineString
from shapely.ops import polygonize, unary_union

def convert_lines_to_polygons(gdf):
    """
    Converts LineString geometries in a GeoDataFrame to Polygons.

    This function iterates through the geometries and, if a LineString is found,
    it converts it to a Polygon based on its coordinates. This relies on the
    assumption that the lines are closed (start and end at the same point).
    """
    # Create a new list to hold the converted geometries
    new_geometries = []
    for geom in gdf.geometry:
        if isinstance(geom, LineString):
            # Convert LineString to Polygon
            new_geometries.append(Polygon(geom.coords))
        else:
            # Keep existing geometry (e.g., if it's already a Polygon)
            new_geometries.append(geom)

    # Replace the old geometry column with the new one
    gdf.geometry = new_geometries
    return gdf


def save_output(gdf, base_name, to_kml, to_shp):
    """Saves the GeoDataFrame to the specified formats."""
    if gdf.empty:
        print("Result is empty, no files will be written.")
        return

    # Ensure the CRS is set for KML compatibility
    gdf = gdf.to_crs("EPSG:4326")
    
    # Flags to track success
    shp_saved = False
    kml_saved = False

    if to_shp:
        try:
            output_path_shp = f"{base_name}.shp"
            print(f"Saving ShapeFile to '{output_path_shp}'...")
            # Truncate field names for Shapefile compatibility
            gdf_shp = gdf.copy()
            gdf_shp.columns = [col[:10] for col in gdf_shp.columns]
            gdf_shp.to_file(output_path_shp, driver='ESRI Shapefile')
            shp_saved = True
        except Exception as e:
            print(f"\n--- ShapeFile Export Error ---")
            print(f"Could not write ShapeFile. Error: {e}")
            print("----------------------------\n")


    if to_kml:
        # We need to import the specific error type from fiona
        try:
            from fiona.errors import DriverError
        except ImportError:
            # Fallback if fiona is not installed as expected
            DriverError = Exception 

        output_path_kml = f"{base_name}.kml"
        print(f"Saving KML to '{output_path_kml}'...")
        try:
            # Enable KML driver for geopandas. This is often necessary.
            gpd.io.file.fiona.drvsupport.supported_drivers['KML'] = 'rw'
            gdf.to_file(output_path_kml, driver='KML')
            kml_saved = True
        except DriverError as e:
            print("\n--- KML EXPORT FAILED ---")
            print(f"Error Message: {e}")
            print("\nCould not write KML file because the required 'LIBKML' driver is not supported")
            print("in your system's installation of the Fiona/GDAL libraries.")
            print("\nThis is a common environment issue. The most reliable fix is to reinstall")
            print("GeoPandas using Conda (which manages these complex dependencies correctly):")
            print("  conda install -c conda-forge geopandas")
            print("--------------------------\n")
        except Exception as e:
            print(f"\nAn unexpected error occurred during KML export: {e}\n")

    if shp_saved or kml_saved:
        print("Output generation complete.")

def perform_overlap_analysis(gdf):
    """
    Performs a topological overlay to find all unique intersecting polygons
    and counts the number of original polygons that overlap each one.
    This method uses polygonization of all boundary lines.
    """
    if gdf.empty:
        return gpd.GeoDataFrame(geometry=[], crs=gdf.crs)

    # 1. Collect all boundary lines from all input polygons
    all_lines = []
    for geom in gdf.geometry:
        # Check if the geometry is a Polygon or MultiPolygon
        if geom.geom_type == 'Polygon':
            all_lines.append(geom.exterior)
            for interior in geom.interiors:
                all_lines.append(interior)
        elif geom.geom_type == 'MultiPolygon':
            for poly in geom.geoms:
                all_lines.append(poly.exterior)
                for interior in poly.interiors:
                    all_lines.append(interior)
    
    # 2. Create a single "MultiLineString" of all unique boundary segments
    boundary_union = unary_union(all_lines)
    
    # 3. "Polygonize" the flat network of lines to create all enclosed areas
    new_polygons = list(polygonize(boundary_union))

    if not new_polygons:
        print("Warning: Polygonization resulted in no new polygons.")
        return gpd.GeoDataFrame(geometry=[], crs=gdf.crs)

    # 4. Create a new GeoDataFrame from these small polygons
    intersection_gdf = gpd.GeoDataFrame(geometry=new_polygons, crs=gdf.crs)

    # 5. Count overlaps for each new polygon
    overlap_counts = []
    spatial_index = gdf.sindex
    
    for poly_to_check in intersection_gdf.geometry:
        # Using representative_point is robust for checking containment
        rep_point = poly_to_check.representative_point()
        
        # Find potential candidates from the original gdf using the spatial index
        possible_matches_index = list(spatial_index.intersection(rep_point.bounds))
        possible_matches = gdf.iloc[possible_matches_index]
        
        # Perform the precise check on the candidates
        count = possible_matches.contains(rep_point).sum()
        overlap_counts.append(count)

    intersection_gdf['overlaps'] = overlap_counts
    
    return intersection_gdf[intersection_gdf['overlaps'] > 0]


def process_one_folder(folder, to_kml, to_shp):
    """
    Processes a single folder to create an overlap counter map.
    """
    print(f"--- Processing single folder: {folder} ---")
    shp_path = Path(folder) / 'shape' / 'contours.shp'
    if not shp_path.is_file():
        print(f"Error: ShapeFile not found at '{shp_path}'")
        return

    print("Reading input ShapeFile...")
    gdf = gpd.read_file(shp_path)

    print("Converting line geometries to polygons...")
    gdf = convert_lines_to_polygons(gdf)
    gdf = gdf[gdf.geometry.is_valid] # Ensure polygons are valid

    if gdf.empty:
        print("Input ShapeFile is empty or contains no valid geometries.")
        return

    print("Calculating overlaps... (this may take a while)")
    
    # Use the robust overlap analysis function
    final_gdf = perform_overlap_analysis(gdf)

    print("Overlap calculation complete.")
    save_output(final_gdf, "overlap_analysis", to_kml, to_shp)


def process_three_folders(folder1, folder2, folder3, to_kml, to_shp):
    """
    Processes three folders to create a "change" map.
    folder1: Before polygon
    folder2: After polygon
    folder3: Many other polygons
    """
    print("--- Processing three folders for change analysis ---")
    paths = {
        'before': Path(folder1) / 'shape' / 'contours.shp',
        'after': Path(folder2) / 'shape' / 'contours.shp',
        'many': Path(folder3) / 'shape' / 'contours.shp',
        'params_before': Path(folder1) / 'parameters.csv',
        'params_after': Path(folder2) / 'parameters.csv',
        'params_many': Path(folder3) / 'parameters.csv',
    }

    # Validate all paths
    for name, path in paths.items():
        if not path.is_file():
            print(f"Error: Required file not found at '{path}'")
            return

    # --- 1. Read all data ---
    print("Reading input files...")
    gdf_before = gpd.read_file(paths['before'])
    gdf_after = gpd.read_file(paths['after'])
    gdf_many = gpd.read_file(paths['many'])

    print("Converting line geometries to polygons...")
    gdf_before = convert_lines_to_polygons(gdf_before)
    gdf_after = convert_lines_to_polygons(gdf_after)
    gdf_many = convert_lines_to_polygons(gdf_many)

    # Ensure geometries are valid
    gdf_before = gdf_before[gdf_before.geometry.is_valid]
    gdf_after = gdf_after[gdf_after.geometry.is_valid]
    gdf_many = gdf_many[gdf_many.geometry.is_valid]
    
    # Use header on row 5 (0-indexed means skiprows=4)
    params_before = pd.read_csv(paths['params_before'], skiprows=4)
    params_after = pd.read_csv(paths['params_after'], skiprows=4)
    params_many = pd.read_csv(paths['params_many'], skiprows=4)

    # --- 2. De-duplication ---
    print("De-duplicating records...")
    recid_before = params_before['RecID'].iloc[0]
    recid_after = params_after['RecID'].iloc[0]
    
    srckeys_to_ignore = params_many[params_many['RecID'].isin([recid_before, recid_after])]['SrcKey']
    gdf_many_filtered = gdf_many[~gdf_many['SOURCEKEY'].isin(srckeys_to_ignore)]
    
    print(f"Removed {len(gdf_many) - len(gdf_many_filtered)} polygons from 'many' dataset based on RecID.")
    if gdf_many_filtered.empty:
        print("No service polygons remain after de-duplication.")
        return

    # --- 3. Perform Overlap Analysis on the FULL set of service polygons FIRST ---
    print("Calculating full service overlap map... (this may take a while)")
    full_overlap_map = perform_overlap_analysis(gdf_many_filtered)
    
    if full_overlap_map.empty:
        print("Full overlap map is empty. Cannot proceed with change analysis.")
        return

    # --- 4. Define Before and After geometries for flagging and clipping ---
    print("Defining Before and After geometries...")
    before_poly = gdf_before.geometry.iloc[0]
    # Put the 'After' polygon into its own GeoDataFrame for the overlay operation
    after_poly_gdf = gpd.GeoDataFrame(geometry=[gdf_after.geometry.iloc[0]], crs=gdf_after.crs)

    # --- 5. Remove the 'After' area from the full overlap map ---
    print("Clipping overlap map to exclude the 'After' area...")
    # This gives us all the overlap areas that are outside the "After" contour
    map_outside_after = gpd.overlay(full_overlap_map, after_poly_gdf, how='difference')

    if map_outside_after.empty:
        print("The final map is empty after removing the 'After' area.")
        return

    # --- 6. Add a flag indicating which areas are inside the 'Before' contour ---
    print("Flagging areas that fall within the 'Before' contour...")
    # .within() returns a boolean Series (True/False)
    is_inside = map_outside_after.within(before_poly)
    # Convert boolean to integer (1 for True, 0 for False) for easier use in GIS
    map_outside_after['in_before'] = is_inside.astype(int)
    
    # The 'overlaps' column from full_overlap_map is preserved in the difference operation.

    print("Change analysis complete.")
    # Keep only the columns we care about for the final output
    final_change_map = map_outside_after[['geometry', 'overlaps', 'in_before']]
    save_output(final_change_map, "change_analysis", to_kml, to_shp)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Geospatial analysis of overlapping polygons."
    )
    parser.add_argument(
        'folders',
        nargs='+',
        help="One to three input folders containing shape/contours.shp files."
    )
    parser.add_argument(
        '-k', '--kml',
        action='store_true',
        help="Output a KML file."
    )
    parser.add_argument(
        '-s', '--shp',
        action='store_true',
        help="Output an ESRI ShapeFile."
    )
    args = parser.parse_args()

    num_folders = len(args.folders)

    if not args.kml and not args.shp:
        parser.error("No output format selected. Please specify -k (KML) and/or -s (ShapeFile).")

    if num_folders == 1:
        process_one_folder(args.folders[0], args.kml, args.shp)
    elif num_folders == 2:
        print("Placeholder: Two-folder analysis is not yet implemented.")
    elif num_folders == 3:
        process_three_folders(args.folders[0], args.folders[1], args.folders[2], args.kml, args.shp)
    else:
        parser.error(f"Error: Expected 1, 2, or 3 folders, but got {num_folders}.")
