#!/usr/bin/env python3
# The above line is a "shebang". It tells the system to use the Python 3
# interpreter to run this script if it's executed directly.

# --- Shell Guard ---
# The following block of code will cause the script to exit with an error
# if it is accidentally run with a shell like bash or sh, instead of Python.
# In Python, this is just a multi-line string, which is safely ignored.
"""
echo "ERROR: This is a Python script, not a shell script." >&2
echo "Please run it with the 'python' command: python $0" >&2
exit 1
"""
# --- End Shell Guard ---

import argparse
from pathlib import Path
import pandas as pd
import geopandas as gpd
import simplekml

def extract_population(name_str):
    """
    Extracts the population integer from the 'Name' column string.
    Example: '153855-255288-1034' -> 1034
    """
    try:
        # Split the string by the hyphen and return the last part as an integer
        return int(name_str.split('-')[-1])
    except (ValueError, IndexError):
        # Handle cases where the format is unexpected or a part is not a number
        return 0

def create_loss_map(folder1, folder2, to_kml, to_shp):
    """
    Compares two tvstudy.csv files and creates a map of service changes.
    """
    # --- 1. Construct paths and validate file existence ---
    path1 = Path(folder1) / 'tvstudy.csv'
    path2 = Path(folder2) / 'tvstudy.csv'

    if not path1.is_file():
        print(f"Error: File not found at '{path1}'")
        return
    if not path2.is_file():
        print(f"Error: File not found at '{path2}'")
        return

    print(f"Comparing '{path1}' (Before) and '{path2}' (After)...")

    try:
        # --- 2. Load data, skipping the first 7 rows ---
        df1 = pd.read_csv(path1, skiprows=7)
        df2 = pd.read_csv(path2, skiprows=7)
        
        # --- 3. Extract population data into new columns ---
        df1['pop_before'] = df1['Name'].apply(extract_population)
        df2['pop_after'] = df2['Name'].apply(extract_population)

        # --- 4. Merge the two dataframes based on the 'Name' column ---
        # Include 'Field' column from both files.
        merged_df = pd.merge(
            df1[['Name', 'Serv', 'Latitude', 'Longitude', 'pop_before', 'Field']],
            df2[['Name', 'Serv', 'pop_after', 'Field']],
            on='Name',
            suffixes=('_before', '_after')
        )
        
        if merged_df.empty:
            print("No matching records found between the two files. Cannot create a map.")
            return

        # --- 5. Check for population mismatches ---
        mismatched_pop = merged_df[merged_df['pop_before'] != merged_df['pop_after']]
        if not mismatched_pop.empty:
            print("\n--- WARNING: Population mismatch found! ---")
            print("The following locations have different population values in the two files:")
            print(mismatched_pop[['Name', 'pop_before', 'pop_after']])
            print("-------------------------------------------\n")

        # Create a single population column for consistency
        merged_df['population'] = merged_df['pop_before']
        merged_df = merged_df.drop(columns=['pop_before', 'pop_after'])


        # --- 6. Determine 'cell_status' for each location ---
        print("Determining cell status for each location...")
        statuses = []
        for _, row in merged_df.iterrows():
            serv1 = row['Serv_before']
            serv2 = row['Serv_after']
            status = f"{int(serv1)}{int(serv2)}"
            statuses.append(status)
        
        merged_df['cell_status'] = statuses

        # --- 7. Create a GeoDataFrame from the points ---
        print("Creating point geometries from coordinates...")
        gdf = gpd.GeoDataFrame(
            merged_df,
            geometry=gpd.points_from_xy(merged_df.Longitude, merged_df.Latitude),
            crs="EPSG:4326"  # Assume WGS84, which is standard for lat/lon
        )
        
        # --- 8. Save the output files ---
        if to_shp:
            save_shapefile(gdf)

        if to_kml:
            save_kml_file(gdf)

    except Exception as e:
        print(f"An error occurred during processing: {e}")

def save_shapefile(gdf):
    """Saves the GeoDataFrame to an ESRI Shapefile."""
    output_path = "loss_code_map.shp"
    print(f"Saving ShapeFile to '{output_path}'...")
    
    # Select and rename columns for Shapefile output
    # Shapefile column names are limited to 10 characters.
    shp_gdf = gdf[['geometry', 'Name', 'cell_status', 'population', 'Field_before', 'Field_after']].copy()
    shp_gdf = shp_gdf.rename(columns={
        'cell_status': 'cell_stat',
        'population': 'pop',
        'Field_before': 'fld_bef',
        'Field_after': 'fld_aft'
    })

    try:
        shp_gdf.to_file(output_path, driver='ESRI Shapefile')
        print("ShapeFile saved successfully.")
    except Exception as e:
        print(f"\n--- ShapeFile Export Error ---")
        print(f"Could not write ShapeFile. Error: {e}")
        print("----------------------------\n")

def save_kml_file(gdf):
    """Saves the GeoDataFrame to a styled KML file using simplekml."""
    output_path = "loss_code_map.kml"
    print(f"Saving KML to '{output_path}'...")

    try:
        kml = simplekml.Kml(name="Service Change Analysis")

        # --- Define Shared Styles (more efficient) ---
        style_map = {
            '10': 'http://maps.google.com/mapfiles/kml/paddle/R.png',    # Red for Loss (1 -> 0)
            '01': 'http://maps.google.com/mapfiles/kml/paddle/Y.png',    # Yellow for Gain (0 -> 1)
            '11': 'http://maps.google.com/mapfiles/kml/paddle/G.png',    # Green for Service (1 -> 1)
            '00': 'http://maps.google.com/mapfiles/kml/paddle/wht-blank.png' # White/Gray for No Service (0 -> 0)
        }

        shared_styles = {}
        for status, icon_href in style_map.items():
            style = simplekml.Style()
            style.iconstyle.icon.href = icon_href
            shared_styles[status] = style
        
        # --- Create points and assign styles ---
        for _, row in gdf.iterrows():
            pnt = kml.newpoint(name=row['Name'])
            pnt.coords = [(row['Longitude'], row['Latitude'])]
            
            # Add population and field data to the description
            pnt.description = (
                f"Population: {row['population']:,}<br>"
                f"Field Before: {row['Field_before']:.2f}<br>"
                f"Field After: {row['Field_after']:.2f}"
            )
            
            # Assign the appropriate shared style
            pnt.style = shared_styles.get(row['cell_status'])

        kml.save(output_path)
        print("KML file saved successfully.")

    except Exception as e:
        print(f"\n--- KML Export Error ---")
        print(f"Could not write KML file. Error: {e}")
        print("This may be due to a problem with the 'simplekml' library.")
        print("------------------------\n")


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Create a map of service changes from two tvstudy files."
    )
    parser.add_argument(
        "folder1",
        help="The 'Before' folder containing a tvstudy.csv file."
    )
    parser.add_argument(
        "folder2",
        help="The 'After' folder containing a tvstudy.csv file."
    )
    parser.add_argument(
        '-k', '--kml',
        action='store_true',
        help="Output a KML file."
    )
    parser.add_argument(
        '-s', '--shp',
        action='store_true',
        help="Output an ESRI ShapeFile."
    )
    args = parser.parse_args()

    if not args.kml and not args.shp:
        parser.error("No output format selected. Please specify -k (KML) and/or -s (ShapeFile).")

    create_loss_map(args.folder1, args.folder2, args.kml, args.shp)
