#!/usr/bin/env python3
# The above line is a "shebang". It tells the system to use the Python 3
# interpreter to run this script if it's executed directly.

# --- Shell Guard ---
# The following block of code will cause the script to exit with an error
# if it is accidentally run with a shell like bash or sh, instead of Python.
# In Python, this is just a multi-line string, which is safely ignored.
"""
echo "ERROR: This is a Python script, not a shell script." >&2
echo "Please run it with the 'python' command: python $0" >&2
exit 1
"""
# --- End Shell Guard ---

import argparse
import pandas as pd
from pathlib import Path

def transform_and_save(file_path, receive_height=10):
    """
    Reads a CSV file, transforms its data, and saves it to a new file.

    The script filters for rows where 'country_key' is 1, negates the
    longitude, and formats the output as specified.

    The output filename is derived from the input filename by inserting
    '.pointset' before the extension (e.g., 'input.csv' -> 'input.pointset.csv').

    Args:
        file_path (str): The path to the input CSV file.
        receive_height (int): The receive height value for the output.
    """
    # --- 1. Validate input file path ---
    input_path = Path(file_path)
    if not input_path.is_file():
        print(f"Error: The file '{file_path}' was not found.")
        return

    # --- 2. Construct the output file path ---
    # Example: 'LossCode-Sample.csv' becomes 'LossCode-Sample.pointset.csv'
    output_path = input_path.with_name(f"{input_path.stem}.pointset.csv")

    try:
        # --- 3. Read and process the data using pandas ---
        # Read the source CSV file
        df = pd.read_csv(input_path)

        # Filter for rows where country_key is 1.
        # .copy() is used to avoid SettingWithCopyWarning.
        df_filtered = df[df['country_key'] == 1].copy()

        # Negate the 'longitude' column
        df_filtered['longitude'] = -df_filtered['longitude']

        # --- 4. Write the transformed data to the output file ---
        print(f"Processing data and writing to '{output_path}'...")
        with open(output_path, 'w') as f_out:
            # Iterate over each row of the filtered DataFrame
            for index, row in df_filtered.iterrows():
                # Format the output string according to the required format
                # <latindex>-<lonindex>-<population>,<latitude>,<longitude>,<receive_height>
                output_string = (
                    f"{int(row['lat_index'])}-{int(row['lon_index'])}-{int(row['population'])},"
                    f"{row['latitude']},{row['longitude']},{receive_height}"
                )
                # Write the formatted string to the file, followed by a newline
                f_out.write(output_string + '\n')

        print("Transformation complete.")
        print(f"Output successfully saved to: {output_path}")

    except Exception as e:
        print(f"An error occurred during processing: {e}")

if __name__ == "__main__":
    # --- Set up command-line argument parsing ---
    parser = argparse.ArgumentParser(
        description="Transform CSV data to a specific pointset format and save to a new file."
    )
    # Required argument: The name of the input file
    parser.add_argument(
        "file_name",
        help="The input CSV file (e.g., 'LossCode-Sample.csv')."
    )
    # Optional positional argument: The receive height
    parser.add_argument(
        "receive_height",
        type=int,
        nargs='?',  # This makes the argument optional
        default=10, # Default value if not provided
        help="Optional receive height value (defaults to 10)."
    )
    args = parser.parse_args()

    # --- Run the main function ---
    transform_and_save(args.file_name, args.receive_height)
