//
//  study.c
//  TVStudy
//
//  Copyright (c) 2012-2024 Hammett & Edison, Inc.  All rights reserved.


// Functions to open a study database and run initial setup for a study.


#include "tvstudy.h"

#include <ctype.h>
#include <sys/wait.h>


//---------------------------------------------------------------------------------------------------------------------

#define WAIT_TIME_SECONDS  10   // Time to wait for study lock to become available, see discussion below.
#define MAX_WAIT_COUNT     18

static void close_at_exit();
static void clear_state(int clearAll);

// Public globals.

MYSQL *MyConnection = NULL;   // Database connection.

char DbName[MAX_STRING];       // Root database name, also prefix for study and external database names.
char DatabaseID[MAX_STRING];   // Database UUID.
char HostDbName[MAX_STRING];   // Host and root database names combined, for reports and file names.

int StudyKey = 0;                         // Study key, type, template key, lock, name, etc.
int StudyType = STUDY_TYPE_UNKNOWN;
int StudyMode = STUDY_MODE_GRID;
int StudyModCount = 0;
char StudyName[MAX_STRING];
char ExtDbName[MAX_STRING];
char PointSetName[MAX_STRING];
int PropagationModel = 1;
int StudyAreaMode = STUDY_AREA_SERVICE;
int StudyAreaGeoKey = 0;
int StudyAreaModCount = 0;
char AreaGeoName[MAX_STRING];

char OutputCodes[MAX_STRING];        // Output codes and flags, may be set before calling open_study(), if not,
int OutputFlags[MAX_OUTPUT_FLAGS];   //   flag codes in the study record will be used.
int OutputFlagsSet = 0;

char MapOutputCodes[MAX_STRING];
int MapOutputFlags[MAX_MAP_OUTPUT_FLAGS];
int MapOutputFlagsSet = 0;

char *ReportPreamble = NULL;   // Report preamble, see report.c.

SCENARIO_PAIR *ScenarioPairList = NULL;   // List of scenario pairs, see run_ixstudy.c and run_scenario.c.

// Run behavior flags, set externally before calling open_study().

int RunNumber = 0;                // Run ID number, optional, used for temporary output files.  See open_temp_file().
int CacheUndesired = 1;           // True to update undesired cell caches, may be set false.  In some situations the
                                  //   hit rate is so low performance is actually worse with caching due to I/O wait.
int CreateResultTablesOpt = -1;   // 1 to create result tables, 0 to not, -1 if option not specified.
int CreateResultTables = 0;       // Set per CreateResultTablesOpt and related state to actually create tables.
int DidCreateResultTables = 0;    // True post-run if result tables were actually created in run_scenario().
int Debug = 0;                    // Activates debugging log messages.

char *RunComment = NULL;   // Comment text appearing in reports.

// Private globals, see open_study().

static int StudyLock = LOCK_NONE;
static int LockCount = 0;
static int KeepLock = 0;
static int KeepExclLock = 0;

static char *ParameterSummary = NULL;
static char *IxRuleSummary = NULL;


//---------------------------------------------------------------------------------------------------------------------
// Open a connection to a study database, query the study record, check and configure the persistent lock flag as
// needed.  If open and locking are successful, load parameters, load and pre-process sources, and get everything set
// up to load and run scenarios.  If anything goes seriously wrong the return is <0, process should probably exit.  If
// the study does not exist, or the lock checks fail, or any of a variety of minor errors occur during setup, return
// is >0; in that case the run can continue and this can be called again with different arguments.  For any non-zero
// return the connection is not opened and study data not loaded, or incompletely loaded.

// This will not return success unless and until a valid lock state exists in the study database.  Under normal
// conditions, the existing state of the lock is checked and the response for each possible state is described below.

// LOCK_NONE
// Set LOCK_RUN_EXCL, giving this process exclusive use of the database and the ability to write updates back during
// the pre-processing done in load_sources().  Once load_sources() is done the lock may be demoted to LOCK_RUN_SHARE,
// unless study features requiring an exclusive lock are active.

// LOCK_EDIT
// Immediately return a non-fatal failure.

// LOCK_ADMIN, LOCK_RUN_EXCL
// Enter a loop waiting for the lock to become LOCK_NONE or LOCK_RUN_SHARE, then proceed as described for those cases.
// The wait loop has a timeout, return a non-fatal failure if that expires.

// LOCK_RUN_SHARE
// Share the existing lock.  Continue with pre-processing and return success, however load_sources() will not attempt
// to make any updates.  The assumption is there won't be any since some other engine process must previously have had
// LOCK_RUN_EXCL and made any updates needed.  But even if there are records still needing update that does not
// invalidate this run.  The new values computed, used, and cached here are assumed to be identical to corresponding
// values computed by any other process.  Some study features require an exclusive lock, in those cases a share lock
// will return a non-fatal failure.

// However if inheritLockCount is >0, the behavior is different.  In that case, attempt to inherit an existing lock
// state set by another application.  The lock must be set to LOCK_RUN_EXCL or LOCK_RUN_SHARE and the lock_count must
// match the argument else immediate (non-fatal) failure occurs.  When the lock is inherited, it will not be cleared
// on exit.  However it may still be downgraded to a shared lock, unless the keepExclLock flag is true or features
// requiring the exclusive lock are active, in which case the lock will not be changed at all.

// Note some features, such as creating result tables, require the exclusive lock so that may be kept even if the
// caller does not request it.

// Arguments:

//   host                Database host name, NULL to use the MySQL API default which is usually localhost.
//   name                Root database name, or NULL to use default.
//   user                Database user name, NULL to use the MySQL API default which is usually the OS login user name.
//   pass                Database password.  Must not be NULL or empty.
//   studyKey            Primary key of the study record in the root database study table, if 0 find study by name.
//   studyName           Study name to find, or NULL.
//   inheritLockCount    >0 to inherit existing run lock, lock_count must match this value.
//   keepExclLock        When inheritLockCount >0, if this is true the lock will not be changed to shared.

// Return <0 for a serious error, >0 for a minor error, 0 for no error.

int open_study(char *host, char *name, char *user, char *pass, int studyKey, char *studyName, int inheritLockCount,
	int keepExclLock) {

	if (MyConnection) {
		log_error("open_study() called with study already open");
		return 1;
	}

	// On the first open, install an exit hook to call close_study().

	static int init = 1;
	if (init) {
		atexit(close_at_exit);
		init = 0;
	}

	// Paranoia, make sure study state is clear.  Set database name.

	clear_state(0);

	if (name) {
		lcpystr(DbName, name, MAX_STRING);
	} else {
		lcpystr(DbName, DEFAULT_DB_NAME, MAX_STRING);
	}

	char query[MAX_QUERY];
	MYSQL_RES *myResult;
	my_ulonglong rowCount, rowIndex;
	MYSQL_ROW fields;

	// Parse out hostname and port number from host for the connection call only, elsewhere the host string is used as
	// an identifier and in file paths so the port number is preserved to allow multiple servers on the same host.

	char *connectHost = host;
	int connectPort = 0;

	char hostBuf[MAX_STRING];
	if (host) {
		char *p = index(host, ':');
		if (p) {
			*p = '\0';
			lcpystr(hostBuf, host, MAX_STRING);
			*p = ':';
			connectHost = hostBuf;
			connectPort = atoi(p + 1);
		}
	}

	// Outer loop in case this needs to wait for a lock change.  An inability to open the initial connection is not a
	// fatal error, the most likely explanation is the user provided incorrect login credentials.

	int err, shareCount, templateKey, needsUpdate, len, pointSetKey, lockOK, newLock, wait, waitCount = MAX_WAIT_COUNT,
		waitMessage = 1, hadResultTables = 0;

	do {

		MyConnection = mysql_init(NULL);

		if (!mysql_real_connect(MyConnection, connectHost, user, pass, DbName, connectPort, NULL, 0)) {
			log_db_error("Database connection failed");
			mysql_close(MyConnection);
			MyConnection = NULL;
			return 1;
		}

		// The study table is locked until the persistent lock flag (a field in the table) can be checked and set.

		snprintf(query, MAX_QUERY, "LOCK TABLES %s.study WRITE, %s.version WRITE, %s.ext_db WRITE, %s.geography AS point_geo WRITE, %s.geography AS area_geo WRITE;", DbName, DbName, DbName, DbName, DbName);
		if (mysql_query(MyConnection, query)) {
			log_db_error("Could not lock study table (1)");
			mysql_close(MyConnection);
			MyConnection = NULL;
			return -1;
		}

		err = -1;
		wait = 0;

		if (studyKey) {
			snprintf(query, MAX_QUERY, "SELECT study.name, version.version, study.study_lock, study.lock_count, study.share_count, study.study_type, study.study_mode, study.template_key, study.needs_update, study.mod_count, version.uuid, ext_db.db_type, ext_db.name, ext_db.id, study.point_set_key, point_geo.name, study.propagation_model, study.study_area_mode, study.study_area_geo_key, area_geo.mod_count, area_geo.name, study.output_config_file_codes, study.output_config_map_codes, study.report_preamble, study.parameter_summary, study.ix_rule_summary, study.had_result_tables FROM %s.study JOIN %s.version LEFT JOIN %s.ext_db USING (ext_db_key) LEFT JOIN %s.geography AS point_geo ON (point_geo.geo_key = study.point_set_key) LEFT JOIN %s.geography AS area_geo ON (area_geo.geo_key = study.study_area_geo_key) WHERE study.study_key = %d;", DbName, DbName, DbName, DbName, DbName, studyKey);
		} else {
			char nameBuf[512];
			unsigned long ulen = strlen(studyName);
			if (ulen > 255) {
				ulen = 255;
			}
			mysql_real_escape_string(MyConnection, nameBuf, studyName, ulen);
			snprintf(query, MAX_QUERY, "SELECT study.study_key, version.version, study.study_lock, study.lock_count, study.share_count, study.study_type, study.study_mode, study.template_key, study.needs_update, study.mod_count, version.uuid, ext_db.db_type, ext_db.name, ext_db.id, study.point_set_key, point_geo.name, study.propagation_model, study.study_area_mode, study.study_area_geo_key, area_geo.mod_count, area_geo.name, study.output_config_file_codes, study.output_config_map_codes, study.report_preamble, study.parameter_summary, study.ix_rule_summary, study.had_result_tables FROM %s.study JOIN %s.version LEFT JOIN %s.ext_db USING (ext_db_key) LEFT JOIN %s.geography AS point_geo ON (point_geo.geo_key = study.point_set_key) LEFT JOIN %s.geography AS area_geo ON (area_geo.geo_key = study.study_area_geo_key) WHERE study.name = '%s';", DbName, DbName, DbName, DbName, DbName, nameBuf);
		}

		if (mysql_query(MyConnection, query)) {
			log_db_error("Study query failed (1)");

		} else {
			myResult = mysql_store_result(MyConnection);
			if (!myResult) {
				log_db_error("Study query failed (2)");

			} else {
				rowCount = mysql_num_rows(myResult);
				if (!rowCount) {
					if (studyKey) {
						log_error("Study not found for studyKey=%d", studyKey);
					} else {
						log_error("Study '%s' not found", studyName);
					}
					err = 1;

				} else {
					fields = mysql_fetch_row(myResult);
					if (!fields) {
						log_db_error("Study query failed (3)");

					} else {

						if (studyKey) {
							lcpystr(StudyName, fields[0], MAX_STRING);
						} else {
							studyKey = atoi(fields[0]);
							lcpystr(StudyName, studyName, MAX_STRING);
						}

						if (TVSTUDY_DATABASE_VERSION != atoi(fields[1])) {
							log_error("Incorrect database version");
							err = 1;

						} else {

							StudyLock = atoi(fields[2]);
							LockCount = atoi(fields[3]);
							shareCount = atoi(fields[4]);
							StudyType = atoi(fields[5]);
							StudyMode = atoi(fields[6]);
							templateKey = atoi(fields[7]);
							needsUpdate = atoi(fields[8]);
							StudyModCount = atoi(fields[9]);

							lcpystr(DatabaseID, fields[10], MAX_STRING);

							if (fields[11]) {
								switch (atoi(fields[11])) {
									case DB_TYPE_CDBS: {
										lcpystr(ExtDbName, "CDBS TV ", MAX_STRING);
										break;
									}
									case DB_TYPE_LMS: {
										lcpystr(ExtDbName, "LMS TV ", MAX_STRING);
										break;
									}
									case DB_TYPE_CDBS_FM: {
										lcpystr(ExtDbName, "CDBS FM ", MAX_STRING);
										break;
									}
								}
								len = strlen(fields[12]);
								if (len > 0) {
									lcatstr(ExtDbName, fields[12], MAX_STRING);
								} else {
									lcatstr(ExtDbName, fields[13], MAX_STRING);
								}
							}

							pointSetKey = atoi(fields[14]);
							if (fields[15]) {
								len = strlen(fields[15]);
								if (len > 0) {
									lcpystr(PointSetName, fields[15], MAX_STRING);
								}
							}

							PropagationModel = atoi(fields[16]);

							StudyAreaMode = atoi(fields[17]);
							StudyAreaGeoKey = atoi(fields[18]);
							if (fields[19]) {
								StudyAreaModCount = atoi(fields[19]);
							}
							if (fields[20]) {
								len = strlen(fields[20]);
								if (len > 0) {
									lcpystr(AreaGeoName, fields[20], MAX_STRING);
								}
							}

							lcpystr(OutputCodes, fields[21], MAX_STRING);

							lcpystr(MapOutputCodes, fields[22], MAX_STRING);

							len = strlen(fields[23]);
							if (len > 0) {
								len++;
								ReportPreamble = (char *)mem_alloc(len);
								lcpystr(ReportPreamble, fields[23], len);
							}

							len = strlen(fields[24]);
							if (len > 0) {
								len++;
								ParameterSummary = (char *)mem_alloc(len);
								lcpystr(ParameterSummary, fields[24], len);
							}

							len = strlen(fields[25]);
							if (len > 0) {
								len++;
								IxRuleSummary = (char *)mem_alloc(len);
								lcpystr(IxRuleSummary, fields[25], len);
							}

							// The had_result_tables flag is used by UI to set the default for a user-initiated run,
							// but runs not initiated directly by the user should not affect that default so if the
							// result tables option was not specified the had-tables flag is not changed.  Otherwise
							// it is set to the specified option for this run, regardless of whether table creation is
							// actually successful.  The has_result_tables flag always indicates if tables exist, on
							// successful open has-tables is cleared and existing tables are dropped, has-tables will
							// be set at close only if tables were successfully created.

							if (CreateResultTablesOpt < 0) {
								hadResultTables = atoi(fields[26]);
							} else {
								hadResultTables = CreateResultTablesOpt;
								CreateResultTables = CreateResultTablesOpt;
							}

							lockOK = 0;

							if (inheritLockCount > 0) {

								if (((StudyLock != LOCK_RUN_EXCL) && (StudyLock != LOCK_RUN_SHARE)) ||
										(LockCount != inheritLockCount)) {
									log_error("Could not inherit study lock, the lock was modified");

								} else {
									KeepLock = 1;
									KeepExclLock = keepExclLock;
									lockOK = 1;
								}

							} else {

								newLock = LOCK_NONE;
								switch (StudyLock) {
									case LOCK_NONE:
										newLock = LOCK_RUN_EXCL;
										LockCount++;
										shareCount = 0;
										break;
									case LOCK_EDIT:
										log_error("Study '%s' is in use by another application", StudyName);
										err = 1;
										break;
									case LOCK_ADMIN:
									case LOCK_RUN_EXCL:
										if (waitCount-- > 0) {
											if (waitMessage) {
												log_message("Waiting for study '%s' to be unlocked...", StudyName);
												waitMessage = 0;
											}
											wait = 1;
											err = 0;
										} else {
											log_error("Study '%s' is in use by another application", StudyName);
											err = 1;
										}
										break;
									case LOCK_RUN_SHARE:
										if (CreateResultTables) {
											log_error("Study '%s' is in use by another application", StudyName);
											err = 1;
										} else {
											newLock = LOCK_RUN_SHARE;
											shareCount++;
										}
										break;
								}

								if (newLock != LOCK_NONE) {
									snprintf(query, MAX_QUERY,
						"UPDATE %s.study SET study_lock = %d, lock_count = %d, share_count = %d WHERE study_key = %d;",
										DbName, newLock, LockCount, shareCount, studyKey);
									if (mysql_query(MyConnection, query)) {
										log_db_error("Study lock update query failed (1)");
									} else {
										StudyLock = newLock;
										lockOK = 1;
									}
								}
							}

							// Set a database identifying string to use in file names and reports.  If the database
							// name is not the default that is included otherwise this is just the host name, however
							// if the host name was not set or is "localhost" or "127.0.0.1" this is not used at all.

							if (lockOK) {
								if (strcmp(DbName, DEFAULT_DB_NAME)) {
									if (!host) {
										snprintf(HostDbName, MAX_STRING, "localhost-%s", DbName);
									} else {
										snprintf(HostDbName, MAX_STRING, "%s-%s", host, DbName);
									}
								} else {
									if (host && strcmp(host, "localhost") && strcmp(host, "127.0.0.1")) {
										lcpystr(HostDbName, host, MAX_STRING);
									}
								}
								StudyKey = studyKey;
								err = 0;
							}
						}
					}
				}
			}

			mysql_free_result(myResult);
		}

		if (mysql_query(MyConnection, "UNLOCK TABLES;")) {
			err = -1;
			log_db_error("Could not unlock study table (1)");
		}

		if (err) {
			if (LOCK_NONE != StudyLock) {
				close_study(1);
			} else {
				mysql_close(MyConnection);
				MyConnection = NULL;
				clear_state(1);
			}
			return err;
		}

		if (wait) {
			mysql_close(MyConnection);
			MyConnection = NULL;
			clear_state(0);
			sleep(WAIT_TIME_SECONDS);
		}

	} while (wait);

	// TV interference-check and TV6 vs. FM studies must use grid mode and indivudal service area mode, do not allow
	// anything else.

	if ((STUDY_TYPE_TV_IX == StudyType) || (STUDY_TYPE_TV6_FM == StudyType)) {
		StudyMode = STUDY_MODE_GRID;
		StudyAreaMode = STUDY_AREA_SERVICE;
	}

	// Study opened and lock acquired.  Do the study setup, first load parameters.

	log_message("Study open for studyKey=%d '%s'", StudyKey, StudyName);

	err = load_study_parameters(templateKey);
	if (err) {
		close_study(1);
		return err;
	}

	// Inform the terrain routines what types of terrain are being requested, mainly this sets UserTerrainRequested
	// if appropriate.  But someday this might also do some cache optimization.

	add_requested_terrain(Params.TerrAvgDb);
	add_requested_terrain(Params.TerrPathDb);

	// Check the cache.

	err = cache_setup(needsUpdate);
	if (err) {
		close_study(1);
		return err;
	}

	// Make sure mode state is consistent.  In points mode load points.

	if (STUDY_MODE_GRID == StudyMode) {

		if (STUDY_AREA_GEOGRAPHY != StudyAreaMode) {
			StudyAreaGeoKey = 0;
			StudyAreaModCount = 0;
			AreaGeoName[0] = '\0';
		}

		PointSetName[0] = '\0';

		// Result tables can only be created in global grid mode in general-purpose study types, and must have an
		// exclusive study lock.

		if (CreateResultTables && ((GRID_TYPE_GLOBAL != Params.GridType) || (LOCK_RUN_EXCL != StudyLock) ||
				((STUDY_TYPE_TV != StudyType) && (STUDY_TYPE_FM != StudyType)))) {
			CreateResultTables = 0;
		}
		if (CreateResultTables) {
			KeepExclLock = 1;
		}

	} else {

		err = load_points(pointSetKey);
		if (err) {
			close_study(1);
			return err;
		}

		StudyAreaMode = STUDY_AREA_SERVICE;
		StudyAreaGeoKey = 0;
		StudyAreaModCount = 0;
		AreaGeoName[0] = '\0';
	}

	// Load all source records, this will also do all initial processing such as replication.

	err = load_sources(needsUpdate);
	if (err) {
		close_study(1);
		return err;
	}

	// Load the scenario pair list.  Each pairing links two scenarios, and specific desired sources in each, for
	// comparison analysis in some special study types e.g. IX check.  The source pointers are resolved as these are
	// loaded, if any are invalid the pairing is silently ignored.  For an IX check study, the query uses a special
	// ordering based on the desired source in the "after" case.

	if (STUDY_TYPE_TV_IX == StudyType) {
		snprintf(query, MAX_QUERY, "SELECT scenario_pair.scenario_key_a, scenario_pair.source_key_a, scenario_pair.scenario_key_b, scenario_pair.source_key_b, scenario_pair.name FROM %s_%d.scenario_pair JOIN %s_%d.source ON (source.source_key = scenario_pair.source_key_b) ORDER BY (CASE WHEN scenario_pair.name LIKE 'MX%%' THEN 1 ELSE 0 END), source.country_key, source.channel, source.state, source.city, scenario_pair.scenario_key_b", DbName, StudyKey, DbName, StudyKey);
	} else {
		snprintf(query, MAX_QUERY, "SELECT scenario_key_a, source_key_a, scenario_key_b, source_key_b, name FROM %s_%d.scenario_pair ORDER BY 3", DbName, StudyKey);
	}
	if (mysql_query(MyConnection, query)) {
		log_db_error("Scenario pair query failed (1)");
		err = -1;

	} else {

		myResult = mysql_store_result(MyConnection);
		if (!myResult) {
			log_db_error("Scenario pair query failed (2)");
			err =  -1;

		} else {

			rowCount = mysql_num_rows(myResult);

			SCENARIO_PAIR *thePair, **pairLink = &ScenarioPairList;
			int scenKeyA, srcKey, scenKeyB;
			SOURCE *srcA, *srcB;

			for (rowIndex = 0; rowIndex < rowCount; rowIndex++) {

				fields = mysql_fetch_row(myResult);
				if (!fields) {
					mysql_free_result(myResult);
					log_db_error("Scenario pair query failed (3)");
					err = -1;
					break;
				}

				scenKeyA = atoi(fields[0]);
				srcKey = atoi(fields[1]);
				if (scenKeyA < 1) {
					continue;
				}
				if ((srcKey < 1) || (srcKey >= SourceIndexSize)) {
					continue;
				}
				srcA = SourceKeyIndex[srcKey];
				if (!srcA) {
					continue;
				}

				scenKeyB = atoi(fields[2]);
				srcKey = atoi(fields[3]);
				if (scenKeyB < 1) {
					continue;
				}
				if ((srcKey < 1) || (srcKey >= SourceIndexSize)) {
					continue;
				}
				srcB = SourceKeyIndex[srcKey];
				if (!srcB) {
					continue;
				}

				thePair = mem_zalloc(sizeof(SCENARIO_PAIR));
				*pairLink = thePair;
				pairLink = &(thePair->next);

				thePair->scenarioKeyA = scenKeyA;
				thePair->sourceA = srcA;

				thePair->scenarioKeyB = scenKeyB;
				thePair->sourceB = srcB;

				lcpystr(thePair->name, fields[4], MAX_STRING);
			}

			mysql_free_result(myResult);
		}
	}

	if (err) {
		close_study(1);
		return err;
	}

	// If the study lock is exclusive, delete all existing result tables.  This is regardless of whether new tables
	// will be created or not; the tables must only contain the results of the most-recent run.

	if (LOCK_RUN_EXCL == StudyLock) {

		snprintf(query, MAX_QUERY, "SHOW TABLES IN %s_%d LIKE 'result\\_%%';", DbName, StudyKey);
		if (mysql_query(MyConnection, query)) {
			log_db_error("Result tables query failed (1)");
			err = -1;

		} else {

			myResult = mysql_store_result(MyConnection);
			if (!myResult) {
				log_db_error("Result tables query failed (2)");
				err = -1;

			} else {

				rowCount = mysql_num_rows(myResult);
				if (rowCount > 0) {

					char **tables = (char **)mem_zalloc(rowCount * sizeof(char *));
					int l;

					for (rowIndex = 0; rowIndex < rowCount; rowIndex++) {
						fields = mysql_fetch_row(myResult);
						if (!fields) {
							log_db_error("Result tables query failed (3)");
							err = -1;
							break;
						}
						l = strlen(fields[0]) + 1;
						tables[rowIndex] = (char *)mem_alloc(l);
						lcpystr(tables[rowIndex], fields[0], l);
					}

					mysql_free_result(myResult);

					if (!err) {
						for (rowIndex = 0; rowIndex < rowCount; rowIndex++) {
							snprintf(query, MAX_QUERY, "DROP TABLE %s_%d.%s;", DbName, StudyKey, tables[rowIndex]);
							if (mysql_query(MyConnection, query)) {
								log_db_error("Result table delete failed");
								err = -1;
								break;
							}
						}
					}

					for (rowIndex = 0; rowIndex < rowCount; rowIndex++) {
						if (tables[rowIndex]) {
							mem_free(tables[rowIndex]);
						}
					}
					mem_free(tables);

				} else {
					mysql_free_result(myResult);
				}
			}
		}

		if (err) {
			close_study(1);
			return err;
		}

		// Change the lock to shared, unless KeepExclLock is set.  In any case, clear the study needs_update flag.
		// Also set the last-run time to now.

		snprintf(query, MAX_QUERY, "LOCK TABLES %s.study WRITE;", DbName);
		if (mysql_query(MyConnection, query)) {
			log_db_error("Could not lock study table (2)");
			err = -1;

		} else {

			if (check_study_lock()) {

				if (KeepExclLock) {

					snprintf(query, MAX_QUERY, "UPDATE %s.study SET needs_update=0, has_result_tables=0, had_result_tables=%d, last_run='%s' WHERE study_key=%d;",
						DbName, hadResultTables, current_time(), StudyKey);
					if (mysql_query(MyConnection, query)) {
						log_db_error("Study record update query failed (2)");
						err = -1;
					}

				} else {

					int lockCount = LockCount + 1;
					snprintf(query, MAX_QUERY, "UPDATE %s.study SET needs_update=0, has_result_tables=0, had_result_tables=%d, last_run='%s', study_lock=%d, lock_count=%d, share_count=1 WHERE study_key=%d;",
						DbName, hadResultTables, current_time(), LOCK_RUN_SHARE, lockCount, StudyKey);
					if (mysql_query(MyConnection, query)) {
						log_db_error("Study lock update query failed (2)");
						err = -1;
					} else {
						StudyLock = LOCK_RUN_SHARE;
						LockCount = lockCount;
					}
				}

			} else {
				err = -1;
			}

			if (mysql_query(MyConnection, "UNLOCK TABLES;")) {
				log_db_error("Could not unlock study table (2)");
				err = -1;
			}
		}

		if (err) {
			close_study(1);
			return err;
		}
	}

	// Ready to run scenarios.

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Exit hook to make sure study is closed.  If it is still open this is an error exit.

static void close_at_exit() {

	close_study(1);
}


//---------------------------------------------------------------------------------------------------------------------
// Test if study lock is exclusive.

int is_lock_exclusive() {

	return (LOCK_RUN_EXCL == StudyLock);
}


//---------------------------------------------------------------------------------------------------------------------
// Check the status of the persistent study lock.  Return is non-zero if the lock has not changed since it was set or
// checked in open_study().  A change means another process violated the locking protocol and data integrity cannot be
// guaranteed, the run should abort.

// Arguments:

//   (none)

// Return is non-zero on success, the current share_count if the lock type is LOCK_RUN_SHARE else -1; return is 0 if
// the lock does not match or an error occurs.

int check_study_lock() {

	if (LOCK_NONE == StudyLock) {
		return 0;
	}

	char query[MAX_QUERY];
	MYSQL_RES *myResult;
	my_ulonglong rowCount;
	MYSQL_ROW fields;

	int result = 0;

	snprintf(query, MAX_QUERY, "SELECT study_lock, lock_count, share_count FROM %s.study WHERE study_key = %d;",
		DbName, StudyKey);
	if (mysql_query(MyConnection, query)) {
		log_db_error("Lock query failed (1)");

	} else {
		myResult = mysql_store_result(MyConnection);
		if (!myResult) {
			log_db_error("Lock query failed (2)");

		} else {
			rowCount = mysql_num_rows(myResult);
			if (rowCount) {
				fields = mysql_fetch_row(myResult);
				if (!fields) {
					log_db_error("Lock query failed (3)");

				} else {
					if ((atoi(fields[0]) == StudyLock) && (atoi(fields[1]) == LockCount)) {
						if (LOCK_RUN_SHARE == StudyLock) {
							result = atoi(fields[2]);
							if (0 == result) {
								result = -1;
							}
						} else {
							result = -1;
						}

					} else {
						log_error("Lock check failed, lock does not match");
					}
				}

			} else {
				log_error("Lock check failed, study no longer exists");
			}

			mysql_free_result(myResult);
		}
	}

	return result;
}


//---------------------------------------------------------------------------------------------------------------------
// Release the persistent study lock and close the database connection.  If database is not open, do nothing.  Check
// the lock state first, if it is still valid as initially acquired in open_study() clear it, otherwise do nothing.
// If the lock is LOCK_RUN_SHARE the share_count is decremented, the lock is only cleared if that is now 0.  However
// if the KeepLock flag is true the lock will not be changed; but it is still checked to verify no change since open.

// Arguments:

//   hadError  True if the close follows an error condition, don't do final analysis/reporting actions.

void close_study(int hadError) {

	if (!MyConnection) {
		return;
	}

	// Do run state cleanup, see run_scenario.c.

	study_closing(hadError);

	// Write settings file if needed.

	if (OutputFlags[SETTING_FILE]) {

		FILE *outFile = open_sum_file(SETTING_FILE_NAME);
		if (outFile) {

			write_report_preamble(REPORT_FILE_SUMMARY, outFile, 1, 0);

			if (ParameterSummary) {
				fprintf(outFile, "Study parameter settings:\n%s\n\n", ParameterSummary);
			}

			if (IxRuleSummary) {
				fprintf(outFile, "Active interference rules:\n\n%s\n\n", IxRuleSummary);
			}

			file_close(outFile);
		}
	}

	// Clear the requested-terrain state, see terrain.c.

	clear_requested_terrain();

	// Unlock study.

	char query[MAX_QUERY];

	snprintf(query, MAX_QUERY, "LOCK TABLES %s.study WRITE;", DbName);
	if (mysql_query(MyConnection, query)) {
		log_db_error("Could not lock study table (3)");

	} else {

		int shareCount = check_study_lock();

		if (shareCount) {

			if (!KeepLock) {

				int lockCount = LockCount;
				int newLock = LOCK_NONE;

				if ((LOCK_RUN_SHARE == StudyLock) && (--shareCount > 0)) {
					newLock = LOCK_RUN_SHARE;
				} else {
					lockCount++;
					shareCount = 0;
				}

				snprintf(query, MAX_QUERY,
		"UPDATE %s.study SET has_result_tables=%d, study_lock=%d, lock_count=%d, share_count=%d WHERE study_key=%d;",
					DbName, DidCreateResultTables, newLock, lockCount, shareCount, StudyKey);
				if (mysql_query(MyConnection, query)) {
					log_db_error("Study lock update query failed (3)");
				}

			} else {

				if (DidCreateResultTables) {
					snprintf(query, MAX_QUERY, "UPDATE %s.study SET has_result_tables=1 WHERE study_key=%d;",
						DbName, StudyKey);
					if (mysql_query(MyConnection, query)) {
						log_db_error("Study index update query failed");
					}
				}
			}
		}

		if (mysql_query(MyConnection, "UNLOCK TABLES;")) {
			log_db_error("Could not unlock study table (3)");
		}
	}

	// Close connection and clear state.

	mysql_close(MyConnection);
	MyConnection = NULL;

	clear_state(1);

	log_message("Study closed");
}


//---------------------------------------------------------------------------------------------------------------------
// Clear global state for study.

// Arguments:

//   clearAll  If true also clear run configuration, otherwise just clear state loaded from study record.

static void clear_state(int clearAll) {

	ScenarioKey = 0;

	DatabaseID[0] = '\0';
	HostDbName[0] = '\0';

	StudyKey = 0;
	StudyType = STUDY_TYPE_UNKNOWN;
	StudyMode = STUDY_MODE_GRID;
	StudyModCount = 0;
	StudyName[0] = '\0';
	ExtDbName[0] = '\0';
	PointSetName[0] = '\0';
	PropagationModel = 1;
	StudyAreaMode = STUDY_AREA_SERVICE;
	StudyAreaGeoKey = 0;
	StudyAreaModCount = 0;
	AreaGeoName[0] = '\0';

	StudyLock = LOCK_NONE;
	LockCount = 0;
	KeepLock = 0;
	KeepExclLock = 0;

	OutputCodes[0] = '\0';
	MapOutputCodes[0] = '\0';

	if (ReportPreamble) {
		mem_free(ReportPreamble);
		ReportPreamble = NULL;
	}
	if (ParameterSummary) {
		mem_free(ParameterSummary);
		ParameterSummary = NULL;
	}
	if (IxRuleSummary) {
		mem_free(IxRuleSummary);
		IxRuleSummary = NULL;
	}

	SCENARIO_PAIR *nextPair, *thePair = ScenarioPairList;
	while (thePair) {
		nextPair = thePair->next;
		thePair->next = NULL;
		mem_free(thePair);
		thePair = nextPair;
	}
	ScenarioPairList = NULL;

	if (clearAll) {

		OutputFlagsSet = 0;
		MapOutputFlagsSet = 0;

		RunNumber = 0;
		CacheUndesired = 1;
		CreateResultTables = 0;
		DidCreateResultTables = 0;
	}
}


//---------------------------------------------------------------------------------------------------------------------
// Misc. support functions.  Find a scenario by name in the open study.

// Arguments:

//   scenarioName  Name of the scenario.

// Return is the scenario key if found, 0 if not found, <0 on error.

int find_scenario_name(char *scenarioName) {

	if (!StudyKey) {
		log_error("find_scenario_name() called with no study open");
		return -1;
	}

	char query[MAX_QUERY];
	MYSQL_RES *myResult;
	my_ulonglong rowCount;
	MYSQL_ROW fields;

	char nameBuf[512];
	unsigned long ulen = strlen(scenarioName);
	if (ulen > 255) {
		ulen = 255;
	}
	mysql_real_escape_string(MyConnection, nameBuf, scenarioName, ulen);
	snprintf(query, MAX_QUERY, "SELECT scenario_key FROM %s_%d.scenario WHERE name = '%s';", DbName, StudyKey,
		nameBuf);

	if (mysql_query(MyConnection, query)) {
		log_db_error("Scenario name query failed (1)");
		return -1;
	}

	myResult = mysql_store_result(MyConnection);
	if (!myResult) {
		log_db_error("Scenario name query failed (2)");
		return -1;
	}

	rowCount = mysql_num_rows(myResult);
	if (!rowCount) {
		mysql_free_result(myResult);
		log_error("Scenario '%s' not found", scenarioName);
		return 0;
	}

	fields = mysql_fetch_row(myResult);
	if (!fields) {
		mysql_free_result(myResult);
		log_db_error("Scenario name query failed (3)");
		return -1;
	}

	int scenarioKey = atoi(fields[0]);
	mysql_free_result(myResult);

	return scenarioKey;
}


//---------------------------------------------------------------------------------------------------------------------
// Parse a flags argument string into globals.  This does not fail, bad flags are just ignored.

// Arguments:

//   flags     Output flags, a string of letter codes selecting desired options or file-related behaviors.  Flag codes
//               may be followed by a positive integer value customizing the behavior.  In general anything not
//               recognized is simply ignored.
//   outFlags  Flags translated to array of integers, character value of flag code is index into the array, value is
//                0 if code did not appear, >0 if it did, 1 if no number followed, else the following number value.
//   maxFlag   Size of outflags array, maximum flag index.

void parse_flags(char *flags, int *outFlags, int maxFlag) {

	static char *opt = NULL;
	static int optLenMax = 0;

	int i, flag, lastFlag = -1, len = strlen(flags), optLen = 0, optVal;

	if (len > optLenMax) {
		optLenMax = len + 10;
		opt = (char *)mem_realloc(opt, optLenMax);
	}

	for (i = 0; i < maxFlag; i++) {
		outFlags[i] = 0;
	}

	for (i = 0; i < len; i++) {

		flag = toupper(flags[i]) - 'A';

		if (flag < 0) {
			flag = flags[i] - '0';
			if ((flag >= 0) && (flag < 10)) {
				opt[optLen++] = flags[i];
			} else {
				optLen = 0;
			}
			continue;
		}

		if (optLen > 0) {
			if (lastFlag >= 0) {
				opt[optLen] = '\0';
				optVal = atoi(opt);
				if (optVal > 0) {
					outFlags[lastFlag] = optVal;
				}
			}
			optLen = 0;
		}

		if (flag < maxFlag) {
			outFlags[flag] = 1;
			lastFlag = flag;
		} else {
			lastFlag = -1;
		}
	}

	if ((optLen > 0) && (lastFlag >= 0)) {
		opt[optLen] = '\0';
		optVal = atoi(opt);
		if (optVal > 0) {
			outFlags[lastFlag] = optVal;
		}
	}
}
