//
//  scenario.c
//  TVStudy
//
//  Copyright (c) 2012-2023 Hammett & Edison, Inc.  All rights reserved.


// The main study code, load and run a scenario from an open study database.


#include "tvstudy.h"

#include <ctype.h>
#include <sys/errno.h>
#include <sys/wait.h>


//---------------------------------------------------------------------------------------------------------------------

typedef struct {                       // Structure for accumulating composite coverage and image data.
	unsigned char countryKey:2;        // Country for point in main grid array CompositePoints, 0 if unoccupied.
	unsigned char extraCountryKey:2;   // Country for extra point in cell, 0 if none.  Not used in extra point.
	unsigned char result:2;            // Composite result code for the point.
	unsigned char hasPop:1;            // True if point has population.
	unsigned char didOutput:1;         // True once cell is written to point file output.  Not used in extra point.
	IMAGE_POINT imagePoint;            // Image data for grid composite image.  Not used in extra point.
	short margin;                      // Desired signal margin at 0.01 dB resolution for best-case composite result.
	unsigned short sourceKey;          // Desired source key for best-case composite result.
	unsigned char serviceCount;        // Count of sources providing coverage at the point.
} COMPOSITE_POINT;

static int do_load_scenario(int scenarioKey);
static int do_run_scenario();
static int do_start_mapfiles();
static int do_image_setup();
static int do_image_legend(FILE *kmlFile, int fromTop);
static int do_finish_kml();
static int do_global_run();
static int do_write_compcov();
static int do_write_image(int regionID, SOURCE *source, FILE *kmlFile);
static void do_result_report_tables();
static int do_local_run();
static int do_points_run();
static int do_bysource_kml(SOURCE *source);
static int render_image(char *psFileName, char *imgFileName, int width, int height);
static int write_source_shapes(SOURCE *source, MAPFILE *mapSrcs, MAPFILE *mapConts, int desiredSourceKey,
	double cullingDistance, FILE *kmlFile, int kmlVis);
static int write_points(MAPFILE *mapPointsShp, FILE *pointsFile);
static int analyze_source(SOURCE *source);
static int analyze_grid();
static int analyze_points(POINT *points, SOURCE *dsource);
static double dtv_codu_adjust(int, double);
static COMPOSITE_POINT *get_composite_point(int countryKey, int latGridIndex, int lonGridIndex, int makePoint);
static int run_process(char **args, char *workDir);

// Public globals.

int ScenarioKey = 0;             // Scenario key and name.
char ScenarioName[MAX_STRING];

int IXMarginSourceKey = 0;   // Set to activate IX margin output, see run_scenario().

UND_TOTAL WirelessUndesiredTotals[MAX_COUNTRY];   // Used to tally net wireless interference in OET-74 study.

DES_TOTAL CompositeTotals[MAX_COUNTRY];   // Tally net coverage for scenario composite output, global grid mode only.

// Private globals.

#define CALC_CACHE_COUNT  1000000   // Max number of field calculations between cache updates.

static FILE *OutputFile[N_OUTPUT_FILES];             // Output file state.
static MAPFILE *MapOutputFile[N_MAP_OUTPUT_FILES];
static MAPFILE *DesiredSourcesKML = NULL;
static MAPFILE *UndesiredSourcesKML = NULL;
static MAPFILE *ContoursKML = NULL;
static FILE *ImageFileKML = NULL;
static int ReportSumHeader = 0;
static int CSVSumHeader = 0;

static double *IXMarginAz = NULL;   // Data for IX margin CSV output.
static double *IXMarginDb = NULL;
static int *IXMarginPop = NULL;
static int IXMarginCount = 0;
static int IXMarginMaxCount = -1;

static int DoPoints = 0;      // Internal function flags, see do_run_scenario().
static int DoComposite = 0;
static int DoTable = 0;
static int DoImage = 0;
static int DoCompositeImage = 0;

static int DidWriteImageKML = 0;    // Flags setting content for top-level KML map file.
static int DidWriteCovPtsKML = 0;
static int DidWriteSelfIXKML = 0;

static int ResultsRowCount = 0;   // Count of rows written to temporary database table.

// All-points array used for composite coverage, image, and point output flags, see do_global_run().

static GRID *CompositeGrid = NULL;
static COMPOSITE_POINT *CompositePoints = NULL;

// A small percentage of cells may have an extra study point due to population in multiple countries.  However there
// are relatively very few of those, quite often none at all, so extra points are stored in a separate list.  The main
// array element has extraCountryKey set when an extra point exists.  Since population data is provided only for the
// U.S., Canada, and Mexico, it is not possible for a cell to have more than two points.

static COMPOSITE_POINT *ExtraCompositePoints = NULL;
static int MaxExtraCompositePointCount = 0;
static int ExtraCompositePointCount = 0;
static int *ExtraCompositePointLatGridIndex = NULL;
static int *ExtraCompositePointLonGridIndex = NULL;

// Color information for image output, see do_image_setup().

#define MAX_IMAGE_COLOR_COUNT 64

static int ImageColorCount = 0;
static double ImageLevel[MAX_IMAGE_COLOR_COUNT];
static double ImageColorR[MAX_IMAGE_COLOR_COUNT];
static double ImageColorG[MAX_IMAGE_COLOR_COUNT];
static double ImageColorB[MAX_IMAGE_COLOR_COUNT];

static int ImageMapStartIndex = -1;

#define IMAGE_LEVEL_NO_SERVICE   -999.
#define IMAGE_LEVEL_INTERFERENCE -998.
#define IMAGE_LEVEL_BACKGROUND   -997.

static int ImageNoServiceIndex = -1;
static int ImageInterferenceIndex = -1;
static int ImageBackgroundIndex = -1;

static int ImageColorIndexUsed[MAX_IMAGE_COLOR_COUNT];

#define IMAGE_OVERLAY_COLOR "90ffffff"

// Image rendering parameters, see do_write_image().

#define IMAGE_RESOLUTION 300

#define IMAGE_TILE_SIZE 3600
#define IMAGE_LOD 128

#define IMAGE_LEGEND_NAME "legend.png"
#define IMAGE_LEGEND_WIDTH 180
#define IMAGE_LEGEND_TITLE "Legend"

// Attribute lists and parameters for KML output.

#define COV_MAX_ATTRIBUTE_COUNT 35
static SHAPEATTR CovPtsAttributes[COV_MAX_ATTRIBUTE_COUNT];
static int CovPtsAttributeCount = 0;

#define SELFIX_ATTRIBUTE_COUNT 8
static SHAPEATTR SelfIXFileAttributes[SELFIX_ATTRIBUTE_COUNT];

#define MAX_KML_FILE_COUNT 5000

static char CovPtsKMLSubDir[MAX_STRING];
static char SelfIXKMLSubDir[MAX_STRING];

static float *Profile = NULL;    // The terrain profile retrieved by a project_field() call, if any.  If the call did
static int ProfileCount = 0;     //  not extract a profile for any reason, ProfileCount is set to 0.  In that case
static double ProfilePpk = 0.;   //  Profile may or may not be NULL.


//---------------------------------------------------------------------------------------------------------------------
// Run a scenario, generate output.  A study must be open.

// Arguments:

//   scenarioKey  Primary key for scenario record in current study database.

// Return is <0 for major error, >0 for minor error, 0 for no error.

int run_scenario(int scenarioKey) {

	if (!StudyKey) {
		log_error("run_scenario() called with no study open");
		return 1;
	}

	// Initialize output file arrays on first call only, some files may remain open across multiple calls.

	static int init = 1;
	if (init) {
		int i;
		for (i = 0; i < N_OUTPUT_FILES; i++) {
			OutputFile[i] = NULL;
		}
		for (i = 0; i < N_MAP_OUTPUT_FILES; i++) {
			MapOutputFile[i] = NULL;
		}
		init = 0;
	}

	// Initialize output flags if needed, caller may have done that already.

	if (!OutputFlagsSet) {
		parse_flags(OutputCodes, OutputFlags, MAX_OUTPUT_FLAGS);
		OutputFlagsSet = 1;
	}
	if (!MapOutputFlagsSet) {
		parse_flags(MapOutputCodes, MapOutputFlags, MAX_MAP_OUTPUT_FLAGS);
		MapOutputFlagsSet = 1;
	}

	// Open summary report, CSV, and cell files if needed, these accumulate output across all scenarios run for the
	// open study so these stay open across calls and will be closed in study_closing().  In the special pair study
	// cell format temporary output files are used which are post-processed by an external utility to generate final
	// output, those are also at the summary level and stay open across all scenarios.

	FILE *outFile;

	if (OutputFlags[REPORT_FILE_SUMMARY] && !OutputFile[REPORT_FILE_SUMMARY]) {
		outFile = open_sum_file(REPORT_FILE_NAME);
		if (!outFile) return 1;
		OutputFile[REPORT_FILE_SUMMARY] = outFile;
		write_report_preamble(REPORT_FILE_SUMMARY, outFile, 1, 1);
		ReportSumHeader = 1;
	}

	if (OutputFlags[CSV_FILE_SUMMARY] && !OutputFile[CSV_FILE_SUMMARY]) {
		outFile = open_sum_file(CSV_FILE_NAME);
		if (!outFile) return 1;
		OutputFile[CSV_FILE_SUMMARY] = outFile;
		write_csv_preamble(CSV_FILE_SUMMARY, outFile, 1, 1);
		CSVSumHeader = 1;
	}

	if (OutputFlags[CELL_FILE_SUMMARY] && !OutputFile[CELL_FILE_SUMMARY]) {
		outFile = open_sum_file(CELL_FILE_NAME);
		if (!outFile) return 1;
		OutputFile[CELL_FILE_SUMMARY] = outFile;
	}

	if (OutputFlags[CELL_FILE_PAIRSTUDY] && !OutputFile[CELL_FILE_PAIRSTUDY]) {
		outFile = open_temp_file();
		if (!outFile) return 1;
		OutputFile[CELL_FILE_PAIRSTUDY] = outFile;
	}

	// Load and run the scenario.

	log_message("Loading scenarioKey=%d", scenarioKey);

	int err = do_load_scenario(scenarioKey);
	if (err) return err;

	log_message("Running scenario '%s'", ScenarioName);

	err = do_run_scenario();

	// Write abort to cell files if error occurred.

	if (err) {
		if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
			fputs("[abort]\n", outFile);
		}
		if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
			fputs("[abort]\n", outFile);
		}
		if ((outFile = OutputFile[CELL_FILE_PAIRSTUDY])) {
			fputs("[abort]\n", outFile);
		}
	}

	// Close all scenario-level files that might be open.

	if (OutputFile[REPORT_FILE_DETAIL]) {
		file_close(OutputFile[REPORT_FILE_DETAIL]);
		OutputFile[REPORT_FILE_DETAIL] = NULL;
	}
	if (OutputFile[CSV_FILE_DETAIL]) {
		file_close(OutputFile[CSV_FILE_DETAIL]);
		OutputFile[CSV_FILE_DETAIL] = NULL;
	}
	if (OutputFile[CELL_FILE_DETAIL]) {
		file_close(OutputFile[CELL_FILE_DETAIL]);
		OutputFile[CELL_FILE_DETAIL] = NULL;
	}
	if (OutputFile[CELL_FILE_CSV]) {
		file_close(OutputFile[CELL_FILE_CSV]);
		OutputFile[CELL_FILE_CSV] = NULL;
	}
	if (OutputFile[CELL_FILE_CSV_D]) {
		file_close(OutputFile[CELL_FILE_CSV_D]);
		OutputFile[CELL_FILE_CSV_D] = NULL;
	}
	if (OutputFile[CELL_FILE_CSV_U]) {
		file_close(OutputFile[CELL_FILE_CSV_U]);
		OutputFile[CELL_FILE_CSV_U] = NULL;
	}
	if (OutputFile[CELL_FILE_CSV_WL_U]) {
		file_close(OutputFile[CELL_FILE_CSV_WL_U]);
		OutputFile[CELL_FILE_CSV_WL_U] = NULL;
	}
	if (OutputFile[CELL_FILE_CSV_WL_U_RSS]) {
		file_close(OutputFile[CELL_FILE_CSV_WL_U_RSS]);
		OutputFile[CELL_FILE_CSV_WL_U_RSS] = NULL;
	}
	if (OutputFile[SELFIX_CSV]) {
		file_close(OutputFile[SELFIX_CSV]);
		OutputFile[SELFIX_CSV] = NULL;
	}
	if (MapOutputFile[MAP_OUT_SHAPE_POINTS]) {
		close_mapfile(MapOutputFile[MAP_OUT_SHAPE_POINTS]);
		MapOutputFile[MAP_OUT_SHAPE_POINTS] = NULL;
	}
	if (MapOutputFile[MAP_OUT_SHAPE_COVPTS]) {
		close_mapfile(MapOutputFile[MAP_OUT_SHAPE_COVPTS]);
		MapOutputFile[MAP_OUT_SHAPE_COVPTS] = NULL;
	}
	if (MapOutputFile[MAP_OUT_SHAPE_CMPPTS]) {
		close_mapfile(MapOutputFile[MAP_OUT_SHAPE_CMPPTS]);
		MapOutputFile[MAP_OUT_SHAPE_CMPPTS] = NULL;
	}
	if (MapOutputFile[MAP_OUT_SHAPE_SELFIX]) {
		close_mapfile(MapOutputFile[MAP_OUT_SHAPE_SELFIX]);
		MapOutputFile[MAP_OUT_SHAPE_SELFIX] = NULL;
	}
	if (MapOutputFile[MAP_OUT_KML_COVPTS]) {
		close_mapfile(MapOutputFile[MAP_OUT_KML_COVPTS]);
		MapOutputFile[MAP_OUT_KML_COVPTS] = NULL;
	}
	if (MapOutputFile[MAP_OUT_KML_SELFIX]) {
		close_mapfile(MapOutputFile[MAP_OUT_KML_SELFIX]);
		MapOutputFile[MAP_OUT_KML_SELFIX] = NULL;
	}
	if (OutputFile[POINTS_FILE]) {
		file_close(OutputFile[POINTS_FILE]);
		OutputFile[POINTS_FILE] = NULL;
	}

	SOURCE *source = Sources;
	int sourceIndex, i;
	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
		for (i = 0; i < RESULT_COUNT; i++) {
			if (source->kmlCovPts[i]) {
				close_mapfile(source->kmlCovPts[i]);
				source->kmlCovPts[i] = NULL;
			}
		}
		if (source->kmlSelfIX) {
			close_mapfile(source->kmlSelfIX);
			source->kmlSelfIX = NULL;
		}
	}
	if (ImageFileKML) {
		kml_close(ImageFileKML);
		ImageFileKML = NULL;
	}
	if (DesiredSourcesKML) {
		close_mapfile(DesiredSourcesKML);
		DesiredSourcesKML = NULL;
	}
	if (UndesiredSourcesKML) {
		close_mapfile(UndesiredSourcesKML);
		UndesiredSourcesKML = NULL;
	}
	if (ContoursKML) {
		close_mapfile(ContoursKML);
		ContoursKML = NULL;
	}

	// Clear state, free the composite points grid if allocated.

	IXMarginSourceKey = 0;
	IXMarginCount = 0;

	DoPoints = 0;
	DoComposite = 0;
	DoTable = 0;
	DoImage = 0;
	DoCompositeImage = 0;
	DidWriteImageKML = 0;
	DidWriteCovPtsKML = 0;
	DidWriteSelfIXKML = 0;

	if (CompositeGrid) {
		free_grid(CompositeGrid);
		CompositeGrid = NULL;
	}
	if (CompositePoints) {
		mem_free(CompositePoints);
		CompositePoints = NULL;
	}

	return err;
}

// Load scenario source list, set up sources.  Sets the inScenario, isDesired, and isUndesired flags on all sources,
// then makes sure all with isDesired set have service areas and undesired source lists.  Also loads scenario
// parameters.  Usual error return; <0 is a serious error, >0 a minor error.

static int do_load_scenario(int scenarioKey) {

	if (scenarioKey == ScenarioKey) {
		return 0;
	}

	if (!check_study_lock()) {
		return -1;
	}

	ScenarioKey = 0;

	int err, i;

	char query[MAX_QUERY];
	MYSQL_RES *myResult;
	MYSQL_ROW fields;

	snprintf(query, MAX_QUERY, "SELECT name FROM %s_%d.scenario WHERE scenario_key = %d;", DbName, StudyKey,
		scenarioKey);
	if (mysql_query(MyConnection, query)) {
		log_db_error("Scenario query failed (1)");
		return -1;
	}

	myResult = mysql_store_result(MyConnection);
	if (!myResult) {
		log_db_error("Scenario query failed (2)");
		return -1;
	}

	if (!mysql_num_rows(myResult)) {
		mysql_free_result(myResult);
		log_error("Scenario not found for scenarioKey=%d", scenarioKey);
		return 1;
	}

	fields = mysql_fetch_row(myResult);
	if (!fields) {
		mysql_free_result(myResult);
		log_db_error("Scenario query failed (3)");
		return -1;
	}

	lcpystr(ScenarioName, fields[0], MAX_STRING);

	mysql_free_result(myResult);

	// Load scenario parameters.

	err = load_scenario_parameters(scenarioKey);
	if (err) return err;

	// Clear state from a previous scenario.

	SOURCE *source;
	int sourceIndex, sourceCount, sourceKey;

	source = Sources;
	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
		source->inScenario = 0;
		source->isDesired = 0;
		source->isUndesired = 0;
		if (source->undesireds) {
			mem_free(source->undesireds);
			source->undesireds = NULL;
			source->undesiredCount = 0;
		}
		if (source->selfIXTotals) {
			mem_free(source->selfIXTotals);
			source->selfIXTotals = NULL;
		}
		memset(source->totals, 0, (MAX_COUNTRY * sizeof(DES_TOTAL)));
		source->next = NULL;
	}

	memset(WirelessUndesiredTotals, 0, (MAX_COUNTRY * sizeof(UND_TOTAL)));

	memset(CompositeTotals, 0, (MAX_COUNTRY * sizeof(DES_TOTAL)));

	// Load the list of sources in the scenario.

	snprintf(query, MAX_QUERY,
		"SELECT source_key, is_desired, is_undesired FROM %s_%d.scenario_source WHERE scenario_key = %d",
		DbName, StudyKey, scenarioKey);
	if (mysql_query(MyConnection, query)) {
		log_db_error("Scenario source query failed (1)");
		return -1;
	}

	myResult = mysql_store_result(MyConnection);
	if (!myResult) {
		log_db_error("Scenario source query failed (2)");
		return -1;
	}

	sourceCount = (int)mysql_num_rows(myResult);
	if (!sourceCount) {
		mysql_free_result(myResult);
		log_error("No sources found for scenarioKey=%d", scenarioKey);
		return 1;
	}

	int desiredCount = 0, wirelessCount = 0;

	for (sourceIndex = 0; sourceIndex < sourceCount; sourceIndex++) {

		fields = mysql_fetch_row(myResult);
		if (!fields) {
			mysql_free_result(myResult);
			log_db_error("Scenario source query failed (3)");
			return -1;
		}

		sourceKey = atoi(fields[0]);
		if ((sourceKey >= SourceIndexSize) || (NULL == (source = SourceKeyIndex[sourceKey]))) {
			mysql_free_result(myResult);
			log_error("Source not found for sourceKey=%d in scenarioKey=%d", sourceKey, scenarioKey);
			return 1;
		}

		source->inScenario = 1;
		source->isDesired = (short)atoi(fields[1]);
		source->isUndesired = (short)atoi(fields[2]);

		// Wireless records cannot be desireds, also these will need further setup as undesireds so keep a count.

		if (RECORD_TYPE_WL == source->recordType) {
			source->isDesired = 0;
			if (source->isUndesired) {
				wirelessCount++;
			}
		}

		if (source->isDesired) {
			desiredCount++;
		}
	}

	mysql_free_result(myResult);

	// Set up service areas as needed and build undesired source lists.

	hb_log_begin(desiredCount);
	if (Terminate) return -1;

	SOURCE *sources, *asource;
	GEOPOINTS *pts;
	double dist, maxdist;

	source = Sources;
	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {

		if (source->isDesired) {

			hb_log_tick();
			if (Terminate) return -1;

			if (source->isParent) {
				sources = source->dtsSources;
			} else {
				sources = source;
				source->next = NULL;
			}

			// Load service area geographies as needed.  Also for sources with SERVAREA_RADIUS or SERVAREA_NO_BOUNDS
			// modes, implement those here by creating a custom circle geography.

			for (asource = sources; asource; asource = asource->next) {

				if (asource->geography) {
					continue;
				}

				switch (asource->serviceAreaMode) {

					case SERVAREA_GEOGRAPHY_FIXED: {
						asource->geography = get_geography(asource->serviceAreaKey);
						if (!asource->geography) return 1;
						break;
					}

					case SERVAREA_GEOGRAPHY_RELOCATED: {
						asource->geography = get_geography(asource->serviceAreaKey);
						if (!asource->geography) return 1;
						relocate_geography(asource);
						break;
					}

					case SERVAREA_RADIUS:
					case SERVAREA_NO_BOUNDS: {
						asource->geography = geography_alloc(-(asource->sourceKey), GEO_TYPE_CIRCLE, asource->latitude,
							asource->longitude, 0);
						asource->geography->a.radius = asource->serviceAreaArg;
						break;
					}
				}
			}

			// Define service areas as needed, this will project contours and render geographies.  Clear the cache
			// flag so the source cache is updated below.

			if (!source->serviceAreaValid) {
				err = set_service_area(source);
				if (err) return err;
				source->serviceAreaValid = 1;
				source->cached = 0;
			}

			// In the front-end app undesired searches use the "rule extra distance" algorithm which provides a single
			// extra distance added to the interference rule culling distance to find undesired records by a site-to-
			// site distance check.  With default values for the parameters controlling that algorithm it provides a
			// worst-case maximum distance to an FCC curves contour.  However if the actual service area is defined by
			// some other method that distance may be insufficient causing undesireds that would be within culling
			// distance of some study points to be missing from the scenario.  To alert the user to that possibility,
			// check the actual rendered service area boundary vs. the extra distance and show a warning as needed.

			for (asource = sources; asource; asource = asource->next) {
				pts = render_service_area(asource);
				if (!pts) {
					continue;
				}
				maxdist = 0.;
				for (i = 0; i < pts->nPts; i++) {
					bear_distance(asource->latitude, asource->longitude, pts->ptLat[i], pts->ptLon[i], NULL, NULL,
						&dist, Params.KilometersPerDegree);
					if (dist > maxdist) {
						maxdist = dist;
					}
				}
				if ((maxdist - asource->ruleExtraDistance) > 0.1) {
					log_message("Max service area distance %.2f > rule extra %.2f for sourceKey=%d  %s", maxdist,
						asource->ruleExtraDistance, asource->sourceKey, source_label(asource));
				}
			}

			err = find_undesired(source);
			if (err) return err;

			if (source->isParent && Params.CheckSelfInterference) {
				source->selfIXTotals = (UND_TOTAL *)mem_zalloc(MAX_COUNTRY * sizeof(UND_TOTAL));
			}
		}
	}

	hb_log_end();
	if (Terminate) return -1;

	// Do setup on wireless sources.  The frequency is defined by a scenario parameter, however when the frequency
	// changes the source's cache has to be reset.  The frequency was initialized in read_source_cache(), if there was
	// an existing cache, so this can just compare the frequency in that case and clear the cache if it changes.  In
	// any case, a source cache is written if needed; wireless records were deliberately not cached with others in
	// source_setup() since at that point the frequency was undefined if the source had no existing cache.  Note the
	// wireless bandwidth is not relevant here, changing only the bandwidth does not affect pathloss predictions, it
	// just affects D/U analysis the results of which are not cached.  Also update source caches for desired sources
	// that had service area updates above.

	log_message("Updating caches");

	hb_log_begin(wirelessCount + desiredCount);
	if (Terminate) return -1;

	source = Sources;
	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {

		if ((RECORD_TYPE_WL == source->recordType) && source->isUndesired) {

			hb_log_tick();
			if (Terminate) return -1;

			if (source->cached && (source->frequency != Params.WirelessFrequency)) {
				clear_cache(source);
			}
			source->frequency = Params.WirelessFrequency;
			if (!source->cached) {
				write_source_cache(source);
			}
		}

		if (source->isDesired) {

			hb_log_tick();
			if (Terminate) return -1;

			if (!source->cached) {
				write_source_cache(source);
			}
		}
	}

	hb_log_end();
	if (Terminate) return -1;

	// Done.  

	ScenarioKey = scenarioKey;

	return 0;
}

// Run the scenario.  First open scenario-level cell files, these are closed as soon as the scenario run is done.  The
// detail CSV cell file format writes different data to multiple files, see analyze_points().  Some are specific to an
// OET-74 study.

static int do_run_scenario() {

	int err = 0;

	FILE *outFile;

	if (OutputFlags[CELL_FILE_DETAIL]) {
		outFile = open_out_file(CELL_FILE_NAME);
		if (!outFile) return 1;
		OutputFile[CELL_FILE_DETAIL] = outFile;
	}

	if (OutputFlags[CELL_FILE_CSV]) {

		outFile = open_out_file_dir(CELL_CSV_SOURCE_FILE_NAME, CELL_CSV_SUBDIR);
		if (!outFile) return 1;
		OutputFile[CELL_FILE_CSV] = outFile;

		outFile = open_out_file_dir(CELL_CSV_D_FILE_NAME, CELL_CSV_SUBDIR);
		if (!outFile) return 1;
		OutputFile[CELL_FILE_CSV_D] = outFile;

		outFile = open_out_file_dir(CELL_CSV_U_FILE_NAME, CELL_CSV_SUBDIR);
		if (!outFile) return 1;
		OutputFile[CELL_FILE_CSV_U] = outFile;

		if (STUDY_TYPE_TV_OET74 == StudyType) {

			outFile = open_out_file_dir(CELL_CSV_WL_U_FILE_NAME, CELL_CSV_SUBDIR);
			if (!outFile) return 1;
			OutputFile[CELL_FILE_CSV_WL_U] = outFile;

			outFile = open_out_file_dir(CELL_CSV_WL_U_RSS_FILE_NAME, CELL_CSV_SUBDIR);
			if (!outFile) return 1;
			OutputFile[CELL_FILE_CSV_WL_U_RSS] = outFile;
		}
	}

	// Open DTS self-interference CSV file if requested.

	if (OutputFlags[SELFIX_CSV] && Params.CheckSelfInterference) {
		outFile = open_out_file(SELFIX_CSV_FILE_NAME);
		if (!outFile) return 1;
		if (STUDY_MODE_POINTS == StudyMode) {
			fputs("SourceKey,PointName,CountryKey,Latitude,Longitude,", outFile);
		} else {
			fputs("SourceKey,LatIndex,LonIndex,CountryKey,Latitude,Longitude,Area,Population,", outFile);
		}
		fputs("DU,DUMargin,DSite,DSignal,URSS,USite,USignal,UDeltaT\n", outFile);
		OutputFile[SELFIX_CSV] = outFile;
	}

	// Initial output to cell files.

	if ((outFile = OutputFile[CELL_FILE_PAIRSTUDY])) {
		fprintf(outFile, "[case]\n%d,%d,%d\n", StudyKey, ScenarioKey, CellLatSize);
	}

	if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
		fprintf(outFile, "[scenario]\n%s\n%s\n%s\n%s\n%s\n%.2f,%d,%d\n", HostDbName, ExtDbName, StudyName,
			ScenarioName, log_open_time(), Params.CellSize, Params.GridType, CellLatSize);
		if (RunComment) {
			fprintf(outFile, "[comment]\n%s\n[endcomment]\n", RunComment);
		}
		fputs("[sources]\n", outFile);
	}

	if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
		fprintf(outFile, "[scenario]\n%s\n%s\n%s\n%s\n%s\n%.2f,%d,%d\n", HostDbName, ExtDbName, StudyName,
			ScenarioName, log_open_time(), Params.CellSize, Params.GridType, CellLatSize);
		if (RunComment) {
			fprintf(outFile, "[comment]\n%s\n[endcomment]\n", RunComment);
		}
		fputs("[sources]\n", outFile);
	}

	SOURCE *source = Sources, *dtsSource, *asource;
	int sourceIndex, fieldCount, dflag, uflag, azi, desCount = 0;
	char patName[MAX_STRING];

	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {

		if (source->inScenario && (source->isDesired || source->isUndesired)) {

			if (source->isDesired) {
				dflag = 1;
				desCount++;
			} else {
				dflag = 0;
			}
			if (source->isUndesired) {
				uflag = 1;
			} else {
				uflag = 0;
			}

			if (source->isParent) {
				fieldCount = 0;
				for (dtsSource = source->dtsSources; dtsSource; dtsSource = dtsSource->next) {
					fieldCount++;
				}
			} else {
				fieldCount = 1;
			}

			if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
				if (dflag) {
					if (RECORD_TYPE_WL == source->recordType) {
						fprintf(outFile, "%d,%d,%d,%d,%d,%.2f,,%d,,,%d,%d,%d,%d,%s,%s\n", source->sourceKey, dflag,
							uflag, source->countryKey, source->serviceTypeKey, source->frequency, fieldCount,
							source->grid->cellBounds.southLatIndex, source->grid->cellBounds.eastLonIndex,
							source->grid->cellBounds.northLatIndex, source->grid->cellBounds.westLonIndex,
							source->fileNumber, source->callSign);
					} else {
						fprintf(outFile, "%d,%d,%d,%d,%d,%d,%.6f,%d,%d,%d,%d,%d,%d,%d,%s,%s\n", source->sourceKey,
							dflag, uflag, source->countryKey, source->serviceTypeKey, source->channel,
							source->serviceLevel, fieldCount, source->facility_id, source->siteNumber,
							source->grid->cellBounds.southLatIndex, source->grid->cellBounds.eastLonIndex,
							source->grid->cellBounds.northLatIndex, source->grid->cellBounds.westLonIndex,
							source->fileNumber, source->callSign);
					}
				} else {
					if (RECORD_TYPE_WL == source->recordType) {
						fprintf(outFile, "%d,%d,%d,%d,%d,%.2f,,%d,,,%s,%s\n", source->sourceKey, dflag, uflag,
							source->countryKey, source->serviceTypeKey, source->frequency, fieldCount,
							source->fileNumber, source->callSign);
					} else {
						fprintf(outFile, "%d,%d,%d,%d,%d,%d,%.6f,%d,%d,%d,%s,%s\n", source->sourceKey, dflag, uflag,
							source->countryKey, source->serviceTypeKey, source->channel, source->serviceLevel,
							fieldCount, source->facility_id, source->siteNumber, source->fileNumber, source->callSign);
					}
				}
			}

			if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
				if (RECORD_TYPE_WL == source->recordType) {
					fprintf(outFile, "%d,%d,%d,%d,%d,%.2f,,,%s,%s\n", source->sourceKey, dflag, uflag,
						source->countryKey, source->serviceTypeKey, source->frequency, source->fileNumber,
						source->callSign);
				} else {
					fprintf(outFile, "%d,%d,%d,%d,%d,%d,%d,%d,%s,%s\n", source->sourceKey, dflag, uflag,
						source->countryKey, source->serviceTypeKey, source->channel, source->facility_id,
						source->siteNumber, source->fileNumber, source->callSign);
				}
			}

			if ((outFile = OutputFile[CELL_FILE_CSV])) {
				if (RECORD_TYPE_WL == source->recordType) {
					fprintf(outFile, "%d,%d,%d,%d,%d,%.2f,,%d,,,%s,%s\n", source->sourceKey, dflag, uflag,
						source->countryKey, source->serviceTypeKey, source->frequency, fieldCount, source->fileNumber,
						source->callSign);
				} else {
					fprintf(outFile, "%d,%d,%d,%d,%d,%d,%.6f,%d,%d,%d,%s,%s\n", source->sourceKey, dflag, uflag,
						source->countryKey, source->serviceTypeKey, source->channel, source->serviceLevel, fieldCount,
						source->facility_id, source->siteNumber, source->fileNumber, source->callSign);
				}
			}

			if (OutputFlags[DERIVE_HPAT_FILE] && source->isDesired) {
				if (source->isParent) {
					asource = source->dtsSources;
				} else {
					asource = source;
					source->next = NULL;
				}
				while (asource) {
					if (asource->conthpat) {
						if (asource->siteNumber > 0) {
							snprintf(patName, MAX_STRING, "hpat-%s_%s_%s_%s_%d.csv", asource->callSign,
								channel_label(asource), asource->serviceCode, asource->status, asource->siteNumber);
						} else {
							snprintf(patName, MAX_STRING, "hpat-%s_%s_%s_%s.csv", asource->callSign,
								channel_label(asource), asource->serviceCode, asource->status);
						}
						outFile = open_out_file_dir(patName, HPAT_SUBDIR);
						if (!outFile) return 1;
						for (azi = 0; azi < 360; azi++) {
							fprintf(outFile, "%d,%.4f\n", azi, pow(10., (asource->conthpat[azi] / 20.)));
						}
						file_close(outFile);
					}
					asource = asource->next;
				}
			}
		}
	}

	if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
		fputs("[endsources]\n", outFile);
	}

	if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
		fputs("[endsources]\n", outFile);
	}

	// Do initial map file opens and writes as needed.

	if (MapOutputFlags[MAP_OUT_SHAPE] || MapOutputFlags[MAP_OUT_KML]) {
		err = do_start_mapfiles();
		if (err) return err;
	}

	// Open points file if needed.

	if (OutputFlags[POINTS_FILE]) {
		outFile = open_out_file(POINTS_FILE_NAME);
		if (!outFile) return 1;
		if (STUDY_MODE_POINTS == StudyMode) {
			fputs("PointName,CountryKey,Latitude,Longitude,ReceiveHeight,GroundHeight", outFile);
		} else {
			if (GRID_TYPE_GLOBAL == Params.GridType) {
				fputs("LatIndex,LonIndex,CountryKey,PointKey,Latitude,Longitude,GroundHeight,Area,", outFile);
				fputs("Population,Households", outFile);
			} else {
				fputs("SourceKey,LatIndex,LonIndex,CountryKey,Latitude,Longitude,GroundHeight,Area,", outFile);
				fputs("Population,Households", outFile);
			}
		}
		if (Params.ApplyClutter) {
			fputs(",LandCoverType,ClutterType\n", outFile);
		} else {
			fputc('\n', outFile);
		}
		OutputFile[POINTS_FILE] = outFile;
	}

	// Set up for the IX margin CSV output if needed.  If IXMarginSourceKey is set that will cause extra attributes in
	// the coverage map file (if that is being generated) for any points for any desired that receive unique IX from
	// that source, including the bearing from the source and the amount by which the D/U fails (the margin).  That
	// information is also accumulated for optional CSV file output after the run.  The CSV output may accumulate in
	// 1-degree azimuth windows, keeping the worst margin (smallest number, failed points always have negative margin)
	// along with the total population in all unique IX points, or the output may list all individual points.  In
	// either case, zero-population points may or may not be included.  See analyze_points().

	int i;

	if ((IXMarginSourceKey > 0) && OutputFlags[IXCHK_MARG_CSV]) {
		if ((IXCHK_MARG_CSV_AGG == OutputFlags[IXCHK_MARG_CSV]) ||
				(IXCHK_MARG_CSV_AGGNO0 == OutputFlags[IXCHK_MARG_CSV])) {
			IXMarginCount = 360;
		} else {
			IXMarginCount = 0;
		}
		if (IXMarginCount > IXMarginMaxCount) {
			IXMarginMaxCount = IXMarginCount + 1000;
			IXMarginAz = (double *)mem_realloc(IXMarginAz, (IXMarginMaxCount * sizeof(double)));
			IXMarginDb = (double *)mem_realloc(IXMarginDb, (IXMarginMaxCount * sizeof(double)));
			IXMarginPop = (int *)mem_realloc(IXMarginPop, (IXMarginMaxCount * sizeof(int)));
		}
		for (i = 0; i < IXMarginCount; i++) {
			IXMarginAz[i] = (double)i;
			IXMarginDb[i] = 0.;
			IXMarginPop[i] = 0;
		}
	}

	// Do the scenario run per study mode: global grid, local grid, or points.

	if (STUDY_MODE_GRID == StudyMode) {

		if (GRID_TYPE_GLOBAL == Params.GridType) {

			// Set global function flags.  DoPoints causes the CompositePoints array to be set up to track points for
			// output and/or compositing coverage.  DoComposite causes coverage compositing for output and/or database
			// result tables.  DoTable is a copy of the other flag so table output errors can cancel output without
			// aborting the run or affecting later scenarios.  DoImage means image output requested, that also may be
			// canceled.  All this only works in global grid mode, and compositing only in general-purpose studies.
			// Also none of these are supported in "all population" study point mode since in that case a cell may have
			// many study points for the same country.

			if (POINT_METHOD_ALL != Params.StudyPointMethod) {
				if (OutputFlags[POINTS_FILE] || (MapOutputFlags[MAP_OUT_SHAPE] && MapOutputFlags[MAP_OUT_SHP_PTS])) {
					DoPoints = 1;
				}
				if ((OutputFlags[COMPOSITE_COVERAGE] || CreateResultTables ||
						(MapOutputFlags[MAP_OUT_SHAPE] && MapOutputFlags[MAP_OUT_SHP_CMP])) &&
						((STUDY_TYPE_TV == StudyType) || (STUDY_TYPE_FM == StudyType))) {
					DoPoints = 1;
					DoComposite = 1;
					DoTable = CreateResultTables;
				}
				DoImage = (MapOutputFlags[MAP_OUT_KML] && MapOutputFlags[MAP_OUT_IMAGE]);
				DoCompositeImage = (DoImage && (MAP_OUT_KML_COM == MapOutputFlags[MAP_OUT_KML]));
			}

			// Create temporary points and results database tables if needed, data is inserted by write_points() and
			// do_write_compcov().  The tables are joined at the end of the scenario run to create the final results
			// table.  The final table will have the composite result code and some point metadata, but does not
			// include any source metadata.  Also create the census points table.  Errors just cancel table output.

			if (DoTable) {

				char query[MAX_QUERY];

				snprintf(query, MAX_QUERY, "CREATE TEMPORARY TABLE %s_%d.temp_point_%d (country_key TINYINT, lat_index INT, lon_index INT, latitude DOUBLE, longitude DOUBLE, area DOUBLE, population INT, households INT, INDEX(country_key, lat_index, lon_index));",
					DbName, StudyKey, ScenarioKey);
				if (mysql_query(MyConnection, query)) {
					log_db_error("Cannot create temporary points database table");
					DoTable = 0;
				}

				snprintf(query, MAX_QUERY, "CREATE TEMPORARY TABLE %s_%d.temp_result_%d (country_key TINYINT, lat_index INT, lon_index INT, source_key SMALLINT, margin SMALLINT, result TINYINT, service_count TINYINT, INDEX(country_key, lat_index, lon_index));",
					DbName, StudyKey, ScenarioKey);
				if (mysql_query(MyConnection, query)) {
					log_db_error("Cannot create temporary results database table");
					DoTable = 0;
				}
				ResultsRowCount = 0;

				snprintf(query, MAX_QUERY, "CREATE TABLE %s_%d.result_%d_cenpt (country_key TINYINT, lat_index INT, lon_index INT, census INT, id VARCHAR(255), INDEX(country_key, lat_index, lon_index));",
					DbName, StudyKey, ScenarioKey);
				if (mysql_query(MyConnection, query)) {
					log_db_error("Cannot create census points database table");
					DoTable = 0;
				}
			}

			// Set up image output if needed.  Errors do not abort, just cancel image output.

			if (DoImage) {
				if (do_image_setup()) {
					DoImage = 0;
					DoCompositeImage = 0;
				}
			}

			err = do_global_run();

		} else {
			err = do_local_run();
		}

	// In points mode, the report and CSV outputs are actually written during analysis in analyze_points() so those
	// have to be opened before the run starts.

	} else {

		if (OutputFlags[REPORT_FILE_DETAIL]) {
			outFile = open_out_file(REPORT_FILE_NAME);
			if (!outFile) return 1;
			write_report_preamble(REPORT_FILE_DETAIL, outFile, 1, 1);
			write_report(REPORT_FILE_DETAIL, OutputFlags[REPORT_FILE_DETAIL], outFile, 1);
			OutputFile[REPORT_FILE_DETAIL] = outFile;
		}

		if ((outFile = OutputFile[REPORT_FILE_SUMMARY])) {
			write_report(REPORT_FILE_SUMMARY, OutputFlags[REPORT_FILE_SUMMARY], outFile, ReportSumHeader);
			ReportSumHeader = 0;
			fprintf(outFile, "%s\n\n", ScenarioName);
		}

		if (OutputFlags[CSV_FILE_DETAIL]) {
			outFile = open_out_file(CSV_FILE_NAME);
			if (!outFile) return 1;
			write_csv_preamble(CSV_FILE_DETAIL, outFile, 1, 1);
			write_csv(CSV_FILE_DETAIL, OutputFlags[CSV_FILE_DETAIL], outFile, 1);
			OutputFile[CSV_FILE_DETAIL] = outFile;
		}

		if ((outFile = OutputFile[CSV_FILE_SUMMARY])) {
			write_csv(CSV_FILE_SUMMARY, 0, outFile, CSVSumHeader);
			CSVSumHeader = 0;
		}

		err = do_points_run();
	}

	if (err) return err;

	// Write the IX margin CSV file if needed.  From here on errors are ignored, they just cancel individual functions.

	if ((IXMarginSourceKey > 0) && OutputFlags[IXCHK_MARG_CSV]) {
		outFile = open_out_file(MARG_CSV_FILE_NAME);
		if (outFile) {
			for (i = 0; i < IXMarginCount; i++) {
				fprintf(outFile, "%.2f,%.3f,%d\n", IXMarginAz[i], IXMarginDb[i], IXMarginPop[i]);
			}
			file_close(outFile);
		}
	}

	// Final output to cell files as needed.

	if ((outFile = OutputFile[CELL_FILE_PAIRSTUDY])) {

		fputs("[coverage]\n", outFile);

		int countryIndex, countryKey;
		DES_TOTAL *dtot;

		source = Sources;
		for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
			if (source->isDesired) {
				for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {
					countryKey = countryIndex + 1;
					dtot = source->totals + countryIndex;
					if ((countryKey != source->countryKey) && (0. == dtot->contourArea)) {
						continue;
					}
					if (dtot->contourPop < 0) {
						fprintf(outFile, "%d,%d,%d,,,%.3f,%d,%.3f,%d\n", source->facility_id, source->channel,
							countryKey, dtot->serviceArea, dtot->servicePop, dtot->ixFreeArea, dtot->ixFreePop);
					} else {
						fprintf(outFile, "%d,%d,%d,%.3f,%d,%.3f,%d,%.3f,%d\n", source->facility_id, source->channel,
							countryKey, dtot->contourArea, dtot->contourPop, dtot->serviceArea, dtot->servicePop,
							dtot->ixFreeArea, dtot->ixFreePop);
					}
				}
			}
		}

		fputs("[endcoverage]\n[endcase]\n", outFile);
	}

	if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
		fputs("[endscenario]\n", outFile);
	}

	if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
		fputs("[endscenario]\n", outFile);
	}

	// Generate parameters output file as needed.

	if (OutputFlags[PARAMS_FILE]) {
		outFile = open_out_file(PARAMETER_FILE_NAME);
		if (outFile) {
			write_parameters(OutputFlags[PARAMS_FILE], outFile);
			file_close(outFile);
		}
	}

	// If KML map files created, finish up with files that link together other files.

	if (MapOutputFlags[MAP_OUT_KML]) {
		err = do_finish_kml();
		if (err) return err;
	}

	// In points mode, that's all.

	if (STUDY_MODE_POINTS == StudyMode) {
		return 0;
	}

	// In grid mode, write report and CSV files as needed.

	if (OutputFlags[REPORT_FILE_DETAIL]) {
		outFile = open_out_file(REPORT_FILE_NAME);
		if (outFile) {
			write_report_preamble(REPORT_FILE_DETAIL, outFile, 1, 1);
			write_report(REPORT_FILE_DETAIL, OutputFlags[REPORT_FILE_DETAIL], outFile, 1);
			file_close(outFile);
		}
	}

	if ((outFile = OutputFile[REPORT_FILE_SUMMARY])) {
		write_report(REPORT_FILE_SUMMARY, OutputFlags[REPORT_FILE_SUMMARY], outFile, ReportSumHeader);
		ReportSumHeader = 0;
	}

	if (OutputFlags[CSV_FILE_DETAIL]) {
		outFile = open_out_file(CSV_FILE_NAME);
		if (outFile) {
			write_csv_preamble(CSV_FILE_DETAIL, outFile, 1, 1);
			write_csv(CSV_FILE_DETAIL, OutputFlags[CSV_FILE_DETAIL], outFile, 1);
			file_close(outFile);
		}
	}

	if ((outFile = OutputFile[CSV_FILE_SUMMARY])) {
		write_csv(CSV_FILE_SUMMARY, 0, outFile, CSVSumHeader);
		CSVSumHeader = 0;
	}

	// Save totals for scenario pairing as needed, those have to be copied because the source may be studied again in
	// a later scenario.  Also check the entire list because the same scenario may be involved in more than one pair.
	// After any update if both scenarios in the pair have been run, update the percentages.

	int didA, didB, countryIndex;
	DES_TOTAL *dtotA, *dtotB;

	SCENARIO_PAIR *thePair = ScenarioPairList;

	while (thePair) {

		didA = 0;
		didB = 0;

		if ((thePair->scenarioKeyA == ScenarioKey) && thePair->sourceA->inScenario && thePair->sourceA->isDesired) {
			memcpy(&(thePair->totalsA), &(thePair->sourceA->totals), (MAX_COUNTRY * sizeof(DES_TOTAL)));
			thePair->didStudyA = 1;
			didA = 1;
			didB = thePair->didStudyB;
		}

		if ((thePair->scenarioKeyB == ScenarioKey) && thePair->sourceB->inScenario && thePair->sourceB->isDesired) {
			memcpy(&(thePair->totalsB), &(thePair->sourceB->totals), (MAX_COUNTRY * sizeof(DES_TOTAL)));
			thePair->didStudyB = 1;
			didB = 1;
			didA = thePair->didStudyA;
		}

		if (didA && didB) {

			for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {

				dtotA = thePair->totalsA + countryIndex;
				dtotB = thePair->totalsB + countryIndex;
				if ((0. == dtotA->contourArea) && (0. == dtotB->contourArea)) {
					continue;
				}

				if (dtotA->ixFreeArea > 0.) {
					thePair->areaPercent[countryIndex] =
						((dtotA->ixFreeArea - dtotB->ixFreeArea) / dtotA->ixFreeArea) * 100.;
				} else {
					thePair->areaPercent[countryIndex] = 0.;
				}

				if (dtotA->ixFreePop > 0) {
					thePair->popPercent[countryIndex] =
						((double)(dtotA->ixFreePop - dtotB->ixFreePop) / (double)dtotA->ixFreePop) * 100.;
				} else {
					thePair->popPercent[countryIndex] = 0.;
				}

				if (dtotA->ixFreeHouse > 0) {
					thePair->housePercent[countryIndex] =
						((double)(dtotA->ixFreeHouse - dtotB->ixFreeHouse) / (double)dtotA->ixFreeHouse) * 100.;
				} else {
					thePair->housePercent[countryIndex] = 0.;
				}
			}
		}

		thePair = thePair->next;
	}

	// Done.

	return 0;
}

// Do source and service area map file outputs for the loaded scenario, open points and coverage map files for output
// during analysis as needed.  This always sets up attributes, but may only open shapefiles.  KML files may be broken
// out into individual files per desired source which will be opened in do_global_run()/do_local_run(); the cov file
// attributes list is global to support that.

#define DSRC_NATTR  21
#define USRC_NATTR  22
#define CONT_NATTR  1

#define CELL_MAX_ATTR  15

#define COMPCOV_NATTR  5

static int do_start_mapfiles() {

	// Attribute lists for source and contour files.

	static SHAPEATTR dattrs[DSRC_NATTR] = {
		ATTR_SOURCEKEY,
		ATTR_DTSKEY,
		ATTR_SITENUMBER,
		ATTR_FACILITYID,
		ATTR_SERVICE,
		ATTR_CHANNEL,
		ATTR_FREQUENCY,
		ATTR_CALLSIGN,
		ATTR_CITY,
		ATTR_STATE,
		ATTR_COUNTRY,
		ATTR_STATUS,
		ATTR_FILENUMBER,
		ATTR_HAMSL,
		ATTR_HAAT,
		ATTR_ERP,
		ATTR_SVCLEVEL,
		ATTR_AZORIENT,
		ATTR_ETILT,
		ATTR_MTILT,
		ATTR_MTLTORIENT
	};

	static SHAPEATTR uattrs[USRC_NATTR] = {
		ATTR_SOURCEKEY,
		ATTR_DTSKEY,
		ATTR_SITENUMBER,
		ATTR_FACILITYID,
		ATTR_SERVICE,
		ATTR_CHANNEL,
		ATTR_FREQUENCY,
		ATTR_CALLSIGN,
		ATTR_CITY,
		ATTR_STATE,
		ATTR_COUNTRY,
		ATTR_STATUS,
		ATTR_FILENUMBER,
		ATTR_HAMSL,
		ATTR_HAAT,
		ATTR_ERP,
		ATTR_AZORIENT,
		ATTR_ETILT,
		ATTR_MTILT,
		ATTR_MTLTORIENT,
		ATTR_DSOURCEKEY,
		ATTR_CULLDIST
	};

	static SHAPEATTR cattrs[CONT_NATTR] = {
		ATTR_SOURCEKEY
	};

	// The ATTR_* macros are structure constants, have to copy to variables for run-time assignments below.  Meh.

	static SHAPEATTR attrPointName = ATTR_POINTNAME;
	static SHAPEATTR attrCountryKey = ATTR_COUNTRYKEY;
	static SHAPEATTR attrSourceKey = ATTR_SOURCEKEY;
	static SHAPEATTR attrLatIndex = ATTR_LATINDEX;
	static SHAPEATTR attrLonIndex = ATTR_LONINDEX;
	static SHAPEATTR attrStudyLat = ATTR_STUDYLAT;
	static SHAPEATTR attrStudyLon = ATTR_STUDYLON;
	static SHAPEATTR attrArea = ATTR_AREA;
	static SHAPEATTR attrPopulation = ATTR_POPULATION;
	static SHAPEATTR attrHouseholds = ATTR_HOUSEHOLDS;
	static SHAPEATTR attrReceiveHgt = ATTR_RECEIVEHGT;
	static SHAPEATTR attrGroundHgt = ATTR_GROUNDHGT;
	static SHAPEATTR attrLandCover = ATTR_LANDCOVER;
	static SHAPEATTR attrClutter = ATTR_CLUTTER;
	static SHAPEATTR attrClutterDb = ATTR_CLUTTERDB;
	static SHAPEATTR attrResult = ATTR_RESULT;
	static SHAPEATTR attrIXBearing = ATTR_IXBEARING;
	static SHAPEATTR attrIXMargin = ATTR_IXMARGIN;
	static SHAPEATTR attrSiteNumber = ATTR_SITENUMBER;
	static SHAPEATTR attrDSignal = ATTR_DSIGNAL;
	static SHAPEATTR attrDMargin = ATTR_DMARGIN;
	static SHAPEATTR attrUSourceKey = ATTR_USOURCEKEY;
	static SHAPEATTR attrDU = ATTR_DU;
	static SHAPEATTR attrDUMargin = ATTR_DUMARGIN;
	static SHAPEATTR attrWLSignal = ATTR_WLSIGNAL;
	static SHAPEATTR attrWLDU = ATTR_WLDU;
	static SHAPEATTR attrWLDUMargin = ATTR_WLDUMARGIN;
	static SHAPEATTR attrSelfDU = ATTR_SELFDU;
	static SHAPEATTR attrSelfDUMarg = ATTR_SELFDUMARG;
	static SHAPEATTR attrMargin = ATTR_MARGIN;
	static SHAPEATTR attrCause = ATTR_CAUSE;
	static SHAPEATTR attrRampAlpha = ATTR_RAMPALPHA;
	static SHAPEATTR attrUSignalRSS = ATTR_USIGNALRSS;
	static SHAPEATTR attrUSiteNumbr = ATTR_USITENUMBR;
	static SHAPEATTR attrUSignal = ATTR_USIGNAL;
	static SHAPEATTR attrDeltaT = ATTR_DELTAT;

	// Get main flags, one of these is true or this would not have been called.

	int doShape = MapOutputFlags[MAP_OUT_SHAPE];
	int doKML = MapOutputFlags[MAP_OUT_KML];

	// If generating KML map output, delete the entire KML subdirectory rather than depending on overwrite of files
	// from a previous run; even minor changes to study conditions can result in a different set of file names.

	int err = 0;

	if (doKML) {
		char *args[4];
		args[0] = "/bin/rm";
		args[1] = "-rf";
		args[2] = KML_SUBDIR;
		args[3] = NULL;
		err = run_process(args, get_file_path());
		if (err) return err;
	}

	// Write source and contour (service area boundary) files for the entire scenario, first open files.

	DesiredSourcesKML = NULL;
	UndesiredSourcesKML = NULL;
	ContoursKML = NULL;

	MAPFILE *shpDSrcs = NULL, *shpUSrcs = NULL, *shpConts = NULL, *kmlDSrcs = NULL, *kmlUSrcs = NULL, *kmlConts = NULL;

	while (1) {

		if (doShape) {

			shpDSrcs = open_mapfile(MAP_FILE_SHAPE, DSOURCE_MAPFILE_NAME, SHP_SUBDIR, SHP_TYPE_POINT, DSRC_NATTR,
				dattrs, NULL);
			if (!shpDSrcs) {
				err = 1;
				break;
			}

			shpUSrcs = open_mapfile(MAP_FILE_SHAPE, USOURCE_MAPFILE_NAME, SHP_SUBDIR, SHP_TYPE_POINT, USRC_NATTR,
				uattrs, NULL);
			if (!shpUSrcs) {
				err = 1;
				break;
			}

			shpConts = open_mapfile(MAP_FILE_SHAPE, CONTOUR_MAPFILE_NAME, SHP_SUBDIR, SHP_TYPE_POLYLINE, CONT_NATTR,
				cattrs, NULL);
			if (!shpConts) {
				err = 1;
				break;
			}
		}

		if (doKML) {

			// KML output may be in by-station mode which writes data to individual files for each desired source as
			// processed, in that case no files are actually written here, but "null" MAPFILE structures are created
			// and kept in globals to be used by the run code later.

			if (MAP_OUT_KML_IND == doKML) {

				DesiredSourcesKML = create_kml_mapfile(DSOURCE_MAPFILE_NAME, SHP_TYPE_POINT, DSRC_NATTR, dattrs);
				if (!DesiredSourcesKML) {
					err = 1;
					break;
				}

				UndesiredSourcesKML = create_kml_mapfile(USOURCE_MAPFILE_NAME, SHP_TYPE_POINT, USRC_NATTR, uattrs);
				if (!UndesiredSourcesKML) {
					err = 1;
					break;
				}

				ContoursKML = create_kml_mapfile(CONTOUR_MAPFILE_NAME, SHP_TYPE_POLYLINE, CONT_NATTR, cattrs);
				if (!ContoursKML) {
					err = 1;
					break;
				}

			} else {

				kmlDSrcs = open_mapfile(MAP_FILE_KML, DSOURCE_MAPFILE_NAME, KML_SUBDIR, SHP_TYPE_POINT, DSRC_NATTR,
					dattrs, DSOURCE_MAPFILE_TITLE);
				if (!kmlDSrcs) {
					err = 1;
					break;
				}

				kmlUSrcs = open_mapfile(MAP_FILE_KML, USOURCE_MAPFILE_NAME, KML_SUBDIR, SHP_TYPE_POINT, USRC_NATTR,
					uattrs, USOURCE_MAPFILE_TITLE);
				if (!kmlUSrcs) {
					err = 1;
					break;
				}

				kmlConts = open_mapfile(MAP_FILE_KML, CONTOUR_MAPFILE_NAME, KML_SUBDIR, SHP_TYPE_POLYLINE, CONT_NATTR,
					cattrs, CONTOUR_MAPFILE_TITLE);
				if (!kmlConts) {
					err = 1;
					break;
				}
			}
		}

		break;
	}

	if (!err) {

		SOURCE *source = Sources, *dtsSource, *usource;
		UNDESIRED *undesireds;
		int sourceIndex, undesiredCount, undesiredIndex;
		double cullDist;

		for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {

			if (source->isDesired) {

				hb_log();
				if (Terminate) {
					err = -1;
					break;
				}

				if (shpDSrcs) {
					err = write_source_shapes(source, shpDSrcs, shpConts, 0, 0., NULL, 1);
					if (err) break;
				}

				if (kmlDSrcs) {
					err = write_source_shapes(source, kmlDSrcs, kmlConts, 0, 0., NULL, (source->isParent ? 0 : 1));
					if (err) break;
				}

				if (source->isParent) {

					if (shpDSrcs) {
						err = write_source_shapes(source->dtsAuthSource, shpDSrcs, shpConts, 0, 0., NULL, 1);
						if (err) break;
					}

					if (kmlDSrcs) {
						err = write_source_shapes(source->dtsAuthSource, kmlDSrcs, kmlConts, 0, 0., NULL, 0);
						if (err) break;
					}

					for (dtsSource = source->dtsSources; dtsSource; dtsSource = dtsSource->next) {

						if (shpDSrcs) {
							err = write_source_shapes(dtsSource, shpDSrcs, shpConts, 0, 0., NULL, 1);
							if (err) break;
						}

						if (kmlDSrcs) {
							err = write_source_shapes(dtsSource, kmlDSrcs, kmlConts, 0, 0., NULL, 1);
							if (err) break;
						}
					}

					if (err) break;
				}

				undesireds = source->undesireds;
				undesiredCount = source->undesiredCount;

				for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {

					usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
					if (!usource) {
						log_error("Source structure index is corrupted");
						exit(1);
					}
					cullDist = undesireds[undesiredIndex].ixDistance;

					if (usource->isParent) {

						for (dtsSource = usource->dtsSources; dtsSource; dtsSource = dtsSource->next) {

							if (shpUSrcs) {
								err = write_source_shapes(dtsSource, shpUSrcs, NULL, source->sourceKey, cullDist,
									NULL, 1);
								if (err) break;
							}

							if (kmlUSrcs) {
								err = write_source_shapes(dtsSource, kmlUSrcs, NULL, source->sourceKey, cullDist,
									NULL, 0);
								if (err) break;
							}
						}

						if (err) break;

					} else {

						if (shpUSrcs) {
							err = write_source_shapes(usource, shpUSrcs, NULL, source->sourceKey, cullDist, NULL, 1);
							if (err) break;
						}

						if (kmlUSrcs) {
							err = write_source_shapes(usource, kmlUSrcs, NULL, source->sourceKey, cullDist, NULL, 0);
							if (err) break;
						}
					}
				}

				if (err) break;
			}
		}
	}

	if (shpDSrcs) {
		close_mapfile(shpDSrcs);
	}
	if (shpUSrcs) {
		close_mapfile(shpUSrcs);
	}
	if (shpConts) {
		close_mapfile(shpConts);
	}
	if (kmlDSrcs) {
		close_mapfile(kmlDSrcs);
	}
	if (kmlUSrcs) {
		close_mapfile(kmlUSrcs);
	}
	if (kmlConts) {
		close_mapfile(kmlConts);
	}

	if (err) return err;

	// Set up coverage points attribute list, may be used here for file open, or later during the scenario run for per-
	// source opens.  If the global IXMarginSourceKey is set in grid mode, the coverage file has extra attributes
	// containing the bearing and the D/U margin from that specific interfering source.  The fields are populated only
	// when the undesired source causes unique interference at the point.  For any type of coverage output, additional
	// fields may be added per the output option flags.

	CovPtsAttributeCount = 0;

	if (STUDY_MODE_GRID == StudyMode) {
		CovPtsAttributes[CovPtsAttributeCount++] = attrCountryKey;
		CovPtsAttributes[CovPtsAttributeCount++] = attrSourceKey;
		CovPtsAttributes[CovPtsAttributeCount++] = attrLatIndex;
		CovPtsAttributes[CovPtsAttributeCount++] = attrLonIndex;
		CovPtsAttributes[CovPtsAttributeCount++] = attrResult;
		if (IXMarginSourceKey > 0) {
			CovPtsAttributes[CovPtsAttributeCount++] = attrIXBearing;
			CovPtsAttributes[CovPtsAttributeCount++] = attrIXMargin;
		}
		if (MapOutputFlags[MAP_OUT_COORDS]) {
			CovPtsAttributes[CovPtsAttributeCount++] = attrStudyLat;
			CovPtsAttributes[CovPtsAttributeCount++] = attrStudyLon;
		}
		if (MapOutputFlags[MAP_OUT_AREAPOP]) {
			CovPtsAttributes[CovPtsAttributeCount++] = attrArea;
			CovPtsAttributes[CovPtsAttributeCount++] = attrPopulation;
			CovPtsAttributes[CovPtsAttributeCount++] = attrHouseholds;
		}
	} else {
		CovPtsAttributes[CovPtsAttributeCount++] = attrPointName;
		CovPtsAttributes[CovPtsAttributeCount++] = attrCountryKey;
		CovPtsAttributes[CovPtsAttributeCount++] = attrSourceKey;
		CovPtsAttributes[CovPtsAttributeCount++] = attrResult;
		if (MapOutputFlags[MAP_OUT_HEIGHT]) {
			CovPtsAttributes[CovPtsAttributeCount++] = attrReceiveHgt;
		}
	}
	if (MapOutputFlags[MAP_OUT_HEIGHT]) {
		CovPtsAttributes[CovPtsAttributeCount++] = attrGroundHgt;
	}
	if (MapOutputFlags[MAP_OUT_CLUTTER] && Params.ApplyClutter) {
		CovPtsAttributes[CovPtsAttributeCount++] = attrLandCover;
		CovPtsAttributes[CovPtsAttributeCount++] = attrClutter;
		CovPtsAttributes[CovPtsAttributeCount++] = attrClutterDb;
	}
	if (MapOutputFlags[MAP_OUT_DESINFO]) {
		CovPtsAttributes[CovPtsAttributeCount++] = attrSiteNumber;
		CovPtsAttributes[CovPtsAttributeCount++] = attrDSignal;
		CovPtsAttributes[CovPtsAttributeCount++] = attrDMargin;
	}
	if (MapOutputFlags[MAP_OUT_UNDINFO]) {
		CovPtsAttributes[CovPtsAttributeCount++] = attrUSourceKey;
		CovPtsAttributes[CovPtsAttributeCount++] = attrDU;
		CovPtsAttributes[CovPtsAttributeCount++] = attrDUMargin;
	}
	if (MapOutputFlags[MAP_OUT_WLINFO] && (STUDY_TYPE_TV_OET74 == StudyType)) {
		CovPtsAttributes[CovPtsAttributeCount++] = attrWLSignal;
		CovPtsAttributes[CovPtsAttributeCount++] = attrWLDU;
		CovPtsAttributes[CovPtsAttributeCount++] = attrWLDUMargin;
	}
	if (MapOutputFlags[MAP_OUT_SELFIX] && Params.CheckSelfInterference) {
		CovPtsAttributes[CovPtsAttributeCount++] = attrSelfDU;
		CovPtsAttributes[CovPtsAttributeCount++] = attrSelfDUMarg;
	}
	if (MapOutputFlags[MAP_OUT_MARGIN]) {
		CovPtsAttributes[CovPtsAttributeCount++] = attrMargin;
		CovPtsAttributes[CovPtsAttributeCount++] = attrCause;
	}
	if (MapOutputFlags[MAP_OUT_RAMP]) {
		CovPtsAttributes[CovPtsAttributeCount++] = attrRampAlpha;
	}

	// Self-interference file attributes, always the same.

	SelfIXFileAttributes[0] = attrSourceKey;
	SelfIXFileAttributes[1] = attrSiteNumber;
	SelfIXFileAttributes[2] = attrDSignal;
	SelfIXFileAttributes[3] = attrUSignalRSS;
	SelfIXFileAttributes[4] = attrPopulation;
	SelfIXFileAttributes[5] = attrUSiteNumbr;
	SelfIXFileAttributes[6] = attrUSignal;
	SelfIXFileAttributes[7] = attrDeltaT;

	// Open shapefiles as needed, these are always single files holding all data from the scenario run.  A separate
	// unique study points file and a coverage composite file are now both optional.  In the points file, include the
	// study point coordinates in metadata if the cell center will be the point in the shapefile itself.

	MAPFILE *mapFile;

	if (doShape) {

		mapFile = open_mapfile(MAP_FILE_SHAPE, COVPTS_MAPFILE_NAME, SHP_SUBDIR, SHP_TYPE_POINT,
			CovPtsAttributeCount, CovPtsAttributes, NULL);
		if (!mapFile) return 1;
		MapOutputFile[MAP_OUT_SHAPE_COVPTS] = mapFile;

		if (MapOutputFlags[MAP_OUT_SHP_PTS]) {

			SHAPEATTR cellAttrs[CELL_MAX_ATTR];
			int nCellAttrs = 0;

			if (STUDY_MODE_POINTS == StudyMode) {
				cellAttrs[nCellAttrs++] = attrPointName;
			}
			cellAttrs[nCellAttrs++] = attrCountryKey;
			if (STUDY_MODE_GRID == StudyMode) {
				cellAttrs[nCellAttrs++] = attrLatIndex;
				cellAttrs[nCellAttrs++] = attrLonIndex;
				if (MapOutputFlags[MAP_OUT_CENTER]) {
					cellAttrs[nCellAttrs++] = attrStudyLat;
					cellAttrs[nCellAttrs++] = attrStudyLon;
				}
				cellAttrs[nCellAttrs++] = attrArea;
				cellAttrs[nCellAttrs++] = attrPopulation;
				cellAttrs[nCellAttrs++] = attrHouseholds;
			} else {
				cellAttrs[nCellAttrs++] = attrReceiveHgt;
			}
			cellAttrs[nCellAttrs++] = attrGroundHgt;
			if (Params.ApplyClutter) {
				cellAttrs[nCellAttrs++] = attrLandCover;
				cellAttrs[nCellAttrs++] = attrClutter;
			}

			mapFile = open_mapfile(MAP_FILE_SHAPE, POINTS_MAPFILE_NAME, SHP_SUBDIR, SHP_TYPE_POINT, nCellAttrs,
				cellAttrs, NULL);
			if (!mapFile) return 1;
			MapOutputFile[MAP_OUT_SHAPE_POINTS] = mapFile;
		}

		// Composite output can only occur in general-purpose study types.

		if (MapOutputFlags[MAP_OUT_SHP_CMP] && ((STUDY_TYPE_TV == StudyType) || (STUDY_TYPE_FM == StudyType))) {

			SHAPEATTR compAttrs[COMPCOV_NATTR];

			compAttrs[0] = attrCountryKey;
			compAttrs[1] = attrLatIndex;
			compAttrs[2] = attrLonIndex;
			compAttrs[3] = attrSourceKey;
			compAttrs[4] = attrResult;

			mapFile = open_mapfile(MAP_FILE_SHAPE, COMPCOV_MAPFILE_NAME, SHP_SUBDIR, SHP_TYPE_POINT, COMPCOV_NATTR,
				compAttrs, NULL);
			if (!mapFile) return 1;
			MapOutputFile[MAP_OUT_SHAPE_CMPPTS] = mapFile;
		}

		if (MapOutputFlags[MAP_OUT_SELFIX] && Params.CheckSelfInterference) {
			mapFile = open_mapfile(MAP_FILE_SHAPE, SELFIX_MAPFILE_NAME, SHP_SUBDIR, SHP_TYPE_POINT,
				SELFIX_ATTRIBUTE_COUNT, SelfIXFileAttributes, NULL);
			if (!mapFile) return 1;
			MapOutputFile[MAP_OUT_SHAPE_SELFIX] = mapFile;
		}
	}

	// In global-layer points mode, open KML files for coverage and self IX; both are optional.  In grid mode or in
	// individual-station-layer mode those are individual files per desired source and will be opened later, just set
	// up subdirectory paths.  The KML file set does not include unique study points or composite result points because
	// those would be mostly redundant in a map viewer; the coverage points provide all the same information for
	// browsing purposes.  The assumption is that KML would never be used for data post-processing, shapefiles are much
	// better suited to that purpose.

	if (doKML) {

		if ((STUDY_MODE_POINTS == StudyMode) && (MAP_OUT_KML_COM == doKML)) {

			if (MapOutputFlags[MAP_OUT_KML_CPT]) {
				mapFile = open_mapfile(MAP_FILE_KML, COVPTS_MAPFILE_NAME, KML_SUBDIR, SHP_TYPE_POINT,
					CovPtsAttributeCount, CovPtsAttributes, COVPTS_MAPFILE_TITLE);
				if (!mapFile) return 1;
				MapOutputFile[MAP_OUT_KML_COVPTS] = mapFile;
			}

			if (MapOutputFlags[MAP_OUT_SELFIX] && Params.CheckSelfInterference) {
				mapFile = open_mapfile(MAP_FILE_KML, SELFIX_MAPFILE_NAME, KML_SUBDIR, SHP_TYPE_POINT,
					SELFIX_ATTRIBUTE_COUNT, SelfIXFileAttributes, SELFIX_MAPFILE_TITLE);
				if (!mapFile) return 1;
				MapOutputFile[MAP_OUT_KML_SELFIX] = mapFile;
			}

		} else {

			snprintf(CovPtsKMLSubDir, MAX_STRING, "%s/%s", KML_SUBDIR, COVPTS_MAPFILE_NAME);
			snprintf(SelfIXKMLSubDir, MAX_STRING, "%s/%s", KML_SUBDIR, SELFIX_MAPFILE_NAME);
		}
	}

	return 0;
}

// Set up map image output.  The image is generated by writing PostScript code then rendering with the GhostScript
// package, so a "gs" command must be available.  To keep things simple, rather than trying to figure out where that
// command might be, the user is required to create a symlink to "gs" in TVStudy's lib directory.  If that link is
// not present, image output is disabled.  All errors here are minor; the caller will disable image output.

static int do_image_setup() {

	static int hasGS = -1;

	char fname[MAX_STRING];

	if (hasGS < 0) {
		snprintf(fname, MAX_STRING, "%s/gs", LIB_DIRECTORY_NAME);
		if (access(fname, X_OK)) {
			hasGS = 0;
		} else {
			hasGS = 1;
		}
	}

	if (!hasGS) {
		return 1;
	}

	// Read the color map, the map index is provided by a second map file flag that should always be set if the image
	// file flag is set.  Otherwise write PostScript to set up the drawing environment.  During analysis each study
	// point defining the value will cause the cell region for that point to be filled with a color from the map.

	if (MapOutputFlags[MAP_OUT_COLORS] < 1) {
		log_error("Missing color map option, image output canceled");
		return 1;
	}

	char query[MAX_QUERY];
	MYSQL_RES *myResult;
	my_ulonglong rowCount;
	MYSQL_ROW fields;

	snprintf(query, MAX_QUERY,
		"SELECT level, color_r, color_g, color_b FROM %s.color_map_data WHERE color_map_key = %d ORDER BY 1;",
		DbName, MapOutputFlags[MAP_OUT_COLORS]);
	if (mysql_query(MyConnection, query)) {
		log_db_error("Color map query failed, image output canceled (1)");
		return 1;
	}

	myResult = mysql_store_result(MyConnection);
	if (!myResult) {
		log_db_error("Color map query failed, image output canceled (2)");
		return 1;
	}

	rowCount = mysql_num_rows(myResult);
	if (!rowCount) {
		log_error("Missing color map data for key %d, image output canceled", MapOutputFlags[MAP_OUT_COLORS]);
		mysql_free_result(myResult);
		return 1;
	}

	ImageColorCount = (int)rowCount;
	if (ImageColorCount > MAX_IMAGE_COLOR_COUNT) {
		ImageColorCount = MAX_IMAGE_COLOR_COUNT;
	}

	ImageMapStartIndex = -1;
	ImageNoServiceIndex = -1;
	ImageInterferenceIndex = -1;
	ImageBackgroundIndex = -1;

	int colorIndex;

	for (colorIndex = 0; colorIndex < ImageColorCount; colorIndex++) {

		fields = mysql_fetch_row(myResult);
		if (!fields) {
			log_db_error("Color map query failed, image output canceled (3)");
			mysql_free_result(myResult);
			return 1;
		}

		ImageLevel[colorIndex] = atof(fields[0]);
		ImageColorR[colorIndex] = atof(fields[1]) / 255.;
		ImageColorG[colorIndex] = atof(fields[2]) / 255.;
		ImageColorB[colorIndex] = atof(fields[3]) / 255.;

		if (IMAGE_LEVEL_NO_SERVICE == ImageLevel[colorIndex]) {
			if ((ImageColorR[colorIndex] >= 0.) && (ImageColorG[colorIndex] >= 0.) &&
					(ImageColorB[colorIndex] >= 0.)) {
				ImageNoServiceIndex = colorIndex;
			}
		} else {
			if (IMAGE_LEVEL_INTERFERENCE == ImageLevel[colorIndex]) {
				if ((ImageColorR[colorIndex] >= 0.) && (ImageColorG[colorIndex] >= 0.) &&
						(ImageColorB[colorIndex] >= 0.)) {
					ImageInterferenceIndex = colorIndex;
				}
			} else {
				if (IMAGE_LEVEL_BACKGROUND == ImageLevel[colorIndex]) {
					ImageBackgroundIndex = colorIndex;
				} else {
					if (ImageMapStartIndex < 0) {
						ImageMapStartIndex = colorIndex;
					}
				}
			}
		}
	}

	mysql_free_result(myResult);

	if ((ImageMapStartIndex < 0) || (ImageBackgroundIndex < 0)) {
		log_error("Bad color map data for key %d, image output canceled", MapOutputFlags[MAP_OUT_COLORS]);
		return 1;
	}

	// Clear flags used to show colors in the image legend, see do_write_image() and do_image_legend().

	for (colorIndex = 0; colorIndex < MAX_IMAGE_COLOR_COUNT; colorIndex++) {
		ImageColorIndexUsed[colorIndex] = 0;
	}

	// In all-stations-combined KML mode open the file that will combine all of the composite-coverage image tile
	// pieces, see do_write_image().  In individual by-station mode, each desired source has a separate image layer
	// and the overlays are written to the per-source KML file opened later.

	if (DoCompositeImage) {
		snprintf(fname, MAX_STRING, "%s.kml", IMAGE_MAPFILE_NAME);
		ImageFileKML = open_out_file_dir(fname, KML_SUBDIR);
		if (!ImageFileKML) {
			return 1;
		}
		kml_start(ImageFileKML, IMAGE_MAPFILE_TITLE);
	}

	return 0;
}

// Finalize KML output, write files linking together other files.

static int do_finish_kml() {

	char topName[MAX_STRING], fileName[MAX_STRING];
	FILE *topFile, *outFile;
	SOURCE *source;
	int sourceIndex, i;

	// Open a top-level file linking all of the other major files.

	snprintf(topName, MAX_STRING, "%s.kml", KML_TOP_FILE_NAME);
	topFile = open_out_file(topName);
	if (!topFile) return 1;

	kml_start(topFile, ScenarioName);
	fputs("<open>1</open>\n", topFile);

	// Behavior depends on whether data is being written to global layers or by-station layers.  In the by-station
	// case, each desired source has a KML file with all features for that source.  Write links to all individual
	// files to the top-level file.  This applies regardless of grid or points mode.

	int err = 0;

	if (MAP_OUT_KML_IND == MapOutputFlags[MAP_OUT_KML]) {

		int sourceIndex;
		SOURCE *source = Sources;
		for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
			if (source->isDesired) {
				fprintf(topFile, "<NetworkLink>\n<name>%s</name>\n", kmlclean(source->callSign));
				fprintf(topFile, "<Link><href>%s/%d.kml</href></Link>\n", KML_SUBDIR, source->sourceKey);
				fputs("</NetworkLink>\n", topFile);
			}
		}

		// In individual-layer mode, the image legend goes to the top-level file as it applies to all of the images
		// for the individual sources but only needs to be displayed once.

		if (DidWriteImageKML) {
			err = do_image_legend(topFile, 1);
		}

	// In the global-layer case for grid mode, create files linking all of the per-source points files together.  These
	// may have been created for all desired sources, for each there are separate files for each possible result code
	// and associated did-write flags in each source.

	} else {

		if (STUDY_MODE_GRID == StudyMode) {

			if (DidWriteCovPtsKML) {

				snprintf(fileName, MAX_STRING, "%s.kml", COVPTS_MAPFILE_NAME);
				outFile = open_out_file_dir(fileName, KML_SUBDIR);
				if (outFile) {

					kml_start(outFile, COVPTS_MAPFILE_TITLE);

					char *title;

					for (i = 0; i < RESULT_COUNT; i++) {

						title = NULL;
						switch (i) {
							case RESULT_COVERAGE: {
								title = RESULT_COVERAGE_TITLE;
								break;
							}
							case RESULT_INTERFERE: {
								title = RESULT_INTERFERE_TITLE;
								break;
							}
							case RESULT_NOSERVICE: {
								if (!MapOutputFlags[MAP_OUT_NOSERV]) {
									title = RESULT_NOSERVICE_TITLE;
								}
								break;
							}
						}
						if (!title) {
							continue;
						}

						fprintf(outFile, "<Folder>\n<name>%s</name>\n", kmlclean(title));

						source = Sources;
						for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
							if (source->didKMLCovPts[i]) {
								fprintf(outFile, "<NetworkLink>\n<name>%s</name>\n", kmlclean(source->callSign));
								fprintf(outFile, "<Link><href>%s/%d-%d.kml</href></Link>\n", COVPTS_MAPFILE_NAME,
									source->sourceKey, i);
								fputs("</NetworkLink>\n", outFile);
								source->didKMLCovPts[i] = 0;
							}
						}

						fputs("</Folder>\n", outFile);
					}

					kml_close(outFile);
				}
			}

			// Similar link file for DTS self-interference files, if any were created.  Again this is only grid mode,
			// in points mode this all went to a single file.

			if (DidWriteSelfIXKML) {

				snprintf(fileName, MAX_STRING, "%s.kml", SELFIX_MAPFILE_NAME);
				outFile = open_out_file_dir(fileName, KML_SUBDIR);
				if (outFile) {

					kml_start(outFile, SELFIX_MAPFILE_TITLE);

					source = Sources;
					for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
						if (source->didKMLSelfIX) {
							fprintf(outFile, "<NetworkLink>\n<name>%s</name>\n", kmlclean(source->callSign));
							fprintf(outFile, "<Link><href>%s/%d.kml</href></Link>\n", SELFIX_MAPFILE_NAME,
								source->sourceKey);
							fputs("</NetworkLink>\n", outFile);
							source->didKMLSelfIX = 0;
						}
					}

					kml_close(outFile);
				}
			}

			// If the image file is open close it, if there was actual image output first write the color legend.

			if (ImageFileKML) {
				if (DidWriteImageKML) {
					err = do_image_legend(ImageFileKML, 0);
				}
				kml_close(ImageFileKML);
				ImageFileKML = NULL;
			}

		// In global-layer points mode the data was all written to global files, using the same names as for the
		// linking files in grid mode created above so the later top-level file is the same either way.

		} else {

			if (MapOutputFile[MAP_OUT_KML_COVPTS]) {
				close_mapfile(MapOutputFile[MAP_OUT_KML_COVPTS]);
				MapOutputFile[MAP_OUT_KML_COVPTS] = NULL;
			}

			if (MapOutputFile[MAP_OUT_KML_SELFIX]) {
				close_mapfile(MapOutputFile[MAP_OUT_KML_SELFIX]);
				MapOutputFile[MAP_OUT_KML_SELFIX] = NULL;
			}
		}

		// Write links for the major global layers to the top-level file.

		if (!err) {

			fprintf(topFile, "<NetworkLink>\n<name>%s</name>\n", kmlclean(CONTOUR_MAPFILE_TITLE));
			fprintf(topFile, "<Link><href>%s/%s.kml</href></Link>\n", KML_SUBDIR, CONTOUR_MAPFILE_NAME);
			fputs("<flyToView>1</flyToView>\n</NetworkLink>\n", topFile);

			fprintf(topFile, "<NetworkLink>\n<name>%s</name>\n", kmlclean(DSOURCE_MAPFILE_TITLE));
			fprintf(topFile, "<Link><href>%s/%s.kml</href></Link>\n", KML_SUBDIR, DSOURCE_MAPFILE_NAME);
			fputs("</NetworkLink>\n", topFile);

			fprintf(topFile, "<NetworkLink>\n<name>%s</name><visibility>0</visibility>\n",
				kmlclean(USOURCE_MAPFILE_TITLE));
			fprintf(topFile, "<Link><href>%s/%s.kml</href></Link>\n", KML_SUBDIR, USOURCE_MAPFILE_NAME);
			fputs("</NetworkLink>\n", topFile);

			if (DidWriteImageKML) {
				fprintf(topFile, "<NetworkLink>\n<name>%s</name>\n", kmlclean(IMAGE_MAPFILE_TITLE));
				fprintf(topFile, "<Link><href>%s/%s.kml</href></Link>\n", KML_SUBDIR, IMAGE_MAPFILE_NAME);
				fputs("</NetworkLink>\n", topFile);
			}

			if (DidWriteCovPtsKML) {
				fprintf(topFile, "<NetworkLink>\n<name>%s</name><visibility>0</visibility>\n",
					kmlclean(COVPTS_MAPFILE_TITLE));
				fprintf(topFile, "<Link><href>%s/%s.kml</href></Link>\n", KML_SUBDIR, COVPTS_MAPFILE_NAME);
				fputs("</NetworkLink>\n", topFile);
			}

			if (DidWriteSelfIXKML) {
				fprintf(topFile, "<NetworkLink>\n<name>%s</name><visibility>0</visibility>\n",
					kmlclean(SELFIX_MAPFILE_TITLE));
				fprintf(topFile, "<Link><href>%s/%s.kml</href></Link>\n", KML_SUBDIR, SELFIX_MAPFILE_NAME);
				fputs("</NetworkLink>\n", topFile);
			}
		}
	}

	kml_close(topFile);

	if (err) return err;

	// Add KMZ file if needed, open and close the file first so it appears in the output file set.  The file then has
	// to be deleted, otherwise zip sees the empty file as an invalid archive and aborts.

	if (MapOutputFlags[MAP_OUT_KML_KMZ]) {

		log_message("Creating KMZ map package");

		int nameLen = strlen(ScenarioName) + 20;
		char *kmzName = (char *)mem_alloc(nameLen);

		char *path = get_file_path();
		int pathLen = strlen(path) + nameLen;
		char *kmzPath = (char *)mem_alloc(pathLen);

		snprintf(kmzName, nameLen, "%s.kmz", ScenarioName);
		snprintf(kmzPath, pathLen, "%s/%s", path, kmzName);

		outFile = open_out_file(kmzName);
		if (!outFile) {
			err = 1;
		} else {

			file_close(outFile);
			unlink(kmzPath);

			char *args[6];
			args[0] = "/usr/bin/zip";
			args[1] = "-rq";
			args[2] = kmzName;
			args[3] = topName;
			args[4] = KML_SUBDIR;
			args[5] = NULL;

			err = run_process(args, path);
		}

		mem_free(kmzName);
		mem_free(kmzPath);
	}

	return err;
}

// Do scenario run for a global grid.  In this mode, the cell grid is defined independent of any source coverage area
// so the cells can be shared between all sources, meaning where desired coverages overlap the study is using the same
// cells and the undesired field strength calculations can be shared between desired sources.  In this type of study,
// the grid can be set up to encompass all sources being studied and all calculations done simultaneously.  However a
// limit has to be set on the grid size to avoid unreasonable memory allocation demands, also when just a few widely-
// separated sources are being studied a single large grid that is mostly empty is inefficient.  So this is a multi-
// pass process that dynamically groups sources onto multiple grids as needed.  The limit is based on the size of the
// terrain data cache, the goal being that all terrain needed for any one grid can be cached, the limit is determined
// by a function in the terrain module, see terrain.c.

static int do_global_run() {

	static int maxRegionCount = 0;
	static SOURCE **regionSources = NULL;
	static INDEX_BOUNDS *regionBounds = NULL;
	static int *regionLonSizes = NULL;

	SOURCE *source, *sourceList;
	POINT **pointPtr, *point;
	FIELD *field;
	long maxGridCount, tempGridCount, gridCount, gridCountSum, gridCellIndex, cacheCount, calcCount, doneCount;
	int err, i, j, regionIndex, regionCount, sourceIndex, cellLonSize, tempCellLonSize, loop, runCount, nCalc, showPcnt,
		calcPcnt, fileCount, fileCountSum;
	INDEX_BOUNDS cellBounds, tempCellBounds;
	char mesg[MAX_STRING];
	FILE *outFile;

	// Set a flag if census points within each study point are needed, in that case population data must be re-queried
	// even if study points are loaded from cache because census points are not cached (need to fix that).  Currently
	// the census points list is needed for one format of the detail cell file, and for result table output.

	int reloadCensusPoints = (CELL_DETAIL_CENPT == OutputFlags[CELL_FILE_DETAIL]) || DoTable;

	// Begin by sorting desired stations into study grid regions based on overlapping coverage areas, to support
	// features that need an all-points grid.  Any two stations with overlapping coverage will be in the same region,
	// so multiple regions will always be non-overlapping.  Most scenarios will have just one region but when there
	// are multiple geographically-distant stations it can have several.  The regions are studied separately so the
	// composite points grid is allocated for only one region at a time.  As discussed above, the stations in a region
	// may be further divided to form sub-grids combining fewer stations with overlapping coverage, as needed to stay
	// within memory limits for the full point and field data structures.  Usually those sub-grids will overlap, so
	// the composite points grid is needed to accumulate results across the sub-grids.  If none of the features that
	// require the composite points grid are active, all stations can be in a single region.

	regionCount = 0;

	while (1) {

		regionIndex = -1;
		source = Sources;
		for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
			if (source->isDesired > 0) {
				regionIndex = regionCount++;
				break;
			}
		}
		if (regionIndex < 0) {
			break;
		}

		if (regionCount >= maxRegionCount) {
			maxRegionCount += 10;
			regionSources = (SOURCE **)mem_realloc(regionSources, (maxRegionCount * sizeof(SOURCE *)));
			regionBounds = (INDEX_BOUNDS *)mem_realloc(regionBounds, (maxRegionCount * sizeof(INDEX_BOUNDS)));
			regionLonSizes = (int *)mem_realloc(regionLonSizes, (maxRegionCount * sizeof(int)));
		}
		source->regionNext = NULL;
		regionSources[regionIndex] = source;
		regionBounds[regionIndex] = source->grid->cellBounds;
		regionLonSizes[regionIndex] = source->grid->cellLonSize;
		source->isDesired = -1;

		do {
			loop = 0;
			source = Sources;
			for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
				if (source->isDesired > 0) {
					if ((DoPoints || DoCompositeImage) &&
							!overlaps_bounds((regionBounds + regionIndex), &(source->grid->cellBounds))) {
						continue;
					}
					source->regionNext = regionSources[regionIndex];
					regionSources[regionIndex] = source;
					extend_bounds_bounds((regionBounds + regionIndex), &(source->grid->cellBounds));
					if (source->grid->cellLonSize < regionLonSizes[regionIndex]) {
						regionLonSizes[regionIndex] = source->grid->cellLonSize;
					}
					source->isDesired = -1;
					loop = 1;
				}
			}
		} while (loop);
	}

	source = Sources;
	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
		if (source->isDesired) {
			source->isDesired = 1;
		}
	}

	// Loop over grid regions, allocate the all-points grid if needed, see discussion above.  Get the max study grid
	// size based on terrain cache limits, see get_max_grid_count() in terrain.c.  If that returns -1 there is not
	// enough memory for any reasonble operations, abort.  That shouldn't happen assuming there was an earlier call to
	// initialize_terrain(), which can fail due to total memory size.  The test done for a region grid will not fail,
	// at worst it will return 0 forcing every desired source to be studied in isolation on it's own grid.  That could
	// theoretically still result in allocation failures, however it will be attempted regardless.  A warning will be
	// logged if a sub-grid size exceeds the limit returned by get_max_grid_count().

	for (regionIndex = 0; regionIndex < regionCount; regionIndex++) {

		if (DoPoints || DoCompositeImage) {

			if (CompositeGrid) {
				free_grid(CompositeGrid);
			}
			CompositeGrid = make_grid(regionBounds[regionIndex], regionLonSizes[regionIndex]);

			long compSize = CompositeGrid->count * sizeof(COMPOSITE_POINT);

			maxGridCount = get_max_grid_count(Params.CellSize, Params.KilometersPerDegree, compSize);
			if (maxGridCount < 0L) {
				log_error("Insufficient memory available for terrain caching");
				return -1;
			}

			if (CompositePoints) {
				mem_free(CompositePoints);
			}
			CompositePoints = (COMPOSITE_POINT *)mem_zalloc(compSize);

			ExtraCompositePointCount = 0;

		} else {

			maxGridCount = get_max_grid_count(Params.CellSize, Params.KilometersPerDegree, 0L);
			if (maxGridCount < 0L) {
				log_error("Insufficient memory available for terrain caching");
				return -1;
			}
		}

		// Make multiple passes to form sub-grid groupings based on memory limit.  On each pass, loop repeatedly over
		// the region source list looking for sources not yet studied and collect those with overlapping coverage that
		// would not make too large a grid.  Of course any one source must be studied regardless of the size of it's
		// isolated grid, so in some cases this may study only one station per pass.

		while (1) {

			sourceList = NULL;
			initialize_bounds(&cellBounds);
			cellLonSize = 999999;
			gridCount = 0L;
			gridCountSum = 0L;
			runCount = 0;
			fileCountSum = 0;

			do {

				loop = 0;

				for (source = regionSources[regionIndex]; source; source = source->regionNext) {

					if (source->isDesired < 0) {
						continue;
					}

					if (!sourceList) {

						tempCellBounds = source->grid->cellBounds;
						tempCellLonSize = source->grid->cellLonSize;
						tempGridCount = source->grid->count;
						source->isDesired = -1;

					} else {

						tempCellBounds = cellBounds;
						tempCellLonSize = cellLonSize;
						if (overlaps_bounds(&tempCellBounds, &(source->grid->cellBounds))) {
							extend_bounds_bounds(&tempCellBounds, &(source->grid->cellBounds));
							if (source->grid->cellLonSize < tempCellLonSize) {
								tempCellLonSize = source->grid->cellLonSize;
							}
							tempGridCount =
								(long)((tempCellBounds.northLatIndex - tempCellBounds.southLatIndex) / CellLatSize) *
								(long)((tempCellBounds.westLonIndex - tempCellBounds.eastLonIndex) / tempCellLonSize);
							if (tempGridCount < maxGridCount) {
								source->isDesired = -1;
							}
						}
					}

					if (source->isDesired < 0) {

						runCount++;

						source->next = sourceList;
						sourceList = source;

						cellBounds = tempCellBounds;
						cellLonSize = tempCellLonSize;
						gridCount = tempGridCount;

						// Sum up the approximate cell count in each source's grid for those that won't load from
						// cache, this is used to decide how to do the population query below.

						if (!source->hasDesiredCache) {
							gridCountSum += (long)((source->grid->cellBounds.northLatIndex -
									source->grid->cellBounds.southLatIndex) / CellLatSize) *
								(long)((source->grid->cellBounds.westLonIndex -
									source->grid->cellBounds.eastLonIndex) / source->grid->cellLonSize);
						}

						loop = 1;

						// When generating KML map output there may be a large number of temporary files open to
						// segment map output by source and into geographic tiles.  Check the estimated open file
						// count, if that that gets too high stop adding sources to this group.

						if (MapOutputFlags[MAP_OUT_KML] && ((STUDY_MODE_GRID == StudyMode) ||
								(MAP_OUT_KML_IND == MapOutputFlags[MAP_OUT_KML]))) {

							if (STUDY_MODE_GRID == StudyMode) {
								fileCount = get_kml_tile_count(source->grid->cellBounds);
							} else {
								fileCount = 1;
							}
							if (MapOutputFlags[MAP_OUT_KML_CPT]) {
								fileCountSum += fileCount * RESULT_COUNT;
							}
							if (MapOutputFlags[MAP_OUT_SELFIX] && Params.CheckSelfInterference && source->dts) {
								fileCountSum += fileCount;
							}

							if (fileCountSum > MAX_KML_FILE_COUNT) {
								loop = 0;
								break;
							}
						}
					}
				}

			} while (loop);

			// If no more sources to study, done.

			if (!sourceList) {
				break;
			}

			snprintf(mesg, MAX_STRING, "%d", runCount);
			status_message(STATUS_KEY_RUNCOUNT, mesg);

			// Set up the grid for this group of sources.

			log_message("Building study grid");
			if (Debug) {
				log_message("Grid setup for %d %d %d %d", cellBounds.southLatIndex, cellBounds.northLatIndex,
					cellBounds.eastLonIndex, cellBounds.westLonIndex);
			}
			if (gridCount > maxGridCount) {
				log_message("Grid size exceeds recommended limit, memory allocation may fail");
			}
			err = grid_setup(cellBounds, cellLonSize);
			if (err) return err;
			if (Debug) {
				log_message("Grid size %d %d %ld", CellLatSize, Grid->cellLonSize, Grid->count);
			}

			if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
				fprintf(outFile, "[grid]\n%d,%d,%d,%d\n", Grid->cellBounds.southLatIndex,
					Grid->cellBounds.eastLonIndex, Grid->cellBounds.northLatIndex, Grid->cellBounds.westLonIndex);
			}

			if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
				fprintf(outFile, "[grid]\n%d,%d,%d,%d\n", Grid->cellBounds.southLatIndex,
					Grid->cellBounds.eastLonIndex, Grid->cellBounds.northLatIndex, Grid->cellBounds.westLonIndex);
			}

			// Compare the combined cell count of all the sources that will not be loaded from cache to the count of
			// the grid.  If the combined count is relatively large, it is more efficient to do a single whole-grid
			// population query rather than having cell_setup() do separate overlapping queries for each source.  If
			// queries must be done regardless of caching to load census points, always use the whole-grid query.

			if ((gridCountSum > (Grid->count / 2)) || reloadCensusPoints) {
				err = load_grid_population(NULL);
				if (err) return err;
			}

			// Set caching flags (cleared as caches are loaded, to prevent multiple cache loads of the same source), do
			// cell setup for sources.

			source = Sources;
			for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
				if (source->inScenario) {
					source->dcache = 1;
					source->ucache = 1;
				}
			}

			cacheCount = 0L;

			for (source = sourceList; source; source = source->next) {

				if (Debug) {
					log_message("Cell setup for sourceKey=%d %d %d %d %d", source->sourceKey,
						source->grid->cellBounds.southLatIndex, source->grid->cellBounds.northLatIndex,
						source->grid->cellBounds.eastLonIndex, source->grid->cellBounds.westLonIndex);
				}

				err = cell_setup(source, &cacheCount, reloadCensusPoints);
				if (err) return err;
			}

			if (cacheCount) {
				log_message("Read %ld fields from cache", cacheCount);
			}

			// Clear caching flags (now they will be set as new fields are calculated to indicate which sources need
			// data written to cache), traverse the grid and calculate all fields needed.  That could be none, if
			// everything loaded from cache.

			source = Sources;
			for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
				if (source->inScenario) {
					source->dcache = 0;
					source->ucache = 0;
					initialize_bounds(&(source->ugridIndex));
				}
			}

			// Check if any fields need to be calculated.

			calcCount = 0L;
			pointPtr = Cells;
			for (gridCellIndex = 0L; gridCellIndex < Grid->count; gridCellIndex++, pointPtr++) {
				for (point = *pointPtr; point; point = point->next) {
					for (field = point->fields; field; field = field->next) {
						if (field->status < 0) {
							calcCount++;
						}
					}
				}
			}

			// Calculate new fields, periodically write to cache.  Ignore any errors from the writes.

			if (calcCount) {

				log_message("Calculating %ld new fields", calcCount);
				doneCount = 0L;

				if (calcCount > CALC_CACHE_COUNT) {
					nCalc = CALC_CACHE_COUNT;
					showPcnt = 1;
				} else {
					nCalc = (int)calcCount;
					showPcnt = 0;
				}
				hb_log_begin(nCalc);
				if (Terminate) return -1;

				pointPtr = Cells;
				for (gridCellIndex = 0L; gridCellIndex < Grid->count; gridCellIndex++, pointPtr++) {
					for (point = *pointPtr; point; point = point->next) {
						for (field = point->fields; field; field = field->next) {
							if (field->status < 0) {

								hb_log_tick();
								if (Terminate) return -1;
								err = project_field(point, field);
								if (err) return err;
								doneCount++;

								if (--nCalc == 0) {

									hb_log_end();
									if (Terminate) return -1;
									if (showPcnt) {
										calcPcnt = (int)(((double)doneCount / (double)calcCount) * 100.);
										log_message("%d%% done, updating caches", calcPcnt);
										if ((calcCount - doneCount) > CALC_CACHE_COUNT) {
											nCalc = CALC_CACHE_COUNT;
										} else {
											nCalc = (int)(calcCount - doneCount);
										}
									} else {
										log_message("Updating caches");
									}
									source = Sources;
									for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
										if (source->inScenario) {
											hb_log();
											if (Terminate) return -1;
											if (source->dcache) {
												write_cell_cache(source, CACHE_DES, 0, NULL);
												source->dcache = 0;
											}
											if (source->ucache && CacheUndesired) {
												write_cell_cache(source, CACHE_UND, 0, NULL);
												source->ucache = 0;
												initialize_bounds(&(source->ugridIndex));
											}
										}
									}

									if (nCalc > 0) {
										hb_log_begin(nCalc);
										if (Terminate) return -1;
									}
								}
							}
						}
					}
				}
			}

			// Write points to files as needed.

			log_message("Processing study points");

			err = write_points(MapOutputFile[MAP_OUT_SHAPE_POINTS], OutputFile[POINTS_FILE]);
			if (err) return err;

			// Determine final coverage for all sources, final output to cell and map files.  For individual-station
			// image output, allocate image arrays in each desired source; otherwise image goes to CompositePoints.

			if (DoImage && !DoCompositeImage) {
				for (source = sourceList; source; source = source->next) {
					if (source->imagePoints) {
						mem_free(source->imagePoints);
					}
					source->imagePoints = (IMAGE_POINT *)mem_zalloc(source->grid->count * sizeof(IMAGE_POINT));
				}
			}

			err = analyze_grid();
			if (err) return err;

			if (MapOutputFlags[MAP_OUT_KML]) {
				j = 1;
				for (source = sourceList; source; source = source->next) {
					for (i = 0; i < RESULT_COUNT; i++) {
						if (source->kmlCovPts[i]) {
							close_mapfile(source->kmlCovPts[i]);
							source->kmlCovPts[i] = NULL;
						}
					}
					if (source->kmlSelfIX) {
						close_mapfile(source->kmlSelfIX);
						source->kmlSelfIX = NULL;
					}
					if (MAP_OUT_KML_IND == MapOutputFlags[MAP_OUT_KML]) {
						if (DoImage) {
							log_message("Rendering image output %d of %d", j++, runCount);
						}
						err = do_bysource_kml(source);
						if (err) return err;
					}
					if (source->imagePoints) {
						mem_free(source->imagePoints);
						source->imagePoints = NULL;
					}
				}
			}

			if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
				fputs("[endgrid]\n", outFile);
			}

			if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
				fputs("[endgrid]\n", outFile);
			}
		}

		// Output to composite map files and database result table as needed.

		if (DoComposite) {
			status_message(STATUS_KEY_RUNCOUNT, "-1");
			err = do_write_compcov();
			if (err) return err;
		}

		// Render image output for this region grid if in global-layer KML mode.

		if (DoCompositeImage) {
			log_message("Rendering image output");
			err = do_write_image(regionIndex, NULL, NULL);
			if (err) return err;
		}
	}

	// Create the final database results table by joining the temporary points and results tables.  If there were no
	// points (can happen with a geography-based study area) don't create the table at all, just log a warning.

	if (DoTable) {

		if (ResultsRowCount > 0) {

			if (ResultsRowCount > 300000) {
				log_message("Running results table query, no progress bar will appear...");
			}

			char query[MAX_QUERY];

			snprintf(query, MAX_QUERY,
				"CREATE TABLE %s_%d.result_%d (INDEX(country_key, lat_index, lon_index)) AS (SELECT country_key, lat_index, lon_index, latitude, longitude, area, population, households, source_key, margin, result, service_count FROM %s_%d.temp_point_%d JOIN %s_%d.temp_result_%d USING (country_key, lat_index, lon_index));",
				DbName, StudyKey, ScenarioKey, DbName, StudyKey, ScenarioKey, DbName, StudyKey, ScenarioKey);
			if (mysql_query(MyConnection, query)) {
				log_db_error("Error creating results database table");
				DoTable = 0;

			} else {
				DidCreateResultTables = 1;
			}

			do_result_report_tables();

		} else {
			log_message("No study points found, results table not created");
			char query[MAX_QUERY];
			snprintf(query, MAX_QUERY, "DROP TABLE IF EXISTS %s_%d.result_%d_cenpt;", DbName, StudyKey, ScenarioKey);
			mysql_query(MyConnection, query);
		}
	}

	// Always attempt to drop temporary database tables if those might have been created.  They would be dropped when
	// the connection is closed but if there are more scenarios to run that won't happen right away so free up storage
	// early.  Also this may be cleaning up after table output was canceled due to error.

	if (CreateResultTables) {
		char query[MAX_QUERY];
		snprintf(query, MAX_QUERY, "DROP TABLE IF EXISTS %s_%d.temp_point_%d;", DbName, StudyKey, ScenarioKey);
		mysql_query(MyConnection, query);
		snprintf(query, MAX_QUERY, "DROP TABLE IF EXISTS %s_%d.temp_result_%d;", DbName, StudyKey, ScenarioKey);
		mysql_query(MyConnection, query);
		ResultsRowCount = 0;
	}

	status_message(STATUS_KEY_RUNCOUNT, "0");

	return 0;
}

// Write composite coverage results to shapefile and/or a database table as needed.  This will only return a major
// error (< 0), if a minor error occurs the output is disabled and a message logged.  The composite shapefile has just
// the point keys, a result code, and the source key for the best-case result in metadata.  Ideally it would also have
// the full metadata for the point and source, but it's often not possible to have the entire study grid in memory at
// once and only minimal data is held in the composite points array.  The metadata tables from other shapefiles can be
// joined in GIS software using the source and point keys.  The database table will have some point metadata because a
// points table is also being created, see write_points(), but it will not have source metadata.  The actual source
// table is of course in the study database so can easily be joined to the results table using the source key.  This
// may be called multiple times for the same scenario in case of a region split.

static int do_write_compcov() {

	MAPFILE *mapFileShp = MapOutputFile[MAP_OUT_SHAPE_CMPPTS];
	if (!mapFileShp && !DoTable) {
		return 0;
	}
	log_message("Processing composite coverage");

	char *attrData[COMPCOV_NATTR], attrCntKey[LEN_COUNTRYKEY + 1], attrLatIdx[LEN_INDEX + 1], attrLonIdx[LEN_INDEX + 1],
		attrSrcKey[LEN_SOURCEKEY + 1];

	attrData[0] = attrCntKey;
	attrData[1] = attrLatIdx;
	attrData[2] = attrLonIdx;
	attrData[3] = attrSrcKey;
	attrCntKey[0] = '\0';
	attrLatIdx[0] = '\0';
	attrLonIdx[0] = '\0';
	attrSrcKey[0] = '\0';

	// Traverse the composite result, write to shapefile and/or table.

	int latIndex = CompositeGrid->cellBounds.southLatIndex, lonIndex, lonSize, lonCount, latGridIndex, lonGridIndex,
		countryKey, compResult, queryLen = 0;
	char values[MAX_STRING], query[MAX_QUERY];
	double lat, lon, halfCellLat = (double)CellLatSize / 7200., halfCellLon;
	COMPOSITE_POINT *compositePoint;

	hb_log_begin(CompositeGrid->latCount);

	for (latGridIndex = 0; latGridIndex < CompositeGrid->latCount; latGridIndex++, latIndex += CellLatSize) {

		hb_log_tick();
		if (Terminate) return -1;

		snprintf(attrLatIdx, (LEN_INDEX + 1), "%d", latIndex);

		lonIndex = CompositeGrid->cellEastLonIndex[latGridIndex];
		lonSize = CompositeGrid->cellLonSizes[latGridIndex];
		lonCount = (((CompositeGrid->cellBounds.westLonIndex - 1) - lonIndex) / lonSize) + 1;

		halfCellLon = (double)lonSize / 7200.;

		for (lonGridIndex = 0; lonGridIndex < lonCount; lonGridIndex++, lonIndex += lonSize) {

			snprintf(attrLonIdx, (LEN_INDEX + 1), "%d", lonIndex);

			lat = ((double)latIndex / 3600.) + halfCellLat;
			lon = ((double)lonIndex / 3600.) + halfCellLon;

			for (countryKey = 1; countryKey <= MAX_COUNTRY; countryKey++) {

				compositePoint = get_composite_point(countryKey, latGridIndex, lonGridIndex, 0);
				if (!compositePoint) {
					continue;
				}

				compResult = compositePoint->result;
				if (0 == compResult) {
					continue;
				}

				// All points are included in the database table, regardless of no-service or no-pop flags.

				if (DoTable) {

					if ((MAX_QUERY - queryLen) < MAX_STRING) {
						lcatstr(query, ";", MAX_QUERY);
						queryLen = 0;
						if (mysql_query(MyConnection, query)) {
							log_db_error("Error writing to temporary results table (1)");
							DoTable = 0;
						}
					}

					if (0 == queryLen) {
						queryLen = snprintf(query, MAX_QUERY,
							"INSERT INTO %s_%d.temp_result_%d VALUES (%d,%d,%d,%d,%d,%d,%d)",
							DbName, StudyKey, ScenarioKey, countryKey, latIndex, lonIndex, compositePoint->sourceKey,
							compositePoint->margin, compResult, compositePoint->serviceCount);
					} else {
						snprintf(values, MAX_STRING, ",(%d,%d,%d,%d,%d,%d,%d)", countryKey, latIndex, lonIndex,
							compositePoint->sourceKey, compositePoint->margin, compResult,
							compositePoint->serviceCount);
						queryLen = lcatstr(query, values, MAX_QUERY);
					}

					ResultsRowCount++;
				}

				// No-service and/or zero-population points may be excluded from shapefile output.

				if (mapFileShp && (!MapOutputFlags[MAP_OUT_NOSERV] || (RESULT_NOSERVICE != compResult)) &&
						(!MapOutputFlags[MAP_OUT_NOPOP] || compositePoint->hasPop)) {

					snprintf(attrCntKey, (LEN_COUNTRYKEY + 1), "%d", countryKey);
					snprintf(attrSrcKey, (LEN_SOURCEKEY + 1), "%d", compositePoint->sourceKey);

					switch (compResult) {
						case RESULT_COVERAGE: {
							attrData[4] = "1";
							break;
						}
						case RESULT_INTERFERE: {
							attrData[4] = "2";
							break;
						}
						case RESULT_NOSERVICE: {
							attrData[4] = "3";
							break;
						}
					}

					write_shape(mapFileShp, lat, lon, NULL, 0, NULL, attrData, NULL, NULL, 0, 1);
				}
			}
		}
	}

	if (DoTable && (queryLen > 0)) {
		lcatstr(query, ";", MAX_QUERY);
		if (mysql_query(MyConnection, query)) {
			log_db_error("Error writing to temporary results table (2)");
			DoTable = 0;
		}
	}

	hb_log_end();
	if (Terminate) return -1;

	return 0;
}

// Generate image output.  Write PostScript to a file to draw and color-fill the cells, then use the "gs" command to
// render to a pixel image.  If rendering succeeds write to a KML file that loads the image as a ground overlay, and
// delete the PostScript file.  This may return major error (< 0) for termination, but will not return a minor error
// (> 0).  A minor error will be logged and image output canceled but 0 is returned.  A minor failure here is isolated
// and does not need to propagate.

// To handle arbitrarily-large areas, this will render the image in sections on a grid, then tile the pieces back
// togther with multiple overlay features in the KML.  The tiling is based on a threshold size in latitude and
// longitude (in integer arc-seconds), so the tile size in land area will vary with latitude.  Each overlay also has
// a <Region> element with an appropriate LOD to control visibility.

// This may be called multiple times for the same scenario if that is broken up into non-overlapping regions, or it may
// be called once for each desired station regardless of region if the by-station map option is active.  If the source
// is null the image data is in the CompositePoints array and regionID is used in the image tile names for uniqueness,
// in that case KML output goes to a global file.  If source is not null image data is in a per-source array, source
// key is used in file names, and the KML output file must be provided as argument, see do_bysource_kml().

static int do_write_image(int regionID, SOURCE *source, FILE *kmlFile) {

	GRID *imageGrid;
	if (source) {
		regionID = source->sourceKey;
		imageGrid = source->grid;
	} else {
		imageGrid = CompositeGrid;
		kmlFile = ImageFileKML;
	}
	if (!kmlFile) return 0;

	// Set up image tiling, loop over tiles.  The tile boundaries are set at arbitrary latitude and longitude index
	// (arc-second) points, so those boundaries will not necessarily fall on cell edges.  In the PostScript, the tile
	// boundary is set as a clipping path then an extra cell is drawn all around, so the final images will line up
	// correctly edge-to-edge.

	int latTileSize, latTileCount, lonTileSize, lonTileCount, latTile, lonTile, southLatIndex, northLatIndex,
		eastLonIndex, westLonIndex, latIndex, latSize, lonIndex, lonSize, southLatGridIndex, northLatGridIndex,
		eastLonGridIndex, westLonGridIndex, latGridIndex, lonGridIndex, pointGridIndex, imageResult, colorIndex,
		err, didWrite;
	double slat, nlat, elon, wlon;
	FILE *outFile;
	char fileName[MAX_STRING], tempFileName[MAX_STRING];

	latSize = (imageGrid->cellBounds.northLatIndex - imageGrid->cellBounds.southLatIndex) + CellLatSize;
	latTileCount = (latSize / IMAGE_TILE_SIZE) + 1;
	latTileSize = latSize / latTileCount;

	lonSize = (imageGrid->cellBounds.westLonIndex - imageGrid->cellBounds.eastLonIndex) + Grid->cellLonSize;
	lonTileCount = (lonSize / IMAGE_TILE_SIZE) + 1;
	lonTileSize = lonSize / lonTileCount;

	didWrite = 0;
	err = 0;

	hb_log_begin(latTileCount * lonTileCount);

	for (latTile = 0; latTile < latTileCount; latTile++) {

		southLatIndex = imageGrid->cellBounds.southLatIndex + (latTile * latTileSize);
		if ((latTileCount - 1) == latTile) {
			northLatIndex = imageGrid->cellBounds.northLatIndex + CellLatSize;
		} else {
			northLatIndex = southLatIndex + latTileSize;
		};

		southLatGridIndex = (southLatIndex - imageGrid->cellBounds.southLatIndex) / CellLatSize;
		northLatGridIndex = ((northLatIndex - imageGrid->cellBounds.southLatIndex) / CellLatSize) + 1;
		if (northLatGridIndex >= imageGrid->latCount) {
			northLatGridIndex = imageGrid->latCount - 1;
		}

		for (lonTile = 0; lonTile < lonTileCount; lonTile++) {

			hb_log_tick();
			if (Terminate) {
				err = -1;
				break;
			}

			eastLonIndex = imageGrid->cellBounds.eastLonIndex + (lonTile * lonTileSize);
			if ((lonTileCount - 1) == lonTile) {
				westLonIndex = imageGrid->cellBounds.westLonIndex + Grid->cellLonSize;
			} else {
				westLonIndex = eastLonIndex + lonTileSize;
			}

			outFile = NULL;

			// Loop over cells.

			latIndex = imageGrid->cellBounds.southLatIndex + (southLatGridIndex * CellLatSize);
			for (latGridIndex = southLatGridIndex; latGridIndex <= northLatGridIndex;
					latGridIndex++, latIndex += CellLatSize) {

				lonSize = imageGrid->cellLonSizes[latGridIndex];
				eastLonGridIndex = (eastLonIndex - imageGrid->cellEastLonIndex[latGridIndex]) / lonSize;
				westLonGridIndex = ((westLonIndex - imageGrid->cellEastLonIndex[latGridIndex]) / lonSize) + 1;
				if (westLonGridIndex >= imageGrid->lonCounts[latGridIndex]) {
					westLonGridIndex = imageGrid->lonCounts[latGridIndex] - 1;
				}

				lonIndex = imageGrid->cellEastLonIndex[latGridIndex] + (eastLonGridIndex * lonSize);
				for (lonGridIndex = eastLonGridIndex; lonGridIndex <= westLonGridIndex;
						lonGridIndex++, lonIndex += lonSize) {

					pointGridIndex = (latGridIndex * imageGrid->lonCount) + lonGridIndex;
					if (source) {
						imageResult = source->imagePoints[pointGridIndex].result;
						colorIndex = source->imagePoints[pointGridIndex].colorIndex;
					} else {
						imageResult = CompositePoints[pointGridIndex].imagePoint.result;
						colorIndex = CompositePoints[pointGridIndex].imagePoint.colorIndex;
					}

					if (0 != imageResult) {

						// Open the temporary PostScript file only when encountering the first cell to actually draw;
						// if it turns out there are none, output for this tile will be skipped.

						if (!outFile) {
							outFile = make_tempfile(tempFileName);
							if (!outFile) {
								err = 1;
								break;
							}
							fprintf(outFile, "%%!PS-Adobe\n");
							fputs("/Draw {moveto dup 0 exch rlineto exch neg 0 rlineto neg 0 exch rlineto\n", outFile);
							fputs("closepath setrgbcolor fill} bind def\n", outFile);
							fprintf(outFile, "0.1 0.1 scale %d %d translate\n", westLonIndex, -southLatIndex);
							fprintf(outFile,
									"%d %d moveto %d %d lineto %d %d lineto %d %d lineto closepath clip newpath\n",
									-westLonIndex, southLatIndex, -eastLonIndex, southLatIndex, -eastLonIndex,
									northLatIndex, -westLonIndex, northLatIndex);
						}

						fprintf(outFile, "%.3f %.3f %.3f %d %d %d %d Draw\n", ImageColorR[colorIndex],
							ImageColorG[colorIndex], ImageColorB[colorIndex], lonSize, CellLatSize, -lonIndex,
							latIndex);
						ImageColorIndexUsed[colorIndex] = 1;
					}
				}

				if (err) break;
			}

			if (err) break;

			if (!outFile) continue;

			fputs("showpage\n", outFile);
			file_close(outFile);

			snprintf(fileName, MAX_STRING, "%d_%d_%d.png", regionID, latTile, lonTile);

			// Do the render.

			err = render_image(tempFileName, fileName, ((westLonIndex - eastLonIndex) / 10),
				((northLatIndex - southLatIndex) / 10));

			unlink(tempFileName);

			if (err) break;

			// Add feature to the KML file for this tile.  Open folder first if needed.

			if (!didWrite && source) {
				fprintf(kmlFile, "<Folder>\n<name>%s</name>\n", IMAGE_MAPFILE_TITLE);
			}

			slat = (double)southLatIndex / 3600.;
			nlat = (double)northLatIndex / 3600.;
			elon = (double)eastLonIndex / 3600.;
			wlon = (double)westLonIndex / 3600.;

			fprintf(kmlFile, "<GroundOverlay>\n<name>%d-%d</name>\n", latTile, lonTile);
			fprintf(kmlFile, "<color>%s</color><Icon><href>%s/%s</href></Icon>\n", IMAGE_OVERLAY_COLOR,
				IMAGE_MAPFILE_NAME, fileName);
			fprintf(kmlFile, "<LatLonBox><north>%.8f</north><south>%.8f</south>", nlat, slat);
			fprintf(kmlFile, "<east>%.8f</east><west>%.8f</west></LatLonBox>\n", -elon, -wlon);
			fprintf(kmlFile, "<Region>\n<LatLonAltBox><north>%.8f</north><south>%.8f</south>", nlat, slat);
			fprintf(kmlFile, "<east>%.8f</east><west>%.8f</west></LatLonAltBox>\n", -elon, -wlon);
			fprintf(kmlFile, "<Lod><minLodPixels>%d</minLodPixels></Lod>\n</Region>\n", IMAGE_LOD);
			fputs("</GroundOverlay>\n", kmlFile);

			didWrite = 1;
		}

		if (err) break;
	}

	// Close folder if needed.

	if (didWrite) {
		DidWriteImageKML = 1;
		if (source) {
			fputs("</Folder>\n", kmlFile);
		}
	}

	// Done.

	hb_log_end();

	if (err < 0) {
		return -1;
	}
	return 0;
}

// Write an image output legend and place in KML file as a screen overlay.

static int do_image_legend(FILE *kmlFile, int fromTop) {

	if (!kmlFile) return 0;

	char tempFileName[MAX_STRING];

	FILE *outFile = make_tempfile(tempFileName);
	if (!outFile) return 1;

	char *units = "", *line1 = "", *line2 = NULL;
	switch (MapOutputFlags[MAP_OUT_IMAGE]) {
		case MAP_OUT_IMAGE_DMARG: {
			units = "dB";
			line1 = "Desired Signal Margin";
			break;
		}
		case MAP_OUT_IMAGE_DUMARG: {
			units = "dB";
			line1 = "D/U Margin";
			break;
		}
		case MAP_OUT_IMAGE_WLDUMARG: {
			units = "dB";
			line1 = "D/U Margin";
			line2 = "Wireless Interference";
			break;
		}
		case MAP_OUT_IMAGE_SELFDUMARG: {
			units = "dB";
			line1 = "D/U Margin";
			line2 = "DTS Self-Interference";
			break;
		}
		case MAP_OUT_IMAGE_MARG: {
			units = "dB";
			line1 = "Worst-case Margin";
			break;
		}
		case MAP_OUT_IMAGE_DESSIG: {
			units = "dBu";
			line1 = "Desired signal";
			break;
		}
	}

	fprintf(outFile, "%%!PS-Adobe\n");
	fputs("/Label {gsave dup false charpath 1.4 setlinewidth 0 setgray\n", outFile);
	fputs("stroke grestore 1 setgray show} bind def\n", outFile);
	fputs("/Draw {20 exch moveto gsave 20 0 rlineto 0 20 rlineto -20 0 rlineto\n", outFile);
	fputs("closepath setrgbcolor fill grestore 30 5 rmoveto Label} bind def\n", outFile);
	fputs("/Helvetica-Bold findfont 14 scalefont setfont\n", outFile);

	int colorIndex, rowY = 20;
	char label[MAX_STRING];

	for (colorIndex = 0; colorIndex < ImageColorCount; colorIndex++) {

		if (ImageColorIndexUsed[colorIndex]) {

			if (colorIndex == ImageNoServiceIndex) {
				lcpystr(label, "No Service", MAX_STRING);
			} else {
				if (colorIndex == ImageInterferenceIndex) {
					lcpystr(label, "Interference", MAX_STRING);
				} else {
					if (colorIndex == ImageBackgroundIndex) {
						snprintf(label, MAX_STRING, "< %.0f %s", ImageLevel[ImageMapStartIndex], units);
					} else {
						if (colorIndex == (ImageColorCount - 1)) {
							snprintf(label, MAX_STRING, "> %.0f %s", ImageLevel[colorIndex], units);
						} else {
							snprintf(label, MAX_STRING, "%.0f to %.0f %s", ImageLevel[colorIndex],
								ImageLevel[colorIndex + 1], units);
						}
					}
				}
			}

			fprintf(outFile, "(%s) %.3f %.3f %.3f %d Draw\n", label, ImageColorR[colorIndex], ImageColorG[colorIndex],
				ImageColorB[colorIndex], rowY);
			rowY += 30;

			ImageColorIndexUsed[colorIndex] = 0;
		}
	}

	if (line2) {
		fprintf(outFile, "20 %d moveto (%s) Label\n", rowY, line2);
		rowY += 15;
		fprintf(outFile, "20 %d moveto (%s) Label\n", rowY, line1);
	} else {
		fprintf(outFile, "20 %d moveto (%s) Label\n", rowY, line1);
	}
	rowY += 35;

	fputs("showpage\n", outFile);
	file_close(outFile);

	int err = render_image(tempFileName, IMAGE_LEGEND_NAME, IMAGE_LEGEND_WIDTH, rowY);

	unlink(tempFileName);

	if (!err) {
		fprintf(kmlFile, "<ScreenOverlay>\n<name>%s</name>\n", IMAGE_LEGEND_TITLE);
		if (fromTop) {
			fprintf(kmlFile, "<color>ffffffff</color><Icon><href>%s/%s/%s</href></Icon>\n", KML_SUBDIR,
				IMAGE_MAPFILE_NAME, IMAGE_LEGEND_NAME);
		} else {
			fprintf(kmlFile, "<color>ffffffff</color><Icon><href>%s/%s</href></Icon>\n", IMAGE_MAPFILE_NAME,
				IMAGE_LEGEND_NAME);
		}
		fputs("<overlayXY x=\"0\" y=\"0\" xunits=\"pixels\" yunits=\"pixels\"/>\n", kmlFile);
		fputs("<screenXY x=\"0\" y=\"50\" xunits=\"pixels\" yunits=\"pixels\"/>\n", kmlFile);
		fprintf(kmlFile, "<size x=\"%d\" y=\"%d\" xunits=\"pixels\" yunits=\"pixels\"/>\n", IMAGE_LEGEND_WIDTH,
			rowY);
		fputs("</ScreenOverlay>\n", kmlFile);
	}

	return err;
}

// Write coverage and interference totals to database tables, when creating tables.

void do_result_report_tables() {

	if (!DoTable) {
		return;
	}

	SOURCE *source, *usource;
	UNDESIRED *undesireds;
	int sourceIndex, undesiredIndex, undesiredCount, countryIndex, countryKey;
	DES_TOTAL *dtot;
	UND_TOTAL *utot;
	char query[MAX_QUERY];

	snprintf(query, MAX_QUERY,
		"CREATE TABLE %s_%d.result_%d_desired (source_key INT, country_key INT, contour_area DOUBLE, contour_population INT, contour_households INT, service_area DOUBLE, service_population INT, service_households INT, ix_free_area DOUBLE, ix_free_population INT, ix_free_households INT);",
		DbName, StudyKey, ScenarioKey);
	if (mysql_query(MyConnection, query)) {
		log_db_error("Error creating results report table (1)");
		DoTable = 0;
		return;
	}

	snprintf(query, MAX_QUERY,
		"CREATE TABLE %s_%d.result_%d_undesired (source_key INT, country_key INT, undesired_source_key INT, ix_area DOUBLE, ix_population INT, ix_households INT, unique_ix_area DOUBLE, unique_ix_population INT, unique_ix_households INT);",
		DbName, StudyKey, ScenarioKey);
	if (mysql_query(MyConnection, query)) {
		log_db_error("Error creating results report table (2)");
		DoTable = 0;
		return;
	}

	source = Sources;
	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {

		if (!source->isDesired) {
			continue;
		}

		undesireds = source->undesireds;
		undesiredCount = source->undesiredCount;

		for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {

			countryKey = countryIndex + 1;
			dtot = source->totals + countryIndex;
			if ((countryKey != source->countryKey) && (0. == dtot->contourArea)) {
				continue;
			}

			snprintf(query, MAX_QUERY,
				"INSERT INTO %s_%d.result_%d_desired VALUES (%d,%d,%.1f,%d,%d,%.1f,%d,%d,%.1f,%d,%d);",
				DbName, StudyKey, ScenarioKey, source->sourceKey, countryKey, dtot->contourArea, dtot->contourPop,
				dtot->contourHouse, dtot->serviceArea, dtot->servicePop, dtot->serviceHouse, dtot->ixFreeArea,
				dtot->ixFreePop, dtot->ixFreeHouse);
			if (mysql_query(MyConnection, query)) {
				log_db_error("Error writing to results report table (1)");
				DoTable = 0;
				return;
			}

			if (source->isParent && Params.CheckSelfInterference) {

				utot = source->selfIXTotals + countryIndex;

				if (utot->ixArea > 0.) {
					snprintf(query, MAX_QUERY,
						"INSERT INTO %s_%d.result_%d_undesired VALUES (%d,%d,%d,%.1f,%d,%d,%.1f,%d,%d);",
						DbName, StudyKey, ScenarioKey, source->sourceKey, countryKey, source->sourceKey, utot->ixArea,
						utot->ixPop, utot->ixHouse, utot->uniqueIxArea, utot->uniqueIxPop, utot->uniqueIxHouse);
					if (mysql_query(MyConnection, query)) {
						log_db_error("Error writing to results report table (2)");
						DoTable = 0;
						return;
					}
				}
			}

			for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {

				usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
				if (!usource) {
					log_error("Source structure index is corrupted");
					exit(1);
				}

				if (RECORD_TYPE_WL == usource->recordType) {
					continue;
				}

				utot = undesireds[undesiredIndex].totals + countryIndex;

				if (utot->ixArea > 0.) {
					snprintf(query, MAX_QUERY,
						"INSERT INTO %s_%d.result_%d_undesired VALUES (%d,%d,%d,%.1f,%d,%d,%.1f,%d,%d);",
						DbName, StudyKey, ScenarioKey, source->sourceKey, countryKey, usource->sourceKey, utot->ixArea,
						utot->ixPop, utot->ixHouse, utot->uniqueIxArea, utot->uniqueIxPop, utot->uniqueIxHouse);
					if (mysql_query(MyConnection, query)) {
						log_db_error("Error writing to results report table (3)");
						DoTable = 0;
						return;
					}
				}
			}
		}
	}
}

// Do scenario run for a local grid.  In this mode, each desired source is studied independently on it's own cell grid,
// the grid is defined based on the bounds of the individual coverage contour and so the cells are never the same even
// where there is coverage overlap between sources.  Undesired signal projections thus cannot be shared with other
// desired source studies.  In this type of study, simply loop over the sources and study each one on it's own grid.
// Otherwise the logic is similar to global mode above; see that code for detailed comments.

static int do_local_run() {

	SOURCE *source, *usource;
	UNDESIRED *undesireds;
	POINT **pointPtr, *point;
	FIELD *field;
	long gridCellIndex, cacheCount, calcCount, doneCount;
	int err, sourceIndex, undesiredIndex, undesiredCount, nCalc, showPcnt, calcPcnt, i;
	FILE *outFile;

	int reloadCensusPoints = (CELL_DETAIL_CENPT == OutputFlags[CELL_FILE_DETAIL]);

	source = Sources;
	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {

		if (source->isDesired) {

			undesireds = source->undesireds;
			undesiredCount = source->undesiredCount;

			source->dcache = 1;
			source->ucache = 1;
			for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {
				usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
				if (!usource) {
					log_error("Source structure index is corrupted");
					exit(1);
				}
				usource->ucache = 1;
			}

			log_message("Building study grid");

			if (Debug) {
				log_message("Grid and cell setup for sourceKey=%d %d %d %d %d", source->sourceKey,
					source->grid->cellBounds.southLatIndex, source->grid->cellBounds.northLatIndex,
					source->grid->cellBounds.eastLonIndex, source->grid->cellBounds.westLonIndex);
			}

			status_message(STATUS_KEY_RUNCOUNT, "1");

			err = grid_setup(source->grid->cellBounds, source->grid->cellLonSize);
			if (err) return err;
			if (Debug) {
				log_message("Grid size %d %d %ld", CellLatSize, Grid->cellLonSize, Grid->count);
			}

			if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
				fprintf(outFile, "[grid]\n%d,%d,%d,%d\n", Grid->cellBounds.southLatIndex,
					Grid->cellBounds.eastLonIndex, Grid->cellBounds.northLatIndex, Grid->cellBounds.westLonIndex);
			}

			if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
				fprintf(outFile, "[grid]\n%d,%d,%d,%d\n", Grid->cellBounds.southLatIndex,
					Grid->cellBounds.eastLonIndex, Grid->cellBounds.northLatIndex, Grid->cellBounds.westLonIndex);
			}

			cacheCount = 0L;

			err = cell_setup(source, &cacheCount, reloadCensusPoints);
			if (err) return err;

			if (cacheCount) {
				log_message("Read %ld fields from cache", cacheCount);
			}

			source->dcache = 0;
			source->ucache = 0;
			for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {
				usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
				if (!usource) {
					log_error("Source structure index is corrupted");
					exit(1);
				}
				usource->ucache = 0;
				initialize_bounds(&(usource->ugridIndex));
			}

			calcCount = 0L;
			pointPtr = Cells;
			for (gridCellIndex = 0L; gridCellIndex < Grid->count; gridCellIndex++, pointPtr++) {
				for (point = *pointPtr; point; point = point->next) {
					for (field = point->fields; field; field = field->next) {
						if (field->status < 0) {
							calcCount++;
						}
					}
				}
			}

			if (calcCount) {

				log_message("Calculating %ld new fields", calcCount);
				doneCount = 0L;

				if (calcCount > CALC_CACHE_COUNT) {
					nCalc = CALC_CACHE_COUNT;
					showPcnt = 1;
				} else {
					nCalc = (int)calcCount;
					showPcnt = 0;
				}
				hb_log_begin(nCalc);
				if (Terminate) return -1;

				pointPtr = Cells;
				for (gridCellIndex = 0L; gridCellIndex < Grid->count; gridCellIndex++, pointPtr++) {
					for (point = *pointPtr; point; point = point->next) {
						for (field = point->fields; field; field = field->next) {
							if (field->status < 0) {

								hb_log_tick();
								if (Terminate) return -1;
								err = project_field(point, field);
								if (err) return err;
								doneCount++;

								if (--nCalc == 0) {

									hb_log_end();
									if (Terminate) return -1;
									if (showPcnt) {
										calcPcnt = (int)(((double)doneCount / (double)calcCount) * 100.);
										log_message("%d%% done, updating caches", calcPcnt);
										if ((calcCount - doneCount) > CALC_CACHE_COUNT) {
											nCalc = CALC_CACHE_COUNT;
										} else {
											nCalc = (int)(calcCount - doneCount);
										}
									} else {
										log_message("Updating caches");
									}
									if (source->dcache) {
										write_cell_cache(source, CACHE_DES, 0, NULL);
										source->dcache = 0;
									}
									for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {
										usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
										if (!usource) {
											log_error("Source structure index is corrupted");
											exit(1);
										}
										if (usource->ucache && CacheUndesired) {
											write_cell_cache(usource, CACHE_UND, source->sourceKey,
												&(undesireds[undesiredIndex].ucacheChecksum));
											usource->ucache = 0;
											initialize_bounds(&(usource->ugridIndex));
										}
									}

									// Must do a separate check for caching a source as an undesired to itself.  When
									// DTS self-interference is being analyzed the desired source also has undesired
									// fields calculated, but it does not appear in it's own list of undesireds so the
									// loop above does not handle this case.

									if (source->ucache && CacheUndesired) {
										write_cell_cache(source, CACHE_UND, source->sourceKey,
											&(source->ucacheChecksum));
									}

									if (nCalc > 0) {
										hb_log_begin(nCalc);
										if (Terminate) return -1;
									}
								}
							}
						}
					}
				}
			}

			log_message("Processing study points");

			err = write_points(MapOutputFile[MAP_OUT_SHAPE_POINTS], OutputFile[POINTS_FILE]);
			if (err) return err;

			err = analyze_source(source);
			if (err) return err;

			if (MapOutputFlags[MAP_OUT_KML]) {
				for (i = 0; i < RESULT_COUNT; i++) {
					if (source->kmlCovPts[i]) {
						close_mapfile(source->kmlCovPts[i]);
						source->kmlCovPts[i] = NULL;
					}
				}
				if (source->kmlSelfIX) {
					close_mapfile(source->kmlSelfIX);
					source->kmlSelfIX = NULL;
				}
				if (MAP_OUT_KML_IND == MapOutputFlags[MAP_OUT_KML]) {
					err = do_bysource_kml(source);
					if (err) return err;
				}
			}

			if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
				fputs("[endgrid]\n", outFile);
			}

			if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
				fputs("[endgrid]\n", outFile);
			}
		}
	}

	status_message(STATUS_KEY_RUNCOUNT, "0");

	return 0;
}

// Do scenario run for points mode.  This is fundamentally different than the other modes because the focus is the
// individual study points, not the desired stations.  Service and interference conditions for all desired stations,
// within a maximum distance limit, analyzed at each study point regardless of whether that point is or is not in the
// desired station's service area.  The set of points is arbitrarily defined by user input.

static int do_points_run() {

	SOURCE *source;
	POINT *point;
	FIELD *field;
	int err, sourceIndex, calcCount, i;
	FILE *outFile;

	status_message(STATUS_KEY_RUNCOUNT, "-1");

	clear_points();

	source = Sources;
	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
		if (source->isDesired) {
			err = points_setup(source);
			if (err) return err;
		}
	}

	if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
		fputs("[points]\n", outFile);
	}

	if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
		fputs("[points]\n", outFile);
	}

	calcCount = 0;
	for (point = Points; point; point = point->next) {
		for (field = point->fields; field; field = field->next) {
			if (field->status < 0) {
				calcCount++;
			}
		}
	}

	if (calcCount) {
		log_message("Calculating %d new fields", calcCount);
		hb_log_begin(calcCount);
		if (Terminate) return -1;
		for (point = Points; point; point = point->next) {
			for (field = point->fields; field; field = field->next) {
				if (field->status < 0) {

					hb_log_tick();
					if (Terminate) return -1;
					err = project_field(point, field);
					if (err) return err;

					if (OutputFlags[POINT_FILE_PROF] && !field->a.isUndesired && ProfileCount) {

						char profName[MAX_STRNGL];

						source = SourceKeyIndex[field->sourceKey];
						snprintf(profName, MAX_STRNGL, "profile-%s_%s_%s_%s-%s.csv", source->callSign,
							channel_label(source), source->serviceCode, source->status,
							PointInfos[point->a.pointIndex].pointName);
						outFile = open_out_file_dir(profName, PROFILE_SUBDIR);
						if (!outFile) return 1;

						int pointIndex;
						for (pointIndex = 0; pointIndex < ProfileCount; pointIndex++) {
							fprintf(outFile, "%.2f,%.2f\n", ((double)pointIndex / ProfilePpk), Profile[pointIndex]);
						}

						file_close(outFile);
					}
				}
			}
		}
		hb_log_end();
		if (Terminate) return -1;
	}

	log_message("Processing study points");

	err = write_points(MapOutputFile[MAP_OUT_SHAPE_POINTS], OutputFile[POINTS_FILE]);
	if (err) return err;

	analyze_points(Points, NULL);

	if (MAP_OUT_KML_IND == MapOutputFlags[MAP_OUT_KML]) {
		source = Sources;
		for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
			if (source->isDesired) {
				for (i = 0; i < RESULT_COUNT; i++) {
					if (source->kmlCovPts[i]) {
						close_mapfile(source->kmlCovPts[i]);
						source->kmlCovPts[i] = NULL;
					}
				}
				if (source->kmlSelfIX) {
					close_mapfile(source->kmlSelfIX);
					source->kmlSelfIX = NULL;
				}
				err = do_bysource_kml(source);
				if (err) return err;
			}
		}
	}

	if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
		fputs("[endpoints]\n", outFile);
	}

	if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
		fputs("[endpoints]\n", outFile);
	}

	status_message(STATUS_KEY_RUNCOUNT, "0");

	return 0;
}

// Write KML file for an individual source in individual-layer output mode, in this mode each desired source gets all
// of it's KML map features in a separate file just for that source, including image output in global grid mode.  This
// may be used in any study mode, global grid, local grid, or points.

static int do_bysource_kml(SOURCE *source) {

	SOURCE *dtsSource, *usource;
	UNDESIRED *undesireds;
	int err, undesiredCount, undesiredIndex, i;
	double cullDist;
	char fileName[MAX_STRING], *title;

	snprintf(fileName, MAX_STRING, "%d.kml", source->sourceKey);
	FILE *kmlFile = open_out_file_dir(fileName, KML_SUBDIR);
	if (!kmlFile) return 1;

	kml_start(kmlFile, source->callSign);

	// Write station point and contour features, see source.c for details.  For a DTS these are grouped in folders.

	if (source->isParent) {

		fprintf(kmlFile, "<Folder>\n<name>%s</name>\n", CONTOUR_MAPFILE_TITLE);
		write_source_shapes(source, NULL, ContoursKML, 0, 0., kmlFile, 0);
		write_source_shapes(source->dtsAuthSource, NULL, ContoursKML, 0, 0., kmlFile, 0);
		for (dtsSource = source->dtsSources; dtsSource; dtsSource = dtsSource->next) {
			write_source_shapes(dtsSource, NULL, ContoursKML, 0, 0., kmlFile, 1);
		}
		fputs("</Folder>\n", kmlFile);

		fprintf(kmlFile, "<Folder>\n<name>%s</name>\n", DSOURCE_MAPFILE_TITLE);
		write_source_shapes(source, DesiredSourcesKML, NULL, 0, 0., kmlFile, 0);
		write_source_shapes(source->dtsAuthSource, DesiredSourcesKML, NULL, 0, 0., kmlFile, 0);
		for (dtsSource = source->dtsSources; dtsSource; dtsSource = dtsSource->next) {
			write_source_shapes(dtsSource, DesiredSourcesKML, NULL, 0, 0., kmlFile, 1);
		}
		fputs("</Folder>\n", kmlFile);

	} else {

		write_source_shapes(source, DesiredSourcesKML, ContoursKML, 0, 0., kmlFile, 1);
	}

	int doFolder = 1;

	undesireds = source->undesireds;
	undesiredCount = source->undesiredCount;

	for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {

		usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
		if (!usource) {
			log_error("Source structure index is corrupted");
			exit(1);
		}

		if (doFolder) {
			fprintf(kmlFile, "<Folder>\n<name>%s</name>\n", USOURCE_MAPFILE_TITLE);
			doFolder = 0;
		}

		cullDist = undesireds[undesiredIndex].ixDistance;

		if (usource->isParent) {
			for (dtsSource = usource->dtsSources; dtsSource; dtsSource = dtsSource->next) {
				write_source_shapes(dtsSource, UndesiredSourcesKML, NULL, source->sourceKey, cullDist, kmlFile, 0);
			}
		} else {
			write_source_shapes(usource, UndesiredSourcesKML, NULL, source->sourceKey, cullDist, kmlFile, 0);
		}
	}

	if (!doFolder) {
		fputs("</Folder>\n", kmlFile);
	}

	if (DoImage) {
		err = do_write_image(0, source, kmlFile);
		if (err) return err;
	}

	doFolder = 1;

	for (i = 0; i < RESULT_COUNT; i++) {

		if (source->didKMLCovPts[i]) {

			if (doFolder) {
				fprintf(kmlFile, "<Folder>\n<name>%s</name>\n", COVPTS_MAPFILE_TITLE);
				doFolder = 0;
			}

			switch (i) {
				case RESULT_COVERAGE: {
					title = RESULT_COVERAGE_TITLE;
					break;
				}
				case RESULT_INTERFERE: {
					title = RESULT_INTERFERE_TITLE;
					break;
				}
				case RESULT_NOSERVICE: {
					title = RESULT_NOSERVICE_TITLE;
					break;
				}
			}

			fprintf(kmlFile, "<NetworkLink>\n<name>%s</name><visibility>0</visibility>\n", title);
			fprintf(kmlFile, "<Link><href>%s/%d-%d.kml</href></Link>\n", COVPTS_MAPFILE_NAME, source->sourceKey, i);
			fputs("</NetworkLink>\n", kmlFile);

			source->didKMLCovPts[i] = 0;
		}
	}

	if (!doFolder) {
		fputs("</Folder>\n", kmlFile);
	}

	if (source->didKMLSelfIX) {

		fprintf(kmlFile, "<NetworkLink>\n<name>%s</name><visibility>0</visibility>\n", SELFIX_MAPFILE_TITLE);
		fprintf(kmlFile, "<Link><href>%s/%d.kml</href></Link>\n", SELFIX_MAPFILE_NAME, source->sourceKey);
		fputs("</NetworkLink>\n", kmlFile);

		source->didKMLSelfIX = 0;
	}

	kml_close(kmlFile);

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Render a PostScript file to an image using GhostScript, see do_write_image().  On first call compose most of the
// arguments for the "gs" command, only the image dimensions and file name vary.  The image tiles go in a deeper
// subdirectory below the KML file.  May return minor error.

// Arguments:

//   psFilePath   Full path to the PostScript file to render.
//   imgFileName  Name for the image output file, in scenario subdirectory.
//   width        Image width in points.
//   height       Image height in points.

static int render_image(char *psFilePath, char *imgFileName, int width, int height) {

	static int pathMax = 0, fnameMax = 0, initForStudyKey = 0, initForScenarioKey = 0;
	static char subDir[MAX_STRING];
	static char *path = NULL, *gsArgs[12];

	if ((StudyKey != initForStudyKey) || (ScenarioKey != initForScenarioKey)) {

		char *thePath = get_file_path();
		int len = strlen(thePath) + 40;
		if (len > pathMax) {
			pathMax = len;
			path = (char *)mem_realloc(path, pathMax);
		}

		// GhostScript doesn't like output file paths that contain '%', those have to be escaped by doubling.

		char *i = thePath, *o = path;
		len = 0;
		while (*i) {
			if ('%' == *i) {
				*(o++) = '%';
				len++;
			}
			*(o++) = *(i++);
			len++;
			if (len > (pathMax - 3)) {
				pathMax += 40;
				path = (char *)mem_realloc(path, pathMax);
				o = path + len;
			}
		}
		*o = '\0';

		// Set up argument list on first call only, most of this doesn't vary with study or scenario names.

		if (0 == fnameMax) {

			snprintf(subDir, MAX_STRING, "%s/%s", KML_SUBDIR, IMAGE_MAPFILE_NAME);

			gsArgs[0] = (char *)mem_alloc(20);
			snprintf(gsArgs[0], 20, "%s/gs", LIB_DIRECTORY_NAME);
			gsArgs[1] = "-q";
			gsArgs[2] = "-dNOPAUSE";
			gsArgs[3] = "-dBATCH";
			gsArgs[4] = "-sDEVICE=pngalpha";
			gsArgs[5] = "-dGraphicsAlphaBits=1";
			gsArgs[6] = (char *)mem_alloc(20);
			snprintf(gsArgs[6], 20, "-r%d", IMAGE_RESOLUTION);
			gsArgs[7] = (char *)mem_alloc(40);
			gsArgs[8] = (char *)mem_alloc(40);
			gsArgs[9] = NULL;
			gsArgs[11] = NULL;
		}

		len += strlen(subDir) + 50;
		if (len > fnameMax) {
			fnameMax = len;
			gsArgs[9] = (char *)mem_realloc(gsArgs[9], fnameMax);
		}

		initForStudyKey = StudyKey;
		initForScenarioKey = ScenarioKey;
	}

	// Open the image output file and immediately close it to put it in the files list.

	FILE *outFile = open_out_file_dir(imgFileName, subDir);
	if (!outFile) return 1;
	file_close(outFile);

	// Complete the argument list for the "gs" command, run it.

	snprintf(gsArgs[7], 40, "-dDEVICEWIDTHPOINTS=%d", width);
	snprintf(gsArgs[8], 40, "-dDEVICEHEIGHTPOINTS=%d", height);
	snprintf(gsArgs[9], fnameMax, "-sOutputFile=%s/%s/%s", path, subDir, imgFileName);
	gsArgs[10] = psFilePath;

	return run_process(gsArgs, NULL);
}


//---------------------------------------------------------------------------------------------------------------------
// Write shapes for one source.  This may be writing to global files containing features for all stations, either KML
// or shapefile, or to an individual-station KML file containing features only for one source.  In the latter case
// kmlFile is non-null and all output is directed to that file; however the MAPFILE structures are still used to
// provide attribute definitions.

// Arguments:

//   source            The source for which features are being written, may be an individual DTS transmitter.
//   mapSrcs           Output file for point feature at source coordinates, may be NULL for no output.
//   mapConts          Output file for contour or geography boundary defining service area for desired, may be NULL.
//   desiredSourceKey  If >0, source is an undesired versus this source as desired.
//   cullingDistance   If desiredSourceKey >0, the interference distance limit for this desired-undesired pair.
//   kmlFile           NULL for output directly to MAPFILEs, otherwise KML output sent to this file.
//   kmlVis            Visibility flag for KML features.

// Return is 0 for success, >0 for error.

static int write_source_shapes(SOURCE *source, MAPFILE *mapSrcs, MAPFILE *mapConts, int desiredSourceKey,
		double cullingDistance, FILE *kmlFile, int kmlVis) {

	static char srcKey[LEN_SOURCEKEY + 1], dtsKey[LEN_SOURCEKEY + 1], siteNum[LEN_SITENUMBER + 1],
		facID[LEN_FACILITYID + 1], chan[LEN_CHANNEL + 1], freq[LEN_FREQUENCY + 1], hamsl[LEN_HEIGHT + 1],
		haat[LEN_HEIGHT + 1], erp[LEN_ERP + 1], svclvl[LEN_SVCLEVEL + 1], azorient[LEN_BEARING + 1],
		etilt[LEN_TILT + 1], mtilt[LEN_TILT + 1], mtiltorient[LEN_BEARING + 1], dSrcKey[LEN_SOURCEKEY + 1],
		cullDist[LEN_DISTANCE + 1], *dattrData[DSRC_NATTR], *uattrData[USRC_NATTR], *cattrData[CONT_NATTR];

	static int doInit = 1;

	if (doInit) {

		dattrData[0] = srcKey;
		dattrData[1] = dtsKey;
		dattrData[2] = siteNum;
		dattrData[3] = facID;
		dattrData[5] = chan;
		dattrData[6] = freq;
		dattrData[13] = hamsl;
		dattrData[14] = haat;
		dattrData[15] = erp;
		dattrData[16] = svclvl;
		dattrData[17] = azorient;
		dattrData[18] = etilt;
		dattrData[19] = mtilt;
		dattrData[20] = mtiltorient;

		uattrData[0] = srcKey;
		uattrData[1] = dtsKey;
		uattrData[2] = siteNum;
		uattrData[3] = facID;
		uattrData[5] = chan;
		uattrData[6] = freq;
		uattrData[13] = hamsl;
		uattrData[14] = haat;
		uattrData[15] = erp;
		uattrData[16] = azorient;
		uattrData[17] = etilt;
		uattrData[18] = mtilt;
		uattrData[19] = mtiltorient;
		uattrData[20] = dSrcKey;
		uattrData[21] = cullDist;

		cattrData[0] = srcKey;

		doInit = 0;
	}

	// For a DTS parent or any of it's secondary sources the DTSKEY field is set to the parent source key to group all
	// the records together.  Identification of the individual records can then be made using the SITENUMBER field;
	// that is >0 on transmitter sources, 0 on the parent and authorized facility, the parent will have SOURCEKEY ==
	// DTSKEY, the authorized will not.  Also for a DTS parent there are no height or ERP values.

	char label[MAX_STRING], *svc = "??";

	snprintf(srcKey, (LEN_SOURCEKEY + 1), "%d", source->sourceKey);

	if (mapSrcs) {

		if (source->parentSource) {
			snprintf(dtsKey, (LEN_SOURCEKEY + 1), "%d", source->parentSource->sourceKey);
		} else {
			if (source->isParent) {
				snprintf(dtsKey, (LEN_SOURCEKEY + 1), "%d", source->sourceKey);
			} else {
				dtsKey[0] = '\0';
			}
		}
		snprintf(siteNum, (LEN_SITENUMBER + 1), "%d", source->siteNumber);
		snprintf(facID, (LEN_FACILITYID + 1), "%d", source->facility_id);
		if (RECORD_TYPE_WL == source->recordType) {
			chan[0] = '\0';
		} else {
			snprintf(chan, (LEN_CHANNEL + 1), "%d", source->channel);
		}
		snprintf(freq, (LEN_FREQUENCY + 1), "%.*f", PREC_FREQUENCY, source->frequency);
		if (source->isParent) {
			hamsl[0] = '\0';
			haat[0] = '\0';
			erp[0] = '\0';
			azorient[0] = '\0';
			etilt[0] = '\0';
			mtilt[0] = '\0';
			mtiltorient[0] = '\0';
		} else {
			snprintf(hamsl, (LEN_HEIGHT + 1), "%.*f", PREC_HEIGHT, source->actualHeightAMSL);
			snprintf(haat, (LEN_HEIGHT + 1), "%.*f", PREC_HEIGHT, source->actualOverallHAAT);
			snprintf(erp, (LEN_ERP + 1), "%.*f", PREC_ERP, atof(erpkw_string(source->peakERP)));
			if (source->hasHpat) {
				snprintf(azorient, (LEN_BEARING + 1), "%.*f", PREC_BEARING, source->hpatOrientation);
			} else {
				azorient[0] = '\0';
			}
			if (source->hasVpat) {
				snprintf(etilt, (LEN_TILT + 1), "%.*f", PREC_TILT, source->vpatElectricalTilt);
				snprintf(mtilt, (LEN_TILT + 1), "%.*f", PREC_TILT, source->vpatMechanicalTilt);
				snprintf(mtiltorient, (LEN_BEARING + 1), "%.*f", PREC_BEARING, source->vpatTiltOrientation);
			} else {
				etilt[0] = '\0';
				mtilt[0] = '\0';
				mtiltorient[0] = '\0';
			}
		}

		switch (source->serviceTypeKey) {
			case SERVTYPE_DTV_FULL:
				svc = "DT";
				break;
			case SERVTYPE_NTSC_FULL:
				svc = "TV";
				break;
			case SERVTYPE_DTV_CLASS_A:
				svc = "DC";
				break;
			case SERVTYPE_NTSC_CLASS_A:
				svc = "CA";
				break;
			case SERVTYPE_DTV_LPTV:
				svc = "LD";
				break;
			case SERVTYPE_NTSC_LPTV:
				svc = "TX";
				break;
			case SERVTYPE_WIRELESS:
				svc = "WL";
				break;
			case SERVTYPE_FM_FULL:
				svc = "FM";
				break;
			case SERVTYPE_FM_IBOC:
				svc = "FM";
				break;
			case SERVTYPE_FM_LP:
				svc = "FL";
				break;
			case SERVTYPE_FM_TX:
				svc = "FX";
				break;
		}
	}

	if (desiredSourceKey) {

		if (mapSrcs) {

			uattrData[4] = svc;
			uattrData[7] = source->callSign;
			uattrData[8] = source->city;
			uattrData[9] = source->state;
			uattrData[10] = CountryName[source->countryKey - 1];
			uattrData[11] = source->status;
			uattrData[12] = source->fileNumber;
			snprintf(dSrcKey, (LEN_SOURCEKEY + 1), "%d", desiredSourceKey);
			snprintf(cullDist, (LEN_DISTANCE + 1), "%.*f", PREC_DISTANCE, cullingDistance);

			if (source->parentSource) {
				snprintf(label, MAX_STRING, "%s.%d", source->callSign, source->siteNumber);
			} else {
				lcpystr(label, source->callSign, MAX_STRING);
			}
			if (kmlFile) {
				kml_write_placemark(mapSrcs, kmlFile, source->latitude, source->longitude, NULL, 0, NULL, uattrData,
					label, kmlVis);
			} else {
				if (write_shape(mapSrcs, source->latitude, source->longitude, NULL, 0, NULL, uattrData,
						label, NULL, 0, kmlVis)) {
					return 1;
				}
			}
		}

	} else {

		// The contours file is really a service area boundary file; if a source uses geography to define service area,
		// that geography is output here.  However, if the service area mode is unrestricted the source has a geography
		// that is just a circle at the maximum cell distance.  That feature is pretty much useless on a map, so ignore
		// the geography and output an FCC contour, which probably has to be projected first.  All other code tolerates
		// both a contour and a geography on a source, the geography is always preferred.

		if (mapConts) {

			GEOPOINTS *points;
			if (SERVAREA_NO_BOUNDS == source->serviceAreaMode) {
				if (!source->contour) {
					int err = set_service_contour(source);
					if (err) return err;
				}
				points = render_contour(source->contour, Params.KilometersPerDegree);
			} else {
				points = render_service_area(source);
			}

			if (points) {
				if (kmlFile) {
					if (source->parentSource) {
						snprintf(label, MAX_STRING, "Contour.%d", source->siteNumber);
					} else {
						if (source->isParent) {
							lcpystr(label, "Limit", MAX_STRING);
						} else {
							lcpystr(label, "Contour", MAX_STRING);
						}
					}
					kml_write_placemark(mapConts, kmlFile, 0., 0., points, 1, NULL, cattrData, label, kmlVis);
				} else {
					if (source->parentSource) {
						snprintf(label, MAX_STRING, "%s.%d", source->callSign, source->siteNumber);
					} else {
						lcpystr(label, source->callSign, MAX_STRING);
					}
					if (write_shape(mapConts, 0., 0., points, 1, NULL, cattrData, label, NULL, 0, kmlVis)) {
						return 1;
					}
				}
			}
		}

		if (mapSrcs) {

			dattrData[4] = svc;
			dattrData[7] = source->callSign;
			dattrData[8] = source->city;
			dattrData[9] = source->state;
			dattrData[10] = CountryName[source->countryKey - 1];
			dattrData[11] = source->status;
			dattrData[12] = source->fileNumber;
			if (source->isParent) {
				svclvl[0] = '\0';
			} else {
				snprintf(svclvl, (LEN_SVCLEVEL + 1), "%.*f", PREC_SVCLEVEL, source->serviceLevel);
			}

			if (source->parentSource) {
				snprintf(label, MAX_STRING, "%s.%d", source->callSign, source->siteNumber);
			} else {
				lcpystr(label, source->callSign, MAX_STRING);
			}
			if (kmlFile) {
				kml_write_placemark(mapSrcs, kmlFile, source->latitude, source->longitude, NULL, 0, NULL, dattrData,
					label, kmlVis);
			} else {
				if (write_shape(mapSrcs, source->latitude, source->longitude, NULL, 0, NULL, dattrData,
						label, NULL, 0, kmlVis)) {
					return 1;
				}
			}
		}
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Write studied points from the current grid or points list to the points map file (shapefile only) and/or CSV file.
// Include any point with a desired field, regardless of calculation status.  In global grid mode, this may be called
// multiple times with separate but often overlapping sub-grids.  In that case the points map is used to track which
// points have previously been output to avoid duplication.  Also in global mode the pointKey that would be assigned
// in a pair study output (see pair_study_post.c) is included in the CSV file.  However that may be -1 if the key
// cannot be defined for a point.  Pair study grid range and cell size are restricted and the pointKey algorithm is
// only valid over those restricted conditions.  In local grid mode, this will only be writing output to the points
// CSV file, the map file is only created in global mode when composite coverage is active.  In points study mode the
// cell information is undefined so each record has just the point name and country key, in addition to coordinates.
// In global grid mode this may also write point data to a temporary database table to be combined with composite
// coverage results to create a results table, see do_write_compcov().

// Arguments:

//   mapPointsShp  The shapefile map file, or NULL.
//   pointsFile    The CSV points file, or NULL.

// Return 0 for success, non-zero on error.

static int write_points(MAPFILE *mapPointsShp, FILE *pointsFile) {

	if (!mapPointsShp && !pointsFile && !DoTable) {
		return 0;
	}

	static char attrPtName[LEN_POINTNAME + 1], attrCntKey[LEN_COUNTRYKEY + 1], attrLatIdx[LEN_INDEX + 1],
		attrLonIdx[LEN_INDEX + 1], attrLat[LEN_LATLON + 1], attrLon[LEN_LATLON + 1], attrArea[LEN_AREA + 1],
		attrPop[LEN_POPULATION + 1], attrHouse[LEN_HOUSEHOLDS + 1], attrRhgt[LEN_RXHEIGHT + 1],
		attrElev[LEN_RXHEIGHT + 1], attrLand[LEN_LANDCOVER + 1], attrCltr[LEN_CLUTTER + 1];

	char *attrData[CELL_MAX_ATTR];

	if (mapPointsShp) {

		attrPtName[0] = '\0';
		attrCntKey[0] = '\0';
		attrLatIdx[0] = '\0';
		attrLonIdx[0] = '\0';
		attrLat[0] = '\0';
		attrLon[0] = '\0';
		attrArea[0] = '\0';
		attrPop[0] = '\0';
		attrHouse[0] = '\0';
		attrRhgt[0] = '\0';
		attrElev[0] = '\0';
		attrLand[0] = '\0';
		attrCltr[0] = '\0';

		int i = 0;

		if (STUDY_MODE_POINTS == StudyMode) {
			attrData[i++] = attrPtName;
		}
		attrData[i++] = attrCntKey;
		if (STUDY_MODE_GRID == StudyMode) {
			attrData[i++] = attrLatIdx;
			attrData[i++] = attrLonIdx;
			if (MapOutputFlags[MAP_OUT_CENTER]) {
				attrData[i++] = attrLat;
				attrData[i++] = attrLon;
			}
			attrData[i++] = attrArea;
			attrData[i++] = attrPop;
			attrData[i++] = attrHouse;
		} else {
			attrData[i++] = attrRhgt;
		}
		attrData[i++] = attrElev;
		if (Params.ApplyClutter) {
			attrData[i++] = attrLand;
			attrData[i++] = attrCltr;
		}

		for (; i < CELL_MAX_ATTR; i++) {
			attrData[i] = NULL;
		}
	}

	POINT **pointPtr, *point;
	CEN_POINT *cenPoint;
	FIELD *field;
	long loopIndex, loopCount;
	int countryKey, countryIndex, latIndex, lonIndex, pointKey, maxLatIndex = 75 * 3600, maxLonIndex = 180 * 3600,
		checkPointFlag = 0, latGridIndex, lonGridIndex, pointGridIndex, queryLen = 0, cpQueryLen = 0;
	double latitude, longitude;
	char query[MAX_QUERY], cpQuery[MAX_QUERY], values[MAX_STRING], *infoName = NULL;

	if (STUDY_MODE_POINTS == StudyMode) {
		pointPtr = &Points;
		loopCount = 1L;
	} else {
		pointPtr = Cells;
		loopCount = Grid->count;
	}

	for (loopIndex = 0L; loopIndex < loopCount; loopIndex++, pointPtr++) {

		hb_log();
		if (Terminate) return -1;

		checkPointFlag = DoPoints;

		for (point = *pointPtr; point; point = point->next) {

			for (field = point->fields; field; field = field->next) {
				if (!field->a.isUndesired) {
					break;
				}
			}

			if (field) {

				countryKey = point->countryKey;
				countryIndex = countryKey - 1;
				latIndex = point->cellLatIndex;
				lonIndex = point->cellLonIndex;

				// If needed, check the point map on the first point in the cell.  The output flag in the map applies
				// to all points in the cell, if the flag is already set skip to the next cell.

				if (checkPointFlag) {
					latGridIndex = (latIndex - CompositeGrid->cellBounds.southLatIndex) / CellLatSize;
					lonGridIndex = (lonIndex - CompositeGrid->cellEastLonIndex[latGridIndex]) /
						CompositeGrid->cellLonSizes[latGridIndex];
					pointGridIndex = (latGridIndex * CompositeGrid->lonCount) + lonGridIndex;
					if (CompositePoints[pointGridIndex].didOutput) {
						break;
					}
					CompositePoints[pointGridIndex].didOutput = 1;
					checkPointFlag = 0;
				}

				if (mapPointsShp) {

					snprintf(attrCntKey, (LEN_COUNTRYKEY + 1), "%d", countryKey);

					latitude = point->latitude;
					longitude = point->longitude;

					if (STUDY_MODE_GRID == StudyMode) {

						snprintf(attrLatIdx, (LEN_INDEX + 1), "%d", latIndex);
						snprintf(attrLonIdx, (LEN_INDEX + 1), "%d", lonIndex);

						if (MapOutputFlags[MAP_OUT_CENTER]) {
							latitude = ((double)latIndex / 3600.) + ((double)CellLatSize / 7200.);
							if (GRID_TYPE_GLOBAL == Params.GridType) {
								longitude = ((double)lonIndex / 3600.) +
									((double)Grid->cellLonSizes[(latIndex - Grid->cellBounds.southLatIndex) /
									CellLatSize] / 7200.);
							} else {
								longitude = ((double)lonIndex / 3600.) + ((double)Grid->cellLonSize / 7200.);
							}
							snprintf(attrLat, (LEN_LATLON + 1), "%.*f", PREC_LATLON, point->latitude);
							snprintf(attrLon, (LEN_LATLON + 1), "%.*f", PREC_LATLON, point->longitude);
						}

						snprintf(attrArea, (LEN_AREA + 1), "%.*f", PREC_AREA, point->b.area);
						snprintf(attrPop, (LEN_POPULATION + 1), "%d", point->a.population);
						snprintf(attrHouse, (LEN_HOUSEHOLDS + 1), "%d", point->households);

					} else {

						infoName = PointInfos[point->a.pointIndex].pointName;

						lcpystr(attrPtName, infoName, (LEN_POINTNAME + 1));
						snprintf(attrRhgt, (LEN_RXHEIGHT + 1), "%.*f", PREC_RXHEIGHT, point->b.receiveHeight);
					}

					snprintf(attrElev, (LEN_RXHEIGHT + 1), "%.*f", PREC_RXHEIGHT, point->elevation);

					if (Params.ApplyClutter) {
						snprintf(attrLand, (LEN_LANDCOVER + 1), "%d", point->landCoverType);
						snprintf(attrCltr, (LEN_CLUTTER + 1), "%d", point->clutterType);
					}

					write_shape(mapPointsShp, latitude, longitude, NULL, 0, NULL, attrData, infoName, NULL, 0, 1);
				}

				if (pointsFile) {

					if (STUDY_MODE_POINTS == StudyMode) {

						fprintf(pointsFile, "\"%s\",%d,%.*f,%.*f,%.*f,%.*f", PointInfos[point->a.pointIndex].pointName,
							countryKey, PREC_LATLON, point->latitude, PREC_LATLON, point->longitude, PREC_RXHEIGHT,
							point->b.receiveHeight, PREC_RXHEIGHT, point->elevation);

					} else {

						if (GRID_TYPE_GLOBAL == Params.GridType) {

							if ((CellLatSize < 16) || (latIndex < 0) || (latIndex > maxLatIndex) || (lonIndex < 0) ||
									(lonIndex > maxLonIndex) || (countryKey < 1) || (countryKey > 3)) {
								pointKey = -1;
							} else {
								pointKey = ((((latIndex / CellLatSize) * ((maxLonIndex / CellLatSize) + 1)) +
									(lonIndex / CellLatSize)) * 3) + (countryKey - 1);
							}
							fprintf(pointsFile, "%d,%d,%d,%d,%.*f,%.*f,%.*f,%.*f,%d,%d", latIndex, lonIndex,
								countryKey, pointKey, PREC_LATLON, point->latitude, PREC_LATLON, point->longitude,
								PREC_RXHEIGHT, point->elevation, PREC_AREA, point->b.area, point->a.population,
								point->households);

						} else {

							fprintf(pointsFile, "%d,%d,%d,%d,%.*f,%.*f,%.*f,%.*f,%d,%d", field->sourceKey,
								latIndex, lonIndex, countryKey, PREC_LATLON, point->latitude, PREC_LATLON,
								point->longitude, PREC_RXHEIGHT, point->elevation, PREC_AREA, point->b.area,
								point->a.population, point->households);
						}
					}

					if (Params.ApplyClutter) {
						fprintf(pointsFile, ",%d,%d\n", point->landCoverType, point->clutterType);
					} else {
						fputc('\n', pointsFile);
					}
				}

				if (DoTable) {

					if ((MAX_QUERY - queryLen) < MAX_STRING) {
						lcatstr(query, ";", MAX_QUERY);
						queryLen = 0;
						if (mysql_query(MyConnection, query)) {
							log_db_error("Error writing to temporary points table (1)");
							DoTable = 0;
						}
					}

					if (DoTable) {

						if (0 == queryLen) {
							queryLen = snprintf(query, MAX_QUERY,
								"INSERT INTO %s_%d.temp_point_%d VALUES (%d,%d,%d,%f,%f,%f,%d,%d)", DbName, StudyKey,
								ScenarioKey, countryKey, latIndex, lonIndex, point->latitude, point->longitude,
								point->b.area, point->a.population, point->households);
						} else {
							snprintf(values, MAX_STRING, ",(%d,%d,%d,%f,%f,%f,%d,%d)", countryKey, latIndex, lonIndex,
								point->latitude, point->longitude, point->b.area, point->a.population,
								point->households);
							queryLen = lcatstr(query, values, MAX_QUERY);
						}

						for (cenPoint = point->censusPoints; cenPoint; cenPoint = cenPoint->next) {

							if ((MAX_QUERY - cpQueryLen) < MAX_STRING) {
								lcatstr(cpQuery, ";", MAX_QUERY);
								cpQueryLen = 0;
								if (mysql_query(MyConnection, cpQuery)) {
									log_db_error("Error writing to census points table (1)");
									DoTable = 0;
									break;
								}
							}

							if (0 == cpQueryLen) {
								cpQueryLen = snprintf(cpQuery, MAX_QUERY,
									"INSERT INTO %s_%d.result_%d_cenpt VALUES (%d,%d,%d,%d,'%s')", DbName, StudyKey,
									ScenarioKey, countryKey, latIndex, lonIndex, Params.CenYear[countryIndex],
									cenPoint->blockID);
							} else {
								snprintf(values, MAX_STRING, ",(%d,%d,%d,%d,'%s')", countryKey, latIndex, lonIndex,
									Params.CenYear[countryIndex], cenPoint->blockID);
								cpQueryLen = lcatstr(cpQuery, values, MAX_QUERY);
							}
						}
					}
				}
			}
		}
	}

	if (DoTable && (queryLen > 0)) {
		lcatstr(query, ";", MAX_QUERY);
		if (mysql_query(MyConnection, query)) {
			log_db_error("Error writing to temporary points table (2)");
			DoTable = 0;
		}
	}

	if (DoTable && (cpQueryLen > 0)) {
		lcatstr(cpQuery, ";", MAX_QUERY);
		if (mysql_query(MyConnection, cpQuery)) {
			log_db_error("Error writing to census points table (2)");
			DoTable = 0;
		}
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Project a field at a study point.  Return error code passed through from run_model().

// Arguments:

//   point  The point being studied.
//   field  The field that needs to be computed.

// Return is <0 for a serious error, >0 for a minor error, 0 for no error.  Note that minor errors from the propagation
// model will NOT cause a non-zero return from this function; such errors are stored in the field structure.

int project_field(POINT *point, FIELD *field) {

	static MODEL_DATA data;

	// Get source structure, copy some basic values.

	SOURCE *source = SourceKeyIndex[field->sourceKey];
	if (!source) {
		log_error("Source structure index is corrupted");
		exit(1);
	}

	int countryIndex = source->countryKey - 1;

	double bear = (double)field->bearing, dist = (double)field->distance;

	double receiveHeight = (STUDY_MODE_POINTS == StudyMode) ? (double)point->b.receiveHeight : Params.ReceiveHeight;

	// Populate the propagation model data structure, see model.h and model.c.  For the model statistical parameters,
	// wireless records should not be desireds but use digital TV values just in case.  FM uses analog TV values.

	data.model = PropagationModel;

	data.latitude = source->latitude;
	data.longitude = source->longitude;

	data.bearing = bear;
	data.distance = dist;

	data.terrainDb = Params.TerrPathDb;

	data.transmitHeightAGL = source->heightAGL;
	data.receiveHeightAGL = receiveHeight;

	data.clutterType = (int)point->clutterType;

	data.frequency = source->frequency;

	if (field->a.percentTime) {
		data.percentTime = (double)field->a.percentTime / 100.;
	}
	if (RECORD_TYPE_WL == source->recordType) {
		if (field->a.percentTime) {
			data.percentLocation = Params.WirelessUndesiredLocation;
			data.percentConfidence = Params.WirelessUndesiredConfidence;
		} else {
			data.percentTime = Params.DigitalDesiredTime;
			data.percentLocation = Params.DigitalDesiredLocation;
			data.percentConfidence = Params.DigitalDesiredConfidence;
		}
	} else {
		if (source->dtv) {
			if (field->a.percentTime) {
				data.percentLocation = Params.DigitalUndesiredLocation;
				data.percentConfidence = Params.DigitalUndesiredConfidence;
			} else {
				data.percentTime = Params.DigitalDesiredTime;
				data.percentLocation = Params.DigitalDesiredLocation;
				data.percentConfidence = Params.DigitalDesiredConfidence;
			}
		} else {
			if (field->a.percentTime) {
				data.percentLocation = Params.AnalogUndesiredLocation;
				data.percentConfidence = Params.AnalogUndesiredConfidence;
			} else {
				data.percentTime = Params.AnalogDesiredTime;
				data.percentLocation = Params.AnalogDesiredLocation;
				data.percentConfidence = Params.AnalogDesiredConfidence;
			}
		}
	}

	data.atmosphericRefractivity = Params.AtmosphericRefractivity;
	data.groundPermittivity = Params.GroundPermittivity;
	data.groundConductivity = Params.GroundConductivity;

	if (RECORD_TYPE_WL == source->recordType) {
		data.signalPolarization = Params.WirelessSignalPolarization;
		data.serviceMode = Params.WirelessLRServiceMode;
	} else {
		data.signalPolarization = Params.SignalPolarization;
		data.serviceMode = Params.LRServiceMode;
	}
	data.climateType = Params.LRClimateType;

	data.profilePpk = (RECORD_TYPE_WL == source->recordType) ?
		Params.WirelessTerrPathPpk : Params.TerrPathPpk[countryIndex];
	data.kilometersPerDegree = Params.KilometersPerDegree;

	data.profileCount = (int)((data.distance * data.profilePpk) + 0.5) + 1;

	// If the the profile would be null (the distance is less than half the point spacing) use free-space, otherwise
	// run the model.  The result is a reference field strength for 0 dBk.  The profile retrieved here, if any, is
	// made available in globals for use by other code.

	data.profile = NULL;
	data.fieldStrength = 0.;
	data.errorCode = 0;
	data.errorMessage[0] = '\0';

	Profile = NULL;
	ProfileCount = 0;

	if (data.profileCount < 2) {

		if (data.distance > 0.01) {
			data.fieldStrength = 106.92 - (20. * log10(data.distance));
		} else {
			data.fieldStrength = 146.92;
		}

	} else {

		int err = run_model(&data);
		if (err) {
			if (data.errorMessage[0]) {
				log_error(data.errorMessage);
			} else {
				log_error("Propagation model failed, err=%d", err);
			}
			return err;
		}

		if (data.profile) {
			Profile = data.profile;
			ProfileCount = data.profileCount;
			ProfilePpk = data.profilePpk;
		}
	}

	// Apply ERP and pattern to arrive at final field strength.  Note this is deliberately not using the values in
	// the MODEL_DATA structure, if those were modified by the model code this should not be affected.

	data.fieldStrength += erp_lookup(source, bear) + vpat_lookup(source, source->heightAGL, bear, dist,
		(double)point->elevation, receiveHeight, VPAT_MODE_PROP, NULL, NULL);

	// If needed, adjust field strength for receiver clutter.

	if (Params.ApplyClutter && point->clutterType) {
		data.fieldStrength += Params.ClutterValues[((((point->clutterType - 1) * N_CLUTTER_BANDS) +
			source->clutterBand) * MAX_COUNTRY) + countryIndex];
	}

	// Save result.

	field->fieldStrength = (float)data.fieldStrength;
	field->status = (short)data.errorCode;
	if (STUDY_MODE_GRID == StudyMode) {
		field->b.cached = 0;
	}

	// For isolated points not part of an analysis grid, all done.  This occurs in points mode.

	if ((INVALID_LATLON_INDEX == point->cellLatIndex) && (INVALID_LATLON_INDEX == point->cellLonIndex)) {
		return 0;
	}

	// Set flag indicating there is a new result needing to be cached from this source.  For an undesired, add to the
	// bounds that indicate the grid range over which there are new fields to cache.  If this field was for a DTS
	// source, the updates are made to the parent source structure.

	if (source->parentSource) {
		source = source->parentSource;
	}

	if (field->a.isUndesired) {
		source->ucache = 1;
		int latGridIndex = (point->cellLatIndex - Grid->cellBounds.southLatIndex) / CellLatSize;
		int lonGridIndex = 0;
		if (GRID_TYPE_GLOBAL == Params.GridType) {
			lonGridIndex = point->cellLonIndex;
		} else {
			lonGridIndex = (point->cellLonIndex - Grid->cellBounds.eastLonIndex) / Grid->cellLonSize;
		}
		extend_bounds_index(&(source->ugridIndex), latGridIndex, lonGridIndex);
	} else {
		source->dcache = 1;
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Analyze result for a given source in the current study grid, tally population and area.  This traverses the cell
// grid for the source coverage area, finds desired and undesired fields, applies receive-antenna-pattern corrections,
// and checks D/U limits to tally coverage under various conditions.  Optionally data will be written to a cell-level
// output file.  See analyze_points() for full details.

// Arguments:

//   source    The desired source to analyze.

// Return is <0 for a serious error, >0 for a minor error, 0 for no error.

static int analyze_source(SOURCE *source) {

	int eastLonIndex = Grid->cellBounds.eastLonIndex;
	int eastLonGridIndex = source->gridIndex.eastLonIndex;
	int westLonGridIndex = source->gridIndex.westLonIndex;

	FILE *outFile;

	if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
		fprintf(outFile, "[source]\n%d\n", source->sourceKey);
	}

	if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
		fprintf(outFile, "[source]\n%d\n", source->sourceKey);
	}

	int latGridIndex, lonGridIndex, err;

	for (latGridIndex = source->gridIndex.southLatIndex; latGridIndex < source->gridIndex.northLatIndex;
			latGridIndex++) {

		if (GRID_TYPE_GLOBAL == Params.GridType) {
			eastLonIndex = Grid->cellEastLonIndex[latGridIndex];
			eastLonGridIndex = (source->gridIndex.eastLonIndex - eastLonIndex) / Grid->cellLonSizes[latGridIndex];
			westLonGridIndex = (((source->gridIndex.westLonIndex - 1) - eastLonIndex) /
				Grid->cellLonSizes[latGridIndex]) + 1;
			if (westLonGridIndex > Grid->lonCounts[latGridIndex]) {
				westLonGridIndex = Grid->lonCounts[latGridIndex];
			}
		}

		for (lonGridIndex = eastLonGridIndex; lonGridIndex < westLonGridIndex; lonGridIndex++) {
			err = analyze_points(*(Cells + ((latGridIndex * Grid->lonCount) + lonGridIndex)), source);
			if (err) return err;
		}
	}

	if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
		fputs("[endsource]\n", outFile);
	}

	if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
		fputs("[endsource]\n", outFile);
	}

	return 0;
} 


//---------------------------------------------------------------------------------------------------------------------
// Analyze results for all desireds over the entire study grid.

// Return is <0 for a serious error, >0 for a minor error, 0 for no error.

static int analyze_grid() {

	int eastLonIndex = Grid->cellBounds.eastLonIndex;
	int eastLonGridIndex = GridIndex.eastLonIndex;
	int westLonGridIndex = GridIndex.westLonIndex;

	int latGridIndex, lonGridIndex, err;

	for (latGridIndex = GridIndex.southLatIndex; latGridIndex < GridIndex.northLatIndex; latGridIndex++) {

		if (GRID_TYPE_GLOBAL == Params.GridType) {
			eastLonIndex = Grid->cellEastLonIndex[latGridIndex];
			eastLonGridIndex = (GridIndex.eastLonIndex - eastLonIndex) / Grid->cellLonSizes[latGridIndex];
			westLonGridIndex = (((GridIndex.westLonIndex - 1) - eastLonIndex) / Grid->cellLonSizes[latGridIndex]) + 1;
			if (westLonGridIndex > Grid->lonCounts[latGridIndex]) {
				westLonGridIndex = Grid->lonCounts[latGridIndex];
			}
		}

		for (lonGridIndex = eastLonGridIndex; lonGridIndex < westLonGridIndex; lonGridIndex++) {
			err = analyze_points(*(Cells + ((latGridIndex * Grid->lonCount) + lonGridIndex)), NULL);
			if (err) return err;
		}
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Analyze a list of study points, which may be from a grid cell, or a separate list in points mode.  The analysis is
// is either for a specific source, or for all sources with desired service at any point.  This is the common part of
// the coverage analysis used by analyze_source() and analyze_grid(), also used directly in points mode.  In points
// mode this also writes directly to the summary and detail report and CSV files as those are open.  The format of
// data written to the various cell files varies depending on whether this is for a specific source, or all.

// Arguments:

//   points   Head pointer to list of points.
//   dsource  The desired source, or NULL to analyze all desired sources found in each point.

// Return is <0 for a serious error, >0 for a minor error, 0 for no error.

static int analyze_points(POINT *points, SOURCE *dsource) {

	if (!points) {
		return 0;
	}

	int gridMode = (STUDY_MODE_GRID == StudyMode);
	int pointsMode = (STUDY_MODE_POINTS == StudyMode);

	// Set up attribute fields for map file output.

	static char attrCntKey[LEN_COUNTRYKEY + 1], attrSrcKey[LEN_SOURCEKEY + 1], attrLatIdx[LEN_INDEX + 1],
		attrLonIdx[LEN_INDEX + 1], attrIXBear[LEN_BEARING + 1], attrIXMarg[LEN_ATTRDB + 1],
		attrLat[LEN_LATLON + 1], attrLon[LEN_LATLON + 1], attrArea[LEN_AREA + 1], attrPop[LEN_POPULATION + 1],
		attrHouse[LEN_HOUSEHOLDS + 1], attrPtName[LEN_POINTNAME + 1], attrRhgt[LEN_RXHEIGHT + 1],
		attrElev[LEN_RXHEIGHT + 1], attrLand[LEN_LANDCOVER + 1], attrCltr[LEN_CLUTTER + 1], attrCltrDb[LEN_ATTRDB + 1],
		attrDesSig[LEN_ATTRDB + 1], attrDesMarg[LEN_ATTRDB + 1], attrWorstDUSrcKey[LEN_SOURCEKEY + 1],
		attrWorstDU[LEN_ATTRDB + 1], attrWorstDUMarg[LEN_ATTRDB + 1], attrWLSig[LEN_ATTRDB + 1],
		attrWLDU[LEN_ATTRDB + 1], attrWLDUMarg[LEN_ATTRDB + 1], attrSelfDU[LEN_ATTRDB + 1],
		attrSelfDUMarg[LEN_ATTRDB + 1], attrMarg[LEN_ATTRDB + 1], attrRamp[LEN_ATTRDB + 1],
		desSiteNum[LEN_SITENUMBER + 1], attrUndSigRSS[LEN_ATTRDB + 1], attrUndSiteNum[LEN_SITENUMBER + 1],
		attrUndSig[LEN_ATTRDB + 1], attrDeltaT[LEN_DELTAT + 1];

	char *attrData[COV_MAX_ATTRIBUTE_COUNT], *attrSelfIX[SELFIX_ATTRIBUTE_COUNT];
	int doMapCov = (MapOutputFlags[MAP_OUT_SHAPE] || MapOutputFlags[MAP_OUT_KML]);
	int i, j, resultAttr = 0, causeAttr = -1;

	if (doMapCov) {

		attrCntKey[0] = '\0';
		attrSrcKey[0] = '\0';
		attrLatIdx[0] = '\0';
		attrLonIdx[0] = '\0';
		attrLat[0] = '\0';
		attrLon[0] = '\0';
		attrArea[0] = '\0';
		attrPop[0] = '\0';
		attrHouse[0] = '\0';
		attrPtName[0] = '\0';
		attrRhgt[0] = '\0';
		attrElev[0] = '\0';
		attrLand[0] = '\0';
		attrCltr[0] = '\0';
		attrCltrDb[0] = '\0';
		attrDesSig[0] = '\0';
		attrDesMarg[0] = '\0';
		attrWorstDUSrcKey[0] = '\0';
		attrWorstDU[0] = '\0';
		attrWorstDUMarg[0] = '\0';
		attrWLSig[0] = '\0';
		attrWLDU[0] = '\0';
		attrWLDUMarg[0] = '\0';
		attrSelfDU[0] = '\0';
		attrSelfDUMarg[0] = '\0';
		attrMarg[0] = '\0';
		attrRamp[0] = '\0';

		i = 0;

		if (gridMode) {
			attrData[i++] = attrCntKey;
			attrData[i++] = attrSrcKey;
			attrData[i++] = attrLatIdx;
			attrData[i++] = attrLonIdx;
			resultAttr = i++;
			if (IXMarginSourceKey > 0) {
				attrData[i++] = attrIXBear;
				attrData[i++] = attrIXMarg;
			}
			if (MapOutputFlags[MAP_OUT_COORDS]) {
				attrData[i++] = attrLat;
				attrData[i++] = attrLon;
			}
			if (MapOutputFlags[MAP_OUT_AREAPOP]) {
				attrData[i++] = attrArea;
				attrData[i++] = attrPop;
				attrData[i++] = attrHouse;
			}
		} else {
			attrData[i++] = attrPtName;
			attrData[i++] = attrCntKey;
			attrData[i++] = attrSrcKey;
			resultAttr = i++;
			if (MapOutputFlags[MAP_OUT_HEIGHT]) {
				attrData[i++] = attrRhgt;
			}
		}
		if (MapOutputFlags[MAP_OUT_HEIGHT]) {
			attrData[i++] = attrElev;
		}
		if (MapOutputFlags[MAP_OUT_CLUTTER] && Params.ApplyClutter) {
			attrData[i++] = attrLand;
			attrData[i++] = attrCltr;
			attrData[i++] = attrCltrDb;
		}
		if (MapOutputFlags[MAP_OUT_DESINFO]) {
			attrData[i++] = desSiteNum;
			attrData[i++] = attrDesSig;
			attrData[i++] = attrDesMarg;
		}
		if (MapOutputFlags[MAP_OUT_UNDINFO]) {
			attrData[i++] = attrWorstDUSrcKey;
			attrData[i++] = attrWorstDU;
			attrData[i++] = attrWorstDUMarg;
		}
		if (MapOutputFlags[MAP_OUT_WLINFO] && (STUDY_TYPE_TV_OET74 == StudyType)) {
			attrData[i++] = attrWLSig;
			attrData[i++] = attrWLDU;
			attrData[i++] = attrWLDUMarg;
		}
		if (MapOutputFlags[MAP_OUT_SELFIX] && Params.CheckSelfInterference) {
			attrData[i++] = attrSelfDU;
			attrData[i++] = attrSelfDUMarg;
			attrSelfIX[0] = attrSrcKey;
			attrSelfIX[1] = desSiteNum;
			attrSelfIX[2] = attrDesSig;
			attrSelfIX[3] = attrUndSigRSS;
			attrSelfIX[4] = attrPop;
			attrSelfIX[5] = attrUndSiteNum;
			attrSelfIX[6] = attrUndSig;
			attrSelfIX[7] = attrDeltaT;
		}
		if (MapOutputFlags[MAP_OUT_MARGIN]) {
			attrData[i++] = attrMarg;
			causeAttr = i++;
		}
		if (MapOutputFlags[MAP_OUT_RAMP]) {
			attrData[i++] = attrRamp;
		}

		for (; i < COV_MAX_ATTRIBUTE_COUNT; i++) {
			attrData[i] = NULL;
		}
	}

	// Local variables.

	POINT *point;
	POINT_INFO *pointInfo = NULL;
	RECEIVE_ANT *recvAnt = NULL;
	MPAT *recvPat = NULL;
	FIELD *field, *ufield, *desField, *undField, *selfIXUndFields;
	SOURCE *source, *usource, *dtsSource, *desSource, *desTopSource;
	UNDESIRED *undesireds;
	DES_TOTAL *dtot;
	UND_TOTAL *utot, *utotix, *utotwl;
	char *resultStr, *kmlFolder, fileName[MAX_STRING];
	double desServBase, desServAdjust, desServ, recvOrient, desERP, desSigBase, desSigAdjust, desSig, desMargin,
		undSigBase, undSigAdjust, undSig, du, duReqBase, duReqAdjust, duReq, duMargin, ixBearing, ixMargin, mindist,
		dtime, utime, delta, worstDU, worstDUMarg, wlSig, wlDU, wlDUMarg, selfDU, selfDUMarg, ptlat, ptlon, imageVal,
		compMargin, maxField = 0., chkField;
	int countryIndex, totCountryIndex, startCell, startCellSum, startPoint, startPointSum, startPointPair,
		startPointReport, undesiredIndex, undesiredCount, hasError, hasServiceError, hasService, needsWireless,
		hasWirelessIXError, hasWirelessIX, ixCount, hasIXError, hasIX, done, result, latGridIndex, lonGridIndex,
		pointGridIndex, compResult, updateResult, worstDUSrcKey, hasSelfIX, mapOutSelfIX, imageResult, colorIndex,
		recvFixed;
	FILE *outFile;
	MAPFILE *mapFile;
	CEN_POINT *cenPoint;
	COMPOSITE_POINT *compositePoint;
	GRID *imageGrid;

	// Loop over study points, setup for cell and map file output.

	startCell = 1;
	startCellSum = 1;

	for (point = points; point; point = point->next) {

		hb_log();
		if (Terminate) return -1;

		if (pointsMode) {
			pointInfo = PointInfos + point->a.pointIndex;
		}

		startPoint = 1;
		startPointSum = 1;
		startPointPair = 1;
		startPointReport = 1;

		if (doMapCov) {
			if (gridMode) {
				snprintf(attrCntKey, (LEN_COUNTRYKEY + 1), "%d", point->countryKey);
				snprintf(attrLatIdx, (LEN_INDEX + 1), "%d", point->cellLatIndex);
				snprintf(attrLonIdx, (LEN_INDEX + 1), "%d", point->cellLonIndex);
				if (MapOutputFlags[MAP_OUT_COORDS]) {
					snprintf(attrLat, (LEN_LATLON + 1), "%.*f", PREC_LATLON, point->latitude);
					snprintf(attrLon, (LEN_LATLON + 1), "%.*f", PREC_LATLON, point->longitude);
				}
				if (MapOutputFlags[MAP_OUT_AREAPOP]) {
					snprintf(attrArea, (LEN_AREA + 1), "%.*f", PREC_AREA, point->b.area);
					snprintf(attrPop, (LEN_POPULATION + 1), "%d", point->a.population);
					snprintf(attrHouse, (LEN_HOUSEHOLDS + 1), "%d", point->households);
				}
			} else {
				lcpystr(attrPtName, pointInfo->pointName, (LEN_POINTNAME + 1));
				if (MapOutputFlags[MAP_OUT_HEIGHT]) {
					snprintf(attrRhgt, (LEN_RXHEIGHT + 1), "%.*f", PREC_RXHEIGHT, point->b.receiveHeight);
				}
			}
			if (MapOutputFlags[MAP_OUT_HEIGHT]) {
				snprintf(attrElev, (LEN_RXHEIGHT + 1), "%.*f", PREC_RXHEIGHT, point->elevation);
			}
			if (MapOutputFlags[MAP_OUT_CLUTTER] && Params.ApplyClutter) {
				snprintf(attrLand, (LEN_LANDCOVER + 1), "%d", point->landCoverType);
				snprintf(attrCltr, (LEN_CLUTTER + 1), "%d", point->clutterType);
			}
		}

		// Loop over fields at the point looking for desireds, either the specific one requested, or all.  Uncalculated
		// fields should never occur, if one is found abort.

		for (field = point->fields; field; field = field->next) {

			if (field->a.isUndesired) {
				continue;
			}
			if (dsource && (field->sourceKey != dsource->sourceKey)) {
				continue;
			}

			if (field->status < 0) {
				if (pointsMode) {
					log_error("Un-calculated field: '%s' sourceKey=%d percentTime=%.2f", pointInfo->pointName,
						field->sourceKey, ((double)field->a.percentTime / 100.));
				} else {
					log_error(
						"Un-calculated field: countryKey=%d latIndex=%d lonIndex=%d sourceKey=%d percentTime=%.2f",
						point->countryKey, point->cellLatIndex, point->cellLonIndex, field->sourceKey,
						((double)field->a.percentTime / 100.));
				}
				return 1;
			}

			source = SourceKeyIndex[field->sourceKey];
			if (!source) {
				log_error("Source structure index is corrupted");
				exit(1);
			}
			countryIndex = source->countryKey - 1;

			if (doMapCov) {
				snprintf(attrSrcKey, (LEN_SOURCEKEY + 1), "%d", source->sourceKey);
				if (MapOutputFlags[MAP_OUT_CLUTTER] && Params.ApplyClutter) {
					if (point->clutterType) {
						snprintf(attrCltrDb, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB,
							Params.ClutterValues[((((point->clutterType - 1) * N_CLUTTER_BANDS) +
								source->clutterBand) * MAX_COUNTRY) + countryIndex]);
					} else {
						attrCltrDb[0] = '\0';
					}
				}
				attrData[resultAttr] = "";
				attrIXBear[0] = '\0';
				attrIXMarg[0] = '\0';
				attrDesSig[0] = '\0';
				attrDesMarg[0] = '\0';
				attrWorstDUSrcKey[0] = '\0';
				attrWorstDU[0] = '\0';
				attrWorstDUMarg[0] = '\0';
				attrWLSig[0] = '\0';
				attrWLDU[0] = '\0';
				attrWLDUMarg[0] = '\0';
				attrSelfDU[0] = '\0';
				attrSelfDUMarg[0] = '\0';
				attrMarg[0] = '\0';
				if (causeAttr >= 0) {
					attrData[causeAttr] = "";
				}
				attrRamp[0] = '\0';
				attrUndSigRSS[0] = '\0';
				attrUndSiteNum[0] = '\0';
				attrUndSig[0] = '\0';
				attrDeltaT[0] = '\0';
			}

			worstDUSrcKey = 0;
			worstDU = 999.;
			worstDUMarg = 999.;
			wlSig = 999.;
			wlDU = 999.;
			wlDUMarg = 999.;
			selfDU = 999.;
			selfDUMarg = 999.;

			// Set up for a custom receive antenna and/or a fixed receive antenna orientation.  Currently those
			// conditions can only occur in points mode, but all code beyond this supports any future case that might
			// set the properties here, i.e. using custom antenna conditions for all points in a grid study.

			recvAnt = NULL;
			recvPat = NULL;
			recvFixed = 0;
			recvOrient = 0.;

			if (pointsMode) {
				if (pointInfo->receiveAnt) {
					recvAnt = pointInfo->receiveAnt;
					recvPat = recvAnt->rpat;
				}
				if (pointInfo->receiveOrient >= 0.) {
					recvFixed = 1;
					recvOrient = pointInfo->receiveOrient;
				}
			}

			// For DTS, the desired source for this point is the source providing the strongest signal.  Usually the
			// effect of the receive antenna is constant by site because the antenna will be oriented at the chosen
			// site, but the receive pattern does have to be applied if the antenna orientation is fixed.  Note the
			// desired field found is just a placeholder for the parent, which is not a "real" source in this context;
			// fields for the actual sources follow the parent in sequence.  However the placeholder source is still
			// used for by-source data management e.g. image output.  Along the way this also verifies the fields in
			// the list match the DTS sources and are calculated.  Note in points mode fields may be flagged as being
			// calculated but still have no field result (field value is FIELD_NOT_CALCULATED) due to being beyond the
			// maximum calculation distance.  In case all fields for the DTS are in that state be sure to still set
			// values for a desired source, arbitrarily the first one.

			desTopSource = source;

			if (source->isParent) {

				desSource = NULL;
				desField = NULL;
				maxField = FIELD_NOT_CALCULATED;

				for (dtsSource = source->dtsSources; dtsSource; dtsSource = dtsSource->next) {

					field = field->next;

					if (!field || (field->sourceKey != dtsSource->sourceKey) || (field->status < 0)) {
						log_error("Missing or out-of-order fields for DTS sourceKey=%d (1)", source->sourceKey);
						return 1;
					}

					chkField = field->fieldStrength;
					if (recvFixed && (chkField > FIELD_NOT_CALCULATED)) {
						chkField += recv_az_lookup(dtsSource, recvPat, recvOrient, (double)field->reverseBearing,
							dtsSource->frequency);
					}
					if (!desSource || (chkField > maxField)) {
						desSource = dtsSource;
						desField = field;
						maxField = chkField;
					}
				}

				snprintf(desSiteNum, (LEN_SITENUMBER + 1), "%d", desSource->siteNumber);

			} else {

				desSource = source;
				desField = field;
				desSiteNum[0] = '\0';
			}

			if (!recvFixed) {
				recvOrient = (double)desField->reverseBearing;
			}

			// In grid mode, add to the desired source's contour total.  When totalling, if the point's country is
			// disabled for separate study and reporting (CenYear[?] is 0), add to the source's country instead.  The
			// area total in the contour (or whatever defines the service area) is always tallied, because other code
			// depends on checking that area total against 0. to determine if any study points were actually checked.
			// However if the desired source is in unrestricted mode, that area total will not be reported, nor will
			// population or households totals as they are conceptually not meaningful.  To flag that condition set
			// the population and households totals to -1.  Once that is set it will never be un-set, so if any one
			// point is encountered that is based on an unrestricted service area, the entire area is considered to
			// be unrestricted and not reported.  There can be a mix of unrestricted and restricted service areas in
			// the case of DTS, or later when tallying composite coverage.

			if (gridMode) {

				totCountryIndex = point->countryKey - 1;
				if (!Params.CenYear[totCountryIndex]) {
					totCountryIndex = countryIndex;
				}

				dtot = source->totals + totCountryIndex;

				if (dtot->contourPop >= 0) {
					if (SERVAREA_NO_BOUNDS == desSource->serviceAreaMode) {
						dtot->contourPop = -1;
						dtot->contourHouse = -1;
					} else {
						dtot->contourPop += point->a.population;
						dtot->contourHouse += point->households;
					}
				}
				dtot->contourArea += point->b.area;
			}

			// Set up for the analysis.  The service threshold for DTV may need to be adjusted for the gain of a custom
			// receive antenna vs. the OET-69 generic antennas.  That does not affect D/U, it only affects the service
			// test because the OET-69 antenna gains were planning factors when establishing the service thresholds.
			// However the relative gain of the receive antenna does affect D/U, the desired signal may be attenuated
			// by off-axis (for fixed antenna orientation) or off-frequency response of the receive antenna.

			desServBase = source->serviceLevel;
			desSigBase = (double)desField->fieldStrength;
			desERP = erp_lookup(desSource, (double)desField->bearing);

			desServAdjust = 0.;
			if (recvAnt && source->dtv) {
				switch (source->band) {
					case BAND_VLO1:
					case BAND_VLO2:
						desServAdjust = OET69_RECV_GAIN_VLO - recvAnt->gain;
						break;
					case BAND_VHI:
						desServAdjust = OET69_RECV_GAIN_VHI - recvAnt->gain;
						break;
					case BAND_UHF:
						desServAdjust = OET69_RECV_GAIN_UHF - recvAnt->gain;
						break;
				}
			}

			desSigAdjust = 0.;
			if (recvPat || recvFixed) {
				desSigAdjust = recv_az_lookup(source, recvPat, recvOrient, (double)desField->reverseBearing,
					source->frequency);
			}

			desServ = desServBase + desServAdjust;
			desSig = desSigBase + desSigAdjust;
			desMargin = desSig - desServ;

			// In points mode, write to the main and summary report and CSV files if open.  Normally all points are
			// output here but not-calculated points have no other output, optionally those can be skipped here too.

			if (pointsMode && ((desSigBase > FIELD_NOT_CALCULATED) || !OutputFlags[POINT_NOTCALC])) {
				if (startPointReport) {
					if ((outFile = OutputFile[REPORT_FILE_DETAIL])) {
						fprintf(outFile, "%-20.20s  %s", pointInfo->pointName, latlon_string(point->latitude, 0));
						fprintf(outFile, "  %s  %7.1f m  %6.1f m\n", latlon_string(point->longitude, 1),
							point->elevation, point->b.receiveHeight);
					}
					if ((outFile = OutputFile[REPORT_FILE_SUMMARY])) {
						fprintf(outFile, "  %-20.20s  %s", pointInfo->pointName, latlon_string(point->latitude, 0));
						fprintf(outFile, "  %s\n", latlon_string(point->longitude, 1));
					}
					startPointReport = 0;
				}
				if ((outFile = OutputFile[CSV_FILE_DETAIL])) {
					fprintf(outFile, "\"%s\",%.12f,%.12f,%.1f,%.1f,", pointInfo->pointName, point->latitude,
						point->longitude, point->elevation, point->b.receiveHeight);
				}
				if ((outFile = OutputFile[CSV_FILE_SUMMARY])) {
					fprintf(outFile, "\"%s\",\"%s\",%.12f,%.12f,", ScenarioName, pointInfo->pointName, point->latitude,
						point->longitude);
				}
				if ((outFile = OutputFile[REPORT_FILE_DETAIL])) {
					fprintf(outFile, "  %-35.35s  %5.1f deg  %6.1f km", source_label(source),
						desField->reverseBearing, desField->distance);
					if (desSigBase <= FIELD_NOT_CALCULATED) {
						fputs("  not calculated\n", outFile);
					} else {
						fprintf(outFile, "  %7.2f dBu", desSig);
					}
				}
				if ((outFile = OutputFile[REPORT_FILE_SUMMARY])) {
					fprintf(outFile, "    %-35.35s", source_label(source));
					if (desSigBase <= FIELD_NOT_CALCULATED) {
						fputs("  not calculated\n", outFile);
					} else {
						fprintf(outFile, "  %7.2f dBu", desSig);
					}
				}
				if ((outFile = OutputFile[CSV_FILE_DETAIL])) {
					fprintf(outFile, "%d,%d,\"%s\",\"%s\",\"%s\",%s,%s,%.1f,%.1f", source->facility_id,
						source->channel, source->callSign, source->fileNumber, source->city, source->state,
						CountryName[countryIndex], desField->reverseBearing, desField->distance);
					if (desSigBase <= FIELD_NOT_CALCULATED) {
						fputc('\n', outFile);
					} else {
						fprintf(outFile, ",%.6f", desSig);
					}
				}
				if ((outFile = OutputFile[CSV_FILE_SUMMARY])) {
					fprintf(outFile, "%d,\"%s\",%d,%d,%d", source->facility_id, source->fileNumber, source->countryKey,
						source->serviceTypeKey, source->channel);
					if (desSigBase <= FIELD_NOT_CALCULATED) {
						fputc('\n', outFile);
					} else {
						fprintf(outFile, ",%.6f", desSig);
					}
				}
			}

			// A particular invalid field strength value means the desired station was too far away and no calculations
			// were actually done, in that case skip the rest of the processing for this desired source.

			if (desSigBase <= FIELD_NOT_CALCULATED) {
				continue;
			}

			// Write to the cell-level files.

			if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
				if (dsource) {
					if (startPoint) {
						fprintf(outFile, "P,%d,%d,%d,%.12f,%.12f,%.8f,%d,%d,%d\n", point->cellLatIndex,
							point->cellLonIndex, point->countryKey, point->latitude, point->longitude, point->b.area,
							point->a.population, point->households, point->clutterType);
						if (CELL_DETAIL_CENPT == OutputFlags[CELL_FILE_DETAIL]) {
							for (cenPoint = point->censusPoints; cenPoint; cenPoint = cenPoint->next) {
								fprintf(outFile, "C,%s,%d,%d\n", cenPoint->blockID, cenPoint->population,
									cenPoint->households);
							}
						}
						startPoint = 0;
					}
					fprintf(outFile, "D,%s,%.6f,%.2f,%.2f,%.6f,%d", desSiteNum, desSig, desField->reverseBearing,
						desField->bearing, desERP, desField->status);
				} else {
					if (gridMode && startCell) {
						fprintf(outFile, "[cell]\n%d,%d\n", point->cellLatIndex, point->cellLonIndex);
						startCell = 0;
					}
					if (startPoint) {
						if (pointsMode) {
							fprintf(outFile, "P,\"%s\",%d,%.12f,%.12f,%.1f,%.1f,%d\n",
								pointInfo->pointName, point->countryKey, point->latitude, point->longitude,
								point->elevation, point->b.receiveHeight, point->clutterType);
						} else {
							fprintf(outFile, "P,%d,%.12f,%.12f,%.8f,%d,%d,%d\n", point->countryKey, point->latitude,
								point->longitude, point->b.area, point->a.population, point->households,
								point->clutterType);
							if (CELL_DETAIL_CENPT == OutputFlags[CELL_FILE_DETAIL]) {
								for (cenPoint = point->censusPoints; cenPoint; cenPoint = cenPoint->next) {
									fprintf(outFile, "C,%s,%d,%d\n", cenPoint->blockID, cenPoint->population,
										cenPoint->households);
								}
							}
						}
						startPoint = 0;
					}
					if (pointsMode) {
						fprintf(outFile, "D,%d,%s,%.6f,%.6f,%.6f,%.2f,%.2f,%.6f,%d", source->sourceKey, desSiteNum,
							desSigBase, desSigAdjust, desServAdjust, desField->reverseBearing, desField->bearing,
							desERP, desField->status);
					} else {
						fprintf(outFile, "D,%d,%s,%.6f,%.2f,%.2f,%.6f,%d", source->sourceKey, desSiteNum, desSig,
							desField->reverseBearing, desField->bearing, desERP, desField->status);
					}
				}
			}

			if ((outFile = OutputFile[CELL_FILE_PAIRSTUDY])) {
				if (startPointPair) {
					fprintf(outFile, "[point]\n%d,%d,%d,%.12f,%.12f,%.8f,%d\n", point->cellLatIndex,
						point->cellLonIndex, point->countryKey, point->latitude, point->longitude, point->b.area,
						point->a.population);
					startPointPair = 0;
				}
				fprintf(outFile, "D,%d,%d", source->facility_id, source->channel);
			}

			// If a calculation error occurred on the desired, always add it to the error tally.  What happens after
			// that depends on the error-handling option; it may be considered service with no check for possible
			// interference, it may be considered no service, or the error may just be disregarded.

			hasError = 0;
			hasServiceError = 0;
			hasService = 0;
			done = 0;

			if (desField->status > 0) {

				hasServiceError = 1;
				hasError = 1;

				if (ERRORS_SERVICE == Params.ErrorHandling[countryIndex]) {
					hasService = 1;
					done = 1;
				} else {

					if (ERRORS_NOSERVICE == Params.ErrorHandling[countryIndex]) {
						done = 1;
					}
				}
			}

			// If still needed, compare the desired to the service level to determine service.  Even if no service, if
			// there is a map or cell output file, or any output file in points mode, the undesireds will be traversed
			// for the output, otherwise done.

			if (!done) {

				if (desSig >= desServ) {
					hasService = 1;
				} else {

					if (!doMapCov && !OutputFile[CELL_FILE_DETAIL] && !OutputFile[CELL_FILE_PAIRSTUDY] &&
							!OutputFile[CELL_FILE_CSV] && !DoImage && (!pointsMode ||
							(!OutputFile[REPORT_FILE_DETAIL] && !OutputFile[REPORT_FILE_SUMMARY] &&
							!OutputFile[CSV_FILE_DETAIL] && !OutputFile[CSV_FILE_SUMMARY]))) {
						done = 1;
					}
				}
			}

			if (pointsMode) {
				if ((outFile = OutputFile[REPORT_FILE_DETAIL])) {
					if (desField->b.inServiceArea) {
						fputs("  in service area", outFile);
					} else {
						fputs("  not in service area", outFile);
					}
					if (hasService) {
						fputs(", service", outFile);
					} else {
						fputs(", no service", outFile);
					}
					if (hasServiceError && done) {
						fputs(" (assumed)", outFile);
					}
					fputc('\n', outFile);
				}
				if ((outFile = OutputFile[REPORT_FILE_SUMMARY])) {
					if (desField->b.inServiceArea) {
						fputs("  in service area", outFile);
					} else {
						fputs("  not in service area", outFile);
					}
					if (hasService) {
						fputs(", service", outFile);
					} else {
						fputs(", no service", outFile);
					}
					if (hasServiceError && done) {
						fputs(" (assumed)", outFile);
					}
				}
				if ((outFile = OutputFile[CSV_FILE_DETAIL])) {
					fprintf(outFile, ",%d,%d\n", desField->b.inServiceArea, hasService);
				}
				if ((outFile = OutputFile[CSV_FILE_SUMMARY])) {
					fprintf(outFile, ",%d,%d", desField->b.inServiceArea, hasService);
				}
			}

			if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
				fprintf(outFile, ",%d\n", hasService);
			}

			if ((outFile = OutputFile[CELL_FILE_PAIRSTUDY])) {
				fprintf(outFile, ",%d\n", hasService);
			}

			if ((outFile = OutputFile[CELL_FILE_CSV_D])) {
				if (pointsMode) {
					fprintf(outFile, "\"%s\",%d,%.12f,%.12f,%.1f,%.1f,%d,%d,%s,%.6f,%.6f,%.6f,%.2f,%.2f,%.6f,%d,%d\n",
						pointInfo->pointName, point->countryKey, point->latitude, point->longitude, point->elevation,
						point->b.receiveHeight, point->clutterType, source->sourceKey, desSiteNum, desSigBase,
						desSigAdjust, desServAdjust, desField->reverseBearing, desField->bearing, desERP, hasService,
						desField->status);
				} else {
					fprintf(outFile, "%d-%d-%d,%.12f,%.12f,%.8f,%d,%d,%d,%d,%s,%.6f,%.2f,%.2f,%.6f,%d,%d\n",
						point->cellLatIndex, point->cellLonIndex, point->countryKey, point->latitude, point->longitude,
						point->b.area, point->a.population, point->households, point->clutterType, source->sourceKey,
						desSiteNum, desSig, desField->reverseBearing, desField->bearing, desERP, hasService,
						desField->status);
				}
			}

			// In an OET-74 study, interference from wireless sources to a TV desired has to be evaluated separately
			// because the error-handling logic may be different (errors are always disregarded) and the wireless
			// undesired signal is a composite of all wireless sources.  If the point is TV service and the desired
			// signal is above threshold disregarding errors, wireless interference will be checked below.  However
			// the check loop may also be traversed for file output even with no service.

			needsWireless = 0;
			hasWirelessIXError = 0;
			hasWirelessIX = 0;
			utotwl = NULL;

			if ((STUDY_TYPE_TV_OET74 == StudyType) && (RECORD_TYPE_TV == source->recordType) &&
					((hasService && (desSig >= desServ)) || doMapCov || OutputFile[CELL_FILE_DETAIL] ||
					OutputFile[CELL_FILE_CSV] || DoImage || (pointsMode && (OutputFile[REPORT_FILE_DETAIL] ||
					OutputFile[REPORT_FILE_SUMMARY] || OutputFile[CSV_FILE_DETAIL])))) {
				needsWireless = 1;
			}

			// Find fields for undesired sources.  If the desired source is DTS and self-interference is being analyzed
			// also find the desired source appearing as an undesired, save a pointer to those undesired fields for
			// later analysis.

			undesireds = source->undesireds;
			undesiredCount = source->undesiredCount;
			for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {
				undesireds[undesiredIndex].field = NULL;
			}
			selfIXUndFields = NULL;
			short selfIxUndesiredTime = (short)rint(Params.SelfIxUndesiredTime * 100.);

			for (ufield = point->fields; ufield; ufield = ufield->next) {
				if (ufield->a.isUndesired) {
					for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {
						if ((ufield->sourceKey == undesireds[undesiredIndex].sourceKey) &&
								(ufield->a.percentTime == undesireds[undesiredIndex].percentTime)) {
							if (ufield->status < 0) {
								if (pointsMode) {
									log_error("Un-calculated field: '%s' sourceKey=%d percentTime=%.2f",
										pointInfo->pointName, ufield->sourceKey,
										((double)ufield->a.percentTime / 100.));
								} else {
									log_error(
							"Un-calculated field: countryKey=%d latIndex=%d lonIndex=%d sourceKey=%d percentTime=%.2f",
										point->countryKey, point->cellLatIndex, point->cellLonIndex, ufield->sourceKey,
										((double)ufield->a.percentTime / 100.));
								}
								return 1;
							}
							undesireds[undesiredIndex].field = ufield;
							break;
						}
					}
					if (source->isParent && Params.CheckSelfInterference &&
							(ufield->sourceKey == source->sourceKey) &&
							(ufield->a.percentTime == selfIxUndesiredTime)) {
						selfIXUndFields = ufield->next;
					}
				}
			}

			// Do the wireless interference analysis and/or cell file output.  Totals for wireless interference are
			// kept in globals.  Note the required D/U value is in each wireless undesired structure, but it is the
			// same value in all because all wireless sources have the same frequency and bandwidth and so the same
			// interference relationship to the desired.  Just use the D/U from the first undesired found.  Also there
			// is no distance check here.  If a wireless undesired appears in the list for a study point, it is always
			// used regardless of distance.  See discussion in find_undesired() in study.c.

			if (needsWireless) {

				undSig = 0.;

				for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {

					undField = undesireds[undesiredIndex].field;
					if (!undField) {
						continue;
					}

					usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
					if (!usource) {
						log_error("Source structure index is corrupted");
						exit(1);
					}
					if (RECORD_TYPE_WL != usource->recordType) {
						continue;
					}

					if (0. == undSig) {
						duReqBase = undesireds[undesiredIndex].requiredDU;
						duReqAdjust = 0.;
						if (undesireds[undesiredIndex].adjustDU) {
							duReqAdjust = dtv_codu_adjust(1, desMargin);
							if (Params.WirelessCapDURamp && (duReqAdjust > Params.WirelessDURampCap)) {
								duReqAdjust = Params.WirelessDURampCap;
							}
						}
						duReq = duReqBase + duReqAdjust;
					}

					undSigBase = (double)undField->fieldStrength;
					undSigAdjust = recv_az_lookup(source, recvPat, recvOrient, (double)undField->reverseBearing,
						usource->frequency);

					// The undesired signal is a power sum of the signals from the individual wireless sources, meaning
					// it is an RSS of field strengths.  However the expression coded here is simplified.  Using 10
					// instead of 20 for the dBu conversions means the scalars are power ratios so a straight sum can
					// be done saving the explicit square and square-root operations.  Conversion to specific power
					// units is unnecessary since the result is only being converted back to dBu so the conversion
					// constants would just cancel out of the simplified expression.

					undSig += pow(10., ((undSigBase + undSigAdjust) / 10.));

					if (undField->status > 0) {
						hasWirelessIXError = 1;
						if (hasService) {
							hasError = 1;
						}
					}

					if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
						fprintf(outFile, "U,%d,%.6f,%.2f,%.6f,%d\n", usource->sourceKey, undSigBase,
							undField->reverseBearing, undSigAdjust, undField->status);
					}

					if ((outFile = OutputFile[CELL_FILE_CSV_WL_U])) {
						if (pointsMode) {
							fprintf(outFile,
								"\"%s\",%d,%.12f,%.12f,%.1f,%.1f,%d,%d,%s,%.6f,%.6f,%.6f,%d,%.6f,%.6f,%.2f,%d,%d\n",
								pointInfo->pointName, point->countryKey, point->latitude, point->longitude,
								point->elevation, point->b.receiveHeight, usource->sourceKey, source->sourceKey,
								desSiteNum, desSigBase, desSigAdjust, desServAdjust, hasService, undSigBase,
								undSigAdjust, undField->reverseBearing, desField->status, undField->status);
						} else {
							fprintf(outFile, "%d-%d-%d,%.12f,%.12f,%.8f,%d,%d,%d,%d,%s,%.6f,%d,%.6f,%.6f,%.2f,%d,%d\n",
								point->cellLatIndex, point->cellLonIndex, point->countryKey, point->latitude,
								point->longitude, point->b.area, point->a.population, point->households,
								usource->sourceKey, source->sourceKey, desSiteNum, desSig, hasService, undSigBase,
								undSigAdjust, undField->reverseBearing, desField->status, undField->status);
						}
					}
				}

				if (undSig > 0.) {

					undSig = 10. * log10(undSig);
					du = desSig - undSig;
					if (hasService && (desSig >= desServ) && (du < duReq)) {
						hasWirelessIX = 1;
					}

					wlSig = undSig;
					wlDU = du;
					wlDUMarg = du - duReq;

					if (pointsMode) {
						if ((outFile = OutputFile[REPORT_FILE_DETAIL])) {
							fprintf(outFile, "    Wireless base stations               %7.2f dBu  %7.2f dB  %7.2f dB",
								undSig, du, duReq);
							if (hasWirelessIX) {
								fputs("  interference\n", outFile);
							} else {
								fputs("  no interference\n", outFile);
							}
						}
						if ((outFile = OutputFile[CSV_FILE_DETAIL])) {
							fprintf(outFile, ",,,,,,,,,,,,,,,,,,,wireless,,,,%.6f,%.6f,%.6f,%d\n", undSig, du, duReq,
								hasWirelessIX);
						}
					}

					if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
						fprintf(outFile, "DU,%.6f,%.6f,%.6f,%.6f,%d\n", undSig, du, duReqBase, duReqAdjust,
							hasWirelessIX);
					}

					if ((outFile = OutputFile[CELL_FILE_CSV_WL_U_RSS])) {
						if (pointsMode) {
							fprintf(outFile,
								"\"%s\",%d,%.12f,%.12f,%.1f,%.1f,%d,%s,%.6f,%.6f,%.6f,%d,%.6f,%.6f,%.6f,%.6f,%d,%d\n",
								pointInfo->pointName, point->countryKey, point->latitude, point->longitude,
								point->elevation, point->b.receiveHeight, source->sourceKey, desSiteNum, desSigBase,
								desSigAdjust, desServAdjust, hasService, undSig, du, duReqBase, duReqAdjust,
								hasWirelessIX, desField->status);
						} else {
							fprintf(outFile,
								"%d-%d-%d,%.12f,%.12f,%.8f,%d,%d,%d,%s,%.6f,%d,%.6f,%.6f,%.6f,%.6f,%d,%d\n",
								point->cellLatIndex, point->cellLonIndex, point->countryKey, point->latitude,
								point->longitude, point->b.area, point->a.population, point->households,
								source->sourceKey, desSiteNum, desSig, hasService, undSig, du, duReqBase, duReqAdjust,
								hasWirelessIX, desField->status);
						}
					}

					if (gridMode) {
						utotwl = WirelessUndesiredTotals + totCountryIndex;
						utotwl->report = 1;
					}
				}
			}

			// Check DTS self-interference as needed.  The source appears again in the field list as an undesired to
			// provide the signals for this.  The site with the strongest desired signal is the desired, other sites
			// are potential undesireds.  An arrival time difference is computed between the desired and each other
			// undesired.  If that is within a time window that undesired causes no interference regardless of D/U.
			// Undesireds arriving outside the time window are summed to form a composite undesired signal which must
			// meet a minimum required D/U.  A separate points output and/or CSV file may be created here with details
			// of the points where self-interference is predicted.

			ixCount = 0;
			utotix = NULL;
			ixBearing = 0.;
			ixMargin = 0.;
			hasSelfIX = 0;
			mapOutSelfIX = 0;

			if (source->isParent && Params.CheckSelfInterference && hasService && (desSig >= desServ)) {

				double undSigOne, undSigMax = FIELD_NOT_CALCULATED, deltaMax = 0., undSigOther;
				int undSiteNum = 0;

				undSig = 0.;

				if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
					fputs("S", outFile);
				}

				dtime = ((double)desField->distance / Params.KilometersPerMicrosecond) + desSource->dtsTimeDelay;

				for (dtsSource = source->dtsSources, undField = selfIXUndFields; dtsSource;
						dtsSource = dtsSource->next, undField = undField->next) {

					if (dtsSource == desSource) {
						continue;
					}

					if (!undField || (undField->sourceKey != dtsSource->sourceKey) || (undField->status < 0)) {
						log_error("Missing or out-of-order fields for DTS sourceKey=%d (2)", source->sourceKey);
						return 1;
					}

					utime = ((double)undField->distance / Params.KilometersPerMicrosecond) + dtsSource->dtsTimeDelay;
					delta = utime - dtime;

					if ((delta < Params.PreArrivalTimeLimit) || (delta > Params.PostArrivalTimeLimit)) {

						undSigBase = undField->fieldStrength;
						undSigAdjust = recv_az_lookup(desSource, recvPat, recvOrient, (double)undField->reverseBearing,
							dtsSource->frequency);
						undSigOne = undSigBase + undSigAdjust;
						undSig += pow(10., (undSigOne / 10.));

						// For the self-interference points output file, keep track of the one undesired included in
						// the RSS with maximum signal.  This will be reported in the file if it turns out to be the
						// dominant source of interference.

						if (undSigOne > undSigMax) {
							undSiteNum = dtsSource->siteNumber;
							undSigMax = undSigOne;
							deltaMax = delta;
						}

						if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
							fprintf(outFile, ",%d,%.1f,%.6f,%.2f,%.6f", dtsSource->siteNumber, delta, undSigBase,
								undField->reverseBearing, undSigAdjust);
						}

					} else {

						if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
							fprintf(outFile, ",%d,%.1f,0,0,0", dtsSource->siteNumber, delta);
						}
					}
				}

				if (undSig > 0.) {

					// For the self-interference points output, compute a separate RSS of all undesireds excluding the
					// largest; see below.  There may have been only one signal in the sum.

					undSigOther = undSig - pow(10., (undSigMax / 10.));
					if (undSigOther > 0.) {
						undSigOther = 10. * log10(undSigOther);
					} else {
						undSigOther = FIELD_NOT_CALCULATED;
					}

					undSig = 10. * log10(undSig);

					du = desSig - undSig;

					duReqBase = Params.SelfIxRequiredDU;
					duReqAdjust = dtv_codu_adjust(1, desMargin);
					if (Params.CapDURamp && (duReqAdjust > Params.DURampCap)) {
						duReqAdjust = Params.DURampCap;
					}

					duReq = duReqBase + duReqAdjust;

					selfDU = du;
					selfDUMarg = du - duReq;

					if (du < duReq) {
						hasSelfIX = 1;
						ixCount = 1;
					}

					if (pointsMode) {
						if ((outFile = OutputFile[REPORT_FILE_DETAIL])) {
							fprintf(outFile, "    DTS undesired sites                  %7.2f dBu  %7.2f dB  %7.2f dB",
								undSig, du, duReq);
							if (hasSelfIX) {
								fputs("  interference\n", outFile);
							} else {
								fputs("  no interference\n", outFile);
							}
						}
						if ((outFile = OutputFile[CSV_FILE_DETAIL])) {
							fprintf(outFile, ",,,,,,,,,,,,,,,,,,,DTSundesired,,,,%.6f,%.6f,%.6f,%d\n", undSig, du,
								duReq, hasSelfIX);
						}
					}

					if (gridMode) {
						utot = source->selfIXTotals + totCountryIndex;
						utot->report = 1;
						if (hasSelfIX) {
							utot->ixPop += point->a.population;
							utot->ixHouse += point->households;
							utot->ixArea += point->b.area;
							utotix = utot;
						}
					}

					if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
						fprintf(outFile, ",%d\nDS,%.6f,%.6f,%.6f\n", ixCount, du, duReqBase, duReqAdjust);
					}

					// Write to self-interference CSV, in grid mode points with no population may be excluded.

					if (((outFile = OutputFile[SELFIX_CSV])) &&
							(hasSelfIX || (SELFIX_CSV_ALL == OutputFlags[SELFIX_CSV]) ||
								(SELFIX_CSV_ALL_NO0 == OutputFlags[SELFIX_CSV])) &&
							(pointsMode || (point->a.population > 0) || (SELFIX_CSV_IX == OutputFlags[SELFIX_CSV]) ||
								(SELFIX_CSV_ALL == OutputFlags[SELFIX_CSV]))) {
						if (pointsMode) {
							fprintf(outFile, "%d,\"%s\",%d,%.12f,%.12f,", source->sourceKey, pointInfo->pointName,
								point->countryKey, point->latitude, point->longitude);
						} else {
							fprintf(outFile, "%d,%d,%d,%d,%.12f,%.12f,%.8f,%d,", source->sourceKey,
								point->cellLatIndex, point->cellLonIndex, point->countryKey, point->latitude,
								point->longitude, point->b.area, point->a.population);
						}
						fprintf(outFile, "%.6f,%.6f,%d,%.6f,%.6f", selfDU, selfDUMarg, desSource->siteNumber,
							desSig, undSig);
						if ((desSig - undSigOther) > duReq) {
							fprintf(outFile, ",%d,%.6f,%.1f\n", undSiteNum, undSigMax, deltaMax);
						} else {
							fputc('\n', outFile);
						}
					}

					// If the self-interference points map file is being created, set up attributes for the output
					// (actual output is later, along with other map files).  Desired site number, desired signal,
					// undesired RSS, and population are always included.  If the one undesired with largest signal is
					// dominant, it's site number, individual signal, and arrival time delta are also include.  The
					// test for "dominant" is that the D/U would be met if that one signal is removed from the RSS.

					if (doMapCov && MapOutputFlags[MAP_OUT_SELFIX]) {

						mapOutSelfIX = 1;

						snprintf(attrDesSig, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, desSig);
						snprintf(attrUndSigRSS, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, undSig);
						snprintf(attrPop, (LEN_POPULATION + 1), "%d", point->a.population);

						if ((desSig - undSigOther) > duReq) {
							snprintf(attrUndSiteNum, (LEN_SITENUMBER + 1), "%d", undSiteNum);
							snprintf(attrUndSig, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, undSigMax);
							snprintf(attrDeltaT, (LEN_DELTAT + 1), "%.*f", PREC_DELTAT, deltaMax);
						} else {

							// This needs to be non-empty so these points are in a KML folder.

							lcpystr(attrUndSiteNum, "_", (LEN_SITENUMBER + 1));
						}
					}

				} else {

					if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
						fputs(",0\n", outFile);
					}
				}
			}

			// If still needed, loop over undesireds and check for interference from non-wireless sources.  Also this
			// loop may be traversed to generate map or cell file output even when the point is no-service.

			if (!done) {

				for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {

					undField = undesireds[undesiredIndex].field;
					if (!undField) {
						continue;
					}

					usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
					if (!usource) {
						log_error("Source structure index is corrupted");
						exit(1);
					}

					if (RECORD_TYPE_WL == usource->recordType) {
						continue;
					}

					// Check the interference distance limit if needed, skip fields that are too far away.  When that
					// limit applies it was checked during study point setup to avoid adding undesired fields to the
					// point when possible, however it has to be checked again here in case the undesired affects more
					// than one desired at the point and so more than one limit may apply.  For a DTS, if the alternate
					// distance check is in effect this checks the smallest distance to any one of the transmitters,
					// otherwise it just checks the distance to the DTS reference point.  Also check the status and
					// order of the individual DTS field structures, abort if there is any problem.

					if (undesireds[undesiredIndex].checkIxDistance) {

						if (usource->isParent && Params.CheckIndividualDTSDistance) {

							mindist = 9999.;

							ufield = undField->next;
							for (dtsSource = usource->dtsSources; dtsSource;
									dtsSource = dtsSource->next, ufield = ufield->next) {

								if (!ufield || (ufield->sourceKey != dtsSource->sourceKey) || (ufield->status < 0)) {
									log_error("Missing or out-of-order fields for DTS sourceKey=%d (3)",
										usource->sourceKey);
									return 1;
								}

								if (ufield->distance < mindist) {
									mindist = ufield->distance;
								}
							}

						} else {

							mindist = undField->distance;
						}

						if (mindist > undesireds[undesiredIndex].ixDistance) {
							continue;
						}
					}

					// Set up for the analysis.

					hasIXError = 0;
					hasIX = 0;
					duMargin = 0.;
					done = 0;

					// The undesired field is attenuated by the receive antenna pattern.  For DTS, the undesired signal
					// is the power sum (RSS) of the individual sources, each of course attenuated differently by the
					// receive pattern.  However if the alternate distance check is in effect, each distance must be
					// checked again here and only those individual transmitters within the distance are included in
					// the sum.  Placeholders for those not used are written to the cell file as needed so the format
					// is consistent.  If any DTS source used has an error, depending on error handling that signal
					// may be excluded from the RSS.  But as long as at least one signal passes all the tests there
					// will be an aggregate result that is considered error-free.

					if (usource->isParent) {

						if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
							fprintf(outFile, "U,%d", usource->sourceKey);
						}

						undSig = 0.;
						double undSigAll = 0.;

						ufield = undField->next;
						for (dtsSource = usource->dtsSources; dtsSource;
								dtsSource = dtsSource->next, ufield = ufield->next) {

							if (!ufield || (ufield->sourceKey != dtsSource->sourceKey) || (ufield->status < 0)) {
								log_error("Missing or out-of-order fields for DTS sourceKey=%d (4)",
									usource->sourceKey);
								return 1;
							}

							if (!Params.CheckIndividualDTSDistance ||
									(ufield->distance <= undesireds[undesiredIndex].ixDistance)) {

								undSigBase = ufield->fieldStrength;
								undSigAdjust = recv_az_lookup(source, recvPat, recvOrient,
									(double)ufield->reverseBearing, dtsSource->frequency);
								if ((0 == ufield->status) || (ERRORS_IGNORE == Params.ErrorHandling[countryIndex])) {
									undSig += pow(10., ((undSigBase + undSigAdjust) / 10.));
								}
								undSigAll += pow(10., ((undSigBase + undSigAdjust) / 10.));

								if (ufield->status > 0) {
									if (hasService) {
										hasError = 1;
									}
								}

								if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
									fprintf(outFile, ",%d,%.6f,%.2f,%.6f,%d", dtsSource->siteNumber,
										undSigBase, ufield->reverseBearing, undSigAdjust, ufield->status);
								}

							} else {

								if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
									fprintf(outFile, ",%d,-999,0,0,0", dtsSource->siteNumber);
								}
							}
						}

						if (undSig > 0.) {
							undSig = 10. * log10(undSig);
						} else {
							hasIXError = 1;
							undSig = undSigAll;
						}

					} else {

						undSigBase = undField->fieldStrength;
						undSigAdjust = recv_az_lookup(source, recvPat, recvOrient, (double)undField->reverseBearing,
							usource->frequency);
						undSig = undSigBase + undSigAdjust;

						if (undField->status > 0) {
							hasIXError = 1;
							if (hasService) {
								hasError = 1;
							}
						}

						if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
							fprintf(outFile, "U,%d,%.6f,%.2f,%.6f,%d", usource->sourceKey, undSigBase,
								undField->reverseBearing, undSigAdjust, undField->status);
						}
					}

					// Determine the D/U and the required (minimum) value for no interference.

					du = desSig - undSig;

					// For an FM NCE undesired to a TV desired in an FM vs. TV channel 6 study the required D/U may not
					// have been pre-determined because it may vary with the desired TV signal level.  Curves from FCC
					// rules, 47 CFR 73.525, which vary both by FM channel and by TV desired signal level are used to
					// look up the D/U.  This method will always be used for an analog TV, and may also be used for
					// DTV based on a parameter setting.  Since the curves were developed for analog TV, the DTV signal
					// may fall below the low-signal end of the curves, in which case extrapolation will occur.  Also
					// extrapolation may occur for analog TV if the service level currently set fis less than the curve
					// minimum.  If the lookup is off the high-signal end of the curves there is no extrapolation, the
					// last value is used.  This method may not be used for DTV, there is an alternate method using
					// fixed D/U values, but in that case the D/U was set in find_undesireds().

					if (undesireds[undesiredIndex].computeTV6FMDU) {

						int ci = usource->channel - TV6_FM_CHAN_BASE;
						if (ci < 0) {
							ci = 0;
						}
						if (ci >= TV6_FM_CHAN_COUNT) {
							ci = TV6_FM_CHAN_COUNT - 1;
						}

						int si = (int)(desSig - TV6_FM_CURVE_MIN);
						if (si >= (TV6_FM_CURVE_COUNT - 1)) {
							duReqBase = Params.TV6FMCurves[((TV6_FM_CURVE_COUNT - 1) * TV6_FM_CHAN_COUNT) + ci];
						} else {
							if (si < 0) {
								si = 0;
							}
							double sig0 = TV6_FM_CURVE_MIN + (double)si;
							double sig1 = sig0 + TV6_FM_CURVE_STEP;
							double du0 = Params.TV6FMCurves[(si * TV6_FM_CHAN_COUNT) + ci];
							double du1 = Params.TV6FMCurves[((si + 1) * TV6_FM_CHAN_COUNT) + ci];
							duReqBase = du0 + ((du1 - du0) * ((desSig - sig0) / (sig1 - sig0)));
						}

						duReqAdjust = 0.;

					// For any other case the D/U was set in find_undesireds() from interference rules or lookup
					// tables.  However the D/U may be adjusted for a TV/DTV undesired into a DTV desired based on the
					// desired signal level, see dtv_codu_adjust().

					} else {

						duReqBase = undesireds[undesiredIndex].requiredDU;
						duReqAdjust = 0.;

						if (undesireds[undesiredIndex].adjustDU) {
							duReqAdjust = dtv_codu_adjust(usource->dtv, desMargin);
							if (Params.CapDURamp && (duReqAdjust > Params.DURampCap)) {
								duReqAdjust = Params.DURampCap;
							}
						}
					}

					duReq = duReqBase + duReqAdjust;

					// Error handling logic similar to desired; an error here may mean there is no interference, there
					// is interference, or the error may be disregarded and the D/U tested as usual.

					if (hasIXError) {

						if (ERRORS_SERVICE == Params.ErrorHandling[countryIndex]) {
							done = 1;
						} else {

							if (ERRORS_NOSERVICE == Params.ErrorHandling[countryIndex]) {
								hasIX = hasService;
								done = 1;
							}
						}
					}

					// If needed, compare the D/U to the adjusted requirement to determine if interference is present.
					// If the rule has signal threshold conditions the desired and undesired must be greater than the
					// respective thresholds for the rule to apply, see find_undesired() in source.c.

					if (!done) {

						if (((0. == undesireds[undesiredIndex].duThresholdD) ||
									(desSig >= undesireds[undesiredIndex].duThresholdD)) &&
								((0. == undesireds[undesiredIndex].duThresholdU) ||
									(undSig >= undesireds[undesiredIndex].duThresholdU))) {

							duMargin = du - duReq;

							if (duMargin < worstDUMarg) {
								worstDUSrcKey = usource->sourceKey;
								worstDU = du;
								worstDUMarg = duMargin;
							}

							if (hasService && (du < duReq)) {
								hasIX = 1;
							}

							if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
								fprintf(outFile, ",%d\nDU,%.6f,%.6f,%.6f\n", hasIX, du, duReqBase, duReqAdjust);
							}

						} else {

							if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
								fprintf(outFile, ",%d\n", hasIX);
							}
						}

					} else {

						if ((outFile = OutputFile[CELL_FILE_DETAIL])) {
							fprintf(outFile, ",%d\n", hasIX);
						}
					}

					if (pointsMode) {
						if ((outFile = OutputFile[REPORT_FILE_DETAIL])) {
							fprintf(outFile, "    %-35.35s  %7.2f dBu  %7.2f dB  %7.2f dB", source_label(usource),
								undSig, du, duReq);
							if (hasIX) {
								fputs("  interference", outFile);
							} else {
								fputs("  no interference", outFile);
							}
							if (hasIXError && done) {
								fputs(" (assumed)", outFile);
							}
							fputc('\n', outFile);
						}
						if ((outFile = OutputFile[CSV_FILE_DETAIL])) {
							fprintf(outFile, ",,,,,,,,,,,,,,,,,%d,%d,\"%s\",\"%s\",\"%s\",\"%s\",%.6f,%.6f,%.6f,%d\n",
								usource->facility_id, usource->channel, usource->callSign, usource->fileNumber,
								usource->city, usource->state, undSig, du, duReq, hasIX);
						}
					}

					if (((outFile = OutputFile[CELL_FILE_PAIRSTUDY])) && hasIX) {
						fprintf(outFile, "U,%d,%d\n", usource->facility_id, usource->channel);
					}

					if ((outFile = OutputFile[CELL_FILE_CSV_U])) {

						if (usource->isParent) {

							ufield = undField->next;
							for (dtsSource = usource->dtsSources; dtsSource;
									dtsSource = dtsSource->next, ufield = ufield->next) {

								if (!Params.CheckIndividualDTSDistance ||
										(ufield->distance <= undesireds[undesiredIndex].ixDistance)) {

									undSigBase = ufield->fieldStrength;
									undSigAdjust = recv_az_lookup(source, recvPat, recvOrient,
										(double)ufield->reverseBearing, dtsSource->frequency);

									if (pointsMode) {
										fprintf(outFile,
			"\"%s\",%d,%.12f,%.12f,%.1f,%.1f,%d,%d,%d,%s,%.6f,%.6f,%.6f,%d,%.6f,%.6f,%.2f,%.6f,%.6f,%.6f,%d,%d,%d\n",
											pointInfo->pointName, point->countryKey, point->latitude, point->longitude,
											point->elevation, point->b.receiveHeight, usource->sourceKey,
											dtsSource->siteNumber, source->sourceKey, desSiteNum, desSigBase,
											desSigAdjust, desServAdjust, hasService, undSigBase, undSigAdjust,
											ufield->reverseBearing, du, duReqBase, duReqAdjust, hasIX,
											desField->status, ufield->status);
									} else {
										fprintf(outFile,
						"%d-%d-%d,%.12f,%.12f,%.8f,%d,%d,%d,%d,%d,%s,%.6f,%d,%.6f,%.6f,%.2f,%.6f,%.6f,%.6f,%d,%d,%d\n",
											point->cellLatIndex, point->cellLonIndex, point->countryKey,
											point->latitude, point->longitude, point->b.area, point->a.population,
											point->households, usource->sourceKey, dtsSource->siteNumber,
											source->sourceKey, desSiteNum, desSig, hasService, undSigBase,
											undSigAdjust, ufield->reverseBearing, du, duReqBase, duReqAdjust, hasIX,
											desField->status, ufield->status);
									}
								}
							}

						} else {

							if (pointsMode) {
								fprintf(outFile,
				"\"%s\",%d,%.12f,%.12f,%.1f,%.1f,%d,,%d,%s,%.6f,%.6f,%.6f,%d,%.6f,%.6f,%.2f,%.6f,%.6f,%.6f,%d,%d,%d\n",
									pointInfo->pointName, point->countryKey, point->latitude, point->longitude,
									point->elevation, point->b.receiveHeight, usource->sourceKey, source->sourceKey,
									desSiteNum, desSigBase, desSigAdjust, desServAdjust, hasService, undSigBase,
									undSigAdjust, undField->reverseBearing, du, duReqBase, duReqAdjust, hasIX,
									desField->status, undField->status);
							} else {
								fprintf(outFile,
						"%d-%d-%d,%.12f,%.12f,%.8f,%d,%d,%d,,%d,%s,%.6f,%d,%.6f,%.6f,%.2f,%.6f,%.6f,%.6f,%d,%d,%d\n",
									point->cellLatIndex, point->cellLonIndex, point->countryKey, point->latitude,
									point->longitude, point->b.area, point->a.population, point->households,
									usource->sourceKey, source->sourceKey, desSiteNum, desSig, hasService, undSigBase,
									undSigAdjust, undField->reverseBearing, du, duReqBase, duReqAdjust, hasIX,
									desField->status, undField->status);
							}
						}
					}

					// Add to the interference totals for this undesired.  Also set up for adding to the unique IX
					// total for this undesired, and for the IX margin processing features.  Margin information is
					// placed in the coverage map file and accumulated for the margin CSV file only if unique IX is
					// found from a specific undesired.

					if (gridMode) {

						utot = undesireds[undesiredIndex].totals + totCountryIndex;
						utot->report = 1;

						if (hasService && hasIXError) {
							utot->errorPop += point->a.population;
							utot->errorHouse += point->households;
							utot->errorArea += point->b.area;
						}

						if (hasIX) {

							utot->ixPop += point->a.population;
							utot->ixHouse += point->households;
							utot->ixArea += point->b.area;

							if (1 == ++ixCount) {
								utotix = utot;
								if (IXMarginSourceKey == usource->sourceKey) {
									ixBearing = undField->bearing;
									ixMargin = duMargin;
								}
							} else {
								utotix = NULL;
								ixBearing = 0.;
								ixMargin = 0.;
							}
						}
					}

					if (pointsMode && hasIX) {
						++ixCount;
					}
				}
			}

			// Tally the desired service and interference totals, and also undesired wireless interference totals.  If
			// non-wireless interference was from just one undesired, also add to that undesired's unique interference
			// total.  Note wireless interference is not considered with respect to the unique interference totals for
			// non-wireless undesireds; wireless interference does not mask TV interference.  However TV interference
			// does mask wireless interference so is considered for the wireless unique interference total.

			if (gridMode) {

				if (hasServiceError) {
					dtot->errorPop += point->a.population;
					dtot->errorHouse += point->households;
					dtot->errorArea += point->b.area;
				}

				if (hasService) {

					dtot->servicePop += point->a.population;
					dtot->serviceHouse += point->households;
					dtot->serviceArea += point->b.area;

					if (hasWirelessIXError && utotwl) {
						utotwl->errorPop += point->a.population;
						utotwl->errorHouse += point->households;
						utotwl->errorArea += point->b.area;
					}

					if (ixCount || hasWirelessIX) {

						if (utotix) {
							utotix->uniqueIxPop += point->a.population;
							utotix->uniqueIxHouse += point->households;
							utotix->uniqueIxArea += point->b.area;
						}

						if (hasWirelessIX && utotwl) {

							utotwl->ixPop += point->a.population;
							utotwl->ixHouse += point->households;
							utotwl->ixArea += point->b.area;

							if (!ixCount) {
								utotwl->uniqueIxPop += point->a.population;
								utotwl->uniqueIxHouse += point->households;
								utotwl->uniqueIxArea += point->b.area;
							}
						}

					} else {

						dtot->ixFreePop += point->a.population;
						dtot->ixFreeHouse += point->households;
						dtot->ixFreeArea += point->b.area;
					}
				}
			}

			// Update the margin envelope for CSV file output as needed.

			if (OutputFlags[IXCHK_MARG_CSV] && (ixMargin < 0.) && ((point->a.population > 0) ||
					(IXCHK_MARG_CSV_AGG == OutputFlags[IXCHK_MARG_CSV]) ||
					(IXCHK_MARG_CSV_ALL == OutputFlags[IXCHK_MARG_CSV]))) {
				if ((IXCHK_MARG_CSV_AGG == OutputFlags[IXCHK_MARG_CSV]) ||
						(IXCHK_MARG_CSV_AGGNO0 == OutputFlags[IXCHK_MARG_CSV])) {
					i = (int)ixBearing;
					if (ixMargin < IXMarginDb[i]) {
						IXMarginDb[i] = ixMargin;
					}
					IXMarginPop[i] += point->a.population;
				} else {
					i = IXMarginCount++;
					if (i >= IXMarginMaxCount) {
						IXMarginMaxCount = i + 1000;
						IXMarginAz = (double *)mem_realloc(IXMarginAz, (IXMarginMaxCount * sizeof(double)));
						IXMarginDb = (double *)mem_realloc(IXMarginDb, (IXMarginMaxCount * sizeof(double)));
						IXMarginPop = (int *)mem_realloc(IXMarginPop, (IXMarginMaxCount * sizeof(int)));
					}
					IXMarginAz[i] = ixBearing;
					IXMarginDb[i] = ixMargin;
					IXMarginPop[i] = point->a.population;
				}
			}

			// Determine the final result status of this desired at the point.

			result = RESULT_NOSERVICE;
			if (hasService) {
				if (ixCount || hasWirelessIX) {
					result = RESULT_INTERFERE;
				} else {
					result = RESULT_COVERAGE;
				}
			}

			// If compositing coverage, get the composite point structure and create or update as needed.

			if (DoComposite) {

				latGridIndex = (point->cellLatIndex - CompositeGrid->cellBounds.southLatIndex) / CellLatSize;
				lonGridIndex = (point->cellLonIndex - CompositeGrid->cellEastLonIndex[latGridIndex]) /
					CompositeGrid->cellLonSizes[latGridIndex];

				compositePoint = get_composite_point(point->countryKey, latGridIndex, lonGridIndex, 1);
				if (!compositePoint) {
					return 1;
				}

				// Flag for the option to exclude no-population points from output later, see do_write_compcov().

				if (point->a.population > 0) {
					compositePoint->hasPop = 1;
				}

				// Update the service count.  Make sure it doesn't overflow.

				if ((RESULT_COVERAGE == result) && (compositePoint->serviceCount < 255)) {
					compositePoint->serviceCount++;
				}

				compResult = compositePoint->result;
				compMargin = (double)compositePoint->margin / 100.;

				updateResult = 0;

				if (result != compResult) {

					dtot = CompositeTotals + totCountryIndex;

					switch (compResult) {

						case 0: {
							if (dtot->contourPop >= 0) {
								if (SERVAREA_NO_BOUNDS == desSource->serviceAreaMode) {
									dtot->contourPop = -1;
									dtot->contourHouse = -1;
								} else {
									dtot->contourPop += point->a.population;
									dtot->contourHouse += point->households;
								}
							}
							dtot->contourArea += point->b.area;
							if (RESULT_NOSERVICE != result) {
								dtot->servicePop += point->a.population;
								dtot->serviceHouse += point->households;
								dtot->serviceArea += point->b.area;
								if (RESULT_COVERAGE == result) {
									dtot->ixFreePop += point->a.population;
									dtot->ixFreeHouse += point->households;
									dtot->ixFreeArea += point->b.area;
								}
							}
							compMargin = FIELD_NOT_CALCULATED;
							updateResult = 1;
							break;
						}

						case RESULT_NOSERVICE: {
							dtot->servicePop += point->a.population;
							dtot->serviceHouse += point->households;
							dtot->serviceArea += point->b.area;
							if (RESULT_NOSERVICE != result) {
								if (RESULT_COVERAGE == result) {
									dtot->ixFreePop += point->a.population;
									dtot->ixFreeHouse += point->households;
									dtot->ixFreeArea += point->b.area;
								}
								compMargin = FIELD_NOT_CALCULATED;
							}
							updateResult = 1;
							break;
						}

						case RESULT_INTERFERE: {
							if (RESULT_NOSERVICE != result) {
								if (RESULT_COVERAGE == result) {
									dtot->ixFreePop += point->a.population;
									dtot->ixFreeHouse += point->households;
									dtot->ixFreeArea += point->b.area;
									compMargin = FIELD_NOT_CALCULATED;
								}
								updateResult = 1;
							}
							break;
						}

						case RESULT_COVERAGE: {
							if (RESULT_COVERAGE == result) {
								updateResult = 1;
							}
							break;
						}
					}

					if (updateResult && (desMargin > compMargin)) {
						i = (int)rint(desMargin * 100.);
						if (i < -32768) {
							i = -32768;
						}
						if (i > 32767) {
							i = 32767;
						}
						compositePoint->result = result;
						compositePoint->margin = (short)i;
						compositePoint->sourceKey = desSource->sourceKey;
					}
				}
			}

			// Output to cell file in summary format and to coverage map file as needed.

			if (hasError) {
				switch (result) {
					case RESULT_COVERAGE:
						resultStr = "11";
						break;
					case RESULT_INTERFERE:
						resultStr = "12";
						break;
					case RESULT_NOSERVICE:
						resultStr = "13";
						break;
					default:
						resultStr = "0";
						break;
				}
			} else {
				switch (result) {
					case RESULT_COVERAGE:
						resultStr = "1";
						break;
					case RESULT_INTERFERE:
						resultStr = "2";
						break;
					case RESULT_NOSERVICE:
						resultStr = "3";
						break;
					default:
						resultStr = "0";
						break;
				}
			}

			if (pointsMode) {
				if ((outFile = OutputFile[REPORT_FILE_SUMMARY])) {
					if (hasService) {
						if (ixCount || hasWirelessIX) {
							fputs(", interference", outFile);
						} else {
							fputs(", no interference", outFile);
						}
					}
					fputc('\n', outFile);
				}
				if ((outFile = OutputFile[CSV_FILE_SUMMARY])) {
					if (hasService) {
						if (ixCount || hasWirelessIX) {
							fputs(",1", outFile);
						} else {
							fputs(",0", outFile);
						}
					}
					fputc('\n', outFile);
				}
			}

			if ((outFile = OutputFile[CELL_FILE_SUMMARY])) {
				if (dsource) {
					fprintf(outFile, "R,%d,%d,%d,%.8f,%d,%d,%d,%s\n", point->countryKey, point->cellLatIndex,
						point->cellLonIndex, point->b.area, point->a.population, point->households, point->clutterType,
						resultStr);
				} else {
					if (gridMode && startCellSum) {
						fprintf(outFile, "[cell]\n%d,%d\n", point->cellLatIndex, point->cellLonIndex);
						startCellSum = 0;
					}
					if (startPointSum) {
						if (pointsMode) {
							fprintf(outFile, "P,\"%s\",%d,%.1f,%.1f,%d\n", pointInfo->pointName, point->countryKey,
								point->elevation, point->b.receiveHeight, point->clutterType);
						} else {
							fprintf(outFile, "P,%d,%.8f,%d,%d,%d\n", point->countryKey, point->b.area,
								point->a.population, point->households, point->clutterType);
						}
						startPointSum = 0;
					}
					fprintf(outFile, "R,%d,%s,%s\n", source->sourceKey, desSiteNum, resultStr);
				}
			}

			// Output to map files and image may be skipped in grid mode if this is a no-service or no-population
			// point.  Always output all points in points mode.

			if (pointsMode || (((RESULT_NOSERVICE != result) || !MapOutputFlags[MAP_OUT_NOSERV]) &&
					((point->a.population > 0) || !MapOutputFlags[MAP_OUT_NOPOP]))) {

				if (doMapCov) {

					attrData[resultAttr] = resultStr;

					if ((IXMarginSourceKey > 0) && (ixMargin < 0.)) {
						snprintf(attrIXBear, (LEN_BEARING + 1), "%.*f", PREC_BEARING, ixBearing);
						snprintf(attrIXMarg, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, ixMargin);
					}
					if (MapOutputFlags[MAP_OUT_DESINFO]) {
						snprintf(attrDesSig, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, desSig);
						snprintf(attrDesMarg, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, desMargin);
					}
					if ((MapOutputFlags[MAP_OUT_UNDINFO]) && (worstDUSrcKey > 0)) {
						snprintf(attrWorstDUSrcKey, (LEN_SOURCEKEY + 1), "%d", worstDUSrcKey);
						snprintf(attrWorstDU, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, worstDU);
						snprintf(attrWorstDUMarg, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, worstDUMarg);
					}
					if ((MapOutputFlags[MAP_OUT_WLINFO]) && (wlSig < 999.)) {
						snprintf(attrWLSig, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, wlSig);
						snprintf(attrWLDU, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, wlDU);
						snprintf(attrWLDUMarg, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, wlDUMarg);
					}
					if (mapOutSelfIX) {
						snprintf(attrSelfDU, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, selfDU);
						snprintf(attrSelfDUMarg, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, selfDUMarg);
					}
					if (MapOutputFlags[MAP_OUT_MARGIN]) {
						double marg = desMargin;
						attrData[causeAttr] = "D";
						if (worstDUMarg < marg) {
							marg = worstDUMarg;
							attrData[causeAttr] = "U";
						}
						if (wlDUMarg < marg) {
							marg = wlDUMarg;
							attrData[causeAttr] = "W";
						}
						if (selfDUMarg < marg) {
							marg = selfDUMarg;
							attrData[causeAttr] = "S";
						}
						snprintf(attrMarg, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, marg);
					}
					if ((MapOutputFlags[MAP_OUT_RAMP]) && source->dtv) {
						snprintf(attrRamp, (LEN_ATTRDB + 1), "%.*f", PREC_ATTRDB, dtv_codu_adjust(1, desMargin));
					}

					if (gridMode && MapOutputFlags[MAP_OUT_CENTER]) {
						ptlat = ((double)point->cellLatIndex / 3600.) + ((double)CellLatSize / 7200.);
						if (GRID_TYPE_GLOBAL == Params.GridType) {
							ptlon = ((double)point->cellLonIndex / 3600.) +
								((double)Grid->cellLonSizes[(point->cellLatIndex - Grid->cellBounds.southLatIndex) /
								CellLatSize] / 7200.);
						} else {
							ptlon = ((double)point->cellLonIndex / 3600.) + ((double)Grid->cellLonSize / 7200.);
						}
					} else {
						ptlat = point->latitude;
						ptlon = point->longitude;
					}

					if ((mapFile = MapOutputFile[MAP_OUT_SHAPE_COVPTS])) {
						write_shape(mapFile, ptlat, ptlon, NULL, 0, NULL, attrData, NULL, NULL, 0, 1);
					}

					// In grid mode or individual-layer map format regardless of study mode, KML points output goes to
					// separate files for each source, all in a common subdirectory.  The individual files are named
					// with the source key and result code so each result can appear as a layer (folder or link) in a
					// referencing file.  In grid mode the folder structure is used in the files themselves to group
					// points on a geographic grid with <Region> elements controlling when each tile becomes visible.
					// That folder grouping is done automatically by write_shape().  Geographic grouping is not done in
					// points mode, assuming it is more useful in that case to be able to see all points regardless of
					// zoom level.  In global-layer points mode, all points are going to a single global file and the
					// folder structure is used in that file to group by result type.

					if (MapOutputFlags[MAP_OUT_KML_CPT]) {

						if (gridMode || (MAP_OUT_KML_IND == MapOutputFlags[MAP_OUT_KML])) {

							mapFile = desTopSource->kmlCovPts[result];
							if (!mapFile) {
								snprintf(fileName, MAX_STRING, "%d-%d", desTopSource->sourceKey, result);
								mapFile = open_mapfile(MAP_FILE_KML, fileName, CovPtsKMLSubDir, SHP_TYPE_POINT,
									CovPtsAttributeCount, CovPtsAttributes, desTopSource->callSign);
								if (!mapFile) return 1;
								desTopSource->kmlCovPts[result] = mapFile;
								desTopSource->didKMLCovPts[result] = 1;
							}

							write_shape(mapFile, ptlat, ptlon, NULL, 0, NULL, attrData, NULL, NULL, gridMode, 1);
							DidWriteCovPtsKML = 1;

						} else {

							mapFile = MapOutputFile[MAP_OUT_KML_COVPTS];
							if (mapFile) {

								switch (result) {
									case RESULT_COVERAGE:
										kmlFolder = RESULT_COVERAGE_TITLE;
										break;
									case RESULT_INTERFERE:
										kmlFolder = RESULT_INTERFERE_TITLE;
										break;
									case RESULT_NOSERVICE:
										kmlFolder = RESULT_NOSERVICE_TITLE;
										break;
									default:
										kmlFolder = NULL;
										break;
								}

								write_shape(mapFile, ptlat, ptlon, NULL, 0, NULL, attrData, NULL, kmlFolder, 0, 1);
								DidWriteCovPtsKML = 1;
							}
						}
					}

					if (mapOutSelfIX) {

						if ((mapFile = MapOutputFile[MAP_OUT_SHAPE_SELFIX])) {
							write_shape(mapFile, ptlat, ptlon, NULL, 0, NULL, attrSelfIX, NULL, NULL, 0, 1);
						}

						// DTS self-interference KML data also goes to a per-source file in grid mode or individual-
						// layer mode and a single file in global-layer points mode, but in either case folders are
						// used to group points by the site number dominating the interference (if any, see above),
						// geographic grouping (tiling) is not done in any case.

						if (gridMode || (MAP_OUT_KML_IND == MapOutputFlags[MAP_OUT_KML])) {
							mapFile = desTopSource->kmlSelfIX;
							if (!mapFile) {
								snprintf(fileName, MAX_STRING, "%d", desTopSource->sourceKey);
								mapFile = open_mapfile(MAP_FILE_KML, fileName, SelfIXKMLSubDir, SHP_TYPE_POINT,
									SELFIX_ATTRIBUTE_COUNT, SelfIXFileAttributes, source->callSign);
								if (!mapFile) return 1;
								desTopSource->kmlSelfIX = mapFile;
								desTopSource->didKMLSelfIX = 1;
							}
						} else {
							mapFile = MapOutputFile[MAP_OUT_KML_SELFIX];
						}

						if (mapFile) {
							write_shape(mapFile, ptlat, ptlon, NULL, 0, NULL, attrSelfIX, NULL, attrSelfIX[5], 0, 1);
							DidWriteSelfIXKML = 1;
						}
					}
				}

				// If creating an image output, apply logic similar to composite coverage above in case of multiple
				// desired signals.  A coverage result is preferred over anything else, then an interference result,
				// then no-service (but note that this will not even be reached if no-service results are being
				// excluded).  Within the same result type, keep the largest color index (best result).  The image
				// may be accumulated in the global composite points array covering the entire study grid region, or
				// in by-station map output format it may be a separate image points array in each desired source.

				if (DoImage) {

					if (DoCompositeImage) {
						imageGrid = CompositeGrid;
					} else {
						imageGrid = desTopSource->grid;
					}

					latGridIndex = (point->cellLatIndex - imageGrid->cellBounds.southLatIndex) / CellLatSize;
					lonGridIndex = (point->cellLonIndex - imageGrid->cellEastLonIndex[latGridIndex]) /
						imageGrid->cellLonSizes[latGridIndex];
					pointGridIndex = (latGridIndex * imageGrid->lonCount) + lonGridIndex;

					if (DoCompositeImage) {
						imageResult = CompositePoints[pointGridIndex].imagePoint.result;
						colorIndex = CompositePoints[pointGridIndex].imagePoint.colorIndex;
					} else {
						imageResult = desTopSource->imagePoints[pointGridIndex].result;
						colorIndex = desTopSource->imagePoints[pointGridIndex].colorIndex;
					}

					updateResult = 0;

					switch (imageResult) {

						case 0: {
							colorIndex = -1;
							updateResult = 1;
							break;
						}

						case RESULT_NOSERVICE: {
							if (RESULT_NOSERVICE != result) {
								colorIndex = -1;
							}
							updateResult = 1;
							break;
						}

						case RESULT_INTERFERE: {
							if (RESULT_NOSERVICE != result) {
								if (RESULT_INTERFERE != result) {
									colorIndex = -1;
								}
								updateResult = 1;
							}
							break;
						}

						case RESULT_COVERAGE: {
							if (RESULT_COVERAGE == result) {
								updateResult = 1;
							}
							break;
						}
					}

					if (updateResult) {

						if ((RESULT_NOSERVICE == result) && (ImageNoServiceIndex >= 0)) {
							i = ImageNoServiceIndex;
						} else {

							if ((RESULT_INTERFERE == result) && (ImageInterferenceIndex >= 0)) {
								i = ImageInterferenceIndex;
							} else {

								switch (MapOutputFlags[MAP_OUT_IMAGE]) {
									case MAP_OUT_IMAGE_DMARG:
									default: {
										imageVal = desMargin;
										break;
									}
									case MAP_OUT_IMAGE_DUMARG: {
										imageVal = worstDUMarg;
										break;
									}
									case MAP_OUT_IMAGE_WLDUMARG: {
										imageVal = wlDUMarg;
										break;
									}
									case MAP_OUT_IMAGE_SELFDUMARG: {
										imageVal = selfDUMarg;
										break;
									}
									case MAP_OUT_IMAGE_MARG: {
										imageVal = desMargin;
										if (worstDUMarg < imageVal) {
											imageVal = worstDUMarg;
										}
										if (wlDUMarg < imageVal) {
											imageVal = wlDUMarg;
										}
										if (selfDUMarg < imageVal) {
											imageVal = selfDUMarg;
										}
										break;
									}
									case MAP_OUT_IMAGE_DESSIG: {
										imageVal = desSig;
										break;
									}
								}

								i = ImageBackgroundIndex;
								for (j = (ImageColorCount - 1); j >= ImageMapStartIndex; j--) {
									if (imageVal >= ImageLevel[j]) {
										i = j;
										break;
									}
								}
							}
						}
						
						if (i > colorIndex) {
							if (DoCompositeImage) {
								CompositePoints[pointGridIndex].imagePoint.result = result;
								CompositePoints[pointGridIndex].imagePoint.colorIndex = i;
							} else {
								desTopSource->imagePoints[pointGridIndex].result = result;
								desTopSource->imagePoints[pointGridIndex].colorIndex = i;
							}
						}
					}
				}
			}
		}

		if (((outFile = OutputFile[CELL_FILE_PAIRSTUDY])) && !startPointPair) {
			fputs("[endpoint]\n", outFile);
		}

		if (pointsMode) {
			if ((outFile = OutputFile[REPORT_FILE_DETAIL])) {
				fputc('\n', outFile);
			}
			if ((outFile = OutputFile[REPORT_FILE_SUMMARY])) {
				fputc('\n', outFile);
			}
		}
	}

	if (((outFile = OutputFile[CELL_FILE_DETAIL])) && !startCell) {
		fputs("[endcell]\n", outFile);
	}

	if (((outFile = OutputFile[CELL_FILE_SUMMARY])) && !startCellSum) {
		fputs("[endcell]\n", outFile);
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Routine to compute and return the adjustment to the required D/U for co-channel protection to a DTV when the desired
// field strength is close to the noise-limited service level.  The adjustment value is added to the pre-defined
// required D/U for "strong signal" ("low noise") conditions, forcing the undesired field to be weaker when the
// desired field is closer to the noise-limited service level (i.e. has a smaller S/N).

// This routine is based on FORTRAN code obtained from the FCC OET on February 23, 1998.  The FCC Rules imply there
// should be a "cut-off" limit when the relative S/N is above a threshold, however, the FCC software does not implement
// such a feature; it will compute and apply a D/U adjustment, which may be very small or 0., regardless of the value
// of the relative S/N.

// Arguments:

//   udtv    True if undesired is digital, false if analog.
//   relsnr  Difference between desired field strength and the desired service threshold.

// Return D/U adjustment in dB.

static double dtv_codu_adjust(int udtv, double relsnr) {

	static double n_adj[10] = {
		30.00, 18.75, 16.50, 15.25,  6.00,  3.50,  2.50,  1.75,  1.25,  0.00
	};

	double adj;

	// Compute adjustment.  If the undesired is DTV, the adjustment is a function of the relative S/N, this is derived
	// theoretically under the assumption that the interfering DTV signal behaves just like additional noise, hence at
	// any point the following relationship must be satisfied:
	//                               D
	//                         R = -----
	//                             U + N
	// where D is the desired signal, U is undesired, N is noise, and R is the "critical" value for service, that is
	// the S/N at the noise-limited threshold or the required D/U where the desired signal is strong and noise is not a
	// factor (these are the same value).

	if (udtv) {

		double x = relsnr / 10.;
		if (x < 0.0001) {
			x = 0.0001;
		}
		adj = 10. * log10(1. / (1. - pow(10., -x)));

	// If the undesired is NTSC, the interfering signal does not appear as more noise, it has a much more complex
	// effect.  The correction in this case is based on table-lookup from data determined empirically by the ATTC in
	// the original Grand Alliance system tests in October, 1995.

	} else {

		int i = (int)relsnr;
		if (i < 0) {
			i = 0;
		}
		if (i < 8) {
			adj = n_adj[i] + ((relsnr - (double)i) * (n_adj[i + 1] - n_adj[i]));
		} else {
			adj = n_adj[8] + (((relsnr - 8.) / 16.) * (n_adj[9] - n_adj[8]));
			if (adj < 0.) {
				adj = 0.;
			}
		}
	}

	// Done.

	return(adj);
}


//---------------------------------------------------------------------------------------------------------------------
// Do a lookup in the all-points compositing grid.  If makePoint is false this is a post-process lookup, in which case
// if the cell and country are not found this returns NULL.  Otherwise a point structure will be returned, possibly
// after adding it as an extra point for the cell if the main array element is already in use for a different country.

// Arguments:

//   countryKey     Country key.
//   latGridIndex   Grid index values for the cell in the current CompositePoints grid layout.
//   lonGridIndex
//   makePoint      True to always return a point, creating extra point if needed.

// Return is NULL if no match found or an error occurs.

static COMPOSITE_POINT *get_composite_point(int countryKey, int latGridIndex, int lonGridIndex, int makePoint) {

	COMPOSITE_POINT *thePoint;

	// Retrieve the structure from the main array, if the search country is a match to the main point country, done.
	// If the first country key is 0 the cell is empty, if makePoint is true initialize country and return.  If
	// makePoint is false return NULL, that is not an error, post-process code may try all possible country keys.

	int pointGridIndex = (latGridIndex * CompositeGrid->lonCount) + lonGridIndex;
	thePoint = CompositePoints + pointGridIndex;

	int firstCountryKey = thePoint->countryKey;
	if (countryKey == firstCountryKey) {
		return thePoint;
	}

	if (0 == firstCountryKey) {
		if (makePoint) {
			thePoint->countryKey = countryKey;
			return thePoint;
		}
		return NULL;
	}

	// If an extra point for the country exists and matches the desired country, find and return that.  If the extra
	// point exists but does not match the requested country and makePoint is true that is an error; it should be
	// impossible for more than two countries to have points in the same cell.  If the country key is not a match and
	// makePoint is false, return NULL.

	int extraCountryKey = thePoint->extraCountryKey;

	if (extraCountryKey > 0) {

		if (countryKey == extraCountryKey) {

			int extraPointIndex;
			for (extraPointIndex = 0; extraPointIndex < ExtraCompositePointCount; extraPointIndex++) {
				if ((latGridIndex == ExtraCompositePointLatGridIndex[extraPointIndex]) &&
						(lonGridIndex == ExtraCompositePointLonGridIndex[extraPointIndex])) {
					return ExtraCompositePoints + extraPointIndex;
				}
			}

			log_error("Missing point in composite cell array: countryKey=%d latGridIndex=%d lonGridIndex=%d",
				countryKey, latGridIndex, lonGridIndex);
			return NULL;
		}

		if (makePoint) {
			log_error("Cannot add point to composite cell data: countryKey=%d latGridIndex=%d lonGridIndex=%d",
				countryKey, latGridIndex, lonGridIndex);
			return NULL;
		}

		return NULL;
	}

	// If no match, no extra point exists, and makePoint is true, add an extra point for this cell.

	if (makePoint) {

		if (ExtraCompositePointCount >= MaxExtraCompositePointCount) {
			MaxExtraCompositePointCount += 1000;
			ExtraCompositePoints = (COMPOSITE_POINT *)mem_realloc(ExtraCompositePoints,
				(MaxExtraCompositePointCount * sizeof(COMPOSITE_POINT)));
			ExtraCompositePointLatGridIndex = (int *)mem_realloc(ExtraCompositePointLatGridIndex,
				(MaxExtraCompositePointCount * sizeof(int)));
			ExtraCompositePointLonGridIndex = (int *)mem_realloc(ExtraCompositePointLonGridIndex,
				(MaxExtraCompositePointCount * sizeof(int)));
		}

		thePoint->extraCountryKey = countryKey;

		int extraPointIndex = ExtraCompositePointCount++;

		thePoint = ExtraCompositePoints + extraPointIndex;
		memset(thePoint, 0, sizeof(COMPOSITE_POINT));
		thePoint->countryKey = countryKey;

		ExtraCompositePointLatGridIndex[extraPointIndex] = latGridIndex;
		ExtraCompositePointLonGridIndex[extraPointIndex] = lonGridIndex;

		return thePoint;
	}

	// No match, no extra point, and makePoint is false.

	return NULL;
}


//---------------------------------------------------------------------------------------------------------------------
// Called when study database is being closed.  If no error, do final reporting of scenario pairings.  These are only
// written to the summary files so one of those must be open.

// Arguments

//   hadError  1 if closing due to error, else 0.

void study_closing(int hadError) {

	if (!hadError) {

		if (ScenarioPairList && (OutputFile[REPORT_FILE_SUMMARY] || OutputFile[CSV_FILE_SUMMARY])) {

			FILE *repFile = OutputFile[REPORT_FILE_SUMMARY], *csvFile = OutputFile[CSV_FILE_SUMMARY];

			SCENARIO_PAIR *thePair = ScenarioPairList;
			int doHeader = 1;

			while (thePair) {
				if (thePair->didStudyA && thePair->didStudyB) {
					if (repFile) {
						write_pair_report(thePair, repFile, doHeader);
					}
					if (csvFile) {
						write_pair_csv(thePair, csvFile, doHeader);
					}
					doHeader = 0;
				}
				thePair = thePair->next;
			}
		}
	}

	// Close summary files.

	if (OutputFile[REPORT_FILE_SUMMARY]) {
		file_close(OutputFile[REPORT_FILE_SUMMARY]);
		OutputFile[REPORT_FILE_SUMMARY] = NULL;
	}
	if (OutputFile[CSV_FILE_SUMMARY]) {
		file_close(OutputFile[CSV_FILE_SUMMARY]);
		OutputFile[CSV_FILE_SUMMARY] = NULL;
	}
	if (OutputFile[CELL_FILE_SUMMARY]) {
		file_close(OutputFile[CELL_FILE_SUMMARY]);
		OutputFile[CELL_FILE_SUMMARY] = NULL;
	}
	if (OutputFile[CELL_FILE_PAIRSTUDY]) {
		file_close(OutputFile[CELL_FILE_PAIRSTUDY]);
		OutputFile[CELL_FILE_PAIRSTUDY] = NULL;
	}

	ReportSumHeader = 0;
	CSVSumHeader = 0;

	// Clear the pending files list in case it contains anything, always commit the files.

	clear_pending_files(1);
}


//---------------------------------------------------------------------------------------------------------------------
// Do a fork-exec to run a child process, monitor the process and call hb_log() regularily to update UI and check for
// terminate flag.  Optionally may set the working directory path for the child.

// Arguments:

//   args     Array of command arguments, NULL-terminated.
//   workDir  Working directory set after the fork, may be NULL for no change.

// Returns the exit code of the process, -1 for other error or termination.

static int run_process(char **args, char *workDir) {

	int cpid, status = 0, ret;

	if (0 == (cpid = fork())) {
		if (workDir && (chdir(workDir) < 0)) {
			_exit(errno);
		}
		execvp(args[0], args);
		_exit(1);
	}

	if (cpid < 0) {
		log_error("Could not create process for '%s', errno=%d (%s)", args[0], errno, strerror(errno));
		return -1;
	}

	while ((ret = waitpid(cpid, &status, WNOHANG)) <= 0) {
		if ((ret < 0) && (EINTR != errno)) {
			log_error("Could not get process status for '%s', errno=%d (%s)", args[0], errno, strerror(errno));
			return -1;
		}
		hb_log();
		sleep(1);
	}

	// Terminate is not checked until the process exits, to avoid race conditions.

	if (Terminate) {
		return -1;
	}

	int err = 0;

	if (WIFEXITED(status)) {
		err = WEXITSTATUS(status);
		if (err) {
			log_error("Process for '%s' exited with code %d", args[0], err);
		}
	} else {
		log_error("Process for '%s' failed", args[0]);
		err = 1;
	}

	return err;
}
