//
//  map.c
//  TVStudy
//
//  Copyright (c) 2012-2018 Hammett & Edison, Inc.  All rights reserved.


// Functions related to geographic data.


#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "global.h"
#include "map.h"
#include "coordinates.h"
#include "memory.h"
#include "parser.h"


//---------------------------------------------------------------------------------------------------------------------

typedef struct ply {    // Data for an area polygon.
	GEOPOINTS *pts;     // Latitude and longitude data.
	short *isBorder;    // Identify points used for border-distance check.
	struct ply *next;   // For list.
	int countryKey;     // Country key for this area.
} COUNTRY_POLY;

static COUNTRY_POLY *PolyHead = NULL;

// Rendering parameters, maximum straight-line segment length, and maximum deviation of straight segment from curve,
// in kilometers at target map scale.  Constants represent 0.5 inches and 0.002 inches respectively.

static double Dres = 1.27e-5 * RENDER_MAP_SCALE;
static double Flat = 5.08e-8 * RENDER_MAP_SCALE;

static int load_polys();
static void simul(double *ae, double *ce, double *a, double *b, double *c, double *d);
static void curv(double a, double b, double c, double d, double x, double *f0, double *fp, double *fr);


//---------------------------------------------------------------------------------------------------------------------
// Check a set of coordinates to determine country.  This reads a set of bounding polygons from a support file which
// define the known country areas.  May return 0 if the test point does not fall in any known area or if lookup fails;
// will return <0 on a serious error.

// Regarding over-range longitude values (west longitude >180 degrees or east longitude <-180 degrees), no correction
// is made here.  The individual boundaries in the database must always be continuous for the polygon test to succeed,
// so the boundary data itself may contain over-range longitude values.  If a region is near (or spans) the 180-degree
// line and may need to be tested against over-range points on either side, the region boundary must be duplicated in
// the database, one version having west longitudes with points >180 degrees as needed and the other having east
// longitudes with points <-180 degrees as needed.

// Arguments:

//   lat, lon  Point coordinates, degrees, positive north and west.

// Return is <0 for an error, 0 if no country could be found, else CNTRY_USA, CNTRY_CAN, or CNTRY_MEX (all >0).

int find_country(double lat, double lon) {

	if (!PolyHead) {
		int err = load_polys();
		if (err) {
			if (err < 0) return err;
			return 0;
		}
	}

	COUNTRY_POLY *poly;
	for (poly = PolyHead; poly; poly = poly->next) {
		if (inside_poly(lat, lon, poly->pts)) {
			return poly->countryKey;
		}
	}

	// If the point is not in any polygon, default to U.S.  As of this writing only non-U.S. polygons are present in
	// the database, so U.S. is implied by no match to any other country.

	return CNTRY_USA;
}


//---------------------------------------------------------------------------------------------------------------------
// Determine shortest distances to country borders.

// Arguments:

//   lat, lon  Point coordinates, degrees, positive north and west.
//   dist      Array of MAX_COUNTRY size, will be set to distances by country.  Distance is 0 if no data is found.

// Return is <0 for an error.

int get_border_distances(double lat, double lon, double *borderDist, double kmPerDegree) {

	int countryIndex;
	for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {
		borderDist[countryIndex] = 0.;
	}

	if (!PolyHead) {
		int err = load_polys();
		if (err) {
			if (err < 0) return err;
			return 0;
		}
	}

	COUNTRY_POLY *poly;
	GEOPOINTS *pts;
	double dist;
	int i;

	for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {
		borderDist[countryIndex] = 99999.;
	}

	for (poly = PolyHead; poly; poly = poly->next) {
		countryIndex = poly->countryKey - 1;
		pts = poly->pts;
		for (i = 0; i < pts->nPts; i++) {
			if (poly->isBorder[i]) {
				bear_distance(lat, lon, pts->ptLat[i], pts->ptLon[i], NULL, NULL, &dist, kmPerDegree);
				if (dist < borderDist[countryIndex]) {
					borderDist[countryIndex] = dist;
				}
			}
		}
	}

	for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {
		if (99999. == borderDist[countryIndex]) {
			borderDist[countryIndex] = 0.;
		}
	}

	// Unless there actually was data tagged U.S. in the database (usually isn't because it would be redundant for the
	// continental case), the U.S. distance is set to the smaller of all others.

	if (0. == borderDist[CNTRY_USA - 1]) {
		dist = 99999.;
		for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {
			if (CNTRY_USA != (countryIndex + 1)) {
				if (borderDist[countryIndex] < dist) {
					dist = borderDist[countryIndex];
				}
			}
		}
		borderDist[CNTRY_USA - 1] = dist;
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Load polygons from file, return non-zero on error.  This will only make one attempt to read the file, additional
// calls will immediately return error.

static int load_polys() {

	static int NoData = 0;

	if (NoData) return 1;

	char fname[MAX_STRING], line[MAX_STRING];

	snprintf(fname, MAX_STRING, "%s/country_poly.csv", LIB_DIRECTORY_NAME);
	FILE *in = fopen(fname, "r");
	if (!in) {
		NoData = 1;
		return 1;
	}

	double *latbuf = NULL, *lonbuf = NULL;
	short *bordbuf = NULL;
	int maxbuf = 0, countryKey = 0, polyseq = -1, vertexseq = -1, lastvertexseq = -1, np = 0, i;
	COUNTRY_POLY *poly;
	COUNTRY_POLY **polyLink = &PolyHead;
	GEOPOINTS *pts;

	int err = 0;

	while (fgetnlc(line, MAX_STRING, in, NULL) >= 0) {

		err = token_atoi(next_token(line), 0, 9999, &polyseq);
		if (err) break;

		if (0 == polyseq) {

			if (np) {

				poly = (COUNTRY_POLY *)mem_alloc(sizeof(COUNTRY_POLY));
				pts = geopoints_alloc(np);
				poly->pts = pts;
				pts->nPts = np;
				poly->isBorder = (short *)mem_alloc(np * sizeof(short));
				poly->countryKey = countryKey;
				poly->next = NULL;

				pts->xlts = 999.;
				pts->xlne = 999.;
				pts->xltn = -999.;
				pts->xlnw = -999.;

				for (i = 0; i < np; i++) {
					if (latbuf[i] < pts->xlts) {
						pts->xlts = latbuf[i];
					}
					if (latbuf[i] > pts->xltn) {
						pts->xltn = latbuf[i];
					}
					if (lonbuf[i] < pts->xlne) {
						pts->xlne = lonbuf[i];
					}
					if (lonbuf[i] > pts->xlnw) {
						pts->xlnw = lonbuf[i];
					}
					pts->ptLat[i] = latbuf[i];
					pts->ptLon[i] = lonbuf[i];
					poly->isBorder[i] = bordbuf[i];
				}

				*polyLink = poly;
				polyLink = &(poly->next);

				np = 0;
			}

			lastvertexseq = -1;
			continue;
		}

		err = token_atoi(next_token(NULL), 0, 99999, &vertexseq);
		if (err) break;
		if (vertexseq <= lastvertexseq) {
			err = 1;
			break;
		}
		lastvertexseq = vertexseq;

		if (np == maxbuf) {
			maxbuf += 10000;
			latbuf = (double *)mem_realloc(latbuf, (maxbuf * sizeof(double)));
			lonbuf = (double *)mem_realloc(lonbuf, (maxbuf * sizeof(double)));
			bordbuf = (short *)mem_realloc(bordbuf, (maxbuf * sizeof(short)));
		}

		err = token_atof(next_token(NULL), -75., 75., (latbuf + np));
		if (err) break;
		err = token_atof(next_token(NULL), -180., 180., (lonbuf + np));
		if (err) break;
		err = token_atoi(next_token(NULL), 1, 3, &countryKey);
		if (err) break;
		err = token_atoi(next_token(NULL), 0, 1, &i);
		if (err) break;
		bordbuf[np++] = (short)i;
	}

	fclose(in);
	if (maxbuf > 0) {
		mem_free(latbuf);
		mem_free(lonbuf);
		mem_free(bordbuf);
	}

	if (err) NoData = 1;

	return err;
}


//---------------------------------------------------------------------------------------------------------------------
// Render a contour to a series of geographic coordinates, optimized for a particular map scale.  This uses a curve-
// fit to the tabulated contour data rendered at incremental points with spacing adjusted dynamically for curvature.
// The rendered data is cached in the contour structure, if that already exists it is just returned.

// Arguments:

//   contour  The contour to render.

// The return is always the GEOPOINTS structure; no errors are possible.

GEOPOINTS *render_contour(CONTOUR *contour, double kmPerDegree) {

	if (contour->points) {
		return contour->points;
	}

	// Set up for the main loop.

	GEOPOINTS *pts = geopoints_alloc(0);
	contour->points = pts;

	double lat = contour->latitude, lon = contour->longitude, *rd = contour->distance;
	int np = contour->count;

	double at[16], cd[4], ax0, ax1, ax2, r0, r1, r2, d1, d2, sln0, sln1, cx, az, azn, a, b, c, d, ds, ph, rc,
		dseg, dsegl, dsn, phn, rcn, azi, rcdel, lat1, lon1;
	int n, j, pass;

	at[0] = 1.;
	at[1] = 1.;
	at[2] = 0.;
	at[3] = 0.;
	at[6] = 1.;
	at[7] = 1.;

	sln1 = 0.;

	// Loop over the contour points.  First collect azimuth and distance at three sequential points.  On the first
	// loop no actual rendering will occur, just set up parameters for the first curve fit.

	double step = 360. / (double)np;

	for (n = -1; n < np; n++) {

		ax0 = (double)n * step;
		j = n;
		if (j < 0) {
			j += np;
		}
		r0 = rd[j];

		ax1 = ax0 + step;
		j = n + 1;
		if (j >= np) {
			j -= np;
		}
		r1 = rd[j];

		ax2 = ax1 + step;
		j = n + 2;
		if (j >= np) {
			j -= np;
		}
		r2 = rd[j];

		// Determine the desired slopes of the curve at each endpoint.  The first end comes from the previous loop,
		// compute the second.  On the first time through the loop, that's all.

		sln0 = sln1;

		d1 = (r1 - r0) / (ax1 - ax0);
		d2 = (r2 - r1) / (ax2 - ax1);
		if (((d1 >= 0.) && (d2 <= 0.)) || ((d1 <= 0.) && (d2 >= 0.))) {
			sln1 = 0.;
		} else {
			cx = cos(((d1 / (d1 + d2)) - 0.5) * PI);
			sln1 = cx * ((r2 - r0) / (ax2 - ax0));
		}

		if (n < 0) {
			continue;
		}

		// Set up the rest of the matrix and solve for the curve coefficients.

		at[4] = ax0;
		at[5] = ax1;
		at[8] = ax0 * ax0;
		at[9] = ax1 * ax1;
		at[10] = 2. * ax0;
		at[11] = 2. * ax1;
		at[12] = at[8] * ax0;
		at[13] = at[9] * ax1;
		at[14] = 3. * at[8];
		at[15] = 3. * at[9];
		cd[0] = r0;
		cd[1] = r1;
		cd[2] = sln0;
		cd[3] = sln1;

		simul(at, cd, &a, &b, &c, &d);

		// Evaluate the curve at the first point, then enter a loop to render incrementally.

		az = ax0;
		curv(a, b, c, d, az, &ds, &ph, &rc);

		do {

			coordinates(lat, lon, az, ds, &lat1, &lon1, kmPerDegree);
			geopoints_addpoint(pts, lat1, lon1);

			// Determine the next point to render.  Set the target segment length, then estimate the point based on
			// that length assuming the curvature remains constant.

			if (rc > Flat) {
				dseg = 2. * sqrt(Flat * ((2. * rc) - Flat));
			} else {
				dseg = 2. * Flat;
			}
			if (dseg > Dres) {
				dseg = Dres;
			}
			azi = asin((dseg * sin(ph * DEGREES_TO_RADIANS)) /
				sqrt((ds * ds) + (dseg * dseg) - (2. * ds * dseg * cos(ph * DEGREES_TO_RADIANS)))) *
				RADIANS_TO_DEGREES;

			// Evaluate the curve at the estimated point, check the actual segment length and curvature at the point
			// (length within 20% of target, no more than 50% change in curvature), adjust and loop as needed.

			pass = 0;

			while (1) {

				azn = az + azi;
				if (azn > ax1) {
					azn = ax1;
					azi = azn - az;
				}
				curv(a, b, c, d, azn, &dsn, &phn, &rcn);

				dsegl = sqrt((ds * ds) + (dsn * dsn) - (2. * ds * dsn * cos(azi * DEGREES_TO_RADIANS)));
				if (rc > rcn) {
					rcdel = rc / rcn;
				} else {
					rcdel = rcn / rc;
				}

				// On the first pass, looking for reasons to make the segment longer.

				if (!pass) {
					if (((dsegl / dseg) < 0.8) && (rcdel <= 1.5) && (azn < ax1)) {
						azi *= 1.1;
						continue;
					}
					pass = 1;
				}

				// On the second pass, looking for reasons to make it shorter.

				if (((dsegl / dseg) > 1.2) || ((rcdel > 1.5) && (dsegl > (5. * Flat)))) {
					azi *= 0.9;
					continue;
				}

				break;
			}

			az = azn;
			ds = dsn;
			ph = phn;
			rc = rcn;

		} while (az < ax1);
	}

	// The contour must be an explicitly-closed polygon.

	geopoints_addpoint(pts, pts->ptLat[0], pts->ptLon[0]);

	return pts;
}


//---------------------------------------------------------------------------------------------------------------------
// Solve four simultaneous equations.

// Arguments:

//  ae          Equation coefficients.
//  ce          Solution column.
//  a, b, c, d  Return solution coefficients.

static void simul(double *ae, double *ce, double *a, double *b, double *c, double *d) {

	double aq[16], aa[4], bb[4], cc[4], dd[4], a1[4], a2[4], b1[4], b2[4], c1[4], a3, a4;
	int i;

	for (i = 0; i < 16; i++) {
		if (fabs(ae[i]) < TINY) {
			if (ae[i] < 0.) {
				ae[i] = -TINY;
			} else {
				ae[i] = TINY;
			}
		}
	}
	for (i = 0; i < 4; i++) {
		if (fabs(ce[i]) < TINY) {
			if (ce[i] < 0.) {
				ce[i] = -TINY;
			} else {
				ce[i] = TINY;
			}
		}
	}

	for (i = 0; i < 4; i++) {
		aq[i * 4] = ae[i];
		aq[(i * 4) + 1] = ae[i + 4];
		aq[(i * 4) + 2] = ae[i + 8];
		aq[(i * 4) + 3] = ae[i + 12];
	}
	for (i = 0; i < 3; i++) {
		aa[i] = aq[i + 1] / aq[0];
		bb[i] = aq[i + 5] / aq[4];
		cc[i] = aq[i + 9] / aq[8];
		dd[i] = aq[i + 13] / aq[12];
	}
	aa[3] = ce[0] / aq[0];
	bb[3] = ce[1] / aq[4];
	cc[3] = ce[2] / aq[8];
	dd[3] = ce[3] / aq[12];
	for (i = 0; i < 4; i++) {
		a1[i] = bb[i] - aa[i];
		b1[i] = cc[i] - aa[i];
		c1[i] = dd[i] - aa[i];
	}
	for (i = 0; i < 2; i++) {
		aa[i] = a1[i + 1] / a1[0];
		bb[i] = b1[i + 1] / b1[0];
		cc[i] = c1[i + 1] / c1[0];
	}
	aa[2] = a1[3] / a1[0];
	bb[2] = b1[3] / b1[0];
	cc[2] = c1[3] / c1[0];
	for (i = 0; i < 3; i++) {
		a2[i] = bb[i] - aa[i];
		b2[i] = cc[i] - aa[i];
	}
	aa[0] = a2[1] / a2[0];
	aa[1] = a2[2] / a2[0];
	bb[0] = b2[1] / b2[0];
	bb[1] = b2[2] / b2[0];
	a3 = bb[0] - aa[0];
	a4 = bb[1] - aa[1];

	*d = a4 / a3;
	*c = (a2[2] - (*d * a2[1])) / a2[0];
	*b = (a1[3] - (*c * a1[1]) - (*d * a1[2])) / a1[0];
	*a = (ce[0] - (*b * aq[1]) - (*c * aq[2]) - (*d * aq[3])) / aq[0];
}


//---------------------------------------------------------------------------------------------------------------------
// Evaluate polynomial curve in polar coordinates.

// Arguments:

//   a, b, c, d  Coefficients.
//   x           Argument.
//   f0          Return curve value.
//   fp          Return angle between radius vector and tangent line.
//   fr          Return radius of curvature.

static void curv(double a, double b, double c, double d, double x, double *f0, double *fp, double *fr) {

	double f0l, f1, f2, fa, fb, fpl;

	f0l = a + (b * x) + (c * x * x) + (d * x * x * x);
	*f0 = f0l;

	f1 = (b + (2. * c * x) + (3. * d * x * x)) * RADIANS_TO_DEGREES;
	f2 = ((2. * c) + (6. * d * x)) * (RADIANS_TO_DEGREES * RADIANS_TO_DEGREES);

	if (fabs(f1) > 1.e-10) {
		fpl = atan(f0l / f1) * RADIANS_TO_DEGREES;
		if (fpl >= 0.) {
			fpl = 180. - fpl;
		} else {
			fpl = fabs(fpl);
		}
	} else {
		fpl = 90.;
	}
	*fp = fpl;

	fa = pow(((f0l * f0l) + (f1 * f1)), 1.5);
	fb = fabs((f0l * f0l) + (2. * (f1 * f1)) - (f0l * f2));
	if (fb < 1.e-10) {
		fb = 1.e-10;
	}
	*fr = fa / fb;
}


//---------------------------------------------------------------------------------------------------------------------
// Render a geography to a series of geographic coordinates, optimized for a particular map scale.  The rendered data
// is cached in the structure, if that already exists it is just returned.

// Arguments:

//   geo  The geography to render.

// The return is always the GEOPOINTS structure; no errors are possible.

GEOPOINTS *render_geography(GEOGRAPHY *geo, double kmPerDegree) {

	if (geo->points) {
		return geo->points;
	}

	GEOPOINTS *pts = geopoints_alloc(0);
	geo->points = pts;

	double lat = geo->latitude, lon = geo->longitude, lat1, lon1;

	switch (geo->type) {

		case GEO_TYPE_CIRCLE: {

			double ds = geo->a.radius, dseg, azi, az;

			if (ds > Flat) {
				dseg = 2. * sqrt(Flat * ((2. * ds) - Flat));
			} else {
				dseg = 2. * Flat;
			}
			if (dseg > Dres) {
				dseg = Dres;
			}
			azi = asin(dseg / sqrt((ds * ds) + (dseg * dseg))) * RADIANS_TO_DEGREES;

			for (az = 0.; az < 360.; az += azi) {
				coordinates(lat, lon, az, ds, &lat1, &lon1, kmPerDegree);
				geopoints_addpoint(pts, lat1, lon1);
			}

			coordinates(lat, lon, 0., ds, &lat1, &lon1, kmPerDegree);
			geopoints_addpoint(pts, lat1, lon1);

			break;
		}

		case GEO_TYPE_BOX: {

			int ewcount = (int)(geo->a.width / Dres) + 2;
			int nscount = (int)(geo->b.height / Dres) + 2;

			int side, pti, ptinc, ptend, iter;
			double ewdist, ewbear, nsdist, dlat, lat2, lon2, dist, error;

			for (side = 1; side <= 4; side++) {

				switch (side) {
					case 1:
						nsdist = geo->b.height * 0.5;
						pti = 0;
						ptinc = 1;
						ptend = ewcount;
						break;
					case 2:
						ewdist = geo->a.width * 0.5;
						pti = nscount;
						ptinc = -1;
						ptend = 0;
						break;
					case 3:
						nsdist = geo->b.height * -0.5;
						pti = ewcount;
						ptinc = -1;
						ptend = 0;
						break;
					case 4:
						ewdist = geo->a.width * -0.5;
						pti = 0;
						ptinc = 1;
						ptend = nscount;
						break;
				}

				for (; pti != ptend; pti += ptinc) {

					switch (side) {
						case 1:
						case 3:
							ewdist = (((double)pti / (double)ewcount) - 0.5) * geo->a.width;
							if (ewdist < 0.) {
								ewbear = 90.;
								ewdist = -ewdist;
							} else {
								ewbear = 270.;
							}
							break;
						case 2:
						case 4:
							nsdist = (((double)pti / (double)nscount) - 0.5) * geo->b.height;
							break;
					}

					coordinates(lat, lon, ewbear, ewdist, &lat1, &lon1, kmPerDegree);
					dlat = nsdist / kmPerDegree;
					iter = 20;
					do {
						coordinates((lat + dlat), lon, ewbear, ewdist, &lat2, &lon2, kmPerDegree);
						bear_distance(lat1, lon1, lat2, lon2, NULL, NULL, &dist, kmPerDegree);
						error = (dist - fabs(nsdist)) / fabs(nsdist);
						dlat *= 1. + (error * 0.95);
					} while ((fabs(error) > 0.001) && (--iter > 0));

					geopoints_addpoint(pts, lat2, lon2);
				}
			}

			geopoints_addpoint(pts, pts->ptLat[0], pts->ptLon[0]);

			break;
		}

		case GEO_TYPE_SECTORS: {

			double ds, dseg, az1, az2, azi, az;
			int i;

			for (i = 0; i < geo->nSectors; i++) {

				az1 = geo->a.sectorAzimuth[i];
				if (i < (geo->nSectors - 1)) {
					az2 = geo->a.sectorAzimuth[i + 1];
				} else {
					az2 = geo->a.sectorAzimuth[0] + 360.;
				}
				ds = geo->b.sectorRadius[i];

				if (ds > Flat) {
					dseg = 2. * sqrt(Flat * ((2. * ds) - Flat));
				} else {
					dseg = 2. * Flat;
				}
				if (dseg > Dres) {
					dseg = Dres;
				}
				azi = asin(dseg / sqrt((ds * ds) + (dseg * dseg))) * RADIANS_TO_DEGREES;

				for (az = az1; az < az2; az += azi) {
					coordinates(lat, lon, az, ds, &lat1, &lon1, kmPerDegree);
					geopoints_addpoint(pts, lat1, lon1);
				}

				coordinates(lat, lon, az2, ds, &lat1, &lon1, kmPerDegree);
				geopoints_addpoint(pts, lat1, lon1);
			}

			coordinates(lat, lon, geo->a.sectorAzimuth[0], geo->b.sectorRadius[0], &lat1, &lon1, kmPerDegree);
			geopoints_addpoint(pts, lat1, lon1);

			break;
		}

		case GEO_TYPE_POLYGON: {

			GEOPOINTS *poly = geo->a.polygon;
			double *plat = poly->ptLat, *plon = poly->ptLon, bear, dist, dist1, lat2, lon2;

			int np, i, j;

			for (i = 0; i < (poly->nPts - 1); i++) {

				lat1 = plat[i];
				lon1 = plon[i];

				geopoints_addpoint(pts, lat1, lon1);

				bear_distance(lat1, lon1, plat[i + 1], plon[i + 1], &bear, NULL, &dist, kmPerDegree);
				np = (int)(dist / Dres) + 1;

				for (j = 1; j < np; j++) {

					dist1 = ((double)j / (double)np) * dist;
					coordinates(lat1, lon1, bear, dist1, &lat2, &lon2, kmPerDegree);
					geopoints_addpoint(pts, lat2, lon2);			
				}
			}

			geopoints_addpoint(pts, plat[i], plon[i]);

			break;
		}
	}

	return pts;
}


//---------------------------------------------------------------------------------------------------------------------
// Test whether a point is inside a geography.

// Arguments:

//   plat, plon  Coordinates of point to test, north latitude and west longitude.
//   geography   The geography.

// The return is true for inside, false for outside.

int inside_geography(double plat, double plon, GEOGRAPHY *geo, double kmPerDegree) {

	switch (geo->type) {

		case GEO_TYPE_CIRCLE: {
			double dist;
			bear_distance(geo->latitude, geo->longitude, plat, plon, NULL, NULL, &dist, kmPerDegree);
			return (dist <= geo->a.radius);
			break;
		}

		case GEO_TYPE_BOX: {
			return inside_poly(plat, plon, render_geography(geo, kmPerDegree));
			break;
		}

		case GEO_TYPE_SECTORS: {
			double bear, dist;
			bear_distance(geo->latitude, geo->longitude, plat, plon, &bear, NULL, &dist, kmPerDegree);
			if (bear >= geo->a.sectorAzimuth[0]) {
				int i;
				for (i = 0; i < (geo->nSectors - 1); i++) {
					if (bear < geo->a.sectorAzimuth[i + 1]) {
						return (dist <= geo->b.sectorRadius[i]);
					}
				}
			}
			return (dist <= geo->b.sectorRadius[geo->nSectors - 1]);
			break;
		}

		case GEO_TYPE_POLYGON: {
			return inside_poly(plat, plon, render_geography(geo, kmPerDegree));
			break;
		}
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Convenience for using interp() on a contour structure.

double interp_cont(double lookup, CONTOUR *contour) {

	return interp(lookup, contour->distance, contour->count);
}


//---------------------------------------------------------------------------------------------------------------------
// Linear interpolation in a table of azimuthal data with regularily-spaced points.

// Arguments:

//   lookup   Azimuth to look up.
//   table    Data table.
//   count    Number of values in table.

// Return is the interpolated value.

double interp(double lookup, double *table, int count) {

	double step = 360. / (double)count;

	int index0 = lookup / step;
	double lookup0 = (double)index0 * step;
	while (index0 < 0) {
		index0 += count;
	}
	int index1 = index0 + 1;
	while (index1 >= count) {
		index1 -= count;
	}

	return table[index0] + (((lookup - lookup0) / step) * (table[index1] - table[index0]));
}


//---------------------------------------------------------------------------------------------------------------------
// Like interp(), but applies a minimum value to the result.  The minimum may optionally be applied to the table
// values before interpolation which changes the slope between points.

// Arguments:

//   lookup   Azimuth to look up.
//   table    Data table.
//   count    Number of values in table.
//   minVal   Minimum value, result will never be less than this.
//   minMode  True to apply minimum to table values before interpolation.

// Return is the interpolated value.

double interp_min(double lookup, double *table, int count, double minVal, int minMode) {

	double step = 360. / (double)count;

	int index0 = lookup / step;
	double lookup0 = (double)index0 * step;
	while (index0 < 0) {
		index0 += count;
	}
	int index1 = index0 + 1;
	while (index1 >= count) {
		index1 -= count;
	}

	double table0 = table[index0], table1 = table[index1];
	if (minMode) {
		if (table0 < minVal) {
			table0 = minVal;
		}
		if (table1 < minVal) {
			table1 = minVal;
		}
	}
	double result = table0 + (((lookup - lookup0) / step) * (table1 - table0));
	if (result < minVal) {
		result = minVal;
	}

	return result;
}


//---------------------------------------------------------------------------------------------------------------------
// Allocate a CONTOUR structure and a dependent distance array only (points are added later as needed).

// Arguments:

//   latitude   Contour center latitude.
//   longitude  Contour center longitude.
//   mode       Contour mode (method of projection), not defined here.
//   count      Count of contour projection radials.

// Return contour, no error return, allocation errors case process to abort.

CONTOUR *contour_alloc(double latitude, double longitude, int mode, int count) {

	CONTOUR *contour = mem_zalloc(sizeof(CONTOUR));
	contour->latitude = latitude;
	contour->longitude = longitude;
	contour->mode = mode;
	contour->count = count;
	contour->distance = mem_zalloc(count * sizeof(double));

	return contour;
}


//---------------------------------------------------------------------------------------------------------------------
// Free a CONTOUR structure and dependent arrays.

// Arguments:

//   contour  The contour to free.

void contour_free(CONTOUR *contour) {

	if (contour->distance) {
		mem_free(contour->distance);
		contour->distance = NULL;
	}
	if (contour->points) {
		geopoints_free(contour->points);
		contour->points = NULL;
	}
	mem_free(contour);
}


//---------------------------------------------------------------------------------------------------------------------
// Allocate a geography structure.

// Arguments:

//   geoKey  Geography key.
//   type    GEO_TYPE_*.
//   lat     Center/reference point coordinates.
//   lon
//   count   Allocation size for sectors or pre-allocation size for polygon.  May be 0 for polygon.

// Return is the new structure.

GEOGRAPHY *geography_alloc(int geoKey, int type, double lat, double lon, int count) {

	GEOGRAPHY *geo = (GEOGRAPHY *)mem_zalloc(sizeof(GEOGRAPHY));

	geo->geoKey = geoKey;
	geo->type = type;
	geo->latitude = lat;
	geo->longitude = lon;

	switch (geo->type) {

		case GEO_TYPE_SECTORS: {
			geo->nSectors = count;
			geo->a.sectorAzimuth = (double *)mem_alloc(count * sizeof(double));
			geo->b.sectorRadius = (double *)mem_alloc(count * sizeof(double));
			break;
		}

		case GEO_TYPE_POLYGON: {
			geo->a.polygon = geopoints_alloc(count);
			break;
		}
	}

	return geo;
}


//---------------------------------------------------------------------------------------------------------------------
// Free storage for a geography.

// Arguments:

//   geo  The geography to free.

void geography_free(GEOGRAPHY *geo) {

	switch (geo->type) {

		case GEO_TYPE_SECTORS: {
			mem_free(geo->a.sectorAzimuth);
			geo->a.sectorAzimuth = NULL;
			mem_free(geo->b.sectorRadius);
			geo->b.sectorRadius = NULL;
			break;
		}

		case GEO_TYPE_POLYGON: {
			geopoints_free(geo->a.polygon);
			geo->a.polygon = NULL;
			break;
		}
	}

	if (geo->points) {
		geopoints_free(geo->points);
		geo->points = NULL;
	}

	mem_free(geo);
}


//---------------------------------------------------------------------------------------------------------------------
// Allocate a GEOPOINTS structure of specified size.  If size is 0, a default initial size is used.

#define POINTS_ALLOC_INC  100

GEOPOINTS *geopoints_alloc(int np) {

	if (np <= 0) {
		np = POINTS_ALLOC_INC;
	}

	GEOPOINTS *pts = (GEOPOINTS *)mem_zalloc(sizeof(GEOPOINTS));
	pts->ptLat = (double *)mem_alloc(np * sizeof(double));
	pts->ptLon = (double *)mem_alloc(np * sizeof(double));
	pts->maxPts = np;

	return pts;
}


//---------------------------------------------------------------------------------------------------------------------
// Add a point, increase allocated space as needed.

void geopoints_addpoint(GEOPOINTS *pts, double lat, double lon) {

	if (pts->nPts == pts->maxPts) {

		pts->maxPts += POINTS_ALLOC_INC;

		pts->ptLat = (double *)mem_realloc(pts->ptLat, (pts->maxPts * sizeof(double)));
		pts->ptLon = (double *)mem_realloc(pts->ptLon, (pts->maxPts * sizeof(double)));
	}

	pts->ptLat[pts->nPts] = lat;
	pts->ptLon[pts->nPts++] = lon;

	if (1 == pts->nPts) {
		pts->xlts = lat;
		pts->xltn = lat;
		pts->xlne = lon;
		pts->xlnw = lon;
	} else {
		if (lat < pts->xlts) {
			pts->xlts = lat;
		}
		if (lat > pts->xltn) {
			pts->xltn = lat;
		}
		if (lon < pts->xlne) {
			pts->xlne = lon;
		}
		if (lon > pts->xlnw) {
			pts->xlnw = lon;
		}
	}
}


//---------------------------------------------------------------------------------------------------------------------
// Free memory for a GEOPOINTS structure.  Note this does not check for linked structure, caller must do that.

void geopoints_free(GEOPOINTS *pts) {

	if (pts) {

		mem_free(pts->ptLat);
		pts->ptLat = NULL;
		mem_free(pts->ptLon);
		pts->ptLon = NULL;

		mem_free(pts);
	}
}


//---------------------------------------------------------------------------------------------------------------------
// Evaluate a test point against a polygon.  This uses the "even-odd" rule for determining insideness.  The polygon may
// be compound with multiple islands and holes, the parts of a compound poly are separated by 999,999 coordinate value
// pairs in data array.  The test is based on assuming latitude and longitude are planar X-Y coordinates, the vertex
// list for each sub-poly is assumed to be closed (first and last points identical), and the polygons are assumed to
// never cross each other or themselves.

// Because of the treament of geographic coordinates as planar, this does not produce accurate results for regions near
// the poles.  It is assumed those will not occur in practice and the inaccuracy is of no concern.

// Arguments:

//   plat, plon  Coordinates of point being tested, nominally geographic coordinates in degrees positive north and
//                 west; however in fact the units and sign convention are arbitrary, as long as they are consistent
//                 with the polygon data provided and allow the 999,999 break-point convention.
//   pts         Polygon vertex point coordinates.  May contain +999,+999 break points to separate parts of a compound
//                 region, parts are implicitly "lakes" or "islands" depending on nesting.  All parts must be closed
//                 explicitly (first and last point coordinates identical).

// Return is 1 if the point is inside, 0 if not.

int inside_poly(double plat, double plon, GEOPOINTS *pts) {

	if ((plat < pts->xlts) || (plat > pts->xltn) || (plon < pts->xlne) || (plon > pts->xlnw)) {
		return 0;
	}

	int i, q1, q2, count, c;
	double x, y, x1, y1, x2, y2, dx, yt;

	// A table for fast determination of crossing, indexed by the quadrant numbers for two segment endpoints.  A 1
	// means crossing occurs, 0 means it does not, a -1 means the endpoints are in non-adjacent quadrants and further
	// checking is needed to determine whether there is a crossing or not.

	static int ctbl[4][4] = {
		{ 0,  1, -1,  0},
		{ 1,  0,  0, -1},
		{-1,  0,  0,  0},
		{ 0, -1,  0,  0}
	};

	// Set the test point.

	x = plon;
	y = plat;

	// Loop for points, count up the number of polygon segments that cross a line from the test point parallel to the
	// positive Y axis.  The first step is to obtain segment endpoint data, the first endpoint comes from the previous
	// loop (undefined on the first loop, that's OK because the quadrant number will be -1, see below).

	count = 0;
	x2 = 0.;
	y2 = 0.;
	q2 = -1;
	for (i = 0; i < pts->nPts; i++) {
		x1 = x2;
		y1 = y2;
		q1 = q2;
		x2 = pts->ptLon[i];
		y2 = pts->ptLat[i];

		// Sub-polygon breaks in a compound polygon are marked by 999 values in the data.  Nothing special needs to be
		// done except to be sure not to test an invalid segment across the break, flag by setting the quadrant to -1.
		// Using the even-odd rule means the winding directions of the sub-polys don't make any difference, the test
		// works regardless.  But this does assume that none of the sub-polys cross each other or themselves.

		if (y2 == 999.) {
			q2 = -1;
			continue;
		}

		// Determine the quadrant relative to the test point which contains the second endpoint.  The quadrants are
		// numbered 0-1-2-3 clockwise starting with +x,+y, a point that falls on an axis line belongs to the quadrant
		// that is counter-clockwise of the line (actually this is flipped if longitudes are positive west, but the
		// logic works out the same, only the preceding description is backwards).  If a point is coincident with the
		// test point, the test point is arbitrarily considered inside the poly, the test is done.

		if (x2 < x) {
			if (y2 < y) {
				q2 = 2;
			} else {
				q2 = 1;
			}
		} else {
			if (x2 > x) {
				if (y2 > y) {
					q2 = 0;
				} else {
					q2 = 3;
				}
			} else {
				if (y2 < y) {
					q2 = 2;
				} else {
					if (y2 > y) {
						q2 = 0;
					} else {
						count = 1;
						break;
					}
				}
			}
		}

		// If the endpoint from the previous loop was undefined, that's all.

		if (q1 < 0) {
			continue;
		}

		// A static table indexed by the endpoint quadrants tells what to do.  If the table value is < 0, further
		// checking is needed, otherwise the table value is 0 for no crossing or 1 for crossing, and so is simply added
		// to the crossing count.  In the cases that need checking, the possibility exists that the test point falls
		// exactly on the segment; in this case, as with the coincident-point case above, the test point is inside, the
		// test is done.

		if ((c = ctbl[q1][q2]) < 0) {
			if ((dx = x2 - x1) == 0.) {
				count = 1;
				break;
			}
			yt = y1 + (((x - x1) / dx) * (y2 - y1));
			if (yt > y) {
				c = 1;
			} else {
				if (yt < y) {
					c = 0;
				} else {
					count = 1;
					break;
				}
			}
		}
		count += c;
	}

	// Done, return one's bit of crossing count, which will be set for odd counts (inside) and clear for even counts
	// (outside).

	return(count & 1);
}
