//
//  ColorMapEditor.java
//  TVStudy
//
//  Copyright (c) 2015-2019 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui.editor;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.gui.*;

import java.util.*;
import java.sql.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.color.*;
import java.awt.geom.*;

import javax.swing.*;
import javax.swing.event.*;


//=====================================================================================================================
// Dialog to edit image output color maps.

public class ColorMapEditor extends AppDialog {

	public static final String WINDOW_TITLE = "Image Color Maps";

	private static final int COLOR_BUTTON_WIDTH = 48;
	private static final int COLOR_BUTTON_HEIGHT = 16;

	private static final int INDEX_CODE_NO_SERVICE = -1;
	private static final int INDEX_CODE_INTERFERENCE = -2;
	private static final int INDEX_CODE_BACKGROUND = -3;

	private static final int LOAD_NEW = 0;
	private static final int LOAD_DEFAULT = -1;
	private static final int LOAD_COPY = -2;

	private KeyedRecordMenu colorMapMenu;
	private JButton deleteButton;

	private ColorMap map;
	private boolean didEdit;

	private ColorPanel backgroundPanel;
	private ColorPanel[] colorPanels;
	private JPanel mapColorsPanel;

	private JButton addButton;
	private JButton removeButton;

	private ColorPanel noServicePanel;
	private ColorPanel interferencePanel;

	private JTextField nameField;
	private JButton saveButton;


	//-----------------------------------------------------------------------------------------------------------------
	// This is a per-database singleton, managed by a StudyManager.

	public ColorMapEditor(AppEditor theParent) {

		super(theParent, WINDOW_TITLE, Dialog.ModalityType.MODELESS);

		colorMapMenu = new KeyedRecordMenu();

		colorMapMenu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					int newKey = colorMapMenu.getSelectedKey();
					if (doSave(true)) {
						doLoad(newKey);
					}
					blockActionsEnd();
				}
			}
		});

		deleteButton = new JButton("Delete");
		deleteButton.setFocusable(false);
		deleteButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					doDelete();
					blockActionsEnd();
				}
			}
		});

		JButton newButton = new JButton("New");
		newButton.setFocusable(false);
		newButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					if (doSave(true)) {
						doLoad(LOAD_NEW);
					}
					blockActionsEnd();
				}
			}
		});

		JButton copyButton = new JButton("Duplicate");
		copyButton.setFocusable(false);
		copyButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					if (doSave(true)) {
						doLoad(LOAD_COPY);
					}
					blockActionsEnd();
				}
			}
		});

		JPanel menuPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		menuPanel.add(colorMapMenu);

		JPanel newCopyDeletePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		newCopyDeletePanel.add(newButton);
		newCopyDeletePanel.add(copyButton);
		newCopyDeletePanel.add(deleteButton);

		backgroundPanel = new ColorPanel(INDEX_CODE_BACKGROUND);
		colorPanels = new ColorPanel[ColorMap.COUNT_INCREMENT];

		mapColorsPanel = new JPanel();
		mapColorsPanel.setLayout(new BoxLayout(mapColorsPanel, BoxLayout.Y_AXIS));
		mapColorsPanel.add(backgroundPanel);

		addButton = new JButton("Add");
		addButton.setFocusable(false);
		addButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doAddColor();
			}
		});

		removeButton = new JButton("Remove");
		removeButton.setFocusable(false);
		removeButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRemoveColor();
			}
		});

		JPanel addRemovePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		addRemovePanel.add(addButton);
		addRemovePanel.add(removeButton);

		noServicePanel = new ColorPanel(INDEX_CODE_NO_SERVICE);
		interferencePanel = new ColorPanel(INDEX_CODE_INTERFERENCE);

		nameField = new JTextField(15);
		AppController.fixKeyBindings(nameField);
		nameField.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				String str = nameField.getText().trim();
				if (!str.equals(map.name)) {
					map.name = str;
					setDidEdit();
				}
			}
		});
		nameField.addFocusListener(new FocusAdapter() {
			public void focusGained(FocusEvent theEvent) {
				setCurrentField(nameField);
			}
			public void focusLost(FocusEvent theEvent) {
				if (!theEvent.isTemporary()) {
					nameField.postActionEvent();
				}
			}
		});

		saveButton = new JButton("Save");
		saveButton.setFocusable(false);
		saveButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					if (doSave(false)) {
						updateMenu(true);
					}
					blockActionsEnd();
				}
			}
		});

		JPanel savePanel = new JPanel();
		savePanel.add(nameField);
		savePanel.add(saveButton);

		// Do the layout.

		Box northBox = Box.createVerticalBox();
		northBox.add(menuPanel);
		northBox.add(newCopyDeletePanel);

		JPanel wrapperPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		wrapperPanel.add(mapColorsPanel);

		JPanel centerPanel = new JPanel(new BorderLayout());
		centerPanel.setBorder(BorderFactory.createTitledBorder("Color Lookup"));
		centerPanel.add(AppController.createScrollPane(wrapperPanel), BorderLayout.CENTER);
		centerPanel.add(addRemovePanel, BorderLayout.SOUTH);

		Box southBox = Box.createVerticalBox();
		southBox.add(noServicePanel);
		southBox.add(interferencePanel);
		southBox.add(savePanel);

		Container cp = getContentPane();
		cp.setLayout(new BorderLayout());
		cp.add(northBox, BorderLayout.NORTH);
		cp.add(centerPanel, BorderLayout.CENTER);
		cp.add(southBox, BorderLayout.SOUTH);

		pack();

		Dimension theSize = getSize();
		theSize.height = 520;
		setMinimumSize(theSize);
		setSize(theSize);

		setResizable(true);
		setLocationSaved(true);

		updateDocumentName();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void updateDocumentName() {

		setDocumentName(parent.getDocumentName());
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doAddColor() {

		if (!commitCurrentField()) {
			return;
		}

		if (map.addColor() < 0) {
			return;
		}

		setDidEdit();

		updatePanels(false);
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doRemoveColor() {

		if (!commitCurrentField()) {
			return;
		}

		map.removeColor();
		setDidEdit();

		updatePanels(false);
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void updateMenu(boolean reSelect) {

		String saveTitle = errorReporter.getTitle();
		errorReporter.setTitle("Load Color Maps");

		colorMapMenu.removeAllItems();

		ArrayList<KeyedRecord> colorMaps = ColorMap.getColorMaps(getDbID(), errorReporter);
		if (null != colorMaps) {
			colorMapMenu.addAllItems(colorMaps);
		}

		if (reSelect) {
			if (!colorMapMenu.containsKey(map.key)) {
				colorMapMenu.addItem(new KeyedRecord(map.key, map.name));
				didEdit = true;
			}
			colorMapMenu.setSelectedKey(map.key);
		}

		errorReporter.setTitle(saveTitle);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Called after model data is set or changed, colorCount may have changed.

	private void updatePanels(boolean newModel) {

		boolean e = !map.isPermanent;

		if (newModel) {
			nameField.setText(map.name);
		}
		AppController.setComponentEnabled(nameField, e);

		if (colorPanels.length < map.colorCount) {
			int newLength = colorPanels.length + ColorMap.COUNT_INCREMENT;
			ColorPanel[] newColorPanels = new ColorPanel[newLength];
			for (int i = 0; i < colorPanels.length; i++) {
				newColorPanels[i] = colorPanels[i];
			}
			colorPanels = newColorPanels;
		}

		for (int i = map.colorCount; i < colorPanels.length; i++) {
			if (null != colorPanels[i]) {
				colorPanels[i].setVisible(false);
			}
		}

		backgroundPanel.updatePanel(newModel);
		backgroundPanel.setEnabled(e);
		for (int i = 0; i < map.colorCount; i++) {
			if (null == colorPanels[i]) {
				colorPanels[i] = new ColorPanel(i);
				mapColorsPanel.add(colorPanels[i]);
				colorPanels[i].updatePanel(true);
			} else {
				if (!colorPanels[i].isVisible()) {
					colorPanels[i].setVisible(true);
					colorPanels[i].updatePanel(true);
				} else {
					colorPanels[i].updatePanel(newModel);
				}
			}
			colorPanels[i].setEnabled(e);
		}

		noServicePanel.updatePanel(newModel);
		noServicePanel.setEnabled(e);
		interferencePanel.updatePanel(newModel);
		interferencePanel.setEnabled(e);

		updateControls();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update controls.

	private void updateControls() {

		boolean e = !map.isPermanent;
		deleteButton.setEnabled(e);
		saveButton.setEnabled(e && didEdit);
		addButton.setEnabled(e && (map.colorCount < ColorMap.MAX_COUNT));
		removeButton.setEnabled(e && (map.colorCount > 1));
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void setDidEdit() {

		if (!didEdit && (null != map) && !map.isPermanent) {
			didEdit = true;
			saveButton.setEnabled(true);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doLoad(int mapKey) {

		updateMenu(false);

		switch (mapKey) {

			case LOAD_NEW: {
				map = new ColorMap(0);
				didEdit = true;
				break;
			}

			case LOAD_COPY: {
				map = map.copy();
				didEdit = true;
				break;
			}

			case LOAD_DEFAULT:
				mapKey = colorMapMenu.getSelectedKey();
			default: {
				map = ColorMap.getColorMap(getDbID(), mapKey, errorReporter);
				if (null == map) {
					map = new ColorMap(0);
					didEdit = true;
				} else {
					didEdit = false;
				}
				break;
			}
		}

		if (!colorMapMenu.containsKey(map.key)) {
			colorMapMenu.addItem(new KeyedRecord(map.key, map.name));
			didEdit = true;
		}
		colorMapMenu.setSelectedKey(map.key);

		updatePanels(true);
	}


	//-----------------------------------------------------------------------------------------------------------------

	private boolean doSave(boolean confirm) {

		if (!commitCurrentField()) {
			return false;
		}
		if (!didEdit || map.isPermanent) {
			return true;
		}

		String title = "Save Color Map";
		errorReporter.setTitle(title);

		if (confirm) {
			AppController.beep();
			int result = JOptionPane.showConfirmDialog(this, "Data modified, do you want to save the changes?",
				title, JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);
			if (JOptionPane.NO_OPTION == result) {
				return true;
			}
			if (JOptionPane.CANCEL_OPTION == result) {
				return false;
			}
		}

		boolean result = map.save(getDbID(), errorReporter);
		didEdit = !result;

		updateControls();

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doDelete() {

		if (map.isPermanent) {
			return;
		}

		if (map.key > 0) {
			if (!ColorMap.deleteColorMap(getDbID(), map.key, errorReporter)) {
				return;
			}
		}

		doLoad(LOAD_DEFAULT);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean cancel() {

		if (!isVisible()) {
			return true;
		}

		if (windowShouldClose()) {
			AppController.hideWindow(this);
			return true;
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillOpen() {

		blockActionsClear();

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				blockActionsStart();
				doLoad(LOAD_DEFAULT);
				blockActionsEnd();
			}
		});
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean windowShouldClose() {

		if (!isVisible()) {
			return true;
		}

		blockActionsStart();
		boolean result = doSave(true);
		blockActionsEnd();

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillClose() {

		if (!isVisible()) {
			return;
		}

		blockActionsSet();
		didEdit = false;
	}


	//=================================================================================================================
	// Panel to edit one color in a map.

	private class ColorPanel extends JPanel {

		private int colorIndex;

		private Color color;

		private JButton colorButton;
		private JCheckBox optionCheckBox;
		private JTextField levelField;
		private JLabel rangeLabel;

		private boolean enabled;


		//-------------------------------------------------------------------------------------------------------------

		private ColorPanel(int theIndex) {

			colorIndex = theIndex;
			color = ColorMap.DEFAULT_COLOR;
			enabled = true;

			setLayout(new FlowLayout(FlowLayout.LEFT));

			colorButton = new JButton(new ColorButtonIcon(theIndex));
			colorButton.setFocusable(false);
			colorButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					if (!commitCurrentField()) {
						return;
					}
					Color newColor = JColorChooser.showDialog(colorButton, "Pick Color: " + getDescription(), color);
					if (null != newColor) {
						color = newColor;
						switch (colorIndex) {
							case INDEX_CODE_NO_SERVICE:
								map.noServiceColor = newColor;
								break;
							case INDEX_CODE_INTERFERENCE:
								map.interferenceColor = newColor;
								break;
							case INDEX_CODE_BACKGROUND:
								map.backgroundColor = newColor;
								break;
							default:
								map.colors[colorIndex] = newColor;
								break;
						}
						colorButton.repaint();
						setDidEdit();
					}
				}
			});

			add(colorButton);

			switch (colorIndex) {

				case INDEX_CODE_NO_SERVICE: {
					optionCheckBox = new JCheckBox("Set color for no service");
					break;
				}

				case INDEX_CODE_INTERFERENCE: {
					optionCheckBox = new JCheckBox("Set color for interference");
					break;
				}

				case INDEX_CODE_BACKGROUND: {
					break;
				}

				default: {

					levelField = new JTextField(5);
					AppController.fixKeyBindings(levelField);

					levelField.addActionListener(new ActionListener() {
						public void actionPerformed(ActionEvent theEvent) {
							String title = "Edit Level";
							String str = levelField.getText().trim();
							if (str.length() > 0) {
								double d = map.levels[colorIndex];
								try {
									d = Double.parseDouble(str);
									d = Math.rint(d * 100.) / 100.;
								} catch (NumberFormatException ne) {
									errorReporter.reportValidationError(title,
										"The color change level must be a number");
								}
								if (d != map.levels[colorIndex]) {
									double minLevel, maxLevel;
									if (0 == colorIndex) {
										minLevel = ColorMap.MIN_LEVEL;
									} else {
										minLevel = map.levels[colorIndex - 1];
									}
									if ((map.colorCount - 1) == colorIndex) {
										maxLevel = ColorMap.MAX_LEVEL;
									} else {
										maxLevel = map.levels[colorIndex + 1];
									}
									if ((d <= minLevel) || (d >= maxLevel)) {
										errorReporter.reportValidationError(title,
											"The color change level must be between " + minLevel + " and " + maxLevel);
									} else {
										map.levels[colorIndex] = d;
										setDidEdit();
									}
								}
							}
							levelField.setText(AppCore.formatDecimal(map.levels[colorIndex], 2));
							if (0 == colorIndex) {
								backgroundPanel.updatePanel(false);
							} else {
								colorPanels[colorIndex - 1].updatePanel(false);
							}
						}
					});

					levelField.addFocusListener(new FocusAdapter() {
						public void focusGained(FocusEvent theEvent) {
							setCurrentField(levelField);
						}
						public void focusLost(FocusEvent theEvent) {
							if (!theEvent.isTemporary()) {
								levelField.postActionEvent();
							}
						}
					});

					add(levelField);

					break;
				}
			}

			if (null != optionCheckBox) {

				optionCheckBox.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (optionCheckBox.isSelected()) {
							colorButton.setEnabled(enabled);
							switch (colorIndex) {
								case INDEX_CODE_NO_SERVICE:
									map.noServiceColor = color;
									break;
								case INDEX_CODE_INTERFERENCE:
									map.interferenceColor = color;
									break;
							}
						} else {
							colorButton.setEnabled(false);
							switch (colorIndex) {
								case INDEX_CODE_NO_SERVICE:
									map.noServiceColor = null;
									break;
								case INDEX_CODE_INTERFERENCE:
									map.interferenceColor = null;
									break;
							}
						}
						setDidEdit();
					}
				});

				add(optionCheckBox);

			} else {

				rangeLabel = new JLabel();

				add(rangeLabel);
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		public void setEnabled(boolean e) {

			enabled = e;

			if (null != optionCheckBox) {
				AppController.setComponentEnabled(optionCheckBox, enabled);
				colorButton.setEnabled(enabled && optionCheckBox.isSelected());
			} else {
				colorButton.setEnabled(enabled);
			}
			if (null != levelField) {
				AppController.setComponentEnabled(levelField, enabled);
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		private void updatePanel(boolean newModel) {

			Color mapColor = null;

			switch (colorIndex) {

				case INDEX_CODE_NO_SERVICE: {
					mapColor = map.noServiceColor;
					if (null == mapColor) {
						optionCheckBox.setSelected(false);
					} else {
						optionCheckBox.setSelected(true);
					}
					break;
				}

				case INDEX_CODE_INTERFERENCE: {
					mapColor = map.interferenceColor;
					if (null == mapColor) {
						optionCheckBox.setSelected(false);
					} else {
						optionCheckBox.setSelected(true);
					}
					break;
				}

				case INDEX_CODE_BACKGROUND: {
					mapColor = map.backgroundColor;
					rangeLabel.setText(" below " + AppCore.formatDecimal(map.levels[0], 2));
					break;
				}

				default: {
					mapColor = map.colors[colorIndex];
					levelField.setText(AppCore.formatDecimal(map.levels[colorIndex], 2));
					if (colorIndex < (map.colorCount - 1)) {
						rangeLabel.setText(" to " + AppCore.formatDecimal(map.levels[colorIndex + 1], 2));
					} else {
						rangeLabel.setText(" and above");
					}
				}
			}

			colorButton.setEnabled(enabled && (null != mapColor));

			if (newModel) {

				if (null != mapColor) {
					color = mapColor;
				} else {
					color = ColorMap.DEFAULT_COLOR;
				}

				colorButton.repaint();
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		private String getDescription() {

			switch (colorIndex) {

				case INDEX_CODE_NO_SERVICE: {
					return "No service";
				}

				case INDEX_CODE_INTERFERENCE: {
					return "Interference";
				}

				case INDEX_CODE_BACKGROUND: {
					return "Below " + AppCore.formatDecimal(map.levels[0], 2);
				}

				default: {
					String str = AppCore.formatDecimal(map.levels[colorIndex], 2);
					if (colorIndex < (map.colorCount - 1)) {
						return str + " to " + AppCore.formatDecimal(map.levels[colorIndex + 1], 2);
					} else {
						return str + " and above";
					}
				}
			}
		}
	}


	//=================================================================================================================
	// Icon for color buttons, solid-filled rounded rectangle.

	private class ColorButtonIcon implements Icon {

		private int colorIndex;


		//-------------------------------------------------------------------------------------------------------------

		private ColorButtonIcon(int theIndex) {

			colorIndex = theIndex;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getIconWidth() {

			return COLOR_BUTTON_WIDTH;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getIconHeight() {

			return COLOR_BUTTON_HEIGHT;
		}


		//-------------------------------------------------------------------------------------------------------------

		public void paintIcon(Component c, Graphics g, int x, int y) {

			Color color;
			switch (colorIndex) {
				case INDEX_CODE_NO_SERVICE:
					color = map.noServiceColor;
					break;
				case INDEX_CODE_INTERFERENCE:
					color = map.interferenceColor;
					break;
				case INDEX_CODE_BACKGROUND:
					color = map.backgroundColor;
					break;
				default:
					color = map.colors[colorIndex];
					break;
			}

			if (null != color) {
				Graphics2D g2 = (Graphics2D)g;
				g2.setColor(color);
				double wid = (double)COLOR_BUTTON_WIDTH, hgt = (double)COLOR_BUTTON_HEIGHT, arc = wid / 8.;
				g2.fill(new RoundRectangle2D.Double((double)x, (double)y, wid, hgt, arc, arc));
			}
		}
	}
}
