//
//  SourceCompare.java
//  TVStudy
//
//  Copyright (c) 2018-2019 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.GroupLayout.*;


//=====================================================================================================================
// Class for comparing SourceEditData records, using objects returned by SourceEditData.getRecordData().  Up to 3
// objects can be compared.  This is a singleton class, the public API is static.

public class SourceCompare extends AppDialog {

	private static SourceCompare sourceCompare;

	private SourceEditData.RecordData referenceData;
	private SourceEditData.RecordData compare1Data;
	private SourceEditData.RecordData compare2Data;

	private JPanel panel;
	private GroupLayout layout;

	private ArrayList<JLabel> compare1Labels;
	private ArrayList<JLabel> compare2Labels;

	private GroupLayout.ParallelGroup labelGroup;
	private GroupLayout.ParallelGroup referenceGroup;
	private GroupLayout.ParallelGroup compare1Group;
	private GroupLayout.ParallelGroup compare2Group;
	private GroupLayout.SequentialGroup verticalGroup;

	private int labelIndex;


	//-----------------------------------------------------------------------------------------------------------------
	// Constructor is private, this is an application-wide singleton.

	private SourceCompare() {

		super(null, null, "Compare Records", Dialog.ModalityType.MODELESS);

		if (null != sourceCompare) {
			throw new RuntimeException("Comparison manager already exists");
		}
		sourceCompare = this;

		panel = new JPanel();

		add(AppController.createScrollPane(panel));

		setResizable(true);
		setLocationSaved(true);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getDbID() {

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public RootEditor getRootEditor() {

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// This method is the entire public API.  If the window is not showing, this will set the reference for comparison
	// and show the window.  Otherwise this adds data to be compared to the reference.  There are two comparison slots,
	// new data is put in slot 1, anything in slot 1 moves to slot 2, anything in slot 2 is removed.  The window must
	// be closed to start a new comparison.  The reference defines the properties compared, if comparison data has
	// additional properties those are not shown, any reference properties that are missing show "N/A" (highlighted as
	// a difference, of course).  Note this does not restrict the data being compared by record type; it probably does
	// not make sense to compare different types e.g. TV and FM, but it's the callers job to perform UI restrictions
	// needed to ensure that doesn't happen.

	public static boolean compare(SourceEditData source, ErrorReporter errors) {
		SourceEditData.RecordData data = source.getRecordData(errors);
		if (null == data) {
			return false;
		}
		compare(data);
		return true;
	}

	public static boolean compare(ExtDbRecord record, ErrorReporter errors) {
		SourceEditData source = SourceEditData.makeSource(record, null, true, errors);
		if (null == source) {
			return false;
		}
		SourceEditData.RecordData data = source.getRecordData(errors);
		if (null == data) {
			return false;
		}
		compare(data);
		return true;
	}

	public static void compare(SourceEditData.RecordData theData) {

		if (null == sourceCompare) {
			new SourceCompare();
		}

		if (sourceCompare.isVisible()) {

			if (null != sourceCompare.compare1Data) {
				sourceCompare.compare2Data = sourceCompare.compare1Data;
			}
			sourceCompare.compare1Data = theData;

			sourceCompare.updateAll();

			sourceCompare.toFront();

		} else {

			sourceCompare.referenceData = theData;
			sourceCompare.compare1Data = null;
			sourceCompare.compare2Data = null;

			sourceCompare.newLayout();

			AppController.showWindow(sourceCompare);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void newLayout() {

		JPanel mainP = new JPanel();
		layout = new GroupLayout(mainP);
		mainP.setLayout(layout);

		layout.setAutoCreateGaps(true);
		layout.setAutoCreateContainerGaps(true);

		labelGroup = layout.createParallelGroup(GroupLayout.Alignment.TRAILING);
		referenceGroup = layout.createParallelGroup();
		compare1Group = layout.createParallelGroup();
		compare2Group = layout.createParallelGroup();
		layout.setHorizontalGroup(layout.createSequentialGroup().
			addGroup(labelGroup).
			addGroup(referenceGroup).
			addGroup(compare1Group).
			addGroup(compare2Group));

		verticalGroup = layout.createSequentialGroup();
		layout.setVerticalGroup(verticalGroup);

		compare1Labels = new ArrayList<JLabel>();
		compare2Labels = new ArrayList<JLabel>();

		for (String key : referenceData.propertyValues.keySet()) {
			layoutOne(referenceData.propertyNames.get(key), referenceData.propertyValues.get(key));
		}
		if (null != referenceData.horizontalPattern) {
			layoutOne("Horizontal pattern data", "");
		}
		if (null != referenceData.verticalPattern) {
			layoutOne("Vertical pattern data", "");
		}

		if (null != referenceData.dependentData) {

			JLabel depL;
			SourceEditData.RecordData refD;

			for (Integer depKey : referenceData.dependentData.keySet()) {

				depL = new JLabel("Site " + String.valueOf(depKey));
				labelGroup.addComponent(depL);
				verticalGroup.addComponent(depL);

				refD = referenceData.dependentData.get(depKey);
				for (String key : refD.propertyValues.keySet()) {
					layoutOne(refD.propertyNames.get(key), refD.propertyValues.get(key));
				}
				if (null != refD.horizontalPattern) {
					layoutOne("Horizontal pattern data", "");
				}
				if (null != refD.verticalPattern) {
					layoutOne("Vertical pattern data", "");
				}
			}
		}

		panel.removeAll();
		panel.add(mainP);

		pack();
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void layoutOne(String labelS, String refS) {

		JLabel labelL = new JLabel(labelS);
		JLabel refL = new JLabel(refS);

		JLabel comp1L = new JLabel(" ");
		comp1L.setOpaque(true);
		comp1L.setPreferredSize(refL.getSize());
		compare1Labels.add(comp1L);

		JLabel comp2L = new JLabel(" ");
		comp2L.setPreferredSize(refL.getSize());
		comp2L.setOpaque(true);
		compare2Labels.add(comp2L);

		labelGroup.addComponent(labelL);
		referenceGroup.addComponent(refL);
		compare1Group.addComponent(comp1L);
		compare2Group.addComponent(comp2L);

		verticalGroup.addGroup(layout.createParallelGroup(GroupLayout.Alignment.BASELINE).
			addComponent(labelL).
			addComponent(refL).
			addComponent(comp1L).
			addComponent(comp2L));
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void updateAll() {

		if (!isVisible()) {
			return;
		}

		labelIndex = 0;

		updateOne(referenceData, compare1Data, compare2Data);

		if (null != referenceData.dependentData) {

			SourceEditData.RecordData refD, comp1D = null, comp2D = null;

			for (Integer depKey : referenceData.dependentData.keySet()) {

				refD = referenceData.dependentData.get(depKey);
				if ((null != compare1Data) && (null != compare1Data.dependentData)) {
					comp1D = compare1Data.dependentData.get(depKey);
				}
				if ((null != compare2Data) && (null != compare2Data.dependentData)) {
					comp2D = compare2Data.dependentData.get(depKey);
				}

				updateOne(refD, comp1D, comp2D);
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void updateOne(SourceEditData.RecordData refData, SourceEditData.RecordData comp1Data,
			SourceEditData.RecordData comp2Data) {

		Color matchC = AppController.DISABLED_TEXT_BACKGROUND;
		Color diffC = Color.PINK;

		String refS, comp1S, comp2S;
		Color comp1C, comp2C;
		JLabel comp1L, comp2L;

		for (String key : refData.propertyValues.keySet()) {

			refS = refData.propertyValues.get(key);

			if (null != compare1Data) {

				if (null != comp1Data) {
					comp1S = comp1Data.propertyValues.get(key);
					if (null == comp1S) {
						comp1S = "";
						comp1C = diffC;
					} else {
						if (refS.equals(comp1S)) {
							comp1C = matchC;
						} else {
							comp1C = diffC;
						}
					}
				} else {
					comp1S = "";
					comp1C = diffC;
				}

				comp1L = compare1Labels.get(labelIndex);
				comp1L.setText(comp1S);
				comp1L.setBackground(comp1C);
			}

			if (null != compare2Data) {

				if (null != comp2Data) {
					comp2S = comp2Data.propertyValues.get(key);
					if (null == comp2S) {
						comp2S = "";
						comp2C = diffC;
					} else {
						if (refS.equals(comp2S)) {
							comp2C = matchC;
						} else {
							comp2C = diffC;
						}
					}
				} else {
					comp2S = "";
					comp2C = diffC;
				}

				comp2L = compare2Labels.get(labelIndex);
				comp2L.setText(comp2S);
				comp2L.setBackground(comp2C);
			}

			labelIndex++;
		}

		if (null != refData.horizontalPattern) {

			if (null != compare1Data) {

				if (null != comp1Data) {
					if (null == comp1Data.horizontalPattern) {
						comp1S = "No pattern";
						comp1C = diffC;
					} else {
						if (refData.horizontalPattern.equalsPattern(comp1Data.horizontalPattern)) {
							comp1S = "Match";
							comp1C = matchC;
						} else {
							comp1S = "Does not match";
							comp1C = diffC;
						}
					}
				} else {
					comp1S = "";
					comp1C = diffC;
				}

				comp1L = compare1Labels.get(labelIndex);
				comp1L.setText(comp1S);
				comp1L.setBackground(comp1C);
			}

			if (null != compare2Data) {

				if (null != comp2Data) {
					if (null == comp2Data.horizontalPattern) {
						comp2S = "No pattern";
						comp2C = diffC;
					} else {
						if (refData.horizontalPattern.equalsPattern(comp2Data.horizontalPattern)) {
							comp2S = "Match";
							comp2C = matchC;
						} else {
							comp2S = "Does not match";
							comp2C = diffC;
						}
					}
				} else {
					comp2S = "";
					comp2C = diffC;
				}

				comp2L = compare2Labels.get(labelIndex);
				comp2L.setText(comp2S);
				comp2L.setBackground(comp2C);
			}

			labelIndex++;
		}

		if (null != refData.verticalPattern) {

			if (null != compare1Data) {

				if (null != comp1Data) {
					if (null == comp1Data.verticalPattern) {
						comp1S = "No pattern";
						comp1C = diffC;
					} else {
						if (refData.verticalPattern.equalsPattern(comp1Data.verticalPattern)) {
							comp1S = "Match";
							comp1C = matchC;
						} else {
							comp1S = "Does not match";
							comp1C = diffC;
						}
					}
				} else {
					comp1S = "";
					comp1C = diffC;
				}

				comp1L = compare1Labels.get(labelIndex);
				comp1L.setText(comp1S);
				comp1L.setBackground(comp1C);
			}

			if (null != compare2Data) {

				if (null != comp2Data) {
					if (null == comp2Data.verticalPattern) {
						comp2S = "No pattern";
						comp2C = diffC;
					} else {
						if (refData.verticalPattern.equalsPattern(comp2Data.verticalPattern)) {
							comp2S = "Match";
							comp2C = matchC;
						} else {
							comp2S = "Does not match";
							comp2C = diffC;
						}
					}
				} else {
					comp2S = "";
					comp2C = diffC;
				}

				comp2L = compare2Labels.get(labelIndex);
				comp2L.setText(comp2S);
				comp2L.setBackground(comp2C);
			}

			labelIndex++;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillOpen() {

		blockActionsClear();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillClose() {

		if (!isVisible()) {
			return;
		}

		panel.removeAll();

		referenceData = null;
		compare1Data = null;
		compare2Data = null;

		blockActionsSet();
	}
}
