//
//  GeoPointSet.java
//  TVStudy
//
//  Copyright (c) 2016-2020 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.core.geo;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;

import java.util.*;
import java.util.regex.*;
import java.sql.*;
import java.io.*;

import org.xml.sax.*;


//=====================================================================================================================
// Model for editing point sets for use in point-mode studies.

public class GeoPointSet extends Geography {

	// Used to set defaults in editor.

	public static final KeyedRecord GENERIC_ANTENNA = new KeyedRecord(0, AntPattern.GENERIC_ANTENNA_NAME);

	// type (super)  Always GEO_TYPE_POINT_SET.
	// key (super)   Key, null for a new set never saved.
	// name (super)  Set name.
	// points        Array of points.

	public ArrayList<StudyPoint> points;

	// Properties used when importing/exporting XML, which may optionally include receive antenna data.

	private boolean exportReceiveAntennas;

	private int nextAntennaKey = -1;
	private HashMap<Integer, AntPattern> importedAntennas;

	private HashMap<String, KeyedRecord> findImportAntennas;
	private HashMap<String, KeyedRecord> findDbAntennas;
	private HashSet<String> antennaNamesNotFound;


	//-----------------------------------------------------------------------------------------------------------------
	// See superclass.

	public GeoPointSet(String theDbID) {

		super(theDbID, GEO_TYPE_POINT_SET);

		points = new ArrayList<StudyPoint>();
	}


	//=================================================================================================================
	// StudyPoint class.  If antenna.key is 0, generic OET-69 antennas are used.  If useAntennaOrientation is true the
	// receive antenna orientation is fixed at the antennaOrientation value, otherwise the usual behavior of orienting
	// the receive antenna toward the current desired station being analyzed is used.

	public static class StudyPoint extends GeoPoint {

		public String name;
		public double receiveHeight;
		public KeyedRecord antenna;
		public boolean useAntennaOrientation;
		public double antennaOrientation;


		//-------------------------------------------------------------------------------------------------------------

		public StudyPoint duplicate() {

			StudyPoint newPoint = new StudyPoint();

			newPoint.name = name;
			newPoint.setLatLon(this);
			newPoint.receiveHeight = receiveHeight;
			newPoint.antenna = antenna;
			newPoint.useAntennaOrientation = useAntennaOrientation;
			newPoint.antennaOrientation = antennaOrientation;

			return newPoint;
		}


		//-------------------------------------------------------------------------------------------------------------

		public boolean isDataValid(ErrorLogger errors) {

			if (null == name) {
				name = "";
			}
			name = name.trim();
			if (name.length() > DbCore.NAME_MAX_LENGTH) {
				name = name.substring(0, DbCore.NAME_MAX_LENGTH);
			}

			if ((0. == latitude) || (0. == longitude)) {
				if (null != errors) {
					errors.reportValidationError("Point latitude and longitude must be provided");
				}
				return false;
			}

			if ((latitude < Source.LATITUDE_MIN) || (latitude > Source.LATITUDE_MAX)) {
				if (null != errors) {
					errors.reportValidationError("Bad point latitude, must be " + Source.LATITUDE_MIN + " to " +
						Source.LATITUDE_MAX + " degrees");
				}
				return false;
			}

			if ((longitude < Source.LONGITUDE_MIN) || (longitude > Source.LONGITUDE_MAX)) {
				if (null != errors) {
					errors.reportValidationError("Bad point longitude, must be " + Source.LONGITUDE_MIN + " to " +
						Source.LONGITUDE_MAX + " degrees");
				}
				return false;
			}

			if ((receiveHeight < MIN_RECEIVE_HEIGHT) || (receiveHeight > MAX_RECEIVE_HEIGHT)) {
				if (null != errors) {
					errors.reportValidationError("Bad point receive height, must be between " + MIN_RECEIVE_HEIGHT +
						" and " + MAX_RECEIVE_HEIGHT);
				}
				return false;
			}

			if (null == antenna) {
				if (null != errors) {
					errors.reportValidationError("Receive antenna must be selected");
				}
				return false;
			}

			if (useAntennaOrientation) {
				if ((antennaOrientation < 0.) || (antennaOrientation >= 360.)) {
					if (null != errors) {
						errors.reportValidationError("Bad antenna orientation, must be 0 to less than 360 degrees");
					}
					return false;
				}
			}

			return true;
		}


		//-------------------------------------------------------------------------------------------------------------
		// XML support.

		public void writeAttributes(Writer xml) throws IOException {

			super.writeAttributes(xml);

			xml.append(" NAME=\"");
			xml.append(AppCore.xmlclean(name));
			xml.append("\" HEIGHT=\"");
			xml.append(AppCore.formatHeight(receiveHeight));
			xml.append('"');

			if ((null != antenna) && (antenna.key > 0)) {
				xml.append(" ANTENNA=\"");
				xml.append(AppCore.xmlclean(antenna.name));
				xml.append('"');
			}

			if (useAntennaOrientation) {
				xml.append(" ORIENT=\"");
				xml.append(AppCore.formatAzimuth(antennaOrientation));
				xml.append('"');
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// The parent GeoPointSet will be used for antenna lookup, if no parent the antenna is always set to generic.

		public boolean parseAttributes(String element, Attributes attrs, ErrorLogger errors) {
			return parseAttributes(element, attrs, (GeoPointSet)null, errors);
		}

		private boolean parseAttributes(String element, Attributes attrs, GeoPointSet parent, ErrorLogger errors) {

			if (!super.parseAttributes(element, attrs, errors)) {
				return false;
			}

			name = attrs.getValue("NAME");
			if (null == name) {
				name = "";
			}
			name = name.trim();
			if (name.length() > DbCore.NAME_MAX_LENGTH) {
				name = name.substring(0, DbCore.NAME_MAX_LENGTH);
			}

			String str = attrs.getValue("HEIGHT");
			receiveHeight = MIN_RECEIVE_HEIGHT - 1.;
			if (null != str) {
				try {
					receiveHeight = Double.parseDouble(str);
				} catch (NumberFormatException nfe) {
				}
			}
			if ((receiveHeight < MIN_RECEIVE_HEIGHT) || (receiveHeight >= MAX_RECEIVE_HEIGHT)) {
				if (null != errors) {
					errors.reportError("Bad HEIGHT attribute in " + element + " tag");
				}
				return false;
			}

			str = attrs.getValue("ANTENNA");
			if (null != str) {
				str = str.trim();
				if (0 == str.length()) {
					str = null;
				}
			}
			if ((null != str) && (null != parent)) {
				antenna = parent.findAntenna(str, errors);
			} else {
				antenna = GENERIC_ANTENNA;
			}

			str = attrs.getValue("ORIENT");
			if (null != str) {
				useAntennaOrientation = true;
				antennaOrientation = -1.;
				try {
					antennaOrientation = Double.parseDouble(str);
				} catch (NumberFormatException nfe) {
				}
				if ((antennaOrientation < 0.) || (antennaOrientation >= 360.)) {
					if (null != errors) {
						errors.reportError("Bad ORIENT attribute in " + element + " tag");
					}
					return false;
				}
			}

			return true;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean loadData(ErrorLogger errors) {

		if (null == key) {
			return false;
		}

		DbConnection db = DbCore.connectDb(dbID, errors);
		if (null == db) {
			return false;
		}

		points = new ArrayList<StudyPoint>();
		StudyPoint point;
		String errmsg = null;

		try {

			db.query("SELECT " +
				"geo_point_set.point_name," +
				"geo_point_set.latitude," +
				"geo_point_set.longitude," +
				"geo_point_set.receive_height," +
				"geo_point_set.antenna_key," +
				"receive_antenna_index.name," +
				"geo_point_set.antenna_orientation " +
			"FROM " +
				"geo_point_set " +
				"LEFT JOIN receive_antenna_index USING (antenna_key) " +
			"WHERE " +
				"geo_key = " + key + " ORDER BY point_name");

			int antKey;
			String antName;

			while (db.next()) {

				point = new StudyPoint();

				point.name = db.getString(1);

				point.setLatLon(db.getDouble(2), db.getDouble(3));

				point.receiveHeight = db.getDouble(4);

				antKey = db.getInt(5);
				antName = db.getString(6);
				if ((0 == antKey) || (null == antName)) {
					point.antenna = GENERIC_ANTENNA;
				} else {
					point.antenna = new KeyedRecord(antKey, antName);
				}

				point.antennaOrientation = db.getDouble(7);
				if (point.antennaOrientation >= 0.) {
					point.useAntennaOrientation = true;
				} else {
					point.antennaOrientation = 0.;
				}

				points.add(point);
			}

			if (points.isEmpty()) {
				errmsg = "Point set geography data not found for key " + key;
			}

		} catch (SQLException se) {
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (null != errmsg) {
			if (null != errors) {
				errors.reportError(errmsg);
			}
			return false;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public Geography duplicate() {

		GeoPointSet newGeo = new GeoPointSet(dbID);

		super.duplicateTo(newGeo);

		if (null != points) {
			newGeo.points = new ArrayList<StudyPoint>();
			for (StudyPoint point : points) {
				newGeo.points.add(point.duplicate());
			}
		}

		return newGeo;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isDataValid(ErrorLogger errors) {

		if (!super.isDataValid(errors)) {
			return false;
		}

		if (points.size() < 1) {
			if (null != errors) {
				errors.reportValidationError("Point set must contain at least 1 point");
			}
			return false;
		}

		for (StudyPoint point : points) {
			if (!point.isDataValid(errors)) {
				return false;
			}
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save the point set.  See superclass.  This simply deletes all then inserts all, no individual point updates.
	// This also updates the key map indicating which receive antennas are in use by this geography.  First check for
	// receive antennas that need to be saved, however only save those actually still in use by study points.  Note the
	// superclass save() locked receive antenna tables so those can be used if needed, see AntPattern.saveAntenna().

	protected String saveGeography(DbConnection db) throws SQLException {

		String errmsg = null;

		if (null != importedAntennas) {

			HashSet<Integer> saveAntKeys = new HashSet<Integer>();
			for (StudyPoint thePoint : points) {
				if ((null != thePoint.antenna) && (thePoint.antenna.key < 0)) {
					saveAntKeys.add(Integer.valueOf(thePoint.antenna.key));
				}
			}

			// Before saving an antenna attempt to load an existing antenna by name, if found and the pattern is an
			// exact match use that, otherwise save the antenna (the name will be made unique by the save method if
			// needed).  In either case remove from the import map and update all points with the correct key and name.

			AntPattern ant, dbAnt;
			KeyedRecord newAnt;

			for (Integer antKey : saveAntKeys) {

				ant = importedAntennas.get(antKey);

				dbAnt = AntPattern.loadReceiveAntenna(dbID, db, ant.name);
				if ((null != dbAnt) && ant.equalsPattern(dbAnt)) {
					newAnt = new KeyedRecord(dbAnt.key.intValue(), dbAnt.name);
				} else {

					errmsg = ant.saveReceiveAntenna(db);
					if (null != errmsg) {
						return errmsg;
					}
					newAnt = new KeyedRecord(ant.key.intValue(), ant.name);
				}

				importedAntennas.remove(antKey);
				for (StudyPoint thePoint : points) {
					if ((null != thePoint.antenna) && (antKey.intValue() == thePoint.antenna.key)) {
						thePoint.antenna = newAnt;
					}
				}
			}

			importedAntennas = null;
		}

		errmsg = super.saveGeography(db, new GeoPoint(), 0., 0., 0.);
		if (null != errmsg) {
			return errmsg;
		}

		db.update("DELETE FROM geo_point_set WHERE geo_key = " + key);

		HashSet<Integer> antKeySet = new HashSet<Integer>();

		StringBuilder query = new StringBuilder("INSERT INTO geo_point_set VALUES");
		int startLength = query.length(), antKey;
		String sep = " (";

		for (StudyPoint point : points) {

			antKey = 0;
			if (null != point.antenna) {
				antKey = point.antenna.key;
				if (antKey > 0) {
					antKeySet.add(Integer.valueOf(antKey));
				} else {

					// This should never happen, but saving negative keys would be bad so make sure it can't.

					if (antKey < 0) {
						point.antenna = GENERIC_ANTENNA;
						antKey = 0;
					}
				}
			}

			query.append(sep);
			query.append(String.valueOf(key));
			query.append(",'");
			query.append(db.clean(point.name));
			query.append("',");
			query.append(String.valueOf(point.latitude));
			query.append(',');
			query.append(String.valueOf(point.longitude));
			query.append(',');
			query.append(String.valueOf(point.receiveHeight));
			query.append(',');
			query.append(String.valueOf(antKey));
			query.append(',');
			query.append(point.useAntennaOrientation ? String.valueOf(point.antennaOrientation) : "-1");

			if (query.length() > DbCore.MAX_QUERY_LENGTH) {
				query.append(')');
				db.update(query.toString());
				query.setLength(startLength);
				sep = " (";
			} else {
				sep = "),(";
			}
		}

		if (query.length() > startLength) {
			query.append(')');
			db.update(query.toString());
		}

		db.update("DELETE FROM geography_receive_antenna WHERE geo_key = " + key);

		if (!antKeySet.isEmpty()) {

			query = new StringBuilder("INSERT INTO geography_receive_antenna VALUES");
			startLength = query.length();
			sep = " (";

			for (Integer theKey : antKeySet) {

				query.append(sep);
				query.append(String.valueOf(key));
				query.append(',');
				query.append(String.valueOf(theKey));

				if (query.length() > DbCore.MAX_QUERY_LENGTH) {
					query.append(')');
					db.update(query.toString());
					query.setLength(startLength);
					sep = " (";
				} else {
					sep = "),(";
				}
			}

			if (query.length() > startLength) {
				query.append(')');
				db.update(query.toString());
			}
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// XML support.  This has a non-standard method for write with option to export receive antennas.

	public boolean writeToXML(Writer xml, boolean exportAntennas) {
		return writeToXML(xml, exportAntennas, null);
	}

	public boolean writeToXML(Writer xml, boolean exportAntennas, ErrorLogger errors) {
		exportReceiveAntennas = exportAntennas;
		return writeToXML(xml, errors);
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected boolean hasElements() {

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// If exporting antennas, collect all that are needed and write those first.  This will correctly handle the case
	// of antennas imported from XML but never saved, those have temporary keys with negative values.

	protected void writeElements(Writer xml) throws IOException {

		if (exportReceiveAntennas) {

			HashSet<Integer> exportKeys = new HashSet<Integer>();
			for (StudyPoint thePoint : points) {
				if ((null != thePoint.antenna) && (0 != thePoint.antenna.key)) {
					exportKeys.add(Integer.valueOf(thePoint.antenna.key));
				}
			}

			AntPattern ant;

			for (Integer theKey : exportKeys) {

				if (theKey.intValue() < 0) {
					if (null != importedAntennas) {
						ant = importedAntennas.get(theKey);
					} else {
						ant = null;
					}
				} else {
					ant = AntPattern.getReceiveAntenna(dbID, theKey.intValue());
				}

				if (null != ant) {

					xml.append("<ANTENNA NAME=\"");
					xml.append(AppCore.xmlclean(ant.name));

					xml.append("\" GAIN=\"");
					xml.append(String.valueOf(ant.gain));

					if (ant.isMatrix()) {

						xml.append("\" MULTIPAT>\n");
						for (AntPattern.AntSlice antSlice : ant.getSlices()) {
							for (AntPattern.AntPoint antPoint : antSlice.points) {
								xml.append(String.format(Locale.US, "%.6f,%.3f,%.4f\n", antSlice.value, antPoint.angle,
									antPoint.relativeField));
							}
						}

					} else {

						xml.append("\">\n");
						for (AntPattern.AntPoint antPoint : ant.getPoints()) {
							xml.append(String.format(Locale.US, "%.3f,%.4f\n", antPoint.angle, antPoint.relativeField));
						}
					}

					xml.append("</ANTENNA>\n");
				}
			}
		}

		for (StudyPoint thePoint : points) {
			xml.append("<POINT");
			thePoint.writeAttributes(xml);
			xml.append("/>\n");
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// If receive antennas are found those are cached locally and will be written to the database on save as needed.

	private Pattern patternParser = Pattern.compile("[,\\n]");

	protected boolean parseElement(String element, Attributes attrs, String content, ErrorLogger errors) {

		if (element.equals("ANTENNA")) {

			String antName = attrs.getValue("NAME");
			if (null != antName) {
				antName = antName.trim();
			}
			if ((null == antName) || (0 == antName.length())) {
				if (null != errors) {
					errors.reportError("Missing NAME attribute in " + element + " tag");
				}
				return false;
			}

			String str = attrs.getValue("GAIN");
			double antGain = AntPattern.GAIN_MIN;
			if (null != str) {
				try {
					antGain = Double.parseDouble(str);
				} catch (NumberFormatException nfe) {
					antGain = AntPattern.GAIN_MIN - 1.;
				}
				if ((antGain < AntPattern.GAIN_MIN) || (antGain > AntPattern.GAIN_MAX)) {
					if (null != errors) {
						errors.reportError("Bad GAIN attribute in " + element + " tag");
					}
					return false;
				}
			}

			boolean hasSlices = (null != attrs.getValue("MULTIPAT"));

			boolean bad = false;

			ArrayList<AntPattern.AntSlice> antSlices = new ArrayList<AntPattern.AntSlice>();
			ArrayList<AntPattern.AntPoint> antPoints = null;

			double freq, az, rf, lastFreq = AntPattern.FREQUENCY_MIN - 1.;

			String[] tokens = patternParser.split(content);
			if (hasSlices) {

				for (int i = 2; i < tokens.length; i += 3) {
					try {
						freq = Double.parseDouble(tokens[i - 2]);
						az = Double.parseDouble(tokens[i - 1]);
						rf = Double.parseDouble(tokens[i]);
					} catch (NumberFormatException nfe) {
						bad = true;
						break;
					}
					if (freq != lastFreq) {
						antPoints = new ArrayList<AntPattern.AntPoint>();
						antSlices.add(new AntPattern.AntSlice(freq, antPoints));
						lastFreq = freq;
					}
					antPoints.add(new AntPattern.AntPoint(az, rf));
				}

			} else {

				for (int i = 1; i < tokens.length; i += 2) {
					try {
						az = Double.parseDouble(tokens[i - 1]);
						rf = Double.parseDouble(tokens[i]);
					} catch (NumberFormatException nfe) {
						bad = true;
						break;
					}
					if (null == antPoints) {
						antPoints = new ArrayList<AntPattern.AntPoint>();
					}
					antPoints.add(new AntPattern.AntPoint(az, rf));
				}
			}

			if (null == antPoints) {
				bad = true;
			}

			AntPattern ant = null;
			if (!bad) {
				if (hasSlices) {
					ant = new AntPattern(dbID, antName, AntPattern.PATTERN_TYPE_RECEIVE, antSlices);
				} else {
					ant = new AntPattern(dbID, AntPattern.PATTERN_TYPE_RECEIVE, antName, antPoints);
				}
				ant.gain = antGain;
				bad = !ant.isDataValid();
			}

			if (bad) {
				if (null != errors) {
					errors.reportError("Bad data in " + element + " element");
				}
				return false;
			}

			// Temporary keys are assigned to imported antennas to support the editor, see getReceiveAntennaList().

			if (null == importedAntennas) {
				importedAntennas = new HashMap<Integer, AntPattern>();
				findImportAntennas = new HashMap<String, KeyedRecord>();
			}
			KeyedRecord theAnt = new KeyedRecord(nextAntennaKey--, ant.name + " (imported)");
			importedAntennas.put(Integer.valueOf(theAnt.key), ant);
			findImportAntennas.put(ant.name, theAnt);
		}

		if (element.equals("POINT")) {
			StudyPoint thePoint = new StudyPoint();
			if (!thePoint.parseAttributes(element, attrs, this, errors)) {
				return false;
			}
			points.add(thePoint);
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Find a receive antenna by name, used during XML import.  If antennas were imported from XML those are searched
	// first, that is an exact name match.  If not found there search current database contents, that is a case-
	// insensitive match.  Otherwise revert to generic and log a message once for each name not found.  This assumes
	// XML import will only happen once per instance so there is no mechanism to refresh the database antenna list.

	private KeyedRecord findAntenna(String theName, ErrorLogger errors) {

		KeyedRecord result = null;

		if (null != findImportAntennas) {
			result = findImportAntennas.get(theName);
		}

		if (null == result) {

			if (null == findDbAntennas) {
				findDbAntennas = new HashMap<String, KeyedRecord>();
				antennaNamesNotFound = new HashSet<String>();
				ArrayList<KeyedRecord> antList = AntPattern.getReceiveAntennaList(dbID);
				if (null != antList) {
					for (KeyedRecord ant : antList) {
						findDbAntennas.put(ant.name.toUpperCase(), ant);
					}
				}
				findDbAntennas.put(GENERIC_ANTENNA.name.toUpperCase(), GENERIC_ANTENNA);
			}

			result = findDbAntennas.get(theName.toUpperCase());
			if (null == result) {
				if ((null != errors) && antennaNamesNotFound.add(theName)) {
					errors.logMessage("Receive antenna '" + theName + "' not found, reverting to generic");
				}
				result = GENERIC_ANTENNA;
			}
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Return list of receive antennas for use in the editor, this gets the current database list and then may add
	// antennas imported from XML that have not yet been saved, those have temporary keys, see parseElement().

	public ArrayList<KeyedRecord> getReceiveAntennaList(ErrorLogger errors) {

		ArrayList<KeyedRecord> result = AntPattern.getReceiveAntennaList(dbID, errors);
		if ((null != result) && (null != importedAntennas)) {
			for (Map.Entry<Integer, AntPattern> e : importedAntennas.entrySet()) {
				result.add(new KeyedRecord(e.getKey().intValue(), e.getValue().name + " (imported)"));
			}
		}
		return result;
	}
}
