//
//  SourceEditData.java
//  TVStudy
//
//  Copyright (c) 2012-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.core.editdata;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.geo.*;

import java.util.*;
import java.util.regex.*;
import java.io.*;
import java.nio.file.*;
import java.sql.*;

import javax.xml.parsers.*;
import org.xml.sax.*;
import org.xml.sax.helpers.*;


//=====================================================================================================================
// Mutable model class to support editing of source records, see SourceEditor, also ScenarioEditor, SourceListData and
// StudyEditor, and to manipulate source data outside an existing study context.  This is an abstract superclass; a
// concrete subclass exists for each different record type, wrapping SourceTV, SourceWL, and SourceFM record objects.
// Core properties are immutable to ensure that arbitrary changes can't be made that would violate the object contract.
// The association of a source with a particular study and the source object primary key are immutable.  The key is a
// unique identifier throughout the models but is only unique within a particular study.  A source object can exist
// outside any study (the study property is null), but in that case the key is an arbitrary value unique only for the
// current application instance, and the object cannot be saved.  Properties such as the service and country, and for
// some record types the facility ID, are used by scenario-building logic that assumes those values are persistent and
// immutable, changing them directly could invalidate the state of existing scenarios.  The user record ID, or station
// data key and record ID, associate the source with a primary record that is presumed immutable.  That association
// cannot be changed arbitrarily, it implies the other immutable properties are identical to the underlying primary
// record, and if the isLocked property is true, that further implies ALL source properties are identical to the
// underlying record.  A source can be associated with a primary record in either locked or unlocked state, but
// unlocked sources are handled differently (e.g. see StudyEditData), and an unlocked source cannot be locked again
// since it may have been modified.  See the subclasses, particularily SourceEditDataTV, for more specific details.
// As with the Source class, common properties are here particularily pattern data.  The pattern data properties have
// special interpretation because the data is not automatically loaded with the record for performance reasons, it will
// be loaded on demand when needed by an editor.  So the pattern data properties are non-null only if the data has been
// loaded, in that case the *PatternChanged flags indicate if there were any actual changes.

// Source records have a general-purpose attributes field, represented in the database as a string in name=value
// format.  The attributes may be set on any object, even when locked.  Other code may use the attributes for any
// desired purpose.  For convenience and consistency names should be defined in Source.java, but those are not pre-
// determined or limited by either Source or this class.

public abstract class SourceEditData implements StationRecord {

	public final StudyEditData study;
	public final String dbID;

	public final int recordType;

	public final Integer key;
	public final Service service;
	public final Country country;
	public final boolean isLocked;
	public final Integer userRecordID;
	public final Integer extDbKey;
	public final String extRecordID;

	public String callSign;
	public String city;
	public String state;
	public String fileNumber;
	public final GeoPoint location;
	public double heightAMSL;
	public double overallHAAT;
	public double peakERP;
	public String antennaID;
	public boolean hasHorizontalPattern;
	public AntPattern horizontalPattern;
	public boolean horizontalPatternChanged;
	public double horizontalPatternOrientation;
	public boolean hasVerticalPattern;
	public AntPattern verticalPattern;
	public boolean verticalPatternChanged;
	public double verticalPatternElectricalTilt;
	public double verticalPatternMechanicalTilt;
	public double verticalPatternMechanicalTiltOrientation;
	public boolean hasMatrixPattern;
	public AntPattern matrixPattern;
	public boolean matrixPatternChanged;
	public boolean useGenericVerticalPattern;

	// Attributes are set and retrieved through methods.

	protected final HashMap<String, String> attributes;
	protected boolean attributesChanged;

	// A temporary sequence used for objects not associated with an existing study.

	private static int nextTemporaryKey = 1;


	//-----------------------------------------------------------------------------------------------------------------
	// Constructor for use by subclasses, sets only the final properties.

	protected SourceEditData(StudyEditData theStudy, String theDbID, int theRecordType, Integer theKey,
			Service theService, Country theCountry, boolean theIsLocked, Integer theUserRecordID, Integer theExtDbKey,
			String theExtRecordID) {

		super();

		study = theStudy;
		dbID = theDbID;

		recordType = theRecordType;

		key = theKey;
		service = theService;
		country = theCountry;
		isLocked = theIsLocked;
		userRecordID = theUserRecordID;
		extDbKey = theExtDbKey;
		extRecordID = theExtRecordID;

		location = new GeoPoint();

		attributes = new HashMap<String, String>();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Accessor for the underlying record object.  May be null.

	public abstract Source getSource();


	//-----------------------------------------------------------------------------------------------------------------
	// Create an object of appropriate subclass for an existing source record from a study database.

	public static SourceEditData getInstance(StudyEditData theStudy, Source theSource) {

		switch (theSource.recordType) {

			case Source.RECORD_TYPE_TV: {
				return new SourceEditDataTV(theStudy, (SourceTV)theSource);
			}

			case Source.RECORD_TYPE_WL: {
				return new SourceEditDataWL(theStudy, (SourceWL)theSource);
			}

			case Source.RECORD_TYPE_FM: {
				return new SourceEditDataFM(theStudy, (SourceFM)theSource);
			}
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Make a source object from an external record object.

	public static SourceEditData makeSource(ExtDbRecord record, StudyEditData study, boolean isLocked) {
		return makeSource(record, study, isLocked, null);
	}

	public static SourceEditData makeSource(ExtDbRecord record, StudyEditData study, boolean isLocked,
			ErrorLogger errors) {

		switch (record.recordType) {
			case Source.RECORD_TYPE_TV: {
				return SourceEditDataTV.makeSourceTV((ExtDbRecordTV)record, study, isLocked, errors);
			}
			case Source.RECORD_TYPE_FM: {
				return SourceEditDataFM.makeSourceFM((ExtDbRecordFM)record, study, isLocked, errors);
			}
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public int hashCode() {

		return key.hashCode();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean equals(Object other) {

		return (null != other) && ((SourceEditData)other).key.equals(key);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String toString() {

		return getCallSign() + " " + getChannel() + " " + getServiceCode() + " " + getStatus() + " " +
			getCity() + " " + getState();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get the next temporary key, used for sources not part of an existing study.

	protected static synchronized Integer getTemporaryKey() {

		return Integer.valueOf(nextTemporaryKey++);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Make an identical copy of an instance, used by editors to support cancel/undo actions.

	public abstract SourceEditData copy();


	//-----------------------------------------------------------------------------------------------------------------
	// Methods to derive a new source from an existing source, optionally changing the locked flag and study context.
	// See SourceEditDataTV for typical implementations.

	public SourceEditData deriveSource(boolean newIsLocked) {
		return deriveSource(study, newIsLocked, null);
	}

	public SourceEditData deriveSource(boolean newIsLocked, ErrorLogger errors) {
		return deriveSource(study, newIsLocked, errors);
	}

	public SourceEditData deriveSource(StudyEditData newStudy, boolean newIsLocked) {
		return deriveSource(newStudy, newIsLocked, null);
	}

	public abstract SourceEditData deriveSource(StudyEditData newStudy, boolean newIsLocked, ErrorLogger errors);


	//-----------------------------------------------------------------------------------------------------------------
	// Overridden by subclasses that have a geography property.

	public boolean isGeographyInUse(int theGeoKey) {
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public int getGeographyKey() {
		return 0;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Accessors for the attributes.  Attributes may or may not have a value; boolean attributes can simply be a test
	// for non-null return from getAttribute().

	public void setAttribute(String name) {

		attributes.put(name, "");
		attributesChanged = true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void setAttribute(String name, String value) {

		attributes.put(name, value);
		attributesChanged = true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getAttribute(String name) {

		return attributes.get(name);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String removeAttribute(String name) {

		attributesChanged = true;
		return attributes.remove(name);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public ArrayList<String> getAttributeKeys() {

		return new ArrayList<String>(attributes.keySet());
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Attributes can be set by copying from another map or parsing from string representation.  These are used by the
	// subclasses when constructing new objects, so attributesChanged is not set.  This may include all attributes, or
	// only non-transient attributes.  Transient attributes are relevant only in the study context where the source was
	// first created, and so are not included when exporting to XML.

	protected void setAllAttributes(HashMap<String, String> theAttrs) {

		attributes.clear();
		attributes.putAll(theAttrs);
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected void setAllAttributesNT(HashMap<String, String> theAttrs) {

		attributes.clear();
		for (Map.Entry<String, String> e : theAttrs.entrySet()) {
			if (!e.getKey().startsWith(Source.TRANSIENT_ATTR_PREFIX)) {
				attributes.put(e.getKey(), e.getValue());
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected void setAllAttributes(String theAttrs) {

		doParseAttributes(theAttrs, false);
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected void setAllAttributesNT(String theAttrs) {

		doParseAttributes(theAttrs, true);
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doParseAttributes(String theAttrs, boolean ntOnly) {

		attributes.clear();

		String name, value;
		int e;
		for (String attr : theAttrs.split("\\n")) {
			e = attr.indexOf('=');
			if (e > 0) {
				name = attr.substring(0, e).trim();
				value = attr.substring(e + 1).trim();
			} else {
				name = attr.trim();
				value = "";
			}
			if (ntOnly && name.startsWith(Source.TRANSIENT_ATTR_PREFIX)) {
				continue;
			}
			attributes.put(name, value);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get string representation of attributes, for save and export code.  Can include all, or only non-transient.

	protected String getAllAttributes() {

		return formatAttributes(false);
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected String getAllAttributesNT() {

		return formatAttributes(true);
	}


	//-----------------------------------------------------------------------------------------------------------------

	private String formatAttributes(boolean ntOnly) {

		if (attributes.isEmpty()) {
			return "";
		}

		StringBuilder result = new StringBuilder();
		String value;
		for (Map.Entry<String, String> e : attributes.entrySet()) {
			if (ntOnly && e.getKey().startsWith(Source.TRANSIENT_ATTR_PREFIX)) {
				continue;
			}
			result.append(e.getKey());
			value = e.getValue();
			if (value.length() > 0) {
				result.append('=');
				result.append(value);
			}
			result.append('\n');
		}

		return result.toString();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Copy only transient attributes from another source.

	public void copyTransientAttributes(SourceEditData theSource) {

		for (Map.Entry<String, String> e : theSource.attributes.entrySet()) {
			if (e.getKey().startsWith(Source.TRANSIENT_ATTR_PREFIX)) {
				attributes.put(e.getKey(), e.getValue());
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Accessor for the horizontal pattern, will load the pattern if needed, returns null if no pattern exists or if
	// an error occurs during retrieval.

	public AntPattern getHorizontalPattern(ErrorLogger errors) {

		if (hasHorizontalPattern) {
			if (null == horizontalPattern) {
				Source theSource = getSource();
				if (null != theSource) {
					horizontalPattern = theSource.getHorizontalPattern(errors);
				}
			}
			return horizontalPattern;
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Accessor for the vertical or matrix pattern, as above.  A record should never have both but if somehow that
	// were to occur the matrix is preferred.

	public AntPattern getVerticalPattern(ErrorLogger errors) {

		if (hasMatrixPattern) {
			if (null == matrixPattern) {
				Source theSource = getSource();
				if (null != theSource) {
					matrixPattern = theSource.getMatrixPattern(errors);
				}
			}
			return matrixPattern;
		}

		if (hasVerticalPattern) {
			if (null == verticalPattern) {
				Source theSource = getSource();
				if (null != theSource) {
					verticalPattern = theSource.getVerticalPattern(errors);
				}
			}
			return verticalPattern;
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Validity and change checks per usual pattern.  Subclasses will usually override to check additional properties,
	// if so they must call super if all other properties are valid, but if other invalidities are detected first this
	// does not have to be called, there are no side-effects here.  A locked object based on a database record object
	// may be assumed valid without checking, since the database object is always assumed valid.

	public boolean isDataValid() {
		return isDataValid(null);
	}

	public boolean isDataValid(ErrorLogger errors) {

		if (isLocked && (null != getSource())) {
			return true;
		}

		if (service.key < 1) {
			if (null != errors) {
				errors.reportValidationError("A service must be selected");
			}
			return false;
		}
		if (country.key < 1) {
			if (null != errors) {
				errors.reportValidationError("A country must be selected");
			}
			return false;
		}
		if (0 == callSign.length()) {
			if (null != errors) {
				errors.reportValidationError("A " + (Source.RECORD_TYPE_WL == recordType ? "site ID" : "call sign") +
					" must be provided");
			}
			return false;
		}

		// To detect failure to enter coordinate values on new data, 0 is illegal for latitude and longitude.  That
		// means legitimate coordinates exactly on the equator or central meridian are rejected.  But the odds of such
		// coordinates actually being needed are very very small, whereas the odds of a user forgetting some fields on
		// new data entry are not so small.  Easy workaround if ever needed is to just enter coordinate 0-00-00.01.

		if ((0. == location.latitude) || (0. == location.longitude)) {
			if (null != errors) {
				errors.reportValidationError("Latitude and longitude must be provided");
			}
			return false;
		}
		if ((location.latitude < Source.LATITUDE_MIN) || (location.latitude > Source.LATITUDE_MAX)) {
			if (null != errors) {
				errors.reportValidationError("Bad latitude, must be " + Source.LATITUDE_MIN + " to " +
					Source.LATITUDE_MAX + " degrees");
			}
			return false;
		}
		if ((location.longitude < Source.LONGITUDE_MIN) || (location.longitude > Source.LONGITUDE_MAX)) {
			if (null != errors) {
				errors.reportValidationError("Bad longitude, must be " + Source.LONGITUDE_MIN + " to " +
					Source.LONGITUDE_MAX + " degrees");
			}
			return false;
		}
		if ((heightAMSL < Source.HEIGHT_MIN) || (heightAMSL > Source.HEIGHT_MAX)) {
			if (null != errors) {
				errors.reportValidationError("Bad height AMSL, must be " + Source.HEIGHT_MIN + " to " +
					Source.HEIGHT_MAX);
			}
			return false;
		}
		if ((overallHAAT < Source.HEIGHT_MIN) || (overallHAAT > Source.HEIGHT_MAX)) {
			if (null != errors) {
				errors.reportValidationError("Bad HAAT, must be " + Source.HEIGHT_MIN + " to " + Source.HEIGHT_MAX);
			}
			return false;
		}
		if ((peakERP < Source.ERP_MIN) || (peakERP > Source.ERP_MAX)) {
			if (null != errors) {
				errors.reportValidationError("Bad ERP, must be " + Source.ERP_MIN + " to " + Source.ERP_MAX);
			}
			return false;
		}

		// Check the horizontal pattern.

		if (hasHorizontalPattern) {
			if (null != horizontalPattern) {
				if ((AntPattern.PATTERN_TYPE_HORIZONTAL != horizontalPattern.type) || !horizontalPattern.isSimple()) {
					if (null != errors) {
						errors.reportValidationError("Bad azimuth pattern, invalid object type or format");
					}
					return false;
				}
				if (!horizontalPattern.isDataValid(errors)) {
					return false;
				}
			}
		} else {
			horizontalPattern = null;
		}

		// Azimuth and elevation pattern orientation and tilt values are always checked even if there is currently no
		// actual pattern data set; these are preserved regardless.

		if ((horizontalPatternOrientation < 0.) || (horizontalPatternOrientation >= 360.)) {
			if (null != errors) {
				errors.reportValidationError("Bad azimuth pattern orientation, must be 0 to less than 360");
			}
			return false;
		}

		// Check vertical pattern similar to horizontal.

		if (hasVerticalPattern) {
			if (null != verticalPattern) {
				if ((AntPattern.PATTERN_TYPE_VERTICAL != verticalPattern.type) || !verticalPattern.isSimple()) {
					if (null != errors) {
						errors.reportValidationError("Bad elevation pattern, invalid object type or format");
					}
					return false;
				}
				if (!verticalPattern.isDataValid(errors)) {
					return false;
				}
			}
		} else {
			verticalPattern = null;
		}

		if ((verticalPatternElectricalTilt < AntPattern.TILT_MIN) ||
				(verticalPatternElectricalTilt > AntPattern.TILT_MAX)) {
			if (null != errors) {
				errors.reportValidationError("Bad elevation pattern electrical tilt, must be " + AntPattern.TILT_MIN +
					" to " + AntPattern.TILT_MAX);
			}
			return false;
		}
		if ((verticalPatternMechanicalTilt < AntPattern.TILT_MIN) ||
				(verticalPatternMechanicalTilt > AntPattern.TILT_MAX)) {
			if (null != errors) {
				errors.reportValidationError("Bad elevation pattern mechanical tilt, must be " + AntPattern.TILT_MIN +
					" to " + AntPattern.TILT_MAX);
			}
			return false;
		}
		if ((verticalPatternMechanicalTiltOrientation < 0.) ||
				(verticalPatternMechanicalTiltOrientation >= 360.)) {
			if (null != errors) {
				errors.reportValidationError("Bad elevation pattern mechanical tilt orientation, " +
					"must be 0 to less than 360");
			}
			return false;
		}

		// Check matrix pattern, as above.

		if (hasMatrixPattern) {
			if (null != matrixPattern) {
				if ((AntPattern.PATTERN_TYPE_VERTICAL != matrixPattern.type) || matrixPattern.isSimple()) {
					if (null != errors) {
						errors.reportValidationError("Bad elevation pattern, invalid object type or format");
					}
					return false;
				}
				if (!matrixPattern.isDataValid(errors)) {
					return false;
				}
			}
		} else {
			matrixPattern = null;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The attributes map can always be edited.  A record that does not have a backing database record object will
	// always report changed since it has never been saved.  Otherwise a locked record is non-editable and so cannot
	// have changes to individual properties.

	public boolean isDataChanged() {

		if (attributesChanged) {
			return true;
		}

		Source theSource = getSource();
		if (null == theSource) {
			return true;
		}
		if (isLocked) {
			return false;
		}

		if (service.key != theSource.service.key) {
			return true;
		}
		if (country.key != theSource.country.key) {
			return true;
		}
		if (!callSign.equals(theSource.callSign)) {
			return true;
		}
		if (!city.equals(theSource.city)) {
			return true;
		}
		if (!state.equals(theSource.state)) {
			return true;
		}
		if (!fileNumber.equals(theSource.fileNumber)) {
			return true;
		}
		if (!location.equals(theSource.location)) {
			return true;
		}
		if (heightAMSL != theSource.heightAMSL) {
			return true;
		}
		if (overallHAAT != theSource.overallHAAT) {
			return true;
		}
		if (peakERP != theSource.peakERP) {
			return true;
		}
		if (hasHorizontalPattern != theSource.hasHorizontalPattern) {
			return true;
		}
		if ((null != horizontalPattern) && !horizontalPattern.name.equals(theSource.horizontalPatternName)) {
			return true;
		}
		if (horizontalPatternChanged) {
			return true;
		}
		if (horizontalPatternOrientation != theSource.horizontalPatternOrientation) {
			return true;
		}
		if (hasVerticalPattern != theSource.hasVerticalPattern) {
			return true;
		}
		if ((null != verticalPattern) && !verticalPattern.name.equals(theSource.verticalPatternName)) {
			return true;
		}
		if (verticalPatternChanged) {
			return true;
		}
		if (verticalPatternElectricalTilt != theSource.verticalPatternElectricalTilt) {
			return true;
		}
		if (verticalPatternMechanicalTilt != theSource.verticalPatternMechanicalTilt) {
			return true;
		}
		if (verticalPatternMechanicalTiltOrientation != theSource.verticalPatternMechanicalTiltOrientation) {
			return true;
		}
		if (hasMatrixPattern != theSource.hasMatrixPattern) {
			return true;
		}
		if ((matrixPattern != null) && !matrixPattern.name.equals(theSource.matrixPatternName)) {
			return true;
		}
		if (matrixPatternChanged) {
			return true;
		}
		if (useGenericVerticalPattern != theSource.useGenericVerticalPattern) {
			return true;
		}

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save this record into source tables in a study, user record table, or import data set table.  Which of those is
	// current is up to the caller, this just needs an open database connection set to the correct context.  The code
	// in the subclasses can assume that isDataValid() and isDataChanged() have both previously been called, that is
	// the responsibility of the caller.  See e.g. StudyEditData.save().

	public abstract void save(DbConnection db) throws SQLException;


	//-----------------------------------------------------------------------------------------------------------------
	// Save pattern data, called from subclass save() implementations.

	protected void savePatterns(DbConnection db) throws SQLException {

		StringBuilder query;
		int startLength;
		String sep;

		if (horizontalPatternChanged) {

			db.update("DELETE FROM source_horizontal_pattern WHERE source_key=" + key);

			if (null != horizontalPattern) {

				query = new StringBuilder(
				"INSERT INTO source_horizontal_pattern (" +
					"source_key," +
					"azimuth," +
					"relative_field) " +
				"VALUES");
				startLength = query.length();
				sep = " (";
				for (AntPattern.AntPoint thePoint : horizontalPattern.getPoints()) {
					query.append(sep);
					query.append(String.valueOf(key));
					query.append(',');
					query.append(String.valueOf(thePoint.angle));
					query.append(',');
					query.append(String.valueOf(thePoint.relativeField));
					if (query.length() > DbCore.MAX_QUERY_LENGTH) {
						query.append(')');
						db.update(query.toString());
						query.setLength(startLength);
						sep = " (";
					} else {
						sep = "),(";
					}
				}
				if (query.length() > startLength) {
					query.append(')');
					db.update(query.toString());
				}
			}
		}

		if (verticalPatternChanged) {

			db.update("DELETE FROM source_vertical_pattern WHERE source_key=" + key);

			if (null != verticalPattern) {

				query = new StringBuilder(
				"INSERT INTO source_vertical_pattern (" +
					"source_key," +
					"depression_angle," +
					"relative_field) " +
				"VALUES");
				startLength = query.length();
				sep = " (";
				for (AntPattern.AntPoint thePoint : verticalPattern.getPoints()) {
					query.append(sep);
					query.append(String.valueOf(key));
					query.append(',');
					query.append(String.valueOf(thePoint.angle));
					query.append(',');
					query.append(String.valueOf(thePoint.relativeField));
					if (query.length() > DbCore.MAX_QUERY_LENGTH) {
						query.append(')');
						db.update(query.toString());
						query.setLength(startLength);
						sep = " (";
					} else {
						sep = "),(";
					}
				}
				if (query.length() > startLength) {
					query.append(')');
					db.update(query.toString());
				}
			}
		}

		if (matrixPatternChanged) {

			db.update("DELETE FROM source_matrix_pattern WHERE source_key=" + key);

			if (null != matrixPattern) {

				query = new StringBuilder(
				"INSERT INTO source_matrix_pattern (" +
					"source_key," +
					"azimuth," +
					"depression_angle," +
					"relative_field) " +
				"VALUES");
				startLength = query.length();
				sep = " (";
				for (AntPattern.AntSlice theSlice : matrixPattern.getSlices()) {
					for (AntPattern.AntPoint thePoint : theSlice.points) {
						query.append(sep);
						query.append(String.valueOf(key));
						query.append(',');
						query.append(String.valueOf(theSlice.value));
						query.append(',');
						query.append(String.valueOf(thePoint.angle));
						query.append(',');
						query.append(String.valueOf(thePoint.relativeField));
						if (query.length() > DbCore.MAX_QUERY_LENGTH) {
							query.append(')');
							db.update(query.toString());
							query.setLength(startLength);
							sep = " (";
						} else {
							sep = "),(";
						}
					}
				}
				if (query.length() > startLength) {
					query.append(')');
					db.update(query.toString());
				}
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set and retrieve a comment for a source.  This is not part of the main data model, it is stored directly in the
	// database.  This is a non-critical auxiliary function that has no concurrency protection and ignores errors.
	// Currently it only works for sources derived from user records, however it takes the source object as argument
	// so the capability may be expanded in the future with additional backing stores using various other keys.

	private static HashMap<String, HashMap<Integer, String>> userRecordCommentCaches =
		new HashMap<String, HashMap<Integer, String>>();

	public static void setSourceComment(SourceEditData theSource, String theComment) {

		if (null == theSource.userRecordID) {
			return;
		}

		if (null == theComment) {
			theComment = "";
		} else {
			theComment = theComment.trim();
		}

		HashMap<Integer, String> theCache = userRecordCommentCaches.get(theSource.dbID);
		if (null == theCache) {
			theCache = new HashMap<Integer, String>();
			userRecordCommentCaches.put(theSource.dbID, theCache);
		}
		theCache.put(theSource.userRecordID, theComment);

		DbConnection db = DbCore.connectDb(theSource.dbID);
		if (null != db) {
			try {
				db.update("UPDATE user_record SET comment = '" + db.clean(theComment) + "' WHERE user_record_id = " +
					theSource.userRecordID);
			} catch (SQLException se) {
				db.reportError(se);
			}
			DbCore.releaseDb(db);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// A not-found condition from the query puts an empty string in the cache.  Any error returns an empty string.

	public static String getSourceComment(SourceEditData theSource) {

		if (null == theSource.userRecordID) {
			return "";
		}

		HashMap<Integer, String> theCache = userRecordCommentCaches.get(theSource.dbID);
		if (null == theCache) {
			theCache = new HashMap<Integer, String>();
			userRecordCommentCaches.put(theSource.dbID, theCache);
		}

		String theComment = theCache.get(theSource.userRecordID);
		if (null != theComment) {
			return theComment;
		}

		theComment = "";

		DbConnection db = DbCore.connectDb(theSource.dbID);
		if (null != db) {
			try {
				db.query("SELECT comment FROM user_record WHERE user_record_id = " + theSource.userRecordID);
				if (db.next()) {
					theComment = db.getString(1);
				}
			} catch (SQLException se) {
				db.reportError(se);
			}
			DbCore.releaseDb(db);
		}

		theCache.put(theSource.userRecordID, theComment);

		return theComment;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save this source as a user record.  If the save is successful, a new source representing the user record is
	// returned.  That source will have a new user record ID and is always locked.  The new source may or may not have
	// a study context, if a study is provided it does not have to be the same as this source.  This is typically
	// called on a new source just created from user input, however it may be used on any source except a replication.
	// First do a validity check.

	public SourceEditData saveAsUserRecord(ErrorLogger errors) {
		return saveAsUserRecord(null, errors);
	}

	public SourceEditData saveAsUserRecord(StudyEditData theStudy, ErrorLogger errors) {

		if (isReplication()) {
			if (null != errors) {
				errors.reportError("Replication record cannot be saved as a user record");
			}
			return null;
		}

		if (!isDataValid(errors)) {
			return null;
		}

		// Create the XML data.  This needs an error logger for trapping errors in the writer, create one if needed.

		if (null != errors) {
			errors.clearErrors();
		} else {
			errors = new ErrorLogger(null, null);
		}

		StringWriter theWriter = new StringWriter();

		try {
			writeToXML(theWriter, errors);
		} catch (IOException e) {
			errors.reportError(e.toString());
		}

		if (errors.hasErrors()) {
			return null;
		}

		String sourceData = theWriter.toString();

		// Save the new data, assign a record ID in the process.  The full source data is saved as XML, however a
		// subset of source properties are also saved as separate fields in the record table to support SQL searches.
		// Note an empty comment is added to the cache, see getRecordComment().

		Integer theUserRecordID = null;
		boolean error = false;
		String errmsg = "";

		DbConnection db = DbCore.connectDb(dbID, errors);
		if (null != db) {

			HashMap<Integer, String> theCache = userRecordCommentCaches.get(dbID);
			if (null == theCache) {
				theCache = new HashMap<Integer, String>();
				userRecordCommentCaches.put(dbID, theCache);
			}

			try {

				db.update("LOCK TABLES user_record WRITE, user_record_id_sequence WRITE");

				db.update("UPDATE user_record_id_sequence SET user_record_id = user_record_id + 1");
				db.query("SELECT user_record_id FROM user_record_id_sequence");
				db.next();
				theUserRecordID = Integer.valueOf(db.getInt(1));

				int facID = 0, chan = 0;
				switch (recordType) {
					case Source.RECORD_TYPE_TV: {
						facID = ((SourceEditDataTV)this).facilityID;
						chan = ((SourceEditDataTV)this).channel;
						break;
					}
					case Source.RECORD_TYPE_FM: {
						facID = ((SourceEditDataFM)this).facilityID;
						chan = ((SourceEditDataFM)this).channel;
						break;
					}
				}

				db.update(
				"INSERT INTO user_record (" +
					"user_record_id," +
					"record_type," +
					"xml_data," +
					"facility_id," +
					"service_key," +
					"call_sign," +
					"status," +
					"channel," +
					"city," +
					"state," +
					"country," +
					"file_number," +
					"comment) " +
				"VALUES (" +
					theUserRecordID + "," +
					recordType + "," +
					"'" + db.clean(sourceData) + "'," +
					facID + "," +
					service.key + "," +
					"'" + db.clean(getCallSign()) + "'," +
					"'" + db.clean(getStatus()) + "'," +
					chan + "," +
					"'" + db.clean(getCity()) + "'," +
					"'" + db.clean(getState()) + "'," +
					"'" + db.clean(getCountryCode()) + "'," +
					"'" + db.clean(getFileNumber()) + "'," +
					"'')");

				theCache.put(theUserRecordID, "");

			} catch (SQLException se) {
				error = true;
				errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
				db.reportError(se);
			}

			try {
				db.update("UNLOCK TABLES");
			} catch (SQLException se) {
				db.reportError(se);
			}

			DbCore.releaseDb(db);

			if (error) {
				errors.reportError(errmsg);
				return null;
			}

		} else {
			return null;
		}

		return readSourceFromXML(dbID, new StringReader(sourceData), theStudy, theUserRecordID, errors);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Delete a user record.

	public static boolean deleteUserRecord(String theDbID, Integer theUserRecordID) {
		return deleteUserRecord(theDbID, theUserRecordID, null);
	}

	public static boolean deleteUserRecord(String theDbID, Integer theUserRecordID, ErrorLogger errors) {

		if (null == theUserRecordID) {
			return false;
		}

		boolean error = false;
		String errmsg = "";

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null != db) {

			try {

				db.update("LOCK TABLES user_record WRITE");

				db.update("DELETE FROM user_record WHERE user_record_id = " + theUserRecordID);

			} catch (SQLException se) {
				error = true;
				errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
				db.reportError(se);
			}

			try {
				db.update("UNLOCK TABLES");
			} catch (SQLException se) {
				db.reportError(se);
			}

			DbCore.releaseDb(db);

			if (error) {
				errors.reportError(errmsg);
				return false;
			}

			HashMap<Integer, String> theCache = userRecordCommentCaches.get(theDbID);
			if (null != theCache) {
				theCache.remove(theUserRecordID);
			}

			return true;

		} else {
			return false;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Do a search of the user record table.  Note errors parsing the XML are ignored here, those just cause the
	// record to not be included in the results.  This must always match a specific record type and is assumed to be
	// occurring outside any study context; user records are never added to a study en masse so a multi-record query
	// is always a pre-search in advance of the user selecting specific records to actually be added to a study.

	public static ArrayList<SourceEditData> findUserRecords(String theDbID, int recordType, String query) {
		return findUserRecords(theDbID, recordType, query, null);
	}

	public static ArrayList<SourceEditData> findUserRecords(String theDbID, int recordType, String query,
			ErrorLogger errors) {

		ArrayList<SourceEditData> results = new ArrayList<SourceEditData>();

		SourceEditData theSource;
		Integer theID;

		String whrStr;
		if ((null != query) && (query.length() > 0)) {
			whrStr =
			"WHERE " +
				"(" + query + ") AND ";
		} else {
			whrStr =
			"WHERE ";
		}

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null != db) {

			HashMap<Integer, String> theCache = userRecordCommentCaches.get(theDbID);
			if (null == theCache) {
				theCache = new HashMap<Integer, String>();
				userRecordCommentCaches.put(theDbID, theCache);
			}

			// The XML parser always needs a logger, use a dummy so it doesn't have to always create a temporary one.

			ErrorLogger xmlErrors = new ErrorLogger(null, null);

			try {

				db.query(
				"SELECT " +
					"user_record_id, " +
					"xml_data, " +
					"comment " +
				"FROM " +
					"user_record " +
				whrStr +
					"(record_type = " + recordType + ")");

				while (db.next()) {
					theID = Integer.valueOf(db.getInt(1));
					theSource = readSourceFromXML(theDbID, new StringReader(db.getString(2)), null, theID, xmlErrors);
					if (null != theSource) {
						results.add(theSource);
						theCache.put(theID, db.getString(3));
					}
				}

				DbCore.releaseDb(db);

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				results = null;
				db.reportError(errors, se);
			}
		}

		return results;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Load a source from the user record table.  Note record comments are included in the queries and added to the
	// comment cache directly, see getRecordComment().

	public static SourceEditData findUserRecord(String theDbID, Integer theUserRecordID) {
		return findUserRecord(theDbID, null, theUserRecordID, null);
	}

	public static SourceEditData findUserRecord(String theDbID, Integer theUserRecordID, ErrorLogger errors) {
		return findUserRecord(theDbID, null, theUserRecordID, errors);
	}

	public static SourceEditData findUserRecord(String theDbID, StudyEditData theStudy, Integer theUserRecordID) {
		return findUserRecord(theDbID, theStudy, theUserRecordID, null);
	}

	public static SourceEditData findUserRecord(String theDbID, StudyEditData theStudy, Integer theUserRecordID,
			ErrorLogger errors) {

		String sourceData = null;
		boolean notfound = false;

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null != db) {

			HashMap<Integer, String> theCache = userRecordCommentCaches.get(theDbID);
			if (null == theCache) {
				theCache = new HashMap<Integer, String>();
				userRecordCommentCaches.put(theDbID, theCache);
			}

			try {
				db.query("SELECT xml_data, comment FROM user_record WHERE user_record_id = " + theUserRecordID);
				if (db.next()) {
					sourceData = db.getString(1);
					theCache.put(theUserRecordID, db.getString(2));
				} else {
					notfound = true;
				}
				DbCore.releaseDb(db);
			} catch (SQLException se) {
				DbCore.releaseDb(db);
				DbConnection.reportError(errors, se);
				return null;
			}
		}

		if (notfound) {
			if (null != errors) {
				errors.reportError("User record not found for record ID '" + theUserRecordID + "'");
			}
			return null;
		}

		return readSourceFromXML(theDbID, new StringReader(sourceData), theStudy, theUserRecordID, errors);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Determine if a station data set is supported by code in this class.  Although there are distinct import data
	// set types for each record type, since all are stored using the same table structure there is no variation
	// in the query composition between record types hence this is entirely in the superclass.

	public static boolean isExtDbSupported(ExtDb extDb) {

		return (!extDb.deleted && extDb.isSupported() && extDb.isGeneric());
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Do a search of a generic import data set, which are stored using Source tables exactly as in a study database.
	// The query methods in Source are used to retrieve records which are then wrapped in SourceEditData objects.
	// Although stored using internal formats, these data sets are managed in the UI like other specific-format import
	// data sets hence the API is similar to those; see ExtDb and ExtDbRecord for details on the concepts and patterns.

	public static LinkedList<SourceEditData> findImportRecords(ExtDb extDb, int recordType, String query) {
		return findImportRecords(extDb, recordType, query, null, 0., 0., null);
	}

	public static LinkedList<SourceEditData> findImportRecords(ExtDb extDb, int recordType, String query,
			ErrorLogger errors) {
		return findImportRecords(extDb, recordType, query, null, 0., 0., errors);
	}

	public static LinkedList<SourceEditData> findImportRecords(ExtDb extDb, int recordType, String query,
			GeoPoint searchCenter, double searchRadius, double kmPerDegree) {
		return findImportRecords(extDb, recordType, query, searchCenter, searchRadius, kmPerDegree, null);
	}

	public static LinkedList<SourceEditData> findImportRecords(ExtDb extDb, int recordType, String query,
			GeoPoint searchCenter, double searchRadius, double kmPerDegree, ErrorLogger errors) {

		if (!isExtDbSupported(extDb) || !extDb.canProvide(recordType)) {
			return new LinkedList<SourceEditData>();
		}

		DbConnection db = extDb.connectDb(errors);
		if (null == db) {
			return null;
		}

		// First retrieve Source subclass objects matching the SQL, the distance search criteria can't be done in SQL
		// so that has to be done separately below.

		ArrayList<Source> theSources = new ArrayList<Source>();

		try {

			switch (recordType) {

				case Source.RECORD_TYPE_TV: {
					SourceTV.getSources(db, extDb.dbID, extDb.dbName, query, theSources);
					break;
				}

				case Source.RECORD_TYPE_WL: {
					SourceWL.getSources(db, extDb.dbID, extDb.dbName, query, theSources);
					break;
				}

				case Source.RECORD_TYPE_FM: {
					SourceFM.getSources(db, extDb.dbID, extDb.dbName, query, theSources);
					break;
				}
			}

			extDb.releaseDb(db);

		} catch (SQLException se) {
			extDb.releaseDb(db);
			DbConnection.reportError(errors, se);
			return null;
		}

		// Apply distance search as needed, create SourceEditData objects.

		LinkedList<SourceEditData> results = new LinkedList<SourceEditData>();

		SourceEditData newSource;
		boolean skip;

		for (Source theSource : theSources) {

			if ((null != searchCenter) && (searchRadius > 0.)) {
				skip = true;
				if (theSource.service.isDTS) {
					for (SourceTV dtsSource : ((SourceTV)theSource).dtsSources) {
						if (searchCenter.distanceTo(dtsSource.location, kmPerDegree) <= searchRadius) {
							skip = false;
							break;
						}
					}
				} else {
					if (searchCenter.distanceTo(theSource.location, kmPerDegree) <= searchRadius) {
						skip = false;
					}
				}
				if (skip) {
					continue;
				}
			}

			newSource = getInstance(null, theSource);
			if (null != newSource) {
				results.add(newSource);
			}
		}

		return results;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Load a specific source from a generic import record table by record ID.

	public static SourceEditData findImportRecord(ExtDb extDb, int recordType, String theRecordID) {
		return findImportRecord(extDb, recordType, null, theRecordID, null);
	}

	public static SourceEditData findImportRecord(ExtDb extDb, int recordType, String theRecordID, ErrorLogger errors) {
		return findImportRecord(extDb, recordType, null, theRecordID, errors);
	}

	public static SourceEditData findImportRecord(ExtDb extDb, int recordType, StudyEditData theStudy,
			String theRecordID) {
		return findImportRecord(extDb, recordType, theStudy, theRecordID, null);
	}

	public static SourceEditData findImportRecord(ExtDb extDb, int recordType, StudyEditData theStudy,
			String theRecordID, ErrorLogger errors) {

		if (!isExtDbSupported(extDb) || !extDb.canProvide(recordType)) {
			return null;
		}

		StringBuilder query = new StringBuilder();
		try {
			addRecordIDQuery(extDb.type, theRecordID, query, false);
		} catch (IllegalArgumentException ie) {
			if (null != errors) {
				errors.logMessage(ie.getMessage());
			}
			return null;
		}

		LinkedList<SourceEditData> theSources = findImportRecords(extDb, recordType, query.toString(), null, 0., 0.,
			errors);

		if (null == theSources) {
			return null;
		}
		if (theSources.isEmpty()) {
			if (null != errors) {
				errors.logMessage("Record not found for record ID '" + theRecordID + "'");
			}
			return null;
		}

		return theSources.getFirst();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Methods to support composing query clauses for findUserRecords() and findImportRecords().  These are patterned
	// after like-named methods in ExtDbRecord and use constants and some support methods from that class, see there
	// for details.  Note neither user nor import record tables are versioned so there is no version argument here.  If
	// the dbType is DB_TYPE_NOT_SET this is for the user record table, otherwise for a generic import data set.  There
	// are only a few minor difference in syntax between those two cases.  Generic import sets are stored in native
	// Source tables as in a study database, user records in a special root database table where most of the data is
	// embedded as XML, however the searchable field names in the user record table match those in the Source tables.
	// Also note these do not need to vary by record type since the storage format is the same for all, hence there is
	// no record type argument, and this does not need type-specific methods in the subclasses as in ExtDbRecord.

	public static boolean addRecordIDQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (ExtDb.DB_TYPE_NOT_SET == dbType) {
			int theID = 0;
			try {
				theID = Integer.parseInt(str);
			} catch (NumberFormatException e) {
				throw new IllegalArgumentException("The user record ID must be a number");
			}
			if (theID <= 0) {
				throw new IllegalArgumentException("The user record ID must be greater than 0");
			}
		}

		if (combine) {
			query.append(" AND ");
		}

		if (ExtDb.DB_TYPE_NOT_SET == dbType) {
			query.append("(user_record_id = ");
			query.append(str);
		} else {
			query.append("(ext_record_id = '");
			query.append(str);
			query.append("'");
		}
		query.append(')');

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Search by file number.

	public static boolean addFileNumberQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		String[] parts = ExtDbRecord.parseFileNumber(str);

		return addFileNumberQuery(dbType, parts[0], parts[1], query, combine);
	}

	public static boolean addFileNumberQuery(int dbType, String prefix, String arn, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		if (prefix.length() > 0) {
			query.append("(UPPER(file_number) LIKE '");
			query.append(DbConnection.clean(prefix.toUpperCase()));
			query.append('%');
			query.append(DbConnection.clean(arn.toUpperCase()));
		} else {
			query.append("(UPPER(file_number) LIKE '%");
			query.append(DbConnection.clean(arn.toUpperCase()));
		}
		query.append("')");

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query clause to search by facility ID.

	public static boolean addFacilityIDQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		int facilityID = 0;
		try {
			facilityID = Integer.parseInt(str);
		} catch (NumberFormatException ne) {
			throw new IllegalArgumentException("The facility ID must be a number");
		}

		return addFacilityIDQuery(dbType, facilityID, query, combine);
	}

	public static boolean addFacilityIDQuery(int dbType, int facilityID, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		query.append("(facility_id = ");
		query.append(String.valueOf(facilityID));
		query.append(')');

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query clause to search by service.

	public static boolean addServiceQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		Service theService = Service.getService(str);
		if (null == theService) {
			throw new IllegalArgumentException("Unknown service code");
		}

		return addServiceQuery(dbType, theService.key, query, combine);
	}

	public static boolean addServiceQuery(int dbType, int serviceKey, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		query.append("(service_key = ");
		query.append(String.valueOf(serviceKey));
		query.append(')');

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for call sign.

	public static boolean addCallSignQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		query.append("(UPPER(call_sign) REGEXP '^D*");
		query.append(DbConnection.clean(str.toUpperCase()));
		query.append(".*')");

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for a specific channel search.

	public static boolean addChannelQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {
		return addChannelQuery(dbType, str, 0, 0, query, combine);
	}

	public static boolean addChannelQuery(int dbType, String str, int minimumChannel, int maximumChannel,
			StringBuilder query, boolean combine) throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		int channel = 0;
		try {
			channel = Integer.parseInt(str);
		} catch (NumberFormatException e) {
			throw new IllegalArgumentException("The channel must be a number");
		}

		return addChannelQuery(dbType, channel, minimumChannel, maximumChannel, query, combine);
	}

	public static boolean addChannelQuery(int dbType, int channel, StringBuilder query, boolean combine)
			throws IllegalArgumentException {
		return addChannelQuery(dbType, channel, 0, 0, query, combine);
	}

	public static boolean addChannelQuery(int dbType, int channel, int minimumChannel, int maximumChannel,
			StringBuilder query, boolean combine) throws IllegalArgumentException {

		if ((minimumChannel > 0) && (maximumChannel > 0) &&
				((channel < minimumChannel) || (channel > maximumChannel))) {
			throw new IllegalArgumentException("The channel must be in the range " + minimumChannel + " to " +
				maximumChannel);
		}

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		query.append("(channel = ");
		query.append(String.valueOf(channel));
		query.append(')');

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for a channel range search.

	public static boolean addChannelRangeQuery(int dbType, int minimumChannel, int maximumChannel, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		query.append("(channel BETWEEN ");
		query.append(String.valueOf(minimumChannel));
		query.append(" AND ");
		query.append(String.valueOf(maximumChannel));
		query.append(')');

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for matching a list of channels.

	public static boolean addMultipleChannelQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		query.append("(channel IN ");
		query.append(str);
		query.append(')');

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for status.

	public static boolean addStatusQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		return addStatusQuery(dbType, ExtDbRecord.getStatusType(str), query, combine);
	}

	public static boolean addStatusQuery(int dbType, int statusType, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		switch (statusType) {

			case ExtDbRecord.STATUS_TYPE_STA:
			case ExtDbRecord.STATUS_TYPE_CP:
			case ExtDbRecord.STATUS_TYPE_LIC:
			case ExtDbRecord.STATUS_TYPE_APP:
			case ExtDbRecord.STATUS_TYPE_EXP:
			case ExtDbRecord.STATUS_TYPE_AMD: {
				break;
			}

			default: {
				throw new IllegalArgumentException("Unknown status code");
			}
		}

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		query.append("(UPPER(status) = '");
		query.append(ExtDbRecord.STATUS_CODES[statusType]);
		query.append("')");

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add query clause for city name search.

	public static boolean addCityQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		query.append("(UPPER(city) LIKE '%");
		query.append(DbConnection.clean(str.toUpperCase()).replace('*', '%'));
		query.append("%')");

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add query clause for state code search.

	public static boolean addStateQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		query.append("(UPPER(state) = '");
		query.append(DbConnection.clean(str.toUpperCase()));
		query.append("')");

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add clause for country search.

	public static boolean addCountryQuery(int dbType, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		Country country = Country.getCountry(str);
		if (null == country) {
			throw new IllegalArgumentException("Unknown country code");
		}

		return addCountryQuery(dbType, country, query, combine);
	}

	public static boolean addCountryQuery(int dbType, int countryKey, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		Country country = Country.getCountry(countryKey);
		if (null == country) {
			throw new IllegalArgumentException("Unknown country key");
		}

		return addCountryQuery(dbType, country, query, combine);
	}

	public static boolean addCountryQuery(int dbType, Country country, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if (!ExtDb.isGeneric(dbType) && (ExtDb.DB_TYPE_NOT_SET != dbType)) {
			throw new IllegalArgumentException(ExtDbRecord.BAD_TYPE_MESSAGE);
		}

		if (combine) {
			query.append(" AND ");
		}

		if (ExtDb.DB_TYPE_NOT_SET == dbType) {
			query.append("(UPPER(country) = '");
			query.append(country.countryCode);
			query.append("')");
		} else {
			query.append("(country_key = ");
			query.append(country.key);
			query.append(")");
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Encode source data as an XML description, usually as part of a scenario structure but this may also be used
	// for a standalone source export.  Locked sources based on station data records are usually exported by reference
	// using the record ID, without any other attributes.  Those will be re-loaded on import.  The station data key is
	// specific to the database installation so is not valid in an export context.  On import, external data references
	// are resolved against whatever station data is current for the import context.  That means the resulting source
	// may not be identical to that appearing in the export context, and import may fail if the export and import
	// contexts are based on different station data types.  However some cross-data cases can be resolved so the
	// decision is left up to the import code.  All of that is desired behavior for scenario exports since the purpose
	// is to move a scenario "in concept" from one study to another, not necessarily moving identical parameters.
	// Unlocked sources whether based on a data record or not, and locked sources not based on a data record, are
	// exported with all attributes and so are fully reconstructed on import.  However if an unlocked source was based
	// on a data record and is later reverted after import, it may revert to a different state than in the original
	// export context.  For a replication source, the original source is actually exported with an extra attribute
	// providing the replication channel.  On import, the original source is first reconstructed from the exported
	// element, then the replication source is re-derived from that.  If the standalone flag is true this is not part
	// of a scenario export.  In that case the desired/undesired flags are irrelevant and those arguments are ignored.
	// The exported record will always be unlocked regardless of it's actual state, hence all parameters will always
	// be explicit, and the record ID will never be included in any case.  When a standalone-exported record is
	// imported again it is always an isolated editable record, see RecordFind.  Note this is also used to store user
	// records internally, those are always locked but the parser will force the locked flag when it knows it is
	// parsing internal data for a user record.  A user record ID is never included in an export, that is valid only
	// for a specific database installation.

	public boolean writeToXML(Writer xml) throws IOException {
		return writeToXML(xml, null);
	}

	public boolean writeToXML(Writer xml, ErrorLogger errors) throws IOException {

		xml.append("<TVSTUDY VERSION=\"" + AppCore.XML_VERSION + "\">\n");
		boolean result = writeToXML(xml, true, false, false, errors);
		xml.append("</TVSTUDY>\n");

		return result;
	}

	public boolean writeToXML(Writer xml, boolean isDesiredFlag, boolean isUndesiredFlag) throws IOException {
		return writeToXML(xml, false, isDesiredFlag, isUndesiredFlag, null);
	}

	public boolean writeToXML(Writer xml, boolean isDesiredFlag, boolean isUndesiredFlag, ErrorLogger errors)
			throws IOException {
		return writeToXML(xml, false, isDesiredFlag, isUndesiredFlag, errors);
	}

	protected abstract boolean writeToXML(Writer xml, boolean standalone, boolean isDesiredFlag,
			boolean isUndesiredFlag, ErrorLogger errors) throws IOException;


	//-----------------------------------------------------------------------------------------------------------------
	// Write multiple isolated sources to an XML file.

	public static boolean writeSourcesToXML(Writer xml, ArrayList<SourceEditData> sources) throws IOException {
		return writeSourcesToXML(xml, sources, null);
	}

	public static boolean writeSourcesToXML(Writer xml, ArrayList<SourceEditData> sources, ErrorLogger errors)
			throws IOException {

		xml.append("<TVSTUDY VERSION=\"" + AppCore.XML_VERSION + "\">\n");
		boolean result = true;
		for (SourceEditData theSource : sources) {
			if (!theSource.writeToXML(xml, true, false, false, errors)) {
				result = false;
				break;
			}
		}
		xml.append("</TVSTUDY>\n");

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Called by the subclass implementation of writeToXML(), write out the superclass property attributes to a tag
	// being written for an element, then close the tag and add non-transient attributes (in the element content) and
	// nested pattern data elements as needed.  That may involve actually loading the pattern data if needed.

	protected void writeAttributes(Writer xml, ErrorLogger errors) throws IOException {

		location.writeAttributes(xml);

		xml.append(" HAMSL=\"" + AppCore.formatHeight(heightAMSL) + '"');
		xml.append(" HAAT=\"" + AppCore.formatHeight(overallHAAT) + '"');
		xml.append(" ERP=\"" + AppCore.formatERP(peakERP) + '"');
		if ((null != antennaID) && (antennaID.length() > 0)) {
			xml.append(" ANTENNA_ID=\"" + antennaID + '"');
		}
		xml.append(" USE_GENERIC=\"" + useGenericVerticalPattern + '"');

		// Orientation and tilt parameters are always included, they don't apply in all cases but the values are
		// preserved in case they might be relevant again if the patterns are changed.

		xml.append(" APAT_ORIENT=\"" + AppCore.formatAzimuth(horizontalPatternOrientation) + '"');
		xml.append(" EPAT_ETILT=\"" + AppCore.formatDepression(verticalPatternElectricalTilt) + '"');
		xml.append(" EPAT_MTILT=\"" + AppCore.formatDepression(verticalPatternMechanicalTilt) + '"');
		xml.append(" EPAT_ORIENT=\"" + AppCore.formatDepression(verticalPatternMechanicalTiltOrientation) + '"');

		// Try to load pattern data as needed.  If that fails an error is reported but the XML is still created with
		// the pattern(s) reverted to omni/generic.

		Source source = getSource();

		boolean hasHpat = hasHorizontalPattern;
		if (hasHpat) {
			if ((null == horizontalPattern) && (null != source)) {
				horizontalPattern = source.getHorizontalPattern(errors);
			}
			if (null == horizontalPattern) {
				hasHpat = false;
			}
		}
		xml.append(" HAS_APAT=\"" + hasHpat + '"');
		if (hasHpat) {
			xml.append(" APAT_NAME=\"" + AppCore.xmlclean(horizontalPattern.name) + '"');
		}

		boolean hasVpat = hasVerticalPattern && !hasMatrixPattern;
		if (hasVpat) {
			if ((null == verticalPattern) && (null != source)) {
				verticalPattern = source.getVerticalPattern(errors);
			}
			if (null == verticalPattern) {
				hasVpat = false;
			}
		}
		xml.append(" HAS_EPAT=\"" + hasVpat + '"');
		if (hasVpat) {
			xml.append(" EPAT_NAME=\"" + AppCore.xmlclean(verticalPattern.name) + '"');
		}

		boolean hasMpat = hasMatrixPattern;
		if (hasMpat) {
			if ((null == matrixPattern) && (null != source)) {
				matrixPattern = source.getMatrixPattern(errors);
			}
			if (null == matrixPattern) {
				hasMpat = false;
			}
		}
		xml.append(" HAS_MPAT=\"" + hasMpat + '"');
		if (hasMpat) {
			xml.append(" MPAT_NAME=\"" + AppCore.xmlclean(matrixPattern.name) + '"');
		}

		xml.append(">\n");

		// Write all non-transient attributes.

		xml.append(AppCore.xmlclean(getAllAttributesNT()));

		// Write pattern data elements as needed.

		if (hasHpat) {
			xml.append("<APAT>\n");
			for (AntPattern.AntPoint point : horizontalPattern.getPoints()) {
				xml.append(AppCore.formatAzimuth(point.angle));
				xml.append(',');
				xml.append(AppCore.formatRelativeField(point.relativeField));
				xml.append('\n');
			}
			xml.append("</APAT>\n");
		}

		if (hasVpat) {
			xml.append("<EPAT>\n");
			for (AntPattern.AntPoint point : verticalPattern.getPoints()) {
				xml.append(AppCore.formatDepression(point.angle));
				xml.append(',');
				xml.append(AppCore.formatRelativeField(point.relativeField));
				xml.append('\n');
			}
			xml.append("</EPAT>\n");
		}

		if (hasMpat) {
			xml.append("<MPAT>\n");
			for (AntPattern.AntSlice slice : matrixPattern.getSlices()) {
				for (AntPattern.AntPoint point : slice.points) {
					xml.append(AppCore.formatAzimuth(slice.value));
					xml.append(',');
					xml.append(AppCore.formatDepression(point.angle));
					xml.append(',');
					xml.append(AppCore.formatRelativeField(point.relativeField));
					xml.append('\n');
				}
			}
			xml.append("</MPAT>\n");
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Parse XML to obtain a user record, this is private, the only public way to obtain user records is with the
	// findUserRecord*() methods, or saveAsUserRecord().  Other classes use readSourcesFromXML().

	private static SourceEditData readSourceFromXML(String theDbID, Reader xml, StudyEditData theStudy,
			Integer theUserRecordID, ErrorLogger errors) {

		SourceXMLHandler handler = new SourceXMLHandler(theStudy, theDbID, theUserRecordID, errors);
		if (!AppCore.parseXML(xml, handler, errors)) {
			return null;
		}

		if (null == handler.source) {
			errors.reportError("Invalid XML data in user record");
			return null;
		}
		return handler.source;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Parse XML to obtain all source elements, optionally filtered by record or study type.  Create a temporary error
	// logger if one is not provided so parsing code doesn't have to worry about a null logger.  An external data set
	// key may be provided to use for resolving external record ID references found in the XML.

	public static ArrayList<SourceEditData> readSourcesFromXML(String theDbID, Reader xml, Integer theExtDbKey,
			Integer theAltExtDbKey, int theRecordType, int theStudyType) {
		return readSourcesFromXML(theDbID, xml, theExtDbKey, theAltExtDbKey, theRecordType, theStudyType, null);
	}

	public static ArrayList<SourceEditData> readSourcesFromXML(String theDbID, Reader xml, Integer theExtDbKey,
			Integer theAltExtDbKey, int theRecordType, int theStudyType, ErrorLogger errors) {

		SourceXMLHandler handler = new SourceXMLHandler(theDbID, theExtDbKey, theAltExtDbKey, theRecordType,
			theStudyType, errors);
		if (!AppCore.parseXML(xml, handler, errors)) {
			return null;
		}

		if ((null == handler.sources) || handler.sources.isEmpty()) {
			if (handler.hadStudy) {
				errors.reportWarning("No compatible records found");
			} else {
				errors.reportWarning("No recognized XML structure found");
			}
			return null;
		}
		return handler.sources;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Parse XML attributes to superclass properties, called under subclass makeSourceWithAttributes??().  The element
	// is passed for error messages as this may be parsing different containing elements.  Note the call sign attribute
	// is not handled here, even though that is a superclass property the subclass must handle that because it is not
	// always an explicit attribute.

	protected boolean parseAttributes(String element, Attributes attrs, ErrorLogger errors) {

		// Latitude and longitude are required.

		if (!location.parseAttributes(element, attrs, recordType, errors)) {
			return false;
		}

		// Height AMSL is required.  Note Source.HEIGHT_DERIVE code may appear here, but that is in the valid range.

		heightAMSL = Source.HEIGHT_MIN - 1.;
		String str = attrs.getValue("HAMSL");
		if (null != str) {
			try {
				heightAMSL = Double.parseDouble(str);
			} catch (NumberFormatException nfe) {
			}
		}
		if ((heightAMSL < Source.HEIGHT_MIN) || (heightAMSL > Source.HEIGHT_MAX)) {
			if (null != errors) {
				errors.reportError("Missing or bad HAMSL attribute in " + element + " tag");
			}
			return false;
		}

		// HAAT is not required, zero is valid.

		str = attrs.getValue("HAAT");
		if (null != str) {
			overallHAAT = Source.HEIGHT_MIN - 1.;
			try {
				overallHAAT = Double.parseDouble(str);
			} catch (NumberFormatException nfe) {
			}
			if ((overallHAAT < Source.HEIGHT_MIN) || (overallHAAT > Source.HEIGHT_MAX)) {
				if (null != errors) {
					errors.reportError("Bad HAAT attribute in " + element + " tag");
				}
				return false;
			}
		}

		// Peak ERP is required.

		peakERP = Source.ERP_MIN - 1.;
		str = attrs.getValue("ERP");
		if (null != str) {
			try {
				peakERP = Double.parseDouble(str);
			} catch (NumberFormatException nfe) {
			}
		}
		if ((peakERP < Source.ERP_MIN) || (peakERP > Source.ERP_MAX)) {
			if (null != errors) {
				errors.reportError("Missing or bad ERP attribute in " + element + " tag");
			}
			return false;
		}

		// Antenna ID is not required, null is valid.

		antennaID = attrs.getValue("ANTENNA_ID");

		// Azimuth pattern orientation is not required, any number is valid, do a modulo 360.

		str = attrs.getValue("APAT_ORIENT");
		if (null != str) {
			try {
				horizontalPatternOrientation = Math.IEEEremainder(Double.parseDouble(str), 360.);
				if (horizontalPatternOrientation < 0.) horizontalPatternOrientation += 360.;
			} catch (NumberFormatException nfe) {
				if (null != errors) {
					errors.reportError("Bad APAT_ORIENT attribute in " + element + " tag");
				}
				return false;
			}
		}

		// Elevation pattern tilt values are not required, zero is valid.

		str = attrs.getValue("EPAT_ETILT");
		if (null != str) {
			verticalPatternElectricalTilt = AntPattern.TILT_MIN - 1.;
			try {
				verticalPatternElectricalTilt = Double.parseDouble(str);
			} catch (NumberFormatException nfe) {
			}
			if ((verticalPatternElectricalTilt < AntPattern.TILT_MIN) ||
					(verticalPatternElectricalTilt > AntPattern.TILT_MAX)) {
				if (null != errors) {
					errors.reportError("Bad EPAT_ETILT attribute in " + element + " tag");
				}
				return false;
			}
		}

		str = attrs.getValue("EPAT_MTILT");
		if (null != str) {
			verticalPatternMechanicalTilt = AntPattern.TILT_MIN - 1.;
			try {
				verticalPatternMechanicalTilt = Double.parseDouble(str);
			} catch (NumberFormatException nfe) {
			}
			if ((verticalPatternMechanicalTilt < AntPattern.TILT_MIN) ||
					(verticalPatternMechanicalTilt > AntPattern.TILT_MAX)) {
				if (null != errors) {
					errors.reportError("Bad EPAT_MTILT attribute in " + element + " tag");
				}
				return false;
			}
		}

		// Elevation pattern mechanical tilt orientation is required if tilt is non-zero, else is not required.

		str = attrs.getValue("EPAT_ORIENT");
		if (null != str) {
			try {
				verticalPatternMechanicalTiltOrientation = Math.IEEEremainder(Double.parseDouble(str), 360.);
				if (verticalPatternMechanicalTiltOrientation < 0.)
					verticalPatternMechanicalTiltOrientation += 360.;
			} catch (NumberFormatException nfe) {
				if (null != errors) {
					errors.reportError("Bad EPAT_ORIENT attribute in " + element + " tag");
				}
				return false;
			}
		} else {
			if (0. != verticalPatternMechanicalTilt) {
				if (null != errors) {
					errors.reportError("Missing EPAT_ORIENT attribute in " + element + " tag");
				}
				return false;
			}
		}

		// Generic pattern flag is not required, default to true.

		str = attrs.getValue("USE_GENERIC");
		useGenericVerticalPattern = ((null == str) || Boolean.parseBoolean(str));

		// Pattern flags are not required, default to false.  However a matrix pattern is mutually-exclusive with a
		// vertical pattern so the matrix pattern flag gets priority.

		str = attrs.getValue("HAS_APAT");
		hasHorizontalPattern = ((null != str) && Boolean.parseBoolean(str));

		str = attrs.getValue("HAS_MPAT");
		hasMatrixPattern = ((null != str) && Boolean.parseBoolean(str));

		if (!hasMatrixPattern) {
			str = attrs.getValue("HAS_EPAT");
			hasVerticalPattern = ((null != str) && Boolean.parseBoolean(str));
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Build comment text intended to appear as a tool-tip pop-up in various display tables.  This shows items from
	// the original data set record that were put in attributes, comment lines provided by subclass override of the
	// getComments() method, and comment text for user records.

	public String makeCommentText() {

		StringBuilder s = new StringBuilder();
		String pfx = "<HTML>";

		ArrayList<String> comments = getComments();
		if ((null != comments) && !comments.isEmpty()) {
			for (String theComment : comments) {
				s.append(pfx);
				s.append(theComment);
				pfx = "<BR>";
			}
		}

		String str = getSourceComment(this);
		if ((null != str) && (str.length() > 0)) {
			s.append(pfx);
			String[] words = str.split("\\s+");
			int wl, ll = 0;
			for (int i = 0; i < words.length; i++) {
				wl = words[i].length(); 
				if (wl > 0) {
					s.append(words[i]);
					ll += wl;
					if (ll > 35) {
						s.append("<BR>");
						ll = 0;
					} else {
						s.append(" ");
					}
				}
			}
		}

		if (s.length() > 0) {
			s.append("</HTML>");
			return s.toString();
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// May be overridden by subclass to provide lines for the comment text.

	protected ArrayList<String> getComments() {

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Many methods in the StationRecord interface will be overridden by the subclasses.

	public String getRecordType() {

		if (null != userRecordID) {
			return "User " + Source.getRecordTypeName(recordType);
		}
		if (null != extDbKey) {
			return ExtDb.getExtDbTypeName(dbID, extDbKey);
		}
		return "New " + Source.getRecordTypeName(recordType);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isSource() {

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean hasRecordID() {

		return ((null != userRecordID) || ((null != extDbKey) && (null != extRecordID)));
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isReplication() {

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getStationData() {

		if (null != userRecordID) {
			return "User record";
		}
		if (null != extDbKey) {
			return ExtDb.getExtDbDescription(dbID, extDbKey);
		}
		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getRecordID() {

		if (null != userRecordID) {
			return String.valueOf(userRecordID);
		}
		if ((null != extDbKey) && (null != extRecordID)) {
			return extRecordID;
		}
		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFacilityID() {

		return "";
	}

	public String getSortFacilityID() {

		return "0";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getService() {

		return service.name;
	}

	public String getServiceCode() {

		return service.serviceCode;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isDigital() {

		return service.serviceType.digital;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isDTS() {

		return service.isDTS;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getSiteCount() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getSiteNumber() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getCallSign() {

		return callSign;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getChannel() {

		return "";
	}

	public int getChannelNumber() {

		return 0;
	}

	public String getSortChannel() {

		return "0";
	}

	public String getOriginalChannel() {

		return getChannel();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFrequency() {

		return "";
	}

	public double getFrequencyValue() {

		return 0.;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getCity() {

		return city;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getState() {

		return state;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getCountry() {

		return country.name;
	}

	public String getCountryCode() {

		return country.countryCode;
	}

	public String getSortCountry() {

		return String.valueOf(country.key);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getZone() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getStatus() {

		return "";
	}

	public String getSortStatus() {

		return "0";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFileNumber() {

		return fileNumber;
	}

	public String getARN() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getSequenceDate() {

		String s = getAttribute(Source.ATTR_SEQUENCE_DATE);
		if (null != s) {
			return s;
		}
		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getSortSequenceDate() {

		java.util.Date d = AppCore.parseDate(getAttribute(Source.ATTR_SEQUENCE_DATE));
		if (null != d) {
			return String.format(Locale.US, "%013d", d.getTime());
		}
		return "9999999999999";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFrequencyOffset() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getEmissionMask() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getLatitude() {

		return AppCore.formatLatitude(location.latitude);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getLongitude() {

		return AppCore.formatLongitude(location.longitude);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getHeightAMSL() {

		if (Source.HEIGHT_DERIVE == heightAMSL) {
			return Source.HEIGHT_DERIVE_LABEL;
		}
		return AppCore.formatHeight(heightAMSL) + " m";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getOverallHAAT() {

		if (Source.HEIGHT_DERIVE == overallHAAT) {
			return Source.HEIGHT_DERIVE_LABEL;
		}
		return AppCore.formatHeight(overallHAAT) + " m";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getPeakERP() {

		return AppCore.formatERP(peakERP) + " kW";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean hasHorizontalPattern() {

		return hasHorizontalPattern;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getHorizontalPatternName() {

		if (hasHorizontalPattern) {
			String theName = "unknown";
			if (null != horizontalPattern) {
				theName = horizontalPattern.name;
			} else {
				Source theSource = getSource();
				if (null != theSource) {
					theName = theSource.horizontalPatternName;
				}
			}
			if ((null != antennaID) && (antennaID.length() > 0)) {
				return theName + " (ID " + antennaID + ")";
			}
			return theName;
		}
		return "Omnidirectional";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getHorizontalPatternOrientation() {

		if (hasHorizontalPattern) {
			return AppCore.formatAzimuth(horizontalPatternOrientation) + " deg";
		}
		return "";
	}


	//=================================================================================================================
	// Class and method to provide properties based on the StationRecord methods, suitable for generic UI.  This
	// primarily exists to support the SourceCompare class, see that for details.  Property values are returned in a
	// TreeMap using the keys below, which provide a natural ordering appropriate to sequence properties in the UI.
	// Copies of antenna pattern data may also be provided, and for subclasses that have dependent sources (i.e. DTS)
	// a map of data for those objects is provided.  A static property also provides descriptive names.  Note the
	// values map does not necessarily have entries for all keys, it will only contain those properties which have
	// defined values in the subclass instance.

	public static class RecordData {

		public static final String KEY_CALL_SIGN = "110callSign";
		public static final String KEY_SERVICE = "120service";
		public static final String KEY_CHANNEL = "130channel";
		public static final String KEY_STATUS = "140status";
		public static final String KEY_CITY = "150city";
		public static final String KEY_STATE = "160state";
		public static final String KEY_COUNTRY = "170country";
		public static final String KEY_FACILITY_ID = "180facilityID";
		public static final String KEY_FILE_NUMBER = "190fileNumber";
		public static final String KEY_SITE_COUNT = "200siteCount";
		public static final String KEY_SITE_NUMBER = "210siteNumber";
		public static final String KEY_LATITUDE = "220latitude";
		public static final String KEY_LONGITUDE = "230longitude";
		public static final String KEY_ZONE = "240zone";
		public static final String KEY_DTS_MAXIMUM_DISTANCE = "250dtsMaximumDistance";
		public static final String KEY_HEIGHT_AMSL = "260heightAMSL";
		public static final String KEY_OVERALL_HAAT = "270overallHAAT";
		public static final String KEY_PEAK_ERP = "280peakERP";
		public static final String KEY_FREQUENCY_OFFSET = "290frequencyOffset";
		public static final String KEY_EMISSION_MASK = "300emissionMask";
		public static final String KEY_DTS_TIME_DELAY = "310dtsTimeDelay";
		public static final String KEY_RECORD_TYPE = "320recordType";
		public static final String KEY_STATION_DATA = "330stationData";
		public static final String KEY_RECORD_ID = "340recordID";
		public static final String KEY_SEQUENCE_DATE = "350sequenceDate";
		public static final String KEY_HORIZONTAL_PATTERN_NAME = "360horizontalPatternName";
		public static final String KEY_HORIZONTAL_PATTERN_ORIENTATION = "370horizontalPatternOrientation";
		public static final String KEY_VERTICAL_PATTERN_NAME = "380verticalPatternName";
		public static final String KEY_VERTICAL_PATTERN_ELECTRICAL_TILT = "390verticalPatternElectricalTilt";
		public static final String KEY_VERTICAL_PATTERN_MECHANICAL_TILT = "400verticalPatternMechanicalTilt";
		public static final String KEY_VERTICAL_PATTERN_MECHANICAL_TILT_ORIENTATION =
			"410verticalPatternMechanicalTiltOrientation";

		public static final HashMap<String, String> propertyNames;

		static {

			propertyNames = new HashMap<String, String>();

			propertyNames.put(KEY_RECORD_TYPE, "Record type");
			propertyNames.put(KEY_STATION_DATA, "Station data");
			propertyNames.put(KEY_RECORD_ID, "Record ID");
			propertyNames.put(KEY_FILE_NUMBER, "File number");
			propertyNames.put(KEY_SEQUENCE_DATE, "Date");
			propertyNames.put(KEY_SERVICE, "Service");
			propertyNames.put(KEY_CALL_SIGN, "Call sign");
			propertyNames.put(KEY_CHANNEL, "Channel");
			propertyNames.put(KEY_STATUS, "Status");
			propertyNames.put(KEY_CITY, "City");
			propertyNames.put(KEY_STATE, "State");
			propertyNames.put(KEY_COUNTRY, "Country");
			propertyNames.put(KEY_FACILITY_ID, "Facility ID");
			propertyNames.put(KEY_SITE_COUNT, "Site count");
			propertyNames.put(KEY_SITE_NUMBER, "Site number");
			propertyNames.put(KEY_LATITUDE, "Latitude");
			propertyNames.put(KEY_LONGITUDE, "Longitude");
			propertyNames.put(KEY_ZONE, "Zone");
			propertyNames.put(KEY_DTS_MAXIMUM_DISTANCE, "DTS limit distance");
			propertyNames.put(KEY_HEIGHT_AMSL, "Height AMSL");
			propertyNames.put(KEY_OVERALL_HAAT, "HAAT");
			propertyNames.put(KEY_PEAK_ERP, "Peak ERP");
			propertyNames.put(KEY_FREQUENCY_OFFSET, "Frequency offset");
			propertyNames.put(KEY_EMISSION_MASK, "Emission mask");
			propertyNames.put(KEY_DTS_TIME_DELAY, "DTS time delay");
			propertyNames.put(KEY_HORIZONTAL_PATTERN_NAME, "Horizontal pattern");
			propertyNames.put(KEY_HORIZONTAL_PATTERN_ORIENTATION, "Pattern orientation");
			propertyNames.put(KEY_VERTICAL_PATTERN_NAME, "Vertical pattern");
			propertyNames.put(KEY_VERTICAL_PATTERN_ELECTRICAL_TILT, "Electrical tilt");
			propertyNames.put(KEY_VERTICAL_PATTERN_MECHANICAL_TILT, "Mechanical tilt");
			propertyNames.put(KEY_VERTICAL_PATTERN_MECHANICAL_TILT_ORIENTATION, "Mechanical tilt orientation");
		};

		public TreeMap<String, String> propertyValues;

		public AntPattern horizontalPattern;
		public AntPattern verticalPattern;

		public TreeMap<Integer, RecordData> dependentData;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Subclasses may override this, call super to get the initial object, then modify as appropriate.  That may
	// include adding or even removing properties from the value map, and defining the dependent data map.

	public RecordData getRecordData(ErrorLogger errors) {

		RecordData data = new RecordData();

		TreeMap<String, String> values = new TreeMap<String, String>();
		data.propertyValues = values;

		values.put(RecordData.KEY_RECORD_TYPE, getRecordType());
		values.put(RecordData.KEY_STATION_DATA, getStationData());
		values.put(RecordData.KEY_RECORD_ID, getRecordID());
		values.put(RecordData.KEY_SERVICE, getService());
		values.put(RecordData.KEY_CALL_SIGN, getCallSign());
		values.put(RecordData.KEY_CITY, getCity());
		values.put(RecordData.KEY_STATE, getState());
		values.put(RecordData.KEY_COUNTRY, getCountry());
		values.put(RecordData.KEY_FILE_NUMBER, getFileNumber());
		values.put(RecordData.KEY_SEQUENCE_DATE, getSequenceDate());
		values.put(RecordData.KEY_LATITUDE, getLatitude());
		values.put(RecordData.KEY_LONGITUDE, getLongitude());
		values.put(RecordData.KEY_HEIGHT_AMSL, getHeightAMSL());
		values.put(RecordData.KEY_OVERALL_HAAT, getOverallHAAT());
		values.put(RecordData.KEY_PEAK_ERP, getPeakERP());

		String theName = "Omnidirectional";

		if (hasHorizontalPattern) {

			theName = getHorizontalPatternName();

			values.put(RecordData.KEY_HORIZONTAL_PATTERN_ORIENTATION, getHorizontalPatternOrientation());

			if (null == horizontalPattern) {
				Source theSource = getSource();
				if (null != theSource) {
					horizontalPattern = theSource.getHorizontalPattern(errors);
					if (null == horizontalPattern) {
						return null;
					}
				}
			}
			if (null != horizontalPattern) {
				data.horizontalPattern = horizontalPattern.copy();
			}
		}

		values.put(RecordData.KEY_HORIZONTAL_PATTERN_NAME, theName);

		theName = "None";

		if (hasVerticalPattern || hasMatrixPattern || useGenericVerticalPattern) {

			if (hasVerticalPattern) {

				if (null != verticalPattern) {
					theName = verticalPattern.name;
				} else {
					Source theSource = getSource();
					if (null != theSource) {
						theName = theSource.verticalPatternName;
						verticalPattern = theSource.getVerticalPattern(errors);
						if (null == verticalPattern) {
							return null;
						}
					}
				}
				if (null != verticalPattern) {
					data.verticalPattern = verticalPattern.copy();
				}

			} else {

				if (hasMatrixPattern) {

					if (null != matrixPattern) {
						theName = matrixPattern.name;
					} else {
						Source theSource = getSource();
						if (null != theSource) {
							theName = theSource.matrixPatternName;
							matrixPattern = theSource.getMatrixPattern(errors);
							if (null == matrixPattern) {
								return null;
							}
						}
					}

					if (null != matrixPattern) {
						data.verticalPattern = matrixPattern.copy();
					}

				} else {

					if (useGenericVerticalPattern) {
						theName = "Generic";
					}
				}
			}

			if (hasVerticalPattern || useGenericVerticalPattern) {
				values.put(RecordData.KEY_VERTICAL_PATTERN_ELECTRICAL_TILT,
					AppCore.formatDepression(verticalPatternElectricalTilt) + " deg");
				values.put(RecordData.KEY_VERTICAL_PATTERN_MECHANICAL_TILT,
					AppCore.formatDepression(verticalPatternMechanicalTilt) + " deg");
				values.put(RecordData.KEY_VERTICAL_PATTERN_MECHANICAL_TILT_ORIENTATION,
					AppCore.formatAzimuth(verticalPatternMechanicalTiltOrientation) + " deg");
			}
		}

		values.put(RecordData.KEY_VERTICAL_PATTERN_NAME, theName);

		return data;
	}


	//=================================================================================================================
	// XML parsing handler for source and scenario data.  This can work both with and without a study context.  If a
	// study is provided this can build scenarios and will add them directly to the study model, else it will just
	// parse source objects and make those available in the sources public array.  This may also be used to parse a
	// user record stored as XML, if a user record ID is provided this will read until one full source record is
	// parsed, assign that the ID, and then ignore all subsequent data in the XML file.  The single record is available
	// in the source public property.

	public static class SourceXMLHandler extends DefaultHandler {

		// This returns all unique sources loaded or created when those aren't being added directly to a study.

		public ArrayList<SourceEditData> sources;

		// This returns a single source in user record parsing mode, else it is null.

		public SourceEditData source;

		// Number of scenarios added to the study in scenario mode.

		public int scenarioCount;

		// Indicate if TVSTUDY element was seen, so caller can report a reason for a non-error but empty result.

		public boolean hadStudy;

		// Functional state.

		private StudyEditData study;
		private String dbID;
		private Integer userRecordID;
		private int parseRecordType;
		private int parseStudyType;
		private ErrorLogger errors;
		private Locator locator;

		private boolean userRecordMode;
		private boolean scenarioMode;

		// This is the data set used to attempt to resolve by-reference sources in any mode but user-record.  For by-
		// reference records found during parsing, search is deferred to the end of the enclosing element so a batch
		// search can be run.  This is performance optimization, record-by-record searches can be really slow.  There
		// may be more than one lookup data set used, a primary and alternate.

		private LookupIndex lookupIndex;
		private LookupIndex alternateIndex;

		private HashSet<String> searchIDs;
		private ArrayList<SearchItem> searchItems;

		// Parsing state.

		private boolean ignoreAll;

		private boolean inStudy;
		private int xmlVersion;
		private ArrayList<SourceEditData> studyNewSources;
		private ArrayList<ScenarioItem> scenarioItems;

		private boolean inScenario;
		private String scenarioName;
		private String scenarioDescription;
		private ArrayList<Scenario.SourceListItem> scenarioSourceItems;
		private HashMap<Integer, ArrayList<String>> scenarioParameters;
		private boolean hasDesiredTV;

		private boolean inParameter;
		private int parameterKey;
		private ArrayList<String> parameterValues;

		private boolean inValue;
		private int valueIndex;

		private boolean inSource;
		private boolean isDesired;
		private boolean isUndesired;
		private SourceEditData newSource;
		private SourceEditDataTV newSourceTV;
		private SourceEditDataWL newSourceWL;
		private SourceEditDataFM newSourceFM;
		private boolean didMakeSource;
		private int replicateToChannel;

		private boolean inDTSSource;
		private SourceEditDataTV newDTSSource;

		private String horizontalPatternName;
		private String verticalPatternName;
		private String matrixPatternName;

		private boolean inPat;

		// Stack of element names and content buffers for nested elements.

		private ArrayDeque<String> elements;
		private ArrayDeque<StringWriter> buffers;

		// For parsing APAT, EPAT, and MPAT data in XML.

		private Pattern patternParser;


		//-------------------------------------------------------------------------------------------------------------
		// Constructor for object that imports scenarios and sources into a study.  Objects will be added directly to
		// the study as parsing proceeds.  The study argument must not be null here.  A data set key may be provided
		// for resolving by-reference sources.  If that argument is null the study default data set is used if that
		// exists, else by-reference sources are ignored.  The parsed records are filtered according to the study type,
		// ignoring any record types not allowed for the study.  If the lookup data set key is provided, an alternate
		// lookup key may also be provided, if a record is not found in the primary the alternate is checked.  The main
		// use of that capability is for TV import, so both CDBS and LMS sets can be provided.

		public SourceXMLHandler(StudyEditData theStudy, Integer theLookupExtDbKey, Integer theAlternateExtDbKey,
				ErrorLogger theErrors) {

			super();
			doSetup(theStudy, theStudy.dbID, theLookupExtDbKey, theAlternateExtDbKey, null, 0,
				theStudy.study.studyType, theErrors);
		}


		//-------------------------------------------------------------------------------------------------------------
		// Constructor for object used to parse a user record.  This is private, the public interface for obtaining
		// user records is findUserRecord*().  This mode may or may not have a study context, if study is null the dbID
		// must be passed explicitly.  This never has a by-reference search context as user records will always be
		// standalone XML blocks.  This will parse until one valid source is found then ignore the rest of the input;
		// XML from the user record table should only ever contain one source.

		private SourceXMLHandler(StudyEditData theStudy, String theDbID, Integer theUserRecordID,
				ErrorLogger theErrors) {

			super();
			doSetup(theStudy, theDbID, null, null, theUserRecordID, 0, 0, theErrors);
		}


		//-------------------------------------------------------------------------------------------------------------
		// Constructor for general-purpose parsing of sources outside any particular context, the resulting objects are
		// transient and presumably will be shown to the user in a selection UI then selected sources transferred to a
		// study context via derivation, see deriveSource().  This may have a by-reference context, or not, and the
		// records may be filtered by record type, study type, or neither.  If the recordType argument is >0 only that
		// type will be accepted, all others will be ignored, else if the studyType argument is >0 only record types
		// that are allowed for that study type will be accepted.

		public SourceXMLHandler(String theDbID, Integer theLookupExtDbKey, Integer theAlternateExtDbKey,
				int theRecordType, int theStudyType, ErrorLogger theErrors) {

			super();
			doSetup(null, theDbID, theLookupExtDbKey, theAlternateExtDbKey, null, theRecordType, theStudyType,
				theErrors);
		}


		//-------------------------------------------------------------------------------------------------------------
		// The common part of construction, see comments above.

		private void doSetup(StudyEditData theStudy, String theDbID, Integer theLookupExtDbKey,
				Integer theAlternateExtDbKey, Integer theUserRecordID, int theRecordType, int theStudyType,
				ErrorLogger theErrors) {

			study = theStudy;
			dbID = theDbID;
			userRecordID = theUserRecordID;
			parseRecordType = theRecordType;
			parseStudyType = theStudyType;
			errors = theErrors;

			// Set flags controlling the major branchings of the logic.  In user record mode the study may be used but
			// only for ownership of the source object, so the scenarioMode flag is not set in that case.  Note if both
			// of these flags are false this is in "generic read mode", all unique sources found are returned in the
			// public list and all are transient, and any enclosing scenario elements are ignored.

			userRecordMode = (null != userRecordID);
			if (!userRecordMode) {
				scenarioMode = (null != study);
			}

			// Get an ExtDb context to use for resolving by-reference sources.  This is always optional, but if no key
			// is provided and a study context is present, try using the default data set key from the study.  Also do
			// some sanity checks.  Generic data sets cannot be used for this since the record IDs are not portable,
			// those should not even be offerred by any UI leading to this so just ignore.  This is irrelevant in user-
			// record mode, there can never be a by-reference record in that case.  A second alternate key may also be
			// provided, it is used only if the primary key is set, subject to the same checks.

			if (!userRecordMode) {

				if (null == theLookupExtDbKey) {
					theAlternateExtDbKey = null;
				}

				ExtDb theExtDb = null;

				if ((null == theLookupExtDbKey) && scenarioMode) {
					theLookupExtDbKey = study.extDbKey;
				}
				if (null != theLookupExtDbKey) {
					theExtDb = ExtDb.getExtDb(dbID, theLookupExtDbKey);
					if (null != theExtDb) {
						switch (parseRecordType) {
							case 0:
								if (!ExtDbRecordTV.isExtDbSupported(theExtDb) &&
										!ExtDbRecordFM.isExtDbSupported(theExtDb)) {
									theExtDb = null;
								}
								break;
							case Source.RECORD_TYPE_TV:
								if (!ExtDbRecordTV.isExtDbSupported(theExtDb)) {
									theExtDb = null;
								}
								break;
							case Source.RECORD_TYPE_FM:
								if (!ExtDbRecordFM.isExtDbSupported(theExtDb)) {
									theExtDb = null;
								}
								break;
							default:
								theExtDb = null;
								break;
						}
					}
				}

				// If a lookup data set was found, parsed shareable sources are indexed to resolve later references.
				// If a study is available the index is seeded with sources from the lookup set that are already in the
				// study.  In that case replications can also occur, and may also be shared.  See doScenarioAdd().

				if (null != theExtDb) {

					lookupIndex = new LookupIndex();
					lookupIndex.lookupExtDb = theExtDb;
					lookupIndex.sharedSources = new HashMap<String, SourceEditData>();

					searchIDs = new HashSet<String>();

					if (scenarioMode) {

						lookupIndex.sharedReplicationSources = new ArrayList<HashMap<String, SourceEditDataTV>>();
						study.loadSharedSourceIndex(lookupIndex.lookupExtDb.key, lookupIndex.sharedSources,
							lookupIndex.sharedReplicationSources);

						searchItems = new ArrayList<SearchItem>();
					}

					// Set up the alternate the same way as needed.

					theExtDb = null;

					if (null != theAlternateExtDbKey) {
					theExtDb = ExtDb.getExtDb(dbID, theAlternateExtDbKey);
						if (null != theExtDb) {
							switch (parseRecordType) {
								case 0:
									if (!ExtDbRecordTV.isExtDbSupported(theExtDb) &&
											!ExtDbRecordFM.isExtDbSupported(theExtDb)) {
										theExtDb = null;
									}
									break;
								case Source.RECORD_TYPE_TV:
									if (!ExtDbRecordTV.isExtDbSupported(theExtDb)) {
										theExtDb = null;
									}
									break;
								case Source.RECORD_TYPE_FM:
									if (!ExtDbRecordFM.isExtDbSupported(theExtDb)) {
										theExtDb = null;
									}
									break;
								default:
									theExtDb = null;
									break;
							}
						}
					}

					if (null != theExtDb) {

						alternateIndex = new LookupIndex();
						alternateIndex.lookupExtDb = theExtDb;
						alternateIndex.sharedSources = new HashMap<String, SourceEditData>();

						if (scenarioMode) {

							alternateIndex.sharedReplicationSources = new ArrayList<HashMap<String, SourceEditDataTV>>();
							study.loadSharedSourceIndex(alternateIndex.lookupExtDb.key, alternateIndex.sharedSources,
								alternateIndex.sharedReplicationSources);
						}
					}
				}
			}

			// If not in scenario mode or user record mode, all sources found will be returned in the list property.

			if (!scenarioMode && !userRecordMode) {
				sources = new ArrayList<SourceEditData>();
			}

			// Setup needed for all modes.

			valueIndex = -1;

			elements = new ArrayDeque<String>();
			buffers = new ArrayDeque<StringWriter>();

			patternParser = Pattern.compile("[,\\n]");
		}


		//-------------------------------------------------------------------------------------------------------------
		// If a locator is set, store it for error reporting.

		public void setDocumentLocator(Locator theLocator) {

			locator = theLocator;
		}


		//-------------------------------------------------------------------------------------------------------------
		// Report an error to the error logger and throw an exception to abort parsing, show line number if possible.

		private void throwError(String message) throws SAXException {

			if (null != errors) {
				if (null != locator) {
					if (null == message) {
						message = "Error occurred ";
					}
					errors.reportError(message + " at line " + String.valueOf(locator.getLineNumber()));
				} else {
					if (null != message) {
						errors.reportError(message);
					}
				}
			}
			throw new SAXException();
		}


		//=============================================================================================================
		// Index for a data set used for resolving record IDs, more than one set may be checked.

		private class LookupIndex {

			private ExtDb lookupExtDb;
			private HashMap<String, SourceEditData> sharedSources;
			private ArrayList<HashMap<String, SourceEditDataTV>> sharedReplicationSources;
		}


		//=============================================================================================================
		// Data for a record in the deferred search list.

		private class SearchItem {

			private String extRecordID;
			private boolean isDesired;
			private boolean isUndesired;
			private int replicateToChannel;
		}


		//=============================================================================================================
		// Data for a new scenario, deferred until all parsed with no errors.

		private class ScenarioItem {

			private String name;
			private String description;
			private ArrayList<Scenario.SourceListItem> sourceItems;
			private HashMap<Integer, ArrayList<String>> parameters;
		}


		//-------------------------------------------------------------------------------------------------------------
		// Start of element.  Push the element name and a new content buffer on to the stacks.

		public void startElement(String nameSpc, String locName, String qName, Attributes attrs) throws SAXException {

			String str;

			elements.push(qName);
			buffers.push(new StringWriter());

			// The TVSTUDY element must be present and include a version number, and nothing before 1.3 is supported.
			// Only the first study element is processed, all later are ignored.

			if (qName.equals("TVSTUDY")) {

				if (hadStudy || ignoreAll) {
					return;
				}

				if (inStudy) {
					throwError("TVSTUDY elements may not be nested");
				}
				inStudy = true;
				hadStudy = true;

				str = attrs.getValue("VERSION");
				if (null != str) {
					try {
						xmlVersion = Integer.parseInt(str);
					} catch (NumberFormatException nfe) {
					}
				}
				if (xmlVersion <= 0) {
					throwError("Missing or bad VERSION attribute in TVSTUDY tag");
				}

				if ((xmlVersion < 103000) || (xmlVersion > AppCore.XML_VERSION)) {
					throwError("Format version is not supported");
				}

				if (scenarioMode) {
					studyNewSources = new ArrayList<SourceEditData>();
					scenarioItems = new ArrayList<ScenarioItem>();
				}

				return;
			}

			// Start of a new scenario.  Ignore this outside of a TVSTUDY element, if not in scenario parsing mode, or
			// if ignoreAll is set.  Else error if parsing already inside a SCENARIO element.  Note SOURCE elements
			// within the scenario may still be parsed even if this element is ignored.

			if (qName.equals("SCENARIO")) {

				if (!inStudy || !scenarioMode || ignoreAll) {
					return;
				}

				if (inScenario) {
					throwError("SCENARIO elements may not be nested");
				}
				inScenario = true;

				// If the name is not provided or the value doesn't pass checks assign a generic default, however the
				// name does not have to be unique, the ScenarioEditData constructor will make the name unique if
				// needed.

				scenarioName = attrs.getValue("NAME");
				if ((null == scenarioName) || !DbCore.checkScenarioName(scenarioName, study, false)) {
					scenarioName = "Import";
				}

				scenarioDescription = "";
				scenarioSourceItems = new ArrayList<Scenario.SourceListItem>();
				scenarioParameters = new HashMap<Integer, ArrayList<String>>();

				return;
			}

			// A DESCRIPTION element is handled at the end of the element.  This is a trivial element so not worth the
			// trouble to do nesting or context checks here.

			if (qName.equals("DESCRIPTION")) {
				return;
			}

			// A PARAMETER element is ignored if not in a SCENARIO, which will always be true if SCENARIO is being
			// ignored.  Otherwise the only check is for nesting, it's not worth enforcing structure any more strictly
			// for these.  The only attribute is the key, that must be present, but is just checked for general
			// validity (> 0).  If a key does not exist in the scenario it will be ignored later; this is just
			// providing values for existing parameters and is always optional.  Duplicate keys are not an error, the
			// last values found will be used.  

			if (qName.equals("PARAMETER")) {

				if (!inScenario || ignoreAll) {
					return;
				}

				if (inParameter) {
					throwError("PARAMETER elements may not be nested");
				}
				inParameter = true;

				str = attrs.getValue("KEY");
				if (null != str) {
					try {
						parameterKey = Integer.parseInt(str);
					} catch (NumberFormatException nfe) {
					}
				}
				if (parameterKey <= 0) {
					throwError("Missing or bad KEY attribute in PARAMETER tag");
				}

				parameterValues = new ArrayList<String>();

				return;
			}

			// A VALUE element provides one value for a parameter, identified by 0..N index, these are ignored outside
			// a PARAMETER and cannot be nested.  Indices may be out of sequence and not all represented, the value
			// list will be expanded by adding nulls as needed.  Also duplicate indices are not an error, the last is
			// used.

			if (qName.equals("VALUE")) {

				if (!inParameter || ignoreAll) {
					return;
				}

				if (inValue) {
					throwError("VALUE elements cannot be nested");
				}
				inValue = true;

				str = attrs.getValue("INDEX");
				if (null != str) {
					try {
						valueIndex = Integer.parseInt(str);
					} catch (NumberFormatException nfe) {
					}
				}
				if ((valueIndex < 0) || (valueIndex >= Parameter.MAX_VALUE_COUNT)) {
					throwError("Missing or bad INDEX attribute in VALUE tag");
				}

				if (valueIndex >= parameterValues.size()) {
					for (int i = parameterValues.size(); i <= valueIndex; i++) {
						parameterValues.add(null);
					}
				}

				return;
			}

			// Start of a SOURCE element.  This may be in a SCENARIO element or not, that will be handled at the end of
			// the element.  However when parsing scenarios, sources outside a scenario are ignored.  Outside of a
			// TVSTUDY element this is ignored.  Usual check for illegal nesting.

			if (qName.equals("SOURCE")) {

				if (!inStudy || (scenarioMode && !inScenario) || ignoreAll) {
					return;
				}

				if (inSource) {
					throwError("SOURCE elements may not be nested");
				}
				inSource = true;

				// The service attribute is extracted first to identify the record type.  However in format versions
				// prior to 1.5 the service was not included on by-reference records, only TV records could exist then
				// so the type was implicit.  In that case assume TV and do not report the missing attribute here.

				Service service = null;
				str = attrs.getValue("SERVICE");
				if (null != str) {
					service = Service.getService(str);
					if (null == service) {
						throwError("Bad SERVICE attribute in SOURCE tag");
					}
				}

				int recordType = Source.RECORD_TYPE_TV;
				if (xmlVersion >= 105000) {
					if (null == service) {
						throwError("Missing SERVICE attribute in SOURCE tag");
					}
					recordType = service.serviceType.recordType;
				}

				// Make sure the record type is allowed, if not ignore it.

				if (parseRecordType > 0) {
					if (recordType != parseRecordType) {
						inSource = false;
						return;
					}
				} else {
					if (parseStudyType > 0) {
						if (!Study.isRecordTypeAllowed(parseStudyType, recordType)) {
							inSource = false;
							return;
						}
					}
				}

				// If a scenario is being loaded, extract the study flags.  The flags are optional, the default is
				// true.  However the flags are not always honored, they may be altered in special cases, see
				// endElement().  For legacy support the attribute STUDY is a synonym for DESIRED.

				if (inScenario) {

					str = attrs.getValue("DESIRED");
					if (null == str) {
						str = attrs.getValue("STUDY");
					}
					isDesired = ((null == str) || Boolean.valueOf(str).booleanValue());

					str = attrs.getValue("UNDESIRED");
					isUndesired = ((null == str) || Boolean.valueOf(str).booleanValue());
				}

				// Extract the locked flag, also optional, default is true.

				str = attrs.getValue("LOCKED");
				boolean isLocked = ((null == str) || Boolean.valueOf(str).booleanValue());

				// If a study is available, check for a REPLICATE attribute on a TV record.  Replication cannot be done
				// without a study context to retain the original source.  The source defined by the SOURCE element is
				// the original, REPLICATE means to create another source replicating that original, that will be done
				// at the end of the SOURCE element.  Skip this in user record mode, that does not support replication.

				if (inScenario && (Source.RECORD_TYPE_TV == recordType)) {

					str = attrs.getValue("REPLICATE");
					if (null != str) {

						try {
							replicateToChannel = Integer.parseInt(str);
						} catch (NumberFormatException nfe) {
						}

						if ((replicateToChannel < SourceTV.CHANNEL_MIN) || (replicateToChannel > SourceTV.CHANNEL_MAX)) {
							throwError("Bad REPLICATE attribute in SOURCE tag");
						}
					}
				}

				// A source based on a record from a data set with persistent, portable IDs is indicated by the
				// presence of a RECORD_ID attribute.  This does not apply and the attribute is ignored for wireless
				// records, and in user-record mode.  IDs for wireless records are valid only for a specific data set
				// so should never be exported.  User records should never be by-reference.  In either case assume all
				// attributes will be present, if not an error should and will occur.  CDBS_ID is a legacy synonym for
				// RECORD_ID.

				String extRecordID = null;
				if (!userRecordMode && (Source.RECORD_TYPE_WL != recordType)) {
					extRecordID = attrs.getValue("RECORD_ID");
						if (null == extRecordID) {
						extRecordID = attrs.getValue("CDBS_ID");
					}
				}

				// A locked data set record is a reference only, no further properties are present in attributes or
				// nested elements.  If a data set is available these records may be loaded from that, otherwise the
				// element is ignored.  It is not an error if a record does not exist in the lookup data set, again the
				// element is just ignored.  Sources loaded this way are kept in a temporary sharing index so later
				// references to the same record don't have to be re-loaded.  If a study is available the index was
				// seeded with sources from the data set already in the study so those also do not have to be loaded.
				// Otherwise a search query is run to find the record.  For performance the search is not done
				// immediately, the record is added to a search list which will be run as a batch at the end of the
				// scenario or study element, see endElement().

				if (isLocked && (null != extRecordID)) {

					if (null == lookupIndex) {
						inSource = false;
					} else {

						newSource = lookupIndex.sharedSources.get(extRecordID);
						if ((null == newSource) && (null != alternateIndex)) {
							newSource = alternateIndex.sharedSources.get(extRecordID);
						}

						if (null == newSource) {

							searchIDs.add(extRecordID);

							if (inScenario) {

								SearchItem theItem = new SearchItem();
								theItem.extRecordID = extRecordID;
								theItem.isDesired = isDesired;
								theItem.isUndesired = isUndesired;
								theItem.replicateToChannel = replicateToChannel;

								searchItems.add(theItem);
							}

							inSource = false;

						} else {

							// In a scenario the same source may be seen more than once because it was used in a
							// previous scenario, or even in the same scenario if there are multiple replications.  In
							// any other mode sources should never be duplicated so ignore. 

							if (inScenario) {

								switch (recordType) {

									case Source.RECORD_TYPE_TV: {
										newSourceTV = (SourceEditDataTV)newSource;
										break;
									}

									case Source.RECORD_TYPE_FM: {
										newSourceFM = (SourceEditDataFM)newSource;
										break;
									}
								}

							} else {
								inSource = false;
							}
						}
					}

					return;
				}

				// This is an editable source or one not based on a portable data set record.  In either case the
				// source will never be shared so this will always create a new source object from attributes.  At this
				// point if the SERVICE attribute was not found it is reported as an error.  Also extract the COUNTRY
				// attribute which must always be present.

				if (null == service) {
					throwError("Missing SERVICE attribute in SOURCE tag");
				}

				Country country = null;
				str = attrs.getValue("COUNTRY");
				if (null != str) {
					country = Country.getCountry(str);
				}
				if (null == country) {
					throwError("Missing or bad COUNTRY attribute in SOURCE tag");
				}

				// Create the new source.  If a RECORD_ID attribute is present and that ID exists in the lookup data
				// set (if any) the data set key and ID are included on the new source, but that is advisory only.  In
				// any case all source properties must be present in attributes or element content so the source is
				// never loaded from the data set here.  However in user record mode, the first source seen in this
				// context is saved as the user record, in that case the record is always locked and any record ID from
				// the tag is ignored.

				Integer extDbKey = null;

				if (userRecordMode) {

					isLocked = true;
					extRecordID = null;

				} else {

					if ((null != lookupIndex) && (null != extRecordID) &&
							ExtDbRecord.doesRecordIDExist(lookupIndex.lookupExtDb, extRecordID)) {
						extDbKey = lookupIndex.lookupExtDb.key;
					} else {
						extRecordID = null;
					}
				}

				switch (recordType) {

					case Source.RECORD_TYPE_TV: {

						newSourceTV = SourceEditDataTV.makeSourceWithAttributesTV(qName, attrs, study, dbID, service,
							country, isLocked, userRecordID, extDbKey, extRecordID, errors);
						if (null == newSourceTV) {
							throwError(null);
						}
						newSource = newSourceTV;

						// If a study is available check channel for TV records, if out of range log it and ignore the
						// element.  The channel was already checked for validity by the attributes parser so this is
						// not an error, the record is valid but it is just not allowed in this particular study.  If a
						// replication attribute was found this checks the replication channel not the original; the
						// original can be out of range as long as the replication is in range.

						if (inScenario) {
							if (0 == replicateToChannel) {
								if ((newSourceTV.channel < study.getMinimumChannel()) ||
										(newSourceTV.channel > study.getMaximumChannel())) {
									if (null != errors) {
										errors.logMessage(ExtDbRecord.makeMessage(newSource,
											"Ignored, channel is out of range for study"));
									}
									inSource = false;
									return;
								}
							} else {
								if ((replicateToChannel < study.getMinimumChannel()) ||
										(replicateToChannel > study.getMaximumChannel())) {
									if (null != errors) {
										errors.logMessage(ExtDbRecord.makeMessage(newSource,
											"Ignored, replication channel " + String.valueOf(replicateToChannel) +
											" is out of range for study"));
									}
									inSource = false;
									return;
								}
							}
						}

						break;
					}

					// Wireless records never have a data set reference, see comments above.

					case Source.RECORD_TYPE_WL: {

						newSourceWL = SourceEditDataWL.makeSourceWithAttributesWL(qName, attrs, study, dbID, service,
							country, isLocked, userRecordID, null, null, errors);
						if (null == newSourceWL) {
							throwError(null);
						}
						newSource = newSourceWL;

						break;
					}

					case Source.RECORD_TYPE_FM: {

						newSourceFM = SourceEditDataFM.makeSourceWithAttributesFM(qName, attrs, study, dbID, service,
							country, isLocked, userRecordID, extDbKey, extRecordID, errors);
						if (null == newSourceFM) {
							throwError(null);
						}
						newSource = newSourceFM;

						break;
					}
				}

				didMakeSource = true;

				// Get pattern names from attributes and store until pattern data is parsed.

				str = attrs.getValue("APAT_NAME");
				if (null != str) {
					if (str.length() > Source.MAX_PATTERN_NAME_LENGTH) {
						str = str.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
					}
					horizontalPatternName = str;
				} else {
					horizontalPatternName = "";
				}

				str = attrs.getValue("EPAT_NAME");
				if (null != str) {
					if (str.length() > Source.MAX_PATTERN_NAME_LENGTH) {
						str = str.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
					}
					verticalPatternName = str;
				} else {
					verticalPatternName = "";
				}

				str = attrs.getValue("MPAT_NAME");
				if (null != str) {
					if (str.length() > Source.MAX_PATTERN_NAME_LENGTH) {
						str = str.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
					}
					matrixPatternName = str;
				} else {
					matrixPatternName = "";
				}

				return;
			}

			// Start a DTS_SOURCE element, as usual no nesting.  If this occurs outside a standalone SOURCE element it
			// is ignored.  Otherwise the containing SOURCE element must be a DTS TV record (the parent flag is true).

			if (qName.equals("DTS_SOURCE")) {

				if (!inSource || !didMakeSource || ignoreAll) {
					return;
				}

				if (inDTSSource) {
					throwError("DTS_SOURCE elements may not be nested");
				}
				inDTSSource = true;

				if ((null == newSourceTV) || !newSourceTV.isParent) {
					throwError("DTS_SOURCE not allowed in non-parent SOURCE element");
				}

				newDTSSource = newSourceTV.addDTSSourceWithAttributes(qName, attrs, errors);
				if (null == newDTSSource) {
					throwError(null);
				}

				// Get pattern names from attributes and store until pattern data is parsed.  Note this is not
				// overwriting anything from the enclosing parent SOURCE since a DTS parent never has patterns.

				str = attrs.getValue("APAT_NAME");
				if (null != str) {
					if (str.length() > Source.MAX_PATTERN_NAME_LENGTH) {
						str = str.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
					}
					horizontalPatternName = str;
				} else {
					horizontalPatternName = "";
				}

				str = attrs.getValue("EPAT_NAME");
				if (null != str) {
					if (str.length() > Source.MAX_PATTERN_NAME_LENGTH) {
						str = str.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
					}
					verticalPatternName = str;
				} else {
					verticalPatternName = "";
				}

				str = attrs.getValue("MPAT_NAME");
				if (null != str) {
					if (str.length() > Source.MAX_PATTERN_NAME_LENGTH) {
						str = str.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
					}
					matrixPatternName = str;
				} else {
					matrixPatternName = "";
				}

				return;
			}

			// APAT/EPAT/MPAT elements may occur inside SOURCE or DTS_SOURCE.  These are ignored outside a containing
			// standalone source.  Also if the related HAS_?PAT flag is false ignore the element.

			boolean isHpat = qName.equals("APAT");
			boolean isVpat = qName.equals("EPAT");
			boolean isMpat = qName.equals("MPAT");

			if (isHpat || isVpat || isMpat) {

				if (!inSource || !didMakeSource || ignoreAll) {
					return;
				}

				SourceEditData theSource = newSource;
				if (null != newDTSSource) {
					theSource = newDTSSource;
				}

				if ((isHpat && !theSource.hasHorizontalPattern) || (isVpat && !theSource.hasVerticalPattern) ||
						(isMpat && !theSource.hasMatrixPattern)) {
					return;
				}

				if (inPat) {
					throwError("APAT/EPAT/MPAT elements may not be nested");
				}
				inPat = true;

				return;
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Add content characters to the buffer for the current element.

		public void characters(char[] ch, int start, int length) {

			if (!buffers.isEmpty()) {
				buffers.peek().write(ch, start, length);
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// End of element, pop the element name and content off the stacks.  If the element name does not match it
		// means either elements overlap or nested elements were not explicitly closed.

		public void endElement(String nameSpc, String locName, String qName) throws SAXException {

			String element = elements.pop();
			String content = buffers.pop().toString().trim();

			if (!element.equals(qName)) {
				throwError("Overlapping or un-terminated elements");
			}

			// End of the TVSTUDY element.  If the deferred search list is not empty, run the search.  That will occur
			// if scenarios were not being processed so this is a generic load of all sources found.  Otherwise in
			// scenario mode, if any scenarios were parsed add them to the study.  Add all new sources to the study
			// first, then create the new scenarios, apply parameter values, and add to the study.

			if (element.equals("TVSTUDY")) {

				if (inStudy) {

					if ((null != lookupIndex) && !searchIDs.isEmpty()) {
						doSearch(lookupIndex);
						if (!searchIDs.isEmpty() && (null != alternateIndex)) {
							doSearch(alternateIndex);
						}
						if (!searchIDs.isEmpty()) {
							errors.logMessage("Record IDs not found in data sets:");
							for (String theID : searchIDs) {
								errors.logMessage(theID);
							}
						}
						searchIDs.clear();
					}

					if (scenarioMode && !scenarioItems.isEmpty()) {

						for (SourceEditData theSource : studyNewSources) {
							study.addOrReplaceSource(theSource);
						}

						ScenarioEditData newScenario;
						ParameterEditData theParam;
						ArrayList<String> theValues;
						String value;
						int nValues, vIndex;

						for (ScenarioItem theItem : scenarioItems) {

							newScenario = new ScenarioEditData(study, theItem.name, theItem.description,
								theItem.sourceItems);

							if (!theItem.parameters.isEmpty()) {
								for (Integer theKey : theItem.parameters.keySet()) {
									theParam = newScenario.getParameter(theKey.intValue());
									if (null != theParam) {
										theValues = theItem.parameters.get(theKey);
										nValues = theValues.size();
										if (nValues > theParam.parameter.valueCount) {
											nValues = theParam.parameter.valueCount;
										}
										for (vIndex = 0; vIndex < nValues; vIndex++) {
											value = theValues.get(vIndex);
											if (null != value) {
												theParam.value[vIndex] = value;
											}
										}
									}
								}
							}

							study.scenarioData.addOrReplace(newScenario);
							scenarioCount++;
						}
					}

					inStudy = false;
				}

				xmlVersion = 0;
				studyNewSources = null;
				scenarioItems = null;
	
				return;
			}

			// The end of a SCENARIO element.  Ignore if not in a scenario.  If the deferred search list is not empty,
			// run the search.  Ignore empty scenarios.  For some study types the scenario must have a desired TV, if
			// one is not present ignore the scenario.  See doScenarioAdd().  The scenario information is stored for
			// later, see above, they are created and added all-or-nothing in case of error.

			if (element.equals("SCENARIO")) {

				if (inScenario) {

					if ((null != lookupIndex) && !searchIDs.isEmpty()) {
						doSearch(lookupIndex);
						if (!searchIDs.isEmpty() && (null != alternateIndex)) {
							doSearch(alternateIndex);
						}
						if (!searchIDs.isEmpty()) {
							errors.logMessage("Record IDs not found in data sets:");
							for (String theID : searchIDs) {
								errors.logMessage(theID);
							}
						}
						searchIDs.clear();
						searchItems.clear();
					}

					if (!scenarioSourceItems.isEmpty() && (((Study.STUDY_TYPE_TV_IX != study.study.studyType) &&
							(Study.STUDY_TYPE_TV_OET74 != study.study.studyType) &&
							(Study.STUDY_TYPE_TV6_FM != study.study.studyType)) || hasDesiredTV)) {

						ScenarioItem theItem = new ScenarioItem();

						theItem.name = scenarioName;
						theItem.description = scenarioDescription;
						theItem.sourceItems = scenarioSourceItems;
						theItem.parameters = scenarioParameters;

						scenarioItems.add(theItem);
					}

					inScenario = false;
				}

				// Clear state for the next SCENARIO element.

				scenarioName = null;
				scenarioDescription = null;
				scenarioSourceItems = null;
				scenarioParameters = null;
				hasDesiredTV = false;

				return;
			}

			// Save description if in a scenario, no check for multiple occurrences or nesting, just use the last one.

			if (element.equals("DESCRIPTION")) {

				if (inScenario) {
					scenarioDescription = content;
				}

				return;
			}

			// If the parameter value list is empty, just ignore the parameter.

			if (element.equals("PARAMETER")) {

				if (inParameter) {

					if (!parameterValues.isEmpty()) {
						scenarioParameters.put(Integer.valueOf(parameterKey), parameterValues);
					}

					inParameter = false;
				}

				parameterKey = 0;
				parameterValues = null;

				return;
			}

			// Parameter value is just the VALUE element content, ignore if empty.

			if (element.equals("VALUE")) {

				if (inValue) {

					if (content.length() > 0) {
						parameterValues.set(valueIndex, content);
					}

					inValue = false;
				}

				valueIndex = -1;

				return;
			}

			// If the source was constructed from attributes and nested elements do additional checks for completeness.
			// For DTS, make sure a valid set of DTS_SOURCE elements was found.

			if (element.equals("SOURCE")) {

				if (inSource) {

					if (didMakeSource) {

						if ((null != newSourceTV) && newSourceTV.isParent) {

							boolean hasSite = false, err = false;
							SourceEditDataTV authSource = null;
							for (SourceEditDataTV dtsSource : newSourceTV.getDTSSources()) {
								if (0 == dtsSource.siteNumber) {
									if (null != authSource) {
										err = true;
										break;
									}
									authSource = dtsSource;
								} else {
									hasSite = true;
								}
							}
							if ((null == authSource) || !hasSite) {
								err = true;
							}
							if (err) {
								throwError("Incomplete or invalid set of DTS_SOURCE elements within SOURCE element");
							}

							// For a Class A or LPTV DTS, copy the authorized facility record coordinates to the parent.
							// On a full-service record the parent coordinates are a distinct reference point defined
							// in external formats, but no such point exists for CA/LP so the coordinates are undefined
							// externally.  However internally the parent coordinates are also used for such things as
							// distance checks and sorting, so they must be set to something appropriate.

							if (!newSourceTV.service.isFullService()) {
								newSourceTV.location.setLatLon(authSource.location);
							}
						}

						// Make sure pattern data elements were found if the attributes indicated they should be
						// present.

						if (newSource.hasHorizontalPattern && (null == newSource.horizontalPattern)) {
							throwError("Missing APAT element in SOURCE element");
						}

						if (newSource.hasVerticalPattern && (null == newSource.verticalPattern)) {
							throwError("Missing EPAT element in SOURCE element");
						}

						// Normalize matrix pattern data if needed, if so apply the derived azimuth pattern which
						// replaces any azimuth pattern in the record.  Normalization is never needed for data exported
						// from TVStudy so usually this does nothing, but it is supported in case other software uses
						// the XML format.

						if (newSource.hasMatrixPattern) {
							if (null == newSource.matrixPattern) {
								throwError("Missing MPAT element in SOURCE element");
							}
							AntPattern matAzPat = newSource.matrixPattern.normalizeVerticalMatrix();
							if (null != matAzPat) {
								newSource.hasHorizontalPattern = true;
								newSource.horizontalPattern = matAzPat;
								newSource.horizontalPatternChanged = true;
							}
						}

						// Source attributes may occur in the element content, apply those.

						newSource.setAllAttributesNT(content);

						// Add the source to the results, in scenario mode it goes in the new scenario sources, in user
						// record mode it goes in the single source public property and all else is ignored, else it
						// goes in the sources list property.  There is no concern about duplicates here because
						// sources seen in this context are always unique by definition.  Below, outside of this
						// context, the source is definitely not unique, it is a by-reference source already retrieved
						// and added while parsing a previous scenario, so it does not need to be added again.

						if (scenarioMode) {

							if (inScenario) {
								studyNewSources.add(newSource);
							}

						} else {

							if (userRecordMode) {

								source = newSource;
								ignoreAll = true;

							} else {

								sources.add(newSource);
							}
						}
					}

					// Do scenario-related processing e.g. replication, see doScenarioAdd().

					if (inScenario) {
						doScenarioAdd(newSource, isDesired, isUndesired, newSourceTV, replicateToChannel);
					}

					inSource = false;
				}

				// Clear state for the next SOURCE element.

				isDesired = false;
				isUndesired = false;
				newSource = null;
				newSourceTV = null;
				newSourceWL = null;
				newSourceFM = null;
				didMakeSource = false;
				replicateToChannel = 0;

				horizontalPatternName = null;
				verticalPatternName = null;
				matrixPatternName = null;

				return;
			}

			// For DTS_SOURCE, check that patterns were found, apply content, then clear state and continue.

			if (element.equals("DTS_SOURCE")) {

				if (inDTSSource) {

					if (didMakeSource) {

						if (newDTSSource.hasHorizontalPattern && (null == newDTSSource.horizontalPattern)) {
							throwError("Missing APAT element in DTS_SOURCE element");
						}

						if (newDTSSource.hasVerticalPattern && (null == newDTSSource.verticalPattern)) {
							throwError("Missing EPAT element in DTS_SOURCE element");
						}

						if (newDTSSource.hasMatrixPattern) {
							if (null == newDTSSource.matrixPattern) {
								throwError("Missing MPAT element in DTS_SOURCE element");
							}
							AntPattern matAzPat = newDTSSource.matrixPattern.normalizeVerticalMatrix();
							if (null != matAzPat) {
								newDTSSource.hasHorizontalPattern = true;
								newDTSSource.horizontalPattern = matAzPat;
								newDTSSource.horizontalPatternChanged = true;
							}
						}

						newDTSSource.setAllAttributesNT(content);
					}

					inDTSSource = false;
				}

				newDTSSource = null;

				return;
			}

			// For APAT/EPAT/MPAT, parse the content data, add to the source or DTS source as needed.  Checks done in
			// startElement() ensure a valid state; if there is a DTS transmitter source being parsed the pattern data
			// goes there, else it goes to the regular (non-DTS) source.

			boolean isHpat = element.equals("APAT");
			boolean isVpat = element.equals("EPAT");
			boolean isMpat = element.equals("MPAT");

			if (isHpat || isVpat || isMpat) {

				if (inPat) {

					boolean bad = false;

					SourceEditData theSource = newSource;
					if (null != newDTSSource) {
						theSource = newDTSSource;
					}

					String[] tokens = patternParser.split(content);

					if (isHpat || isVpat) {

						ArrayList<AntPattern.AntPoint> thePoints = new ArrayList<AntPattern.AntPoint>();

						double azdep, rf, minazdep, maxazdep, lazdep;
						if (isHpat) {
							minazdep = AntPattern.AZIMUTH_MIN;
							maxazdep = AntPattern.AZIMUTH_MAX;
							lazdep = AntPattern.AZIMUTH_MIN - 1.;
						} else {
							minazdep = AntPattern.DEPRESSION_MIN;
							maxazdep = AntPattern.DEPRESSION_MAX;
							lazdep = AntPattern.DEPRESSION_MIN - 1.;
						}

						for (int i = 1; i < tokens.length; i += 2) {
							try {
								azdep = Double.parseDouble(tokens[i - 1]);
								rf = Double.parseDouble(tokens[i]);
							} catch (NumberFormatException nfe) {
								bad = true;
								break;
							}
							if ((azdep < minazdep) || (azdep > maxazdep)) {
								bad = true;
								break;
							}
							if (azdep <= lazdep) {
								bad = true;
								break;
							}
							lazdep = azdep;
							if ((rf < AntPattern.FIELD_MIN) || (rf > AntPattern.FIELD_MAX)) {
								bad = true;
								break;
							}
							thePoints.add(new AntPattern.AntPoint(azdep, rf));
						}

						if (!bad && (thePoints.size() < AntPattern.PATTERN_REQUIRED_POINTS)) {
							bad = true;
						}

						if (!bad) {
							if (isHpat) {
								theSource.hasHorizontalPattern = true;
								theSource.horizontalPattern = new AntPattern(dbID, AntPattern.PATTERN_TYPE_HORIZONTAL,
									horizontalPatternName, thePoints);
								theSource.horizontalPatternChanged = true;
							} else {
								theSource.hasVerticalPattern = true;
								theSource.verticalPattern = new AntPattern(dbID, AntPattern.PATTERN_TYPE_VERTICAL,
									verticalPatternName, thePoints);
								theSource.verticalPatternChanged = true;
							}
						}

					} else {

						ArrayList<AntPattern.AntSlice> theSlices = new ArrayList<AntPattern.AntSlice>();
						ArrayList<AntPattern.AntPoint> thePoints = null;
						double az, dep, rf, laz = AntPattern.AZIMUTH_MIN - 1., ldep = AntPattern.DEPRESSION_MIN - 1.;

						for (int i = 2; i < tokens.length; i += 3) {
							try {
								az = Double.parseDouble(tokens[i - 2]);
								dep = Double.parseDouble(tokens[i - 1]);
								rf = Double.parseDouble(tokens[i]);
							} catch (NumberFormatException nfe) {
								bad = true;
								break;
							}
							if ((az < AntPattern.AZIMUTH_MIN) || (az > AntPattern.AZIMUTH_MAX)) {
								bad = true;
								break;
							}
							if ((dep < AntPattern.DEPRESSION_MIN) || (dep > AntPattern.DEPRESSION_MAX)) {
								bad = true;
								break;
							}
							if (az != laz) {
								if (az <= laz) {
									bad = true;
									break;
								}
								laz = az;
								if ((null != thePoints) && (thePoints.size() < AntPattern.PATTERN_REQUIRED_POINTS)) {
									bad = true;
									break;
								}
								thePoints = new ArrayList<AntPattern.AntPoint>();
								theSlices.add(new AntPattern.AntSlice(az, thePoints));
								ldep = AntPattern.DEPRESSION_MIN - 1.;
							}
							if (dep <= ldep) {
								bad = true;
								break;
							}
							ldep = dep;
							if ((rf < AntPattern.FIELD_MIN) || (rf > AntPattern.FIELD_MAX)) {
								bad = true;
								break;
							}
							thePoints.add(new AntPattern.AntPoint(dep, rf));
						}

						if (!bad && (null != thePoints) && (thePoints.size() < AntPattern.PATTERN_REQUIRED_POINTS)) {
							bad = true;
						}

						if (!bad && (theSlices.size() < AntPattern.PATTERN_REQUIRED_POINTS)) {
							bad = true;
						}

						if (!bad) {
							theSource.hasMatrixPattern = true;
							theSource.matrixPattern = new AntPattern(dbID, matrixPatternName,
								AntPattern.PATTERN_TYPE_VERTICAL, theSlices);
							theSource.matrixPatternChanged = true;
						}
					}

					if (bad) {
						throwError("Bad data in APAT/EPAT/MPAT element");
					}

					inPat = false;
				}

				return;
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Do a search for all by-reference records not previously loaded, called at the end of either scenario or
		// study context as needed.  May be multiple calls for the primary and alternate lookup sets.  This may be a
		// bit paranoid checking the record and study type filters since any record ID in the search list was already
		// type-checked, but relying on that check assumes the XML will never incorrectly specify type for a record ID.

		private void doSearch(LookupIndex theIndex) throws SAXException {

			if (theIndex.lookupExtDb.canProvide(Source.RECORD_TYPE_TV) &&
					((0 == parseRecordType) || (Source.RECORD_TYPE_TV == parseRecordType)) &&
					((0 == parseStudyType) || Study.isRecordTypeAllowed(parseStudyType, Source.RECORD_TYPE_TV))) {

				HashMap<String, ExtDbRecordTV> theRecords = ExtDbRecordTV.batchFindRecordTV(theIndex.lookupExtDb,
					searchIDs, errors);
				if (null == theRecords) {
					throwError(null);
				}

				if (!theRecords.isEmpty()) {

					SourceEditData theSource;
					SourceEditDataTV theSourceTV;

					// Adding to a scenario, run through the search list and retrieve individual records.

					if (inScenario) {

						ExtDbRecordTV theRecord;

						for (SearchItem theItem : searchItems) {

							theRecord = theRecords.get(theItem.extRecordID);
							if (null == theRecord) {
								continue;
							}

							searchIDs.remove(theItem.extRecordID);

							// If a REPLICATE attribute was not found, check if this record requests automatic
							// replication, see ExtDbRecordTV for details.  Then check the ultimate channel for range,
							// the original channel may be out of range if the replication channel is not.  If out of
							// range just ignore the element, that is not an error, the source is valid it just can't
							// be used in this study.

							if (0 == theItem.replicateToChannel) {
								theItem.replicateToChannel = theRecord.replicateToChannel;
							}
							if (0 == theItem.replicateToChannel) {
								if ((theRecord.channel < study.getMinimumChannel()) ||
										(theRecord.channel > study.getMaximumChannel())) {
									if (null != errors) {
										errors.logMessage(ExtDbRecord.makeMessage(theRecord,
											"Ignored, channel is out of range for study"));
									}
									continue;
								}
							} else {
								if ((theItem.replicateToChannel < study.getMinimumChannel()) ||
										(theItem.replicateToChannel > study.getMaximumChannel())) {
									if (null != errors) {
										errors.logMessage(ExtDbRecord.makeMessage(theRecord,
											"Ignored, replication channel " +
											String.valueOf(theItem.replicateToChannel) +
											" is out of range for study"));
									}
									continue;
								}
							}

							// Check to see if the source is already converted.  This is not entirely paranoia; it is
							// possible for the same record to be used more than once within the same scenario.

							theSource = theIndex.sharedSources.get(theItem.extRecordID);
							if (null == theSource) {
								theSourceTV = SourceEditDataTV.makeSourceTV(theRecord, study, true, errors);
								if (null == theSourceTV) {
									throwError(null);
								}
								theSource = theSourceTV;
								theIndex.sharedSources.put(theItem.extRecordID, theSource);
								studyNewSources.add(theSource);
							} else {
								theSourceTV = (SourceEditDataTV)theSource;
							}

							doScenarioAdd(theSource, theItem.isDesired, theItem.isUndesired, theSourceTV,
								theItem.replicateToChannel);
						}
					}

					// For a generic read-all run, just put all records found in the results.  Ignore records that need
					// replication.  Note this assumes there can be no duplicates, in this case there will only be one
					// search so there is no existing state to merge with (although two data sets may be searched, the
					// IDs set is used to be sure there cannot be any duplication between those searchs).

					if (!scenarioMode) {

						for (ExtDbRecordTV theRecord : theRecords.values()) {

							searchIDs.remove(theRecord.extRecordID);

							if (theRecord.replicateToChannel > 0) {
								continue;
							}

							theSourceTV = SourceEditDataTV.makeSourceTV(theRecord, null, true, errors);
							if (null == theSourceTV) {
								throwError(null);
							}

							sources.add(theSourceTV);
						}
					}
				}
			}

			if (theIndex.lookupExtDb.canProvide(Source.RECORD_TYPE_FM) &&
					((0 == parseRecordType) || (Source.RECORD_TYPE_FM == parseRecordType)) &&
					((0 == parseStudyType) || Study.isRecordTypeAllowed(parseStudyType, Source.RECORD_TYPE_FM))) {

				HashMap<String, ExtDbRecordFM> theRecords = ExtDbRecordFM.batchFindRecordFM(theIndex.lookupExtDb,
					searchIDs, errors);
				if (null == theRecords) {
					throwError(null);
				}

				if (!theRecords.isEmpty()) {

					SourceEditData theSource;
					SourceEditDataFM theSourceFM;

					if (inScenario) {

						ExtDbRecordFM theRecord;

						for (SearchItem theItem : searchItems) {

							theRecord = theRecords.get(theItem.extRecordID);
							if (null == theRecord) {
								continue;
							}

							searchIDs.remove(theItem.extRecordID);

							theSource = theIndex.sharedSources.get(theItem.extRecordID);
							if (null == theSource) {
								theSourceFM = SourceEditDataFM.makeSourceFM(theRecord, study, true, errors);
								if (null == theSourceFM) {
									throwError(null);
								}
								theSource = theSourceFM;
								theIndex.sharedSources.put(theRecord.extRecordID, theSource);
								studyNewSources.add(theSource);
							}

							doScenarioAdd(theSource, theItem.isDesired, theItem.isUndesired, null, 0);
						}
					}

					if (!scenarioMode) {

						for (ExtDbRecordFM theRecord : theRecords.values()) {

							searchIDs.remove(theRecord.extRecordID);

							theSourceFM = SourceEditDataFM.makeSourceFM(theRecord, null, true, errors);
							if (null == theSourceFM) {
								throwError(null);
							}

							sources.add(theSourceFM);
						}
					}
				}
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Add a source to the scenario, first do replication if needed.  If the replication channel matches the
		// original and the original is digital, the replication would do nothing so don't bother.  Only analog sources
		// can be replicated on-channel, which converts them to digital.  For a locked source from external data, check
		// shared sources in case the replication source already exists, if not do the replication and add it.

		private void doScenarioAdd(SourceEditData theSource, boolean isDesired, boolean isUndesired,
				SourceEditDataTV theSourceTV, int replicateToChannel) throws SAXException {

			if ((null != theSourceTV) && (replicateToChannel > 0) &&
					((replicateToChannel != theSourceTV.channel) || !theSourceTV.service.serviceType.digital)) {

				SourceEditDataTV origSource = theSourceTV;
				theSourceTV = null;

				LookupIndex theIndex = null;
				if ((null != origSource.extDbKey) && (null != origSource.extRecordID) && (null != lookupIndex)) {
					if (origSource.extDbKey.equals(lookupIndex.lookupExtDb)) {
						theIndex = lookupIndex;
					} else {
						if ((null != alternateIndex) && origSource.extDbKey.equals(alternateIndex.lookupExtDb)) {
							theIndex = alternateIndex;
						}
					}
				}

				if ((null != theIndex) && origSource.isLocked) {
					theSourceTV = theIndex.sharedReplicationSources.get(replicateToChannel - SourceTV.CHANNEL_MIN).
						get(origSource.extRecordID);
				}

				if (null == theSourceTV) {

					theSourceTV = origSource.replicate(replicateToChannel, errors);
					if (null == theSourceTV) {
						throwError(null);
					}

					if ((null != theIndex) && origSource.isLocked) {
						theIndex.sharedReplicationSources.get(replicateToChannel - SourceTV.CHANNEL_MIN).
							put(origSource.extRecordID, theSourceTV);
					}

					studyNewSources.add(theSourceTV);
				}

				theSource = theSourceTV;
			}

			// Check for duplication, this is a paranoia check that just guards against the same source key appearing
			// more than once in the scenario because that would cause a key-uniqueness exception in SQL.  Such
			// duplicates are silently ignored.  Other rules such as checking for MX records are not implemented here.
			// The scenario is assumed valid because it must have been valid when exported, so those tests are
			// unnecessary.  Manual editing of the XML file may of course make that assumption wrong, hence the
			// paranoia check for duplicates.  But if a user manually edits the XML making the scenario invalid in
			// other ways that corrupt study results but do not cause fatal errors, too bad, they dug their own hole.

			for (Scenario.SourceListItem theItem : scenarioSourceItems) {
				if (theSource.key.intValue() == theItem.key) {
					return;
				}
			}

			// Add the source key to the list for the scenario.  For TV interference-check and wireless to TV
			// interference studies, a scenario can only have one desired TV source (it also must have one, that is
			// checked at the end of the scenario element, see above).  In those cases the first desired found is added
			// as a permanent entry and the undesired flag is cleared.  After that all TV sources have desired cleared
			// and permanent false.  All non-TV sources also have desired cleared and are not permanent.  One further
			// rule, for a TV channel 6 and FM study only a TV channel 6 record can be desired, all other TVs must be
			// undesired-only.  The channel 6 must also be undesired in that case as it may cause interference to the
			// FM records in the scenario, which may also be desireds in that study type.

			boolean isPermanent = false;
			if ((Study.STUDY_TYPE_TV_IX == study.study.studyType) ||
					(Study.STUDY_TYPE_TV_OET74 == study.study.studyType) ||
					(Study.STUDY_TYPE_TV6_FM == study.study.studyType)) {
				if (null != theSourceTV) {
					if (hasDesiredTV) {
						isDesired = false;
					} else {
						if (isDesired) {
							if (Study.STUDY_TYPE_TV6_FM == study.study.studyType) {
								if (6 == theSourceTV.channel) {
									hasDesiredTV = true;
									isUndesired = true;
									isPermanent = true;
								} else {
									isDesired = false;
								}
							} else {
								hasDesiredTV = true;
								isUndesired = false;
								isPermanent = true;
							}
						}
					}
				} else {
					if (Study.STUDY_TYPE_TV6_FM != study.study.studyType) {
						isDesired = false;
					}
				}
			}

			scenarioSourceItems.add(new Scenario.SourceListItem(theSource.key.intValue(), isDesired, isUndesired,
				isPermanent));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Read source records from text files, this may be for import into a generic data set, a study, or outside of any
	// context.  Various delimiter characters are supported which may be inferred from the file name or auto-detected,
	// see AppCore.readAndParseLine().  There are different formats for the main station file depending on record type.
	// Wireless is an older format that previously was imported to create a single-import data set accessed through an
	// ExtDbRecord subclass, before generic data sets existed.  The generic structure provides greater functionality so
	// the old single-import type for wireless is no longer supported.  The import file format is still used for
	// backwards compatibility.  The TV and FM formats are similar to wireless.

	private static final String PATTERN_DATA_START = "$pattern";
	private static final String PATTERN_DATA_END = "$end";

	public static ArrayList<SourceEditData> readFromText(ExtDb extDb, File stationFile, File patternFile,
			ErrorLogger errors) {
		if (!extDb.isGeneric()) {
			if (null != errors) {
				errors.reportError("Unsupported import format for station data type");
			}
			return null;
		}
		return readFromText(extDb.dbID, extDb.getDefaultRecordType(), stationFile, patternFile, extDb, null, errors);
	}

	public static ArrayList<SourceEditData> readFromText(StudyEditData study, int recordType, File stationFile,
			File patternFile, ErrorLogger errors) {
		if (!Study.isRecordTypeAllowed(study.study.studyType, recordType)) {
			if (null != errors) {
				errors.reportError("Unsupported record type for study type");
			}
			return null;
		}
		return readFromText(study.dbID, recordType, stationFile, patternFile, null, study, errors);
	}

	public static ArrayList<SourceEditData> readFromText(String dbID, int recordType, File stationFile,
			File patternFile, ErrorLogger errors) {
		return readFromText(dbID, recordType, stationFile, patternFile, null, null, errors);
	}

	private static ArrayList<SourceEditData> readFromText(String dbID, int recordType, File stationFile,
			File patternFile, ExtDb extDb, StudyEditData study, ErrorLogger errors) {

		ArrayList<SourceEditData> sources = new ArrayList<SourceEditData>();

		BufferedReader reader = null;

		String fileName = "", errmsg = null;
		AppCore.LineCounter lineCount = null;

		try {

			// If the patternFile argument is non-null first read pattern data from that file.  Pattern data may also
			// be in-line in the main file, see readSources().

			HashMap<Integer, AntPattern> patterns = new HashMap<Integer, AntPattern>();

			if (null != patternFile) {

				fileName = patternFile.getName();

				reader = new BufferedReader(new FileReader(patternFile));

				lineCount = new AppCore.LineCounter();
				lineCount.setDelimiterForFile(patternFile);

				errmsg = readPatterns(dbID, reader, fileName, lineCount, patterns, errors);

				try {reader.close();} catch (IOException ie) {};

				reader = null;
				fileName = "";
				lineCount = null;
			}

			// If no error, read from the main station file.

			if (null == errmsg) {

				fileName = stationFile.getName();

				reader = new BufferedReader(new FileReader(stationFile));

				lineCount = new AppCore.LineCounter();
				lineCount.setDelimiterForFile(stationFile);

				errmsg = readSources(dbID, recordType, extDb, study, reader, fileName, lineCount, patterns, sources,
					errors);
			}

		} catch (FileNotFoundException fe) {
			errmsg = "Data file '" + fileName + "' could not be opened";

		} catch (IOException ie) {
			errmsg = "An I/O error occurred:\n" + ie + "\n";

		} catch (Throwable t) {
			errmsg = "An unexpected error occurred:\n" + t + "\n";
			AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", t);
		}

		// Close file if needed, check for error.

		if (null != reader) {
			try {reader.close();} catch (IOException ie) {};
		}

		if (null != errmsg) {
			if (null != errors) {
				if ((null != lineCount) && (lineCount.get() > 0)) {
					errmsg = errmsg +  "in '" + fileName + "' at line " + lineCount;
				}
				errors.reportError(errmsg);
			}
			return null;
		}

		return sources;
	}

	// Read pattern data from text.  The full pattern tabulation for azimuth and elevation patterns is in a single
	// row, with ID, type, and name, then pattern points.  The pattern types are 'A' for azimuth, 'E' for elevation.
	// Matrix patterns have type 'M' and span multiple rows, each row is an elevation pattern slice.  Each point has
	// the degree and field values separated by a semicolon within one text field.

	private static String readPatterns(String dbID, BufferedReader reader, String fileName,
			AppCore.LineCounter lineCount, HashMap<Integer, AntPattern> patterns, ErrorLogger errors)
			throws IOException {

		ArrayList<AntPattern.AntPoint> points;

		Integer patKey = null;
		AntPattern pat = null;

		int antID, i, pointStart;
		char patType;
		boolean isAzPat, inMatrixPat = false;
		String str, patName = null;
		String[] fields, patFields;
		double degree, field, lastDegree, fieldMax, matrixFieldMax = 0.,
			lastMatrixDegree = AntPattern.AZIMUTH_MIN - 1.;
		AntPattern matAzPat;

		Pattern pointPattern = Pattern.compile(";");

		while (true) {

			fields = AppCore.readAndParseLine(reader, lineCount);
			if (null == fields) {
				break;
			}

			// Check for explicit end code, this will appear when the data is embedded in the main station data file.

			if ((1 == fields.length) && fields[0].toLowerCase().startsWith(PATTERN_DATA_END)) {
				break;
			}

			if (fields.length < (AntPattern.PATTERN_REQUIRED_POINTS + 3)) {
				return "Bad field count ";
			}

			// Extract and check the ID.  A matrix pattern spans multiple rows with the same antenna ID, it is not
			// validated and saved until all rows have been parsed.

			antID = -1;
			str = fields[0];
			if (str.length() > 0) {
				try {
					antID = Integer.parseInt(str);
				} catch (NumberFormatException ne) {
				}
			}
			if (antID <= 0) {
				return "Missing or bad antenna ID ";
			}

			if (inMatrixPat && (antID != patKey.intValue())) {
				if ((matrixFieldMax < AntPattern.FIELD_MAX_CHECK) && (null != errors)) {
					errors.logMessage("Pattern does not contain a 1 for antenna ID " + patKey + " in '" + fileName +
						"' at line " + lineCount);
				}
				if (pat.isDataValid(errors)) {
					matAzPat = pat.normalizeVerticalMatrix();
					if (null != matAzPat) {
						patterns.put(Integer.valueOf(-(patKey.intValue())), matAzPat);
					}
					patterns.put(patKey, pat);
					inMatrixPat = false;
					matrixFieldMax = 0.;
					lastMatrixDegree = AntPattern.AZIMUTH_MIN - 1.;
				} else {
					return "Error occurred ";
				}
			}

			patKey = Integer.valueOf(antID);

			// Check pattern type.

			patType = ' ';
			str = fields[1];
			if (str.length() > 0) {
				patType = str.toUpperCase().charAt(0);
			}
			if (('A' != patType) && ('E' != patType) && ('M' != patType)) {
				return "Missing or bad pattern type ";
			}
			isAzPat = ('A' == patType);

			// The name field is ignored after the first line of a matrix pattern, otherwise must not be empty.

			if (!inMatrixPat) {
				patName = fields[2];
				if (0 == patName.length()) {
					return "Missing pattern name ";
				}
				if (patName.length() > Source.MAX_PATTERN_NAME_LENGTH) {
					patName = patName.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
					if (null != errors) {
						errors.logMessage("Pattern name too long, truncated, in '" + fileName + "' at line " +
							lineCount);
					}
				}
			}

			// Matrix pattern rows have an extra field providing the slice azimuth.

			if ('M' == patType) {

				str = fields[3];
				degree = AntPattern.AZIMUTH_MIN - 1.;
				if (str.length() > 0) {
					try {
						degree = Double.parseDouble(str);
					} catch (NumberFormatException ne) {
					}
				}
				degree = Math.rint(degree * AntPattern.AZIMUTH_ROUND) / AntPattern.AZIMUTH_ROUND;
				if ((degree < AntPattern.AZIMUTH_MIN) || (degree > AntPattern.AZIMUTH_MAX)) {
					return "Bad matrix slice azimuth ";
				}
				if (degree <= lastMatrixDegree) {
					return "Matrix slices out of order or duplicated ";
				}
				lastMatrixDegree = degree;

				if (!inMatrixPat) {
					pat = new AntPattern(dbID, AntPattern.PATTERN_TYPE_VERTICAL, patName);
					inMatrixPat = true;
				}

				points = pat.addSlice(Double.valueOf(degree));
				pointStart = 4;

			} else {

				if (isAzPat) {
					pat = new AntPattern(dbID, AntPattern.PATTERN_TYPE_HORIZONTAL, patName);
				} else {
					pat = new AntPattern(dbID, AntPattern.PATTERN_TYPE_VERTICAL, patName);
				}

				points = pat.getPoints();
				pointStart = 3;
			}

			// Extract points.

			if (isAzPat) {
				lastDegree = AntPattern.AZIMUTH_MIN - 1.;
			} else {
				lastDegree = AntPattern.DEPRESSION_MIN - 1.;
			}
			fieldMax = 0.;

			for (i = pointStart; i < fields.length; i++) {

				patFields = pointPattern.split(fields[i]);
				if (patFields.length != 2) {
					return "Bad pattern point format at point " + (i - 2) + " ";
				}

				str = patFields[0];
				if (isAzPat) {
					degree = AntPattern.AZIMUTH_MIN - 1.;
				} else {
					degree = AntPattern.DEPRESSION_MIN - 1.;
				}
				if (str.length() > 0) {
					try {
						degree = Double.parseDouble(str);
					} catch (NumberFormatException ne) {
					}
				}
				if (isAzPat) {
					degree = Math.rint(degree * AntPattern.AZIMUTH_ROUND) / AntPattern.AZIMUTH_ROUND;
					if ((degree < AntPattern.AZIMUTH_MIN) || (degree > AntPattern.AZIMUTH_MAX)) {
						return "Bad azimuth at point " + (i - 2) + " ";
					}
				} else {
					degree = Math.rint(degree * AntPattern.DEPRESSION_ROUND) / AntPattern.DEPRESSION_ROUND;
					if ((degree < AntPattern.DEPRESSION_MIN) || (degree > AntPattern.DEPRESSION_MAX)) {
						return "Bad vertical angle at point " + (i - 2) + " ";
					}
				}
				if (degree <= lastDegree) {
					return "Pattern points out of order or duplicated at point " + (i - 2) + " ";
				}
				lastDegree = degree;

				field = -1.;
				str = patFields[1];
				if (str.length() > 0) {
					try {
						field = Double.parseDouble(str);
					} catch (NumberFormatException ne) {
					}
				}
				field = Math.rint(field * AntPattern.FIELD_ROUND) / AntPattern.FIELD_ROUND;
				if ((field < 0.) || (field > AntPattern.FIELD_MAX)) {
					return "Bad relative field at point " + (i - 2) + " ";
				}
				if (field < AntPattern.FIELD_MIN) {
					field = AntPattern.FIELD_MIN;
				}
				if (field > fieldMax) {
					fieldMax = field;
				}

				points.add(new AntPattern.AntPoint(degree, field));
			}

			// In matrix pattern data, the max-field test applies to the entire set of slices not each slice.  The
			// pattern data will be normalized and an azimuth pattern derived if needed later.

			if (inMatrixPat) {
				if (fieldMax > matrixFieldMax) {
					matrixFieldMax = fieldMax;
				}
			} else {
				if ((fieldMax < AntPattern.FIELD_MAX_CHECK) && (null != errors)) {
					errors.logMessage("Pattern does not contain a 1 for antenna ID " + patKey + " in '" + fileName +
						"' at line " + lineCount);
				}
			}

			// Matrix patterns can't be validated and added until all rows are read.

			if (!inMatrixPat) {
				if (pat.isDataValid(errors)) {
					patterns.put(patKey, pat);
				} else {
					return "Error occurred ";
				}
			}
		}

		if (inMatrixPat) {
			if ((matrixFieldMax < AntPattern.FIELD_MAX_CHECK) && (null != errors)) {
				errors.logMessage("Pattern does not contain a 1 for antenna ID " + patKey + " in '" + fileName +
					"' at line " + lineCount);
			}
			if (pat.isDataValid(errors)) {
				matAzPat = pat.normalizeVerticalMatrix();
				if (null != matAzPat) {
					patterns.put(Integer.valueOf(-(patKey.intValue())), matAzPat);
				}
				patterns.put(patKey, pat);
			} else {
				return "Error occurred ";
			}
		}

		return null;
	}

	// Read station record data from text, format varies with record type.  Generally one row is one record, except for
	// a TV DTS record which is a sequential group of rows, the first is the parent, followed usually by a authorized
	// facility row but that is optional, then two or more individual site rows.  There may also be embedded in-line
	// pattern data sections, see readPatterns().

	private static String readSources(String dbID, int recordType, ExtDb extDb, StudyEditData study,
			BufferedReader reader, String fileName, AppCore.LineCounter lineCount,
			HashMap<Integer, AntPattern> patterns, ArrayList<SourceEditData> sources, ErrorLogger errors)
			throws IOException {

		String callSign, sectorID, serviceStr, channelStr, stationClassStr, status, fileNumber, facilityIDStr, city,
			state, countryStr, zoneStr, dtsDistanceStr, latitudeStr, longitudeStr, hamslStr, haatStr, erpStr,
			offsetStr, maskStr, ibocStr, azAntIDStr, orientStr, elAntIDStr, eTiltStr, mTiltStr, mTiltOrientStr;
		Service service;
		boolean isTV, isDRT, isNonDTS, isDTSParent, isDTSAuth, isDTSSite, isWL, isFM, isIBOC;
		int channel, stationClass, statusType, facilityID, siteNumber, siteCount = 0, antID = 0;
		Country country;
		Zone zone;
		double dtsDistance, latitude, longitude, hamsl, haat, erp, iboc, orient, eTilt, mTilt, mTiltOrient;
		FrequencyOffset offset;
		EmissionMask mask;
		Integer patKey;
		AntPattern azPat, elPat, matAzPat;

		SourceEditData source = null;
		SourceEditDataTV sourceTV = null, sourceDTS = null, sourceDTSAuth = null;
		SourceEditDataWL sourceWL = null;
		SourceEditDataFM sourceFM = null;

		String importDate = AppCore.formatDate(new java.util.Date());

		String str;
		String[] fields;

		while (true) {

			fields = AppCore.readAndParseLine(reader, lineCount);
			if (null == fields) {
				break;
			}

			// Check for an in-line pattern data section.  This can be anywhere and can appear more than once, and a
			// pattern may replace one with the same key read previously.

			if ((1 == fields.length) && fields[0].toLowerCase().startsWith(PATTERN_DATA_START)) {
				String errmsg = readPatterns(dbID, reader, fileName, lineCount, patterns, errors);
				if (null != errmsg) {
					return errmsg;
				}
				continue;
			}

			// Set all properties to undefined/invalid/default.

			callSign = null;
			sectorID = null;
			serviceStr = null;
			service = null;
			isTV = false;
			isDRT = false;
			isNonDTS = false;
			isDTSParent = false;
			isDTSAuth = false;
			isDTSSite = false;
			isWL = false;
			isFM = false;
			channelStr = null;
			channel = -1;
			stationClassStr = null;
			stationClass = ExtDbRecordFM.FM_CLASS_OTHER;
			status = null;
			statusType = ExtDbRecord.STATUS_TYPE_OTHER;
			fileNumber = null;
			facilityIDStr = null;
			facilityID = -1;
			city = null;
			state = null;
			countryStr = null;
			country = null;
			zoneStr = null;
			zone = Zone.getNullObject();
			siteNumber = -1;
			dtsDistanceStr = null;
			dtsDistance = Source.DISTANCE_MIN - 1.;
			latitudeStr = null;
			latitude = Source.LATITUDE_MIN - 1.;
			longitudeStr = null;
			longitude = Source.LONGITUDE_MIN - 1.;
			hamslStr = null;
			hamsl = Source.HEIGHT_MIN - 1.;
			haatStr = null;
			haat = Source.HEIGHT_MIN - 1.;
			erpStr = null;
			erp = Source.ERP_MIN - 1.;
			offsetStr = null;
			offset = FrequencyOffset.getNullObject();
			maskStr = null;
			mask = EmissionMask.getNullObject();
			ibocStr = null;
			iboc = SourceFM.IBOC_FRACTION_MIN - 1.;
			isIBOC = false;
			azAntIDStr = null;
			azPat = null;
			orientStr = null;
			orient = AntPattern.AZIMUTH_MIN - 1.;
			elAntIDStr = null;
			patKey = null;
			elPat = null;
			matAzPat = null;
			eTiltStr = null;
			eTilt = AntPattern.DEPRESSION_MIN - 1.;
			mTiltStr = null;
			mTilt = AntPattern.DEPRESSION_MIN - 1.;
			mTiltOrientStr = null;
			mTiltOrient = AntPattern.AZIMUTH_MIN - 1.;

			// Check field count per record type, extract fields.

			switch (recordType) {

				case Source.RECORD_TYPE_TV: {

					isTV = true;

					if ((fields.length < 13) || (fields.length > 25)) {
						return "Bad field count ";
					}

					// TV records may be non-DTS, DTS parent, DTS authorized facility, or DTS site.  Use site number
					// and service fields to determine the type.  The site number contains "P" for a DTS parent or "R"
					// for a DTS authorized facility.  DTS site vs. non-DTS is determined by the service code and
					// whether or not a DTS parent preceded the current set of DTS records.  Service code is optional
					// on a DTS site, if provided it must match the parent, else it is copied from the parent.

					str = fields[10].toUpperCase();
					if (str.length() > 0) {
						if (str.equals("P")) {
							isDTSParent = true;
							siteNumber = 0;
						} else {
							if (str.equals("R")) {
								if (null == sourceDTS) {
									return "No parent record for DTS authorized facility ";
								}
								if (null != sourceDTSAuth) {
									return "Multiple DTS authorized facilities for same parent ";
								}
								isDTSAuth = true;
								siteNumber = 0;
							} else {
								try {
									siteNumber = Integer.parseInt(str);
								} catch (NumberFormatException ne) {
									return "Bad site number ";
								}
							}
						}
					}

					str = fields[1].toUpperCase();
					if (str.equals("DRT")) {
						isDRT = true;
						str = Service.LD_CODE;
					}
					if (str.length() > 0) {
						service = service.getService(str);
						if (null != service) {
							if (service.isDTS) {
								if (!isDTSParent) {
									if (isDTSAuth) {
										return "Authorized facility record cannot be DTS service ";
									}
									if (null == sourceDTS) {
										return "No parent record for DTS site record ";
									}
									if (service.key != sourceDTS.service.key) {
										return "DTS site must be same service as parent ";
									}
									isDTSSite = true;
								}
							} else {
								if (isDTSParent) {
									return "Parent record must be DTS service ";
								}
								if (!isDTSAuth) {
									isNonDTS = true;
								}
							}
						}
					} else {
						if (!isDTSParent && !isDTSAuth && (null != sourceDTS)) {
							isDTSSite = true;
							service = sourceDTS.service;
						}
					}

					if (null == service) {
						return "Unknown or missing service code ";
					}

					if (!service.isTV()) {
						return "Invalid service for record type ";
					}

					// Site number must be specified for DRT or DTS site, for non-DTS it is optional, default 1.

					if (siteNumber < 0) {
						if (isDRT || isDTSSite) {
							return "Missing site number ";
						}
						siteNumber = 1;
					}

					// If previous records were a DTS and this one is either non-DTS or a new DTS parent, process the
					// previous record.  If the parent did not have an authorized facility record for full service just
					// make a "dummy" authorized facility record.  For a non-full-service there must be an authorized
					// facility, and the parent coordinates are set to that.  Then check validity and add to results.

					if ((isNonDTS || isDTSParent) && (null != sourceDTS)) {
						if (sourceDTS.service.isFullService()) {
							if ((null == sourceDTSAuth) && (null == sourceDTS.makeDTSAuthorizedSource(extDb, errors))) {
								return "Error occurred ";
							}
						} else {
							if (null == sourceDTSAuth) {
								return "No authorized facility record for DTS parent ";
							}
							sourceDTS.location.setLatLon(sourceDTSAuth.location);
						}
						if (!sourceDTS.isDataValid(errors)) {
							return "Error occurred ";
						}
						sources.add(sourceDTS);
						sourceDTS = null;
						sourceDTSAuth = null;
					}

					// Extract fields for parsing below.

					if (isNonDTS) {
						callSign = fields[0];
						channelStr = fields[2];
						status = fields[3];
						fileNumber = fields[4];
						facilityIDStr = fields[5];
						city = fields[6];
						state = fields[7];
						countryStr = fields[8].toUpperCase();
						zoneStr = fields[9].toUpperCase();
						latitudeStr = fields[11];
						longitudeStr = fields[12];
						if (fields.length > 13) {
							hamslStr = fields[13];
						} else {
							hamslStr = "";
						}
						if (fields.length > 14) {
							haatStr = fields[14];
						} else {
							haatStr = "";
						}
						if (fields.length > 15) {
							erpStr = fields[15];
						} else {
							erpStr = "";
						}
						if (fields.length > 16) {
							azAntIDStr = fields[16];
						} else {
							azAntIDStr = "";
						}
						if (fields.length > 17) {
							orientStr = fields[17];
						} else {
							orientStr = "";
						}
						if (fields.length > 18) {
							elAntIDStr = fields[18];
						} else {
							elAntIDStr = "";
						}
						if (fields.length > 19) {
							eTiltStr = fields[19];
						} else {
							eTiltStr = "";
						}
						if (fields.length > 20) {
							mTiltStr = fields[20];
						} else {
							mTiltStr = "";
						}
						if (fields.length > 21) {
							mTiltOrientStr = fields[21];
						} else {
							mTiltOrientStr = "";
						}
						if (fields.length > 22) {
							maskStr = fields[22].toUpperCase();
						} else {
							maskStr = "";
						}
						if (fields.length > 23) {
							offsetStr = fields[23].toUpperCase();
						} else {
							offsetStr = "";
						}
					}

					// Latitude, longitude, and the DTS boundary definition are undefined on a non-full-service DTS
					// parent, anything in those fields is ignored.  The authorized facility coordinates get assigned
					// to the parent in this case, see elsewhere.

					if (isDTSParent) {
						callSign = fields[0];
						channelStr = fields[2];
						status = fields[3];
						fileNumber = fields[4];
						facilityIDStr = fields[5];
						city = fields[6];
						state = fields[7];
						countryStr = fields[8].toUpperCase();
						zoneStr = fields[9].toUpperCase();
						if (service.isFullService()) {
							latitudeStr = fields[11];
							longitudeStr = fields[12];
							if (fields.length > 24) {
								dtsDistanceStr = fields[24].toUpperCase();
							} else {
								dtsDistanceStr = "";
							}
						} else {
							dtsDistanceStr = "";
						}
						if (fields.length > 22) {
							maskStr = fields[22].toUpperCase();
						} else {
							maskStr = "";
						}
						if (fields.length > 23) {
							offsetStr = fields[23].toUpperCase();
						} else {
							offsetStr = "";
						}
					}

					if (isDTSAuth) {
						channelStr = fields[2];
						status = fields[3];
						fileNumber = fields[4];
						zoneStr = fields[9].toUpperCase();
						latitudeStr = fields[11];
						longitudeStr = fields[12];
						if (fields.length > 13) {
							hamslStr = fields[13];
						} else {
							hamslStr = "";
						}
						if (fields.length > 14) {
							haatStr = fields[14];
						} else {
							haatStr = "";
						}
						if (fields.length > 15) {
							erpStr = fields[15];
						} else {
							erpStr = "";
						}
						if (fields.length > 16) {
							azAntIDStr = fields[16];
						} else {
							azAntIDStr = "";
						}
						if (fields.length > 17) {
							orientStr = fields[17];
						} else {
							orientStr = "";
						}
						if (fields.length > 18) {
							elAntIDStr = fields[18];
						} else {
							elAntIDStr = "";
						}
						if (fields.length > 19) {
							eTiltStr = fields[19];
						} else {
							eTiltStr = "";
						}
						if (fields.length > 20) {
							mTiltStr = fields[20];
						} else {
							mTiltStr = "";
						}
						if (fields.length > 21) {
							mTiltOrientStr = fields[21];
						} else {
							mTiltOrientStr = "";
						}
						if (fields.length > 22) {
							maskStr = fields[22].toUpperCase();
						} else {
							maskStr = "";
						}
						if (fields.length > 23) {
							offsetStr = fields[23].toUpperCase();
						} else {
							offsetStr = "";
						}
					}

					if (isDTSSite) {
						latitudeStr = fields[11];
						longitudeStr = fields[12];
						if (fields.length > 13) {
							hamslStr = fields[13];
						} else {
							hamslStr = "";
						}
						if (fields.length > 14) {
							haatStr = fields[14];
						} else {
							haatStr = "";
						}
						if (fields.length > 15) {
							erpStr = fields[15];
						} else {
							erpStr = "";
						}
						if (fields.length > 16) {
							azAntIDStr = fields[16];
						} else {
							azAntIDStr = "";
						}
						if (fields.length > 17) {
							orientStr = fields[17];
						} else {
							orientStr = "";
						}
						if (fields.length > 18) {
							elAntIDStr = fields[18];
						} else {
							elAntIDStr = "";
						}
						if (fields.length > 19) {
							eTiltStr = fields[19];
						} else {
							eTiltStr = "";
						}
						if (fields.length > 20) {
							mTiltStr = fields[20];
						} else {
							mTiltStr = "";
						}
						if (fields.length > 21) {
							mTiltOrientStr = fields[21];
						} else {
							mTiltOrientStr = "";
						}
					}

					break;
				}

				// Wireless record.

				case Source.RECORD_TYPE_WL: {

					isWL = true;

					if ((fields.length < 7) || (fields.length > 17)) {
						return "Bad field count ";
					}

					serviceStr = Service.WL_CODE;

					callSign = fields[0];
					sectorID = fields[1];
					latitudeStr = fields[2];
					longitudeStr = fields[3];
					hamslStr = fields[4];
					haatStr = fields[5];
					erpStr = fields[6];
					if (fields.length > 7) {
						azAntIDStr = fields[7];
					} else {
						azAntIDStr = "";
					}
					if (fields.length > 8) {
						orientStr = fields[8];
					} else {
						orientStr = "";
					}
					if (fields.length > 9) {
						elAntIDStr = fields[9];
					} else {
						elAntIDStr = "";
					}
					if (fields.length > 10) {
						eTiltStr = fields[10];
					} else {
						eTiltStr = "";
					}
					if (fields.length > 11) {
						mTiltStr = fields[11];
					} else {
						mTiltStr = "";
					}
					if (fields.length > 12) {
						mTiltOrientStr = fields[12];
					} else {
						mTiltOrientStr = "";
					}
					if (fields.length > 13) {
						fileNumber = fields[13];
					} else {
						fileNumber = "";
					}
					if (fields.length > 14) {
						city = fields[14];
					} else {
						city = "";
					}
					if (fields.length > 15) {
						state = fields[15];
					} else {
						state = "";
					}
					if (fields.length > 16) {
						countryStr = fields[16].toUpperCase();
					} else {
						countryStr = Country.US_CODE;
					}

					break;
				}

				// FM record.

				case Source.RECORD_TYPE_FM: {

					isFM = true;
	
					if ((fields.length < 15) || (fields.length > 22)) {
						return "Bad field count ";
					}

					callSign = fields[0];
					serviceStr = fields[1].toUpperCase();
					channelStr = fields[2];
					stationClassStr = fields[3].toUpperCase();
					status = fields[4].toUpperCase();
					fileNumber = fields[5];
					facilityIDStr = fields[6];
					city = fields[7];
					state = fields[8];
					countryStr = fields[9].toUpperCase();
					latitudeStr = fields[10];
					longitudeStr = fields[11];
					hamslStr = fields[12];
					haatStr = fields[13];
					erpStr = fields[14];
					if (fields.length > 15) {
						ibocStr = fields[15];
					} else {
						ibocStr = "";
					}
					if (fields.length > 16) {
						azAntIDStr = fields[16];
					} else {
						azAntIDStr = "";
					}
					if (fields.length > 17) {
						orientStr = fields[17];
					} else {
						orientStr = "";
					}
					if (fields.length > 18) {
						elAntIDStr = fields[18];
					} else {
						elAntIDStr = "";
					}
					if (fields.length > 19) {
						eTiltStr = fields[19];
					} else {
						eTiltStr = "";
					}
					if (fields.length > 20) {
						mTiltStr = fields[20];
					} else {
						mTiltStr = "";
					}
					if (fields.length > 21) {
						mTiltOrientStr = fields[21];
					} else {
						mTiltOrientStr = "";
					}

					break;
				}

				default: {
					return "Unknown or unsupported station data type ";
				}
			}

			// Parse and error-check all defined field values.  The call sign (or cell site ID for wireless) must not
			// be empty and has limited length, if exceeded truncate and log an informational message.  The wireless
			// sector ID may be blank, if not it also has limited length.

			if (null != callSign) {
				if (0 == callSign.length()) {
					return "Missing " + (isWL ? "cell site ID " : "call sign ");
				}
				if (callSign.length() > Source.MAX_CALL_SIGN_LENGTH) {
					callSign = callSign.substring(0, Source.MAX_CALL_SIGN_LENGTH);
					if (null != errors) {
						errors.logMessage((isWL ? "Cell site ID" : "Call sign") + " too long, truncated, in '" +
							fileName + "' at line " + lineCount);
					}
				}
			}

			if (null != sectorID) {
				if (sectorID.length() > Source.MAX_SECTOR_ID_LENGTH) {
					sectorID = sectorID.substring(0, Source.MAX_SECTOR_ID_LENGTH);
					if (null != errors) {
						errors.logMessage("Sector ID too long, truncated, in '" + fileName + "' at line " +
							lineCount);
					}
				}
			}

			if (null != serviceStr) {
				service = Service.getService(serviceStr);
				if (null == service) {
					return "Unknown or missing service code ";
				}
				if ((isTV && !service.isTV()) || (isWL && !service.isWL()) || (isFM && !service.isFM())) {
					return "Invalid service for record type ";
				}
			}

			if (null != channelStr) {
				try {
					channel = Integer.parseInt(channelStr);
				} catch (NumberFormatException ne) {
				}
				if ((isFM && ((channel < SourceFM.CHANNEL_MIN) || (channel > SourceFM.CHANNEL_MAX))) ||
						(isTV && ((channel < SourceTV.CHANNEL_MIN) || (channel > SourceTV.CHANNEL_MAX)))) {
					return "Bad channel number ";
				}
			}

			if (null != stationClassStr) {
				for (int i = 1; i < ExtDbRecordFM.FM_CLASS_CODES.length; i++) {
					if (stationClassStr.equals(ExtDbRecordFM.FM_CLASS_CODES[i])) {
						stationClass = i;
						break;
					}
				}
			}

			// Status may be blank.

			if (null != status) {
				statusType = ExtDbRecord.getStatusType(status);
			}

			// File number has restricted length, truncate silently.  May be blank.

			if (null != fileNumber) {
				if (fileNumber.length() > Source.MAX_FILE_NUMBER_LENGTH) {
					fileNumber = fileNumber.substring(0, Source.MAX_FILE_NUMBER_LENGTH);
				}
			}

			if (null != facilityIDStr) {
				try {
					facilityID = Integer.parseInt(facilityIDStr);
				} catch (NumberFormatException ne) {
				}
				if (facilityID <= 0) {
					return "Missing or bad facility ID ";
				}
			}

			// City and state have restricted length, truncate silently.  May be blank for WL but not TV or FM.

			if (null != city) {
				if (city.length() > Source.MAX_CITY_LENGTH) {
					city = city.substring(0, Source.MAX_CITY_LENGTH);
				}
				if (!isWL) {
					if (0 == city.length()) {
						return "Missing city name ";
					}
				}
			}

			if (null != state) {
				if (state.length() > Source.MAX_STATE_LENGTH) {
					state = state.substring(0, Source.MAX_STATE_LENGTH);
				}
				if (!isWL) {
					if (0 == state.length()) {
						return "Missing state code ";
					}
				}
			}

			if (null != countryStr) {
				country = Country.getCountry(countryStr);
				if (null == country) {
					return "Missing or unknown country code ";
				}
			}

			if (null != zoneStr) {
				zone = Zone.getZone(zoneStr);
			}

			// The DTS boundary field may be a single radius distance, or a sectors definition.  Parse as a radius
			// first, if that fails try as a sectors list.  If blank, default to radius 0.

			if (null != dtsDistanceStr) {
				if (dtsDistanceStr.length() > 0) {
					try {
						dtsDistance = Double.parseDouble(dtsDistanceStr);
						if ((dtsDistance != 0.) && ((dtsDistance < Source.DISTANCE_MIN) ||
								(dtsDistance > Source.DISTANCE_MAX))) {
							return "Bad DTS maximum distance ";
						}
						dtsDistanceStr = "";
					} catch (NumberFormatException ne) {
						str = GeoSectors.validateString(dtsDistanceStr);
						if (null != str) {
							return str + " ";
						} else {
							dtsDistance = 0.;
						}
					}
				} else {
					dtsDistance = 0.;
				}
			}

			if (null != latitudeStr) {
				if (latitudeStr.length() > 0) {
					try {
						latitude = Double.parseDouble(latitudeStr);
					} catch (NumberFormatException ne) {
					}
				}
				if ((latitude < Source.LATITUDE_MIN) || (latitude > Source.LATITUDE_MAX)) {
					return "Missing or bad latitude ";
				}
			}

			// Longitude is negative west for wireless records, positive west for others.

			if (null != longitudeStr) {
				if (longitudeStr.length() > 0) {
					try {
						longitude = Double.parseDouble(longitudeStr);
						if (isWL) {
							longitude = -longitude;
						}
					} catch (NumberFormatException ne) {
					}
				}
				if ((longitude < Source.LONGITUDE_MIN) || (longitude > Source.LONGITUDE_MAX)) {
					return "Missing or bad longitude ";
				}
			}

			if (null != hamslStr) {
				if (hamslStr.length() > 0) {
					try {
						hamsl = Double.parseDouble(hamslStr);
					} catch (NumberFormatException ne) {
					}
				}
				if ((hamsl < Source.HEIGHT_MIN) || (hamsl > Source.HEIGHT_MAX)) {
					return "Missing or bad AMSL height ";
				}
			}

			// HAAT is optional, default to derive.

			if (null != haatStr) {
				if (haatStr.length() > 0) {
					try {
						haat = Double.parseDouble(haatStr);
					} catch (NumberFormatException ne) {
					}
					if ((haat < Source.HEIGHT_MIN) || (haat > Source.HEIGHT_MAX)) {
						return "Missing or bad HAAT ";
					}
				} else {
					haat = Source.HEIGHT_DERIVE;
				}
			}

			if (null != erpStr) {
				if (erpStr.length() > 0) {
					try {
						erp = Double.parseDouble(erpStr);
					} catch (NumberFormatException ne) {
					}
				}
				if ((erp < Source.ERP_MIN) || (erp > Source.ERP_MAX)) {
					return "Missing or bad ERP ";
				}
			}

			if (null != offsetStr) {
				offset = FrequencyOffset.getFrequencyOffset(offsetStr);
			}

			// Emission mask field is ignored unless required by the service type.

			if ((null != maskStr) && service.serviceType.needsEmissionMask) {
				mask = EmissionMask.getEmissionMask(maskStr);
				if (mask.key <= 0) {
					return "Missing or bad emission mask code ";
				}
			}

			if (null != ibocStr) {
				if (ibocStr.length() > 0) {
					try {
						iboc = Double.parseDouble(ibocStr);
					} catch (NumberFormatException ne) {
					}
					if (iboc >= (SourceFM.IBOC_FRACTION_MIN * 100.)) {
						iboc /= 100.;
					}
					if ((iboc < SourceFM.IBOC_FRACTION_MIN) || (iboc > SourceFM.IBOC_FRACTION_MAX)) {
						return "Bad IBOC power ";
					}
					isIBOC = true;
				} else {
					iboc = SourceFM.IBOC_FRACTION_DEF;
				}
			}

			// Because patterns are in a single ID space, have to check to be sure the correct type of pattern is
			// being referenced, i.e. horizontal pattern for azimuth, vertical pattern for elevation.

			if ((null != azAntIDStr) && (azAntIDStr.length() > 0)) {
				try {
					antID = Integer.parseInt(azAntIDStr);
					if (antID > 0) {
						azPat = patterns.get(Integer.valueOf(antID));
					}
				} catch (NumberFormatException ne) {
				}
				if (null == azPat) {
					return "Bad or unknown azimuth antenna ID ";
				}
				if (AntPattern.PATTERN_TYPE_HORIZONTAL != azPat.type) {
					return "Wrong type for azimuth pattern ";
				}
			}

			// Orientation and tilt values are always parsed even if pattern ID indicates omni, and all default to 0
			// if the field is empty, again regardless of omni.

			if (null != orientStr) {
				if (orientStr.length() > 0) {
					try {
						orient = Math.IEEEremainder(Double.parseDouble(orientStr), 360.);
						if (orient < 0.) orient += 360.;
					} catch (NumberFormatException ne) {
					}
					if ((orient < AntPattern.AZIMUTH_MIN) || (orient > AntPattern.AZIMUTH_MAX)) {
						return "Bad pattern orientation ";
					}
				} else {
					orient = 0.;
				}
			}

			// If the elevation pattern is a matrix it may have been normalized producing a peak-value azimuth envelope
			// that will be used as the azimuth pattern for the record, overriding any explicit azimuth pattern.  Log
			// a warning if an override happens, it really shouldn't.

			if ((null != elAntIDStr) && (elAntIDStr.length() > 0)) {
				try {
					antID = Integer.parseInt(elAntIDStr);
					if (antID > 0) {
						elPat = patterns.get(Integer.valueOf(antID));
					}
				} catch (NumberFormatException ne) {
				}
				if (null == elPat) {
					return "Bad elevation antenna ID ";
				}
				if (AntPattern.PATTERN_TYPE_VERTICAL != elPat.type) {
					return "Wrong type for elevation pattern ";
				}
				matAzPat = patterns.get(Integer.valueOf(-antID));
				if (null != matAzPat) {
					if ((null != azPat) && (null != errors)) {
						errors.logMessage("Matrix envelope overriding azimuth pattern, in '" + fileName +
							"' at line " + lineCount);
					}
					azPat = matAzPat;
				}
			}

			// Electrical and mechanical tilt values are optional and may be blank, default to 0.

			if (null != eTiltStr) {
				if (eTiltStr.length() > 0) {
					try {
						eTilt = Double.parseDouble(eTiltStr);
					} catch (NumberFormatException ne) {
					}
					if ((eTilt < AntPattern.TILT_MIN) || (eTilt > AntPattern.TILT_MAX)) {
						return "Bad electrical tilt ";
					}
				} else {
					eTilt = 0.;
				}
			}

			if (null != mTiltStr) {
				if (mTiltStr.length() > 0) {
					try {
						mTilt = Double.parseDouble(mTiltStr);
					} catch (NumberFormatException ne) {
					}
					if ((mTilt < AntPattern.TILT_MIN) || (mTilt > AntPattern.TILT_MAX)) {
						return "Bad mechanical tilt ";
					}
				} else {
					mTilt = 0.;
				}
			}

			// The mechanical tilt orientation defaults to the pattern orientation if blank.

			if (null != mTiltOrientStr) {
				if (mTiltOrientStr.length() > 0) {
					try {
						mTiltOrient = Math.IEEEremainder(Double.parseDouble(mTiltOrientStr), 360.);
						if (mTiltOrient < 0.) mTiltOrient += 360.;
					} catch (NumberFormatException ne) {
						return "Bad mechanical tilt orientation ";
					}
				} else {
					mTiltOrient = orient;
				}
			}

			// Create the source.  Different APIs are used in SourceEditData depending on whether this is for a generic
			// data set import or not.

			source = null;
			if (null != extDb) {
				if (isTV) {
					if (isDTSAuth || isDTSSite) {
						sourceTV = sourceDTS.addExtDTSSource(extDb, service, siteNumber, errors);
						source = sourceTV;
					} else {
						sourceTV = SourceEditDataTV.createExtSource(extDb, facilityID, service, isDRT, country,
							errors);
						source = sourceTV;
					}
				} else {
					if (isWL) {
						sourceWL = SourceEditDataWL.createExtSource(extDb, service, country, errors);
						source = sourceWL;
					} else {
						if (isFM) {
							sourceFM = SourceEditDataFM.createExtSource(extDb, facilityID, service, stationClass,
								country, errors);
							source = sourceFM;
						}
					}
				}
			} else {
				if (isTV) {
					if (isDTSAuth || isDTSSite) {
						sourceTV = sourceDTS.addDTSSource(service, siteNumber, errors);
						source = sourceTV;
					} else {
						sourceTV = SourceEditDataTV.createSource(study, dbID, facilityID, service, isDRT,
							country, false, errors);
						source = sourceTV;
					}
				} else {
					if (isWL) {
						sourceWL = SourceEditDataWL.createSource(study, dbID, service, country, false, errors);
						source = sourceWL;
					} else {
						if (isFM) {
							sourceFM = SourceEditDataFM.createSource(study, dbID, facilityID, service,
								stationClass, country, false, errors);
							source = sourceFM;
						}
					}
				}
			}
			if (null == source) {
				return "Error occurred ";
			}

			// Populate.

			source.location.setLatLon(latitude, longitude);

			if (isDTSParent) {

				sourceDTS = sourceTV;

				sourceTV.dtsMaximumDistance = dtsDistance;
				sourceTV.dtsSectors = dtsDistanceStr;

			} else {

				source.heightAMSL = hamsl;
				source.overallHAAT = haat;
				source.peakERP = erp;
				if (null != azPat) {
					source.hasHorizontalPattern = true;
					source.horizontalPattern = azPat;
					source.horizontalPatternChanged = true;
				}
				source.horizontalPatternOrientation = orient;
				if (null != elPat) {
					if (elPat.isSimple()) {
						source.hasVerticalPattern = true;
						source.verticalPattern = elPat;
						source.verticalPatternChanged = true;
					} else {
						source.hasMatrixPattern = true;
						source.matrixPattern = elPat;
						source.matrixPatternChanged = true;
					}
				}
				source.verticalPatternElectricalTilt = eTilt;
				source.verticalPatternMechanicalTilt = mTilt;
				source.verticalPatternMechanicalTiltOrientation = mTiltOrient;
			}

			if (isTV) {

				if (isDTSAuth) {

					sourceDTSAuth = sourceTV;

					sourceTV.channel = channel;
					sourceTV.status = status;
					sourceTV.statusType = statusType;
					sourceTV.fileNumber = fileNumber;
					sourceTV.zone = zone;
					sourceTV.frequencyOffset = offset;
					sourceTV.emissionMask = mask;

				} else {

					if (!isDTSSite) {

						sourceTV.callSign = callSign;
						sourceTV.channel = channel;
						sourceTV.status = status;
						sourceTV.statusType = statusType;
						sourceTV.fileNumber = fileNumber;
						sourceTV.city = city;
						sourceTV.state = state;
						sourceTV.zone = zone;
						sourceTV.frequencyOffset = offset;
						sourceTV.emissionMask = mask;
						sourceTV.siteNumber = siteNumber;
					}
				}

				if ((null != status) && status.equals(ExtDbRecordTV.BASELINE_STATUS)) {
					source.setAttribute(Source.ATTR_IS_BASELINE);
				}

			} else {

				if (isWL) {

					sourceWL.callSign = callSign;
					sourceWL.sectorID = sectorID;
					sourceWL.fileNumber = fileNumber;
					sourceWL.city = city;
					sourceWL.state = state;

				} else {

					if (isFM) {

						sourceFM.callSign = callSign;
						sourceFM.channel = channel;
						sourceFM.status = status;
						sourceFM.statusType = statusType;
						sourceFM.fileNumber = fileNumber;
						sourceFM.city = city;
						sourceFM.state = state;
						sourceFM.isIBOC = isIBOC;
						sourceFM.ibocFraction = iboc;
					}
				}
			}

			source.setAttribute(Source.ATTR_SEQUENCE_DATE, importDate);

			// Checking and adding a DTS is deferred until all records for the same parent are read, see elsewhere.

			if (!isTV || isNonDTS) {
				if (!source.isDataValid(errors)) {
					return "Error occurred ";
				}
				sources.add(source);
			}
		}

		// Extra steps if last record was DTS, see comments above.

		if (null != sourceDTS) {
			if (sourceDTS.service.isFullService()) {
				if ((null == sourceDTSAuth) && (null == sourceDTS.makeDTSAuthorizedSource(extDb, errors))) {
					return "Error occurred ";
				}
			} else {
				if (null == sourceDTSAuth) {
					return "No authorized facility record for DTS parent ";
				}
				sourceDTS.location.setLatLon(sourceDTSAuth.location);
			}
			if (!sourceDTS.isDataValid(errors)) {
				return "Error occurred ";
			}
			sources.add(sourceDTS);
			sourceDTS = null;
			sourceDTSAuth = null;
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Write sources to CSV file, always uses the inline pattern format.

	public static void writeToCSV(BufferedWriter writer, int recordType, ArrayList<SourceEditData> sources,
			ErrorLogger errors) throws IOException {

		SourceEditDataTV sourceTV;
		SourceEditDataWL sourceWL;
		SourceEditDataFM sourceFM;

		int patKey;
		boolean patStart;

		for (SourceEditData source : sources) {

			if (recordType != source.recordType) {
				continue;
			}

			if (source.service.isDTS) {

				patStart = true;
				patKey = 1;

				for (SourceEditDataTV dtsSource : ((SourceEditDataTV)source).getDTSSources()) {

					if (dtsSource.hasHorizontalPattern || dtsSource.hasMatrixPattern || dtsSource.hasVerticalPattern) {

						if (patStart) {
							writer.write(PATTERN_DATA_START);
							writer.write('\n');
							patStart = false;
						}

						if (!writePatternsToCSV(writer, dtsSource, patKey, errors)) {
							return;
						}
					}

					patKey += 2;
				}

				if (!patStart) {
					writer.write(PATTERN_DATA_END);
					writer.write('\n');
				}

			} else {

				if (source.hasHorizontalPattern || source.hasMatrixPattern || source.hasVerticalPattern) {

					writer.write(PATTERN_DATA_START);
					writer.write('\n');

					if (!writePatternsToCSV(writer, source, 1, errors)) {
						return;
					}

					writer.write(PATTERN_DATA_END);
					writer.write('\n');
				}
			}

			switch (recordType) {

				case Source.RECORD_TYPE_TV: {

					sourceTV = (SourceEditDataTV)source;

					writer.write(sourceTV.callSign);
					writer.write(',');
					writer.write(sourceTV.service.serviceCode);
					writer.write(',');
					writer.write(String.valueOf(sourceTV.channel));
					writer.write(',');
					writer.write(sourceTV.status);
					writer.write(',');
					writer.write(sourceTV.fileNumber);
					writer.write(',');
					writer.write(String.valueOf(sourceTV.facilityID));
					writer.write(",\"");
					writer.write(sourceTV.city);
					writer.write("\",");
					writer.write(sourceTV.state);
					writer.write(',');
					writer.write(sourceTV.country.countryCode);
					writer.write(',');
					writer.write(sourceTV.zone.zoneCode);
					writer.write(',');
					if (sourceTV.service.isDTS) {
						writer.write('P');
					} else {
						writer.write(String.valueOf(sourceTV.siteNumber));
					}
					writer.write(',');
					if (sourceTV.service.isDTS) {
						if (sourceTV.service.isFullService()) {
							writer.write(String.valueOf(sourceTV.location.latitude));
							writer.write(',');
							writer.write(String.valueOf(sourceTV.location.longitude));
						} else {
							writer.write(',');
						}
						writer.write(",,,,,,,,,,");
						if (sourceTV.service.serviceType.needsEmissionMask) {
							writer.write(sourceTV.emissionMask.emissionMaskCode);
						}
						writer.write(',');
						writer.write(sourceTV.frequencyOffset.frequencyOffsetCode);
						if (sourceTV.service.isFullService()) {
							writer.write(',');
							if (sourceTV.dtsSectors.length() > 0) {
								writer.write("\"");
								writer.write(sourceTV.dtsSectors);
								writer.write("\"");
							} else {
								if (sourceTV.dtsMaximumDistance > 0.) {
									writer.write(String.valueOf(sourceTV.dtsMaximumDistance));
								}
							}
						}
					} else {
						writer.write(String.valueOf(sourceTV.location.latitude));
						writer.write(',');
						writer.write(String.valueOf(sourceTV.location.longitude));
						writer.write(',');
						writer.write(String.valueOf(sourceTV.heightAMSL));
						writer.write(',');
						writer.write(String.valueOf(sourceTV.overallHAAT));
						writer.write(',');
						writer.write(String.valueOf(sourceTV.peakERP));
						writer.write(',');
						if (source.hasHorizontalPattern) {
							writer.write('1');
						}
						writer.write(',');
						writer.write(String.valueOf(sourceTV.horizontalPatternOrientation));
						writer.write(',');
						if (sourceTV.hasMatrixPattern || sourceTV.hasVerticalPattern) {
							writer.write('2');
						}
						writer.write(',');
						writer.write(String.valueOf(sourceTV.verticalPatternElectricalTilt));
						writer.write(',');
						writer.write(String.valueOf(sourceTV.verticalPatternMechanicalTilt));
						writer.write(',');
						writer.write(String.valueOf(sourceTV.verticalPatternMechanicalTiltOrientation));
						writer.write(',');
						if (sourceTV.service.serviceType.needsEmissionMask) {
							writer.write(sourceTV.emissionMask.emissionMaskCode);
						}
						writer.write(',');
						writer.write(sourceTV.frequencyOffset.frequencyOffsetCode);
					}

					writer.write('\n');

					if (sourceTV.service.isDTS) {

						patKey = 1;

						for (SourceEditDataTV dtsSource : sourceTV.getDTSSources()) {

							writer.write(',');
							if (0 == dtsSource.siteNumber) {
								writer.write(dtsSource.service.serviceCode);
								writer.write(',');
								writer.write(String.valueOf(dtsSource.channel));
								writer.write(',');
								writer.write(dtsSource.status);
								writer.write(',');
								writer.write(dtsSource.fileNumber);
								writer.write(',');
							} else {
								writer.write(",,,,");
							}
							writer.write(",,,,");
							if (0 == dtsSource.siteNumber) {
								writer.write(dtsSource.zone.zoneCode);
							}
							writer.write(',');
							if (0 == dtsSource.siteNumber) {
								writer.write('R');
							} else {
								writer.write(String.valueOf(dtsSource.siteNumber));
							}
							writer.write(',');
							writer.write(String.valueOf(dtsSource.location.latitude));
							writer.write(',');
							writer.write(String.valueOf(dtsSource.location.longitude));
							writer.write(',');
							writer.write(String.valueOf(dtsSource.heightAMSL));
							writer.write(',');
							writer.write(String.valueOf(dtsSource.overallHAAT));
							writer.write(',');
							writer.write(String.valueOf(dtsSource.peakERP));
							writer.write(',');
							if (dtsSource.hasHorizontalPattern) {
								writer.write(String.valueOf(patKey));
							}
							writer.write(',');
							writer.write(String.valueOf(dtsSource.horizontalPatternOrientation));
							writer.write(',');
							if (dtsSource.hasMatrixPattern || dtsSource.hasVerticalPattern) {
								writer.write(String.valueOf(patKey + 1));
							}
							writer.write(',');
							writer.write(String.valueOf(dtsSource.verticalPatternElectricalTilt));
							writer.write(',');
							writer.write(String.valueOf(dtsSource.verticalPatternMechanicalTilt));
							writer.write(',');
							writer.write(String.valueOf(dtsSource.verticalPatternMechanicalTiltOrientation));
							writer.write(',');
							if (0 == dtsSource.siteNumber) {
								writer.write(dtsSource.emissionMask.emissionMaskCode);
								writer.write(',');
								writer.write(dtsSource.frequencyOffset.frequencyOffsetCode);
							} else {
								writer.write(',');
							}

							writer.write('\n');

							patKey += 2;
						}
					}
					break;
				}

				case Source.RECORD_TYPE_WL: {

					sourceWL = (SourceEditDataWL)source;

					writer.write(sourceWL.callSign);
					writer.write(',');
					writer.write(sourceWL.sectorID);
					writer.write(',');
					writer.write(String.valueOf(sourceWL.location.latitude));
					writer.write(',');
					writer.write(String.valueOf(-(sourceWL.location.longitude)));
					writer.write(',');
					writer.write(String.valueOf(sourceWL.heightAMSL));
					writer.write(',');
					writer.write(String.valueOf(sourceWL.overallHAAT));
					writer.write(',');
					writer.write(String.valueOf(sourceWL.peakERP));
					writer.write(',');
					if (sourceWL.hasHorizontalPattern) {
						writer.write('1');
					}
					writer.write(',');
					writer.write(String.valueOf(sourceWL.horizontalPatternOrientation));
					writer.write(',');
					if (sourceWL.hasMatrixPattern || sourceWL.hasVerticalPattern) {
						writer.write('2');
					}
					writer.write(',');
					writer.write(String.valueOf(sourceWL.verticalPatternElectricalTilt));
					writer.write(',');
					writer.write(String.valueOf(sourceWL.verticalPatternMechanicalTilt));
					writer.write(',');
					writer.write(String.valueOf(sourceWL.verticalPatternMechanicalTiltOrientation));
					writer.write(',');
					writer.write(sourceWL.fileNumber);
					writer.write(",\"");
					writer.write(sourceWL.city);
					writer.write("\",");
					writer.write(sourceWL.state);
					writer.write(',');
					writer.write(sourceWL.country.countryCode);

					writer.write('\n');

					break;
				}

				case Source.RECORD_TYPE_FM: {

					sourceFM = (SourceEditDataFM)source;

					writer.write(sourceFM.callSign);
					writer.write(',');
					writer.write(sourceFM.service.serviceCode);
					writer.write(',');
					writer.write(String.valueOf(sourceFM.channel));
					writer.write(',');
					writer.write(ExtDbRecordFM.FM_CLASS_CODES[sourceFM.stationClass]);
					writer.write(',');
					writer.write(sourceFM.status);
					writer.write(',');
					writer.write(sourceFM.fileNumber);
					writer.write(',');
					writer.write(String.valueOf(sourceFM.facilityID));
					writer.write(",\"");
					writer.write(sourceFM.city);
					writer.write("\",");
					writer.write(sourceFM.state);
					writer.write(',');
					writer.write(sourceFM.country.countryCode);
					writer.write(',');
					writer.write(String.valueOf(sourceFM.location.latitude));
					writer.write(',');
					writer.write(String.valueOf(sourceFM.location.longitude));
					writer.write(',');
					writer.write(String.valueOf(sourceFM.heightAMSL));
					writer.write(',');
					writer.write(String.valueOf(sourceFM.overallHAAT));
					writer.write(',');
					writer.write(String.valueOf(sourceFM.peakERP));
					writer.write(',');
					if (sourceFM.isIBOC) {
						writer.write(String.valueOf(sourceFM.ibocFraction * 100.));
					}
					writer.write(',');
					if (sourceFM.hasHorizontalPattern) {
						writer.write('1');
					}
					writer.write(',');
					writer.write(String.valueOf(sourceFM.horizontalPatternOrientation));
					writer.write(',');
					if (sourceFM.hasMatrixPattern || sourceFM.hasVerticalPattern) {
						writer.write('2');
					}
					writer.write(',');
					writer.write(String.valueOf(sourceFM.verticalPatternElectricalTilt));
					writer.write(',');
					writer.write(String.valueOf(sourceFM.verticalPatternMechanicalTilt));
					writer.write(',');
					writer.write(String.valueOf(sourceFM.verticalPatternMechanicalTiltOrientation));
	
					writer.write('\n');

					break;
				}
			}
		}
	}

	private static boolean writePatternsToCSV(BufferedWriter writer, SourceEditData source, int patKey,
			ErrorLogger errors) throws IOException {

		AntPattern pattern;
		boolean first;

		if (source.hasHorizontalPattern) {

			pattern = source.getHorizontalPattern(errors);
			if (null == pattern) {
				return false;
			}

			writer.write(String.valueOf(patKey));
			writer.write(",A,\"");
			writer.write(pattern.name);
			writer.write('"');
			for (AntPattern.AntPoint point : pattern.getPoints()) {
				writer.write(',');
				writer.write(String.valueOf(point.angle));
				writer.write(';');
				writer.write(String.valueOf(point.relativeField));
			}
			writer.write('\n');
		}

		if (source.hasMatrixPattern || source.hasVerticalPattern) {

			pattern = source.getVerticalPattern(errors);
			if (null == pattern) {
				return false;
			}

			if (pattern.isMatrix()) {

				first = true;
				for (AntPattern.AntSlice slice : pattern.getSlices()) {
					writer.write(String.valueOf(patKey + 1));
					writer.write(",M,");
					if (first) {
						writer.write('"');
						writer.write(pattern.name);
						writer.write('"');
						first = false;
					}
					writer.write(',');
					writer.write(String.valueOf(slice.value));
					for (AntPattern.AntPoint point : slice.points) {
						writer.write(',');
						writer.write(String.valueOf(point.angle));
						writer.write(';');
						writer.write(String.valueOf(point.relativeField));
					}
					writer.write('\n');
				}

			} else {

				writer.write(String.valueOf(patKey + 1));
				writer.write(",E,\"");
				writer.write(pattern.name);
				writer.write('"');
				for (AntPattern.AntPoint point : pattern.getPoints()) {
					writer.write(',');
					writer.write(String.valueOf(point.angle));
					writer.write(';');
					writer.write(String.valueOf(point.relativeField));
				}
				writer.write('\n');
			}
		}

		return true;
	}
}
