//
//  ExtDbRecord.java
//  TVStudy
//
//  Copyright (c) 2012-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.core;

import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;
import gov.fcc.tvstudy.core.geo.*;

import java.util.*;
import java.text.*;
import java.sql.*;


//=====================================================================================================================
// Data class to hold a record from external station data.  This is an abstract superclass, concrete subclasses exist
// for different record types.  This is not a database record representation class, nor is it part of an editor model.
// It's role is to encapsulate the SQL code and translation logic needed to create SourceEditData objects from external
// station data, using databases created by the ExtDb class methods.  However the subclasses here are not one-to-one
// with ExtDb data types, they follow the Source and SourceEditData record type subclasses, so a subclass here may draw
// data from several different station data types.  Subclasses implement two main methods, static method findRecords()
// is used for search queries, and an instance method updateSource??() is used to apply data to a source object.  Both
// are typed specifically to the subclass so are not declared here.  As with the related class sets like Source, the
// properties and code related to antenna patterns is here in the superclass.

public abstract class ExtDbRecord implements StationRecord {

	// Status code enumeration used by some subclasses, here for convenience.

	public static final int STATUS_TYPE_STA = 0;
	public static final int STATUS_TYPE_CP = 1;
	public static final int STATUS_TYPE_LIC = 2;
	public static final int STATUS_TYPE_APP = 3;
	public static final int STATUS_TYPE_OTHER = 4;
	public static final int STATUS_TYPE_EXP = 5;
	public static final int STATUS_TYPE_AMD = 6;

	public static final String[] STATUS_CODES = {
		"STA", "CP", "LIC", "APP", "", "EXP", "AMD"
	};

	// Options for query composing methods that match logical flags, tested flag may be set, clear, or any.

	public static final int FLAG_MATCH_SET = 1;
	public static final int FLAG_MATCH_CLEAR = 2;
	public static final int FLAG_MATCH_ANY = 3;

	// Instance properties common to all record types.

	public final ExtDb extDb;
	public final int recordType;

	public String extRecordID;

	public Service service;
	public String city;
	public String state;
	public Country country;
	public final GeoPoint location;
	public double heightAMSL;
	public double overallHAAT;
	public double peakERP;
	public String antennaID;
	public String antennaRecordID;
	public double horizontalPatternOrientation;
	public String elevationAntennaRecordID;
	public double verticalPatternElectricalTilt;
	public double verticalPatternMechanicalTilt;
	public double verticalPatternMechanicalTiltOrientation;
	public boolean isPending;
	public boolean isArchived;
	public java.util.Date sequenceDate;

	// A message string used many times.

	public static final String BAD_TYPE_MESSAGE = "Unknown or unsupported station data type";


	//-----------------------------------------------------------------------------------------------------------------
	// Get list of KeyedRecord objects representing status codes.

	private static ArrayList<KeyedRecord> statusList = null;

	public static ArrayList<KeyedRecord> getStatusList() {

		if (null == statusList) {
			statusList = new ArrayList<KeyedRecord>();
			statusList.add(new KeyedRecord(STATUS_TYPE_LIC, STATUS_CODES[STATUS_TYPE_LIC]));
			statusList.add(new KeyedRecord(STATUS_TYPE_CP, STATUS_CODES[STATUS_TYPE_CP]));
			statusList.add(new KeyedRecord(STATUS_TYPE_APP, STATUS_CODES[STATUS_TYPE_APP]));
			statusList.add(new KeyedRecord(STATUS_TYPE_STA, STATUS_CODES[STATUS_TYPE_STA]));
			statusList.add(new KeyedRecord(STATUS_TYPE_AMD, STATUS_CODES[STATUS_TYPE_AMD]));
			statusList.add(new KeyedRecord(STATUS_TYPE_EXP, STATUS_CODES[STATUS_TYPE_EXP]));
		}

		return statusList;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Do a fast check to determine if a given record ID exists in a data set.  Returns false on errors.

	public static boolean doesRecordIDExist(String theDbID, Integer theExtDbKey, String theExtRecordID) {

		ExtDb theExtDb = ExtDb.getExtDb(theDbID, theExtDbKey);
		if (null == theExtDb) {
			return false;
		}

		return doesRecordIDExist(theExtDb, theExtRecordID);
	}

	public static boolean doesRecordIDExist(ExtDb theExtDb, String theExtRecordID) {

		if (theExtDb.canProvide(Source.RECORD_TYPE_TV)) {
			if (ExtDbRecordTV.doesRecordIDExistTV(theExtDb, theExtRecordID)) {
				return true;
			}
		}

		if (theExtDb.canProvide(Source.RECORD_TYPE_FM)) {
			if (ExtDbRecordFM.doesRecordIDExistFM(theExtDb, theExtRecordID)) {
				return true;
			}
		}

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Methods to support composing query clauses for findRecords() searching, subclasses provide data-specific SQL
	// code in methods with TV/FM name suffixes.  These superclass implementations will branch out to subclasses by
	// data set type.  In general the search arguments to these are assumed to be user input, thus validation or other
	// pre-processing is often appropriate.  An IllegalArgumentException is thrown if validation fails or an unknown
	// data set type is passed.  If the input string is null or empty the methods do nothing.  If a clause will be
	// added and the combine argument is true the new clause is preceded with " AND ".  The return is true if a clause
	// is added.  See RecordFind for typical use.

	public static boolean addRecordIDQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addRecordIDQueryTV(dbType, version, str, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addRecordIDQueryFM(dbType, version, str, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// A file number conceptually consists of a file prefix and an ARN (application reference number).  The prefix is
	// alphabetic, the ARN suffix is alphanumeric but usually begins with a numerical digit, and the two may or may not
	// be separated by a '-'.  Also a prefix string of "BLANK" may be used when there is no prefix.  The ARN itself is
	// sufficient to be unique in most contexts so the prefix is always optional.

	public static String[] parseFileNumber(String str) {

		String[] parts = new String[2];

		// If the search string starts with "BLANK" there is no prefix, strip "BLANK" and possibly a '-' following.

		String upStr = str.toUpperCase();

		if (upStr.startsWith("BLANK")) {

			parts[0] = "";
			str = str.substring(5);
			if (str.startsWith("-")) {
				parts[1] = str.substring(1);
			} else {
				parts[1] = str;
			}

		} else {

			// If string starts with "B" there is a prefix, break ARN at the first digit or after a '-'.

			if (upStr.startsWith("B")) {

				char cc;
				StringBuilder pre = new StringBuilder();

				parts[1] = "";
				for (int i = 0; i < str.length(); i++) {
					cc = str.charAt(i);
					if (Character.isDigit(cc)) {
						parts[1] = DbConnection.clean(str.substring(i));
						break;
					}
					if ('-' == cc) {
						parts[1] = DbConnection.clean(str.substring(i + 1));
						break;
					}
					pre.append(cc);
				}

				parts[0] = DbConnection.clean(pre.toString());

			// No leading "BLANK" or "B" means no prefix, the whole string matches the ARN.

			} else {

				parts[0] = "";
				parts[1] = str;
			}
		}

		return parts;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Search by file number.

	public static boolean addFileNumberQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		String[] parts = parseFileNumber(str);

		return addFileNumberQuery(dbType, version, recordType, parts[0], parts[1], query, combine);
	}

	public static boolean addFileNumberQuery(int dbType, int version, int recordType, String prefix, String arn,
			StringBuilder query, boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addFileNumberQueryTV(dbType, version, prefix, arn, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addFileNumberQueryFM(dbType, version, prefix, arn, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query clause to search by facility ID.  The first version converts string input, second takes a numerical
	// facility ID argument.

	public static boolean addFacilityIDQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		int facilityID = 0;
		try {
			facilityID = Integer.parseInt(str);
		} catch (NumberFormatException ne) {
			throw new IllegalArgumentException("The facility ID must be a number");
		}

		return addFacilityIDQuery(dbType, version, recordType, facilityID, query, combine);
	}

	public static boolean addFacilityIDQuery(int dbType, int version, int recordType, int facilityID,
			StringBuilder query, boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addFacilityIDQueryTV(dbType, version, facilityID, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addFacilityIDQueryFM(dbType, version, facilityID, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query clause to search by service.  The string argument is a two-letter service code that must resolve to
	// a Service object, the second version takes an integer service key.

	public static boolean addServiceQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		Service theService = Service.getService(str);
		if (null == theService) {
			throw new IllegalArgumentException("Unknown service code");
		}

		return addServiceQuery(dbType, version, recordType, theService.key, query, combine);
	}

	public static boolean addServiceQuery(int dbType, int version, int recordType, int serviceKey, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addServiceQueryTV(dbType, version, serviceKey, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addServiceQueryFM(dbType, version, serviceKey, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for call sign, case-insensitive head-anchored match.

	public static boolean addCallSignQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addCallSignQueryTV(dbType, version, str, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addCallSignQueryFM(dbType, version, str, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for a specific channel search.  If the caller provides a channel range the channel
	// must be in that range, otherwise anything is allowed.  The range arguments are assumed to be valid if >0.

	public static boolean addChannelQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {
		return addChannelQuery(dbType, version, recordType, str, 0, 0, query, combine);
	}

	public static boolean addChannelQuery(int dbType, int version, int recordType, String str, int minimumChannel,
			int maximumChannel, StringBuilder query, boolean combine) throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		int channel = 0;
		try {
			channel = Integer.parseInt(str);
		} catch (NumberFormatException e) {
			throw new IllegalArgumentException("The channel must be a number");
		}

		return addChannelQuery(dbType, version, recordType, channel, minimumChannel, maximumChannel, query, combine);
	}

	public static boolean addChannelQuery(int dbType, int version, int recordType, int channel, StringBuilder query,
			boolean combine) throws IllegalArgumentException {
		return addChannelQuery(dbType, version, recordType, channel, 0, 0, query, combine);
	}

	public static boolean addChannelQuery(int dbType, int version, int recordType, int channel, int minimumChannel,
			int maximumChannel, StringBuilder query, boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addChannelQueryTV(dbType, version, channel, minimumChannel, maximumChannel, query,
					combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addChannelQueryFM(dbType, version, channel, minimumChannel, maximumChannel, query,
					combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for a channel range search.  The range arguments are assumed valid.

	public static boolean addChannelRangeQuery(int dbType, int version, int recordType, int minimumChannel,
			int maximumChannel, StringBuilder query, boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addChannelRangeQueryTV(dbType, version, minimumChannel, maximumChannel, query,
					combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addChannelRangeQueryFM(dbType, version, minimumChannel, maximumChannel, query,
					combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for matching a list of channels.  This is not meant to process direct user input, the
	// argument string is assumed to be a valid SQL value list composed by other code e.g. "(7,8,12,33)", containing
	// valid channel numbers.

	public static boolean addMultipleChannelQuery(int dbType, int version, int recordType, String str,
			StringBuilder query, boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addMultipleChannelQueryTV(dbType, version, str, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addMultipleChannelQueryFM(dbType, version, str, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for status.  The string argument is treated as a source record status code matched to
	// just the internal status type list, see getStatusType() below.  Second form takes a status type directly.

	public static boolean addStatusQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((null == str) || (0 == str.length())) {
			return false;
		}

		return addStatusQuery(dbType, version, recordType, getStatusType(str), query, combine);
	}

	public static boolean addStatusQuery(int dbType, int version, int recordType, int statusType, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addStatusQueryTV(dbType, version, statusType, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addStatusQueryFM(dbType, version, statusType, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add query clause for city name search, case-insensitive unanchored match, '*' wildcards are allowed.

	public static boolean addCityQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addCityQueryTV(dbType, version, str, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addCityQueryFM(dbType, version, str, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add query clause for state code search, this is an exact string match but still case-insensitive.

	public static boolean addStateQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addStateQueryTV(dbType, version, str, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addStateQueryFM(dbType, version, str, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add clause for country search, this is resolved to a Country object by string code or key.

	public static boolean addCountryQuery(int dbType, int version, int recordType, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		Country country = Country.getCountry(str);
		if (null == country) {
			throw new IllegalArgumentException("Unknown country code");
		}

		return addCountryQuery(dbType, version, recordType, country, query, combine);
	}

	public static boolean addCountryQuery(int dbType, int version, int recordType, int countryKey, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		Country country = Country.getCountry(countryKey);
		if (null == country) {
			throw new IllegalArgumentException("Unknown country key");
		}

		return addCountryQuery(dbType, version, recordType, country, query, combine);
	}

	public static boolean addCountryQuery(int dbType, int version, int recordType, Country country, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addCountryQueryTV(dbType, version, country, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addCountryQueryFM(dbType, version, country, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add record type search query clause.

	public static boolean addRecordTypeQuery(int dbType, int version, int recordType, boolean includeArchived,
			StringBuilder query, boolean combine) {

		switch (recordType) {

			case Source.RECORD_TYPE_TV: {
				return ExtDbRecordTV.addRecordTypeQueryTV(dbType, version, includeArchived, query, combine);
			}

			case Source.RECORD_TYPE_FM: {
				return ExtDbRecordFM.addRecordTypeQueryFM(dbType, version, includeArchived, query, combine);
			}
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Translate status string to type code.  This is the generic case for general use and the user record table.

	public static int getStatusType(String theStatus) {

		for (int i = 0; i < STATUS_CODES.length; i++) {
			if ((STATUS_TYPE_OTHER != i) && STATUS_CODES[i].equalsIgnoreCase(theStatus)) {
				return i;
			}
		}

		return STATUS_TYPE_OTHER;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Compose a data warning message including record identifiers, used with logMessage() in error reporter objects.

	public static String makeMessage(StationRecord theRecord, String theMessage) {

		return String.format(Locale.US, "%-6.6s %-8.8s %-3.3s %-2.2s %-6.6s: %s", theRecord.getFacilityID(),
			theRecord.getCallSign(), theRecord.getChannel(), theRecord.getServiceCode(), theRecord.getStatus(),
			theMessage);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Fill the cache of CDBS/LMS service code mappings for use by subclasses.  This is called after similar methods
	// in core.data classes the first time any database is opened, see comments in DbCore.openDb().

	private static HashMap<String, Service> cdbsCodeCache = new HashMap<String, Service>();
	private static HashMap<String, Service> lmsCodeCache = new HashMap<String, Service>();

	public static void loadCache(DbConnection db) throws SQLException {

		cdbsCodeCache.clear();
		lmsCodeCache.clear();

		Service theService;

		db.query("SELECT cdbs_service_code, service_key FROM cdbs_service");
		while (db.next()) {
			theService = Service.getService(Integer.valueOf(db.getInt(2)));
			if (null != theService) {
				cdbsCodeCache.put(db.getString(1), theService);
			}
		}

		db.query("SELECT lms_service_code, service_key FROM lms_service");
		while (db.next()) {
			theService = Service.getService(Integer.valueOf(db.getInt(2)));
			if (null != theService) {
				lmsCodeCache.put(db.getString(1), theService);
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Accessor for the service code caches.

	public static Service getService(int dbType, String serviceCode) {

		switch (dbType) {
			case ExtDb.DB_TYPE_CDBS:
			case ExtDb.DB_TYPE_CDBS_FM: {
				return cdbsCodeCache.get(serviceCode);
			}
			case ExtDb.DB_TYPE_LMS:
			case ExtDb.DB_TYPE_LMS_LIVE: {
				return lmsCodeCache.get(serviceCode);
			}
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add service code list to a query using the code caches.  May be restricted by record type or a specific service
	// key, either form may have a 0 argument to match all.  May also be restricted based on isOperating and isDigital
	// flags.  Does not fail, if arguments don't match anything this will add a list containing just an empty string.

	public static void addServiceCodeList(int dbType, int recordType, StringBuilder query) {
		addServiceCodeList(dbType, recordType, 0, FLAG_MATCH_ANY, FLAG_MATCH_ANY, query);
	}

	public static void addServiceCodeList(int dbType, int recordType, int serviceKey, StringBuilder query) {
		addServiceCodeList(dbType, recordType, serviceKey, FLAG_MATCH_ANY, FLAG_MATCH_ANY, query);
	}

	public static void addServiceCodeList(int dbType, int recordType, int operatingMatch, int digitalMatch,
			StringBuilder query) {
		addServiceCodeList(dbType, recordType, 0, operatingMatch, digitalMatch, query);
	}

	private static void addServiceCodeList(int dbType, int recordType, int serviceKey, int operatingMatch,
			int digitalMatch, StringBuilder query) {

		query.append("('");

		HashMap<String, Service> codeCache = null;
		switch (dbType) {
			case ExtDb.DB_TYPE_CDBS:
			case ExtDb.DB_TYPE_CDBS_FM: {
				codeCache = cdbsCodeCache;
				break;
			}
			case ExtDb.DB_TYPE_LMS:
			case ExtDb.DB_TYPE_LMS_LIVE: {
				codeCache = lmsCodeCache;
				break;
			}
		}

		if (null != codeCache) {

			Service theService;
			String s = "";

			for (Map.Entry<String, Service> e : codeCache.entrySet()) {
				theService = e.getValue();
				if (((0 == serviceKey) || (serviceKey == theService.key)) &&
						((0 == recordType) || (recordType == theService.serviceType.recordType)) &&
						((FLAG_MATCH_SET != operatingMatch) || theService.isOperating) &&
						((FLAG_MATCH_CLEAR != operatingMatch) || !theService.isOperating) &&
						((FLAG_MATCH_SET != digitalMatch) || theService.serviceType.digital) &&
						((FLAG_MATCH_CLEAR != digitalMatch) || !theService.serviceType.digital)) {
					query.append(s);
					query.append(e.getKey());
					s = "','";
				}
			}
		}

		query.append("')");
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The constructor is private, instances are obtained from findRecords() search method.

	protected ExtDbRecord(ExtDb theExtDb, int theRecordType) {

		super();

		extDb = theExtDb;
		recordType = theRecordType;

		location = new GeoPoint();

		sequenceDate = new java.util.Date();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String toString() {

		return getCallSign() + " " + getChannel() + " " + getServiceCode() + " " + getStatus() + " " +
			getCity() + ", " + getState();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Build comment text intended to show as a tool-tip pop-up in tables.  Shows various details that generally are
	// not significant enough to have separate columns in a display table.

	public String makeCommentText() {

		StringBuilder s = new StringBuilder();
		ArrayList<String> comments = getComments();
		if ((null != comments) && !comments.isEmpty()) {
			String pfx = "<HTML>";
			for (String theComment : comments) {
				s.append(pfx);
				s.append(theComment);
				pfx = "<BR>";
			}
			s.append("</HTML>");
		}
		return s.toString();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// May be overridden by subclass to provide lines for the comment text.

	protected ArrayList<String> getComments() {

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Methods in the StationRecord interface.

	public String getRecordType() {

		return extDb.getTypeName();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isSource() {

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean hasRecordID() {

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isReplication() {

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getStationData() {

		return ExtDb.getExtDbDescription(extDb.dbID, extDb.key);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getRecordID() {

		return extRecordID;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFacilityID() {

		return "";
	}

	public String getSortFacilityID() {

		return "0";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getService() {

		return service.name;
	}

	public String getServiceCode() {

		return service.serviceCode;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isDigital() {

		return service.serviceType.digital;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isDTS() {

		return service.isDTS;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getSiteCount() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getCallSign() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getChannel() {

		return "";
	}

	public int getChannelNumber() {

		return 0;
	}


	public String getSortChannel() {

		return "0";
	}

	public String getOriginalChannel() {

		return getChannel();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFrequency() {

		return "";
	}

	public double getFrequencyValue() {

		return 0.;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getCity() {

		return city;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getState() {

		return state;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getCountry() {

		return country.name;
	}

	public String getCountryCode() {

		return country.countryCode;
	}

	public String getSortCountry() {

		return String.valueOf(country.key);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getZone() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getStatus() {

		return "";
	}

	public String getSortStatus() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFileNumber() {

		return "";
	}

	public String getARN() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getSequenceDate() {

		return AppCore.formatDate(sequenceDate);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getSortSequenceDate() {

		return String.format(Locale.US, "%013d", sequenceDate.getTime());
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFrequencyOffset() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getEmissionMask() {

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getLatitude() {

		return AppCore.formatLatitude(location.latitude);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getLongitude() {

		return AppCore.formatLongitude(location.longitude);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getHeightAMSL() {

		if (Source.HEIGHT_DERIVE == heightAMSL) {
			return Source.HEIGHT_DERIVE_LABEL;
		}
		return AppCore.formatHeight(heightAMSL) + " m";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getOverallHAAT() {

		if (Source.HEIGHT_DERIVE == overallHAAT) {
			return Source.HEIGHT_DERIVE_LABEL;
		}
		return AppCore.formatHeight(overallHAAT) + " m";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getPeakERP() {

		return AppCore.formatERP(peakERP) + " kW";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean hasHorizontalPattern() {

		return (null != antennaRecordID);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getHorizontalPatternName() {

		if (null != antennaRecordID) {
			if ((null != antennaID) && (antennaID.length() > 0)) {
				return "ID " + antennaID;
			} else {
				return "unknown";
			}
		}
		return "Omnidirectional";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getHorizontalPatternOrientation() {

		if (null != antennaRecordID) {
			return AppCore.formatAzimuth(horizontalPatternOrientation) + " deg";
		}
		return "";
	}
}
