//
//  ColorMap.java
//  TVStudy
//
//  Copyright (c) 2019-2021 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.core;

import java.util.*;
import java.sql.*;

import java.awt.Color;


//=====================================================================================================================
// Data class for image color map.

public class ColorMap {

	public static final String NEW_OBJECT_NAME = "(new)";

	public static final double MIN_LEVEL = -40.;
	public static final double MAX_LEVEL = 100.;

	public static final double LEVEL_CODE_NO_SERVICE = -999.;
	public static final double LEVEL_CODE_INTERFERENCE = -998.;
	public static final double LEVEL_CODE_BACKGROUND = -997.;

	public static final int COUNT_INCREMENT = 20;

	public static final int MAX_COUNT = 50;   // Color index must be 0-63 including override colors.

	public static final Color DEFAULT_COLOR = Color.GREEN;
	public int key;
	public String name;
	public boolean isPermanent;

	public Color backgroundColor;

	public int colorCount;
	public double[] levels;
	public Color[] colors;

	public Color noServiceColor;
	public Color interferenceColor;


	//-----------------------------------------------------------------------------------------------------------------
	// If the key is <=0 this is a new unsaved object.  Default properties are set so this is always functional.

	public ColorMap(int theKey) {

		key = theKey;

		name = NEW_OBJECT_NAME;

		backgroundColor = Color.BLUE;

		levels = new double[COUNT_INCREMENT];
		colors = new Color[COUNT_INCREMENT];
		colorCount = 1;
		levels[0] = 0.;
		colors[0] = DEFAULT_COLOR;

		noServiceColor = null;
		interferenceColor = null;
	}


	//-----------------------------------------------------------------------------------------------------------------

	private ColorMap() {
	}


	//-----------------------------------------------------------------------------------------------------------------

	public ColorMap copy() {

		ColorMap newMap = new ColorMap();

		newMap.name = name + " (copy)";

		newMap.backgroundColor = backgroundColor;

		newMap.levels = new double[levels.length];
		newMap.colors = new Color[levels.length];
		newMap.colorCount = colorCount;
		for (int i = 0; i < colorCount; i++) {
			newMap.levels[i] = levels[i];
			newMap.colors[i] = colors[i];
		}

		newMap.noServiceColor = noServiceColor;
		newMap.interferenceColor = interferenceColor;

		return newMap;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a color to the map, return new color index, -1 if at max count.

	public int addColor() {

		if (MAX_COUNT == colorCount) {
			return -1;
		}

		int newIndex = colorCount++;
		expandArrays();

		levels[newIndex] = levels[newIndex - 1] + 3.;
		colors[newIndex] = DEFAULT_COLOR;

		return newIndex;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Remove a color.

	public void removeColor() {

		if (colorCount > 1) {
			colorCount--;
		}
	}


	//-------------------------------------------------------------------------------------------------------------
	// Called after colorCount changed and may have increased.

	private void expandArrays() {

		if (colorCount <= levels.length) {
			return;
		}

		int newLength = colorCount + COUNT_INCREMENT;

		double[] newLevels = new double[newLength];
		Color[] newColors = new Color[newLength];

		for (int i = 0; i < levels.length; i++) {
			newLevels[i] = levels[i];
			newColors[i] = colors[i];
		}

		levels = newLevels;
		colors = newColors;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save data, assign a new key if needed, return false on error.  Also verify the name is unique and prevent the
	// new-object name from being saved.  In case of concurrent edits, last save wins.  This will not save to a map
	// with the permanent flag true, that is always checked by re-querying.  A new map will always have permanent
	// forced false.  Permanent maps can only be created by database installation/update code.

	public boolean save(String theDbID) {
		return save(theDbID, null);
	}

	public boolean save(String theDbID, ErrorLogger errors) {

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			return false;
		}

		String errmsg = null;

		try {

			db.update("LOCK TABLES color_map WRITE, color_map_data WRITE");

			String theKey;
			boolean doInsert = false;

			if (key > 0) {
				theKey = String.valueOf(key);
				db.query("SELECT permanent FROM color_map WHERE color_map_key = " + theKey);
				if (db.next()) {
					isPermanent = db.getBoolean(1);
				} else {
					isPermanent = false;
					doInsert = true;
				}
			} else {
				db.query("SELECT MAX(color_map_key) FROM color_map");
				if (db.next()) {
					key = db.getInt(1) + 1;
				} else {
					key = 1;
				}
				theKey = String.valueOf(key);
				isPermanent = false;
				doInsert = true;
			}

			if (isPermanent) {
				errmsg = "Color map cannot be modified";
			} else {

				if ((null == name) || name.equalsIgnoreCase(NEW_OBJECT_NAME)) {
					errmsg = "Missing or invalid color map name";
				} else {

					db.query("SELECT color_map_key FROM color_map WHERE (UPPER(name) = '" +
						db.clean(name.toUpperCase()) + "') AND (color_map_key <> " + theKey + ")");
					if (db.next()) {
						errmsg = "Color map name already exists";
					} else {

						if (doInsert) {
							db.update("INSERT INTO color_map VALUES(" + theKey + ",'" + db.clean(name) + "',false)");
						} else {
							db.update("UPDATE color_map SET name = '" + db.clean(name) + "' WHERE color_map_key = " +
								theKey);
						}
				
						db.update("DELETE FROM color_map_data WHERE color_map_key = " + theKey);

						StringBuilder q = new StringBuilder("INSERT INTO color_map_data VALUES (");

						Color theColor = backgroundColor;
						q.append(theKey);
						q.append(',');
						q.append(String.valueOf(LEVEL_CODE_BACKGROUND));
						q.append(',');
						q.append(String.valueOf(theColor.getRed()));
						q.append(',');
						q.append(String.valueOf(theColor.getGreen()));
						q.append(',');
						q.append(String.valueOf(theColor.getBlue()));

						q.append("),(");

						theColor = noServiceColor;
						q.append(theKey);
						q.append(',');
						q.append(LEVEL_CODE_NO_SERVICE);
						q.append(',');
						if (null == theColor) {
							q.append("-1,-1,-1");
						} else {
							q.append(String.valueOf(theColor.getRed()));
							q.append(',');
							q.append(String.valueOf(theColor.getGreen()));
							q.append(',');
							q.append(String.valueOf(theColor.getBlue()));
						}

						q.append("),(");

						theColor = interferenceColor;
						q.append(theKey);
						q.append(',');
						q.append(LEVEL_CODE_INTERFERENCE);
						q.append(',');
						if (null == theColor) {
							q.append("-1,-1,-1");
						} else {
							q.append(String.valueOf(theColor.getRed()));
							q.append(',');
							q.append(String.valueOf(theColor.getGreen()));
							q.append(',');
							q.append(String.valueOf(theColor.getBlue()));
						}

						for (int ci = 0; ci < colorCount; ci++) {

							q.append("),(");

							theColor = colors[ci];
							q.append(theKey);
							q.append(',');
							q.append(String.valueOf(levels[ci]));
							q.append(',');
							q.append(String.valueOf(theColor.getRed()));
							q.append(',');
							q.append(String.valueOf(theColor.getGreen()));
							q.append(',');
							q.append(String.valueOf(theColor.getBlue()));
						}

						q.append(')');

						db.update(q.toString());
					}
				}
			}

		} catch (SQLException se) {
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		try {
			db.update("UNLOCK TABLES");
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (null != errmsg) {
			if (null != errors) {
				errors.reportError(errmsg);
			}
			return false;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Load full list of color maps for UI pick-list.

	public static ArrayList<KeyedRecord> getColorMaps(String theDbID) {
		return getColorMaps(theDbID, null);
	}

	public static ArrayList<KeyedRecord> getColorMaps(String theDbID, ErrorLogger errors) {

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			return null;
		}

		ArrayList<KeyedRecord> result = new ArrayList<KeyedRecord>();
		String errmsg = null;

		try {
			db.query("SELECT color_map_key, name FROM color_map ORDER BY 1");
			while (db.next()) {
				result.add(new KeyedRecord(db.getInt(1), db.getString(2)));
			}
		} catch (SQLException se) {
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (null != errmsg) {
			if (null != errors) {
				errors.reportError(errmsg);
			}
			return null;
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Load a color map by key, return null on error.

	public static ColorMap getColorMap(String theDbID, int mapKey) {
		return getColorMap(theDbID, mapKey, null);
	}

	public static ColorMap getColorMap(String theDbID, int mapKey, ErrorLogger errors) {

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			return null;
		}

		ColorMap result = new ColorMap(mapKey);
		String errmsg = null;

		try {

			db.update("LOCK TABLES color_map WRITE, color_map_data WRITE");

			String theKey = String.valueOf(mapKey);

			db.query("SELECT name, permanent FROM color_map WHERE color_map_key = " + theKey);
			if (db.next()) {

				result.name = db.getString(1);
				result.isPermanent = db.getBoolean(2);

				db.query("SELECT level, color_r, color_g, color_b FROM color_map_data WHERE color_map_key = " +
					theKey + " ORDER BY 1");

				double lvl;
				int r, g, b, colIndex = -1;

				while (db.next()) {

					lvl = db.getDouble(1);
					r = db.getInt(2);
					g = db.getInt(3);
					b = db.getInt(4);

					if (LEVEL_CODE_NO_SERVICE == lvl) {
						if ((r >= 0) && (g >= 0) && (b >= 0)) {
							result.noServiceColor = new Color(r, g, b);
						}
					} else {
						if (LEVEL_CODE_INTERFERENCE == lvl) {
							if ((r >= 0) && (g >= 0) && (b >= 0)) {
								result.interferenceColor = new Color(r, g, b);
							}
						} else {
							if (LEVEL_CODE_BACKGROUND == lvl) {
								result.backgroundColor = new Color(r, g, b);
							} else {
								if (colIndex < 0) {
									result.colorCount = 0;
								}
								colIndex = result.colorCount++;
								result.expandArrays();
								result.levels[colIndex] = lvl;
								result.colors[colIndex] = new Color(r, g, b);
							}
						}
					}
				}

			} else {
				errmsg = "Color map not found";
			}

		} catch (SQLException se) {
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		try {
			db.update("UNLOCK TABLES");
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (null != errmsg) {
			if (null != errors) {
				errors.reportError(errmsg);
			}
			return null;
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Delete a color map.  This will not delete a permanent map.

	public static boolean deleteColorMap(String theDbID, int mapKey) {
		return deleteColorMap(theDbID, mapKey, null);
	}

	public static boolean deleteColorMap(String theDbID, int mapKey, ErrorLogger errors) {

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			return false;
		}

		String errmsg = null;

		try {

			db.update("LOCK TABLES color_map WRITE, color_map_data WRITE");

			String theKey = String.valueOf(mapKey);

			db.query("SELECT permanent FROM color_map WHERE color_map_key = " + theKey);
			if (db.next()) {

				if (db.getBoolean(1)) {
					errmsg = "Color map cannot be deleted";
				} else {

					db.update("DELETE FROM color_map WHERE color_map_key = " + theKey);
					db.update("DELETE FROM color_map_data WHERE color_map_key = " + theKey);
				}
			}

		} catch (SQLException se) {
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		try {
			db.update("UNLOCK TABLES");
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (null != errmsg) {
			if (null != errors) {
				errors.reportError(errmsg);
			}
			return false;
		}

		return true;
	}
}
