//
//  AppCore.java
//  TVStudy
//
//  Copyright (c) 2012-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.core;

import codeid.CodeID;

import gov.fcc.tvstudy.core.data.*;

import java.util.*;
import java.util.logging.*;
import java.util.regex.*;
import java.text.*;
import java.io.*;
import java.nio.file.*;

import javax.xml.parsers.*;
import org.xml.sax.*;
import org.xml.sax.helpers.*;


//=====================================================================================================================
// Collection of core application support methods and properties, this is all static and is never instantiated.  This
// includes low-level logging of messages not meant to be reported to the user; user error reporting is handled by the
// ErrorLogger class.  A local file-backed name-value property store is managed here, to hold persistent state.  Also
// a similar read-only store for configuration settings.  Various formatting and parsing methods are provided.

public class AppCore {

	// The application version.  Version numbers are X.Y.Z version strings expressed as XYYZZZ.

	public static final String APP_VERSION_STRING = "2.3.0";
	public static final int APP_VERSION = 203000;

	public static String appVersionString = AppCore.APP_VERSION_STRING + " (" + CodeID.ID + ")";

	// Current XML export version.

	public static final int XML_VERSION = 105000;

	// Study engine executable name and standard output file name.

	public static final String STUDY_ENGINE_NAME = "tvstudy";
	public static final String STUDY_ENGINE_REPORT = "tvstudy.txt";

	// The study engine has special behaviors to support indirect control through this front-end application, those
	// include a prompt-and-response system and status messages.  See e.g. ProcessPanel and IxCheckAPI.

	public static final String ENGINE_PROMPT_PREFIX = "#*#*#";

	public static final String ENGINE_MESSAGE_PREFIX = "$$";
	public static final int ENGINE_MESSAGE_PREFIX_LENGTH = 2;

	public static final String ENGINE_PROGRESS_KEY = "progress";
	public static final String ENGINE_FILE_KEY = "outfile";
	public static final String ENGINE_REPORT_KEY = "report";
	public static final String ENGINE_RUNCOUNT_KEY = "runcount";
	public static final String ENGINE_RESULT_KEY = "result";
	public static final String ENGINE_PID_KEY = "pid";

	// Message types for logging and reporting.

	public static final int INFORMATION_MESSAGE = 1;
	public static final int WARNING_MESSAGE = 2;
	public static final int ERROR_MESSAGE = 3;

	// Debug flag.

	public static boolean Debug = (null != System.getProperty("DEBUG"));

	// Determines if image output options are available, see initialize().

	public static boolean showImageOptions = false;

	// Path to various directory and files, see initialize().

	public static Path workingDirectoryPath;

	public static final String LIB_DIRECTORY_NAME = "lib";
	public static final String DATA_DIRECTORY_NAME = "data";
	public static final String XML_DIRECTORY_NAME = "xml";
	public static final String DBASE_DIRECTORY_NAME = "dbase";
	public static final String CACHE_DIRECTORY_NAME = "cache";
	public static final String HELP_DIRECTORY_NAME = "help";
	public static final String OUT_DIRECTORY_NAME = "out";

	public static Path libDirectoryPath;
	public static Path dataDirectoryPath;
	public static Path xmlDirectoryPath;
	public static Path dbaseDirectoryPath;
	public static Path cacheDirectoryPath;
	public static Path helpDirectoryPath;
	public static Path outDirectoryPath;
	public static String studyOutDirectory;

	// Logger.

	private static final String LOGGER_NAME = "gov.fcc.tvstudy";
	private static final String LOG_FILE_NAME = "tvstudy_err.log";
	private static Logger logger;

	// Configuration and properties.  All CONFIG_* keys are defined in the read-only configuration, but may also be
	// defined in properties.  Code should look up these keys with getPreference(), which will check properties first
	// and fall back to the configuration if needed, so it will always return a value for a CONFIG_* key.  The PREF_*
	// keys are only stored in properties so will return null if not defined.

	private static final String CONFIG_FILE_NAME = "config.props";
	private static Properties configProperties;

	public static final String CONFIG_SHOW_DB_NAME = "showDbName";
	public static final String CONFIG_MYSQL_SOCKET_FILE = "mysqlSocketFile";
	public static final String CONFIG_SHOW_CREATE_RESULT_TABLES = "showCreateResultTables";
	public static final String CONFIG_HIDE_USER_RECORD_DELETE = "hideUserRecordDelete";
	public static final String CONFIG_ALLOW_FREQUENCY_EDIT = "allowFrequencyEdit";
	public static final String CONFIG_LMS_DOWNLOAD_URL = "lmsDownloadURL";
	public static final String CONFIG_AUTO_DELETE_PREVIOUS_DOWNLOAD = "autoDeletePreviousDownload";
	public static final String CONFIG_LMS_BASELINE_DATE = "lmsBaselineDate";
	public static final String CONFIG_TVIX_WINDOW_END_DATE = "ixCheckFilingWindowEndDate";
	public static final String CONFIG_TVIX_PROTECT_PRE_BASELINE_DEFAULT = "ixCheckProtectPreBaselineDefault";
	public static final String CONFIG_TVIX_INCLUDE_FOREIGN_DEFAULT = "ixCheckIncludeForeignDefault";
	public static final String CONFIG_TVIX_DEFAULT_CELL_SIZE = "ixCheckDefaultCellSize";
	public static final String CONFIG_TVIX_DEFAULT_PROFILE_RESOLUTION = "ixCheckDefaultProfileResolution";
	public static final String CONFIG_TVIX_DEFAULT_CELL_SIZE_LPTV = "ixCheckDefaultCellSizeLPTV";
	public static final String CONFIG_TVIX_DEFAULT_PROFILE_RESOLUTION_LPTV = "ixCheckDefaultProfileResolutionLPTV";
	public static final String CONFIG_TVIX_AM_SEARCH_DISTANCE_ND = "ixCheckAMSearchDistanceND";
	public static final String CONFIG_TVIX_AM_SEARCH_DISTANCE_DA = "ixCheckAMSearchDistanceDA";

	private static final String PROPS_FILE_NAME = "tvstudy.props";
	private static Properties localProperties;
	private static Path propsPath;

	public static final String PREF_CACHE_DIRECTORY = "cacheDirectory";
	public static final String PREF_OUT_DIRECTORY = "outDirectory";
	public static final String PREF_DEFAULT_ENGINE_MEMORY_LIMIT = "defaultEngineMemoryLimit";
	public static final String PREF_TVIX_DEFAULT_CP_EXCLUDES_BL = "ixCheckDefaultCPExcludesBL";
	public static final String PREF_TVIX_DEFAULT_EXCLUDE_NEW_LPTV = "ixCheckDefaultExcludeNewLPTV";
	public static final String PREF_STUDY_MANAGER_NAME_COLUMN_FIRST = "studyManagerNameColumnFirst";
	public static final String PREF_USE_SCREEN_MENU_BAR = "useScreenMenuBar";

	public static final String LAST_FILE_DIRECTORY_KEY = "last_file_directory";

	// Separate properties file storing database server login information, see APIOperation and ExtDb.

	public static final String API_PROPS_FILE_NAME = "api_login.props";

	// An estimate value for cache disk space needed to study one source, see isStudyCacheSpaceAvailable().

	private static final long SOURCE_CACHE_SPACE_NEEDED = 2000000L;

	// Number of CPU cores, see initialize(), also AppTask.

	public static int availableCPUCount = Runtime.getRuntime().availableProcessors();

	// Engine version number, maximum number of study engine processes, propagation model list, see initialize().  If
	// the max engine process count is zero it means there is not enough memory for the engine to run at all, the UI
	// can still be used for editing but no study runs can be performed.  If it is -1 it means the engine executable
	// was not reachable when probed for memory requirements, again there should be no attempt to run the engine.

	public static String engineVersionString = "(unknown)";
	public static int maxEngineProcessCount = -1;
	public static ArrayList<KeyedRecord> propagationModels = new ArrayList<KeyedRecord>();

	// Prefix strings for comment and metadata lines in text files, see readLineSkipCommments().

	public static final char[] TEXT_FILE_COMMENT_PREFIX = {'#', '%'};
	public static final char TEXT_FILE_METADATA_PREFIX = '$';
	public static final char TEXT_FILE_METADATA_SEPARATOR = '=';

	// Comment patterns used for version number in support files, subset of metadata comments, separate from general
	// text parsing because this also needs to support versioning in XML files.  See getFileVersion().

	private static final String FILE_VERSION_COMMENT_PREFIX = "#";
	private static final String FILE_VERSION_PREFIX = "#$version=";
	private static final String FILE_VERSION_XML_COMMENT_PREFIX = "<!--";
	private static final String FILE_VERSION_XML_PREFIX = "<!--$version=";

	// State set in initialize() when it checks support file installation, if fileCheckError is non-null some file was
	// not found or was the wrong version, the application should probably exit immediately.

	public static final String FILE_CHECK_LIST_FILE = "versions.dat";
	public static final int FILE_CHECK_MINIMUM_VERSION = 16;
	public static int fileCheckVersion;
	public static String fileCheckID = "(unknown)";
	public static String fileCheckError;

	// Separator used in path names listed in FILE_CHECK_LIST_FILE, so the contents of that are platform-independent.

	public static final String FILE_PATH_SEPARATOR = "/";

	// Static initialization, the logger and properties will be associated with files later, see initialize().

	static {

		logger = Logger.getLogger(LOGGER_NAME);
		logger.setUseParentHandlers(false);
		if (Debug) {
			logger.setLevel(Level.ALL);
		} else {
			logger.setLevel(Level.SEVERE);
		}

		configProperties = new Properties();

		localProperties = new Properties();
		Runtime.getRuntime().addShutdownHook(new Thread() {
			public void run() {
				saveProperties();
			}
		});
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Initialization, set the working directory and derive paths and file names, associate the logger and properties
	// with files, and optionally redirect the standard error stream to the log file.  This must be called early in
	// application startup, and can only be used once.

	private static boolean didInit = false;

	public static synchronized void initialize(String workDir, boolean redirectStderr, boolean doEngineChecks) {

		if (didInit) {
			return;
		}
		didInit = true;

		// The default locations for application support directories are relative to the install directory, make
		// absolute paths for those off the current working directory.  The cache and output locations can be changed
		// by preference but those will be saved as relative paths if below the install directory, so that directory
		// can be relocated.  The output directory is saved in each study record and can be changed per study with
		// similar logic, also keep the output location in a potentially relative string form to use for new studies.

		workingDirectoryPath = Paths.get(workDir).toAbsolutePath();

		libDirectoryPath = workingDirectoryPath.resolve(LIB_DIRECTORY_NAME);
		dataDirectoryPath = workingDirectoryPath.resolve(DATA_DIRECTORY_NAME);
		xmlDirectoryPath = workingDirectoryPath.resolve(XML_DIRECTORY_NAME);
		dbaseDirectoryPath = workingDirectoryPath.resolve(DBASE_DIRECTORY_NAME);
		cacheDirectoryPath = workingDirectoryPath.resolve(CACHE_DIRECTORY_NAME);
		helpDirectoryPath = workingDirectoryPath.resolve(HELP_DIRECTORY_NAME);
		outDirectoryPath = workingDirectoryPath.resolve(OUT_DIRECTORY_NAME);
		studyOutDirectory = OUT_DIRECTORY_NAME;

		// Direct the logger to a file.

		OutputStream logStream = null;
		try {
			logStream = Files.newOutputStream(libDirectoryPath.resolve(LOG_FILE_NAME), StandardOpenOption.CREATE,
				StandardOpenOption.APPEND);
			logger.addHandler(new StreamHandler(logStream, new LogFormatter()));
		} catch (Throwable t) {
			log(ERROR_MESSAGE, "Could not initialize logger: ", t);
		}

		// In some environments (e.g. desktop on MacOS X), stderr dumps into the bit bucket and useful messages like
		// uncaught exceptions just vanish.  In that case it is desireable to redirect stderr to the local log file.
		// However when running from other contexts (e.g. a servlet container) that would not be a nice thing to do
		// since messages from unrelated code would end up in TVStudy's log.  Hence the caller has to decide.

		if ((null != logStream) && redirectStderr) {
			System.setErr(new PrintStream(logStream, true));
		}

		// Check support files for correct installation and current version.  The list of files and expected versions
		// are stored in the file "lib/versions.dat".  That file itself has a version number which is checked against
		// a minimum expected value here.  Many of the files may be updated without any update to the code, but such
		// updates will always change the versions list and it's version number.  The main purpose of this check is to
		// catch incomplete installation of such file-only updates.  But at the next code update, the minimum version
		// for the versions list will change so any missed file-only updates must be installed with the code update.
		// The versions list also has a descriptive ID string which appears in the "About" dialog.  If any problem is
		// found or error occurs, fileCheckError is set and other properties are cleared.  It's up to the caller to
		// decide how to behave in that situation; the initialization will continue here.

		String indexFileName, fileName = null;
		Path indexFilePath;
		BufferedReader indexReader = null;

		try {

			indexFileName = LIB_DIRECTORY_NAME + FILE_PATH_SEPARATOR + FILE_CHECK_LIST_FILE;
			fileName = indexFileName;

			try {
				indexReader = Files.newBufferedReader(libDirectoryPath.resolve(FILE_CHECK_LIST_FILE));
			} catch (IOException ie) {
				throw new FileNotFoundException(ie.getMessage());
			}

			fileCheckVersion = getFileVersion(indexReader);
			if (fileCheckVersion < FILE_CHECK_MINIMUM_VERSION) {
				fileCheckError = "File check failed, '" + indexFileName + "' is the wrong version";
			} else {

				fileCheckID = readLineSkipComments(indexReader);
				if (null == fileCheckID) {
					fileCheckError = "File check failed, bad format in '" + indexFileName + "'";
				} else {

					String str;
					int requiredVersion, fileVersion;

					while (true) {

						fileName = readLineSkipComments(indexReader);
						if (null == fileName) {
							break;
						}
						str = readLineSkipComments(indexReader);
						requiredVersion = 0;
						if (null != str) {
							try {
								requiredVersion = Integer.parseInt(str);
							} catch (NumberFormatException ne) {
							}
						}
						if (requiredVersion <= 0) {
							fileCheckError = "File check failed, bad format in '" + indexFileName + "'";
							break;
						}

						if (fileName.toLowerCase().endsWith(".xml")) {
							fileVersion = getXMLFileVersion(getSupportFilePath(fileName));
						} else {
							fileVersion = getFileVersion(getSupportFilePath(fileName));
						}
						if (fileVersion != requiredVersion) {
							fileCheckError = "File check failed, '" + fileName + "' is the wrong version";
							break;
						}
					}
				}
			}

		} catch (FileNotFoundException fe) {
			fileCheckError = "File check failed, '" + fileName + "' not found";
		} catch (IOException ie) {
			fileCheckError = "File check failed, I/O error:\n" + ie.getMessage();
		}

		try {
			if (null != indexReader) {
				indexReader.close();
			}
		} catch (IOException ie) {
		}

		if (null != fileCheckError) {
			fileCheckVersion = 0;
			fileCheckID = "(failed)";
		}

		// Load configuration and properties from files.

		InputStream theStream;

		try {
			theStream = Files.newInputStream(libDirectoryPath.resolve(CONFIG_FILE_NAME));
			configProperties.load(theStream);
			theStream.close();
		} catch (Throwable t) {
		}

		propsPath = libDirectoryPath.resolve(PROPS_FILE_NAME);
		try {
			theStream = Files.newInputStream(propsPath);
			localProperties.load(theStream);
			theStream.close();
		} catch (Throwable t) {
		}

		// Check for custom locations of cache and output directories.  These may be relative or absolute as discussed
		// above, convert to absolute using the current working directory if needed.

		String str = getPreference(PREF_CACHE_DIRECTORY);
		if (null != str) {
			cacheDirectoryPath = Paths.get(str);
			if (!cacheDirectoryPath.isAbsolute()) {
				cacheDirectoryPath = workingDirectoryPath.resolve(str);
			}
		}

		str = getPreference(PREF_OUT_DIRECTORY);
		if (null != str) {
			studyOutDirectory = str;
			outDirectoryPath = Paths.get(str);
			if (!outDirectoryPath.isAbsolute()) {
				outDirectoryPath = workingDirectoryPath.resolve(str);
			}
		}

		// If this is initializing for a utility function that won't be using the study engine, skip the rest.

		if (!doEngineChecks) {
			return;
		}

		// Query the study engine for various information including it's verison number, a maximum process count based
		// on available memory, and a list of available propagation models.  If this succeeds maxEngineProcessCount is
		// set to the maximum number of parallel engine processes that can safely run at the same time.  That is the
		// smaller of the memory limit the study engine reports and the number of CPU cores.  However if CPU cores is
		// limiting that is never less than 2, even on a single-CPU system there is a benefit to running parallel
		// processes if memory permits.  Note the engine may return a limit of 0 meaning the total memory is below its
		// minimum limit and it would refuse to do anything, it would start but immediately abort with a message about
		// insufficient memory.  In that case all run UI should be disabled but editing can still be performed.  If the
		// process count remains at -1 it means the engine query failed, again other code should not allow any attempt
		// to start an engine process.  But a -1 should probably show a different error message than a 0.

		maxEngineProcessCount = -1;
		int memCount = 1;

		try {

			// Set the dynamic linker library search path, this sets environment variables for both MacOS and Linux.

			ProcessBuilder pb = new ProcessBuilder(libDirectoryPath.resolve(STUDY_ENGINE_NAME).toString(), "-q");
			Map<String, String> env = pb.environment();
			String dyldpath = env.get("DYLD_LIBRARY_PATH");
			if (null == dyldpath) {
				dyldpath = env.get("LD_LIBRARY_PATH");
			}
			if (null == dyldpath) {
				dyldpath = libDirectoryPath.toString();
			} else {
				dyldpath = dyldpath + ":" + libDirectoryPath.toString();
			}
			env.put("DYLD_LIBRARY_PATH", dyldpath);
			env.put("LD_LIBRARY_PATH", dyldpath);
			Process p = pb.start();

			BufferedReader br = new BufferedReader(new InputStreamReader(p.getInputStream()));

			String line;
			String[] parts;
			int i = 0, model;

			while (true) {

				line = br.readLine();
				if (null == line) {
					break;
				}

				switch (++i) {

					case 1: {
						engineVersionString = line;
						break;
					}

					case 2: {
						try {
							memCount = Integer.parseInt(line);
							int cpuCount = availableCPUCount;
							if (cpuCount < 2) {
								cpuCount = 2;
							}
							maxEngineProcessCount = (memCount < cpuCount) ? memCount : cpuCount;
						} catch (NumberFormatException ne) {
						}
						break;
					}

					default: {
						parts = line.split("=");
						if (2 == parts.length) {
							model = 0;
							try {
								model = Integer.parseInt(parts[0].trim());
							} catch (NumberFormatException ne) {
							}
							if (model > 0) {
								propagationModels.add(new KeyedRecord(model, parts[1].trim()));
							}
						}
						break;
					}
				}
			}

		} catch (Throwable t) {
			log(ERROR_MESSAGE, "Unexpected error", t);
		}

		// Image output depends on separately-installed GhostScript imaging software.  There must be a symlink to that
		// software at $lib/gs if it is installed.  If not, image output options are hidden, see OutputConfig.

		showImageOptions = Files.exists(libDirectoryPath.resolve("gs"));
	}


	//=================================================================================================================
	// Formatter for log messages.

	private static class LogFormatter extends java.util.logging.Formatter {

		public String format(LogRecord record) {

			StringWriter result = new StringWriter();
			result.write(formatTimestamp(record.getMillis()));
			result.write(" ");
			result.write(record.getLoggerName());
			result.write(" [");
			result.write(record.getLevel().getLocalizedName());
			result.write("] : ");
			result.write(record.getMessage());
			result.write("\n");

			Throwable theThrown = record.getThrown();
			if (null != theThrown) {
				theThrown.printStackTrace(new PrintWriter(result));
			}

			return result.toString();
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Logging methods.  Mostly just wrappers around the Logger, but providing message type mapping.

	public static void log(int type, String msg) {

		Level level = Level.INFO;
		if (ERROR_MESSAGE == type) {
			level = Level.SEVERE;
		} else {
			if (WARNING_MESSAGE == type) {
				level = Level.WARNING;
			}
		}
		logger.log(level, msg);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public static void log(int type, String msg, Throwable thrown) {

		Level level = Level.INFO;
		if (ERROR_MESSAGE == type) {
			level = Level.SEVERE;
		} else {
			if (WARNING_MESSAGE == type) {
				level = Level.WARNING;
			}
		}
		logger.log(level, msg, thrown);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Local properties.

	public static String getProperty(String name) {

		return localProperties.getProperty(name);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Values stored in the properties can never be null.

	public static void setProperty(String name, String value) {

		if (null == value) {
			return;
		}
		localProperties.setProperty(name, value);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Commit properties to the file.

	public static void saveProperties() {

		if (null != propsPath) {
			try {
				localProperties.store(Files.newOutputStream(propsPath), "");
			} catch (Throwable t) {
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Preferences.  These usually access the local property store, but if a get finds no value in local properties
	// the configuration properties are checked, and setting null will remove the key from properties.

	public static String getPreference(String name) {

		String value = localProperties.getProperty(name);
		if (null == value) {
			value = configProperties.getProperty(name);
		}
		return value;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public static void setPreference(String name, String value) {

		if (null == value) {
			localProperties.remove(name);
		} else {
			localProperties.setProperty(name, value);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Methods to change the cache and output directories.  Setting to null will revert to the default path and remove
	// any existing preference, also setting the default path is the same as setting null.  If the new directory is
	// below the current working directory it will be saved as a relative path.

	public static void setCacheDirectoryPath(Path thePath) {

		Path defPath = workingDirectoryPath.resolve(CACHE_DIRECTORY_NAME);

		Path newPath = defPath;
		if (null != thePath) {
			newPath = thePath.toAbsolutePath();
			if (newPath.equals(defPath)) {
				thePath = null;
				newPath = defPath;
			}
		}

		cacheDirectoryPath = newPath;

		if (null == thePath) {
			setPreference(PREF_CACHE_DIRECTORY, null);
		} else {
			if (newPath.startsWith(workingDirectoryPath)) {
				newPath = workingDirectoryPath.relativize(newPath);
			}
			setPreference(PREF_CACHE_DIRECTORY, newPath.toString());
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The output directory is also kept as a path string to use when setting the output directory on a study, that
	// may be a relative path string if the output directory is below the working directory.

	public static void setOutDirectoryPath(Path thePath) {

		Path defPath = workingDirectoryPath.resolve(OUT_DIRECTORY_NAME);

		Path newPath = defPath;
		if (null != thePath) {
			newPath = thePath.toAbsolutePath();
			if (newPath.equals(defPath)) {
				thePath = null;
				newPath = defPath;
			}
		}

		outDirectoryPath = newPath;
		if (newPath.startsWith(workingDirectoryPath)) {
			studyOutDirectory = workingDirectoryPath.relativize(newPath).toString();
		} else {
			studyOutDirectory = newPath.toString();
		}

		if (null == thePath) {
			setPreference(PREF_OUT_DIRECTORY, null);
		} else {
			setPreference(PREF_OUT_DIRECTORY, studyOutDirectory);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Utility methods for working with cache and output files and directories.  First get the storage space used by
	// cache files for a specified database and study.  If the study key is 0, this will return the total size for all
	// studies in the database.

	public static long getStudyCacheSize(String theDbID) {
		return getStudyCacheSize(theDbID, 0);
	}

	public static long getStudyCacheSize(String theDbID, int studyKey) {

		long newSize = 0L;

		File cacheRoot;

		if (studyKey > 0) {
			cacheRoot = cacheDirectoryPath.resolve(theDbID).resolve(String.valueOf(studyKey)).toFile();
		} else {
			cacheRoot = cacheDirectoryPath.resolve(theDbID).toFile();
		}

		if (cacheRoot.exists() && cacheRoot.isDirectory()) {
			newSize = sizeOfDirectoryContents(cacheRoot);
		}

		return newSize;
	}


	//=================================================================================================================
	// Information returned by getFreeSpace(), usable free space in bytes on the file store containing the cache
	// directory, and the output directory, for a particular database ID.  If sameFileStore is true, the cache and
	// output are on the same file store so are sharing the same free space.

	public static class FreeSpaceInfo {

		public static String dbID;

		public long totalFreeSpace;

		public boolean sameFileStore;

		public long cacheFreeSpace;
		public long outputFreeSpace;


		//-------------------------------------------------------------------------------------------------------------

		public FreeSpaceInfo(String theDbID) {

			dbID = theDbID;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get free space for cache and output file allocation for a database.  This tries to check the database-specific
	// directories to allow for any possible use of symlinks to separate storage into different physical filesystems,
	// but if the database-specific directory doesn't yet exist the root cache or output directory is checked.  This
	// always returns an result, if either free space value can't be determined it is set to -1.

	public static FreeSpaceInfo getFreeSpace(String theDbID) {

		FreeSpaceInfo theInfo = new FreeSpaceInfo(theDbID);
		FileStore cacheStore = null, outputStore = null;
		try {
			Path cachePath = cacheDirectoryPath.resolve(theDbID);
			if (!Files.exists(cachePath)) {
				cachePath = cacheDirectoryPath;
				if (!Files.exists(cachePath)) {
					cachePath = workingDirectoryPath;
				}
			}
			cacheStore = Files.getFileStore(cachePath);
			theInfo.cacheFreeSpace = cacheStore.getUsableSpace();
		} catch (Throwable t) {
		}
		if (theInfo.cacheFreeSpace <= 0L) {
			theInfo.cacheFreeSpace = -1L;
		}

		try {
			String hostDir = DbCore.getHostDbName(theDbID);
			Path outPath = null;
			if (null != hostDir) {
				outPath = outDirectoryPath.resolve(hostDir);
				if (!Files.exists(outPath)) {
					outPath = outDirectoryPath;
				}
			} else {
				outPath = outDirectoryPath;
			}
			if (!Files.exists(outPath)) {
				outPath = workingDirectoryPath;
			}
			outputStore = Files.getFileStore(outPath);
			theInfo.outputFreeSpace = outputStore.getUsableSpace();
		} catch (Throwable t) {
		}
		if (theInfo.outputFreeSpace <= 0L) {
			theInfo.outputFreeSpace = -1L;
		}

		if ((null != cacheStore) && (null != outputStore)) {
			theInfo.sameFileStore = cacheStore.equals(outputStore);
		}
		if (theInfo.sameFileStore) {
			theInfo.totalFreeSpace = theInfo.cacheFreeSpace;
		} else {
			if ((theInfo.cacheFreeSpace > 0L) && (theInfo.outputFreeSpace > 0L)) {
				theInfo.totalFreeSpace = theInfo.cacheFreeSpace + theInfo.outputFreeSpace;
			} else {
				theInfo.totalFreeSpace = -1L;
			}
		}

		return theInfo;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Test if disk space is available for estimate allocations for a study run, cache size is checked based on the
	// number of desired sources to be studied, output file size is passed in bytes and must be estimated by the
	// caller.  Zero for either will skip that test.  If free space cannot be determined the return is always false.
	// This is only used to generate a warning alert to the user which can be ignored.

	public static boolean isFreeSpaceAvailable(String theDbID, int sourceCount, long outputNeeded) {

		long cacheNeeded = (long)sourceCount * SOURCE_CACHE_SPACE_NEEDED;

		FreeSpaceInfo theInfo = getFreeSpace(theDbID);

		if (theInfo.sameFileStore) {
			return (cacheNeeded + outputNeeded) < theInfo.totalFreeSpace;
		}

		boolean cacheOK = true, outputOK = true;
		if (cacheNeeded > 0L) {
			cacheOK = (cacheNeeded < theInfo.cacheFreeSpace);
		}
		if (outputNeeded > 0L) {
			outputOK = (outputNeeded < theInfo.outputFreeSpace);
		}
		return (cacheOK && outputOK);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Delete study engine cache files, see comments above.

	public static boolean deleteStudyCache(String theDbID, int studyKey) {

		boolean result = false;

		File cacheRoot;

		if (studyKey > 0) {
			cacheRoot = cacheDirectoryPath.resolve(theDbID).resolve(String.valueOf(studyKey)).toFile();
		} else {
			cacheRoot = cacheDirectoryPath.resolve(theDbID).toFile();
		}

		if (cacheRoot.exists() && cacheRoot.isDirectory()) {
			result = deleteDirectoryAndContents(cacheRoot);
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Delete study engine cache files for sources that no longer exist, based on a source key use map.

	public static boolean purgeStudyCache(String theDbID, int studyKey, boolean[] sourceKeyMap) {

		boolean result = false;

		File cacheRoot = cacheDirectoryPath.resolve(theDbID).resolve(String.valueOf(studyKey)).toFile();

		if (cacheRoot.exists() && cacheRoot.isDirectory()) {
			result = deleteUnusedCacheFiles(cacheRoot, sourceKeyMap);
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Recurse through the cache directory for a study.  For each file extract the source key, check map of keys in
	// use, if not found delete the file.  For most files the key is the entire file name, undesired cache files in
	// local grid mode have a second key in the name following a '_', but the first key is the relevant one.  If a
	// file name doesn't parse leave it alone.  If a key is found that is beyond the range of the map, delete it.
	// This is private because only purgeStudyCache() needs it, it has no generic use.

	private static boolean deleteUnusedCacheFiles(File theDir, boolean[] sourceKeyMap) {

		File[] contents = theDir.listFiles();
		if (null == contents) {
			return false;
		}

		boolean result = true;

		String theName;
		int theKey, pos;

		for (int i = 0; i < contents.length; i++) {
			if (contents[i].isDirectory()) {
				if (!deleteUnusedCacheFiles(contents[i], sourceKeyMap)) {
					result = false;
				}
			} else {
				theName = contents[i].getName();
				pos = theName.indexOf('_');
				if (pos > 0) {
					theName = theName.substring(0, pos);
				}
				theKey = 0;
				try {
					theKey = Integer.parseInt(theName);
				} catch (NumberFormatException nfe) {
				}
				if (theKey > 0) {
					if ((theKey >= sourceKeyMap.length) || !sourceKeyMap[theKey]) {
						if (!contents[i].delete()) {
							result = false;
						}
					}
				}
			}
		}

		return result;
	}


	//-------------------------------------------------------------------------------------------------------------
	// Sum the length of all files in a directory tree.

	public static long sizeOfDirectoryContents(File theDir) {

		File[] contents = theDir.listFiles();
		if (null == contents) {
			return 0;
		}

		long result = 0;

		for (int i = 0; i < contents.length; i++) {
			if (contents[i].isDirectory()) {
				result += sizeOfDirectoryContents(contents[i]);
			} else {
				result += contents[i].length();
			}
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Delete a directory tree.  The return indicates if any errors occurred, but this will continue trying to delete
	// files and directories through the entire tree regardless of previous errors.

	public static boolean deleteDirectoryAndContents(File theDir) {

		File[] contents = theDir.listFiles();
		if (null == contents) {
			return false;
		}

		boolean result = true;

		for (int i = 0; i < contents.length; i++) {
			if (contents[i].isDirectory()) {
				if (!deleteDirectoryAndContents(contents[i])) {
					result = false;
				}
			} else {
				if (!contents[i].delete()) {
					result = false;
				}
			}
		}

		if (result && !theDir.delete()) {
			result = false;
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Methods for formatting and parsing coordinates, first geographic latitude in decimal degrees broken out to
	// degrees-minutes-seconds.

	public static String formatLatitude(double theValue) {

		if (theValue < 0) {
			return formatLatLon(-theValue) + " S";
		} else {
			return formatLatLon(theValue) + " N";
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Geographic longitude, as above.

	public static String formatLongitude(double theValue) {

		if (theValue < 0) {
			return formatLatLon(-theValue) + " E";
		} else {
			return formatLatLon(theValue) + " W";
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private static String formatLatLon(double theValue) {

		int deg = (int)theValue;
		int min = (int)((theValue - (double)deg) * 60.);
		double sec = (((theValue - (double)deg) * 60.) - (double)min) * 60.;
		if (sec >= 59.995) {
			sec = 0.;
			if (60 == ++min) {
				min = 0;
				++deg;
			}
		}

		return String.format(Locale.US, "%3d %02d %05.2f", deg, min, sec);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Parse latitude/longitude strings as created by the formatting methods above.  These will also allow the string
	// to just contain a signed decimal value.

	public static double parseLatitude(String str) {

		return parseLatLon(str, (Source.LATITUDE_MIN - 1.), "S");
	}


	//-----------------------------------------------------------------------------------------------------------------

	public static double parseLongitude(String str) {

		return parseLatLon(str, (Source.LONGITUDE_MIN - 1.), "E");
	}


	//-----------------------------------------------------------------------------------------------------------------

	private static Pattern latLonPattern = Pattern.compile("\\s");

	private static double parseLatLon(String str, double bad, String neg) {

		if (null == str) {
			return bad;
		}

		double latlon = bad;

		String[] tokens = latLonPattern.split(str.trim());

		if (1 == tokens.length) {

			try {

				latlon = Double.parseDouble(tokens[0]);

			} catch (NumberFormatException nfe) {
			}

		} else {

			if ((3 == tokens.length) || (4 == tokens.length)) {

				try {

					int deg = Integer.parseInt(tokens[0]);
					int min = Integer.parseInt(tokens[1]);
					double sec = Double.parseDouble(tokens[2]);

					latlon = (double)deg + ((double)min / 60.) + (sec / 3600.);

					if ((4 == tokens.length) && (tokens[3].equalsIgnoreCase(neg))) {
						latlon = -latlon;
					}

				} catch (NumberFormatException nfe) {
				}
			}
		}

		return latlon;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Methods for formatting and parsing numbers so the display and input behavior are consistent application-wide.
	// First decimal values with varying precision.  NumberFormat is not synchronized, so these methods are.

	public static final int MAX_DECIMAL_PREC = 8;

	private static NumberFormat decimalFormatter = null;

	static {
		decimalFormatter = NumberFormat.getInstance(Locale.US);
		decimalFormatter.setMinimumIntegerDigits(1);
		decimalFormatter.setGroupingUsed(false);
	}

	public static String formatDecimal(double theValue, int prec) {
		return formatDecimal(theValue, prec, prec);
	}

	public static synchronized String formatDecimal(double theValue, int minPrec, int maxPrec) {

		if (maxPrec < 0) {
			maxPrec = 0;
		}
		if (maxPrec > MAX_DECIMAL_PREC) {
			maxPrec = MAX_DECIMAL_PREC;
		}
		if (minPrec < 0) {
			minPrec = 0;
		}
		if (minPrec > maxPrec) {
			minPrec = maxPrec;
		}
		decimalFormatter.setMinimumFractionDigits(minPrec);
		decimalFormatter.setMaximumFractionDigits(maxPrec);

		return decimalFormatter.format(theValue);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Wrappers for formatting varius real-world decimal values with assumed units, i.e. distance in km, ERP in kW.
	// Code displaying such values always used these methods, so appearance can be altered application-wide by just
	// changing these.  First, a geographic coordinate in arc-seconds.

	public static String formatSeconds(double theValue) {

		return formatDecimal(theValue, 2, 2);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Height in meters.

	public static String formatHeight(double theValue) {

		return formatDecimal(theValue, 1, 1);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Distance in kilometers.

	public static String formatDistance(double theValue) {

		return formatDecimal(theValue, 2, 2);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Azimuth or bearing in degrees true.

	public static String formatAzimuth(double theValue) {

		return formatDecimal(theValue, 1, 1);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Depression angle in degrees.

	public static String formatDepression(double theValue) {

		return formatDecimal(theValue, 2, 2);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// ERP in kilowatts, the precision is range-dependent.

	public static String formatERP(double theValue) {

		if (theValue < 0.001) {
			return String.valueOf(theValue);
		} else {
			if (theValue < 0.9995) {
				return formatDecimal(theValue, 3, 3);
			} else {
				if (theValue < 9.995) {
					return formatDecimal(theValue, 2, 2);
				} else {
					if (theValue < 99.95) {
						return formatDecimal(theValue, 1, 1);
					} else {
						return formatDecimal(theValue, 0, 0);
					}
				}
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Relative field, 0-1.  If less than 0.001 don't limit the number of fraction digits, this should never display as
	// 0 unless it actually is 0, which it should never be since 0 is invalid in most contexts.

	public static String formatRelativeField(double theValue) {

		if (theValue < 0.001) {
			return String.valueOf(theValue);
		} else {
			return formatDecimal(theValue, 3, 3);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Field strength in dBu.

	public static String formatField(double theValue) {

		return formatDecimal(theValue, 1, 1);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// D/U value in dB.

	public static String formatDU(double theValue) {

		return formatDecimal(theValue, 1, 1);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Percentage.

	public static String formatPercent(double theValue) {

		return formatDecimal(theValue, 2, 2);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Format an integer or decimal value with grouping, decimal precision fixed at 1.

	private static NumberFormat iCountFormatter = null;
	private static NumberFormat dCountFormatter = null;

	static {

		iCountFormatter = NumberFormat.getIntegerInstance(Locale.US);
		iCountFormatter.setGroupingUsed(true);

		dCountFormatter = NumberFormat.getInstance(Locale.US);
		dCountFormatter.setMinimumFractionDigits(1);
		dCountFormatter.setMaximumFractionDigits(1);
		dCountFormatter.setMinimumIntegerDigits(1);
		dCountFormatter.setGroupingUsed(true);
	}

	public static synchronized String formatCount(int theValue) {

		return iCountFormatter.format(theValue);
	}

	public static synchronized String formatCount(double theValue) {

		return dCountFormatter.format(theValue);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Convert a version number string in the form X, X.Y, or X.Y.Z to XYYZZZ format, return -1 if format not valid.
	// This will ignore additional dot-separated values beyond the first three, so version strings may include other
	// info as long as all three X, Y, and Z are present and Z is followed by a dot.

	public static int parseVersion(String theVers) {

		if (null == theVers) {
			return -1;
		}

		theVers = theVers.trim();
		if (0 == theVers.length()) {
			return -1;
		}

		String[] parts = theVers.split("\\.");

		int major = 0, minor = 0, dot = 0;

		try {
			major = Integer.parseInt(parts[0]);
			if (parts.length > 1) {
				minor = Integer.parseInt(parts[1]);
			}
			if (parts.length > 2) {
				dot = Integer.parseInt(parts[2]);
			}
		} catch (NumberFormatException ne) {
			return -1;
		}

		if ((major < 1) || (minor < 0) || (minor > 99) || (dot < 0) || (dot > 999)) {
			return -1;
		}

		return (major * 100000) + (minor * 1000) + dot;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Format an XYYZZZ version number into X.Y.Z format.  Also takes format XYYZZZSS, but SS is not shown unless > 0.

	public static String formatVersion(int version) {

		if (version <= 0) {
			return "";
		}

		int sub = 0;
		if (version > 9999999) {
			sub = version % 100;
			version /= 100;
		}

		int major = version / 100000;
		int minor = (version % 100000) / 1000;
		int dot = version % 1000;

		if (sub > 0) {
			return String.format(Locale.US, "%d.%d.%d.%d", major, minor, dot, sub);
		} else {
			return String.format(Locale.US, "%d.%d.%d", major, minor, dot);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Formatting and parsing methods for dates and times.  SimpleDateFormat is not thread-safe.

	private static final DateFormat dateTimeFormatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

	public static synchronized String formatDateTime(java.util.Date theDate) {

		if (null != theDate) {
			return dateTimeFormatter.format(theDate);
		}
		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	private static final DateFormat dateFormatter = new SimpleDateFormat("yyyy-MM-dd");

	public static synchronized String formatDate(java.util.Date theDate) {

		if (null != theDate) {
			return dateFormatter.format(theDate);
		}
		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	private static final DateFormat[] dateParsers = {
		new SimpleDateFormat("yyyy-MM-dd"),
		new SimpleDateFormat("M/d/yyyy"),
		new SimpleDateFormat("ddMMMyyyy")
	};

	public static synchronized java.util.Date parseDate(String s) {

		if (null == s) {
			return null;
		}

		java.util.Date d = null;
		for (DateFormat f : dateParsers) {
			try {
				d = f.parse(s);
			} catch (ParseException pe) {
			}
			if (null != d) {
				break;
			}
		}
		return d;
	}


	//-----------------------------------------------------------------------------------------------------------------

	private static final DateFormat dayFormatter = new SimpleDateFormat("EEEEEEEEE");

	public static synchronized String formatDay(java.util.Date theDate) {

		if (null != theDate) {
			return dayFormatter.format(theDate);
		}
		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	private static SimpleDateFormat timestampFormatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS", Locale.US);

	public static synchronized String formatTimestamp(long theTime) {

		return timestampFormatter.format(new java.util.Date(theTime));
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Format a number of bytes into a string, converting to gigabytes, megabytes, or kilobytes depending on value.
	// If the value is less than 0 return "(unknown)"; if it is exactly 0, return "-".

	public static String formatBytes(long bytes) {

		if (bytes < 0L) {
			return "(unknown)";
		}
		if (bytes >= 995000000L) {
			return String.format(Locale.US, "%.2f GB", ((double)bytes / 1.e9));
		}
		if (bytes >= 950000L) {
			return String.format(Locale.US, "%.1f MB", ((double)bytes / 1.e6));
		}
		if (bytes > 0L) {
			return String.format(Locale.US, "%.0f kB", ((double)bytes / 1.e3));
		}
		return "-";
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Run an XML parser.  Convenience.

	private static SAXParserFactory parserFactory = SAXParserFactory.newInstance();

	public static boolean parseXML(Reader xml, DefaultHandler handler, ErrorLogger errors) {

		try {
			parserFactory.newSAXParser().parse(new InputSource(xml), handler);

		} catch (SAXException se) {
			if (null != errors) {
				String msg = se.getMessage();
				if ((null != msg) && (msg.length() > 0)) {
					errors.reportError("XML error: " + msg);
				}
			}
			return false;

		} catch (Throwable t) {
			AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", t);
			if (null != errors) {
				errors.reportError("An unexpected error occurred:\n" + t);
			}
			return false;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Replace special characters in a string for XML output.

	public static String xmlclean(String theText) {

		StringBuilder result = new StringBuilder();
		char c;

		for (int i = 0; i < theText.length(); i++) {

			c = theText.charAt(i);

			switch (c) {

				case '&':
					result.append("&amp;");
					break;

				case '<':
					result.append("&lt;");
					break;

				case '>':
					result.append("&gt;");
					break;

				case '"':
					result.append("&quot;");
					break;

				default:
					result.append(c);
					break;
			}
		}

		return result.toString();
	}


	//=================================================================================================================
	// Class to store data from a DBF file, see readDBFData().

	public static class DBFData {

		public String fileName;

		public int fieldCount;
		public String[] fieldName;
		public char[] fieldType;
		public int[] fieldLength;

		public ArrayList<String[]> fieldData;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Read the contents of a DBF file.  Returns null on error.

	public static DBFData readDBFData(File dbfFile, boolean doTrim, ErrorLogger errors) {

		String fileName = dbfFile.getName();

		BufferedInputStream dbfStream = null;
		try {
			dbfStream = new BufferedInputStream(new FileInputStream(dbfFile));
		} catch (IOException ie) {
			if (null != errors) {
				errors.reportError("Could not open file '" + fileName + "':\n" + ie.getMessage());
			}
			return null;
		}

		DBFData data = new DBFData();
		data.fileName = fileName;

		byte[] buf = new byte[65536];
		int i, j, k, reclen = 0;
		StringBuilder s = new StringBuilder();
		String[] attrs;
		boolean firstRecord = true;

		String errmsg = null;

		try {

			if (32 != dbfStream.read(buf, 0, 32)) {
				errmsg = "Unexpected end-of-file";
				return null;
			}

			data.fieldCount = (((buf[8] & 0xFF) | (buf[9] & 0xFF) << 8) - 33) / 32;
			if ((data.fieldCount <= 0) || (data.fieldCount > 255)) {
				errmsg = "Bad field count in file header";
				return null;
			}

			data.fieldName = new String[data.fieldCount];
			data.fieldType = new char[data.fieldCount];
			data.fieldLength = new int[data.fieldCount];

			reclen = 1;

			for (i = 0; i < data.fieldCount; i++) {

				if (32 != dbfStream.read(buf, 0, 32)) {
					errmsg = "Unexpected end-of-file";
					return null;
				}

				s.setLength(0);
				for (j = 0; j < 11; j++) {
					if (0 == buf[j]) {
						break;
					}
					s.append((char)buf[j]);
				}
				if (doTrim) {
					data.fieldName[i] = s.toString().trim();
				} else {
					data.fieldName[i] = s.toString();
				}

				data.fieldType[i] = (char)buf[11];

				data.fieldLength[i] = (buf[16] & 0xFF);
				reclen += data.fieldLength[i];
			}

			dbfStream.read(buf, 0, 1);

			data.fieldData = new ArrayList<String[]>();

			while (dbfStream.read(buf, 0, reclen) == reclen) {

				if (firstRecord) {
					if (10 == buf[0]) {
						for (k = 1; k < reclen; k++) {
							buf[k - 1] = buf[k];
						}
						dbfStream.read(buf, (reclen - 1), 1);
					}
					firstRecord = false;
				}

				attrs = new String[data.fieldCount];
				data.fieldData.add(attrs);

				k = 1;
				for (i = 0; i < data.fieldCount; i++) {
					s.setLength(0);
					for (j = 0; j < data.fieldLength[i]; j++) {
						s.append((char)buf[k++]);
					}
					if (doTrim) {
						attrs[i] = s.toString().trim();
					} else {
						attrs[i] = s.toString();
					}
				}
			}

			if (firstRecord) {
				errmsg = "No records found in file";
				return null;
			}

			return data;

		} catch (IOException ie) {
			if (null != errors) {
				errors.reportError("An I/O error occurred reading '" + fileName + "':\n" + ie);
			}
			return null;

		} catch (Throwable t) {
			AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", t);
			if (null != errors) {
				errors.reportError("An unexpected error occurred reading '" + fileName + "':\n" + t);
			}
			return null;

		} finally {
			try {dbfStream.close();} catch (IOException ie) {}
			if ((null != errmsg) && (null != errors)) {
				errors.reportError("An error occurred reading '" + fileName + "': " + errmsg);
			}
		}
	}


	//=================================================================================================================
	// Line counter for reading text files, can store a delimiter character for parsing, see readAndParseLine().

	public static class LineCounter {

		private int count;
		private char delimiter;

		public void set(int theCount) {
			count = theCount;
		}

		public int get() {
			return count;
		}

		public int increment() {
			return ++count;
		}

		public int decrement() {
			return --count;
		}

		public void setDelimiter(char s) {
			delimiter = s;
		}

		public void setDelimiterForFile(File theFile) {
			String theName = theFile.getName().toLowerCase();
			if (theName.endsWith(".csv")) {
				delimiter = ',';
			} else {
				if (theName.endsWith(".txt")) {
					delimiter = '\t';
				} else {
					if (theName.endsWith(".dat")) {
						delimiter = '|';
					}
				}
			}
		}

		public char getDelimiter() {
			return delimiter;
		}

		public boolean hasDelimiter() {
			return ('\0' != delimiter);
		}

		public void reset() {
			count = 0;
			delimiter = '\0';
		}

		public String toString() {
			return String.valueOf(count);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Wrapper around BufferedReader.readLine(), skip over lines that being with a comment prefix.  Optionally will
	// increment a line counter for each line read including comments.  Optionally will also parse comment lines for
	// possible metadata content, which is of the form "#$name=value" (where "#" may be any recognized comment prefix).
	// If a metadata hashmap is provided name-value pairs are added to that, otherwise ignored.  Return null at EOF.

	public static String readLineSkipComments(BufferedReader reader) throws IOException {
		return readLineSkipComments(reader, null, null);
	}

	public static String readLineSkipComments(BufferedReader reader, LineCounter counter) throws IOException {
		return readLineSkipComments(reader, counter, null);
	}

	public static String readLineSkipComments(BufferedReader reader, LineCounter counter,
			HashMap<String, String> metadata) throws IOException {

		String line, name, value;
		boolean skip = false;
		int len, eq, i;
		char p;

		do {

			line = reader.readLine();
			if (null == line) {
				return null;
			}
			if (null != counter) {
				counter.increment();
			}

			skip = false;
			len = line.length();
			if (len > 0) {
				p = line.charAt(0);
				for (i = 0; i < TEXT_FILE_COMMENT_PREFIX.length; i++) {
					if (p == TEXT_FILE_COMMENT_PREFIX[i]) {
						skip = true;

						if ((null != metadata) && (len > 2) && (TEXT_FILE_METADATA_PREFIX == line.charAt(1))) {
							name = "";
							value = "";
							eq = line.indexOf(TEXT_FILE_METADATA_SEPARATOR);
							if (eq < 0) {
								name = line.substring(2).trim();
							} else {
								name = line.substring(2, eq).trim();
								value = line.substring(eq + 1).trim();
							}
							if (name.length() > 0) {
								metadata.put(name, value);
							}
						}

						break;
					}
				}
			}

		} while (skip);

		return line;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Read a line, skipping comments, and parse into fields.  Field delimiter may be comma, tab, or pipe.  Quotation
	// of field contents is supported.  Leading/trailing whitespace is trimmed from field contents unless quoted, and
	// trailing empty fields are dropped.  May optionally use a line counter and also parse metadata comments, see
	// readLineSkipComments().  Returns null at EOF.

	public static String[] readAndParseLine(BufferedReader reader) throws IOException {
		return readAndParseLine(reader, null, null);
	}

	public static String[] readAndParseLine(BufferedReader reader, LineCounter counter) throws IOException {
		return readAndParseLine(reader, counter, null);
	}

	public static String[] readAndParseLine(BufferedReader reader, LineCounter counter,
			HashMap<String, String> metadata) throws IOException {

		String line = readLineSkipComments(reader, counter, metadata);
		if (null == line) {
			return null;
		}

		ArrayList<String> fields = new ArrayList<String>();

		char sepc = '\0';
		if (null != counter) {
			sepc = counter.getDelimiter();
		}

		char c = '\0', lc;
		boolean quoted = false, started = false;
		int i, trail = 0, blanks = 0;

		StringBuilder field = new StringBuilder();

		for (i = 0; i < line.length(); i++) {

			lc = c;
			c = line.charAt(i);

			// If this character is not escaped, check for delimiter, escape, and quoting.  First check for delimiter,
			// unless quoted.  If the delimiter is not set match any, once the first is found only that one is matched.
			// If a counter is provided store the delimiter there so it applies until the counter is reset.

			if ('\\' != lc) {

				if (!quoted) {

					if (('\0' == sepc) && ((',' == c) || ('\t' == c) || ('|' == c))) {
						sepc = c;
						if (null != counter) {
							counter.setDelimiter(c);
						}
					}

					// If delimiter, commit this field, strip trailing whitespace first.  Then reset for next field.
					// Blank fields are not added until the next non-blank field, so trailing blanks are never added.

					if (c == sepc) {

						if (trail > 0) {
							field.setLength(field.length() - trail);
						}
						if (field.length() > 0) {
							while (blanks > 0) {
								fields.add("");
								blanks--;
							}
							fields.add(field.toString());
							field.setLength(0);
						} else {
							blanks++;
						}

						started = false;
						trail = 0;
						continue;
					}
				}

				// Escape, continue to next character.

				if ('\\' == c) {
					continue;
				}

				// Quote, toggle quote state and continue to next character.

				if ('"' == c) {
					quoted = !quoted;
					continue;
				}
			}

			// Have a character to add, if whitespace strip when leading otherwise add and increment trailing count,
			// else reset trailing count.  Leading/trailing whitespace is stripped regardless of quoting.

			if (Character.isWhitespace(c)) {
				if (!started) {
					continue;
				}
				trail++;
			} else {
				trail = 0;
			}
			started = true;

			field.append(c);
		}

		if (trail > 0) {
			field.setLength(field.length() - trail);
		}
		if (field.length() > 0) {
			while (blanks > 0) {
				fields.add("");
				blanks--;
			}
			fields.add(field.toString());
		}

		return fields.toArray(new String[fields.size()]);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Write a text metadata line, see readLineSkipComments() for details.

	public static void writeTextMetadata(String key, String value, Writer writer) throws IOException {

		writer.write(TEXT_FILE_COMMENT_PREFIX[0]);
		writer.write(TEXT_FILE_METADATA_PREFIX);
		writer.write(key);
		writer.write(TEXT_FILE_METADATA_SEPARATOR);
		writer.write(value);
		writer.write('\n');
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Take a file name string that may include a directory relative off the installation root and convert to a path
	// using the current values of the support directory path properties.  The argument will use FILE_PATH_SEPARATOR
	// as a separator character regardless of platform, this is used for filenames stored in the version list file.

	private static Path getSupportFilePath(String theName) {

		String dirName = null, fileName = theName;

		int sep = theName.indexOf(FILE_PATH_SEPARATOR);
		if (sep >= 0) {
			if (sep > 0) {
				dirName = theName.substring(0, sep);
			}
			fileName = theName.substring(sep + 1);
			if ((0 == fileName.length()) && (null != dirName)) {
				fileName = dirName;
				dirName = null;
			}
		}

		Path thePath;

		if (null != dirName) {
			if (dirName.equals(LIB_DIRECTORY_NAME)) {
				thePath = libDirectoryPath;
			} else {
				if (dirName.equals(DATA_DIRECTORY_NAME)) {
					thePath = dataDirectoryPath;
				} else {
					if (dirName.equals(XML_DIRECTORY_NAME)) {
						thePath = xmlDirectoryPath;
					} else {
						if (dirName.equals(DBASE_DIRECTORY_NAME)) {
							thePath = dbaseDirectoryPath;
						} else {
							if (dirName.equals(HELP_DIRECTORY_NAME)) {
								thePath = helpDirectoryPath;
							} else {
								thePath = workingDirectoryPath.resolve(dirName);
							}
						}
					}
				}
			}
		} else {
			thePath = workingDirectoryPath;
		}

		return thePath.resolve(fileName);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get a version number embedded in a text or XML file (different prefix strings used for comments).  The number
	// is in a comment line with a special format which must appear in the file before the first non-comment line.
	// Returns 0 if no version is found or any error occurs.  A negative version is invalid so returns 0.  If the form
	// taking a BufferedReader is used, the read will proceed from the current stream position which usually would be
	// the start of the file.  One or more comment lines may be skipped but no non-comment lines are skipped.  If this
	// reads past comments without finding a version it will reset the position back to before the first non-comment
	// line past the starting position, if there is one.

	public static int getFileVersion(Path thePath) throws FileNotFoundException {
		return getFileVersion(thePath, FILE_VERSION_COMMENT_PREFIX, FILE_VERSION_PREFIX);
	}

	public static int getXMLFileVersion(Path thePath) throws FileNotFoundException {
		return getFileVersion(thePath, FILE_VERSION_XML_COMMENT_PREFIX, FILE_VERSION_XML_PREFIX);
	}

	private static int getFileVersion(Path thePath, String commentPrefix, String versionPrefix)
			throws FileNotFoundException {
		BufferedReader reader = null;
		try {
			reader = Files.newBufferedReader(thePath);
		} catch (IOException ie) {
			throw new FileNotFoundException(ie.getMessage());
		}
		int version = getFileVersion(reader, commentPrefix, versionPrefix);
		try {reader.close();} catch (IOException ie) {}
		return version;
	}

	public static int getFileVersion(BufferedReader theReader) {
		return getFileVersion(theReader, FILE_VERSION_COMMENT_PREFIX, FILE_VERSION_PREFIX);
	}

	private static int getFileVersion(BufferedReader theReader, String commentPrefix, String versionPrefix) {

		int version = 0;
		String line = null;

		do {

			try {
				theReader.mark(10000);
				line = theReader.readLine();
			} catch (IOException ie) {
				line = null;
				break;
			}

			if ((null != line) && line.startsWith(versionPrefix)) {
				int s = versionPrefix.length(), e = s, n = line.length();
				while ((e < n) && Character.isDigit(line.charAt(e))) e++;
				try {
					version = Integer.parseInt(line.substring(s, e));
					if (version < 0) {
						version = 0;
					}
				} catch (NumberFormatException ne) {
				}
				line = null;
				break;
			}

		} while ((null != line) && line.startsWith(commentPrefix));

		if (null != line) {
			try {
				theReader.reset();
			} catch (IOException ie) {
			}
		}

		return version;
	}
}
