//
//  AntPattern.java
//  TVStudy
//
//  Copyright (c) 2016-2021 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.core;

import gov.fcc.tvstudy.core.data.*;

import java.util.*;
import java.sql.*;
import java.io.*;


//=====================================================================================================================
// Mutable data model for all types of pattern data.  This can contain horizontal, vertical, and receive patterns.
// Horizontal and receive are tabulated by azimuth, vertical by depression angle.  Any type can be "simple" with just
// one horizontal/vertical slice.  Vertical and receive can be "matrix" patterns with multiple horizontal/vertical
// slices, those are tabulated by azimuth for vertical, or by frequency for receive.  Receive also has an additional
// property for pattern gain.

public class AntPattern {

	// Constants for value range checking and input processing.

	public static final double AZIMUTH_MIN = 0.;
	public static final double AZIMUTH_MAX = 359.999;
	public static final double AZIMUTH_ROUND = 1000.;

	public static final double FREQUENCY_MIN = 10.;
	public static final double FREQUENCY_MAX = 5000.;

	public static final double DEPRESSION_MIN = -90.;
	public static final double DEPRESSION_MAX = 90.;
	public static final double DEPRESSION_ROUND = 1000.;

	public static final double FIELD_MIN = 0.001;
	public static final double FIELD_MAX = 1.;
	public static final double FIELD_ROUND = 1000.;

	public static final double FIELD_MAX_CHECK = 0.977;

	public static final int PATTERN_REQUIRED_POINTS = 2;

	public static final double TILT_MIN = -10.;
	public static final double TILT_MAX = 16.;

	public static final double GAIN_MIN = 0.;
	public static final double GAIN_MAX = 60.;

	// Type constants.

	public static final int PATTERN_TYPE_HORIZONTAL = 1;   // Transmit azimuth pattern, always non-matrix.
	public static final int PATTERN_TYPE_VERTICAL = 2;     // Transmit elevation pattern, may be matrix by azimuth.
	public static final int PATTERN_TYPE_RECEIVE = 3;      // Receive azimuth pattern, may be matrix by frequency.

	// Reserved names.

	public static final String NEW_ANTENNA_NAME = "(new)";
	public static final String GENERIC_ANTENNA_NAME = "(generic)";

	// Keys for metadata in imported text files, see importAntenna().

	public static final String TEXT_META_KEY_NAME = "name";
	public static final String TEXT_META_KEY_GAIN = "gain";
	public static final String TEXT_META_KEY_DB = "dbvalues";

	// Properties.  Only one or the other of points or slices is non-null.  Note the key is only used for receive
	// patterns, it is null on others.  Gain is also only used for receive.

	public final String dbID;

	public final int type;
	public Integer key;
	public String name;

	public double gain;

	private ArrayList<AntPoint> points;
	private TreeMap<Double, AntSlice> slices;


	//=================================================================================================================
	// Data class for points.  The angle may be azimuth in degrees true or vertical angle in degrees of depression.

	public static class AntPoint {

		public double angle;
		public double relativeField;


		//-------------------------------------------------------------------------------------------------------------

		public AntPoint(double theAngle, double theRelativeField) {

			angle = theAngle;
			relativeField = theRelativeField;
		}


		//-------------------------------------------------------------------------------------------------------------

		public AntPoint copy() {

			return new AntPoint(angle, relativeField);
		}
	}


	//=================================================================================================================
	// Data class for slices in a matrix pattern.  The pattern may be a vertical pattern, in which case value is an
	// azimuth, or a receive pattern, in which case value is a frequency.

	public static class AntSlice {

		public double value;
		public ArrayList<AntPoint> points;


		//-------------------------------------------------------------------------------------------------------------

		public AntSlice(double theValue) {

			value = theValue;
			points = new ArrayList<AntPoint>();
		}


		//-------------------------------------------------------------------------------------------------------------

		public AntSlice(double theValue, ArrayList<AntPoint> thePoints) {

			value = theValue;
			points = thePoints;
		}


		//-------------------------------------------------------------------------------------------------------------

		public AntSlice copy() {

			ArrayList<AntPoint> newPoints = new ArrayList<AntPoint>();
			for (AntPoint thePoint : points) {
				newPoints.add(thePoint.copy());
			}
			return new AntSlice(value, newPoints);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public AntPattern(String theDbID, int theType, String theName) {

		dbID = theDbID;
		type = theType;
		name = theName;
		points = new ArrayList<AntPoint>();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public AntPattern(String theDbID, int theType, String theName, ArrayList<AntPoint> thePoints) {

		dbID = theDbID;
		type = theType;
		name = theName;
		points = thePoints;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// A horizontal pattern cannot be a matrix.  Note varying argument order to prevent name clash.

	public AntPattern(String theDbID, String theName, int theType, ArrayList<AntSlice> theSlices) {

		dbID = theDbID;
		type = theType;
		name = theName;
		if ((PATTERN_TYPE_HORIZONTAL == type) || (theSlices.size() < 2)) {
			if (!theSlices.isEmpty()) {
				points = theSlices.get(0).points;
			} else {
				points = new ArrayList<AntPoint>();
			}
		} else {
			slices = new TreeMap<Double, AntSlice>();
			for (AntSlice theSlice : theSlices) {
				slices.put(Double.valueOf(theSlice.value), theSlice);
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public AntPattern copy() {

		AntPattern newPattern;

		if (null != points) {

			ArrayList<AntPoint> newPoints = new ArrayList<AntPoint>();
			for (AntPoint thePoint : points) {
				newPoints.add(thePoint.copy());
			}
			newPattern = new AntPattern(dbID, type, name, newPoints);

		} else {

			ArrayList<AntSlice> newSlices = new ArrayList<AntSlice>();
			for (AntSlice theSlice : slices.values()) {
				newSlices.add(theSlice.copy());
			}
			newPattern = new AntPattern(dbID, name, type, newSlices);
		}

		newPattern.key = key;
		newPattern.gain = gain;

		return newPattern;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public AntPattern duplicate() {

		AntPattern newPattern = copy();

		newPattern.key = null;
		newPattern.name = "";

		return newPattern;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isSimple() {

		return (null != points);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The points array and point objects are modified directly by the editor.

	public ArrayList<AntPoint> getPoints() {

		return points;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isMatrix() {

		return (null != slices);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public double minimumValue() {

		if (null != slices) {
			return slices.firstEntry().getValue().value;
		}
		return -1.;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public double maximumValue() {

		if (null != slices) {
			return slices.lastEntry().getValue().value;
		}
		return -1.;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The editor does not use this, it's for the save code.  Slice objects are not modified directly.

	public Collection<AntSlice> getSlices() {

		if (null == slices) {
			return null;
		}
		return slices.values();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// When converting a simple pattern to a matrix, a slice value must be provided for the existing pattern plus a
	// value for the new second slice.  A horizontal pattern cannot be a matrix.  The values must be different.

	public boolean convertToMatrix(Double firstValue, Double secondValue) {
		return convertToMatrix(firstValue, secondValue, null);
	}

	public boolean convertToMatrix(Double firstValue, Double secondValue, ArrayList<AntPoint>newPoints) {

		if ((PATTERN_TYPE_HORIZONTAL == type) || (null == points) || firstValue.equals(secondValue)) {
			return false;
		}

		slices = new TreeMap<Double, AntSlice>();

		AntSlice theSlice = new AntSlice(firstValue.doubleValue(), points);
		slices.put(firstValue, theSlice);

		points = null;

		if (null != newPoints) {
			theSlice = new AntSlice(secondValue.doubleValue(), newPoints);
		} else {
			theSlice = new AntSlice(secondValue.doubleValue());
		}
		slices.put(secondValue, theSlice);

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void convertToSimple() {

		if (null == slices) {
			return;
		}

		points = slices.firstEntry().getValue().points;
		slices = null;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public ArrayList<Double> getSliceValues() {

		if (null == slices) {
			return null;
		}

		ArrayList<Double> theValues = new ArrayList<Double>();
		for (AntSlice theSlice : slices.values()) {
			theValues.add(Double.valueOf(theSlice.value));
		}
		return theValues;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean containsSlice(Double theValue) {

		if (null == slices) {
			return false;
		}

		return slices.containsKey(theValue);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The AntSlice objects are private, this returns the points array for a slice.

	public ArrayList<AntPoint> getSlice(Double theValue) {

		if (null == slices) {
			return null;
		}

		AntSlice theSlice = slices.get(theValue);
		if (null == theSlice) {
			return null;
		}
		return theSlice.points;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Changing the slice value can't be done directly by the editor since the map has to be updated.  If the slice
	// does not exist or the new value already exists this fails.

	public boolean changeSliceValue(Double oldValue, Double newValue) {

		if ((null == slices) || !slices.containsKey(oldValue) || slices.containsKey(newValue)) {
			return false;
		}

		AntSlice theSlice = slices.remove(oldValue);
		theSlice.value = newValue.doubleValue();
		slices.put(newValue, theSlice);
		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a slice and return new points, or null on error.  If the slice already exists just return that.  Adding a
	// slice to a simple pattern may convert to matrix if the pattern is empty; if not, convertToMatrix() must be used.

	public ArrayList<AntPoint> addSlice(Double newValue) {
		return addSlice(newValue, null);
	}

	public ArrayList<AntPoint> addSlice(Double newValue, ArrayList<AntPoint> newPoints) {

		if (null == slices) {
			if ((PATTERN_TYPE_HORIZONTAL == type) || !points.isEmpty()) {
				return null;
			}
			points = null;
			slices = new TreeMap<Double, AntSlice>();
		}

		AntSlice theSlice = slices.get(newValue);
		if (null == theSlice) {
			if (null != newPoints) {
				theSlice = new AntSlice(newValue.doubleValue(), newPoints);
			} else {
				theSlice = new AntSlice(newValue.doubleValue());
			}
			slices.put(newValue, theSlice);
		} else {
			if (null != newPoints) {
				theSlice.points = newPoints;
			}
		}

		return theSlice.points;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Removing the next-to-last slice converts to a simple pattern.

	public boolean removeSlice(Double theValue) {

		if (null == slices) {
			return false;
		}

		boolean result = (null != slices.remove(theValue));
		if (1 == slices.size()) {
			points = slices.firstEntry().getValue().points;
			slices = null;
		}
		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// For a matrix vertical pattern, perform a normalization process to give each pattern slice a maximum value of 1
	// and derive a peak-value horizontal pattern envelope using the slice azimuths.  Return the horizontal pattern,
	// or if normalization is not needed (this is not a vertical matrix or slices are already normalized) return null.

	public AntPattern normalizeVerticalMatrix() {

		if ((PATTERN_TYPE_VERTICAL != type) || (null == slices)) {
			return null;
		}

		ArrayList<AntPoint> newHpat = new ArrayList<AntPoint>();
		AntPoint maxPoint;
		boolean usePat = false;

		for (AntSlice theSlice : slices.values()) {
			maxPoint = null;
			for (AntPoint thePoint : theSlice.points) {
				if ((null == maxPoint) || (thePoint.relativeField > maxPoint.relativeField)) {
					maxPoint = thePoint;
				}
			}
			newHpat.add(new AntPoint(theSlice.value, maxPoint.relativeField));
			if (maxPoint.relativeField < 1.) {
				usePat = true;
			}
		}

		if (usePat) {
			AntPoint hPoint;
			int i = 0;
			for (AntSlice theSlice : slices.values()) {
				hPoint = newHpat.get(i++);
				for (AntPoint vPoint : theSlice.points) {
					vPoint.relativeField /= hPoint.relativeField;
				}
			}
			return new AntPattern(dbID, PATTERN_TYPE_HORIZONTAL, "(matrix derived)", newHpat);
		}

		return null;
	}

		
	//-----------------------------------------------------------------------------------------------------------------
	// Do a point-by-point comparison to another pattern.  This is not an object-equals concept, it is comparing the
	// pattern tabulation data itself regardless of validity, so name and dbID are not considered.

	public boolean equalsPattern(AntPattern other) {

		if (type != other.type) {
			return false;
		}
		if (PATTERN_TYPE_RECEIVE == type) {
			if (gain != other.gain) {
				return false;
			}
		}
		if ((null != points) && (null == other.points)) {
			return false;
		}
		if ((null != slices) && (null == other.slices)) {
			return false;
		}
		if (null != points) {
			if (points.size() != other.points.size()) {
				return false;
			}
			AntPoint aPoint, bPoint;
			for (int i = 0; i < points.size(); i++) {
				aPoint = points.get(i);
				bPoint = other.points.get(i);
				if (aPoint.angle != bPoint.angle) {
					return false;
				}
				if (aPoint.relativeField != bPoint.relativeField) {
					return false;
				}
			}
		} else {
			if (slices.size() != other.slices.size()) {
				return false;
			}
			AntSlice aSlice, bSlice;
			AntPoint aPoint, bPoint;
			Iterator<AntSlice> it = slices.values().iterator();
			Iterator<AntSlice> oit = other.slices.values().iterator();
			while (it.hasNext()) {
				aSlice = it.next();
				bSlice = oit.next();
				if (aSlice.value != bSlice.value) {
					return false;
				}
				if (aSlice.points.size() != bSlice.points.size()) {
					return false;
				}
				for (int i = 0; i < aSlice.points.size(); i++) {
					aPoint = aSlice.points.get(i);
					bPoint = bSlice.points.get(i);
					if (aPoint.angle != bPoint.angle) {
						return false;
					}
					if (aPoint.relativeField != bPoint.relativeField) {
						return false;
					}
				}
			}
		}
		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check validity.

	public boolean isDataValid() {
		return isDataValid(null);
	}

	public boolean isDataValid(ErrorLogger errors) {

		switch (type) {

			case PATTERN_TYPE_HORIZONTAL: {

				if (points.size() < PATTERN_REQUIRED_POINTS) {
					if (null != errors) {
						errors.reportValidationError("Bad azimuth pattern, must have " + PATTERN_REQUIRED_POINTS +
							" or more points");
					}
					return false;
				}

				double lastAz = AZIMUTH_MIN - 1.;

				for (AntPoint thePoint : points) {
					if ((thePoint.angle < AZIMUTH_MIN) || (thePoint.angle > AZIMUTH_MAX)) {
						if (null != errors) {
							errors.reportValidationError("Bad pattern point azimuth, must be " + AZIMUTH_MIN + " to " +
								AZIMUTH_MAX);
						}
						return false;
					}
					if (thePoint.angle <= lastAz) {
						if (null != errors) {
							errors.reportValidationError("Bad azimuth pattern, duplicate or out-of-order points");
						}
						return false;
					}
					lastAz = thePoint.angle;
					if ((thePoint.relativeField < FIELD_MIN) || (thePoint.relativeField > FIELD_MAX)) {
						if (null != errors) {
							errors.reportValidationError("Bad pattern point relative field, must be " + FIELD_MIN +
								" to " + FIELD_MAX);
						}
						return false;
					}
				}

				return true;
			}

			case PATTERN_TYPE_VERTICAL: {

				Collection<AntSlice> theSlices;

				if (null == slices) {

					if (points.size() < PATTERN_REQUIRED_POINTS) {
						if (null != errors) {
							errors.reportValidationError("Bad elevation pattern, must have " +
								PATTERN_REQUIRED_POINTS + " or more points");
						}
						return false;
					}

					theSlices = new ArrayList<AntSlice>();
					theSlices.add(new AntSlice(AZIMUTH_MIN, points));

				} else {

					if (slices.size() < PATTERN_REQUIRED_POINTS) {
						if (null != errors) {
							errors.reportValidationError("Bad matrix elevation pattern, must have " +
								PATTERN_REQUIRED_POINTS + " or more azimuths");
						}
						return false;
					}

					theSlices = slices.values();
				}

				double lastDep = 0.;

				for (AntSlice theSlice : theSlices) {

					if ((theSlice.value < AZIMUTH_MIN) || (theSlice.value > AZIMUTH_MAX)) {
						if (null != errors) {
							errors.reportValidationError("Bad matrix elevation pattern azimuth, must be " +
								AZIMUTH_MIN + " to " + AZIMUTH_MAX);
						}
						return false;
					}
					if (theSlice.points.size() < PATTERN_REQUIRED_POINTS) {
						if (null != errors) {
							errors.reportValidationError("Bad elevation pattern, must have " +
								PATTERN_REQUIRED_POINTS + " or more points");
						}
						return false;
					}

					lastDep = DEPRESSION_MIN - 1.;

					for (AntPoint thePoint : theSlice.points) {
						if ((thePoint.angle < DEPRESSION_MIN) || (thePoint.angle > DEPRESSION_MAX)) {
							if (null != errors) {
								errors.reportValidationError("Bad pattern point vertical angle, must be " +
									DEPRESSION_MIN + " to " + DEPRESSION_MAX);
							}
							return false;
						}
						if (thePoint.angle <= lastDep) {
							if (null != errors) {
								errors.reportValidationError("Bad elevation pattern, duplicate or out-of-order " +
									"points");
							}
							return false;
						}
						lastDep = thePoint.angle;
						if ((thePoint.relativeField < FIELD_MIN) || (thePoint.relativeField > FIELD_MAX)) {
							if (null != errors) {
								errors.reportValidationError("Bad pattern point relative field, must be " + FIELD_MIN +
									" to " + FIELD_MAX);
							}
							return false;
						}
					}
				}
			
				return true;
			}

			// A receive pattern must have a 1.0 value somewhere in the data, but not necessarily in every slice of a
			// matrix.  This is not enforced for horizontal or vertical pattern types because data from CDBS/LMS may
			// not always contain a 1.0 however that data is considered valid anyway.  The 1.0 in those pattern types
			// will be required when the pattern is edited so that is checked by the editor.

			case PATTERN_TYPE_RECEIVE: {

				if ((gain < GAIN_MIN) || (gain > GAIN_MAX)) {
					if (null != errors) {
						errors.reportValidationError("Bad receive antenna gain, must be " + GAIN_MIN + " to " +
							GAIN_MAX);
					}
					return false;
				}

				Collection<AntSlice> theSlices;

				if (null == slices) {

					if (points.size() < PATTERN_REQUIRED_POINTS) {
						if (null != errors) {
							errors.reportValidationError("Bad receive pattern, must have " + PATTERN_REQUIRED_POINTS +
								" or more points");
						}
						return false;
					}

					theSlices = new ArrayList<AntSlice>();
					theSlices.add(new AntSlice(FREQUENCY_MIN, points));

				} else {

					if (slices.size() < PATTERN_REQUIRED_POINTS) {
						if (null != errors) {
							errors.reportValidationError("Bad receive pattern, must have " + PATTERN_REQUIRED_POINTS +
								" or more frequencies");
						}
						return false;
					}

					theSlices = slices.values();
				}

				double lastAz = 0., patmax = 0.;

				for (AntSlice theSlice : theSlices) {

					if ((theSlice.value < FREQUENCY_MIN) || (theSlice.value > FREQUENCY_MAX)) {
						if (null != errors) {
							errors.reportValidationError("Bad receive pattern frequency, must be " + FREQUENCY_MIN +
								" to " + FREQUENCY_MAX);
						}
						return false;
					}
					if (theSlice.points.size() < PATTERN_REQUIRED_POINTS) {
						if (null != errors) {
							errors.reportValidationError("Bad receive pattern, must have " + PATTERN_REQUIRED_POINTS +
								" or more points");
						}
						return false;
					}

					lastAz = AZIMUTH_MIN - 1.;

					for (AntPoint thePoint : theSlice.points) {
						if ((thePoint.angle < AZIMUTH_MIN) || (thePoint.angle > AZIMUTH_MAX)) {
							if (null != errors) {
								errors.reportValidationError("Bad pattern point azimuth, must be " + AZIMUTH_MIN +
									" to " + AZIMUTH_MAX);
							}
							return false;
						}
						if (thePoint.angle <= lastAz) {
							if (null != errors) {
								errors.reportValidationError("Bad receive pattern, duplicate or out-of-order points");
							}
							return false;
						}
						lastAz = thePoint.angle;
						if ((thePoint.relativeField < FIELD_MIN) || (thePoint.relativeField > FIELD_MAX)) {
							if (null != errors) {
								errors.reportValidationError("Bad pattern point relative field, must be " + FIELD_MIN +
									" to " + FIELD_MAX);
							}
							return false;
						}
						if (thePoint.relativeField > patmax) {
							patmax = thePoint.relativeField;
						}
					}
				}

				if (patmax < 1.) {
					if (null != errors) {
						errors.reportValidationError("Bad receive antenna, pattern must have a 1");
					}
					return false;
				}

				return true;
			}
		}

		if (null != errors) {
			errors.reportValidationError("Bad pattern, unknown type");
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Export to a file, format is usually CSV including for matrix patterns.  For a vertical pattern only this can
	// use an LMS-compatible XML format, in that case this will also verify the pattern data meets LMS requirements,
	// if not a warning message is shown that the file may not be suitable for LMS submittal.

	public void export(File theFile, ErrorLogger errors) {
		export(theFile, false, errors);
	}

	public void export(File theFile, boolean useXML, ErrorLogger errors) {

		if (PATTERN_TYPE_VERTICAL != type) {
			useXML = false;
		}

		boolean isMatrix = (null != slices);

		// For a matrix pattern the slices are columns but the file formats are written by row, invert the matrix for
		// output.  The slices don't necessarily have matching row values so the inverted matrix may have null values
		// for empty cells.  The CSV format is always a full matrix but may have empty fields for missing values.

		TreeMap<Double, AntSlice> rowSlices = null;

		if (isMatrix) {

			rowSlices = new TreeMap<Double, AntSlice>();

			Double rowValue;
			AntSlice rowSlice;
			int colIndex = 0, i;

			for (AntSlice slice : slices.values()) {
				for (AntPoint point : slice.points) {
					rowValue = Double.valueOf(point.angle);
					rowSlice = rowSlices.get(rowValue);
					if (null == rowSlice) {
						rowSlice = new AntSlice(point.angle);
						for (i = 0; i < slices.size(); i++) {
							rowSlice.points.add(null);
						}
						rowSlices.put(rowValue, rowSlice);
					}
					rowSlice.points.set(colIndex, new AntPoint(slice.value, point.relativeField));
				}
				colIndex++;
			}
		}

		// Open file, write data.  During XML output check for empty cells which will cause a warning message about
		// LMS compatibility; LMS may not allow missing values, although that is not known for certain as of this
		// writing.  Note XML tags are not typos, the LMS format mis-spells angle as "Agle".  For CSV output include
		// pattern name, and gain for receive patterns, as text metadata lines; see AppCore.

		FileWriter theWriter = null;
		try {
			theWriter = new FileWriter(theFile);
		} catch (IOException ie) {
			if (null != errors) {
				errors.reportError("Could not open the file:\n" + ie.getMessage());
			}
			return;
		}
		BufferedWriter writer = new BufferedWriter(theWriter);

		boolean badMatrix = false;

		try {

			if (useXML) {
				writer.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
					"<elevationPattern xmlns=\"http://mb.fcc.gov/elevationPattern\">\n");
			} else {
				AppCore.writeTextMetadata(TEXT_META_KEY_NAME, name, writer);
				if (PATTERN_TYPE_RECEIVE == type) {
					AppCore.writeTextMetadata(TEXT_META_KEY_GAIN, AppCore.formatDecimal(gain, 2), writer);
				}
			}

			if (isMatrix) {

				if (useXML) {
					for (AntSlice rowSlice : rowSlices.values()) {
						writer.write(String.format(Locale.US,
							"<elevationData>\n<depressionAgle>%.3f</depressionAgle>\n", rowSlice.value));
						for (AntPattern.AntPoint point : rowSlice.points) {
							if (null == point) {
								badMatrix = true;
							} else {
								writer.write(String.format(Locale.US,
									"<fieldValues><azimuth>%.3f</azimuth><fieldValue>%.4f</fieldValue></fieldValues>\n",
									point.angle, point.relativeField));
							}
						}
						writer.write("</elevationData>\n");
					}
				} else {
					for (AntSlice slice : slices.values()) {
						writer.write(String.format(Locale.US, ",%.3f", slice.value));
					}
					writer.write('\n');
					for (AntSlice rowSlice : rowSlices.values()) {
						writer.write(String.format(Locale.US, "%.3f", rowSlice.value));
						for (AntPoint point: rowSlice.points) {
							if (null == point) {
								writer.write(',');
							} else {
								writer.write(String.format(Locale.US, ",%.4f", point.relativeField));
							}
						}
						writer.write('\n');
					}
				}

			} else {

				for (AntPoint point : points) {
					if (useXML) {
						writer.write(String.format(Locale.US,
			"<elevationData>\n<depressionAgle>%.3f</depressionAgle><fieldValue>%.4f</fieldValue>\n</elevationData>\n",
							point.angle, point.relativeField));
					} else {
						writer.write(String.format(Locale.US, "%.3f,%.4f\n", point.angle, point.relativeField));
					}
				}
			}

			if (useXML) {
				writer.write("</elevationPattern>\n");
			}

		} catch (IOException ie) {
			if (null != errors) {
				errors.reportError("Could not write to the file:\n" + ie.getMessage());
			}
		}

		try {writer.close();} catch (IOException ie) {}

		// Do LMS compatibility checks, show warning as needed.  These include restrictions on depression angle range
		// and step size, and for a matrix pattern, no empty cells in the matrix (that was checked above).

		if (useXML && (null != errors) && !errors.hasErrors()) {

			boolean outRange = false, noMinMax = false, badStep = false, badFineStep = false;

			double lastAngle = 0., delta;
			boolean checkDelta, hasMin, hasMax;

			for (AntSlice slice : slices.values()) {
				checkDelta = false;
				hasMin = false;
				hasMax = false;
				for (AntPoint point : slice.points) {
					if ((point.angle < -10.) || (point.angle > 90.)) {
						outRange = true;
					}
					if (-10. == point.angle) {
						hasMin = true;
					}
					if (90. == point.angle) {
						hasMax = true;
					}
					if (checkDelta) {
						delta = point.angle - lastAngle;
						if ((lastAngle < -5.) || (lastAngle >= 10.)) {
							if (delta > 5.) {
								badStep = true;
							}
						} else {
							if (delta > 0.5) {
								badFineStep = true;
							}
						}
					}
					lastAngle = point.angle;
					checkDelta = true;
				}
				if (!hasMin || !hasMax) {
					noMinMax = true;
				}
			}

			if (outRange || noMinMax || badStep || badFineStep || badMatrix) {
				String warnmsg = "The XML file does not comply with LMS requirements:\n";
				if (outRange) {
					warnmsg = warnmsg + "Vertical angles are outside -10 to 90 degrees\n";
				} else {
					if (noMinMax) {
						warnmsg = warnmsg + "Vertical angles do not span -10 to 90 degrees\n";
					}
				}
				if (badStep) {
					warnmsg = warnmsg + "Vertical angles are more than 5 degrees apart\n";
				}
				if (badFineStep) {
					warnmsg = warnmsg + "Vertical angles are more than 0.5 degrees apart between -5 and 10 degrees\n";
				}
				if (badMatrix) {
					warnmsg = warnmsg + "Matrix pattern elevation slices do not have matching vertical angles\n";
				}
				errors.reportWarning(warnmsg);
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save.  This can only be used on a receive antenna, and caller must check validity.  This is factored out so
	// saveReceiveAntenna() can be used with an already-open connection, note the relevant tables must be locked on
	// that connection by the caller, see GeoPointSet.saveGeography().

	public boolean save() {
		return save(null);
	}

	public boolean save(ErrorLogger errors) {

		if (PATTERN_TYPE_RECEIVE != type) {
			if (null != errors) {
				errors.reportError("Antenna save failed, invalid pattern type");
			}
			return false;
		}

		DbConnection db = DbCore.connectDb(dbID, errors);
		if (null == db) {
			return false;
		}

		String errmsg = null;

		try {

			db.update("LOCK TABLES receive_antenna_index WRITE, antenna_key_sequence WRITE, " +
				"receive_pattern WRITE, geography_receive_antenna WRITE, study_geography WRITE, study WRITE");

			errmsg = saveReceiveAntenna(db);

		} catch (SQLException se) {
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		try {
			db.update("UNLOCK TABLES");
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (null != errmsg) {
			if (null != errors) {
				errors.reportError(errmsg);
			}
			return false;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save a receive antenna, confirm type first, then check for reference from running studies via geographies to
	// defer the save; see comments in Geography.save().  Returns null on success, else an error message string.

	public String saveReceiveAntenna(DbConnection db) throws SQLException {

		if (PATTERN_TYPE_RECEIVE != type) {
			return "Antenna save failed, invalid pattern type";
		}

		if (null != key) {
			db.query(
			"SELECT " +
				"COUNT(*) " +
			"FROM " +
				"geography_receive_antenna " +
				"JOIN study_geography USING (geo_key)" +
				"JOIN study USING (study_key) " +
			"WHERE " +
				"(geography_receive_antenna.antenna_key = " + key + ") " +
				"AND (study.study_lock IN (" + Study.LOCK_RUN_EXCL + "," + Study.LOCK_RUN_SHARE + "))");
			if (db.next() && (db.getInt(1) > 0)) {
				return "Changes cannot be saved now, the antenna is in use by a running study";
			}
		}

		// Check the name for uniqueness, append a suffix if needed.  See checkReceiveAntennaName().

		db.query("SELECT antenna_key FROM receive_antenna_index WHERE UPPER(name) = '" +
			db.clean(name.toUpperCase()) + "'");
		if (db.next() && ((null == key) || (db.getInt(1) != key.intValue()))) {
			name = name + " " + String.valueOf(DbCore.NAME_UNIQUE_CHAR) + String.valueOf(key);
		}

		// Generate a key for a new save, else delete old data for existing key.

		if (null == key) {
			db.update("UPDATE antenna_key_sequence SET antenna_key = antenna_key + 1");
			db.query("SELECT antenna_key FROM antenna_key_sequence");
			db.next();
			key = Integer.valueOf(db.getInt(1));
		} else {
			db.update("DELETE FROM receive_antenna_index WHERE antenna_key = " + key);
			db.update("DELETE FROM receive_pattern WHERE antenna_key = " + key);
		}

		// Save data.

		db.update("INSERT INTO receive_antenna_index VALUES (" + key + ",'" + db.clean(name) + "'," + gain + ")");

		StringBuilder query = new StringBuilder("INSERT INTO receive_pattern VALUES");
		int startLength = query.length();
		String sep = " (";

		if (null != points) {

			for (AntPoint thePoint : points) {
				query.append(sep);
				query.append(String.valueOf(key));
				query.append(",-1,");
				query.append(String.valueOf(thePoint.angle));
				query.append(',');
				query.append(String.valueOf(thePoint.relativeField));
				if (query.length() > DbCore.MAX_QUERY_LENGTH) {
					query.append(')');
					db.update(query.toString());
					query.setLength(startLength);
					sep = " (";
				} else {
					sep = "),(";
				}
			}

		} else {

			for (AntSlice theSlice : slices.values()) {
				for (AntPoint thePoint : theSlice.points) {
					query.append(sep);
					query.append(String.valueOf(key));
					query.append(',');
					query.append(String.valueOf(theSlice.value));
					query.append(',');
					query.append(String.valueOf(thePoint.angle));
					query.append(',');
					query.append(String.valueOf(thePoint.relativeField));
					if (query.length() > DbCore.MAX_QUERY_LENGTH) {
						query.append(')');
						db.update(query.toString());
						query.setLength(startLength);
						sep = " (";
					} else {
						sep = "),(";
					}
				}
			}
		}

		if (query.length() > startLength) {
			query.append(')');
			db.update(query.toString());
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check a receive antenna name for validity.  An existing name may be provided for comparison.  If the new name
	// matches the old name ignoring case the change is allowed without any further checks.  A reserved character is
	// used as part of a suffix that may be added to make the name unique (see save()), that character cannot be used
	// in a new name.  But since uniqueness is case-insensitive, changes to an existing name that only affect case will
	// not violate uniqueness and so can be allowed while preserving an existing uniqueness suffix.  If the new name
	// does not match or there is no old name, the new name is checked for length, for the reserved character, and for
	// match to fixed reserved names.  Finally, the name is checked against the database for uniqueness.  Note by the
	// time a save occurs it's possible the name may no longer being unique, in that case the suffix will be added.

	public static boolean checkReceiveAntennaName(String theDbID, String newName) {
		return checkReceiveAntennaName(theDbID, newName, null, null);
	}

	public static boolean checkReceiveAntennaName(String theDbID, String newName, ErrorLogger errors) {
		return checkReceiveAntennaName(theDbID, newName, null, errors);
	}

	public static boolean checkReceiveAntennaName(String theDbID, String newName, String oldName) {
		return checkReceiveAntennaName(theDbID, newName, oldName, null);
	}

	public static boolean checkReceiveAntennaName(String theDbID, String newName, String oldName, ErrorLogger errors) {

		if ((null != oldName) && (oldName.length() > 0)) {
			if (newName.equalsIgnoreCase(oldName)) {
				return true;
			}
		}

		if (0 == newName.length()) {
			if (null != errors) {
				errors.reportWarning("An antenna name must be provided");
			}
			return false;
		}

		if (newName.length() > DbCore.NAME_MAX_LENGTH) {
			if (null != errors) {
				errors.reportWarning("The antenna name cannot be more than " + DbCore.NAME_MAX_LENGTH +
					" characters long");
			}
			return false;
		}

		if (newName.contains(String.valueOf(DbCore.NAME_UNIQUE_CHAR))) {
			if (null != errors) {
				errors.reportWarning("The antenna name cannot contain the character '" + DbCore.NAME_UNIQUE_CHAR + "'");
			}
			return false;
		}

		if (newName.equalsIgnoreCase(NEW_ANTENNA_NAME) || newName.equalsIgnoreCase(GENERIC_ANTENNA_NAME)) {
			if (null != errors) {
				errors.reportWarning("That antenna name cannot be used, please try again");
			}
			return false;
		}

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null != db) {
			boolean exists = false;
			try {
				db.query("SELECT antenna_key FROM receive_antenna_index WHERE UPPER(name) = '" +
					db.clean(newName.toUpperCase()) + "'");
				exists = db.next();
				DbCore.releaseDb(db);
			} catch (SQLException se) {
				DbCore.releaseDb(db);
				DbConnection.reportError(errors, se);
				return false;
			}
			if (exists) {
				if (null != errors) {
					errors.reportWarning("That antenna name is already in use, please try again");
				}
				return false;
			}
		} else {
			return false;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get a list of receive antenna patterns in a database.

	public static ArrayList<KeyedRecord> getReceiveAntennaList(String theDbID) {
		return getReceiveAntennaList(theDbID, null);
	}

	public static ArrayList<KeyedRecord> getReceiveAntennaList(String theDbID, ErrorLogger errors) {

		ArrayList<KeyedRecord> result = null;

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null != db) {
			try {

				result = new ArrayList<KeyedRecord>();

				db.query(
				"SELECT " +
					"antenna_key, " +
					"name " +
				"FROM " +
					 "receive_antenna_index " +
				"ORDER BY 2");

				while (db.next()) {
					result.add(new KeyedRecord(db.getInt(1), db.getString(2)));
				}

				DbCore.releaseDb(db);

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				result = null;
				DbConnection.reportError(errors, se);
			}
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get receive antenna pattern data by key or name.  This is factored out so loadReceiveAntenna() can be used from
	// a context that already has an open database connection.  Returns null on error.  Note a not-found result is
	// reported as an error for a key search but not for a name search.

	public static AntPattern getReceiveAntenna(String theDbID, int theKey) {
		return getReceiveAntenna(theDbID, theKey, null, null);
	}

	public static AntPattern getReceiveAntenna(String theDbID, int theKey, ErrorLogger errors) {
		return getReceiveAntenna(theDbID, theKey, null, errors);
	}

	public static AntPattern getReceiveAntenna(String theDbID, String theName) {
		return getReceiveAntenna(theDbID, 0, theName, null);
	}

	public static AntPattern getReceiveAntenna(String theDbID, String theName, ErrorLogger errors) {
		return getReceiveAntenna(theDbID, 0, theName, errors);
	}

	private static AntPattern getReceiveAntenna(String theDbID, int theKey, String theName, ErrorLogger errors) {

		AntPattern result = null;

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null != db) {

			try {

				result = loadReceiveAntenna(theDbID, db, theKey, theName);

				DbCore.releaseDb(db);

				if ((null == result) && (null != errors) && (null == theName)) {
					errors.reportError("Receive antenna not found for key " + theKey);
				}

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				result = null;
				DbConnection.reportError(errors, se);
			}
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Load a receive antenna by key or by name, returns null if not found.  Errors throw exception.

	public static AntPattern loadReceiveAntenna(String dbID, DbConnection db, int theKey) throws SQLException {
		return loadReceiveAntenna(dbID, db, theKey, null);
	}

	public static AntPattern loadReceiveAntenna(String dbID, DbConnection db, String theName) throws SQLException {
		return loadReceiveAntenna(dbID, db, 0, theName);
	}

	private static AntPattern loadReceiveAntenna(String dbID, DbConnection db, int theKey, String theName)
			throws SQLException {

		AntPattern result = null;

		double theGain = 0.;
		ArrayList<AntSlice> theSlices = null;

		if (null == theName) {
			db.query(
			"SELECT " +
				"antenna_key, " +
				"name, " +
				"gain " +
			"FROM " +
				"receive_antenna_index " +
			"WHERE " +
				"antenna_key = " + theKey);
		} else {
			db.query(
			"SELECT " +
				"antenna_key, " +
				"name, " +
				"gain " +
			"FROM " +
				"receive_antenna_index " +
			"WHERE " +
				"UPPER(name) = '" + db.clean(theName.toUpperCase()) + "'");
		}

		if (db.next()) {

			theKey = db.getInt(1);
			theName = db.getString(2);
			theGain = db.getDouble(3);

			db.query(
			"SELECT " +
				"frequency, " +
				"azimuth, " +
				"relative_field " +
			"FROM " +
				 "receive_pattern " +
			"WHERE " +
				"antenna_key = " + theKey + " " +
			"ORDER BY 1, 2");

			ArrayList<AntPoint> thePoints = null;
			double theFreq, lastFreq = FREQUENCY_MIN - 1.;

			while (db.next()) {

				theFreq = db.getDouble(1);

				if (theFreq != lastFreq) {
					if (null == theSlices) {
						theSlices = new ArrayList<AntSlice>();
					}
					thePoints = new ArrayList<AntPoint>();
					theSlices.add(new AntSlice(theFreq, thePoints));
					lastFreq = theFreq;
				}

				thePoints.add(new AntPoint(db.getDouble(2), db.getDouble(3)));
			}

			if (1 == theSlices.size()) {
				result = new AntPattern(dbID, PATTERN_TYPE_RECEIVE, theName, theSlices.get(0).points);
			} else {
				result = new AntPattern(dbID, theName, PATTERN_TYPE_RECEIVE, theSlices);
			}
			result.key = Integer.valueOf(theKey);
			result.gain = theGain;
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Delete a receive antenna.  First check if it is in use by existing geographies, this just checks as-saved data,
	// the caller has to check active UI directly to find out about usage in unsaved state.

	public static boolean deleteReceiveAntenna(String theDbID, int theKey) {
		return deleteReceiveAntenna(theDbID, theKey, null);
	}

	public static boolean deleteReceiveAntenna(String theDbID, int theKey, ErrorLogger errors) {

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			return false;
		}

		String errmsg = null;
		int refCount = 0, errtyp = AppCore.ERROR_MESSAGE;

		try {

			db.update("LOCK TABLES receive_antenna_index WRITE, receive_pattern WRITE, " +
				"geography_receive_antenna WRITE");

			db.query("SELECT COUNT(*) FROM geography_receive_antenna WHERE antenna_key = " + theKey);
			db.next();
			refCount = db.getInt(1);

			if (0 == refCount) {

				db.update("DELETE FROM receive_antenna_index WHERE antenna_key = " + theKey);
				db.update("DELETE FROM receive_pattern WHERE antenna_key = " + theKey);

			} else {
				errmsg = "The antenna is in use and cannot be deleted";
				errtyp = AppCore.WARNING_MESSAGE;
			}

		} catch (SQLException se) {
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		try {
			db.update("UNLOCK TABLES");
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (null != errmsg) {
			if (null != errors) {
				errors.reportError(errmsg, errtyp);
			}
			return false;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Read a pattern from a text file, typically in CSV format but can be any delimited text format recognized by
	// AppCore.readAndParseLine().  A simple (single-slice) pattern is supported for all pattern types, for vertical
	// and receive patterns matrix (multi-slice) formats are also supported.  This will also parse metadata comments
	// to set pattern properties including name, and gain for receive.  Returns null on error.

	public static AntPattern readFromText(String theDbID, int patternType, File patternFile, ErrorLogger errors) {

		// Set min/max and rounding values per pattern type for processing column and row values.

		double mincol, maxcol, lcol, colrnd, minrow, maxrow, lrownew, rowrnd;
		String coltypeuc, coltypelc, rowtypeuc, rowtypelc;

		switch (patternType) {

			case PATTERN_TYPE_HORIZONTAL: {
				mincol = 0.;
				maxcol = 0.;
				lcol = 0.;
				colrnd = 0.;
				coltypeuc = "";
				coltypelc = "";
				minrow = AZIMUTH_MIN;
				maxrow = AZIMUTH_MAX;
				lrownew = AZIMUTH_MIN - 1.;
				rowrnd = AZIMUTH_ROUND;
				rowtypeuc = "Azimuth";
				rowtypelc = "azimuth";
				break;
			}

			case PATTERN_TYPE_VERTICAL: {
				mincol = AZIMUTH_MIN;
				maxcol = AZIMUTH_MAX;
				lcol = AZIMUTH_MIN - 1.;
				colrnd = AZIMUTH_ROUND;
				coltypeuc = "Azimuth";
				coltypelc = "azimuth";
				minrow = DEPRESSION_MIN;
				maxrow = DEPRESSION_MAX;
				lrownew = DEPRESSION_MIN - 1.;
				rowrnd = DEPRESSION_ROUND;
				rowtypeuc = "Vertical angle";
				rowtypelc = "vertical angle";
				break;
			}

			case PATTERN_TYPE_RECEIVE: {
				mincol = FREQUENCY_MIN;
				maxcol = FREQUENCY_MAX;
				lcol = FREQUENCY_MIN - 1.;
				colrnd = 0.;
				coltypeuc = "Frequency";
				coltypelc = "frequency";
				minrow = AZIMUTH_MIN;
				maxrow = AZIMUTH_MAX;
				lrownew = AZIMUTH_MIN - 1.;
				rowrnd = AZIMUTH_ROUND;
				rowtypeuc = "Azimuth";
				rowtypelc = "azimuth";
				break;
			}

			default: {
				if (null != errors) {
					errors.reportError("Unknown or unsupported pattern type");
				}
				return null;
			}
		}

		// Open file, read the first data line and check to auto-detect the file format.  For vertical or receive
		// patterns this may be a simple or matrix pattern, a horizontal pattern can only be a simple pattern.

		// If the first line of the file contains more than 3 fields it is a matrix pattern (multiple slices) in full-
		// grid format.  The first line provides all slice (column) values, with the first field ignored, it should be
		// blank.  Those values are azimuths for a vertical pattern or frequencies for a receive pattern.  Subsequent
		// lines start with a row value, which is depression angle for vertical or azimuth for receive, then pattern
		// values for each slice in sequence.  There may be empty fields to represent "holes" in the matrix.  If an
		// entire row or column is empty it will be ignored.

		// If the first line contains exactly 3 fields, it may be a full-grid matrix with only two slices, or it may
		// be a matrix in triples format.  If the first field is blank it's full-grid, otherwise it's triples format.
		// In triples format, slices are tabulated sequentially with each line having both column and row values, that
		// is azimuth, depression angle, pattern value for vertical; or frequency, azimuth, pattern value for receive.

		// If the first line contains exactly 2 fields, it may be a simple pattern or a full-grid matrix with only one
		// slice.  The latter is not supported by the model and is silently converted to a simple pattern.  If the
		// first field is blank it is full-grid, otherwise it is simple.

		// In all formats pattern values may be relative field or dB, the latter will be converted to relative field.

		// Note in all formats, azimuths, depression angles, and frequencies must always occur in sequence.

		FileReader theReader = null;
		try {
			theReader = new FileReader(patternFile);
		} catch (IOException ie) {
			if (null != errors) {
				errors.reportError("Could not open file:\n" + ie.getMessage());
			}
			return null;
		}
		BufferedReader reader = new BufferedReader(theReader);

		AppCore.LineCounter lineNumber = new AppCore.LineCounter();
		lineNumber.setDelimiterForFile(patternFile);

		ArrayList<AntSlice> newSlices = new ArrayList<AntSlice>();

		HashMap<String, String> metadata = new HashMap<String, String>();

		boolean error = false;
		String errmsg = "";

		String theLine;
		String[] tokens;

		final int SIMPLE = 1;
		final int GRID = 2;
		final int TRIPLES = 3;

		AntSlice newSlice = null;
		ArrayList<AntPoint> newPoints = null;

		double row, lrow = lrownew, col, pat, patmin = 999., patmax = -999.;
		int format = SIMPLE, ncol = 0, icol, rowi;

		try {

			tokens = AppCore.readAndParseLine(reader, lineNumber, metadata);
			if (null == tokens) {
				error = true;
				errmsg = "Could not read first data line from file";
			} else {
				if (tokens.length < 2) {
					error = true;
					errmsg = "Bad data format, missing values";
				} else {
					if (tokens.length > 3) {
						format = GRID;
					} else {
						if (0 == tokens[0].length()) {
							format = GRID;
						} else {
							if (3 == tokens.length) {
								format = TRIPLES;
							}
						}
					}
				}
			}

			if (!error && (PATTERN_TYPE_HORIZONTAL == patternType) && (SIMPLE != format)) {
				error = true;
				errmsg = "Bad data format, too many values";
			}

			// Simple format is stored in a temporary slice object to generalize the code below.  In full-grid matrix
			// format, parse slice values from the first line and create slices, then read the first row line.  

			if (!error) {

				switch (format) {

					case SIMPLE: {
						newSlice = new AntSlice(0.);
						newSlices.add(newSlice);
						newPoints = newSlice.points;
						break;
					}

					case GRID: {
						ncol = tokens.length - 1;
						for (icol = 0; icol < ncol; icol++) {
							try {
								col = Double.parseDouble(tokens[icol + 1]);
							} catch (NumberFormatException nfe) {
								error = true;
								errmsg = "Bad number format for " + coltypelc;
								break;
							}
							if (colrnd > 0.) {
								col = Math.rint(col * colrnd) / colrnd;
							}
							if ((col < mincol) || (col > maxcol)) {
								error = true;
								errmsg = coltypeuc + " out of range";
								break;
							}
							if (col <= lcol) {
								error = true;
								errmsg = coltypeuc + " out of sequence or duplicated";
								break;
							}
							lcol = col;
							newSlice = new AntSlice(col);
							newSlices.add(newSlice);
						}
						if (!error) {
							tokens = AppCore.readAndParseLine(reader, lineNumber, metadata);
						}
						break;
					}
				}
			}

			// Parse and continue reading data lines, check field count.  In full-grid matrix format only the first
			// field is required; fields may be blank for missing pattern values and the parser drops trailing blank
			// fields, and a row may be entirely empty.

			while (!error && (null != tokens)) {

				switch (format) {
					case SIMPLE:
					default: {
						if (2 != tokens.length) {
							error = true;
						}
						break;
					}
					case GRID: {
						if ((tokens.length < 1) || (tokens.length > (ncol + 1))) {
							error = true;
						}
						break;
					}
					case TRIPLES: {
						if (3 != tokens.length) {
							error = true;
						}
						break;
					}
				}
				if (error) {
					errmsg = "Bad data format, missing or extra values";
					break;
				}

				// In triples format, parse the column value from the first field and create a new slice as needed.
				// Check the number of rows in the previous slice, if any, must be the pattern minimum.

				rowi = 0;
				if (TRIPLES == format) {
					rowi = 1;
					try {
						col = Double.parseDouble(tokens[0]);
					} catch (NumberFormatException nfe) {
						error = true;
						errmsg = "Bad number format for " + coltypelc;
						break;
					}
					if (colrnd > 0.) {
						col = Math.rint(col * colrnd) / colrnd;
					}
					if ((col < mincol) || (col > maxcol)) {
						error = true;
						errmsg = coltypeuc + " out of range";
						break;
					}
					if (col != lcol) {
						if (col < lcol) {
							error = true;
							errmsg = coltypeuc + " out of sequence";
							break;
						}
						newSlice = new AntSlice(col);
						newSlices.add(newSlice);
						newPoints = newSlice.points;
						lcol = col;
						lrow = lrownew;
					}
				}

				// Parse the row value.

				try {
					row = Double.parseDouble(tokens[rowi]);
				} catch (NumberFormatException nfe) {
					error = true;
					errmsg = "Bad number format for " + rowtypelc;
					break;
				}
				row = Math.rint(row * rowrnd) / rowrnd;
				if ((row < minrow) || (row > maxrow)) {
					error = true;
					errmsg = rowtypeuc + " out of range";
					break;
				}
				if (row <= lrow) {
					error = true;
					errmsg = rowtypeuc + " out of sequence or duplicated";
					break;
				}
				lrow = row;

				// Parse the pattern value or values, in full-grid format skipping empty fields.  Note the min and max
				// are tracked here but the values are not rounded or range-checked.  These may be in relative field
				// or dB which is determined from the min/max range so that can't be done until all data is read, at
				// which time all values will be converted and/or rounded as needed.

				if (GRID == format) {

					for (icol = 0; icol < (tokens.length - 1); icol++) {
						if (0 == tokens[icol + 1].length()) {
							continue;
						}
						try {
							pat = Double.parseDouble(tokens[icol + 1]);
						} catch (NumberFormatException nfe) {
							error = true;
							errmsg = "Bad number format for field";
							break;
						}
						if (pat < patmin) {
							patmin = pat;
						}
						if (pat > patmax) {
							patmax = pat;
						}
						newSlices.get(icol).points.add(new AntPoint(row, pat));
					}
					if (error) {
						break;
					}

				} else {

					try {
						pat = Double.parseDouble(tokens[rowi + 1]);
					} catch (NumberFormatException nfe) {
						error = true;
						errmsg = "Bad number format for field";
						break;
					}
					if (pat < patmin) {
						patmin = pat;
					}
					if (pat > patmax) {
						patmax = pat;
					}
					newPoints.add(new AntPoint(row, pat));
				}

				// Read the next line, continue.

				tokens = AppCore.readAndParseLine(reader, lineNumber, metadata);
			}

			// Check for minimum point count in all slices.  For a matrix remove any entirely-empty slices, those may
			// arise due to empty columns in full-grid format, and make sure that does not result in no slices at all.
			// There may be only one slice in a matrix, that case is silently converted to a simple pattern.

			if (!error) {

				if (SIMPLE == format) {

					if (newPoints.size() < PATTERN_REQUIRED_POINTS) {
						error = true;
						errmsg = "Not enough points in pattern";
					}

				} else {

					Iterator<AntSlice> it = newSlices.iterator();
					while (it.hasNext()) {
						newSlice = it.next();
						if (newSlice.points.isEmpty()) {
							it.remove();
						} else {
							if (newSlice.points.size() < PATTERN_REQUIRED_POINTS) {
								error = true;
								errmsg = "Not enough points in pattern, for " + coltypelc + " " +
									String.valueOf(newSlice.value);
								break;
							}
						}
					}

					if (!error && (0 == newSlices.size())) {
						error = true;
						errmsg = "No pattern slices found in file";
					}

					if (1 == newSlices.size()) {
						format = SIMPLE;
						newPoints = newSlices.get(0).points;
					}
				}

				if (error) {
					lineNumber.reset();
				}
			}

		} catch (IOException ie) {
			error = true;
			errmsg = "Could not read from file:\n" + ie.getMessage();
			lineNumber.reset();
		}

		try {reader.close();} catch (IOException ie) {}

		// Determine if values are dB or relative field.  A metadata key may force dB.  Otherwise if all values are
		// between 0. and 1. inclusive data is relative field, else dB.  Values in dB will be normalized so the result
		// will always have a 1 peak value.  For relative field check the actual peak, if less than 0.5 data is bad.

		boolean isDb = false;
		if (metadata.containsKey(TEXT_META_KEY_DB)) {
			isDb = true;
		} else {
			isDb = ((patmin < 0.) || (patmax > 1.));
		}

		if (!error && !isDb && (patmax < 0.5)) {
			error = true;
			errmsg = "Pattern maximum value is too small";
			lineNumber.reset();
		}
	
		// Report fatal error if needed.

		if (error) {
			if (null != errors) {
				if (lineNumber.get() > 0) {
					errmsg = errmsg + " at line " + lineNumber;
				}
				errors.reportError(errmsg);
			}
			return null;
		}

		// Convert and round as needed.  Report if data converted from dB, relative field is typical and a small error
		// could incorrectly trigger dB.  If not dB, show a warning if the peak value is not (very nearly) 1.

		for (AntSlice theSlice : newSlices) {
			for (AntPoint thePoint : theSlice.points) {
				pat = thePoint.relativeField;
				if (isDb) {
					pat = Math.pow(10., ((pat - patmax) / 20.));
				}
				pat = Math.rint(pat * FIELD_ROUND) / FIELD_ROUND;
				if (pat < FIELD_MIN) {
					pat = FIELD_MIN;
				}
				thePoint.relativeField = pat;
			}
		}

		if (null != errors) {
			if (isDb) {
				errors.reportMessage("Pattern values converted from dB to relative field");
			} else {
				if (patmax < FIELD_MAX_CHECK) {
					errors.reportWarning("Pattern does not have a 1");
				}
			}
		}

		// Create the pattern, check for metadata providing name, and gain for a receive pattern.  A receive pattern
		// in dB will have gain set to the peak dB value, but that may be overridden by an explicit metadata value.

		String theName = metadata.get(TEXT_META_KEY_NAME);
		if (null == theName) {
			theName = "";
		}

		AntPattern thePattern = null;
		if (SIMPLE == format) {
			thePattern = new AntPattern(theDbID, patternType, theName, newPoints);
		} else {
			thePattern = new AntPattern(theDbID, theName, patternType, newSlices);
		}

		if (PATTERN_TYPE_RECEIVE == patternType) {
			if (isDb) {
				thePattern.gain = patmax;
			}
			String theVal = metadata.get(TEXT_META_KEY_GAIN);
			if (null != theVal) {
				double g = GAIN_MIN - 1.;
				try {
					g = Double.parseDouble(theVal);
				} catch (NumberFormatException nfe) {
				}
				if ((g < GAIN_MIN) || (g > GAIN_MAX)) {
					if (null != errors) {
						errors.reportWarning("Bad receive pattern gain value, ignored");
					}
				} else {
					thePattern.gain = g;
				}
			}
		}

		return thePattern;
	}
}
