//
//  map.h
//  TVStudy
//
//  Copyright (c) 2018 Hammett & Edison, Inc.  All rights reserved.


#define CNTRY_USA  1   // Country keys records - USA.
#define CNTRY_CAN  2   // Canada.
#define CNTRY_MEX  3   // Mexico.

#define MAX_COUNTRY  3   // Number of possible countries.  WARNING!  The code uses country keys as index values in
                         // arrays, after subtracting 1.  The key range must be 1 through MAX_COUNTRY without gaps.

#define GEO_TYPE_CIRCLE   2   // Geography types.
#define GEO_TYPE_BOX      3
#define GEO_TYPE_POLYGON  4
#define GEO_TYPE_SECTORS  5

#define RENDER_MAP_SCALE  500000.   // Map scale for rendering contours for map file output.

typedef struct gpt {    // General-purpose geographic data structure, a list of coordinates.
	struct gpt *next;   // For linked lists (often not used).
	double *ptLat;      // Arrays of point coordinates, positive north and west, NAD83.
	double *ptLon;
	double xlts;        // Bounding box around points.
	double xlne;
	double xltn;
	double xlnw;
	int nPts;           // Current number of points stored, never greater than maxPts.
	int maxPts;         // Allocated size of the arrays, this may be 0 in which case ptLat and ptLon are NULL.
} GEOPOINTS;

typedef struct {         // Data for a signal level contour.
	double *distance;    // Projected contour distances in km for evenly-spaced radials.
	GEOPOINTS *points;   // Rendered contour points, may be NULL.
	double latitude;     // Contour center point coordinates, NAD83.
	double longitude;
	int mode;            // Contour projection mode.
	int count;           // Number of contour projection radials.
} CONTOUR;

typedef struct geo {             // Data for a service area geography.
	struct geo *next;            // For cache list, see map.c.
	union {
		double radius;           // Radius for circle, km.
		double width;            // Width for box, km.
		double *sectorAzimuth;   // Azimuths for sectors.
		GEOPOINTS *polygon;      // Data for polygon.
	} a;
	union {
		double height;           // Height for box, km.
		double *sectorRadius;    // Radii for sectors.
	} b;
	GEOPOINTS *points;           // Rendered boundary.
	double latitude;             // Center for circle, box, or sectors, reference point for polygon.  NAD83.
	double longitude;
	int geoKey;                  // Geography key.
	int type;                    // Geography type, GEO_TYPE_*.
	int nSectors;                // Number of sectors for sectors.
} GEOGRAPHY;

int find_country(double lat, double lon);
int get_border_distances(double lat, double lon, double *borderDist, double kmPerDegree);
GEOPOINTS *render_contour(CONTOUR *contour, double kmPerDegree);
GEOPOINTS *render_geography(GEOGRAPHY *geo, double kmPerDegree);
int inside_geography(double plat, double plon, GEOGRAPHY *geo, double kmPerDegree);
double interp_cont(double lookup, CONTOUR *contour);
double interp(double lookup, double *table, int count);
double interp_min(double lookup, double *table, int count, double minVal, int minMode);
CONTOUR *contour_alloc(double latitude, double longitude, int mode, int count);
void contour_free(CONTOUR *contour);
GEOGRAPHY *geography_alloc(int geoKey, int type, double lat, double lon, int count);
void geography_free(GEOGRAPHY *geo);
GEOPOINTS *geopoints_alloc(int np);
void geopoints_addpoint(GEOPOINTS *pts, double lat, double lon);
void geopoints_free(GEOPOINTS *pts);
int inside_poly(double plat, double plon, GEOPOINTS *pts);
