//
//  ixstudy.c
//  TVStudy
//
//  Copyright (c) 2012-2024 Hammett & Edison, Inc.  All rights reserved.


// Run an interference-check ("IX check") study, a special study type for iteratively testing proposal records.


#include "tvstudy.h"


//---------------------------------------------------------------------------------------------------------------------

static int do_run_ix_study();
static void ix_source_descrip(SOURCE *source, SOURCE *dsource, FILE *repFile, int doLabel);
static char *ix_source_label(SOURCE *source);
static int check_haat(SOURCE *target, FILE *repFile);
static int check_base_coverage(SOURCE *target, SOURCE *targetBase, FILE *repFile);
static int check_dts_coverage(SOURCE *target, FILE *repFile);
static int check_border_distance(SOURCE *target, FILE *repFile);
static int check_monitor_station(SOURCE *target, FILE *repFile);
static int check_va_quiet_zone(SOURCE *target, FILE *repFile);
static int check_table_mountain(SOURCE *target, FILE *repFile);
static int check_land_mobile(SOURCE *target, FILE *repFile);
static int check_offshore_radio(SOURCE *target, FILE *repFile);
static int check_top_markets(SOURCE *target, FILE *repFile);

// Private globals.

static FILE *ixReportFile = NULL;
static UNDESIRED *uBefore = NULL;

// See check_dts_coverage().

#define DTS_AUTH_CONT_ADJUST -0.5
#define DTS_AUTH_COORD_MATCH 3.


//---------------------------------------------------------------------------------------------------------------------
// Run an interference-check study, this is a different processing sequence than normal studies and it writes a
// separate report output file.  It traverses the scenario pair list and runs the pairs in sequence.  Normal output
// files may also be written by run_scenario(), but those are not required.

// Return is <0 for major error, >0 for minor error, 0 for no error.

int run_ix_study() {

	if (!StudyKey || (STUDY_TYPE_TV_IX != StudyType)) {
		log_error("run_ix_study() called with no study open, or wrong study type");
		return 1;
	}

	int err = do_run_ix_study();

	if (uBefore) {
		mem_free(uBefore);
		uBefore = NULL;
	}

	if (ixReportFile) {
		file_close(ixReportFile);
		ixReportFile = NULL;
	}

	return err;
}

static int do_run_ix_study() {

	int err = 0;

	// Identify the study target, and perform auxiliary checks.  The target source, and possibly a baseline record for
	// the target, are expected to be in the first scenario in the study.  That scenario is also the "before" for
	// scenario pairs that analyze interference to the proposal.  Load and run that scenario, then identify the target
	// and possibly baseline.  Both target and baseline must be desireds, no other records should exist.  The target is
	// identified by an attribute.

	err = run_scenario(1);
	if (err) return err;

	SOURCE *source = Sources, *target = NULL, *targetBase = NULL;
	int sourceIndex;
	for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
		if (source->inScenario) {
			if (source->isDesired) {
				if (get_source_attribute(source, ATTR_KEY_IS_PROPOSAL)) {
					if (target) {
						target = NULL;
						break;
					}
					target = source;
				} else {
					if (targetBase) {
						target = NULL;
						break;
					}
					targetBase = source;
				}
			} else {
				target = NULL;
				break;
			}
		}
	}

	if (!target || (RECORD_TYPE_TV != target->recordType)) {
		log_error("Could not identify proposal record in IX check study");
		return 1;
	}

	// Verify the target and any baseline are using FCC curves contours to define service areas.  All of the check_*()
	// functions assume those contours already exist on the sources.

	SOURCE *sources;

	if (target->isParent) {
		sources = target->dtsSources;
	} else {
		sources = target;
		target->next = NULL;
	}
	for (source = sources; source; source = source->next) {
		if (!source->contour || (SERVAREA_CONTOUR_FCC != source->contour->mode)) {
			log_error("Proposal records must use FCC curves service contours for IX check study");
			return 1;
		}
	}

	if (targetBase) {
		if (targetBase->isParent) {
			sources = targetBase->dtsSources;
		} else {
			sources = targetBase;
			targetBase->next = NULL;
		}
		for (source = sources; source; source = source->next) {
			if (!source->contour || (SERVAREA_CONTOUR_FCC != source->contour->mode)) {
				log_error("Proposal records must use FCC curves service contours for IX check study");
				return 1;
			}
		}
	}

	// Open the report file, write the report preamble.

	ixReportFile = open_sum_file(TV_IX_FILE_NAME);
	if (!ixReportFile) return 1;

	write_report_preamble(REPORT_FILE_SUMMARY, ixReportFile, 0, 0);

	// Run the various special checks.

	err = check_haat(target, ixReportFile);
	if (err) return err;

	err = check_base_coverage(target, targetBase, ixReportFile);
	if (err) return err;

	err = check_dts_coverage(target, ixReportFile);
	if (err) return err;

	err = check_border_distance(target, ixReportFile);
	if (err) return err;

	err = check_monitor_station(target, ixReportFile);
	if (err) return err;

	err = check_va_quiet_zone(target, ixReportFile);
	if (err) return err;

	err = check_table_mountain(target, ixReportFile);
	if (err) return err;

	err = check_land_mobile(target, ixReportFile);
	if (err) return err;

	err = check_offshore_radio(target, ixReportFile);
	if (err) return err;

	err = check_top_markets(target, ixReportFile);
	if (err) return err;

	// Report cell size and profile resolution and the interference limit percentages.  For an LPTV or Class A target
	// if the cell size is greater than 1 km include a warning that the results may not be accepted.

	char mesg[MAX_STRING];

	snprintf(mesg, MAX_STRING, "Study cell size: %.2f km", Params.CellSize);
	status_message(STATUS_KEY_REPORT, "");
	status_message(STATUS_KEY_REPORT, mesg);
	fprintf(ixReportFile, "\n%s\n", mesg);
	if ((SERV_TV != target->service) && (Params.CellSize > 1.)) {
		lcpystr(mesg, "Warning: Study results at this cell size for Class A/LPTV proposal may not be accepted",
			MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		fprintf(ixReportFile, "%s\n", mesg);
	}

	snprintf(mesg, MAX_STRING, "Profile point spacing: %.2f km", (1. / Params.TerrPathPpk[target->countryKey - 1]));
	status_message(STATUS_KEY_REPORT, mesg);
	status_message(STATUS_KEY_REPORT, "");
	fprintf(ixReportFile, "%s\n\n", mesg);

	snprintf(mesg, MAX_STRING, "Maximum new IX to full-service and Class A: %.2f%%", Params.IxCheckLimitPercent);
	status_message(STATUS_KEY_REPORT, mesg);
	fprintf(ixReportFile, "%s\n", mesg);

	snprintf(mesg, MAX_STRING, "Maximum new IX to LPTV: %.2f%%", Params.IxCheckLimitPercentLPTV);
	status_message(STATUS_KEY_REPORT, mesg);
	status_message(STATUS_KEY_REPORT, "");
	fprintf(ixReportFile, "%s\n\n", mesg);

	// Run the study, loop over scenario pairs, run before and after scenarios as needed, analyze and report.

	static char *countryLabel[MAX_COUNTRY + 1] = {"", "  (in U.S.)", "  (in Canada)", "  (in Mexico)"};

	SOURCE *usource;
	UNDESIRED *undesireds;
	int i, isIXPair, hasAPP, iAPP, countryKey, countryIndex, ubCount = 0, ubIndex, undesiredCount, undesiredIndex,
		showLbl, cLblIndex, showHdr, skip, fails, failed = 0, reportWorst = 0, desSourceKey = 0, firstMX = 1;
	double failLimit, pcta, pctp, worstCase;
	DES_TOTAL *dtotBefore, *dtotAfter;
	UND_TOTAL *utotBefore, *utotAfter, *ttotBefore, *ttotAfter;
	char *scenID, worstMesg[MAX_STRING];

	SCENARIO_PAIR *thePair = ScenarioPairList;

	while (thePair) {

		source = thePair->sourceB;
		isIXPair = (source != target);

		// By convention the pair name contains a # followed by a numerical identifier for the pairing, extract that
		// for use in reporting.  If # is not found use the entire pair name as the ID.

		scenID = thePair->name;
		for (i = strlen(scenID) - 2; i > 0; i--) {
			if ('#' == scenID[i]) {
				scenID += i + 1;
				break;
			}
		}

		// Set up to report the worst-case interference for this desired if there are no actual failures.

		if (source->sourceKey != desSourceKey) {
			if (reportWorst) {
				status_message(STATUS_KEY_REPORT, worstMesg);
			}
			desSourceKey = source->sourceKey;
			reportWorst = OutputFlags[IXCHK_REPORT_WORST];
			if (reportWorst) {
				if (isIXPair) {
					snprintf(worstMesg, MAX_STRING, "Proposal causes no interference to %s %s %s", source->callSign,
						source->status, source->fileNumber);
				} else {
					snprintf(worstMesg, MAX_STRING, "Proposal receives no interference");
				}
			}
			worstCase = 0.;
		}

		// The pending-files feature may be used to delete files for scenario pairs that pass, if the output flag is
		// set to only output for failed scenarios.  Due to the code design in analyze_points() it is not possible to
		// defer file output until after the pass/fail test result is known.  So the files are always created during
		// the analysis, then once pass/fail is determined the files will be deleted or committed as appropriate.
		// This is cleared after the pair analysis so has to be re-set for every new scenario pair.

		UsePendingFiles = OutputFlags[IXCHK_DEL_PASS];

		// The "before" scenario may be shared between pairs for the cases that examine interference received by the
		// target station.  Results from shared scenarios are updated in all pair structures when the scenario is run
		// so it does not need to be re-run, see run_scenario().  In the coverage cases the "before" does not provide
		// any undesired information as it contains only the desired station.  In other pairs the scenarios should not
		// be shared so both are run regardless to get the undesired lists.  The "before" undesired list is copied
		// since it will be cleared by running the "after".  Note this assumes the two desired sources in the pair are
		// the same however it doesn't fail if they aren't, although the report may be incomplete in that case.

		if (isIXPair || !thePair->didStudyA) {

			err = run_scenario(thePair->scenarioKeyA);
			if (err) return err;

			if (isIXPair) {
				ubCount = thePair->sourceA->undesiredCount;
				if (ubCount) {
					uBefore = (UNDESIRED *)mem_alloc(ubCount * sizeof(UNDESIRED));
					memcpy(uBefore, thePair->sourceA->undesireds, (ubCount * sizeof(UNDESIRED)));
				}
			}
		}

		// During the after scenario run, activate the IX margin feature in analyze_points(), see run_scenario.c.

		if (isIXPair) {
			IXMarginSourceKey = target->sourceKey;
		}

		err = run_scenario(thePair->scenarioKeyB);
		if (err) return err;

		undesireds = source->undesireds;
		undesiredCount = source->undesiredCount;

		dtotBefore = thePair->totalsA;
		dtotAfter = thePair->totalsB;

		// For an IX pairing, locate the target totals for "before" and "after".  If there is a "before" undesireds
		// list search it for the target's baseline record to get the totals for that source as undesired.  In any case
		// these loops also validate the source index for both lists so that does not have to be done again later.

		ttotBefore = NULL;
		ttotAfter = NULL;

		for (ubIndex = 0; ubIndex < ubCount; ubIndex++) {
			usource = SourceKeyIndex[uBefore[ubIndex].sourceKey];
			if (!usource) {
				log_error("Source structure index is corrupted");
				exit(1);
			}
			if (isIXPair && (usource == targetBase)) {
				ttotBefore = uBefore[ubIndex].totals;
			}
		}

		for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {
			usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
			if (!usource) {
				log_error("Source structure index is corrupted");
				exit(1);
			}
			if (isIXPair && (usource == target)) {
				ttotAfter = undesireds[undesiredIndex].totals;
			}
		}

		// Start the report.  In an IX pairing if the target causes no interference either "before" or "after" there is
		// nothing more to report.  Otherwise check the new interference percentage against limits, report any failures
		// to the front-end app by inline message as well.  An interference failure is based only on coverage in the
		// desired station's country, interference in other countries is reported but never considered a failure.  A
		// coverage pair (interference to the target) is always reported.

		fputs(
		"\n--------------------------------------------------------------------------------------------------------\n",
			ixReportFile);

		if (isIXPair) {

			fprintf(ixReportFile, "Interference to %s %s %s scenario %s\n", source->callSign, source->status,
				source->fileNumber, scenID);
			skip = 1;
			for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {
				if (ttotBefore && (ttotBefore[countryIndex].ixArea > 0.)) {
					skip = 0;
					break;
				}
				if (ttotAfter && (ttotAfter[countryIndex].ixArea > 0.)) {
					skip = 0;
					break;
				}
			}

		} else {

			fprintf(ixReportFile, "Interference to proposal %s %s %s scenario %s\n", target->callSign, target->status,
				target->fileNumber, scenID);
			skip = 0;
		}

		fails = 0;

		if (skip) {

			fprintf(ixReportFile, "Proposal causes no interference.\n");

		} else {

			countryIndex = source->countryKey - 1;
			if (SERV_TVLP == source->service) {
				failLimit = Params.IxCheckLimitPercentLPTV;
			} else {
				failLimit = Params.IxCheckLimitPercent;
			}
			fails = (thePair->popPercent[countryIndex] > failLimit);

			// Report failures.  For the IX pairs if the desired is an APP or AMD it is reported as an MX failure, else
			// an IX failure.  For the MX check pairs, it is reported as an MX failure only if the scenario includes
			// one or more APP/AMD records actually causing interference.  Else it is reported but not as a failure.

			if (fails) {
				if (isIXPair) {
					failed = 1;
					if ((0 == strcasecmp(source->status, "APP")) || (0 == strcasecmp(source->status, "AMD"))) {
						fprintf(ixReportFile, "**MX: %.2f%% interference caused\n", thePair->popPercent[countryIndex]);
						snprintf(mesg, MAX_STRING, "**MX with %s %s %s scenario %s, %.2f%% interference caused",
							source->callSign, source->status, source->fileNumber, scenID,
							thePair->popPercent[countryIndex]);
					} else {
						fprintf(ixReportFile, "**IX: %.2f%% interference caused\n", thePair->popPercent[countryIndex]);
						snprintf(mesg, MAX_STRING,
							"**IX check failure to %s %s %s scenario %s, %.2f%% interference caused",
							source->callSign, source->status, source->fileNumber, scenID,
							thePair->popPercent[countryIndex]);
					}
				} else {
					if (firstMX) {
						snprintf(mesg, MAX_STRING, "---- Below is IX received by proposal %s %s %s ----",
							target->callSign, target->status, target->fileNumber);
						status_message(STATUS_KEY_REPORT, "");
						status_message(STATUS_KEY_REPORT, mesg);
						status_message(STATUS_KEY_REPORT, "");
						firstMX = 0;
					}
					hasAPP = 0;
					iAPP = -1;
					for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {
						usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
						if (usource != target) {
							if (((0 == strcasecmp(usource->status, "APP")) ||
										(0 == strcasecmp(usource->status, "AMD"))) &&
									(undesireds[undesiredIndex].totals[countryIndex].ixPop > 0)) {
								if (hasAPP) {
									iAPP = -1;
								} else {
									hasAPP = 1;
									iAPP = undesiredIndex;
								}
							}
						}
					}
					if (hasAPP) {
						failed = 1;
						fprintf(ixReportFile, "**MX: %.2f%% interference received\n",
							thePair->popPercent[countryIndex]);
						if (iAPP >= 0) {
							usource = SourceKeyIndex[undesireds[iAPP].sourceKey];
							snprintf(mesg, MAX_STRING, "**MX with %s %s %s scenario %s, %.2f%% interference received",
								usource->callSign, usource->status, usource->fileNumber, scenID,
								thePair->popPercent[countryIndex]);
						} else {
							snprintf(mesg, MAX_STRING, "**MX with scenario %s, %.2f%% interference received",
								scenID, thePair->popPercent[countryIndex]);
						}
					} else {
						fprintf(ixReportFile, "%.2f%% interference received\n", thePair->popPercent[countryIndex]);
						snprintf(mesg, MAX_STRING, "Proposal receives %.2f%% interference from scenario %s",
							thePair->popPercent[countryIndex], scenID);
					}
				}
				status_message(STATUS_KEY_REPORT, mesg);

				reportWorst = 0;

			} else {

				// If the worst case is being reported update the message as needed, to be reported later.

				if (reportWorst && (thePair->popPercent[countryIndex] > worstCase)) {
					worstCase = thePair->popPercent[countryIndex];
					if (isIXPair) {
						snprintf(worstMesg, MAX_STRING, "Proposal causes %.2f%% interference to %s %s %s scenario %s",
							thePair->popPercent[countryIndex], source->callSign, source->status, source->fileNumber,
							scenID);
					} else {
						if (firstMX) {
							snprintf(mesg, MAX_STRING, "---- Below is IX received by proposal %s %s %s ----",
								target->callSign, target->status, target->fileNumber);
							status_message(STATUS_KEY_REPORT, "");
							status_message(STATUS_KEY_REPORT, mesg);
							status_message(STATUS_KEY_REPORT, "");
							firstMX = 0;
						}
						snprintf(worstMesg, MAX_STRING, "Proposal receives %.2f%% interference from scenario %s",
							thePair->popPercent[countryIndex], scenID);
					}
				}

				// Report masked interference from the proposal over a threshold percentage.  High masked interference
				// may indicate an MX relationship among three (or more) records that won't be reported because two of
				// the records will always mask each other when studying the third.  This is only a warning since this
				// is not a definitive test, and can be disabled by setting the threshold to 0.

				if (isIXPair && ttotAfter && (dtotAfter[countryIndex].servicePop > 0) &&
						(Params.IxCheckMaskingLimit > 0.)) {
					pctp = ((double)(ttotAfter[countryIndex].ixPop - ttotAfter[countryIndex].uniqueIxPop) /
						(double)dtotAfter[countryIndex].servicePop) * 100.;
					if (pctp >= Params.IxCheckMaskingLimit) {
						fputs("!!Possible MX, excessive masked interference caused\n", ixReportFile);
						snprintf(mesg, MAX_STRING, "!!Possible MX, excessive masked interference caused in scenario %s",
							scenID);
						status_message(STATUS_KEY_REPORT, mesg);
					}
				}
			}

			// List all stations involved in the scenario.  In an IX pairing list the target first, including it's
			// "before" if that exists.

			fputs(
			"\n             Call      Chan  Svc Status  City, State               File Number             Distance\n",
				ixReportFile);

			ix_source_descrip(source, NULL, ixReportFile, 1);
			fputc('\n', ixReportFile);

			showLbl = 1;

			if (isIXPair) {
				if (targetBase) {
					ix_source_descrip(targetBase, source, ixReportFile, showLbl);
					showLbl = 0;
				}
				ix_source_descrip(target, source, ixReportFile, showLbl);
				showLbl = 0;
			}

			for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {
				usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
				if (usource != target) {
					ix_source_descrip(usource, source, ixReportFile, showLbl);
					showLbl = 0;
				}
			}

			// Report the coverage and total interference for the desired.  For an interference pairing, this shows
			// interference-free coverage in both "before" and "after", and the percentage change.  A desired line is
			// always shown for the desired station's country even if the contour total is zero, other countries are
			// reported only if there were some study points in the contour.

			if (isIXPair) {

				fputs(
		"\n        Service area       Terrain-limited       IX-free, before        IX-free, after    Percent New IX\n",
					ixReportFile);

				for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {

					countryKey = countryIndex + 1;
					if ((countryKey != source->countryKey) && (0. == dtotBefore[countryIndex].contourArea) &&
							(0. == dtotAfter[countryIndex].contourArea)) {
						continue;
					}
					if (countryKey == source->countryKey) {
						cLblIndex = 0;
					} else {
						cLblIndex = countryKey;
					}

					fprintf(ixReportFile, "%8.1f %11s", dtotAfter[countryIndex].contourArea,
						pop_commas(dtotAfter[countryIndex].contourPop));
					fprintf(ixReportFile, "  %8.1f %11s", dtotAfter[countryIndex].serviceArea,
						pop_commas(dtotAfter[countryIndex].servicePop));
					fprintf(ixReportFile, "  %8.1f %11s", dtotBefore[countryIndex].ixFreeArea,
						pop_commas(dtotBefore[countryIndex].ixFreePop));
					fprintf(ixReportFile, "  %8.1f %11s  %7.2f  %7.2f%s\n", dtotAfter[countryIndex].ixFreeArea,
						pop_commas(dtotAfter[countryIndex].ixFreePop), thePair->areaPercent[countryIndex],
						thePair->popPercent[countryIndex], countryLabel[cLblIndex]);
				}

				// Report the total and unique interference from undesireds, with the unique reported for both "before"
				// and "after" cases.  Report the target first, it's "before" and "after" totals were located earlier.
				// Don't show lines where the total IX is zero; there will be at least one line for the target in some
				// country, see above.  If the target has a "before" record report that on a separate line showing only
				// "before" totals, then only "after" totals on the target's line.  Separate total IX columns are not
				// not used because for all the other undesireds those would always be identical.

				showHdr = 1;

				for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {

					countryKey = countryIndex + 1;
					if ((!ttotBefore || (0. == ttotBefore[countryIndex].ixArea)) &&
							(!ttotAfter || (0. == ttotAfter[countryIndex].ixArea))) {
						continue;
					}
					if (countryKey == source->countryKey) {
						cLblIndex = 0;
					} else {
						cLblIndex = countryKey;
					}

					if (showHdr) {
						fputs(
						"\nUndesired                         Total IX     Unique IX, before      Unique IX, after\n",
							ixReportFile);
						showHdr = 0;
					}

					if (ttotBefore) {
						fprintf(ixReportFile, "%-20s  %8.1f %11s", ix_source_label(targetBase),
							ttotBefore[countryIndex].ixArea, pop_commas(ttotBefore[countryIndex].ixPop));
						fprintf(ixReportFile, "  %8.1f %11s", ttotBefore[countryIndex].uniqueIxArea,
							pop_commas(ttotBefore[countryIndex].uniqueIxPop));
						fprintf(ixReportFile, "                      %s\n", countryLabel[cLblIndex]);
					}

					if (ttotAfter) {
						fprintf(ixReportFile, "%-20s  %8.1f %11s", ix_source_label(target),
							ttotAfter[countryIndex].ixArea, pop_commas(ttotAfter[countryIndex].ixPop));
						fprintf(ixReportFile, "                        %8.1f %11s%s\n",
							ttotAfter[countryIndex].uniqueIxArea, pop_commas(ttotAfter[countryIndex].uniqueIxPop),
							countryLabel[cLblIndex]);
					}
				}

				// Now report all of the other undesireds.  First locate the "before" totals for the undesired source,
				// the same source should exist in the "before" list but this will not fail if it does not.  Again do
				// not report any lines that have zero total interference.

				for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {

					usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
					if (usource == target) {
						continue;
					}

					utotBefore = NULL;
					for (ubIndex = 0; ubIndex < ubCount; ubIndex++) {
						if (uBefore[ubIndex].sourceKey == usource->sourceKey) {
							utotBefore = uBefore[ubIndex].totals;
							break;
						}
					}

					utotAfter = undesireds[undesiredIndex].totals;

					for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {

						countryKey = countryIndex + 1;
						if ((!utotBefore || (0. == utotBefore[countryIndex].ixArea)) &&
								(0. == utotAfter[countryIndex].ixArea)) {
							continue;
						}
						if (countryKey == source->countryKey) {
							cLblIndex = 0;
						} else {
							cLblIndex = countryKey;
						}

						if (showHdr) {
							fputs(
						"\nUndesired                         Total IX     Unique IX, before      Unique IX, after\n",
								ixReportFile);
							showHdr = 0;
						}

						if (utotBefore) {

							fprintf(ixReportFile, "%-20s  %8.1f %11s", ix_source_label(usource),
								utotAfter[countryIndex].ixArea, pop_commas(utotAfter[countryIndex].ixPop));
							fprintf(ixReportFile, "  %8.1f %11s", utotBefore[countryIndex].uniqueIxArea,
								pop_commas(utotBefore[countryIndex].uniqueIxPop));
							fprintf(ixReportFile, "  %8.1f %11s%s\n", utotAfter[countryIndex].uniqueIxArea,
								pop_commas(utotAfter[countryIndex].uniqueIxPop), countryLabel[cLblIndex]);

						} else {

							fprintf(ixReportFile, "%-20s  %8.1f %11s", ix_source_label(usource),
								utotAfter[countryIndex].ixArea, pop_commas(utotAfter[countryIndex].ixPop));
							fprintf(ixReportFile, "                        %8.1f %11s%s\n",
								utotAfter[countryIndex].uniqueIxArea, pop_commas(utotAfter[countryIndex].uniqueIxPop),
								countryLabel[cLblIndex]);
						}
					}
				}

			// Report for a coverage case, this is much simpler as there is no "before".  That scenario exists just as
			// a placeholder for the pairing, it contains no undesireds at all and is there just so the generic pairing
			// logic correctly calculates total interference percentages, see do_scenario().

			} else {

				fputs("\n        Service area       Terrain-limited               IX-free        Percent IX\n",
					ixReportFile);

				for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {

					countryKey = countryIndex + 1;
					if ((countryKey != source->countryKey) && (0. == dtotAfter[countryIndex].contourArea)) {
						continue;
					}
					if (countryKey == source->countryKey) {
						cLblIndex = 0;
					} else {
						cLblIndex = countryKey;
					}

					fprintf(ixReportFile, "%8.1f %11s", dtotAfter[countryIndex].contourArea,
						pop_commas(dtotAfter[countryIndex].contourPop));
					fprintf(ixReportFile, "  %8.1f %11s", dtotAfter[countryIndex].serviceArea,
						pop_commas(dtotAfter[countryIndex].servicePop));
					fprintf(ixReportFile, "  %8.1f %11s  %7.2f  %7.2f%s\n", dtotAfter[countryIndex].ixFreeArea,
						pop_commas(dtotAfter[countryIndex].ixFreePop), thePair->areaPercent[countryIndex],
						thePair->popPercent[countryIndex], countryLabel[cLblIndex]);
				}

				showHdr = 1;

				for (undesiredIndex = 0; undesiredIndex < undesiredCount; undesiredIndex++) {

					usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
					utotAfter = undesireds[undesiredIndex].totals;

					for (countryIndex = 0; countryIndex < MAX_COUNTRY; countryIndex++) {

						countryKey = countryIndex + 1;
						if (0. == utotAfter[countryIndex].ixArea) {
							continue;
						}
						if (countryKey == source->countryKey) {
							cLblIndex = 0;
						} else {
							cLblIndex = countryKey;
						}

						if (showHdr) {
							fputs(
							"\nUndesired                         Total IX             Unique IX   Prcnt Unique IX\n",
								ixReportFile);
							showHdr = 0;
						}

						fprintf(ixReportFile, "%-20s  %8.1f %11s", ix_source_label(usource),
							utotAfter[countryIndex].ixArea, pop_commas(utotAfter[countryIndex].ixPop));
						fprintf(ixReportFile, "  %8.1f %11s", utotAfter[countryIndex].uniqueIxArea,
							pop_commas(utotAfter[countryIndex].uniqueIxPop));
						pcta = (undesireds[undesiredIndex].totals[countryIndex].uniqueIxArea /
							dtotBefore[countryIndex].ixFreeArea) * 100.;
						if (dtotBefore[countryIndex].ixFreePop > 0) {
							pctp = ((double)undesireds[undesiredIndex].totals[countryIndex].uniqueIxPop /
								(double)dtotBefore[countryIndex].ixFreePop) * 100.;
						} else {
							pctp = 0.;
						}
						fprintf(ixReportFile, "  %7.2f  %7.2f%s\n", pcta, pctp, countryLabel[cLblIndex]);
					}
				}
			}
		}

		// Commit or delete files (if pending files not active this does nothing).

		if (!fails && OutputFlags[IXCHK_DEL_PASS] &&
				(isIXPair || (IXCHK_DEL_PASS_ALL == OutputFlags[IXCHK_DEL_PASS]))) {
			clear_pending_files(0);
		} else {
			clear_pending_files(1);
		}

		// Next pair.

		if (uBefore) {
			mem_free(uBefore);
			uBefore = NULL;
			ubCount = 0;
		}

		thePair = thePair->next;
	}

	if (reportWorst) {
		status_message(STATUS_KEY_REPORT, worstMesg);
	}

	if (!failed) {
		status_message(STATUS_KEY_REPORT, "No IX check failures found");
	}

	// All runs complete.

	status_message(STATUS_KEY_RUNCOUNT, "0");

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Descriptive text for a source in an IX check study report table.  If dsource is NULL source is a desired, otherwise
// it is an undesired relative to the desired in dsource, in that case this will include a distance column.  If
// showLabel is true prefix and suffix labels are shown.

static void ix_source_descrip(SOURCE *source, SOURCE *dsource, FILE *repFile, int showLabel) {

	if (showLabel) {
		if (dsource) {
			fputs("Undesireds:  ", repFile);
		} else {
			fputs("Desired:     ", repFile);
		}
	} else {
		fputs("             ", repFile);
	}

	char cityst[MAX_STRING];
	snprintf(cityst, MAX_STRING, "%s, %s", source->city, source->state);

	fprintf(repFile, "%-10s%-6s%-4s%-8s%-26s%-24s", source->callSign, channel_label(source), source->serviceCode,
		source->status, cityst, source->fileNumber);

	if (dsource) {

		double dist = 0.;
		bear_distance(dsource->latitude, dsource->longitude, source->latitude, source->longitude, NULL, NULL, &dist,
			Params.KilometersPerDegree);

		fprintf(repFile, "%5.1f", dist);

		if (showLabel) {
			fputs(" km", repFile);
		}
	}

	fputc('\n', repFile);
}


//---------------------------------------------------------------------------------------------------------------------
// Label text for a source in an IX check study report.

static char *ix_source_label(SOURCE *source) {

	static char str[MAX_STRING];

	snprintf(str, MAX_STRING, "%s %s %s %s", source->callSign, channel_label(source), source->serviceCode,
		source->status);

	return str;
}


//---------------------------------------------------------------------------------------------------------------------
// First of several target-record checks, these are auxiliary, they perform various tests and immediately report the
// result, they do not affect the rest of the study run.  The report file argument may be NULL in which case reporting
// is only to the front-end app by inline message.  Some tests assume the target record has just been studied as a
// desired station and so has service contours and desired coverage totals.  Some tests read data from CSV files in
// the lib/ directory, data is cached on the first read.

// Check a target source for overall HAAT matching computed value, check overall HAAT vs. ERP limits, also print an
// HAAT, ERP, and contour distance table.  For DTS, each individual transmitter source is checked and tabulated.  The
// target contours must exist.

static int check_haat(SOURCE *target, FILE *repFile) {

	// Lookup tables for HAAT vs. ERP check, from 73.622(f).

#define NUM_LOW_VHF 11
	static double lowVhfHAAT[NUM_LOW_VHF] =
		{305., 335., 365., 395., 425., 460., 490., 520., 550., 580., 610.};
	static double lowVhfERP[NUM_LOW_VHF] = 
		{ 45.,  37.,  31.,  26.,  22.,  19.,  16.,  14.,  12.,  11.,  10.};

#define NUM_HIGH_VHF 11
	static double highVhfHAAT[NUM_HIGH_VHF] =
		{305., 335., 365., 395., 425., 460., 490., 520., 550., 580., 610.};
	static double highVhfERP[NUM_HIGH_VHF] =
		{160., 132., 110.,  92.,  76.,  64.,  54.,  47.,  40.,  34.,  30.};

#define NUM_UHF 9
	static double uhfHAAT[NUM_UHF] =
		{ 365., 395., 425., 460., 490., 520., 550., 580., 610.};
	static double uhfERP[NUM_UHF] =
		{1000., 900., 750., 630., 540., 460., 400., 350., 316.};

	SOURCE *sources, *source;
	double *haat, erp, azm, dist, overallHAAT, maxerp;
	int i, i0, i1;
	char mesg[MAX_STRING], *msk;

	int countryIndex = target->countryKey - 1;

	int radialCount = Params.OverallHAATCount[countryIndex];
	double radialStep = 360. / (double)radialCount;

	if (target->isParent) {
		sources = target->dtsSources;
	} else {
		sources = target;
		target->next = NULL;
	}

	for (source = sources; source; source = source->next) {

		haat = compute_source_haat(source, radialCount);
		if (!haat) return -1;

		if (source->parentSource) {
			snprintf(mesg, MAX_STRING, "Record parameters as studied, DTS site # %d:", source->siteNumber);
		} else {
			lcpystr(mesg, "Record parameters as studied:", MAX_STRING);
		}
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);

		snprintf(mesg, MAX_STRING, "    Channel: %s", channel_label(source));
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		msk = NULL;
		switch (source->emissionMaskKey) {
			case LPTV_MASK_SIMPLE:
				msk = "Simple";
				break;
			case LPTV_MASK_STRINGENT:
				msk = "Stringent";
				break;
			case LPTV_MASK_FULL_SERVICE:
				msk = "Full Service";
				break;
		}
		if (msk) {
			snprintf(mesg, MAX_STRING, "       Mask: %s", msk);
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
		}

		snprintf(mesg, MAX_STRING, "   Latitude: %s (NAD83)", latlon_string(source->latitude, 0));
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		snprintf(mesg, MAX_STRING, "  Longitude: %s", latlon_string(source->longitude, 1));
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		if ((HEIGHT_DERIVE != source->heightAMSL) && ((source->actualHeightAMSL - source->heightAMSL) > HEIGHT_ROUND)) {
			snprintf(mesg, MAX_STRING, "Height AMSL: %.1f m (Adjusted based on actual ground elevation calculation)",
				source->actualHeightAMSL);
		} else {
			snprintf(mesg, MAX_STRING, "Height AMSL: %.1f m", source->actualHeightAMSL);
		}
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		snprintf(mesg, MAX_STRING, "       HAAT: %.1f m", source->actualOverallHAAT);
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		snprintf(mesg, MAX_STRING, "   Peak ERP: %s kW", erpkw_string(source->peakERP));
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		if (source->hasHpat) {
			if (source->hasMpat) {
				if (strlen(source->antennaID)) {
					snprintf(mesg, MAX_STRING, "    Antenna: %s (ID %s) (matrix)", source->mpatName,
						source->antennaID);
				} else {
					snprintf(mesg, MAX_STRING, "    Antenna: %s (matrix)", source->mpatName);
				}
			} else {
				if (strlen(source->antennaID)) {
					snprintf(mesg, MAX_STRING, "    Antenna: %s (ID %s) %.1f deg", source->hpatName, source->antennaID,
						source->hpatOrientation);
				} else {
					snprintf(mesg, MAX_STRING, "    Antenna: %s %.1f deg", source->hpatName, source->hpatOrientation);
				}
			}
		} else {
			lcpystr(mesg, "    Antenna: Omnidirectional", MAX_STRING);
		}
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		if (source->hasVpat) {
			snprintf(mesg, MAX_STRING, "Elev Pattrn: %s", source->vpatName);
		} else {
			if (source->useGeneric) {
				lcpystr(mesg, "Elev Pattrn: Generic", MAX_STRING);
			} else {
				lcpystr(mesg, "Elev Pattrn: None", MAX_STRING);
			}
		}
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		if ((source->hasVpat || source->useGeneric) &&
				((source->vpatElectricalTilt != 0.) || (source->vpatMechanicalTilt != 0.))) {
			if (source->vpatElectricalTilt != 0.) {
				if (source->vpatMechanicalTilt != 0.) {
					snprintf(mesg, MAX_STRING, "      Tilts: elec %.2f, mech %.2f @ %.1f deg",
						source->vpatElectricalTilt, source->vpatMechanicalTilt, source->vpatTiltOrientation);
				} else {
					snprintf(mesg, MAX_STRING, "  Elec Tilt: %.2f", source->vpatElectricalTilt);
				}
			} else {
				snprintf(mesg, MAX_STRING, "  Mech Tilt: %.2f @ %.1f deg", source->vpatMechanicalTilt,
					source->vpatTiltOrientation);
			}
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
		}

		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fputc('\n', repFile);

		snprintf(mesg, MAX_STRING, "%.1f dBu contour:", source->contourLevel);
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		lcpystr(mesg, "Azimuth      ERP       HAAT   Distance", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		for (i = 0; i < radialCount; i++) {
			azm = (double)i * radialStep;
			erp = contour_erp_lookup(source, azm);
			dist = interp_cont(azm, source->contour);
			if (0 == i) {
				snprintf(mesg, MAX_STRING, "%5.1f deg  %5.5s kW  %6.1f m  %5.1f km", azm, erpkw_string(erp), haat[i],
					dist);
			} else {
				snprintf(mesg, MAX_STRING, "%5.1f      %5.5s     %6.1f    %5.1f", azm, erpkw_string(erp), haat[i],
					dist);
			}
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
		}

		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fputc('\n', repFile);

		overallHAAT = 0;
		for (i = 0; i < radialCount; i++) {
			overallHAAT += haat[i];
		}
		overallHAAT /= (double)radialCount;

		if (rint(overallHAAT) != rint(source->actualOverallHAAT)) {
			lcpystr(mesg, "Database HAAT does not agree with computed HAAT", MAX_STRING);
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
			snprintf(mesg, MAX_STRING, "Database HAAT: %d m   Computed HAAT: %d m",
				(int)rint(source->actualOverallHAAT), (int)rint(overallHAAT));
			status_message(STATUS_KEY_REPORT, mesg);
			status_message(STATUS_KEY_REPORT, "");
			if (repFile) fprintf(repFile, "%s\n\n", mesg);
		};

		// For full-service U.S. DTV, check ERP vs. HAAT against limits.  For VHF this depends on the zone, so if the
		// zone is not defined on the record print a warning and skip the test.

		if ((SERV_TV == source->service) && source->dtv && (CNTRY_USA == source->countryKey)) {

			if ((ZONE_NONE == source->zoneKey) && (BAND_UHF != source->band)) {

				lcpystr(mesg, "!!Proposal record zone is undefined, ERP vs. HAAT check not performed", MAX_STRING);
				status_message(STATUS_KEY_REPORT, mesg);
				status_message(STATUS_KEY_REPORT, "");
				if (repFile) fprintf(repFile, "%s\n\n", mesg);

			} else {

				switch (source->band) {

					case BAND_VLO1:
					case BAND_VLO2: {
						if (source->zoneKey == ZONE_I) {
							if (overallHAAT <= 305.) {
								maxerp = 10.;
							} else {
								maxerp = 92.57 - 33.24 * log10(overallHAAT);
							}
						} else {
							if (overallHAAT <= 305.) {
								maxerp = 16.53;
							} else {
								if (overallHAAT <= 610.) {
									for (i1 = 1; i1 < (NUM_LOW_VHF - 1); i1++) {
										if (overallHAAT < lowVhfHAAT[i1]) {
											break;
										}
									}
									i0 = i1 - 1;
									maxerp = 10. * log10(lowVhfERP[i0] + ((lowVhfERP[i1] - lowVhfERP[i0]) *
										((overallHAAT - lowVhfHAAT[i0]) / (lowVhfHAAT[i1] - lowVhfHAAT[i0]))));
								} else {
									maxerp = 57.57 - 17.08 * log10(overallHAAT);
								}
							}
						}
						break;
					}

					case BAND_VHI: {
						if (source->zoneKey == ZONE_I) {
							if (overallHAAT <= 305.) {
								maxerp = 14.77;
							} else {
								maxerp = 97.35 - 33.24 * log10(overallHAAT);
							}
						} else {
							if (overallHAAT <= 305.) {
								maxerp = 22.04;
							} else {
								if (overallHAAT <= 610.) {
									for (i1 = 1; i1 < (NUM_HIGH_VHF - 1); i1++) {
										if (overallHAAT < highVhfHAAT[i1]) {
											break;
										}
									}
									i0 = i1 - 1;
									maxerp = 10. * log10(highVhfERP[i0] + ((highVhfERP[i1] - highVhfERP[i0]) *
										((overallHAAT - highVhfHAAT[i0]) / (highVhfHAAT[i1] - highVhfHAAT[i0]))));
								} else {
									maxerp = 62.34 - 17.08 * log10(overallHAAT);
								}
							}
						}
						break;
					}

					case BAND_UHF:
					default: {
						if (overallHAAT <= 365.) {
							maxerp = 30.;
						} else {
							if (overallHAAT <= 610.) {
								for (i1 = 1; i1 < (NUM_UHF - 1); i1++) {
									if (overallHAAT < uhfHAAT[i1]) {
										break;
									}
								}
								i0 = i1 - 1;
								maxerp = 10. * log10(uhfERP[i0] + ((uhfERP[i1] - uhfERP[i0]) *
									((overallHAAT - uhfHAAT[i0]) / (uhfHAAT[i1] - uhfHAAT[i0]))));
							} else {
								maxerp = 72.57 - 17.08 * log10(overallHAAT);
							}
						}
						break;
					}
				}

				if (source->peakERP > maxerp) {
					lcpystr(mesg, "ERP exceeds maximum", MAX_STRING);
					status_message(STATUS_KEY_REPORT, mesg);
					if (repFile) fprintf(repFile, "%s\n", mesg);
					lcpystr(mesg, "ERP: ", MAX_STRING);
					lcatstr(mesg, erpkw_string(source->peakERP), MAX_STRING);
					lcatstr(mesg, " kW   ERP maximum: ", MAX_STRING);
					lcatstr(mesg, erpkw_string(maxerp), MAX_STRING);
					lcatstr(mesg, " kW", MAX_STRING);
					status_message(STATUS_KEY_REPORT, mesg);
					status_message(STATUS_KEY_REPORT, "");
					if (repFile) fprintf(repFile, "%s\n\n", mesg);
				}
			}
		}

		mem_free(haat);
		haat = NULL;
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Check coverage of the target vs. it's baseline record if any, and if enabled by parameter.  There are two tests.
// First determine if the target service area fits inside the baseline area expanded by a percentage (of distance on
// each contour radial) set by parameter.  For a DTS baseline the area is the union of the individual DTS source
// contours each expanded by the percentage.  For a DTS target all individual DTS source contours must fit in the
// baseline area, regardless of whether the baseline is DTS.  Also ignore an LPTV baseline, for a digital flash-cut
// proposal the analog license may appear in the baseline role, but the coverage test is not relevant in that case.
// Note this may still function even if there is no baseline or the baseline is not being checked, because this may
// also generate the proposed contour output files.

static int check_base_coverage(SOURCE *target, SOURCE *targetBase, FILE *repFile) {

	int doBaseCheck = (targetBase && Params.CheckBaselineAreaPop && (SERV_TVLP != targetBase->service));
	if (!doBaseCheck && !OutputFlags[IXCHK_PROP_CONT]) {
		return 0;
	}

	GEOPOINTS *baseContours = NULL, **linkContour, *contour, *baseContour;
	SOURCE *source, *sources;
	CONTOUR *contourExp;
	int i, fails, didfail = 0;
	char mesg[MAX_STRING];

	int countryIndex = target->countryKey - 1;

	double *haat = NULL, azm, dist, eah, erp, minHAAT = Params.MinimumHAAT[countryIndex];
	int haatCount =
		(SERV_TVLP == target->service) ? Params.HAATCountLPTV[countryIndex] : Params.HAATCount[countryIndex];

	FILE *outfile = NULL;

	// The contour(s) being tested, both baseline and proposed, may be output to CSV files.  If so the HAAT and ERP are
	// determined on each radial to a rendered contour point.  Note the rendering will always put points on all actual
	// contour radials, but it may also include intermediate curve-fit points.

	if (doBaseCheck) {

		double areaExtend = 1. + (Params.BaselineAreaExtendPercent / 100.);

		if (OutputFlags[IXCHK_PROP_CONT]) {
			snprintf(mesg, MAX_STRING, "%s_base.csv", PROP_CONT_BASE_NAME);
			outfile = open_sum_file(mesg);
		}

		if (targetBase->isParent) {
			sources = targetBase->dtsSources;
		} else {
			sources = targetBase;
			targetBase->next = NULL;
		}

		linkContour = &baseContours;
		for (source = sources; source; source = source->next) {

			if (outfile) {
				if (source->parentSource) {
					fprintf(outfile, "DTS site # %d\n", source->siteNumber);
				}
				fputs("Azimuth,Distance,Latitude,Longitude,HAAT,ERP\n", outfile);
				if (haat) {
					mem_free(haat);
				}
				haat = compute_source_haat(source, haatCount);
				if (!haat) return -1;
			}

			contourExp = contour_alloc(source->contour->latitude, source->contour->longitude, source->contour->mode,
				source->contour->count);
			for (i = 0; i < contourExp->count; i++) {
				contourExp->distance[i] = source->contour->distance[i] * areaExtend;
			}
			contour = render_contour(contourExp, Params.KilometersPerDegree);

			*linkContour = contour;
			linkContour = &(contour->next);
			contourExp->points = NULL;
			contour_free(contourExp);
			contourExp = NULL;

			if (outfile) {
				for (i = 0; i < contour->nPts; i++) {
					bear_distance(source->latitude, source->longitude, contour->ptLat[i], contour->ptLon[i], &azm,
						NULL, &dist, Params.KilometersPerDegree);
					eah = interp_min(azm, haat, haatCount, minHAAT, 1);
					erp = contour_erp_lookup(source, azm);
					fprintf(outfile, "%.2f,%.2f,%.8f,%.8f,%.2f,%.2f\n", azm, dist, contour->ptLat[i],
						contour->ptLon[i], eah, erp);
				}
			}
		}
	}

	if (outfile) {
		file_close(outfile);
		outfile = NULL;
	}
	if (haat) {
		mem_free(haat);
		haat = NULL;
	}

	if (OutputFlags[IXCHK_PROP_CONT]) {
		snprintf(mesg, MAX_STRING, "%s.csv", PROP_CONT_BASE_NAME);
		outfile = open_sum_file(mesg);
	}

	if (target->isParent) {
		sources = target->dtsSources;
	} else {
		sources = target;
		target->next = NULL;
	}

	for (source = sources; source; source = source->next) {

		if (outfile) {
			if (source->parentSource) {
				fprintf(outfile, "DTS site # %d\n", source->siteNumber);
			}
			fputs("Azimuth,Distance,Latitude,Longitude,HAAT,ERP,Fails\n", outfile);
			if (haat) {
				mem_free(haat);
			}
			haat = compute_source_haat(source, haatCount);
			if (!haat) return -1;
		}

		contour = render_contour(source->contour, Params.KilometersPerDegree);

		for (i = 0; i < contour->nPts; i++) {

			fails = 0;

			if (doBaseCheck) {
				for (baseContour = baseContours; baseContour; baseContour = baseContour->next) {
					if (inside_poly(contour->ptLat[i], contour->ptLon[i], baseContour)) {
						break;
					}
				}
				if (!baseContour) {
					fails = 1;
					didfail = 1;
					if (!outfile) {
						break;
					}
				}
			}

			if (outfile) {
				bear_distance(source->latitude, source->longitude, contour->ptLat[i], contour->ptLon[i], &azm, NULL,
					&dist, Params.KilometersPerDegree);
				eah = interp_min(azm, haat, haatCount, minHAAT, 1);
				erp = contour_erp_lookup(source, azm);
				fprintf(outfile, "%.2f,%.2f,%.8f,%.8f,%.2f,%.2f,%d\n", azm, dist, contour->ptLat[i], contour->ptLon[i],
					eah, erp, fails);
			}
		}

		if (didfail && !outfile) {
			break;
		}
	}

	// In addition to CSV, the proposal and baseline contours may also output to a shapefile or KML file.

	if (outfile) {
		file_close(outfile);
		outfile = NULL;
	}
	if (haat) {
		mem_free(haat);
		haat = NULL;
	}

	if ((IXCHK_PROP_CONT_CSVSHP == OutputFlags[IXCHK_PROP_CONT]) ||
			(IXCHK_PROP_CONT_CSVKML == OutputFlags[IXCHK_PROP_CONT])) {

#define IXMAP_NUM_ATTR 6

#define IXMAP_LEN_LABEL 30
#define IXMAP_LEN_NUM   7
#define IXMAP_PREC_NUM  2

		SHAPEATTR attrs[IXMAP_NUM_ATTR] = {
			ATTR_CALLSIGN,
			ATTR_FILENUMBER,
			ATTR_SITENUMBER,
			{"TYPE", SHP_ATTR_CHAR, IXMAP_LEN_LABEL, 0},
			{"LEVEL", SHP_ATTR_NUM, IXMAP_LEN_NUM, IXMAP_PREC_NUM},
			{"EXPANDPCT", SHP_ATTR_NUM, IXMAP_LEN_NUM, IXMAP_PREC_NUM}
		};

		int fmt = MAP_FILE_SHAPE;
		if (IXCHK_PROP_CONT_CSVKML == OutputFlags[IXCHK_PROP_CONT]) {
			fmt = MAP_FILE_KML;
		}
		MAPFILE *mapOut = open_sum_mapfile(fmt, PROP_CONT_BASE_NAME, SHP_TYPE_POLYLINE, IXMAP_NUM_ATTR, attrs, NULL);

		if (mapOut) {

			char label[IXMAP_LEN_LABEL + 1], siteNum[LEN_SITENUMBER + 1], contLvl[IXMAP_LEN_NUM + 1],
				expPct[IXMAP_LEN_NUM + 1], *attrData[IXMAP_NUM_ATTR];

			attrData[2] = siteNum;
			attrData[4] = contLvl;
			attrData[5] = expPct;

			if (targetBase) {
				if (targetBase->isParent) {
					sources = targetBase->dtsSources;
				} else {
					sources = targetBase;
					targetBase->next = NULL;
				}
				baseContour = baseContours;
				attrData[0] = targetBase->callSign;
				attrData[1] = targetBase->fileNumber;
				snprintf(contLvl, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, targetBase->contourLevel);
				for (source = sources; source; source = source->next) {
					if (targetBase->isParent) {
						snprintf(label, (IXMAP_LEN_LABEL + 1), "%s.%d BL", targetBase->callSign, source->siteNumber);
						snprintf(siteNum, (LEN_SITENUMBER + 1), "%d", source->siteNumber);
					} else {
						snprintf(label, (IXMAP_LEN_LABEL + 1), "%s BL", targetBase->callSign);
						siteNum[0] = '\0';
					}
					attrData[3] = "Baseline";
					expPct[0] = '\0';
					if (write_shape(mapOut, 0., 0., render_service_area(source), 1, NULL, attrData, label, NULL, 0,
							1)) {
						close_mapfile(mapOut);
						mapOut = NULL;
						break;
					} else {
						if (baseContour) {
							if (targetBase->isParent) {
								snprintf(label, (IXMAP_LEN_LABEL + 1), "%s.%d BL exp", targetBase->callSign,
									source->siteNumber);
							} else {
								snprintf(label, (IXMAP_LEN_LABEL + 1), "%s BL exp", targetBase->callSign);
							}
							attrData[3] = "Baseline expanded";
							snprintf(expPct, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM,
								Params.BaselineAreaExtendPercent);
							if (write_shape(mapOut, 0., 0., baseContour, 1, NULL, attrData, label, NULL, 0, 1)) {
								close_mapfile(mapOut);
								mapOut = NULL;
								break;
							}
							baseContour = baseContour->next;
						}
					}
				}
			}

			if (mapOut) {
				if (target->isParent) {
					sources = target->dtsSources;
				} else {
					sources = target;
					target->next = NULL;
				}
				attrData[0] = target->callSign;
				attrData[1] = target->fileNumber;
				attrData[3] = "Proposal";
				snprintf(contLvl, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, target->contourLevel);
				expPct[0] = '\0';
				for (source = sources; source; source = source->next) {
					if (target->isParent) {
						snprintf(label, (IXMAP_LEN_LABEL + 1), "%s.%d", target->callSign, source->siteNumber);
						snprintf(siteNum, (LEN_SITENUMBER + 1), "%d", source->siteNumber);
					} else {
						snprintf(label, (IXMAP_LEN_LABEL + 1), "%s", target->callSign);
						siteNum[0] = '\0';
					}
					if (write_shape(mapOut, 0., 0., render_service_area(source), 1, NULL, attrData, label, NULL, 0,
							1)) {
						break;
					}
				}
			}

			close_mapfile(mapOut);
			mapOut = NULL;
		}
	}

	while (baseContours) {
		baseContour = baseContours;
		baseContours = baseContour->next;
		baseContour->next = NULL;
		geopoints_free(baseContour);
	}
	baseContour = NULL;

	if (!doBaseCheck) {
		return 0;
	}

	if (didfail) {
		snprintf(mesg, MAX_STRING, "**Proposal service area extends beyond baseline plus %.1f%%",
			Params.BaselineAreaExtendPercent);
	} else {
		snprintf(mesg, MAX_STRING, "Proposal service area is within baseline plus %.1f%%",
			Params.BaselineAreaExtendPercent);
	}
	status_message(STATUS_KEY_REPORT, mesg);
	if (repFile) fprintf(repFile, "%s\n", mesg);

	// The second test is just a straight percentage change of the terrain-limited population, fails if the target
	// reduces that by more than a percentage parameter.  Both sources should have desired totals, meaning both have
	// just been studied as desireds in a scenario.

	double frac = 1.;
	if (targetBase->totals[countryIndex].servicePop > 0) {
		frac = (double)target->totals[countryIndex].servicePop / (double)targetBase->totals[countryIndex].servicePop;
	}

	if (frac < (1. - (Params.BaselinePopReducePercent / 100.))) {
		snprintf(mesg, MAX_STRING, "**Proposal service area population is less than %.1f%% of baseline",
			(100. - Params.BaselinePopReducePercent));
	} else {
		snprintf(mesg, MAX_STRING, "Proposal service area population is more than %.1f%% of baseline",
			(100. - Params.BaselinePopReducePercent));
	}
	status_message(STATUS_KEY_REPORT, mesg);
	status_message(STATUS_KEY_REPORT, "");
	if (repFile) fprintf(repFile, "%s\n\n", mesg);

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Check a DTS operation to determine if there are coverage areas outside the rule limit.  That limit is at least the
// authorized facility contour for all records, for full-service it is the union of the authorized facility contour
// and the boundary distance/sectors around the reference point.  If there are DTS contours extending beyond that limit
// area, check DTS sites which must be within the limit area.  If that test passes then check F(50,50) and F(50,10)
// contours per 73.626(f) for full-service, or just F(50,50) per 73.6023(f)/74.720(e) for Class A/LPTV, for contour
// extension.  If output of a contour mapfile is also selected (see check_base_coverage()), this will generate an
// additional map file containing all of the service limit areas and checked contours including F(50,50) and F(50,10).

static int check_dts_coverage(SOURCE *target, FILE *repFile) {

	// Values from 73.626(c) table.

#define NUM_TABLE_ROWS 5
	static double tableDist90[NUM_TABLE_ROWS] = {108., 128., 101., 123., 103.};
	static double tableDist50[NUM_TABLE_ROWS] = {132., 158., 121., 149., 142.};
	static double tableDist10[NUM_TABLE_ROWS] = {183., 209., 182., 208., 246.};
	static double tableAuthIX[NUM_TABLE_ROWS] = {28., 28., 33., 33., 36.};
	static double tableNodeIX[NUM_TABLE_ROWS] = {18.8, 18.8, 23.8, 23.8, 26.8};
	static double tableHAAT[NUM_TABLE_ROWS] = {305., 610., 305., 610., 365.};

	// Do nothing for a non-DTS or non-U.S. record.

	if (!target->dts || (CNTRY_USA != target->countryKey)) {
		return 0;
	}
	int targetIsFullService = (SERV_TV == target->service);

	// These checks can't be done for VHF if the zone is undefined.  Only for full-service.

	char mesg[MAX_STRING];

	if (targetIsFullService && (ZONE_NONE == target->zoneKey) && (BAND_UHF != target->band)) {
		lcpystr(mesg, "!!Proposal record zone is undefined, DTS contour check not performed", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
		return 0;
	}

	// Set up for map file output if needed.

	SHAPEATTR attrs[IXMAP_NUM_ATTR] = {
		ATTR_CALLSIGN,
		ATTR_FILENUMBER,
		ATTR_SITENUMBER,
		{"TYPE", SHP_ATTR_CHAR, IXMAP_LEN_LABEL, 0},
		{"LEVEL", SHP_ATTR_NUM, IXMAP_LEN_NUM, IXMAP_PREC_NUM},
		{"RADIUS", SHP_ATTR_NUM, IXMAP_LEN_NUM, IXMAP_PREC_NUM}
	};

	char label[IXMAP_LEN_LABEL + 1], siteNum[LEN_SITENUMBER + 1], contLvl[IXMAP_LEN_NUM + 1], dist[IXMAP_LEN_NUM + 1],
		*attrData[IXMAP_NUM_ATTR];
	attrData[2] = siteNum;
	attrData[4] = contLvl;
	attrData[5] = dist;

	// No RADIUS attribute for Class A/LPTV.

	int attrCount = IXMAP_NUM_ATTR;
	if (!targetIsFullService) {
		attrCount--;
	}

	MAPFILE *mapOut = NULL;
	if ((IXCHK_PROP_CONT_CSVSHP == OutputFlags[IXCHK_PROP_CONT]) ||
			(IXCHK_PROP_CONT_CSVKML == OutputFlags[IXCHK_PROP_CONT])) {
		int fmt = MAP_FILE_SHAPE;
		if (IXCHK_PROP_CONT_CSVKML == OutputFlags[IXCHK_PROP_CONT]) {
			fmt = MAP_FILE_KML;
		}
		char mapName[MAX_STRING];
		snprintf(mapName, MAX_STRING, "%s_dts", PROP_CONT_BASE_NAME);
		mapOut = open_sum_mapfile(fmt, mapName, SHP_TYPE_POLYLINE, attrCount, attrs, NULL);
	}

	// Check the service contours of all sites against the authorized contour and distance limit or sectors geography,
	// if any.  The authorized facility contour is expanded to prevent failures when a DTS facility is essentially, but
	// not precisely, identical to the authorized.  Per FCC policy ERP can vary by a small amount on any given azimuth
	// for a "minor change", so the contour is expanded by reducing the contour level by that amount which for curve
	// lookup has the same effect as increasing ERP.

	CONTOUR *authContour, *contour;
	GEOPOINTS *authContourPts, *contourPts;
	SOURCE *dtsSource;
	int i, fails = 0, done = 0;
	double authContLvl = target->dtsAuthSource->contourLevel + DTS_AUTH_CONT_ADJUST;

	authContour = project_fcc_contour(target->dtsAuthSource, FCC_F90, authContLvl);
	if (NULL == authContour) return -1;
	authContourPts = render_contour(authContour, Params.KilometersPerDegree);

	// When writing to the map file if an error occurs cancel map output but continue with the tests.

	if (mapOut) {
		snprintf(label, (IXMAP_LEN_LABEL + 1), "%s auth", target->dtsAuthSource->callSign);
		attrData[0] = target->dtsAuthSource->callSign;
		attrData[1] = target->dtsAuthSource->fileNumber;
		siteNum[0] = '\0';
		attrData[3] = "Authorized";
		snprintf(contLvl, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, authContLvl);
		dist[0] = '\0';
		if (write_shape(mapOut, 0., 0., authContourPts, 1, NULL, attrData, label, NULL, 0, 1)) {
			close_mapfile(mapOut);
			mapOut = NULL;
		} else {
			if (targetIsFullService) {
				snprintf(label, (IXMAP_LEN_LABEL + 1), "%s lim", target->callSign);
				attrData[3] = "Limit";
				contLvl[0] = '\0';
				snprintf(dist, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, target->dtsMaximumDistance);
				if (write_shape(mapOut, 0., 0., render_geography(target->geography, Params.KilometersPerDegree), 1,
						NULL, attrData, label, NULL, 0, 1)) {
					close_mapfile(mapOut);
					mapOut = NULL;
				}
			}
		}
	}

	if (mapOut) {
		attrData[0] = target->callSign;
		attrData[1] = target->fileNumber;
		attrData[3] = "Proposal";
		snprintf(contLvl, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, target->contourLevel);
		dist[0] = '\0';
	}

	for (dtsSource = target->dtsSources; dtsSource; dtsSource = dtsSource->next) {

		contourPts = render_contour(dtsSource->contour, Params.KilometersPerDegree);

		if (mapOut) {
			snprintf(label, (IXMAP_LEN_LABEL + 1), "%s.%d", dtsSource->callSign, dtsSource->siteNumber);
			snprintf(siteNum, (LEN_SITENUMBER + 1), "%d", dtsSource->siteNumber);
			if (write_shape(mapOut, 0., 0., contourPts, 1, NULL, attrData, label, NULL, 0, 1)) {
				close_mapfile(mapOut);
				mapOut = NULL;
			}
		}

		if (!fails) {
			for (i = 0; i < contourPts->nPts; i++) {
				if ((!targetIsFullService || !inside_geography(contourPts->ptLat[i], contourPts->ptLon[i],
						target->geography, Params.KilometersPerDegree)) && !inside_poly(contourPts->ptLat[i],
						contourPts->ptLon[i], authContourPts)) {
					fails = 1;
					break;
				}
			}
		}

		if (fails && !mapOut) {
			break;
		}
	}

	// If contours fit no further checks are needed, however this may continue to write map output.

	if (!fails) {
		done = 1;
		if (targetIsFullService) {
			lcpystr(mesg, "DTS proposal coverage is within authorized facility and distance limit", MAX_STRING);
		} else {
			lcpystr(mesg, "DTS proposal coverage is within authorized facility limit", MAX_STRING);
		}
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
	}

	// If contours do not fit, check if any DTS sites are outside the limit area, if so that's immediate failure.

	if (!done) {
		fails = 0;
		for (dtsSource = target->dtsSources; dtsSource; dtsSource = dtsSource->next) {
			if ((!targetIsFullService || !inside_geography(dtsSource->latitude, dtsSource->longitude,
					target->geography, Params.KilometersPerDegree)) && !inside_poly(dtsSource->latitude,
					dtsSource->longitude, authContourPts)) {
				fails = 1;
				break;
			}
		}
		if (fails) {
			done = 1;
			if (targetIsFullService) {
				lcpystr(mesg, "**DTS proposal has sites outside authorized facility and distance limit", MAX_STRING);
			} else {
				lcpystr(mesg, "**DTS proposal has sites outside authorized facility limit", MAX_STRING);
			}
			status_message(STATUS_KEY_REPORT, mesg);
			status_message(STATUS_KEY_REPORT, "");
			if (repFile) fprintf(repFile, "%s\n\n", mesg);
		}
	}

	contour_free(authContour);
	authContour = NULL;
	authContourPts = NULL;

	if (done && !mapOut) return 0;

	// Sites are OK, now check the F(50,50) and F(50,10) contours as needed, may be continuing just for map output.

	if (!done) {
		if (targetIsFullService) {
			lcpystr(mesg, "DTS proposal coverage extends outside authorized facility and distance limit", MAX_STRING);
		} else {
			lcpystr(mesg, "DTS proposal coverage extends outside authorized facility limit", MAX_STRING);
		}
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);
	}

	// Set up F(50,10) contour levels and limit distances for full-service case, F(50,50) levels are just the service
	// contour level.  Not needed for Class A/LPTV, those have no distance limit and only F(50,50) is checked.

	int tableCase = 0, authTableCase = 0;
	double nodeIX = 0., nodeAuthIX = 0., authIX = 0., dist90 = 0, dist50 = 0., dist10 = 0.;

	if (targetIsFullService) {

		tableCase = 4;
		if ((BAND_VLO1 == target->band) || (BAND_VLO2 == target->band)) {
			if (ZONE_I == target->zoneKey) {
				tableCase = 0;
			} else {
				tableCase = 1;
			}
		} else {
			if (BAND_VHI == target->band) {
				if (ZONE_I == target->zoneKey) {
					tableCase = 2;
				} else {
					tableCase = 3;
				}
			}
		}

		// The F(50,10) contour levels from the table may need to be dipole-adjusted for UHF.

		nodeIX = tableNodeIX[tableCase];
		nodeAuthIX = tableAuthIX[tableCase];
		if ((BAND_UHF == target->band) && (Params.UseDipoleCont[CNTRY_USA - 1])) {
			double adj = 20. * log10(target->channelFrequency / Params.DipoleCenterFreqCont[CNTRY_USA - 1]);
			nodeIX += adj;
			nodeAuthIX += adj;
		}

		// Authorized facility not necessarily the same band, zone, or channel as main record.

		authTableCase = 4;
		if ((BAND_VLO1 == target->dtsAuthSource->band) || (BAND_VLO2 == target->dtsAuthSource->band)) {
			if (ZONE_I == target->dtsAuthSource->zoneKey) {
				authTableCase = 0;
			} else {
				authTableCase = 1;
			}
		} else {
			if (BAND_VHI == target->dtsAuthSource->band) {
				if (ZONE_I == target->dtsAuthSource->zoneKey) {
					authTableCase = 2;
				} else {
					authTableCase = 3;
				}
			}
		}

		authIX = tableAuthIX[authTableCase];
		if ((BAND_UHF == target->dtsAuthSource->band) && (Params.UseDipoleCont[CNTRY_USA - 1])) {
			authIX += 20. * log10(target->dtsAuthSource->channelFrequency / Params.DipoleCenterFreqCont[CNTRY_USA - 1]);
		}

		// Determine the limit distances for F(50,50) and F(50,10).  These usually come from the table, but if the
		// limit distance on the record does not match the F(50,90) table distance, the other table distance are not
		// applicable either.  In that case use the distance and service level from the record plus the HAAT used to
		// define the table to get an ERP for the F(50,90) case, then use that to project corresponding F(50,50) and
		// F(50,10) distances.

		dist90 = target->dtsMaximumDistance;
		dist50 = tableDist50[tableCase];
		dist10 = tableDist10[tableCase];
		if (dist90 != tableDist90[tableCase]) {
			double erp = 0., sig = target->contourLevel;
			fcc_curve(&erp, &sig, &dist90, tableHAAT[tableCase], target->band, FCC_PWR, FCC_F90,
				Params.OffCurveLookupMethod, NULL, 0., NULL, NULL, NULL);
			fcc_curve(&erp, &sig, &dist50, tableHAAT[tableCase], target->band, FCC_DST, FCC_F50,
				Params.OffCurveLookupMethod, NULL, 0., NULL, NULL, NULL);
			dist50 = rint(dist50 * 100.) / 100.;
			fcc_curve(&erp, &authIX, &dist10, tableHAAT[tableCase], target->band, FCC_DST, FCC_F10,
				Params.OffCurveLookupMethod, NULL, 0., NULL, NULL, NULL);
			dist10 = rint(dist10 * 100.) / 100.;
		}
	}

	// Check the F(50,50) contours, these are projected using the service contour level.

	if (!done) {
		if (targetIsFullService) {
			snprintf(mesg, MAX_STRING, "Checking F(50,50) contour extension, limit distance = %.2f km", dist50);
		} else {
			lcpystr(mesg, "Checking F(50,50) contour extension", MAX_STRING);
		}
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);
	}

	GEOGRAPHY *geo = NULL;
	if (targetIsFullService) {
		geo = geography_alloc(0, GEO_TYPE_CIRCLE, target->latitude, target->longitude, 0);
		geo->a.radius = dist50;
	}

	authContour = project_fcc_contour(target->dtsAuthSource, FCC_F50, authContLvl);
	if (NULL == authContour) return -1;
	authContourPts = render_contour(authContour, Params.KilometersPerDegree);

	if (mapOut) {
		snprintf(label, (IXMAP_LEN_LABEL + 1), "%s auth F50", target->dtsAuthSource->callSign);
		attrData[0] = target->dtsAuthSource->callSign;
		attrData[1] = target->dtsAuthSource->fileNumber;
		siteNum[0] = '\0';
		attrData[3] = "Authorized F(50,50)";
		snprintf(contLvl, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, authContLvl);
		dist[0] = '\0';
		if (write_shape(mapOut, 0., 0., authContourPts, 1, NULL, attrData, label, NULL, 0, 1)) {
			close_mapfile(mapOut);
			mapOut = NULL;
		} else {
			if (targetIsFullService) {
				snprintf(label, (IXMAP_LEN_LABEL + 1), "%s lim F50", target->callSign);
				attrData[3] = "Limit F(50,50)";
				contLvl[0] = '\0';
				snprintf(dist, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, dist50);
				if (write_shape(mapOut, 0., 0., render_geography(geo, Params.KilometersPerDegree), 1, NULL, attrData,
						label, NULL, 0, 1)) {
					close_mapfile(mapOut);
					mapOut = NULL;
				}
			}
		}
	}

	if (!done) fails = 0;

	if (mapOut) {
		attrData[0] = target->callSign;
		attrData[1] = target->fileNumber;
		attrData[3] = "Proposal F(50,50)";
		snprintf(contLvl, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, target->contourLevel);
		dist[0] = '\0';
	}

	for (dtsSource = target->dtsSources; dtsSource; dtsSource = dtsSource->next) {

		contour = project_fcc_contour(dtsSource, FCC_F50, dtsSource->contourLevel);
		if (NULL == contour) return -1.;
		contourPts = render_contour(contour, Params.KilometersPerDegree);

		if (mapOut) {
			snprintf(label, (IXMAP_LEN_LABEL + 1), "%s.%d F50", dtsSource->callSign, dtsSource->siteNumber);
			snprintf(siteNum, (LEN_SITENUMBER + 1), "%d", dtsSource->siteNumber);
			if (write_shape(mapOut, 0., 0., contourPts, 1, NULL, attrData, label, NULL, 0, 1)) {
				close_mapfile(mapOut);
				mapOut = NULL;
			}
		}

		if (!done && !fails) {
			for (i = 0; i < contourPts->nPts; i++) {
				if ((!targetIsFullService || !inside_geography(contourPts->ptLat[i], contourPts->ptLon[i], geo,
						Params.KilometersPerDegree)) && !inside_poly(contourPts->ptLat[i], contourPts->ptLon[i],
						authContourPts)) {
					fails = 1;
					break;
				}
			}
		}

		contour_free(contour);
		contour = NULL;
		contourPts = NULL;

		if (fails && !mapOut) {
			break;
		}
	}

	contour_free(authContour);
	authContour = NULL;
	authContourPts = NULL;

	if (geo) {
		geography_free(geo);
		geo = NULL;
	}

	if (!done && fails) {
		done = 1;
		if (targetIsFullService) {
			lcpystr(mesg, "**F(50,50) contours extend outside authorized facility and distance limit", MAX_STRING);
		} else {
			lcpystr(mesg, "**F(50,50) contours extend outside authorized facility limit", MAX_STRING);
		}
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
		if (!mapOut) return 0;
	}

	// Finally, if full-service, check the F(50,10) contours, these are projected using the table contour levels.  This
	// is complicated by having a different contour level if a site is at the authorized facility location (within +/-
	// a small amount of latitude and longitude).

	if (targetIsFullService) {

		if (!done) {
			snprintf(mesg, MAX_STRING, "Checking F(50,10) contour extension, limit distance = %.2f km", dist10);
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
		}

		geo = geography_alloc(0, GEO_TYPE_CIRCLE, target->latitude, target->longitude, 0);
		geo->a.radius = dist10;

		authContLvl = authIX + DTS_AUTH_CONT_ADJUST;
		authContour = project_fcc_contour(target->dtsAuthSource, FCC_F10, authContLvl);
		if (NULL == authContour) return -1;
		authContourPts = render_contour(authContour, Params.KilometersPerDegree);

		if (mapOut) {
			snprintf(label, (IXMAP_LEN_LABEL + 1), "%s auth F10", target->dtsAuthSource->callSign);
			attrData[0] = target->dtsAuthSource->callSign;
			attrData[1] = target->dtsAuthSource->fileNumber;
			siteNum[0] = '\0';
			attrData[3] = "Authorized F(50,10)";
			snprintf(contLvl, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, authContLvl);
			dist[0] = '\0';
			if (write_shape(mapOut, 0., 0., authContourPts, 1, NULL, attrData, label, NULL, 0, 1)) {
				close_mapfile(mapOut);
				mapOut = NULL;
			} else {
				snprintf(label, (IXMAP_LEN_LABEL + 1), "%s lim F10", target->callSign);
				attrData[3] = "Limit F(50,10)";
				contLvl[0] = '\0';
				snprintf(dist, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, dist10);
				if (write_shape(mapOut, 0., 0., render_geography(geo, Params.KilometersPerDegree), 1, NULL, attrData,
						label, NULL, 0, 1)) {
					close_mapfile(mapOut);
					mapOut = NULL;
				}
			}
		}

		double authLat = target->dtsAuthSource->latitude, authLon = target->dtsAuthSource->longitude, ixCont = 0.;

		if (!done) fails = 0;

		if (mapOut) {
			attrData[0] = target->callSign;
			attrData[1] = target->fileNumber;
			attrData[3] = "Proposal F(50,10)";
			dist[0] = '\0';
		}

		for (dtsSource = target->dtsSources; dtsSource; dtsSource = dtsSource->next) {

			if (((fabs(dtsSource->latitude - authLat) * 3600.) <= DTS_AUTH_COORD_MATCH) &&
					((fabs(dtsSource->longitude - authLon) * 3600.) <= DTS_AUTH_COORD_MATCH)) {
				ixCont = nodeAuthIX;
			} else {
				ixCont = nodeIX;
			}
			contour = project_fcc_contour(dtsSource, FCC_F10, ixCont);
			if (NULL == contour) return -1.;
			contourPts = render_contour(contour, Params.KilometersPerDegree);

			if (mapOut) {
				snprintf(label, (IXMAP_LEN_LABEL + 1), "%s.%d F10", dtsSource->callSign, dtsSource->siteNumber);
				snprintf(siteNum, (LEN_SITENUMBER + 1), "%d", dtsSource->siteNumber);
				snprintf(contLvl, (IXMAP_LEN_NUM + 1), "%.*f", IXMAP_PREC_NUM, ixCont);
				if (write_shape(mapOut, 0., 0., contourPts, 1, NULL, attrData, label, NULL, 0, 1)) {
					close_mapfile(mapOut);
					mapOut = NULL;
				}
			}

			if (!done && !fails) {
				for (i = 0; i < contourPts->nPts; i++) {
					if (!inside_geography(contourPts->ptLat[i], contourPts->ptLon[i], geo, Params.KilometersPerDegree) &&
							!inside_poly(contourPts->ptLat[i], contourPts->ptLon[i], authContourPts)) {
						fails = 1;
						break;
					}
				}
			}

			contour_free(contour);
			contour = NULL;
			contourPts = NULL;

			if (fails && !mapOut) {
				break;
			}
		}

		contour_free(authContour);
		authContour = NULL;
		authContourPts = NULL;

		if (geo) {
			geography_free(geo);
			geo = NULL;
		}

		if (!done && fails) {
			done = 1;
			lcpystr(mesg, "**F(50,10) contours extend outside authorized facility and distance limit", MAX_STRING);
			status_message(STATUS_KEY_REPORT, mesg);
			status_message(STATUS_KEY_REPORT, "");
			if (repFile) fprintf(repFile, "%s\n\n", mesg);
		}
	}

	if (!done) {
		done = 1;
		lcpystr(mesg, "Contour checks passed", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
	}

	if (mapOut) {
		close_mapfile(mapOut);
		mapOut = NULL;
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Report distances to Canadian and Mexican borders and check if the distances are within coordination limits.  For a
// DTS source, check all of the actual transmitter locations and use the smallest distance; the reference point and
// authorized facility location are irrelevant for this test.  The coordination test for Canada may also be a contour
// check, project a particular F(50,10) contour and test if it crosses the Canadian border.

static int check_border_distance(SOURCE *target, FILE *repFile) {

	static int dtsSize = 0;
	static int *dtsSiteNumber = NULL;
	static double *dtsCanDist = NULL;
	static double *dtsMexDist = NULL;

	double bordDist[MAX_COUNTRY], minCanDist = 99999., minMexDist = 99999.;
	int err, i, fails, dtsCount = 0;
	char mesg[MAX_STRING];

	SOURCE *source, *sources;

	if (target->isParent) {
		sources = target->dtsSources;
	} else {
		sources = target;
		target->next = NULL;
	}

	for (source = sources; source; source = source->next) {
		err = get_border_distances(source->latitude, source->longitude, bordDist, Params.KilometersPerDegree);
		if (err) return err;
		if (bordDist[CNTRY_CAN - 1] < minCanDist) {
			minCanDist = bordDist[CNTRY_CAN - 1];
		}
		if (bordDist[CNTRY_MEX - 1] < minMexDist) {
			minMexDist = bordDist[CNTRY_MEX - 1];
		}
		if (target->isParent) {
			if (dtsCount >= dtsSize) {
				dtsSize += 10;
				dtsSiteNumber = (int *)mem_realloc(dtsSiteNumber, (dtsSize * sizeof(int)));
				dtsCanDist = (double *)mem_realloc(dtsCanDist, (dtsSize * sizeof(double)));
				dtsMexDist = (double *)mem_realloc(dtsMexDist, (dtsSize * sizeof(double)));
			}
			dtsSiteNumber[dtsCount] = source->siteNumber;
			dtsCanDist[dtsCount] = bordDist[CNTRY_CAN - 1];
			dtsMexDist[dtsCount++] = bordDist[CNTRY_MEX - 1];
		}
	}

	// For full-service, if less than 300 km from the Canadian border coordination is always required.  If between
	// 300 and 360 km, do the contour check.  For Class A and LPTV, if less than 300 km do the contour check.

	int doCont = 0;
	if (SERV_TV == target->service) {
		if (minCanDist <= 300.) {
			lcpystr(mesg, "**Proposal is within coordination distance of Canadian border", MAX_STRING);
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
		} else {
			if (minCanDist <= 360.) {
				doCont = 1;
			}
		}
	} else {
		if (minCanDist <= 300.) {
			doCont = 1;
		}
	}

	if (doCont) {

		CONTOUR *srcContour;
		GEOPOINTS *contour;
		double contLevel;

		fails = 0;

		switch (target->band) {
			case BAND_VLO1:
			case BAND_VLO2: {
				contLevel = 13.;
				break;
			}
			case BAND_VHI: {
				contLevel = 21.;
				break;
			}
			case BAND_UHF:
			default: {
				contLevel = 26. + (20. * log10(target->channelFrequency / Params.DipoleCenterFreqCont[CNTRY_USA - 1]));
				break;
			}
		}

		for (source = sources; source; source = source->next) {
			srcContour = project_fcc_contour(source, FCC_F10, contLevel);
			if (!srcContour) return -1;
			contour = render_contour(srcContour, Params.KilometersPerDegree);
			for (i = 0; i < contour->nPts; i++) {
				if (CNTRY_CAN == find_country(contour->ptLat[i], contour->ptLon[i])) {
					fails = 1;
					break;
				}
			}
			contour_free(srcContour);
			srcContour = NULL;
			contour = NULL;
			if (fails) {
				break;
			}
		}

		if (fails) {
			snprintf(mesg, MAX_STRING,
				"**Proposal %.2f dBu contour crosses Canadian border, coordination required", contLevel);
		} else {
			snprintf(mesg, MAX_STRING, "Proposal %.2f dBu contour does not cross Canadian border", contLevel);
		}
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);
	}

	if (target->isParent) {
		for (i = 0; i < dtsCount; i++) {
			if (0 == i) {
				snprintf(mesg, MAX_STRING, "Distance to Canadian border: DTS site # %d   %.1f km", dtsSiteNumber[i],
					dtsCanDist[i]);
			} else {
				snprintf(mesg, MAX_STRING, "                             DTS site # %d   %.1f km", dtsSiteNumber[i],
					dtsCanDist[i]);
			}
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
		}
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fputc('\n', repFile);
	} else {
		snprintf(mesg, MAX_STRING, "Distance to Canadian border: %.1f km", minCanDist);
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
	}

	if (SERV_TV == target->service) {
		fails = (minMexDist <= 250.);
	} else {
		fails = (minMexDist <= 275.);
	}
	if (fails) {
		lcpystr(mesg, "**Proposal is within coordination distance of Mexican border", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);
	}

	if (target->isParent) {
		for (i = 0; i < dtsCount; i++) {
			if (0 == i) {
				snprintf(mesg, MAX_STRING, "Distance to Mexican border: DTS site # %d   %.1f km", dtsSiteNumber[i],
					dtsMexDist[i]);
			} else {
				snprintf(mesg, MAX_STRING, "                            DTS site # %d   %.1f km", dtsSiteNumber[i],
					dtsMexDist[i]);
			}
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
		}
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fputc('\n', repFile);
	} else {
		snprintf(mesg, MAX_STRING, "Distance to Mexican border: %.1f km", minMexDist);
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
	}

	// Report if a Class A in the Mexican border zone does not use the full-service emission mask.

	if (fails && (SERV_TVCA == target->service) && (LPTV_MASK_FULL_SERVICE != target->emissionMaskKey)) {
		lcpystr(mesg, "**Class A proposal in Mexican border zone does not specify full-service emission mask",
			MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
	}

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Check distance and field strength to FCC monitoring stations.  The list of locations to check is in a CSV data file
// so this does not involve any database access, on the first call read the file.

static int check_monitor_station(SOURCE *target, FILE *repFile) {

	static struct monsta {
		char name[MAX_STRNGS];
		double latitude;
		double longitude;
		struct monsta *next;
	} *stationList = NULL;
	static int fileReadStatus = 0;

	struct monsta *station;
	int err = 0;

	if (!fileReadStatus) {

		char fname[MAX_STRING];

		snprintf(fname, MAX_STRING, "%s/monitor_station.csv", LIB_DIRECTORY_NAME);
		FILE *in = file_open(fname, "r");
		if (in) {

			struct monsta **link = &stationList;
			char line[MAX_STRING], *token;
			int deg, min;
			double sec;

			while (fgetnlc(line, MAX_STRING, in, NULL) >= 0) {

				token = next_token(line);
				if (!token) {
					err = 1;
					break;
				}

				station = (struct monsta *)mem_zalloc(sizeof(struct monsta));
				*link = station;
				link = &(station->next);

				lcpystr(station->name, token, MAX_STRNGS);

				err = token_atoi(next_token(NULL), -75, 75, &deg);
				if (err) break;
				err = token_atoi(next_token(NULL), 0, 59, &min);
				if (err) break;
				err = token_atof(next_token(NULL), 0, 60., &sec);
				if (err) break;
				station->latitude = (double)deg + ((double)min / 60.) + (sec / 3600.);

				err = token_atoi(next_token(NULL), -180, 180, &deg);
				if (err) break;
				err = token_atoi(next_token(NULL), 0, 59, &min);
				if (err) break;
				err = token_atof(next_token(NULL), 0, 60., &sec);
				if (err) break;
				station->longitude = (double)deg + ((double)min / 60.) + (sec / 3600.);
			}

			file_close(in);

			if (!stationList) {
				err = 1;
			}

		} else {
			err = 1;
		}

		if (err) {
			while (stationList) {
				station = stationList;
				stationList = station->next;
				station->next = NULL;
				mem_free(station);
			}
			fileReadStatus = -1;
		} else {
			fileReadStatus = 1;
		}
	}

	// If the file read failed report it but still return success, this does not abort the run.

	char mesg[MAX_STRING];

	if (fileReadStatus < 0) {
		lcpystr(mesg, "Unable to check FCC monitoring stations, error reading data file", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
		return 0;
	}

	// Check stations, report all that fail distance or field strength checks (the FCC stations are probably never
	// close enough for failures to more than one station, but to be safe that is not assumed).  If no failures are
	// found, just report conditions at the nearest station.  A one-off point structure is created at the station
	// location, then field strength(s) are computed for nearby transmitter(s), using FCC curves method.  For a DTS
	// there will be multiple fields.  All transmitters that fail the checks are reported.  If the distance check
	// fails the field strength conditions are also reported even if those do not fail the check.

	POINT *point, *nearestPoint = NULL;
	FIELD *field;
	SOURCE *source;
	char *nearestName = NULL, erpkw[MAX_STRING], erpkwVpat[MAX_STRING];
	double bear, dist, distInc, *haat, *erp, *vpat, *nearestHAAT, *nearestERP, *nearestVpat, sigdbu, sigmvm, minDist,
		nearestDist = 99999.;
	int curv, fieldCount, fieldIndex, failsDist, failsSig, didFail = 0, showFailsDist = 1, showFailsSig = 1;

	int countryIndex = target->countryKey - 1;

	distInc = 1. / Params.TerrAvgPpk[countryIndex];
	if (target->dtv) {
		curv = Params.CurveSetDigital[countryIndex];
	} else {
		curv = Params.CurveSetAnalog[countryIndex];
	}

	// This needs to use FCC curves with HAAT calculated by the contour-projection method, so it can't just use the
	// propagation model code.  Still using point and field structures for convenience and to hold results for later
	// printout, but also need local arrays to hold HAAT, ERP, and vertical pattern adjustments.

	if (target->isParent) {
		fieldCount = 0;
		for (source = target->dtsSources; source; source = source->next) fieldCount++;
	} else {
		fieldCount = 1;
	}
	haat = (double *)mem_alloc((6 * fieldCount) * sizeof(double));
	erp = haat + fieldCount;
	vpat = erp + fieldCount;
	nearestHAAT = vpat + fieldCount;
	nearestERP = nearestHAAT + fieldCount;
	nearestVpat = nearestERP + fieldCount;

	for (station = stationList; station; station = station->next) {

		point = make_point(station->latitude, station->longitude, target);
		if (!point) {
			if (nearestPoint) {
				free_point(nearestPoint);
			}
			mem_free(haat);
			return -1;
		}

		minDist = 99999.;

		failsDist = 0;
		failsSig = 0;

		for (field = point->fields, fieldIndex = 0; field; field = field->next, fieldIndex++) {

			source = SourceKeyIndex[field->sourceKey];
			if (!source) {
				log_error("Source structure index is corrupted");
				exit(1);
			}

			bear = field->bearing;
			dist = field->distance;
			erp[fieldIndex] = erp_lookup(source, bear);
			if (dist < minDist) {
				minDist = dist;
			}
			if ((dist < 2.4) || ((dist < 4.8) && (erp[fieldIndex] > -13.01)) ||
					((dist < 16.) && (erp[fieldIndex] > 0.)) || ((dist < 80.) && (erp[fieldIndex] > 13.98))) {
				failsDist = 1;
				didFail = 1;
			}

			if (failsDist || (dist < Params.MaximumDistance)) {

				err = compute_haat_radial(source->latitude, source->longitude, bear, source->actualHeightAMSL,
					(haat + fieldIndex), Params.TerrAvgDb, Params.AVETStartDistance[countryIndex],
					Params.AVETEndDistance[countryIndex], distInc, Params.KilometersPerDegree, NULL);
				if (err) {
					log_error("Terrain lookup failed: db=%d err=%d", Params.TerrAvgDb, err);
					free_point(point);
					if (nearestPoint) {
						free_point(nearestPoint);
					}
					mem_free(haat);
					return err;
				}

				fcc_curve((erp + fieldIndex), &sigdbu, &dist, haat[fieldIndex], source->band, FCC_FLD, curv,
					Params.OffCurveLookupMethod, source, bear, NULL, NULL, (vpat + fieldIndex));
				field->fieldStrength = (float)sigdbu;
				field->status = 0;

				if (sigdbu > 80.) {
					failsSig = 1;
					didFail = 1;
				}
			}
		}

		if (failsDist || failsSig) {

			if (failsDist && showFailsDist) {
				lcpystr(mesg, "**Proposal is within coordination distance of FCC monitoring station", MAX_STRING);
				status_message(STATUS_KEY_REPORT, mesg);
				if (repFile) fprintf(repFile, "%s\n", mesg);
				showFailsDist = 0;
			}

			if (failsSig && showFailsSig) {
				lcpystr(mesg, "**Proposal exceeds field strength limit at FCC monitoring station", MAX_STRING);
				status_message(STATUS_KEY_REPORT, mesg);
				if (repFile) fprintf(repFile, "%s\n", mesg);
				showFailsSig = 0;
			}

			snprintf(mesg, MAX_STRING, "Conditions at FCC monitoring station: %s", station->name);
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);

			for (field = point->fields, fieldIndex = 0; field; field = field->next, fieldIndex++) {

				source = SourceKeyIndex[field->sourceKey];
				if (!source) {
					log_error("Source structure index is corrupted");
					exit(1);
				}

				bear = field->bearing;
				dist = field->distance;
				if (source->parentSource) {
					snprintf(mesg, MAX_STRING, "DTS site # %d   Bearing: %.1f degrees   Distance: %.1f km",
						source->siteNumber, bear, dist);
				} else {
					snprintf(mesg, MAX_STRING, "Bearing: %.1f degrees   Distance: %.1f km", bear, dist);
				}
				status_message(STATUS_KEY_REPORT, mesg);
				if (repFile) fprintf(repFile, "%s\n", mesg);

				if (field->status >= 0) {
					sigdbu = field->fieldStrength;
					sigmvm = pow(10., (sigdbu / 20.)) / 1000.;
					lcpystr(erpkw, erpkw_string(erp[fieldIndex]), MAX_STRING);
					if (vpat[fieldIndex] != 0.) {
						lcpystr(erpkwVpat, erpkw_string(erp[fieldIndex] + vpat[fieldIndex]), MAX_STRING);
						snprintf(mesg, MAX_STRING,
							"ERP: %s kW  Adj. ERP: %s kW  HAAT: %.1f m  Field strength: %.1f dBu, %.1f mV/m",
							erpkw, erpkwVpat, haat[fieldIndex], sigdbu, sigmvm);
					} else {
						snprintf(mesg, MAX_STRING, "ERP: %s kW  HAAT: %.1f m  Field strength: %.1f dBu, %.1f mV/m",
							erpkw, haat[fieldIndex], sigdbu, sigmvm);
					}
					status_message(STATUS_KEY_REPORT, mesg);
					if (repFile) fprintf(repFile, "%s\n", mesg);
				}
			}
		}

		if (minDist < nearestDist) {
			nearestDist = minDist;
			nearestName = station->name;
			if (nearestPoint) {
				free_point(nearestPoint);
			}
			nearestPoint = point;
			for (fieldIndex = 0; fieldIndex < fieldCount; fieldIndex++) {
				nearestHAAT[fieldIndex] = haat[fieldIndex];
				nearestERP[fieldIndex] = erp[fieldIndex];
				nearestVpat[fieldIndex] = vpat[fieldIndex];
			}
		} else {
			free_point(point);
		}
		point = NULL;
	}

	if (!didFail) {

		snprintf(mesg, MAX_STRING, "Conditions at FCC monitoring station: %s", nearestName);
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		for (field = nearestPoint->fields, fieldIndex = 0; field; field = field->next, fieldIndex++) {

			source = SourceKeyIndex[field->sourceKey];
			if (!source) {
				log_error("Source structure index is corrupted");
				exit(1);
			}

			bear = field->bearing;
			dist = field->distance;
			if (source->parentSource) {
				snprintf(mesg, MAX_STRING, "DTS site # %d   Bearing: %.1f degrees   Distance: %.1f km",
					source->siteNumber, bear, dist);
			} else {
				snprintf(mesg, MAX_STRING, "Bearing: %.1f degrees   Distance: %.1f km", bear, dist);
			}
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);

			if (field->status >= 0) {
				sigdbu = field->fieldStrength;
				sigmvm = pow(10., (sigdbu / 20.)) / 1000.;
				lcpystr(erpkw, erpkw_string(nearestERP[fieldIndex]), MAX_STRING);
				if (nearestVpat[fieldIndex] != 0.) {
					lcpystr(erpkwVpat, erpkw_string(nearestERP[fieldIndex] + nearestVpat[fieldIndex]), MAX_STRING);
					snprintf(mesg, MAX_STRING,
						"ERP: %s kW  Adj. ERP: %s kW  HAAT: %.1f m  Field strength: %.1f dBu, %.1f mV/m",
						erpkw, erpkwVpat, nearestHAAT[fieldIndex], sigdbu, sigmvm);
				} else {
					snprintf(mesg, MAX_STRING, "ERP: %s kW  HAAT: %.1f m  Field strength: %.1f dBu, %.1f mV/m",
						erpkw, nearestHAAT[fieldIndex], sigdbu, sigmvm);
				}
				status_message(STATUS_KEY_REPORT, mesg);
				if (repFile) fprintf(repFile, "%s\n", mesg);
			}
		}
	}

	status_message(STATUS_KEY_REPORT, "");
	if (repFile) fputc('\n', repFile);

	free_point(nearestPoint);
	mem_free(haat);
	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Check target versus the West Virgina quiet zone area.  This is a simple boundary box check.

static int check_va_quiet_zone(SOURCE *target, FILE *repFile) {

	SOURCE *dtsSource;
	int fail;
	char mesg[MAX_STRING];

	if (target->dts) {

		for (dtsSource = target->dtsSources; dtsSource; dtsSource = dtsSource->next) {
			fail = ((dtsSource->latitude >= 37.5) && (dtsSource->latitude <= 39.25) &&
				(dtsSource->longitude >= 78.5) && (dtsSource->longitude <= 80.5));
			if (fail) {
				break;
			}
		}

	} else {

		fail = ((target->latitude >= 37.5) && (target->latitude <= 39.25) &&
			(target->longitude >= 78.5) && (target->longitude <= 80.5));
	}

	if (fail) {
		lcpystr(mesg, "**Proposal is within the West Virginia quiet zone area", MAX_STRING);
	} else {
		lcpystr(mesg, "Proposal is not within the West Virginia quiet zone area", MAX_STRING);
	}
	status_message(STATUS_KEY_REPORT, mesg);
	status_message(STATUS_KEY_REPORT, "");
	if (repFile) fprintf(repFile, "%s\n\n", mesg);

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Check the Table Mountain receiving zone, test distance and field strength at the four corners of the area.  The
// approach is similar to the monitoring station check, except that only one corner point with worst-case conditions
// is reported, that is the point with highest field strength if those are projected, else it is the closest point.

static int check_table_mountain(SOURCE *target, FILE *repFile) {

	static double tmLat[2] = {40.11805556, 40.15277778};
	static double tmLon[2] = {105.2252778, 105.2536111};

	POINT *point, *worstPoint = NULL;
	FIELD *field;
	SOURCE *source;
	double lat, lon, bear, dist, erp, sigdbu, sigmvm, minDist, maxSig, worstDist = 99999.,
		worstSig = FIELD_NOT_CALCULATED;
	int err, ilat, ilon, failsDist, didFailDist = 0, didFailSig = 0;
	char mesg[MAX_STRING];

	for (ilat = 0; ilat < 2; ilat++) {
		for (ilon = 0; ilon < 2; ilon++) {

			lat = tmLat[ilat];
			lon = tmLon[ilon];

			point = make_point(lat, lon, target);
			if (!point) return -1;

			minDist = 99999.;
			maxSig = FIELD_NOT_CALCULATED;

			for (field = point->fields; field; field = field->next) {

				source = SourceKeyIndex[field->sourceKey];
				if (!source) {
					log_error("Source structure index is corrupted");
					exit(1);
				}

				dist = field->distance;
				bear = field->bearing;
				erp = erp_lookup(source, bear);

				if (dist < minDist) {
					minDist = dist;
				}

				failsDist = 0;

				if ((dist < 2.4) || ((dist < 4.8) && (erp > -13.01)) || ((dist < 16.) && (erp > 0.)) ||
						((dist < 80.) && (erp > 13.98))) {
					failsDist = 1;
					didFailDist = 1;
				}

				if (failsDist || (dist <= Params.MaximumDistance)) {

					err = project_field(point, field);
					if (err) {
						free_point(point);
						return err;
					}

					sigdbu = field->fieldStrength;
					sigmvm = pow(10., (sigdbu / 20.)) / 1000.;

					if (sigdbu > maxSig) {
						maxSig = sigdbu;
					}

					if (BAND_UHF == target->band) {
						if (sigdbu > 89.5) {
							didFailSig = 1;
						}
					} else {
						if (sigdbu > 80.) {
							didFailSig = 1;
						}
					}
				}
			}

			if ((maxSig > worstSig) || ((FIELD_NOT_CALCULATED == worstSig) && (minDist < worstDist))) {
				worstDist = minDist;
				worstSig = maxSig;
				if (worstPoint) {
					free_point(worstPoint);
					worstPoint = NULL;
				}
				worstPoint = point;
			} else {
				free_point(point);
			}
			point = NULL;
		}
	}

	if (didFailDist) {
		lcpystr(mesg, "**Proposal is within coordination distance of Table Mountain receiving zone", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);
	}

	if (didFailSig) {
		lcpystr(mesg, "**Proposal exceeds field strength limit at Table Mountain receiving zone", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);
	}

	lcpystr(mesg, "Conditions at Table Mountain receiving zone:", MAX_STRING);
	status_message(STATUS_KEY_REPORT, mesg);
	if (repFile) fprintf(repFile, "%s\n", mesg);

	for (field = worstPoint->fields; field; field = field->next) {

		source = SourceKeyIndex[field->sourceKey];
		if (!source) {
			log_error("Source structure index is corrupted");
			exit(1);
		}

		bear = field->bearing;
		dist = field->distance;
		erp = erp_lookup(source, bear);

		if (source->parentSource) {
			snprintf(mesg, MAX_STRING, "DTS site # %d   Bearing: %.1f degrees   Distance: %.1f km",
				source->siteNumber, bear, dist);
		} else {
			snprintf(mesg, MAX_STRING, "Bearing: %.1f degrees   Distance: %.1f km", bear, dist);
		}
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);

		if (field->status >= 0) {

			sigdbu = field->fieldStrength;
			sigmvm = pow(10., (sigdbu / 20.)) / 1000.;

			snprintf(mesg, MAX_STRING, "ERP: %s kW   Field strength: %.1f dBu, %.1f mV/m", erpkw_string(erp), sigdbu,
				sigmvm);
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
		}
	}

	free_point(worstPoint);

	status_message(STATUS_KEY_REPORT, "");
	if (repFile) fputc('\n', repFile);

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Check for protection to land mobile stations.  This reads various land mobile city and coordinate lists and test
// conditions from data files, read those on the first call.  The main stationList is the set of station locations to
// be protected per 73.617(a) and 74.709(a) (the lists in those two sections are nominally the same, however the
// coordinates of some locations vary slightly so there may be multiple entries in the data files here).  A list of
// waiver locations is also loaded, that data is provided and updated by FCC staff as needed.  Each entry can define a
// full-service distance check and/or a Class A/LPTV/translator distance and contour overlap check, optionally with
// exclusions to the land-mobile protected area as defined in 74.709(b) (currently the exclusions do not apply to the
// waiver list).  The exclusion list is read from a separate file.  All distances and contour levels are provided in
// the files.  This only applies to channel 14-20 according to the rules, however channel 21 is included as it may be
// adjacent to channel 20 land mobile.  In that case any match is reported as an advisory message not a failure.

static int check_land_mobile(SOURCE *target, FILE *repFile) {

	if ((target->channel < 14) || (target->channel > 21)) {
		return 0;
	}

	static struct lanmob {
		char name[MAX_STRNGS];
		int channel;
		double latitude;
		double longitude;
		double fsCoDistance;
		double fsAdjDistance;
		double lpCoDistance;
		double lpAdjDistance;
		double lpCoContourLevel;
		double lpAdjContourLevel;
		double lpCoExcludeDistance;
		double lpAdjExcludeDistance;
		struct lanmob *next;
	} *stationList = NULL, *waiverList = NULL, *excludeList = NULL;
	static int fileReadStatus = 0;

	struct lanmob *list, *station, *exclude;
	int err = 0;

	if (!fileReadStatus) {

		FILE *in = NULL;
		struct lanmob **link;
		char fname[MAX_STRING], line[MAX_STRING], *token;
		int deg, min, fi;
		double sec;

		for (fi = 0; fi < 3; fi++) {

			switch (fi) {
				case 0: {
					link = &stationList;
					snprintf(fname, MAX_STRING, "%s/land_mobile.csv", LIB_DIRECTORY_NAME);
					in = file_open(fname, "r");
					break;
				}
				case 1: {
					link = &waiverList;
					snprintf(fname, MAX_STRING, "%s/land_mobile_waiver.csv", LIB_DIRECTORY_NAME);
					in = file_open(fname, "r");
					break;
				}
				case 2: {
					link = &excludeList;
					snprintf(fname, MAX_STRING, "%s/land_mobile_exclude.csv", LIB_DIRECTORY_NAME);
					in = file_open(fname, "r");
					break;
				}
			}

			if (!in) {
				err = 1;
				break;
			}

			while (fgetnlc(line, MAX_STRING, in, NULL) >= 0) {

				token = next_token(line);
				if (!token) {
					err = 1;
					break;
				}

				station = (struct lanmob *)mem_zalloc(sizeof(struct lanmob));
				*link = station;
				link = &(station->next);

				lcpystr(station->name, token, MAX_STRNGS);

				err = token_atoi(next_token(NULL), 2, 51, &(station->channel));
				if (err) break;

				err = token_atoi(next_token(NULL), -75, 75, &deg);
				if (err) break;
				err = token_atoi(next_token(NULL), 0, 59, &min);
				if (err) break;
				err = token_atof(next_token(NULL), 0, 60., &sec);
				if (err) break;
				station->latitude = (double)deg + ((double)min / 60.) + (sec / 3600.);

				err = token_atoi(next_token(NULL), -180, 180, &deg);
				if (err) break;
				err = token_atoi(next_token(NULL), 0, 59, &min);
				if (err) break;
				err = token_atof(next_token(NULL), 0, 60., &sec);
				if (err) break;
				station->longitude = (double)deg + ((double)min / 60.) + (sec / 3600.);

				if ((token = next_token(NULL))) {
					err = token_atof(token, 0., 500., &(station->fsCoDistance));
					if (err) break;
				}
				if ((token = next_token(NULL))) {
					err = token_atof(token, 0., 500., &(station->fsAdjDistance));
					if (err) break;
				}

				if ((token = next_token(NULL))) {
					err = token_atof(token, 0., 500., &(station->lpCoDistance));
					if (err) break;
				}
				if ((token = next_token(NULL))) {
					err = token_atof(token, 0., 500., &(station->lpAdjDistance));
					if (err) break;
				}
				if ((token = next_token(NULL))) {
					err = token_atof(token, 0., 110., &(station->lpCoContourLevel));
					if (err) break;
				}
				if ((token = next_token(NULL))) {
					err = token_atof(token, 0., 110., &(station->lpAdjContourLevel));
					if (err) break;
				}
				if ((token = next_token(NULL))) {
					err = token_atof(token, 0., 500., &(station->lpCoExcludeDistance));
					if (err) break;
				}
				if ((token = next_token(NULL))) {
					err = token_atof(token, 0., 500., &(station->lpAdjExcludeDistance));
					if (err) break;
				}
			}

			file_close(in);
			if (err) break;
		}

		if (err) {
			while (stationList) {
				station = stationList;
				stationList = station->next;
				station->next = NULL;
				mem_free(station);
			}
			while (waiverList) {
				station = waiverList;
				waiverList = station->next;
				station->next = NULL;
				mem_free(station);
			}
			while (excludeList) {
				exclude = excludeList;
				excludeList = exclude->next;
				exclude->next = NULL;
				mem_free(exclude);
			}
			fileReadStatus = -1;
		} else {
			fileReadStatus = 1;
		}
	}

	// If any file reads failed report it but still return success, this does not abort the run.

	char mesg[MAX_STRING];

	if (fileReadStatus < 0) {
		lcpystr(mesg, "Unable to check land mobile stations, error reading data file", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
		return 0;
	}

	// Do the checks.  The full-service distance check only applies if the target is full-service of course, but
	// entries in the list may also not have the full-service check applied at all if the distances were not defined
	// in the file.  Likewise the Class A/LPTV/translator contour-overlap check only applies if parameters are in
	// the file.  Also within the overlap check, the exclusions may or may not be applied depending on whether the
	// exclusion distances are in the file.  Note the exclusion distance is a radius around the exclusion points, but
	// those distances are a property of the protected station not the exclusion; per 74.709(b) there are variations
	// in that distance depending on which protected station is being considered.

	GEOPOINTS *coContours = NULL, *adjContours = NULL, **linkContour, *chkContours, *contour;
	CONTOUR *srcContour;
	SOURCE *sources, *source;
	double minDist, exclDist, contLevel, dist, failDist, coContourLevel, adjContourLevel;
	int li, delta, exclDelta, i, fails, didFail, didWarn;
	char *testLabel;

	for (li = 0; li < 2; li++) {

		switch (li) {
			case 0: {
				list = stationList;
				if (SERV_TV == target->service) {
					testLabel = "73.617(a)";
				} else {
					testLabel = "74.709(a)";
				}
				break;
			}
			case 1: {
				list = waiverList;
				testLabel = "waiver";
				break;
			}
		}

		didFail = 0;
		didWarn = 0;

		for (station = list; station; station = station->next) {

			delta = abs(target->channel - station->channel);
			if (delta > 1) {
				continue;
			}

			if (SERV_TV == target->service) {
				if (0. == station->fsCoDistance) {
					continue;
				}
				if (0 == delta) {
					minDist = station->fsCoDistance;
				} else {
					minDist = station->fsAdjDistance;
				}
				exclDist = 0.;
				contLevel = 0.;
			} else {
				if (0. == station->lpCoDistance) {
					continue;
				}
				if (0 == delta) {
					minDist = station->lpCoDistance;
					contLevel = station->lpCoContourLevel;
				} else {
					minDist = station->lpAdjDistance;
					contLevel = station->lpAdjContourLevel;
				}
			}

			// Do the distance check.  For a DTS each transmitter point has to be fully checked not just the closest,
			// due to the exclusions the closest point could pass but the next-closest still fail.  Unlikely but
			// possible.  However once a failure is found the check does not continue, this does not report distances
			// to every DTS transmitter, just the first one that fails.

			fails = 0;
			failDist = 0.;

			if (target->isParent) {
				sources = target->dtsSources;
			} else {
				sources = target;
				target->next = NULL;
			}

			for (source = sources; source; source = source->next) {
				bear_distance(station->latitude, station->longitude, source->latitude, source->longitude, NULL, NULL,
					&dist, Params.KilometersPerDegree);
				if (dist <= minDist) {
					fails = 1;
					failDist = dist;
					if ((SERV_TV != target->service) && (station->lpCoExcludeDistance > 0.)) {
						for (exclude = excludeList; exclude; exclude = exclude->next) {
							exclDelta = abs(station->channel - exclude->channel);
							if (exclDelta > 1) {
								continue;
							}
							if (0 == exclDelta) {
								exclDist = station->lpCoExcludeDistance;
							} else {
								exclDist = station->lpAdjExcludeDistance;
							}
							bear_distance(exclude->latitude, exclude->longitude, source->latitude, source->longitude,
								NULL, NULL, &dist, Params.KilometersPerDegree);
							if (dist <= exclDist) {
								fails = 0;
								break;
							}
						}
					}
				}
				if (fails) {
					break;
				}
			}

			// If the distance check fails, any contour check can be skipped since it would obviously fail too.  If the
			// proposal is on channel 21 this is advisory only, not a failure.

			if (fails) {
				if (target->channel < 21) {
					if (!didFail) {
						snprintf(mesg, MAX_STRING, "**Proposal fails land mobile %s test:", testLabel);
						status_message(STATUS_KEY_REPORT, mesg);
						if (repFile) fprintf(repFile, "%s\n", mesg);
						didFail = 1;
					}
					snprintf(mesg, MAX_STRING, "**  Short-spaced to station: %s ch. %d, %.1f km", station->name,
						station->channel, failDist);
					status_message(STATUS_KEY_REPORT, mesg);
					if (repFile) fprintf(repFile, "%s\n", mesg);
				} else {
					snprintf(mesg, MAX_STRING, "Proposal is short-spaced to land mobile station: %s ch. %d, %.1f km",
						station->name, station->channel, failDist);
					status_message(STATUS_KEY_REPORT, mesg);
					if (repFile) fprintf(repFile, "%s\n", mesg);
					didWarn = 1;
				}
				continue;
			}

			// Proceed to contour overlap if applicable and if the target is close enough for possible overlap; see
			// discussion in set_rule_extra_distance() in source.c.

			if (0. == contLevel) {
				continue;
			}

			for (source = sources; source; source = source->next) {
				bear_distance(station->latitude, station->longitude, source->latitude, source->longitude, NULL, NULL,
					&dist, Params.KilometersPerDegree);
				if (dist <= (minDist + source->ruleExtraDistance)) {
					break;
				}
			}
			if (!source) {
				continue;
			}

			// Project contours from the target station as needed.  All contours are projected from a DTS regardless of
			// the individual distance because once projected these may be re-used for the next land mobile station
			// needing this check.  Generally the same contour levels will apply to all but if the level does change,
			// discard the existing contours.  If there are contour changes the data file should be ordered to minimize
			// re-projection.

			if (0 == delta) {

				linkContour = &coContours;
				if (*linkContour && (coContourLevel != contLevel)) {
					while (coContours) {
						contour = coContours;
						coContours = contour->next;
						contour->next = NULL;
						geopoints_free(contour);
					}
				}

				coContourLevel = contLevel;

			} else {

				linkContour = &adjContours;
				if (*linkContour && (adjContourLevel != contLevel)) {
					while (adjContours) {
						contour = adjContours;
						adjContours = contour->next;
						contour->next = NULL;
						geopoints_free(contour);
					}
				}

				adjContourLevel = contLevel;
			}

			if (!(*linkContour)) {
				for (source = sources; source; source = source->next) {
					srcContour = project_fcc_contour(source, FCC_F10, contLevel);
					if (!srcContour) return -1;
					*linkContour = render_contour(srcContour, Params.KilometersPerDegree);
					linkContour = &((*linkContour)->next);
					srcContour->points = NULL;
					contour_free(srcContour);
					srcContour = NULL;
				}
			}

			if (0 == delta) {
				chkContours = coContours;
			} else {
				chkContours = adjContours;
			}

			for (contour = chkContours; contour; contour = contour->next) {
				for (i = 0; i < contour->nPts; i++) {
					bear_distance(station->latitude, station->longitude, contour->ptLat[i], contour->ptLon[i], NULL,
						NULL, &dist, Params.KilometersPerDegree);
					if (dist <= minDist) {
						fails = 1;
						if ((SERV_TV != target->service) && (station->lpCoExcludeDistance > 0.)) {
							for (exclude = excludeList; exclude; exclude = exclude->next) {
								exclDelta = abs(station->channel - exclude->channel);
								if (exclDelta > 1) {
									continue;
								}
								if (0 == exclDelta) {
									exclDist = station->lpCoExcludeDistance;
								} else {
									exclDist = station->lpAdjExcludeDistance;
								}
								bear_distance(exclude->latitude, exclude->longitude, contour->ptLat[i],
									contour->ptLon[i], NULL, NULL, &dist, Params.KilometersPerDegree);
								if (dist <= exclDist) {
									fails = 0;
									break;
								}
							}
						}
					}
					if (fails) {
						break;
					}
				}
				if (fails) {
					break;
				}
			}

			if (fails) {
				if (target->channel < 21) {
					if (!didFail) {
						snprintf(mesg, MAX_STRING, "**Proposal fails land mobile %s test:", testLabel);
						status_message(STATUS_KEY_REPORT, mesg);
						if (repFile) fprintf(repFile, "%s\n", mesg);
						didFail = 1;
					}
					snprintf(mesg, MAX_STRING, "**  Contour overlap with station: %s ch. %d", station->name,
						station->channel);
					status_message(STATUS_KEY_REPORT, mesg);
					if (repFile) fprintf(repFile, "%s\n", mesg);
				} else {
					snprintf(mesg, MAX_STRING, "Proposal has contour overlap with land mobile station: %s ch. %d",
						station->name, station->channel);
					status_message(STATUS_KEY_REPORT, mesg);
					if (repFile) fprintf(repFile, "%s\n", mesg);
					didWarn = 1;
				}
			}
		}

		if (didFail) {
			status_message(STATUS_KEY_REPORT, "");
			if (repFile) fputc('\n', repFile);
		} else {
			if (target->channel < 21) {
				snprintf(mesg, MAX_STRING, "Proposal passes land mobile %s test", testLabel);
				status_message(STATUS_KEY_REPORT, mesg);
				status_message(STATUS_KEY_REPORT, "");
				if (repFile) fprintf(repFile, "%s\n\n", mesg);
			} else {
				if (didWarn) {
					status_message(STATUS_KEY_REPORT, "");
					if (repFile) fputc('\n', repFile);
				}
			}
		}
	}

	while (coContours) {
		contour = coContours;
		coContours = contour->next;
		contour->next = NULL;
		geopoints_free(contour);
	}
	while (adjContours) {
		contour = adjContours;
		adjContours = contour->next;
		contour->next = NULL;
		geopoints_free(contour);
	}
	contour = NULL;

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Check target versus the offshore radio service protections in 74.709(e).  Only for Class A/LPTV/translator.  This
// is a set of geographic limit checks, the target cannot be between certain ranges of longitude and south of a line
// of latitude which may not be directly east-west so the latitude limit is interpolated.

static int check_offshore_radio(SOURCE *target, FILE *repFile) {

	if (SERV_TV == target->service) {
		return 0;
	}

	double lat1, lon1, lat2, lon2, lat3, lon3;

	switch (target->channel) {
		case 15: {
			lat1 = 30.5;
			lon1 = 92.0;
			lat2 = 30.5;
			lon2 = 96.0;
			lat3 = 28.0;
			lon3 = 98.5;
			break;
		}
		case 16: {
			lat1 = 31.0;
			lon1 = 86.6666666666667;
			lat2 = 31.0;
			lon2 = 95.0;
			lat3 = 29.5;
			lon3 = 96.5;
			break;
		}
		case 17: {
			lat1 = 31.0;
			lon1 = 86.5;
			lat2 = 31.5;
			lon2 = 94.0;
			lat3 = 29.5;
			lon3 = 96.0;
			break;
		}
		case 18: {
			lat1 = 31.0;
			lon1 = 87.0;
			lat2 = 31.0;
			lon2 = 91.0;
			lat3 = 31.0;
			lon3 = 95.0;
			break;
		}
		default: {
			return 0;
		}
	}

	SOURCE *source, *sources;
	double checkLat;
	int fail = 0;
	char mesg[MAX_STRING];

	if (target->isParent) {
		sources = target->dtsSources;
	} else {
		sources = target;
		target->next = NULL;
	}

	for (source = sources; source; source = source->next) {
		if ((source->longitude >= lon1) && (source->longitude <= lon2)) {
			checkLat = lat1 + ((lat2 - lat1) * ((source->longitude - lon1) / (lon2 - lon1)));
			if (source->latitude <= checkLat) {
				fail = 1;
				break;
			}
		} else {
			if ((source->longitude >= lon2) && (source->longitude <= lon3)) {
				checkLat = lat2 + ((lat3 - lat2) * ((source->longitude - lon2) / (lon3 - lon2)));
				if (source->latitude <= checkLat) {
					fail = 1;
					break;
				}
			}
		}
	}

	if (fail) {
		lcpystr(mesg, "**Proposal is within the Offshore Radio Service protected area", MAX_STRING);
	} else {
		lcpystr(mesg, "Proposal is not within the Offshore Radio Service protected area", MAX_STRING);
	}
	status_message(STATUS_KEY_REPORT, mesg);
	status_message(STATUS_KEY_REPORT, "");
	if (repFile) fprintf(repFile, "%s\n\n", mesg);

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Check distance to a list of top market cities from the rural LPTV filing window order.  Read coordinates from a CSV
// file.  For LPTV proposals only.  Currently this will never fail and the check is always advisory.

static int check_top_markets(SOURCE *target, FILE *repFile) {

	if (SERV_TVLP != target->service) {
		return 0;
	}

	static struct mktloc {
		char name[MAX_STRNGS];
		double latitude;
		double longitude;
		struct mktloc *next;
	} *marketList = NULL;
	static int fileReadStatus = 0;

	struct mktloc *market;
	int err = 0;

	if (!fileReadStatus) {

		char fname[MAX_STRING];

		snprintf(fname, MAX_STRING, "%s/top_markets.csv", LIB_DIRECTORY_NAME);
		FILE *in = file_open(fname, "r");
		if (in) {

			struct mktloc **link = &marketList;
			char line[MAX_STRING], *token, *c;
			int deg, min;
			double sec;

			while (fgetnlc(line, MAX_STRING, in, NULL) >= 0) {

				token = next_token(line);
				if (!token) {
					err = 1;
					break;
				}

				market = (struct mktloc *)mem_zalloc(sizeof(struct mktloc));
				*link = market;
				link = &(market->next);

				// Commas in names were replaced with ^ in the file because it's CSV; change those back.

				for (c = token; *c != '\0'; c++) {
					if ('^' == *c) {
						*c = ',';
					}
				}
				lcpystr(market->name, token, MAX_STRNGS);

				err = token_atoi(next_token(NULL), -75, 75, &deg);
				if (err) break;
				err = token_atoi(next_token(NULL), 0, 59, &min);
				if (err) break;
				err = token_atof(next_token(NULL), 0, 60., &sec);
				if (err) break;
				market->latitude = (double)deg + ((double)min / 60.) + (sec / 3600.);

				err = token_atoi(next_token(NULL), -180, 180, &deg);
				if (err) break;
				err = token_atoi(next_token(NULL), 0, 59, &min);
				if (err) break;
				err = token_atof(next_token(NULL), 0, 60., &sec);
				if (err) break;
				market->longitude = (double)deg + ((double)min / 60.) + (sec / 3600.);
			}

			file_close(in);

			if (!marketList) {
				err = 1;
			}

		} else {
			err = 1;
		}

		if (err) {
			while (marketList) {
				market = marketList;
				marketList = market->next;
				market->next = NULL;
				mem_free(market);
			}
			fileReadStatus = -1;
		} else {
			fileReadStatus = 1;
		}
	}

	// If the file read failed report it.

	char mesg[MAX_STRING];

	if (fileReadStatus < 0) {
		lcpystr(mesg, "Unable to check top market distances, error reading data file", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		status_message(STATUS_KEY_REPORT, "");
		if (repFile) fprintf(repFile, "%s\n\n", mesg);
		return 0;
	}

	// Check markets, report any that are closer than 121 km, or report that none are.

	double dist;
	int firstHit = 1;

	for (market = marketList; market; market = market->next) {
		bear_distance(target->latitude, target->longitude, market->latitude, market->longitude, NULL, NULL, &dist,
			Params.KilometersPerDegree);
		if (dist < 121.) {
			if (firstHit) {
				lcpystr(mesg, "Proposal is within 121 km of the following top 100 markets:", MAX_STRING);
				status_message(STATUS_KEY_REPORT, mesg);
				if (repFile) fprintf(repFile, "%s\n", mesg);
				firstHit = 0;
			}
			snprintf(mesg, MAX_STRING, "%s, %.1f km", market->name, dist);
			status_message(STATUS_KEY_REPORT, mesg);
			if (repFile) fprintf(repFile, "%s\n", mesg);
		}
	}

	if (firstHit) {
		lcpystr(mesg, "Proposal is not within 121 km of any top 100 market", MAX_STRING);
		status_message(STATUS_KEY_REPORT, mesg);
		if (repFile) fprintf(repFile, "%s\n", mesg);
	}

	status_message(STATUS_KEY_REPORT, "");
	if (repFile) fputc('\n', repFile);

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Do a run in "probe" mode, no output files are created, results are reported via status messages.  This is used by
// the front-end app to get information about actual interference relationships, e.g. during an IX check study build
// to identify records that actually have to be considered.  However this can be run for any type of study.  All
// default type scenarios are run, for each desired-undesired pair a message is logged providing desired source key,
// undesired source key, a flag indicating if the interference area is non-zero, and the interference population.

// Return is <0 for major error, >0 for minor error, 0 for no error.

int run_probe() {

	if (!StudyKey) {
		log_error("run_probe() called with no study open");
		return 1;
	}

	// Make sure no output files will be generated, even if arguments requested those.  Also no result tables, and
	// status messages must be on since that is the only output.

	parse_flags("", OutputFlags, MAX_OUTPUT_FLAGS);
	OutputFlagsSet = 1;
	parse_flags("", MapOutputFlags, MAX_MAP_OUTPUT_FLAGS);
	MapOutputFlagsSet = 1;
	CreateResultTables = 0;
	set_status_enabled(1);

	int err = 0;

	// Query a list of scenario keys to run.

	char query[MAX_QUERY];
	MYSQL_RES *myResult;
	my_ulonglong rowCount;
	MYSQL_ROW fields;

	int *scenarios = NULL, scenarioCount = 0, i;
	char mesg[MAX_STRING];

	snprintf(query, MAX_QUERY, "SELECT scenario_key FROM %s_%d.scenario WHERE scenario_type = 1 ORDER BY 1;",
		DbName, StudyKey);
	if (mysql_query(MyConnection, query)) {
		log_db_error("Scenario list query failed (1)");
		err = -1;

	} else {
		myResult = mysql_store_result(MyConnection);
		if (!myResult) {
			log_db_error("Scenario list query failed (2)");
			err = -1;

		} else {
			rowCount = mysql_num_rows(myResult);

			if (rowCount) {

				scenarioCount = (int)rowCount;
				scenarios = (int *)mem_alloc(scenarioCount * sizeof(int));
				for (i = 0; i < scenarioCount; i++) {

					fields = mysql_fetch_row(myResult);
					if (!fields) {
						log_db_error("Scenario list query failed (3)");
						err = -1;
						break;
					}

					scenarios[i] = atoi(fields[0]);
				}

			} else {
				log_error("No probe scenarios found");
				err = 1;
			}

			mysql_free_result(myResult);
		}
	}

	if (err) {
		if (scenarios) {
			mem_free(scenarios);
		}
		return err;
	}

	// Run the scenarios, log the results.

	SOURCE *source, *usource;
	UNDESIRED *undesireds;
	int sourceIndex, countryIndex, undesiredIndex, causesIX;

	for (i = 0; i < scenarioCount; i++) {

		err = run_scenario(scenarios[i]);
		if (err) {
			mem_free(scenarios);
			return err;
		}

		source = Sources;
		for (sourceIndex = 0; sourceIndex < SourceCount; sourceIndex++, source++) {
			if (source->inScenario && source->isDesired) {
				countryIndex = source->countryKey - 1;
				undesireds = source->undesireds;
				for (undesiredIndex = 0; undesiredIndex < source->undesiredCount; undesiredIndex++) {
					usource = SourceKeyIndex[undesireds[undesiredIndex].sourceKey];
					if (!usource) {
						log_error("Source structure index is corrupted");
						exit(1);
					}
					causesIX = (undesireds[undesiredIndex].totals[countryIndex].ixArea > 0.);
					snprintf(mesg, MAX_STRING, "%d,%d,%d,%d", source->sourceKey, usource->sourceKey, causesIX,
						undesireds[undesiredIndex].totals[countryIndex].ixPop);
					status_message(STATUS_KEY_RESULT, mesg);
				}
			}
		}
	}

	mem_free(scenarios);
	return 0;
}
