//
//  RunPanelPairStudy.java
//  TVStudy
//
//  Copyright (c) 2012-2021 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui.run;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.gui.*;

import java.util.*;
import java.io.*;
import java.nio.file.*;
import java.sql.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;


//=====================================================================================================================
// RunPanel subclass for managing a pair-wise study.  This study type consists of a series of intermediate study
// engine runs generating temporary data files which are finally post-processed by a separate utility.  This panel
// manages all of those runs in a tabbed layout, in turn this is managed by the run manager in the usual way, and may
// also call back to a StudyManager to report progress.  Although this works with an existing study, there are study
// build and post-run study restore phases, see StudyBuildPair.  The studyBuild object always exists here, it is
// created by the constructor.

public class RunPanelPairStudy extends RunPanel implements StudyLockHolder, ProcessPanel.Handler {

	private StudyManager studyManager;
	private String dbID;

	public StudyBuildPair studyBuild;

	// The study lock state.

	public int studyKey;
	public String studyName;
	public int studyType;
	public int studyLock;
	public int lockCount;

	public Path outDirectoryPath;

	// These must be set, initialize() will validate.

	public OutputConfig fileOutputConfig;
	public OutputConfig mapOutputConfig;

	// Multiple ProcessPanels are displayed.  Status label and cancel button shown outside the tab pane, applying to
	// all panels when multiple processes are running.

	private int runCount;

	private JTabbedPane tabPane;

	private JLabel statusLabel;
	private JButton cancelButton;

	// State.

	private static final int RUN_STATE_INIT = 0;
	private static final int RUN_STATE_WAIT = 1;
	private static final int RUN_STATE_BUILD = 2;
	private static final int RUN_STATE_PRERUN = 3;
	private static final int RUN_STATE_RUNNING = 4;
	private static final int RUN_STATE_POSTRUN = 5;
	private static final int RUN_STATE_RESTORE = 6;
	private static final int RUN_STATE_EXITING = 7;
	private static final int RUN_STATE_EXIT = 8;

	private int runState;

	private boolean runFailed;
	private boolean runCanceled;
	private String runFailedMessage;

	private AppTask task;
	private boolean taskWaiting;

	// The build process runs on a separate thread.

	private Thread buildThread;
	private boolean buildFailed;

	// Run list from the build.

	private ArrayDeque<StudyBuildPair.ScenarioSortItem> scenarioRunList;

	// There may be multiple running ProcessPanels.  See the ProcessPanel.Handler methods for details of other state.

	private ArrayList<ProcessPanel> studyRuns;

	private HashMap<ProcessPanel, StudyBuildPair.ScenarioSortItem> studyRunsPending;
	private HashSet<ProcessPanel> ignoreFailedRuns;

	private boolean hasOutput;

	// The restore process also runs on a separate thread.

	private Thread restoreThread;

	// Temporary output files will be deleted on exit.

	private Path filePath;
	private ArrayList<File> outFiles;

	// This always exists as a place to show messages if needed, but this property is not used for state management.
	// When appropriate, this panel will also be in the studyRuns list.

	private ProcessPanel baselinePanel;

	// State for progress reporting and time estimate.

	private long runStatusStartTime;
	private int runStatusTotalCount;
	private int runStatusRunningCount;
	private int runStatusDoneCount;
	private boolean updateRunStatus;

	// Estimate of disk space used per desired station per channel studied.  Both temporary and final output files are
	// on disk together during post-processing so this is nearly double the expected size of the final output.

	private static final long SOURCE_OUTPUT_SPACE_NEEDED = 3000000L;


	//-----------------------------------------------------------------------------------------------------------------

	public RunPanelPairStudy(StudyManager theManager, String theDbID) {

		super();

		studyManager = theManager;
		dbID = theDbID;

		studyBuild = new StudyBuildPair(dbID);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The StudyBuildPair initializer does most of the work.

	public boolean initialize(ErrorReporter errors) {

		if (initialized) {
			return true;
		}

		if (RUN_STATE_INIT != runState) {
			runState = RUN_STATE_EXIT;
			runFailed = true;
			return false;
		}

		runState = RUN_STATE_EXIT;
		runFailed = true;

		if (!super.initialize(errors)) {
			return false;
		}

		// Make sure the output configurations are set and valid.  If the config has the log file option, set the flag
		// so the run manager automatically saves run output when done, but don't clear it if not.

		if ((null == fileOutputConfig) || fileOutputConfig.isNull() || !fileOutputConfig.isValid()) {
			if (null != errors) {
				errors.reportError("Cannot run study, missing or invalid output file settings");
			}
			return false;
		}
		if ((null == mapOutputConfig) || mapOutputConfig.isNull() || !mapOutputConfig.isValid()) {
			if (null != errors) {
				errors.reportError("Cannot run study, missing or invalid map output settings");
			}
			return false;
		}

		autoSaveOutput = ((1 == fileOutputConfig.flags[OutputConfig.LOG_FILE]) ||
			(3 == fileOutputConfig.flags[OutputConfig.LOG_FILE]));

		// Make sure the file output flag for the pair study custom cell file is set, also the settings file.

		fileOutputConfig.flags[OutputConfig.CELL_FILE_PAIRSTUDY] = 1;
		fileOutputConfig.flags[OutputConfig.SETTING_FILE] = 1;

		if (!studyBuild.initialize(errors)) {
			return false;
		}

		// Make sure database connections can be made later even if the study manager closes.

		if (!DbCore.openDb(dbID, this)) {
			errors.reportError("Invalid database connection ID");
			return false;
		}

		// During the main running phase multiple engine processes may be used, but total memory will be scaled by the
		// memory fraction selected.  Make sure the requested fraction is not unreasonably small, then determine the
		// running process count.  That count started as user input setting a maximum, StudyBuildPair may have reduced
		// that based on the number of scenarios, here it may be further reduced based on the memory fraction.  Note
		// maxEngineProcessCount <= 0 indicates there is either not enough memory to run the engine at all, or the
		// engine executable isn't properly installed.  In that case this should never be reached, but regardless,
		// runCount is never less than 1 here.  If there is an engine problem an error will happen later.

		runCount = studyBuild.runCount;

		if (AppCore.maxEngineProcessCount > 0) {

			double minFrac = 1. / (double)AppCore.maxEngineProcessCount;
			if (memoryFraction < minFrac) {
				memoryFraction = minFrac;
			}

			if ((runCount > 1) && (memoryFraction < 1.)) {
				int maxCount = (int)(memoryFraction * (double)AppCore.maxEngineProcessCount);
				if (maxCount < 1) {
					maxCount = 1;
				}
				if (runCount > maxCount) {
					runCount = maxCount;
				}
			}
		}

		task = new AppTask(memoryFraction, runCount);

		// Verify the lock state is set, it should just be the lock state in the studyBuild.baseStudy object but don't
		// assume that, it's up to whatever created this (usually a RunStart subclass) which "hands off" the lock.  The
		// output path and run names are set here.

		if ((studyKey <= 0) || (Study.LOCK_NONE == studyLock) || (lockCount <= 0)) {
			errors.reportError("Cannot run study, missing or invalid lock state");
			return false;
		}

		outDirectoryPath = Paths.get(studyBuild.baseStudy.outDirectory);
		if (!outDirectoryPath.isAbsolute()) {
			outDirectoryPath = AppCore.workingDirectoryPath.resolve(studyBuild.baseStudy.outDirectory);
		}

		runName = "Run pair study '" + studyName + "'";

		// Other setup.

		String hostDir = DbCore.getHostDbName(dbID);
		if (null == hostDir) {
			filePath = outDirectoryPath.resolve(studyName);
		} else {
			filePath = outDirectoryPath.resolve(hostDir).resolve(studyName);
		}
		outFiles = new ArrayList<File>();

		studyRuns = new ArrayList<ProcessPanel>();
		studyRunsPending = new HashMap<ProcessPanel, StudyBuildPair.ScenarioSortItem>();
		ignoreFailedRuns = new HashSet<ProcessPanel>();

		// A status label tracking progress of the study.

		statusLabel = new JLabel();
		statusLabel.setPreferredSize(AppController.labelSize[60]);

		// Buttons.

		cancelButton = new JButton("Cancel");
		cancelButton.setFocusable(false);
		cancelButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				cancel();
			}
		});
		cancelButton.setEnabled(false);

		// Layout.  Always create the baseline process panel to have something to show, and a place to show messages.

		baselinePanel = new ProcessPanel(this, "Study", null);
		baselinePanel.setProcessHandler(this);
		baselinePanel.setStatusPanelVisible(false);

		tabPane = new JTabbedPane();
		tabPane.addTab("Baseline", baselinePanel);

		JPanel statPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		statPanel.add(cancelButton);
		statPanel.add(statusLabel);

		setLayout(new BorderLayout());
		add(tabPane, BorderLayout.CENTER);
		add(statPanel, BorderLayout.SOUTH);

		// Success.

		runState = RUN_STATE_WAIT;
		runFailed = false;

		initialized = true;

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getDbID() {

		return dbID;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// StudyLockHolder methods.

	public int getStudyKey() {

		return studyKey;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getStudyName() {

		return studyName;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public int getStudyLock() {

		return studyLock;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public int getLockCount() {

		return lockCount;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void toFront() {

		if (null != parent) {
			parent.getWindow().toFront();
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean closeWithoutSave() {

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean studyManagerClosing() {

		studyManager = null;
		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check free disk space.  Check both cache space, based on the baseline scenario source count, and also an
	// estimate of the output file size.  For a pair study, the output files can easily be much bigger than cache.

	public boolean isDiskSpaceAvailable() {

		long outSize =
			(long)(studyBuild.baselineSourceCount * studyBuild.studyChannels.length) * SOURCE_OUTPUT_SPACE_NEEDED;

		return AppCore.isFreeSpaceAvailable(dbID, studyBuild.baselineSourceCount, outSize);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The state runner, called by the displaying run manager.  Handles progression through states, in running states
	// polls running processes to keep UI updated.  Note when a process exits with failure that may be ignored if it is
	// in the ignoreFailedRuns list, see processFailed().

	public void poll() {

		if (!initialized) {
			return;
		}

		// In the exit state, make sure AppTask knows this is done.

		if (RUN_STATE_EXIT == runState) {

			if (null != task) {

				AppTask.taskDone(task);
				task = null;

				DbCore.closeDb(dbID, this);
			}

			return;
		}

		cancelButton.setEnabled(!runCanceled && canCancel());

		// Waiting state.  Check with AppTask to see if this can start.  Always poll the baseline panel in wait and
		// build states so any UI updates there will happen.

		if (RUN_STATE_WAIT == runState) {

			baselinePanel.poll();

			if (!AppTask.canTaskStart(task)) {
				if (!taskWaiting) {
					taskWaiting = true;
					statusLabel.setText("Waiting for other runs to complete...");
				}
				return;
			}
			taskWaiting = false;

			runState = RUN_STATE_BUILD;
		}
			
		// Build state.  Start the thread first time through, poll it later.

		if (RUN_STATE_BUILD == runState) {

			baselinePanel.poll();

			if (null == buildThread) {

				buildThread = new Thread() {
					public void run() {
						try {
							buildStudy();
						} catch (Throwable t) {
							AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", t);
							buildFailed = true;
						}
					}
				};
				buildThread.start();

				statusLabel.setText("Building pair study scenarios...");

				return;

			} else {

				if (buildThread.isAlive()) {
					return;
				}

				// When the thread exits, if it failed or was canceled go straight to exiting state, attempting a
				// restore is pointless since the study is either unchanged or already damaged.  Otherwise inform the
				// study manager about the new study lock state and start the prerun state.

				buildThread = null;

				if (buildFailed) {

					runState = RUN_STATE_EXITING;
					if (studyBuild.isCanceled()) {
						runCanceled = true;
						runFailedMessage = "Study setup canceled";
					} else {
						runFailed = true;
						runFailedMessage = "Study setup failed";
					}
					cancelButton.setEnabled(false);

				} else {

					if (runCanceled) {

						runState = RUN_STATE_RESTORE;
						cancelButton.setEnabled(false);

					} else {

						runState = RUN_STATE_PRERUN;
						statusLabel.setText("Running baseline scenario");
						startPrerun();
						if (null != studyManager) {
							studyManager.applyEditsFrom(this);
						}
					}
				}
			}
		}

		// As long as studyRuns contains one or more active runs, poll all of those, removing any that are no longer
		// running, until all are done.  If another run failed or after a cancel, others still running are canceled.
		// State change will not occur until all runs are finished.

		if (!studyRuns.isEmpty()) {

			Iterator<ProcessPanel> it = studyRuns.iterator();
			ProcessPanel theRun;
			boolean isRunning;

			while (it.hasNext()) {

				theRun = it.next();

				if ((runFailed || runCanceled) && !theRun.isCanceled()) {
					theRun.cancel();
				}

				isRunning = theRun.poll();

				if (!isRunning) {

					it.remove();

					if (theRun.hasOutput()) {
						hasOutput = true;
					}

					// Ignore process exit errors after the first failure, or after a cancel.

					if (!runFailed && !runCanceled && theRun.didProcessFail() && !ignoreFailedRuns.contains(theRun)) {
						runFailed = true;
						runFailedMessage = "Study process failed";
						cancelButton.setEnabled(false);
					}
				}
			}

			// If needed, update a count of sources or scenarios studied along with an estimated time to complete.  The
			// update flag is set whenever an engine process gives feedback indicating it has completed an item, the
			// exact mechanism varies between the baseline study and full pair study but the concept is the same, see
			// details in processResponseConfirmed() and processLogMessage().  However in the pair study phase the
			// count of items (scenarios) is scaled according to the number of sources for the time estimate.

			if (!studyRuns.isEmpty() && updateRunStatus) {
				int totalCount = 0, doneCount = 0;
				if (RUN_STATE_PRERUN == runState) {
					totalCount = runStatusTotalCount;
					doneCount = runStatusDoneCount;
				} else {
					if (RUN_STATE_RUNNING == runState) {
						totalCount = studyBuild.scenarioCount;
						doneCount = studyBuild.scenarioCount - scenarioRunList.size();
					}
				}
				if (totalCount > 0) {
					String status;
					String progress = AppCore.formatCount(doneCount) + " of " +
						AppCore.formatCount(totalCount) + " items done";
					if (doneCount == totalCount) {
						status = progress + ", running";
					} else {
						if (0L == runStatusStartTime) {
							status = progress + ", running";
						} else {
							double fractionDone = (double)runStatusDoneCount / (double)runStatusTotalCount;
							double minutesElapsed = (double)(System.currentTimeMillis() - runStatusStartTime) / 60000.;
							if ((fractionDone < 0.1) && (minutesElapsed < 2.)) {
								status = progress + ", running";
							} else {
								double minutesRemaining = minutesElapsed * ((1. / fractionDone) - 1.);
								if (fractionDone < 0.5) {
									minutesRemaining *= 1.5 - fractionDone;
								}
								if (minutesRemaining < 1.) {
									status = progress + ", less than 1 minute remaining";
								} else {
									if (minutesRemaining < 60.) {
										int minutes = (int)Math.rint(minutesRemaining);
										status = progress + ", about " + minutes +
											((1 == minutes) ? " minute" : " minutes") + " remaining";
									} else {
										int hours = (int)Math.rint(minutesRemaining / 60.);
										status = progress + ", about " + hours +
											((1 == hours) ? " hour" : " hours") + " remaining";
									}
								}
							}
						}
					}
					statusLabel.setText(status);
				}
				updateRunStatus = false;
			}

			// If any still running, that's all for now.

			if (!studyRuns.isEmpty()) {
				return;
			}

			// All processes done.  In the running state do a fail-safe check to be sure all scenarios were run, in
			// case processes exited but didn't report errors.

			if ((RUN_STATE_RUNNING == runState) && !runFailed && !runCanceled && !scenarioRunList.isEmpty()) {

				runFailed = true;
				runFailedMessage = "Study failed, did not process all scenarios";
				runState = RUN_STATE_RESTORE;
				cancelButton.setEnabled(false);

			} else {

				// Proceed to the next state as needed.

				if (runFailed || runCanceled || (RUN_STATE_POSTRUN == runState)) {

					runState = RUN_STATE_RESTORE;
					cancelButton.setEnabled(false);

				} else {

					if (RUN_STATE_PRERUN == runState) {
						runState = RUN_STATE_RUNNING;
						statusLabel.setText("Running pair study scenarios");
						startRunning();
						return;
					}

					if (RUN_STATE_RUNNING == runState) {
						runState = RUN_STATE_POSTRUN;
						statusLabel.setText("Post-processing study results");
						cancelButton.setEnabled(false);
						startPostrun();
						return;
					}
				}
			}
		}

		// Restore state runs on a secondary thread, similar to build state above.  Errors during the restore do not
		// cause the overall state to change to failed.  When the thread exits, inform the study manager of another
		// lock state change.  See StudyBuildPair.restoreStudy().

		if (RUN_STATE_RESTORE == runState) {

			if (null == restoreThread) {

				statusLabel.setText("Restoring study database");

				restoreThread = new Thread() {
					public void run() {
						try {
							restoreStudy();
						} catch (Throwable t) {
							AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", t);
						}
					}
				};
				restoreThread.start();

				return;

			} else {

				if (restoreThread.isAlive()) {
					return;
				}

				restoreThread = null;

				runState = RUN_STATE_EXITING;
				statusLabel.setText("Study complete");
			}
		}

		// Final fall through, must be in exiting state.  Set exit, inform the task manager this is done, tell the
		// study manager this object is done with the study, and delete the temporary output files.

		runState = RUN_STATE_EXIT;
		if (runFailed || runCanceled) {
			statusLabel.setText(runFailedMessage);
		}
		cancelButton.setEnabled(false);

		if (null != studyManager) {
			studyManager.editorClosing(this);
		}

		for (File theFile : outFiles) {
			theFile.delete();
		}
		outFiles.clear();

		if (null != task) {

			AppTask.taskDone(task);
			task = null;

			DbCore.closeDb(dbID, this);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Build the study.  This is assumed to be called on a secondary thread, it may block for a considerable time.
	// Any errors or messages from the build process are displayed in the baseline process panel.

	private void buildStudy() {

		if (RUN_STATE_BUILD != runState) {
			buildFailed = true;
			return;
		}

		String errmsg = "";

		// Logger to accumulate errors and other messages for display in the baseline panel.

		ErrorLogger errors = new ErrorLogger();

		Study theStudy = studyBuild.buildStudy(errors);
		if (null == theStudy) {
			buildFailed = true;
		} else {

			// Study build succeeded, change the study lock from edit to run.

			if (errors.hasMessages()) {
				errors.reportMessage(errors.getMessages());
				errors.clearMessages();
			}

			DbConnection db = DbCore.connectDb(dbID, errors);
			if (null != db) {
				try {

					db.update("LOCK TABLES study WRITE");

					db.query("SELECT study_lock, lock_count FROM study WHERE study_key = " + theStudy.key);

					if (db.next()) {

						if ((db.getInt(1) == studyLock) && (db.getInt(2) == lockCount)) {

							db.update("UPDATE study SET study_lock = " + Study.LOCK_RUN_EXCL +
								", lock_count = lock_count + 1, share_count = 0 WHERE study_key = " + studyKey);
							studyLock = Study.LOCK_RUN_EXCL;
							lockCount++;

						} else {
							buildFailed = true;
							errmsg = "Could not update study lock, the lock was modified";
						}

					} else {
						buildFailed = true;
						errmsg = "Could not update study lock, the study was deleted";
					}

				} catch (SQLException se) {
					buildFailed = true;
					errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
					db.reportError(se);
				}

				try {
					db.update("UNLOCK TABLES");
				} catch (SQLException se) {
					db.reportError(se);
				}

				DbCore.releaseDb(db);

				if (buildFailed) {
					errors.reportError(errmsg);
				}

			} else {
				buildFailed = true;
			}
		}

		// Show errors and messages in the baseline process panel display area, has to be done on the event thread.

		if (errors.hasErrors()) {
			final String message = "\n" + errors.toString() + "\n";
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					baselinePanel.displayLogMessage(message);
				}
			});
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Start the baseline run state.  This studies just the baseline scenario.  The run creates a temporary data file
	// that will be post-processed along with the pair runs, but it also has an important side-effect of pre-processing
	// all sources in the database, i.e. completing all replications, so the later pair runs can all run with just the
	// shared run lock.  The user may also select output files from this run.

	private void startPrerun() {

		ArrayList<String> arguments = new ArrayList<String>();

		String lockID = String.valueOf(lockCount);

		File outFile = filePath.resolve("outtemp_" + lockID + "_0").toFile();

		arguments.add(AppCore.libDirectoryPath.resolve(AppCore.STUDY_ENGINE_NAME).toString());

		if (AppCore.Debug) {
			arguments.add("-d");
		}

		arguments.add("-w");
		arguments.add(AppCore.workingDirectoryPath.toString());
		arguments.add("-z");
		arguments.add(AppCore.cacheDirectoryPath.toString());
		arguments.add("-o");
		arguments.add(outDirectoryPath.toString());
		arguments.add("-i");

		arguments.add("-h");
		arguments.add(DbCore.getDbHostname(dbID));
		arguments.add("-b");
		arguments.add(DbCore.getDbName(dbID));
		arguments.add("-u");
		arguments.add(DbCore.getDbUsername(dbID));

		arguments.add("-l");
		arguments.add(lockID);
		arguments.add("-k");

		arguments.add("-n");
		arguments.add(lockID);

		arguments.add("-f");
		arguments.add(fileOutputConfig.getCodes());
		arguments.add("-e");
		arguments.add(mapOutputConfig.getCodes());

		// Include some run information in the comment field, followed by any user-provided comment.

		StringBuilder com = new StringBuilder();
		com.append("Pair study baseline run\n");
		com.append("Study country: ");
		if (null == studyBuild.studyCountry) {
			com.append("All");
		} else {
			com.append(studyBuild.studyCountry.name);
		}
		com.append('\n');
		com.append("Study channels:");
		for (int chan : studyBuild.studyChannels) {
			com.append(' ');
			com.append(String.valueOf(chan));
		}
		com.append('\n');
		com.append(runComment);
		arguments.add("-c");
		arguments.add(com.toString());

		arguments.add(String.valueOf(studyKey));

		arguments.add(String.valueOf(studyBuild.baselineScenarioKey));

		// The baseline panel already exists and is in the display.

		baselinePanel.setProcessArguments(arguments, DbCore.getDbPassword(dbID));

		studyRuns.add(baselinePanel);

		outFiles.add(outFile);

		runStatusStartTime = 0L;
		runStatusTotalCount = studyBuild.baselineSourceCount;
		runStatusRunningCount = 0;
		runStatusDoneCount = 0;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Start the running state, queue up all the actual study runs.

	private void startRunning() {

		// Get the run list from the build.

		scenarioRunList = studyBuild.getScenarioRunList();

		// Change the study lock to run shared.

		boolean error = false;
		String errorMessage = "";

		DbConnection db = DbCore.connectDb(dbID);
		if (null != db) {
			try {

				db.update("LOCK TABLES study WRITE");

				db.query("SELECT study_lock, lock_count FROM study WHERE study_key = " + studyKey);

				if (db.next()) {

					if ((db.getInt(1) == studyLock) && (db.getInt(2) == lockCount)) {

						db.update("UPDATE study SET study_lock = " + Study.LOCK_RUN_SHARE +
							", lock_count = lock_count + 1, share_count = 1 WHERE study_key = " + studyKey);
						studyLock = Study.LOCK_RUN_SHARE;
						lockCount++;

					} else {
						error = true;
						errorMessage = "The study lock was modified";
					}

				} else {
					error = true;
					errorMessage = "The study was deleted";
				}

			} catch (SQLException se) {
				error = true;
				errorMessage = "Error: " + se;
				db.reportError(se);
			}

			try {
				db.update("UNLOCK TABLES");
			} catch (SQLException se) {
				db.reportError(se);
			}

			DbCore.releaseDb(db);

		} else {
			error = true;
			errorMessage = "Could not open database connection";
		}

		// If no error, set up to start the runs.  The memory limit argument actually given to the engine is the
		// reciprocal of the memory fraction desired, in other words it is the total number of engine processes
		// expected to run concurrently; here that has to be based on both the process count and memory fraction.

		if (!error) {

			String nProc = String.valueOf((int)Math.rint((double)runCount / memoryFraction));

			// A run ID number is given to each process to use for temporary file names, based on combining the current
			// lock count (which will always be unique for the study) with the run number.  This must be in sync with
			// code in the study engine for naming of the files.  It assumes the directory path was established by the
			// engine in the baseline run so all directories will already exist.  The scenario keys run by each process
			// will be provided dynamically, see the ProcessPanel.Handler methods, the trigger for that behavior in the
			// engine is to pass a '*' as the first scenario key argument.

			String runID;
			File outFile;
			ArrayList<String> arguments;
			ProcessPanel theRun;

			String theHost = DbCore.getDbHostname(dbID);
			String theName = DbCore.getDbName(dbID);
			String theUser = DbCore.getDbUsername(dbID);

			String lockID = String.valueOf(lockCount);

			// These runs generate only the pair study cell file, all other output is off.

			OutputConfig conf = new OutputConfig(OutputConfig.CONFIG_TYPE_FILE, "");
			conf.flags[OutputConfig.CELL_FILE_PAIRSTUDY] = 1;
			String outCodes = conf.getCodes();

			for (int runNumber = 0; runNumber < runCount; runNumber++) {

				runID = String.valueOf((lockCount * runCount) + runNumber);

				outFile = filePath.resolve("outtemp_" + runID + "_0").toFile();

				arguments = new ArrayList<String>();

				arguments.add(AppCore.libDirectoryPath.resolve(AppCore.STUDY_ENGINE_NAME).toString());

				if (AppCore.Debug) {
					arguments.add("-d");
				}

				arguments.add("-w");
				arguments.add(AppCore.workingDirectoryPath.toString());
				arguments.add("-z");
				arguments.add(AppCore.cacheDirectoryPath.toString());
				arguments.add("-o");
				arguments.add(outDirectoryPath.toString());
				arguments.add("-i");

				arguments.add("-h");
				arguments.add(theHost);
				arguments.add("-b");
				arguments.add(theName);
				arguments.add("-u");
				arguments.add(theUser);

				arguments.add("-l");
				arguments.add(lockID);
				arguments.add("-k");

				arguments.add("-n");
				arguments.add(runID);

				arguments.add("-x");

				arguments.add("-m");
				arguments.add(nProc);

				arguments.add("-f");
				arguments.add(outCodes);
				arguments.add("-e");
				arguments.add(OutputConfig.NO_OUTPUT_CODE);

				arguments.add(String.valueOf(studyKey));

				arguments.add("*");

				theRun = new ProcessPanel(this, "Study", null);
				theRun.setProcessHandler(this);
				theRun.setStatusPanelVisible(false);

				theRun.setProcessArguments(arguments, DbCore.getDbPassword(dbID));

				studyRuns.add(theRun);

				outFiles.add(outFile);
			}
		}

		// If any error occurred put message in the UI, set failure flag, and go to restore.  Just to be paranoid,
		// cancel any processes that might have been set up before the error, so they can never start.

		if (error) {

			runFailed = true;
			runFailedMessage = errorMessage;
			runState = RUN_STATE_RESTORE;
			cancelButton.setEnabled(false);

			for (ProcessPanel theRun : studyRuns) {
				theRun.cancel();
			}
			studyRuns.clear();

			return;
		}

		// Display the run UIs, the poll method does the rest, the processes will start the first time they are polled.

		int runNumber = 1;
		for (ProcessPanel theRun : studyRuns) {
			tabPane.addTab("Run " + runNumber++, theRun);
		}

		runStatusStartTime = 0L;
		runStatusTotalCount = studyBuild.scenarioSourceCount;
		runStatusRunningCount = 0;
		runStatusDoneCount = 0;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Start the post-run state, do the data post-processing.  A bit of that is local, but most of it is handled by a
	// separate external utility for performance reasons.  First the local part, write the stations list output file,
	// along the way determine the maximum facility ID value.  For historical reasons this is done by re-querying the
	// source records.  That used to be necessary to get the NAD83 coordinates; those are in the Source model now so
	// this could draw from the in-memory state, but this is working fine so why mess with it.

	private void startPostrun() {

		boolean error = false;
		String errorMessage = null;

		int maxFacilityID = 0;

		BufferedWriter stationsWriter = null;

		String rootName = DbCore.getDbName(dbID);

		DbConnection db = DbCore.connectDb(dbID);
		if (null != db) {
			try {

				stationsWriter = Files.newBufferedWriter(filePath.resolve("stations.csv"));

				db.setDatabase(rootName + "_" + studyKey);

				db.query(
				"SELECT " +
					"source.facility_id, " +
					"source.channel, " +
					"scenario_source.is_desired, " +
					"scenario_source.is_undesired, " +
					"service.service_code, " +
					"source.call_sign, " +
					"source.city, " +
					"source.state, " +
					"country.country_code, " +
					"source.status, " +
					"source.file_number, " +
					"source.latitude, " +
					"source.longitude " +
				"FROM " +
					"scenario_source " +
					"JOIN source USING (source_key) " +
					"JOIN " + rootName + ".service USING (service_key) " +
					"JOIN " + rootName + ".country USING (country_key) " +
				"WHERE " +
					"scenario_source.scenario_key = " + studyBuild.baselineScenarioKey + " " +
					"AND (scenario_source.is_desired OR scenario_source.is_undesired) " +
				"ORDER BY " +
					"source.country_key, source.state, source.city, source.channel");

				int facID;

				while (db.next()) {

					facID = db.getInt(1);
					if (facID > maxFacilityID) {
						maxFacilityID = facID;
					}

					stationsWriter.write(
						String.format(Locale.US, "%d,%d,%d,%d,%s,\"%s\",\"%s\",%s,%s,%s,\"%.22s\",%f,%f\n",
							facID, db.getInt(2), db.getInt(3), db.getInt(4), db.getString(5), db.getString(6),
							db.getString(7), db.getString(8), db.getString(9), db.getString(10), db.getString(11),
							db.getDouble(12), db.getDouble(13)));
				}

			} catch (SQLException se) {
				error = true;
				errorMessage = "Error: " + se;
				db.reportError(se);

			} catch (IOException ie) {
				error = true;
				errorMessage = "Error: " + ie;
			}

			DbCore.releaseDb(db);

			if (null != stationsWriter) {
				try {stationsWriter.close();} catch (IOException ie) {}
			}

		} else {
			error = true;
			errorMessage = "Could not open database connection";
		}

		// If an error occurred, straight to restore.

		if (error) {

			runFailed = true;
			runFailedMessage = errorMessage;
			runState = RUN_STATE_RESTORE;
			cancelButton.setEnabled(false);

			return;
		}

		// Start the post-processing run.

		ArrayList<String> arguments = new ArrayList<String>();

		arguments.add(AppCore.libDirectoryPath.resolve("pair_study_post").toString());

		arguments.add(filePath.toString());

		arguments.add(String.valueOf(maxFacilityID));

		for (File theFile : outFiles) {
			arguments.add(theFile.getName());
		}

		ArrayList<String> mergeLines = new ArrayList<String>();
		mergeLines.add("Completed ");

		ProcessPanel theRun = new ProcessPanel(this, "Process", mergeLines);
		theRun.setProcessHandler(this);
		theRun.setStatusPanelVisible(false);

		theRun.setProcessArguments(arguments, null);

		studyRuns.add(theRun);

		tabPane.addTab("Post", theRun);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Restore the study, called on a secondary thread.  Errors here are essentially ignored, the state will proceed
	// to exiting in any case, and a failure here does not invalidate any of the study results so it should not alter
	// a successful result to this point.  However error messages will be displayed in the baseline log.

	private void restoreStudy() {

		if (RUN_STATE_RESTORE != runState) {
			return;
		}

		// Copy the current lock state to and from the study build object.

		studyBuild.baseStudy.studyLock = studyLock;
		studyBuild.baseStudy.lockCount = lockCount;

		ErrorLogger errors = new ErrorLogger();

		boolean error = !studyBuild.restoreStudy(errors);

		studyLock = studyBuild.baseStudy.studyLock;
		lockCount = studyBuild.baseStudy.lockCount;

		// If no error, unlock the study.

		if (!error) {

			String errorMessage = "";

			DbConnection db = DbCore.connectDb(dbID);
			if (null != db) {
				try {

					db.update("LOCK TABLES study WRITE");

					db.query("SELECT study_lock, lock_count FROM study WHERE study_key=" + studyKey);

					if (db.next()) {

						if ((db.getInt(1) == studyLock) && (db.getInt(2) == lockCount)) {

							db.update("UPDATE study SET study_lock = " + Study.LOCK_NONE +
								", lock_count = lock_count + 1, share_count = 0 WHERE study_key = " + studyKey);
							studyLock = Study.LOCK_NONE;
							lockCount++;

						} else {
							error = true;
							errorMessage = "Could not unlock study, the lock was modified";
						}

					} else {
						error = true;
						errorMessage = "Could not unlock study, the study was deleted";
					}

				} catch (SQLException se) {
					error = true;
					errorMessage = DbConnection.ERROR_TEXT_PREFIX + se;
					db.reportError(se);
				}

				try {
					db.update("UNLOCK TABLES");
				} catch (SQLException se) {
					db.reportError(se);
				}

				DbCore.releaseDb(db);

			} else {
				error = true;
				errorMessage = "Could not open database connection";
			}

			if (error) {
				errors.reportError(errorMessage);
			}
		}

		if (error) {
			final String message = "\n" + errors.toString() + "\n";
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					baselinePanel.displayLogMessage(message);
				}
			});
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// ProcessPanel.Handler interface methods.  When this method is called, a prompt was issued by one of the running
	// engine processes.  Currently this can only be a prompt for another scenario key, just pass them out in queued
	// sequence.  However the response must be confirmed by the process, so this moves each key into a pending state
	// per process until that is either confirmed or the process fails.  If there are no more scenarios, respond just
	// with the prompting prefix which the process will interpret to mean clean up and exit, however even that has to
	// be confirmed by the process.  See ProcessPanel.  Return null on invalid calls, calling process will be killed.

	public String getProcessResponse(ProcessPanel thePanel, String thePrompt) {

		if (RUN_STATE_RUNNING != runState) {
			return null;
		}

		if (!studyRuns.contains(thePanel) || studyRunsPending.containsKey(thePanel)) {
			return null;
		}

		if (0L == runStatusStartTime) {
			runStatusStartTime = System.currentTimeMillis();
		}

		StudyBuildPair.ScenarioSortItem theItem = scenarioRunList.poll();
		studyRunsPending.put(thePanel, theItem);

		if (null == theItem) {
			return AppCore.ENGINE_PROMPT_PREFIX;
		} else {
			return String.valueOf(theItem.key);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Reports confirmation of a response sent to a process, consider the scenario as successfully running.

	public void processResponseConfirmed(ProcessPanel thePanel) {

		if (RUN_STATE_RUNNING != runState) {
			return;
		}

		StudyBuildPair.ScenarioSortItem theItem = studyRunsPending.remove(thePanel);
		if (null != theItem) {
			runStatusDoneCount += theItem.sourceCount;
			updateRunStatus = true;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Reports failure of a process being controlled.

	// If the process failed while not in the prompt-response pending state this is a true failure that must cancel the
	// entire study.  That will be detected and handled by poll() automatically.  But in that case there is no point
	// to continuing with other scenario runs since the overall study has failed, so this empties the scenario queue
	// causing all other processes to exit at the next prompt.

	// If the process failed while in the pending state, this may attempt to recover and finish the overall study.  The
	// failed process was in a "safe" state where any previous scenario runs were completed without error, most likely
	// the failure had to do with the interactive prompt-response protocol.  In that case as long as at least one other
	// process is still able to continue running scenarios, the pending scenario can just be re-queued and the overall
	// study may still finish.  In that case this process is placed in the ignoreFailedRuns list so poll() does not
	// count the failure as canceling the study.  If there are no other processes, as above the queue is cleared.

	// If the process failed in the pending state but the pending response is the exit command, the process can just
	// immediately be placed in the ignoreFailedRuns list regardless of whether or not any other processes are still
	// running, since there is no pending scenario to re-queue.  However when checking for other processes that could
	// process a re-queued scenario, any that have the exit command pending can't be counted.

	public void processFailed(ProcessPanel thePanel) {

		if (RUN_STATE_RUNNING != runState) {
			return;
		}

		if (!studyRuns.contains(thePanel)) {
			return;
		}

		if (studyRunsPending.containsKey(thePanel)) {

			StudyBuildPair.ScenarioSortItem theItem = studyRunsPending.remove(thePanel);

			if (null == theItem) {
				ignoreFailedRuns.add(thePanel);
			} else {

				boolean hasOther = false;
				for (ProcessPanel otherRun : studyRuns) {
					if (otherRun.isProcessRunning()) {
						if (studyRunsPending.containsKey(otherRun)) {
							if (null != studyRunsPending.get(otherRun)) {
								hasOther = true;
								break;
							}
						} else {
							hasOther = true;
							break;
						}
					}
				}
				if (hasOther) {
					scenarioRunList.push(theItem);
					ignoreFailedRuns.add(thePanel);
				} else {
					scenarioRunList.clear();
				}
			}

		} else {
			scenarioRunList.clear();
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Reports a process exited cleanly.  If not in the prompt-response pending state do nothing, that means all is
	// well.  If in the pending state, this shouldn't have happened.  If the pending response was the exit command
	// ignore it, if not simply re-queue the pending scenario.  If there are other processes still active they will
	// pick up the scenario at next prompt.  If not, a fail-safe check in poll() will consider the overall study as
	// failed if all processes have exited but the scenario queue is not empty.

	public void processComplete(ProcessPanel thePanel) {

		if (RUN_STATE_RUNNING != runState) {
			return;
		}

		StudyBuildPair.ScenarioSortItem theItem = studyRunsPending.remove(thePanel);
		if (null != theItem) {
			scenarioRunList.push(theItem);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// In the baseline study phase, status messages are used to follow the study state and update run status and
	// remaining-time estimate.  In the pair study phase, that is handled through the prompt-and-response methods
	// above.  In either case, update of the status happens in poll().

	public void processStatusMessage(ProcessPanel thePanel, String theKey, String theData) {

		if (RUN_STATE_PRERUN != runState) {
			return;
		}

		if (theKey.equals(AppCore.ENGINE_RUNCOUNT_KEY)) {
			if (0L == runStatusStartTime) {
				runStatusStartTime = System.currentTimeMillis();
				updateRunStatus = true;
			}
			if (runStatusRunningCount > 0) {
				runStatusDoneCount += runStatusRunningCount;
				runStatusRunningCount = 0;
				updateRunStatus = true;
			}
			try {
				runStatusRunningCount = Integer.parseInt(theData);
			} catch (NumberFormatException e) {
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Don't care about individual process panel status changes here.

	public void setStatusMessage(String theMessage) {
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isRunning() {

		if (RUN_STATE_EXIT != runState) {
			return true;
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isWaiting() {

		if (RUN_STATE_WAIT == runState) {
			return true;
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void bumpTask() {

		if ((RUN_STATE_WAIT == runState) && (null != task)) {
			AppTask.bumpTask(task);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean runFailed() {

		return runFailed;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// A cancel action is only available in wait, build, pre-run, and running.

	public boolean canCancel() {

		return ((RUN_STATE_WAIT == runState) || (RUN_STATE_BUILD == runState) || (RUN_STATE_PRERUN == runState) ||
			(RUN_STATE_RUNNING == runState));
	}


	//-----------------------------------------------------------------------------------------------------------------
	// In wait or in build before the build thread has been created, just set exiting.  In build if the thread is
	// running, send the cancel to the build object.  In prerun or running, send a cancel to all processes.

	public void cancel() {

		if (runCanceled || !canCancel()) {
			return;
		}
		runCanceled = true;
		runFailedMessage = "Study canceled";
		cancelButton.setEnabled(false);

		if ((RUN_STATE_WAIT == runState) || ((RUN_STATE_BUILD == runState) && (null == buildThread))) {
			runState = RUN_STATE_EXITING;
			return;
		}

		if (RUN_STATE_BUILD == runState) {
			if (!studyBuild.isCanceled()) {
				studyBuild.cancel();
			}
			return;
		}

		for (ProcessPanel theRun : studyRuns) {
			theRun.cancel();
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getStatus() {

		switch (runState) {

			case RUN_STATE_INIT:
			case RUN_STATE_WAIT:
				return "Waiting";

			case RUN_STATE_BUILD:
				return "Building study";

			case RUN_STATE_PRERUN:
				if (runStatusTotalCount > 0) {
					return "Baseline, " + AppCore.formatCount(runStatusDoneCount) + " of " +
						AppCore.formatCount(runStatusTotalCount) + " done";
				} else {
					return "Baseline";
				}

			case RUN_STATE_RUNNING:
				if (studyBuild.scenarioCount > 0) {
					return "Running, " +
						AppCore.formatCount(studyBuild.scenarioCount - scenarioRunList.size()) + " of " +
						AppCore.formatCount(studyBuild.scenarioCount) + " done";
				} else {
					return "Running";
				}

			case RUN_STATE_POSTRUN:
			case RUN_STATE_RESTORE:
				return "Post-process";

			case RUN_STATE_EXITING:
				return "Exiting";

			case RUN_STATE_EXIT:
				if (runCanceled) {
					return "Canceled";
				} else {
					if (runFailed) {
						return "Failed";
					} else {
						return "Complete";
					}
				}
		}

		return "Unknown";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean hasOutput() {

		if (RUN_STATE_EXIT == runState) {
			return hasOutput;
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// This concatenates output from all the run panels showing in the tab pane.  At least one panel actually has some
	// output to save otherwise the hasOutput flag would not be set, see poll().

	public void writeOutputTo(Writer theWriter) throws IOException {

		if ((RUN_STATE_EXIT != runState) || !hasOutput) {
			return;
		}

		Component c;
		ProcessPanel p;
		for (int i = 0; i < tabPane.getTabCount(); i++) {
			c = tabPane.getComponentAt(i);
			if (c instanceof ProcessPanel) {
				p = (ProcessPanel)c;
				if (p.hasOutput()) {
					theWriter.write("\n\n------------------------ Output from " + tabPane.getTitleAt(i) +
						" ------------------------\n\n");
					p.writeOutputTo(theWriter);
				}
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public Path getOutDirectoryPath() {

		return filePath;
	}
}
