//
//  RunPanel.java
//  TVStudy
//
//  Copyright (c) 2012-2021 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui.run;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.gui.*;

import java.util.*;
import java.io.*;
import java.nio.file.*;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;


//=====================================================================================================================
// Abstract panel superclass used by RunManager to manage running processes.  Properties are set directly, then
// initialize() does setup.  Memory fraction and comment are optional, defaults will be used if not set.

public abstract class RunPanel extends AppPanel {

	public static final String LOG_FILE_NAME = "log.txt";
	public static final String REPORT_FILE_NAME = "report.txt";

	public double memoryFraction = 1.;
	public String runComment;

	public String runName;

	// If set, run output and/or report are auto-saved, regardless of errors, when autoSave() is called by manager.

	public boolean autoSaveOutput;
	public boolean autoSaveReport;

	// If set, the run manager will remove the panel from display when the run completes without error.

	public boolean autoRemove;

	// Set by subclass initialize() implementation on successful validation and setup.

	protected boolean initialized;


	//-----------------------------------------------------------------------------------------------------------------
	// Run panels are only displayed in the run manager so that is always the parent.

	public RunPanel() {

		super(RunManager.getRunManager());
	}


	//-----------------------------------------------------------------------------------------------------------------

	public RunPanel(Runnable theCallBack) {

		super(RunManager.getRunManager(), theCallBack);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Runs are not necessarily associated with a database, if they are subclass must override to provide ID.

	public String getDbID() {

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Subclass must always override, call super if desired but always set initialized flag directly.

	public boolean initialize(ErrorReporter errors) {

		if (initialized) {
			return true;
		}

		// Silently fix an invalid memory fraction.

		if (memoryFraction < 0.) {
			memoryFraction = 0.;
		}
		if (memoryFraction > 1.) {
			memoryFraction = 1.;
		}

		// Comment should not be null, set an empty string if needed.

		if (null != runComment) {
			runComment = runComment.trim();
		} else {
			runComment = "";
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isInitialized() {

		return initialized;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getRunName() {

		if (null == runName) {
			return "(unknown)";
		}
		return runName;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Do a check to see if disk space needed for the run is available, other UI might want to warn the user if this
	// returns false, e.g. see RunStart.  Typically overridden, but not required as this is advisory only.

	public boolean isDiskSpaceAvailable() {

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// This is called regularily by the run manager to monitor and progress run state.  Run state is defined by the
	// subclass however desired.

	public abstract void poll();


	//-----------------------------------------------------------------------------------------------------------------

	public abstract boolean isRunning();


	//-----------------------------------------------------------------------------------------------------------------

	public abstract boolean isWaiting();


	//-----------------------------------------------------------------------------------------------------------------

	public abstract void bumpTask();


	//-----------------------------------------------------------------------------------------------------------------

	public abstract boolean runFailed();


	//-----------------------------------------------------------------------------------------------------------------
	// A cancel() is a request only, it is not guaranteed to stop activity immediately, or at all.  If canCancel()
	// returns false cancel() may be ignored, and that state may change as a run proceeds.

	public abstract void cancel();


	//-----------------------------------------------------------------------------------------------------------------

	public boolean canCancel() {

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public abstract String getStatus();


	//-----------------------------------------------------------------------------------------------------------------
	// These are typically wrappers for methods in ProcessPanel, but some subclasses may have multiple such panels.

	public abstract boolean hasOutput();


	//-----------------------------------------------------------------------------------------------------------------

	public abstract void writeOutputTo(Writer theWriter) throws IOException;


	//-----------------------------------------------------------------------------------------------------------------
	// May be overridden if subclass also has a report.

	public boolean hasReport() {

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void writeReportTo(Writer theWriter) throws IOException {
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Return output directory for auto-saved files, overridden to return e.g. study output directory, and method
	// called by the run manager to handle auto-save of output and/or report as needed.

	public Path getOutDirectoryPath() {

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void autoSave() {

		Path outDir = getOutDirectoryPath();
		if (null == outDir) {
			return;
		}

		if (autoSaveOutput && hasOutput()) {

			autoSaveOutput = false;

			BufferedWriter theWriter = null;
			try {
				theWriter = Files.newBufferedWriter(outDir.resolve(LOG_FILE_NAME));
			} catch (IOException ie) {
				return;
			}

			try {writeOutputTo(theWriter);} catch (IOException ie) {}

			try {theWriter.close();} catch (IOException ie) {}
		}

		if (autoSaveReport && hasReport()) {

			autoSaveReport = false;

			BufferedWriter theWriter = null;
			try {
				theWriter = Files.newBufferedWriter(outDir.resolve(REPORT_FILE_NAME));
			} catch (IOException ie) {
				return;
			}

			try {writeReportTo(theWriter);} catch (IOException ie) {}

			try {theWriter.close();} catch (IOException ie) {}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// UI convenience methods to save output or report, these depend on writeOutputTo() and writeReportTo().

	public void saveOutput() {

		if (isRunning() || !hasOutput()) {
			return;
		}

		String title = "Save Run Log";
		errorReporter.setTitle(title);

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.SAVE_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);

		File theFile = null;
		do {
			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Save")) {
				return;
			}
			theFile = chooser.getSelectedFile();
			if (theFile.exists()) {
				AppController.beep();
				if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
						"The file exists, do you want to replace it?", title, JOptionPane.YES_NO_OPTION,
						JOptionPane.WARNING_MESSAGE)) {
					theFile = null;
				}
			}
		} while (null == theFile);

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

		FileWriter theWriter = null;
		try {
			theWriter = new FileWriter(theFile);
		} catch (IOException ie) {
			errorReporter.reportError("Could not open the file:\n" + ie.getMessage());
			return;
		}

		try {
			writeOutputTo(theWriter);
		} catch (IOException ie) {
			errorReporter.reportError("Could not write to the file:\n" + ie.getMessage());
		}

		try {theWriter.close();} catch (IOException ie) {}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void saveReport() {

		if (isRunning() || !hasReport()) {
			return;
		}

		String title = "Save Report";
		errorReporter.setTitle(title);

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.SAVE_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);

		File theFile = null;
		do {
			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Save")) {
				return;
			}
			theFile = chooser.getSelectedFile();
			if (theFile.exists()) {
				AppController.beep();
				if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
						"The file exists, do you want to replace it?", title, JOptionPane.YES_NO_OPTION,
						JOptionPane.WARNING_MESSAGE)) {
					theFile = null;
				}
			}
		} while (null == theFile);

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

		FileWriter theWriter = null;
		try {
			theWriter = new FileWriter(theFile);
		} catch (IOException ie) {
			errorReporter.reportError("Could not open the file:\n" + ie.getMessage());
			return;
		}

		try {
			writeReportTo(theWriter);
		} catch (IOException ie) {
			errorReporter.reportError("Could not write to the file:\n" + ie.getMessage());
		}

		try {theWriter.close();} catch (IOException ie) {}
	}
}
