//
//  ProcessPanel.java
//  TVStudy
//
//  Copyright (c) 2012-2021 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui.run;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.gui.*;

import java.util.*;
import java.util.logging.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;


//=====================================================================================================================
// Panel interface to create and manage a ProcessRun object and accumulate/display it's output.  This is intended to
// be displayed in a window and remain visible as long as the process is running.  A manager must call poll() often
// which in turn calls ProcessRun.poll().  The ProcessRun.Handler interface is used to get process output back from
// the process, and to handle the interactive prompt-and-response protocol some processes may support.  To use that
// capability a parent object must set a ProcessPanel.Handler so methods can be forwarded.  Other methods here forward
// to the ProcessRun object, e.g. isProcessRunning().

// This has a pre-run state, until the argument list is set with setProcessArguments(), the panel is displayable but
// the process will not be started.  The displayLogMessage() method can be used to show messages in the same manner as
// later output from the process.  Once arguments are set, the the process will start on the next poll.

// All process output is accumulated for possible log file output as well as being displayed.  To avoid out-of-memory
// problems, a temporary disk file is used and only a limited number of lines stay in the view for scrollback.  After
// the process ends, hasOutput() will indicate if there is output accumulated regardless of success or failure.  The
// output can be saved with writeOutputTo().

// These are one-shot objects, this will only create a ProcessRun once.  If a cancel() occurs before the process is
// created/started, startup will never occur.

public class ProcessPanel extends AppPanel implements ProcessRun.Handler {

	public static final int TEXT_AREA_ROWS = 15;
	public static final int TEXT_AREA_COLUMNS = 80;

	private String processName;

	private ProcessRun processRun;
	private boolean processStarted;
	private boolean processExited;

	private Handler handler;

	private File outputFile;
	private boolean outputFileOpened;
	private FileWriter outputFileWriter;
	private boolean hasOutput;
	private static final int COPY_BUFFER_SIZE = 1048576;

	private JTextArea outputArea;
	private ArrayList<String> mergeLineStrings;
	private int lastOutputLineType;
	private static final int CR_LINE = -1;
	private int lastOutputLineStart;
	private int scrollbackLineCount;
	private static final int MAX_SCROLLBACK_LINES = 15000;

	private JViewport outputViewport;
	private int autoScrollState;
	private boolean autoScrollLock;

	private JLabel statusLabel;
	private JButton cancelButton;
	private JPanel statusPanel;

	private boolean canceled;

	private ProcessPanel outerThis = this;


	//=================================================================================================================
	// Interface for objects that can be interactive handlers for a ProcessPanel.

	public interface Handler {


		//-------------------------------------------------------------------------------------------------------------
		// Inline key=data messages from the process that aren't known to ProcessPanel are sent to the handler.

		public void processStatusMessage(ProcessPanel thePanel, String theKey, String theData);


		//-------------------------------------------------------------------------------------------------------------
		// Provide response to a prompt from the process, or null if no response is known (invalid or unexpected
		// prompt) in which case process will be killed.  When returning a response, the implementer must enter a
		// pending state until confirmation or failure.  Only one response per process may be pending at any given
		// time.  In the pending state this should not be called, if it is it may return the pending response string
		// again, or return null.

		public String getProcessResponse(ProcessPanel thePanel, String thePrompt);


		//-------------------------------------------------------------------------------------------------------------
		// Process confirmed the response, clear pending state.

		public void processResponseConfirmed(ProcessPanel thePanel);


		//-------------------------------------------------------------------------------------------------------------
		// Process failed for any reason, including but not limited to failure to confirm a response.

		public void processFailed(ProcessPanel thePanel);


		//-------------------------------------------------------------------------------------------------------------
		// Process exited without error.  However if this happens while a response is pending, it might be an error.

		public void processComplete(ProcessPanel thePanel);


		//-------------------------------------------------------------------------------------------------------------
		// The handler can disable cancel if needed, this is polled so it can change.

		public boolean canCancel();


		//-------------------------------------------------------------------------------------------------------------
		// Called from setStatusMessage().

		public void setStatusMessage(String mesg);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// If the name string is null, "Process" is used.  See below for details of the merge-line string list.

	public ProcessPanel(AppEditor theParent, String theProcessName, ArrayList<String> theMergeLineStrings) {

		super(theParent);

		if (null == theProcessName) {
			processName = "Process";
		} else {
			processName = theProcessName;
		}

		if ((null != theMergeLineStrings) && !theMergeLineStrings.isEmpty()) {
			mergeLineStrings = new ArrayList<String>(theMergeLineStrings);
		}

		// Set up the UI, main element is a text area displaying output from the process.  All output is written to a
		// temporary disk file, the most-recent lines are displayed for scroll-back viewing.

		outputArea = new JTextArea(TEXT_AREA_ROWS, TEXT_AREA_COLUMNS);
		outputArea.setFont(new Font("Monospaced", Font.PLAIN, 12));
		AppController.fixKeyBindings(outputArea);
		outputArea.setEditable(false);
		outputArea.setLineWrap(true);

		// The output text area has an auto-scroll behavior.  Initially the viewport follows new text as it is added.
		// If the user moves the scroll bar away from the bottom, auto-scroll stops; if the scroll bar is moved to the
		// bottom again, auto-scroll resumes.  Be sure the caret doesn't cause any scrolling.

		((DefaultCaret)outputArea.getCaret()).setUpdatePolicy(DefaultCaret.NEVER_UPDATE);

		JScrollPane outPane = AppController.createScrollPane(outputArea);
		outputViewport = outPane.getViewport();

		outPane.getVerticalScrollBar().addAdjustmentListener(new AdjustmentListener() {
			public void adjustmentValueChanged(AdjustmentEvent theEvent) {
				if (autoScrollLock) {
					return;
				}
				if (outputViewport.getViewSize().height <= outputViewport.getExtentSize().height) {
					autoScrollState = 0;
				}
				if (0 == autoScrollState) {
					return;
				}
				Adjustable theAdj = theEvent.getAdjustable();
				boolean atBot = ((theAdj.getMaximum() - theAdj.getVisibleAmount() - theEvent.getValue()) < 10);
				if (1 == autoScrollState) {
					if (!atBot) {
						autoScrollState = 2;
					}
				} else {
					if (atBot) {
						autoScrollState = 1;
					}
				}
			}
		});

		// A status message.

		statusLabel = new JLabel(processName + " starting");
		statusLabel.setPreferredSize(AppController.labelSize[60]);

		// Cancel button.

		cancelButton = new JButton("Cancel");
		cancelButton.setFocusable(false);
		cancelButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				cancel();
			}
		});
		cancelButton.setEnabled(false);

		// Layout.

		statusPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		statusPanel.add(cancelButton);
		statusPanel.add(statusLabel);

		setLayout(new BorderLayout());

		add(outPane, BorderLayout.CENTER);
		add(statusPanel, BorderLayout.SOUTH);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Close and delete the temporary output file if needed.

	public void finalize() {

		if (null != outputFile) {

			if (null != outputFileWriter) {
				try {
					outputFileWriter.close();
				} catch (IOException ie) {
				}
				outputFileWriter = null;
			}

			outputFile.delete();
			outputFile = null;
			hasOutput = false;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set a process handler, see poll().

	public void setProcessHandler(Handler theHandler) {

		handler = theHandler;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Hide or show the cancel button and status message.

	public void setStatusPanelVisible(boolean flag) {

		statusPanel.setVisible(flag);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set text in the status label, note there is no priority here, the most-recent message is always shown whether
	// set by this method or by internal state changes.  Also forwarded to handler.

	public void setStatusMessage(String theMessage) {

		statusLabel.setText(theMessage);
		if (null != handler) {
			handler.setStatusMessage(theMessage);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Show an arbitrary message in the display area and write to the log file.  The message may contain multiple
	// lines, blank lines are ignored but otherwise all lines are parsed and displayed/logged as they would be during
	// output processing from a controlled process, see processOutput() for details.

	public void displayLogMessage(String theMessage) {

		if (null == theMessage) {
			return;
		}

		int position = 0, nextPosition = 0, length = theMessage.length(), lineType = 0, stringIndex;
		char nextChar;
		String line = null;

		while (position < length) {

			do {
				nextChar = theMessage.charAt(nextPosition);
			} while (('\n' != nextChar) && ('\r' != nextChar) && (++nextPosition < length));

			if (nextPosition > position) {
				line = theMessage.substring(position, nextPosition);
				if (nextPosition < length) {
					++nextPosition;
				}
				position = nextPosition;
			} else {
				position = ++nextPosition;
				continue;
			}

			if ('\r' == nextChar) {
				lineType = CR_LINE;
			} else {

				writeLine(line);

				lineType = 0;
				if (null != mergeLineStrings) {
					for (stringIndex = 0; stringIndex < mergeLineStrings.size(); stringIndex++) {
						if (line.contains(mergeLineStrings.get(stringIndex))) {
							lineType = stringIndex + 1;
							break;
						}
					}
				}
			}

			displayLine(line, lineType);
		}

		updateScroll();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// If the process has not yet been created, create it and configure, the next poll() will start the run.  If the
	// password string is non-null the first output from the process must be a password prompt, see ProcessRun.

	public void setProcessArguments(ArrayList<String> theArgumentList, String thePassword) {

		if ((null == processRun) && !processStarted) {
			processRun = new ProcessRun(theArgumentList, thePassword);
			processRun.setHandler(this);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Poll the process status and update the UI, this must be called frequently and only from the Swing event thread.
	// Returns true if running or waiting for arguments to start, false otherwise, including if startup was canceled.

	public boolean poll() {

		if (processExited) {
			return false;
		}

		cancelButton.setEnabled(!canceled && ((null == handler) || handler.canCancel()));

		if (!processStarted) {

			if (null == processRun) {
				return true;
			}
			processStarted = true;

			if (processRun.poll()) {

				setStatusMessage(processName + " running");

			} else {

				processExited = true;
				setStatusMessage(processName + " startup failed");
				cancelButton.setEnabled(false);
			}

		} else {

			if (!processRun.poll()) {

				processExited = true;
				if (canceled) {
					setStatusMessage(processName + " canceled");
				} else {
					if (processRun.wasStopped()) {
						setStatusMessage(processName + " aborted");
					} else {
						if (processRun.didFail()) {
							setStatusMessage(processName + " exited with error");
						} else {
							setStatusMessage(processName + " complete");
						}
					}
				}
				cancelButton.setEnabled(false);
			}
		}

		return !processExited;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// ProcessRun.Handler methods, handle output from the process.  Determine a line type based on content, if the
	// mergeLineStrings array is set, when two sequential lines both with a contains() match to the same merge string
	// arrive, the second one will overwrite the first to condense the output when a series of output lines are mostly
	// similar.  Lines ending with carriage return will always be overwritten by the next line regardless of content.
	// CR-terminated lines do not go to the output file, but all others do even if they are the same overwrite type.
	// A maximum number of lines are kept in the view for scrollback, as the limit is reached older lines are removed.

	public void processOutput(ProcessRun theRun, String line, boolean crTerminated) {

		if (theRun != processRun) {
			return;
		}

		int lineType = 0;
		if (crTerminated) {
			lineType = CR_LINE;
		} else {

			writeLine(line);

			if (null != mergeLineStrings) {
				for (int stringIndex = 0; stringIndex < mergeLineStrings.size(); stringIndex++) {
					if (line.contains(mergeLineStrings.get(stringIndex))) {
						lineType = stringIndex + 1;
						break;
					}
				}
			}
		}

		displayLine(line, lineType);

		updateScroll();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Handle key=data messages from a study engine process.  Some are handled here, specifically progress messages
	// written during lengthy operations so the process does not appear stuck.  Those display in the view but are not
	// written to the output file.  Other key=data messages are sent to the handler if set, else are ignored.  Also
	// progress messages are always overwritten by the next message regardless of content.

	public void processMessage(ProcessRun theRun, String key, String data) {

		if (theRun != processRun) {
			return;
		}

		if (key.equals(AppCore.ENGINE_PROGRESS_KEY)) {

			displayLine(data, CR_LINE);
			updateScroll();

		} else {

			if (null != handler) {
				handler.processStatusMessage(this, key, data);
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Forward the interactive processing messages to the handler if set.

	public String getProcessResponse(ProcessRun theRun, String prompt) {

		if ((theRun == processRun) && (null != handler)) {
			return handler.getProcessResponse(this, prompt);
		}
		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void processResponseConfirmed(ProcessRun theRun) {

		if ((theRun == processRun) && (null != handler)) {
			handler.processResponseConfirmed(this);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void processFailed(ProcessRun theRun) {

		if ((theRun == processRun) && (null != handler)) {
			handler.processFailed(this);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void processComplete(ProcessRun theRun) {

		if ((theRun == processRun) && (null != handler)) {
			handler.processComplete(this);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Display a line of text in the display area, possibly over-writing the previous line if that was a matching line
	// type or was terminated with a carriage return.

	private void displayLine(String newLine, int newLineType) {

		int outputEnd = outputArea.getDocument().getLength();

		if (((0 != newLineType) && (newLineType == lastOutputLineType)) || (CR_LINE == lastOutputLineType)) {

			outputArea.replaceRange(newLine, lastOutputLineStart, outputEnd);

		} else {

			lastOutputLineStart = outputEnd;
			outputArea.append(newLine);
			scrollbackLineCount++;
		}

		outputArea.append("\n");

		lastOutputLineType = newLineType;
	}


	//-----------------------------------------------------------------------------------------------------------------	
	// Apply the scroll-back limit and the auto-scroll behavior, called after one or more calls to displayLine().
	// When the maximum scroll-back size is reached 10% of the lines are removed at once, removing line-by-line
	// causes excessive CPU load when lines are being displayed rapidly.

	private void updateScroll() {

		if (scrollbackLineCount > MAX_SCROLLBACK_LINES) {

			try {

				int removeLines = scrollbackLineCount / 10;
				int removeTo = outputArea.getLineEndOffset(removeLines);
				outputArea.getDocument().remove(0, removeTo);
				scrollbackLineCount -= removeLines;
				lastOutputLineStart -= removeTo;

			} catch (BadLocationException ble) {
				outputArea.setText("");
				scrollbackLineCount = 0;
				lastOutputLineStart = 0;
				lastOutputLineType = 0;
			}
		}

		autoScrollLock = true;

		outputViewport.validate();
		if (outputViewport.getViewSize().height <= outputViewport.getExtentSize().height) {
			autoScrollState = 0;
		} else {
			if (0 == autoScrollState) {
				autoScrollState = 1;
			}
			if (1 == autoScrollState) {
				outputViewport.scrollRectToVisible(
					new Rectangle(0, (outputViewport.getViewSize().height - 1), 1, 1));
			}
		}

		autoScrollLock = false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Write a line to the log file, open file as needed.

	private void writeLine(String line) {

		try {
			if (!outputFileOpened) {
				outputFileOpened = true;
				outputFile = File.createTempFile(processName, ".log");
				outputFileWriter = new FileWriter(outputFile);
			}
			if (null != outputFileWriter) {
				outputFileWriter.write(line);
				outputFileWriter.write('\n');
				hasOutput = true;
			}
		} catch (IOException ie) {
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// If the process has not yet started, cancel startup, otherwise this will be picked up by the process in poll()
	// which checks isCanceled() and will attempt an orderly shutdown.  If a handler is set that can disable cancel.

	public void cancel() {

		if (canceled || ((null != handler) && !handler.canCancel())) {
			return;
		}

		canceled = true;
		cancelButton.setEnabled(false);

		if (!processStarted) {
			processStarted = true;
			processExited = true;
			setStatusMessage(processName + " canceled");
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean isCanceled() {

		return canceled;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// This returns true if the process has not yet started.

	public boolean isProcessRunning() {

		if (null != processRun) {
			return processRun.isRunning();
		}
		return !processExited;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean didProcessFail() {

		if (null != processRun) {
			return processRun.didFail();
		}
		return processExited;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean hasOutput() {

		return (!isProcessRunning() && hasOutput);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Copy the output file contents, if any, to a writer.  This closes the output file so further messages (sent to
	// displayLogMessage(), there can't be any further process output) would be displayed but cannot be saved.

	public void writeOutputTo(Writer theWriter) throws IOException {

		if (isProcessRunning() || !hasOutput) {
			return;
		}

		if (null != outputFileWriter) {
			try {outputFileWriter.close();} catch (IOException ie) {}
			outputFileWriter = null;
		}

		IOException rethrow = null;

		try {

			FileReader theReader = new FileReader(outputFile);

			char[] cbuf = new char[COPY_BUFFER_SIZE];
			int len;

			do {
				len = theReader.read(cbuf, 0, COPY_BUFFER_SIZE);
				if (len > 0) {
					theWriter.write(cbuf, 0, len);
				}
			} while (len >= 0);

			theReader.close();

		} catch (IOException ie) {
			rethrow = ie;
		}

		if (null != rethrow) {
			throw rethrow;
		}			
	}
}
