//
//  AnalyzeRunResult.java
//  TVStudy
//
//  Copyright (c) 2018-2023 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui.run;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.gui.*;

import java.util.*;
import java.io.*;
import java.sql.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.table.*;
import javax.swing.filechooser.*;


//=====================================================================================================================
// Class for running custom queries on study run result database tables.  The study to analyze is set with setStudy().
// This holds an edit lock on the study to prevent concurrent changes.  Manually-composed queries can be saved in a
// study database table, changes to that table are made directly here so this does not hold editor state.

public class AnalyzeRunResult extends AppFrame implements StudyLockHolder {
	
	public static final String WINDOW_TITLE = "Analyze Run Result";

	private static final int KEY_COVERAGE = -1;
	private static final int KEY_COVERAGE_LOSS = -2;
	private static final int KEY_COVERAGE_GAIN = -3;
	private static final int KEY_COVERAGE_OVERLAP = -4;
	private static final int KEY_CUSTOM_QUERY = -5;

	private static final String NAME_COVERAGE = "Coverage";
	private static final String NAME_COVERAGE_LOSS = "Coverage loss";
	private static final String NAME_COVERAGE_GAIN = "Coverage gain";
	private static final String NAME_COVERAGE_OVERLAP = "Coverage overlap";

	private Study study;

	private ArrayList<KeyedRecord> scenarioList;
	private HashMap<Integer, ScenarioResult> scenarioMap;

	private HashMap<Integer, SavedQuery> queryMap;

	private KeyedRecordMenu queryMenu;
	private JButton editButton;
	private JMenuItem editMenuItem;
	private JMenuItem deleteMenuItem;
	private KeyedRecordMenu scenarioAMenu;
	private KeyedRecordMenu scenarioBMenu;
	private KeyedRecordMenu countryMenu;
	private JCheckBox excludeZeroPopCheckBox;
	private JButton modifyButton;
	private JMenuItem modifyMenuItem;
	private JButton newButton;
	private JMenuItem newMenuItem;
	private JButton runButton;
	private JMenuItem runMenuItem;
	private JButton viewButton;
	private JMenuItem viewMenuItem;
	private JButton saveCSVButton;
	private JMenuItem saveCSVMenuItem;
	private JButton saveKMLButton;
	private JMenuItem saveKMLMenuItem;
	private JButton closeButton;

	private QueryEditDialog editDialog;
	private SavedQuery customQuery;

	private String runDescription = "";
	private String runQuery = "";
	private boolean noResult;
	private Integer baseKey;
	private int baseCountry;
	private String pointQuery;

	private QueryRunner queryRunner;

	private boolean queryRunning;
	private Thread queryThread;
	private static final int TIMER_INTERVAL = 200;   // milliseconds
	private javax.swing.Timer queryTimer;

	private JButton cancelButton;

	private JTextArea resultArea;
	private ResultTableModel tableModel;

	private boolean didRun;
	private ArrayList<String[]> results;
	private String[] resultColumnNames;
	private int resultNameColumn = -1;
	private int resultLatitudeColumn = -1;
	private int resultLongitudeColumn = -1;
	private boolean canSaveKML;


	//-----------------------------------------------------------------------------------------------------------------

	public AnalyzeRunResult(AppEditor theParent) {

		super(theParent, WINDOW_TITLE);

		// Data for scenarios with results tables, see setStudy().

		scenarioList = new ArrayList<KeyedRecord>();
		scenarioMap = new HashMap<Integer, ScenarioResult>();

		// UI actions clear previous query results.

		ActionListener clearResultListener = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (queryRunning) {
					return;
				}
				if (blockActions()) {
					clearResult();
					blockActionsEnd();
				}
			}
		};

		// Map of saved queries, menu of all available queries, will be updated in updateMenu().

		queryMap = new HashMap<Integer, SavedQuery>();

		queryMenu = new KeyedRecordMenu();
		queryMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXyXyXyXy"));
		queryMenu.addActionListener(clearResultListener);

		// Menus for selecting scenarios for analysis, will be populated in setStudy().

		scenarioAMenu = new KeyedRecordMenu();
		scenarioAMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXyXyXyXy"));
		scenarioAMenu.addActionListener(clearResultListener);

		scenarioBMenu = new KeyedRecordMenu();
		scenarioBMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXyXyXyXy"));
		scenarioBMenu.addActionListener(clearResultListener);

		// Restrict query result to selected country.

		ArrayList<KeyedRecord> countries = Country.getCountries();
		countries.add(0, new KeyedRecord(0, "(all)"));
		countryMenu = new KeyedRecordMenu(countries);
		countryMenu.addActionListener(clearResultListener);

		// Exclude zero-population points from result.

		excludeZeroPopCheckBox = new JCheckBox("Exclude zero-pop points");
		excludeZeroPopCheckBox.addActionListener(clearResultListener);
		excludeZeroPopCheckBox.setSelected(true);

		// Dialog for editing query.

		editDialog = new QueryEditDialog(this);

		// Area to display results.

		resultArea = new JTextArea(8, 40);
		resultArea.setFont(new Font("Monospaced", Font.PLAIN, 12));
		AppController.fixKeyBindings(resultArea);
		resultArea.setEditable(false);
		resultArea.setLineWrap(true);

		// Model to view results in table.

		tableModel = new ResultTableModel();

		// Buttons.

		editButton = new JButton("Edit");
		editButton.setFocusable(false);
		editButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doEdit();
			}
		});

		modifyButton = new JButton("Modify Query");
		modifyButton.setFocusable(false);
		modifyButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doModify();
			}
		});

		newButton = new JButton("New Query");
		newButton.setFocusable(false);
		newButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doNew();
			}
		});

		runButton = new JButton("Run");
		runButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRun();
			}
		});

		viewButton = new JButton("View");
		viewButton.setFocusable(false);
		viewButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doView();
			}
		});

		saveCSVButton = new JButton("Save CSV");
		saveCSVButton.setFocusable(false);
		saveCSVButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSaveCSV();
			}
		});

		saveKMLButton = new JButton("Save KML");
		saveKMLButton.setFocusable(false);
		saveKMLButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSaveKML();
			}
		});

		closeButton = new JButton("Close");
		closeButton.setFocusable(false);
		closeButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				closeWithoutSave();
			}
		});

		cancelButton = new JButton("Cancel");
		cancelButton.setFocusable(false);
		cancelButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				cancelQuery();
			}
		});
		cancelButton.setEnabled(false);

		// Layout.

		JPanel row1P = new JPanel(new FlowLayout(FlowLayout.LEFT));
		row1P.add(new JLabel("Query"));
		row1P.add(queryMenu);
		row1P.add(editButton);

		JPanel row2P = new JPanel(new FlowLayout(FlowLayout.LEFT));
		row2P.add(new JLabel("Scenario A"));
		row2P.add(scenarioAMenu);
		row2P.add(new JLabel("Country"));
		row2P.add(countryMenu);

		JPanel row3P = new JPanel(new FlowLayout(FlowLayout.LEFT));
		row3P.add(new JLabel("Scenario B"));
		row3P.add(scenarioBMenu);
		row3P.add(excludeZeroPopCheckBox);

		Box row4lB = Box.createVerticalBox();
		row4lB.add(modifyButton);
		row4lB.add(newButton);

		JPanel row4lP = new JPanel(new FlowLayout(FlowLayout.LEFT));
		row4lP.add(row4lB);

		Box row4rB = Box.createVerticalBox();
		row4rB.add(saveCSVButton);
		row4rB.add(saveKMLButton);

		JPanel row4rP = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		row4rP.add(runButton);
		row4rP.add(viewButton);
		row4rP.add(row4rB);

		Box row4B = Box.createHorizontalBox();
		row4B.add(row4lP);
		row4B.add(row4rP);

		Box topB = Box.createVerticalBox();
		topB.add(row1P);
		topB.add(row2P);
		topB.add(row3P);
		topB.add(row4B);

		JPanel butlP = new JPanel(new FlowLayout(FlowLayout.LEFT));
		butlP.add(cancelButton);

		JPanel butrP = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		butrP.add(closeButton);

		Box butB = Box.createHorizontalBox();
		butB.add(butlP);
		butB.add(butrP);

		setLayout(new BorderLayout());
		add(topB, BorderLayout.NORTH);
		add(AppController.createScrollPane(resultArea), BorderLayout.CENTER);
		add(butB, BorderLayout.SOUTH);

		getRootPane().setDefaultButton(runButton);

		setResizable(true);
		setLocationSaved(true);

		pack();

		setMinimumSize(getSize());

		// Build the Query menu.

		fileMenu.removeAll();

		// Edit

		editMenuItem = new JMenuItem("Edit");
		editMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, AppController.MENU_SHORTCUT_KEY_MASK));
		editMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doEdit();
			}
		});
		fileMenu.add(editMenuItem);

		// Delete

		deleteMenuItem = new JMenuItem("Delete");
		deleteMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doDelete();
			}
		});
		fileMenu.add(deleteMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// Modify

		modifyMenuItem = new JMenuItem("Modify");
		modifyMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, AppController.MENU_SHORTCUT_KEY_MASK));
		modifyMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doModify();
			}
		});
		fileMenu.add(modifyMenuItem);

		// New

		newMenuItem = new JMenuItem("New");
		newMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, AppController.MENU_SHORTCUT_KEY_MASK));
		newMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doNew();
			}
		});
		fileMenu.add(newMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// Run

		runMenuItem = new JMenuItem("Run");
		runMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, AppController.MENU_SHORTCUT_KEY_MASK));
		runMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRun();
			}
		});
		fileMenu.add(runMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// View Result

		viewMenuItem = new JMenuItem("View Table");
		viewMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T, AppController.MENU_SHORTCUT_KEY_MASK));
		viewMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doView();
			}
		});
		fileMenu.add(viewMenuItem);

		// Save CSV File

		saveCSVMenuItem = new JMenuItem("Save CSV File");
		saveCSVMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, AppController.MENU_SHORTCUT_KEY_MASK));
		saveCSVMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSaveCSV();
			}
		});
		fileMenu.add(saveCSVMenuItem);

		// Save KML File

		saveKMLMenuItem = new JMenuItem("Save KML File");
		saveKMLMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_K, AppController.MENU_SHORTCUT_KEY_MASK));
		saveKMLMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSaveKML();
			}
		});
		fileMenu.add(saveKMLMenuItem);

		// Timer to check on a running query.

		queryTimer = new javax.swing.Timer(TIMER_INTERVAL, new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				checkQuery();
			}
		});
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected String getFileMenuName() {

		return "Query";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void updateDocumentName() {

		String docName = parent.getDocumentName();
		if (null != study) {
			if (null == docName) {
				docName = study.name;
			} else {
				docName = docName + "/" + study.name;
			}
		}
		setDocumentName(docName);
	}


	//=================================================================================================================
	// Data class to hold information on a scenario with a result table, including total coverage for the scenario.

	private static class ScenarioResult {

		Integer key;
		String name;

		boolean hasTable;

		boolean hasResult;
		int resultCountry;
		double area;
		int population;
		int households;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set the study, populate the scenario list.  Must be called before showing the window, and only called once.  A
	// query is run to get the existing results tables in the study database.  Only scenarios with results are shown
	// in the menus.  Caller is responsible for locking/unlocking the study.

	public boolean setStudy(Study theStudy, ErrorReporter errors) {

		if (isVisible() || (null != study)) {
			return false;
		}

		// Get list of result tables.

		DbConnection db = DbCore.connectDb(theStudy.dbID, errors);
		if (null == db) {
			return false;
		}

		scenarioList.clear();
		scenarioMap.clear();

		ScenarioResult theScenario;
		for (Scenario scen : theStudy.scenarios) {
			theScenario = new ScenarioResult();
			theScenario.key = Integer.valueOf(scen.key);
			theScenario.name = scen.name;
			scenarioMap.put(theScenario.key, theScenario);
		}

		try {

			String sdbname = DbCore.getDbName(theStudy.dbID) + "_" + String.valueOf(theStudy.key);

			String str;
			Integer skey;

			db.query("SHOW TABLES IN " + sdbname + " LIKE 'result\\_%'");
			while (db.next()) {
				str = db.getString(1);
				skey = null;
				try {
					skey = Integer.valueOf(str.substring(7));
				} catch (NumberFormatException e) {
				}
				if (null != skey) {
					theScenario = scenarioMap.get(skey);
					if (null != theScenario) {
						theScenario.hasTable = true;
						scenarioList.add(new KeyedRecord(skey.intValue(), theScenario.name));
					}
				}
			}

			// Databases don't by default have the table used to save queries; create that table here if needed.

			db.query("SHOW TABLES IN " + sdbname + " LIKE 'saved_query'");
			if (!db.next()) {
				db.update("CREATE TABLE " + sdbname + ".saved_query (query_key INT NOT NULL PRIMARY KEY, " +
					"name VARCHAR(255) NOT NULL, description VARCHAR(255) NOT NULL, query MEDIUMTEXT NOT NULL, " +
					"no_result BOOLEAN NOT NULL)");
			}

		} catch (SQLException se) {
			DbCore.releaseDb(db);
			db.reportError(errors, se);
			return false;
		}

		DbCore.releaseDb(db);

		if (scenarioList.isEmpty()) {
			errors.reportWarning("The study does not have any result data to analyze");
			return false;
		}

		// Ready to go, set state.

		study = theStudy;

		Comparator<KeyedRecord> comp = new Comparator<KeyedRecord>() {
			public int compare(KeyedRecord theRecord, KeyedRecord otherRecord) {
				if (theRecord.key < otherRecord.key) {
					return -1;
				} else {
					if (theRecord.key > otherRecord.key) {
						return 1;
					}
				}
				return 0;
			}
		};
		scenarioList.sort(comp);

		editDialog.setScenarios(scenarioList);

		scenarioAMenu.addAllItems(scenarioList);
		scenarioBMenu.addAllItems(scenarioList);
		if (scenarioList.size() > 1) {
			scenarioBMenu.setSelectedIndex(1);
		}

		updateDocumentName();

		updateMenu(null);

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update UI state per query selection and past run state, use argument if non-null else use menu selection.

	private void updateControls(SavedQuery theQuery) {

		if (queryRunning) {

			AppController.setComponentEnabled(queryMenu, false);

			editButton.setEnabled(false);
			editMenuItem.setEnabled(false);
			deleteMenuItem.setEnabled(false);

			AppController.setComponentEnabled(scenarioAMenu, false);
			AppController.setComponentEnabled(scenarioBMenu, false);
			AppController.setComponentEnabled(countryMenu, false);
			AppController.setComponentEnabled(excludeZeroPopCheckBox, false);

			modifyButton.setEnabled(false);
			modifyMenuItem.setEnabled(false);
			newButton.setEnabled(false);
			newMenuItem.setEnabled(false);

			runButton.setEnabled(false);
			runMenuItem.setEnabled(false);

			viewButton.setEnabled(false);
			viewMenuItem.setEnabled(false);
			saveCSVButton.setEnabled(false);
			saveCSVMenuItem.setEnabled(false);
			saveKMLButton.setEnabled(false);
			saveKMLMenuItem.setEnabled(false);

			closeButton.setEnabled(false);

		} else {

			int qkey;
			if (null != theQuery) {
				qkey = theQuery.key.intValue();
			} else {
				qkey = queryMenu.getSelectedKey();
			}

			AppController.setComponentEnabled(queryMenu, true);

			boolean enable = (qkey > 0);
			editButton.setEnabled(enable);
			editMenuItem.setEnabled(enable);
			deleteMenuItem.setEnabled(enable);

			enable = ((KEY_COVERAGE == qkey) || (KEY_COVERAGE_LOSS == qkey) || (KEY_COVERAGE_GAIN == qkey) ||
				(KEY_COVERAGE_OVERLAP == qkey));
			AppController.setComponentEnabled(scenarioAMenu, enable);
			AppController.setComponentEnabled(countryMenu, enable);
			AppController.setComponentEnabled(excludeZeroPopCheckBox, enable);
			enable = ((KEY_COVERAGE_LOSS == qkey) || (KEY_COVERAGE_GAIN == qkey) || (KEY_COVERAGE_OVERLAP == qkey));
			AppController.setComponentEnabled(scenarioBMenu, enable);

			modifyButton.setEnabled(true);
			modifyMenuItem.setEnabled(true);
			newButton.setEnabled(true);
			newMenuItem.setEnabled(true);

			runButton.setEnabled(!didRun);
			runMenuItem.setEnabled(!didRun);

			enable = (didRun && (null != results));
			viewButton.setEnabled(enable);
			viewMenuItem.setEnabled(enable);
			saveCSVButton.setEnabled(enable);
			saveCSVMenuItem.setEnabled(enable);
			saveKMLButton.setEnabled(enable && canSaveKML);
			saveKMLMenuItem.setEnabled(enable & canSaveKML);

			closeButton.setEnabled(true);
		}
	}


	//=================================================================================================================
	// Data class to hold a saved query.

	private static class SavedQuery {

		Integer key;
		String name;
		String description;
		String query;
		boolean noResult;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Load saved queries and populate the query menu.  There are always hard-coded items for gain, loss, etc., plus
	// saved queries.  A desired menu selection may be passed and will be selected regardless of whether it appears in
	// the menu (so may be added as a temporary item) else the existing selection is preserved if possible.

	private void updateMenu(SavedQuery selectThis) {

		if (queryRunning) {
			return;
		}

		blockActionsStart();

		queryMap.clear();

		int selectKey = KEY_COVERAGE;
		if ((null == selectThis) && (queryMenu.getItemCount() > 0)) {
			selectKey = queryMenu.getSelectedKey();
		}

		queryMenu.removeAllItems();

		queryMenu.addItem(new KeyedRecord(KEY_COVERAGE, NAME_COVERAGE));
		queryMenu.addItem(new KeyedRecord(KEY_COVERAGE_LOSS, NAME_COVERAGE_LOSS));
		queryMenu.addItem(new KeyedRecord(KEY_COVERAGE_GAIN, NAME_COVERAGE_GAIN));
		queryMenu.addItem(new KeyedRecord(KEY_COVERAGE_OVERLAP, NAME_COVERAGE_OVERLAP));

		DbConnection db = DbCore.connectDb(study.dbID);
		if (null != db) {
			try {

				SavedQuery q;

				db.query("SELECT query_key, name, description, query, no_result FROM " + DbCore.getDbName(study.dbID) +
					"_" + String.valueOf(study.key) + ".saved_query ORDER BY 1");

				while (db.next()) {

					q = new SavedQuery();
					q.key = Integer.valueOf(db.getInt(1));
					q.name = db.getString(2);
					q.description = db.getString(3);
					q.query = db.getString(4);
					q.noResult = db.getBoolean(5);

					queryMap.put(q.key, q);

					queryMenu.addItem(new KeyedRecord(q.key.intValue(), q.name));
				}

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				db.reportError(se);
			}

			DbCore.releaseDb(db);
		}

		if (null != selectThis) {
			queryMenu.setSelectedItem(new KeyedRecord(selectThis.key.intValue(), selectThis.name));
		} else {
			if (queryMenu.containsKey(selectKey)) {
				queryMenu.setSelectedKey(selectKey);
			}
		}

		clearResult(selectThis);

		blockActionsEnd();
	}


	//=================================================================================================================
	// Edit dialog for composing a query.  Must call setQuery() before showing to load state, may be creating a new
	// query, modifying the last-built query, or editing a saved query.

	private class QueryEditDialog extends AppDialog {

		private ScenarioModel scenarioModel;

		private SavedQuery query;

		private JTextField nameField;
		private JTextArea descriptionArea;
		private JTextArea queryArea;
		private JCheckBox noResultCheckBox;

		private JButton okButton;

		private boolean canceled;


		//-------------------------------------------------------------------------------------------------------------

		private QueryEditDialog(AppEditor theParent) {

			super(theParent, null, "Edit Query", Dialog.ModalityType.APPLICATION_MODAL);

			// Table to display scenario list for reference.

			scenarioModel = new ScenarioModel();

			// Field for saved query name.

			nameField = new JTextField(30);
			AppController.fixKeyBindings(nameField);

			// Area for optional description appearing in query output, if blank name is used.

			descriptionArea = new JTextArea(6, 30);
			AppController.fixKeyBindings(descriptionArea);
			descriptionArea.setLineWrap(true);
			descriptionArea.setWrapStyleWord(true);

			// Area for editing query.

			queryArea = new JTextArea(25, 50);
			AppController.fixKeyBindings(queryArea);
			queryArea.setLineWrap(true);
			queryArea.setWrapStyleWord(true);

			// Check box for queries that do not return a result set.

			noResultCheckBox = new JCheckBox("Update query (does not return a result set)");

			// Buttons.

			okButton = new JButton("OK");
			okButton.setFocusable(false);
			okButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doOK();
				}
			});

			JButton saveButton = new JButton("Save");
			saveButton.setFocusable(false);
			saveButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doSave();
				}
			});

			JButton cancelButton = new JButton("Cancel");
			cancelButton.setFocusable(false);
			cancelButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					canceled = cancel();
				}
			});

			// Layout.

			JPanel nameP = new JPanel(new FlowLayout(FlowLayout.LEFT));
			nameP.setBorder(BorderFactory.createTitledBorder("Name"));
			nameP.add(nameField);

			JPanel butP = new JPanel(new FlowLayout(FlowLayout.RIGHT));
			butP.add(cancelButton);
			butP.add(saveButton);
			butP.add(okButton);

			Box topB = Box.createHorizontalBox();
			topB.add(nameP);
			topB.add(butP);

			JScrollPane descP = AppController.createScrollPane(descriptionArea);
			descP.setBorder(BorderFactory.createTitledBorder("Description"));

			JScrollPane listP = AppController.createScrollPane(scenarioModel.table);
			listP.setBorder(BorderFactory.createTitledBorder("Scenarios"));

			JPanel topP = new JPanel(new BorderLayout());
			topP.add(topB, BorderLayout.NORTH);
			topP.add(descP, BorderLayout.CENTER);
			topP.add(listP, BorderLayout.EAST);

			JScrollPane queryP = AppController.createScrollPane(queryArea);
			queryP.setBorder(BorderFactory.createTitledBorder("Query"));

			JPanel optP = new JPanel(new FlowLayout(FlowLayout.LEFT));
			optP.add(noResultCheckBox);

			setLayout(new BorderLayout());
			add(topP, BorderLayout.NORTH);
			add(queryP, BorderLayout.CENTER);
			add(optP, BorderLayout.SOUTH);

			setResizable(true);
			setLocationSaved(true);

			pack();

			setMinimumSize(getSize());
		}


		//=============================================================================================================
		// Table model for the scenario list, content set by setScenarios().

		private class ScenarioModel extends AbstractTableModel {

			private ArrayList<KeyedRecord> modelRows;

			private JTable table;


			//---------------------------------------------------------------------------------------------------------

			private ScenarioModel() {

				super();

				modelRows = new ArrayList<KeyedRecord>();

				table = new JTable(this);
				AppController.configureTable(table);
				table.setRowSelectionAllowed(false);

				TableColumn theColumn = table.getColumn(getColumnName(0));
				theColumn.setMinWidth(AppController.textFieldWidth[2]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[3]);

				theColumn = table.getColumn(getColumnName(1));
				theColumn.setMinWidth(AppController.textFieldWidth[5]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[15]);

				table.setPreferredScrollableViewportSize(new Dimension(table.getPreferredSize().width,
					(table.getRowHeight() * 5)));
			}


			//---------------------------------------------------------------------------------------------------------

			private void setScenarios(ArrayList<KeyedRecord> theScenarios) {

				modelRows.clear();
				modelRows.addAll(theScenarios);

				fireTableDataChanged();
			}


			//---------------------------------------------------------------------------------------------------------

			private KeyedRecord get(int rowIndex) {

				return modelRows.get(rowIndex);
			}


			//---------------------------------------------------------------------------------------------------------

			public int getColumnCount() {

				return 2;
			}


			//---------------------------------------------------------------------------------------------------------

			public String getColumnName(int columnIndex) {

				switch (columnIndex) {
					case 0:
						return "Key";
					case 1:
						return "Name";
				}

				return "";
			}


			//---------------------------------------------------------------------------------------------------------

			public int getRowCount() {

				return modelRows.size();
			}


			//---------------------------------------------------------------------------------------------------------

			public Object getValueAt(int rowIndex, int columnIndex) {

				KeyedRecord theItem = modelRows.get(rowIndex);

				switch (columnIndex) {
					case 0:
						return String.valueOf(theItem.key);
					case 1:
						return theItem.name;
				}

				return "";
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Forward to table model.

		private void setScenarios(ArrayList<KeyedRecord> theScenarios) {

			scenarioModel.setScenarios(theScenarios);
		}


		//-------------------------------------------------------------------------------------------------------------
		// Must be called before showing dialog.  Argument is >0 to edit an existing saved query, KEY_CUSTOM_QUERY to
		// modify the last-built query, or 0 to create a new query.

		private void setQuery(int queryKey) {
			setQuery(queryKey, false);
		}

		private void setQuery(int queryKey, boolean usePointQuery) {

			if (isVisible()) {
				return;
			}

			query = new SavedQuery();

			if (queryKey <= 0) {

				query.name = "";

				if (queryKey < 0) {

					query.description = runDescription;
					if (usePointQuery) {
						query.query = pointQuery;
					} else {
						query.query = runQuery;
					}
					query.noResult = noResult;

				} else {

					query.description = "";
					query.query = "";
				}

			} else {

				SavedQuery q = queryMap.get(Integer.valueOf(queryKey));

				if (null == q) {

					query.name = "";
					query.description = "";
					query.query = "";

				} else {

					query.key = q.key;
					query.name = q.name;
					query.description = q.description;
					query.query = q.query;
					query.noResult = q.noResult;
				}
			}

			okButton.setVisible(queryKey <= 0);

			setDocumentName(query.name);

			nameField.setText(query.name);
			descriptionArea.setText(query.description);
			queryArea.setText(query.query);
			noResultCheckBox.setSelected(query.noResult);
		}


		//-------------------------------------------------------------------------------------------------------------

		private SavedQuery getQuery() {

			return query;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void doOK() {

			if (null != query.key) {
				return;
			}

			query.query = queryArea.getText().trim();
			if (0 == query.query.length()) {
				errorReporter.reportWarning("Please provide the query");
				return;
			}

			query.noResult = noResultCheckBox.isSelected();

			query.key = Integer.valueOf(KEY_CUSTOM_QUERY);
			query.name = "(custom)";
			query.description = descriptionArea.getText();

			AppController.hideWindow(this);
			canceled = false;
		}


		//-------------------------------------------------------------------------------------------------------------
		// Save an existing or new query after edits.  New query must have a unique name.

		private void doSave() {

			String title = "Save Query";
			errorReporter.setTitle(title);

			query.name = nameField.getText().trim();
			if (0 == query.name.length()) {
				errorReporter.reportWarning("Please provide a name");
				return;
			}
			if (query.name.length() > 40) {
				errorReporter.reportWarning("The name is too long");
				return;
			}
			if (query.name.equalsIgnoreCase(NAME_COVERAGE) || query.name.equalsIgnoreCase(NAME_COVERAGE_LOSS) ||
					query.name.equalsIgnoreCase(NAME_COVERAGE_GAIN) ||
					query.name.equalsIgnoreCase(NAME_COVERAGE_OVERLAP)) {
				errorReporter.reportWarning("The name cannot be used");
				return;
			}

			query.description = descriptionArea.getText();

			query.query = queryArea.getText().trim();
			if (0 == query.query.length()) {
				errorReporter.reportWarning("Please provide the query");
				return;
			}

			query.noResult = noResultCheckBox.isSelected();

			DbConnection db = DbCore.connectDb(study.dbID, errorReporter);
			if (null == db) {
				return;
			}

			boolean badName = false;
			int theKey = 0;

			try {

				String tblname = DbCore.getDbName(study.dbID) + "_" + String.valueOf(study.key) + ".saved_query";

				if (null == query.key) {

					db.query("SELECT query_key FROM " + tblname + " WHERE UPPER(name) = '" +
						db.clean(query.name.toUpperCase()) + "'");
					if (db.next()) {
						badName = true;
					} else {

						db.query("SELECT MAX(query_key) FROM " + tblname);
						db.next();
						query.key = Integer.valueOf(db.getInt(1) + 1);

						db.update("INSERT INTO " + tblname + " (query_key, name, description, query, no_result) " +
							"VALUES (" + String.valueOf(query.key) + ", '" + db.clean(query.name) + "', '" +
							db.clean(query.description) + "', '" + db.clean(query.query) + "', " +
							String.valueOf(query.noResult) + ")");
					}

				} else {

					db.update("UPDATE " + tblname + " SET name = '" + db.clean(query.name) + "', description = '" +
						db.clean(query.description) + "', query = '" + db.clean(query.query) +
						"', no_result = " + String.valueOf(query.noResult) + " WHERE query_key = " +
						String.valueOf(query.key));
				}

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				db.reportError(errorReporter, se);
				return;
			}

			DbCore.releaseDb(db);

			if (badName) {
				errorReporter.reportWarning("The name is already in use");
				return;
			}

			AppController.hideWindow(this);
			canceled = false;
		}


		//-------------------------------------------------------------------------------------------------------------

		public void windowWillOpen() {

			blockActionsClear();
		}


		//-------------------------------------------------------------------------------------------------------------

		public void windowWillClose() {

			blockActionsSet();
			canceled = true;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doNew() {

		if (!isVisible() || (null == study) || queryRunning) {
			return;
		}

		editDialog.setQuery(0);
		AppController.showWindow(editDialog);
		if (editDialog.canceled) {
			return;
		}

		customQuery = editDialog.getQuery();
		updateMenu(customQuery);
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doModify() {

		if (!isVisible() || (null == study) || !buildQuery() || queryRunning) {
			return;
		}

		boolean usePoint = false;
		if (null != pointQuery) {
			String[] opts = {"Cancel", "View/Save query", "Summary query"};
			switch (JOptionPane.showOptionDialog(this, "There are separate queries for summary and\n" +
					"for view/save output, which do you want to modify?", "Choose Query", JOptionPane.DEFAULT_OPTION,
					JOptionPane.QUESTION_MESSAGE, null, opts, null)) {
				case 0:
				default:
					return;
				case 1:
					usePoint = true;
					break;
				case 2:
					usePoint = false;
					break;
			}
		}

		editDialog.setQuery(KEY_CUSTOM_QUERY, usePoint);
		AppController.showWindow(editDialog);
		if (editDialog.canceled) {
			return;
		}

		customQuery = editDialog.getQuery();
		updateMenu(customQuery);
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doEdit() {

		if (!isVisible() || (null == study) || queryRunning) {
			return;
		}

		editDialog.setQuery(queryMenu.getSelectedKey());
		AppController.showWindow(editDialog);
		if (editDialog.canceled) {
			return;
		}

		updateMenu(editDialog.getQuery());
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doDelete() {

		if (!isVisible() || (null == study) || queryRunning) {
			return;
		}

		int qkey = queryMenu.getSelectedKey();
		SavedQuery q = queryMap.get(Integer.valueOf(qkey));
		if (null == q) {
			return;
		}

		String title = "Delete Query";
		errorReporter.setTitle(title);

		AppController.beep();
		if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
				"Are you sure you want to delete query '" + q.name + "'?", title, JOptionPane.YES_NO_OPTION,
				JOptionPane.WARNING_MESSAGE)) {
			return;
		}

		DbConnection db = DbCore.connectDb(study.dbID, errorReporter);
		if (null == db) {
			return;
		}

		try {

			db.update("DELETE FROM " + DbCore.getDbName(study.dbID) + "_" + String.valueOf(study.key) +
				".saved_query WHERE query_key = " + String.valueOf(q.key));

		} catch (SQLException se) {
			DbCore.releaseDb(db);
			db.reportError(errorReporter, se);
			return;
		}

		DbCore.releaseDb(db);

		updateMenu(null);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check input and set up the query state, called before either a run or a modify.

	private boolean buildQuery() {

		if (queryRunning) {
			return false;
		}

		String title = "Build Query";
		errorReporter.setTitle(title);

		// For the pre-composed queries, the first query does area and population sums.  If a base scenario key is set,
		// the totals from that scenario will be used as the base to compute area and population percentages in the
		// main query.  A second query is also composed to get point-by-point results for CSV and/or KML output.  The
		// pre-composed queries may also be restricted by country.

		int qkey = queryMenu.getSelectedKey();

		switch (qkey) {

			case KEY_COVERAGE: {

				int scen = scenarioAMenu.getSelectedKey();
				if (scen <= 0) {
					errorReporter.reportWarning("Please select a scenario");
					return false;
				}

				runDescription = queryMenu.getSelectedName();

				runQuery = 
				"SELECT\n" +
					"  SUM(area) AS area,\n" +
					"  SUM(population) AS population,\n" +
					"  SUM(households) AS households\n" +
				"FROM\n" +
					"  result_" + scen + "\n" +
				"WHERE\n" +
					"  (result = 1)\n";

				noResult = false;
				baseKey = null;
				baseCountry = 0;

				pointQuery =
				"SELECT\n" +
					"  country_key,\n" +
					"  lat_index,\n" +
					"  lon_index,\n" +
					"  latitude,\n" +
					"  longitude,\n" +
					"  area,\n" +
					"  population,\n" +
					"  households,\n" +
					"  source_key,\n" +
					"  (margin / 100.) AS service_margin\n" +
				"FROM\n" +
					"  result_" + scen + "\n" +
				"WHERE\n" +
					"  (result = 1)\n";

				int cntry = countryMenu.getSelectedKey();
				if (cntry > 0) {
					runQuery = runQuery + "  AND (country_key = " + cntry + ")\n";
					pointQuery = pointQuery + "  AND (country_key = " + cntry + ")\n";
				}

				if (excludeZeroPopCheckBox.isSelected()) {
					pointQuery = pointQuery + "  AND (result_" + scen + ".population > 0)\n";
				}

				break;
			}

			case KEY_COVERAGE_LOSS:
			case KEY_COVERAGE_GAIN: {

				int scenA = scenarioAMenu.getSelectedKey();
				int scenB = scenarioBMenu.getSelectedKey();
				if ((scenA <= 0) || (scenB <= 0) || (scenA == scenB)) {
					errorReporter.reportWarning("Please select two different scenarios to compare");
					return false;
				}
				int left, right;
				if (KEY_COVERAGE_LOSS == qkey) {
					left = scenA;
					right = scenB;
				} else {
					left = scenB;
					right = scenA;
				}

				runDescription = queryMenu.getSelectedName();

				runQuery = 
				"SELECT\n" +
					"  SUM(result_" + left + ".area) AS area,\n" +
					"  SUM(result_" + left + ".population) AS population,\n" +
					"  SUM(result_" + left + ".households) AS households\n" +
				"FROM\n" +
					"  result_" + left + "\n" +
					"  LEFT JOIN result_" + right + " USING (country_key, lat_index, lon_index)\n" +
				"WHERE\n" +
					"  (result_" + left + ".result = 1)\n" +
					"  AND ((result_" + right + ".result IS NULL)\n" +
					"    OR (result_" + right + ".result <> 1))\n";

				noResult = false;
				baseKey = Integer.valueOf(left);
				baseCountry = 0;

				pointQuery =
				"SELECT\n" +
					"  result_" + left + ".country_key,\n" +
					"  result_" + left + ".lat_index,\n" +
					"  result_" + left + ".lon_index,\n" +
					"  result_" + left + ".latitude,\n" +
					"  result_" + left + ".longitude,\n" +
					"  result_" + left + ".area,\n" +
					"  result_" + left + ".population,\n" +
					"  result_" + left + ".households\n" +
				"FROM\n" +
					"  result_" + left + "\n" +
					"  LEFT JOIN result_" + right + " USING (country_key, lat_index, lon_index)\n" +
				"WHERE\n" +
					"  (result_" + left + ".result = 1)\n" +
					"  AND ((result_" + right + ".result IS NULL)\n" +
					"    OR (result_" + right + ".result <> 1))\n";

				int cntry = countryMenu.getSelectedKey();
				if (cntry > 0) {
					runQuery = runQuery + "  AND (country_key = " + cntry + ")\n";
					baseCountry = cntry;
					pointQuery = pointQuery + " AND (country_key = " + cntry + ")";
				}

				if (excludeZeroPopCheckBox.isSelected()) {
					pointQuery = pointQuery + " AND (result_" + left + ".population > 0)";
				}

				break;
			}

			case KEY_COVERAGE_OVERLAP: {

				int scenA = scenarioAMenu.getSelectedKey();
				int scenB = scenarioBMenu.getSelectedKey();
				if ((scenA <= 0) || (scenB <= 0) || (scenA == scenB)) {
					errorReporter.reportWarning("Please select two different scenarios to compare");
					return false;
				}

				runDescription = queryMenu.getSelectedName();

				runQuery = 
				"SELECT\n" +
					"  SUM(result_" + scenA + ".area) AS area,\n" +
					"  SUM(result_" + scenA + ".population) AS population,\n" +
					"  SUM(result_" + scenA + ".households) AS households\n" +
				"FROM\n" +
					"  result_" + scenA + "\n" +
					"  JOIN result_" + scenB + " USING (country_key, lat_index, lon_index)\n" +
				"WHERE\n" +
					"  (result_" + scenA + ".result = 1)\n" +
					"  AND (result_" + scenB + ".result = 1)\n";

				noResult = false;
				baseKey = null;
				baseCountry = 0;

				pointQuery =
				"SELECT\n" +
					"  result_" + scenA + ".country_key,\n" +
					"  result_" + scenA + ".lat_index,\n" +
					"  result_" + scenA + ".lon_index,\n" +
					"  result_" + scenA + ".latitude,\n" +
					"  result_" + scenA + ".longitude,\n" +
					"  result_" + scenA + ".area,\n" +
					"  result_" + scenA + ".population,\n" +
					"  result_" + scenA + ".households\n" +
				"FROM\n" +
					"  result_" + scenA + "\n" +
					"  JOIN result_" + scenB + " USING (country_key, lat_index, lon_index)\n" +
				"WHERE\n" +
					"  (result_" + scenA + ".result = 1)\n" +
					"  AND (result_" + scenB + ".result = 1)\n";

				int cntry = countryMenu.getSelectedKey();
				if (cntry > 0) {
					runQuery = runQuery + "  AND (country_key = " + cntry + ")\n";
					pointQuery = pointQuery + "  AND (country_key = " + cntry + ")\n";
				}

				if (excludeZeroPopCheckBox.isSelected()) {
					pointQuery = pointQuery + "  AND (result_" + scenA + ".population > 0)\n";
				}

				break;
			}

			// For a custom or saved query, the main query will be used for CSV/KML output.  CSV is always output
			// regardless of the number and content of tuples, KML output will be made only if the tuples include
			// latitude and longitude values.

			case KEY_CUSTOM_QUERY: {

				if (null == customQuery) {
					return false;
				}

				runDescription = customQuery.description;
				if (0 == runDescription.length()) {
					runDescription = customQuery.name;
				}

				runQuery = customQuery.query;
				noResult = customQuery.noResult;

				baseKey = null;
				baseCountry = 0;
				pointQuery = null;

				break;
			}

			default: {

				SavedQuery q = queryMap.get(Integer.valueOf(qkey));
				if (null == q) {
					return false;
				}

				runDescription = q.description;
				if (0 == runDescription.length()) {
					runDescription = q.name;
				}

				runQuery = q.query;
				noResult = q.noResult;

				baseKey = null;
				baseCountry = 0;
				pointQuery = null;

				break;
			}
		}

		return true;
	}


	//=================================================================================================================
	// Class to run the query on a background thread.

	private class QueryRunner implements Runnable {

		private ErrorLogger errors;

		private DbConnection db;

		private int updateCount;

		private boolean hasOutput;
		private String[] outputLabels;
		private String[] outputValues;

		private ArrayList<String[]> newResults;
		private String[] columnNames;

		private boolean canceled;


		//-------------------------------------------------------------------------------------------------------------

		private QueryRunner() {

			errors = new ErrorLogger(new StringBuilder());
		}


		//-------------------------------------------------------------------------------------------------------------

		public void run() {

			db = DbCore.connectDb(study.dbID, errors);
			if (null == db) {
				cleanup();
				return;
			}

			try {

				db.setDatabase(DbCore.getDbName(study.dbID) + "_" + String.valueOf(study.key));

				// For a no-result query just run it and return.

				if (noResult) {
					updateCount = db.update(runQuery);
					cleanup();
					return;
				}

				// If a base scenario key is set look that up, if it exists and has a table, query totals for the base
				// coverage as needed.  These are used to compute percentages for values from the main query.

				ScenarioResult baseScenario = null;
				String theQuery;

				if (null != baseKey) {

					ScenarioResult theScenario = scenarioMap.get(baseKey);
					if ((null != theScenario) && theScenario.hasTable) {

						if (theScenario.hasResult && (theScenario.resultCountry == baseCountry)) {
							baseScenario = theScenario;
						} else {

							theQuery = "SELECT SUM(area), SUM(population), SUM(households) FROM result_" +
								baseKey + " WHERE (result = 1)";
							if (baseCountry > 0) {
								theQuery = theQuery + " AND (country_key = " + baseCountry + ")";
							}
							db.query(theQuery);

							if (db.next()) {
								theScenario.area = db.getDouble(1);
								theScenario.population = db.getInt(2);
								theScenario.households = db.getInt(3);
								theScenario.hasResult = true;
								theScenario.resultCountry = baseCountry;
								baseScenario = theScenario;
							}
						}
					}
				}

				// Run the main query, and if set, the separate point query.  Results are accumualted from the point
				// query if it exists, else from the main query.

				ResultSetMetaData md;
				String name, value;
				int colCount = 0, c;

				boolean doPoint = (null != pointQuery);

				theQuery = runQuery;

				while (true) {

					db.query(theQuery);

					md = db.getMetaData();
					colCount = md.getColumnCount();

					while (db.next()) {

						if (canceled) {
							break;
						}

						// On the first tuple of the main query, extract labels and values for later display.  Look for
						// for area, population, and households values which get special formatting, also if the base
						// scenario is set percentages are added to the display values where possible.

						if (!hasOutput) {

							hasOutput = true;

							outputLabels = new String[colCount];
							outputValues = new String[colCount];

							for (c = 1; c <= colCount; c++) {

								outputLabels[c - 1] = md.getColumnLabel(c);

								name = md.getColumnName(c);

								if (name.equals("area")) {
									double dv = db.getDouble(c);
									value = AppCore.formatCount(dv);
									if ((null != baseScenario) && (baseScenario.area > 0.)) {
										value = value + String.format(Locale.US, " (%.2f%%)",
											((dv / baseScenario.area) * 100.));
									}
								} else {

									if (name.equals("population")) {
										int iv = db.getInt(c);
										value = AppCore.formatCount(iv);
										if ((null != baseScenario) && (baseScenario.population > 0)) {
											value = value + String.format(Locale.US, " (%.2f%%)",
												(((double)iv / (double)baseScenario.population) * 100.));
										}
									} else {

										if (name.equals("households")) {
											int iv = db.getInt(c);
											value = AppCore.formatCount(iv);
											if ((null != baseScenario) && (baseScenario.households > 0)) {
												value = value + String.format(Locale.US, " (%.2f%%)",
													(((double)iv / (double)baseScenario.households) * 100.));
											}
										} else {

											value = db.getString(c);
											if (null == value) {
												value = "";
											}
										}
									}
								}

								outputValues[c - 1] = value;
							}
						}

						// If a point query is still to be run only the first tuple of the main query is processed.

						if (doPoint) {
							continue;
						}

						// Process and store results from query.  On the first tuple store the column names.

						if (null == newResults) {

							newResults = new ArrayList<String[]>();

							columnNames = new String[colCount];
							for (c = 1; c <= colCount; c++) {
								columnNames[c - 1] = md.getColumnLabel(c);
							}
						}

						String[] row = new String[colCount];
						for (c = 1; c <= colCount; c++) {
							row[c - 1] = db.getString(c);
						}
						newResults.add(row);
					}

					if (canceled) {
						break;
					}

					// Do second pass for point query if needed.

					if (doPoint) {
						theQuery = pointQuery;
						doPoint = false;
					} else {
						break;
					}
				}

			} catch (Exception e) {
				if (!canceled) {
					AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", e);
					errors.reportError(e.toString());
				}
			}

			cleanup();
		}


		//-------------------------------------------------------------------------------------------------------------
		// Clean up state when queries are done.

		private void cleanup() {

			if (null != db) {
				DbCore.releaseDb(db);
				db = null;
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set up and run a query.

	private void doRun() {

		if (!isVisible() || (null == study) || !buildQuery() || queryRunning) {
			return;
		}

		errorReporter.setTitle("Run Query");

		// Start the query.

		queryRunner = new QueryRunner();
		queryThread = new Thread(queryRunner);
		queryThread.start();

		queryRunning = true;
		queryTimer.start();
		cancelButton.setEnabled(true);

		clearResult();
		resultArea.setText("Query running...");
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Cancel a running query, this finds the running query in the backend process list and sends a KILL query on a
	// separate connection.  This is a one-shot, and it's fire-and-forget, it may or may not actually work.

	private void cancelQuery() {

		if (!queryRunning || (null == queryRunner.db) || queryRunner.canceled) {
			return;
		}

		queryRunner.canceled = true;
		cancelButton.setEnabled(false);

		String user = queryRunner.db.username;
		String dbName = queryRunner.db.getDatabase();
		if (null == dbName) {
			return;
		}

		DbConnection db = DbCore.connectDb(study.dbID, errorReporter);
		if (null == db) {
			return;
		}

		try {

			db.query("SHOW PROCESSLIST");

			int id = 0;
			while (db.next()) {
				if ("Query".equals(db.getString("Command")) && user.equals(db.getString("User")) &&
						dbName.equals(db.getString("db"))) {
					id = db.getInt("Id");
					break;
				}
			}

			if (id > 0) {
				db.update("KILL QUERY " + String.valueOf(id));
			}

		} catch (SQLException se) {
			db.reportError(errorReporter, se);
		}

		DbCore.releaseDb(db);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check on status of running query, called by timer when it is running.

	private void checkQuery() {

		if (!queryRunning) {
			queryTimer.stop();
			return;
		}

		if (queryThread.isAlive()) {
			return;
		}
		queryThread = null;
		queryRunning = false;
		queryTimer.stop();
		cancelButton.setEnabled(false);

		// Query done, report/store results as needed.

		if (queryRunner.errors.hasErrors()) {

			resultArea.setText(queryRunner.errors.toString());

		} else {

			if (queryRunner.canceled) {

				resultArea.setText("Query canceled\n");

			} else {

				didRun = true;

				resultArea.setText(runDescription);
				resultArea.append("\n");

				if (noResult) {

					resultArea.append("\nQuery modified " + String.valueOf(queryRunner.updateCount) + " rows\n");

				} else {

					if (!queryRunner.hasOutput) {

						resultArea.append("\nQuery returned empty result\n");

					} else {

						resultArea.append("\n");
						for (int c = 0; c < queryRunner.outputLabels.length; c++) {
							resultArea.append(String.format("%20s  %s\n", queryRunner.outputLabels[c],
								queryRunner.outputValues[c]));
						}

						if ((null != queryRunner.newResults) && !queryRunner.newResults.isEmpty()) {

							results = queryRunner.newResults;
							resultColumnNames = queryRunner.columnNames;

							// KML output can only occur if query returned latitude and longitude.  Also optionally a
							// name column will be used for the KML placemarks.

							for (int i = 0; i < resultColumnNames.length; i++) {
								if (resultColumnNames[i].equalsIgnoreCase("name")) {
									resultNameColumn = i;
								}
								if (resultColumnNames[i].equalsIgnoreCase("latitude")) {
									resultLatitudeColumn = i;
								}
								if (resultColumnNames[i].equalsIgnoreCase("longitude")) {
									resultLongitudeColumn = i;
								}
							}
							canSaveKML = ((resultLatitudeColumn >= 0) && (resultLongitudeColumn >= 0));

							resultArea.append("\n");
							resultArea.append(String.valueOf(results.size()));
							resultArea.append(" rows in result\n");
						}
					}
				}

				if (queryRunner.errors.hasMessages()) {
					resultArea.append("\n");
					resultArea.append(queryRunner.errors.getMessages());
				}
			}
		}

		queryRunner = null;

		updateControls(null);
	}


	//=================================================================================================================
	// Table model for displaying query results.

	private class ResultTableModel extends AbstractTableModel {


		//-------------------------------------------------------------------------------------------------------------

		private JTable createTable() {

			if (null == results) {
				return null;
			}

			JTable theTable = new JTable(this);
			AppController.configureTable(theTable, null);
			theTable.setAutoCreateRowSorter(true);
			theTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);

			int n = resultColumnNames.length, c, l;
			int[] sizes = new int[n];
			for (c = 0; c < n; c++) {
				l = resultColumnNames[c].length() / 2;
				if (l < 5) {
					sizes[c] = 5;
				} else {
					sizes[c] = l;
				}
			}
			for (String[] values : results) {
				for (c = 0; c < n; c++) {
					if (null != values[c]) {
						l = values[c].length() / 2;
						if (l > sizes[c]) {
							sizes[c] = l;
						}
					}
				}
			}

			TableColumnModel theColumns = theTable.getColumnModel();
			TableColumn theColumn;
			for (c = 0; c < n; c++) {
				theColumn = theColumns.getColumn(c);
				theColumn.setMinWidth(AppController.textFieldWidth[5]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[sizes[c]]);
			}

			return theTable;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getColumnCount() {

			if (null == results) {
				return 0;
			}
			return resultColumnNames.length;
		}


		//-------------------------------------------------------------------------------------------------------------

		public String getColumnName(int columnIndex) {

			if (null == results) {
				return "";
			}
			return resultColumnNames[columnIndex];
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getRowCount() {

			if (null == results) {
				return 0;
			}
			return results.size();
		}


		//-------------------------------------------------------------------------------------------------------------

		public Object getValueAt(int rowIndex, int columnIndex) {

			if (null == results) {
				return "";
			}
			return results.get(rowIndex)[columnIndex];
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// View results of last query in a table.

	private void doView() {

		if (!isVisible() || (null == study) || queryRunning || (null == results)) {
			return;
		}

		AppDialog viewDialog = new AppDialog(this, "Query Results", Dialog.ModalityType.DOCUMENT_MODAL) {
			public void windowWillOpen() {
				blockActionsClear();
			}
			public void windowWillClose() {
				blockActionsSet();
			}
		};
		viewDialog.setLayout(new BorderLayout());
		viewDialog.add(AppController.createScrollPane(tableModel.createTable(), JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
			JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS), BorderLayout.CENTER);
		viewDialog.setResizable(true);
		viewDialog.setLocationSaved(true);
		viewDialog.pack();
		viewDialog.setMinimumSize(viewDialog.getSize());

		AppController.showWindow(viewDialog);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save results of last query to CSV file.

	private void doSaveCSV() {

		if (!isVisible() || (null == study) || queryRunning || (null == results)) {
			return;
		}

		String title = "Save results to CSV file";
		errorReporter.setTitle(title);

		File csvFile = null;
		String theName;

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.SAVE_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new FileNameExtensionFilter("CSV (*.csv)", "csv"));
		chooser.setAcceptAllFileFilterUsed(false);

		do {
			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Save")) {
				return;
			}
			csvFile = chooser.getSelectedFile();
			theName = csvFile.getName().toLowerCase();
			if (!theName.endsWith(".csv")) {
				csvFile = new File(csvFile.getAbsolutePath() + ".csv");
			}
			if (csvFile.exists()) {
				AppController.beep();
				if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
						"The file exists, do you want to replace it?", title, JOptionPane.YES_NO_OPTION,
						JOptionPane.WARNING_MESSAGE)) {
					csvFile = null;
				}
			}
		} while (null == csvFile);

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, csvFile.getParentFile().getAbsolutePath());

		BufferedWriter csvWriter = null;
		try {
			csvWriter = new BufferedWriter(new FileWriter(csvFile));
		} catch (IOException ie) {
			errorReporter.reportError("File could not be created:\n" + ie.getMessage());
			return;
		}

		try {

			String sep = "";
			for (String col : resultColumnNames) {
				csvWriter.write(sep);
				csvWriter.write(col);
				sep = ",";
			}
			csvWriter.write("\n");

			for (String[] values : results) {
				sep = "";
				for (String value : values) {
					csvWriter.write(sep);
					csvWriter.write(value);
					sep = ",";
				}
				csvWriter.write("\n");
			}

		} catch (IOException ie) {
			errorReporter.reportError("Error writing to file:\n" + ie.getMessage());
			return;
		}

		try {csvWriter.close();} catch (IOException ie) {}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save results of last query to KML file.

	private void doSaveKML() {

		if (!isVisible() || (null == study) || queryRunning || (null == results) || !canSaveKML) {
			return;
		}

		String title = "Save results to KML file";
		errorReporter.setTitle(title);

		File kmlFile = null;
		String theName;

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.SAVE_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new FileNameExtensionFilter("KML (*.kml)", "kml"));
		chooser.setAcceptAllFileFilterUsed(false);

		do {
			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Save")) {
				return;
			}
			kmlFile = chooser.getSelectedFile();
			theName = kmlFile.getName().toLowerCase();
			if (!theName.endsWith(".kml")) {
				kmlFile = new File(kmlFile.getAbsolutePath() + ".kml");
			}
			if (kmlFile.exists()) {
				AppController.beep();
				if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
						"The file exists, do you want to replace it?", title, JOptionPane.YES_NO_OPTION,
						JOptionPane.WARNING_MESSAGE)) {
					kmlFile = null;
				}
			}
		} while (null == kmlFile);

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, kmlFile.getParentFile().getAbsolutePath());

		BufferedWriter kmlWriter = null;
		try {
			kmlWriter = new BufferedWriter(new FileWriter(kmlFile));
		} catch (IOException ie) {
			errorReporter.reportError("File could not be created:\n" + ie.getMessage());
			return;
		}

		int colCount = resultColumnNames.length, c;
		boolean hasData = false;
		if (resultNameColumn >= 0) {
			hasData = (resultColumnNames.length > 3);
		} else {
			hasData = (resultColumnNames.length > 2);
		}
		String lon;

		try {

			kmlWriter.write("<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n");
			kmlWriter.write("<kml xmlns=\"http://www.opengis.net/kml/2.2\">\n");
			kmlWriter.write("<Document>\n<name>");
			kmlWriter.write(AppCore.xmlclean(runDescription));
			kmlWriter.write("</name>\n");
			kmlWriter.write("<Style id=\"styles\">\n");
			kmlWriter.write("<IconStyle><color>ffff0000</color><scale>0.5</scale></IconStyle>\n");
			kmlWriter.write("<LineStyle><color>ff00ff00</color><width>3</width></LineStyle>\n");
			kmlWriter.write("<PolyStyle><outline>1</outline><fill>0</fill></PolyStyle>\n");
			kmlWriter.write("</Style>\n");

			for (String[] values : results) {

				kmlWriter.write("<Placemark>\n");
				kmlWriter.write("<styleUrl>#styles</styleUrl>\n");

				if (resultNameColumn >= 0) {
					kmlWriter.write("<name>");
					kmlWriter.write(AppCore.xmlclean(values[resultNameColumn]));
					kmlWriter.write("</name>\n");
				}

				if (hasData) {
					kmlWriter.write("<ExtendedData>\n");
					for (c = 0; c < colCount; c++) {
						if ((c != resultNameColumn) && (c != resultLatitudeColumn) && (c != resultLongitudeColumn)) {
							kmlWriter.write("<Data name=\"");
							kmlWriter.write(resultColumnNames[c]);
							kmlWriter.write("\"><value>");
							if (null != values[c]) {
								kmlWriter.write(values[c]);
							}
							kmlWriter.write("</value></Data>\n");
						}
					}
					kmlWriter.write("</ExtendedData>\n");
				}

				kmlWriter.write("<Point>\n<coordinates>");
				lon = values[resultLongitudeColumn];
				if (lon.startsWith("-")) {
					lon = lon.substring(1);
				} else {
					kmlWriter.write("-");
				}
				kmlWriter.write(lon);
				kmlWriter.write(',');
				kmlWriter.write(values[resultLatitudeColumn]);
				kmlWriter.write("</coordinates>\n</Point>\n");

				kmlWriter.write("</Placemark>\n");
			}

			kmlWriter.write("</Document>\n</kml>\n");

		} catch (IOException ie) {
			errorReporter.reportError("Error writing to file:\n" + ie.getMessage());
			return;
		}

		try {kmlWriter.close();} catch (IOException ie) {}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Clear result from past query.

	private void clearResult() {
		clearResult(null);
	}

	private void clearResult(SavedQuery theQuery) {

		didRun = false;

		results = null;
		resultColumnNames = null;
		resultNameColumn = -1;
		resultLatitudeColumn = -1;
		resultLongitudeColumn = -1;
		canSaveKML = false;

		resultArea.setText("");

		updateControls(theQuery);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// StudyLockHolder methods.

	public int getStudyKey() {

		if (null != study) {
			return study.key;
		}
		return 0;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getStudyName() {

		if (null != study) {
			return study.name;
		}
		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public int getStudyLock() {

		if (null != study) {
			return study.studyLock;
		}
		return 0;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public int getLockCount() {

		if (null != study) {
			return study.lockCount;
		}
		return 0;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void toFront() {

		super.toFront();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean closeWithoutSave() {

		if (windowShouldClose()) {
			AppController.hideWindow(this);
			return true;
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean studyManagerClosing() {

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillOpen() {

		blockActionsClear();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean windowShouldClose() {

		if (queryRunning) {
			toFront();
			AppController.beep();
			errorReporter.reportMessage("This window can't be closed while a query is running");
			return false;
		}

		if (editDialog.isVisible()) {
			return editDialog.cancel();
		}
		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The parent will clear the study lock if needed in editorClosing().

	public void windowWillClose() {

		if (queryRunning) {
			return;
		}
		blockActionsSet();
		queryTimer.stop();
		parent.editorClosing(this);
		study = null;
	}
}
