//
//  ScenarioEditor.java
//  TVStudy
//
//  Copyright (c) 2012-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui.editor;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;
import gov.fcc.tvstudy.core.geo.*;
import gov.fcc.tvstudy.gui.*;

import java.util.*;
import java.sql.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.table.*;
import javax.swing.filechooser.*;


//=====================================================================================================================
// Scenario editor, allows adding and removing sources in the scenario, and editing sources when permitted.  This is
// a secondary UI window that always belongs to a parent StudyEditor, which is responsible for loading and saving data.
// This is constructed with a specific ScenarioEditData object and can only edit that object.  There is no "editing
// canceled" behavior here.

// UI labelling has been changed vs. the code and database design, "Source" is now labeled "Station" or "Record".

public class ScenarioEditor extends AppFrame {

	public static final String WINDOW_TITLE = "Scenario";

	private ScenarioEditData scenario;
	private int studyType;

	// UI fields.

	private JTextField scenarioNameField;
	private JTextArea scenarioDescriptionArea;

	private ArrayList<ParameterEditor> parameterEditors;

	private SourceTableModel sourceModel;
	private JTable sourceTable;

	// Buttons and menu items.

	private JButton addSourceButton;
	private JButton openSourceButton;
	private JButton removeSourceButton;

	private JMenuItem addSourceMenuItem;
	private JMenuItem addSourcesMenuItem;
	private JMenuItem importSourcesMenuItem;
	private JMenu importSourcesMenu;
	private JMenuItem openSourceMenuItem;
	private JMenuItem removeSourceMenuItem;

	private JMenuItem replicateSourceMenuItem;
	private JMenuItem unlockSourceMenuItem;
	private JMenuItem revertSourceMenuItem;
	private JMenuItem copyToScenarioMenuItem;
	private JMenuItem compareSourceMenuItem;
	private JMenuItem exportSourcesMenuItem;
	private JMenuItem saveAsUserRecordMenuItem;

	private JMenuItem setDesiredMenuItem;
	private JMenuItem clearDesiredMenuItem;
	private JMenuItem toggleDesiredMenuItem;
	private JMenuItem setUndesiredMenuItem;
	private JMenuItem clearUndesiredMenuItem;
	private JMenuItem toggleUndesiredMenuItem;

	private JComboBox<ExtDbSearch> addSourcesMenu;

	// Source table contextual popup menu, and menu items.

	private JPopupMenu sourceTablePopupMenu;

	private JMenuItem cmOpenSourceMenuItem;
	private JMenuItem cmRemoveSourceMenuItem;
	private JMenuItem cmReplicateSourceMenuItem;
	private JMenuItem cmUnlockSourceMenuItem;
	private JMenuItem cmRevertSourceMenuItem;
	private JMenuItem cmSetDesiredMenuItem;
	private JMenuItem cmClearDesiredMenuItem;
	private JMenuItem cmToggleDesiredMenuItem;
	private JMenuItem cmSetUndesiredMenuItem;
	private JMenuItem cmClearUndesiredMenuItem;
	private JMenuItem cmToggleUndesiredMenuItem;

	// Dependent dialogs.  May be SourceEditors open for any/many sources in the list, indexed by sourceKey.

	private HashMap<Integer, SourceEditor> sourceEditors;

	private RecordFind addSourceFinder;
	private ExtDbSearchDialog addSourcesDialog;

	// Disambiguation.

	private ScenarioEditor outerThis = this;


	//-----------------------------------------------------------------------------------------------------------------

	public ScenarioEditor(AppEditor theParent, ScenarioEditData theScenario) {

		super(theParent, WINDOW_TITLE);

		scenario = theScenario;
		studyType = scenario.study.study.studyType;

		// See also the override of getKeyTitle().

		setTitleKey(scenario.key.intValue());

		// Create UI components, the scenario name first.

		scenarioNameField = new JTextField(30);
		AppController.fixKeyBindings(scenarioNameField);

		scenarioNameField.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					String newName = scenarioNameField.getText().trim();
					if ((newName.length() > 0) && !scenario.name.equals(newName)) {
						boolean changeOK = false;
						if (scenario.name.equalsIgnoreCase(newName)) {
							changeOK = true;
						} else {
							errorReporter.setTitle("Change Scenario Name");
							changeOK = DbCore.checkScenarioName(newName, scenario, true, errorReporter);
						}
						if (changeOK) {
							scenario.name = newName;
							setDidEdit();
							updateDocumentName();
							parent.applyEditsFrom(outerThis);
						}
					}
					blockActionsEnd();
				}
				scenarioNameField.setText(scenario.name);
			}
		});
		scenarioNameField.addFocusListener(new FocusAdapter() {
			public void focusGained(FocusEvent theEvent) {
				setCurrentField(scenarioNameField);
			}
			public void focusLost(FocusEvent theEvent) {
				if (!theEvent.isTemporary()) {
					scenarioNameField.postActionEvent();
				}
			}
		});

		JPanel innerNamePanel = new JPanel();
		innerNamePanel.setBorder(BorderFactory.createTitledBorder("Scenario Name"));
		innerNamePanel.add(scenarioNameField);

		JPanel namePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		namePanel.add(innerNamePanel);

		scenarioNameField.setText(scenario.name);

		// Scenario description.

		scenarioDescriptionArea = new JTextArea(6, 40);
		AppController.fixKeyBindings(scenarioDescriptionArea);

		scenarioDescriptionArea.setLineWrap(true);
		scenarioDescriptionArea.setWrapStyleWord(true);
		scenarioDescriptionArea.getDocument().addDocumentListener(new DocumentListener() {
			public void insertUpdate(DocumentEvent theEvent) {
				String str = scenarioDescriptionArea.getText().trim();
				if (!str.equals(scenario.description)) {
					scenario.description = str;
					setDidEdit();
				}
			}
			public void removeUpdate(DocumentEvent theEvent) {
				String str = scenarioDescriptionArea.getText().trim();
				if (!str.equals(scenario.description)) {
					scenario.description = str;
					setDidEdit();
				}
			}
			public void changedUpdate(DocumentEvent theEvent) {
			}
		});
		scenarioDescriptionArea.addFocusListener(new FocusAdapter() {
			public void focusGained(FocusEvent theEvent) {
			}
			public void focusLost(FocusEvent theEvent) {
				if (!theEvent.isTemporary()) {
					parent.applyEditsFrom(outerThis);
				}
			}
		});

		JPanel descriptionPanel = new JPanel(new BorderLayout());
		descriptionPanel.setBorder(BorderFactory.createTitledBorder("Scenario Description"));
		descriptionPanel.add(AppController.createScrollPane(scenarioDescriptionArea), BorderLayout.CENTER);

		scenarioDescriptionArea.setText(scenario.description);
		scenario.description = scenarioDescriptionArea.getText().trim();

		// Create the parameter editor layout if needed.

		JPanel parameterEditPanel = null;

		if (null != scenario.parameters) {

			parameterEditors = new ArrayList<ParameterEditor>();
			JComponent paramEdit = ParameterEditor.createEditorLayout(this, errorReporter, scenario.parameters,
				parameterEditors);

			parameterEditPanel = new JPanel(new BorderLayout());
			parameterEditPanel.setBorder(BorderFactory.createTitledBorder("Scenario Parameters"));
			parameterEditPanel.add(paramEdit, BorderLayout.CENTER);
		}

		// Set up the source list table.

		JPanel sourcePanel = new JPanel(new BorderLayout());
		sourceModel = new SourceTableModel(sourcePanel);
		sourceTable = sourceModel.createTable(editMenu);

		sourceTable.addMouseListener(new MouseAdapter() {
			public void mouseClicked(MouseEvent e) {
				if (2 == e.getClickCount()) {
					doOpenSource();
				}
			}
			public void mousePressed(MouseEvent e) {
				if (e.isPopupTrigger()) {
					sourceTablePopupMenu.show(e.getComponent(), e.getX(), e.getY());
				}
			}
			public void mouseReleased(MouseEvent e) {
				if (e.isPopupTrigger()) {
					sourceTablePopupMenu.show(e.getComponent(), e.getX(), e.getY());
				}
			}
		});

		sourceTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
			public void valueChanged(ListSelectionEvent theEvent) {
				updateControls();
			}
		});

		sourceModel.updateBorder();
		sourcePanel.add(AppController.createScrollPane(sourceTable), BorderLayout.CENTER);
		sourcePanel.add(sourceModel.filterPanel, BorderLayout.SOUTH);

		sourceEditors = new HashMap<Integer, SourceEditor>();

		// Menu of saved searches, this is populated later by the updateSearchMenu() method.

		addSourcesMenu = new JComboBox<ExtDbSearch>();
		addSourcesMenu.setFocusable(false);

		addSourcesMenu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					ExtDbSearch newSearch = (ExtDbSearch)(addSourcesMenu.getSelectedItem());
					if ((null != newSearch) && (newSearch.name.length() > 0)) {
						doAddSources(newSearch);
					}
					addSourcesMenu.setSelectedIndex(0);
					blockActionsEnd();
				}
			}
		});

		// Create action buttons.

		addSourceButton = new JButton("Add One");
		addSourceButton.setFocusable(false);
		addSourceButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doAddSource();
			}
		});

		openSourceButton = new JButton("View");
		openSourceButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doOpenSource();
			}
		});

		removeSourceButton = new JButton("Remove");
		removeSourceButton.setFocusable(false);
		removeSourceButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRemoveSource();
			}
		});

		JButton closeButton = new JButton("Close");
		closeButton.setFocusable(false);
		closeButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (windowShouldClose()) {
					AppController.hideWindow(outerThis);
				}
			}
		});

		// Do the layout.

		JPanel nameDescPanel = new JPanel(new BorderLayout());
		nameDescPanel.add(namePanel, BorderLayout.NORTH);
		nameDescPanel.add(descriptionPanel, BorderLayout.CENTER);

		JPanel topPanel = nameDescPanel;
		if (null != parameterEditPanel) {
			topPanel = new JPanel(new BorderLayout());
			topPanel.add(nameDescPanel, BorderLayout.NORTH);
			topPanel.add(parameterEditPanel, BorderLayout.CENTER);
		}

		JPanel leftButPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		leftButPanel.add(addSourceButton);
		leftButPanel.add(addSourcesMenu);

		JPanel midButPanel = new JPanel();
		midButPanel.add(removeSourceButton);
		midButPanel.add(openSourceButton);

		JPanel rightButPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		rightButPanel.add(closeButton);

		JPanel buttonPanel = new JPanel();
		buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
		buttonPanel.add(leftButPanel);
		buttonPanel.add(midButPanel);
		buttonPanel.add(rightButPanel);

		Container cp = getContentPane();
		cp.setLayout(new BorderLayout());
		cp.add(topPanel, BorderLayout.NORTH);
		cp.add(sourcePanel, BorderLayout.CENTER);
		cp.add(buttonPanel, BorderLayout.SOUTH);

		getRootPane().setDefaultButton(openSourceButton);

		pack();

		Dimension theSize = new Dimension(860, 650);
		setSize(theSize);
		setMinimumSize(theSize);

		// Build the Source menu.

		fileMenu.removeAll();

		// Previous

		JMenuItem miPrevious = new JMenuItem("Previous");
		miPrevious.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_UP, AppController.MENU_SHORTCUT_KEY_MASK));
		miPrevious.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doPrevious();
			}
		});
		fileMenu.add(miPrevious);

		// Next

		JMenuItem miNext = new JMenuItem("Next");
		miNext.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, AppController.MENU_SHORTCUT_KEY_MASK));
		miNext.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doNext();
			}
		});
		fileMenu.add(miNext);

		// __________________________________

		fileMenu.addSeparator();

		// Add One...

		addSourceMenuItem = new JMenuItem("Add One...");
		addSourceMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, AppController.MENU_SHORTCUT_KEY_MASK));
		addSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doAddSource();
			}
		});
		fileMenu.add(addSourceMenuItem);

		// Add Many...

		addSourcesMenuItem = new JMenuItem("Add Many...");
		addSourcesMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doAddSources(null);
			}
		});
		fileMenu.add(addSourcesMenuItem);

		// Import... [ -> ]

		ArrayList<KeyedRecord> list = Source.getRecordTypes(studyType);

		if (1 == list.size()) {

			final int recType = list.get(0).key;
			importSourcesMenuItem = new JMenuItem("Import...");
			importSourcesMenuItem.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doImportSources(recType);
				}
			});
			fileMenu.add(importSourcesMenuItem);

		} else {

			importSourcesMenu = new JMenu("Import");
			JMenuItem miImport;
			for (KeyedRecord theType : list) {
				miImport = new JMenuItem(theType.name);
				final int recType = theType.key;
				miImport.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doImportSources(recType);
					}
				});
				importSourcesMenu.add(miImport);
			}
			fileMenu.add(importSourcesMenu);
		}

		// __________________________________

		fileMenu.addSeparator();

		// View/Edit

		openSourceMenuItem = new JMenuItem("View");
		openSourceMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, AppController.MENU_SHORTCUT_KEY_MASK));
		openSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doOpenSource();
			}
		});
		fileMenu.add(openSourceMenuItem);

		// Remove

		removeSourceMenuItem = new JMenuItem("Remove");
		removeSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRemoveSource();
			}
		});
		fileMenu.add(removeSourceMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// Replicate...

		replicateSourceMenuItem = new JMenuItem("Replicate...");
		replicateSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doReplicateSource();
			}
		});
		fileMenu.add(replicateSourceMenuItem);

		// Allow Editing

		unlockSourceMenuItem = new JMenuItem("Allow Editing");
		unlockSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doUnlockSource();
			}
		});
		fileMenu.add(unlockSourceMenuItem);

		// Revert

		revertSourceMenuItem = new JMenuItem("Revert");
		revertSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRevertSource();
			}
		});
		fileMenu.add(revertSourceMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// Copy To Scenario...

		copyToScenarioMenuItem = new JMenuItem("Copy To Scenario...");
		copyToScenarioMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doCopyToScenario();
			}
		});
		fileMenu.add(copyToScenarioMenuItem);

		// Export...

		exportSourcesMenuItem = new JMenuItem("Export...");
		exportSourcesMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doExportSources();
			}
		});
		fileMenu.add(exportSourcesMenuItem);

		// Save As User Record...

		saveAsUserRecordMenuItem = new JMenuItem("Save As User Record...");
		saveAsUserRecordMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSaveAsUserRecord();
			}
		});
		fileMenu.add(saveAsUserRecordMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// Compare

		compareSourceMenuItem = new JMenuItem("Compare");
		compareSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doCompareSource();
			}
		});
		fileMenu.add(compareSourceMenuItem);

		// Add to the Edit menu, items to set/clear desired and undesired flags on a multiple selection.

		editMenu.addSeparator();

		setDesiredMenuItem = new JMenuItem("Set Desired");
		setDesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSetFlags(true, true);
			}
		});
		editMenu.add(setDesiredMenuItem);

		clearDesiredMenuItem = new JMenuItem("Clear Desired");
		clearDesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSetFlags(true, false);
			}
		});
		editMenu.add(clearDesiredMenuItem);

		toggleDesiredMenuItem = new JMenuItem("Toggle Desired");
		toggleDesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doToggleFlags(true);
			}
		});
		editMenu.add(toggleDesiredMenuItem);

		setUndesiredMenuItem = new JMenuItem("Set Undesired");
		setUndesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSetFlags(false, true);
			}
		});
		editMenu.add(setUndesiredMenuItem);

		clearUndesiredMenuItem = new JMenuItem("Clear Undesired");
		clearUndesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSetFlags(false, false);
			}
		});
		editMenu.add(clearUndesiredMenuItem);

		toggleUndesiredMenuItem = new JMenuItem("Toggle Undesired");
		toggleUndesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doToggleFlags(false);
			}
		});
		editMenu.add(toggleUndesiredMenuItem);

		// Create the contextual popup menu and items, subset of file menu items plus extra edit menu items.

		sourceTablePopupMenu = new JPopupMenu();

		cmOpenSourceMenuItem = new JMenuItem("View");
		cmOpenSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doOpenSource();
			}
		});
		sourceTablePopupMenu.add(cmOpenSourceMenuItem);

		cmRemoveSourceMenuItem = new JMenuItem("Remove");
		cmRemoveSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRemoveSource();
			}
		});
		sourceTablePopupMenu.add(cmRemoveSourceMenuItem);

		cmReplicateSourceMenuItem = new JMenuItem("Replicate...");
		cmReplicateSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doReplicateSource();
			}
		});
		sourceTablePopupMenu.add(cmReplicateSourceMenuItem);

		cmUnlockSourceMenuItem = new JMenuItem("Allow Editing");
		cmUnlockSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doUnlockSource();
			}
		});
		sourceTablePopupMenu.add(cmUnlockSourceMenuItem);

		cmRevertSourceMenuItem = new JMenuItem("Revert");
		cmRevertSourceMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRevertSource();
			}
		});
		sourceTablePopupMenu.add(cmRevertSourceMenuItem);

		cmSetDesiredMenuItem = new JMenuItem("Set Desired");
		cmSetDesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSetFlags(true, true);
			}
		});
		sourceTablePopupMenu.add(cmSetDesiredMenuItem);

		cmClearDesiredMenuItem = new JMenuItem("Clear Desired");
		cmClearDesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSetFlags(true, false);
			}
		});
		sourceTablePopupMenu.add(cmClearDesiredMenuItem);

		cmToggleDesiredMenuItem = new JMenuItem("Toggle Desired");
		cmToggleDesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doToggleFlags(true);
			}
		});
		sourceTablePopupMenu.add(cmToggleDesiredMenuItem);

		cmSetUndesiredMenuItem = new JMenuItem("Set Undesired");
		cmSetUndesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSetFlags(false, true);
			}
		});
		sourceTablePopupMenu.add(cmSetUndesiredMenuItem);

		cmClearUndesiredMenuItem = new JMenuItem("Clear Undesired");
		cmClearUndesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSetFlags(false, false);
			}
		});
		sourceTablePopupMenu.add(cmClearUndesiredMenuItem);

		cmToggleUndesiredMenuItem = new JMenuItem("Toggle Undesired");
		cmToggleUndesiredMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doToggleFlags(false);
			}
		});
		sourceTablePopupMenu.add(cmToggleUndesiredMenuItem);

		// Initial update of UI state.

		updateControls();
		updateDocumentName();
	}


	//=================================================================================================================
	// Table model for the source list in a scenario.  This is a wrapper around the SourceListData model object which
	// is actually managing the list.

	private class SourceTableModel extends AbstractTableModel implements TableFilterModel {

		private static final String SOURCE_DESIRED_COLUMN = "Des";
		private static final String SOURCE_UNDESIRED_COLUMN = "Und";
		private static final String SOURCE_CALLSIGN_COLUMN = "Call Sign";
		private static final String SOURCE_CALLSIGN_COLUMN_WL = "Call/ID";
		private static final String SOURCE_CHANNEL_COLUMN = "Channel";
		private static final String SOURCE_SERVICE_COLUMN = "Svc";
		private static final String SOURCE_STATUS_COLUMN = "Status";
		private static final String SOURCE_CITY_COLUMN = "City";
		private static final String SOURCE_STATE_COLUMN = "State";
		private static final String SOURCE_COUNTRY_COLUMN = "Cntry";
		private static final String SOURCE_FACILITY_ID_COLUMN = "Facility ID";
		private static final String SOURCE_FILE_COLUMN = "File Number";
		private static final String SOURCE_FILE_COLUMN_WL = "File/Ref Number";
		private static final String SOURCE_DATE_COLUMN = "Date";

		private String[] columnNamesNoWL = {
			SOURCE_DESIRED_COLUMN,
			SOURCE_UNDESIRED_COLUMN,
			SOURCE_CALLSIGN_COLUMN,
			SOURCE_CHANNEL_COLUMN,
			SOURCE_SERVICE_COLUMN,
			SOURCE_STATUS_COLUMN,
			SOURCE_CITY_COLUMN,
			SOURCE_STATE_COLUMN,
			SOURCE_COUNTRY_COLUMN,
			SOURCE_FACILITY_ID_COLUMN,
			SOURCE_FILE_COLUMN,
			SOURCE_DATE_COLUMN
		};

		private String[] columnNamesWL = {
			SOURCE_DESIRED_COLUMN,
			SOURCE_UNDESIRED_COLUMN,
			SOURCE_CALLSIGN_COLUMN_WL,
			SOURCE_CHANNEL_COLUMN,
			SOURCE_SERVICE_COLUMN,
			SOURCE_STATUS_COLUMN,
			SOURCE_CITY_COLUMN,
			SOURCE_STATE_COLUMN,
			SOURCE_COUNTRY_COLUMN,
			SOURCE_FACILITY_ID_COLUMN,
			SOURCE_FILE_COLUMN_WL,
			SOURCE_DATE_COLUMN
		};

		private String[] columnNames;

		private static final int SOURCE_DESIRED_INDEX = 0;
		private static final int SOURCE_UNDESIRED_INDEX = 1;
		private static final int SOURCE_CALLSIGN_INDEX = 2;
		private static final int SOURCE_CHANNEL_INDEX = 3;
		private static final int SOURCE_SERVICE_INDEX = 4;
		private static final int SOURCE_STATUS_INDEX = 5;
		private static final int SOURCE_CITY_INDEX = 6;
		private static final int SOURCE_STATE_INDEX = 7;
		private static final int SOURCE_COUNTRY_INDEX = 8;
		private static final int SOURCE_FACILITY_ID_INDEX = 9;
		private static final int SOURCE_FILE_INDEX = 10;
		private static final int SOURCE_DATE_INDEX = 11;

		private JPanel panel;

		private TableFilterPanel filterPanel;


		//-------------------------------------------------------------------------------------------------------------

		private SourceTableModel(JPanel thePanel) {

			if (Study.isRecordTypeAllowed(studyType, Source.RECORD_TYPE_WL)) {
				columnNames = columnNamesWL;
			} else {
				columnNames = columnNamesNoWL;
			}

			panel = thePanel;

			filterPanel = new TableFilterPanel(outerThis, this);
		}


		//-------------------------------------------------------------------------------------------------------------
		// Set up a table to display this model.  Custom renderer to apply color per locked state.  Also the table
		// gets custom sorting.

		private JTable createTable(EditMenu theEditMenu) {

			JTable theTable = new JTable(this);
			AppController.configureTable(theTable, theEditMenu);

			// Subclass TableRowSorter so that any call to sortKeys() substitutes a fixed list of keys based on the
			// first column in the argument list.  Also if the argument is null or empty rather than no sort, use a
			// default sort.

			TableRowSorter<SourceTableModel> theSorter = new TableRowSorter<SourceTableModel>(this) {
				public void setSortKeys(java.util.List<? extends RowSorter.SortKey> sortKeys) {

					int columnIndex = SOURCE_DESIRED_INDEX;
					SortOrder normal = SortOrder.ASCENDING;
					SortOrder reverse = SortOrder.DESCENDING;

					if ((null != sortKeys) && (sortKeys.size() > 0)) {
						RowSorter.SortKey theKey = sortKeys.get(0);
						columnIndex = theKey.getColumn();
						if ((SOURCE_DESIRED_INDEX == columnIndex) || (SOURCE_UNDESIRED_INDEX == columnIndex)) {
							if (theKey.getSortOrder().equals(SortOrder.ASCENDING)) {
								normal = SortOrder.DESCENDING;
								reverse = SortOrder.ASCENDING;
							}
						} else {
							if (theKey.getSortOrder().equals(SortOrder.DESCENDING)) {
								normal = SortOrder.DESCENDING;
								reverse = SortOrder.ASCENDING;
							}
						}
					}

					ArrayList<RowSorter.SortKey> theKeys = new ArrayList<RowSorter.SortKey>();

					switch (columnIndex) {

						case SOURCE_DESIRED_INDEX:   // Desired (country, state, city, channel).  Default.
						default:
							theKeys.add(new RowSorter.SortKey(SOURCE_DESIRED_INDEX, reverse));
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_UNDESIRED_INDEX:   // Undesired (country, state, city, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_UNDESIRED_INDEX, reverse));
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_CALLSIGN_INDEX:   // Call sign (status, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_CALLSIGN_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATUS_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_CHANNEL_INDEX:   // Channel (country, state, city).
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							break;

						case SOURCE_SERVICE_INDEX:   // Service (country, state, city, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_SERVICE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_STATUS_INDEX:   // Status (country, state, city, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_STATUS_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_CITY_INDEX:   // City (country, state, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_STATE_INDEX:   // State (country, city, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_COUNTRY_INDEX:   // Country (state, city, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_FACILITY_ID_INDEX:   // Facility ID (status, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_FACILITY_ID_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATUS_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_FILE_INDEX:   // File number (country, state, city, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_FILE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;

						case SOURCE_DATE_INDEX:   // Sequence date (country, state, city, channel).
							theKeys.add(new RowSorter.SortKey(SOURCE_DATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_COUNTRY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_STATE_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CITY_INDEX, normal));
							theKeys.add(new RowSorter.SortKey(SOURCE_CHANNEL_INDEX, normal));
							break;
					}

					super.setSortKeys(theKeys);
				}
			};

			// The string converter may provide different properties for sorting that are displayed, see implementation
			// of the StationRecord interface in SourceEditData.

			TableStringConverter theConverter = new TableStringConverter() {
				public String toString(TableModel theModel, int rowIndex, int columnIndex) {

					Scenario.SourceListItem theItem = scenario.sourceData.get(filterPanel.forwardIndex[rowIndex]);
					SourceEditData theSource = scenario.sourceData.getSource(filterPanel.forwardIndex[rowIndex]);

					switch (columnIndex) {

						case SOURCE_DESIRED_INDEX:
							return (theItem.isDesired ? "0" : "1");

						case SOURCE_UNDESIRED_INDEX:
							return (theItem.isUndesired ? "0" : "1");

						case SOURCE_CALLSIGN_INDEX:
							return theSource.getCallSign();

						case SOURCE_CHANNEL_INDEX:
							return theSource.getSortChannel();

						case SOURCE_SERVICE_INDEX:
							return theSource.getServiceCode();

						case SOURCE_STATUS_INDEX:
							return theSource.getSortStatus();

						case SOURCE_CITY_INDEX:
							return theSource.getCity();

						case SOURCE_STATE_INDEX:
							return theSource.getState();

						case SOURCE_COUNTRY_INDEX:
							return theSource.getSortCountry();

						case SOURCE_FACILITY_ID_INDEX:
							return theSource.getSortFacilityID();

						case SOURCE_FILE_INDEX:
							return theSource.getFileNumber();

						case SOURCE_DATE_INDEX:
							return theSource.getSortSequenceDate();
					}

					return "";
				}
			};

			theSorter.setStringConverter(theConverter);
			theTable.setRowSorter(theSorter);
			theSorter.setSortKeys(null);

			DefaultTableCellRenderer theRend = new DefaultTableCellRenderer() {
				public Component getTableCellRendererComponent(JTable t, Object o, boolean s, boolean f, int r, int c) {

					JLabel comp = (JLabel)super.getTableCellRendererComponent(t, o, s, f, r, c);

					SourceEditData theSource =
						scenario.sourceData.getSource(filterPanel.forwardIndex[t.convertRowIndexToModel(r)]);

					if (!s) {
						if (theSource.isLocked) {
							comp.setForeground(Color.BLACK);
						} else {
							comp.setForeground(Color.GREEN.darker());
						}
					}

					if (SOURCE_CALLSIGN_INDEX == c) {
						String cmnt = theSource.makeCommentText();
						if ((null != cmnt) && (cmnt.length() > 0)) {
							if (!s) {
								comp.setForeground(Color.BLUE);
							}
							comp.setToolTipText(cmnt);
						} else {
							comp.setToolTipText(null);
						}
					} else {
						comp.setToolTipText(null);
					}

					return comp;
				}
			};

			TableColumnModel columnModel = theTable.getColumnModel();

			TableColumn theColumn = columnModel.getColumn(SOURCE_DESIRED_INDEX);
			theColumn.setMinWidth(AppController.textFieldWidth[3]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[3]);

			theColumn = columnModel.getColumn(SOURCE_UNDESIRED_INDEX);
			theColumn.setMinWidth(AppController.textFieldWidth[3]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[3]);

			theColumn = columnModel.getColumn(SOURCE_CALLSIGN_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[7]);

			theColumn = columnModel.getColumn(SOURCE_CHANNEL_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[7]);

			theColumn = columnModel.getColumn(SOURCE_SERVICE_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[2]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[2]);

			theColumn = columnModel.getColumn(SOURCE_STATUS_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[4]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[4]);

			theColumn = columnModel.getColumn(SOURCE_CITY_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[20]);

			theColumn = columnModel.getColumn(SOURCE_STATE_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[3]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[3]);

			theColumn = columnModel.getColumn(SOURCE_COUNTRY_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[3]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[3]);

			theColumn = columnModel.getColumn(SOURCE_FACILITY_ID_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[5]);

			theColumn = columnModel.getColumn(SOURCE_FILE_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[23]);

			theColumn = columnModel.getColumn(SOURCE_DATE_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[8]);

			return theTable;
		}


		//-------------------------------------------------------------------------------------------------------------

		private int addOrReplace(SourceEditData theSource, boolean theDesFlag, boolean theUndFlag) {

			postChange(scenario.sourceData.addOrReplace(theSource, theDesFlag, theUndFlag));

			int modelIndex = scenario.sourceData.getLastRowChanged();
			if (modelIndex >= 0) {
				return filterPanel.reverseIndex[modelIndex];
			}
			return modelIndex;
		}


		//--------------------------------------------------------------------------------------------------------------

		private boolean remove(int rowIndex) {

			return postChange(scenario.sourceData.remove(filterPanel.forwardIndex[rowIndex]));
		}


		//-------------------------------------------------------------------------------------------------------------

		private boolean remove(int[] rows) {

			int[] modRows = new int[rows.length];
			for (int i = 0; i < rows.length; i++) {
				modRows[i] = filterPanel.forwardIndex[rows[i]];
			}
			return postChange(scenario.sourceData.remove(modRows));
		}


		//-------------------------------------------------------------------------------------------------------------

		private boolean set(int rowIndex, SourceEditData newSource) {

			return postChange(scenario.sourceData.set(filterPanel.forwardIndex[rowIndex], newSource));
		}


		//-------------------------------------------------------------------------------------------------------------

		private boolean setUnfiltered(int rowIndex, SourceEditData newSource) {

			return postChange(scenario.sourceData.set(rowIndex, newSource));
		}


		//-------------------------------------------------------------------------------------------------------------

		private Scenario.SourceListItem get(int rowIndex) {

			return scenario.sourceData.get(filterPanel.forwardIndex[rowIndex]);
		}


		//-------------------------------------------------------------------------------------------------------------

		private SourceEditData getSource(int rowIndex) {

			return scenario.sourceData.getSource(filterPanel.forwardIndex[rowIndex]);
		}


		//-------------------------------------------------------------------------------------------------------------

		private boolean setIsDesired(int rowIndex, boolean flag) {

			if (isCellEditable(rowIndex, SOURCE_DESIRED_INDEX)) {
				return postChange(scenario.sourceData.setIsDesired(filterPanel.forwardIndex[rowIndex], flag));
			}
			return false;
		}


		//-------------------------------------------------------------------------------------------------------------

		private boolean setIsUndesired(int rowIndex, boolean flag) {

			if (isCellEditable(rowIndex, SOURCE_UNDESIRED_INDEX)) {
				return postChange(scenario.sourceData.setIsUndesired(filterPanel.forwardIndex[rowIndex], flag));
			}
			return false;
		}


		//-------------------------------------------------------------------------------------------------------------
		// Called after table data may have changed, fire appropriate change events.  Gets complicated due to the
		// filter, the filter is updated and the filtered model row determined both before and after that update.
		// Which filtered model index matters depends on the type of change.  The model row affected may not be in the
		// filtered model before, after, neither, or both; which means an update may have to be notified as a delete or
		// an insert if the change caused the row to leave or enter the filtered model.

		private boolean postChange(boolean didChange) {

			if (didChange) {

				int modelIndex = scenario.sourceData.getLastRowChanged();
				int rowBefore = -1, rowAfter = -1;
				if ((modelIndex >= 0) && (modelIndex < filterPanel.reverseIndex.length)) {
					rowBefore = filterPanel.reverseIndex[modelIndex];
				}
				filterPanel.updateFilter();
				if ((modelIndex >= 0) && (modelIndex < filterPanel.reverseIndex.length)) {
					rowAfter = filterPanel.reverseIndex[modelIndex];
				}

				switch (scenario.sourceData.getLastChange()) {

					case ListDataChange.NO_CHANGE:
					default:
						break;

					case ListDataChange.ALL_CHANGE:
						fireTableDataChanged();
						updateBorder();
						break;

					case ListDataChange.INSERT:
						if (rowAfter >= 0) {
							fireTableRowsInserted(rowAfter, rowAfter);
							updateBorder();
						}
						break;

					case ListDataChange.UPDATE:
						if (rowBefore >= 0) {
							if (rowAfter >= 0) {
								fireTableRowsUpdated(rowBefore, rowAfter);
							} else {
								fireTableRowsDeleted(rowBefore, rowBefore);
								updateBorder();
							}
						} else {
							if (rowAfter >= 0) {
								fireTableRowsInserted(rowAfter, rowAfter);
								updateBorder();
							}
						}
						break;

					case ListDataChange.DELETE:
						if (rowBefore >= 0) {
							fireTableRowsDeleted(rowBefore, rowBefore);
							updateBorder();
						}
						break;
				}
			}

			return didChange;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void dataWasChanged() {

			filterPanel.updateFilter();
			fireTableDataChanged();
			updateBorder();
		}


		//-------------------------------------------------------------------------------------------------------------

		public void filterDidChange() {

			fireTableDataChanged();
			updateBorder();
		}


		//-------------------------------------------------------------------------------------------------------------

		private void updateBorder() {

			int n = filterPanel.forwardIndex.length;
			if (n > 0) {
				panel.setBorder(BorderFactory.createTitledBorder(String.valueOf(n) + " stations"));
			} else {
				panel.setBorder(BorderFactory.createTitledBorder("Stations"));
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getColumnCount() {

			return columnNames.length;
		}


		//-------------------------------------------------------------------------------------------------------------

		public String getColumnName(int columnIndex) {

			return columnNames[columnIndex];
		}


		//-------------------------------------------------------------------------------------------------------------

		public boolean filterByColumn(int columnIndex) {

			switch (columnIndex) {
				case SOURCE_CALLSIGN_INDEX:
				case SOURCE_CHANNEL_INDEX:
				case SOURCE_SERVICE_INDEX:
				case SOURCE_STATE_INDEX:
				case SOURCE_COUNTRY_INDEX: {
					return true;
				}
			}

			return false;
		}


		//-------------------------------------------------------------------------------------------------------------

		public boolean collapseFilterChoices(int columnIndex) {

			switch (columnIndex) {
				case SOURCE_CALLSIGN_INDEX:
				case SOURCE_CHANNEL_INDEX:
				case SOURCE_STATE_INDEX: {
					return true;
				}
			}

			return false;
		}


		//-------------------------------------------------------------------------------------------------------------

		public Class getColumnClass(int columnIndex) {

			if ((SOURCE_DESIRED_INDEX == columnIndex) || (SOURCE_UNDESIRED_INDEX == columnIndex)) {
				return Boolean.class;
			}

			return Object.class;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getRowCount() {

			return filterPanel.forwardIndex.length;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getUnfilteredRowCount() {

			return scenario.sourceData.getRowCount();
		}


		//-------------------------------------------------------------------------------------------------------------
		// Permanent items cannot have desired/undesired state changed.  In various study types including TV IX check
		// and wireless IX, there can be only one desired station per scenario which was added programmatically when
		// the scenario was created, so the desired column can never be changed.  Also wireless records can never be
		// desireds, those should only appear in OET-74 studies, but check the record type too just in case.  In an FM
		// and TV channel 6 study there can only be one desired TV, like TV and wireless IX that is a permanent entry
		// added when the scenario was created.  But FMs in that study type can be both desired and undesired, so in
		// that case the desired lockout only applies to TV records.  Also in TVIX studies, some scenario types cannot
		// have desired or undesired flags changed at all.

		public boolean isCellEditable(int rowIndex, int columnIndex) {

			if ((Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == scenario.scenarioType) ||
					(Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType)) {
				return false;
			}

			rowIndex = filterPanel.forwardIndex[rowIndex];

			if (SOURCE_DESIRED_INDEX == columnIndex) {
				Scenario.SourceListItem theItem = scenario.sourceData.get(rowIndex);
				if (theItem.isPermanent) {
					return false;
				}
				if ((Study.STUDY_TYPE_TV_IX == studyType) || (Study.STUDY_TYPE_TV_OET74 == studyType)) {
					return false;
				}
				SourceEditData theSource = scenario.sourceData.getSource(rowIndex);
				if (Source.RECORD_TYPE_WL == theSource.recordType) {
					return false;
				}
				if ((Study.STUDY_TYPE_TV6_FM == studyType) && (Source.RECORD_TYPE_TV == theSource.recordType)) {
					return false;
				}
				return true;
			}

			if (SOURCE_UNDESIRED_INDEX == columnIndex) {
				Scenario.SourceListItem theItem = scenario.sourceData.get(rowIndex);
				if (theItem.isPermanent) {
					return false;
				}
				return true;
			}

			return false;
		}


		//-------------------------------------------------------------------------------------------------------------

		public Object getValueAt(int rowIndex, int columnIndex) {

			return getCellValue(filterPanel.forwardIndex[rowIndex], columnIndex);
		}


		//-------------------------------------------------------------------------------------------------------------

		public String getUnfilteredValueAt(int rowIndex, int columnIndex) {

			return getCellValue(rowIndex, columnIndex).toString();
		}


		//-------------------------------------------------------------------------------------------------------------

		private Object getCellValue(int rowIndex, int columnIndex) {

			Scenario.SourceListItem theItem = scenario.sourceData.get(rowIndex);
			SourceEditData theSource = scenario.sourceData.getSource(rowIndex);

			switch (columnIndex) {

				case SOURCE_DESIRED_INDEX:
					return Boolean.valueOf(theItem.isDesired);

				case SOURCE_UNDESIRED_INDEX:
					return Boolean.valueOf(theItem.isUndesired);

				case SOURCE_CALLSIGN_INDEX:
					return theSource.getCallSign();

				case SOURCE_CHANNEL_INDEX:
					return theSource.getChannel() + " " + theSource.getFrequency();

				case SOURCE_SERVICE_INDEX:
					return theSource.getServiceCode();

				case SOURCE_STATUS_INDEX:
					return theSource.getStatus();

				case SOURCE_CITY_INDEX:
					return theSource.getCity();

				case SOURCE_STATE_INDEX:
					return theSource.getState();

				case SOURCE_COUNTRY_INDEX:
					return theSource.getCountryCode();

				case SOURCE_FACILITY_ID_INDEX:
					return theSource.getFacilityID();

				case SOURCE_FILE_INDEX:
					return theSource.getFileNumber();

				case SOURCE_DATE_INDEX:
					return theSource.getSequenceDate();
			}

			return "";
		}


		//-------------------------------------------------------------------------------------------------------------
		// Note this does not apply the filter because the setter methods will.

		public void setValueAt(Object value, int rowIndex, int columnIndex) {

			switch (columnIndex) {

				case SOURCE_DESIRED_INDEX:
					if (setIsDesired(rowIndex, ((Boolean)value).booleanValue())) {
						setDidEdit();
					}
					break;

				case SOURCE_UNDESIRED_INDEX:
					if (setIsUndesired(rowIndex, ((Boolean)value).booleanValue())) {
						setDidEdit();
					}
					break;
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update the state of buttons and menu items per current table selection.

	private void updateControls() {

		int rowCount = sourceTable.getSelectedRowCount();
		boolean isEditing = false, viewOnly = false, eAdd = false, eOpen = false, eRemove = false, eReplicate = false,
			eUnlock = false, eRevert = false, eCopy = false, eCompare = false, eExport = false, eFlags = false,
			eSaveAs = false;

		if (1 == rowCount) {

			int rowIndex = sourceTable.convertRowIndexToModel(sourceTable.getSelectedRow());
			Scenario.SourceListItem theItem = sourceModel.get(rowIndex);
			SourceEditData theSource = sourceModel.getSource(rowIndex);
			SourceEditor theEditor = sourceEditors.get(theSource.key);
			if (null != theEditor) {
				if (theEditor.isVisible()) {
					isEditing = !theSource.isLocked;
				} else {
					sourceEditors.remove(theSource.key);
				}
			}

			viewOnly = theSource.isLocked;
			eAdd = true;
			eOpen = true;
			eRemove = !theItem.isPermanent;
			eReplicate = (Source.RECORD_TYPE_TV == theSource.recordType) && !isEditing;
			eUnlock = theSource.isLocked && !theSource.isReplication();
			eRevert = theSource.isReplication() || (!theSource.isLocked && theSource.hasRecordID() && !isEditing);
			eCopy = !isEditing;
			eCompare = !isEditing;
			eExport = !isEditing;
			eFlags = true;
			eSaveAs = !theSource.isLocked && !isEditing;

			// For a permanent entry in an OET74 or TV6FM study, replication is not allowed, and revert is not allowed
			// if the entry is already a replication.  In those studies, the permanent entry in a scenario is a target
			// record that was used to automatically build the rest of the scenario based on interference rules, so
			// channel, service, and location must not change else the entire scenario is invalidated.  Replication
			// will always change either channel or service, or both.  An initial replication may have been performed
			// on the record before the scenario was built, that must not be reverted.  However the record can be made
			// editable, so reverting an editable record is allowed.  The source editor will not allow channel or
			// coordinates to be edited in this case, so the original will always match.  Except, it is now possible
			// to change the channel of a record during the initial build, in which case the record is made editable
			// but the original may have a different channel.  But that case has to be checked for separately in the
			// revert method, it will abort an editable-record revert if the original record has a different channel.

			if (theItem.isPermanent && ((Study.STUDY_TYPE_TV_OET74 == studyType) ||
					(Study.STUDY_TYPE_TV6_FM == studyType))) {
				eReplicate = false;
				if (theSource.isReplication()) {
					eRevert = false;
				}
			}

		} else {

			viewOnly = true;
			eAdd = true;
			if (rowCount > 1) {
				eRemove = true;
				eCopy = true;
				if (rowCount <= 3) {
					eCompare = true;
				}
				eExport = true;
				eFlags = true;
			}
		}

		// In a TV IX check study, special scenario types identify scenarios that are part of the auto-build.  The
		// inteference scenarios are entirely non-editable.  The proposal scenario can't have records added or removed,
		// but the existing record(s) can be edited or replicated.

		if (Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType) {
			eAdd = false;
			eRemove = false;
			eReplicate = false;
			eUnlock = false;
			eRevert = false;
			eFlags = false;
		} else {
			if (Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == scenario.scenarioType) {
				eAdd = false;
				eRemove = false;
				eFlags = false;
			}
		}

		addSourceButton.setEnabled(eAdd);
		addSourcesMenu.setEnabled(eAdd);

		addSourceMenuItem.setEnabled(eAdd);
		addSourcesMenuItem.setEnabled(eAdd);
		if (null != importSourcesMenuItem) {
			importSourcesMenuItem.setEnabled(eAdd);
		}
		if (null != importSourcesMenu) {
			importSourcesMenu.setEnabled(eAdd);
		}

		if (viewOnly) {
			openSourceButton.setText("View");
			openSourceMenuItem.setText("View");
			cmOpenSourceMenuItem.setText("View");
		} else {
			openSourceButton.setText("Edit");
			openSourceMenuItem.setText("Edit");
			cmOpenSourceMenuItem.setText("Edit");
		}

		openSourceButton.setEnabled(eOpen);
		openSourceMenuItem.setEnabled(eOpen);
		cmOpenSourceMenuItem.setEnabled(eOpen);

		removeSourceButton.setEnabled(eRemove);
		removeSourceMenuItem.setEnabled(eRemove);
		cmRemoveSourceMenuItem.setEnabled(eRemove);

		replicateSourceMenuItem.setEnabled(eReplicate);
		cmReplicateSourceMenuItem.setEnabled(eReplicate);
		unlockSourceMenuItem.setEnabled(eUnlock);
		cmUnlockSourceMenuItem.setEnabled(eUnlock);
		revertSourceMenuItem.setEnabled(eRevert);
		cmRevertSourceMenuItem.setEnabled(eRevert);
		copyToScenarioMenuItem.setEnabled(eCopy);
		compareSourceMenuItem.setEnabled(eCompare);
		exportSourcesMenuItem.setEnabled(eExport);
		saveAsUserRecordMenuItem.setEnabled(eSaveAs);

		setDesiredMenuItem.setEnabled(eFlags);
		cmSetDesiredMenuItem.setEnabled(eFlags);
		clearDesiredMenuItem.setEnabled(eFlags);
		cmClearDesiredMenuItem.setEnabled(eFlags);
		toggleDesiredMenuItem.setEnabled(eFlags);
		cmToggleDesiredMenuItem.setEnabled(eFlags);
		setUndesiredMenuItem.setEnabled(eFlags);
		cmSetUndesiredMenuItem.setEnabled(eFlags);
		clearUndesiredMenuItem.setEnabled(eFlags);
		cmClearUndesiredMenuItem.setEnabled(eFlags);
		toggleUndesiredMenuItem.setEnabled(eFlags);
		cmToggleUndesiredMenuItem.setEnabled(eFlags);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Forward messages from geography editor to secondary editors.

	public void geographyChanged(int scopeStudyKey, int scopeSourceKey, Integer geoKey, int geoType) {

		for (SourceEditor theEditor : sourceEditors.values()) {
			theEditor.geographyChanged(scopeStudyKey, scopeSourceKey, geoKey, geoType);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void geographyScopeChanged(int scopeStudyKey, int scopeSourceKey) {

		for (SourceEditor theEditor : sourceEditors.values()) {
			theEditor.geographyScopeChanged(scopeStudyKey, scopeSourceKey);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void setDidEdit() {

		parent.setDidEdit();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getKeyTitle() {

		Window theWin = parent.getWindow();
		if (theWin instanceof AppFrame) {
			return ((AppFrame)theWin).getKeyTitle() + "." + super.getKeyTitle();
		}

		return super.getKeyTitle();
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected String getFileMenuName() {

		return "Station";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void updateDocumentName() {

		String docName = parent.getDocumentName();
		if (null == docName) {
			setDocumentName(scenario.name);
		} else {
			setDocumentName(docName + "/" + scenario.name);
		}

		if (null != addSourcesDialog) {
			addSourcesDialog.updateDocumentName();
		}
		if (null != addSourceFinder) {
			addSourceFinder.updateDocumentName();
		}
		for (SourceEditor theEditor : sourceEditors.values()) {
			theEditor.updateDocumentName();
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public ScenarioEditData getScenario() {

		return scenario;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Called by the parent when performing a validity check.  This only checks pending edit state, it does not call
	// isDataValid() in the model objects, the parent will do that.  See StudyEditor.isDataValid().

	public boolean isDataValid(String title) {

		for (SourceEditor theEditor : sourceEditors.values()) {
			if (theEditor.isVisible() && theEditor.isEditing()) {
				AppController.beep();
				errorReporter.reportValidationError("Please commit or cancel record edits then re-try the operation");
				theEditor.toFront();
				return false;
			}
		}

		return commitCurrentField();
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doPrevious() {

		int rowCount = sourceTable.getRowCount();
		int rowIndex = sourceTable.getSelectedRow();
		if ((rowCount > 0) && (rowIndex != 0)) {
			if (rowIndex < 0) {
				rowIndex = rowCount - 1;
			} else {
				rowIndex--;
			}
			sourceTable.setRowSelectionInterval(rowIndex, rowIndex);
			sourceTable.scrollRectToVisible(sourceTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doNext() {

		int rowCount = sourceTable.getRowCount();
		int rowIndex = sourceTable.getSelectedRow();
		if ((rowCount > 0) && (rowIndex < (rowCount - 1))) {
			if (rowIndex < 0) {
				rowIndex = 0;
			} else {
				rowIndex++;
			}
			sourceTable.setRowSelectionInterval(rowIndex, rowIndex);
			sourceTable.scrollRectToVisible(sourceTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Show a RecordFind window configured to allow individual records to be added to the scenario.  Note the window
	// stays open to allow multiple adds, it must be explicitly closed when no longer needed.  The actual add process
	// happens in applyEditsFrom(), called by the find window when user clicks the apply button.

	private void doAddSource() {

		if ((Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == scenario.scenarioType) ||
				(Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType)) {
			return;
		}

		if (null != addSourceFinder) {
			if (addSourceFinder.isVisible()) {
				addSourceFinder.toFront();
				return;
			}
			addSourceFinder = null;
		}

		RecordFind theFinder = new RecordFind(this, "Add Station", studyType, 0);
		theFinder.setDefaultExtDbKey(scenario.study.extDbKey);
		theFinder.setAccessoryPanel(new OptionsPanel.AddRecord(this));
		theFinder.setApply(new String[] {"Add"}, new int[] {1}, true, false);
		theFinder.setStudy(scenario.study);

		addSourceFinder = theFinder;
		AppController.showWindow(theFinder);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Populate the menu of saved searchs.  This always contains an initial "null" object just to identify the menu,
	// also two default selections for desired and undesired searches.

	private void updateSearchMenu() {

		blockActionsStart();

		ArrayList<ExtDbSearch> theSearches = ExtDbSearch.getSearches(getDbID(), studyType);

		addSourcesMenu.removeAllItems();

		ExtDbSearch defSearch = new ExtDbSearch(0);
		defSearch.name = "Add Many...";
		addSourcesMenu.addItem(defSearch);

		// Some study types do not allow a desired or protected search.

		if ((Study.STUDY_TYPE_TV == studyType) || (Study.STUDY_TYPE_FM == studyType) ||
				(Study.STUDY_TYPE_TV6_FM == studyType)) {

			defSearch = new ExtDbSearch(studyType);
			defSearch.name = ExtDbSearch.DEFAULT_DESIREDS_SEARCH_NAME;
			defSearch.searchType = ExtDbSearch.SEARCH_TYPE_DESIREDS;
			addSourcesMenu.addItem(defSearch);

			defSearch = new ExtDbSearch(studyType);
			defSearch.name = ExtDbSearch.DEFAULT_PROTECTEDS_SEARCH_NAME;
			defSearch.searchType = ExtDbSearch.SEARCH_TYPE_PROTECTEDS;
			addSourcesMenu.addItem(defSearch);
		}

		defSearch = new ExtDbSearch(studyType);
		defSearch.name = ExtDbSearch.DEFAULT_UNDESIREDS_SEARCH_NAME;
		defSearch.searchType = ExtDbSearch.SEARCH_TYPE_UNDESIREDS;
		defSearch.preferOperating = true;
		addSourcesMenu.addItem(defSearch);

		for (ExtDbSearch theSearch : theSearches) {
			addSourcesMenu.addItem(theSearch);
		}

		blockActionsEnd();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add multiple records from an external data search.  If a search object is provided, it has the auto-run flag
	// set, and there is a valid default data set, immediately run the search.  Otherwise, open the search editing
	// dialog or if already showing update with the new object.

	private void doAddSources(ExtDbSearch search) {

		if ((Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == scenario.scenarioType) ||
				(Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType)) {
			return;
		}

		if ((null != search) && search.autoRun) {
			ExtDb theDb = null;
			if (null != scenario.study.extDbKey) {
				theDb = ExtDb.getExtDb(getDbID(), scenario.study.extDbKey, errorReporter);
			} else {
				ArrayList<KeyedRecord> dbList = ExtDb.getExtDbList(getDbID(), Study.getDefaultRecordType(studyType),
					errorReporter);
				if (null != dbList) {
					theDb = ExtDb.getExtDb(getDbID(), Integer.valueOf(dbList.get(0).key), errorReporter);
				}
			}
			if (null != theDb) {
				runExtDbSearch(theDb, false, search, errorReporter);
				return;
			}
		}

		if (null != addSourcesDialog) {
			if (addSourcesDialog.isVisible()) {
				if (null != search) {
					addSourcesDialog.setSearch(search);
				}
				addSourcesDialog.toFront();
				return;
			}
			addSourcesDialog = null;
		}

		ExtDbSearchDialog theDialog = new ExtDbSearchDialog(this, scenario.study.extDbKey, studyType);
		if (null != search) {
			theDialog.setSearch(search);
		}
		addSourcesDialog = theDialog;
		AppController.showWindow(theDialog);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Run an add-many search defined by an ExtDbSearch object.  First make sure this is a valid search, the study
	// type must match, the record type must be allowed, and some study types do not allow desired or protected
	// searches.  The search input dialog shouldn't allow invalid combinations, but check anyway to be safe.

	private boolean runExtDbSearch(ExtDb extDb, boolean useBaseline, ExtDbSearch search, ErrorReporter errors) {

		if ((search.studyType != studyType) || !Study.isRecordTypeAllowed(studyType, search.recordType) ||
				(((ExtDbSearch.SEARCH_TYPE_DESIREDS == search.searchType) ||
						(ExtDbSearch.SEARCH_TYPE_PROTECTEDS == search.searchType)) &&
					(Study.STUDY_TYPE_TV != studyType) && (Study.STUDY_TYPE_FM != studyType) &&
					((Study.STUDY_TYPE_TV6_FM != studyType) || (Source.RECORD_TYPE_FM != search.recordType))) ||
				((Study.STUDY_TYPE_TV_IX == studyType) && !search.disableMX)) {
			errors.reportWarning("Search is not valid in this study, no stations were added");
			return false;
		}

		// If this is a protecteds or undesireds search, there must be undesired or desired records already in the
		// scenario of an appropriate type to apply rules during the search.  For a protecteds search, undesireds of
		// the same record type as the search must exist.  For an undesireds search, if the search type is TV or FM
		// desireds of the same type must exist, but if the undesired is WL a TV desired must exist.

		if (ExtDbSearch.SEARCH_TYPE_PROTECTEDS == search.searchType) {
			if (!scenario.sourceData.hasUndesiredSources(search.recordType)) {
				errors.reportWarning(
					"The scenario must have at least one undesired station\n" +
					"before a search for protecteds can be performed");
				return false;
			}

		} else {

			if (ExtDbSearch.SEARCH_TYPE_UNDESIREDS == search.searchType) {
				int checkType = search.recordType;
				if (Source.RECORD_TYPE_WL == checkType) {
					checkType = Source.RECORD_TYPE_TV;
				}
				if (!scenario.sourceData.hasDesiredSources(checkType)) {
					errors.reportWarning(
						"The scenario must have at least one desired station\n" +
						"before a search for undesireds can be performed");
					return false;
				}
			}
		}

		// If MX checks are disabled, remind the user that records will not be flagged as undesireds, so individual
		// records that should cause interference must be selected manually.  Skip this for an auto-run search.  Also
		// skip this for an IX check study, in which manually-edited scenarios are expected to contain MX undesireds.

		if (search.disableMX && !search.autoRun && (Study.STUDY_TYPE_TV_IX != studyType)) {
			errors.reportWarning(
				"MX checks are disabled, new records will not be marked as undesireds\n" +
				"Records that should be undesireds must be identified manually");
		}

		// Run it.

		final ExtDb theDb = extDb;
		final boolean theBaseline = useBaseline;
		final ExtDbSearch theSearch = search;

		BackgroundWorker<Integer> theWorker = new BackgroundWorker<Integer>(errors.getWindow(), errors.getTitle()) {
			public Integer doBackgroundWork(ErrorLogger errors) {
				int count = scenario.addRecords(theDb, theBaseline, theSearch, errors);
				if (count < 0) {
					return null;
				}
				return Integer.valueOf(count);
			}
		};

		errors.clearMessages();

		Integer result = theWorker.runWork("Searching for stations, please wait...", errors);
		if (null == result) {
			return false;
		}

		errors.showMessages();

		sourceModel.dataWasChanged();
		if (result.intValue() > 0) {
			setDidEdit();
		}

		errors.reportMessage(result.toString() + " records were added");

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Import records from a text file in the generic text format.  This is mostly a convenience, a generic data set
	// could instead be created and the file imported there, then the set used to add to the scenario by search.  But
	// direct import can be useful when the records in the file are already specifically tailored for a scenario so
	// the search and selection logic is unnecessary.  There are no filtering conditions or MX checks here, the only
	// thing that may exclude records is TV channel range.  The desired/undesired flags are set per user input (however
	// desired can't be set for some conditions, see SourceModel.isCellEditable()).  All records will be editable and
	// cannot be shared.  For that reason direct import should only be used if the records are needed in just one
	// scenario, otherwise a generic set is preferable so the records can be locked and shared.

	private void doImportSources(int theRecordType) {

		if ((Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == scenario.scenarioType) ||
				(Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType)) {
			return;
		}

		String title = "Import " + Source.getRecordTypeName(theRecordType) + " Records";
		errorReporter.setTitle(title);

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.OPEN_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new FileNameExtensionFilter("Text (*.csv,*.txt,*.dat)", "csv", "txt", "dat"));
		chooser.setAcceptAllFileFilterUsed(false);

		JCheckBox patternFileCheckBox = new JCheckBox("Separate pattern file");

		JPanel textPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		textPanel.setBorder(BorderFactory.createTitledBorder("Text options"));
		textPanel.add(patternFileCheckBox);

		JCheckBox desiredCheckBox = null;
		if ((Study.STUDY_TYPE_TV_IX != studyType) && (Study.STUDY_TYPE_TV_OET74 != studyType) &&
				(Source.RECORD_TYPE_WL != theRecordType) &&
				((Study.STUDY_TYPE_TV6_FM != studyType) || (Source.RECORD_TYPE_TV != theRecordType))) {
			desiredCheckBox = new JCheckBox("Desired");
		}
		JCheckBox undesiredCheckBox = new JCheckBox("Undesired");

		Box flagsBox = Box.createVerticalBox();
		if (null != desiredCheckBox) {
			flagsBox.add(desiredCheckBox);
		}
		flagsBox.add(undesiredCheckBox);

		JPanel flagsPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		flagsPanel.setBorder(BorderFactory.createTitledBorder("Scenario flags"));
		flagsPanel.add(flagsBox);

		Box optionsBox = Box.createVerticalBox();
		optionsBox.add(textPanel);
		optionsBox.add(flagsPanel);

		JPanel accessoryPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		accessoryPanel.add(optionsBox);

		chooser.setAccessory(accessoryPanel);

		if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Import")) {
			return;
		}
		File theFile = chooser.getSelectedFile();

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

		boolean isDesired = false;
		if (null != desiredCheckBox) {
			isDesired = desiredCheckBox.isSelected();
		}
		boolean isUndesired = undesiredCheckBox.isSelected();

		final int importRecordType = theRecordType;
		final File importFile = theFile;

		// Text format is usually self-contained, but may have pattern data in a separate file for legacy support.

		theFile = null;

		if (patternFileCheckBox.isSelected()) {

			chooser = new JFileChooser(importFile.getParentFile().getAbsolutePath());
			chooser.setDialogType(JFileChooser.OPEN_DIALOG);
			chooser.setDialogTitle("Choose pattern data file");
			chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
			chooser.setMultiSelectionEnabled(false);
			chooser.addChoosableFileFilter(new FileNameExtensionFilter("Text (*.csv,*.txt,*.dat)", "csv", "txt",
				"dat"));
			chooser.setAcceptAllFileFilterUsed(false);

			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Import")) {
				return;
			}
			theFile = chooser.getSelectedFile();

			AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());
		}

		final File patternFile = theFile;

		BackgroundWorker<ArrayList<SourceEditData>> theWorker =
				new BackgroundWorker<ArrayList<SourceEditData>>(this, title) {
			protected ArrayList<SourceEditData> doBackgroundWork(ErrorLogger errors) {
				return SourceEditData.readFromText(scenario.study, importRecordType, importFile, patternFile, errors);
			}
		};

		errorReporter.clearMessages();
		ArrayList<SourceEditData> newSources = theWorker.runWork("Importing records, please wait...",
			errorReporter);
		if (null == newSources) {
			return;
		}

		// Apply the results.

		errorReporter.showMessages();

		int add = 0, chan, minChan = scenario.study.getMinimumChannel(), maxChan = scenario.study.getMaximumChannel();
		for (SourceEditData newSource : newSources) {
			if (Source.RECORD_TYPE_TV == newSource.recordType) {
				chan = ((SourceEditDataTV)newSource).channel;
				if ((chan < minChan) || (chan > maxChan)) {
					continue;
				}
			}
			scenario.sourceData.addOrReplace(newSource, isDesired, isUndesired);
			add++;
		}

		if (add > 0) {
			sourceModel.dataWasChanged();
			setDidEdit();
		}

		errorReporter.reportMessage(String.valueOf(add) + " records were added");
	}


	//-----------------------------------------------------------------------------------------------------------------
	// View or edit a source in a SourceEditor window.  If editing, the change is made in applyEditsFrom().

	private void doOpenSource() {

		if (sourceTable.getSelectedRowCount() != 1) {
			return;
		}

		int rowIndex = sourceTable.convertRowIndexToModel(sourceTable.getSelectedRow());
		SourceEditData theSource = sourceModel.getSource(rowIndex);

		SourceEditor theEditor = sourceEditors.get(theSource.key);
		if (null != theEditor) {
			if (theEditor.isVisible()) {
				theEditor.toFront();
				return;
			} else {
				sourceEditors.remove(theSource.key);
			}
		}

		Scenario.SourceListItem theItem = sourceModel.get(rowIndex);

		theEditor = new SourceEditor(this);
		theEditor.setSource(theSource, theItem.isPermanent);

		AppController.showWindow(theEditor);
		sourceEditors.put(theSource.key, theEditor);

		updateControls();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Remove records from the scenario.  If any have open source editors those are canceled.

	private void doRemoveSource() {

		if ((Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == scenario.scenarioType) ||
				(Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType)) {
			return;
		}

		if (0 == sourceTable.getSelectedRowCount()) {
			return;
		}

		int[] selRows = sourceTable.getSelectedRows();
		ArrayList<Integer> toDelete = new ArrayList<Integer>();

		int i, rowIndex;
		Scenario.SourceListItem theItem;
		boolean hasPerm = false;
		SourceEditor theEditor;

		for (i = 0; i < selRows.length; i++) {
			rowIndex = sourceTable.convertRowIndexToModel(selRows[i]);
			theItem = sourceModel.get(rowIndex);
			if (theItem.isPermanent) {
				hasPerm = true;
			} else {
				theEditor = sourceEditors.get(Integer.valueOf(theItem.key));
				if (null != theEditor) {
					if (theEditor.isVisible()) {
						if (!theEditor.cancel()) {
							AppController.beep();
							theEditor.toFront();
							return;
						}
					} else {
						sourceEditors.remove(Integer.valueOf(theItem.key));
					}
				}
				toDelete.add(Integer.valueOf(rowIndex));
			}
		}

		if (hasPerm) {
			errorReporter.reportMessage("Remove Stations", "Some records are permanent and cannot be removed");
		}

		if (toDelete.isEmpty()) {
			return;
		}

		int[] rows = new int[toDelete.size()];
		for (i = 0; i < toDelete.size(); i++) {
			rows[i] = toDelete.get(i).intValue();
		}

		if (sourceModel.remove(rows)) {
			setDidEdit();
		}

		updateControls();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Replace a locked source with an editable copy, and open a source editor.  May need to cancel an existing source
	// editor viewing the record first.

	private void doUnlockSource() {

		if (Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType) {
			return;
		}

		if (sourceTable.getSelectedRowCount() != 1) {
			return;
		}

		int rowIndex = sourceTable.convertRowIndexToModel(sourceTable.getSelectedRow());
		SourceEditData theSource = sourceModel.getSource(rowIndex);
		if (!theSource.isLocked || theSource.isReplication()) {
			return;
		}

		SourceEditor theEditor = sourceEditors.get(theSource.key);
		if (null != theEditor) {
			if (theEditor.isVisible()) {
				if (!theEditor.cancel()) {
					AppController.beep();
					theEditor.toFront();
					return;
				}
			} else {
				sourceEditors.remove(theSource.key);
			}
		}

		errorReporter.setTitle("Allow Editing");

		SourceEditData newSource = theSource.deriveSource(false, errorReporter);
		if (null == newSource) {
			return;
		}

		sourceModel.set(rowIndex, newSource);
		setDidEdit();

		doOpenSource();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Replicate a TV source, or change the channel of replication.  If there is an open editor, this is not allowed
	// if the source is editable otherwise the open editor is cancelled.

	private void doReplicateSource() {

		if (Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType) {
			return;
		}

		if (sourceTable.getSelectedRowCount() != 1) {
			return;
		}

		int rowIndex = sourceTable.convertRowIndexToModel(sourceTable.getSelectedRow());
		SourceEditData theSource = sourceModel.getSource(rowIndex);
		if (Source.RECORD_TYPE_TV != theSource.recordType) {
			return;
		}
		SourceEditDataTV theSourceTV = (SourceEditDataTV)theSource;

		SourceEditor theEditor = sourceEditors.get(theSource.key);
		if (null != theEditor) {
			if (theEditor.isVisible()) {
				if (!theSource.isLocked || !theEditor.cancel()) {
					AppController.beep();
					theEditor.toFront();
					return;
				}
			} else {
				sourceEditors.remove(theSource.key);
			}
		}

		// This is not allowed on a permanent entry in an OET74 or TV6FM study; see comments in updateControls().

		Scenario.SourceListItem theItem = sourceModel.get(rowIndex);

		if (theItem.isPermanent && ((Study.STUDY_TYPE_TV_OET74 == studyType) ||
				(Study.STUDY_TYPE_TV6_FM == studyType))) {
			return;
		}

		String title = "Replicate";
		errorReporter.setTitle(title);

		// Retrieve the original source for replication, that may be the current source if it is not already a
		// replication source.

		SourceEditDataTV origSource = theSourceTV;
		if (null != theSourceTV.originalSourceKey) {
			SourceEditData aSource = scenario.study.getSource(theSourceTV.originalSourceKey);
			if ((null == aSource) || (Source.RECORD_TYPE_TV != aSource.recordType)) {
				errorReporter.reportError("The original record for replication does not exist");
				return;
			}
			origSource = (SourceEditDataTV)aSource;
		}

		// Get the new channel.  The channel entered must be different than the original, unless the original is
		// analog.  All replications are digital; analog sources can replicate to digital on the same channel.

		int repChan = 0, minChannel = scenario.study.getMinimumChannel(),
			maxChannel = scenario.study.getMaximumChannel();
		String str;

		do {

			str = JOptionPane.showInputDialog(this, "Enter the replication channel", title,
				JOptionPane.QUESTION_MESSAGE);
			if (null == str) {
				return;
			}
			str = str.trim();

			if (str.length() > 0) {
				try {
					repChan = Integer.parseInt(str);
					if ((repChan < minChannel) || (repChan > maxChannel)) {
						errorReporter.reportWarning("The channel must be in the range " + minChannel + " to " +
							maxChannel);
						repChan = 0;
					} else {
						if (origSource.service.serviceType.digital && (repChan == origSource.channel)) {
							errorReporter.reportWarning("The channel must be different than the original");
							repChan = 0;
						}
					}
				} catch (NumberFormatException ne) {
					errorReporter.reportWarning("The channel must be a number");
					repChan = 0;
				}
			}

		} while (0 == repChan);

		// If the current source is already a replication and the current channel was re-entered, nothing to do!

		if ((null != theSourceTV.originalSourceKey) && (repChan == theSourceTV.channel)) {
			return;
		}

		// Find or create the replication source, set it in the scenario, replacing the current.

		errorReporter.clearMessages();

		SourceEditData newSource = null;
		if (origSource.isLocked) {
			if (null != origSource.userRecordID) {
				newSource = scenario.study.findSharedReplicationSource(origSource.userRecordID, repChan);
			} else {
				if ((null != origSource.extDbKey) && (null != origSource.extRecordID)) {
					newSource = scenario.study.findSharedReplicationSource(origSource.extDbKey,
						origSource.extRecordID, repChan);
				}
			}
		}
		if (null == newSource) {
			newSource = origSource.replicate(repChan, errorReporter);
			if (null == newSource) {
				return;
			}
		}

		// Check if this replication is already in the scenario, if so abort.  See comments in doRevertSource().

		String newChan = newSource.getSortChannel();

		for (SourceEditData otherSource : scenario.sourceData.getSources()) {
			if ((((null != newSource.userRecordID) && (newSource.userRecordID.equals(otherSource.userRecordID))) ||
					((null != newSource.extDbKey) && newSource.extDbKey.equals(otherSource.extDbKey) &&
					(null != newSource.extRecordID) && newSource.extRecordID.equals(otherSource.extRecordID))) &&
					newChan.equals(otherSource.getSortChannel())) {
				errorReporter.reportWarning(
					"The original record is already replicated on that channel in the scenario");
				return;
			}
		}

		errorReporter.showMessages();

		sourceModel.set(rowIndex, newSource);
		setDidEdit();

		updateControls();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Revert a source.  If the selected source is a replication, revert to the original.  If it is an editable source
	// based on a user or external record, revert to the non-editable record directly from the user record table or
	// external data (that may have to be retrieved, there is no guarantee it is currently part of the shared source
	// model).  A non-replication locked source, or an editable source that is not based on a user or external record,
	// can't be reverted because there is nothing to revert to (note an editable source can never be locked again).

	private void doRevertSource() {

		if (Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType) {
			return;
		}

		if (sourceTable.getSelectedRowCount() != 1) {
			return;
		}

		int rowIndex = sourceTable.convertRowIndexToModel(sourceTable.getSelectedRow());
		Scenario.SourceListItem theItem = sourceModel.get(rowIndex);
		SourceEditData theSource = sourceModel.getSource(rowIndex);

		SourceEditor theEditor = sourceEditors.get(theSource.key);
		if (null != theEditor) {
			if (theEditor.isVisible()) {
				if (!theEditor.cancel()) {
					AppController.beep();
					theEditor.toFront();
					return;
				}
			} else {
				sourceEditors.remove(theSource.key);
			}
		}

		String title = "Revert";
		errorReporter.setTitle(title);

		SourceEditData newSource = null;

		errorReporter.clearErrors();

		if (theSource.isReplication()) {

			// In an OET74 or TV6FM study if the entry is permanent the record cannot be reverted, because the channel
			// and service cannot be allowed to change.  See comments in updateControls().

			if (theItem.isPermanent && ((Study.STUDY_TYPE_TV_OET74 == studyType) ||
					(Study.STUDY_TYPE_TV6_FM == studyType))) {
				return;
			}

			newSource = scenario.study.getSource(((SourceEditDataTV)theSource).originalSourceKey);
			if (null == newSource) {
				errorReporter.reportError("The original pre-replication record does not exist");
				return;
			}

			// A replication cannot be reverted if the original channel is illegal for the study.  This can happen
			// because most appearances of the RecordFind window offer the chance to replicate the selected record
			// before it is actually added to a scenario, in which case the original channel is not restricted by the
			// study's range, only the replication channel is restricted.

			int origChan = ((SourceEditDataTV)newSource).channel;
			if ((origChan < scenario.study.getMinimumChannel()) || (origChan > scenario.study.getMaximumChannel())) {
				errorReporter.reportWarning(
					"Replication cannot be reverted, the original channel is out of range for the study");
				return;
			}

			// A replication cannot be reverted if the original record already appears in the scenario.  The test is
			// the same identical-record test used when adding new entries, matching primary record IDs and channel.
			// Note the replication record will never match it's original by this test because the getSortChannel()
			// result includes a D/N suffix so even on-channel replication from analog to digital will have non-
			// matching channel strings from that method.

			String newChan = newSource.getSortChannel();

			for (SourceEditData otherSource : scenario.sourceData.getSources()) {
				if ((((null != newSource.userRecordID) && (newSource.userRecordID.equals(otherSource.userRecordID))) ||
						((null != newSource.extDbKey) && newSource.extDbKey.equals(otherSource.extDbKey) &&
						(null != newSource.extRecordID) && newSource.extRecordID.equals(otherSource.extRecordID))) &&
						newChan.equals(otherSource.getSortChannel())) {
					errorReporter.reportWarning(
						"Replication cannot be reverted, the original record is already in the scenario");
					return;
				}
			}

		} else {

			if (theSource.isLocked || !theSource.hasRecordID()) {
				return;
			}

			// If the record has to be re-loaded (that can happen if unused records were deleted since the record was
			// made editable), warning messages are not reported to the user.  The assumption is those messages were
			// already seen the first time the record was added to the scenario, before it was replaced by the editable
			// duplicate.  Also if the record requests automatic replication that is ignored here.  That replication
			// was done when the record was first added so the user had to deliberately revert that before making the
			// source editable.

			if (null != theSource.userRecordID) {

				newSource = scenario.study.findSharedSource(theSource.userRecordID);
				if (null == newSource) {
					newSource = SourceEditData.findUserRecord(getDbID(), scenario.study, theSource.userRecordID,
						errorReporter);
					if (null == newSource) {
						if (!errorReporter.hasErrors()) {
							errorReporter.reportError("The original user record does not exist");
						}
						return;
					}
					newSource.copyTransientAttributes(theSource);
				}

			} else {

				newSource = scenario.study.findSharedSource(theSource.extDbKey, theSource.extRecordID);
				if (null == newSource) {
					ExtDb theExtDb = ExtDb.getExtDb(getDbID(), theSource.extDbKey, errorReporter);
					if (null == theExtDb) {
						return;
					}
					if (theExtDb.isGeneric()) {
						newSource = SourceEditData.findImportRecord(theExtDb, theSource.recordType, scenario.study,
							theSource.extRecordID, errorReporter);
						if (null == newSource) {
							if (!errorReporter.hasErrors()) {
								errorReporter.reportError("The original station data record does not exist");
							}
							return;
						}
					} else {
						ExtDbRecord theRecord = null;
						switch (theSource.recordType) {
							case Source.RECORD_TYPE_TV: {
								theRecord = ExtDbRecordTV.findRecordTV(theExtDb, theSource.extRecordID, errorReporter);
								break;
							}
							case Source.RECORD_TYPE_FM: {
								theRecord = ExtDbRecordFM.findRecordFM(theExtDb, theSource.extRecordID, errorReporter);
								break;
							}
						}
						if (null == theRecord) {
							if (!errorReporter.hasErrors()) {
								errorReporter.reportError("The original station data record does not exist");
							}
							return;
						}
						newSource = SourceEditData.makeSource(theRecord, scenario.study, true, errorReporter);
					}
					newSource.copyTransientAttributes(theSource);
				}
			}

			// In an OET74 or TV6FM study if the entry is permanent check the channel on the original record, if that
			// is not the same as the current record it can't be reverted.  See comments in updateControls().

			if (theItem.isPermanent &&
					((Study.STUDY_TYPE_TV_OET74 == studyType) || (Study.STUDY_TYPE_TV6_FM == studyType)) &&
					(((SourceEditDataTV)newSource).channel != ((SourceEditDataTV)theSource).channel)) {
				errorReporter.reportWarning("The record cannot be reverted, the original has a different channel");
				return;
			}
		}

		// Final safety check, make sure the new source itself is not already in the scenario.  There are several ways
		// that could happen, e.g. if records have been added with MX checks disabled.

		if (scenario.sourceData.indexOfSourceKey(newSource.key) >= 0) {
			errorReporter.reportWarning("The record cannot be reverted, the original is already in the scenario");
			return;
		}

		sourceModel.set(rowIndex, newSource);
		setDidEdit();

		updateControls();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Copy selected record(s) to a different scenario in the study.  Prompt for selection of the scenario, the list
	// will not include scenarios that cannot be edited, or this editor's scenario of course.  Normally this will not
	// copy records that are MX to any record already in the other scenario, but the MX check can be disabled, or can
	// also be restricted to just consider facility ID.  Identical records (same source key) are never copied.

	private void doCopyToScenario() {

		if (0 == sourceTable.getSelectedRowCount()) {
			return;
		}

		String title = "Copy To Scenario";
		errorReporter.setTitle(title);

		ArrayList<KeyedRecord> scenarioList = new ArrayList<KeyedRecord>();
		int rowIndex;
		ScenarioEditData theScenario;

		for (rowIndex = 0; rowIndex < scenario.study.scenarioData.getRowCount(); rowIndex++) {
			theScenario = scenario.study.scenarioData.get(rowIndex);
			if (scenario.key.equals(theScenario.key)) {
				continue;
			}
			if ((Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == theScenario.scenarioType) ||
					(Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == theScenario.scenarioType)) {
				continue;
			}
			scenarioList.add(new KeyedRecord(rowIndex, theScenario.name));
		}
		if (scenarioList.isEmpty()) {
			errorReporter.reportWarning("No other scenarios are available for the copy");
			return;
		}

		class PromptDialog extends AppDialog {

			private KeyedRecordMenu scenarioMenu;
			private JCheckBox disableMXCheckBox;
			private JCheckBox mxFacilityIDOnlyCheckBox;

			private boolean canceled;

			private PromptDialog(ArrayList<KeyedRecord> theList) {

				super(outerThis, "Copy To Scenario", Dialog.ModalityType.APPLICATION_MODAL);

				scenarioMenu = new KeyedRecordMenu(theList);

				JPanel menuP = new JPanel(new FlowLayout(FlowLayout.LEFT));
				menuP.setBorder(BorderFactory.createTitledBorder("Copy To"));
				menuP.add(scenarioMenu);

				disableMXCheckBox = new JCheckBox("Disable all MX checks");
				JPanel disP = new JPanel(new FlowLayout(FlowLayout.LEFT));
				disP.add(disableMXCheckBox);

				mxFacilityIDOnlyCheckBox = new JCheckBox("Use only facility ID for MX check");
				JPanel facidP = new JPanel(new FlowLayout(FlowLayout.LEFT));
				facidP.add(mxFacilityIDOnlyCheckBox);

				// MX checks are never applied in a TV interference-check study, see below.

				if (Study.STUDY_TYPE_TV_IX == studyType) {
					disableMXCheckBox.setSelected(true);
					AppController.setComponentEnabled(disableMXCheckBox, false);
					AppController.setComponentEnabled(mxFacilityIDOnlyCheckBox, false);
				} else {
					disableMXCheckBox.addActionListener(new ActionListener() {
						public void actionPerformed(ActionEvent theEvent) {
							if (blockActions()) {
								if (disableMXCheckBox.isSelected()) {
									mxFacilityIDOnlyCheckBox.setSelected(false);
									AppController.setComponentEnabled(mxFacilityIDOnlyCheckBox, false);
								} else {
									AppController.setComponentEnabled(mxFacilityIDOnlyCheckBox, true);
								}
								blockActionsEnd();
							}
						}
					});
				}

				JButton okButton = new JButton("OK");
				okButton.setFocusable(false);
				okButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doOK();
					}
				});

				JButton canButton = new JButton("Cancel");
				canButton.setFocusable(false);
				canButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doCancel();
					}
				});

				Box mainB = Box.createVerticalBox();
				mainB.add(menuP);
				mainB.add(disP);
				mainB.add(facidP);
				add(mainB, BorderLayout.CENTER);

				JPanel butP = new JPanel(new FlowLayout(FlowLayout.RIGHT));
				butP.add(canButton);
				butP.add(okButton);
				add(butP, BorderLayout.SOUTH);

				getRootPane().setDefaultButton(okButton);

				pack();
				setMinimumSize(getSize());
				setLocationRelativeTo(outerThis);
			}

			private void doOK() {

				AppController.hideWindow(this);
			}

			private void doCancel() {

				canceled = true;
				AppController.hideWindow(this);
			}

			public void windowWillOpen() {

				blockActionsClear();
			}

			public boolean windowShouldClose() {

				canceled = true;
				return true;
			}

			public void windowWillClose() {

				blockActionsSet();
			}
		};

		PromptDialog theDialog = new PromptDialog(scenarioList);
		AppController.showWindow(theDialog);
		if (theDialog.canceled) {
			return;
		}

		ScenarioEditData otherScenario = null;
		boolean disableMX = false, mxFacilityIDOnly = false;

		rowIndex = theDialog.scenarioMenu.getSelectedKey();
		if (rowIndex >= 0) {
			otherScenario = scenario.study.scenarioData.get(rowIndex);
		} else {
			errorReporter.reportError("Invalid scenario selection");
			return;
		}

		if (Study.STUDY_TYPE_TV_IX == studyType) {
			disableMX = true;
		} else {
			disableMX = theDialog.disableMXCheckBox.isSelected();
			if (!disableMX) {
				mxFacilityIDOnly = theDialog.mxFacilityIDOnlyCheckBox.isSelected();
			}
			if (disableMX) {
				errorReporter.reportWarning(
					"MX checks are disabled, copied records will not be marked as undesireds\n" +
					"Records that should be undesireds must be identified manually");
			}
		}

		// Process the selected sources, make duplications and replications as needed for TV, do identical-record and
		// MX checks, and build a list of items to be copied.  Nothing is actually copied until all checks are done in
		// case errors occur during the checks.

		class CopyItem {
			SourceEditData source;
			SourceEditData originalSource;
			boolean isDesired;
			boolean isUndesired;
		}

		ArrayList<CopyItem> copyItems = new ArrayList<CopyItem>();

		ArrayList<SourceEditData> otherSources = otherScenario.sourceData.getSources();

		double coChanMX = scenario.study.getCoChannelMxDistance();
		double kmPerDeg = scenario.study.getKilometersPerDegree();

		int[] selRows = sourceTable.getSelectedRows();

		Scenario.SourceListItem theItem;
		CopyItem copyItem;
		SourceEditor theEditor;
		SourceEditDataTV theSourceTV, origSourceTV;

		for (int i = 0; i < selRows.length; i++) {

			rowIndex = sourceTable.convertRowIndexToModel(selRows[i]);

			copyItem = new CopyItem();
			copyItem.source = sourceModel.getSource(rowIndex);

			// Set flags.  In a TV interference-check, OET-74, or channel 6 vs. FM study, the editable scenarios can
			// only have a single desired TV source which had to be picked when the scenario was created and cannot be
			// removed or un-flagged later.  In those cases, the copied records never have the desired flag set.  Also
			// wireless records can never be desireds in any case.  In an interference-check study, the undesired flag
			// must be set on new records since they haven't yet been checked for actual interference and so must be
			// included in the interference-scenario iteration.  In other studies if MX checks are disabled, never set
			// the undesired flag because the user must manually determine which undesireds to activate.  Otherwise,
			// copy the flags from the current scenario.

			theItem = sourceModel.get(rowIndex);
			if ((Study.STUDY_TYPE_TV_IX == studyType) || (Study.STUDY_TYPE_TV_OET74 == studyType) ||
					((Study.STUDY_TYPE_TV6_FM == studyType) && (Source.RECORD_TYPE_TV == copyItem.source.recordType)) ||
					(Source.RECORD_TYPE_WL == copyItem.source.recordType)) {
				copyItem.isDesired = false;
			} else {
				copyItem.isDesired = theItem.isDesired;
			}
			if (Study.STUDY_TYPE_TV_IX == studyType) {
				copyItem.isUndesired = true;
			} else {
				if (disableMX) {
					copyItem.isUndesired = false;
				} else {
					copyItem.isUndesired = theItem.isUndesired;
				}
			}

			// If an editable source has an active editor, it must be closed first to commit any edits.  An editable
			// source is duplicated so it will be unique in the new scenario, but still needs to be checked for MX.

			if (!copyItem.source.isLocked) {

				theEditor = sourceEditors.get(copyItem.source.key);
				if (null != theEditor) {
					if (theEditor.isVisible()) {
						AppController.beep();
						errorReporter.reportWarning("Please commit or cancel record edits then re-try the operation");
						theEditor.toFront();
						return;
					} else {
						sourceEditors.remove(copyItem.source.key);
					}
				}

				copyItem.source = copyItem.source.deriveSource(false, errorReporter);
			}

			switch (copyItem.source.recordType) {

				case Source.RECORD_TYPE_TV: {

					theSourceTV = (SourceEditDataTV)copyItem.source;

					// If the record is a TV replication and the original is an editable record, must duplicate that
					// editable original and create a new replication from that duplicate.  Although replications are
					// always locked, the problem is that editable records created by UI are scenario-private.  If
					// a replication of an editable record were copied then later reverted, the editable record could
					// end up being shared by the scenarios.  If the original does not exist that isn't an error here,
					// in that case it's safe to just copy the replication.

					if (theSourceTV.isReplication()) {
						origSourceTV = (SourceEditDataTV)scenario.study.getSource(theSourceTV.originalSourceKey);
						if ((null != origSourceTV) && !origSourceTV.isLocked) {
							origSourceTV = (SourceEditDataTV)origSourceTV.deriveSource(false, errorReporter);
							if (null == origSourceTV) {
								return;
							}
							theSourceTV = origSourceTV.replicate(theSourceTV.channel, errorReporter);
							if (null == theSourceTV) {
								return;
							}
							copyItem.source = theSourceTV;
							copyItem.originalSource = origSourceTV;
						}
					}

					// Check for identical and MX records in the other scenario.

					for (SourceEditData otherSource : otherSources) {
						if (Source.RECORD_TYPE_TV == otherSource.recordType) {
							if (copyItem.source.key.equals(otherSource.key)) {
								copyItem = null;
								break;
							}
							if (!disableMX && ExtDbRecordTV.areRecordsMX(theSourceTV, (SourceEditDataTV)otherSource,
									mxFacilityIDOnly, coChanMX, kmPerDeg)) {
								copyItem = null;
								break;
							}
						}
					}

					break;
				}

				case Source.RECORD_TYPE_FM: {

					for (SourceEditData otherSource : otherSources) {
						if (Source.RECORD_TYPE_FM == otherSource.recordType) {
							if (copyItem.source.key.equals(otherSource.key)) {
								copyItem = null;
								break;
							}
							if (!disableMX && ExtDbRecordFM.areRecordsMX((SourceEditDataFM)copyItem.source,
									(SourceEditDataFM)otherSource, mxFacilityIDOnly, coChanMX, kmPerDeg)) {
								copyItem = null;
								break;
							}
						}
					}

					break;
				}

				case Source.RECORD_TYPE_WL: {

					// Wireless records have no MX concept so just check for identicals.

					for (SourceEditData otherSource : otherSources) {
						if (Source.RECORD_TYPE_WL == otherSource.recordType) {
							if (copyItem.source.key.equals(otherSource.key)) {
								copyItem = null;
								break;
							}
						}
					}

					break;
				}
			}

			if (null != copyItem) {
				copyItems.add(copyItem);
			}
		}

		// Do the copies if there are any.

		if (!copyItems.isEmpty()) {

			for (CopyItem anItem : copyItems) {
				if (null != anItem.originalSource) {
					otherScenario.study.addOrReplaceSource(anItem.originalSource);
				}
				otherScenario.sourceData.addOrReplace(anItem.source, anItem.isDesired, anItem.isUndesired);
			}
			setDidEdit();

			// If the other scenario has an open editor, tell it to refresh it's UI.

			ScenarioEditor otherEditor = (ScenarioEditor)parent.getEditorFor(otherScenario);
			if (null != otherEditor) {
				otherEditor.sourceModel.dataWasChanged();
			}
		}

		// Report number copied, and also number excluded if any.

		int copied = copyItems.size(), notCopied = selRows.length - copied;
		if (notCopied > 0) {
			errorReporter.reportMessage(String.valueOf(copied) + " records copied,\n" + String.valueOf(notCopied) +
				" MX records not copied");
		} else {
			errorReporter.reportMessage(String.valueOf(copied) + " records copied");
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Send up to three selected sources to the SourceCompare class, see there for details.

	private void doCompareSource() {

		if (0 == sourceTable.getSelectedRowCount()) {
			return;
		}

		errorReporter.setTitle("Compare Records");

		int[] selRows = sourceTable.getSelectedRows();

		for (int i = 0; (i < selRows.length) && (i < 3); i++) {
			if (!SourceCompare.compare(sourceModel.getSource(sourceTable.convertRowIndexToModel(selRows[i])),
					errorReporter)) {
				return;
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Export selected record(s) to a CSV file.  If any are editable, check for open editors, also all of the records
	// must be the same record type; text files cannot have mixed types.

	private void doExportSources() {

		if (sourceTable.getSelectedRowCount() < 1) {
			return;
		}

		ArrayList<SourceEditData> theSources = new ArrayList<SourceEditData>();
		int theRecordType = 0;

		SourceEditData theSource;
		SourceEditor theEditor;

		for (int row : sourceTable.getSelectedRows()) {

			theSource = sourceModel.getSource(sourceTable.convertRowIndexToModel(row));

			if (!theSource.isLocked) {
				theEditor = sourceEditors.get(theSource.key);
				if (null != theEditor) {
					if (theEditor.isVisible()) {
						AppController.beep();
						theEditor.toFront();
						return;
					} else {
						sourceEditors.remove(theSource.key);
					}
				}
			}

			if (0 == theRecordType) {
				theRecordType = theSource.recordType;
			} else {
				if (theRecordType != theSource.recordType) {
					errorReporter.reportWarning("All selected records must be the same type");
					return;
				}
			}

			theSources.add(theSource);
		}

		String title = "Export " + Source.getRecordTypeName(theRecordType) + " Records";
		errorReporter.setTitle(title);

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.SAVE_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new FileNameExtensionFilter("CSV (*.csv)", "csv"));
		chooser.setAcceptAllFileFilterUsed(false);

		File theFile = null;
		String theName;
		do {
			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Export")) {
				return;
			}
			theFile = chooser.getSelectedFile();
			theName = theFile.getName();
			if (!theName.toLowerCase().endsWith(".csv")) {
				theFile = new File(theFile.getAbsolutePath() + ".csv");
			}
			if (theFile.exists()) {
				AppController.beep();
				if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
						"The file exists, do you want to replace it?", title, JOptionPane.YES_NO_OPTION,
						JOptionPane.WARNING_MESSAGE)) {
					theFile = null;
				}
			}
		} while (null == theFile);

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

		BufferedWriter writer = null;
		try {
			writer = new BufferedWriter(new FileWriter(theFile));
		} catch (IOException ie) {
			errorReporter.reportError("Could not open the file:\n" + ie.getMessage());
			return;
		}

		try {
			SourceEditData.writeToCSV(writer, theRecordType, theSources, errorReporter);
		} catch (IOException ie) {
			errorReporter.reportError(ie.toString());
		}

		try {writer.close();} catch (IOException ie) {}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save selected record as a user record, the record must be editable and not have an open editor.  Optionally
	// the new user record may replace the selected record.  Also prompt for comment.

	private void doSaveAsUserRecord() {

		if (sourceTable.getSelectedRowCount() != 1) {
			return;
		}

		int rowIndex = sourceTable.convertRowIndexToModel(sourceTable.getSelectedRow());
		SourceEditData theSource = sourceModel.getSource(rowIndex);
		if (theSource.isLocked) {
			return;
		}
		SourceEditor theEditor = sourceEditors.get(theSource.key);
		if (null != theEditor) {
			if (theEditor.isVisible()) {
				theEditor.toFront();
				return;
			} else {
				sourceEditors.remove(theSource.key);
			}
		}

		String title = "Save As User Record";
		errorReporter.setTitle(title);

		TextInputDialog theDialog = new TextInputDialog(outerThis, title, "Comment");
		JCheckBox replaceRecordCheckBox = new JCheckBox("Replace record in scenario");
		theDialog.add(replaceRecordCheckBox, BorderLayout.NORTH);
		theDialog.pack();

		AppController.showWindow(theDialog);
		if (theDialog.canceled) {
			return;
		}

		SourceEditData newSource = theSource.saveAsUserRecord(scenario.study, errorReporter);
		if (null == newSource) {
			return;
		}

		String theComment = theDialog.getInput();
		if (theComment.length() > 0) {
			SourceEditData.setSourceComment(newSource, theComment);
		} else {
			theComment = null;
		}

		if (replaceRecordCheckBox.isSelected()) {
			sourceModel.set(rowIndex, newSource);
			setDidEdit();
		} else {
			errorReporter.reportMessage("Saved as user record " + String.valueOf(newSource.userRecordID));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Change desired/undesired flags on all selected rows.

	private void doSetFlags(boolean doDes, boolean newFlag) {

		if ((Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == scenario.scenarioType) ||
				(Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType)) {
			return;
		}

		if (0 == sourceTable.getSelectedRowCount()) {
			return;
		}

		for (int row : sourceTable.getSelectedRows()) {
			if (doDes) {
				if (sourceModel.setIsDesired(sourceTable.convertRowIndexToModel(row), newFlag)) {
					setDidEdit();
				}
			} else {
				if (sourceModel.setIsUndesired(sourceTable.convertRowIndexToModel(row), newFlag)) {
					setDidEdit();
				}
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// As above, but toggle the existing flag state.

	private void doToggleFlags(boolean doDes) {

		if ((Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == scenario.scenarioType) ||
				(Scenario.SCENARIO_TYPE_TVIX_INTERFERENCE == scenario.scenarioType)) {
			return;
		}

		if (0 == sourceTable.getSelectedRowCount()) {
			return;
		}

		int rowIndex;
		Scenario.SourceListItem theItem;
		for (int row : sourceTable.getSelectedRows()) {
			rowIndex = sourceTable.convertRowIndexToModel(row);
			theItem = sourceModel.get(rowIndex);
			if (doDes) {
				if (sourceModel.setIsDesired(rowIndex, !theItem.isDesired)) {
					setDidEdit();
				}
			} else {
				if (sourceModel.setIsUndesired(rowIndex, !theItem.isUndesired)) {
					setDidEdit();
				}
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Called by dependent dialogs when user commits edits or requests action.  First process request from the add-
	// multiple-records dialog, doing a search to add all matching records for user-entered criteria.

	public boolean applyEditsFrom(AppEditor theEditor) {

		if (theEditor instanceof ExtDbSearchDialog) {

			if ((ExtDbSearchDialog)theEditor != addSourcesDialog) {
				return false;
			}

			return runExtDbSearch(addSourcesDialog.extDb, addSourcesDialog.useBaseline, addSourcesDialog.search,
				addSourcesDialog.getErrorReporter());
		}

		// If the single-station add dialog sends an apply a single record has been selected in the dialog, retrieve
		// it and process the selection and options input.

		if (theEditor instanceof RecordFind) {

			if ((RecordFind)theEditor != addSourceFinder) {
				return false;
			}

			ErrorReporter theReporter = addSourceFinder.getErrorReporter();
			OptionsPanel theOptions = (OptionsPanel)(addSourceFinder.getAccessoryPanel());

			SourceEditData theSource = addSourceFinder.source;
			ExtDbRecord theRecord = addSourceFinder.record;

			int theType = 0;
			if (null != theSource) {
				theType = theSource.recordType;
			} else {
				if (null != theRecord) {
					theType = theRecord.recordType;
				}
			}

			// Do a paranoia check of the record type; the dialog should only allow compatible types to be chosen.

			if (!Study.isRecordTypeAllowed(studyType, theType)) {
				theReporter.reportError("That record type is not allowed here");
				return false;
			}

			// If a TV record is selected, check the channel against the legal range for the study.  The search dialog
			// does not restrict the channel range for search because an out-of-range record may be replicated or
			// changed to an in-range channel.  If the channel will change check the final channel not the original.
			// This supports automatic replication, see ExtDbRecordTV for details.

			boolean replicate = false, changeChannel = false;
			int studyChannel = 0;

			if (Source.RECORD_TYPE_TV == theType) {

				replicate = theOptions.replicate;
				changeChannel = theOptions.changeChannel;

				if (replicate || changeChannel) {
					studyChannel = theOptions.studyChannel;
				} else {
					if (null != theSource) {
						studyChannel = ((SourceEditDataTV)theSource).channel;
					} else {
						studyChannel = ((ExtDbRecordTV)theRecord).replicateToChannel;
						if (0 == studyChannel) {
							studyChannel = ((ExtDbRecordTV)theRecord).channel;
						} else {
							replicate = true;
						}
					}
				}

				int minChannel = scenario.study.getMinimumChannel(), maxChannel = scenario.study.getMaximumChannel();

				if ((studyChannel < minChannel) || (studyChannel > maxChannel)) {
					theReporter.reportWarning("A TV record must have a channel in the range " + minChannel + " to " +
						maxChannel);
					return false;
				}
			}

			// Prepare to create the source.

			SourceEditData newSource = null;
			SourceEditDataTV originalSource = null;
			boolean wasLocked = true, isLocked = true;

			theReporter.clearErrors();
			theReporter.clearMessages();

			// A source object from the find may be a new source never saved, or one loaded from a generic data set or
			// the user record table.  If either of the latter it is shareable if it will be locked so check for an
			// existing shareable source.  If not found, derive a new source to set the study and unlock if needed.

			if (null != theSource) {

				wasLocked = theSource.isLocked;
				isLocked = wasLocked && theOptions.isLocked;

				if (isLocked) {
					if ((null != theSource.extDbKey) && (null != theSource.extRecordID)) {
						newSource = scenario.study.findSharedSource(theSource.extDbKey, theSource.extRecordID);
					} else {
						if (null != theSource.userRecordID) {
							newSource = scenario.study.findSharedSource(theSource.userRecordID);
						}
					}
				}
				if (null == newSource) {
					newSource = theSource.deriveSource(scenario.study, isLocked, theReporter);
				}

				// Do replication or channel change as needed, again if the new source is locked check for existing
				// shared sources.  Replication sources are always locked however the original may not be and the
				// replication is only shareable if it's original is locked.  To change channel, an unlocked source
				// will be derived if needed otherwise the selected source is modified directly.

				if (null != newSource) {
					if (replicate) {
						originalSource = (SourceEditDataTV)newSource;
						newSource = null;
						if (isLocked) {
							if ((null != originalSource.extDbKey) && (null != originalSource.extRecordID)) {
								newSource = scenario.study.findSharedReplicationSource(originalSource.extDbKey,
									originalSource.extRecordID, studyChannel);
							} else {
								if (null != originalSource.userRecordID) {
									newSource = scenario.study.findSharedReplicationSource(originalSource.userRecordID,
										studyChannel);
								}
							}
						}
						if (null == newSource) {
							newSource = originalSource.replicate(studyChannel, theReporter);
						} else {
							originalSource = null;
						}
					} else {
						if (changeChannel) {
							if (isLocked) {
								originalSource = (SourceEditDataTV)newSource;
								newSource = originalSource.deriveSource(scenario.study, false, theReporter);
							}
							if (null != newSource) {
								((SourceEditDataTV)newSource).channel = studyChannel;
							}
						}
					}
				}

			} else {

				// For an external record, a source must be created.  It is shareable if it will be non-editable.

				if (null != theRecord) {

					isLocked = theOptions.isLocked;

					if (isLocked) {
						newSource = scenario.study.findSharedSource(theRecord.extDb.key, theRecord.extRecordID);
					}
					if (null == newSource) {
						newSource = SourceEditData.makeSource(theRecord, scenario.study, isLocked, theReporter);
					}
					if (null != newSource) {
						if (replicate) {
							originalSource = (SourceEditDataTV)newSource;
							newSource = null;
							if (isLocked) {
								newSource = scenario.study.findSharedReplicationSource(originalSource.extDbKey,
									originalSource.extRecordID, studyChannel);
							}
							if (null == newSource) {
								newSource = originalSource.replicate(studyChannel, theReporter);
							} else {
								originalSource = null;
							}
						} else {
							if (changeChannel) {
								if (isLocked) {
									originalSource = (SourceEditDataTV)newSource;
									newSource = originalSource.deriveSource(scenario.study, false, theReporter);
								}
								if (null != newSource) {
									((SourceEditDataTV)newSource).channel = studyChannel;
								}
							}
						}
					}
				}
			}

			// If a source object was not created successfully, fail.  In some cases makeSource() will fail to create a
			// source but only log a message, report the message as an error if needed.

			if (null == newSource) {
				if (!theReporter.hasErrors()) {
					if (theReporter.hasMessages()) {
						theReporter.reportError(theReporter.getMessages());
					} else {
						theReporter.reportError("The record could not be added due to an unknown error");
					}
				}
				return false;
			}

			// Check to be sure an identical source (matching user record ID or external data key and record ID) is not
			// already in the scenario, fail in that case.  Also check if the new source appears to be MX to any source
			// already in the scenario.  If MX the new source probably should not be added since the study engine does
			// not apply any MX logic, all active undesireds potentially interfere with all desireds based only on
			// matching an interference rule.  That means inappropriate self-interference could easily be predicted if
			// MX records are included.  However the user can manually manage MX records in a scenario by selecting
			// which are active as undesireds; so an MX record can still be added, but the addition must be confirmed.
			// Also this now requires a channel match for the identical-record test, to allow a record and one or more
			// replications of that same record to appear in the same scenario.  In an IX check study the scenarios
			// being edited are expected to contain many MX undesireds, because they are used to build interference
			// scenarios by iterating through non-MX undesired combinations.  In that case the MX test only looks at
			// desired records, and if MX is found the warning message is different.

			boolean isMX = false;
			double coChanMX = scenario.study.getCoChannelMxDistance();
			double kmPerDeg = scenario.study.getKilometersPerDegree();

			String newChan = newSource.getSortChannel();

			SourceEditData otherSource;
			Scenario.SourceListItem theItem;
			for (int rowIndex = 0; rowIndex < scenario.sourceData.getRowCount(); rowIndex++) {
				otherSource = scenario.sourceData.getSource(rowIndex);
				if ((((null != newSource.userRecordID) && (newSource.userRecordID.equals(otherSource.userRecordID))) ||
						((null != newSource.extDbKey) && newSource.extDbKey.equals(otherSource.extDbKey) &&
						(null != newSource.extRecordID) && newSource.extRecordID.equals(otherSource.extRecordID))) &&
						newChan.equals(otherSource.getSortChannel())) {
					theReporter.reportWarning("That record is already in the scenario");
					return false;
				}
				if (!isMX) {
					if (newSource.recordType == otherSource.recordType) {
						if ((Source.RECORD_TYPE_TV == newSource.recordType) &&
								ExtDbRecordTV.areRecordsMX((SourceEditDataTV)newSource, (SourceEditDataTV)otherSource,
									false, coChanMX, kmPerDeg)) {
							if (Study.STUDY_TYPE_TV_IX == studyType) {
								theItem = scenario.sourceData.get(rowIndex);
								if (theItem.isDesired) {
									isMX = true;
								}
							} else {
								isMX = true;
							}
						}
						if ((Source.RECORD_TYPE_FM == newSource.recordType) &&
								ExtDbRecordFM.areRecordsMX((SourceEditDataFM)newSource, (SourceEditDataFM)otherSource,
									false, coChanMX, kmPerDeg)) {
							isMX = true;
						}
					}
				}
			}

			// Extra test for a TV IX check study, records that are MX to the proposal record being evaluated by the
			// study should not be added, because the proposal record will always be added to the interference
			// scenarios automatically by the auto-build code later.

			if (!isMX && (Study.STUDY_TYPE_TV_IX == studyType)) {
				ScenarioEditData propScenario = scenario.study.scenarioData.get(0);
				if (Scenario.SCENARIO_TYPE_TVIX_PROPOSAL == propScenario.scenarioType) {
					SourceEditDataTV propSource = null;
					for (SourceEditData aSource : propScenario.sourceData.getSources(Source.RECORD_TYPE_TV)) {
						if (null != aSource.getAttribute(Source.ATTR_IS_PROPOSAL)) {
							propSource = (SourceEditDataTV)aSource;
							break;
						}
					}
					if ((null != propSource) && ExtDbRecordTV.areRecordsMX((SourceEditDataTV)newSource, propSource,
							true, 0., 0.)) {
						isMX = true;
					}
				}
			}

			if (isMX) {
				AppController.beep();
				String[] opts = {"No", "Yes"};
				if (Study.STUDY_TYPE_TV_IX == studyType) {
					if (1 != JOptionPane.showOptionDialog(addSourceFinder,
							"The new record appears to be mutually-exclusive with the desired record in\n" +
							"this scenario, or with the proposal record for the study.  All records in\n" +
							"the scenario will be included as undesireds in the interference scenarios,\n" +
							"regardless of the undesired flag.  Are you sure you want to add this record?",
							addSourceFinder.getBaseTitle(), 0, JOptionPane.WARNING_MESSAGE, null, opts, opts[0])) {
						return false;
					}
				} else {
					if (1 != JOptionPane.showOptionDialog(addSourceFinder,
							"The new record appears to be mutually-exclusive with one already in the\n" +
							"scenario.  The study will always consider all active undesired records as\n" +
							"potential interference sources to all desireds, regardless of apparent MX\n" +
							"relationships.  Are you sure you want to add this record?",
							addSourceFinder.getBaseTitle(), 0, JOptionPane.WARNING_MESSAGE, null, opts, opts[0])) {
						return false;
					}
					theReporter.reportWarning("The new record will not be marked as an undesired");
				}
			}

			// Add the new source.  If this was a replication add the original first.  For general-purpose study types
			// the new source is always a desired, for most others it never is (for those types there is always one
			// desired station that was added to the scenario when it was created, with the permanent flag set), except
			// in a FM and TV channel 6 study where FM records may also be desireds.  For any type, the new station is
			// an undesired, unless an MX relationship was detected, except in a TV IX check study where MX is ignored.
			// Also if the record is a sharing guest, it is never flagged as either desired or undesired.  Since those
			// are effectively MX to the host and other guests but that is deliberately ignored by the MX test, the
			// user must manually determine if a guest should be active.

			theReporter.showMessages();

			if (null != originalSource) {
				scenario.study.addOrReplaceSource(originalSource);
				setDidEdit();
			}
			boolean isDes = ((Study.STUDY_TYPE_TV == studyType) || (Study.STUDY_TYPE_FM == studyType) ||
				((Study.STUDY_TYPE_TV6_FM == studyType) && (Source.RECORD_TYPE_FM == newSource.recordType)));
			boolean isUnd = (!isMX || (Study.STUDY_TYPE_TV_IX == studyType));
			if (null != newSource.getAttribute(Source.ATTR_IS_SHARING_GUEST)) {
				isDes = false;
				isUnd = false;
			}
			int rowIndex = sourceModel.addOrReplace(newSource, isDes, isUnd);
			if (rowIndex < 0) {
				return false;
			}
			setDidEdit();

			rowIndex = sourceTable.convertRowIndexToView(rowIndex);
			sourceTable.setRowSelectionInterval(rowIndex, rowIndex);
			sourceTable.scrollRectToVisible(sourceTable.getCellRect(rowIndex, 0, true));

			// If the new source is editable but the selected record was not, immediately show an edit dialog.  If the
			// record selected in the find dialog was editable it must have been a newly-created or derived source
			// meaning the user has just seen it in the source editor off the find dialog, no need to show that again.

			if (!theOptions.isLocked && wasLocked) {
				doOpenSource();
			}

			return true;
		}

		// An apply from a source editor means editing is complete on an editable record, update the model.  But be
		// paranoid and make sure the source really is in the model and is editable first.

		if (theEditor instanceof SourceEditor) {

			SourceEditor sourceEditor = (SourceEditor)theEditor;
			Integer theKey = sourceEditor.getOriginalSourceKey();
			if (sourceEditor != sourceEditors.get(theKey)) {
				return false;
			}

			int rowIndex = scenario.sourceData.indexOfSourceKey(theKey);
			if (rowIndex < 0) {
				return false;
			}
			SourceEditData oldSource = scenario.sourceData.getSource(rowIndex);
			if (oldSource.isLocked) {
				return false;
			}

			SourceEditData newSource = sourceEditor.getSource();
			sourceModel.setUnfiltered(rowIndex, newSource);
			if (!newSource.equals(oldSource) || newSource.isDataChanged()) {
				setDidEdit();
			}

			return true;
		}

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Called by dependent dialogs when closing.

	public void editorClosing(AppEditor theEditor) {

		if (theEditor instanceof ExtDbSearchDialog) {
			if ((ExtDbSearchDialog)theEditor == addSourcesDialog) {
				addSourcesDialog = null;
				updateSearchMenu();
			}
			return;
		}

		if (theEditor instanceof RecordFind) {
			if ((RecordFind)theEditor == addSourceFinder) {
				addSourceFinder = null;
			}
			return;
		}

		if (theEditor instanceof SourceEditor) {
			sourceEditors.remove(((SourceEditor)theEditor).getOriginalSourceKey(), (SourceEditor)theEditor);
			updateControls();
			return;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Force the editor to go away without preserving or saving any state; cancel showing dialogs and hide.

	public boolean closeWithoutSave() {

		if (!sourceEditors.isEmpty()) {
			for (SourceEditor theEditor : new ArrayList<SourceEditor>(sourceEditors.values())) {
				if (theEditor.isVisible()) {
					if (!theEditor.cancel()) {
						AppController.beep();
						theEditor.toFront();
						return false;
					}
				} else {
					sourceEditors.remove(theEditor.getOriginalSourceKey());
				}
			}
		}

		if (null != addSourceFinder) {
			if (addSourceFinder.isVisible() && !addSourceFinder.closeWithoutSave()) {
				AppController.beep();
				addSourceFinder.toFront();
				return false;
			}
			addSourceFinder = null;
		}

		if (null != addSourcesDialog) {
			if (addSourcesDialog.isVisible() && !addSourcesDialog.cancel()) {
				AppController.beep();
				addSourcesDialog.toFront();
				return false;
			}
			addSourcesDialog = null;
		}

		AppController.hideWindow(this);

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillOpen() {

		DbController.restoreColumnWidths(getDbID(), getKeyTitle(), sourceTable);

		updateSearchMenu();

		blockActionsClear();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// When closing, the dialogs for adding sources are just canceled, as are any source editors for non-editable
	// records.  However source editors for editable records will block the close and must be closed manually first.

	public boolean windowShouldClose() {

		if (!isVisible()) {
			return true;
		}

		if (!commitCurrentField()) {
			toFront();
			return false;
		}

		if (!sourceEditors.isEmpty()) {
			for (SourceEditor theEditor : new ArrayList<SourceEditor>(sourceEditors.values())) {
				if (theEditor.isVisible()) {
					if (theEditor.isEditing() || !theEditor.cancel()) {
						AppController.beep();
						theEditor.toFront();
						return false;
					}
				} else {
					sourceEditors.remove(theEditor.getOriginalSourceKey());
				}
			}
		}

		if (null != addSourceFinder) {
			if (addSourceFinder.isVisible() && !addSourceFinder.closeIfPossible()) {
				AppController.beep();
				addSourceFinder.toFront();
				return false;
			}
			addSourceFinder = null;
		}

		if (null != addSourcesDialog) {
			if (addSourcesDialog.isVisible() && !addSourcesDialog.cancel()) {
				AppController.beep();
				addSourcesDialog.toFront();
				return false;
			}
			addSourcesDialog = null;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillClose() {

		if (!isVisible()) {
			return;
		}

		DbController.saveColumnWidths(getDbID(), getKeyTitle(), sourceTable);

		blockActionsSet();
		parent.editorClosing(this);
	}
}
