//
//  PatternEditor.java
//  TVStudy
//
//  Copyright (c) 2012-2020 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui.editor;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;
import gov.fcc.tvstudy.core.geo.*;
import gov.fcc.tvstudy.gui.*;

import java.util.*;
import java.util.regex.*;
import java.sql.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.table.*;
import javax.swing.filechooser.*;


//=====================================================================================================================
// Pattern editor dialog UI for all types of pattern data, also this can at as a pattern search tool for loading
// pattern data from external data sets, and provides preview of the pattern data and plot while searching.  When used
// as an editor from a parent source editor, antenna orientation is used only in horizontal antenna mode, it is set and
// read directly by the parent.

public class PatternEditor extends AppDialog implements ExtDbListener {

	private AntPattern pattern;
	private int patternType;

	public double antennaOrientation;

	// UI fields.

	private JTextField patternNameField;

	private JTextField antennaOrientationField;
	private JCheckBox rotateHorizontalPlotCheckBox;

	private JTextField antennaGainField;

	private JComboBox<Double> sliceMenu;
	private JButton previousSliceButton;
	private JButton nextSliceButton;
	private boolean ignoreSliceChange;
	private Double currentSliceValue;

	private PatternTableModel patternModel;

	private PatternPlotPanel patternPlotPanel;

	private PatternSearchPanel patternSearchPanel;
	private boolean isSearch;
	private int recordType;

	// Buttons.

	private JButton changeSliceValueButton;
	private JButton removeSliceButton;

	private JButton insertPointButton;
	private JButton deletePointButton;

	private JButton applyButton;

	// Flags.

	private boolean canEdit;
	private boolean didEdit;

	// Disambiguation.

	private PatternEditor outerThis = this;


	//-----------------------------------------------------------------------------------------------------------------
	// First constructor used for editor, no search UI will appear.  The AntPattern model object will be modified
	// directly if editing is allowed, so always make a copy.

	public PatternEditor(AppEditor theParent, AntPattern thePat, boolean theCanEdit, String theTitle) {

		super(theParent, theTitle, Dialog.ModalityType.MODELESS);

		pattern = thePat.copy();
		patternType = thePat.type;
		canEdit = theCanEdit;

		doSetup();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Constructor for search mode, pattern type must be either horizontal or vertical.  May be restricted by record
	// type, but that argument may be 0 to match regardless.

	public PatternEditor(AppEditor theParent, boolean searchVert, int theRecType, String theTitle) {

		super(theParent, theTitle, Dialog.ModalityType.APPLICATION_MODAL);

		patternType = (searchVert ? AntPattern.PATTERN_TYPE_VERTICAL : AntPattern.PATTERN_TYPE_HORIZONTAL);
		isSearch = true;
		recordType = theRecType;

		doSetup();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Common part of construction.  Create the UI components, field for the name first.

	private void doSetup() {

		patternNameField = new JTextField(30);
		AppController.fixKeyBindings(patternNameField);

		JPanel namePanel = new JPanel();
		namePanel.setBorder(BorderFactory.createTitledBorder("Name"));
		namePanel.add(patternNameField);

		AppController.setComponentEnabled(patternNameField, canEdit);

		// Field for editing the horizontal pattern orientation when in horizontal mode.  Also a check-box here will
		// rotate the pattern plot according to the orientation.  Not shown in search mode.

		JPanel horizontalPanel = null;

		if ((AntPattern.PATTERN_TYPE_HORIZONTAL == patternType) && !isSearch) {

			antennaOrientationField = new JTextField(7);
			AppController.fixKeyBindings(antennaOrientationField);

			AppController.setComponentEnabled(antennaOrientationField, canEdit);

			if (canEdit) {

				antennaOrientationField.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (blockActions()) {
							String title = "Edit Pattern Orientation";
							String str = antennaOrientationField.getText().trim();
							if (str.length() > 0) {
								double d = antennaOrientation;
								try {
									d = Math.IEEEremainder(Double.parseDouble(str), 360.);
									if (d < 0.) d += 360.;
								} catch (NumberFormatException ne) {
									errorReporter.reportValidationError(title, "The orientation must be a number");
								}
								if (d != antennaOrientation) {
									antennaOrientation = d;
									if (rotateHorizontalPlotCheckBox.isSelected()) {
										patternPlotPanel.repaint();
									}
								}
							}
							antennaOrientationField.setText(AppCore.formatAzimuth(antennaOrientation));
							blockActionsEnd();
						}
					}
				});

				antennaOrientationField.addFocusListener(new FocusAdapter() {
					public void focusGained(FocusEvent theEvent) {
						setCurrentField(antennaOrientationField);
					}
					public void focusLost(FocusEvent theEvent) {
						if (!theEvent.isTemporary()) {
							antennaOrientationField.postActionEvent();
						}
					}
				});
			}

			rotateHorizontalPlotCheckBox = new JCheckBox("Rotate pattern plot");
			rotateHorizontalPlotCheckBox.setFocusable(false);

			rotateHorizontalPlotCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					patternPlotPanel.repaint();
				}
			});

			horizontalPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			horizontalPanel.setBorder(BorderFactory.createTitledBorder("Orientation"));
			horizontalPanel.add(antennaOrientationField);
			horizontalPanel.add(rotateHorizontalPlotCheckBox);
		}

		// Gain field appears in receive mode only.  Receive mode can never occur in search mode.

		JPanel gainPanel = null;

		if (AntPattern.PATTERN_TYPE_RECEIVE == patternType) {

			antennaGainField = new JTextField(7);
			AppController.fixKeyBindings(antennaGainField);

			AppController.setComponentEnabled(antennaGainField, canEdit);

			if (canEdit) {

				antennaGainField.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (blockActions()) {
							String title = "Edit Antenna Gain";
							String str = antennaGainField.getText().trim();
							if (str.length() > 0) {
								double d = pattern.gain;
								try {
									d = Double.parseDouble(str);
								} catch (NumberFormatException ne) {
									errorReporter.reportValidationError(title, "The gain must be a number");
								}
								if (d != pattern.gain) {
									if ((d < AntPattern.GAIN_MIN) || (d > AntPattern.GAIN_MAX)) {
										errorReporter.reportValidationError(title, "The gain must be in the range " +
											AntPattern.GAIN_MIN + " to " + AntPattern.GAIN_MAX);
									} else {
										pattern.gain = d;
										didEdit = true;
									}
								}
							}
							antennaGainField.setText(AppCore.formatDecimal(pattern.gain, 2));
							blockActionsEnd();
						}
					}
				});

				antennaGainField.addFocusListener(new FocusAdapter() {
					public void focusGained(FocusEvent theEvent) {
						setCurrentField(antennaGainField);
					}
					public void focusLost(FocusEvent theEvent) {
						if (!theEvent.isTemporary()) {
							antennaGainField.postActionEvent();
						}
					}
				});
			}

			gainPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			gainPanel.setBorder(BorderFactory.createTitledBorder("Antenna Gain, dBd"));
			gainPanel.add(antennaGainField);
		}

		// The slice selection menu is used to navigate vertical or receive matrix patterns.  It always appears when
		// the current pattern is a matrix.  It also appears in search mode or when editing since in either case the
		// pattern could change to a matrix.  Controls to add/remove slices only appear when editing.

		JPanel matrixPanel = null;

		if ((AntPattern.PATTERN_TYPE_HORIZONTAL != patternType) &&
				(((null != pattern) && pattern.isMatrix()) || isSearch || canEdit)) {

			sliceMenu = new JComboBox<Double>();
			sliceMenu.setPrototypeDisplayValue(Double.valueOf(999.9));
			sliceMenu.setFocusable(false);
			sliceMenu.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doChangeSlice();
				}
			});

			Dimension butSize = new Dimension(20, 20);
			Insets butMarg = new Insets(0, 0, 0, 0);

			previousSliceButton = new JButton("<");
			previousSliceButton.setPreferredSize(butSize);
			previousSliceButton.setMargin(butMarg);
			previousSliceButton.setFocusable(false);
			previousSliceButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doPreviousSlice();
				}
			});

			nextSliceButton = new JButton(">");
			nextSliceButton.setPreferredSize(butSize);
			nextSliceButton.setMargin(butMarg);
			nextSliceButton.setFocusable(false);
			nextSliceButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doNextSlice();
				}
			});

			matrixPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			if (AntPattern.PATTERN_TYPE_VERTICAL != patternType) {
				matrixPanel.setBorder(BorderFactory.createTitledBorder("Frequency, MHz"));
			} else {
				matrixPanel.setBorder(BorderFactory.createTitledBorder("Azimuth"));
			}
			matrixPanel.add(sliceMenu);
			matrixPanel.add(previousSliceButton);
			matrixPanel.add(nextSliceButton);

			if (canEdit) {

				changeSliceValueButton = new JButton("Edit");
				changeSliceValueButton.setFocusable(false);
				changeSliceValueButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doChangeSliceValue();
					}
				});

				KeyedRecordMenu addSliceMenu = new KeyedRecordMenu();
				addSliceMenu.addItem(new KeyedRecord(0, "Add..."));
				addSliceMenu.addItem(new KeyedRecord(1, "New..."));
				addSliceMenu.addItem(new KeyedRecord(2, "Import..."));
				addSliceMenu.setFocusable(false);
				addSliceMenu.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (blockActions()) {
							doAddSlice(2 == addSliceMenu.getSelectedKey());
							addSliceMenu.setSelectedIndex(0);
							blockActionsEnd();
						}
					}
				});

				removeSliceButton = new JButton("Remove");
				removeSliceButton.setFocusable(false);
				removeSliceButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doRemoveSlice();
					}
				});

				matrixPanel.add(changeSliceValueButton);
				matrixPanel.add(addSliceMenu);
				matrixPanel.add(removeSliceButton);
			}
		}

		// Table for the points.

		patternModel = new PatternTableModel();

		patternModel.table.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
			public void valueChanged(ListSelectionEvent theEvent) {
				patternModel.updateButtons();
			}
		});

		JPanel patternEditPanel = new JPanel(new BorderLayout());
		patternEditPanel.add(AppController.createScrollPane(patternModel.table), BorderLayout.CENTER);

		// Don't move this line!  Table must be in a container before being enabled/disabled.

		AppController.setComponentEnabled(patternModel.table, canEdit);

		// Plot panel.

		patternPlotPanel = new PatternPlotPanel();

		// Search panel if in search mode.

		if (isSearch) {
			patternSearchPanel = new PatternSearchPanel();
		}

		// Buttons to change points only appear when editable.

		JButton addPointButton = null;

		if (canEdit) {

			addPointButton = new JButton("Add");
			addPointButton.setFocusable(false);
			addPointButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doAddPoint();
				}
			});

			insertPointButton = new JButton("Insert");
			insertPointButton.setFocusable(false);
			insertPointButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doInsertPoint();
				}
			});

			deletePointButton = new JButton("Delete");
			deletePointButton.setFocusable(false);
			deletePointButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doDeletePoint();
				}
			});
		}

		// Button to commit edits and/or close window, does not appear in non-editable, non-search mode.

		if (canEdit || isSearch) {
			if (AntPattern.PATTERN_TYPE_RECEIVE == patternType) {
				applyButton = new JButton("Save");
			} else {
				applyButton = new JButton("OK");
			}
			applyButton.setFocusable(false);
			applyButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doApply();
				}
			});
		}

		JButton closeButton = null;
		if (canEdit || isSearch) {
			closeButton = new JButton("Cancel");
		} else {
			closeButton = new JButton("Close");
		}
		closeButton.setFocusable(false);
		closeButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (isSearch) {
					pattern = null;
				}
				cancel();
			}
		});

		// Do the layout.

		JPanel topPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		topPanel.add(namePanel);

		JPanel midPanel = new JPanel(new BorderLayout());

		if ((null != horizontalPanel) || (null != gainPanel) || (null != matrixPanel)) {
			Box extrasBox = Box.createVerticalBox();
			if (null != horizontalPanel) {
				extrasBox.add(horizontalPanel);
			}
			if (null != gainPanel) {
				extrasBox.add(gainPanel);
			}
			if (null != matrixPanel) {
				extrasBox.add(matrixPanel);
			}
			JPanel midTopPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			midTopPanel.add(extrasBox);
			midPanel.add(midTopPanel, BorderLayout.NORTH);
		}

		if (null != addPointButton) {
			JPanel patButPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			patButPanel.add(addPointButton);
			patButPanel.add(insertPointButton);
			patButPanel.add(deletePointButton);
			patternEditPanel.add(patButPanel, BorderLayout.SOUTH);
		}

		JTabbedPane tabPane = new JTabbedPane();
		tabPane.addTab("Data", patternEditPanel);
		tabPane.addTab("Plot", patternPlotPanel);

		midPanel.add(tabPane, BorderLayout.CENTER);

		JPanel bottomPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		bottomPanel.add(closeButton);
		if (null != applyButton) {
			bottomPanel.add(applyButton);
		}

		Container cp = getContentPane();
		cp.setLayout(new BorderLayout());

		if (null != patternSearchPanel) {

			JPanel mainPanel = new JPanel(new BorderLayout());
			mainPanel.add(topPanel, BorderLayout.NORTH);
			mainPanel.add(midPanel, BorderLayout.CENTER);

			JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, patternSearchPanel, mainPanel);
			cp.add(splitPane, BorderLayout.CENTER);

		} else {

			cp.add(topPanel, BorderLayout.NORTH);
			cp.add(midPanel, BorderLayout.CENTER);
		}

		cp.add(bottomPanel, BorderLayout.SOUTH);

		if (null != applyButton) {
			getRootPane().setDefaultButton(applyButton);
		}

		pack();

		Dimension theSize = getSize();
		theSize.height = 650;
		setSize(theSize);
		setMinimumSize(theSize);

		setResizable(true);
		setLocationSaved(true);

		updateDocumentName();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void updateDocumentName() {

		setDocumentName(parent.getDocumentName());
	}


	//-----------------------------------------------------------------------------------------------------------------

	public AntPattern getPattern() {

		return pattern;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean didEdit() {

		return didEdit;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Change the slice being displayed in a matrix pattern.  If editing, this is not allowed to change until the
	// current slice is valid.

	private void doChangeSlice() {

		if ((null == sliceMenu) || (null == currentSliceValue) || ignoreSliceChange || pattern.isSimple()) {
			return;
		}

		if (canEdit && !patternModel.checkPattern()) {
			ignoreSliceChange = true;
			sliceMenu.setSelectedItem(currentSliceValue);
			ignoreSliceChange = false;
			return;
		}

		updateState((Double)sliceMenu.getSelectedItem());
	}

		
	//-----------------------------------------------------------------------------------------------------------------

	private void doPreviousSlice() {

		if ((null == sliceMenu) || (null == currentSliceValue) || ignoreSliceChange || pattern.isSimple() ||
				(canEdit && !patternModel.checkPattern())) {
			return;
		}

		int theIndex = sliceMenu.getSelectedIndex() - 1;
		if (theIndex < 0) {
			theIndex = sliceMenu.getItemCount() - 1;
		}
		ignoreSliceChange = true;
		sliceMenu.setSelectedIndex(theIndex);
		ignoreSliceChange = false;

		updateState((Double)sliceMenu.getSelectedItem());
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doNextSlice() {

		if ((null == sliceMenu) || (null == currentSliceValue) || ignoreSliceChange || pattern.isSimple() ||
				(canEdit && !patternModel.checkPattern())) {
			return;
		}

		int theIndex = sliceMenu.getSelectedIndex() + 1;
		if (theIndex >= sliceMenu.getItemCount()) {
			theIndex = 0;
		}
		ignoreSliceChange = true;
		sliceMenu.setSelectedIndex(theIndex);
		ignoreSliceChange = false;

		updateState((Double)sliceMenu.getSelectedItem());
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Prompts for a new value for the current slice, it must be valid and not already exist in the pattern.

	private void doChangeSliceValue() {

		if (!canEdit || (null == currentSliceValue) || pattern.isSimple()) {
			return;
		}

		String uclbl, lclbl;
		double minVal, maxVal;
		if (AntPattern.PATTERN_TYPE_VERTICAL == patternType) {
			uclbl = "Azimuth";
			lclbl = "azimuth";
			minVal = AntPattern.AZIMUTH_MIN;
			maxVal = AntPattern.AZIMUTH_MAX;
		} else {
			uclbl = "Frequency";
			lclbl = "frequency";
			minVal = AntPattern.FREQUENCY_MIN;
			maxVal = AntPattern.FREQUENCY_MAX;
		}

		String title = "Change Pattern " + uclbl;
		errorReporter.setTitle(title);

		String str;
		double val;
		Double newValue = null;

		do {

			str = (String)JOptionPane.showInputDialog(this, "New " + lclbl, title, JOptionPane.QUESTION_MESSAGE, null,
				null, AppCore.formatDecimal(currentSliceValue.doubleValue(), 2));
			if (null == str) {
				return;
			}
			str = str.trim();

			if (str.length() > 0) {
				try {
					val = Double.parseDouble(str);
					if ((val < minVal) || (val > maxVal)) {
						errorReporter.reportWarning("The " + lclbl + " must be in the range " + minVal + " to " +
							maxVal);
					} else {
						newValue = Double.valueOf(val);
						if (newValue.equals(currentSliceValue)) {
							return;
						}
						if (pattern.containsSlice(newValue)) {
							errorReporter.reportWarning("A pattern with that " + lclbl + " already exists");
							newValue = null;
						}
					}
				} catch (NumberFormatException ne) {
					errorReporter.reportWarning("The " + lclbl + " must be a number");
				}
			}

		} while (null == newValue);

		pattern.changeSliceValue(currentSliceValue, newValue);

		updateState(newValue);
		didEdit = true;
	}

		
	//-----------------------------------------------------------------------------------------------------------------
	// If the pattern is currently simple, this prompts for two new slice values, one for the current simple pattern
	// when it converts to a slice, and a second for the new slice.  Those must both be valid and different.  If the
	// pattern is already a matrix prompt for a new slice value, that must be valid and not already exist.  This is
	// not allowed unless the current pattern is valid.  This is never allowed in horizontal mode.

	private void doAddSlice(boolean doImport) {

		if (!canEdit || (AntPattern.PATTERN_TYPE_HORIZONTAL == patternType) || !patternModel.checkPattern()) {
			return;
		}

		String tuclbl, tlclbl, uclbl, lclbl;
		double minVal, maxVal;
		if (AntPattern.PATTERN_TYPE_VERTICAL != patternType) {
			tuclbl = "Azimuth";
			tlclbl = "azimuth";
			uclbl = "Frequency";
			lclbl = "frequency";
			minVal = AntPattern.FREQUENCY_MIN;
			maxVal = AntPattern.FREQUENCY_MAX;
		} else {
			tuclbl = "Elevation";
			tlclbl = "elevation";
			uclbl = "Azimuth";
			lclbl = "azimuth";
			minVal = AntPattern.AZIMUTH_MIN;
			maxVal = AntPattern.AZIMUTH_MAX;
		}

		String title = "Add " + tuclbl + " Pattern";
		errorReporter.setTitle(title);

		String str;
		double val;
		Double firstValue = null, newValue = null;

		if (pattern.isSimple()) {

			do {

				str = JOptionPane.showInputDialog(this, "Existing pattern " + lclbl, title,
						JOptionPane.QUESTION_MESSAGE);
				if (null == str) {
					return;
				}
				str = str.trim();

				if (str.length() > 0) {
					try {
						val = Double.parseDouble(str);
						if ((val < minVal) || (val > maxVal)) {
							errorReporter.reportWarning("The " + lclbl + " must be in the range " + minVal + " to " +
								maxVal);
						} else {
							firstValue = Double.valueOf(val);
						}
					} catch (NumberFormatException ne) {
						errorReporter.reportWarning("The " + lclbl + " must be a number");
					}
				}

			} while (null == firstValue);
		}

		do {

			str = JOptionPane.showInputDialog(this, "New pattern " + lclbl, title, JOptionPane.QUESTION_MESSAGE);
			if (null == str) {
				return;
			}
			str = str.trim();

			if (str.length() > 0) {
				try {
					val = Double.parseDouble(str);
					if ((val < minVal) || (val > maxVal)) {
						errorReporter.reportWarning("The " + lclbl + " must be in the range " + minVal + " to " +
							maxVal);
					} else {
						newValue = Double.valueOf(val);
						if (pattern.containsSlice(newValue) || ((null != firstValue) && newValue.equals(firstValue))) {
							errorReporter.reportWarning("A pattern with that " + lclbl + " already exists");
							newValue = null;
						}
					}
				} catch (NumberFormatException ne) {
					errorReporter.reportWarning("The " + lclbl + " must be a number");
				}
			}

		} while (null == newValue);

		ArrayList<AntPattern.AntPoint> newPoints = null;

		if (doImport) {

			JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
			chooser.setDialogType(JFileChooser.OPEN_DIALOG);
			chooser.setDialogTitle(title);
			chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
			chooser.setMultiSelectionEnabled(false);
			chooser.addChoosableFileFilter(new FileNameExtensionFilter("Text (*.csv,*.txt,*.dat)", "csv", "txt",
				"dat"));
			chooser.setAcceptAllFileFilterUsed(false);

			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Import")) {
				return;
			}

			File theFile = chooser.getSelectedFile();

			AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

			AntPattern newPat = AntPattern.readFromText(getDbID(), patternType, theFile, errorReporter);
			if (null == newPat) {
				return;
			}

			if (newPat.isSimple()) {
				newPoints = newPat.getPoints();
			} else {
				Double pickValue = null;
				ArrayList<Double> values = newPat.getSliceValues();
				if ((null == values) || values.isEmpty()) {
					return;
				}
				Object[] options = values.toArray();
				pickValue = (Double)JOptionPane.showInputDialog(this, "Select " + tlclbl + " pattern to add", title,
					JOptionPane.INFORMATION_MESSAGE, null, options, options[0]);
				if (null == pickValue) {
					return;
				}
				newPoints = newPat.getSlice(pickValue);
			}

			if (null == newPoints) {
				return;
			}
		}

		if (pattern.isSimple()) {
			pattern.convertToMatrix(firstValue, newValue, newPoints);
		} else {
			pattern.addSlice(newValue, newPoints);
		}

		updateState(newValue);
		didEdit = true;
	}

		
	//-----------------------------------------------------------------------------------------------------------------
	// Removing the next-to-last slice turns the pattern back into a simple pattern.

	private void doRemoveSlice() {

		if (!canEdit || (null == currentSliceValue) || pattern.isSimple()) {
			return;
		}

		pattern.removeSlice(currentSliceValue);

		updateState(null);
		didEdit = true;
	}

		
	//-----------------------------------------------------------------------------------------------------------------
	// Update model and UI state typically after a change to the slice selection.  The pattern can only be null in
	// search mode, at any other time it will always be set.

	private void updateState(Double loadValue) {

		if (null == pattern) {

			if (null != applyButton) {
				applyButton.setEnabled(false);
			}

			patternModel.setPoints(null);
			currentSliceValue = null;

			if (null != sliceMenu) {
				ignoreSliceChange = true;
				sliceMenu.removeAllItems();
				ignoreSliceChange = false;
				sliceMenu.setEnabled(false);
				previousSliceButton.setEnabled(false);
				nextSliceButton.setEnabled(false);
			}

			return;
		}

		if (null != applyButton) {
			applyButton.setEnabled(true);
		}

		if (null == sliceMenu) {
			patternModel.setPoints(pattern.getPoints());
			currentSliceValue = null;
			return;
		}

		ignoreSliceChange = true;

		sliceMenu.removeAllItems();

		if (pattern.isSimple()) {

			sliceMenu.setEnabled(false);
			previousSliceButton.setEnabled(false);
			nextSliceButton.setEnabled(false);
			if (canEdit) {
				changeSliceValueButton.setEnabled(false);
				removeSliceButton.setEnabled(false);
			}

			patternModel.setPoints(pattern.getPoints());
			currentSliceValue = null;

		} else {

			for (Double theValue : pattern.getSliceValues()) {
				sliceMenu.addItem(theValue);
			}

			if ((null != loadValue) && pattern.containsSlice(loadValue)) {
				sliceMenu.setSelectedItem(loadValue);
			} else {
				loadValue = (Double)sliceMenu.getSelectedItem();
			}

			sliceMenu.setEnabled(true);
			previousSliceButton.setEnabled(true);
			nextSliceButton.setEnabled(true);
			if (canEdit) {
				changeSliceValueButton.setEnabled(true);
				removeSliceButton.setEnabled(true);
			}

			patternModel.setPoints(pattern.getSlice(loadValue));
			currentSliceValue = loadValue;
		}

		ignoreSliceChange = false;
	}


	//=================================================================================================================
	// Table model for the pattern editing table.  This object also creates and manages the table itself.

	private class PatternTableModel extends AbstractTableModel {

		private ArrayList<AntPattern.AntPoint> points;

		private JTable table;


		//-------------------------------------------------------------------------------------------------------------

		private PatternTableModel() {

			super();

			points = new ArrayList<AntPattern.AntPoint>();

			table = new JTable(this);
			AppController.configureTable(table);

			TableColumn theColumn = table.getColumn(getColumnName(0));
			theColumn.setMinWidth(AppController.textFieldWidth[4]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[6]);

			theColumn = table.getColumn(getColumnName(1));
			theColumn.setMinWidth(AppController.textFieldWidth[4]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[6]);

			table.setPreferredScrollableViewportSize(new Dimension(table.getPreferredSize().width,
				(table.getRowHeight() * 30)));
		}


		//-------------------------------------------------------------------------------------------------------------
		// A points array from the AntPattern object is modified directly, as are the point objects.

		private void setPoints(ArrayList<AntPattern.AntPoint> thePoints) {

			if (null == thePoints) {
				points = new ArrayList<AntPattern.AntPoint>();
			} else {
				points = thePoints;
			}
			fireTableDataChanged();

			updateButtons();

			patternPlotPanel.repaint();
		}


		//-------------------------------------------------------------------------------------------------------------
		// Points are always kept in order by azimuth/depression, this returns the row at which the point is inserted,
		// and also automatically selects that row.  It will fail if the azimuth/depression already exists in the data,
		// returning -1 in that case.

		private int addPoint(AntPattern.AntPoint newPoint) {

			int rowIndex = 0;
			AntPattern.AntPoint thePoint;

			for (rowIndex = 0; rowIndex < points.size(); rowIndex++) {
				thePoint = points.get(rowIndex);
				if (newPoint.angle <= thePoint.angle) {
					if (newPoint.angle == thePoint.angle) {
						return -1;
					}
					break;
				}
			}

			points.add(rowIndex, newPoint);
			fireTableRowsInserted(rowIndex, rowIndex);

			final int moveToIndex = rowIndex;
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					table.setRowSelectionInterval(moveToIndex, moveToIndex);
					table.scrollRectToVisible(table.getCellRect(moveToIndex, 0, true));
				}
			});

			updateButtons();

			return rowIndex;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void remove(int rowIndex) {

			if (points.size() <= AntPattern.PATTERN_REQUIRED_POINTS) {
				return;
			}

			points.remove(rowIndex);
			fireTableRowsDeleted(rowIndex, rowIndex);

			updateButtons();
		}


		//-------------------------------------------------------------------------------------------------------------

		private void updateButtons() {

			if (canEdit) {
				if (table.getSelectedRow() >= 0) {
					insertPointButton.setEnabled(true);
					deletePointButton.setEnabled(points.size() > AntPattern.PATTERN_REQUIRED_POINTS);
				} else {
					insertPointButton.setEnabled(false);
					deletePointButton.setEnabled(false);
				}
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Convenience.

		private int getSelectedRow() {

			return table.getSelectedRow();
		}


		//-------------------------------------------------------------------------------------------------------------
		// Checks on pattern validity, must have a minimum number of points.  The pattern must have a 1.0 point for
		// horizontal or vertical patterns.  A receive pattern must also have a 1.0 but that will be checked by
		// isDataValid(), for a matrix receive pattern there only has to be a 1.0 in some slice not necessarily all.
		// The validity check does not enforce a 1.0 for horizontal or vertical pattern types because patterns from
		// source databases may not always pass that test, so it is enforced only when the pattern is edited.

		private boolean checkPattern() {

			errorReporter.setTitle("Verify Pattern");

			if (points.size() < AntPattern.PATTERN_REQUIRED_POINTS) {
				errorReporter.reportValidationError("Pattern must have " + AntPattern.PATTERN_REQUIRED_POINTS +
					" or more points");
				return false;
			}

			if (AntPattern.PATTERN_TYPE_RECEIVE != patternType) {

				double patmax = 0.;
				for (AntPattern.AntPoint thePoint : points) {
					if (thePoint.relativeField > patmax) {
						patmax = thePoint.relativeField;
					}
				}

				if (patmax < 1.) {
					errorReporter.reportValidationError("Pattern must have a 1");
					return false;
				}
			}

			return true;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getColumnCount() {

			return 2;
		}


		//-------------------------------------------------------------------------------------------------------------

		public String getColumnName(int columnIndex) {

			if (0 == columnIndex) {
				if (AntPattern.PATTERN_TYPE_VERTICAL != patternType) {
					return "Azimuth";
				} else {
					return "Vertical Angle";
				}
			} else {
				return "Relative Field";
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getRowCount() {

			return points.size();
		}


		//-------------------------------------------------------------------------------------------------------------

		public boolean isCellEditable(int rowIndex, int columnIndex) {

			return canEdit;
		}


		//-------------------------------------------------------------------------------------------------------------

		public Object getValueAt(int rowIndex, int columnIndex) {

			AntPattern.AntPoint thePoint = points.get(rowIndex);

			switch (columnIndex) {

				case 0:
					if (patternType != AntPattern.PATTERN_TYPE_VERTICAL) {
						return AppCore.formatAzimuth(thePoint.angle);
					} else {
						return AppCore.formatDepression(thePoint.angle);
					}

				case 1:
					return AppCore.formatRelativeField(thePoint.relativeField);
			}

			return "";
		}


		//-------------------------------------------------------------------------------------------------------------
		// As points are edited, they are rounded appropriately and error-checked.  Edits that are no-change after
		// rounding are simply ignored.  Otherwise if azimuth/depression is edited, make sure the change does not
		// create a duplicate, and re-order the points if needed.

		public void setValueAt(Object value, int rowIndex, int columnIndex) {

			if (!canEdit) {
				return;
			}

			errorReporter.setTitle("Edit Pattern Point");

			double newVal = 0.;
			try {
				newVal = Double.parseDouble(value.toString());
			} catch (NumberFormatException ne) {
				errorReporter.reportValidationError("The value must be a number");
				return;
			}

			AntPattern.AntPoint thePoint = points.get(rowIndex);

			switch (columnIndex) {

				case 0: {

					String lbl = "";

					if (AntPattern.PATTERN_TYPE_VERTICAL != patternType) {

						newVal = Math.rint(newVal * AntPattern.AZIMUTH_ROUND) / AntPattern.AZIMUTH_ROUND;
						if (newVal == thePoint.angle) {
							return;
						}
						if ((newVal < AntPattern.AZIMUTH_MIN) || (newVal > AntPattern.AZIMUTH_MAX)) {
							errorReporter.reportValidationError("Azimuth must be from " + AntPattern.AZIMUTH_MIN +
								" to " + AntPattern.AZIMUTH_MAX);
							return;
						}
						lbl = "azimuth";

					} else {

						newVal = Math.rint(newVal * AntPattern.DEPRESSION_ROUND) / AntPattern.DEPRESSION_ROUND;
						if (newVal == thePoint.angle) {
							return;
						}
						if ((newVal < AntPattern.DEPRESSION_MIN) || (newVal > AntPattern.DEPRESSION_MAX)) {
							errorReporter.reportValidationError("Vertical angle must be from " +
								AntPattern.DEPRESSION_MIN + " to " + AntPattern.DEPRESSION_MAX);
							return;
						}
						lbl = "vertical angle";
					}

					AntPattern.AntPoint oldPoint;
					int newRowIndex = 0;

					for (newRowIndex = 0; newRowIndex < points.size(); newRowIndex++) {
						if (newRowIndex == rowIndex) {
							continue;
						}
						oldPoint = points.get(newRowIndex);
						if (newVal <= oldPoint.angle) {
							if (newVal == oldPoint.angle) {
								errorReporter.reportValidationError("A point at that " + lbl + " already exists");
								return;
							}
							break;
						}
					}
					if (newRowIndex > rowIndex) {
						newRowIndex--;
					}

					thePoint.angle = newVal;
					didEdit = true;

					if (newRowIndex == rowIndex) {

						fireTableRowsUpdated(rowIndex, rowIndex);

					} else {

						points.remove(rowIndex);
						fireTableRowsDeleted(rowIndex, rowIndex);

						points.add(newRowIndex, thePoint);
						fireTableRowsInserted(newRowIndex, newRowIndex);

						final int moveToIndex = newRowIndex;
						SwingUtilities.invokeLater(new Runnable() {
							public void run() {
								table.setRowSelectionInterval(moveToIndex, moveToIndex);
								table.scrollRectToVisible(table.getCellRect(moveToIndex, 0, true));
							}
						});
					}

					return;
				}

				case 1: {

					if ((newVal < 0.) || (newVal > AntPattern.FIELD_MAX)) {
						errorReporter.reportValidationError("Relative field must be from " + AntPattern.FIELD_MIN +
							" to " + AntPattern.FIELD_MAX);
						return;
					}

					newVal = Math.rint(newVal * AntPattern.FIELD_ROUND) / AntPattern.FIELD_ROUND;
					if (newVal < AntPattern.FIELD_MIN) {
						newVal = AntPattern.FIELD_MIN;
					}

					if (newVal == thePoint.relativeField) {
						return;
					}

					thePoint.relativeField = newVal;
					didEdit = true;

					fireTableRowsUpdated(rowIndex, rowIndex);

					return;
				}
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The pattern point model automatically keeps points in order, an add or insert uses appropriate default values
	// for the new point.  For add, try to extend an existing sequence by stepping along at the same increment of
	// azimuth/depression, defaulting to 10 degrees for azimuth or 1 degree for depression.  This will fail if the last
	// point in the pattern is already at the maximum value for azimuth/depression.

	private void doAddPoint() {

		if (!canEdit) {
			return;
		}

		double newAzDep = 0., newRel = 1.;

		int lastIndex = patternModel.points.size() - 1;
		if (lastIndex >= 0) {

			double azDepInc, maxAzDep;
			if (AntPattern.PATTERN_TYPE_VERTICAL != patternType) {
				azDepInc = 10.;
				maxAzDep = AntPattern.AZIMUTH_MAX - 0.1;
			} else {
				azDepInc = 1.;
				maxAzDep = AntPattern.DEPRESSION_MAX - 0.1;
			}

			AntPattern.AntPoint lastPoint = patternModel.points.get(lastIndex);

			if (lastIndex > 0) {
				AntPattern.AntPoint prevPoint = patternModel.points.get(lastIndex - 1);
				azDepInc = lastPoint.angle - prevPoint.angle;
			}

			newAzDep = lastPoint.angle + azDepInc;
			if (newAzDep > maxAzDep) {
				newAzDep = maxAzDep;
			}
			if (newAzDep <= lastPoint.angle) {
				AppController.beep();
				return;
			}

			newRel = lastPoint.relativeField;
		}

		if (patternModel.addPoint(new AntPattern.AntPoint(newAzDep, newRel)) >= 0) {
			didEdit = true;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Insert is similar to add, in this case the usual default is to interpolate between bracketing points.  If the
	// first point is selected, this offsets downward by an increment similar to the add case above.  This will fail
	// if the bracketing points are separated by just the minimum increment, or if the first point is selected and is
	// already at the minimum azimuth/depression.

	private void doInsertPoint() {

		if (!canEdit) {
			return;
		}

		int rowIndex = patternModel.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		double newAzDep = 0., newRel = 1.;

		AntPattern.AntPoint thisPoint = patternModel.points.get(rowIndex);

		double azDepInc, minAzDep, azDepRnd;
		if (AntPattern.PATTERN_TYPE_VERTICAL != patternType) {
			azDepInc = 10.;
			minAzDep = AntPattern.AZIMUTH_MIN;
			azDepRnd = AntPattern.AZIMUTH_ROUND;
		} else {
			azDepInc = 1.;
			minAzDep = AntPattern.DEPRESSION_MIN;
			azDepRnd = AntPattern.DEPRESSION_ROUND;
		}

		if (rowIndex > 0) {

			AntPattern.AntPoint prevPoint = patternModel.points.get(rowIndex - 1);

			if ((thisPoint.angle - prevPoint.angle) > (2. / azDepRnd)) {

				newAzDep = (thisPoint.angle + prevPoint.angle) / 2.;
				newAzDep = Math.rint(newAzDep * azDepRnd) / azDepRnd;

				newRel = (thisPoint.relativeField + prevPoint.relativeField) / 2.;
				newRel = Math.rint(newRel * AntPattern.FIELD_ROUND) / AntPattern.FIELD_ROUND;

			} else {

				AppController.beep();
				return;
			}

		} else {

			if (rowIndex < (patternModel.points.size() - 1)) {
				AntPattern.AntPoint nextPoint = patternModel.points.get(rowIndex + 1);
				azDepInc = nextPoint.angle - thisPoint.angle;
			}

			newAzDep = thisPoint.angle - azDepInc;
			if (newAzDep < minAzDep) {
				newAzDep = minAzDep;
			}
			if (newAzDep >= thisPoint.angle) {
				AppController.beep();
				return;
			}

			newRel = thisPoint.relativeField;
		}

		if (patternModel.addPoint(new AntPattern.AntPoint(newAzDep, newRel)) >= 0) {
			didEdit = true;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doDeletePoint() {

		if (!canEdit || (patternModel.points.size() <= AntPattern.PATTERN_REQUIRED_POINTS)) {
			return;
		}

		int rowIndex = patternModel.getSelectedRow();
		if (rowIndex >= 0) {
			patternModel.remove(rowIndex);
			didEdit = true;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doApply() {

		if (!canEdit || isSearch) {
			AppController.hideWindow(this);
			return;
		}

		// Commit any pending edits, check for errors.

		errorReporter.clearTitle();

		if (!commitCurrentField()) {
			return;
		}

		errorReporter.clearErrors();
		if (patternModel.table.isEditing()) {
			patternModel.table.getCellEditor().stopCellEditing();
		}
		if (errorReporter.hasErrors()) {
			return;
		}

		// Call both the local checkPattern() and isDataValid(), checkPattern() does more checks than are required for
		// object validity.

		if (!patternModel.checkPattern()) {
			return;
		}
		if (!pattern.isDataValid(errorReporter)) {
			return;
		}

		// A name must be provided for a receive antenna pattern, it is optional for all others.  Certain names are
		// reserved by other UIs, treat those as blank.

		String theName = patternNameField.getText().trim();
		if (theName.equals(AntPattern.NEW_ANTENNA_NAME) || theName.equals(AntPattern.GENERIC_ANTENNA_NAME)) {
			theName = "";
		}
		if (theName.length() > Source.MAX_PATTERN_NAME_LENGTH) {
			theName = theName.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
		}
		if ((AntPattern.PATTERN_TYPE_RECEIVE == patternType) && (0 == theName.length())) {
			errorReporter.reportWarning("Please provide a name for the antenna");
			return;
		}

		if (!theName.equals(pattern.name)) {
			if ((AntPattern.PATTERN_TYPE_RECEIVE == patternType) && 
					!AntPattern.checkReceiveAntennaName(getDbID(), theName, pattern.name, errorReporter)) {
				return;
			}
			pattern.name = theName;
			didEdit = true;
		}

		// Inform the parent of the change, close if successful.

		if (parent.applyEditsFrom(this)) {
			AppController.hideWindow(this);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean cancel() {

		if (windowShouldClose()) {
			AppController.hideWindow(this);
			return true;
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update station data list for changes.

	public void updateExtDbList() {

		if (!isSearch) {
			return;
		}

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {

				ArrayList<KeyedRecord> list = ExtDb.getExtDbList(getDbID(), 0);
				if (null == list) {
					return;
				}

				int selectKey = patternSearchPanel.extDbMenu.getSelectedKey();
				patternSearchPanel.extDbMenu.removeAllItems();
				if (!list.isEmpty()) {
					patternSearchPanel.extDbMenu.addAllItems(list);
					if (patternSearchPanel.extDbMenu.containsKey(selectKey)) {
						patternSearchPanel.extDbMenu.setSelectedKey(selectKey);
					}
				}
			}
		});
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set initial state.

	public void windowWillOpen() {

		updateState(null);

		if (isSearch) {
			updateExtDbList();
		} else {

			patternNameField.setText(pattern.name);

			if (null != antennaOrientationField) {
				antennaOrientationField.setText(AppCore.formatAzimuth(antennaOrientation));
			}

			if (null != antennaGainField) {
				antennaGainField.setText(AppCore.formatDecimal(pattern.gain, 2));
			}
		}

		setLocationRelativeTo(getOwner());

		blockActionsClear();

		if (isSearch) {
			ExtDb.addListener(this);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillClose() {

		if (!isVisible()) {
			return;
		}

		if (isSearch) {
			ExtDb.removeListener(this);
		}

		blockActionsSet();
		if (!isSearch) {
			parent.editorClosing(this);
		}
	}


	//=================================================================================================================
	// Plot the pattern.  Nothing fancy, but it works.

	private class PatternPlotPanel extends Canvas {


		//-------------------------------------------------------------------------------------------------------------

		public void paint(Graphics og) {

			Graphics2D g = (Graphics2D)og;
			g.setColor(Color.GRAY);
			g.setStroke(new BasicStroke((float)1.));

			double topy = (double)getHeight();
			double xscl = 1., yscl = 1., scl = 1., xo, yo, adi, ad, rf, x, y;
			int xi1, yi1, xi2, yi2;

			if (AntPattern.PATTERN_TYPE_VERTICAL != patternType) {

				xo = (double)getWidth() / 2.;
				yo = topy / 2.;

				xscl = xo * 0.95;
				yscl = yo * 0.95;
				if (xscl < yscl) {
					scl = xscl;
				} else {
					scl = yscl;
				}

				adi = 0.02;

				for (rf = 0.2; rf < 1.01; rf += 0.2) {
					x = rf * scl * 2.;
					xi1 = (int)(xo - (x / 2.));
					yi1 = (int)(topy - (yo + (x / 2.)));
					xi2 = (int)x;
					yi2 = (int)x;
					g.drawOval(xi1, yi1, xi2, yi2);
				}
				for (ad = 0.; ad < 359.99; ad += 10.) {
					xi1 = (int)(xo + (0.2 * scl * Math.sin(ad * GeoPoint.DEGREES_TO_RADIANS)));
					yi1 = (int)(topy - (yo + (0.2 * scl * Math.cos(ad * GeoPoint.DEGREES_TO_RADIANS))));
					xi2 = (int)(xo + (scl * Math.sin(ad * GeoPoint.DEGREES_TO_RADIANS)));
					yi2 = (int)(topy - (yo + (scl * Math.cos(ad * GeoPoint.DEGREES_TO_RADIANS))));
					g.drawLine(xi1, yi1, xi2, yi2);
				}

			} else {

				x = (double)getWidth() / 100.;
				xscl = x * 0.95;
				xo = ((x - xscl) / 2.) * 100.;

				yscl = topy * 0.95;
				yo = (topy - yscl) / 2.;

				adi = 0.25;

				xi1 = (int)xo;
				xi2 = (int)(xo + (100. * xscl));
				for (rf = 0.; rf < 1.01; rf += 0.1) {
					yi1 = (int)(topy - (yo + (rf * yscl)));
					yi2 = yi1;
					g.drawLine(xi1, yi1, xi2, yi2);
				}
				yi1 = (int)(topy - yo);
				yi2 = (int)(topy - (yo + yscl));
				for (ad = 0.; ad < 100.01; ad += 5.) {
					xi1 = (int)(xo + (ad * xscl));
					xi2 = xi1;
					g.drawLine(xi1, yi1, xi2, yi2);
				}
				xi1 = (int)(xo + (10. * xscl));
				xi2 = xi1;
				g.setStroke(new BasicStroke((float)2.));
				g.drawLine(xi1, yi1, xi2, yi2);
			}

			ArrayList<AntPattern.AntPoint> thePoints = patternModel.points;
			int rowCount = thePoints.size();

			if (rowCount < 2) {
				return;
			}

			Path2D.Double thePath = new Path2D.Double();

			int row1, row2;
			AntPattern.AntPoint point1, point2;
			double x0 = 0., y0 = 0., xl = 0., yl = 0., ad1, ad2, rf1, rf2, dx, dy;
			boolean first = true;

			if (AntPattern.PATTERN_TYPE_VERTICAL != patternType) {

				double rot = 0.;
				if ((null != rotateHorizontalPlotCheckBox) && rotateHorizontalPlotCheckBox.isSelected()) {
					rot = antennaOrientation;
				}

				for (row1 = 0; row1 < rowCount; row1++) {

					point1 = thePoints.get(row1);
					ad1 = (point1.angle + rot) * GeoPoint.DEGREES_TO_RADIANS;
					rf1 = point1.relativeField;

					row2 = row1 + 1;
					if (row2 == rowCount) {
						row2 = 0;
					}
					point2 = thePoints.get(row2);
					ad2 = (point2.angle + rot) * GeoPoint.DEGREES_TO_RADIANS;
					if (ad2 < ad1) {
						ad2 += GeoPoint.TWO_PI;
					}
					rf2 = point2.relativeField;

					for (ad = ad1; ad < ad2; ad += adi) {
						rf = rf1 + ((rf2 - rf1) * ((ad - ad1) / (ad2 - ad1)));
						x = xo + (rf * scl * Math.sin(ad));
						y = topy - (yo + (rf * scl * Math.cos(ad)));
						if (first) {
							first = false;
							thePath.moveTo(x, y);
							xl = x;
							yl = y;
							x0 = x;
							y0 = y;
						} else {
							dx = x - xl;
							dy = y - yl;
							if (Math.sqrt((dx * dx) + (dy * dy)) > 2.) {
								thePath.lineTo(x, y);
								xl = x;
								yl = y;
							}
						}
					}
				}

				thePath.lineTo(x0, y0);

			} else {

				for (row1 = 0; row1 < (rowCount - 1); row1++) {

					point1 = thePoints.get(row1);
					ad1 = point1.angle + 10.;
					rf1 = point1.relativeField;

					point2 = thePoints.get(row1 + 1);
					ad2 = point2.angle + 10.;
					rf2 = point2.relativeField;

					for (ad = ad1; ad < ad2; ad += adi) {

						if (ad < 0.) {
							continue;
						}

						rf = rf1 + ((rf2 - rf1) * ((ad - ad1) / (ad2 - ad1)));
						x = xo + (ad * xscl);
						y = topy - (yo + (rf * yscl));

						if (first) {
							first = false;
							thePath.moveTo(x, y);
							xl = x;
							yl = y;
						} else {
							dx = x - xl;
							dy = y - yl;
							if (Math.sqrt((dx * dx) + (dy * dy)) > 2.) {
								thePath.lineTo(x, y);
								xl = x;
								yl = y;
							}
						}
					}
				}
			}

			g.setColor(Color.BLACK);
			g.setStroke(new BasicStroke((float)3., BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(thePath);
		}
	}


	//=================================================================================================================
	// Search UI.

	private class PatternSearchPanel extends JPanel {

		private KeyedRecordMenu extDbMenu;

		private JTextField searchField;

		private SearchListModel searchModel;

		// Buttons.

		private JButton searchButton;


		//-------------------------------------------------------------------------------------------------------------
		// Create the UI components.  The data set menu will be re-populated as needed, see updateExtDbList().

		private PatternSearchPanel() {

			extDbMenu = new KeyedRecordMenu(new ArrayList<KeyedRecord>());

			JPanel extDbPanel = new JPanel();
			extDbPanel.setBorder(BorderFactory.createTitledBorder("Station Data"));
			extDbPanel.add(extDbMenu);

			searchField = new JTextField(15);
			AppController.fixKeyBindings(searchField);
			searchField.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doSearch();
				}
			});

			searchModel = new SearchListModel();

			searchModel.list.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
				public void valueChanged(ListSelectionEvent theEvent) {
					doLoad();
				}
			});

			// Buttons.

			searchButton = new JButton("Search");
			searchButton.setFocusable(false);
			searchButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doSearch();
				}
			});

			// Do the layout.

			JPanel row1Panel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			row1Panel.add(extDbPanel);

			JPanel row2Panel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			row2Panel.add(searchField);
			row2Panel.add(searchButton);

			JPanel topPanel = new JPanel();
			topPanel.setLayout(new BoxLayout(topPanel, BoxLayout.Y_AXIS));
			topPanel.add(row1Panel);
			topPanel.add(row2Panel);

			setLayout(new BorderLayout());
			add(topPanel, BorderLayout.NORTH);
			add(AppController.createScrollPane(searchModel.list), BorderLayout.CENTER);
		}


		//=============================================================================================================
		// List model for the search results list.

		private class SearchListModel extends AbstractListModel<String> {

			private ArrayList<ExtDb.PatternID> modelRows;

			private JList<String> list;


			//---------------------------------------------------------------------------------------------------------

			private SearchListModel() {

				modelRows = new ArrayList<ExtDb.PatternID>();

				list = new JList<String>(this);
				list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			}


			//---------------------------------------------------------------------------------------------------------

			private void setItems(ArrayList<ExtDb.PatternID> newList) {

				int lastIndex = modelRows.size() - 1;
				if (lastIndex >= 0) {
					modelRows.clear();
					fireIntervalRemoved(this, 0, lastIndex);
				}

				if (null != newList) {
					modelRows.addAll(newList);
				}

				lastIndex = modelRows.size() - 1;
				if (lastIndex >= 0) {
					fireIntervalAdded(this, 0, lastIndex);
				}
			}


			//---------------------------------------------------------------------------------------------------------

			private ExtDb.PatternID getSelectedItem() {

				int rowIndex = list.getSelectedIndex();
				if (rowIndex >= 0) {
					return modelRows.get(rowIndex);
				}
				return null;
			}


			//---------------------------------------------------------------------------------------------------------

			public int getSize() {

				return modelRows.size();
			}


			//---------------------------------------------------------------------------------------------------------

			public String getElementAt(int index) {

				ExtDb.PatternID theItem = modelRows.get(index);

				if (theItem.isMatrix) {
					return theItem.name + " (matrix)";
				} else {
					return theItem.name;
				}
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Do the search.

		private void doSearch() {

			errorReporter.setTitle("Pattern Search");

			int theKey = extDbMenu.getSelectedKey();
			if (theKey <= 0) {
				errorReporter.reportWarning("Please select a station data set");
				return;
			}

			final Integer extDbKey = Integer.valueOf(theKey);

			final String str = searchField.getText().trim();
			if (0 == str.length()) {
				return;
			}

			BackgroundWorker<ArrayList<ExtDb.PatternID>> theWorker =
					new BackgroundWorker<ArrayList<ExtDb.PatternID>>(outerThis, getTitle()) {
				protected ArrayList<ExtDb.PatternID> doBackgroundWork(ErrorLogger errors) {
					return ExtDb.findPatterns(getDbID(), extDbKey, str,
						(AntPattern.PATTERN_TYPE_VERTICAL == patternType), recordType, errors);
				}
			};

			ArrayList<ExtDb.PatternID> theItems = theWorker.runWork("Searching for patterns, please wait...",
				errorReporter);
			if (null == theItems) {
				return;
			}

			pattern = null;
			updateState(null);
			patternNameField.setText("");

			searchModel.setItems(theItems);
		}


		//-------------------------------------------------------------------------------------------------------------
		// Load pattern data for the selected antenna into the pattern display UI.

		private void doLoad() {

			ExtDb.PatternID theAnt = searchModel.getSelectedItem();
			if (null == theAnt) {
				return;
			}

			String title = "Load Pattern Data";
			errorReporter.setTitle(title);

			errorReporter.clearMessages();

			pattern = ExtDb.getPattern(theAnt, errorReporter);
			updateState(null);
			if (null != pattern) {
				patternNameField.setText(pattern.name);
				errorReporter.showMessages();
			} else {
				patternNameField.setText("");
			}
		}
	}
}
