//
//  TemplateManager.java
//  TVStudy
//
//  Copyright (c) 2015-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;
import gov.fcc.tvstudy.gui.editor.*;

import java.util.*;
import java.sql.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.table.*;
import javax.swing.filechooser.*;


//=====================================================================================================================
// Editor for the list of templates in a database.  Allows templates to be created, imported, exported, and deleted,
// also opens and manages template editors, see TemplateEditor.  These are managed as standalone instances one per
// open database as with StudyManager, see showManager().

public class TemplateManager extends AppFrame {

	public static final String WINDOW_TITLE = "Template Manager";

	private String dbID;

	private TemplateListModel templateModel;
	private JTable templateTable;

	private HashMap<Integer, TemplateEditor> templateEditors;

	private JButton duplicateTemplateButton;
	private JButton exportTemplateButton;
	private JButton openTemplateButton;

	private JMenuItem openTemplateMenuItem;
	private JMenuItem renameTemplateMenuItem;
	private JMenuItem duplicateTemplateMenuItem;
	private JMenuItem deleteTemplateMenuItem;
	private JMenuItem exportTemplateMenuItem;

	private JMenuItem templateInfoMenuItem;


	//-----------------------------------------------------------------------------------------------------------------

	private static HashMap<String, TemplateManager> managers = new HashMap<String, TemplateManager>();

	public static boolean showManager(String theDbID) {

		TemplateManager theManager = managers.get(theDbID);
		if (null != theManager) {
			theManager.toFront();
			return true;
		}

		if (DbCore.isDbRegistered(theDbID)) {
			theManager = new TemplateManager(theDbID);
			managers.put(theDbID, theManager);
			AppController.showWindow(theManager);
			return true;
		}

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	private TemplateManager(String theDbID) {

		super(null, WINDOW_TITLE);

		dbID = theDbID;

		// Table for the template list.

		templateModel = new TemplateListModel();
		templateTable = templateModel.createTable();

		templateTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
			public void valueChanged(ListSelectionEvent theEvent) {
				updateControls();
			}
		});

		templateTable.addMouseListener(new MouseAdapter() {
			public void mouseClicked(MouseEvent e) {
				if (2 == e.getClickCount()) {
					doOpenTemplate();
				}
			}
		});

		JPanel templatePanel = new JPanel(new BorderLayout());
		templatePanel.setBorder(BorderFactory.createTitledBorder("Templates"));
		templatePanel.add(AppController.createScrollPane(templateTable), BorderLayout.CENTER);

		templateEditors = new HashMap<Integer, TemplateEditor>();

		// Buttons.

		duplicateTemplateButton = new JButton("Duplicate");
		duplicateTemplateButton.setFocusable(false);
		duplicateTemplateButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doDuplicateTemplate();
			}
		});

		JButton importTemplateButton = new JButton("Import");
		importTemplateButton.setFocusable(false);
		importTemplateButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doImportTemplate();
			}
		});

		exportTemplateButton = new JButton("Export");
		exportTemplateButton.setFocusable(false);
		exportTemplateButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doExportTemplate();
			}
		});

		openTemplateButton = new JButton("Open");
		openTemplateButton.setFocusable(false);
		openTemplateButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doOpenTemplate();
			}
		});

		// Do the layout.

		JPanel butL = new JPanel(new FlowLayout(FlowLayout.LEFT));
		butL.add(duplicateTemplateButton);
		butL.add(importTemplateButton);
		butL.add(exportTemplateButton);

		JPanel butR = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		butR.add(openTemplateButton);

		JPanel buttonPanel = new JPanel();
		buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
		buttonPanel.add(butL);
		buttonPanel.add(butR);

		Container cp = getContentPane();
		cp.setLayout(new BorderLayout());
		cp.add(templatePanel, BorderLayout.CENTER);
		cp.add(buttonPanel, BorderLayout.SOUTH);

		pack();

		Dimension theSize = new Dimension(500, 400);
		setMinimumSize(theSize);
		setSize(theSize);

		// Build the file menu.

		fileMenu.removeAll();

		// Previous

		JMenuItem miPrevious = new JMenuItem("Previous");
		miPrevious.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_UP, AppController.MENU_SHORTCUT_KEY_MASK));
		miPrevious.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doPrevious();
			}
		});
		fileMenu.add(miPrevious);

		// Next

		JMenuItem miNext = new JMenuItem("Next");
		miNext.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, AppController.MENU_SHORTCUT_KEY_MASK));
		miNext.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doNext();
			}
		});
		fileMenu.add(miNext);

		// __________________________________

		fileMenu.addSeparator();

		// Refresh List

		JMenuItem miRefresh = new JMenuItem("Refresh List");
		miRefresh.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				updateTemplateList(true);
			}
		});
		fileMenu.add(miRefresh);

		// __________________________________

		fileMenu.addSeparator();

		// Import...

		JMenuItem miImport = new JMenuItem("Import...");
		miImport.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doImportTemplate();
			}
		});
		fileMenu.add(miImport);

		// Export...

		exportTemplateMenuItem = new JMenuItem("Export...");
		exportTemplateMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E,
			AppController.MENU_SHORTCUT_KEY_MASK));
		exportTemplateMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doExportTemplate();
			}
		});
		fileMenu.add(exportTemplateMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// Open

		openTemplateMenuItem = new JMenuItem("Open");
		openTemplateMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O,
			AppController.MENU_SHORTCUT_KEY_MASK));
		openTemplateMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doOpenTemplate();
			}
		});
		fileMenu.add(openTemplateMenuItem);

		// Rename...

		renameTemplateMenuItem = new JMenuItem("Rename...");
		renameTemplateMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRenameTemplate();
			}
		});
		fileMenu.add(renameTemplateMenuItem);

		// Duplicate...

		duplicateTemplateMenuItem = new JMenuItem("Duplicate...");
		duplicateTemplateMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doDuplicateTemplate();
			}
		});
		fileMenu.add(duplicateTemplateMenuItem);

		// Delete

		deleteTemplateMenuItem = new JMenuItem("Delete");
		deleteTemplateMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doDeleteTemplate();
			}
		});
		fileMenu.add(deleteTemplateMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// Get Info

		templateInfoMenuItem = new JMenuItem("Get Info");
		templateInfoMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_I, AppController.MENU_SHORTCUT_KEY_MASK));
		templateInfoMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doTemplateInfo();
			}
		});
		fileMenu.add(templateInfoMenuItem);

		// Build the extra menu.

		extraMenu.removeAll();

		// Study Manager

		JMenuItem miStudy = new JMenuItem("Study Manager");
		miStudy.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				StudyManager.showManager(dbID);
			}
		});
		extraMenu.add(miStudy);

		// Station Data Manager

		JMenuItem miExtDb = new JMenuItem("Station Data Manager");
		miExtDb.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				ExtDbManager.showManager(dbID);
			}
		});
		extraMenu.add(miExtDb);

		// Initial update of UI state.

		updateControls();

		updateDocumentName();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getDbID() {

		return dbID;
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected String getFileMenuName() {

		return "Template";
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected boolean showsEditMenu() {

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected boolean showsExtraMenu() {

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected String getExtraMenuName() {

		return "Database";
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void updateDocumentName() {

		setDocumentName(DbCore.getHostDbName(dbID));
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update controls per table selection.  Note permanent templates cannot be exported.  Only database installation
	// or update code can create permanent templates which means those must be present in every database so there is
	// no reason to export/import those.  Permanent templates can be duplicated (the duplicate will not be permanent)
	// so if the user really wants data from a permanent template in XML form it can be duplicated then exported.  Any
	// template can be opened, the editor has view-only mode for locked and locked-in-study templates.

	private void updateControls() {

		int rowIndex = templateTable.getSelectedRow();
		boolean eOpen = false, eRename = false, eDuplicate = false, eDelete = false, eExport = false, eInfo = false;

		if (rowIndex >= 0) {

			eOpen = true;
			eDuplicate = true;
			eInfo = true;

			TemplateListItem theItem = templateModel.get(templateTable.convertRowIndexToModel(rowIndex));
			if (!theItem.isPermanent) {
				if (!templateEditors.containsKey(theItem.key)) {
					eRename = true;
				}
				if (0 == theItem.useCount) {
					eDelete = true;
				}
				eExport = true;
			}
		}

		duplicateTemplateButton.setEnabled(eDuplicate);
		exportTemplateButton.setEnabled(eExport);

		openTemplateButton.setEnabled(eOpen);

		openTemplateMenuItem.setEnabled(eOpen);
		renameTemplateMenuItem.setEnabled(eRename);
		duplicateTemplateMenuItem.setEnabled(eDuplicate);
		deleteTemplateMenuItem.setEnabled(eDelete);
		exportTemplateMenuItem.setEnabled(eExport);

		templateInfoMenuItem.setEnabled(eInfo);
	}


	//=================================================================================================================
	// Data class for the template list.

	private class TemplateListItem {

		private Integer key;
		private String name;
		private boolean isPermanent;
		private boolean isLocked;
		private boolean isLockedInStudy;
		private int useCount;
	}


	//=================================================================================================================

	private class TemplateListModel extends AbstractTableModel {

		private static final String TEMPLATE_NAME_COLUMN = "Name";
		private static final String TEMPLATE_LOCKED_COLUMN = "Locked";
		private static final String TEMPLATE_STUDY_LOCKED_COLUMN = "Study Locked";
		private static final String TEMPLATE_IN_USE_COLUMN = "In Use";

		private String[] columnNames = {
			TEMPLATE_NAME_COLUMN,
			TEMPLATE_LOCKED_COLUMN,
			TEMPLATE_STUDY_LOCKED_COLUMN,
			TEMPLATE_IN_USE_COLUMN
		};

		private static final int TEMPLATE_NAME_INDEX = 0;
		private static final int TEMPLATE_LOCKED_INDEX = 1;
		private static final int TEMPLATE_STUDY_LOCKED_INDEX = 2;
		private static final int TEMPLATE_IN_USE_INDEX = 3;

		private ArrayList<TemplateListItem> modelRows;


		//-------------------------------------------------------------------------------------------------------------

		private TemplateListModel() {

			super();

			modelRows = new ArrayList<TemplateListItem>();
		}


		//-------------------------------------------------------------------------------------------------------------

		private JTable createTable() {

			JTable theTable = new JTable(this);
			AppController.configureTable(theTable);

			TableColumn theColumn = theTable.getColumn(TEMPLATE_NAME_COLUMN);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[20]);

			theColumn = theTable.getColumn(TEMPLATE_LOCKED_COLUMN);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[6]);

			theColumn = theTable.getColumn(TEMPLATE_STUDY_LOCKED_COLUMN);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[6]);

			theColumn = theTable.getColumn(TEMPLATE_IN_USE_COLUMN);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[6]);

			return theTable;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void setItems(ArrayList<TemplateListItem> newItems) {

			modelRows.clear();
			modelRows.addAll(newItems);
			fireTableDataChanged();
		}


		//-------------------------------------------------------------------------------------------------------------

		private TemplateListItem get(int rowIndex) {

			return modelRows.get(rowIndex);
		}


		//-------------------------------------------------------------------------------------------------------------

		private void remove(int rowIndex) {

			modelRows.remove(rowIndex);
			fireTableRowsDeleted(rowIndex, rowIndex);
		}


		//-------------------------------------------------------------------------------------------------------------

		private void itemWasChanged(int rowIndex) {

			fireTableRowsUpdated(rowIndex, rowIndex);
		}


		//-------------------------------------------------------------------------------------------------------------

		private int indexOf(TemplateListItem theItem) {

			return modelRows.indexOf(theItem);
		}


		//-------------------------------------------------------------------------------------------------------------

		private int indexOfKey(Integer theKey) {

			int rowIndex = 0;
			for (TemplateListItem theItem : modelRows) {
				if (theItem.key.equals(theKey)) {
					return rowIndex;
				}
				rowIndex++;
			}

			return -1;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getColumnCount() {

			return columnNames.length;
		}


		//-------------------------------------------------------------------------------------------------------------

		public String getColumnName(int columnIndex) {

			return columnNames[columnIndex];
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getRowCount() {

			return modelRows.size();
		}
			

		//-------------------------------------------------------------------------------------------------------------

		public Object getValueAt(int rowIndex, int columnIndex) {

			TemplateListItem theItem = modelRows.get(rowIndex);

			switch (columnIndex) {

				case TEMPLATE_NAME_INDEX:
					return theItem.name;

				case TEMPLATE_LOCKED_INDEX:
					return (theItem.isLocked ? "Yes" : "No");

				case TEMPLATE_STUDY_LOCKED_INDEX:
					return (theItem.isLockedInStudy ? "Yes" : "No");

				case TEMPLATE_IN_USE_INDEX:
					return ((theItem.useCount > 0) ? "Yes" : "No");
			}

			return "";
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Refresh template list for specified manager if visible, called by study editor after template save.

	public static void updateTemplates(String theDbID) {

		TemplateManager theManager = managers.get(theDbID);
		if ((null != theManager) && theManager.isVisible()) {
			theManager.updateTemplateList(true);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update the template list to current database contents, optionally preserving the table selection.

	private void updateTemplateList(boolean preserveSelection) {

		String saveTitle = errorReporter.getTitle();
		errorReporter.setTitle("Load Template List");

		ArrayList<TemplateListItem> list = getItems(errorReporter);
		if (null == list) {
			errorReporter.setTitle(saveTitle);
			return;
		}

		TemplateListItem selectedItem = null;
		if (preserveSelection) {
			int rowIndex = templateTable.getSelectedRow();
			if (rowIndex >= 0) {
				selectedItem = templateModel.get(templateTable.convertRowIndexToModel(rowIndex));
			}
		}

		templateModel.setItems(list);

		if (selectedItem != null) {
			int rowIndex = templateModel.indexOf(selectedItem);
			if (rowIndex >= 0) {
				rowIndex = templateTable.convertRowIndexToView(rowIndex);
				templateTable.setRowSelectionInterval(rowIndex, rowIndex);
				templateTable.scrollRectToVisible(templateTable.getCellRect(rowIndex, 0, true));
			}
		}

		updateControls();

		errorReporter.setTitle(saveTitle);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Retrieve a list of all templates in the database.  Note any with a value in the hidden date field are ignored,
	// see TemplateEditData.createNewTemplate().  However during the lookup this will watch for any hidden templates
	// that need to be activated and call Template.checkHiddenTemplates() if needed.  That will only be attempted once.

	private ArrayList<TemplateListItem> getItems(ErrorReporter errors) {

		ArrayList<TemplateListItem> result = new ArrayList<TemplateListItem>();

		TemplateListItem theItem;

		DbConnection db = DbCore.connectDb(dbID, errors);
		if (null != db) {
			try {

				boolean checkHidden = true, hasHidden;

				do {

					hasHidden = false;

					result.clear();

					if (checkHidden) {
						db.query(
						"SELECT " +
							"template_key, " +
							"name, " +
							"permanent, " +
							"locked, " +
							"locked_in_study, " +
							"(SELECT COUNT(*) FROM study WHERE template_key = template.template_key), " +
							"hidden " +
						"FROM " +
							"template " +
						"WHERE " +
							"(hidden IS NULL) " +
							"OR (hidden <= NOW()) " +
						"ORDER BY 1");
					} else {
						db.query(
						"SELECT " +
							"template_key, " +
							"name, " +
							"permanent, " +
							"locked, " +
							"locked_in_study, " +
							"(SELECT COUNT(*) FROM study WHERE template_key = template.template_key) " +
						"FROM " +
							"template " +
						"WHERE " +
							"(hidden IS NULL) " +
						"ORDER BY 1");
					}

					while (db.next()) {

						if (checkHidden && (null != db.getDate(7))) {
							hasHidden = true;
							break;
						}

						theItem = new TemplateListItem();

						theItem.key = Integer.valueOf(db.getInt(1));
						theItem.name = db.getString(2);
						theItem.isPermanent = db.getBoolean(3);
						theItem.isLocked = db.getBoolean(4);
						theItem.isLockedInStudy = db.getBoolean(5);
						theItem.useCount = db.getInt(6);

						result.add(theItem);
					}

					checkHidden = false;

					if (hasHidden) {
						Template.checkHiddenTemplates(db);
					}

				} while (hasHidden);

				DbCore.releaseDb(db);

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				DbConnection.reportError(errors, se);
			}
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Re-query the database record for a template, update list item properties.  Return the list item as updated, or
	// null if the record no longer exists (the item is removed from the list) or an error occurs.

	private TemplateListItem checkTemplate(int rowIndex, ErrorReporter errors) {

		TemplateListItem theItem = templateModel.get(rowIndex);

		boolean error = false, notfound = false;

		DbConnection db = DbCore.connectDb(dbID, errors);
		if (null != db) {
			try {

				db.query(
				"SELECT " +
					"name, " +
					"permanent, " +
					"locked, " +
					"locked_in_study, " +
					"(SELECT COUNT(*) FROM study WHERE template_key = template.template_key) " +
				"FROM " +
					"template " +
				"WHERE " +
					"template_key = " + theItem.key);

				if (db.next()) {

					theItem.name = db.getString(1);
					theItem.isPermanent = db.getBoolean(2);
					theItem.isLocked = db.getBoolean(3);
					theItem.isLockedInStudy = db.getBoolean(4);
					theItem.useCount = db.getInt(5);

				} else {
					notfound = true;
				}

				DbCore.releaseDb(db);

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				error = true;
				DbConnection.reportError(errors, se);
			}

		} else {
			return null;
		}

		if (notfound) {
			if (null != errors) {
				errors.reportError("The template no longer exists");
			}
			templateModel.remove(rowIndex);
			return null;
		}

		if (error) {
			return null;
		}

		templateModel.itemWasChanged(rowIndex);

		return theItem;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Move selection up or down in the table.

	private void doPrevious() {

		int size = templateTable.getRowCount();
		int rowIndex = templateTable.getSelectedRow();
		if ((size > 0) && (rowIndex != 0)) {
			if (rowIndex < 0) {
				rowIndex = size - 1;
			} else {
				rowIndex--;
			}
			templateTable.setRowSelectionInterval(rowIndex, rowIndex);
			templateTable.scrollRectToVisible(templateTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doNext() {

		int size = templateTable.getRowCount();
		int rowIndex = templateTable.getSelectedRow();
		if ((size > 0) && (rowIndex < (size - 1))) {
			if (rowIndex < 0) {
				rowIndex = 0;
			} else {
				rowIndex++;
			}
			templateTable.setRowSelectionInterval(rowIndex, rowIndex);
			templateTable.scrollRectToVisible(templateTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Import a study template from an XML file, the import method in TemplateEditData creates the template directly
	// in the database.  In fact the import code will support more than one template in the same file, but that rarely
	// if ever occurs since the export function below will only export one template.

	private void doImportTemplate() {

		String title = "Import Template";
		errorReporter.setTitle(title);

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.OPEN_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new FileNameExtensionFilter("XML (*.xml)", "xml"));
		chooser.setAcceptAllFileFilterUsed(false);

		if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Import")) {
			return;
		}

		File theFile = chooser.getSelectedFile();

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

		FileReader theReader = null;
		try {
			theReader = new FileReader(theFile);
		} catch (FileNotFoundException fnfe) {
			errorReporter.reportError("Could not open the file:\n" + fnfe.getMessage());
			return;
		}

		final BufferedReader xml = new BufferedReader(theReader);

		BackgroundWorker<Integer> theWorker = new BackgroundWorker<Integer>(this, title) {
			protected Integer doBackgroundWork(ErrorLogger errors) {
				return TemplateEditData.readTemplateFromXML(dbID, xml, errors);
			}
		};

		Integer newKey = theWorker.runWork("Importing template, please wait...", errorReporter);

		try {xml.close();} catch (IOException ie) {}

		if (null == newKey) {
			return;
		}

		updateTemplateList(false);

		int rowIndex = templateModel.indexOfKey(newKey);
		if (rowIndex >= 0) {
			rowIndex = templateTable.convertRowIndexToView(rowIndex);
			templateTable.setRowSelectionInterval(rowIndex, rowIndex);
			templateTable.scrollRectToVisible(templateTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Export the selected template to an XML file.

	private void doExportTemplate() {

		int rowIndex = templateTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		String title = "Export Template";
		errorReporter.setTitle(title);

		TemplateListItem theItem = checkTemplate(templateTable.convertRowIndexToModel(rowIndex), errorReporter);
		if (null == theItem) {
			return;
		}
		if (theItem.isPermanent) {
			return;
		}

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.SAVE_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new FileNameExtensionFilter("XML (*.xml)", "xml"));
		chooser.setAcceptAllFileFilterUsed(false);

		File theFile = null;
		String theName;
		do {
			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Export")) {
				return;
			}
			theFile = chooser.getSelectedFile();
			theName = theFile.getName();
			if (!theName.toLowerCase().endsWith(".xml")) {
				theFile = new File(theFile.getAbsolutePath() + ".xml");
			}
			if (theFile.exists()) {
				AppController.beep();
				if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
						"The file exists, do you want to replace it?", title, JOptionPane.YES_NO_OPTION,
						JOptionPane.WARNING_MESSAGE)) {
					theFile = null;
				}
			}
		} while (null == theFile);

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

		FileWriter theWriter = null;
		try {
			theWriter = new FileWriter(theFile);
		} catch (IOException ie) {
			errorReporter.reportError("Could not open the file:\n" + ie.getMessage());
			return;
		}

		BufferedWriter xml = new BufferedWriter(theWriter);

		TemplateEditData.writeTemplateToXML(dbID, theItem.key.intValue(), xml, errorReporter);

		try {xml.close();} catch (IOException ie) {}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Open a template editor.

	private void doOpenTemplate() {

		int rowIndex = templateTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		String title = "Open Template";
		errorReporter.setTitle(title);

		TemplateListItem theItem = checkTemplate(templateTable.convertRowIndexToModel(rowIndex), errorReporter);
		if (null == theItem) {
			return;
		}

		TemplateEditor theEditor = templateEditors.get(theItem.key);
		if (null != theEditor) {
			if (theEditor.isVisible()) {
				theEditor.toFront();
				return;
			}
			templateEditors.remove(theItem.key);
		}

		Template theTemplate = Template.getTemplate(dbID, theItem.key.intValue(), errorReporter);
		if (null == theTemplate) {
			return;
		}

		try {
			TemplateEditData theTemplateEditData = new TemplateEditData(dbID, theTemplate);
			theEditor = new TemplateEditor(this, theTemplateEditData);
		} catch (Throwable t) {
			AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", t);
			errorReporter.reportError("Unexpected error:\n" + t);
			return;
		}

		AppController.showWindow(theEditor);
		templateEditors.put(theItem.key, theEditor);

		updateControls();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Rename an existing template, this can of course also be done by editing the template so if an editor is open
	// this operation is disabled.  However this rename will also work on locked templates.  It will not work on
	// permanent templates.  Prompt for a new name, make sure it is unique, then directly modify in the database.  Do
	// a concurrent-safe name check during the save.

	private void doRenameTemplate() {

		int rowIndex = templateTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		String title = "Rename Template";
		errorReporter.setTitle(title);

		TemplateListItem theItem = checkTemplate(templateTable.convertRowIndexToModel(rowIndex), errorReporter);
		if ((null == theItem) || theItem.isPermanent || templateEditors.containsKey(theItem.key)) {
			return;
		}

		String theName = "";

		while (true) {
			theName = (String)(JOptionPane.showInputDialog(this, "Enter a new name for the template", title,
				JOptionPane.QUESTION_MESSAGE, null, null, theItem.name));
			if (null == theName) {
				return;
			}
			theName = theName.trim();
			if (theName.equals(theItem.name)) {
				return;
			}
			if (Template.checkTemplateName(dbID, theName, theItem.name, errorReporter)) {
				break;
			}
		};

		Template.renameTemplate(dbID, theItem.key.intValue(), theName, errorReporter);

		updateTemplateList(true);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Duplicate an existing template.  Prompt for a new name, make sure it is unique.

	private void doDuplicateTemplate() {

		int rowIndex = templateTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		String title = "Duplicate Template";
		errorReporter.setTitle(title);

		TemplateListItem theItem = checkTemplate(templateTable.convertRowIndexToModel(rowIndex), errorReporter);
		if (null == theItem) {
			return;
		}

		String theName = "";

		while (true) {
			theName = (String)(JOptionPane.showInputDialog(this, "Enter a name for the new template", title,
				JOptionPane.QUESTION_MESSAGE, null, null, theItem.name));
			if (null == theName) {
				return;
			}
			theName = theName.trim();
			if (Template.checkTemplateName(dbID, theName, errorReporter)) {
				break;
			}
		};

		final int oldKey = theItem.key.intValue();
		final String newName = theName;

		BackgroundWorker<Integer> theWorker = new BackgroundWorker<Integer>(this, title) {
			protected Integer doBackgroundWork(ErrorLogger errors) {
				return Template.duplicateTemplate(dbID, oldKey, newName, errors);
			}
		};

		Integer newKey = theWorker.runWork("Duplicating template, please wait...", errorReporter);
		if (null == newKey) {
			return;
		}

		updateTemplateList(false);

		rowIndex = templateModel.indexOfKey(newKey);
		if (rowIndex < 0) {
			return;
		}

		rowIndex = templateTable.convertRowIndexToView(rowIndex);
		templateTable.setRowSelectionInterval(rowIndex, rowIndex);
		templateTable.scrollRectToVisible(templateTable.getCellRect(rowIndex, 0, true));
	}


	//-----------------------------------------------------------------------------------------------------------------
	// A template cannot be deleted if it is permanent or if it is still in use by studies.

	private void doDeleteTemplate() {

		int rowIndex = templateTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		String title = "Delete Template";
		errorReporter.setTitle(title);

		TemplateListItem theItem = checkTemplate(templateTable.convertRowIndexToModel(rowIndex), errorReporter);
		if (null == theItem) {
			return;
		}
		if (theItem.isPermanent) {
			return;
		}

		// Confirm the operation, if an editor window is open, force it closed.

		AppController.beep();
		int result = JOptionPane.showConfirmDialog(this, "Are you sure you want to delete template '" +
			theItem.name + "'?", title, JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
		if (JOptionPane.YES_OPTION != result) {
			return;
		}

		TemplateEditor theEditor = templateEditors.get(theItem.key);
		if (null != theEditor) {
			if (theEditor.isVisible() && !theEditor.closeWithoutSave()) {
				errorReporter.reportWarning("Could not close editor window for the template\n" +
					"Please close the window manually and try again");
				theEditor.toFront();
				return;
			}
			templateEditors.remove(theItem.key);
		}

		Template.deleteTemplate(dbID, theItem.key.intValue(), errorReporter);

		updateTemplateList(false);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Display a template info dialog, shows study key and various other information not directly visible in UI.

	private void doTemplateInfo() {

		int rowIndex = templateTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		String title = "Template Info";
		errorReporter.setTitle(title);

		TemplateListItem theItem = checkTemplate(templateTable.convertRowIndexToModel(rowIndex), errorReporter);
		if (null == theItem) {
			return;
		}

		StringBuilder mesg = new StringBuilder();

		mesg.append("Name: ");
		mesg.append(theItem.name);
		mesg.append('\n');
		mesg.append("Primary key: ");
		mesg.append(String.valueOf(theItem.key));
		mesg.append('\n');
		mesg.append("Permanent: ");
		mesg.append(theItem.isPermanent ? "Yes" : "No");
		mesg.append('\n');
		mesg.append("Locked: ");
		mesg.append(theItem.isLocked ? "Yes" : "No");
		mesg.append('\n');
		mesg.append("Study locked: ");
		mesg.append(theItem.isLockedInStudy ? "Yes" : "No");
		mesg.append('\n');
		mesg.append("Use count: ");
		mesg.append(String.valueOf(theItem.useCount));
		mesg.append('\n');

		AppController.showMessage(this, mesg.toString(), "Template Info");
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Called by editors after a save or other operation that might have changed the information being displayed.
	// Not bothering to confirm editor identity, this is only a UI state update so harmless in any case.

	public boolean applyEditsFrom(AppEditor theEditor) {

		if (theEditor instanceof TemplateEditor) {
			int rowIndex = templateModel.indexOfKey(Integer.valueOf(((TemplateEditor)theEditor).getTemplateKey()));
			if (rowIndex >= 0) {
				checkTemplate(rowIndex, null);
				updateControls();
			}
			return true;
		}

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Sent by template editors when they close.

	public void editorClosing(AppEditor theEditor) {

		if (theEditor instanceof TemplateEditor) {
			templateEditors.remove(Integer.valueOf(((TemplateEditor)theEditor).getTemplateKey()),
				(TemplateEditor)theEditor);
			updateControls();
			return;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillOpen() {

		if (isVisible()) {
			return;
		}

		DbCore.openDb(dbID, this);

		DbController.restoreColumnWidths(dbID, getKeyTitle(), templateTable);

		blockActionsClear();

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				errorReporter.setTitle("Load Template List");
				templateModel.setItems(getItems(errorReporter));
			}
		});
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean windowShouldClose() {

		if (!isVisible()) {
			return true;
		}

		for (TemplateEditor theEditor : templateEditors.values()) {
			if (!theEditor.closeIfPossible()) {
				return false;
			}
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillClose() {

		if (!isVisible()) {
			return;
		}

		DbController.saveColumnWidths(dbID, getKeyTitle(), templateTable);

		blockActionsSet();

		managers.remove(dbID);
		DbCore.closeDb(dbID, this);
	}
}
