//
//  RecordFindPanel.java
//  TVStudy
//
//  Copyright (c) 2012-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;
import gov.fcc.tvstudy.core.geo.*;
import gov.fcc.tvstudy.gui.editor.*;

import java.util.*;
import java.sql.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.table.*;
import javax.swing.filechooser.*;


//=====================================================================================================================
// Panel UI for finding and creating records.  Searches can be made on external station data sets or the custom user
// record table.  Search can target a specific record by record ID or file number.  A general search can match any
// combination of facility ID, service, call sign, channel, status, city, and state.  User can choose from multiple
// results displayed in a table, and preview those in a SourceEditor.  New records can also be created, either all-new
// or from duplicating another record.  When a new record is created it may be made into a permanent user record.  A
// default data set can be specified if appropriate, that will be the default in the menu of available data sets.  The
// legal channel range for a search may also be restricted.  See the RecordFind class for typical use.

public class RecordFindPanel extends AppPanel implements ExtDbListener {

	// Configurable properties, see set methods.  These can only be changed when the panel is not visible.

	private int studyType;
	private int recordType;
	private boolean userRecordsOnly;

	private Integer defaultExtDbKey;

	private int minimumTVChannel;
	private int maximumTVChannel;

	private StudyEditData study;

	// UI components.

	private KeyedRecordMenu searchRecordTypeMenu;
	private KeyedRecordMenu extDbMenu;
	private JCheckBox baselineCheckBox;
	private ExtDb extDb;
	private int searchRecordType;
	private boolean useBaseline;

	private JTextField recordIDField;
	private JTextField fileNumberField;
	private JPanel fileNumberPanel;

	private JTextField callSignField;
	private JPanel callSignPanel;
	private JTextField channelField;
	private KeyedRecordMenu statusMenu;
	private KeyedRecordMenu serviceMenu;
	private JTextField cityField;
	private JTextField stateField;
	private KeyedRecordMenu countryMenu;
	private JTextField facilityIDField;
	private JCheckBox includeArchivedCheckBox;
	private JCheckBox includeSharingGuestsCheckBox;

	private JTextArea additionalSQLTextArea;

	private JCheckBox radiusSearchCheckBox;
	private CoordinatePanel latitudePanel;
	private CoordinatePanel longitudePanel;
	private CoordinateCopyPastePanel copyPastePanel;
	private GeoPoint searchCenter;
	private JTextField radiusField;
	private double searchRadius;

	private AppDialog moreOptionsDialog;
	private JButton moreOptionsButton;

	private RecordListTableModel listModel;
	private JTable listTable;

	private RecordListItem selectedItem;
	private int selectedItemIndex = -1;

	private JLabel noteLabel;
	private JPanel notePanel;

	// Buttons and menu items.

	private boolean canViewEdit;

	private boolean showUserDelete;

	private KeyedRecordMenu newRecordMenu;
	private KeyedRecordMenu importMenu;

	private JButton searchButton;
	private JButton duplicateButton;
	private JButton exportButton;
	private JButton saveButton;
	private JButton deleteButton;
	private JButton openButton;

	private JMenuItem duplicateMenuItem;
	private JMenuItem exportMenuItem;
	private JMenuItem saveMenuItem;
	private JMenuItem deleteMenuItem;
	private JMenuItem openMenuItem;
	private JMenuItem compareMenuItem;

	// Disambiguation.

	private RecordFindPanel outerThis = this;


	//-----------------------------------------------------------------------------------------------------------------
	// The study type restricts the data set menu to sets providing record types allowed in that type of study, also
	// the options for creating new records are similarily restricted, and records imported from XML files must be of
	// an allowed type.  However the study type may be 0 to show and allow all types.  Alternately a single record
	// type may be specified and only that type is allowed, but that also may be 0 for no restriction.  The record
	// type may also be negated which means only user records of that type may be chosen.  If both type arguments are
	// non-0 the record type takes priority, regardless of whether that record type is compatible with the specified
	// study type.  The callback will be called when the selected record changes.  If canViewEdit is false, all UI
	// related to creating, editing, and viewing records is omitted; the panel is used to search and select one record
	// from the results table, and nothing more.

	public RecordFindPanel(AppEditor theParent, int theStudyType, int theRecordType) {
		super(theParent);
		doSetup(theStudyType, theRecordType, true);
	}

	public RecordFindPanel(AppEditor theParent, Runnable theCallBack, int theStudyType, int theRecordType) {
		super(theParent, theCallBack);
		doSetup(theStudyType, theRecordType, true);
	}

	public RecordFindPanel(AppEditor theParent, Runnable theCallBack, int theRecordType, boolean theCanViewEdit) {
		super(theParent, theCallBack);
		doSetup(0, theRecordType, theCanViewEdit);
	}

	private void doSetup(int theStudyType, int theRecordType, boolean theCanViewEdit) {

		canViewEdit = theCanViewEdit;

		minimumTVChannel = SourceTV.CHANNEL_MIN;
		maximumTVChannel = SourceTV.CHANNEL_MAX;

		ArrayList<KeyedRecord> list = new ArrayList<KeyedRecord>();

		// UI for creating a new record, also set the record and study type restrictions.  If a single record type was
		// specified, or if a study type was specified that only allows a single record type, recordType is set to the
		// type and the new-record UI is a button.  Otherwise studyType is set and the UI is a menu listing the types.
		// If the record type is a negative number, only user records of that type may be selected.  For TV records the
		// alternate baseline record table may be searched.

		if (theRecordType > 0) {
			recordType = theRecordType;
		} else {
			if (theRecordType < 0) {
				recordType = -theRecordType;
				userRecordsOnly = true;
			} else {
				list.addAll(Source.getRecordTypes(theStudyType));
				if (1 == list.size()) {
					recordType = list.get(0).key;
				} else {
					studyType = theStudyType;
				}
			}
		}

		JPanel recordTypePanel = new JPanel();
		recordTypePanel.setBorder(BorderFactory.createTitledBorder("Search Record Type"));

		if (recordType > 0) {

			searchRecordType = recordType;
			recordTypePanel.add(new JLabel(Source.getRecordTypeName(recordType)));

		} else {

			searchRecordTypeMenu = new KeyedRecordMenu(list);
			searchRecordType = searchRecordTypeMenu.getSelectedKey();

			searchRecordTypeMenu.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					if (blockActions()) {
						updateSearchUI(false);
						blockActionsEnd();
					}
				}
			});

			recordTypePanel.add(searchRecordTypeMenu);
		}

		JButton newRecordButton = null, importButton = null;

		if (canViewEdit) {

			if (recordType > 0) {

				newRecordButton = new JButton("New");
				newRecordButton.setFocusable(false);
				newRecordButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doNew(recordType);
					}
				});

				importButton = new JButton("Import");
				importButton.setFocusable(false);
				importButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doImport(recordType);
					}
				});

			} else {

				list.add(0, new KeyedRecord(0, "New..."));
				newRecordMenu = new KeyedRecordMenu(list);

				newRecordMenu.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (blockActions()) {
							int theKey = newRecordMenu.getSelectedKey();
							newRecordMenu.setSelectedIndex(0);
							blockActionsEnd();
							if (theKey > 0) {
								doNew(theKey);
							}
						}
					}
				});

				list.set(0, new KeyedRecord(0, "Import..."));
				importMenu = new KeyedRecordMenu(list);

				importMenu.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (blockActions()) {
							int theKey = importMenu.getSelectedKey();
							importMenu.setSelectedIndex(0);
							blockActionsEnd();
							if (theKey > 0) {
								doImport(theKey);
							}
						}
					}
				});
			}
		}

		baselineCheckBox = new JCheckBox("Baseline");
		baselineCheckBox.setFocusable(false);
		baselineCheckBox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					updateSearchUI(false);
					blockActionsEnd();
				}
			}
		});

		recordTypePanel.add(baselineCheckBox);

		// Create the search fields.  The data set menu will be populated when the dialog is shown or when the search
		// record type is changed.

		extDbMenu = new KeyedRecordMenu();
		extDbMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXyXy"));
		extDbMenu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					updateSearchUI(false);
					blockActionsEnd();
				}
			}
		});

		JPanel dataPanel = new JPanel();
		dataPanel.setBorder(BorderFactory.createTitledBorder("Search Station Data"));
		dataPanel.add(extDbMenu);

		// Record ID.

		recordIDField = new JTextField(18);
		AppController.fixKeyBindings(recordIDField);

		JPanel recordIDPanel = new JPanel();
		recordIDPanel.setBorder(BorderFactory.createTitledBorder("Record ID"));
		recordIDPanel.add(recordIDField);

		// File number.

		fileNumberField = new JTextField(18);
		AppController.fixKeyBindings(fileNumberField);

		fileNumberPanel = new JPanel();
		fileNumberPanel.setBorder(BorderFactory.createTitledBorder("File Number"));
		fileNumberPanel.add(fileNumberField);

		// Call sign.

		callSignField = new JTextField(8);
		AppController.fixKeyBindings(callSignField);

		callSignPanel = new JPanel();
		callSignPanel.setBorder(BorderFactory.createTitledBorder("Call Sign"));
		callSignPanel.add(callSignField);

		// Channel.

		channelField = new JTextField(5);
		AppController.fixKeyBindings(channelField);

		JPanel channelPanel = new JPanel();
		channelPanel.setBorder(BorderFactory.createTitledBorder("Channel"));
		channelPanel.add(channelField);

		// Status.

		list.clear();
		list.add(new KeyedRecord(-1, "(any)"));
		list.addAll(ExtDbRecord.getStatusList());
		statusMenu = new KeyedRecordMenu(list);

		JPanel statusPanel = new JPanel();
		statusPanel.setBorder(BorderFactory.createTitledBorder("Status"));
		statusPanel.add(statusMenu);

		// Service, this will be updated per record type when the data set is changed, see updateSearchUI().

		list.clear();
		list.add(new KeyedRecord(0, "(any)"));
		serviceMenu = new KeyedRecordMenu(list);

		JPanel servicePanel = new JPanel();
		servicePanel.setBorder(BorderFactory.createTitledBorder("Service"));
		servicePanel.add(serviceMenu);

		// City.

		cityField = new JTextField(18);
		AppController.fixKeyBindings(cityField);

		JPanel cityPanel = new JPanel();
		cityPanel.setBorder(BorderFactory.createTitledBorder("City"));
		cityPanel.add(cityField);

		// State.

		stateField = new JTextField(4);
		AppController.fixKeyBindings(stateField);

		JPanel statePanel = new JPanel();
		statePanel.setBorder(BorderFactory.createTitledBorder("State"));
		statePanel.add(stateField);

		// Country.

		list.clear();
		list.add(new KeyedRecord(0, "(any)"));
		list.addAll(Country.getCountries());
		countryMenu = new KeyedRecordMenu(list);

		JPanel countryPanel = new JPanel();
		countryPanel.setBorder(BorderFactory.createTitledBorder("Country"));
		countryPanel.add(countryMenu);

		// Facility ID.

		facilityIDField = new JTextField(7);
		AppController.fixKeyBindings(facilityIDField);

		JPanel facilityIDPanel = new JPanel();
		facilityIDPanel.setBorder(BorderFactory.createTitledBorder("Facility ID"));
		facilityIDPanel.add(facilityIDField);

		// External data search is normally just for current and pending records, option to include archived.

		includeArchivedCheckBox = new JCheckBox("Include archived");
		includeArchivedCheckBox.setFocusable(false);

		// Include sharing guest records in an LMS TV search only, this is hidden in any other case.  Because of the
		// way this property is implemented in ExtDbRecordTV it needs a passed argument, can't be done in the SQL.

		includeSharingGuestsCheckBox = new JCheckBox("Include sharing guests");
		includeSharingGuestsCheckBox.setFocusable(false);

		// Text area for input of additional SQL appended to the WHERE clause.

		additionalSQLTextArea = new JTextArea(5, 40);
		AppController.fixKeyBindings(additionalSQLTextArea);
		additionalSQLTextArea.setLineWrap(true);
		additionalSQLTextArea.setWrapStyleWord(true);

		JPanel addSQLPanel = new JPanel();
		addSQLPanel.setBorder(BorderFactory.createTitledBorder("Additional SQL for WHERE clause"));
		addSQLPanel.add(AppController.createScrollPane(additionalSQLTextArea));

		// Search by center point and radius.

		radiusSearchCheckBox = new JCheckBox("Search by center point and radius");
		radiusSearchCheckBox.setFocusable(false);
		radiusSearchCheckBox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				boolean e = radiusSearchCheckBox.isSelected();
				latitudePanel.setEnabled(e);
				longitudePanel.setEnabled(e);
				copyPastePanel.setEnabled(e);
				AppController.setComponentEnabled(radiusField, e);
			}
		});

		searchCenter = new GeoPoint();

		latitudePanel = new CoordinatePanel(this, searchCenter, false, null);

		longitudePanel = new CoordinatePanel(this, searchCenter, true, null);

		copyPastePanel = new CoordinateCopyPastePanel(latitudePanel, longitudePanel);

		radiusField = new JTextField(7);
		AppController.fixKeyBindings(radiusField);

		JPanel radiusPanel = new JPanel();
		radiusPanel.setBorder(BorderFactory.createTitledBorder("Radius, km"));
		radiusPanel.add(radiusField);

		JPanel radiusSearchPanel = new JPanel();
		radiusSearchPanel.setLayout(new BoxLayout(radiusSearchPanel, BoxLayout.Y_AXIS));
		radiusSearchPanel.setBorder(BorderFactory.createEtchedBorder());

		JPanel thePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		thePanel.add(radiusSearchCheckBox);
		radiusSearchPanel.add(thePanel);

		thePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		thePanel.add(latitudePanel);
		thePanel.add(copyPastePanel);
		radiusSearchPanel.add(thePanel);

		thePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		thePanel.add(longitudePanel);
		radiusSearchPanel.add(thePanel);

		thePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		thePanel.add(radiusPanel);
		radiusSearchPanel.add(thePanel);

		// Less-used options appear in a pop-up dialog, do the layout for that.  Dialog is created on demand.

		JPanel recIDPanel = new JPanel();
		recIDPanel.add(recordIDPanel);

		JPanel ctyStCntPanel = new JPanel();
		ctyStCntPanel.add(cityPanel);
		ctyStCntPanel.add(statePanel);
		ctyStCntPanel.add(countryPanel);

		JPanel radPanel = new JPanel();
		radPanel.add(radiusSearchPanel);

		JPanel moreMainP = new JPanel();
		moreMainP.setLayout(new BoxLayout(moreMainP, BoxLayout.Y_AXIS));
		moreMainP.add(ctyStCntPanel);
		moreMainP.add(radPanel);
		moreMainP.add(recIDPanel);
		moreMainP.add(addSQLPanel);

		JButton optsOKBut = new JButton("OK");
		optsOKBut.setFocusable(false);
		optsOKBut.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if ((null != moreOptionsDialog) && moreOptionsDialog.isVisible()) {
					AppController.hideWindow(moreOptionsDialog);
				}
			}
		});

		JPanel moreButP = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		moreButP.add(optsOKBut);

		moreOptionsDialog = new AppDialog(getOriginalParent(), "Additional Options",
				Dialog.ModalityType.APPLICATION_MODAL) {

			public void windowWillOpen() {
				blockActionsClear();
			}

			public void windowWillClose() {
				blockActionsSet();
			}
		};

		moreOptionsDialog.add(moreMainP, BorderLayout.CENTER);
		moreOptionsDialog.add(moreButP, BorderLayout.SOUTH);

		moreOptionsDialog.pack();

		moreOptionsDialog.setMinimumSize(moreOptionsDialog.getSize());

		latitudePanel.setParent(moreOptionsDialog);
		longitudePanel.setParent(moreOptionsDialog);
		copyPastePanel.setParent(moreOptionsDialog);

		moreOptionsButton = new JButton("More");
		moreOptionsButton.setFocusable(false);
		moreOptionsButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (!moreOptionsDialog.isVisible()) {
					moreOptionsDialog.setLocationRelativeTo(moreOptionsButton);
					AppController.showWindow(moreOptionsDialog);
				}
			}
		});

		// Table to display list of matching records.

		JPanel listTablePanel = new JPanel(new BorderLayout());
		listModel = new RecordListTableModel(listTablePanel);
		listTable = listModel.createTable();

		if (canViewEdit) {
			listTable.addMouseListener(new MouseAdapter() {
				public void mouseClicked(MouseEvent e) {
					if (2 == e.getClickCount()) {
						doOpen();
					}
				}
			});
		}

		listTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
			public void valueChanged(ListSelectionEvent theEvent) {
				if (!theEvent.getValueIsAdjusting()) {
					if (1 == listTable.getSelectedRowCount()) {
						selectedItemIndex = listTable.convertRowIndexToModel(listTable.getSelectedRow());
						selectedItem = listModel.get(selectedItemIndex);
					} else {
						selectedItemIndex = -1;
						selectedItem = null;
					}
					updateControls();
					if (null != callBack) {
						callBack.run();
					}
				}
			}
		});

		listTablePanel.add(AppController.createScrollPane(listTable, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
				JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS), BorderLayout.CENTER);
		listTablePanel.add(listModel.filterPanel, BorderLayout.SOUTH);

		// See setNote().

		noteLabel = new JLabel();
		notePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		notePanel.add(noteLabel);
		notePanel.setVisible(false);

		// Property may hide ability to delete user records.

		String str = AppCore.getPreference(AppCore.CONFIG_HIDE_USER_RECORD_DELETE);
		showUserDelete = !((null != str) && Boolean.valueOf(str).booleanValue());

		// Buttons.

		searchButton = new JButton("Search");
		searchButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doSearch();
			}
		});

		JButton clearButton = new JButton("Clear");
		clearButton.setFocusable(false);
		clearButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doClear(true);
			}
		});

		if (canViewEdit) {

			duplicateButton = new JButton("Duplicate");
			duplicateButton.setFocusable(false);
			duplicateButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doDuplicate();
				}
			});

			exportButton = new JButton("Export");
			exportButton.setFocusable(false);
			exportButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doExport();
				}
			});

			saveButton = new JButton("Save");
			saveButton.setFocusable(false);
			saveButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doSave();
				}
			});

			if (showUserDelete) {
				deleteButton = new JButton("Delete");
				deleteButton.setFocusable(false);
				deleteButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doDelete();
					}
				});
			}

			openButton = new JButton("View");
			openButton.setFocusable(false);
			openButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doOpen();
				}
			});
		}

		// Do the layout.

		JPanel fileNumPanel = new JPanel();
		fileNumPanel.add(fileNumberPanel);

		JPanel callFacPanel = new JPanel();
		callFacPanel.add(callSignPanel);
		callFacPanel.add(facilityIDPanel);

		JPanel chanStatPanel = new JPanel();
		chanStatPanel.add(channelPanel);
		chanStatPanel.add(statusPanel);

		JPanel servPanel = new JPanel();
		servPanel.add(servicePanel);

		Box inclBox = Box.createVerticalBox();
		inclBox.add(includeArchivedCheckBox);
		inclBox.add(includeSharingGuestsCheckBox);
		JPanel inclMorePanel = new JPanel();
		inclMorePanel.add(inclBox);
		inclMorePanel.add(moreOptionsButton);

		Box searchBox = Box.createVerticalBox();
		searchBox.add(callFacPanel);
		searchBox.add(chanStatPanel);
		searchBox.add(fileNumPanel);
		searchBox.add(servPanel);
		searchBox.add(inclMorePanel);

		JPanel allPanel = new JPanel();
		allPanel.add(searchBox);

		JPanel searchButtonPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		searchButtonPanel.add(clearButton);
		searchButtonPanel.add(searchButton);

		Box topBox = Box.createVerticalBox();
		topBox.add(recordTypePanel);
		topBox.add(dataPanel);

		JPanel searchPanel = new JPanel(new BorderLayout());
		searchPanel.add(topBox, BorderLayout.NORTH);
		searchPanel.add(AppController.createScrollPane(allPanel), BorderLayout.CENTER);
		searchPanel.add(searchButtonPanel, BorderLayout.SOUTH);

		JPanel listPanel = null;

		if (canViewEdit) {

			JPanel listButtonLeftPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			listButtonLeftPanel.add(openButton);
			listButtonLeftPanel.add(duplicateButton);
			listButtonLeftPanel.add(exportButton);
			listButtonLeftPanel.add(saveButton);
			if (showUserDelete) {
				listButtonLeftPanel.add(deleteButton);
			}

			JPanel listButtonRightPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
			if (null != importMenu) {
				listButtonRightPanel.add(importMenu);
			} else {
				if (null != importButton) {
					listButtonRightPanel.add(importButton);
				}
			}
			if (null != newRecordMenu) {
				listButtonRightPanel.add(newRecordMenu);
			} else {
				if (null != newRecordButton) {
					listButtonRightPanel.add(newRecordButton);
				}
			}

			Box listButtonBox = Box.createHorizontalBox();
			listButtonBox.add(listButtonLeftPanel);
			listButtonBox.add(listButtonRightPanel);

			listPanel = new JPanel(new BorderLayout());
			listPanel.add(listTablePanel, BorderLayout.CENTER);
			listPanel.add(listButtonBox, BorderLayout.SOUTH);

		} else {

			listPanel = listTablePanel;
		}

		setLayout(new BorderLayout());
		add(notePanel, BorderLayout.NORTH);
		add(listPanel, BorderLayout.CENTER);
		add(searchPanel, BorderLayout.EAST);

		// Create persistent menu items, regardless of whether or not they will be used, see addMenuItems().  This is
		// easier than checking for null all the time in the state updater.

		if (canViewEdit) {

			duplicateMenuItem = new JMenuItem("Duplicate");
			duplicateMenuItem.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doDuplicate();
				}
			});

			openMenuItem = new JMenuItem("View");
			openMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, AppController.MENU_SHORTCUT_KEY_MASK));
			openMenuItem.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doOpen();
				}
			});

			compareMenuItem = new JMenuItem("Compare");
			compareMenuItem.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					if (null == selectedItem) {
						return;
					}
					if (selectedItem.isSource) {
						SourceCompare.compare((SourceEditData)(selectedItem.record), errorReporter);
					} else {
						SourceCompare.compare((ExtDbRecord)(selectedItem.record), errorReporter);
					}
				}
			});

			exportMenuItem = new JMenuItem("Export...");
			exportMenuItem.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doExport();
				}
			});

			saveMenuItem = new JMenuItem("Save");
			saveMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, AppController.MENU_SHORTCUT_KEY_MASK));
			saveMenuItem.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doSave();
				}
			});

			if (showUserDelete) {
				deleteMenuItem = new JMenuItem("Delete");
				deleteMenuItem.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doDelete();
					}
				});
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Return button for presenting window to set as root pane default.

	public JButton getDefaultButton() {

		return searchButton;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add items to a menu from a presenting window.

	public void addMenuItems(JMenu fileMenu) {

		// Previous

		JMenuItem miPrevious = new JMenuItem("Previous");
		miPrevious.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_UP, AppController.MENU_SHORTCUT_KEY_MASK));
		miPrevious.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doPrevious();
			}
		});
		fileMenu.add(miPrevious);

		// Next

		JMenuItem miNext = new JMenuItem("Next");
		miNext.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, AppController.MENU_SHORTCUT_KEY_MASK));
		miNext.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doNext();
			}
		});
		fileMenu.add(miNext);

		if (!canViewEdit) {
			return;
		}

		// __________________________________

		fileMenu.addSeparator();

		// New ? [ -> ]
		// Import ? [ -> ]

		if (recordType > 0) {

			JMenuItem miNew = new JMenuItem("New");
			miNew.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, AppController.MENU_SHORTCUT_KEY_MASK));
			miNew.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doNew(recordType);
				}
			});
			fileMenu.add(miNew);

			JMenuItem miImport = new JMenuItem("Import");
			miImport.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doImport(recordType);
				}
			});
			fileMenu.add(miImport);

		} else {

			JMenu meNew = new JMenu("New");
			JMenuItem miNew;
			for (KeyedRecord theType : Source.getRecordTypes(studyType)) {
				miNew = new JMenuItem(theType.name);
				final int typeKey = theType.key;
				miNew.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doNew(typeKey);
					}
				});
				meNew.add(miNew);
			}
			fileMenu.add(meNew);

			JMenu meImport = new JMenu("Import");
			JMenuItem miImport;
			for (KeyedRecord theType : Source.getRecordTypes(studyType)) {
				miImport = new JMenuItem(theType.name);
				final int typeKey = theType.key;
				miImport.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doImport(typeKey);
					}
				});
				meImport.add(miImport);
			}
			fileMenu.add(meImport);
		}

		// Duplicate

		fileMenu.add(duplicateMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// View/Edit

		fileMenu.add(openMenuItem);

		// Save

		fileMenu.add(saveMenuItem);

		// Delete

		if (showUserDelete) {
			fileMenu.add(deleteMenuItem);
		}

		// __________________________________

		fileMenu.addSeparator();

		// Compare

		fileMenu.add(compareMenuItem);

		// Export...

		fileMenu.add(exportMenuItem);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set default station data.  This and all the other configurable properties can only be changed while hidden.

	public void setDefaultExtDbKey(Integer theKey) {

		if (getWindow().isVisible()) {
			return;
		}

		defaultExtDbKey = theKey;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set the TV channel range for record searches (FM searches are never restricted by channel).  If either value is
	// invalid it reverts to the default maximum range, clearing the range may be accomplished by setting 0,0.  Note
	// reversing the arguments is allowed.

	public void setTVChannelRange(int theMinChannel, int theMaxChannel) {

		if (getWindow().isVisible()) {
			return;
		}

		if ((theMinChannel < SourceTV.CHANNEL_MIN) || (theMinChannel > SourceTV.CHANNEL_MAX)) {
			theMinChannel = SourceTV.CHANNEL_MIN;
		}

		if ((theMaxChannel < SourceTV.CHANNEL_MIN) || (theMaxChannel > SourceTV.CHANNEL_MAX)) {
			theMaxChannel = SourceTV.CHANNEL_MAX;
		}

		if (theMinChannel <= theMaxChannel) {
			minimumTVChannel = theMinChannel;
			maximumTVChannel = theMaxChannel;
		} else {
			minimumTVChannel = theMaxChannel;
			maximumTVChannel = theMinChannel;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set note text appearing a label at the top of the layout.  If null or empty string the label is hidden.

	public void setNote(String theNote) {

		if ((null == theNote) || (0 == theNote.length())) {
			noteLabel.setText("");
			notePanel.setVisible(false);
		} else {
			noteLabel.setText(theNote);
			notePanel.setVisible(true);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set a study context, this will be used to obtain study parameter values as needed e.g. for bearing and distance
	// calculations.  If not set, parameter values are loaded from the default template.

	public void setStudy(StudyEditData theStudy) {

		study = theStudy;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void updateDocumentName() {

		listModel.updateEditorDocumentNames();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Public methods for presenting window to access the selection.  The "apply" concept means that window tells some
	// outer editing context to actually do something with the record.  That can only occur if the selection is valid,
	// but some contexts will accept a new (unsaved) record and some will not so caller must specify.  Note even if a
	// record is new and new can be applied, it can't be applied if it has an open editor since that may be holding
	// un-committed editor state.  The caller can reply back when a record has been applied, that will affect the
	// behavior on window close, see windowWillClose().

	public StationRecord getSelectedRecord() {

		if (null != selectedItem) {
			return selectedItem.record;
		}
		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean canApplySelection(boolean canApplyNew) {

		if (null != selectedItem) {
			 return (selectedItem.isValid &&
				(!selectedItem.isNew || (canApplyNew && (null == selectedItem.sourceEditor))));
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void selectionWasApplied() {

		if (null != selectedItem) {
			selectedItem.wasApplied = true;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Used when a record may not currently be the selection but was a previous one and might still be in the list due
	// to being an unlocked new source; those are kept even when the list is otherwise cleared.

	public boolean canApplyRecord(StationRecord theRecord, boolean canApplyNew) {

		RecordListItem theItem = listModel.findItemForRecord(theRecord);
		if (null != theItem) {
			 return (theItem.isValid && (!theItem.isNew || (canApplyNew && (null == theItem.sourceEditor))));
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void recordWasApplied(StationRecord theRecord) {

		RecordListItem theItem = listModel.findItemForRecord(theRecord);
		if (null != theItem) {
			theItem.wasApplied = true;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update the UI when record type or data set selection is changed, adapting to the new record type as needed which
	// may include refreshing the data set menu contents.  Search criteria that don't apply to the new record type are
	// cleared and disabled, also the service menu is updated.  Any content in the record ID or additional SQL fields
	// are also cleared, if the database type actually changes.  Likewise the previous search results are cleared if
	// there is any actual change here.  If the firstUpdate flag is true this ignores existing state and updates as if
	// everything changed.

	private void updateSearchUI(boolean firstUpdate) {

		boolean clearResults = false;

		int newRecordType = searchRecordType;
		if (null != searchRecordTypeMenu) {
			newRecordType = searchRecordTypeMenu.getSelectedKey();
			if (newRecordType != searchRecordType) {
				ArrayList<KeyedRecord> list = getExtDbMenuList(newRecordType, null);
				if (null != list) {
					int selectKey = extDbMenu.getSelectedKey();
					extDbMenu.removeAllItems();
					extDbMenu.addAllItems(list);
					if (extDbMenu.containsKey(selectKey)) {
						extDbMenu.setSelectedKey(selectKey);
					}
				}
			}
		}

		// The key value on "* user records" menu items is the record type negated.  Note because of the "most recent"
		// functions, the key from the menu selection may be translated by the lookup.

		int oldDbKey = -searchRecordType, oldDbType = ExtDb.DB_TYPE_NOT_SET;
		if (null != extDb) {
			oldDbKey = extDb.key.intValue();
			oldDbType = extDb.type;
		}

		int newDbKey = extDbMenu.getSelectedKey(), newDbType = ExtDb.DB_TYPE_NOT_SET;
		ExtDb newExtDb = null;
		if (newDbKey > 0) {
			newExtDb = ExtDb.getExtDb(getDbID(), Integer.valueOf(newDbKey), errorReporter);
			if (null == newExtDb) {
				extDbMenu.setSelectedKey(oldDbKey);
				newExtDb = extDb;
				newDbKey = oldDbKey;
				newDbType = oldDbType;
			} else {
				newDbKey = newExtDb.key.intValue();
				newDbType = newExtDb.type;
			}
		} else {
			if (newDbKey < 0) {
				newRecordType = -newDbKey;
			} else {
				newDbKey = -newRecordType;
			}
		}

		if ((newDbKey != oldDbKey) || (newRecordType != searchRecordType) || firstUpdate) {

			extDb = newExtDb;
			clearResults = true;

			if ((newDbType != oldDbType) || firstUpdate) {
				recordIDField.setText("");
				additionalSQLTextArea.setText("");
			}

			if ((Source.RECORD_TYPE_TV == newRecordType) && (null != extDb) && extDb.hasBaseline()) {
				AppController.setComponentEnabled(baselineCheckBox, true);
			} else {
				baselineCheckBox.setSelected(false);
				AppController.setComponentEnabled(baselineCheckBox, false);
			}

			// User record searches don't have the center-and-radius option.

			if (null == extDb) {
				radiusSearchCheckBox.setSelected(false);
				AppController.setComponentEnabled(radiusSearchCheckBox, false);
				latitudePanel.setEnabled(false);
				longitudePanel.setEnabled(false);
				copyPastePanel.setEnabled(false);
				AppController.setComponentEnabled(radiusField, false);
			} else {
				AppController.setComponentEnabled(radiusSearchCheckBox, true);
				boolean e = radiusSearchCheckBox.isSelected();
				latitudePanel.setEnabled(e);
				longitudePanel.setEnabled(e);
				copyPastePanel.setEnabled(e);
				AppController.setComponentEnabled(radiusField, e);
			}
		}

		boolean newBaseline = baselineCheckBox.isSelected();

		if ((newRecordType != searchRecordType) || (newBaseline != useBaseline) || firstUpdate) {

			searchRecordType = newRecordType;
			useBaseline = newBaseline;
			clearResults = true;

			ArrayList<KeyedRecord> list = new ArrayList<KeyedRecord>();
			list.add(new KeyedRecord(0, "(any)"));
			list.addAll(Service.getServices(searchRecordType));
			serviceMenu.removeAllItems();
			serviceMenu.addAllItems(list);

			switch (searchRecordType) {

				case Source.RECORD_TYPE_TV:
				default: {

					fileNumberPanel.setBorder(BorderFactory.createTitledBorder("File Number"));
					callSignPanel.setBorder(BorderFactory.createTitledBorder("Call Sign"));

					AppController.setComponentEnabled(channelField, true);
					AppController.setComponentEnabled(facilityIDField, true);

					if (useBaseline) {

						fileNumberField.setText("");
						statusMenu.setSelectedIndex(0);
						includeArchivedCheckBox.setSelected(false);
						includeSharingGuestsCheckBox.setSelected(false);

						AppController.setComponentEnabled(fileNumberField, false);
						AppController.setComponentEnabled(statusMenu, false);
						AppController.setComponentEnabled(includeArchivedCheckBox, false);
						AppController.setComponentEnabled(includeSharingGuestsCheckBox, false);

						if (null != extDb) {

							if (ExtDb.DB_TYPE_CDBS == extDb.type) {

								serviceMenu.setSelectedIndex(0);
								countryMenu.setSelectedIndex(0);

								AppController.setComponentEnabled(serviceMenu, false);
								AppController.setComponentEnabled(countryMenu, false);

							} else {

								AppController.setComponentEnabled(serviceMenu, true);
								AppController.setComponentEnabled(countryMenu, true);
							}
						}

					} else {

						AppController.setComponentEnabled(fileNumberField, true);
						AppController.setComponentEnabled(statusMenu, true);
						AppController.setComponentEnabled(serviceMenu, true);
						AppController.setComponentEnabled(includeArchivedCheckBox, true);
						AppController.setComponentEnabled(includeSharingGuestsCheckBox, true);
						AppController.setComponentEnabled(countryMenu, true);
					}

					break;
				}

				case Source.RECORD_TYPE_WL: {

					fileNumberPanel.setBorder(BorderFactory.createTitledBorder("Reference Number"));
					callSignPanel.setBorder(BorderFactory.createTitledBorder("Cell Site ID"));

					channelField.setText("");
					statusMenu.setSelectedIndex(0);
					facilityIDField.setText("");
					includeArchivedCheckBox.setSelected(false);
					includeSharingGuestsCheckBox.setSelected(false);

					AppController.setComponentEnabled(fileNumberField, true);
					AppController.setComponentEnabled(channelField, false);
					AppController.setComponentEnabled(statusMenu, false);
					AppController.setComponentEnabled(serviceMenu, true);
					AppController.setComponentEnabled(facilityIDField, false);
					AppController.setComponentEnabled(includeArchivedCheckBox, false);
					AppController.setComponentEnabled(includeSharingGuestsCheckBox, false);
					AppController.setComponentEnabled(countryMenu, true);

					break;
				}

				case Source.RECORD_TYPE_FM: {

					fileNumberPanel.setBorder(BorderFactory.createTitledBorder("File Number"));
					callSignPanel.setBorder(BorderFactory.createTitledBorder("Call Sign"));
					includeSharingGuestsCheckBox.setSelected(false);

					AppController.setComponentEnabled(fileNumberField, true);
					AppController.setComponentEnabled(channelField, true);
					AppController.setComponentEnabled(statusMenu, true);
					AppController.setComponentEnabled(serviceMenu, true);
					AppController.setComponentEnabled(facilityIDField, true);
					AppController.setComponentEnabled(includeArchivedCheckBox, true);
					AppController.setComponentEnabled(includeSharingGuestsCheckBox, false);
					AppController.setComponentEnabled(countryMenu, true);

					break;
				}
			}
		}

		if (clearResults) {
			listModel.setItems(null, true);
		}
	}


	//=================================================================================================================
	// Item class for table list model.  A SourceEditor may be open for any/many records in the list.  The isSource,
	// isUserRecord, and isNew flags are convenience.  The isValid flag is true for most records.  It is only false
	// temporarily on completely new records while the initial SourceEditor is open, there it will be set true in
	// applyEditsFrom(), SourceEditor will not apply edits until data is valid.  If that initial editor is canceled
	// the invalid new record is removed again in editorClosing().  The wasApplied flag is set on any new record when
	// it is applied to the parent, and is cleared again if the record is edited again, see hasUnsavedData().  The
	// comment text for source objects may be set to a user record comment, on data set records it may be set to some
	// useful accessory information from the record.  It appears as a pop-up in the results table.  When records are
	// imported from XML, isNew will be true on a record that is not locked, but in that case isImport is also true.
	// If a new imported record is actually edited isImport is set false.  That is so new imported records that have
	// not been edited do not trigger warning messages about unsaved changes, since there really aren't any.

	private class RecordListItem {

		private StationRecord record;

		private boolean isSource;
		private boolean isUserRecord;
		private boolean isNew;
		private boolean isImport;
		private boolean isValid;
		private boolean wasApplied;

		private String comment;

		private SourceEditor sourceEditor;
	}


	//=================================================================================================================
	// Table model for list of search results.

	private class RecordListTableModel extends AbstractTableModel implements TableFilterModel {

		private static final String RECORD_TYPE_COLUMN = "Type";
		private static final String RECORD_CALLSIGN_COLUMN = "Call Sign";
		private static final String RECORD_CALLSIGN_COLUMN_WL = "Call/ID";
		private static final String RECORD_CHANNEL_COLUMN = "Channel";
		private static final String RECORD_SERVICE_COLUMN = "Svc";
		private static final String RECORD_STATUS_COLUMN = "Status";
		private static final String RECORD_CITY_COLUMN = "City";
		private static final String RECORD_STATE_COLUMN = "State";
		private static final String RECORD_COUNTRY_COLUMN = "Cntry";
		private static final String RECORD_FACILITY_ID_COLUMN = "Facility ID";
		private static final String RECORD_FILE_COLUMN = "File Number";
		private static final String RECORD_FILE_COLUMN_WL = "File/Ref Number";
		private static final String RECORD_DATE_COLUMN = "Date";

		private String[] columnNamesNoWL = {
			RECORD_TYPE_COLUMN,
			RECORD_CALLSIGN_COLUMN,
			RECORD_CHANNEL_COLUMN,
			RECORD_SERVICE_COLUMN,
			RECORD_STATUS_COLUMN,
			RECORD_CITY_COLUMN,
			RECORD_STATE_COLUMN,
			RECORD_COUNTRY_COLUMN,
			RECORD_FACILITY_ID_COLUMN,
			RECORD_FILE_COLUMN,
			RECORD_DATE_COLUMN
		};

		private String[] columnNamesWL = {
			RECORD_TYPE_COLUMN,
			RECORD_CALLSIGN_COLUMN_WL,
			RECORD_CHANNEL_COLUMN,
			RECORD_SERVICE_COLUMN,
			RECORD_STATUS_COLUMN,
			RECORD_CITY_COLUMN,
			RECORD_STATE_COLUMN,
			RECORD_COUNTRY_COLUMN,
			RECORD_FACILITY_ID_COLUMN,
			RECORD_FILE_COLUMN_WL,
			RECORD_DATE_COLUMN
		};

		private String[] columnNames;

		private static final int RECORD_TYPE_INDEX = 0;
		private static final int RECORD_CALLSIGN_INDEX = 1;
		private static final int RECORD_CHANNEL_INDEX = 2;
		private static final int RECORD_SERVICE_INDEX = 3;
		private static final int RECORD_STATUS_INDEX = 4;
		private static final int RECORD_CITY_INDEX = 5;
		private static final int RECORD_STATE_INDEX = 6;
		private static final int RECORD_COUNTRY_INDEX = 7;
		private static final int RECORD_FACILITY_ID_INDEX = 8;
		private static final int RECORD_FILE_INDEX = 9;
		private static final int RECORD_DATE_INDEX = 10;

		private JPanel panel;
		private ArrayList<RecordListItem> modelRows;

		private TableFilterPanel filterPanel;


		//-------------------------------------------------------------------------------------------------------------

		private RecordListTableModel(JPanel thePanel) {

			super();

			if (((0 == recordType) || (Source.RECORD_TYPE_WL == recordType)) &&
					((0 == studyType) || Study.isRecordTypeAllowed(studyType, Source.RECORD_TYPE_WL))) {
				columnNames = columnNamesWL;
			} else {
				columnNames = columnNamesNoWL;
			}

			panel = thePanel;
			modelRows = new ArrayList<RecordListItem>();

			filterPanel = new TableFilterPanel(outerThis, this);
		}


		//-------------------------------------------------------------------------------------------------------------
		// Custom sorting similar to the source listing table in the scenario editor.  Sort order is based on multiple
		// columns, some are sorted differently than natural ordering of display values so a custom string converter
		// is used.  Methods in the StationRecord interface provide the formatting, see SourceEditData and ExtDbRecord.

		private JTable createTable() {

			JTable theTable = new JTable(this);
			AppController.configureTable(theTable);

			TableRowSorter<RecordListTableModel> theSorter = new TableRowSorter<RecordListTableModel>(this) {
				public void setSortKeys(java.util.List<? extends RowSorter.SortKey> sortKeys) {

					int columnIndex = RECORD_CALLSIGN_INDEX;
					SortOrder order = SortOrder.ASCENDING;
					if ((null != sortKeys) && (sortKeys.size() > 0)) {
						RowSorter.SortKey theKey = sortKeys.get(0);
						columnIndex = theKey.getColumn();
						order = theKey.getSortOrder();
					}

					ArrayList<RowSorter.SortKey> theKeys = new ArrayList<RowSorter.SortKey>();

					switch (columnIndex) {

						case RECORD_TYPE_INDEX:   // Record type (call sign, status, channel).
							theKeys.add(new RowSorter.SortKey(RECORD_TYPE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CALLSIGN_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATUS_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;

						case RECORD_CALLSIGN_INDEX:   // Call sign (status, channel).  Default.
						default:
							theKeys.add(new RowSorter.SortKey(RECORD_CALLSIGN_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATUS_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;

						case RECORD_CHANNEL_INDEX:   // Channel (country, state, city).
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_COUNTRY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CITY_INDEX, order));
							break;

						case RECORD_SERVICE_INDEX:   // Service (country, state, city, channel).
							theKeys.add(new RowSorter.SortKey(RECORD_SERVICE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_COUNTRY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CITY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;

						case RECORD_STATUS_INDEX:   // Status (country, state, city, channel).
							theKeys.add(new RowSorter.SortKey(RECORD_STATUS_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_COUNTRY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CITY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;

						case RECORD_CITY_INDEX:   // City (state, channel).
							theKeys.add(new RowSorter.SortKey(RECORD_CITY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;

						case RECORD_STATE_INDEX:   // State (city, channel).
							theKeys.add(new RowSorter.SortKey(RECORD_STATE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CITY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;

						case RECORD_COUNTRY_INDEX:   // Country (state, city, channel).
							theKeys.add(new RowSorter.SortKey(RECORD_COUNTRY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CITY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;

						case RECORD_FACILITY_ID_INDEX:   // Facility ID (status, channel).
							theKeys.add(new RowSorter.SortKey(RECORD_FACILITY_ID_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATUS_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;

						case RECORD_FILE_INDEX:   // File number (country, state, city, channel).
							theKeys.add(new RowSorter.SortKey(RECORD_FILE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_COUNTRY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CITY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;

						case RECORD_DATE_INDEX:   // Date (country, state, city, channel).
							theKeys.add(new RowSorter.SortKey(RECORD_DATE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_COUNTRY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_STATE_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CITY_INDEX, order));
							theKeys.add(new RowSorter.SortKey(RECORD_CHANNEL_INDEX, order));
							break;
					}

					super.setSortKeys(theKeys);
				}
			};

			TableStringConverter theConverter = new TableStringConverter() {
				public String toString(TableModel theModel, int rowIndex, int columnIndex) {

					StationRecord theRecord = modelRows.get(filterPanel.forwardIndex[rowIndex]).record;

					switch (columnIndex) {

						case RECORD_TYPE_INDEX:
							return theRecord.getRecordType();

						case RECORD_CALLSIGN_INDEX:
							return theRecord.getCallSign();

						case RECORD_CHANNEL_INDEX:
							return theRecord.getSortChannel();

						case RECORD_SERVICE_INDEX:
							return theRecord.getServiceCode();

						case RECORD_STATUS_INDEX:
							return theRecord.getSortStatus();

						case RECORD_CITY_INDEX:
							return theRecord.getCity();

						case RECORD_STATE_INDEX:
							return theRecord.getState();

						case RECORD_COUNTRY_INDEX:
							return theRecord.getSortCountry();

						case RECORD_FACILITY_ID_INDEX:
							return theRecord.getSortFacilityID();

						case RECORD_FILE_INDEX:
							return theRecord.getFileNumber();

						case RECORD_DATE_INDEX:
							return theRecord.getSortSequenceDate();
					}

					return "";
				}
			};

			theSorter.setStringConverter(theConverter);
			theTable.setRowSorter(theSorter);
			theSorter.setSortKeys(null);

			// Customize the cell renderer to change color on new/invalid records, and also set any comment text to
			// appear in tool-tip pop-up over the call sign field.

			DefaultTableCellRenderer theRend = new DefaultTableCellRenderer() {
				public Component getTableCellRendererComponent(JTable t, Object o, boolean s, boolean f, int r, int c) {

					JLabel comp = (JLabel)super.getTableCellRendererComponent(t, o, s, f, r, c);

					int row = t.convertRowIndexToModel(r);
					int col = t.convertColumnIndexToModel(c);

					RecordListItem theItem = modelRows.get(filterPanel.forwardIndex[row]);

					if (!s) {
						if (theItem.isNew) {
							if (theItem.isValid) {
								comp.setForeground(Color.GREEN.darker());
							} else {
								comp.setForeground(Color.RED);
							}
						} else {
							comp.setForeground(Color.BLACK);
						}
					}

					if ((RECORD_CALLSIGN_INDEX == col) && (null != theItem.comment) && (theItem.comment.length() > 0)) {
						if (!s) {
							comp.setForeground(Color.BLUE);
						}
						comp.setToolTipText(theItem.comment);
					} else {
						comp.setToolTipText(null);
					}

					return comp;
				}
			};

			TableColumnModel columnModel = theTable.getColumnModel();

			TableColumn theColumn = columnModel.getColumn(RECORD_TYPE_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[4]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[4]);

			theColumn = columnModel.getColumn(RECORD_CALLSIGN_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[5]);

			theColumn = columnModel.getColumn(RECORD_CHANNEL_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[7]);

			theColumn = columnModel.getColumn(RECORD_SERVICE_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[2]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[2]);

			theColumn = columnModel.getColumn(RECORD_STATUS_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[4]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[4]);

			theColumn = columnModel.getColumn(RECORD_CITY_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[8]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[14]);

			theColumn = columnModel.getColumn(RECORD_STATE_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[2]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[2]);

			theColumn = columnModel.getColumn(RECORD_COUNTRY_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[2]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[2]);

			theColumn = columnModel.getColumn(RECORD_FACILITY_ID_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[5]);

			theColumn = columnModel.getColumn(RECORD_FILE_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[8]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[16]);

			theColumn = columnModel.getColumn(RECORD_DATE_INDEX);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[8]);

			theTable.setPreferredScrollableViewportSize(new Dimension(theTable.getPreferredSize().width,
				(theTable.getRowHeight() * 5)));

			return theTable;
		}


		//-------------------------------------------------------------------------------------------------------------
		// Set records appearing in the table.  Existing new records may be preserved.  Open SourceEditor dialogs for
		// existing records are canceled, including those for new records if keepNew is false so that should not be
		// false unless the user has confirmed discarding unsaved state.  See windowShouldClose().  This has to be
		// done in two passes, carefully, since canceling dialogs will call editorClosing() which will modify the
		// model here.  Also canceling a dialog theoretically could fail, currently that is impossible but it has to
		// be supported.

		private boolean setItems(ArrayList<RecordListItem> newItems, boolean keepNew) {

			ArrayList<RecordListItem> keepItems = new ArrayList<RecordListItem>();
			for (RecordListItem theItem : new ArrayList<RecordListItem>(modelRows)) {
				if (theItem.isNew && !theItem.isImport && keepNew) {
					keepItems.add(theItem);
				} else {
					if (null != theItem.sourceEditor) {
						if (theItem.sourceEditor.isVisible() && !theItem.sourceEditor.cancel()) {
							AppController.beep();
							theItem.sourceEditor.toFront();
							return false;
						}
						theItem.sourceEditor = null;
					}
				}
			}

			modelRows.clear();

			if (null != newItems) {
				modelRows.addAll(newItems);
			}
			modelRows.addAll(keepItems);

			filterPanel.clearFilter();
			fireTableDataChanged();
			updateBorder();

			return true;
		}


		//-------------------------------------------------------------------------------------------------------------
		// Check if there is any unsaved data, that is new edited records that haven't been applied.  Note wasApplied
		// and isImport are both cleared if a new record is actually edited, even if it was previously applied.

		private boolean hasUnsavedData() {

			for (RecordListItem theItem : modelRows) {
				if (theItem.isNew && !theItem.isImport && !theItem.wasApplied) {
					return true;
				}
			}

			return false;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void updateEditorDocumentNames() {

			for (RecordListItem theItem : modelRows) {
				if (null != theItem.sourceEditor) {
					theItem.sourceEditor.updateDocumentName();
				}
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Adding a new record does not clear the filter state since here the only add is from a duplicate, the new
		// record should have the same filter state as the original which must be visible, so the new will be visible.
		// But don't assume that of course.

		private int add(RecordListItem theItem) {

			int modelIndex = modelRows.size();
			modelRows.add(theItem);

			filterPanel.updateFilter();
			int rowIndex = filterPanel.reverseIndex[modelIndex];
			if (rowIndex >= 0) {
				fireTableRowsInserted(rowIndex, rowIndex);
				updateBorder();
			}

			return rowIndex;
		}


		//-------------------------------------------------------------------------------------------------------------

		private RecordListItem get(int rowIndex) {

			return modelRows.get(filterPanel.forwardIndex[rowIndex]);
		}


		//-------------------------------------------------------------------------------------------------------------

		private void remove(int rowIndex) {

			modelRows.remove(filterPanel.forwardIndex[rowIndex]);

			filterPanel.updateFilter();
			fireTableRowsDeleted(rowIndex, rowIndex);
			updateBorder();
		}


		//-------------------------------------------------------------------------------------------------------------
		// A changed item may mean a change to the filtered state so this may appear as an update or a delete.

		private void itemWasChanged(int rowIndex) {

			int modelIndex = filterPanel.forwardIndex[rowIndex];
			filterPanel.updateFilter();
			int newRowIndex = filterPanel.reverseIndex[modelIndex];
			if (newRowIndex >= 0) {
				fireTableRowsUpdated(newRowIndex, newRowIndex);
			} else {
				fireTableRowsDeleted(rowIndex, rowIndex);
				updateBorder();
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Try to find a list item for an exact match to a give record object, returns null if not found.

		private RecordListItem findItemForRecord(StationRecord theRecord) {

			for (RecordListItem theItem : modelRows) {
				if (theRecord == theItem.record) {
					return theItem;
				}
			}
			return null;
		}


		//-------------------------------------------------------------------------------------------------------------
		// Methods using unfiltered model index.

		private RecordListItem ufGet(int modelIndex) {

			return modelRows.get(modelIndex);
		}


		//-------------------------------------------------------------------------------------------------------------

		private void ufRemove(int modelIndex) {

			modelRows.remove(modelIndex);

			int rowIndex = filterPanel.reverseIndex[modelIndex];
			filterPanel.updateFilter();
			if (rowIndex >= 0) {
				fireTableRowsDeleted(rowIndex, rowIndex);
				updateBorder();
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		private int ufIndexOfEditor(SourceEditor theEditor) {

			RecordListItem theItem;
			for (int modelIndex = 0; modelIndex < modelRows.size(); modelIndex++) {
				theItem = modelRows.get(modelIndex);
				if (theEditor == theItem.sourceEditor) {
					return modelIndex;
				}
			}

			return -1;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void ufItemWasChanged(int modelIndex) {

			int rowIndex = filterPanel.reverseIndex[modelIndex];
			filterPanel.updateFilter();
			int newRowIndex = filterPanel.reverseIndex[modelIndex];
			if (rowIndex >= 0) {
				if (newRowIndex >= 0) {
					fireTableRowsUpdated(newRowIndex, newRowIndex);
				} else {
					fireTableRowsDeleted(rowIndex, rowIndex);
					updateBorder();
				}
			} else {
				if (newRowIndex >= 0) {
					fireTableRowsInserted(newRowIndex, newRowIndex);
					updateBorder();
				}
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		public void filterDidChange() {

			fireTableDataChanged();
			updateBorder();
		}


		//-------------------------------------------------------------------------------------------------------------

		private void updateBorder() {

			int n = filterPanel.forwardIndex.length;
			if (n > 0) {
				panel.setBorder(BorderFactory.createTitledBorder(String.valueOf(n) + " records"));
			} else {
				panel.setBorder(BorderFactory.createTitledBorder("Records"));
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getColumnCount() {

			return columnNames.length;
		}


		//-------------------------------------------------------------------------------------------------------------

		public String getColumnName(int columnIndex) {

			return columnNames[columnIndex];
		}


		//-------------------------------------------------------------------------------------------------------------

		public boolean filterByColumn(int columnIndex) {

			switch (columnIndex) {
				case RECORD_CALLSIGN_INDEX:
				case RECORD_CHANNEL_INDEX:
				case RECORD_SERVICE_INDEX:
				case RECORD_STATUS_INDEX:
				case RECORD_COUNTRY_INDEX: {
					return true;
				}
			}

			return false;
		}


		//-------------------------------------------------------------------------------------------------------------

		public boolean collapseFilterChoices(int columnIndex) {

			switch (columnIndex) {
				case RECORD_CALLSIGN_INDEX:
				case RECORD_CHANNEL_INDEX: {
					return true;
				}
			}

			return false;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getRowCount() {

			return filterPanel.forwardIndex.length;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getUnfilteredRowCount() {

			return modelRows.size();
		}


		//-------------------------------------------------------------------------------------------------------------

		public Object getValueAt(int rowIndex, int columnIndex) {

			return getUnfilteredValueAt(filterPanel.forwardIndex[rowIndex], columnIndex);
		}


		//-------------------------------------------------------------------------------------------------------------

		public String getUnfilteredValueAt(int rowIndex, int columnIndex) {

			StationRecord theRecord = modelRows.get(rowIndex).record;

			switch (columnIndex) {

				case RECORD_TYPE_INDEX:
					return theRecord.getRecordType();

				case RECORD_CALLSIGN_INDEX:
					return theRecord.getCallSign();

				case RECORD_CHANNEL_INDEX:
					return theRecord.getChannel() + " " + theRecord.getFrequency();

				case RECORD_SERVICE_INDEX:
					return theRecord.getServiceCode();

				case RECORD_STATUS_INDEX:
					return theRecord.getStatus();

				case RECORD_CITY_INDEX:
					return theRecord.getCity();

				case RECORD_STATE_INDEX:
					return theRecord.getState();

				case RECORD_COUNTRY_INDEX:
					return theRecord.getCountryCode();

				case RECORD_FACILITY_ID_INDEX:
					return theRecord.getFacilityID();

				case RECORD_FILE_INDEX:
					return theRecord.getFileNumber();

				case RECORD_DATE_INDEX:
					return theRecord.getSequenceDate();
			}

			return "";
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Clear all input fields, except the record type and data set.  Also clear search results, optionally preserve
	// new records.  Theoretically could fail due to dependent dialogs not closing, see setItems() in the list model.

	private boolean doClear(boolean keepNew) {

		blockActionsStart();

		recordIDField.setText("");
		fileNumberField.setText("");

		facilityIDField.setText("");
		serviceMenu.setSelectedIndex(0);
		callSignField.setText("");
		channelField.setText("");
		statusMenu.setSelectedIndex(0);
		cityField.setText("");
		stateField.setText("");
		countryMenu.setSelectedIndex(0);
		includeArchivedCheckBox.setSelected(false);
		includeSharingGuestsCheckBox.setSelected(false);

		additionalSQLTextArea.setText("");

		radiusSearchCheckBox.setSelected(false);
		latitudePanel.setEnabled(false);
		longitudePanel.setEnabled(false);
		copyPastePanel.setEnabled(false);
		AppController.setComponentEnabled(radiusField, false);
		searchCenter.setLatLon(0., 0.);
		latitudePanel.updatePanel();
		longitudePanel.updatePanel();
		radiusField.setText("");
		searchRadius = 0.;

		boolean result = listModel.setItems(null, keepNew);

		updateControls();

		blockActionsEnd();

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Do the search.  First build the search query.  If the record ID or file number are set, all other criteria are
	// ignored as those are expected to match just one specific record.  For a file number search if archived records
	// are included there may be multiple matches, however those are historical versions of the same record and it is
	// assumed they would not vary with respect to other search criteria.  Otherwise a search is performed combining
	// all criteria with AND.  The search is always restricted by record type even for an ID or file number search,
	// also for TV records the search is always restricted by channel.

	private void doSearch() {

		errorReporter.clearTitle();

		int dbType = ExtDb.DB_TYPE_NOT_SET;
		int version = 0;
		boolean isSrc = true;
		if (null != extDb) {
			dbType = extDb.type;
			version = extDb.version;
			isSrc = extDb.isGeneric();
		}

		// Parse the radius field, check for validity.  Also get the parameter value for distance calculations.

		searchRadius = 0.;
		double kpd = 0.;

		if (radiusSearchCheckBox.isSelected()) {

			String str = radiusField.getText().trim();
			if (str.length() > 0) {
				try {
					double d = Double.parseDouble(str);
					if ((d < 1.) || (d > 1000.)) {
						errorReporter.reportWarning("The radius value must be in the range 1 to 1000");
						return;
					} else {
						searchRadius = d;
					}
				} catch (NumberFormatException ne) {
					errorReporter.reportWarning("The radius value must be a number");
					return;
				}
			}

			if (null != study) {
				kpd = study.getKilometersPerDegree();
			} else {
				kpd = 111.15;
				str = Parameter.getTemplateParameterValue(getDbID(), 1, Parameter.PARAM_EARTH_SPH_DIST, 0);
				if (null != str) {
					try {
						kpd = Double.parseDouble(str);
					} catch (NumberFormatException ne) {
					}
				}
			}
		}

		ArrayList<RecordListItem> searchResults = null;
		StringBuilder query = new StringBuilder();

		if (!useBaseline || (Source.RECORD_TYPE_TV != searchRecordType)) {

			try {

				boolean hasCrit = false, hasChannel = false;

				if (isSrc) {
					if (SourceEditData.addRecordIDQuery(dbType, recordIDField.getText().trim(), query, false)) {
						hasCrit = true;
					}
				} else {
					if (ExtDbRecord.addRecordIDQuery(dbType, version, searchRecordType, recordIDField.getText().trim(),
							query, false)) {
						hasCrit = true;
					}
				}

				if (!hasCrit) {
					if (isSrc) {
						if (SourceEditData.addFileNumberQuery(dbType, fileNumberField.getText().trim(), query,
							false)) {
							hasCrit = true;
						}
					} else {
						if (ExtDbRecord.addFileNumberQuery(dbType, version, searchRecordType,
								fileNumberField.getText().trim(), query, false)) {
							hasCrit = true;
						}
					}
				}

				if (!hasCrit) {

					if (radiusSearchCheckBox.isSelected()) {
						if ((0. == searchCenter.latitude) || (0. == searchCenter.longitude) || (0. == searchRadius)) {
							errorReporter.reportWarning("Please enter search radius and center coordinates");
							return;
						}
					}

					if (isSrc) {
						if (SourceEditData.addFacilityIDQuery(dbType, facilityIDField.getText().trim(), query,
								false)) {
							hasCrit = true;
						}
					} else {
						if (ExtDbRecord.addFacilityIDQuery(dbType, version, searchRecordType,
								facilityIDField.getText().trim(), query, false)) {
							hasCrit = true;
						}
					}

					int serviceKey = serviceMenu.getSelectedKey();
					if (serviceKey > 0) {
						if (isSrc) {
							if (SourceEditData.addServiceQuery(dbType, serviceKey, query, hasCrit)) {
								hasCrit = true;
							}
						} else {
							if (ExtDbRecord.addServiceQuery(dbType, version, searchRecordType, serviceKey, query,
									hasCrit)) {
								hasCrit = true;
							}
						}
					}

					if (isSrc) {
						if (SourceEditData.addCallSignQuery(dbType, callSignField.getText().trim(), query, hasCrit)) {
							hasCrit = true;
						}
					} else {
						if (ExtDbRecord.addCallSignQuery(dbType, version, searchRecordType,
								callSignField.getText().trim(), query, hasCrit)) {
							hasCrit = true;
						}
					}

					if (Source.RECORD_TYPE_TV == searchRecordType) {
						if (isSrc) {
							if (SourceEditData.addChannelQuery(dbType, channelField.getText().trim(), minimumTVChannel,
									maximumTVChannel, query, hasCrit)) {
								hasChannel = true;
								hasCrit = true;
							}
						} else {
							if (ExtDbRecordTV.addChannelQuery(dbType, version, searchRecordType,
									channelField.getText().trim(), minimumTVChannel, maximumTVChannel, query,
									hasCrit)) {
								hasChannel = true;
								hasCrit = true;
							}
						}
					} else {
						if (Source.RECORD_TYPE_FM == searchRecordType) {
							if (isSrc) {
								if (SourceEditData.addChannelQuery(dbType, channelField.getText().trim(),
										SourceFM.CHANNEL_MIN, SourceFM.CHANNEL_MAX, query, hasCrit)) {
									hasChannel = true;
									hasCrit = true;
								}
							} else {
								if (ExtDbRecordFM.addChannelQuery(dbType, version, searchRecordType,
										channelField.getText().trim(), SourceFM.CHANNEL_MIN, SourceFM.CHANNEL_MAX,
										query, hasCrit)) {
									hasChannel = true;
									hasCrit = true;
								}
							}
						}
					}

					int statusType = statusMenu.getSelectedKey();
					if (statusType >= 0) {
						if (isSrc) {
							if (SourceEditData.addStatusQuery(dbType, statusType, query, hasCrit)) {
								hasCrit = true;
							}
						} else {
							if (ExtDbRecord.addStatusQuery(dbType, version, searchRecordType, statusType, query,
									hasCrit)) {
								hasCrit = true;
							}
						}
					}

					if (isSrc) {
						if (SourceEditData.addCityQuery(dbType, cityField.getText().trim(), query, hasCrit)) {
							hasCrit = true;
						}
					} else {
						if (ExtDbRecord.addCityQuery(dbType, version, searchRecordType, cityField.getText().trim(),
								query, hasCrit)) {
							hasCrit = true;
						}
					}

					if (isSrc) {
						if (SourceEditData.addStateQuery(dbType, stateField.getText().trim(), query, hasCrit)) {
							hasCrit = true;
						}
					} else {
						if (ExtDbRecord.addStateQuery(dbType, version, searchRecordType, stateField.getText().trim(),
								query, hasCrit)) {
							hasCrit = true;
						}
					}

					int countryKey = countryMenu.getSelectedKey();
					if (countryKey > 0) {
						if (isSrc) {
							if (SourceEditData.addCountryQuery(dbType, countryKey, query, hasCrit)) {
								hasCrit = true;
							}
						} else {
							if (ExtDbRecord.addCountryQuery(dbType, version, searchRecordType, countryKey, query,
									hasCrit)) {
								hasCrit = true;
							}
						}
					}

					// Add any additional SQL with AND.

					String addSQL = additionalSQLTextArea.getText().trim();
					if (addSQL.length() > 0) {
						if (hasCrit) {
							query.append(" AND ");
						}
						query.append('(');
						query.append(addSQL);
						query.append(')');
						hasCrit = true;
					}

				} else {
					searchRadius = 0.;
				}

				// A TV search is always restricted by channel, if a specific one was not entered use full range.

				if ((Source.RECORD_TYPE_TV == searchRecordType) && !hasChannel) {
					if (isSrc) {
						if (SourceEditData.addChannelRangeQuery(dbType, minimumTVChannel, maximumTVChannel, query,
								hasCrit)) {
							hasCrit = true;
						}
					} else {
						if (ExtDbRecordTV.addChannelRangeQueryTV(dbType, version, minimumTVChannel, maximumTVChannel,
								query, hasCrit)) {
							hasCrit = true;
						}
					}
				}

				// Add record type clause, generally this will include only current and pending records, but
				// optionally may include archived records as well.  There is no record type concept in the user
				// record table or generic import data sets so skip this in those cases.

				if (!isSrc) {
					if (ExtDbRecord.addRecordTypeQuery(dbType, version, searchRecordType,
							includeArchivedCheckBox.isSelected(), query, hasCrit)) {
						hasCrit = true;
					}
				}

			} catch (IllegalArgumentException ie) {
				errorReporter.reportError(ie.getMessage());
				return;
			}

			// Do the search.

			final String theQuery = query.toString();
			final boolean inclGuests = includeSharingGuestsCheckBox.isSelected();
			final double kmPerDegree = kpd;

			BackgroundWorker<ArrayList<RecordListItem>> theWorker =
					new BackgroundWorker<ArrayList<RecordListItem>>(getWindow(), getTitle()) {
				protected ArrayList<RecordListItem> doBackgroundWork(ErrorLogger errors) {

					ArrayList<RecordListItem> results = new ArrayList<RecordListItem>();
					RecordListItem theItem;

					int dbType = ExtDb.DB_TYPE_NOT_SET;
					boolean isSrc = true;
					if (null != extDb) {
						dbType = extDb.type;
						isSrc = extDb.isGeneric();
					}

					if (isSrc) {

						java.util.List<SourceEditData> sources = null;

						boolean isUsr = false;
						if (ExtDb.DB_TYPE_NOT_SET == dbType) {
							sources = SourceEditData.findUserRecords(getDbID(), searchRecordType, theQuery, errors);
							isUsr = true;
						} else {
							sources = SourceEditData.findImportRecords(extDb, searchRecordType, theQuery, searchCenter,
								searchRadius, kmPerDegree, errors);
						}

						if (null == sources) {
							return null;
						}

						for (SourceEditData theSource : sources) {

							theItem = new RecordListItem();
							theItem.record = theSource;
							theItem.isSource = true;
							theItem.isUserRecord = isUsr;
							theItem.isValid = true;

							theItem.comment = theSource.makeCommentText();

							results.add(theItem);
						}

					} else {

						LinkedList<ExtDbRecord> records = null;
						switch (searchRecordType) {
							case Source.RECORD_TYPE_TV: {
								records = ExtDbRecordTV.findRecords(extDb, theQuery, inclGuests, searchCenter,
									searchRadius, kmPerDegree, errors);
								break;
							}
							case Source.RECORD_TYPE_FM: {
								records = ExtDbRecordFM.findRecords(extDb, theQuery, searchCenter, searchRadius,
									kmPerDegree, errors);
								break;
							}
						}
						if (null == records) {
							return null;
						}

						for (ExtDbRecord theRecord : records) {

							theItem = new RecordListItem();
							theItem.record = theRecord;
							theItem.isValid = true;

							theItem.comment = theRecord.makeCommentText();

							results.add(theItem);
						}
					}

					return results;
				}
			};

			errorReporter.clearMessages();

			searchResults = theWorker.runWork("Searching for records, please wait...", errorReporter);
			if (null == searchResults) {
				return;
			}

		} else {

			// A TV baseline record search is handled by separate query-building methods.  Facility ID here is expected
			// single match, that is the unique identifier in the baseline tables.

			try {

				if (!ExtDbRecordTV.addBaselineRecordIDQuery(dbType, version, recordIDField.getText().trim(), query,
							false) &&
						!ExtDbRecordTV.addBaselineFacilityIDQuery(dbType, version, facilityIDField.getText().trim(),
							query, false)) {

					if (radiusSearchCheckBox.isSelected()) {
						if ((0. == searchCenter.latitude) || (0. == searchCenter.longitude) || (0. == searchRadius)) {
							errorReporter.reportWarning("Please enter search radius and center coordinates");
							return;
						}
					}

					boolean hasCrit = false;

					int serviceKey = serviceMenu.getSelectedKey();
					if (serviceKey > 0) {
						if (ExtDbRecordTV.addBaselineServiceQuery(dbType, version, serviceKey, query, hasCrit)) {
							hasCrit = true;
						}
					}

					if (ExtDbRecordTV.addBaselineCallSignQuery(dbType, version, callSignField.getText().trim(), query,
							hasCrit)) {
						hasCrit = true;
					}

					boolean hasChannel = false;
					if (ExtDbRecordTV.addBaselineChannelQuery(dbType, version, channelField.getText().trim(),
							minimumTVChannel, maximumTVChannel, query, hasCrit)) {
						hasChannel = true;
						hasCrit = true;
					}

					if (ExtDbRecordTV.addBaselineCityQuery(dbType, version, cityField.getText().trim(), query,
							hasCrit)) {
						hasCrit = true;
					}

					if (ExtDbRecordTV.addBaselineStateQuery(dbType, version, stateField.getText().trim(), query,
							hasCrit)) {
						hasCrit = true;
					}

					int countryKey = countryMenu.getSelectedKey();
					if (countryKey > 0) {
						if (ExtDbRecordTV.addBaselineCountryQuery(dbType, version, countryKey, query, hasCrit)) {
							hasCrit = true;
						}
					}

					String addSQL = additionalSQLTextArea.getText().trim();
					if (addSQL.length() > 0) {
						if (hasCrit) {
							query.append(" AND ");
						}
						query.append('(');
						query.append(addSQL);
						query.append(')');
						hasCrit = true;
					}

					if (!hasChannel) {
						if (ExtDbRecordTV.addBaselineChannelRangeQuery(dbType, version, minimumTVChannel,
								maximumTVChannel, query, hasCrit)) {
							hasCrit = true;
						}
					}

				} else {
					searchRadius = 0.;
				}

			} catch (IllegalArgumentException ie) {
				errorReporter.reportError(ie.getMessage());
				return;
			}

			final String theQuery = query.toString();
			final double kmPerDegree = kpd;

			BackgroundWorker<ArrayList<RecordListItem>> theWorker =
					new BackgroundWorker<ArrayList<RecordListItem>>(getWindow(), getTitle()) {
				protected ArrayList<RecordListItem> doBackgroundWork(ErrorLogger errors) {

					ArrayList<RecordListItem> results = new ArrayList<RecordListItem>();
					RecordListItem theItem;

					LinkedList<ExtDbRecordTV> records = ExtDbRecordTV.findBaselineRecords(extDb, theQuery,
						searchCenter, searchRadius, kmPerDegree, errors);
					if (null == records) {
						return null;
					}

					for (ExtDbRecordTV theRecord : records) {

						theItem = new RecordListItem();
						theItem.record = theRecord;
						theItem.isValid = true;

						theItem.comment = theRecord.makeCommentText();

						results.add(theItem);
					}

					return results;
				}
			};

			errorReporter.clearMessages();

			searchResults = theWorker.runWork("Searching for records, please wait...", errorReporter);
			if (null == searchResults) {
				return;
			}
		}

		if (searchResults.size() > 0) {

			if (!listModel.setItems(searchResults, true)) {
				return;
			}

			errorReporter.showMessages();

			listTable.setRowSelectionInterval(0, 0);
			listTable.scrollRectToVisible(listTable.getCellRect(0, 0, true));

		} else {

			if (!listModel.setItems(null, true)) {
				return;
			}

			errorReporter.reportMessage("No matching records found");
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update the state of various controls after a table selection change.

	private void updateControls() {

		if (!canViewEdit) {
			return;
		}

		if (null == selectedItem) {

			duplicateButton.setEnabled(false);
			duplicateMenuItem.setEnabled(false);
			exportButton.setEnabled(false);
			exportMenuItem.setEnabled(false);
			saveButton.setEnabled(false);
			saveMenuItem.setEnabled(false);
			openButton.setText("View");
			openButton.setEnabled(false);
			openMenuItem.setText("View");
			openMenuItem.setEnabled(false);
			compareMenuItem.setEnabled(false);
			if (showUserDelete) {
				deleteButton.setEnabled(false);
				deleteMenuItem.setEnabled(false);
			}

		} else {

			// A new record is temporarily invalid if it has an open source editor.

			boolean isValid = (selectedItem.isValid && (!selectedItem.isNew || (null == selectedItem.sourceEditor)));

			duplicateButton.setEnabled(isValid);
			duplicateMenuItem.setEnabled(isValid);
			exportButton.setEnabled(isValid);
			exportMenuItem.setEnabled(isValid);

			// Only new records can be saved as user records.  It may or may not be possible to apply a new record.

			if (selectedItem.isNew) {
				saveButton.setEnabled(isValid);
				saveMenuItem.setEnabled(isValid);
				openButton.setText("Edit");
				openMenuItem.setText("Edit");
			} else {
				saveButton.setEnabled(false);
				saveMenuItem.setEnabled(false);
				openButton.setText("View");
				openMenuItem.setText("View");
			}
			openButton.setEnabled(true);
			openMenuItem.setEnabled(true);
			compareMenuItem.setEnabled(true);
			if (showUserDelete) {
				deleteButton.setEnabled(selectedItem.isUserRecord);
				deleteMenuItem.setEnabled(selectedItem.isUserRecord);
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doPrevious() {

		int rowCount = listTable.getRowCount();
		int rowIndex = listTable.getSelectedRow();
		if ((rowCount > 0) && (rowIndex != 0)) {
			if (rowIndex < 0) {
				rowIndex = rowCount - 1;
			} else {
				rowIndex--;
			}
			listTable.setRowSelectionInterval(rowIndex, rowIndex);
			listTable.scrollRectToVisible(listTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doNext() {

		int rowCount = listTable.getRowCount();
		int rowIndex = listTable.getSelectedRow();
		if ((rowCount > 0) && (rowIndex < (rowCount - 1))) {
			if (rowIndex < 0) {
				rowIndex = 0;
			} else {
				rowIndex++;
			}
			listTable.setRowSelectionInterval(rowIndex, rowIndex);
			listTable.scrollRectToVisible(listTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// View or edit the selected record.  Even with a new source the editor is told to derive another new record so
	// all properties including facility ID etc. are still editable.  A user record is just viewed, an ExtDbRecord is
	// converted temporarily to a locked source and viewed.

	private void doOpen() {

		if (!canViewEdit || (null == selectedItem)) {
			return;
		}

		if (null != selectedItem.sourceEditor) {
			if (selectedItem.sourceEditor.isVisible()) {
				selectedItem.sourceEditor.toFront();
				return;
			}
			selectedItem.sourceEditor = null;
		}

		if (selectedItem.isNew) {
			errorReporter.setTitle("Edit Record");
		} else {
			errorReporter.setTitle("View Record");
		}
		errorReporter.clearMessages();

		SourceEditData theSource = null;
		String theNote = null;
		if (selectedItem.isSource) {
			theSource = (SourceEditData)(selectedItem.record);
		} else {
			theSource = SourceEditData.makeSource((ExtDbRecord)(selectedItem.record), null, true, errorReporter);
			if (null == theSource) {
				return;
			}
			if (Source.RECORD_TYPE_TV == theSource.recordType) {
				ExtDbRecordTV theRecord = (ExtDbRecordTV)(selectedItem.record);
				if (theRecord.replicateToChannel > 0) {
					theNote = "will replicate to D" + theRecord.replicateToChannel;
				}
			}
		}

		errorReporter.showMessages();

		SourceEditor theEditor = new SourceEditor(this);
		theEditor.setChannelNote(theNote);
		if (!theEditor.setSource(theSource, selectedItem.isNew, errorReporter)) {
			return;
		}

		AppController.showWindow(theEditor);

		selectedItem.sourceEditor = theEditor;

		listModel.itemWasChanged(selectedItemIndex);

		updateControls();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Create an entirely new source record, immediately open a SourceEditor for the record.  The blank new record is
	// added to the list, but will be removed again in editorClosing() if still invalid meaning the edit was canceled.

	private void doNew(int theRecordType) {

		if (!canViewEdit) {
			return;
		}

		errorReporter.setTitle("Create New Record");

		SourceEditData newSource = null;
		switch (theRecordType) {
			case Source.RECORD_TYPE_TV: {
				newSource = SourceEditDataTV.createSource(null, getDbID(), 0, Service.getInvalidObject(), false,
					Country.getInvalidObject(), false, errorReporter);
				break;
			}
			case Source.RECORD_TYPE_WL: {
				newSource = SourceEditDataWL.createSource(null, getDbID(), Service.getInvalidObject(),
					Country.getInvalidObject(), false, errorReporter);
				break;
			}
			case Source.RECORD_TYPE_FM: {
				newSource = SourceEditDataFM.createSource(null, getDbID(), 0, Service.getInvalidObject(), 0,
					Country.getInvalidObject(), false, errorReporter);
				break;
			}
		}
		if (null == newSource) {
			return;
		}

		SourceEditor theEditor = new SourceEditor(this);
		if (!theEditor.setSource(newSource, true, errorReporter)) {
			return;
		}

		AppController.showWindow(theEditor);

		RecordListItem newItem = new RecordListItem();
		newItem.record = newSource;
		newItem.isSource = true;
		newItem.isNew = true;
		newItem.sourceEditor = theEditor;

		int rowIndex = listModel.add(newItem);
		if (rowIndex >= 0) {
			rowIndex = listTable.convertRowIndexToView(rowIndex);
			listTable.setRowSelectionInterval(rowIndex, rowIndex);
			listTable.scrollRectToVisible(listTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Create a new record by duplicating the selected record.  The selected record must be valid and if new not have
	// an open source editor.  If a source is selected derive a new one, if an ExtDbRecord make a source from that.
	// Presumably the user wants to make changes to the duplicate so immediately open a source editor.  That also means
	// if the user cancels the editor they will not expect the duplicate to remain in the list, so the duplicate is
	// flagged invalid even though it isn't to trigger that behavior in editorClosing().

	private void doDuplicate() {

		if (!canViewEdit || (null == selectedItem) || !selectedItem.isValid ||
				(selectedItem.isNew && (null != selectedItem.sourceEditor))) {
			return;
		}

		errorReporter.setTitle("Duplicate Record");

		errorReporter.clearMessages();

		SourceEditData newSource = null;
		if (selectedItem.isSource) {
			newSource = ((SourceEditData)(selectedItem.record)).deriveSource(null, false, errorReporter);
		} else {
			newSource = SourceEditData.makeSource((ExtDbRecord)(selectedItem.record), null, false, errorReporter);
		}
		if (null == newSource) {
			return;
		}

		// Strip the baseline state from the duplicate.  This is just establishing a default because it's unlikely the
		// user wants a baseline record here.  Baseline can be re-selected in the editor if desired.

		newSource.removeAttribute(Source.ATTR_IS_BASELINE);

		errorReporter.showMessages();

		SourceEditor theEditor = new SourceEditor(this);
		if (!theEditor.setSource(newSource, true, errorReporter)) {
			return;
		}

		AppController.showWindow(theEditor);

		RecordListItem newItem = new RecordListItem();
		newItem.record = newSource;
		newItem.isSource = true;
		newItem.isNew = true;
		newItem.isValid = false;
		newItem.sourceEditor = theEditor;

		int rowIndex = listModel.add(newItem);
		if (rowIndex >= 0) {
			rowIndex = listTable.convertRowIndexToView(rowIndex);
			listTable.setRowSelectionInterval(rowIndex, rowIndex);
			listTable.scrollRectToVisible(listTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Import records from a text file.

	private void doImport(int theRecordType) {

		if (!canViewEdit) {
			return;
		}

		String title = "Import " + Source.getRecordTypeName(theRecordType) + " Records";
		errorReporter.setTitle(title);

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.OPEN_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new FileNameExtensionFilter("Text (*.csv,*.txt,*.dat)", "csv", "txt", "dat"));
		chooser.setAcceptAllFileFilterUsed(false);

		JCheckBox patternFileCheckBox = new JCheckBox("Separate pattern file");

		JPanel textPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		textPanel.setBorder(BorderFactory.createTitledBorder("Text options"));
		textPanel.add(patternFileCheckBox);

		JPanel accessoryPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		accessoryPanel.add(textPanel);

		chooser.setAccessory(accessoryPanel);

		if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Import")) {
			return;
		}
		File theFile = chooser.getSelectedFile();
		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

		final int importRecordType = theRecordType;
		final File importFile = theFile;

		// Optionally pattern data may be in a separate file, prompt for that if needed.

		theFile = null;

		if (patternFileCheckBox.isSelected()) {

			chooser = new JFileChooser(importFile.getParentFile().getAbsolutePath());
			chooser.setDialogType(JFileChooser.OPEN_DIALOG);
			chooser.setDialogTitle("Choose pattern data file");
			chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
			chooser.setMultiSelectionEnabled(false);
			chooser.addChoosableFileFilter(new FileNameExtensionFilter("Text (*.csv,*.txt,*.dat)", "csv", "txt",
				"dat"));
			chooser.setAcceptAllFileFilterUsed(false);

			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Import")) {
				return;
			}
			theFile = chooser.getSelectedFile();

			AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());
		}

		final File patternFile = theFile;

		BackgroundWorker<ArrayList<SourceEditData>> theWorker =
				new BackgroundWorker<ArrayList<SourceEditData>>(getWindow(), title) {
			protected ArrayList<SourceEditData> doBackgroundWork(ErrorLogger errors) {
				return SourceEditData.readFromText(getDbID(), importRecordType, importFile, patternFile, errors);
			}
		};

		errorReporter.clearMessages();
		ArrayList<SourceEditData> newSources = theWorker.runWork("Importing records, please wait...", errorReporter);
		if (null == newSources) {
			return;
		}

		// Apply the results.

		errorReporter.showMessages();

		ArrayList<RecordListItem> newItems = new ArrayList<RecordListItem>();
		RecordListItem newItem;

		for (SourceEditData newSource : newSources) {

			newItem = new RecordListItem();
			newItem.record = newSource;
			newItem.isSource = true;
			newItem.isNew = !newSource.isLocked;
			newItem.isImport = newItem.isNew;
			newItem.isValid = true;

			newItems.add(newItem);
		}

		listModel.setItems(newItems, true);

		listTable.setRowSelectionInterval(0, 0);
		listTable.scrollRectToVisible(listTable.getCellRect(0, 0, true));
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Export the selected record to a CSV file.  It must be valid, and if new, must not have an open editor.

	private void doExport() {

		if (!canViewEdit || (null == selectedItem) || !selectedItem.isValid ||
				(selectedItem.isNew && (null != selectedItem.sourceEditor))) {
			return;
		}

		String title = "Export Record";
		errorReporter.setTitle(title);

		errorReporter.clearMessages();
		SourceEditData theSource = null;
		if (selectedItem.isSource) {
			theSource = (SourceEditData)(selectedItem.record);
		} else {
			theSource = SourceEditData.makeSource((ExtDbRecord)(selectedItem.record), null, true, errorReporter);
			if (null == theSource) {
				return;
			}
		}
		errorReporter.showMessages();

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.SAVE_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new FileNameExtensionFilter("CSV (*.csv)", "csv"));
		chooser.setAcceptAllFileFilterUsed(false);

		File theFile = null;
		String theName;
		do {
			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Export")) {
				return;
			}
			theFile = chooser.getSelectedFile();
			theName = theFile.getName();
			if (!theName.toLowerCase().endsWith(".csv")) {
				theFile = new File(theFile.getAbsolutePath() + ".csv");
			}
			if (theFile.exists()) {
				AppController.beep();
				if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
						"The file exists, do you want to replace it?", title, JOptionPane.YES_NO_OPTION,
						JOptionPane.WARNING_MESSAGE)) {
					theFile = null;
				}
			}
		} while (null == theFile);

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

		BufferedWriter writer = null;
		try {
			writer = new BufferedWriter(new FileWriter(theFile));
		} catch (IOException ie) {
			errorReporter.reportError("Could not open the file:\n" + ie.getMessage());
			return;
		}

		ArrayList<SourceEditData> theSources = new ArrayList<SourceEditData>();
		theSources.add(theSource);

		try {
			SourceEditData.writeToCSV(writer, theSource.recordType, theSources, errorReporter);
		} catch (IOException ie) {
			errorReporter.reportError(ie.toString());
		}

		try {writer.close();} catch (IOException ie) {}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Save a new record as a user record, the selected record must be valid, new, and not have an open editor.  The
	// newly-created SourceEditData object representing the user record replaces the existing one in the list.  Also
	// prompt for entry of comment text for the user record.

	private void doSave() {

		if (!canViewEdit || (null == selectedItem) || !selectedItem.isValid || !selectedItem.isNew ||
				(null != selectedItem.sourceEditor)) {
			return;
		}

		String title = "Save User Record";
		errorReporter.setTitle(title);

		SourceEditData theSource = (SourceEditData)selectedItem.record;
		final java.util.Date originalDate = AppCore.parseDate(theSource.getAttribute(Source.ATTR_SEQUENCE_DATE));
		final boolean originalSharedFlag = (null != theSource.getAttribute(Source.ATTR_IS_SHARING_HOST));

		TextInputDialog theDialog = new TextInputDialog(outerThis, title, "Comment");

		final DateSelectionPanel sequenceDatePanel = new DateSelectionPanel(theDialog, "Sequence date", false);
		sequenceDatePanel.setFutureAllowed(true);

		final JCheckBox sharingHostCheckBox = new JCheckBox("Shared channel host");

		JPanel topPanel = new JPanel();
		topPanel.add(sequenceDatePanel);
		topPanel.add(sharingHostCheckBox);

		if ((null != originalDate) || originalSharedFlag) {
			JButton useOrigButton = new JButton("Set from original");
			useOrigButton.setFocusable(false);
			useOrigButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					sequenceDatePanel.setDate(originalDate);
					sharingHostCheckBox.setSelected(originalSharedFlag);
				}
			});
			topPanel.add(useOrigButton);
		}

		theDialog.add(topPanel, BorderLayout.NORTH);

		theDialog.pack();

		AppController.showWindow(theDialog);
		if (theDialog.canceled) {
			return;
		}

		String theComment = theDialog.getInput();

		java.util.Date theDate = sequenceDatePanel.getDate();
		if (null == theDate) {
			theSource.removeAttribute(Source.ATTR_SEQUENCE_DATE);
		} else {
			theSource.setAttribute(Source.ATTR_SEQUENCE_DATE, AppCore.formatDate(theDate));
		}

		if (sharingHostCheckBox.isSelected()) {
			theSource.setAttribute(Source.ATTR_IS_SHARING_HOST);
		} else {
			theSource.removeAttribute(Source.ATTR_IS_SHARING_HOST);
		}

		SourceEditData newSource = theSource.saveAsUserRecord(errorReporter);
		if (null == newSource) {
			return;
		}

		if (theComment.length() > 0) {
			SourceEditData.setSourceComment(newSource, theComment);
		} else {
			theComment = null;
		}

		selectedItem.record = newSource;
		selectedItem.isUserRecord = true;
		selectedItem.isNew = false;
		selectedItem.isImport = false;

		selectedItem.comment = newSource.makeCommentText();

		listModel.itemWasChanged(selectedItemIndex);

		updateControls();
		if (null != callBack) {
			callBack.run();
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Delete a user record.

	private void doDelete() {

		if (!canViewEdit || !showUserDelete || (null == selectedItem) || !selectedItem.isUserRecord) {
			return;
		}

		String title = "Delete User Record";
		errorReporter.setTitle(title);

		if (null != selectedItem.sourceEditor) {
			if (selectedItem.sourceEditor.isVisible() && !selectedItem.sourceEditor.cancel()) {
				AppController.beep();
				selectedItem.sourceEditor.toFront();
				return;
			}
			selectedItem.sourceEditor = null;
		}

		SourceEditData theSource = (SourceEditData)selectedItem.record;
		if (SourceEditData.deleteUserRecord(theSource.dbID, theSource.userRecordID, errorReporter)) {
			listModel.remove(selectedItemIndex);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Called by source editors when edits are saved, data is assumed valid.

	public boolean applyEditsFrom(AppEditor theEditor) {

		if (theEditor instanceof SourceEditor) {

			SourceEditor sourceEditor = (SourceEditor)theEditor;

			int rowIndex = listModel.ufIndexOfEditor(sourceEditor);
			if (rowIndex < 0) {
				return false;
			}

			RecordListItem theItem = listModel.ufGet(rowIndex);

			theItem.record = sourceEditor.getSource();
			theItem.isImport = false;
			theItem.wasApplied = false;
			theItem.isValid = true;

			listModel.ufItemWasChanged(rowIndex);

			return true;
		}

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// When an editor closes, if the record being edited is still invalid that means it was a completely new record
	// and editing was canceled; in that case remove the record from the list.

	public void editorClosing(AppEditor theEditor) {

		if (theEditor instanceof SourceEditor) {

			SourceEditor sourceEditor = (SourceEditor)theEditor;

			int rowIndex = listModel.ufIndexOfEditor(sourceEditor);
			if (rowIndex < 0) {
				return;
			}

			RecordListItem theItem = listModel.ufGet(rowIndex);
			theItem.sourceEditor = null;

			if (!theItem.isValid) {
				listModel.ufRemove(rowIndex);
			} else {
				listModel.ufItemWasChanged(rowIndex);
			}

			updateControls();
			if (null != callBack) {
				callBack.run();
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void clearFields() {

		doClear(true);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Called by ExtDb when the list of data sets may have changed, may be called on a secondary thread.

	public void updateExtDbList() {

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				doExtDbUpdate(false);
			}
		});
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update the menu of data sets, called on first open, also for later updates from ExtDb.

	private void doExtDbUpdate(boolean firstUpdate) {

		ErrorLogger errors = null;
		if (firstUpdate) {
			errors = errorReporter;
			errorReporter.setTitle("Load Station Data List");
		}

		ArrayList<KeyedRecord> list = getExtDbMenuList(searchRecordType, errors);

		blockActionsStart();

		int selectKey = 0;
		if (firstUpdate) {
			if (null != defaultExtDbKey) {
				selectKey = defaultExtDbKey.intValue();
			}
		} else {
			selectKey = extDbMenu.getSelectedKey();
		}

		extDbMenu.removeAllItems();
		extDbMenu.addAllItems(list);
		if (extDbMenu.containsKey(selectKey)) {
			extDbMenu.setSelectedKey(selectKey);
			if (firstUpdate) {
				updateSearchUI(true);
			}
		} else {
			updateSearchUI(firstUpdate);
		}

		if (firstUpdate) {
			doClear(false);
		}

		blockActionsEnd();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Return a list of data sets for the current search record type, including a user-record item which may be the
	// only item in the list.  If there are other data sets, the user records item is added before the first actual
	// data set in the list, which puts it just after data sets with keys in the reserved range which represent special
	// functions like most-recent or live servers; ExtDb.getExtDbList() always puts those at the top.

	private ArrayList<KeyedRecord>getExtDbMenuList(int theRecordType, ErrorLogger errors) {

		ArrayList<KeyedRecord> list = new ArrayList<KeyedRecord>();

		boolean doUser = true;

		if (!userRecordsOnly) {
			ArrayList<KeyedRecord> addList = ExtDb.getExtDbList(getDbID(), theRecordType, errors);
			if (null != addList) {
				for (KeyedRecord theDb : addList) {
					if (doUser && ((theDb.key < ExtDb.RESERVED_KEY_RANGE_START) ||
							(theDb.key > ExtDb.RESERVED_KEY_RANGE_END))) {
						list.add(new KeyedRecord(-theRecordType, "User records"));
						doUser = false;
					}
					list.add(theDb);
				}
			}
		}

		if (doUser) {
			list.add(new KeyedRecord(-theRecordType, "User records"));
		}

		return list;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Re-populate the record type and data set menu when showing the dialog.

	public void windowWillOpen() {

		ExtDb.addListener(this);

		extDb = null;

		callSignField.requestFocusInWindow();

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				doExtDbUpdate(true);
			}
		});
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check if there is any unsaved data, that is new records not saved as user records or open editors for new
	// records, if so confirm the user wants to discard those.  Called when the presenting window is being closed.

	public boolean windowShouldClose() {

		if (listModel.hasUnsavedData()) {
			AppController.beep();
			if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
					"New records were edited but have not been saved.  Unsaved\n" +
					"records will be discarded when this window is closed.\n\n" +
					"Are you sure you want to continue?", "Confirm Close",
					JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE)) {
				return false;
			}
		}

		// This will close all open editors as a side-effect, it can fail if one refuses to close.

		if (!doClear(false)) {
			return false;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillClose() {

		ExtDb.removeListener(this);
	}
}
