//
//  OptionsPanel.java
//  TVStudy
//
//  Copyright (c) 2016-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui;

import gov.fcc.tvstudy.gui.run.*;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;

import java.util.*;
import java.sql.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.table.*;


//=====================================================================================================================
// Accessory panel class used to provide a variety of common UIs needed in contexts such as adding new records to
// scenarios, creating new scenarios, setting up build-and-run studies, and running studies.  Often used as an
// accessory panel in a RecordFind dialog.  The abstract superclass provides a core set of UIs that are individually
// activated by constructor argument.  That constructor is used only by concrete subclasses.  The UIs include a field
// to enter a study or scenario name, button and pop-up text dialog to enter a study or scenario description, option to
// make the selected record (e.g. in a RecordFind context) editable, option to replicate the selected record or change
// it's channel if it is TV, UI to select/edit a run output configuration, and UI for run settings including memory
// limit and a button and pop-up dialog for entering a comment.  The constructor creates components then calls
// createLayout() with this as argument, subclasses may override to substitute a different panel and call super, or lay
// out superclass components directly as desired.   A set of concrete inner subclasses is provided for most common
// uses, some of those are just constructor argument patterns, others provide additional UI e.g. for the build-and-run
// study types.

public abstract class OptionsPanel extends AppPanel {

	protected boolean showName;
	protected String nameLabel;
	protected JTextField nameField;
	protected JPanel namePanel;

	protected boolean showDescription;
	protected String descriptionLabel;
	protected JButton descriptionButton;
	protected JPanel descriptionPanel;

	protected boolean showEdit;
	protected JCheckBox editableCheckBox;
	protected boolean isNew;
	protected JPanel editPanel;

	protected boolean showChannel;
	protected boolean allowMultiChannel;
	protected JCheckBox replicateCheckBox;
	protected JCheckBox changeChannelCheckBox;
	protected JTextField channelField;
	protected boolean isDigital;
	protected boolean isLPTV;
	protected boolean isClassA;
	protected int channel;
	protected java.util.Date sequenceDate;
	protected JPanel channelPanel;

	protected boolean showOutput;
	protected boolean showOutputMenus;
	protected JComboBox<OutputConfig> fileOutputConfigMenu;
	protected JComboBox<OutputConfig> mapOutputConfigMenu;
	protected JButton editConfigButton;
	protected JLabel fileOutputConfigLabel;
	protected JLabel mapOutputConfigLabel;
	protected JPanel outputPanel;

	protected boolean showRun;
	protected JComboBox<String> memoryFractionMenu;
	protected JButton commentButton;
	protected JPanel runPanel;

	protected boolean enabled = true;

	// Defaults for UI may be set by subclass or other, applied in clearFields().

	public String defaultName;

	public String defaultDescription;

	public OutputConfig defaultFileOutputConfig;
	public OutputConfig defaultMapOutputConfig;

	// Inputs available in these properties, if enabled, once validateInput() returns true.

	public String name;

	public String description;

	public boolean isLocked;

	public boolean replicate;
	public boolean changeChannel;
	public int studyChannel;
	public int[] studyChannels;
	public boolean includeOriginalChannelStudy;

	public OutputConfig fileOutputConfig;
	public OutputConfig mapOutputConfig;

	public double memoryFraction;
	public String comment;


	//-----------------------------------------------------------------------------------------------------------------

	protected OptionsPanel(AppEditor theParent, boolean theShowName, String theNameLabel, boolean theShowDescrip,
			String theDescripLabel, boolean theShowEdit, boolean theShowChan, boolean theMultiChan, boolean theShowOut,
			boolean theShowOutMenus, boolean theShowRun) {

		super(theParent);

		showName = theShowName;
		nameLabel = theNameLabel;
		showDescription = theShowDescrip;
		descriptionLabel = theDescripLabel;
		showEdit = theShowEdit;
		showChannel = theShowChan;
		allowMultiChannel = theMultiChan;
		showOutput = theShowOut;
		if (showOutput) {
			showOutputMenus = theShowOutMenus;
		}
		showRun = theShowRun;

		// Name field.  This can actually be any arbitrary text input since caller provides the label.

		if (showName) {

			nameField = new JTextField(20);
			AppController.fixKeyBindings(nameField);

			namePanel = new JPanel();
			namePanel.add(nameField);
			namePanel.setBorder(BorderFactory.createTitledBorder(nameLabel));
		}

		// Description pop-up, again could actually be anything since caller provides the label.

		if (showDescription) {

			descriptionButton = new JButton(descriptionLabel);

			descriptionButton.setFocusable(false);
			descriptionButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doEditDescription();
				}
			});

			descriptionPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			descriptionPanel.add(descriptionButton);

			description = "";
		}

		// Editable-record option.  If editable, replication is not allowed; the combination doesn't make sense here.
		// The scenario model supports it, but it wouldn't make sense to the user in context.  The editable request
		// would apply to the source before replication, but the replication source is all the user initially sees and
		// that is never editable.  So it would appear the editable request was ignored.  Later if the user reverts
		// the replicated record, the editable original would unexpectedly appear.  Having these options is just a
		// convenience anyway, the user can always do the steps manually, including replicating an editable record.

		if (showEdit) {

			editableCheckBox = new JCheckBox("Allow editing");
			editableCheckBox.setFocusable(false);
			editableCheckBox.setToolTipText(
				"<HTML>Allow the station record to be edited.  Changes apply only in the current scenario.<BR>" +
				"Use only if necessary, editable records can significantly increase study run times.</HTML>");
			AppController.setComponentEnabled(editableCheckBox, false);

			if (showChannel) {
				editableCheckBox.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (blockActions()) {
							if (editableCheckBox.isSelected()) {
								replicateCheckBox.setSelected(false);
								AppController.setComponentEnabled(channelField, changeChannelCheckBox.isSelected());
							} else {
								changeChannelCheckBox.setSelected(false);
								AppController.setComponentEnabled(channelField, replicateCheckBox.isSelected());
							}
							blockActionsEnd();
						}
					}
				});
			}

			editPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			editPanel.add(editableCheckBox);
		}

		// Replication and channel change option.  Replication sources have to be locked, channel changes must be
		// editable, so these interact with the editable check-box.

		if (showChannel) {

			replicateCheckBox = new JCheckBox("Replicate");
			replicateCheckBox.setFocusable(false);
			replicateCheckBox.setToolTipText(
				"<HTML>Study a TV station on a different channel using a derived<BR>" +
				"ERP and azimuth pattern replicating the original coverage.</HTML>");
			AppController.setComponentEnabled(replicateCheckBox, false);

			replicateCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					if (blockActions()) {
						if (replicateCheckBox.isSelected()) {
							changeChannelCheckBox.setSelected(false);
							if (showEdit) {
								editableCheckBox.setSelected(false);
							}
							AppController.setComponentEnabled(channelField, true);
						} else {
							AppController.setComponentEnabled(channelField, changeChannelCheckBox.isSelected());
						}
						blockActionsEnd();
					}
				}
			});

			changeChannelCheckBox = new JCheckBox("Change Channel");
			changeChannelCheckBox.setFocusable(false);
			changeChannelCheckBox.setToolTipText(
				"<HTML>Change the channel of the study record,<BR>" +
				"preserving the existing ERP and pattern.</HTML>");
			AppController.setComponentEnabled(changeChannelCheckBox, false);

			changeChannelCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					if (blockActions()) {
						if (changeChannelCheckBox.isSelected()) {
							replicateCheckBox.setSelected(false);
							if (showEdit) {
								editableCheckBox.setSelected(true);
							}
							AppController.setComponentEnabled(channelField, true);
						} else {
							AppController.setComponentEnabled(channelField, replicateCheckBox.isSelected());
						}
						blockActionsEnd();
					}
				}
			});

			// Channel field for replication or channel change, activates if either option is selected.

			channelField = new JTextField(8);
			AppController.fixKeyBindings(channelField);
			AppController.setComponentEnabled(channelField, false);

			// Layout.

			JPanel chkBoxP1 = new JPanel(new FlowLayout(FlowLayout.LEFT));
			chkBoxP1.add(replicateCheckBox);

			JPanel chkBoxP2 = new JPanel(new FlowLayout(FlowLayout.LEFT));
			chkBoxP2.add(changeChannelCheckBox);

			Box chkBoxB = Box.createVerticalBox();
			chkBoxB.add(chkBoxP1);
			chkBoxB.add(chkBoxP2);

			JPanel chanP = new JPanel();
			if (allowMultiChannel) {
				chanP.setBorder(BorderFactory.createTitledBorder("Channel(s)"));
			} else {
				chanP.setBorder(BorderFactory.createTitledBorder("Channel"));
			}
			chanP.add(channelField);

			channelPanel = new JPanel();
			channelPanel.add(chkBoxB);
			channelPanel.add(chanP);
		}

		// Output configuration UI.  See OutputConfigDialog.  Menus will be populated in clearFields().

		if (showOutput) {

			outputPanel = new JPanel();

			if (showOutputMenus) {

				fileOutputConfigMenu = new JComboBox<OutputConfig>();
				fileOutputConfigMenu.setFocusable(false);
				OutputConfig proto = new OutputConfig(OutputConfig.CONFIG_TYPE_FILE, "");
				proto.name = "XyXyXyXyXyXyXyXyXy";
				fileOutputConfigMenu.setPrototypeDisplayValue(proto);

				fileOutputConfigMenu.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (blockActions()) {
							fileOutputConfig = (OutputConfig)fileOutputConfigMenu.getSelectedItem();
							blockActionsEnd();
						}
					}
				});

				mapOutputConfigMenu = new JComboBox<OutputConfig>();
				mapOutputConfigMenu.setFocusable(false);
				proto = new OutputConfig(OutputConfig.CONFIG_TYPE_MAP, "");
				proto.name = "XyXyXyXyXyXyXyXyXy";
				mapOutputConfigMenu.setPrototypeDisplayValue(proto);

				mapOutputConfigMenu.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (blockActions()) {
							mapOutputConfig = (OutputConfig)mapOutputConfigMenu.getSelectedItem();
							blockActionsEnd();
						}
					}
				});

				outputPanel.setBorder(BorderFactory.createTitledBorder("Output Settings"));
				outputPanel.add(new JLabel("File:"));
				outputPanel.add(fileOutputConfigMenu);
				outputPanel.add(new JLabel("Map:"));
				outputPanel.add(mapOutputConfigMenu);

				editConfigButton = new JButton("Edit");

				outputPanel.add(editConfigButton);

			} else {

				editConfigButton = new JButton("Output Settings");

				fileOutputConfigLabel = new JLabel();
				mapOutputConfigLabel = new JLabel();

				JPanel butP = new JPanel();
				butP.add(editConfigButton);
				JPanel fileP = new JPanel();
				fileP.add(fileOutputConfigLabel);
				JPanel mapP = new JPanel();
				mapP.add(mapOutputConfigLabel);

				outputPanel.setLayout(new BoxLayout(outputPanel, BoxLayout.Y_AXIS));
				outputPanel.add(butP);
				outputPanel.add(fileP);
				outputPanel.add(mapP);
			}

			editConfigButton.setFocusable(false);
			editConfigButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doEditOutputConfigs();
				}
			});
		}

		// Memory fraction and comment for study run.  The comment is actually generically labeled "Comment" so it is
		// not necessarily a run comment, but currently that's the only application.  Note the maxEngineProcessCount
		// can be <=0 in which case the engine will not run at all (see AppCore.initialize()), but in that case other
		// code needs to pre-check and not even show this UI.  This will always show at least an "All" option.

		if (showRun) {

			memoryFractionMenu = new JComboBox<String>();
			memoryFractionMenu.setFocusable(false);
			memoryFractionMenu.addItem("All");
			for (int frac = 2; frac <= AppCore.maxEngineProcessCount; frac++) {
				memoryFractionMenu.addItem("1/" + String.valueOf(frac));
			}

			commentButton = new JButton("Comment");

			commentButton.setFocusable(false);
			commentButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doEditComment();
				}
			});

			JPanel memP = new JPanel();
			memP.setBorder(BorderFactory.createTitledBorder("Memory Use"));
			memP.add(memoryFractionMenu);

			JPanel comP = new JPanel(new FlowLayout(FlowLayout.LEFT));
			comP.add(commentButton);

			runPanel = new JPanel();
			runPanel.add(memP);
			runPanel.add(comP);

			comment = "";
		}

		// Default layout, just add everything to the default FlowLayout.

		createLayout(this);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// This just adds the enabled UIs in sequence to the argument panel without interacting with the layout manager
	// directly.  That usually works, if it is too limited the subclass can override this to do something else.

	protected void createLayout(JPanel thePanel) {

		if (showName) {
			thePanel.add(namePanel);
		}

		if (showDescription) {
			thePanel.add(descriptionPanel);
		}

		if (showEdit) {
			thePanel.add(editPanel);
		}

		if (showChannel) {
			thePanel.add(channelPanel);
		}

		if (showOutput) {
			thePanel.add(outputPanel);
		}

		if (showRun) {
			thePanel.add(runPanel);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Show a pop-up text dialog to edit/input description text.

	private void doEditDescription() {

		TextInputDialog theDialog = new TextInputDialog(parent, descriptionLabel, descriptionLabel);
		theDialog.setInput(description);

		AppController.showWindow(theDialog);
		if (theDialog.canceled) {
			return;
		}

		description = theDialog.getInput();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Open an OutputConfigDialog to edit the output file configurations.

	private void doEditOutputConfigs() {

		if (!showOutput) {
			return;
		}

		ArrayList<OutputConfig> theList = new ArrayList<OutputConfig>();
		theList.add(fileOutputConfig);
		theList.add(mapOutputConfig);

		OutputConfigDialog theDialog = new OutputConfigDialog(this, theList, false);
		AppController.showWindow(theDialog);

		for (OutputConfig theConfig : theDialog.getConfigs()) {
			switch (theConfig.type) {
				case OutputConfig.CONFIG_TYPE_FILE: {
					fileOutputConfig = theConfig;
					break;
				}
				case OutputConfig.CONFIG_TYPE_MAP: {
					mapOutputConfig = theConfig;
					break;
				}
			}
		}

		updateOutputConfigUI();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Re-populate the output config menus and select current objects, which may or may not be in the saved list.  Or
	// if menus are not showing, just put current config names in the labels.

	private void updateOutputConfigUI() {

		if (showOutputMenus) {

			blockActionsStart();

			ArrayList<OutputConfig> theList = OutputConfig.getConfigs(getDbID(), OutputConfig.CONFIG_TYPE_FILE);
			fileOutputConfigMenu.removeAllItems();
			for (OutputConfig config : theList) {
				fileOutputConfigMenu.addItem(config);
			}

			if (!theList.contains(fileOutputConfig)) {
				fileOutputConfigMenu.addItem(fileOutputConfig);
			}
			fileOutputConfigMenu.setSelectedItem(fileOutputConfig);

			theList = OutputConfig.getConfigs(getDbID(), OutputConfig.CONFIG_TYPE_MAP);
			mapOutputConfigMenu.removeAllItems();
			for (OutputConfig config : theList) {
				mapOutputConfigMenu.addItem(config);
			}

			if (!theList.contains(mapOutputConfig)) {
				mapOutputConfigMenu.addItem(mapOutputConfig);
			}
			mapOutputConfigMenu.setSelectedItem(mapOutputConfig);

			blockActionsEnd();

		} else {

			fileOutputConfigLabel.setText("File: " + fileOutputConfig.name);
			mapOutputConfigLabel.setText("Map: " + mapOutputConfig.name);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Pop-up for comment text.

	private void doEditComment() {

		TextInputDialog theDialog = new TextInputDialog(parent, "Comment", "Comment");
		theDialog.setInput(comment);

		AppController.showWindow(theDialog);
		if (theDialog.canceled) {
			return;
		}

		comment = theDialog.getInput();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void setEnabled(boolean flag) {

		if (showName) {
			AppController.setComponentEnabled(nameField, flag);
		}

		if (showDescription) {
			descriptionButton.setEnabled(flag);
		}

		if (showEdit) {
			if (isNew) {
				AppController.setComponentEnabled(editableCheckBox, false);
			} else {
				AppController.setComponentEnabled(editableCheckBox, flag);
			}
		}

		if (showChannel) {
			boolean eRep = flag;
			if (isNew || (0 == channel)) {
				eRep = false;
			}
			AppController.setComponentEnabled(replicateCheckBox, eRep);
			AppController.setComponentEnabled(changeChannelCheckBox, flag);
			if (replicateCheckBox.isSelected() || changeChannelCheckBox.isSelected()) {
				AppController.setComponentEnabled(channelField, flag);
			} else {
				AppController.setComponentEnabled(channelField, false);
			}
		}

		if (showOutput) {
			if (showOutputMenus) {
				AppController.setComponentEnabled(fileOutputConfigMenu, flag);
				AppController.setComponentEnabled(mapOutputConfigMenu, flag);
			}
			editConfigButton.setEnabled(flag);
		}

		if (showRun) {
			AppController.setComponentEnabled(memoryFractionMenu, flag);
			commentButton.setEnabled(flag);
		}

		enabled = flag;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void clearFields() {

		blockActionsStart();

		if (showName) {
			if (null == defaultName) {
				defaultName = "";
			}
			nameField.setText(defaultName);
		}

		if (showDescription) {
			if (null == defaultDescription) {
				defaultDescription = "";
			}
			description = defaultDescription;
		}

		if (showEdit) {
			if (isNew) {
				editableCheckBox.setSelected(true);
				AppController.setComponentEnabled(editableCheckBox, false);
			} else {
				editableCheckBox.setSelected(false);
				AppController.setComponentEnabled(editableCheckBox, true);
			}
		}

		if (showChannel) {
			replicateCheckBox.setSelected(false);
			changeChannelCheckBox.setSelected(false);
			channelField.setText("");
			AppController.setComponentEnabled(channelField, false);
		}

		if (showOutput) {
			if ((null == defaultFileOutputConfig) || defaultFileOutputConfig.isNull()) {
				defaultFileOutputConfig = OutputConfig.getLastUsed(getDbID(), OutputConfig.CONFIG_TYPE_FILE);
			}
			fileOutputConfig = defaultFileOutputConfig;
			if ((null == defaultMapOutputConfig) || defaultMapOutputConfig.isNull()) {
				defaultMapOutputConfig = OutputConfig.getLastUsed(getDbID(), OutputConfig.CONFIG_TYPE_MAP);
			}
			mapOutputConfig = defaultMapOutputConfig;
			updateOutputConfigUI();
		}

		if (showRun) {
			String str = AppCore.getPreference(AppCore.PREF_DEFAULT_ENGINE_MEMORY_LIMIT);
			if (null != str) {
				int lim = 0;
				try {
					lim = Integer.parseInt(str);
				} catch (NumberFormatException ne) {
				}
				if (lim > AppCore.maxEngineProcessCount) {
					lim = AppCore.maxEngineProcessCount;
				}
				if (lim < 0) {
					lim = 0;
				}
				memoryFractionMenu.setSelectedIndex(lim);
			} else {
				memoryFractionMenu.setSelectedIndex(0);
			}
			comment = "";
		}

		blockActionsEnd();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// An unlocked SourceEditData is assumed to be a new record just created or imported but never saved to any data
	// set or as a user record.  It must remain unlocked because no unlocked record can be locked again, but it can
	// be replicated or have channel changed.  Non-new records of any type can be made editable and have the channel
	// changed, or can be replicated, but as discussed earlier editable and replication can't be selected together.

	public void selectionChanged(Object newSelection) {

		if (null == newSelection) {
			isNew = false;
			isDigital = false;
			isLPTV = false;
			isClassA = false;
			channel = 0;
			sequenceDate = null;
			if (showEdit) {
				editableCheckBox.setSelected(false);
				AppController.setComponentEnabled(editableCheckBox,false);
			}
			if (showChannel) {
				replicateCheckBox.setSelected(false);
				AppController.setComponentEnabled(replicateCheckBox, false);
				changeChannelCheckBox.setSelected(false);
				AppController.setComponentEnabled(changeChannelCheckBox, false);
				AppController.setComponentEnabled(channelField, false);
			}
			return;
		}

		if (newSelection instanceof SourceEditData) {

			SourceEditData theSource = (SourceEditData)newSelection;
			isNew = !theSource.isLocked;
			isDigital = theSource.service.isDigital();
			isLPTV = theSource.service.isLPTV();
			isClassA = theSource.service.isClassA();
			channel = theSource.getChannelNumber();
			sequenceDate = AppCore.parseDate(theSource.getAttribute(Source.ATTR_SEQUENCE_DATE));
			if (null == sequenceDate) {
				sequenceDate = new java.util.Date();
			}

		} else {

			if (newSelection instanceof ExtDbRecord) {

				ExtDbRecord theRecord = (ExtDbRecord)newSelection;
				isNew = false;
				isDigital = theRecord.service.isDigital();
				isLPTV = theRecord.service.isLPTV();
				isClassA = theRecord.service.isClassA();
				channel = theRecord.getChannelNumber();
				if (Source.RECORD_TYPE_TV == theRecord.recordType) {
					ExtDbRecordTV theRecordTV = (ExtDbRecordTV)theRecord;
					if ((theRecordTV.replicateToChannel > 0) && showChannel && !replicateCheckBox.isSelected()) {
						replicateCheckBox.setSelected(true);
						channelField.setText(String.valueOf(theRecordTV.replicateToChannel));
					}
				}
				sequenceDate = theRecord.sequenceDate;

			} else {

				isNew = false;
				isDigital = false;
				isLPTV = false;
				isClassA = false;
				channel = 0;
				sequenceDate = null;
			}
		}

		if (showEdit) {
			if (isNew) {
				editableCheckBox.setSelected(false);
				AppController.setComponentEnabled(editableCheckBox, false);
			} else {
				AppController.setComponentEnabled(editableCheckBox, true);
			}
		}

		if (showChannel) {
			if (isNew || (0 == channel)) {
				replicateCheckBox.setSelected(false);
				AppController.setComponentEnabled(replicateCheckBox, false);
			} else {
				AppController.setComponentEnabled(replicateCheckBox, true);
			}
			AppController.setComponentEnabled(changeChannelCheckBox, true);
			AppController.setComponentEnabled(channelField, (replicateCheckBox.isSelected() ||
				changeChannelCheckBox.isSelected()));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Subclass can override to allow the name input to be blank.

	protected boolean isNameRequired() {

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Override to suppress error when replicate or change channel requested on the same channel.

	protected boolean noReplicateChangeSameChannel() {

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check for valid input. 

	public boolean validateInput() {

		name = "";

		if (showName) {
			name = nameField.getText().trim();
			if (isNameRequired() && (0 == name.length())) {
				errorReporter.reportWarning("Please provide a name");
				return false;
			}
		}

		isLocked = true;

		if (showEdit) {
			isLocked = !editableCheckBox.isSelected();
		}

		// Optionally this can allow multiple channels to be entered for replication or channel change.  In either case
		// multiple studies will be built all with the same settings except for the channel.

		replicate = false;
		changeChannel = false;
		studyChannel = 0;
		studyChannels = null;
		includeOriginalChannelStudy = false;

		if (showChannel) {

			replicate = replicateCheckBox.isSelected();
			changeChannel = (changeChannelCheckBox.isSelected() && !replicate);

			if (replicate || changeChannel) {

				boolean error = false;
				TreeSet<Integer> theChannels = new TreeSet<Integer>();

				String inputString = channelField.getText().trim();
				if (inputString.length() > 0) {

					String[] parts = inputString.split(","), chans = null;

					int chan = 0, chan1 = 0, chan2 = 0;

					for (String part : parts) {

						if (part.contains("-")) {
							chans = part.split("-");
							if (2 == chans.length) {
								try {
									chan1 = Integer.parseInt(chans[0].trim());
								} catch (NumberFormatException ne) {
									error = true;
								}
								if (error) {
									break;
								}
								if ((chan1 < SourceTV.CHANNEL_MIN) || (chan1 > SourceTV.CHANNEL_MAX)) {
									error = true;
									break;
								}
								try {
									chan2 = Integer.parseInt(chans[1].trim());
								} catch (NumberFormatException ne) {
									error = true;
								}
								if (error) {
									break;
								}
								if ((chan2 < SourceTV.CHANNEL_MIN) || (chan2 > SourceTV.CHANNEL_MAX)) {
									error = true;
									break;
								}
								if (chan1 <= chan2) {
									for (chan = chan1; chan <= chan2; chan++) {
										theChannels.add(Integer.valueOf(chan));
									}
								} else {
									for (chan = chan2; chan <= chan1; chan++) {
										theChannels.add(Integer.valueOf(chan));
									}
								}
							} else {
								error = true;
								break;
							}

						} else {

							try {
								chan = Integer.parseInt(part.trim());
							} catch (NumberFormatException ne) {
								error = true;
							}
							if (error) {
								break;
							}
							if ((chan < SourceTV.CHANNEL_MIN) || (chan > SourceTV.CHANNEL_MAX)) {
								error = true;
								break;
							}
							theChannels.add(Integer.valueOf(chan));
						}
					}

				} else {
					error = true;
				}

				// Usually entering the same channel for replication of a digital record or for channel change of any
				// type of record is an error (it is not an error for replication of an analog record, that sets up
				// on-channel replication to digital).  However as a convenience, if multiple channels are entered
				// and the same channel appears in the list it is not reported as an error, instead a separate flag is
				// set so a normal study on the original channel will also be included in the build list.

				if (replicate || changeChannel) {

					String ctype = (replicate ? "replication" : "study");

					if (allowMultiChannel) {

						if (error || theChannels.isEmpty()) {
							errorReporter.reportWarning("Please enter valid " + ctype + " channels\n" +
								"Use commas and dashes for multiple channels and ranges");
							return false;
						}

						if ((isDigital || changeChannel) && theChannels.remove(Integer.valueOf(channel))) {
							if (theChannels.isEmpty() && noReplicateChangeSameChannel()) {
								errorReporter.reportWarning("The " + ctype +
									" channel must be different than the original channel");
								return false;
							} else {
								includeOriginalChannelStudy = true;
							}
						}

						studyChannels = new int[theChannels.size()];
						int i = 0;
						for (Integer theChan : theChannels) {
							studyChannels[i++] = theChan.intValue();
						}

						studyChannel = studyChannels[0];

					} else {

						if (error || theChannels.isEmpty()) {
							errorReporter.reportWarning("Please enter a valid " + ctype + " channel");
							return false;
						}

						if (theChannels.size() > 1) {
							errorReporter.reportWarning("Please enter a single " + ctype + " channel");
							return false;
						}

						studyChannel = theChannels.first().intValue();
						if ((isDigital || changeChannel) && (studyChannel == channel) &&
								noReplicateChangeSameChannel()) {
							errorReporter.reportWarning("The " + ctype +
								" channel must be different than the original channel");
							studyChannel = 0;
							return false;
						}
					}
				}
			}
		}

		if (showOutput) {

			if ((null == fileOutputConfig) || fileOutputConfig.isNull() || !fileOutputConfig.isValid()) {
				errorReporter.reportWarning("Please provide valid output file settings");
				return false;
			}

			if ((null == mapOutputConfig) || mapOutputConfig.isNull() || !mapOutputConfig.isValid()) {
				errorReporter.reportWarning("Please provide valid map output settings");
				return false;
			}

			fileOutputConfig.saveAsLastUsed(getDbID());
			mapOutputConfig.saveAsLastUsed(getDbID());

		} else {

			fileOutputConfig = null;
			mapOutputConfig = null;
		}

		memoryFraction = 1.;

		if (showRun) {

			memoryFraction = 1. / (double)(memoryFractionMenu.getSelectedIndex() + 1);
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// For use by subclasses that use a StudyBuild subclass to manage the study creation process.  Some may cause
	// multiple studies to be created.

	public ArrayList<StudyBuild> getStudyBuilds() {

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillOpen() {

		clearFields();
	}


	//=================================================================================================================
	// Subclass for a basic add-new-record case, just the superclass UI for record edit and replicate.  The superclass
	// doesn't really need to be abstract, it's fully functional so trivial subclasses like this can exist to protect
	// the world from having to use that superclass contructor with the long cryptic argument list.

	public static class AddRecord extends OptionsPanel {


		//-------------------------------------------------------------------------------------------------------------

		public AddRecord(AppEditor theParent) {

			super(theParent, false, null, false, null, true, true, false, false, false, false);

			setBorder(BorderFactory.createTitledBorder("Record Options"));
		}
	}
		

	//=================================================================================================================
	// Subclass for a new-scenario case, show scenario name and replicate options.

	public static class NewScenario extends OptionsPanel {


		//-------------------------------------------------------------------------------------------------------------

		public NewScenario(AppEditor theParent) {

			super(theParent, true, "Scenario Name", false, null, false, true, false, false, false, false);

			setBorder(BorderFactory.createTitledBorder("Scenario Settings"));
		}
	}
		

	//=================================================================================================================
	// Subclass used when creating a new interference check study.

	public static class IxCheckStudy extends OptionsPanel implements ExtDbListener {

		private KeyedRecordMenu templateMenu;

		private KeyedRecordMenu extDbMenu;

		private JComboBox<String> cellSizeMenu;
		private JTextField profilePtIncField;

		private JButton setBeforeButton;
		private JButton clearBeforeButton;
		private JLabel beforeLabel;
		private JCheckBox useDefaultBeforeCheckBox;
		private JPanel beforePanel;

		private JCheckBox protectPreBaselineCheckBox;
		private JCheckBox protectBaselineFromLPTVCheckBox;
		private JCheckBox protectLPTVFromClassACheckBox;

		private JCheckBox includeForeignCheckBox;
		private JTextArea includeUserRecordsTextArea;
		private JButton includeUserRecordButton;
		private JCheckBox cpExcludesBaselineCheckBox;
		private JCheckBox excludeAppsCheckBox;
		private JCheckBox includeAmendmentsCheckBox;
		private JCheckBox excludePendingCheckBox;
		private JCheckBox excludePostTransitionCheckBox;
		private JCheckBox excludeNewLPTVCheckBox;
		private JTextArea excludeCommandsTextArea;
		private JButton excludeARNButton;

		private JPanel moreOptionsMainPanel;
		private JPanel moreOptionsButtonPanel;
		private AppDialog moreOptionsDialog;
		private JButton moreOptionsButton;

		private DateSelectionPanel filingCutoffDatePanel;

		private boolean isPostWindow;

		private StationRecord selectedRecord;

		// Values are set directly in a StudyBuildIxCheck object provided to the constructor.

		private StudyBuildIxCheck studyBuild;

		// This may cause multiple studies to be built on multiple channels so the return is an array.

		private ArrayList<StudyBuild> studyBuilds;


		//-------------------------------------------------------------------------------------------------------------
		// Show the name, description, channel, output, and run options from the superclass.  Except, multiple
		// channels, output, and run are not shown if this is not a full study build; see StudyBuildIxCheck.

		public IxCheckStudy(AppEditor theParent, StudyBuildIxCheck theStudy) {

			super(theParent, true, "Study Name", true, "Study Description", false, true, theStudy.buildFullStudy,
				theStudy.buildFullStudy, false, theStudy.buildFullStudy);

			if (theStudy.buildFullStudy) {
				setBorder(BorderFactory.createTitledBorder("Study Build Settings"));
			} else {
				setBorder(BorderFactory.createTitledBorder("Study Settings"));
			}

			studyBuild = theStudy;

			defaultName = theStudy.studyName;
			defaultDescription = theStudy.studyDescription;
			defaultFileOutputConfig = theStudy.fileOutputConfig;
			defaultMapOutputConfig = theStudy.mapOutputConfig;

			templateMenu = new KeyedRecordMenu();
			templateMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXy"));

			JPanel templatePanel = new JPanel();
			templatePanel.setBorder(BorderFactory.createTitledBorder("Template"));
			templatePanel.add(templateMenu);

			extDbMenu = new KeyedRecordMenu();
			extDbMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXy"));

			JPanel extDbPanel = new JPanel();
			extDbPanel.setBorder(BorderFactory.createTitledBorder("Station Data"));
			extDbPanel.add(extDbMenu);

			cellSizeMenu = new JComboBox<String>();
			cellSizeMenu.setFocusable(false);
			for (double siz : StudyBuildIxCheck.CELL_SIZES) {
				cellSizeMenu.addItem(AppCore.formatDecimal(siz, 0, 1));
			}

			JPanel cellSizePanel = new JPanel();
			cellSizePanel.setBorder(BorderFactory.createTitledBorder("Cell size, km"));
			cellSizePanel.add(cellSizeMenu);

			profilePtIncField = new JTextField(8);
			AppController.fixKeyBindings(profilePtIncField);
			profilePtIncField.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					String str = profilePtIncField.getText().trim();
					if (str.length() > 0) {
						try {
							double d = Double.parseDouble(str);
							if (d > 1.) {
								d = 1. / d;
							}
							profilePtIncField.setText(AppCore.formatDecimal(d, 0, 3));
						} catch (NumberFormatException ne) {
						}
					}
				}
			});
			profilePtIncField.addFocusListener(new FocusAdapter() {
				public void focusLost(FocusEvent theEvent) {
					if (!theEvent.isTemporary()) {
						profilePtIncField.postActionEvent();
					}
				}
			});

			JPanel profilePtIncPanel = new JPanel();
			profilePtIncPanel.setBorder(BorderFactory.createTitledBorder("Profile inc., km"));
			profilePtIncPanel.add(profilePtIncField);

			// Options to choose the "before" record for the study.  Location in layout will vary depending on whether
			// this is a full build or not, see below.

			setBeforeButton = new JButton("Set");
			setBeforeButton.setFocusable(false);
			setBeforeButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					AppEditor theParent = parent;
					if (null != moreOptionsDialog) {
						theParent = moreOptionsDialog;
					}
					RecordFindDialog theDialog = new RecordFindDialog(theParent, "Choose proposal \"before\" record",
						Source.RECORD_TYPE_TV);
					AppController.showWindow(theDialog);
					if (!theDialog.canceled) {
						StationRecord theRecord = theDialog.getSelectedRecord();
						if (theRecord instanceof SourceEditDataTV) {
							studyBuild.didSetBefore = true;
							studyBuild.beforeSource = (SourceEditDataTV)theRecord;
							beforeLabel.setText(studyBuild.beforeSource.toString());
							studyBuild.beforeRecord = null;
						} else {
							if (theRecord instanceof ExtDbRecordTV) {
								studyBuild.didSetBefore = true;
								studyBuild.beforeSource = null;
								studyBuild.beforeRecord = (ExtDbRecordTV)theRecord;
								beforeLabel.setText(studyBuild.beforeRecord.toString());
							} else {
								errorReporter.reportError("That record type is not allowed here");
							}
						}
					}
				}
			});

			clearBeforeButton = new JButton("Clear");
			clearBeforeButton.setFocusable(false);
			clearBeforeButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					studyBuild.didSetBefore = true;
					studyBuild.beforeSource = null;
					studyBuild.beforeRecord = null;
					beforeLabel.setText("(none)");
				}
			});

			// If this is a full study build, show all UI; the "before" record UI shows option to use default record
			// found during the build, and that UI goes in the secondary dialog with other less-used options.  If this
			// is not a full build, various build options and that secondary dialog will not be shown.  This is just
			// going to create a "blank slate" new study with only a proposal record and scenario, to be opened in an
			// editor.  In that case the "before" record UI is in the main panel and there is no default, the record
			// must be set manually or no record will be used.

			if (theStudy.buildFullStudy) {

				useDefaultBeforeCheckBox = new JCheckBox("Default");
				useDefaultBeforeCheckBox.setFocusable(false);
				useDefaultBeforeCheckBox.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (useDefaultBeforeCheckBox.isSelected()) {
							studyBuild.didSetBefore = false;
							setBeforeButton.setEnabled(false);
							clearBeforeButton.setEnabled(false);
							beforeLabel.setText("(default)");
						} else {
							studyBuild.didSetBefore = true;
							setBeforeButton.setEnabled(true);
							clearBeforeButton.setEnabled(true);
							if (null != studyBuild.beforeSource) {
								beforeLabel.setText(studyBuild.beforeSource.toString());
							} else {
								if (null != studyBuild.beforeRecord) {
									beforeLabel.setText(studyBuild.beforeRecord.toString());
								} else {
									beforeLabel.setText("(none)");
								}
							}
						}
					}
				});

				beforeLabel = new JLabel("(default)");

				// Build options.

				protectPreBaselineCheckBox = new JCheckBox("Protect records not on baseline channel");
				protectPreBaselineCheckBox.setFocusable(false);
				protectBaselineFromLPTVCheckBox = new JCheckBox("Protect baseline records from LPTV");
				protectBaselineFromLPTVCheckBox.setFocusable(false);
				protectLPTVFromClassACheckBox = new JCheckBox("Protect LPTV records from Class A");
				protectLPTVFromClassACheckBox.setFocusable(false);

				// Options for including and excluding records during searches appear in a separate dialog.

				includeForeignCheckBox = new JCheckBox("Include non-U.S. records");
				includeForeignCheckBox.setFocusable(false);

				includeUserRecordsTextArea = new JTextArea(4, 16);
				AppController.fixKeyBindings(includeUserRecordsTextArea);
				includeUserRecordsTextArea.setLineWrap(true);
				includeUserRecordsTextArea.setWrapStyleWord(true);

				includeUserRecordButton = new JButton("Add");
				includeUserRecordButton.setFocusable(false);
				includeUserRecordButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						AppEditor theParent = parent;
						if (null != moreOptionsDialog) {
							theParent = moreOptionsDialog;
						}
						RecordFindDialog theDialog = new RecordFindDialog(theParent, "Choose user record to include",
							-Source.RECORD_TYPE_TV);
						AppController.showWindow(theDialog);
						if (!theDialog.canceled) {
							String theID = theDialog.getSelectedRecord().getRecordID();
							if ((null != theID) && (theID.length() > 0)) {
								includeUserRecordsTextArea.append(theID + "\n");
							}
						}
					}
				});

				JPanel incUserButP = new JPanel(new FlowLayout(FlowLayout.LEFT));
				incUserButP.add(includeUserRecordButton);

				JPanel incUserPanel = new JPanel(new BorderLayout());
				incUserPanel.setBorder(BorderFactory.createTitledBorder("User records to include"));
				incUserPanel.add(AppController.createScrollPane(includeUserRecordsTextArea), BorderLayout.CENTER);
				incUserPanel.add(incUserButP, BorderLayout.SOUTH);

				cpExcludesBaselineCheckBox = new JCheckBox("CP excludes station's baseline");
				cpExcludesBaselineCheckBox.setFocusable(false);

				excludeAppsCheckBox = new JCheckBox("Exclude all APP records");
				excludeAppsCheckBox.setFocusable(false);

				includeAmendmentsCheckBox = new JCheckBox("Include all AMD records");
				includeAmendmentsCheckBox.setFocusable(false);

				excludePendingCheckBox = new JCheckBox("Exclude pending license records");
				excludePendingCheckBox.setFocusable(false);

				// Checking the excludePostTransition option will automatically check the protectPreBaseline option too
				// because that is almost always what the user will want, however this is just advisory and one-way,
				// the user can still un-check protectPreBaseline and that does not affect the exclude option.

				excludePostTransitionCheckBox = new JCheckBox("Exclude all post-transition CP, APP, and BL");
				excludePostTransitionCheckBox.setFocusable(false);
				excludePostTransitionCheckBox.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (excludePostTransitionCheckBox.isSelected()) {
							protectPreBaselineCheckBox.setSelected(true);
						}
					}
				});

				excludeNewLPTVCheckBox = new JCheckBox("Exclude records for new LPTV stations");
				excludeNewLPTVCheckBox.setFocusable(false);

				excludeCommandsTextArea = new JTextArea(4, 16);
				AppController.fixKeyBindings(excludeCommandsTextArea);
				excludeCommandsTextArea.setLineWrap(true);
				excludeCommandsTextArea.setWrapStyleWord(true);

				excludeARNButton = new JButton("Add");
				excludeARNButton.setFocusable(false);
				excludeARNButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						AppEditor theParent = parent;
						if (null != moreOptionsDialog) {
							theParent = moreOptionsDialog;
						}
						RecordFindDialog theDialog = new RecordFindDialog(theParent, "Choose record to exclude",
							Source.RECORD_TYPE_TV);
						AppController.showWindow(theDialog);
						if (!theDialog.canceled) {
							String theARN = theDialog.getSelectedRecord().getARN();
							if ((null != theARN) && (theARN.length() > 0)) {
								excludeCommandsTextArea.append(theARN + "\n");
							}
						}
					}
				});

				JPanel excARNButP = new JPanel(new FlowLayout(FlowLayout.LEFT));
				excARNButP.add(excludeARNButton);

				JPanel excARNsPanel = new JPanel(new BorderLayout());
				excARNsPanel.setBorder(BorderFactory.createTitledBorder("Records to exclude (ARNs)"));
				excARNsPanel.add(AppController.createScrollPane(excludeCommandsTextArea), BorderLayout.CENTER);
				excARNsPanel.add(excARNButP, BorderLayout.SOUTH);

				filingCutoffDatePanel = new DateSelectionPanel(this, "Cutoff Date", false);
				filingCutoffDatePanel.setFutureAllowed(true);

				Box moreOptsBox = Box.createVerticalBox();
				moreOptsBox.add(cpExcludesBaselineCheckBox);
				moreOptsBox.add(excludeAppsCheckBox);
				moreOptsBox.add(includeAmendmentsCheckBox);
				moreOptsBox.add(excludePendingCheckBox);
				moreOptsBox.add(excludePostTransitionCheckBox);
				moreOptsBox.add(excludeNewLPTVCheckBox);

				JPanel moreOptsP = new JPanel(new FlowLayout(FlowLayout.LEFT));
				moreOptsP.add(moreOptsBox);

				moreOptionsMainPanel = new JPanel();
				moreOptionsMainPanel.setLayout(new BoxLayout(moreOptionsMainPanel, BoxLayout.Y_AXIS));
				moreOptionsMainPanel.add(moreOptsP);
				moreOptionsMainPanel.add(excARNsPanel);
				moreOptionsMainPanel.add(incUserPanel);
				moreOptionsMainPanel.add(filingCutoffDatePanel);

				JButton optsOKBut = new JButton("OK");
				optsOKBut.setFocusable(false);
				optsOKBut.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if ((null != moreOptionsDialog) && moreOptionsDialog.isVisible()) {
							AppController.hideWindow(moreOptionsDialog);
						}
					}
				});

				moreOptionsButtonPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
				moreOptionsButtonPanel.add(optsOKBut);

				moreOptionsButton = new JButton("More");
				moreOptionsButton.setFocusable(false);
				moreOptionsButton.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						if (null == moreOptionsDialog) {
							moreOptionsDialog = new AppDialog(parent, "Additional Options",
									Dialog.ModalityType.APPLICATION_MODAL) {
								public void windowWillOpen() {
									blockActionsClear();
								}
								public void windowWillClose() {
									blockActionsSet();
								}
							};
							filingCutoffDatePanel.setParent(moreOptionsDialog);
							moreOptionsDialog.add(moreOptionsMainPanel, BorderLayout.CENTER);
							moreOptionsDialog.add(moreOptionsButtonPanel, BorderLayout.SOUTH);
							moreOptionsDialog.pack();
							moreOptionsDialog.setMinimumSize(moreOptionsDialog.getSize());
							moreOptionsDialog.setLocationRelativeTo(moreOptionsButton);
						}
						if (!moreOptionsDialog.isVisible()) {
							AppController.showWindow(moreOptionsDialog);
						}
					}
				});

			} else {

				beforeLabel = new JLabel("(none)");
			}

			// Layout.

			Box col1Box = Box.createVerticalBox();
			col1Box.add(channelPanel);
			col1Box.add(templatePanel);

			add(col1Box);

			Box col2Box = Box.createVerticalBox();
			col2Box.add(extDbPanel);
			col2Box.add(namePanel);
			col2Box.add(descriptionPanel);

			add(col2Box);

			Box col3Box = Box.createVerticalBox();
			col3Box.add(cellSizePanel);
			col3Box.add(profilePtIncPanel);

			add(col3Box);

			JPanel befTopP = new JPanel(new FlowLayout(FlowLayout.LEFT));
			befTopP.add(beforeLabel);

			JPanel befBotP = new JPanel(new FlowLayout(FlowLayout.LEFT));
			befBotP.add(setBeforeButton);
			befBotP.add(clearBeforeButton);

			beforePanel = new JPanel();
			beforePanel.setBorder(BorderFactory.createTitledBorder("Proposal \"before\" case"));
			beforePanel.setLayout(new BoxLayout(beforePanel, BoxLayout.Y_AXIS));
			beforePanel.add(befTopP);
			beforePanel.add(befBotP);

			if (theStudy.buildFullStudy) {

				moreOptionsMainPanel.add(beforePanel);

				runPanel.remove(commentButton);
				descriptionPanel.add(commentButton);

				befBotP.add(useDefaultBeforeCheckBox);

				Box optsBox = Box.createVerticalBox();
				optsBox.add(protectPreBaselineCheckBox);
				optsBox.add(protectBaselineFromLPTVCheckBox);
				optsBox.add(protectLPTVFromClassACheckBox);
				optsBox.add(includeForeignCheckBox);

				JPanel topRowP = new JPanel();
				topRowP.add(optsBox);
				topRowP.add(moreOptionsButton);

				JPanel botRowP = new JPanel();
				botRowP.add(outputPanel);
				botRowP.add(runPanel);

				Box col4Box = Box.createVerticalBox();
				col4Box.add(topRowP);
				col4Box.add(botRowP);

				add(col4Box);

			} else {

				add(beforePanel);
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// The layout is done directly above.

		protected void createLayout(JPanel thePanel) {

			return;
		}


		//-------------------------------------------------------------------------------------------------------------

		public void setEnabled(boolean flag) {

			super.setEnabled(flag);

			AppController.setComponentEnabled(templateMenu, flag);

			AppController.setComponentEnabled(extDbMenu, flag);

			AppController.setComponentEnabled(cellSizeMenu, flag);
			AppController.setComponentEnabled(profilePtIncField, flag);

			if (studyBuild.buildFullStudy) {

				AppController.setComponentEnabled(useDefaultBeforeCheckBox, flag);
				if (useDefaultBeforeCheckBox.isSelected()) {
					setBeforeButton.setEnabled(false);
					clearBeforeButton.setEnabled(false);
				} else {
					setBeforeButton.setEnabled(flag);
					clearBeforeButton.setEnabled(flag);
				}

				AppController.setComponentEnabled(protectPreBaselineCheckBox,
					(flag && (null != studyBuild.baselineDate)));
				AppController.setComponentEnabled(protectBaselineFromLPTVCheckBox, (flag && isLPTV));
				AppController.setComponentEnabled(protectLPTVFromClassACheckBox, (flag && isClassA && !isPostWindow));

				AppController.setComponentEnabled(includeForeignCheckBox, flag);
				AppController.setComponentEnabled(includeUserRecordsTextArea, flag);
				includeUserRecordButton.setEnabled(flag);

				AppController.setComponentEnabled(cpExcludesBaselineCheckBox, flag);
				AppController.setComponentEnabled(excludeAppsCheckBox, flag);
				AppController.setComponentEnabled(includeAmendmentsCheckBox, flag);
				AppController.setComponentEnabled(excludePendingCheckBox, flag);
				AppController.setComponentEnabled(excludePostTransitionCheckBox, flag);
				AppController.setComponentEnabled(excludeNewLPTVCheckBox, flag);
				AppController.setComponentEnabled(excludeCommandsTextArea, flag);
				excludeARNButton.setEnabled(flag);

				moreOptionsButton.setEnabled(flag);

				filingCutoffDatePanel.setEnabled(flag);

			} else {

				setBeforeButton.setEnabled(flag);
				clearBeforeButton.setEnabled(flag);
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Update data set menu when notified by ExtDb.

		public void updateExtDbList() {

			SwingUtilities.invokeLater(new Runnable() {
				public void run() {

					ArrayList<KeyedRecord> list = ExtDb.getExtDbList(getDbID(), ExtDb.DB_TYPE_LMS,
						StudyBuildIxCheck.MIN_LMS_VERSION);
					if (null == list) {
						return;
					}

					blockActionsStart();

					int selectKey = extDbMenu.getSelectedKey();
					extDbMenu.removeAllItems();
					if (!list.isEmpty()) {
						extDbMenu.addAllItems(list);
						if (extDbMenu.containsKey(selectKey)) {
							extDbMenu.setSelectedKey(selectKey);
						}
					}

					blockActionsEnd();
				}
			});
		}


		//-------------------------------------------------------------------------------------------------------------
		// Populate menus when the containing window is shown.

		public void windowWillOpen() {

			ExtDb.addListener(this);

			templateMenu.removeAllItems();
			extDbMenu.removeAllItems();

			SwingUtilities.invokeLater(new Runnable() {
				public void run() {

					blockActionsStart();

					errorReporter.setTitle("Load Template List");

					ArrayList<KeyedRecord> list = Template.getTemplateInfoList(getDbID(), errorReporter);
					if (null != list) {
						Template.Info theInfo;
						for (KeyedRecord theItem : list) {
							theInfo = (Template.Info)theItem;
							if (theInfo.isLocked && !theInfo.isLockedInStudy) {
								templateMenu.addItem(theItem);
							}
						}
					}

					errorReporter.setTitle("Load Station Data List");

					list = ExtDb.getExtDbList(getDbID(), ExtDb.DB_TYPE_LMS, StudyBuildIxCheck.MIN_LMS_VERSION,
						errorReporter);
					if (null != list) {
						if (list.isEmpty()) {
							errorReporter.reportError(
								"No compatible LMS station data found, use Station Data Manager to add data");
						} else {
							extDbMenu.addAllItems(list);
						}
					}

					clearFields();

					blockActionsEnd();
				}
			});
		}


		//-------------------------------------------------------------------------------------------------------------

		public void windowWillClose() {

			ExtDb.removeListener(this);
		}


		//-------------------------------------------------------------------------------------------------------------
		// The IxCheckStudy object provided at construction may provide defaults for some or all fields.

		public void clearFields() {

			super.clearFields();

			blockActionsStart();

			if ((studyBuild.templateKey > 0) && templateMenu.containsKey(studyBuild.templateKey)) {
				templateMenu.setSelectedKey(studyBuild.templateKey);
			} else {
				templateMenu.setSelectedIndex(0);
			}

			applyDefaults();

			if (null != studyBuild.extDb) {
				int theKey = studyBuild.extDb.key.intValue();
				if (extDbMenu.containsKey(theKey)) {
					extDbMenu.setSelectedKey(theKey);
				} else {
					extDbMenu.setSelectedIndex(0);
				}
			} else {
				extDbMenu.setSelectedIndex(0);
			}

			studyBuild.beforeSource = null;
			studyBuild.beforeRecord = null;
			setBeforeButton.setEnabled(false);
			clearBeforeButton.setEnabled(false);

			if (studyBuild.buildFullStudy) {

				studyBuild.didSetBefore = false;
				beforeLabel.setText("(default)");
				useDefaultBeforeCheckBox.setSelected(true);

				protectPreBaselineCheckBox.setSelected(studyBuild.protectPreBaseline);
				protectBaselineFromLPTVCheckBox.setSelected(studyBuild.protectBaselineFromLPTV && isLPTV);
				protectLPTVFromClassACheckBox.setSelected(studyBuild.protectLPTVFromClassA && isClassA);

				includeForeignCheckBox.setSelected(studyBuild.includeForeign);
				if (null != studyBuild.includeUserRecords) {
					StringBuilder e = new StringBuilder();
					for (Integer uid : studyBuild.includeUserRecords) {
						e.append(String.valueOf(uid));
						e.append("\n");
					}
					includeUserRecordsTextArea.setText(e.toString());
				} else {
					includeUserRecordsTextArea.setText("");
				}

				cpExcludesBaselineCheckBox.setSelected(studyBuild.cpExcludesBaseline);
				excludeAppsCheckBox.setSelected(studyBuild.excludeApps);
				includeAmendmentsCheckBox.setSelected(studyBuild.includeAmendments);
				excludePendingCheckBox.setSelected(studyBuild.excludePending);
				excludePostTransitionCheckBox.setSelected(studyBuild.excludePostTransition);
				excludeNewLPTVCheckBox.setSelected(studyBuild.excludeNewLPTV);
				if (null != studyBuild.excludeCommands) {
					StringBuilder e = new StringBuilder();
					for (String cmd : studyBuild.excludeCommands) {
						e.append(cmd);
						e.append("\n");
					}
					excludeCommandsTextArea.setText(e.toString());
				} else {
					excludeCommandsTextArea.setText("");
				}

				filingCutoffDatePanel.setDate(studyBuild.filingCutoffDate);

			} else {

				studyBuild.didSetBefore = true;
				beforeLabel.setText("(none)");
			}

			blockActionsEnd();
		}


		//-------------------------------------------------------------------------------------------------------------
		// Set UI fields to defaults from the study build object.

		private void applyDefaults() {

			String str;
			if (isLPTV || (isClassA && isPostWindow)) {
				str = studyBuild.defaultCellSizeLPTV;
			} else {
				str = studyBuild.defaultCellSize;
			}
			if ((null != str) && (str.length() > 0)) {
				try {
					double d = Double.parseDouble(str);
					str = AppCore.formatDecimal(d, 0, 1);
					cellSizeMenu.setSelectedItem(str);
				} catch (NumberFormatException nfe) {
				}
			}

			if (isLPTV || (isClassA && isPostWindow)) {
				str = studyBuild.defaultProfilePpkLPTV;
			} else {
				str = studyBuild.defaultProfilePpk;
			}
			if ((null != str) && (str.length() > 0)) {
				try {
					double d = Double.parseDouble(str);
					if (d > 0.) {
						d = 1. / d;
					}
					profilePtIncField.setText(AppCore.formatDecimal(d, 0, 3));
				} catch (NumberFormatException nfe) {
				}
			}

			if (studyBuild.buildFullStudy) {

				if (null != studyBuild.baselineDate) {
					AppController.setComponentEnabled(protectPreBaselineCheckBox, enabled);
				} else {
					protectPreBaselineCheckBox.setSelected(false);
					AppController.setComponentEnabled(protectPreBaselineCheckBox, false);
				}

				protectLPTVFromClassACheckBox.setSelected(studyBuild.protectLPTVFromClassA && isClassA);
				AppController.setComponentEnabled(protectLPTVFromClassACheckBox,
					(enabled && isClassA && !isPostWindow));
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// When a record from a data set is selected, set default values for cell size and profile resolution, also
		// set the file number as the study name, and select the station data set.

		public void selectionChanged(Object newSelection) {

			super.selectionChanged(newSelection);

			selectedRecord = null;

			isPostWindow = false;

			if (null == newSelection) {
				return;
			}

			if (newSelection instanceof SourceEditData) {

				SourceEditData theSource = (SourceEditData)newSelection;
				if (Source.RECORD_TYPE_TV == theSource.recordType) {
					selectedRecord = theSource;
					if (null != theSource.extDbKey) {
						int theKey = theSource.extDbKey.intValue();
						if (extDbMenu.containsKey(theKey)) {
							extDbMenu.setSelectedKey(theKey);
						}
					}
				}

			} else {

				if (newSelection instanceof ExtDbRecord) {

					ExtDbRecord theRecord = (ExtDbRecord)newSelection;
					if (Source.RECORD_TYPE_TV == theRecord.recordType) {
						selectedRecord = theRecord;
						int theKey = theRecord.extDb.key.intValue();
						if (extDbMenu.containsKey(theKey)) {
							extDbMenu.setSelectedKey(theKey);
						}
					}
				}
			}

			isPostWindow = ((null != studyBuild.filingWindowEndDate) && (null != sequenceDate) &&
				sequenceDate.after(studyBuild.filingWindowEndDate));

			String str;
			if (isLPTV || (isClassA && isPostWindow)) {
				str = studyBuild.defaultCellSizeLPTV;
			} else {
				str = studyBuild.defaultCellSize;
			}
			if ((null != str) && (str.length() > 0)) {
				try {
					double d = Double.parseDouble(str);
					str = AppCore.formatDecimal(d, 0, 1);
					cellSizeMenu.setSelectedItem(str);
				} catch (NumberFormatException nfe) {
				}
			}

			if (isLPTV || (isClassA && isPostWindow)) {
				str = studyBuild.defaultProfilePpkLPTV;
			} else {
				str = studyBuild.defaultProfilePpk;
			}
			if ((null != str) && (str.length() > 0)) {
				try {
					double d = Double.parseDouble(str);
					if (d > 0.) {
						d = 1. / d;
					}
					profilePtIncField.setText(AppCore.formatDecimal(d, 0, 3));
				} catch (NumberFormatException nfe) {
				}
			}

			if (newSelection instanceof StationRecord) {
				String theName = ((StationRecord)newSelection).getFileNumber();
				if ((null != theName) && (theName.length() > 0)) {
					nameField.setText(theName);
				}
			}

			if (studyBuild.buildFullStudy) {

				protectPreBaselineCheckBox.setSelected(studyBuild.protectPreBaseline);

				protectBaselineFromLPTVCheckBox.setSelected(studyBuild.protectBaselineFromLPTV && isLPTV);
				AppController.setComponentEnabled(protectBaselineFromLPTVCheckBox, (enabled && isLPTV));

				protectLPTVFromClassACheckBox.setSelected(studyBuild.protectLPTVFromClassA && isClassA);
				AppController.setComponentEnabled(protectLPTVFromClassACheckBox,
					(enabled && isClassA && !isPostWindow));
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		public boolean validateInput() {

			errorReporter.clearTitle();

			if (null == selectedRecord) {
				errorReporter.reportWarning("Please select a TV record to study");
				return false;
			}

			if (!super.validateInput()) {
				return false;
			}

			// If replicating or changing channel, space is reserved in the name to append the new channel(s).

			if (!DbCore.checkStudyName(name, getDbID(), false, ((replicate || changeChannel) ? 10 : 0),
					errorReporter)) {
				return false;
			}

			int templateKey = templateMenu.getSelectedKey();
			if (templateKey <= 0) {
				errorReporter.reportWarning("Please select a study template");
				return false;
			}

			ExtDb extDb = null;
			int theKey = extDbMenu.getSelectedKey();
			if (theKey > 0) {
				extDb = ExtDb.getExtDb(getDbID(), Integer.valueOf(theKey));
			} else {
				errorReporter.reportWarning("Please select a station data set");
				return false;
			}
			if ((null == extDb) || extDb.deleted) {
				errorReporter.reportWarning("That station data set has been deleted, please select another");
				return false;
			}

			String cellSize = (String)cellSizeMenu.getSelectedItem();

			String str = profilePtIncField.getText().trim();
			if (0 == str.length()) {
				errorReporter.reportWarning("Please enter a profile point spacing");
				return false;
			}
			double d = Parameter.MIN_PATH_TERR_RES - 1.;
			try {
				d = Double.parseDouble(str);
				if (d > 0.) {
					if (d <= 1.) {
						d = 1. / d;
					}
					profilePtIncField.setText(AppCore.formatDecimal((1. / d), 0, 3));
				}
			} catch (NumberFormatException ne) {
				errorReporter.reportWarning("The profile point spacing must be a number");
				return false;
			}
			if ((d < Parameter.MIN_PATH_TERR_RES) || (d > Parameter.MAX_PATH_TERR_RES)) {
				errorReporter.reportWarning("The profile point spacing must be in the range " +
					(1. / Parameter.MAX_PATH_TERR_RES) + " to " + (1. / Parameter.MIN_PATH_TERR_RES));
				return false;
			}
			String profilePpk = String.valueOf(d);

			TreeSet<String> newExcludeCommands = null;
			TreeSet<Integer> newIncludeUserRecords = null;

			if (studyBuild.buildFullStudy) {

				str = excludeCommandsTextArea.getText().trim();
				if (str.length() > 0) {
					newExcludeCommands = new TreeSet<String>();
					for (String cmd : str.split("\\s+")) {
						newExcludeCommands.add(cmd);
					}
				}

				str = includeUserRecordsTextArea.getText().trim();
				if (str.length() > 0) {
					newIncludeUserRecords = new TreeSet<Integer>();
					int id;
					for (String uid : str.split("\\s+")) {
						id = 0;
						try {
							id = Integer.parseInt(uid);
						} catch (NumberFormatException nfe) {
						}
						if (id <= 0) {
							errorReporter.reportWarning("Only user record IDs may appear in the include list");
							return false;
						}
						newIncludeUserRecords.add(Integer.valueOf(id));
					}
				}
			}

			// All good, save values.

			if (selectedRecord.isSource()) {
				studyBuild.source = (SourceEditData)selectedRecord;
			} else {
				studyBuild.record = (ExtDbRecord)selectedRecord;
			}

			studyBuild.templateKey = templateKey;

			studyBuild.extDb = extDb;

			studyBuild.studyName = name;

			studyBuild.studyDescription = null;
			if (description.length() > 0) {
				studyBuild.studyDescription = description;
			}

			studyBuild.cellSize = cellSize;
			studyBuild.profilePpk = profilePpk;

			// If this is to be a full study build (and possible auto-run), all UI was active, update everything then
			// build the list of study build objects the main UI will use to add to the run manager.

			if (studyBuild.buildFullStudy) {

				studyBuild.fileOutputConfig = fileOutputConfig;
				studyBuild.mapOutputConfig = mapOutputConfig;

				studyBuild.protectPreBaseline = protectPreBaselineCheckBox.isSelected();
				studyBuild.protectBaselineFromLPTV = protectBaselineFromLPTVCheckBox.isSelected();
				studyBuild.protectLPTVFromClassA = protectLPTVFromClassACheckBox.isSelected();

				studyBuild.includeForeign = includeForeignCheckBox.isSelected();
				studyBuild.includeUserRecords = newIncludeUserRecords;

				studyBuild.cpExcludesBaseline = cpExcludesBaselineCheckBox.isSelected();
				studyBuild.excludeApps = excludeAppsCheckBox.isSelected();
				studyBuild.includeAmendments = includeAmendmentsCheckBox.isSelected();
				studyBuild.excludePending = excludePendingCheckBox.isSelected();
				studyBuild.excludePostTransition = excludePostTransitionCheckBox.isSelected();
				studyBuild.excludeNewLPTV = excludeNewLPTVCheckBox.isSelected();
				studyBuild.excludeCommands = newExcludeCommands;

				studyBuild.filingCutoffDate = filingCutoffDatePanel.getDate();

				// If replicating or changing channel, a normal study on the original channel may be added first, see
				// super.validateInput(), then add duplicates for each channel.  Otherwise just add the normal study.

				studyBuilds = new ArrayList<StudyBuild>();

				if (replicate || changeChannel) {

					if (includeOriginalChannelStudy) {
						studyBuilds.add(studyBuild);
					}

					StudyBuildIxCheck theBuild;

					for (int theChan : studyChannels) {

						theBuild = studyBuild.copy();

						theBuild.replicate = replicate;
						theBuild.changeChannel = changeChannel;
						theBuild.studyChannel = theChan;

						theBuild.studyName = name + " (on " + String.valueOf(theChan) + ")";

						studyBuilds.add(theBuild);
					}

				} else {

					studyBuilds.add(studyBuild);
				}

			// If this is not a full build, the main study build object will be retrieved and used in a different
			// context by the study manager.

			} else {

				if (replicate || changeChannel) {

					studyBuild.replicate = replicate;
					studyBuild.changeChannel = changeChannel;
					studyBuild.studyChannel = studyChannel;

					studyBuild.studyName = name + " (on " + String.valueOf(studyChannel) + ")";
				}
			}

			return true;
		}


		//-------------------------------------------------------------------------------------------------------------

		public StudyBuildIxCheck getStudyBuild() {

			return studyBuild;
		}


		//-------------------------------------------------------------------------------------------------------------

		public ArrayList<StudyBuild> getStudyBuilds() {

			return studyBuilds;
		}
	}


	//=================================================================================================================
	// Subclass used when creating a wireless interference study.  User inputs study name, selects template, and
	// selects the wireless station data set to use for the initial scenario creation.  In the containing RecordFind
	// dialog the user picks the TV data set and the desired TV record.  This will allow multiple TV records to be
	// selected, accumulating a list, to build multiple studies using the same settings.

	public static class WirelessStudy extends OptionsPanel implements ExtDbListener {

		private KeyedRecordMenu templateMenu;

		private KeyedRecordMenu wirelessExtDbMenu;
		private JCheckBox maskingCheckBox;
		private KeyedRecordMenu maskingExtDbMenu;

		private JTextField frequencyField;
		private JTextField bandwidthField;

		private StationRecord selectedRecord;
		private RecordsTableModel selectedRecordsModel;
		private JTable selectedRecordsTable;
		private JButton addRecordButton;
		private JButton removeRecordButton;

		// Values are set directly in the study creation object.

		public StudyBuildWireless studyBuild;

		private ArrayList<StudyBuild> studyBuilds;


		//-------------------------------------------------------------------------------------------------------------
		// Show name, description, replication, output configuration, and run options from superclass.

		public WirelessStudy(AppEditor theParent, StudyBuildWireless theStudy) {

			super(theParent, true, "Study Name", true, "Study Description", false, true, false, true, false, true);

			setBorder(BorderFactory.createTitledBorder("Study Build Settings"));

			studyBuild = theStudy;

			defaultName = theStudy.studyName;
			defaultDescription = theStudy.studyDescription;

			templateMenu = new KeyedRecordMenu();
			templateMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXy"));

			templateMenu.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					if (blockActions()) {
						updateParams();
						blockActionsEnd();
					}
				}
			});

			JPanel templatePanel = new JPanel();
			templatePanel.setBorder(BorderFactory.createTitledBorder("Template"));
			templatePanel.add(templateMenu);

			wirelessExtDbMenu = new KeyedRecordMenu();
			wirelessExtDbMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXy"));

			JPanel wlMenuPanel = new JPanel();
			wlMenuPanel.setBorder(BorderFactory.createTitledBorder("Wireless Station Data"));
			wlMenuPanel.add(wirelessExtDbMenu);

			maskingCheckBox = new JCheckBox("Include masking interference");
			maskingCheckBox.setFocusable(false);
			maskingCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					AppController.setComponentEnabled(maskingExtDbMenu, maskingCheckBox.isSelected());
				}
			});

			maskingExtDbMenu = new KeyedRecordMenu();
			maskingExtDbMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXy"));
			AppController.setComponentEnabled(maskingExtDbMenu, false);

			JPanel tvMenuPanel = new JPanel();
			tvMenuPanel.setBorder(BorderFactory.createTitledBorder("TV Station Data"));
			tvMenuPanel.add(maskingExtDbMenu);

			frequencyField = new JTextField(7);
			AppController.fixKeyBindings(frequencyField);

			JPanel freqPanel = new JPanel();
			freqPanel.setBorder(BorderFactory.createTitledBorder("WL Freq., MHz"));
			freqPanel.add(frequencyField);

			bandwidthField = new JTextField(7);
			AppController.fixKeyBindings(bandwidthField);

			JPanel bandPanel = new JPanel();
			bandPanel.setBorder(BorderFactory.createTitledBorder("WL B/W, MHz"));
			bandPanel.add(bandwidthField);

			JPanel selectP = new JPanel(new BorderLayout());
			selectedRecordsModel = new RecordsTableModel(selectP);
			selectedRecordsTable = selectedRecordsModel.createTable();

			selectedRecordsTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
				public void valueChanged(ListSelectionEvent theEvent) {
					updateControls();
				}
			});

			addRecordButton = new JButton("Add");
			addRecordButton.setFocusable(false);
			addRecordButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doAddRecord();
				}
			});

			removeRecordButton = new JButton("Remove");
			removeRecordButton.setFocusable(false);
			removeRecordButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doRemoveRecord();
				}
			});

			// Layout.

			Box col1Box = Box.createVerticalBox();
			col1Box.add(channelPanel);
			col1Box.add(templatePanel);

			JPanel chkP = new JPanel(new FlowLayout(FlowLayout.LEFT));
			chkP.add(maskingCheckBox);

			Box col2Box = Box.createVerticalBox();
			col2Box.add(wlMenuPanel);
			col2Box.add(chkP);
			col2Box.add(tvMenuPanel);

			Box col3Box = Box.createVerticalBox();
			col3Box.add(freqPanel);
			col3Box.add(bandPanel);

			JPanel nameDescP = new JPanel();
			nameDescP.add(namePanel);
			nameDescP.add(descriptionPanel);

			JPanel outRunP = new JPanel();
			outRunP.add(outputPanel);
			outRunP.add(runPanel);

			Box col4Box = Box.createVerticalBox();
			col4Box.add(nameDescP);
			col4Box.add(outRunP);

			JPanel topP = new JPanel();
			topP.add(col1Box);
			topP.add(col2Box);
			topP.add(col3Box);
			topP.add(col4Box);

			Box srButB = Box.createVerticalBox();
			srButB.add(addRecordButton);
			srButB.add(removeRecordButton);

			selectP.add(AppController.createScrollPane(selectedRecordsTable), BorderLayout.CENTER);
			selectP.add(srButB, BorderLayout.WEST);

			setLayout(new BorderLayout());
			add(topP, BorderLayout.NORTH);
			add(selectP, BorderLayout.CENTER);

			selectedRecordsModel.updateBorder();
			updateControls();
		}


		//-------------------------------------------------------------------------------------------------------------
		// The layout is done directly.

		protected void createLayout(JPanel thePanel) {

			return;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void updateParams() {

			int tempKey = templateMenu.getSelectedKey();
			if (tempKey > 0) {
				String str = Parameter.getTemplateParameterValue(getDbID(), tempKey, Parameter.PARAM_SCEN_WL_FREQ, 0);
				if (null != str) {
					frequencyField.setText(str);
				}
				str = Parameter.getTemplateParameterValue(getDbID(), tempKey, Parameter.PARAM_SCEN_WL_BW, 0);
				if (null != str) {
					bandwidthField.setText(str);
				}
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		public void setEnabled(boolean flag) {

			super.setEnabled(flag);

			AppController.setComponentEnabled(templateMenu, flag);

			AppController.setComponentEnabled(wirelessExtDbMenu, flag);
			AppController.setComponentEnabled(maskingCheckBox, flag);
			if (maskingCheckBox.isSelected()) {
				AppController.setComponentEnabled(maskingExtDbMenu, flag);
			} else {
				AppController.setComponentEnabled(maskingExtDbMenu, false);
			}

			AppController.setComponentEnabled(frequencyField, flag);
			AppController.setComponentEnabled(bandwidthField, flag);

			AppController.setComponentEnabled(selectedRecordsTable, flag);
			updateControls();
		}


		//-------------------------------------------------------------------------------------------------------------
		// Update data set menus when notified by ExtDb.

		public void updateExtDbList() {

			SwingUtilities.invokeLater(new Runnable() {
				public void run() {

					ArrayList<KeyedRecord> list = ExtDb.getExtDbList(getDbID(), Source.RECORD_TYPE_WL, false, true);
					if (null == list) {
						return;
					}

					blockActionsStart();

					int selectKey = wirelessExtDbMenu.getSelectedKey();
					wirelessExtDbMenu.removeAllItems();
					if (!list.isEmpty()) {
						wirelessExtDbMenu.addAllItems(list);
						if (wirelessExtDbMenu.containsKey(selectKey)) {
							wirelessExtDbMenu.setSelectedKey(selectKey);
						}
					}

					blockActionsEnd();

					list = ExtDb.getExtDbList(getDbID(), Source.RECORD_TYPE_TV, false, true);
					if (null == list) {
						return;
					}

					blockActionsStart();

					selectKey = maskingExtDbMenu.getSelectedKey();
					maskingExtDbMenu.removeAllItems();
					if (!list.isEmpty()) {
						maskingExtDbMenu.addAllItems(list);
						if (maskingExtDbMenu.containsKey(selectKey)) {
							maskingExtDbMenu.setSelectedKey(selectKey);
						}
					}

					blockActionsEnd();
				}
			});
		}


		//-------------------------------------------------------------------------------------------------------------
		// Populate menus when the containing window is shown.

		public void windowWillOpen() {

			ExtDb.addListener(this);

			templateMenu.removeAllItems();
			wirelessExtDbMenu.removeAllItems();
			maskingExtDbMenu.removeAllItems();

			SwingUtilities.invokeLater(new Runnable() {
				public void run() {

					blockActionsStart();

					errorReporter.setTitle("Load Template List");

					ArrayList<KeyedRecord> list = Template.getTemplateInfoList(getDbID(), errorReporter);
					if (null != list) {
						templateMenu.addAllItems(list);
					}

					errorReporter.setTitle("Load Station Data List");

					list = ExtDb.getExtDbList(getDbID(), Source.RECORD_TYPE_WL, true, true, errorReporter);
					if (null != list) {
						if (list.isEmpty()) {
							errorReporter.reportError(
								"No wireless station data found, use Station Data Manager to add data");
						} else {
							wirelessExtDbMenu.addAllItems(list);
						}
					}

					list = ExtDb.getExtDbList(getDbID(), Source.RECORD_TYPE_TV, false, true, errorReporter);
					if (null != list) {
						maskingExtDbMenu.addAllItems(list);
					}

					clearFields();

					blockActionsEnd();
				}
			});
		}


		//-------------------------------------------------------------------------------------------------------------

		public void windowWillClose() {

			ExtDb.removeListener(this);
		}


		//-------------------------------------------------------------------------------------------------------------
		// The StudyBuild object may provide defaults.

		public void clearFields() {

			super.clearFields();

			blockActionsStart();

			if ((studyBuild.templateKey > 0) && templateMenu.containsKey(studyBuild.templateKey)) {
				templateMenu.setSelectedKey(studyBuild.templateKey);
			} else {
				templateMenu.setSelectedIndex(0);
			}

			updateParams();

			if (null != studyBuild.wirelessExtDb) {
				int theKey = studyBuild.wirelessExtDb.key.intValue();
				if (wirelessExtDbMenu.containsKey(theKey)) {
					wirelessExtDbMenu.setSelectedKey(theKey);
				} else {
					wirelessExtDbMenu.setSelectedIndex(0);
				}
			} else {
				wirelessExtDbMenu.setSelectedIndex(0);
			}
			if (null != studyBuild.maskingExtDb) {
				maskingCheckBox.setSelected(true);
				int theKey = studyBuild.maskingExtDb.key.intValue();
				if (maskingExtDbMenu.containsKey(theKey)) {
					maskingExtDbMenu.setSelectedKey(theKey);
				} else {
					maskingExtDbMenu.setSelectedIndex(0);
				}
			} else {
				maskingCheckBox.setSelected(false);
				maskingExtDbMenu.setSelectedIndex(0);
				AppController.setComponentEnabled(maskingExtDbMenu, false);
			}

			blockActionsEnd();
		}


		//-------------------------------------------------------------------------------------------------------------
		// When a record from a data set is selected, set the selection for the masking TV interference search to use
		// the same data set.  If the new record does not have a data set reference the menu selection is not changed.
		// The record should always be TV, just ignore if it is not.  Also set the study name to the file number.

		public void selectionChanged(Object newSelection) {

			super.selectionChanged(newSelection);

			selectedRecord = null;

			if (null != newSelection) {

				if (newSelection instanceof SourceEditData) {

					SourceEditData theSource = (SourceEditData)newSelection;
					if (Source.RECORD_TYPE_TV == theSource.recordType) {
						selectedRecord = theSource;
						if (null != theSource.extDbKey) {
							int theKey = theSource.extDbKey.intValue();
							if (maskingExtDbMenu.containsKey(theKey)) {
								maskingExtDbMenu.setSelectedKey(theKey);
							}
						}
					}

				} else {

					if (newSelection instanceof ExtDbRecord) {

						ExtDbRecord theRecord = (ExtDbRecord)newSelection;
						if (Source.RECORD_TYPE_TV == theRecord.recordType) {
							selectedRecord = theRecord;
							int theKey = theRecord.extDb.key.intValue();
							if (maskingExtDbMenu.containsKey(theKey)) {
								maskingExtDbMenu.setSelectedKey(theKey);
							}
						}
					}
				}
			}

			if ((null != selectedRecord) && (0 == selectedRecordsModel.getRowCount())) {

				String theName = "", sep = "";

				String theCall = selectedRecord.getCallSign();
				if ((null != theCall) && (theCall.length() > 0)) {
					theName = theCall;
					sep = "-";
				}

				String theFile = selectedRecord.getFileNumber();
				if ((null != theFile) && (theFile.length() > 0)) {
					theName = theName + sep + theFile;
				}

				nameField.setText(theName);
			}

			updateControls();
		}


		//-------------------------------------------------------------------------------------------------------------
		// Add a record to the list of studies to build.  Check first if it is already in the list.

		private void doAddRecord() {

			if (null == selectedRecord) {
				return;
			}

			int rowIndex = selectedRecordsModel.indexOfRecord(selectedRecord);
			if (rowIndex < 0) {
				rowIndex = selectedRecordsModel.add(selectedRecord);

				if (1 == selectedRecordsModel.getRowCount()) {
					nameField.setText("");
				}
			}

			if (rowIndex >= 0) {
				selectedRecordsTable.setRowSelectionInterval(rowIndex, rowIndex);
				selectedRecordsTable.scrollRectToVisible(selectedRecordsTable.getCellRect(rowIndex, 0, true));
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Remove a record from the list.

		private void doRemoveRecord() {

			if (0 == selectedRecordsTable.getSelectedRowCount()) {
				return;
			}
			selectedRecordsModel.remove(selectedRecordsTable.getSelectedRow());
		}


		//-------------------------------------------------------------------------------------------------------------

		private void updateControls() {

			boolean e = AppController.isComponentEnabled(selectedRecordsTable);
			addRecordButton.setEnabled(e && (null != selectedRecord) &&
				(selectedRecordsModel.indexOfRecord(selectedRecord) < 0));
			removeRecordButton.setEnabled(e && (1 == selectedRecordsTable.getSelectedRowCount()));
		}


		//=============================================================================================================
		// Table model for list of selected records.

		private class RecordsTableModel extends AbstractTableModel {

			private static final String RECORD_TYPE_COLUMN = "Type";
			private static final String RECORD_CALLSIGN_COLUMN = "Call Sign";
			private static final String RECORD_CHANNEL_COLUMN = "Channel";
			private static final String RECORD_SERVICE_COLUMN = "Svc";
			private static final String RECORD_STATUS_COLUMN = "Status";
			private static final String RECORD_CITY_COLUMN = "City";
			private static final String RECORD_STATE_COLUMN = "State";
			private static final String RECORD_COUNTRY_COLUMN = "Cntry";
			private static final String RECORD_FACILITY_ID_COLUMN = "Facility ID";
			private static final String RECORD_FILE_COLUMN = "File Number";

			private String[] columnNames = {
				RECORD_TYPE_COLUMN,
				RECORD_CALLSIGN_COLUMN,
				RECORD_CHANNEL_COLUMN,
				RECORD_SERVICE_COLUMN,
				RECORD_STATUS_COLUMN,
				RECORD_CITY_COLUMN,
				RECORD_STATE_COLUMN,
				RECORD_COUNTRY_COLUMN,
				RECORD_FACILITY_ID_COLUMN,
				RECORD_FILE_COLUMN
			};

			private static final int RECORD_TYPE_INDEX = 0;
			private static final int RECORD_CALLSIGN_INDEX = 1;
			private static final int RECORD_CHANNEL_INDEX = 2;
			private static final int RECORD_SERVICE_INDEX = 3;
			private static final int RECORD_STATUS_INDEX = 4;
			private static final int RECORD_CITY_INDEX = 5;
			private static final int RECORD_STATE_INDEX = 6;
			private static final int RECORD_COUNTRY_INDEX = 7;
			private static final int RECORD_FACILITY_ID_INDEX = 8;
			private static final int RECORD_FILE_INDEX = 9;

			private JPanel panel;

			private ArrayList<StationRecord> modelRows;


			//---------------------------------------------------------------------------------------------------------

			private RecordsTableModel(JPanel thePanel) {

				super();

				panel = thePanel;

				modelRows = new ArrayList<StationRecord>();
			}


			//---------------------------------------------------------------------------------------------------------

			private JTable createTable() {

				JTable theTable = new JTable(this);
				AppController.configureTable(theTable);

				TableColumnModel columnModel = theTable.getColumnModel();

				TableColumn theColumn = columnModel.getColumn(RECORD_TYPE_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[4]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[4]);

				theColumn = columnModel.getColumn(RECORD_CALLSIGN_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[5]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[5]);

				theColumn = columnModel.getColumn(RECORD_CHANNEL_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[5]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[7]);

				theColumn = columnModel.getColumn(RECORD_SERVICE_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[2]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[2]);

				theColumn = columnModel.getColumn(RECORD_STATUS_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[4]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[4]);

				theColumn = columnModel.getColumn(RECORD_CITY_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[8]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[14]);

				theColumn = columnModel.getColumn(RECORD_STATE_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[2]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[2]);

				theColumn = columnModel.getColumn(RECORD_COUNTRY_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[2]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[2]);

				theColumn = columnModel.getColumn(RECORD_FACILITY_ID_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[5]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[5]);

				theColumn = columnModel.getColumn(RECORD_FILE_INDEX);
				theColumn.setMinWidth(AppController.textFieldWidth[8]);
				theColumn.setPreferredWidth(AppController.textFieldWidth[16]);

				theTable.setPreferredScrollableViewportSize(new Dimension(theTable.getPreferredSize().width,
					(theTable.getRowHeight() * 3)));

				return theTable;
			}


			//---------------------------------------------------------------------------------------------------------
			// If the object is already in the array just return the index, don't add again.

			private int add(StationRecord theRecord) {

				int rowIndex = modelRows.size();
				modelRows.add(theRecord);
				fireTableRowsInserted(rowIndex, rowIndex);

				updateBorder();

				return rowIndex;
			}


			//---------------------------------------------------------------------------------------------------------
			// Determine if a record, or an equivalent object representing the same underlying database record, is
			// already in the list.  Return index if found, else -1.

			private int indexOfRecord(StationRecord theRecord) {

				int rowCount = modelRows.size();
				if (0 == rowCount) {
					return -1;
				}

				SourceEditData theSource;
				ExtDbRecord theExtRecord;
				Integer theUserRecordID;
				Integer theExtDbKey;
				String theExtRecordID;

				if (theRecord.isSource()) {
					theSource = (SourceEditData)theRecord;
					theExtRecord = null;
					theUserRecordID = theSource.userRecordID;
					theExtDbKey = theSource.extDbKey;
					theExtRecordID = theSource.extRecordID;
				} else {
					theSource = null;
					theExtRecord = (ExtDbRecord)theRecord;
					theUserRecordID = null;
					theExtDbKey = theExtRecord.extDb.key;
					theExtRecordID = theExtRecord.extRecordID;
				}

				StationRecord otherRecord;
				SourceEditData otherSource;
				ExtDbRecord otherExtRecord;
				Integer otherUserRecordID;
				Integer otherExtDbKey;
				String otherExtRecordID;

				for (int rowIndex = 0; rowIndex < rowCount; rowIndex++) {

					otherRecord = modelRows.get(rowIndex);

					// First check for a same object, that is of course always a match.

					if (theRecord == otherRecord) {
						return rowIndex;
					}

					// If the new record is an unlocked SourceEditData object, the only match is the same object.

					if ((null != theSource) && !theSource.isLocked) {
						continue;
					}

					if (otherRecord.isSource()) {
						otherSource = (SourceEditData)otherRecord;
						otherExtRecord = null;
						otherUserRecordID = otherSource.userRecordID;
						otherExtDbKey = otherSource.extDbKey;
						otherExtRecordID = otherSource.extRecordID;
					} else {
						otherSource = null;
						otherExtRecord = (ExtDbRecord)otherRecord;
						otherUserRecordID = null;
						otherExtDbKey = otherExtRecord.extDb.key;
						otherExtRecordID = otherExtRecord.extRecordID;
					}

					// An existing record that is an unlocked SourceEditData can only match the same-object test.

					if ((null != otherSource) && !otherSource.isLocked) {
						continue;
					}

					// If the new record has a userRecordID match is based on that, if it has an extRecordID on that.

					if (null != theUserRecordID) {
						if ((null != otherUserRecordID) && theUserRecordID.equals(otherUserRecordID)) {
							return rowIndex;
						}
						continue;
					}

					if (null != theExtRecordID) {
						if ((null != otherExtRecordID) && theExtDbKey.equals(otherExtDbKey) &&
								theExtRecordID.equals(otherExtRecordID)) {
							return rowIndex;
						}
						continue;
					}
				}

				return -1;
			}


			//---------------------------------------------------------------------------------------------------------

			private StationRecord get(int rowIndex) {

				return modelRows.get(rowIndex);
			}


			//---------------------------------------------------------------------------------------------------------

			private void remove(int rowIndex) {

				modelRows.remove(rowIndex);
				fireTableRowsDeleted(rowIndex, rowIndex);

				updateBorder();
			}


			//---------------------------------------------------------------------------------------------------------

			private void updateBorder() {

				int n = modelRows.size();
				if (n > 0) {
					panel.setBorder(BorderFactory.createTitledBorder(String.valueOf(n) + " selected records"));
				} else {
					panel.setBorder(BorderFactory.createTitledBorder("Selected records"));
				}
			}


			//---------------------------------------------------------------------------------------------------------

			public int getColumnCount() {

				return columnNames.length;
			}


			//---------------------------------------------------------------------------------------------------------

			public String getColumnName(int columnIndex) {

				return columnNames[columnIndex];
			}


			//---------------------------------------------------------------------------------------------------------

			public int getRowCount() {

				return modelRows.size();
			}


			//---------------------------------------------------------------------------------------------------------

			public Object getValueAt(int rowIndex, int columnIndex) {

				StationRecord theRecord = modelRows.get(rowIndex);

				switch (columnIndex) {

					case RECORD_TYPE_INDEX:
						return theRecord.getRecordType();

					case RECORD_CALLSIGN_INDEX:
						return theRecord.getCallSign();

					case RECORD_CHANNEL_INDEX:
						return theRecord.getChannel() + " " + theRecord.getFrequency();

					case RECORD_SERVICE_INDEX:
						return theRecord.getServiceCode();

					case RECORD_STATUS_INDEX:
						return theRecord.getStatus();

					case RECORD_CITY_INDEX:
						return theRecord.getCity();

					case RECORD_STATE_INDEX:
						return theRecord.getState();

					case RECORD_COUNTRY_INDEX:
						return theRecord.getCountryCode();

					case RECORD_FACILITY_ID_INDEX:
						return theRecord.getFacilityID();

					case RECORD_FILE_INDEX:
						return theRecord.getFileNumber();
				}

				return "";
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// If there are records in the selection list, even if only one, study names are auto-generated so the name
		// field may be blank.

		protected boolean isNameRequired() {

			return (0 == selectedRecordsModel.getRowCount());
		}


		//-------------------------------------------------------------------------------------------------------------
		// If there are records in the selection list, replication or channel change may be requested but applies only
		// to records that actually need it; suppress super's same-channel test.

		protected boolean noReplicateChangeSameChannel() {

			return (0 == selectedRecordsModel.getRowCount());
		}


		//-------------------------------------------------------------------------------------------------------------

		public boolean validateInput() {

			errorReporter.clearTitle();

			int selectedRecordCount = selectedRecordsModel.getRowCount();

			if ((null == selectedRecord) && (0 == selectedRecordCount)) {
				errorReporter.reportWarning("Please select a TV record to study");
				return false;
			}

			if (!super.validateInput()) {
				return false;
			}

			// Check the study name if needed, in the case of multiple selected records it may be blank.  Keep extra
			// length to append various things as needed.  In the case of multiple records the name is a prefix, other
			// properties for each record will be appended.  In any case if replicating or changing channel the new
			// channel will be appended.  Note this check is advisory; there will always be filtering, truncation, and
			// a uniqueness suffix added if needed later, see Study.createNewStudy().

			if (name.length() > 0) {
				int reserveLength = 0;
				if (selectedRecordCount > 0) {
					reserveLength = 40;
				}
				if (replicate || changeChannel) {
					reserveLength += 10;
				}
				if (!DbCore.checkStudyName(name, getDbID(), false, reserveLength, errorReporter)) {
					return false;
				}
			}

			int templateKey = templateMenu.getSelectedKey();
			if (templateKey <= 0) {
				errorReporter.reportWarning("Please choose a study template");
				return false;
			}

			ExtDb wirelessExtDb = null;
			int theKey = wirelessExtDbMenu.getSelectedKey();
			if (theKey > 0) {
				wirelessExtDb = ExtDb.getExtDb(getDbID(), Integer.valueOf(theKey));
			}
			if ((null == wirelessExtDb) || wirelessExtDb.deleted) {
				errorReporter.reportWarning("That wireless station data set has been deleted, please select another");
				return false;
			}

			ExtDb maskingExtDb = null;
			if (maskingCheckBox.isSelected()) {
				theKey = maskingExtDbMenu.getSelectedKey();
				if (theKey > 0) {
					maskingExtDb = ExtDb.getExtDb(getDbID(), Integer.valueOf(theKey));
				}
				if ((null == maskingExtDb) || maskingExtDb.deleted) {
					errorReporter.reportWarning("That TV station data set has been deleted, please select another");
					return false;
				}
			}

			String str = frequencyField.getText().trim();
			if (0 == str.length()) {
				errorReporter.reportWarning("Please enter a wireless frequency");
				return false;
			}
			double d = 0.;
			try {
				d = Double.parseDouble(str);
			} catch (NumberFormatException ne) {
				errorReporter.reportWarning("The wireless frequency must be a number");
				return false;
			}
			if ((d < 50.) || (d > 800.)) {
				errorReporter.reportWarning("The wireless frequency must be in the range 50 to 800");
				return false;
			}
			String frequency = str;

			str = bandwidthField.getText().trim();
			if (0 == str.length()) {
				errorReporter.reportWarning("Please enter a wireless bandwidth");
				return false;
			}
			d = 0.;
			try {
				d = Double.parseDouble(str);
			} catch (NumberFormatException ne) {
				errorReporter.reportWarning("The wireless bandwidth must be a number");
				return false;
			}
			if ((d < 0.1) || (d > 20.)) {
				errorReporter.reportWarning("The wireless bandwidth must be in the range 0.1 to 20");
				return false;
			}
			String bandwidth = str;

			// All good, save values.

			studyBuild.replicate = replicate;
			studyBuild.changeChannel = changeChannel;
			studyBuild.studyChannel = studyChannel;

			studyBuild.studyDescription = null;
			if (description.length() > 0) {
				studyBuild.studyDescription = description;
			}

			studyBuild.templateKey = templateKey;

			studyBuild.wirelessExtDb = wirelessExtDb;
			studyBuild.maskingExtDb = maskingExtDb;

			studyBuild.frequency = frequency;
			studyBuild.bandwidth = bandwidth;

			// Create the list of study build objects.  If the selection list is empty just the currently-selected
			// record will be used, set the name on the main object.  The parent did the canApply/wasApplied logic.

			studyBuilds = new ArrayList<StudyBuild>();

			int rowCount = selectedRecordsModel.getRowCount();

			if (0 == selectedRecordCount) {

				if (selectedRecord.isSource()) {
					studyBuild.source = (SourceEditData)selectedRecord;
				} else {
					studyBuild.record = (ExtDbRecord)selectedRecord;
				}

				if (replicate || changeChannel) {
					studyBuild.studyName = name + " (on " + String.valueOf(studyChannel) + ")";
				} else {
					studyBuild.studyName = name;
				}
		
				studyBuilds.add(studyBuild);

			// There are selected records in the list.  Make sure the parent is a RecordFind UI (it should be), confirm
			// all selected can be applied.  If the currently-selected record is in the list that can be skipped, the
			// parent will have already checked that.

			} else {

				StationRecord theRecord;
				RecordFind finder = null;

				if (parent instanceof RecordFind) {
					finder = (RecordFind)parent;
					for (int rowIndex = 0; rowIndex < rowCount; rowIndex++) {
						theRecord = selectedRecordsModel.get(rowIndex);
						if (theRecord != selectedRecord) {
							if (!finder.canApplyRecord(theRecord, true)) {
								return false;
							}
						}
					}
				}

				// If the currently-selected record is not in the list, add it.

				if ((null != selectedRecord) && (selectedRecordsModel.indexOfRecord(selectedRecord) < 0)) {
					selectedRecordsModel.add(selectedRecord);
					selectedRecordCount++;
				}

				// Create the build objects.  The study name is auto-generated using any text entered as a prefix
				// followed by the call sign and file number.  If everything is blank use a generic text.  If replicate
				// or change channel is set, check each record to see if it actually needs that action.  If the channel
				// is the same and it's not a non-digital record being replicated on-channel to digital, just clear the
				// replicate/channel change state.  This allows a mix of records some of which are already on the
				// desired channel but some of which are not.

				StudyBuildWireless theBuild;
				String theName, sep, theCall, theFile;

				for (int rowIndex = 0; rowIndex < rowCount; rowIndex++) {

					theRecord = selectedRecordsModel.get(rowIndex);

					theBuild = studyBuild.copy();

					theName = "";
					sep = "";

					if (name.length() > 0) {
						theName = name;
						sep = "-";
					}

					theCall = theRecord.getCallSign();
					if ((null != theCall) && (theCall.length() > 0)) {
						theName = theName + sep + theCall;
						sep = "-";
					}

					theFile = theRecord.getFileNumber();
					if ((null != theFile) && (theFile.length() > 0)) {
						theName = theName + sep + theFile;
					}

					if (0 == theName.length()) {
						theName = "Wireless study";
					}

					if (theRecord.isSource()) {
						theBuild.source = (SourceEditData)theRecord;
					} else {
						theBuild.record = (ExtDbRecord)theRecord;
					}

					if ((replicate || changeChannel) && (theRecord.getChannelNumber() == studyChannel) &&
							(changeChannel || theRecord.isDigital())) {
						theBuild.replicate = false;
						theBuild.changeChannel = false;
						theBuild.studyChannel = 0;
					} else {
						theName = theName + " (on " + String.valueOf(studyChannel) + ")";
					}

					theBuild.studyName = theName;

					studyBuilds.add(theBuild);

					if (null != finder) {
						finder.recordWasApplied(theRecord);
					}
				}
			}

			return true;
		}


		//-------------------------------------------------------------------------------------------------------------

		public ArrayList<StudyBuild> getStudyBuilds() {

			return studyBuilds;
		}
	}


	//=================================================================================================================
	// Subclass used when creating an FM to TV channel 6 interference study.  Input study name, select template, and
	// select the FM station data set to use for the initial scenario creation.  In the containing RecordFind dialog
	// the user picks the TV data set and the target TV record.

	public static class TV6FMStudy extends OptionsPanel implements ExtDbListener {

		private KeyedRecordMenu templateMenu;

		private KeyedRecordMenu fmExtDbMenu;
		private JCheckBox maskingCheckBox;
		private KeyedRecordMenu tvExtDbMenu;

		private StationRecord selectedRecord;

		// Values are set directly in the study creation object.

		public StudyBuildTV6FM studyBuild;


		//-------------------------------------------------------------------------------------------------------------
		// Show name, description, replicate, output, and run options from superclass.

		public TV6FMStudy(AppEditor theParent, StudyBuildTV6FM theStudy) {

			super(theParent, true, "Study Name", true, "Study Description", false, true, false, true, false, true);

			setBorder(BorderFactory.createTitledBorder("Study Build Settings"));

			studyBuild = theStudy;

			defaultName = theStudy.studyName;
			defaultDescription = theStudy.studyDescription;

			templateMenu = new KeyedRecordMenu();
			templateMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXy"));

			JPanel templatePanel = new JPanel();
			templatePanel.setBorder(BorderFactory.createTitledBorder("Template"));
			templatePanel.add(templateMenu);

			fmExtDbMenu = new KeyedRecordMenu();
			fmExtDbMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXy"));

			JPanel fmMenuPanel = new JPanel();
			fmMenuPanel.setBorder(BorderFactory.createTitledBorder("FM Station Data"));
			fmMenuPanel.add(fmExtDbMenu);

			maskingCheckBox = new JCheckBox("Include TV masking interference");
			maskingCheckBox.setFocusable(false);
			maskingCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					AppController.setComponentEnabled(tvExtDbMenu, maskingCheckBox.isSelected());
				}
			});

			tvExtDbMenu = new KeyedRecordMenu();
			tvExtDbMenu.setPrototypeDisplayValue(new KeyedRecord(0, "XyXyXyXyXyXyXyXyXy"));
			AppController.setComponentEnabled(tvExtDbMenu, false);

			JPanel tvMenuPanel = new JPanel();
			tvMenuPanel.setBorder(BorderFactory.createTitledBorder("TV Station Data"));
			tvMenuPanel.add(tvExtDbMenu);

			// Layout.

			Box col1Box = Box.createVerticalBox();
			col1Box.add(channelPanel);
			col1Box.add(templatePanel);

			JPanel chkP = new JPanel(new FlowLayout(FlowLayout.LEFT));
			chkP.add(maskingCheckBox);

			Box col2Box = Box.createVerticalBox();
			col2Box.add(fmMenuPanel);
			col2Box.add(chkP);
			col2Box.add(tvMenuPanel);

			JPanel nameDescP = new JPanel();
			nameDescP.add(namePanel);
			nameDescP.add(descriptionPanel);

			JPanel outRunP = new JPanel();
			outRunP.add(outputPanel);
			outRunP.add(runPanel);

			Box col3Box = Box.createVerticalBox();
			col3Box.add(nameDescP);
			col3Box.add(outRunP);

			add(col1Box);
			add(col2Box);
			add(col3Box);
		}


		//-------------------------------------------------------------------------------------------------------------
		// Layout done directly.

		protected void createLayout(JPanel thePanel) {

			return;
		}


		//-------------------------------------------------------------------------------------------------------------

		public void setEnabled(boolean flag) {

			super.setEnabled(flag);

			AppController.setComponentEnabled(templateMenu, flag);

			AppController.setComponentEnabled(fmExtDbMenu, flag);
			AppController.setComponentEnabled(maskingCheckBox, flag);
			if (maskingCheckBox.isSelected()) {
				AppController.setComponentEnabled(tvExtDbMenu, flag);
			} else {
				AppController.setComponentEnabled(tvExtDbMenu, false);
			}
		}


		//-------------------------------------------------------------------------------------------------------------
		// Update data set menus when notified by ExtDb.

		public void updateExtDbList() {

			SwingUtilities.invokeLater(new Runnable() {
				public void run() {

					ArrayList<KeyedRecord> list = ExtDb.getExtDbList(getDbID(), Source.RECORD_TYPE_FM, false, true);
					if (null == list) {
						return;
					}

					blockActionsStart();

					int selectKey = fmExtDbMenu.getSelectedKey();
					fmExtDbMenu.removeAllItems();
					if (!list.isEmpty()) {
						fmExtDbMenu.addAllItems(list);
						if (fmExtDbMenu.containsKey(selectKey)) {
							fmExtDbMenu.setSelectedKey(selectKey);
						}
					}

					blockActionsEnd();

					list = ExtDb.getExtDbList(getDbID(), Source.RECORD_TYPE_TV, false, true);
					if (null == list) {
						return;
					}

					blockActionsStart();

					selectKey = tvExtDbMenu.getSelectedKey();
					tvExtDbMenu.removeAllItems();
					if (!list.isEmpty()) {
						tvExtDbMenu.addAllItems(list);
						if (tvExtDbMenu.containsKey(selectKey)) {
							tvExtDbMenu.setSelectedKey(selectKey);
						}
					}

					blockActionsEnd();
				}
			});
		}


		//-------------------------------------------------------------------------------------------------------------

		public void windowWillOpen() {

			ExtDb.addListener(this);

			templateMenu.removeAllItems();
			fmExtDbMenu.removeAllItems();
			tvExtDbMenu.removeAllItems();

			SwingUtilities.invokeLater(new Runnable() {
				public void run() {

					blockActionsStart();

					errorReporter.setTitle("Load Template List");

					ArrayList<KeyedRecord> list = Template.getTemplateInfoList(getDbID(), errorReporter);
					if (null != list) {
						templateMenu.addAllItems(list);
					}

					errorReporter.setTitle("Load Station Data List");

					list = ExtDb.getExtDbList(getDbID(), Source.RECORD_TYPE_FM, false, true, errorReporter);
					if (null != list) {
						if (list.isEmpty()) {
							errorReporter.reportError(
								"No FM station data found, use Station Data Manager to add data");
						} else {
							fmExtDbMenu.addAllItems(list);
						}
					}

					list = ExtDb.getExtDbList(getDbID(), Source.RECORD_TYPE_TV, false, true, errorReporter);
					if (null != list) {
						tvExtDbMenu.addAllItems(list);
					}

					clearFields();

					blockActionsEnd();
				}
			});
		}


		//-------------------------------------------------------------------------------------------------------------

		public void windowWillClose() {

			ExtDb.removeListener(this);
		}


		//-------------------------------------------------------------------------------------------------------------

		public void clearFields() {

			super.clearFields();

			blockActionsStart();

			if ((studyBuild.templateKey > 0) && templateMenu.containsKey(studyBuild.templateKey)) {
				templateMenu.setSelectedKey(studyBuild.templateKey);
			} else {
				templateMenu.setSelectedIndex(0);
			}

			if (null != studyBuild.fmExtDb) {
				int theKey = studyBuild.fmExtDb.key.intValue();
				if (fmExtDbMenu.containsKey(theKey)) {
					fmExtDbMenu.setSelectedKey(theKey);
				} else {
					fmExtDbMenu.setSelectedIndex(0);
				}
			} else {
				fmExtDbMenu.setSelectedIndex(0);
			}
			if (null != studyBuild.tvExtDb) {
				maskingCheckBox.setSelected(true);
				int theKey = studyBuild.tvExtDb.key.intValue();
				if (tvExtDbMenu.containsKey(theKey)) {
					tvExtDbMenu.setSelectedKey(theKey);
				} else {
					tvExtDbMenu.setSelectedIndex(0);
				}
			} else {
				maskingCheckBox.setSelected(false);
				tvExtDbMenu.setSelectedIndex(0);
				AppController.setComponentEnabled(tvExtDbMenu, false);
			}

			blockActionsEnd();
		}


		//-------------------------------------------------------------------------------------------------------------
		// When a record from a data set is selected, set the selection for the TV interference search to use the same
		// data set.  If the new record does not have a data set reference the menu selection is not changed.  The
		// record should always be TV, just ignore if it is not.  Also set the study name to the record file number.

		public void selectionChanged(Object newSelection) {

			super.selectionChanged(newSelection);

			if (null == newSelection) {
				return;
			}

			if (newSelection instanceof SourceEditData) {

				SourceEditData theSource = (SourceEditData)newSelection;
				if (Source.RECORD_TYPE_TV == theSource.recordType) {
					selectedRecord = theSource;
					if (null != theSource.extDbKey) {
						int theKey = theSource.extDbKey.intValue();
						if (tvExtDbMenu.containsKey(theKey)) {
							tvExtDbMenu.setSelectedKey(theKey);
						}
					}
				}

			} else {

				if (newSelection instanceof ExtDbRecord) {

					ExtDbRecord theRecord = (ExtDbRecord)newSelection;
					if (Source.RECORD_TYPE_TV == theRecord.recordType) {
						selectedRecord = theRecord;
						int theKey = theRecord.extDb.key.intValue();
						if (tvExtDbMenu.containsKey(theKey)) {
							tvExtDbMenu.setSelectedKey(theKey);
						}
					}
				}
			}


			if (newSelection instanceof StationRecord) {
				String theName = ((StationRecord)newSelection).getFileNumber();
				if ((null != theName) && (theName.length() > 0)) {
					nameField.setText(theName);
				}
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		public boolean validateInput() {

			errorReporter.clearTitle();

			if (null == selectedRecord) {
				errorReporter.reportWarning("Please select a TV record to study");
				return false;
			}

			if (!super.validateInput()) {
				return false;
			}

			int theChan = selectedRecord.getChannelNumber();
			if (replicate || changeChannel) {
				theChan = studyChannel;
			}
			if (6 != theChan) {
				errorReporter.reportWarning("The study record must be on or replicated/changed to channel 6");
				return false;
			}

			if (!DbCore.checkStudyName(name, getDbID(), false, ((replicate || changeChannel) ? 10 : 0),
					errorReporter)) {
				return false;
			}

			int templateKey = templateMenu.getSelectedKey();
			if (templateKey <= 0) {
				errorReporter.reportWarning("Please choose a study template");
				return false;
			}

			ExtDb fmExtDb = null;
			int theKey = fmExtDbMenu.getSelectedKey();
			if (theKey > 0) {
				fmExtDb = ExtDb.getExtDb(getDbID(), Integer.valueOf(theKey));
			}
			if ((null == fmExtDb) || fmExtDb.deleted) {
				errorReporter.reportWarning("That FM station data set has been deleted, please select another");
				return false;
			}

			ExtDb tvExtDb = null;
			if (maskingCheckBox.isSelected()) {
				theKey = tvExtDbMenu.getSelectedKey();
				if (theKey > 0) {
					tvExtDb = ExtDb.getExtDb(getDbID(), Integer.valueOf(theKey));
				}
				if ((null == tvExtDb) || tvExtDb.deleted) {
					errorReporter.reportWarning("That TV station data set has been deleted, please select another");
					return false;
				}
			}

			// All good, save values.

			if (selectedRecord.isSource()) {
				studyBuild.source = (SourceEditData)selectedRecord;
			} else {
				studyBuild.record = (ExtDbRecord)selectedRecord;
			}

			studyBuild.replicate = replicate;
			studyBuild.changeChannel = changeChannel;
			studyBuild.studyChannel = studyChannel;

			if (replicate || changeChannel) {
				studyBuild.studyName = name + " (on " + String.valueOf(studyChannel) + ")";
			} else {
				studyBuild.studyName = name;
			}

			studyBuild.studyDescription = null;
			if (description.length() > 0) {
				studyBuild.studyDescription = description;
			}

			studyBuild.templateKey = templateKey;

			studyBuild.fmExtDb = fmExtDb;
			studyBuild.tvExtDb = tvExtDb;

			return true;
		}


		//-------------------------------------------------------------------------------------------------------------
		// To support other subclasses that might cause multiple studies to be created this returns an array, but here
		// there will only ever be one.

		public ArrayList<StudyBuild> getStudyBuilds() {

			ArrayList<StudyBuild> theBuilds = new ArrayList<StudyBuild>();
			theBuilds.add(studyBuild);
			return theBuilds;
		}
	}


	//=================================================================================================================
	// Subclass for a run-study dialog, show output config including menus, and run options.

	public static class RunStart extends OptionsPanel {


		//-------------------------------------------------------------------------------------------------------------

		public RunStart(AppEditor theParent) {

			super(theParent, false, null, false, null, false, false, false, true, true, true);
		}
	}
}
