//
//  ExtDbManager.java
//  TVStudy
//
//  Copyright (c) 2012-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.gui;

import gov.fcc.tvstudy.core.*;
import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;
import gov.fcc.tvstudy.core.geo.*;
import gov.fcc.tvstudy.gui.editor.*;
import gov.fcc.tvstudy.gui.run.*;

import java.util.*;
import java.sql.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.table.*;
import javax.swing.filechooser.*;

import javax.xml.parsers.*;

import org.xml.sax.*;
import org.xml.sax.helpers.*;


//=====================================================================================================================
// Manager UI for the set of external station data sets in a database.  Allows data sets to be created by importing
// downloaded dump files, also data sets can be re-named and deleted.  There is no actual editor for a data set, it is
// a static mirror of a particular download state of the FCC's database, or other external data source.  This is not a
// dialog, but it makes no sense to have two of these open for the same database so they are managed as singleton top-
// level windows one per database, shown with showManager(), as is done in StudyManager.

// There are now two fundamentally different types of data sets.  The traditional import types, the list returned by
// ExtDb.getImportTypes(), are the dump-file-based "mirror" sets described above.  The data set database is created
// and data imported once, and only once, into that database in one operation, and subsequently the set is read-only.
// There also now "generic" import types, returned by ExtDb.getGenericTypes().  These are more flexible, the data set
// database is created first by separate action, then one more more imports may subsequently be performed drawing from
// a variety of different source data file formats to add records to the set.  The only difference is in setup here,
// throughout the rest of the UI both all types of data sets function the same way.

public class ExtDbManager extends AppFrame implements ExtDbListener {

	public static final String WINDOW_TITLE = "Station Data Manager";

	private String dbID;

	private ExtDbListModel extDbModel;
	private JTable extDbTable;

	private KeyedRecordMenu downloadMenu;
	private JButton downloadButton;
	private KeyedRecordMenu createAndImportMenu;

	private JButton importGenericButton;
	private JMenuItem importGenericMenuItem;

	private JMenuItem renameExtDbMenuItem;
	private JMenuItem deleteExtDbMenuItem;
	private JMenuItem clearBadDataFlagMenuItem;
	private JMenuItem unlockExtDbMenuItem;
	private JMenuItem extDbInfoMenuItem;

	// Disambiguation.

	private ExtDbManager outerThis = this;


	//-----------------------------------------------------------------------------------------------------------------

	private static HashMap<String, ExtDbManager> managers = new HashMap<String, ExtDbManager>();

	public static boolean showManager(String theDbID) {

		ExtDbManager theManager = managers.get(theDbID);
		if (null != theManager) {
			theManager.toFront();
			return true;
		}

		if (DbCore.isDbRegistered(theDbID)) {
			theManager = new ExtDbManager(theDbID);
			managers.put(theDbID, theManager);
			AppController.showWindow(theManager);
			return true;
		}

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	private ExtDbManager(String theDbID) {

		super(null, WINDOW_TITLE);

		dbID = theDbID;

		// Table for the data list.

		extDbModel = new ExtDbListModel();
		extDbTable = extDbModel.createTable();

		extDbTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
			public void valueChanged(ListSelectionEvent theEvent) {
				updateControls();
			}
		});

		JPanel extDbPanel = new JPanel(new BorderLayout());
		extDbPanel.setBorder(BorderFactory.createTitledBorder("Station Data"));
		extDbPanel.add(AppController.createScrollPane(extDbTable), BorderLayout.CENTER);

		// Pop-up menus for various downloading, creating, and importing steps.  If the download list has only one
		// entry, show as a button instead of a menu.

		ArrayList<KeyedRecord> dlTypes = ExtDb.getDownloadTypes();
		if (dlTypes.size() > 1) {

			dlTypes.add(0, new KeyedRecord(0, "Download..."));
			downloadMenu = new KeyedRecordMenu(dlTypes);

			downloadMenu.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					if (blockActions()) {
						int theType = downloadMenu.getSelectedKey();
						downloadMenu.setSelectedIndex(0);
						blockActionsEnd();
						if (theType > 0) {
							doDownloadAndImport(theType);
						}
					}
				}
			});

		} else {

			KeyedRecord dlType = dlTypes.get(0);
			downloadButton = new JButton("Download " + dlType.name);
			downloadButton.setFocusable(false);

			final int downloadType = dlType.key;
			downloadButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doDownloadAndImport(downloadType);
				}
			});
		}

		// Pop-up menu for create-and-import, supports all types.

		ArrayList<KeyedRecord> impTypes = ExtDb.getImportTypes();
		impTypes.add(0, new KeyedRecord(0, "Create & Import..."));
		createAndImportMenu = new KeyedRecordMenu(impTypes);

		createAndImportMenu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				if (blockActions()) {
					int theType = createAndImportMenu.getSelectedKey();
					createAndImportMenu.setSelectedIndex(0);
					blockActionsEnd();
					if (theType > 0) {
						doCreateAndImport(theType);
					}
				}
			}
		});

		// Button to import into a selected generic data set.

		importGenericButton = new JButton("Import More");
		importGenericButton.setFocusable(false);
		importGenericButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doImportGeneric();
			}
		});

		// Do the layout.

		JPanel butLeft = new JPanel(new FlowLayout(FlowLayout.LEFT));
		if (null != downloadMenu) {
			butLeft.add(downloadMenu);
		} else {
			butLeft.add(downloadButton);
		}
		butLeft.add(createAndImportMenu);

		JPanel butRight = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		butRight.add(importGenericButton);

		Box buttonBox = Box.createHorizontalBox();
		buttonBox.add(butLeft);
		buttonBox.add(butRight);

		Container cp = getContentPane();
		cp.setLayout(new BorderLayout());
		cp.add(extDbPanel, BorderLayout.CENTER);
		cp.add(buttonBox, BorderLayout.SOUTH);

		pack();

		Dimension theSize = new Dimension(500, 400);
		setMinimumSize(theSize);
		setSize(theSize);

		// Build the file menu.

		fileMenu.removeAll();

		// Previous

		JMenuItem miPrevious = new JMenuItem("Previous");
		miPrevious.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_UP, AppController.MENU_SHORTCUT_KEY_MASK));
		miPrevious.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doPrevious();
			}
		});
		fileMenu.add(miPrevious);

		// Next

		JMenuItem miNext = new JMenuItem("Next");
		miNext.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, AppController.MENU_SHORTCUT_KEY_MASK));
		miNext.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doNext();
			}
		});
		fileMenu.add(miNext);

		// __________________________________

		fileMenu.addSeparator();

		// Refresh List

		JMenuItem miRefresh = new JMenuItem("Refresh List");
		miRefresh.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				ExtDb.reloadCache(dbID);
			}
		});
		fileMenu.add(miRefresh);

		// __________________________________

		fileMenu.addSeparator();

		// Download ->

		if (dlTypes.size() > 1) {

			JMenu meDownload = new JMenu("Download");
			JMenuItem miDownload;
			for (KeyedRecord theType : dlTypes) {
				miDownload = new JMenuItem(theType.name);
				final int typeKey = theType.key;
				miDownload.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doDownloadAndImport(typeKey);
					}
				});
				meDownload.add(miDownload);
			}
			fileMenu.add(meDownload);

		} else {

			KeyedRecord theType = dlTypes.get(0);
			JMenuItem miDownload = new JMenuItem("Download " + theType.name);
			final int typeKey = theType.key;
			miDownload.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent theEvent) {
					doDownloadAndImport(typeKey);
				}
			});
			fileMenu.add(miDownload);
		}

		// Create & Import ->

		JMenu meImport = new JMenu("Create & Import");
		JMenuItem miImport;
		for (KeyedRecord theType : impTypes) {
			if (theType.key > 0) {
				miImport = new JMenuItem(theType.name);
				final int typeKey = theType.key;
				miImport.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent theEvent) {
						doCreateAndImport(typeKey);
					}
				});
				meImport.add(miImport);
			}
		}
		fileMenu.add(meImport);

		// __________________________________

		fileMenu.addSeparator();

		// Import More...

		importGenericMenuItem = new JMenuItem("Import More...");
		importGenericMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doImportGeneric();
			}
		});
		fileMenu.add(importGenericMenuItem);

		// Rename...

		renameExtDbMenuItem = new JMenuItem("Rename...");
		renameExtDbMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doRenameExtDb();
			}
		});
		fileMenu.add(renameExtDbMenuItem);

		// Delete

		deleteExtDbMenuItem = new JMenuItem("Delete");
		deleteExtDbMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doDeleteExtDb();
			}
		});
		fileMenu.add(deleteExtDbMenuItem);

		// Clear Bad Data Flag

		clearBadDataFlagMenuItem = new JMenuItem("Clear Bad Data Flag");
		clearBadDataFlagMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doClearBadDataFlag();
			}
		});
		fileMenu.add(clearBadDataFlagMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// Unlock

		unlockExtDbMenuItem = new JMenuItem("Unlock");
		unlockExtDbMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doUnlockExtDb();
			}
		});
		fileMenu.add(unlockExtDbMenuItem);

		// __________________________________

		fileMenu.addSeparator();

		// Get Info

		extDbInfoMenuItem = new JMenuItem("Get Info");
		extDbInfoMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_I, AppController.MENU_SHORTCUT_KEY_MASK));
		extDbInfoMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				doExtDbInfo();
			}
		});
		fileMenu.add(extDbInfoMenuItem);

		// Build the extra menu.

		extraMenu.removeAll();

		// Study Manager

		JMenuItem miStudy = new JMenuItem("Study Manager");
		miStudy.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				StudyManager.showManager(dbID);
			}
		});
		extraMenu.add(miStudy);

		// Template Manager

		JMenuItem miTemplates = new JMenuItem("Template Manager");
		miTemplates.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent theEvent) {
				TemplateManager.showManager(dbID);
			}
		});
		extraMenu.add(miTemplates);

		updateControls();

		updateDocumentName();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getDbID() {

		return dbID;
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected String getFileMenuName() {

		return "Data";
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected boolean showsExtraMenu() {

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected String getExtraMenuName() {

		return "Database";
	}


	//-----------------------------------------------------------------------------------------------------------------

	protected boolean showsEditMenu() {

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void updateDocumentName() {

		setDocumentName(DbCore.getHostDbName(dbID));
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void updateControls() {

		boolean eImport = false, eRename = false, eDelete = false, eClearBad = false, eUnlock = false;

		int rowIndex = extDbTable.getSelectedRow();
		if (rowIndex >= 0) {
			ExtDbListItem theItem = extDbModel.get(extDbTable.convertRowIndexToModel(rowIndex));
			eImport = !theItem.isLocked && theItem.isGeneric && theItem.isSupported;
			eRename = !theItem.isLocked && theItem.isSupported;
			eDelete = !theItem.isLocked;
			eClearBad = theItem.hasBadData;
			eUnlock = theItem.isLocked;
		}

		importGenericButton.setEnabled(eImport);
		importGenericMenuItem.setEnabled(eImport);
		renameExtDbMenuItem.setEnabled(eRename);
		deleteExtDbMenuItem.setEnabled(eDelete);
		clearBadDataFlagMenuItem.setEnabled(eClearBad);
		unlockExtDbMenuItem.setEnabled(eUnlock);
	}


	//=================================================================================================================
	// Data class for the station data list.

	private class ExtDbListItem {

		private Integer key;
		private int type;
		private int version;
		private int indexVersion;
		private String id;
		private String name;
		private boolean isLocked;
		private boolean isGeneric;
		private boolean isSupported;
		private boolean hasBadData;
	}


	//=================================================================================================================

	private class ExtDbListModel extends AbstractTableModel {

		private static final String EXT_DB_TYPE_COLUMN = "Type";
		private static final String EXT_DB_ID_COLUMN = "Date";
		private static final String EXT_DB_NAME_COLUMN = "Name";

		private String[] columnNames = {
			EXT_DB_TYPE_COLUMN,
			EXT_DB_ID_COLUMN,
			EXT_DB_NAME_COLUMN
		};

		private static final int EXT_DB_TYPE_INDEX = 0;
		private static final int EXT_DB_ID_INDEX = 1;
		private static final int EXT_DB_NAME_INDEX = 2;

		private ArrayList<ExtDbListItem> modelRows;


		//-------------------------------------------------------------------------------------------------------------

		private ExtDbListModel() {

			super();

			modelRows = new ArrayList<ExtDbListItem>();
		}


		//-------------------------------------------------------------------------------------------------------------

		private JTable createTable() {

			JTable theTable = new JTable(this);
			AppController.configureTable(theTable);

			DefaultTableCellRenderer theRend = new DefaultTableCellRenderer() {
				public Component getTableCellRendererComponent(JTable t, Object o, boolean s, boolean f, int r,
						int c) {
					JLabel comp = (JLabel)super.getTableCellRendererComponent(t, o, s, f, r, c);
					ExtDbListItem theItem = modelRows.get(t.convertRowIndexToModel(r));
					if (!s && theItem.hasBadData) {
						comp.setForeground(Color.RED);
					} else {
						comp.setForeground(Color.BLACK);
					}
					return comp;
				}
			};

			TableColumn theColumn = theTable.getColumn(EXT_DB_TYPE_COLUMN);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[6]);

			theColumn = theTable.getColumn(EXT_DB_ID_COLUMN);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[15]);

			theColumn = theTable.getColumn(EXT_DB_NAME_COLUMN);
			theColumn.setCellRenderer(theRend);
			theColumn.setMinWidth(AppController.textFieldWidth[5]);
			theColumn.setPreferredWidth(AppController.textFieldWidth[20]);

			return theTable;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void setItems(ArrayList<ExtDbListItem> newItems) {

			modelRows.clear();
			modelRows.addAll(newItems);
			fireTableDataChanged();
		}


		//-------------------------------------------------------------------------------------------------------------

		private ExtDbListItem get(int rowIndex) {

			return modelRows.get(rowIndex);
		}


		//-------------------------------------------------------------------------------------------------------------

		private void remove(int rowIndex) {

			modelRows.remove(rowIndex);
			fireTableRowsDeleted(rowIndex, rowIndex);
		}


		//-------------------------------------------------------------------------------------------------------------

		private void itemWasChanged(int rowIndex) {

			fireTableRowsUpdated(rowIndex, rowIndex);
		}


		//-------------------------------------------------------------------------------------------------------------

		private int indexOf(ExtDbListItem theItem) {

			return modelRows.indexOf(theItem);
		}


		//-------------------------------------------------------------------------------------------------------------

		private int indexOfKey(Integer theKey) {

			int rowIndex = 0;
			for (ExtDbListItem theItem : modelRows) {
				if (theItem.key.equals(theKey)) {
					return rowIndex;
				}
				rowIndex++;
			}

			return -1;
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getColumnCount() {

			return columnNames.length;
		}


		//-------------------------------------------------------------------------------------------------------------

		public String getColumnName(int columnIndex) {

			return columnNames[columnIndex];
		}


		//-------------------------------------------------------------------------------------------------------------

		public int getRowCount() {

			return modelRows.size();
		}
			

		//-------------------------------------------------------------------------------------------------------------

		public Object getValueAt(int rowIndex, int columnIndex) {

			ExtDbListItem theItem = modelRows.get(rowIndex);

			switch (columnIndex) {

				case EXT_DB_TYPE_INDEX:
					return ExtDb.getTypeName(theItem.type);

				case EXT_DB_ID_INDEX:
					return theItem.id;

				case EXT_DB_NAME_INDEX:
					return theItem.name;
			}

			return "";
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update the list to current database contents, optionally preserving the table selection.  The ExtDb class is
	// responsible for managing the data set table and a cache of it's contents, so this does not do a direct query.
	// The public version is the ExtDbListener call.

	public void updateExtDbList() {
		updateExtDbList(true);
	}

	private void updateExtDbList(boolean preserveSelection) {

		String saveTitle = errorReporter.getTitle();
		errorReporter.setTitle("Load Station Data List");
	
		ExtDbListItem selectedItem = null;
		if (preserveSelection) {
			int rowIndex = extDbTable.getSelectedRow();
			if (rowIndex >= 0) {
				selectedItem = extDbModel.get(extDbTable.convertRowIndexToModel(rowIndex));
			}
		}

		ArrayList<ExtDb> extdbs = ExtDb.getExtDbs(dbID, true, errorReporter);
		if (null == extdbs) {
			errorReporter.setTitle(saveTitle);
			return;
		}

		ArrayList<ExtDbListItem> list = new ArrayList<ExtDbListItem>();
		ExtDbListItem theItem;
		for (ExtDb theDb : extdbs) {
			theItem = new ExtDbListItem();
			theItem.key = theDb.key;
			theItem.type = theDb.type;
			theItem.version = theDb.version;
			theItem.indexVersion = theDb.indexVersion;
			theItem.id = theDb.id;
			if (theDb.isDownload) {
				theItem.name = ExtDb.DOWNLOAD_SET_NAME;
			} else {
				theItem.name = theDb.name;
			}
			theItem.isLocked = theDb.isLocked;
			theItem.isGeneric = theDb.isGeneric();
			theItem.isSupported = theDb.isSupported();
			theItem.hasBadData = theDb.hasBadData;
			list.add(theItem);
		}

		extDbModel.setItems(list);

		if (selectedItem != null) {
			int rowIndex = extDbModel.indexOf(selectedItem);
			if (rowIndex >= 0) {
				rowIndex = extDbTable.convertRowIndexToView(rowIndex);
				extDbTable.setRowSelectionInterval(rowIndex, rowIndex);
				extDbTable.scrollRectToVisible(extDbTable.getCellRect(rowIndex, 0, true));
			}
		}

		updateControls();

		errorReporter.setTitle(saveTitle);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Re-query the database record for an entry.  This does directly query the database table in case the cache in
	// ExtDb is out of date.  If a change is detected notify that class to update, however it will call back here to
	// reload the entire list so defer that notification.  Return an updated object if anything changed, the existing
	// object if errors occur, or null if the data set is deleted.

	private ExtDbListItem checkExtDb(int rowIndex, ErrorReporter errors) {

		ExtDbListItem theItem = extDbModel.get(rowIndex);

		boolean error = false, notfound = false, didchange = false;
		int newType, newVersion, newIndexVersion;
		String newID, newName;
		boolean newLocked;

		DbConnection db = DbCore.connectDb(dbID, errors);
		if (null != db) {
			try {

				db.query(
				"SELECT " +
					"db_type, " +
					"version, " +
					"index_version, " +
					"id, " +
					"is_download, " +
					"name, " +
					"locked " +
				"FROM " +
					"ext_db " +
				"WHERE " +
					"NOT deleted " +
					"AND ext_db_key = " + theItem.key);

				if (db.next()) {

					newType = db.getInt(1);
					newVersion = db.getInt(2);
					newIndexVersion = db.getInt(3);
					newID = db.getString(4);
					if (db.getBoolean(5)) {
						newName = ExtDb.DOWNLOAD_SET_NAME;
					} else {
						newName = db.getString(6);
					}
					newLocked = db.getBoolean(7);

					if ((newType != theItem.type) || (newVersion != theItem.version) ||
							(newIndexVersion != theItem.indexVersion) || !newID.equals(theItem.id) ||
							!newName.equals(theItem.name) || (newLocked != theItem.isLocked)) {

						theItem.type = newType;
						theItem.version = newVersion;
						theItem.indexVersion = newIndexVersion;
						theItem.id = newID;
						theItem.name = newName;
						theItem.isLocked = newLocked;
						theItem.isGeneric = ExtDb.isGeneric(newType);
						theItem.isSupported = ExtDb.isSupported(newType, newVersion, newIndexVersion);

						didchange = true;
					}

				} else {
					notfound = true;
					didchange = true;
				}

				DbCore.releaseDb(db);

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				error = true;
				DbConnection.reportError(errors, se);
			}

		} else {
			return theItem;
		}

		if (error) {
			return theItem;
		}

		if (notfound) {
			if (null != errors) {
				errors.reportError("The station data no longer exists");
			}
			theItem = null;
		}

		if (didchange) {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					ExtDb.reloadCache(dbID);
				}
			});
		}

		return theItem;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Move selection up or down in the table.

	private void doPrevious() {

		int size = extDbTable.getRowCount();
		int rowIndex = extDbTable.getSelectedRow();
		if ((size > 0) && (rowIndex != 0)) {
			if (rowIndex < 0) {
				rowIndex = size - 1;
			} else {
				rowIndex--;
			}
			extDbTable.setRowSelectionInterval(rowIndex, rowIndex);
			extDbTable.scrollRectToVisible(extDbTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doNext() {

		int size = extDbTable.getRowCount();
		int rowIndex = extDbTable.getSelectedRow();
		if ((size > 0) && (rowIndex < (size - 1))) {
			if (rowIndex < 0) {
				rowIndex = 0;
			} else {
				rowIndex++;
			}
			extDbTable.setRowSelectionInterval(rowIndex, rowIndex);
			extDbTable.scrollRectToVisible(extDbTable.getCellRect(rowIndex, 0, true));
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Do a download-and-import of CDBS or LMS data, managed by a RunPanel in the run manager.  This does not prompt
	// for a data set name.  By default download sets have no name and are flagged as downloads so they are deleted by
	// the next download.  Renaming a set manually clears the download flag so it is not deleted.

	private void doDownloadAndImport(int theType) {

		String title = "Download " + ExtDb.getTypeName(theType) + " station data";
		errorReporter.setTitle(title);

		if (ExtDb.isDownloadInProgress()) {
			errorReporter.reportWarning("Download is already in progress");
			RunManager.showRunManager();
			return;
		}

		final int extDbType = theType;

		RunPanelThread thePanel = new RunPanelThread(title, dbID) {
			public Object runActivity(StatusLogger status, ErrorLogger errors) {
				return ExtDb.downloadDatabase(dbID, extDbType, "", status, errors);
			}
		};
		thePanel.memoryFraction = 0.;

		if (thePanel.initialize(errorReporter)) {
			RunManager.addRunPanel(thePanel);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Do a create-and-import of a data set, including generic types.  When adding another data set no attempt is made
	// to check for duplication.  An optional descriptive name may be set for UI labelling.  If that is blank the UI
	// label will be an ID string appropriate to the data.  The import is handled by a background thread task in the
	// run manager for LMS or CDBS, for generic imports see importGeneric().

	private void doCreateAndImport(int theType) {

		String title = "Import " + ExtDb.getTypeName(theType) + " Station Data";
		errorReporter.setTitle(title);

		switch (theType) {

			// For CDBS and LMS databases the station data is SQL dump files with fixed names, user just selects
			// enclosing directory and the rest of the logic is in ExtDb.createNewDatabase().  This now also supports
			// selecting the data dump ZIP file directly, the import code will extract files from that.

			case ExtDb.DB_TYPE_CDBS:
			case ExtDb.DB_TYPE_LMS:
			case ExtDb.DB_TYPE_CDBS_FM: {

				JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
				chooser.setDialogType(JFileChooser.OPEN_DIALOG);
				chooser.setMultiSelectionEnabled(false);

				JTextField nameField = new JTextField(12);
				AppController.fixKeyBindings(nameField);

				JPanel namePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
				namePanel.setBorder(BorderFactory.createTitledBorder("Data set name (optional)"));
				namePanel.add(nameField);

				JPanel accessoryPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
				accessoryPanel.add(namePanel);

				chooser.setAccessory(accessoryPanel);

				chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
				chooser.addChoosableFileFilter(new FileNameExtensionFilter("ZIP (*.zip)", "zip"));
				chooser.setAcceptAllFileFilterUsed(false);
				chooser.setDialogTitle("Choose " + ExtDb.getTypeName(theType) + " data file directory, or ZIP file");

				// If a name is entered, check it for length and use of the reserved character for uniqueness.

				File theFile = null;
				String theName = "";
				while (true) {
					if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Choose")) {
						break;
					}
					theName = nameField.getText().trim();
					if (ExtDb.checkExtDbName(dbID, theName, errorReporter)) {
						theFile = chooser.getSelectedFile();
						break;
					}
				};
				if (null == theFile) {
					break;
				}

				// This is a bug in the chooser when both files and directories can be selected.  A double-click on a
				// directory opens it but also leaves the directory name in the file name field.  If that state is
				// chosen, the resulting path specifies a file in the directory that has the directory name as the
				// file name, in other words the final path component is erroneously repeated.  That invalid path of
				// course does not exist, in which case choose the enclosing directory.

				if (!theFile.exists()) {
					theFile = theFile.getParentFile();
				}

				AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

				final int extDbType = theType;
				final File sourceFile = theFile;
				final String extDbName = theName;

				RunPanelThread thePanel = new RunPanelThread(title, dbID) {
					public Object runActivity(StatusLogger status, ErrorLogger errors) {
						return ExtDb.createNewDatabase(dbID, extDbType, sourceFile, extDbName, status, errors);
					}
				};
				thePanel.memoryFraction = 0.;

				if (thePanel.initialize(errorReporter)) {
					RunManager.addRunPanel(thePanel);
				}

				break;
			}

			// Generic import handled by another method.

			case ExtDb.DB_TYPE_GENERIC_TV:
			case ExtDb.DB_TYPE_GENERIC_WL:
			case ExtDb.DB_TYPE_GENERIC_FM: {

				importGeneric(null, theType, title);

				break;
			}

			default: {
				errorReporter.reportError("Unknown or unsupported station data type");
				break;
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Import into an existing generic data set.

	private void doImportGeneric() {

		int rowIndex = extDbTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		ExtDbListItem theItem = checkExtDb(extDbTable.convertRowIndexToModel(rowIndex), errorReporter);
		if (theItem.isLocked || !theItem.isGeneric || !theItem.isSupported) {
			return;
		}

		String title = "Import " + ExtDb.getTypeName(theItem.type) + " Station Data";
		errorReporter.setTitle(title);

		ExtDb extDb = ExtDb.getExtDb(dbID, theItem.key, errorReporter);
		if (null == extDb) {
			return;
		}

		importGeneric(extDb, extDb.type, title);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Create and/or import into a generic data set.  If theExtDb is null this will create the data set and also prompt
	// for optional name in the file selection UI.  File selection is done first, if that is cancelled the data set is
	// not created, but if import fails the set will exist after with no content.  The primary import format is the
	// generic delimited text format, but this also supports the Canadian Broadcasting Services DBF format for TV/FM.
	// The generic text format allows several different field delimiters, see AppCore.readAndParseLine().  For wireless
	// records, the original wireless CSV format will be recognized by the generic text parser.  In the past importing
	// that wireless format created a separate single-import data set with an ExtDbRecord API, but that was eliminated
	// since generic provides all the same capability plus multi-import.

	private void importGeneric(ExtDb theExtDb, int theType, String title) {

		if (!ExtDb.isGeneric(theType)) {
			errorReporter.reportError("Unknown or unsupported station data type");
			return;
		}

		JFileChooser chooser = new JFileChooser(AppCore.getProperty(AppCore.LAST_FILE_DIRECTORY_KEY));
		chooser.setDialogType(JFileChooser.OPEN_DIALOG);
		chooser.setDialogTitle(title);
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new FileNameExtensionFilter("Text (*.csv,*.txt,*.dat)", "csv", "txt", "dat"));
		if ((ExtDb.DB_TYPE_GENERIC_TV == theType) || (ExtDb.DB_TYPE_GENERIC_FM == theType)) {
			chooser.addChoosableFileFilter(new FileNameExtensionFilter("Canadian DBF (*.dbf)", "dbf"));
		}
		chooser.setAcceptAllFileFilterUsed(false);

		JCheckBox patternFileCheckBox = new JCheckBox("Separate pattern file");
		if (ExtDb.DB_TYPE_GENERIC_WL == theType) {
			patternFileCheckBox.setSelected(true);
		}

		JPanel textPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		textPanel.setBorder(BorderFactory.createTitledBorder("Text options"));
		textPanel.add(patternFileCheckBox);

		Box optionsBox = Box.createVerticalBox();

		JTextField nameField = null;
		if (null == theExtDb) {

			nameField = new JTextField(12);
			AppController.fixKeyBindings(nameField);

			JPanel namePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			namePanel.setBorder(BorderFactory.createTitledBorder("Data set name (optional)"));
			namePanel.add(nameField);

			optionsBox.add(namePanel);
		}

		optionsBox.add(textPanel);

		JPanel accessoryPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		accessoryPanel.add(optionsBox);

		chooser.setAccessory(accessoryPanel);

		File theFile = null;
		String extDbName = "";
		while (true) {
			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Import")) {
				break;
			}
			if (null != nameField) {
				extDbName = nameField.getText().trim();
				if (!ExtDb.checkExtDbName(dbID, extDbName, errorReporter)) {
					continue;
				}
			}
			theFile = chooser.getSelectedFile();
			break;
		};
		if (null == theFile) {
			return;
		}

		AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());

		// Determine the import format from the selected file name extension.

		String theName = theFile.getName().toLowerCase();
		int theFormat = ExtDb.GENERIC_FORMAT_TEXT;
		if (theName.endsWith(".dbf")) {
			theFormat = ExtDb.GENERIC_FORMAT_DBF;
		}

		final int importFormat = theFormat;
		final File importFile = theFile;

		// Text import has a primary station data file and optionally a separate pattern data file, prompt for the
		// pattern file if needed.

		theFile = null;
		if ((ExtDb.GENERIC_FORMAT_TEXT == theFormat) && patternFileCheckBox.isSelected()) {
			chooser = new JFileChooser(importFile.getParentFile().getAbsolutePath());
			chooser.setDialogType(JFileChooser.OPEN_DIALOG);
			chooser.setDialogTitle("Choose pattern data file");
			chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
			chooser.setMultiSelectionEnabled(false);
			chooser.addChoosableFileFilter(new FileNameExtensionFilter("TVStudy text (*.csv,*.txt,*.dat)", "csv",
				"txt", "dat"));
			chooser.setAcceptAllFileFilterUsed(false);
			if (JFileChooser.APPROVE_OPTION != chooser.showDialog(this, "Import")) {
				return;
			}
			theFile = chooser.getSelectedFile();
			AppCore.setProperty(AppCore.LAST_FILE_DIRECTORY_KEY, theFile.getParentFile().getAbsolutePath());
		}

		final File patternFile = theFile;

		// Create the data set if needed.  This should not take significant time so is done on the main thread.

		if (null == theExtDb) {
			Integer newKey = ExtDb.createNewGenericDatabase(dbID, theType, extDbName, errorReporter);
			if (null == newKey) {
				return;
			}
			theExtDb = ExtDb.getExtDb(dbID, newKey, errorReporter);
			if (null == theExtDb) {
				return;
			}
		}
		final ExtDb extDb = theExtDb;

		// Run the import on a background thread, this is not expected to be a terribly lengthy process so doesn't use
		// a run panel, just a thread worker.

		BackgroundWorker<Integer> theWorker = new BackgroundWorker<Integer>(this, title) {
			protected Integer doBackgroundWork(ErrorLogger errors) {
				int sourceCount = ExtDb.importToGenericDatabase(extDb, importFormat, importFile, patternFile, errors);
				if (sourceCount < 0) {
					return null;
				}
				return Integer.valueOf(sourceCount);
			}
		};

		errorReporter.clearMessages();

		Integer count = theWorker.runWork("Importing records, please wait...", errorReporter);
		if (null != count) {
			errorReporter.showMessages();
			errorReporter.reportMessage("Imported " + count + " records");
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// The name is really just a description, but for consistency this does enforce uniqueness.  The name is always
	// optional so it can be set to an empty string, in which case the permanent ID string appears in the UI.  When a
	// name is set, even to an empty name, the download flag is also cleared; see ExtDb.downloadDatabase().  For that
	// reason an empty new name will always be saved even if the old name is already empty.

	private void doRenameExtDb() {

		String title = "Rename Station Data";
		errorReporter.setTitle(title);

		int rowIndex = extDbTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		ExtDbListItem theItem = checkExtDb(extDbTable.convertRowIndexToModel(rowIndex), errorReporter);
		if ((null == theItem) || theItem.isLocked || !theItem.isSupported) {
			return;
		}

		String oldName = theItem.name, newName = null;
		if (oldName.equals(ExtDb.DOWNLOAD_SET_NAME)) {
			oldName = "";
		}
		while (true) {
			newName = (String)(JOptionPane.showInputDialog(this, "Enter a name for the data (optional)", title,
				JOptionPane.QUESTION_MESSAGE, null, null, oldName));
			if (null == newName) {
				return;
			}
			newName = newName.trim();
			if ((oldName.length() > 0) && newName.equals(oldName)) {
				return;
			}
			if (ExtDb.checkExtDbName(dbID, newName, oldName, errorReporter)) {
				break;
			}
		};

		ExtDb.renameDatabase(dbID, theItem.key, newName, errorReporter);
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doDeleteExtDb() {

		int rowIndex = extDbTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		ExtDbListItem theItem = checkExtDb(extDbTable.convertRowIndexToModel(rowIndex), errorReporter);
		if ((null == theItem) || theItem.isLocked) {
			return;
		}

		errorReporter.setTitle("Delete Station Data");

		ExtDb.deleteDatabase(dbID, theItem.key.intValue(), errorReporter);
	}


	//-----------------------------------------------------------------------------------------------------------------

	private void doClearBadDataFlag() {

		int rowIndex = extDbTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		ExtDbListItem theItem = checkExtDb(extDbTable.convertRowIndexToModel(rowIndex), errorReporter);
		if ((null == theItem) || theItem.isLocked) {
			return;
		}

		errorReporter.setTitle("Clear Bad Data Flag");

		ExtDb.clearBadDataFlag(dbID, theItem.key.intValue(), errorReporter);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Unlock a data set, get confirmation, this should not be done unless a previous crash left the lock hanging.

	private void doUnlockExtDb() {

		String title = "Unlock Station Data";
		errorReporter.setTitle(title);

		int rowIndex = extDbTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		ExtDbListItem theItem = checkExtDb(extDbTable.convertRowIndexToModel(rowIndex), errorReporter);
		if ((null == theItem) || !theItem.isLocked) {
			return;
		}

		AppController.beep();
		if (JOptionPane.YES_OPTION != JOptionPane.showConfirmDialog(this,
				"This will clear the in-use flag on the data set.  Do this only if the flag\n" +
				"was not cleared because of an application crash or network failure.  If\n" +
				"this is done when another application is still using the data set, that\n" +
				"application could fail and data could become corrupted.\n\n" +
				"Do you want to continue?", title, JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE)) {
			return;
		}

		ExtDb extDb = ExtDb.getExtDb(dbID, theItem.key, errorReporter);
		if (null == extDb) {
			return;
		}

		theItem.isLocked = extDb.unlock(errorReporter);

		updateControls();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Display a data set info dialog, shows primary key and other information not otherwise visible in UI.

	private void doExtDbInfo() {

		String title = "Data Set Info";
		errorReporter.setTitle(title);

		int rowIndex = extDbTable.getSelectedRow();
		if (rowIndex < 0) {
			return;
		}

		ExtDbListItem theItem = checkExtDb(extDbTable.convertRowIndexToModel(rowIndex), errorReporter);
		if (null == theItem) {
			return;
		}

		StringBuilder mesg = new StringBuilder();
		mesg.append("Name: ");
		mesg.append(theItem.name);
		mesg.append('\n');
		mesg.append("Type: ");
		mesg.append(ExtDb.getTypeName(theItem.type));
		mesg.append('\n');
		mesg.append("Date: ");
		mesg.append(theItem.id);
		mesg.append('\n');
		mesg.append("Primary key: ");
		mesg.append(String.valueOf(theItem.key));
		mesg.append('\n');
		mesg.append("Database name: ");
		mesg.append(ExtDb.makeDbName(DbCore.getDbName(dbID), theItem.type, theItem.key));
		mesg.append('\n');
		mesg.append("Version: ");
		mesg.append(String.valueOf(theItem.version));
		mesg.append('\n');
		mesg.append("Index version: ");
		mesg.append(String.valueOf(theItem.indexVersion));
		if (theItem.isLocked) {
			mesg.append("\nLocked for data import");
		}
		if (!theItem.isSupported) {
			mesg.append("\n** Unknown/unsupported type or version **");
		}
		if (theItem.hasBadData) {
			mesg.append("\n** Consistency checks failed, may have bad data **");
		}

		AppController.showMessage(this, mesg.toString(), "Data Set Info");
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillOpen() {

		if (isVisible()) {
			return;
		}

		DbCore.openDb(dbID, this);

		ExtDb.addListener(this);

		DbController.restoreColumnWidths(dbID, getKeyTitle(), extDbTable);

		blockActionsClear();

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				updateExtDbList(false);
			}
		});
	}


	//-----------------------------------------------------------------------------------------------------------------

	public void windowWillClose() {

		if (!isVisible()) {
			return;
		}

		DbController.saveColumnWidths(dbID, getKeyTitle(), extDbTable);

		blockActionsSet();

		managers.remove(dbID);
		DbCore.closeDb(dbID, this);
	}
}
