//
//  ProcessRun.java
//  TVStudy
//
//  Copyright (c) 2019-2020 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.core;

import java.util.*;
import java.sql.*;
import java.io.*;
import java.nio.file.*;


//=====================================================================================================================
// Class to run a native process, typically a study engine run however this can handle any non-interactive, command-
// line-driven process.  A study engine may be interactive in a limited way using a prompt-and-response protocol.
// Another object must call poll() repeatedly to start and monitor the process, often enough to keep up with process
// output.  The process can be stopped if running, or startup cancelled with stop().  Check status with isRunning(),
// that will return true even before the process starts.  Once isRunning() returns false, didFail() indicates if the
// process failed to start, was stopped, or exited with a non-zero result; wasStopped() indicates if it was stopped.

// By default process output is discarded, except for some status messages from a study engine, see poll().  To get
// output for display or logging, and to support the prompt-and-response interactive protocol, an object implementing
// ProcessRun.Handler may be set.

// These are one-shot objects, once state progresses to completed/stopped/failed it cannot be re-started.

public class ProcessRun {

	private ArrayList<String> arguments;
	private String password;

	private Process process;
	private boolean processStarted;
	private boolean processRunning;
	private boolean processStopped;
	private boolean processFailed;

	private Handler handler;
	private boolean inPromptedState;
	private String promptResponse;
	private long lastResponseTime;
	private static final long CONFIRMATION_TIMEOUT = 2000;   // milliseconds
	private int responseAttempts;
	private static final int MAX_RESPONSE_ATTEMPTS = 3;

	private static final int READ_BUFFER_SIZE = 10000;
	private BufferedInputStream processOutput;
	private byte[] readBuffer;
	private static final int MAX_READ_PER_POLL = 100000;

	private StringBuilder outputBuffer;
	private boolean skipNextLine;

	private long lastOutputTime;
	private long stuckTimeout = 600000L;   // milliseconds
	private boolean wasAlive;

	private int processID;
	private long terminateSentTime;
	private static final long TERMINATE_WAIT = 10000L;   // millseconds

	public static final int POLL_INTERVAL = 200;   // milliseconds


	//=================================================================================================================
	// Interface for object that receives the process output and optionally handles some limited interactivity.

	public interface Handler {


		//-------------------------------------------------------------------------------------------------------------
		// Handle output from the process.

		public void processOutput(ProcessRun theRun, String line, boolean crTerminated);


		//-------------------------------------------------------------------------------------------------------------
		// Handle inline key=data messaging from the process, used by the study engine.

		public void processMessage(ProcessRun theRun, String key, String data);


		//-------------------------------------------------------------------------------------------------------------
		// Handler may have this return true to have stop() called at the next poll.  If a handler manages multiple
		// processes this would apply to all.  This may just return false, handler may always call stop() directly.

		public boolean isCanceled();


		//-------------------------------------------------------------------------------------------------------------
		// Provide response to a prompt from the process, or null if no response is known.

		public String getProcessResponse(ProcessRun theRun, String prompt);


		//-------------------------------------------------------------------------------------------------------------
		// Process confirmed the response.

		public void processResponseConfirmed(ProcessRun theRun);


		//-------------------------------------------------------------------------------------------------------------
		// Process failed for any reason.

		public void processFailed(ProcessRun theRun);


		//-------------------------------------------------------------------------------------------------------------
		// Process exited without error.

		public void processComplete(ProcessRun theRun);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// If password is non-null, the process is expected to output a password prompt, no other output will be handled
	// until that is seen and the password is written back.

	public ProcessRun(ArrayList<String> theArguments, String thePassword) {

		arguments = new ArrayList<String>(theArguments);
		password = thePassword;

		outputBuffer = new StringBuilder();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set a process handler, see poll().

	public void setHandler(Handler theHandler) {

		handler = theHandler;
		inPromptedState = false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Set a timeout in milliseconds to detect a stuck process, if no output at all for more than the set time the
	// process will be killed.  Set to 0 to disable the check.  Default is 10 minutes.

	public void setStuckTimeout(long theTimeout) {

		stuckTimeout = theTimeout;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Blocking run, poll in a loop until the process exits.

	public void run() {

		while (poll()) {
			try {Thread.currentThread().sleep(POLL_INTERVAL);} catch (Exception e) {};
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// On the first call this will start the process, later calls monitor the process, handle output, and check for
	// timeouts.  If the process generates output this must be called often enough to keep the output buffer clear
	// otherwise the process will stall on tty output.  If a handler is set, start by polling for a cancel, send a
	// stop() if that happens but that is initially a terminate request not an immediate kill so continue with the
	// poll.  Return true if the process is running, false if it has exited or fails to start.

	public synchronized boolean poll() {

		if ((null != handler) && !processStopped && handler.isCanceled()) {
			stop(true);
		}

		long now = System.currentTimeMillis();

		if (!processStarted) {

			processStarted = true;

			try {

				ProcessBuilder pb = new ProcessBuilder(arguments);

				// Set up environment, set the dynamic linker search path to include TVStudy installation directory.

				Map<String, String> env = pb.environment();

				String dyldpath = env.get("DYLD_LIBRARY_PATH");
				if (null == dyldpath) {
					dyldpath = env.get("LD_LIBRARY_PATH");
				}
				if (null == dyldpath) {
					dyldpath = AppCore.libDirectoryPath.toString();
				} else {
					dyldpath = dyldpath + ":" + AppCore.libDirectoryPath.toString();
				}
				env.put("DYLD_LIBRARY_PATH", dyldpath);
				env.put("LD_LIBRARY_PATH", dyldpath);

				// Set the path to the MySQL named socket file, if any; see comments in DbCore.registerDb().

				if (null != DbCore.mysqlSocketFile) {
					env.put("MYSQL_UNIX_PORT", DbCore.mysqlSocketFile);
				}

				pb.redirectErrorStream(true);

				process = pb.start();

				processRunning = true;
				wasAlive = true;

			} catch (Throwable t) {
				AppCore.log(AppCore.ERROR_MESSAGE, "Could not start process", t);
			}

			if (processRunning) {

				processOutput = new BufferedInputStream(process.getInputStream(), READ_BUFFER_SIZE);
				readBuffer = new byte[READ_BUFFER_SIZE];

				lastOutputTime = now;

			} else {

				processFailed = true;

				if (null != handler) {
					handler.processFailed(this);
				}

				return false;
			}
		}

		if (!processRunning) {
			return false;
		}

		// Check if a terminate was sent, see stop().

		if (terminateSentTime > 0L) {
			if ((now - terminateSentTime) > TERMINATE_WAIT) {
				stop(false);
				return false;
			}
		}

		// Collect more output, parse it if there is any.

		boolean sendPassword = false;

		if (collectOutput()) {

			lastOutputTime = now;

			// If still waiting to send the password check for the prompt, that is any text containing "password",
			// case-insensitive, regardless of line termination.  The actual send of the password is done later after
			// any terminated lines are stripped out of the buffer.

			if ((null != password) && outputBuffer.toString().toLowerCase().contains("password")) {
				sendPassword = true;
			}

			// Parse lines from output buffer, nothing is processed (except recognizing the password prompt) until it
			// is a terminated line, meaning not all new output may be processed on this poll.  Both linefeed and
			// carriage return terminators are recognized.

			int newPosition = 0, newLength = outputBuffer.length(), e;
			char nextChar;
			String mesgKey, mesgData, newLine = null;
			boolean crTerm;

			while (newPosition < newLength) {

				do {
					nextChar = outputBuffer.charAt(newPosition);
				} while (('\n' != nextChar) && ('\r' != nextChar) && (++newPosition < newLength));

				if (newPosition < newLength) {

					crTerm = ('\r' == nextChar);

					newLine = outputBuffer.substring(0, newPosition++);
					outputBuffer.delete(0, newPosition);
					newLength -= newPosition;
					newPosition = 0;

					// If still waiting to send a password or if the flag is set to skip a line, ignore the new line.
					// All output is discarded prior to sending the password.

					if ((null != password) || skipNextLine) {
						skipNextLine = false;
						continue;
					}

					// If a process handler is set, check the prompt-and-response state.  Initially watch for a line
					// that begins with the prompt prefix, when seen enter the prompted state and ask the handler for
					// the response, it will return null if the prompt is unknown or unexpected in which case kill the
					// process.  Otherwise the handler enters a pending state that must be terminated by either a
					// confirmation or failure.  The response is sent to the process with timeouts and retries hence
					// that is outside the line-parsing loop, see below.  The prompted state ends and the response is
					// confirmed once any output is seen after the response is sent.  If there is output before the
					// response is sent something is wrong, kill the process.

					if (null != handler) {

						if (inPromptedState) {

							if (responseAttempts > 0) {

								handler.processResponseConfirmed(this);
								inPromptedState = false;

							} else {

								AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected output '" + newLine +
									"' from controlled process");
								stop(false);
								return false;
							}

						} else {

							if (newLine.startsWith(AppCore.ENGINE_PROMPT_PREFIX)) {

								promptResponse = handler.getProcessResponse(this, newLine);

								if (null == promptResponse) {

									AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected prompt '" + newLine +
										"' from controlled process");
									stop(false);
									return false;

								} else {

									inPromptedState = true;
									lastResponseTime = 0;
									responseAttempts = 0;

									continue;
								}
							}
						}
					}

					// Send output to handler, if any.  Also this needs to parse engine status messages and handle
					// the PID message here regardless of whether a handler exists, see stop().

					if (newLine.startsWith(AppCore.ENGINE_MESSAGE_PREFIX)) {

						e = newLine.indexOf('=');
						if (e >= 0) {

							mesgKey = newLine.substring(AppCore.ENGINE_MESSAGE_PREFIX_LENGTH, e);
							mesgData = newLine.substring(e + 1);

							if (mesgKey.equals(AppCore.ENGINE_PID_KEY)) {
								try {
									processID = Integer.parseInt(mesgData);
								} catch (NumberFormatException nfe) {
								}
							}

							if (null != handler) {
								handler.processMessage(this, mesgKey, mesgData);
							}
						}

					} else {

						if (null != handler) {
							handler.processOutput(this, newLine, crTerm);
						}
					}
				}
			}

		} else {

			// Check if the process has exited, if exit value is non-zero set failed status.  Make sure there is always
			// one more poll after the exit, to avoid race conditions where buffered output can be missed.

			if (!process.isAlive()) {

				if (wasAlive) {
					wasAlive = false;
					return true;
				}

				if (process.exitValue() != 0) {
					processFailed = true;
				}

				processRunning = false;

				if (null != handler) {
					if (processFailed) {
						handler.processFailed(this);
					} else {
						handler.processComplete(this);
					}
				}

				return false;
			}
		}

		// If stuck-process timeout is set and there has been no output from the process, kill it.

		if ((stuckTimeout > 0) && ((now - lastOutputTime) > stuckTimeout)) {

			AppCore.log(AppCore.ERROR_MESSAGE, "Process stalled, no output for timeout interval");
			stop(false);

			return false;
		}

		// If the password needs to be written (see above), do it.  When a password is set all output is discarded
		// until the password is written.  The assumption is the user doesn't need to see the prompt or anything that
		// precedes it, which should be nothing.  Set a flag to ignore the next line of output, in case the password
		// echoes, and to suppress an initial blank line in the output.

		if (sendPassword) {

			try {
				OutputStream out = process.getOutputStream();
				out.write(password.getBytes());
				out.write(13);
				out.flush();
			} catch (IOException ie) {
				stop(false);
				return false;
			}

			password = null;
			skipNextLine = true;

			return true;
		}

		// If in the prompted state check to see if response needs to be sent or re-sent.  After sending, if the
		// response is not confirmed or failure detected (see above) within a timeout interval, send again, up to a
		// maximum number of attempts.  If retries are exceeded, kill the process.

		if (inPromptedState && ((now - lastResponseTime) > CONFIRMATION_TIMEOUT)) {

			if (++responseAttempts > MAX_RESPONSE_ATTEMPTS) {

				AppCore.log(AppCore.ERROR_MESSAGE, "No output from controlled process after sending prompt response");
				stop(false);
				return false;

			} else {

				lastResponseTime = now;

				try {
					OutputStream out = process.getOutputStream();
					out.write(promptResponse.getBytes());
					out.write(13);
					out.flush();
				} catch (IOException ie) {
					stop(false);
					return false;
				}
			}
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Collect any new output from the process, add it to the buffer, return true if anything read, else false.  To
	// prevent this from effectively blocking due to a process that generates large amounts of output very rapidly, it
	// will return after a maximum number of characters even if more are still available.

	private boolean collectOutput() {

		boolean result = false;

		if (null == processOutput) {
			return result;
		}

		int count, total = 0;

		try {
			while ((processOutput.available() > 0) && (total < MAX_READ_PER_POLL)) {
				count = processOutput.read(readBuffer);
				if (count > 0) {
					outputBuffer.append(new String(readBuffer, 0, count));
					result = true;
					total += count;
				}
			}
		} catch (IOException ie) {
			stop(false);
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Stop the process.  If the process has already started and exited, do nothing.  If not yet started, abort the
	// startup.  Otherwise, if doTerminate is true and the process provided a PID, send SIGTERM to the process which
	// should result in an orderly shutdown.  That starts a timeout checked in poll(), if the process does not exit
	// within the interval this will be called again with doTerminate false.  During that time repeated calls with
	// doTerminate true do nothing.  Whenever this is called with doTerminate false, or if the process does not send
	// a PID, the process is immediately killed with destroyForcibly().

	public synchronized void stop(boolean doTerminate) {

		if (processStarted && !processRunning) {
			return;
		}

		if (!processStarted) {

			processStarted = true;
			processStopped = true;

			return;
		}

		if (doTerminate && (processID > 0)) {

			if (terminateSentTime > 0L) {
				return;
			}
			terminateSentTime = System.currentTimeMillis();

			processStopped = true;
			processFailed = true;

			try {
				new ProcessBuilder("/bin/kill", "-TERM", String.valueOf(processID)).start();
				return;
			} catch (Throwable t) {
			}
		}

		try {
			process.destroyForcibly();
		} catch (Throwable t) {
		}

		processStopped = true;
		processFailed = true;

		processRunning = false;
		wasAlive = false;

		if (null != handler) {
			handler.processFailed(this);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// This returns true if the process has not yet started.

	public synchronized boolean isRunning() {

		if (processStarted) {
			return processRunning;
		}
		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// This is true as soon as stop() is used, even if process is still running.

	public synchronized boolean wasStopped() {

		return processStopped;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public synchronized boolean didFail() {

		return processFailed;
	}
}
