//
//  ExtDbRecordFM.java
//  TVStudy
//
//  Copyright (c) 2015-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.core;

import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;
import gov.fcc.tvstudy.core.geo.*;

import java.util.*;
import java.text.*;
import java.sql.*;


//=====================================================================================================================
// Concrete subclass of ExtDbRecord for FM records.  See the superclass for details.

public class ExtDbRecordFM extends ExtDbRecord implements StationRecord {

	// Mappings for status codes.  The value order of the types generally defines a record-preference order, see
	// isPreferredRecord().

	private static final String[] CDBS_STATUS_CODES = {
		"APP", "CP", "CP MOD", "LIC", "STA"
	};
	private static final int[] CDBS_STATUS_TYPES = {
		STATUS_TYPE_APP, STATUS_TYPE_CP, STATUS_TYPE_CP, STATUS_TYPE_LIC, STATUS_TYPE_STA
	};
	private static final String[] LMS_APP_STATUS_CODES = {
		"SUB", "DIS", "WIT", "DEN", "RET", "CAN", "REC", "REV"
	};

	// List of station classes, the stationClass property is an index into this array.  There is currently no benefit
	// to putting this in a root database enumeration table.

	public static final int FM_CLASS_OTHER = 0;
	public static final int FM_CLASS_C = 1;
	public static final int FM_CLASS_C0 = 2;
	public static final int FM_CLASS_C1 = 3;
	public static final int FM_CLASS_C2 = 4;
	public static final int FM_CLASS_C3 = 5;
	public static final int FM_CLASS_B = 6;
	public static final int FM_CLASS_B1 = 7;
	public static final int FM_CLASS_A = 8;
	public static final int FM_CLASS_D = 9;
	public static final int FM_CLASS_L1 = 10;
	public static final int FM_CLASS_L2 = 11;

	public static final String[] FM_CLASS_CODES = {
		"", "C", "C0", "C1", "C2", "C3", "B", "B1", "A", "D", "L1", "L2"
	};

	// Properties.

	public int facilityID;
	public boolean isIBOC;
	public double ibocFraction;
	public int stationClass;
	public String callSign;
	public int channel;
	public String status;
	public int statusType;
	public String filePrefix;
	public String appARN;
	public boolean daIndicated;
	public String licensee;
	public boolean contour73215;

	// See isOperating().

	private boolean isOperatingFacility;
	private boolean operatingStatusSet;


	//-----------------------------------------------------------------------------------------------------------------
	// Determine if a station data set is supported by code in this class.

	public static boolean isExtDbSupported(ExtDb extDb) {

		return (!extDb.deleted && extDb.isSupported() && extDb.canProvide(Source.RECORD_TYPE_FM) &&
			((ExtDb.DB_TYPE_CDBS_FM == extDb.type) || (ExtDb.DB_TYPE_LMS == extDb.type) ||
			(ExtDb.DB_TYPE_LMS_LIVE == extDb.type)));
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Search a CDBS or LMS database for FM records.  For CDBS, the query does an inner join on tables fm_eng_data,
	// application, and facility.  For LMS, it is tables app_location, app_antenna, app_antenna_frequency, application,
	// license_filing_version, and application_facility (however app_antenna and app_antenna_frequency are left joins
	// so fields may be null).  Records that have an unknown service or country, bad facility IDs, or bad channel are
	// ignored.  If the database type is not supported return an empty list.

	public static LinkedList<ExtDbRecord> findRecords(ExtDb extDb, String query, ErrorLogger errors) {
		return findRecords(extDb, query, null, 0., 0., errors);
	}

	public static LinkedList<ExtDbRecord> findRecords(ExtDb extDb, String query, GeoPoint searchCenter,
			double searchRadius, double kmPerDegree, ErrorLogger errors) {
		LinkedList<ExtDbRecordFM> records = findRecordsFM(extDb, query, searchCenter, searchRadius, kmPerDegree,
			errors);
		if (null != records) {
			return new LinkedList<ExtDbRecord>(records);
		}
		return null;
	}

	public static LinkedList<ExtDbRecordFM> findRecordsFM(ExtDb extDb, String query) {
		return findRecordsFM(extDb, query, null, 0., 0., null);
	}

	public static LinkedList<ExtDbRecordFM> findRecordsFM(ExtDb extDb, String query, ErrorLogger errors) {
		return findRecordsFM(extDb, query, null, 0., 0., errors);
	}

	private static HashMap<Integer, HashMap<Integer, Double>> ibocMaps =
		new HashMap<Integer, HashMap<Integer, Double>>();

	private static LinkedList<ExtDbRecordFM> findRecordsFM(ExtDb extDb, String query, GeoPoint searchCenter,
			double searchRadius, double kmPerDegree, ErrorLogger errors) {

		if (!isExtDbSupported(extDb)) {
			return new LinkedList<ExtDbRecordFM>();
		}

		boolean isCDBS = (ExtDb.DB_TYPE_CDBS_FM == extDb.type);

		// Compose the query.  See ExtDbRecordTV for comments on the LMS code.

		String fldStr = "", frmStr = "";
		StringBuilder whrStr = new StringBuilder();

		if (isCDBS) {

			fldStr =
			"SELECT " +
				"fm_eng_data.application_id," +
				"fm_eng_data.facility_id," +
				"fm_eng_data.station_class," +
				"fm_eng_data.station_channel," +
				"(CASE WHEN (facility.fac_callsign <> '') " +
					"THEN facility.fac_callsign ELSE application.fac_callsign END)," +
				"(CASE WHEN (facility.comm_city <> '') " +
					"THEN facility.comm_city ELSE application.comm_city END)," +
				"(CASE WHEN (facility.comm_state <> '') " +
					"THEN facility.comm_state ELSE application.comm_state END)," +
				"fm_eng_data.fm_dom_status," +
				"application.file_prefix," +
				"application.app_arn," +
				"fm_eng_data.lat_dir," +
				"fm_eng_data.lat_deg," +
				"fm_eng_data.lat_min," +
				"fm_eng_data.lat_sec," +
				"fm_eng_data.lon_dir," +
				"fm_eng_data.lon_deg," +
				"fm_eng_data.lon_min," +
				"fm_eng_data.lon_sec," +
				"fm_eng_data.rcamsl_horiz_mtr," +
				"fm_eng_data.rcamsl_vert_mtr," +
				"fm_eng_data.haat_horiz_rc_mtr," +
				"fm_eng_data.haat_vert_rc_mtr," +
				"fm_eng_data.max_horiz_erp," +
				"fm_eng_data.horiz_erp," +
				"fm_eng_data.max_vert_erp," +
				"fm_eng_data.vert_erp," +
				"fm_eng_data.antenna_id," +
				"fm_eng_data.ant_rotation," +
				"fm_eng_data.eng_record_type," +
				"fm_app_indicators.da_ind," +
				"fm_eng_data.asd_service," +
				"facility.fac_country," +
				"(CASE WHEN (app_tracking.accepted_date IS NOT NULL " +
					"AND (app_tracking.accepted_date <> '')) " +
					"THEN app_tracking.accepted_date " +
					"ELSE fm_eng_data.last_change_date END) AS sequence_date," +
				"facility.digital_status," +
				"facility.fac_status";

			frmStr =
			" FROM " +
				"fm_eng_data " +
				"JOIN facility USING (facility_id) " +
				"JOIN application USING (application_id) " +
				"LEFT JOIN fm_app_indicators USING (application_id) " +
				"LEFT JOIN app_tracking USING (application_id)";

			if ((null != query) && (query.length() > 0)) {
				whrStr.append(" WHERE (");
				whrStr.append(query);
				whrStr.append(')');
			}

		} else {

			if (ExtDb.DB_TYPE_LMS_LIVE == extDb.type) {

				fldStr =
				"SELECT " +
					"app_location.aloc_aapp_application_id," +
					"application_facility.afac_facility_id," +
					"application_facility.station_class_code," +
					"(CASE WHEN TRIM(application_facility.afac_channel) IN ('','null') THEN 0::INT " +
						"ELSE application_facility.afac_channel::INT END)," +
					"facility.callsign," +
					"(CASE WHEN (facility.community_served_city <> '') " +
						"THEN facility.community_served_city " +
						"ELSE application_facility.afac_community_city END)," +
					"(CASE WHEN (facility.community_served_state <> '') " +
						"THEN facility.community_served_state " +
						"ELSE application_facility.afac_community_state_code END)," +
					"license_filing_version.current_status_code," +
					"'BLANK' AS file_prefix," +
					"application.aapp_file_num," +
					"app_location.aloc_lat_dir," +
					"(CASE WHEN TRIM(app_location.aloc_lat_deg) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_location.aloc_lat_deg::FLOAT END)," +
					"(CASE WHEN TRIM(app_location.aloc_lat_mm) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_location.aloc_lat_mm::FLOAT END)," +
					"(CASE WHEN TRIM(app_location.aloc_lat_ss) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_location.aloc_lat_ss::FLOAT END)," +
					"app_location.aloc_long_dir," +
					"(CASE WHEN TRIM(app_location.aloc_long_deg) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_location.aloc_long_deg::FLOAT END)," +
					"(CASE WHEN TRIM(app_location.aloc_long_mm) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_location.aloc_long_mm::FLOAT END)," +
					"(CASE WHEN TRIM(app_location.aloc_long_ss) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_location.aloc_long_ss::FLOAT END)," +
					"(CASE WHEN TRIM(app_antenna.aant_horiz_rc_amsl) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_antenna.aant_rc_amsl::FLOAT END)," +
					"(CASE WHEN TRIM(app_antenna.aant_vert_rc_amsl) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_antenna.aant_vert_rc_amsl::FLOAT END)," +
					"(CASE WHEN TRIM(app_antenna.aant_rc_haat) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_antenna.aant_rc_haat::FLOAT END)," +
					"(CASE WHEN TRIM(app_antenna.aant_vert_rc_haat) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_antenna.aant_vert_rc_haat::FLOAT END)," +
					"(CASE WHEN TRIM(app_antenna_frequency.aafq_horiz_max_erp_kw) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_antenna_frequency.aafq_horiz_max_erp_kw::FLOAT END)," +
					"(CASE WHEN TRIM(app_antenna_frequency.aafq_horiz_erp_kw) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_antenna_frequency.aafq_horiz_erp_kw::FLOAT END)," +
					"(CASE WHEN TRIM(app_antenna_frequency.aafq_vert_max_erp_kw) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_antenna_frequency.aafq_vert_max_erp_kw::FLOAT END)," +
					"(CASE WHEN TRIM(app_antenna_frequency.aafq_vert_erp_kw) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_antenna_frequency.aafq_vert_erp_kw::FLOAT END)," +
					"app_antenna.aant_antenna_record_id," +
					"(CASE WHEN TRIM(app_antenna.aant_rotation_deg) IN ('','null') THEN 0::FLOAT " +
						"ELSE app_antenna.aant_rotation_deg::FLOAT END)," +
					"license_filing_version.active_ind," +
					"app_antenna.aant_antenna_type_code," +
					"license_filing_version.service_code," +
					"application_facility.country_code," +
					"application.aapp_receipt_date AS sequence_date," +
					"application_facility.afac_facility_type," +
					"facility.facility_status," +
					"license_filing_version.auth_type_code," +
					"app_antenna.aant_antenna_id," +
					"facility_applicant.legal_name," +
					"app_mm_application.contour_215_protection_ind";

				frmStr =
				" FROM " +
					"mass_media.app_location " +
					"LEFT JOIN mass_media.app_antenna ON (app_antenna.aant_aloc_loc_record_id = " +
						"app_location.aloc_loc_record_id) " +
					"LEFT JOIN mass_media.app_antenna_frequency ON " +
						"(app_antenna_frequency.aafq_aant_antenna_record_id = app_antenna.aant_antenna_record_id) " +
					"JOIN common_schema.application ON (application.aapp_application_id = " +
						"app_location.aloc_aapp_application_id) " +
					"JOIN common_schema.license_filing_version ON (license_filing_version.filing_version_id = " +
						"app_location.aloc_aapp_application_id) " +
					"JOIN common_schema.application_facility ON (application_facility.afac_application_id = " +
						"app_location.aloc_aapp_application_id) " +
					"JOIN common_schema.facility ON (facility.facility_id = application_facility.afac_facility_id) " +
					"LEFT JOIN common_schema.facility_applicant ON " +
						"((facility_applicant.facility_uuid = facility.facility_uuid) " +
						"AND (facility_applicant.contact_ind = 'N')) " +
					"LEFT JOIN mass_media.app_mm_application ON (app_mm_application.aapp_application_id = " +
						"app_location.aloc_aapp_application_id)";

			} else {

				fldStr =
				"SELECT DISTINCT " +
					"app_location.aloc_aapp_application_id," +
					"application_facility.afac_facility_id," +
					"application_facility.station_class_code," +
					"application_facility.afac_channel," +
					"facility.callsign," +
					"(CASE WHEN (facility.community_served_city <> '') " +
						"THEN facility.community_served_city " +
						"ELSE application_facility.afac_community_city END)," +
					"(CASE WHEN (facility.community_served_state <> '') " +
						"THEN facility.community_served_state " +
						"ELSE application_facility.afac_community_state_code END)," +
					"license_filing_version.current_status_code," +
					"'BLANK' AS file_prefix," +
					"application.aapp_file_num," +
					"app_location.aloc_lat_dir," +
					"app_location.aloc_lat_deg," +
					"app_location.aloc_lat_mm," +
					"app_location.aloc_lat_ss," +
					"app_location.aloc_long_dir," +
					"app_location.aloc_long_deg," +
					"app_location.aloc_long_mm," +
					"app_location.aloc_long_ss," +
					"app_antenna.aant_rc_amsl," +
					"app_antenna.aant_vert_rc_amsl," +
					"app_antenna.aant_rc_haat," +
					"app_antenna.aant_vert_rc_haat," +
					"app_antenna_frequency.aafq_horiz_max_erp_kw," +
					"app_antenna_frequency.aafq_horiz_erp_kw," +
					"app_antenna_frequency.aafq_vert_max_erp_kw," +
					"app_antenna_frequency.aafq_vert_erp_kw," +
					"app_antenna.aant_antenna_record_id," +
					"app_antenna.aant_rotation_deg," +
					"license_filing_version.active_ind," +
					"app_antenna.aant_antenna_type_code," +
					"license_filing_version.service_code," +
					"application_facility.country_code," +
					"application.aapp_receipt_date AS sequence_date," +
					"application_facility.afac_facility_type," +
					"facility.facility_status," +
					"license_filing_version.auth_type_code," +
					"app_antenna.aant_antenna_id";

				frmStr =
				" FROM " +
					"app_location " +
					"LEFT JOIN app_antenna ON (app_antenna.aant_aloc_loc_record_id = " +
						"app_location.aloc_loc_record_id) " +
					"LEFT JOIN app_antenna_frequency ON (app_antenna_frequency.aafq_aant_antenna_record_id = " +
						"app_antenna.aant_antenna_record_id) " +
					"JOIN application ON (application.aapp_application_id = app_location.aloc_aapp_application_id) " +
					"JOIN license_filing_version ON (license_filing_version.filing_version_id = " +
						"app_location.aloc_aapp_application_id) " +
					"JOIN application_facility ON (application_facility.afac_application_id = " +
						"app_location.aloc_aapp_application_id) " +
					"JOIN facility ON (facility.facility_id = application_facility.afac_facility_id)";

				if (extDb.version > 7) {
					fldStr = fldStr + ", facility_applicant.legal_name";
					frmStr = frmStr +
					" LEFT JOIN facility_applicant ON " +
						"((facility_applicant.facility_uuid = facility.facility_uuid) " +
						"AND (facility_applicant.contact_ind = 'N'))";
				} else {
					fldStr = fldStr + ", '' AS legal_name";
				}

				if (extDb.version > 8) {
					fldStr = fldStr + ", app_mm_application.contour_215_protection_ind";
					frmStr = frmStr +
					" LEFT JOIN app_mm_application ON (app_mm_application.aapp_application_id = " +
						"app_location.aloc_aapp_application_id)";
				} else {
					fldStr = fldStr + ", 'N' AS contour_215_protection_ind";
				}
			}

			whrStr.append(" WHERE ");
			if ((null != query) && (query.length() > 0)) {
				whrStr.append('(');
				whrStr.append(query);
				whrStr.append(") AND ");
			}
			whrStr.append("((CASE WHEN (license_filing_version.purpose_code = 'AMD') ");
			whrStr.append("THEN license_filing_version.original_purpose_code ");
			whrStr.append("ELSE license_filing_version.purpose_code END) IN ('CP','L2C','MOD','RUL','STA')) AND ");
			whrStr.append("(license_filing_version.current_status_code <> 'SAV') AND ");
			whrStr.append("(license_filing_version.service_code IN ");
			addServiceCodeList(extDb.type, Source.RECORD_TYPE_FM, whrStr);
			whrStr.append(')');
		}

		LinkedList<ExtDbRecordFM> result = null;

		DbConnection db = extDb.connectDb(errors);
		if (null != db) {

			SimpleDateFormat dateFormat;
			if (isCDBS) {
				dateFormat = new SimpleDateFormat("MM/dd/yyyy", Locale.US);
			} else {
				dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.US);
			}

			// IBOC digital power levels in LMS are in a separate set of records in the digital_notification table,
			// and the presence of a record there for a given facility ID is the only indication of IBOC operation.
			// The facility ID is the only link so pulling those records in to the main query by join is impractical,
			// so the entire set is loaded and cached on the first call for each data set.  The notification table is
			// only in LMS import version 11 or later.

			HashMap<Integer, Double> ibocMap = null;
			String str = null;
			java.util.Date seqDate = null;
			Double ibocFrac = null;

			if (!isCDBS && (extDb.version > 10)) {

				ibocMap = ibocMaps.get(extDb.key);

				if (null == ibocMap) {

					ibocMap = new HashMap<Integer, Double>();
					ibocMaps.put(extDb.key, ibocMap);

					Integer facID;
					HashMap<Integer, java.util.Date> dateMap = new HashMap<Integer, java.util.Date>();
					java.util.Date otherDate;
					double analogERP, digitalERP;

					try {

						db.query(
						"SELECT " +
							"application_facility.afac_facility_id, " +
							"application.aapp_receipt_date, " +
							"digital_notification.analog_erp_kw, " +
							"digital_notification.digital_erp_kw " +
						"FROM " +
							"digital_notification " +
							"JOIN application ON (application.aapp_application_id = " +
								"digital_notification.application_id) " +
							"JOIN application_facility ON (application_facility.afac_application_id = " +
								"digital_notification.application_id)");

						while (db.next()) {

							facID = Integer.valueOf(db.getInt(1));

							seqDate = null;
							str = db.getString(2);
							if (null != str) {
								try {
									seqDate = dateFormat.parse(str);
								} catch (ParseException pe) {
								}
							}
							if (null == seqDate) {
								seqDate = new java.util.Date(0);
							}
							otherDate = dateMap.get(facID);
							if ((otherDate != null) && otherDate.after(seqDate)) {
								continue;
							}

							analogERP = -1.;
							str = db.getString(3);
							if (null != str) {
								try {
									analogERP = Double.parseDouble(str);
								} catch (NumberFormatException ne) {
								}
							}
							if (analogERP <= 0.) {
								continue;
							}

							digitalERP = -1.;
							str = db.getString(4);
							if (null != str) {
								try {
									digitalERP = Double.parseDouble(str);
								} catch (NumberFormatException ne) {
								}
							}
							if (digitalERP <= 0.) {
								continue;
							}

							ibocFrac = Double.valueOf(digitalERP / analogERP);

							ibocMap.put(facID, ibocFrac);
							dateMap.put(facID, seqDate);
						}

					} catch (SQLException se) {
						db.reportError(se);
					}
				}
			}

			// Run the query.

			try {

				result = new LinkedList<ExtDbRecordFM>();
				ExtDbRecordFM theRecord;

				Service theService;
				Country theCountry;
				int appID, chan, i, antID, statType;
				double d1, d2, d3, d4;
				String recID, dir, stat, pfx, arn, facStat;
				GeoPoint thePoint = new GeoPoint();

				db.query(fldStr + frmStr + whrStr.toString());

				while (db.next()) {

					theService = getService(extDb.type, db.getString(31));
					if (null == theService) {
						continue;
					}

					theCountry = Country.getCountry(db.getString(32));
					if (null == theCountry) {
						continue;
					}

					if (isCDBS) {
						appID = db.getInt(1);
						if (appID <= 0) {
							continue;
						}
						recID = String.valueOf(appID);
					} else {
						recID = db.getString(1);
						if ((null == recID) || (0 == recID.length())) {
							continue;
						}
					}

					chan = db.getInt(4);
					if ((chan < SourceFM.CHANNEL_MIN) || (chan > SourceFM.CHANNEL_MAX)) {
						continue;
					}

					// Extract sequence date, see isPreferredRecord().

					seqDate = null;
					try {
						str = db.getString(33);
						if (null != str) {
							seqDate = dateFormat.parse(str);
						}
					} catch (ParseException pe) {
					}
					if (null == seqDate) {
						seqDate = new java.util.Date(0);
					}

					// Facility status may affect other properties.

					facStat = db.getString(35);
					if (null == facStat) {
						facStat = "";
					}

					// A facility status of "EXPER" is experimental regardless of anything else.

					stat = db.getString(8);
					if (null == stat) {
						stat = "";
					}
					statType = STATUS_TYPE_OTHER;

					if (facStat.equals("EXPER")) {
						statType = STATUS_TYPE_EXP;
					} else {

						if (isCDBS) {
							for (i = 0; i < CDBS_STATUS_CODES.length; i++) {
								if (CDBS_STATUS_CODES[i].equalsIgnoreCase(stat)) {
									statType = CDBS_STATUS_TYPES[i];
									break;
								}
							}
						} else {
							String type = db.getString(36);
							if (type.equalsIgnoreCase("S")) {
								statType = STATUS_TYPE_STA;
							} else {
								if (stat.equalsIgnoreCase("REV")) {
									statType = STATUS_TYPE_AMD;
								} else {
									if (stat.equalsIgnoreCase("PEN") && type.equalsIgnoreCase("C")) {
										statType = STATUS_TYPE_APP;
									} else {
										for (i = 0; i < LMS_APP_STATUS_CODES.length; i++) {
											if (LMS_APP_STATUS_CODES[i].equalsIgnoreCase(stat)) {
												statType = STATUS_TYPE_APP;
												break;
											}
										}
										if (STATUS_TYPE_OTHER == statType) {
											if (type.equalsIgnoreCase("C")) {
												statType = STATUS_TYPE_CP;
											} else {
												if (type.equalsIgnoreCase("L")) {
													statType = STATUS_TYPE_LIC;
												}
											}
										}
									}
								}
							}
						}
					}

					if (STATUS_TYPE_OTHER != statType) {
						stat = STATUS_CODES[statType];
					}

					// Extract coordinates first, if a search radius is set, check that.

					thePoint.latitudeNS = 0;
					dir = db.getString(11);
					if ((null != dir) && dir.equalsIgnoreCase("S")) {
						thePoint.latitudeNS = 1;
					}
					thePoint.latitudeDegrees = db.getInt(12);
					thePoint.latitudeMinutes = db.getInt(13);
					thePoint.latitudeSeconds = db.getDouble(14);

					thePoint.longitudeWE = 0;
					dir = db.getString(15);
					if ((null != dir) && dir.equalsIgnoreCase("E")) {
						thePoint.longitudeWE = 1;
					}
					thePoint.longitudeDegrees = db.getInt(16);
					thePoint.longitudeMinutes = db.getInt(17);
					thePoint.longitudeSeconds = db.getDouble(18);

					thePoint.updateLatLon();
					if (isCDBS) {
						thePoint.convertFromNAD27();
						thePoint.updateDMS();
					}

					if ((null != searchCenter) && (searchRadius > 0.) &&
							(searchCenter.distanceTo(thePoint, kmPerDegree) > searchRadius)) {
						continue;
					}

					// Save the record in the results.

					theRecord = new ExtDbRecordFM(extDb);

					theRecord.service = theService;
					theRecord.country = theCountry;
					theRecord.extRecordID = recID;

					theRecord.facilityID = db.getInt(2);
					theRecord.channel = chan;

					theRecord.stationClass = FM_CLASS_OTHER;
					str = db.getString(3);
					if ((null != str) && (str.length() > 0)) {
						for (i = 1; i < FM_CLASS_CODES.length; i++) {
							if (str.equalsIgnoreCase(FM_CLASS_CODES[i])) {
								theRecord.stationClass = i;
								break;
							}
						}
					}

					str = db.getString(5);
					if (null == str) {
						str = "";
					} else {
						if (str.length() > Source.MAX_CALL_SIGN_LENGTH) {
							str = str.substring(0, Source.MAX_CALL_SIGN_LENGTH);
						}
					}
					theRecord.callSign = str;

					str = db.getString(6);
					if (null == str) {
						str = "";
					} else {
						if (str.length() > Source.MAX_CITY_LENGTH) {
							str = str.substring(0, Source.MAX_CITY_LENGTH);
						}
					}
					theRecord.city = str;
					str = db.getString(7);
					if (null == str) {
						str = "";
					} else {
						if (str.length() > Source.MAX_STATE_LENGTH) {
							str = str.substring(0, Source.MAX_STATE_LENGTH);
						}
					}
					theRecord.state = str;

					if (stat.length() > Source.MAX_STATUS_LENGTH) {
						stat = stat.substring(0, Source.MAX_STATUS_LENGTH);
					}
					theRecord.status = stat;
					theRecord.statusType = statType;

					pfx = db.getString(9);
					arn = db.getString(10);
					if (!isCDBS && (null != arn)) {
						String[] parts = arn.split("-");
						if (2 == parts.length) {
							pfx = parts[0];
							arn = parts[1];
						}
					}
					if (null == arn) {
						arn = "";
					} else {
						if (arn.length() > Source.MAX_FILE_NUMBER_LENGTH) {
							arn = arn.substring(0, Source.MAX_FILE_NUMBER_LENGTH);
						}
					}
					if (null == pfx) {
						pfx = "";
					} else {
						if ((pfx.length() + arn.length()) > Source.MAX_FILE_NUMBER_LENGTH) {
							pfx = pfx.substring(0, Source.MAX_FILE_NUMBER_LENGTH - arn.length());
						}
					}
					theRecord.filePrefix = pfx;
					theRecord.appARN = arn;

					theRecord.location.setLatLon(thePoint);

					d1 = db.getDouble(19);
					d2 = db.getDouble(20);
					theRecord.heightAMSL = (d1 > d2 ? d1 : d2);
					d1 = db.getDouble(21);
					d2 = db.getDouble(22);
					theRecord.overallHAAT = (d1 > d2 ? d1 : d2);

					d1 = db.getDouble(23);
					d2 = db.getDouble(24);
					d3 = (d1 > d2 ? d1 : d2);
					d1 = db.getDouble(25);
					d2 = db.getDouble(26);
					d4 = (d1 > d2 ? d1 : d2);
					theRecord.peakERP = (d3 > d4 ? d3 : d4);

					if (isCDBS) {
						antID = db.getInt(27);
						if (antID > 0) {
							theRecord.antennaRecordID = String.valueOf(antID);
							theRecord.antennaID = theRecord.antennaRecordID;
						}
					} else {
						str = db.getString(27);
						if (null != str) {
							str = str.trim();
							if (0 == str.length()) {
								str = null;
							}
						}
						theRecord.antennaRecordID = str;
						str = db.getString(37);
						if (null != str) {
							str = str.trim();
							if (0 == str.length()) {
								str = null;
							}
						}
						theRecord.antennaID = str;
					}
					theRecord.horizontalPatternOrientation = Math.IEEEremainder(db.getDouble(28), 360.);
					if (theRecord.horizontalPatternOrientation < 0.) theRecord.horizontalPatternOrientation += 360.;

					// If the facility status is FVOID, PRCAN, or LICAN, treat this as an archived record regardless
					// of other fields.

					if ((facStat.equals("FVOID") || facStat.equals("PRCAN") || facStat.equals("LICAN"))) {
						theRecord.isArchived = true;
					} else {
						str = db.getString(29);
						if (null != str) {
							if (isCDBS) {
								if (str.equalsIgnoreCase("P")) {
									theRecord.isPending = true;
								} else {
									if (str.equalsIgnoreCase("A")) {
										theRecord.isArchived = true;
									}
								}
							} else {
								if (str.equalsIgnoreCase("N")) {
									theRecord.isArchived = true;
								} else {
									str = db.getString(8);
									if ((null != str) && str.equalsIgnoreCase("PEN")) {
										theRecord.isPending = true;
									}
								}
							}
						}
					}

					str = db.getString(30);
					if (null != str) {
						if (isCDBS) {
							theRecord.daIndicated = str.equalsIgnoreCase("Y");
						} else {
							theRecord.daIndicated = str.equalsIgnoreCase("DIR");
						}
					}

					theRecord.sequenceDate = seqDate;

					// For CDBS, a flag from the query definitively indicates if the station is operating IBOC so that
					// can be determined here, then the actual power level will be queried later from a separate table
					// of notification records, see updateSource().  For LMS however, the existence of the separate
					// notification record is what indicates an IBOC operation, the flag from the main query appears to
					// no longer be in use.  The set of notification records was loaded and cached earlier, see above.

					theRecord.ibocFraction = SourceFM.IBOC_FRACTION_DEF;

					if (isCDBS) {

						str = db.getString(34);
						if (null != str) {
							theRecord.isIBOC = str.equalsIgnoreCase("H");
						}

					} else {

						if (null != ibocMap) {
							ibocFrac = ibocMap.get(Integer.valueOf(theRecord.facilityID));
							if (null != ibocFrac) {
								theRecord.isIBOC = true;
								theRecord.ibocFraction = ibocFrac.doubleValue();
							}
						}
					}

					if (!isCDBS) {

						theRecord.licensee = db.getString(38);

						str = db.getString(39);
						theRecord.contour73215 = ((null != str) && str.equalsIgnoreCase("Y"));
					}

					result.add(theRecord);
				}

				extDb.releaseDb(db);

			} catch (SQLException se) {
				extDb.releaseDb(db);
				result = null;
				DbConnection.reportError(errors, se);
			}
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Convenience to do a search for a specific record by record ID.

	public static ExtDbRecordFM findRecordFM(String theDbID, Integer extDbKey, String extRecordID) {
		return findRecordFM(theDbID, extDbKey, extRecordID, null);
	}

	public static ExtDbRecordFM findRecordFM(String theDbID, Integer extDbKey, String extRecordID,
			ErrorLogger errors) {
		ExtDb extDb = ExtDb.getExtDb(theDbID, extDbKey, errors);
		if (null == extDb) {
			return null;
		}
		return findRecordFM(extDb, extRecordID, errors);
	}

	public static ExtDbRecordFM findRecordFM(ExtDb extDb, String extRecordID) {
		return findRecordFM(extDb, extRecordID, null);
	}

	public static ExtDbRecordFM findRecordFM(ExtDb extDb, String extRecordID, ErrorLogger errors) {

		if (!isExtDbSupported(extDb)) {
			return null;
		}

		StringBuilder query = new StringBuilder();
		try {
			addRecordIDQueryFM(extDb.type, extDb.version, extRecordID, query, false);
		} catch (IllegalArgumentException ie) {
			if (null != errors) {
				errors.logMessage(ie.getMessage());
			}
			return null;
		}

		LinkedList<ExtDbRecordFM> theRecs = findRecordsFM(extDb, query.toString(), null, 0., 0., errors);

		if (null == theRecs) {
			return null;
		}
		if (theRecs.isEmpty()) {
			if (null != errors) {
				errors.logMessage("Record not found for record ID '" + extRecordID + "'");
			}
			return null;
		}

		return theRecs.getFirst();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Batch search for individual records by ID, used to efficiently find multiple records since individual use of
	// findRecordFM() may have very poor performance.

	public static HashMap<String, ExtDbRecordFM> batchFindRecordFM(ExtDb extDb, HashSet<String> theExtRecordIDs) {
		return batchFindRecordFM(extDb, theExtRecordIDs, null);
	}

	public static HashMap<String, ExtDbRecordFM> batchFindRecordFM(ExtDb extDb, HashSet<String> theExtRecordIDs,
			ErrorLogger errors) {

		HashMap<String, ExtDbRecordFM> result = new HashMap<String, ExtDbRecordFM>();

		if (!isExtDbSupported(extDb)) {
			return result;
		}

		boolean isCDBS = (ExtDb.DB_TYPE_CDBS_FM == extDb.type);

		boolean doSearch = false;
		StringBuilder query = new StringBuilder();
		String sep = "(";
		if (isCDBS) {
			query.append("fm_eng_data.application_id IN ");
		} else {
			query.append("UPPER(app_location.aloc_aapp_application_id) IN ");
			sep = "('";
		}

		int applicationID;

		for (String theID : theExtRecordIDs) {

			if (isCDBS) {
				applicationID = 0;
				try {
					applicationID = Integer.parseInt(theID);
				} catch (NumberFormatException ne) {
				}
				if (applicationID <= 0) {
					continue;
				}
				theID = String.valueOf(applicationID);
			} else {
				theID = DbConnection.clean(theID.toUpperCase());
			}

			query.append(sep);
			query.append(theID);
			if (isCDBS) {
				sep = ",";
			} else {
				sep = "','";
			}

			doSearch = true;
		}

		if (doSearch) {

			if (isCDBS) {
				query.append(")");
			} else {
				query.append("')");
			}

			LinkedList<ExtDbRecordFM> theRecs = findRecordsFM(extDb, query.toString(), null, 0., 0., errors);
			if (null == theRecs) {
				return null;
			}

			for (ExtDbRecordFM theRec : theRecs) {
				result.put(theRec.extRecordID, theRec);
			}
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public static boolean doesRecordIDExistFM(ExtDb extDb, String extRecordID) {

		if (!isExtDbSupported(extDb)) {
			return false;
		}

		StringBuilder query = new StringBuilder();
		try {
			addRecordIDQueryFM(extDb.type, extDb.version, extRecordID, query, false);
		} catch (IllegalArgumentException ie) {
			return false;
		}

		DbConnection db = extDb.connectDb();
		if (null == db) {
			return false;
		}

		boolean result = false;

		try {

			if (ExtDb.DB_TYPE_CDBS_FM == extDb.type) {
				db.query("SELECT COUNT(*) FROM fm_eng_data WHERE " + query.toString());
			} else {
				if (ExtDb.DB_TYPE_LMS_LIVE == extDb.type) {
					db.query("SELECT COUNT(*) FROM mass_media.app_location WHERE " + query.toString());
				} else {
					db.query("SELECT COUNT(*) FROM app_location WHERE " + query.toString());
				}
			}

			if (db.next()) {
				result = (db.getInt(1) > 0);
			}

		} catch (SQLException se) {
			db.reportError(se);
		}

		extDb.releaseDb(db);

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Methods to support composing query clauses for findRecords() searching, see superclass for details.  The first
	// method adds a record ID search.  In CDBS the argument is validated as a number > 0 for application ID match.

	public static boolean addRecordIDQueryFM(int dbType, int version, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if ((null == str) || (0 == str.length())) {
				return false;
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {
				int theID = 0;
				try {
					theID = Integer.parseInt(str);
				} catch (NumberFormatException e) {
					throw new IllegalArgumentException("The record ID must be a number");
				}
				if (theID <= 0) {
					throw new IllegalArgumentException("The record ID must be greater than 0");
				}
			}

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				query.append("(fm_eng_data.application_id = ");
				query.append(str);
				query.append(')');

			} else {

				query.append("(UPPER(app_location.aloc_aapp_application_id) = '");
				query.append(DbConnection.clean(str.toUpperCase()));
				query.append("')");
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Search by file number.  See superclass for details.

	public static boolean addFileNumberQueryFM(int dbType, int version, String prefix, String arn, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				if (prefix.length() > 0) {
					query.append("((UPPER(application.file_prefix) = '");
					query.append(DbConnection.clean(prefix.toUpperCase()));
					query.append("') AND (UPPER(application.app_arn) = '");
					query.append(DbConnection.clean(arn.toUpperCase()));
					query.append("'))");
				} else {
					query.append("(UPPER(application.app_arn) = '");
					query.append(DbConnection.clean(arn.toUpperCase()));
					query.append("')");
				}

			} else {

				if (ExtDb.DB_TYPE_LMS_LIVE == dbType) {

					if (prefix.length() > 0) {
						query.append("(UPPER(application.aapp_file_num) = '");
						query.append(DbConnection.clean(prefix.toUpperCase()));
						query.append('-');
						query.append(DbConnection.clean(arn.toUpperCase()));
					} else {
						query.append("(UPPER(application.aapp_file_num) LIKE '%");
						query.append(DbConnection.clean(arn.toUpperCase()));
					}
					query.append("')");

				} else {

					if (prefix.length() > 0) {
						query.append("(application._search_filenum = '");
						query.append(DbConnection.clean(prefix.toUpperCase()));
						query.append('-');
						query.append(DbConnection.clean(arn.toUpperCase()));
					} else {
						query.append("(application._search_arn = '");
						query.append(DbConnection.clean(arn.toUpperCase()));
					}
					query.append("')");
				}
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query clause to search by facility ID.  The version that takes a string argument is in the superclass.

	public static boolean addFacilityIDQueryFM(int dbType, int version, int facilityID, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {
				query.append("(fm_eng_data.facility_id = ");
			} else {
				query.append("(application_facility.afac_facility_id = ");
			}

			query.append(String.valueOf(facilityID));
			query.append(')');

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query clause to search by service.

	public static boolean addServiceQueryFM(int dbType, int version, int serviceKey, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {
				query.append("(fm_eng_data.asd_service IN ");
			} else {
				query.append("(license_filing_version.service_code IN ");
			}
			addServiceCodeList(dbType, Source.RECORD_TYPE_FM, serviceKey, query);
			query.append(')');

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query clause to search by services matching isOperating flag.

	public static boolean addServiceTypeQueryFM(int dbType, int version, int operatingMatch, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {
				query.append("(fm_eng_data.asd_service IN ");
			} else {
				query.append("(license_filing_version.service_code IN ");
			}
			addServiceCodeList(dbType, Source.RECORD_TYPE_FM, operatingMatch, FLAG_MATCH_ANY, query);
			query.append(')');

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for call sign, case-insensitive head-anchored match.

	public static boolean addCallSignQueryFM(int dbType, int version, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if ((null == str) || (0 == str.length())) {
				return false;
			}

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				query.append("(UPPER(CASE WHEN (facility.fac_callsign <> '') " +
					"THEN facility.fac_callsign ELSE application.fac_callsign END) REGEXP '^D*");
				query.append(DbConnection.clean(str.toUpperCase()));
				query.append(".*')");

			} else {

				if (ExtDb.DB_TYPE_LMS_LIVE == dbType) {

					query.append("(UPPER(facility.callsign) ~ '^D*");
					query.append(DbConnection.clean(str.toUpperCase()));
					query.append(".*')");

				} else {

					query.append("(facility._search_callsign LIKE '");
					query.append(DbConnection.clean(str.toUpperCase()));
					query.append("%')");
				}
			}


			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for a specific channel search.  See superclass for argument-checking forms.

	public static boolean addChannelQueryFM(int dbType, int version, int channel, int minimumChannel,
			int maximumChannel, StringBuilder query, boolean combine) throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if ((minimumChannel > 0) && (maximumChannel > 0) &&
					((channel < minimumChannel) || (channel > maximumChannel))) {
				throw new IllegalArgumentException("The channel must be in the range " + minimumChannel + " to " +
					maximumChannel);
			}

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				query.append("(fm_eng_data.station_channel = ");
				query.append(String.valueOf(channel));
				query.append(')');

			} else {

				if (ExtDb.DB_TYPE_LMS_LIVE == dbType) {
					query.append("(CASE WHEN TRIM(application_facility.afac_channel) IN ('','null') THEN false " +
						"ELSE (application_facility.afac_channel::INT = ");
					query.append(String.valueOf(channel));
					query.append(") END)");
				} else {
					query.append("(application_facility.afac_channel = ");
					query.append(String.valueOf(channel));
					query.append(')');
				}
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for a channel range search.  The range arguments are assumed valid.

	public static boolean addChannelRangeQueryFM(int dbType, int version, int minimumChannel, int maximumChannel,
			StringBuilder query, boolean combine) throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				query.append("(fm_eng_data.station_channel BETWEEN ");
				query.append(String.valueOf(minimumChannel));
				query.append(" AND ");
				query.append(String.valueOf(maximumChannel));
				query.append(')');

			} else {

				if (ExtDb.DB_TYPE_LMS_LIVE == dbType) {
					query.append("(CASE WHEN TRIM(application_facility.afac_channel) IN ('','null') THEN false " +
						"ELSE (application_facility.afac_channel::INT BETWEEN ");
					query.append(String.valueOf(minimumChannel));
					query.append(" AND ");
					query.append(String.valueOf(maximumChannel));
					query.append(") END)");
				} else {
					query.append("(application_facility.afac_channel BETWEEN ");
					query.append(String.valueOf(minimumChannel));
					query.append(" AND ");
					query.append(String.valueOf(maximumChannel));
					query.append(')');
				}
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for matching a list of channels.  This is not meant to process direct user input, the
	// argument string is assumed to be a valid SQL value list composed by other code e.g. "(7,8,12,33)", containing
	// valid channel numbers.

	public static boolean addMultipleChannelQueryFM(int dbType, int version, String str, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				query.append("(fm_eng_data.station_channel IN ");
				query.append(str);
				query.append(')');

			} else {

				if (ExtDb.DB_TYPE_LMS_LIVE == dbType) {
					query.append("(CASE WHEN TRIM(application_facility.afac_channel) IN ('','null') THEN false " +
						"ELSE (application_facility.afac_channel::INT IN ");
					query.append(str);
					query.append(") END)");
				} else {
					query.append("(application_facility.afac_channel IN ");
					query.append(str);
					query.append(')');
				}
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add a query search clause for status.

	public static boolean addStatusQueryFM(int dbType, int version, int statusType, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			switch (statusType) {

				case STATUS_TYPE_STA:
				case STATUS_TYPE_CP:
				case STATUS_TYPE_LIC:
				case STATUS_TYPE_APP:
				case STATUS_TYPE_EXP:
				case STATUS_TYPE_AMD: {
					break;
				}

				default: {
					throw new IllegalArgumentException("Unknown status code");
				}
			}

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				if (STATUS_TYPE_EXP == statusType) {

					query.append("(facility.fac_status = 'EXPER')");

				} else {

					if (STATUS_TYPE_STA == statusType) {

						query.append("(UPPER(application.app_type) IN ('STA', 'STAX'))");

					} else {

						if (STATUS_TYPE_AMD == statusType) {

							query.append("(false)");

						} else {

							query.append("(UPPER(fm_eng_data.fm_dom_status) IN ");
							String s = "('";

							for (int i = 0; i < CDBS_STATUS_TYPES.length; i++) {
								if (CDBS_STATUS_TYPES[i] == statusType) {
									query.append(s);
									query.append(CDBS_STATUS_CODES[i]);
									s = "','";
								}
							}

							query.append("'))");
						}
					}
				}

			} else {

				if (STATUS_TYPE_EXP == statusType) {

					query.append("(facility.facility_status = 'EXPER')");

				} else {

					if (STATUS_TYPE_STA == statusType) {

						query.append("((facility.facility_status <> 'EXPER') AND ");
						query.append("(UPPER(license_filing_version.auth_type_code) = 'S'))");

					} else {

						query.append("((facility.facility_status <> 'EXPER') AND ");
						query.append("(UPPER(license_filing_version.auth_type_code) <> 'S') AND ");

						if (STATUS_TYPE_AMD == statusType) {

							query.append("(UPPER(license_filing_version.current_status_code = 'REV')))");

						} else {

							if (STATUS_TYPE_APP == statusType) {
								query.append("(((UPPER(license_filing_version.auth_type_code) = 'C') AND ");
								query.append("(UPPER(license_filing_version.current_status_code) = 'PEN')) OR ");
								query.append("(UPPER(license_filing_version.current_status_code) IN ");
							} else {
								query.append("(((UPPER(license_filing_version.auth_type_code) <> 'C') OR ");
								query.append("(UPPER(license_filing_version.current_status_code) <> 'PEN')) AND ");
								query.append("(UPPER(license_filing_version.current_status_code) NOT IN ");
							}
							String s = "('";
							for (int i = 0; i < LMS_APP_STATUS_CODES.length; i++) {
								query.append(s);
								query.append(LMS_APP_STATUS_CODES[i]);
								s = "','";
							}
							query.append("')))");

							if (STATUS_TYPE_APP == statusType) {
								query.append(')');
							} else {
								if (STATUS_TYPE_CP == statusType) {
									query.append(" AND (UPPER(license_filing_version.auth_type_code) = 'C'))");
								} else {
									if (STATUS_TYPE_LIC == statusType) {
										query.append(" AND (UPPER(license_filing_version.auth_type_code) = 'L'))");
									}
								}
							}
						}
					}
				}
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add query clause for city name search, case-insensitive unanchored match, '*' wildcards are allowed.

	public static boolean addCityQueryFM(int dbType, int version, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if ((null == str) || (0 == str.length())) {
				return false;
			}

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				query.append("(UPPER(CASE WHEN (facility.comm_city <> '') " +
					"THEN facility.comm_city ELSE application.comm_city END) LIKE '");
				query.append(DbConnection.clean(str.toUpperCase()).replace('*', '%'));
				query.append("%')");

			} else {

				if (ExtDb.DB_TYPE_LMS_LIVE == dbType) {

					query.append("(UPPER(CASE WHEN (facility.community_served_city <> '') " +
						"THEN facility.community_served_city " +
						"ELSE application_facility.afac_community_city END) LIKE '");
					query.append(DbConnection.clean(str.toUpperCase()).replace('*', '%'));
					query.append("%')");

				} else {

					query.append("(facility._search_city LIKE '");
					query.append(DbConnection.clean(str.toUpperCase()).replace('*', '%'));
					query.append("%')");
				}
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add query clause for state code search, this is an exact string match but still case-insensitive.

	public static boolean addStateQueryFM(int dbType, int version, String str, StringBuilder query, boolean combine)
			throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if ((null == str) || (0 == str.length())) {
				return false;
			}

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				query.append("(UPPER(CASE WHEN (facility.comm_state <> '') " +
					"THEN facility.comm_state ELSE application.comm_state END) = '");
				query.append(DbConnection.clean(str.toUpperCase()));
				query.append("')");

			} else {

				query.append("(UPPER(CASE WHEN (facility.community_served_state <> '') " +
					"THEN facility.community_served_state " +
					"ELSE application_facility.afac_community_state_code END) = '");
				query.append(DbConnection.clean(str.toUpperCase()));
				query.append("')");
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add clause for country search.  See superclass for forms that resolve string and key arguments.

	public static boolean addCountryQueryFM(int dbType, int version, Country country, StringBuilder query,
			boolean combine) throws IllegalArgumentException {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				query.append("(UPPER(facility.fac_country) = '");
				query.append(country.countryCode);
				query.append("')");

			} else {

				query.append("(UPPER(application_facility.country_code) = '");
				query.append(country.countryCode);
				query.append("')");
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add record type search query clause.  Current records are always included, archived records may optionally be
	// included.  Any unknown record types are treated as current, so this excludes rather than includes, meaning
	// if includeArchived is true it does nothing.

	public static boolean addRecordTypeQueryFM(int dbType, int version, boolean includeArchived, StringBuilder query,
			boolean combine) {

		if ((ExtDb.DB_TYPE_CDBS_FM == dbType) || (ExtDb.DB_TYPE_LMS == dbType) || (ExtDb.DB_TYPE_LMS_LIVE == dbType)) {

			if (includeArchived) {
				return false;
			}

			if (combine) {
				query.append(" AND ");
			}

			if (ExtDb.DB_TYPE_CDBS_FM == dbType) {

				query.append("((fm_eng_data.eng_record_type <> 'A') AND " +
					"(facility.fac_status NOT IN ('FVOID', 'PRCAN', 'LICAN')))");

			} else {

				query.append("((license_filing_version.active_ind = 'Y') AND ");
				query.append("(facility.facility_status <> 'FVOID')");
				query.append(')');
			}

			return true;
		}

		throw new IllegalArgumentException(BAD_TYPE_MESSAGE);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add records as desired sources to the scenario using a station data search.  See comments in ExtDbRecordTV for
	// details; the concepts are similar here.

	private static class SearchDelta {
		int delta;
		double maximumDistance;
	}

	public static int addRecords(ExtDb extDb, ScenarioEditData scenario, int searchType, String query,
			GeoPoint searchCenter, double searchRadius, int minimumChannel, int maximumChannel, boolean disableMX,
			boolean mxFacilityIDOnly, boolean preferOperating, boolean setUndesired, ErrorLogger errors) {

		// Desired and protected searches are allowed in general-purpose FM studies and TV channel 6 vs. FM studies.

		int studyType = scenario.study.study.studyType;
		if (!isExtDbSupported(extDb) || !Study.isRecordTypeAllowed(studyType, Source.RECORD_TYPE_FM) ||
				(((ExtDbSearch.SEARCH_TYPE_DESIREDS == searchType) ||
					(ExtDbSearch.SEARCH_TYPE_PROTECTEDS == searchType)) &&
					(Study.STUDY_TYPE_FM != studyType) && (Study.STUDY_TYPE_TV6_FM != studyType))) {
			return 0;
		}

		// Get necessary study parameters.

		double coChanMX = scenario.study.getCoChannelMxDistance();
		double kmPerDeg = scenario.study.getKilometersPerDegree();

		// For desired or protected FM record in a TV6-FM study restrict channel to the NCE band.

		int minChannel = SourceFM.CHANNEL_MIN;
		int maxChannel = SourceFM.CHANNEL_MAX;
		if (((ExtDbSearch.SEARCH_TYPE_DESIREDS == searchType) || (ExtDbSearch.SEARCH_TYPE_PROTECTEDS == searchType)) &&
				(Study.STUDY_TYPE_TV6_FM == studyType)) {
			maxChannel = SourceFM.CHANNEL_MAX_NCE;
		}

		if (minimumChannel > minChannel) {
			minChannel = minimumChannel;
		}
		if ((maximumChannel > 0) && (maximumChannel < maxChannel)) {
			maxChannel = maximumChannel;
		}
		if (minChannel > maxChannel) {
			return 0;
		}

		StringBuilder q = new StringBuilder();
		boolean hasCrit = false;
		if ((null != query) && (query.length() > 0)) {
			q.append(query);
			hasCrit = true;
		}

		ArrayList<SourceEditData> theSources = null;
		Collection<SearchDelta> deltas = null;

		if (ExtDbSearch.SEARCH_TYPE_DESIREDS == searchType) {

			try {
				addChannelRangeQueryFM(extDb.type, extDb.version, minChannel, maxChannel, q, hasCrit);
			} catch (IllegalArgumentException ie) {
			}

		} else {

			if (ExtDbSearch.SEARCH_TYPE_PROTECTEDS == searchType) {

				theSources = scenario.sourceData.getUndesiredSources(Source.RECORD_TYPE_FM);
				if (theSources.isEmpty()) {
					if (null != errors) {
						errors.reportError("There are no undesired FM stations in the scenario");
					}
					return -1;
				}

			} else {

				theSources = scenario.sourceData.getDesiredSources(Source.RECORD_TYPE_FM);
				if (theSources.isEmpty()) {
					if (null != errors) {
						errors.reportError("There are no desired FM stations in the scenario");
					}
					return -1;
				}
			}

			// Build a list of search deltas, accumulate worst-case maximum distance.

			HashMap<Integer, SearchDelta> searchDeltas = new HashMap<Integer, SearchDelta>();
			SearchDelta searchDelta;

			for (IxRuleEditData theRule : scenario.study.ixRuleData.getActiveRows()) {

				if (Source.RECORD_TYPE_FM != theRule.serviceType.recordType) {
					continue;
				}

				searchDelta = searchDeltas.get(Integer.valueOf(theRule.channelDelta.delta));

				if (null == searchDelta) {

					searchDelta = new SearchDelta();
					searchDelta.delta = theRule.channelDelta.delta;
					searchDelta.maximumDistance = theRule.distance;

					searchDeltas.put(Integer.valueOf(searchDelta.delta), searchDelta);

				} else {

					if (theRule.distance > searchDelta.maximumDistance) {
						searchDelta.maximumDistance = theRule.distance;
					}
				}
			}

			deltas = searchDeltas.values();

			// Build list of channels to search.

			int desChan, undChan, numChans = 0, maxChans = (maxChannel - minChannel) + 1, iChan;
			SourceEditDataFM theSource;

			boolean[] searchChans = new boolean[maxChans];

			for (SourceEditData aSource : theSources) {
				theSource = (SourceEditDataFM)aSource;

				for (SearchDelta theDelta : deltas) {

					if (ExtDbSearch.SEARCH_TYPE_PROTECTEDS == searchType) {

						undChan = theSource.channel;
						desChan = undChan - theDelta.delta;

						if ((desChan < minChannel) || (desChan > maxChannel)) {
							continue;
						}

						iChan = desChan - minChannel;
						if (searchChans[iChan]) {
							continue;
						}

					} else {

						desChan = theSource.channel;
						undChan = desChan + theDelta.delta;

						if ((undChan < minChannel) || (undChan > maxChannel)) {
							continue;
						}

						iChan = undChan - minChannel;
						if (searchChans[iChan]) {
							continue;
						}
					}

					if (searchChans[iChan]) {
						continue;
					}

					searchChans[iChan] = true;
					numChans++;

					if (numChans == maxChans) {
						break;
					}
				}

				if (numChans == maxChans) {
					break;
				}
			}

			// No channels is not an error.

			if (0 == numChans) {
				return 0;
			}

			// Add the channel list or range to the query.

			if (numChans < maxChans) {

				StringBuilder chanList = new StringBuilder();
				char sep = '(';
				for (iChan = 0; iChan < maxChans; iChan++) {
					if (searchChans[iChan]) {
						chanList.append(sep);
						chanList.append(String.valueOf(iChan + minChannel));
						sep = ',';
					}
				}
				chanList.append(')');
				try {
					addMultipleChannelQueryFM(extDb.type, extDb.version, chanList.toString(), q, hasCrit);
				} catch (IllegalArgumentException ie) {
				}

			} else {

				try {
					addChannelRangeQueryFM(extDb.type, extDb.version, minChannel, maxChannel, q, hasCrit);
				} catch (IllegalArgumentException ie) {
				}
			}
		}

		// Do the search.

		LinkedList<ExtDbRecordFM> records = findRecordsFM(extDb, q.toString(), searchCenter, searchRadius, kmPerDeg,
			errors);

		if (null == records) {
			return -1;
		}
		if (records.isEmpty()) {
			return 0;
		}

		// Remove mutually-exclusive records.

		removeAllMX(scenario, records, disableMX, mxFacilityIDOnly, preferOperating, coChanMX, kmPerDeg);

		// For protecteds or undesireds searches, eliminate any new records that are outside the maximum distance
		// limits from the rules.

		if (ExtDbSearch.SEARCH_TYPE_DESIREDS != searchType) {

			ListIterator<ExtDbRecordFM> lit = records.listIterator(0);
			ExtDbRecordFM theRecord;
			SourceEditDataFM theSource;
			boolean remove;
			int chanDelt;
			double checkDist;

			while (lit.hasNext()) {

				theRecord = lit.next();
				remove = true;

				for (SourceEditData aSource : theSources) {
					theSource = (SourceEditDataFM)aSource;

					if (ExtDbSearch.SEARCH_TYPE_PROTECTEDS == searchType) {
						chanDelt = theSource.channel - theRecord.channel;
					} else {
						chanDelt = theRecord.channel - theSource.channel;
					}

					for (SearchDelta theDelta : deltas) {

						if (theDelta.delta != chanDelt) {
							continue;
						}

						if (ExtDbSearch.SEARCH_TYPE_PROTECTEDS == searchType) {
							checkDist = theDelta.maximumDistance + theRecord.getRuleExtraDistance(scenario.study);
							if (theRecord.location.distanceTo(theSource.location, kmPerDeg) <= checkDist) {
								remove = false;
								break;
							}
						} else {
							checkDist = theDelta.maximumDistance + theSource.getRuleExtraDistance();
							if (theSource.location.distanceTo(theRecord.location, kmPerDeg) <= checkDist) {
								remove = false;
								break;
							}
						}
					}

					if (!remove) {
						break;
					}
				}

				if (remove) {
					lit.remove();
				}
			}
		}

		// Add the new records to the scenario.

		SourceEditData newSource;
		ArrayList<SourceEditData> newSources = new ArrayList<SourceEditData>();

		if (null != errors) {
			errors.clearErrors();
		} else {
			errors = new ErrorLogger(null, null);
		}

		for (ExtDbRecordFM theRecord : records) {
			newSource = scenario.study.findSharedSource(theRecord.extDb.key, theRecord.extRecordID);
			if (null == newSource) {
				newSource = SourceEditData.makeSource(theRecord, scenario.study, true, errors);
				if (null == newSource) {
					if (errors.hasErrors()) {
						return -1;
					}
					continue;
				}
			}
			newSources.add(newSource);
		}

		boolean isDesired = true, isUndesired = true;
		if ((ExtDbSearch.SEARCH_TYPE_DESIREDS == searchType) || (ExtDbSearch.SEARCH_TYPE_PROTECTEDS == searchType)) {
			if (disableMX) {
				isUndesired = false;
			} else {
				isUndesired = setUndesired;
			}
		} else {
			isDesired = false;
			if (disableMX) {
				isUndesired = false;
			}
		}

		for (SourceEditData aSource : newSources) {
			scenario.sourceData.addOrReplace(aSource, isDesired, isUndesired);
		}

		return newSources.size();
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Filter a list of new records for MX relationships, see comments in ExtDbRecordTV.

	private static void removeAllMX(ScenarioEditData scenario, LinkedList<ExtDbRecordFM> records, boolean disableMX,
			boolean mxFacilityIDOnly, boolean preferOperating, double coChanMX, double kmPerDeg) {

		ArrayList<SourceEditData> theSources = scenario.sourceData.getSources(Source.RECORD_TYPE_FM);

		ListIterator<ExtDbRecordFM> lit = records.listIterator(0);
		ExtDbRecordFM theRecord;
		SourceEditDataFM theSource;
		while (lit.hasNext()) {
			theRecord = lit.next();
			for (SourceEditData aSource : theSources) {
				theSource = (SourceEditDataFM)aSource;
				if (theRecord.extRecordID.equals(theSource.extRecordID) ||
						(!disableMX && areRecordsMX(theRecord, theSource, mxFacilityIDOnly, coChanMX, kmPerDeg))) {
					lit.remove();
					break;
				}
			}
		}

		if (disableMX) {
			return;
		}

		// Check remaining records for MX pairs and pick one using isPreferredRecord().

		final boolean prefOp = preferOperating;
		Comparator<ExtDbRecordFM> prefComp = new Comparator<ExtDbRecordFM>() {
			public int compare(ExtDbRecordFM theRecord, ExtDbRecordFM otherRecord) {
				if (theRecord.isPreferredRecord(otherRecord, prefOp)) {
					return -1;
				}
				return 1;
			}
		};

		Collections.sort(records, prefComp);

		int recCount = records.size() - 1;
		for (int recIndex = 0; recIndex < recCount; recIndex++) {
			theRecord = records.get(recIndex);
			lit = records.listIterator(recIndex + 1);
			while (lit.hasNext()) {
				if (areRecordsMX(theRecord, lit.next(), mxFacilityIDOnly, coChanMX, kmPerDeg)) {
					lit.remove();
					recCount--;
				}
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------

	private ExtDbRecordFM(ExtDb theExtDb) {

		super(theExtDb, Source.RECORD_TYPE_FM);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update a source record object from this object's properties, this is called by makeSource() in SourceEditData
	// when creating a new source encapsulating this record.

	public boolean updateSource(SourceEditDataFM theSource) {
		return updateSource(theSource, null);
	}

	public boolean updateSource(SourceEditDataFM theSource, ErrorLogger errors) {

		if (!extDb.dbID.equals(theSource.dbID) || (facilityID != theSource.facilityID) ||
				(stationClass != theSource.stationClass) || !service.equals(theSource.service) ||
				!country.equals(theSource.country)) {
			if (null != errors) {
				errors.reportError("ExtDbRecordFM.updateSource(): non-matching source object");
			}
			return false;
		}

		if (!isExtDbSupported(extDb)) {
			if (null != errors) {
				errors.reportError("ExtDbRecordFM.updateSource(): unsupported station data type");
			}
			return false;
		}

		boolean isCDBS = (ExtDb.DB_TYPE_CDBS_FM == extDb.type);

		// Some of the logic below depends on checking for reported errors from other methods.  If the caller did not
		// provide an error logger, create a temporary one.  Also saves having to check for null everywhere.

		if (null != errors) {
			errors.clearErrors();
		} else {
			errors = new ErrorLogger(null, null);
		}
		boolean error = false;

		// Begin copying properties.

		theSource.isIBOC = isIBOC;
		theSource.ibocFraction = ibocFraction;
		theSource.callSign = callSign;
		theSource.channel = channel;
		theSource.city = city;
		theSource.state = state;
		theSource.status = status;
		theSource.statusType = statusType;
		theSource.fileNumber = filePrefix + appARN;
		theSource.appARN = appARN;
		theSource.location.setLatLon(location);
		theSource.heightAMSL = heightAMSL;
		theSource.overallHAAT = overallHAAT;
		theSource.peakERP = peakERP;

		if (isIBOC) {

			// Determine IBOC power level for a CDBS record, this involves a separate record in the if_notification
			// table indicating the power, which is converted to a fraction.  Records in that table are not directly
			// linked to the engineering data; search for applications with the service FD and matching facility ID.
			// If more than one record is found take the most-recent based on ARN.  For LMS records this was already
			// determined during the main query processing in findRecordsFM().

			if (isCDBS) {

				DbConnection db = extDb.connectDb(errors);
				if (null != db) {
					try {

						db.query(
						"SELECT " +
							"if_notification.digital_erp / if_notification.analog_erp " +
						"FROM " +
							"if_notification " +
							"JOIN application USING (application_id) " +
						"WHERE " +
							"application.app_service = 'FD' " +
							"AND application.facility_id = " + facilityID + " " +
						"ORDER BY " +
							"application.app_arn DESC " +
						"LIMIT 1");

						if (db.next()) {
							theSource.ibocFraction = db.getDouble(1);
						}

						extDb.releaseDb(db);

					} catch (SQLException se) {
						extDb.releaseDb(db);
						error = true;
						DbConnection.reportError(errors, se);
					}

				} else {
					error = true;
				}

				if (error) {
					return false;
				}
			}

			// Regardless of where it came from, silently restrict the IBOC power fraction to the valid range.

			if (theSource.ibocFraction < SourceFM.IBOC_FRACTION_MIN) {
				theSource.ibocFraction = SourceFM.IBOC_FRACTION_MIN;
			}
			if (theSource.ibocFraction > SourceFM.IBOC_FRACTION_MAX) {
				theSource.ibocFraction = SourceFM.IBOC_FRACTION_MAX;
			}
		}

		// Get horizontal pattern data if needed.  If pattern data does not exist set pattern to omni.  A pattern with
		// too few points or bad data values will also revert to omni, see pattern-load methods in the superclass for
		// details.  If the study parameter says to trust the DA indicator and that is not set don't even look for the
		// pattern just use omni.  If there is no study context, default to ignoring the DA indicator.  Note there are
		// no vertical patterns on these records.

		if ((null != antennaRecordID) && (daIndicated || (null == theSource.study) ||
				!theSource.study.getTrustPatternFlag())) {

			String theName = null, make, model;

			DbConnection db = extDb.connectDb(errors);
			if (null != db) {
				try {

					if (isCDBS) {

						db.query(
						"SELECT " +
							"ant_make, " +
							"ant_model_num " +
						"FROM " +
							"ant_make " +
						"WHERE " +
							"antenna_id = " + antennaRecordID);


					} else {

						if (ExtDb.DB_TYPE_LMS_LIVE == extDb.type) {

							db.query(
							"SELECT " +
								"aant_make, " +
								"aant_model " +
							"FROM " +
								"mass_media.app_antenna " +
							"WHERE " +
								"aant_antenna_record_id = '" + antennaRecordID + "'");

						} else {

							db.query(
							"SELECT " +
								"aant_make, " +
								"aant_model " +
							"FROM " +
								"app_antenna " +
							"WHERE " +
								"aant_antenna_record_id = '" + antennaRecordID + "'");
						}
					}

					if (db.next()) {

						make = db.getString(1);
						if (null == make) {
							make = "";
						}
						model = db.getString(2);
						if (null == model) {
							model = "";
						}
						theName = make + "-" + model;
						if (theName.length() > Source.MAX_PATTERN_NAME_LENGTH) {
							theName = theName.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
						}

					} else {

						errors.logMessage(makeMessage(this, "Antenna record ID " + antennaRecordID + " not found"));
					}

					extDb.releaseDb(db);

				} catch (SQLException se) {
					extDb.releaseDb(db);
					error = true;
					DbConnection.reportError(errors, se);
				}

			} else {
				error = true;
			}

			if (error) {
				return false;
			}

			if (null != theName) {
				AntPattern thePattern = ExtDb.getHorizontalPattern(this, theName, errors);
				if (null != thePattern) {
					theSource.antennaID = antennaID;
					theSource.hasHorizontalPattern = true;
					theSource.horizontalPattern = thePattern;
					theSource.horizontalPatternChanged = true;
				} else {
					if (errors.hasErrors()) {
						return false;
					}
				}
			}
		}

		theSource.horizontalPatternOrientation = horizontalPatternOrientation;

		// Save sequence date and other flags in source attributes.

		theSource.setAttribute(Source.ATTR_SEQUENCE_DATE, AppCore.formatDate(sequenceDate));
		if ((null != licensee) && (licensee.length() > 0)) {
			theSource.setAttribute(Source.ATTR_LICENSEE, licensee);
		}
		if (isPending) {
			theSource.setAttribute(Source.ATTR_IS_PENDING);
		}
		if (isArchived) {
			theSource.setAttribute(Source.ATTR_IS_ARCHIVED);
		}
		if (contour73215) {
			theSource.setAttribute(Source.ATTR_CONTOUR_73_215);
		}

		// Apply defaults and corrections, see discussion in ExtDbRecordTV.  Small difference here, if the ERP is 0 do
		// not assume that is dBk assume the value is missing and set a default of 1 watt; only assume dBk if < 0.

		if (null == theSource.study) {
			theSource.useGenericVerticalPattern = true;
		} else {
			theSource.useGenericVerticalPattern = theSource.study.getUseGenericVpat(theSource.country.key - 1);
		}

		if ((Country.MX == country.key) && (ServiceType.SERVTYPE_FM_FULL == service.serviceType.key) &&
				(null != theSource.study)) {

			if (0. == peakERP) {

				switch (stationClass) {

					case FM_CLASS_A:
					default: {
						theSource.peakERP = theSource.study.getDefaultMexicanFMAERP();
						break;
					}

					case FM_CLASS_B: {
						theSource.peakERP = theSource.study.getDefaultMexicanFMBERP();
						break;
					}

					case FM_CLASS_B1: {
						theSource.peakERP = theSource.study.getDefaultMexicanFMB1ERP();
						break;
					}

					case FM_CLASS_C: {
						theSource.peakERP = theSource.study.getDefaultMexicanFMCERP();
						break;
					}

					case FM_CLASS_C1: {
						theSource.peakERP = theSource.study.getDefaultMexicanFMC1ERP();
						break;
					}
				}

				errors.logMessage(makeMessage(this, "Used default for missing ERP"));
			}

			if ((0. == heightAMSL) && (0. == overallHAAT)) {

				switch (stationClass) {

					case FM_CLASS_A:
					default: {
						theSource.overallHAAT = theSource.study.getDefaultMexicanFMAHAAT();
						break;
					}

					case FM_CLASS_B: {
						theSource.overallHAAT = theSource.study.getDefaultMexicanFMBHAAT();
						break;
					}

					case FM_CLASS_B1: {
						theSource.overallHAAT = theSource.study.getDefaultMexicanFMB1HAAT();
						break;
					}

					case FM_CLASS_C: {
						theSource.overallHAAT = theSource.study.getDefaultMexicanFMCHAAT();
						break;
					}

					case FM_CLASS_C1: {
						theSource.overallHAAT = theSource.study.getDefaultMexicanFMC1HAAT();
						break;
					}
				}

				errors.logMessage(makeMessage(this, "Used default for missing HAAT"));
			}
		}

		if ((Country.US != country.key) && (0. == heightAMSL) && (0. != overallHAAT)) {
			theSource.heightAMSL = Source.HEIGHT_DERIVE;
			errors.logMessage(makeMessage(this, "Derived missing AMSL from HAAT"));
		}

		if (0. == peakERP) {
			theSource.peakERP = Source.ERP_DEF;
			errors.logMessage(makeMessage(this, "Used default for missing ERP"));
		}

		if (peakERP < 0.) {
			theSource.peakERP = Math.pow(10., (peakERP / 10.));
			errors.logMessage(makeMessage(this, "Converted ERP from dBk to kilowatts"));
		}

		// Done.

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Determine if two records (contained in ExtDbRecordFM or SourceEditDataFM objects) are mutually-exclusive.  The
	// primary test is facility ID; any two records with the same facility ID are MX.  Beyond that are back-up checks
	// to detect co-channel MX cases with differing facility IDs.  Those apply only for matching channel and country.
	// Most often those checks are needed to detect MX applications for a new facility, but also to catch cases such as
	// DTV rule-making records that have a facility ID that does not match the actual station.  Optionally the backup
	// tests can be skipped so only facility ID is considered.

	public static boolean areRecordsMX(ExtDbRecordFM a, ExtDbRecordFM b, boolean facIDOnly, double mxDist,
			double kmPerDeg) {
		return areRecordsMX(
			a.facilityID, a.channel, a.country.key, a.state, a.city, a.location,
			b.facilityID, b.channel, b.country.key, b.state, b.city, b.location,
			facIDOnly, mxDist, kmPerDeg);
	}

	public static boolean areRecordsMX(ExtDbRecordFM a, SourceEditDataFM b, boolean facIDOnly, double mxDist,
			double kmPerDeg) {
		return areRecordsMX(
			a.facilityID, a.channel, a.country.key, a.state, a.city, a.location,
			b.facilityID, b.channel, b.country.key, b.state, b.city, b.location,
			facIDOnly, mxDist, kmPerDeg);
	}

	public static boolean areRecordsMX(SourceEditDataFM a, SourceEditDataFM b, boolean facIDOnly, double mxDist,
			double kmPerDeg) {
		return areRecordsMX(
			a.facilityID, a.channel, a.country.key, a.state, a.city, a.location,
			b.facilityID, b.channel, b.country.key, b.state, b.city, b.location,
			facIDOnly, mxDist, kmPerDeg);
	}

	private static boolean areRecordsMX(int a_facilityID, int a_channel, int a_country_key, String a_state,
			String a_city, GeoPoint a_location, int b_facilityID, int b_channel, int b_country_key, String b_state,
			String b_city, GeoPoint b_location, boolean facIDOnly, double mxDist, double kmPerDeg) {

		if (a_facilityID == b_facilityID) {
			return true;
		}

		if (facIDOnly) {
			return false;
		}

		if (a_channel != b_channel) {
			return false;
		}

		if (a_country_key != b_country_key) {
			return false;
		}

		if (a_state.equalsIgnoreCase(b_state) && a_city.equalsIgnoreCase(b_city)) {
			return true;
		}

		if ((mxDist > 0.) && (a_location.distanceTo(b_location, kmPerDeg) < mxDist)) {
			return true;
		}

		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check another record and determine if this record is preferred over the other, presumably mutually-exclusive.
	// Returns true if this record is preferred, false otherwise.

	public boolean isPreferredRecord(ExtDbRecordFM otherRecord, boolean preferOperating) {

		// The first test is to prefer an "operating facility" record, see isOperating() for details.  This test can
		// be disabled by argument.

		if (preferOperating) {
			if (isOperating()) {
				if (!otherRecord.isOperating()) {
					return true;
				}
			} else {
				if (otherRecord.isOperating()) {
					return false;
				}
			}
		}

		// Next test the ranking of the record service, in case services are not the same.

		if (service.preferenceRank > otherRecord.service.preferenceRank) {
			return true;
		}
		if (service.preferenceRank < otherRecord.service.preferenceRank) {
			return false;
		}

		// Same operating status, same service ranking, check the record status and prefer according to the order
		// reflected by the STATUS_TYPE_* constants.

		if (statusType < otherRecord.statusType) {
			return true;
		}
		if (statusType > otherRecord.statusType) {
			return false;
		}

		// Prefer the record with the more-recent sequence date.

		if (sequenceDate.after(otherRecord.sequenceDate)) {
			return true;
		}
		if (sequenceDate.before(otherRecord.sequenceDate)) {
			return false;
		}

		// This must return a consistent non-equal result for sorting use (A.isPreferred(B) != B.isPreferred(A)), so if
		// all else was equal compare record IDs.  The ID sequence is arbitrary but since they are unique this is
		// guaranteed to never see the records as equal.

		if (extRecordID.compareTo(otherRecord.extRecordID) > 0) {
			return true;
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Determine if this record represents an actual operating facility.  The test result is cached.

	public boolean isOperating() {

		if (operatingStatusSet) {
			return isOperatingFacility;
		}

		// If the record service is non-operating (allotments, rule-makings, etc.), the record cannot be operating.
		// Archived records are never operating.  Status codes LIC and STA are always operating.  CP and OTHER are
		// operating for non-U.S., not operating for U.S.  OTHER is operating for non-U.S. but not for U.S.  EXP or
		// unknown are never operating.

		if (!service.isOperating || isArchived) {
			isOperatingFacility = false;
		} else {
			switch (statusType) {
				case STATUS_TYPE_LIC:
				case STATUS_TYPE_STA: {
					isOperatingFacility = true;
					break;
				}
				case STATUS_TYPE_CP:
				case STATUS_TYPE_OTHER: {
					if (Country.US == country.key) {
						isOperatingFacility = false;
					} else {
						isOperatingFacility = true;
					}
					break;
				}
				case STATUS_TYPE_EXP:
				default: {
					isOperatingFacility = false;
					break;
				}
			}
		}

		operatingStatusSet = true;

		return isOperatingFacility;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get the rule extra distance, see details in SourceEditDataFM.

	public double getRuleExtraDistance(StudyEditData study) {

		return SourceEditDataFM.getRuleExtraDistance(study, service, stationClass, country, channel, peakERP);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// See makeCommentText() in superclass.

	protected ArrayList<String> getComments() {

		ArrayList<String> result = null;

		boolean hasLic = ((null != licensee) && (licensee.length() > 0));

		if (hasLic || contour73215) {
			result = new ArrayList<String>();
			if (hasLic) {
				result.add("Licensee: " + licensee);
			}
			if (contour73215) {
				result.add("73.215 contour protection requested");
			}
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Methods in the StationRecord interface.  Some are in the superclass.

	public String getFacilityID() {

		return String.valueOf(facilityID);
	}

	public String getSortFacilityID() {

		return String.format(Locale.US, "%07d", facilityID);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getCallSign() {

		return callSign;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getChannel() {

		return String.valueOf(channel) + FM_CLASS_CODES[stationClass];
	}

	public int getChannelNumber() {

		return channel;
	}

	public String getSortChannel() {

		return String.format(Locale.US, "%03d%02d", channel, stationClass);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFrequency() {

		return SourceEditDataFM.getFrequency(channel);
	}

	public double getFrequencyValue() {

		return SourceEditDataFM.getFrequencyValue(channel);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getStatus() {

		return status + (isPending ? " *P" : (isArchived ? " *A" : ""));
	}

	public String getSortStatus() {

		return String.valueOf(statusType);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String getFileNumber() {

		return filePrefix + appARN;
	}

	public String getARN() {

		return appARN;
	}
}
