//
//  ExtDb.java
//  TVStudy
//
//  Copyright (c) 2015-2024 Hammett & Edison, Inc.  All rights reserved.

package gov.fcc.tvstudy.core;

import gov.fcc.tvstudy.core.data.*;
import gov.fcc.tvstudy.core.editdata.*;
import gov.fcc.tvstudy.core.geo.*;

import java.util.*;
import java.text.*;
import java.sql.*;
import java.io.*;
import java.nio.file.*;
import java.net.*;
import java.util.zip.*;


//=====================================================================================================================
// Represents records from the ext_db table in the root database.  Provides static factory methods, as well as static
// methods to create and delete external data set databases.  See ExtDbRecord and gui.ExtDbManager for details.  This
// now also supports external databases on different hosts for direct connection to "live" database servers, and also
// special key values that automatically map to the most recent imported data.

public class ExtDb {

	public static final int DB_TYPE_NOT_SET = -1;
	public static final int DB_TYPE_UNKNOWN = 0;

	public static final int DB_TYPE_CDBS = 1;
	public static final int DB_TYPE_LMS = 2;
	public static final int DB_TYPE_CDBS_FM = 4;
	public static final int DB_TYPE_LMS_LIVE = 5;
	public static final int DB_TYPE_GENERIC_TV = 6;
	public static final int DB_TYPE_GENERIC_WL = 7;
	public static final int DB_TYPE_GENERIC_FM = 8;

	// This type is an obsolete wireless data set format, existing data sets of this type are converted to the generic
	// type on database upgrade, see convertWirelessExtDbs().

	public static final int DB_TYPE_WIRELESS = 3;

	// Sorting map, see getExtDbList().

	private static final int[] DB_TYPE_ORDER = {4, 2, 999, 6, 1, 3, 5, 7};

	// A key range is reserved for internal use and will not be used for imported data sets, see createNewDatabase().

	public static final int RESERVED_KEY_RANGE_START = 10000;
	public static final int RESERVED_KEY_RANGE_END = 19999;

	// Key values in the reserved range for "live" external server objects, currently only LMS has that feature.

	public static final int KEY_LMS_LIVE = 10005;

	// Key values for the "most recent" functions, in the reserved range.  When these keys are retrieved by getExtDb(),
	// an object is returned for an actual imported data set in the category (the most-recent keys never appear on an
	// actual object).  Most recent is based on the actual data date determined during import, not the date of import.
	// See createNewDatabase().  For that reason, wireless and generic imports are not part of the most-recent test as
	// those have no intrinsic dating of content, the database timestamp is the time of import.

	public static final int KEY_MOST_RECENT_LMS = 10102;
	public static final int KEY_MOST_RECENT_CDBS = 10103;
	public static final int KEY_MOST_RECENT_CDBS_FM = 10104;

	// Name used in UI to identify an unnamed download set.

	public static final String DOWNLOAD_SET_NAME = "(download)";

	// File formats for generic import, see importToGenericDatabase().

	public static final int GENERIC_FORMAT_UNKNOWN = 0;
	public static final int GENERIC_FORMAT_TEXT = 1;
	public static final int GENERIC_FORMAT_DBF = 2;

	// Properties.

	public final String dbID;

	public final Integer key;

	public final String dbName;
	public final java.util.Date dbDate;

	public final int type;
	public final int version;
	public final int indexVersion;

	public String id;
	public String name;
	public String description;

	public boolean deleted;
	public boolean isDownload;
	public boolean isLocked;
	public boolean hasBadData;

	// To support databases on different servers, connections are always obtained with connectDb()/releaseDb() methods
	// mirroring those in DbCore.  For imported data sets those are mostly just wrappers for the DbCore methods but if
	// isLive is true a separate server is used with a local connection pool.  These are called "live" because they are
	// usually active servers providing current data being edited through other UIs, e.g. see getLMSLiveExtDb().

	private final boolean isLive;
	private DbConnection liveDb;
	private ArrayDeque<DbConnection> dbPool;
	private HashSet<DbConnection> openDbs;

	// Generic import data sets can be expanded by additional imports, which involves creating new SourceEditData
	// objects to be saved into the data set's database.  See connectAndLock() and getNewRecordKey().

	private boolean dbLocked;
	private int nextRecordKey;

	// Interval for automatic update of cache, milliseconds.  Some calls will always update regardless of the interval.

	private static final long CACHE_UPDATE_INTERVAL = 300000L;

	// Name of table definitions file for CDBS dump files, see loadCDBSFieldNames().

	private static final String CDBS_TABLE_DEFS_FILE = "cdbs_table_defs.dat";

	// Current version numbers for imports.

	// Note generic data sets are not versioned.  The backing store for those is the usual Source tables in separate
	// import databases, as with study databases those will be updated in place when changes occur.  Internally records
	// from those data sets are provided as native SourceEditData objects, there is no ExtDbRecord translation subclass
	// needed.  The SourceEditData superclass is used to compose search queries.

	// Version change history.  Note the import code has a limited ability to detect version from import file content
	// and adjust accordingly, the constant here is the highest version but that may be rolled back in some cases.

	// Any version 0
	//   Imported before app version 1.4.1 when versioning was added (or generic).
	//   NO LONGER SUPPORTED (except for generic).

	// CDBS
	//   Version 1
	//     import app_tracking table for accepted_date field
	//     import am_ant_sys table for AM station checks (optional)

	// LMS
	//   Version 1
	//     new fields dts_reference_ind, dts_waiver_distance in application
	//     NO LONGER SUPPORTED.
	//   Version 2
	//     include facility table to get facility_status field
	//     import tables for baseline record support
	//     import CDBS application ID cross-reference for lookup during XML import (optional)
	//   Version 3
	//     new fields for service and country in lkp_dtv_allotment
	//     import CDBS mirror tables gis_am_ant_sys, gis_facility, and gis_application for AM checks (optional)
	//   Version 4
	//     new field for Class A baseline records, app_dtv_channel_assignment.emission_mask_code
	//   Version 5
	//     new field app_dtv_channel_assignment.electrical_deg for electrical beam tilt
	//     new field app_antenna.foreign_station_beam_tilt for electrical beam tilt on non-U.S. records
	//   Version 6
	//     import shared_channel table
	//   Version 7
	//     support for FM records
	//     requires new fields added to schema recently
	//   Version 8
	//     import facility_applicant table to get licensee name for TV and FM records
	//     no longer importing CDBS cross-reference table
	//   Version 9
	//     import app_mm_application table to get 73.215 flag for FM records
    //   Version 10
    //     import app_am_antenna table, require application_facilty.am_frequency
	//     no longer importing gis_* CDBS mirror tables
	//   Version 11
	//     import digital_notification table to get FM IBOC digital power

	// CDBS_FM
	//   Version 1
	//     first version for new data type

	private static final int CDBS_VERSION = 1;
	private static final int LMS_VERSION = 11;
	private static final int CDBS_FM_VERSION = 1;

	// Minimum version numbers so support for older versions can be removed from query code.  Any older imports have
	// "(unsupported)" in the description and appear only in the data set manager list so they can still be deleted.

	private static final int CDBS_MIN_VERSION = 1;
	private static final int LMS_MIN_VERSION = 2;
	private static final int CDBS_FM_MIN_VERSION = 1;

	// Search index version, see updateSearchIndex().

	public static final int LMS_INDEX_VERSION = 3;

	private static final int DOWNLOAD_TIMEOUT = 30000;   // milliseconds


	//-----------------------------------------------------------------------------------------------------------------
	// Instances come only from factory methods below.  Some data set types can now provide more than one record type
	// so record type is no longer a property; use canProvide().

	private ExtDb(String theDbID, Integer theKey, String theDbName, java.util.Date theDbDate, int theType,
			int theVersion, int theIndexVersion, boolean theIsLive) {

		dbID = theDbID;

		key = theKey;

		dbName = theDbName;
		dbDate = theDbDate;

		type = theType;
		version = theVersion;
		indexVersion = theIndexVersion;

		isLive = theIsLive;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public String toString() {

		return description;
	}


	//-----------------------------------------------------------------------------------------------------------------

	public int hashCode() {

		return key.intValue();
	}


	//-----------------------------------------------------------------------------------------------------------------

	public boolean equals(Object other) {

		return (null != other) && ((ExtDb)other).key.equals(key);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Connect to a generic data set and lock it for updating.  If this is not a generic set this fails, also if the
	// set is already locked this fails, see connectDb().  If this succeeds, update the value for the next record key.

	public DbConnection connectAndLock() {
		return connectAndLock(null);
	}

	public DbConnection connectAndLock(ErrorLogger errors) {

		if (!isGeneric()) {
			if (null != errors) {
				errors.reportError("Connect failed, station data cannot be locked");
			}
			return null;
		}

		DbConnection db = connectDb(true, errors);
		if (null == db) {
			return null;
		}

		nextRecordKey = 1;
		try {
			db.query("SELECT MAX(source_key) FROM source");
			if (db.next()) {
				nextRecordKey = db.getInt(1) + 1;
			}
		} catch (SQLException se) {
			DbCore.releaseDb(db);
			db = null;
			DbConnection.reportError(errors, se);
		}

		return db;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get a connected database object from a pool, the connection must be released by passing it to releaseDb().  For
	// normal imported data sets this is just a wrapper around DbCore methods.  When the database represented by this
	// object is on a live server, a local connection pool is maintained.  That may be shared by other objects.  The
	// private version may attempt to lock the data set, see connectAndLock() above.  The current lock will always be
	// checked and the connect fails if the set is locked, if not locked and doLock is true the lock is set.  Note the
	// lock protocol is really just advisory within this app instance, there is no protection against some overlapping
	// uses of connectDb() and connectAndLock().  Specifically connectAndLock() does not check if there are existing
	// unreleased connections before setting the lock, and once the lock is set it is cleared by the next releaseDb()
	// regardless of which connection is being released.  But the primary purpose is to keep concurrent app instances
	// that are sharing the root database from concurrently modifying the data set, and it does that fine.  Note the
	// live databases are never locked, and currently the only others that can be locked are generic sets which can
	// have additional data added by multiple imports.  The other types of sets are created and imported atomically
	// (the ext_db record is not written until the import is done, see createNewDatabase()) and so don't need locking.

	// Note this deliberately does not check the deleted flag; if the data set database no longer exists an SQL error
	// will occur.  That should rarely happen because of the deferred-delete behavior.  When a set is deleted, the
	// index record is marked deleted however the database itself is not dropped until closeDb() is called, which
	// presumably does not occur until there are no running or pending activities that might still try to access the
	// data set.  The only case that is not guarded is concurrent delete by another application instance which may
	// actually drop the database before activities here complete, but that is assumed to be very unlikely.  This will
	// fail for unsupported sets, so query code does not have to do an unsupported version check.

	public DbConnection connectDb() {
		return connectDb(false, null);
	}

	public DbConnection connectDb(ErrorLogger errors) {
		return connectDb(false, errors);
	}

	private synchronized DbConnection connectDb(boolean doLock, ErrorLogger errors) {

		if (!isSupported()) {
			if (null != errors) {
				errors.reportError("The station data version is not supported");
			}
			return null;
		}

		DbConnection db = null;

		if (isLive) {

			db = dbPool.poll();
			if (null == db) {
				db = liveDb.copy();
			}

			openDbs.add(db);

			if (!db.connect(dbName, errors)) {
				releaseDb(db);
				return null;
			}

		} else {

			// If an unreleased connection in this same app instance has the data set locked, can't open another.

			if (dbLocked) {
				if (null != errors) {
					errors.reportWarning("The station data is in use");
				}
				return null;
			}

			db = DbCore.connectDb(dbID, errors);
			if (null == db) {
				return null;
			}

			// Check the lock state, fail if locked, else set the lock if requested.

			String rootName = DbCore.getDbName(dbID);
			boolean wasLocked = false;

			try {

				db.setDatabase(dbName);

				db.update("LOCK TABLES " + rootName + ".ext_db WRITE");

				db.query("SELECT locked FROM " + rootName + ".ext_db WHERE ext_db_key = " + key);
				if (db.next()) {
					isLocked = db.getBoolean(1);
					wasLocked = isLocked;
				}

				if (!isLocked && doLock) {
					db.update("UPDATE " + rootName + ".ext_db SET locked = true WHERE ext_db_key = " + key);
					dbLocked = true;
					isLocked = true;
				}

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				DbConnection.reportError(errors, se);
				return null;
			}

			try {
				db.update("UNLOCK TABLES");
			} catch (SQLException se) {
				db.reportError(se);
			}

			if (wasLocked) {
				if (null != errors) {
					errors.reportWarning("The station data is in use");
				}
				DbCore.releaseDb(db);
				db = null;
			}
		}

		return db;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// See comments above regarding locking protocol.

	public synchronized void releaseDb(DbConnection db) {

		if (isLive) {

			if (openDbs.remove(db)) {
				db.close();
				dbPool.push(db);
			}

		} else {

			if (dbLocked) {
				try {
					db.update("UPDATE " + DbCore.getDbName(dbID) + ".ext_db SET locked = false WHERE ext_db_key = " +
						key);
				} catch (SQLException se) {
					db.reportError(se);
				}
				dbLocked = false;
				isLocked = false;
			}

			DbCore.releaseDb(db);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Attempt to forcibly unlock the database, should be warnings in UI leading to this.  Returns new lock state.

	public boolean unlock() {
		return unlock(null);
	}

	public boolean unlock(ErrorLogger errors) {

		DbConnection db = DbCore.connectDb(dbID, errors);
		if (null == db) {
			return isLocked;
		}

		try {

			db.update("LOCK TABLES ext_db WRITE");

			db.update("UPDATE ext_db SET locked = false WHERE ext_db_key = " + key);
			dbLocked = false;
			isLocked = false;

		} catch (SQLException se) {
			DbConnection.reportError(errors, se);
		}

		try {
			db.update("UNLOCK TABLES");
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		return isLocked;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get a new record key for new imports to a data set, the set must be locked which can only occur under certain
	// conditions, see connectAndLock() above.

	public Integer getNewRecordKey() {

		if (dbLocked) {
			return Integer.valueOf(nextRecordKey++);
		}
		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Return a string description of the data type.  "Generic" has been removed from the UI, generic sets are now
	// called "TVStudy XX" for TV/FM or just "Wireless" for wireless since there is no other type of wireless now.

	public String getTypeName() {
		return getTypeName(type);
	}

	public static String getTypeName(int theType) {

		switch (theType) {
			case DB_TYPE_CDBS:
				return "CDBS TV";
			case DB_TYPE_LMS:
			case DB_TYPE_LMS_LIVE:
				return "LMS";
			case DB_TYPE_CDBS_FM:
				return "CDBS FM";
			case DB_TYPE_GENERIC_TV:
				return "TVStudy TV";
			case DB_TYPE_GENERIC_WL:
				return "Wireless";
			case DB_TYPE_GENERIC_FM:
				return "TVStudy FM";
			default:
				return "??";
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Return a list of download and import types for UI selection; this does not include the "live" database types.
	// See ExtDbManager.  Since FM has migrated to LMS, that is now the only download type.  CDBS and CDBS_FM can still
	// be imported from archived downloads.

	private static ArrayList<KeyedRecord> downloadTypes = null;

	public static synchronized ArrayList<KeyedRecord> getDownloadTypes() {

		if (null == downloadTypes) {
			downloadTypes = new ArrayList<KeyedRecord>();
			downloadTypes.add(new KeyedRecord(DB_TYPE_LMS, getTypeName(DB_TYPE_LMS)));
		}

		return new ArrayList<KeyedRecord>(downloadTypes);
	}

	private static ArrayList<KeyedRecord> importTypes = null;

	public static synchronized ArrayList<KeyedRecord> getImportTypes() {

		if (null == importTypes) {
			importTypes = new ArrayList<KeyedRecord>();
			importTypes.add(new KeyedRecord(DB_TYPE_LMS, getTypeName(DB_TYPE_LMS)));
			importTypes.add(new KeyedRecord(DB_TYPE_GENERIC_WL, getTypeName(DB_TYPE_GENERIC_WL)));
			importTypes.add(new KeyedRecord(DB_TYPE_GENERIC_TV, getTypeName(DB_TYPE_GENERIC_TV)));
			importTypes.add(new KeyedRecord(DB_TYPE_GENERIC_FM, getTypeName(DB_TYPE_GENERIC_FM)));
			importTypes.add(new KeyedRecord(DB_TYPE_CDBS, getTypeName(DB_TYPE_CDBS)));
			importTypes.add(new KeyedRecord(DB_TYPE_CDBS_FM, getTypeName(DB_TYPE_CDBS_FM)));
		}

		return new ArrayList<KeyedRecord>(importTypes);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Names for keys in the reserved range.  These names are also reserved and can't be used for named imports.

	private static HashMap<Integer, String> reservedKeyNames = new HashMap<Integer, String>();
	static {
		reservedKeyNames.put(Integer.valueOf(KEY_MOST_RECENT_LMS), "Most recent LMS");
		reservedKeyNames.put(Integer.valueOf(KEY_MOST_RECENT_CDBS), "Most recent CDBS TV");
		reservedKeyNames.put(Integer.valueOf(KEY_MOST_RECENT_CDBS_FM), "Most recent CDBS FM");
		reservedKeyNames.put(Integer.valueOf(KEY_LMS_LIVE), "LMS live server");
	}

	public static synchronized String getReservedKeyName(int theKey) {

		String theName = reservedKeyNames.get(Integer.valueOf(theKey));
		if (null == theName) {
			theName = "???";
		}
		return theName;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check if type and version are supported.

	public boolean isSupported() {
		return isSupported(type, version, indexVersion);
	}

	public static boolean isSupported(int theType, int theVersion, int theIndexVersion) {

		switch (theType) {
			case DB_TYPE_CDBS:
				return (theVersion >= CDBS_MIN_VERSION);
			case DB_TYPE_LMS:
				return (theVersion >= LMS_MIN_VERSION) && (LMS_INDEX_VERSION == theIndexVersion);
			case DB_TYPE_CDBS_FM:
				return (theVersion >= CDBS_FM_MIN_VERSION);
			case DB_TYPE_LMS_LIVE:
			case DB_TYPE_GENERIC_TV:
			case DB_TYPE_GENERIC_WL:
			case DB_TYPE_GENERIC_FM:
				return true;
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Return a default single record type for the data set, convenience.

	public int getDefaultRecordType() {

		return getDefaultRecordType(type, version);
	}

	public static int getDefaultRecordType(int theType, int theVersion) {

		switch (theType) {
			case DB_TYPE_CDBS:
			case DB_TYPE_LMS:
			case DB_TYPE_LMS_LIVE:
			case DB_TYPE_GENERIC_TV:
				return Source.RECORD_TYPE_TV;
			case DB_TYPE_GENERIC_WL:
				return Source.RECORD_TYPE_WL;
			case DB_TYPE_CDBS_FM:
			case DB_TYPE_GENERIC_FM:
				return Source.RECORD_TYPE_FM;
		}
		return 0;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Determine if the data set contains a particular record type.  Originally this was a one-to-one mapping, a data
	// set could only contain one record type, but LMS can now provide multiple types.

	public boolean canProvide(int recordType) {

		return canProvide(type, version, recordType);
	}

	public static boolean canProvide(int theType, int theVersion, int recordType) {

		switch (theType) {
			case DB_TYPE_CDBS:
				return (Source.RECORD_TYPE_TV == recordType);
			case DB_TYPE_LMS:
				return ((Source.RECORD_TYPE_TV == recordType) ||
					((theVersion > 6) && (Source.RECORD_TYPE_FM == recordType)));
			case DB_TYPE_CDBS_FM:
				return (Source.RECORD_TYPE_FM == recordType);
			case DB_TYPE_LMS_LIVE:
				return (Source.RECORD_TYPE_TV == recordType);
			case DB_TYPE_GENERIC_TV:
				return (Source.RECORD_TYPE_TV == recordType);
			case DB_TYPE_GENERIC_WL:
				return (Source.RECORD_TYPE_WL == recordType);
			case DB_TYPE_GENERIC_FM:
				return (Source.RECORD_TYPE_FM == recordType);
		}
		return false;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check for any generic type.

	public boolean isGeneric() {

		return isGeneric(type);
	}

	public static boolean isGeneric(int theType) {

		return ((DB_TYPE_GENERIC_TV == theType) || (DB_TYPE_GENERIC_WL == theType) || (DB_TYPE_GENERIC_FM == theType));
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Indicate if TV baseline record searches can be performed, see ExtDbRecordTV.  CDBS imports always had the
	// baseline table.  In LMS those were added in import version 2.  Of course always there on the LMS live server.

	public boolean hasBaseline() {

		return ((DB_TYPE_CDBS == type) || ((DB_TYPE_LMS == type) && (version > 1)) || (DB_TYPE_LMS_LIVE == type));
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Retrieve an object by key, return null on error.  If the key is not found that is an error; since records are
	// never removed from the database just marked deleted, not-found should never occur since other code should never
	// encounter undefined keys.  Also if the data is deleted that is an error unless the findDeleted flag is true.
	// This will not return a data set with an unsupported version, that is also an error.

	public static ExtDb getExtDb(String theDbID, Integer theKey) {
		return getExtDb(theDbID, theKey, false, null);
	}

	public static ExtDb getExtDb(String theDbID, Integer theKey, ErrorLogger errors) {
		return getExtDb(theDbID, theKey, false, errors);
	}

	public static ExtDb getExtDb(String theDbID, Integer theKey, boolean findDeleted) {
		return getExtDb(theDbID, theKey, findDeleted, null);
	}

	public static synchronized ExtDb getExtDb(String theDbID, Integer theKey, boolean findDeleted,
			ErrorLogger errors) {

		if (KEY_LMS_LIVE == theKey.intValue()) {
			return getLMSLiveExtDb(theDbID, errors);
		}

		HashMap<Integer, ExtDb> theMap = getCache(theDbID, errors);
		if (null == theMap) {
			return null;
		}

		ExtDb result = theMap.get(theKey);

		if (null == result) {
			theMap = mrCache.get(theDbID);
			if (null != theMap) {
				result = theMap.get(theKey);
			}
		}

		if (null == result) {
			if (null != errors) {
				errors.reportError("Invalid station data key");
			}
			return null;
		}

		if (result.deleted && !findDeleted) {
			if (null != errors) {
				errors.reportWarning("The station data has been deleted");
			}
			return null;
		}

		if (!result.isSupported()) {
			if (null != errors) {
				errors.reportWarning("The station data version is not supported");
			}
			return null;
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Retrieve an object by name, no index by name so this is a linear search but this is an uncommon lookup so
	// performance not a big concern.  This only has to check non-deleted sets since names are cleared on deleted sets.
	// Returns null on error or not found, not-found is not an error.  Will not return an unsupported set, also does
	// not recognize the most-recent names or the live server names, only imported sets.

	public static ExtDb getExtDb(String theDbID, String theName) {
		return getExtDb(theDbID, theName, null);
	}

	public static synchronized ExtDb getExtDb(String theDbID, String theName, ErrorLogger errors) {

		HashMap<Integer, ExtDb> theMap = getCache(theDbID, errors);
		if (null == theMap) {
			return null;
		}

		ExtDb result = null;
		for (ExtDb theDb : theMap.values()) {
			if (!theDb.deleted && theDb.name.equalsIgnoreCase(theName)) {
				result = theDb;
				break;
			}
		}

		if ((null != result) && !result.isSupported()) {
			if (null != errors) {
				errors.reportWarning("The station data version is not supported");
			}
			return null;
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get the type name or description for an external data set key, regardless of whether the data set is deleted or
	// not.  These do not fail, they will return an empty string if the key is not found or errors occur.  Used for
	// non-critical labelling only.  Live server objects may be unavailable due to connection failure and so not appear
	// in the cache, recognize those directly and return an appropriate string.

	public static String getExtDbTypeName(String theDbID, Integer theKey) {

		ExtDb theDb = getExtDb(theDbID, theKey, true, null);
		if (null == theDb) {
			if (KEY_LMS_LIVE == theKey.intValue()) {
				return "LMS";
			}
			return "";
		}
		return getTypeName(theDb.type);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public static String getExtDbDescription(String theDbID, Integer theKey) {

		ExtDb theDb = getExtDb(theDbID, theKey, true, null);
		if (null == theDb) {
			if (KEY_LMS_LIVE == theKey.intValue()) {
				return getReservedKeyName(KEY_LMS_LIVE) + " (offline)";
			}
			return "";
		}
		return theDb.description;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get a full list of non-deleted ExtDb objects from a database.  See getExtDbList() for comments about sorting.
	// Returns null on error.  This can optionally include non-deleted sets with unsupported version, but does not
	// include live server objects.

	public static ArrayList<ExtDb> getExtDbs(String theDbID, boolean includeUnsupported) {
		return getExtDbs(theDbID, includeUnsupported, null);
	}

	public static synchronized ArrayList<ExtDb> getExtDbs(String theDbID, boolean includeUnsupported,
			ErrorLogger errors) {

		HashMap<Integer, ExtDb> theMap = getCache(theDbID, errors);
		if (null == theMap) {
			return null;
		}

		ArrayList<ExtDb> result = new ArrayList<ExtDb>();
		for (ExtDb theExtDb : theMap.values()) {
			if (!theExtDb.deleted && (includeUnsupported || theExtDb.isSupported())) {
				result.add(theExtDb);
			}
		}

		Collections.sort(result, new Comparator<ExtDb>() {
			public int compare(ExtDb one, ExtDb two) {
				if (DB_TYPE_ORDER[one.type - 1] < DB_TYPE_ORDER[two.type - 1]) {
					return -1;
				} else {
					if (DB_TYPE_ORDER[one.type - 1] > DB_TYPE_ORDER[two.type - 1]) {
						return 1;
					} else {
						if (one.key > two.key) {
							return -1;
						} else {
							if (one.key < two.key) {
								return 1;
							}
						}
					}
				}
				return 0;
			}
		});

		return result;
	}


	//=================================================================================================================
	// A KeyedRecord subclass used to facilitate sorting in getExtDbList.

	private static class DbListItem extends KeyedRecord {

		private final int type;


		//-------------------------------------------------------------------------------------------------------------

		private DbListItem(int theKey, String theName, int theType) {

			super(theKey, theName);

			type = theType;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get a KeyedRecord list of non-deleted data sets in a specified database, updating the cache first.  The list is
	// usually filtered by record type, returning only data sets containing the specified type.  Optionally generic
	// data sets can be excluded.  Alternately, the list can be filtered by database type and version.  Either form
	// with 0 for recordType or dbType will return all, version 0 will return all.  Requesting the LMS database type
	// will also match the LMS live type.  Live server items are included if active, most-recent items may optionally
	// be included.  Returns null on error.  Does not include sets with unsupported version.

	public static ArrayList<KeyedRecord> getExtDbList(String theDbID, int recordType) {
		return getExtDbList(theDbID, recordType, 0, 0, true, true, null);
	}

	public static ArrayList<KeyedRecord> getExtDbList(String theDbID, int recordType, ErrorLogger errors) {
		return getExtDbList(theDbID, recordType, 0, 0, true, true, errors);
	}

	public static ArrayList<KeyedRecord> getExtDbList(String theDbID, int recordType, boolean includeGeneric,
			boolean includeMostRecent) {
		return getExtDbList(theDbID, recordType, 0, 0, includeGeneric, includeMostRecent, null);
	}

	public static ArrayList<KeyedRecord> getExtDbList(String theDbID, int recordType, boolean includeGeneric,
			boolean includeMostRecent, ErrorLogger errors) {
		return getExtDbList(theDbID, recordType, 0, 0, includeGeneric, includeMostRecent, errors);
	}

	public static ArrayList<KeyedRecord> getExtDbList(String theDbID, int dbType, int minVersion) {
		return getExtDbList(theDbID, 0, dbType, minVersion, true, true, null);
	}

	public static ArrayList<KeyedRecord> getExtDbList(String theDbID, int dbType, int minVersion, ErrorLogger errors) {
		return getExtDbList(theDbID, 0, dbType, minVersion, true, true, errors);
	}

	private static synchronized ArrayList<KeyedRecord> getExtDbList(String theDbID, int recordType, int dbType,
			int minVersion, boolean includeGeneric, boolean includeMostRecent, ErrorLogger errors) {

		HashMap<Integer, ExtDb> theMap = getCache(theDbID, errors);
		if (null == theMap) {
			return null;
		}

		ArrayList<DbListItem> dbList = new ArrayList<DbListItem>();
		for (ExtDb theDb : theMap.values()) {
			if (!theDb.deleted && (includeGeneric || !theDb.isGeneric()) &&
					((0 == recordType) || theDb.canProvide(recordType)) &&
					((0 == dbType) || (dbType == theDb.type)) &&
					(theDb.version >= minVersion) && theDb.isSupported()) {
				dbList.add(new DbListItem(theDb.key.intValue(), theDb.description, theDb.type));
			}
		}

		// Live servers are always included if active and the criteria match.

		ExtDb theDb;
		if (((0 == recordType) || (Source.RECORD_TYPE_TV == recordType)) &&
				((0 == dbType) || (DB_TYPE_LMS == dbType) || (DB_TYPE_LMS_LIVE == dbType))) {
			theDb = getLMSLiveExtDb(theDbID, errors);
			if (null != theDb) {
				dbList.add(new DbListItem(theDb.key.intValue(), theDb.description, theDb.type));
			}
		}
				
		// This is the only context from which objects having the most-recent pseudo-keys are returned, those may
		// appear in UI lists but are resolved to real objects on lookup with getExtDb().  Check each of the objects
		// appearing in the most-recent cache map against filtering criteria, add a most-recent item as needed.  Some
		// criteria don't apply; the most-recent objects cannot be deleted or generic sets, or unsupported version.

		if (includeMostRecent) {
			theMap = mrCache.get(theDbID);
			if (null != theMap) {
				int theKey;
				for (Map.Entry<Integer, ExtDb> e : theMap.entrySet()) {
					theDb = e.getValue();
					if (((0 == recordType) || theDb.canProvide(recordType)) &&
							((0 == dbType) || (dbType == theDb.type)) &&
							(theDb.version >= minVersion)) {
						theKey = e.getKey().intValue();
						dbList.add(new DbListItem(theKey, getReservedKeyName(theKey), theDb.type));
					}
				}
			}
		}

		// The list is sorted first by database type using an ordering map, second by keys being in the reserved range
		// or not, third for reserved by ascending order of key, else by descending order of key.

		Comparator<DbListItem> comp = new Comparator<DbListItem>() {
			public int compare(DbListItem one, DbListItem two) {
				if (DB_TYPE_ORDER[one.type - 1] < DB_TYPE_ORDER[two.type - 1]) {
					return -1;
				} else {
					if (DB_TYPE_ORDER[one.type - 1] > DB_TYPE_ORDER[two.type - 1]) {
						return 1;
					} else {
						if ((one.key >= RESERVED_KEY_RANGE_START) && (one.key <= RESERVED_KEY_RANGE_END)) {
							if ((two.key >= RESERVED_KEY_RANGE_START) && (two.key <= RESERVED_KEY_RANGE_END)) {
								if (one.key < two.key) {
									return -1;
								} else {
									if (one.key > two.key) {
										return 1;
									}
								}
							} else {
								return -1;
							}
						} else {
							if ((two.key >= RESERVED_KEY_RANGE_START) && (two.key <= RESERVED_KEY_RANGE_END)) {
								return 1;
							} else {
								if (one.key > two.key) {
									return -1;
								} else {
									if (one.key < two.key) {
										return 1;
									}
								}
							}
						}
					}
				}
				return 0;
			}
		};

		Collections.sort(dbList, comp);

		return new ArrayList<KeyedRecord>(dbList);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Add and remove objects that are notified when cache is reloaded for a database.

	private static HashMap<String, HashSet<ExtDbListener>> listeners = new HashMap<String, HashSet<ExtDbListener>>();

	public static synchronized void addListener(ExtDbListener theListener) {

		String theDbID = theListener.getDbID();
		HashSet<ExtDbListener> theListeners = listeners.get(theDbID);
		if (null == theListeners) {
			theListeners = new HashSet<ExtDbListener>();
			listeners.put(theDbID, theListeners);
		}
		theListeners.add(theListener);
	}


	//-----------------------------------------------------------------------------------------------------------------

	public static synchronized void removeListener(ExtDbListener theListener) {

		String theDbID = theListener.getDbID();
		HashSet<ExtDbListener> theListeners = listeners.get(theDbID);
		if (null != theListeners) {
			theListeners.remove(theListener);
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Check a data set name for validity.  An existing name may be provided for comparison.  If the new name matches
	// the old name ignoring case the change is allowed without any further checks.  A reserved character is used as
	// part of a suffix that may be added to make the name unique (see createNewDatabase()), that character cannot be
	// used in a new name.  But since uniqueness is case-insensitive, changes to an existing name that only affect case
	// will not violate uniqueness and so can be allowed while preserving an existing uniqueness suffix.  If the new
	// name does not match or there is no old name, the new name is checked for length, for the reserved character used
	// in the uniqueness suffix, and for reserved names (most-recent, live server, unnamed download).  An empty new
	// name is allowed.  Finally, the name is checked against the existing data set cache for uniqueness (only non-
	// deleted sets need to be checked since the name is cleared when a set is deleted).  Note by the time the actual
	// database create occurs it's possible concurrent actions will result in the name no longer being unique, but that
	// is the case when the uniqueness suffix will be added.

	public static boolean checkExtDbName(String dbID, String newName) {
		return checkExtDbName(dbID, newName, null, null);
	}

	public static boolean checkExtDbName(String dbID, String newName, ErrorLogger errors) {
		return checkExtDbName(dbID, newName, null, errors);
	}

	public static boolean checkExtDbName(String dbID, String newName, String oldName) {
		return checkExtDbName(dbID, newName, oldName, null);
	}

	public static boolean checkExtDbName(String dbID, String newName, String oldName, ErrorLogger errors) {

		if ((null != oldName) && (oldName.length() > 0)) {
			if (newName.equalsIgnoreCase(oldName)) {
				return true;
			}
		}

		if (0 == newName.length()) {
			return true;
		}

		if (newName.length() > DbCore.NAME_MAX_LENGTH) {
			if (null != errors) {
				errors.reportWarning("The station data name cannot be more than " + DbCore.NAME_MAX_LENGTH +
					" characters long");
			}
			return false;
		}

		if (newName.contains(String.valueOf(DbCore.NAME_UNIQUE_CHAR))) {
			if (null != errors) {
				errors.reportWarning("The station data name cannot contain the character '" + DbCore.NAME_UNIQUE_CHAR +
					"'");
			}
			return false;
		}

		for (String resName : reservedKeyNames.values()) {
			if (newName.equalsIgnoreCase(resName)) {
				if (null != errors) {
					errors.reportWarning("That station data name cannot be used, please try again");
				}
				return false;
			}
		}

		if (newName.equalsIgnoreCase(DOWNLOAD_SET_NAME)) {
			if (null != errors) {
				errors.reportWarning("That station data name cannot be used, please try again");
			}
			return false;
		}

		if (null != getExtDb(dbID, newName, errors)) {
			if (null != errors) {
				errors.reportWarning("That station data name is already in use, please try again");
			}
			return false;
		}

		return true;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Do an immediate cache reload for a database, and notify listeners.

	public static synchronized void reloadCache(String theDbID) {

		if (null == updateCache(theDbID, null)) {
			return;
		}

		HashSet<ExtDbListener> theListeners = listeners.get(theDbID);
		if (null != theListeners) {
			for (ExtDbListener theListener : theListeners) {
				theListener.updateExtDbList();
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get cache map for a database, update if needed based on time since last cache update, or if there is no cache
	// for the requested database.

	private static HashMap<Integer, ExtDb> getCache(String theDbID, ErrorLogger errors) {

		HashMap<Integer, ExtDb> theMap = null;

		Long lastUpdate = cacheLastUpdate.get(theDbID);
		if ((null != lastUpdate) && ((System.currentTimeMillis() - lastUpdate.longValue()) < CACHE_UPDATE_INTERVAL)) {
			theMap = dbCache.get(theDbID);
		}

		if (null == theMap) {
			theMap = updateCache(theDbID, errors);
		}

		return theMap;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update the cache of objects for a given database, returns null on error.  Note because rows in the ext_db table
	// are not ever deleted this does not empty and reload the cache, it updates existing content by key.  A separate
	// set of maps is used to look up psuedo-keys for the "most-recent" functions.

	private static HashMap<String, HashMap<Integer, ExtDb>> dbCache = new HashMap<String, HashMap<Integer, ExtDb>>();
	private static HashMap<String, HashMap<Integer, ExtDb>> mrCache = new HashMap<String, HashMap<Integer, ExtDb>>();
	private static HashMap<String, Long> cacheLastUpdate = new HashMap<String, Long>();

	private static HashMap<Integer, ExtDb> updateCache(String theDbID, ErrorLogger errors) {

		HashMap<Integer, ExtDb> theMap = dbCache.get(theDbID);
		HashMap<Integer, ExtDb> mrMap = mrCache.get(theDbID);

		// If a cache for this dbID does not exist create one.

		if (null == theMap) {

			theMap = new HashMap<Integer, ExtDb>();
			mrMap = new HashMap<Integer, ExtDb>();

			dbCache.put(theDbID, theMap);
			mrCache.put(theDbID, mrMap);
		}

		// Run the lookup query, add/update objects as needed.  The query is ordered by descending key value so the
		// data sets imported later are loaded earlier.  This is for the most-recent logic, if two data sets have the
		// same date the one imported later is preferred and needs to be encountered first.

		String rootName = DbCore.getDbName(theDbID);

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null != db) {
			try {

				db.query("SELECT ext_db_key, db_type, db_date, version, index_version, id, name, deleted, " +
					"is_download, locked, bad_data FROM ext_db ORDER BY 1 DESC");

				Integer theKey, mrKey;
				ExtDb theExtDb, mrExtDb;
				int theType;
				String theDbName = "";

				// Remove any existing most-recent objects, new ones will be added as seen.

				mrMap.remove(Integer.valueOf(KEY_MOST_RECENT_LMS));
				mrMap.remove(Integer.valueOf(KEY_MOST_RECENT_CDBS));
				mrMap.remove(Integer.valueOf(KEY_MOST_RECENT_CDBS_FM));

				while (db.next()) {

					theKey = Integer.valueOf(db.getInt(1));
					theExtDb = theMap.get(theKey);

					if (null == theExtDb) {

						theType = db.getInt(2);
						theDbName = makeDbName(rootName, theType, theKey);

						theExtDb = new ExtDb(
							theDbID,
							theKey,
							theDbName,
							db.getDate(3),
							theType,
							db.getInt(4),
							db.getInt(5),
							false);

						theMap.put(theExtDb.key, theExtDb);
					}

					theExtDb.id = db.getString(6);
					if (null == theExtDb.id) {
						theExtDb.id = "";
					}

					theExtDb.name = db.getString(7);
					if (null == theExtDb.name) {
						theExtDb.name = "";
					}

					theExtDb.deleted = db.getBoolean(8);
					theExtDb.isDownload = db.getBoolean(9);
					theExtDb.isLocked = db.getBoolean(10);
					theExtDb.hasBadData = db.getBoolean(11);

					if (theExtDb.deleted) {
						theExtDb.description = theExtDb.getTypeName() + " " + theExtDb.id + " (deleted)";
					} else {
						if (!theExtDb.isSupported()) {
							theExtDb.name = theExtDb.name + " (unsupported)";
							theExtDb.description = theExtDb.getTypeName() + " " + theExtDb.id + " (unsupported)";
						} else {
							if (theExtDb.name.length() > 0) {
								theExtDb.description = theExtDb.getTypeName() + " " + theExtDb.name;
							} else {
								theExtDb.description = theExtDb.getTypeName() + " " + theExtDb.id;
							}
							if (theExtDb.hasBadData) {
								theExtDb.description = theExtDb.description + " (bad)";
							}
						}
					}

					// Update most-recent objects as needed, if the new object is not deleted, and is supported.  Also
					// don't consider this if the bad-data flag is set, see checkImport().

					if (!theExtDb.deleted && theExtDb.isSupported() && !theExtDb.hasBadData) {

						switch (theExtDb.type) {

							case DB_TYPE_LMS: {
								mrKey = Integer.valueOf(KEY_MOST_RECENT_LMS);
								mrExtDb = mrMap.get(mrKey);
								if ((null == mrExtDb) || theExtDb.dbDate.after(mrExtDb.dbDate)) {
									mrMap.put(mrKey, theExtDb);
								}
								break;
							}

							case DB_TYPE_CDBS: {
								mrKey = Integer.valueOf(KEY_MOST_RECENT_CDBS);
								mrExtDb = mrMap.get(mrKey);
								if ((null == mrExtDb) || theExtDb.dbDate.after(mrExtDb.dbDate)) {
									mrMap.put(mrKey, theExtDb);
								}
								break;
							}

							case DB_TYPE_CDBS_FM: {
								mrKey = Integer.valueOf(KEY_MOST_RECENT_CDBS_FM);
								mrExtDb = mrMap.get(mrKey);
								if ((null == mrExtDb) || theExtDb.dbDate.after(mrExtDb.dbDate)) {
									mrMap.put(mrKey, theExtDb);
								}
								break;
							}
						}
					}
				}

				DbCore.releaseDb(db);

				cacheLastUpdate.put(theDbID, Long.valueOf(System.currentTimeMillis()));

			} catch (SQLException se) {
				DbCore.releaseDb(db);
				theMap = null;
				DbConnection.reportError(errors, se);
			}

		} else {
			theMap = null;
		}

		return theMap;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Get an object wrapping a connection to a "live" LMS server.  The ExtDb objects are specific to a dbID and are
	// cached, however the actual DbConnection, and objects to manage a connection pool, are static and are shared by
	// all of the ExtDb objects.  On first use, this will attempt to read server login properties from a local file
	// and open the initial connection.  If the properties file can't be opened or does not contain all necessary
	// properties this fails silently; however if a connection is attempted failures will be reported.  In any case if
	// the read or connect fails it is not attempted again.  This should only be called from synchronized methods.
	// Note the key assigned to the ExtDb object is constant and the version is always the current version.

	private static HashMap<String, ExtDb> lmsLiveDbCache = new HashMap<String, ExtDb>();

	private static boolean lmsLiveDidTryOpen;

	private static String lmsLiveDbName;
	private static DbConnection lmsLiveDb;
	private static ArrayDeque<DbConnection> lmsLiveDbPool;
	private static HashSet<DbConnection> lmsLiveOpenDbs;

	private static ExtDb getLMSLiveExtDb(String theDbID, ErrorLogger errors) {

		ExtDb theExtDb = lmsLiveDbCache.get(theDbID);
		if (null != theExtDb) {
			return theExtDb;
		}

		if (!lmsLiveDidTryOpen) {

			lmsLiveDidTryOpen = true;

			Properties props = new Properties();
			try {
				props.load(Files.newInputStream(AppCore.libDirectoryPath.resolve(AppCore.API_PROPS_FILE_NAME)));
			} catch (IOException e) {
				return null;
			}

			String theDriver = props.getProperty("lms_driver");
			String theHost = props.getProperty("lms_host");
			lmsLiveDbName = props.getProperty("lms_name");
			String theUser = props.getProperty("lms_user");
			String thePass = props.getProperty("lms_pass");

			if ((null == theDriver) || (null == theHost) || (null == lmsLiveDbName) || (null == theUser) ||
					(null == thePass)) {
				return null;
			}

			DbConnection db = new DbConnection(theDriver, theHost, theUser, thePass);
			if (db.connect(lmsLiveDbName)) {

				db.close();

				lmsLiveDb = db;

				lmsLiveDbPool = new ArrayDeque<DbConnection>();
				lmsLiveDbPool.push(lmsLiveDb);
				lmsLiveOpenDbs = new HashSet<DbConnection>();

			} else {
				if (null != errors) {
					errors.reportError("Cannot open live LMS connection, properties may be invalid");
				}
			}
		}

		if (null != lmsLiveDb) {

			theExtDb = new ExtDb(theDbID, Integer.valueOf(KEY_LMS_LIVE), lmsLiveDbName, new java.util.Date(),
				DB_TYPE_LMS_LIVE, LMS_VERSION, 0, true);

			theExtDb.id = "";
			theExtDb.name = getReservedKeyName(KEY_LMS_LIVE);
			theExtDb.description = theExtDb.name;

			theExtDb.liveDb = lmsLiveDb;
			theExtDb.dbPool = lmsLiveDbPool;
			theExtDb.openDbs = lmsLiveOpenDbs;

			lmsLiveDbCache.put(theDbID, theExtDb);
		}

		return theExtDb;
	}


	//=================================================================================================================
	// Item for antenna pattern search results.

	public static class PatternID {

		public ExtDb extDb;
		public String antennaRecordID;
		public int type;
		public boolean isMatrix;

		public String name;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Search for patterns in a data set, normally horizontal patterns, argument can request vertical pattern which
	// will also match matrix patterns.  If the search string converts to a positive number match the numeric antenna
	// ID in CDBS, the source key in generic data sets, or the descriptive ID in LMS.  Always match both the make and
	// model fields for CDBS/LMS, or the appropriate name field(s) for generic.  All the matches are combined with OR.
	// The queries for CDBS/LMS always join to the pattern tabulation data table so this will only match patterns that
	// actually have data.  Also the query runs a summation of the tabulation values and uses that as part of a key
	// along with the name to determine uniqueness, to eliminate duplicates.  For LMS and generic the search may be
	// restricted by record type, that argument may be 0 to match all types.  That has no effect with CDBS, it would be
	// difficult to implement and CDBS is now obsolete, having only minimal legacy support.  This returns null on
	// error, an empty array if no match.

	public static ArrayList<PatternID> findPatterns(String theDbID, Integer extDbKey, String search,
			boolean searchVertical, int recordType, ErrorLogger errors) {

		ExtDb extDb = getExtDb(theDbID, extDbKey, errors);
		if (null == extDb) {
			return null;
		}

		// Compose the query.

		String str = DbConnection.clean(search.trim().toUpperCase().replace('*', '%'));
		if (0 == str.length()) {
			return new ArrayList<PatternID>();
		}

		int id = 0;
		try {
			id = Integer.parseInt(search);
		} catch (NumberFormatException ne) {
		}

		StringBuilder query = new StringBuilder();

		switch (extDb.type) {

			default: {
				if (null != errors) {
					errors.reportError("Unknown or unsupported station data type");
				}
				return null;
			}

			case DB_TYPE_CDBS:
			case DB_TYPE_CDBS_FM: {

				String fld = "", mfld = "", ufld = "", tbl = "", jtbl = "";
				if (searchVertical) {
					fld = "elevation_antenna_id";
					mfld = "field_value0";
					ufld = "depression_angle * (field_value + field_value0)";
					tbl = "elevation_ant_make";
					jtbl = "elevation_pattern";
				} else {
					fld = "antenna_id";
					mfld = "field_value";
					ufld = "azimuth * field_value";
					tbl = "ant_make";
					jtbl = "ant_pattern";
				}

				query.append("SELECT ");
				query.append(fld);
				query.append(", ant_make, ant_model_num, MAX(");
				query.append(mfld);
				query.append("), SUM(");
				query.append(ufld);
				query.append(") FROM ");
				query.append(tbl);
				query.append(" JOIN ");
				query.append(jtbl);
				query.append(" USING (");
				query.append(fld);
				query.append(") WHERE ");

				if (id > 0) {
					query.append("(");
					query.append(fld);
					query.append(" = ");
					query.append(String.valueOf(id));
					query.append(") OR ");
				}
				query.append("(UPPER(ant_make) LIKE '%");
				query.append(str);
				query.append("%') OR (UPPER(ant_model_num) LIKE '%");
				query.append(str);
				query.append("%') GROUP BY 1, 2, 3");

				break;
			}

			case DB_TYPE_LMS:
			case DB_TYPE_LMS_LIVE: {

				String mfld = "", ufld = "", jtbl = "", jfld = "";
				if (searchVertical) {
					mfld = "aaep_azimuth";
					ufld = "aaep_depression_angle * aaep_field_value";
					jtbl = "app_antenna_elevation_pattern";
					jfld = "aaep_antenna_record_id";
				} else {
					mfld = "aafv_field_value";
					ufld = "aafv_azimuth * aafv_field_value";
					jtbl = "app_antenna_field_value";
					jfld = "aafv_aant_antenna_record_id";
				}

				query.append("SELECT aant_antenna_record_id, aant_make, aant_model, MAX(");
				query.append(mfld);
				query.append("), SUM(");
				query.append(ufld);
				query.append(") FROM ");
				if (DB_TYPE_LMS_LIVE == extDb.type) {
					query.append("mass_media.");
				}
				query.append("app_antenna JOIN ");
				if (DB_TYPE_LMS_LIVE == extDb.type) {
					query.append("mass_media.");
				}
				query.append(jtbl);
				query.append(" ON (");
				query.append(jfld);
				query.append(" = aant_antenna_record_id) JOIN ");
				if (DB_TYPE_LMS_LIVE == extDb.type) {
					query.append("mass_media.");
				}
				query.append("app_location ON (aloc_loc_record_id = aant_aloc_loc_record_id) JOIN ");
				if (DB_TYPE_LMS_LIVE == extDb.type) {
					query.append("common_schema.");
				}
				query.append("license_filing_version ON (filing_version_id = aloc_aapp_application_id) WHERE ");
				query.append("((CASE WHEN (purpose_code = 'AMD') THEN original_purpose_code ELSE purpose_code END) ");
				query.append("IN ('CP','L2C','MOD','RUL','STA')) AND (current_status_code <> 'SAV') AND ");
				query.append("(service_code IN ");
				ExtDbRecord.addServiceCodeList(extDb.type, recordType, query);
				query.append(") AND (");

				if (id > 0) {
					query.append("(aant_antenna_id = '");
					query.append(String.valueOf(id));
					query.append("') OR ");
				}
				query.append("(UPPER(aant_make) LIKE '%");
				query.append(str);
				query.append("%') OR (UPPER(aant_model) LIKE '%");
				query.append(str);
				query.append("%')) GROUP BY 1, 2, 3");

				break;
			}

			// For generic data sets, stored in the Source record format, match the source_key if the search string is
			// numeric (that is the antennaRecordID for these), or the appropriate name by substring match.

			case DB_TYPE_GENERIC_TV:
			case DB_TYPE_GENERIC_WL:
			case DB_TYPE_GENERIC_FM: {

				String nameFld = "horizontal_pattern_name";
				if (searchVertical) {
					nameFld = "(CASE WHEN has_matrix_pattern THEN matrix_pattern_name ELSE vertical_pattern_name END)";
				}

				query.append("SELECT source_key, '', ");
				query.append(nameFld);
				if (searchVertical) {
					query.append(", (CASE WHEN has_matrix_pattern THEN 1. ELSE 0. END), ");
					query.append("SUM(CASE WHEN has_matrix_pattern THEN ");
					query.append("(source_matrix_pattern.depression_angle * ");
					query.append("source_matrix_pattern.relative_field) ELSE ");
					query.append("(source_vertical_pattern.depression_angle * ");
					query.append("source_vertical_pattern.relative_field) END)");
				} else {
					query.append(", 0., SUM(azimuth * relative_field)");
				}
				query.append(" FROM source ");
				if (searchVertical) {
					query.append("LEFT JOIN source_vertical_pattern USING (source_key) ");
					query.append("LEFT JOIN source_matrix_pattern USING (source_key) ");
				} else {
					query.append("JOIN source_horizontal_pattern USING (source_key) ");
				}
				query.append("WHERE ");
				if (recordType > 0) {
					query.append("(record_type = ");
					query.append(String.valueOf(recordType));
					query.append(") AND ");
				}

				if (searchVertical) {
					query.append("(has_matrix_pattern OR has_vertical_pattern) AND (");
				} else {
					query.append("has_horizontal_pattern AND (");
				}
				if (id > 0) {
					query.append("(source_key = ");
					query.append(str);
					query.append(") OR ");
				}
				query.append("(UPPER(");
				query.append(nameFld);
				query.append(") LIKE '%");
				query.append(str);
				query.append("%')) GROUP BY 1, 2, 3");

				break;
			}
		}

		// Do the search.

		TreeMap<String, PatternID> theItems = new TreeMap<String, PatternID>();

		String theName, theKey;
		PatternID theItem;

		DbConnection db = extDb.connectDb(errors);
		if (null != db) {
			try {

				db.query(query.toString());

				while (db.next()) {

					theName = db.getString(2);
					if (theName.length() > 0) {
						theName = theName + "-" + db.getString(3);
					} else {
						theName = db.getString(3);
					}
					theKey = theName + "-" + db.getString(5);

					if (!theItems.containsKey(theKey)) {
						theItem = new PatternID();
						theItem.extDb = extDb;
						theItem.antennaRecordID = db.getString(1);
						theItem.name = theName;
						if (searchVertical) {
							theItem.type = AntPattern.PATTERN_TYPE_VERTICAL;
							theItem.isMatrix = (db.getDouble(4) > 0.);
						} else {
							theItem.type = AntPattern.PATTERN_TYPE_HORIZONTAL;
						}
						theItems.put(theKey, theItem);
					}
				}

				extDb.releaseDb(db);

			} catch (SQLException se) {
				extDb.releaseDb(db);
				DbConnection.reportError(errors, se);
				return null;
			}

		} else {
			return null;
		}

		return new ArrayList<PatternID>(theItems.values());
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Load an antenna pattern from a search-results object.

	public static AntPattern getPattern(PatternID theAnt, ErrorLogger errors) {

		if (theAnt.isMatrix) {
			return getMatrixPattern(null, theAnt.extDb, theAnt.antennaRecordID, theAnt.name, errors);
		} else {
			if (AntPattern.PATTERN_TYPE_VERTICAL == theAnt.type) {
				return getVerticalPattern(null, theAnt.extDb, theAnt.antennaRecordID, "", theAnt.name, errors);
			} else {
				return getHorizontalPattern(null, theAnt.extDb, theAnt.antennaRecordID, theAnt.name, errors);
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Retrieve a horizontal pattern from a data set.  The public form is used by ExtDbRecord subclasses when building
	// source objects, the private version is also used from getPattern() above.  Returns null if the pattern is not
	// found or has any bad data.  A not-found condition is not an error, non-directional patterns also appear in the
	// pattern index, so the caller must check the error logger state to interpret a null return.  Bad data checks
	// include a minimum number of points, azimuths and relative fields in range, and no duplicate points.  Values are
	// also rounded, see PatternEditor.  Relative field values less than a minimum are set to the minimum.  The
	// ExtDbRecord form can also load patterns from baseline table records in data sets that have those, and it also
	// changes error handling, in that case some failures are logged as messages using identifying record information.

	public static AntPattern getHorizontalPattern(ExtDbRecord theRecord, String theName, ErrorLogger errors) {
		return getHorizontalPattern(theRecord, theRecord.extDb, theRecord.antennaRecordID, theName, errors);
	}

	private static AntPattern getHorizontalPattern(ExtDbRecord theRecord, ExtDb theExtDb, String theAntRecordID,
			String theName, ErrorLogger errors) {

		String query = "";

		switch (theExtDb.type) {

			case DB_TYPE_CDBS:
			case DB_TYPE_CDBS_FM: {

				query = 
				"SELECT " +
					"azimuth, " +
					"field_value " +
				"FROM " +
					"ant_pattern " +
				"WHERE " +
					"antenna_id = " + theAntRecordID + " " +
				"ORDER BY 1";

				break;
			}

			// LMS antenna ID is defined in the lkp_antenna table for baseline records, except for DTS records which
			// are actually normal records pulled in by reference from a baseline entry.

			case DB_TYPE_LMS: {

				if ((null != theRecord) && (Source.RECORD_TYPE_TV == theRecord.recordType) &&
						((ExtDbRecordTV)theRecord).isBaseline && !((ExtDbRecordTV)theRecord).service.isDTS) {

					query =
					"SELECT " +
						"lkp_antenna_field_value.rafv_azimuth, " +
						"lkp_antenna_field_value.rafv_field_value " +
					"FROM " +
						"lkp_antenna " +
						"JOIN lkp_antenna_field_value ON (lkp_antenna_field_value.rafv_antenna_record_id = " +
							"lkp_antenna.rant_antenna_record_id) " +
					"WHERE " +
						"lkp_antenna.rant_antenna_id = '" + theAntRecordID + "' " +
					"ORDER BY 1";

				} else {

					query =
					"SELECT " +
						"aafv_azimuth, " +
						"aafv_field_value " +
					"FROM " +
						"app_antenna_field_value " +
					"WHERE " +
						"aafv_aant_antenna_record_id = '" + theAntRecordID + "' " +
					"ORDER BY 1";
				}

				break;
			}

			case DB_TYPE_LMS_LIVE: {

				if ((null != theRecord) && (Source.RECORD_TYPE_TV == theRecord.recordType) &&
						((ExtDbRecordTV)theRecord).isBaseline && !((ExtDbRecordTV)theRecord).service.isDTS) {

					query =
					"SELECT " +
						"(CASE WHEN TRIM(lkp_antenna_field_value.rafv_azimuth) IN ('','null') THEN 0::FLOAT " +
							"ELSE lkp_antenna_field_value.rafv_azimuth::FLOAT END), " +
						"(CASE WHEN TRIM(lkp_antenna_field_value.rafv_field_value) IN ('','null') THEN 0::FLOAT " +
							" ELSE lkp_antenna_field_value.rafv_field_value::FLOAT END) " +
					"FROM " +
						"mass_media.lkp_antenna " +
						"JOIN mass_media.lkp_antenna_field_value ON " +
							"(lkp_antenna_field_value.rafv_antenna_record_id = " +
								"lkp_antenna.rant_antenna_record_id) " +
					"WHERE " +
						"lkp_antenna.rant_antenna_id = '" + theAntRecordID + "' " +
					"ORDER BY 1";

				} else {

					query =
					"SELECT " +
						"(CASE WHEN TRIM(aafv_azimuth) IN ('','null') THEN 0::FLOAT ELSE aafv_azimuth::FLOAT END), " +
						"(CASE WHEN TRIM(aafv_field_value) IN ('','null') THEN 0::FLOAT " +
							"ELSE aafv_field_value::FLOAT END) " +
					"FROM " +
						"mass_media.app_antenna_field_value " +
					"WHERE " +
						"aafv_aant_antenna_record_id = '" + theAntRecordID + "' " +
					"ORDER BY 1";
				}

				break;
			}

			case DB_TYPE_GENERIC_TV:
			case DB_TYPE_GENERIC_WL:
			case DB_TYPE_GENERIC_FM: {

				query = 
				"SELECT " +
					"azimuth, " +
					"relative_field " +
				"FROM " +
					"source_horizontal_pattern " +
				"WHERE " +
					"source_key = " + theAntRecordID + " " +
				"ORDER BY 1";

				break;
			}

			default: {
				return null;
			}
		}

		ArrayList<AntPattern.AntPoint> thePoints = new ArrayList<AntPattern.AntPoint>();

		boolean badData = false;
		String errmsg = null;
		double patMax = AntPattern.FIELD_MIN;

		DbConnection db = theExtDb.connectDb(errors);
		if (null != db) {
			try {

				db.query(query);

				double az, pat, lastAz = AntPattern.AZIMUTH_MIN - 1.;

				while (db.next()) {

					az = Math.rint(db.getDouble(1) * AntPattern.AZIMUTH_ROUND) / AntPattern.AZIMUTH_ROUND;
					if ((az < AntPattern.AZIMUTH_MIN) || (az > AntPattern.AZIMUTH_MAX)) {
						badData = true;
						errmsg = "azimuth out of range";
						break;
					}
					if (az <= lastAz) {
						badData = true;
						errmsg = "duplicate azimuths";
						break;
					}
					lastAz = az;

					pat = Math.rint(db.getDouble(2) * AntPattern.FIELD_ROUND) / AntPattern.FIELD_ROUND;
					if (pat < AntPattern.FIELD_MIN) {
						pat = AntPattern.FIELD_MIN;
					}
					if (pat > AntPattern.FIELD_MAX) {
						badData = true;
						errmsg = "relative field out of range";
						break;
					}
					if (pat > patMax) {
						patMax = pat;
					}

					thePoints.add(new AntPattern.AntPoint(az, pat));
				}

				// Final data checks, must be enough points.  If the pattern maximum is too low a value don't use the
				// data, otherwise if it is not 1.0 (or very close) a warning message will be logged.

				if (!badData && (thePoints.size() < AntPattern.PATTERN_REQUIRED_POINTS)) {
					badData = true;
					errmsg = "not enough points";
				}

				if (!badData && (patMax < 0.5)) {
					badData = true;
					errmsg = "max value is too small";
				}

				theExtDb.releaseDb(db);

			} catch (SQLException se) {
				theExtDb.releaseDb(db);
				DbConnection.reportError(errors, se);
				return null;
			}
		}

		if (thePoints.isEmpty()) {
			return null;
		}

		if (badData) {
			if (null != errors) {
				String msg = "Pattern for antenna record ID " + theAntRecordID + " is bad, " + errmsg;
				if (null != theRecord) {
					errors.logMessage(ExtDbRecord.makeMessage(theRecord, msg));
				} else {
					errors.reportError(msg);
				}
			}
			return null;
		}

		// Show warning message if there is no peak value.

		if ((null != errors) && (patMax < AntPattern.FIELD_MAX_CHECK)) {
			String msg = "Pattern for antenna record ID " + theAntRecordID + " does not have a 1";
			if (null != theRecord) {
				msg = ExtDbRecord.makeMessage(theRecord, msg);
			}
			errors.logMessage(msg);
		}

		return new AntPattern(theExtDb.dbID, AntPattern.PATTERN_TYPE_HORIZONTAL, theName, thePoints);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Retrieve a vertical pattern.  Bad data checks similar to horizontal above.  This is also used during load of a
	// matrix pattern from CDBS to retrieve the 10-degree-increment columns that are in the main pattern table, that is
	// determined by fieldSuffix, an empty string gets the normal elevation pattern, "0", "10", ... will get the matrix
	// pattern columns.  That argument is ignored for LMS; earlier checks will have already determined that the LMS
	// elevation pattern table does not contain a matrix pattern else this would not be called.

	public static AntPattern getVerticalPattern(ExtDbRecord theRecord, String theName, ErrorLogger errors) {
		return getVerticalPattern(theRecord, theRecord.extDb, theRecord.elevationAntennaRecordID, "", theName, errors);
	}

	private static AntPattern getVerticalPattern(ExtDbRecord theRecord, ExtDb theExtDb, String theAntRecordID,
			String fieldSuffix, String theName, ErrorLogger errors) {

		String query = "";
		boolean isCDBS = false;

		switch (theExtDb.type) {

			case DB_TYPE_CDBS:
			case DB_TYPE_CDBS_FM: {

				query = 
				"SELECT " +
					"depression_angle," +
					"field_value" + fieldSuffix + " " +
				"FROM " +
					"elevation_pattern " +
				"WHERE " +
					"elevation_antenna_id = " + theAntRecordID + " " +
				"ORDER BY 1";

				isCDBS = true;

				break;
			}

			case DB_TYPE_LMS: {

				query =
				"SELECT " +
					"aaep_depression_angle," +
					"aaep_field_value " +
				"FROM " +
					"app_antenna_elevation_pattern " +
				"WHERE " +
					"(aaep_antenna_record_id = '" + theAntRecordID + "') " +
					"AND (aaep_azimuth = 0.) " +
				"ORDER BY 1";

				break;
			}

			case DB_TYPE_LMS_LIVE: {

				query =
				"SELECT " +
					"aaep_depression_angle," +
					"aaep_field_value " +
				"FROM " +
					"mass_media.app_antenna_elevation_pattern " +
				"WHERE " +
					"(aaep_antenna_record_id = '" + theAntRecordID + "') " +
					"AND ((aaep_azimuth = 0.) OR (aaep_azimuth IS NULL)) " +
				"ORDER BY 1";

				break;
			}

			case DB_TYPE_GENERIC_TV:
			case DB_TYPE_GENERIC_WL:
			case DB_TYPE_GENERIC_FM: {

				query = 
				"SELECT " +
					"depression_angle, " +
					"relative_field " +
				"FROM " +
					"source_vertical_pattern " +
				"WHERE " +
					"source_key = " + theAntRecordID + " " +
				"ORDER BY 1";

				break;
			}

			default: {
				return null;
			}
		}

		ArrayList<AntPattern.AntPoint> thePoints = new ArrayList<AntPattern.AntPoint>();

		boolean badData = false, allZero = true;
		String errmsg = null;
		double patMax = AntPattern.FIELD_MIN;

		DbConnection db = theExtDb.connectDb(errors);
		if (null != db) {
			try {

				db.query(query);

				double dep, pat, lastDep = AntPattern.DEPRESSION_MIN - 1.;

				while (db.next()) {

					dep = Math.rint(db.getDouble(1) * AntPattern.DEPRESSION_ROUND) / AntPattern.DEPRESSION_ROUND;
					if ((dep < AntPattern.DEPRESSION_MIN) || (dep > AntPattern.DEPRESSION_MAX)) {
						badData = true;
						errmsg = "vertical angle out of range";
						break;
					}
					if (dep <= lastDep) {
						badData = true;
						errmsg = "duplicate vertical angles";
						break;
					}
					lastDep = dep;

					pat = Math.rint(db.getDouble(2) * AntPattern.FIELD_ROUND) / AntPattern.FIELD_ROUND;
					if (pat < AntPattern.FIELD_MIN) {
						pat = AntPattern.FIELD_MIN;
					} else {
						allZero = false;
					}
					if (pat > AntPattern.FIELD_MAX) {
						badData = true;
						errmsg = "relative field out of range";
						break;
					}
					if (pat > patMax) {
						patMax = pat;
					}

					thePoints.add(new AntPattern.AntPoint(dep, pat));
				}

				if (!badData && (thePoints.size() < AntPattern.PATTERN_REQUIRED_POINTS)) {
					badData = true;
					errmsg = "not enough points";
				}

				theExtDb.releaseDb(db);

			} catch (SQLException se) {
				theExtDb.releaseDb(db);
				DbConnection.reportError(errors, se);
				return null;
			}
		}

		// Here, an empty result may be an error; in CDBS, elevation pattern keys are never just identifiers.  But this
		// is not an error in LMS as all records have a common antenna identifier which may or may not have pattern
		// data associated.  Also if the data column is all zeros treat that as a not-found condition, that will occur
		// when a matrix pattern is loaded as a normal pattern or vice-versa.

		if (thePoints.isEmpty() || allZero) {
			if (isCDBS) {
				badData = true;
				errmsg = "pattern data not found";
			} else {
				return null;
			}
		}

		if (badData) {
			if (null != errors) {
				String msg = "Pattern for elevation antenna record ID " + theAntRecordID + " is bad, " + errmsg;
				if (null != theRecord) {
					errors.logMessage(ExtDbRecord.makeMessage(theRecord, msg));
				} else {
					errors.reportError(msg);
				}
			}
			return null;
		}

		// Peak value check is skipped when loading individual matrix slices from CDBS, the whole pattern will be
		// checked once all slices are loaded.

		if ((null != errors) && fieldSuffix.equals("") && (patMax < AntPattern.FIELD_MAX_CHECK)) {
			String msg = "Pattern for elevation antenna record ID " + theAntRecordID + " does not have a 1";
			if (null != theRecord) {
				msg = ExtDbRecord.makeMessage(theRecord, msg);
			}
			errors.logMessage(msg);
		}

		return new AntPattern(theExtDb.dbID, AntPattern.PATTERN_TYPE_VERTICAL, theName, thePoints);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Retrieve a per-azimuth vertical pattern, a.k.a. a matrix pattern.  Similar to vertical pattern retrieval, but
	// iterated.  Additional checks on the azimuths similar to horizontal pattern retrieval.  See comments above.

	public static AntPattern getMatrixPattern(ExtDbRecord theRecord, String theName, ErrorLogger errors) {
		return getMatrixPattern(theRecord, theRecord.extDb, theRecord.elevationAntennaRecordID, theName, errors);
	}

	private static AntPattern getMatrixPattern(ExtDbRecord theRecord, ExtDb theExtDb, String theAntRecordID,
			String theName, ErrorLogger errors) {

		// For CDBS, start by getting the standard 10-degree azimuth slices from the main elevation pattern table, this
		// is handled by the private version of getVerticalPattern(), see above.

		ArrayList<AntPattern.AntSlice> theSlices = new ArrayList<AntPattern.AntSlice>();

		AntPattern thePattern = null;
		AntPattern.AntSlice theSlice = null;
		double patMax = AntPattern.FIELD_MIN;

		if ((DB_TYPE_CDBS == theExtDb.type) || (DB_TYPE_CDBS_FM == theExtDb.type)) {
			for (int iaz = 0; iaz < 360; iaz += 10) {
				thePattern = getVerticalPattern(null, theExtDb, theAntRecordID, String.valueOf(iaz), "", errors);
				if (null == thePattern) {
					return null;
				}
				theSlice = new AntPattern.AntSlice((double)iaz, thePattern.getPoints());
				theSlices.add(theSlice);
				for (AntPattern.AntPoint thePoint : theSlice.points) {
					if (thePoint.relativeField > patMax) {
						patMax = thePoint.relativeField;
					}
				}
			}
		}

		// Now a query on the additional-points pattern table for CDBS, or the main elevation pattern table for LMS.
		// In CDBS this may provide additional full pattern slices at azimuths other than the 10-degree set, and/or it
		// may contain additional depression angles for the 10-degree slices.  For LMS this is the only query and
		// provides all data.  Likewise for generic sets where this queries the separate matrix pattern table.

		String query = "";

		switch (theExtDb.type) {

			case DB_TYPE_CDBS:
			case DB_TYPE_CDBS_FM: {

				query =
				"SELECT " +
					"azimuth," +
					"depression_angle," +
					"field_value " +
				"FROM " +
					"elevation_pattern_addl " +
				"WHERE " +
					"elevation_antenna_id = " + theAntRecordID + " " +
				"ORDER BY 1, 2";

				break;
			}

			case DB_TYPE_LMS: {

				query =
				"SELECT " +
					"aaep_azimuth," +
					"aaep_depression_angle," +
					"aaep_field_value " +
				"FROM " +
					"app_antenna_elevation_pattern " +
				"WHERE " +
					"aaep_antenna_record_id = '" + theAntRecordID + "' " +
				"ORDER BY 1, 2";

				break;
			}

			case DB_TYPE_LMS_LIVE: {

				query =
				"SELECT " +
					"aaep_azimuth," +
					"aaep_depression_angle," +
					"aaep_field_value " +
				"FROM " +
					"mass_media.app_antenna_elevation_pattern " +
				"WHERE " +
					"aaep_antenna_record_id = '" + theAntRecordID + "' " +
				"ORDER BY 1, 2";

				break;
			}

			case DB_TYPE_GENERIC_TV:
			case DB_TYPE_GENERIC_WL:
			case DB_TYPE_GENERIC_FM: {

				query = 
				"SELECT " +
					"azimuth, " +
					"depression_angle, " +
					"relative_field " +
				"FROM " +
					"source_matrix_pattern " +
				"WHERE " +
					"source_key = " + theAntRecordID + " " +
				"ORDER BY 1, 2";

				break;
			}

			default: {
				return null;
			}
		}

		boolean badData = false;
		String errmsg = null;

		DbConnection db = theExtDb.connectDb(errors);
		if (null != db) {
			try {

				db.query(query);

				ArrayList<AntPattern.AntPoint> thePoints = null;
				AntPattern.AntPoint thePoint, newPoint;

				double az, dep, pat, lastAz = AntPattern.AZIMUTH_MIN - 1., lastDep = AntPattern.DEPRESSION_MIN - 1.;
				boolean newpat = false;
				int i;

				while (db.next()) {

					az = Math.rint(db.getDouble(1) * AntPattern.AZIMUTH_ROUND) / AntPattern.AZIMUTH_ROUND;
					if ((az < AntPattern.AZIMUTH_MIN) || (az > AntPattern.AZIMUTH_MAX)) {
						badData = true;
						errmsg = "azimuth out of range";
						break;
					}

					if (az != lastAz) {

						if (newpat && (thePoints.size() < AntPattern.PATTERN_REQUIRED_POINTS)) {
							badData = true;
							errmsg = "not enough points, at azimuth " + theSlice.value;
							break;
						}

						newpat = true;
						for (i = 0; i < theSlices.size(); i++) {
							theSlice = theSlices.get(i);
							if (az == theSlice.value) {
								newpat = false;
								break;
							}
							if (az < theSlice.value) {
								break;
							}
						}

						if (newpat) {
							theSlice = new AntPattern.AntSlice(az);
							theSlices.add(i, theSlice);
						}
						thePoints = theSlice.points;

						lastAz = az;
						lastDep = AntPattern.DEPRESSION_MIN - 1.;
					}

					dep = Math.rint(db.getDouble(2) * AntPattern.DEPRESSION_ROUND) / AntPattern.DEPRESSION_ROUND;
					if ((dep < AntPattern.DEPRESSION_MIN) || (dep > AntPattern.DEPRESSION_MAX)) {
						badData = true;
						errmsg = "vertical angle out of range, at azimuth " + theSlice.value;
						break;
					}
					if (dep <= lastDep) {
						badData = true;
						errmsg = "duplicate vertical angles, at azimuth " + theSlice.value;
						break;
					}
					lastDep = dep;

					pat = Math.rint(db.getDouble(3) * AntPattern.FIELD_ROUND) / AntPattern.FIELD_ROUND;
					if (pat < AntPattern.FIELD_MIN) {
						pat = AntPattern.FIELD_MIN;
					}
					if (pat > AntPattern.FIELD_MAX) {
						badData = true;
						errmsg = "field value greater than 1, at azimuth " + theSlice.value;
						break;
					}
					if (pat > patMax) {
						patMax = pat;
					}

					newPoint = new AntPattern.AntPoint(dep, pat);

					if (newpat) {

						thePoints.add(newPoint);

					} else {

						for (i = 0; i < thePoints.size(); i++) {
							thePoint = thePoints.get(i);
							if (dep == thePoint.angle) {
								badData = true;
								errmsg = "duplicate vertical angles, at azimuth " + theSlice.value;
								break;
							}
							if (dep < thePoint.angle) {
								break;
							}
						}
						if (badData) {
							break;
						}

						thePoints.add(i, newPoint);
					}
				}

				if (!badData && newpat && (thePoints.size() < AntPattern.PATTERN_REQUIRED_POINTS)) {
					badData = true;
					errmsg = "not enough points, at azimuth " + theSlice.value;
				}

				if (!badData && (patMax < 0.5)) {
					badData = true;
					errmsg = "max value is too small";
				}

				theExtDb.releaseDb(db);

			} catch (SQLException se) {
				theExtDb.releaseDb(db);
				DbConnection.reportError(errors, se);
				return null;
			}
		}

		if (theSlices.isEmpty()) {
			return null;
		}

		if (badData) {
			if (null != errors) {
				String msg = "Pattern for elevation antenna record ID " + theAntRecordID + " is bad, " + errmsg;
				if (null != theRecord) {
					errors.logMessage(ExtDbRecord.makeMessage(theRecord, msg));
				} else {
					errors.reportError(msg);
				}
			}
			return null;
		}

		if ((null != errors) && (patMax < AntPattern.FIELD_MAX_CHECK)) {
			String msg = "Pattern for elevation antenna record ID " + theAntRecordID + " does not have a 1";
			if (null != theRecord) {
				msg = ExtDbRecord.makeMessage(theRecord, msg);
			}
			errors.logMessage(msg);
		}

		return new AntPattern(theExtDb.dbID, theName, AntPattern.PATTERN_TYPE_VERTICAL, theSlices);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Do an auxiliary search for AM stations within specified distance of a set of coordinates (or a coordinates list,
	// to check a DTS facility), append results to a text report.  Used by automated-run study types e.g. interference
	// check.  It is here rather than in the study engine because it needs direct access to the external station data
	// tables.  This does not fail; if the data set does not support AM, or if any error happens during the query, this
	// simply writes in the report that the check could not be performed.

	public static void checkForAMStations(ExtDb extDb, GeoPoint targetLocation, ArrayList<GeoPoint> targetLocations,
			double searchDistanceND, double searchDistanceDA, double kilometersPerDegree, StringBuilder report) {

		boolean error = false;

		DbConnection db = extDb.connectDb();
		if (null == db) {
			error = true;
		} else {

			String dir, call, freq, stat, mode, hours, city, state, file;
			StringBuilder repND = new StringBuilder(), repDA = new StringBuilder();
			GeoPoint point = new GeoPoint();
			boolean isDA, foundND = false, foundDA = false;

			try {

				switch (extDb.type) {

					case DB_TYPE_CDBS: {
						db.query(
						"SELECT " +
							"am_ant_sys.lat_dir," +
							"am_ant_sys.lat_deg," +
							"am_ant_sys.lat_min," +
							"am_ant_sys.lat_sec," +
							"am_ant_sys.lon_dir," +
							"am_ant_sys.lon_deg," +
							"am_ant_sys.lon_min," +
							"am_ant_sys.lon_sec," +
							"facility.fac_callsign, " +
							"application.fac_frequency, " +
							"am_ant_sys.am_dom_status," +
							"am_ant_sys.ant_mode," +
							"am_ant_sys.hours_operation," +
							"facility.comm_city, " +
							"facility.comm_state, " +
							"CONCAT(application.file_prefix, application.app_arn) " +
						"FROM " +
							"am_ant_sys " +
							"JOIN application USING (application_id) " +
							"JOIN facility USING (facility_id) " +
						"WHERE " +
							"(am_ant_sys.eng_record_type NOT IN ('P','A','R')) " +
						"ORDER BY " +
							"15, 14, 9, 16");
						break;
					}

					// As of LMS import version 10, AM records are in primary LMS tables and the legacy gis_* tables
					// that contained a copy of CDBS are obsolete.  Note an older version doesn't necessarily have the
					// gis_* tables, those were always optional and are no longer imported even if an old LMS archive
					// with those files is re-imported.

					case DB_TYPE_LMS: {

						if (extDb.version > 9) {

							db.query(
							"SELECT DISTINCT " +
								"app_am_antenna.lat_dir," +
								"app_am_antenna.lat_deg," +
								"app_am_antenna.lat_min," +
								"app_am_antenna.lat_sec," +
								"app_am_antenna.long_dir," +
								"app_am_antenna.long_deg," +
								"app_am_antenna.long_min," +
								"app_am_antenna.long_sec," +
								"facility.callsign," +
								"application_facility.am_frequency," +
								"(CASE WHEN (license_filing_version.purpose_code = 'AMD') " +
									"THEN license_filing_version.original_purpose_code " +
									"ELSE license_filing_version.purpose_code END)," +
								"app_am_antenna.dir_ind," +
								"app_am_antenna.am_ant_mode_code," +
								"(CASE WHEN (facility.community_served_city <> '') " +
									"THEN facility.community_served_city " +
									"ELSE application_facility.afac_community_city END)," +
								"(CASE WHEN (facility.community_served_state <> '') " +
									"THEN facility.community_served_state " +
									"ELSE application_facility.afac_community_state_code END)," +
								"application.aapp_file_num " +
							"FROM " +
								"app_am_antenna " +
								"JOIN license_filing_version ON (license_filing_version.filing_version_id = " +
									"app_am_antenna.aapp_application_id) " +
								"JOIN application_facility ON (application_facility.afac_application_id = " +
									"app_am_antenna.aapp_application_id) " +
								"JOIN facility ON (facility.facility_id = application_facility.afac_facility_id) " +
								"JOIN application ON (application.aapp_application_id = " +
									"app_am_antenna.aapp_application_id) " +
							"WHERE " +
								"(license_filing_version.active_ind = 'Y') AND " +
								"((CASE WHEN (license_filing_version.purpose_code = 'AMD') " +
									"THEN license_filing_version.original_purpose_code " +
									"ELSE license_filing_version.purpose_code END) IN ('CP','L2C','MOD','STA')) AND " +
								"(license_filing_version.current_status_code <> 'SAV') AND " +
								"(license_filing_version.service_code = 'AM') AND " +
								"(application_facility.country_code = 'US')" +
							"ORDER BY " +
								"15, 14, 9, 16");

						} else {

							db.query(
							"SELECT " +
								"gis_am_ant_sys.lat_dir," +
								"gis_am_ant_sys.lat_deg," +
								"gis_am_ant_sys.lat_min," +
								"gis_am_ant_sys.lat_sec," +
								"gis_am_ant_sys.lon_dir," +
								"gis_am_ant_sys.lon_deg," +
								"gis_am_ant_sys.lon_min," +
								"gis_am_ant_sys.lon_sec," +
								"gis_facility.fac_callsign," +
								"gis_application.fac_frequency," +
								"gis_am_ant_sys.am_dom_status," +
								"gis_am_ant_sys.ant_mode," +
								"gis_am_ant_sys.hours_operation," +
								"gis_facility.comm_city," +
								"gis_facility.comm_state," +
								"CONCAT(gis_application.file_prefix, gis_application.app_arn) " +
							"FROM " +
								"gis_am_ant_sys " +
								"JOIN gis_application USING (application_id) " +
								"JOIN gis_facility USING (facility_id) " +
							"WHERE " +
								"(gis_am_ant_sys.eng_record_type NOT IN ('P','A','R')) " +
							"ORDER BY " +
								"15, 14, 9, 16");
						}

						break;
					}

					case DB_TYPE_LMS_LIVE: {

						db.query(
						"SELECT " +
							"app_am_antenna.lat_dir," +
							"(CASE WHEN TRIM(app_am_antenna.lat_deg) IN ('','null') THEN 0::FLOAT " +
								"ELSE app_am_antenna.lat_deg::FLOAT END), " +
							"(CASE WHEN TRIM(app_am_antenna.lat_min) IN ('','null') THEN 0::FLOAT " +
								"ELSE app_am_antenna.lat_min::FLOAT END), " +
							"(CASE WHEN TRIM(app_am_antenna.lat_sec) IN ('','null') THEN 0::FLOAT " +
								"ELSE app_am_antenna.lat_sec::FLOAT END), " +
							"app_am_antenna.long_dir," +
							"(CASE WHEN TRIM(app_am_antenna.long_deg) IN ('','null') THEN 0::FLOAT " +
								"ELSE app_am_antenna.long_deg::FLOAT END), " +
							"(CASE WHEN TRIM(app_am_antenna.long_min) IN ('','null') THEN 0::FLOAT " +
								"ELSE app_am_antenna.long_min::FLOAT END), " +
							"(CASE WHEN TRIM(app_am_antenna.long_sec) IN ('','null') THEN 0::FLOAT " +
								"ELSE app_am_antenna.long_sec::FLOAT END), " +
							"facility.callsign," +
							"application_facility.am_frequency," +
							"(CASE WHEN (license_filing_version.purpose_code = 'AMD') " +
								"THEN license_filing_version.original_purpose_code " +
								"ELSE license_filing_version.purpose_code END)," +
							"app_am_antenna.dir_ind," +
							"app_am_antenna.am_ant_mode_code," +
							"(CASE WHEN (facility.community_served_city <> '') " +
								"THEN facility.community_served_city " +
								"ELSE application_facility.afac_community_city END)," +
							"(CASE WHEN (facility.community_served_state <> '') " +
								"THEN facility.community_served_state " +
								"ELSE application_facility.afac_community_state_code END)," +
							"application.aapp_file_num " +
						"FROM " +
							"mass_media.app_am_antenna " +
							"JOIN common_schema.license_filing_version ON " +
								"(license_filing_version.filing_version_id = app_am_antenna.aapp_application_id) " +
							"JOIN common_schema.application_facility ON (application_facility.afac_application_id = " +
								"app_am_antenna.aapp_application_id) " +
							"JOIN common_schema.facility ON (facility.facility_id = " +
								"application_facility.afac_facility_id) " +
							"JOIN common_schema.application ON (application.aapp_application_id = " +
								"app_am_antenna.aapp_application_id) " +
						"WHERE " +
							"(license_filing_version.active_ind = 'Y') AND " +
							"((CASE WHEN (license_filing_version.purpose_code = 'AMD') " +
								"THEN license_filing_version.original_purpose_code " +
								"ELSE license_filing_version.purpose_code END) IN ('CP','L2C','MOD','STA')) AND " +
							"(license_filing_version.current_status_code <> 'SAV') AND " +
							"(license_filing_version.service_code = 'AM') AND " +
							"(application_facility.country_code = 'US')" +
						"ORDER BY " +
							"15, 14, 9, 16");

						break;
					}

					default: {
						error = true;
						break;
					}
				}

				if (!error) {

					double dist, dist1;

					while (db.next()) {

						point.latitudeNS = 0;
						dir = db.getString(1);
						if ((null != dir) && dir.equalsIgnoreCase("S")) {
							point.latitudeNS = 1;
						}
						point.latitudeDegrees = db.getInt(2);
						point.latitudeMinutes = db.getInt(3);
						point.latitudeSeconds = db.getDouble(4);

						point.longitudeWE = 0;
						dir = db.getString(5);
						if ((null != dir) && dir.equalsIgnoreCase("E")) {
							point.longitudeWE = 1;
						}
						point.longitudeDegrees = db.getInt(6);
						point.longitudeMinutes = db.getInt(7);
						point.longitudeSeconds = db.getDouble(8);

						point.updateLatLon();
						if ((0. == point.latitude) || (0. == point.longitude)) {
							continue;
						}
						if ((DB_TYPE_CDBS == extDb.type) || (extDb.version < 10)) {
							point.convertFromNAD27();
							point.updateDMS();
						}

						call = db.getString(9);
						if (null == call) {
							continue;
						}
						freq = db.getString(10);
						if (null == freq) {
							continue;
						}
						stat = db.getString(11);
						if (null == stat) {
							continue;
						}
						mode = db.getString(12);
						if (null == mode) {
							continue;
						}
						isDA = false;
						if ((DB_TYPE_CDBS == extDb.type) || (extDb.version < 10)) {
							isDA = mode.startsWith("DA");
						} else {
							isDA = mode.equals("Y");
						}
						hours = db.getString(13);
						if (null == hours) {
							continue;
						}
						city = db.getString(14);
						if (null == city) {
							continue;
						}
						state = db.getString(15);
						if (null == state) {
							continue;
						}
						file = db.getString(16);
						if (null == file) {
							continue;
						}


						if ((null != targetLocations) && !targetLocations.isEmpty()) {
							dist = 9999.;
							for (GeoPoint location : targetLocations) {
								dist1 = location.distanceTo(point, kilometersPerDegree);
								if (dist1 < dist) {
									dist = dist1;
								}
							}
						} else {
							dist = targetLocation.distanceTo(point, kilometersPerDegree);
						}

						if (isDA) {

							if (dist <= searchDistanceDA) {
								if (!foundDA) {
									repDA.append(String.format(Locale.US,
										"Directional AM stations within %.1f km:\n", searchDistanceDA));
									foundDA = true;
								}
								repDA.append(call + " " + freq + " " + stat + " " + hours + " " + city + ", " + state +
									" " + file + ", " + AppCore.formatDistance(dist) + " km\n");
							}

						} else {

							if (dist <= searchDistanceND) {
								if (!foundND) {
									repND.append(String.format(Locale.US,
										"Non-directional AM stations within %.1f km:\n", searchDistanceND));
									foundND = true;
								}
								repND.append(call + " " + freq + " " + stat + " " + hours + " " + city + ", " + state +
									" " + file + ", " + AppCore.formatDistance(dist) + " km\n");
							}
						}
					}

					if (foundND) {
						repND.append("\n");
					} else {
						repND.append(String.format(Locale.US,
							"No non-directional AM stations found within %.1f km\n\n", searchDistanceND));
					}
					report.append(repND);

					if (foundDA) {
						repDA.append("\n");
					} else {
						repDA.append(String.format(Locale.US,
							"No directional AM stations found within %.1f km\n\n", searchDistanceDA));
					}
					report.append(repDA);
				}

			} catch (SQLException se) {
				DbConnection.reportError(se);
				error = true;
			}

			extDb.releaseDb(db);
		}

		if (error) {
			report.append("Data is not available for AM station check\n\n");
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Create a new generic import data set database, which stores data in a table structure created by the Source
	// class, identical to what is used in an actual study database.  Searches are managed by SourceEditData rather
	// than ExtDbRecord.  The database is initially empty, the caller will populate it by creating SourceEditData
	// objects and saving those directly.  It is possible to add to a generic data set with multiple imports so the
	// API for import is separate, see below.

	public static Integer createNewGenericDatabase(String theDbID, int dataType, String theName) {
		return createNewGenericDatabase(theDbID, dataType, theName, null);
	}

	public static Integer createNewGenericDatabase(String theDbID, int dataType, String theName, ErrorLogger errors) {

		if (!isGeneric(dataType)) {
			if (null != errors) {
				errors.reportError("Cannot create station data, unknown or unsupported data type");
			}
			return null;
		}

		// Open database connection, lock tables, get a new key for the data set.

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			return null;
		}

		String errmsg = null, theDbName = null, rootName = DbCore.getDbName(theDbID);
		boolean error = false;
		int newKey = 0;

		try {

			db.update("LOCK TABLES ext_db_key_sequence WRITE");

			db.update("UPDATE ext_db_key_sequence SET ext_db_key = ext_db_key + 1");
			db.query("SELECT ext_db_key FROM ext_db_key_sequence");
			db.next();
			newKey = db.getInt(1);

			// A range of keys is reserved for internally-composed objects, skip if needed.

			if (RESERVED_KEY_RANGE_START == newKey) {
				newKey = RESERVED_KEY_RANGE_END + 1;
				db.update("UPDATE ext_db_key_sequence SET ext_db_key = " + newKey);
			}

			db.update("UNLOCK TABLES");

			// Create the database and tables.

			theDbName = makeDbName(rootName, dataType, Integer.valueOf(newKey));
			db.update("CREATE DATABASE " + theDbName + " CHARACTER SET latin1");
			Source.createTables(db, theDbName);

			// Generate ID string and save the index record.

			String theID = AppCore.formatDateTime(new java.util.Date());

			if (null == theName) {
				theName = "";
			}

			db.setDatabase(rootName);

			// Check the ID and name for uniqueness, append the key as a suffix if needed.  Save the new index entry in
			// the ext_db table.

			db.update("LOCK TABLES ext_db WRITE");

			db.query("SELECT ext_db_key FROM ext_db WHERE (id = '" + db.clean(theID) + "')");
			if (db.next()) {
				theID = theID + " " + String.valueOf(DbCore.NAME_UNIQUE_CHAR) + String.valueOf(newKey);
			}

			if (theName.length() > 0) {
				db.query("SELECT ext_db_key FROM ext_db WHERE (UPPER(name) = '" +
					db.clean(theName.toUpperCase()) + "')");
				if (db.next()) {
					theName = theName + " " + String.valueOf(DbCore.NAME_UNIQUE_CHAR) + String.valueOf(newKey);
				}
			}

			db.update(
			"INSERT INTO ext_db (" +
				"ext_db_key, " +
				"db_type, " +
				"db_date, " +
				"version, " +
				"index_version, " +
				"id, " +
				"name, " +
				"deleted, " +
				"locked, " +
				"is_download, " +
				"bad_data) "+
			"VALUES (" +
				newKey + ", "  +
				dataType + ", " +
				"NOW(), " +
				"0, " +
				"0, " +
				"'" + db.clean(theID) + "', " +
				"'" + db.clean(theName) + "', " +
				"false, " +
				"false, " +
				"false, " +
				"false)");

		} catch (SQLException se) {
			errmsg = "A database error occurred:\n" + se;
			error = true;
			db.reportError(se);
		}

		// Make sure table locks are released, if an error occurred also drop the database.

		try {
			db.update("UNLOCK TABLES");
			if (error && (null != theDbName)) {
				db.update("DROP DATABASE IF EXISTS " + theDbName);
			}
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if ((null != errmsg) && (null != errors)) {
			if (error) {
				errors.reportError(errmsg);
			} else {
				errors.reportWarning(errmsg);
			}
		}

		if (error) {
			return null;
		}

		reloadCache(theDbID);

		return Integer.valueOf(newKey);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Import data into a generic data set database.  Returns the count of imported records, <0 on error.  Keys for
	// other data sets may be needed to resolve record references in the imported data, depending on file format.

	public static int importToGenericDatabase(ExtDb extDb, int importFormat, File importFile, File patternFile,
			ErrorLogger errors) {
		return importToGenericDatabase(extDb, importFormat, importFile, patternFile, null, null, null);
	}

	public static int importToGenericDatabase(ExtDb extDb, int importFormat, File importFile, File patternFile,
			Integer lookupKey, Integer altLookupKey, ErrorLogger errors) {

		if (!extDb.isGeneric()) {
			if (null != errors) {
				errors.reportError("Unknown or unsupported station data type");
			}
			return -1;
		}

		DbConnection db = extDb.connectAndLock(errors);
		if (null == db) {
			return -1;
		}

		int sourceCount = 0;

		switch (importFormat) {

			default: {
				if (null != errors) {
					errors.reportError("Unknown import file type");
				}
				sourceCount = -1;
				break;
			}

			// Text format, separate station data and pattern files.  Station data has different formats depending on
			// record type.  See SourceEditData.readFromText().

			case GENERIC_FORMAT_TEXT: {

				try {

					ArrayList<SourceEditData> sources = SourceEditData.readFromText(extDb, importFile, patternFile,
						errors);
					if (null == sources) {
						sourceCount = -1;
						break;
					}

					for (SourceEditData theSource : sources) {
						theSource.isDataChanged();
						theSource.save(db);
						sourceCount++;
					}

				} catch (SQLException se) {
					if (null != errors) {
						db.reportError(errors, se);
					}

				} catch (Throwable t) {
					AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", t);
					if (null != errors) {
						errors.reportError("An unexpected error occurred:\n" + t);
					}
					sourceCount = -1;
				}

				break;
			}

			// For Canadian DBF format, import is from a set of related files located in the same directory.  The file
			// argument may be the actual directory, if not it's a specific file in that directory but which one is
			// irrelevant, use the parent directory.

			case GENERIC_FORMAT_DBF: {

				File theDir = importFile;
				if (!theDir.isDirectory()) {
					theDir = theDir.getParentFile();
				}

				try {

					// Facility ID numbers are assigned to call signs as those are imported; build an index of
					// existing records already in the data set.

					HashMap<String, Integer> facIDs = new HashMap<String, Integer>();
					int facID, nextFacID = 0;
					db.query("SELECT facility_id, call_sign FROM source");
					while (db.next()) {
						facID = db.getInt(1);
						facIDs.put(db.getString(2), Integer.valueOf(facID));
						if (facID > nextFacID) {
							nextFacID = facID;
						}
					}
					nextFacID++;

					ArrayList<SourceEditData> sources = readFromDBF(extDb, theDir, facIDs, nextFacID, errors);
					if (null == sources) {
						sourceCount = -1;
						break;
					}

					for (SourceEditData theSource : sources) {
						theSource.isDataChanged();
						theSource.save(db);
						sourceCount++;
					}

				} catch (SQLException se) {
					if (null != errors) {
						db.reportError(errors, se);
					}

				} catch (Throwable t) {
					AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", t);
					if (null != errors) {
						errors.reportError("An unexpected error occurred:\n" + t);
					}
					sourceCount = -1;
				}

				break;
			}
		}

		extDb.releaseDb(db);

		return sourceCount;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Read from Canadian-format DBF files.  Wireless is not supported by this format.  Returns null on error.

	private static ArrayList<SourceEditData> readFromDBF(ExtDb extDb, File fileDir, HashMap<String, Integer> facIDs,
			int nextFacID, ErrorLogger errors) {

		if ((DB_TYPE_GENERIC_TV != extDb.type) && (DB_TYPE_GENERIC_FM != extDb.type)) {
			if (null != errors) {
				errors.reportError("Unsupported import format for station data type");
			}
			return null;
		}

		// Start by loading all data from files.

		AppCore.DBFData patLinksData = AppCore.readDBFData(new File(fileDir, "apatstat.dbf"), true, errors);
		if (null == patLinksData) {
			return null;
		}
		if (patLinksData.fieldCount < 2) {
			if (null != errors) {
				errors.reportError("Bad field count in data file '" + patLinksData.fileName + "'");
			}
			return null;
		}

		AppCore.DBFData patIndexData = AppCore.readDBFData(new File(fileDir, "apatdesc.dbf"), true, errors);
		if (null == patIndexData) {
			return null;
		}
		if (patIndexData.fieldCount < 7) {
			if (null != errors) {
				errors.reportError("Bad field count in data file '" + patIndexData.fileName + "'");
			}
			return null;
		}

		AppCore.DBFData patData = AppCore.readDBFData(new File(fileDir, "apatdat.dbf"), true, errors);
		if (null == patData) {
			return null;
		}
		if (patData.fieldCount < 3) {
			if (null != errors) {
				errors.reportError("Bad field count in data file '" + patData.fileName + "'");
			}
			return null;
		}

		File dbfFile = null;
		int minCount = 0;
		if (DB_TYPE_GENERIC_TV == extDb.type) {
			dbfFile = new File(fileDir, "tvstatio.dbf");
			minCount = 41;
		} else {
			dbfFile = new File(fileDir, "fmstatio.dbf");
			minCount = 37;
		}
		AppCore.DBFData stationData = AppCore.readDBFData(dbfFile, true, errors);
		if (null == stationData) {
			return null;
		}
		if (stationData.fieldCount < minCount) {
			if (null != errors) {
				errors.reportError("Bad field count in data file '" + stationData.fileName + "'");
			}
			return null;
		}

		// Build pattern data objects, these will be matched to records using a station key formed from the call sign
		// and banner fields, with trailing blanks removed from the call sign.  That key is used as the pattern name
		// for matching to records later.  First process the index of station keys to pattern keys.  Patterns not
		// linked to a station record are ignored.  Then create the empty pattern objects.

		HashMap<Integer, String> patLinks = new HashMap<Integer, String>();

		String[] values;
		String stationKey;
		int i;

		for (i = 0; i < patLinksData.fieldData.size(); i++) {
			values = patLinksData.fieldData.get(i);
			stationKey = values[0].substring(0, 12).trim().toUpperCase() + "-" + values[0].substring(12).toUpperCase();
			try {
				patLinks.put(Integer.valueOf(values[1]), stationKey);
			} catch (NumberFormatException ne) {
			}
		}

		HashMap<Integer, AntPattern> allPats = new HashMap<Integer, AntPattern>();

		Integer patKey;
		AntPattern pat;

		for (i = 0; i < patIndexData.fieldData.size(); i++) {
			values = patIndexData.fieldData.get(i);
			try {
				patKey = Integer.valueOf(values[0]);
				stationKey = patLinks.get(patKey);
				if (null != stationKey) {
					if (values[1].equalsIgnoreCase("V")) {
						pat = new AntPattern(extDb.dbID, AntPattern.PATTERN_TYPE_VERTICAL, stationKey);
					} else {
						pat = new AntPattern(extDb.dbID, AntPattern.PATTERN_TYPE_HORIZONTAL, stationKey);
					}
					allPats.put(patKey, pat);
				}
			} catch (NumberFormatException ne) {
			}
		}

		// Load the pattern data.  Usually the pattern data records are in order by pattern key and angle so this is
		// optimized for that case, but it works with any order.  If an error occurs in the data for a pattern, that
		// pattern is removed from the map so it's data is ignored.

		Integer lastKey = null;
		double ang, rf, lastAngle = 999.;
		ArrayList<AntPattern.AntPoint> points;
		AntPattern.AntPoint pt;
		int idx, badPatCount = 0;

		pat = null;
		points = null;

		for (i = 0; i < patData.fieldData.size(); i++) {

			values = patData.fieldData.get(i);
			patKey = null;

			try {

				patKey = Integer.valueOf(values[0]);
				if (!patKey.equals(lastKey)) {
					pat = allPats.get(patKey);
					if (null != pat) {
						points = pat.getPoints();
					} else {
						points = null;
					}
					lastKey = patKey;
					lastAngle = 999.;
				}
				if (null == pat) {
					continue;
				}

				ang = Double.parseDouble(values[1]);
				rf = Math.pow(10., Double.parseDouble(values[2]) / 20.);
				if (rf < AntPattern.FIELD_MIN) {
					rf = AntPattern.FIELD_MIN;
				}
				if (rf > AntPattern.FIELD_MAX) {
					rf = AntPattern.FIELD_MAX;
				}
				pt = new AntPattern.AntPoint(ang, rf);

				if (ang > lastAngle) {
					points.add(pt);
					lastAngle = ang;
				} else {
					for (idx = 0; idx < points.size(); idx++) {
						if (ang > points.get(idx).angle) {
							break;
						}
					}
					if (idx == points.size()) {
						lastAngle = ang;
					}
					points.add(idx, pt);
				}

			} catch (NumberFormatException ne) {
				if (null != patKey) {
					allPats.remove(patKey);
					pat = null;
					points = null;
				}
				badPatCount++;
			}
		}

		// Transfer patterns that have valid data to maps by station key, separate maps for azimuth and elevation
		// patterns, a given station key can have one of each.

		HashMap<String, AntPattern> hPats = new HashMap<String, AntPattern>();
		HashMap<String, AntPattern> vPats = new HashMap<String, AntPattern>();

		for (AntPattern p : allPats.values()) {
			if (p.isDataValid()) {
				if (AntPattern.PATTERN_TYPE_HORIZONTAL == p.type) {
					hPats.put(p.name, p);
				} else {
					vPats.put(p.name, p);
				}
			} else {
				badPatCount++;
			}
		}

		// Finally process the station records, build source objects.

		ArrayList<SourceEditData> result = new ArrayList<SourceEditData>();

		SourceEditData source = null;
		SourceEditDataTV sourceTV = null;
		SourceEditDataFM sourceFM = null;

		String callSign, banner, classStr, state, status;
		Integer facilityID;
		Country country;
		Service service;
		int j, channel, stationClass, statusType, ll, ld, lm, ls, badRecCount = 0;
		double lat, lon, erp;

		String importDate = AppCore.formatDate(new java.util.Date());

		for (i = 0; i < stationData.fieldData.size(); i++) {

			values = stationData.fieldData.get(i);

			try {

				stationKey = values[2].toUpperCase() + "-" + values[7].toUpperCase();

				callSign = values[2];

				facilityID = facIDs.get(callSign);
				if (null == facilityID) {
					facilityID = Integer.valueOf(nextFacID++);
					facIDs.put(callSign, facilityID);
				}

				banner = values[7].toUpperCase();

				classStr = values[4].toUpperCase();

				state = values[0].toUpperCase();
				if (state.equals("AB") || state.equals("BC") || state.equals("MB") ||
						state.equals("NB") || state.equals("NF") || state.equals("NS") ||
						state.equals("NT") || state.equals("NU") || state.equals("ON") ||
						state.equals("PE") || state.equals("QC") || state.equals("SK") ||
						state.equals("YT")) {
					country = Country.getCountry(Country.CA);
				} else {
					country = Country.getCountry(Country.US);
				}

				statusType = ExtDbRecord.STATUS_TYPE_OTHER;
				if (banner.equals("IC") || banner.equals("PC") || banner.equals("RE") || banner.equals("TP") ||
						banner.equals("UC") || banner.equals("UN") || banner.equals("UX")) {
					statusType = ExtDbRecord.STATUS_TYPE_APP;
				} else {
					if (banner.equals("AP") || banner.equals("AU") || banner.equals("CP")) {
						statusType = ExtDbRecord.STATUS_TYPE_CP;
					} else {
						if (banner.equals("AX") || banner.equals("OP")) {
							statusType = ExtDbRecord.STATUS_TYPE_LIC;
						} else {
							if (banner.equals("TO")) {
								statusType = ExtDbRecord.STATUS_TYPE_STA;
							}
						}
					}
				}
				if (ExtDbRecord.STATUS_TYPE_OTHER != statusType) {
					status = ExtDbRecord.STATUS_CODES[statusType];
				} else {
					status = banner;
				}

				if (DB_TYPE_GENERIC_TV == extDb.type) {

					channel = Integer.parseInt(values[40]);
					if (channel > 69) {
						continue;
					}

					if (0 == Integer.parseInt(values[36])) {
						if (banner.equals("AX") || banner.equals("UX")) {
							service = Service.getService(Service.TS_CODE);
						} else {
							if (classStr.equals("LP") || classStr.equals("VLP")) {
								service = Service.getService(Service.TX_CODE);
							} else {
								if (classStr.equals("A")) {
									service = Service.getService(Service.CA_CODE);
								} else {
									service = Service.getService(Service.TV_CODE);
								}
							}
						}
					} else {
						if (banner.equals("AX") || banner.equals("UX")) {
							service = Service.getService(Service.DX_CODE);
						} else {
							if (classStr.equals("LP") || classStr.equals("VLP")) {
								service = Service.getService(Service.LD_CODE);
							} else {
								if (classStr.equals("A")) {
									service = Service.getService(Service.DC_CODE);
								} else {
									service = Service.getService(Service.DT_CODE);
								}
							}
						}
					}

					sourceTV = SourceEditDataTV.createExtSource(extDb, facilityID.intValue(), service, false, country,
						errors);
					if (null == sourceTV) {
						return null;
					}
					source = sourceTV;

					sourceTV.appARN = "CANADA" + facilityID + "-" + banner;
					sourceTV.fileNumber = "BLANK" + sourceTV.appARN;

					sourceTV.channel = channel;

					sourceTV.status = status;
					sourceTV.statusType = statusType;

					sourceTV.frequencyOffset = FrequencyOffset.getFrequencyOffset(values[12]);
					sourceTV.emissionMask = EmissionMask.getDefaultObject();

					if (values[39].length() > 0) {
						sourceTV.heightAMSL = Double.parseDouble(values[39]);
					} else {
						sourceTV.heightAMSL = Source.HEIGHT_DERIVE;
					}
					if (values[30].length() > 0) {
						sourceTV.overallHAAT = Double.parseDouble(values[30]);
						if (sourceTV.overallHAAT < Source.HEIGHT_MIN) {
							sourceTV.overallHAAT = Source.HEIGHT_DERIVE;
						}
					} else {
						sourceTV.overallHAAT = Source.HEIGHT_DERIVE;
					}

					if (values[32].length() > 0) {
						sourceTV.peakERP = Double.parseDouble(values[32]) / 1000.;
					} else {
						sourceTV.peakERP = Source.ERP_DEF;
					}

					if (values[29].length() > 0) {
						sourceTV.verticalPatternElectricalTilt = Double.parseDouble(values[29]);
					}

				} else {

					channel = Integer.parseInt(values[36]);

					stationClass = ExtDbRecordFM.FM_CLASS_OTHER;

					if (banner.equals("AX") || banner.equals("UX")) {
						service = Service.getService(Service.FS_CODE);
					} else {
						if (classStr.equals("LP")) {
							service = Service.getService(Service.FL_CODE);
							stationClass = ExtDbRecordFM.FM_CLASS_L1;
						} else {
							if (classStr.equals("VLP")) {
								service = Service.getService(Service.FL_CODE);
								stationClass = ExtDbRecordFM.FM_CLASS_L2;
							} else {
								service = Service.getService(Service.FM_CODE);
								if (classStr.equals("A1")) {
									stationClass = ExtDbRecordFM.FM_CLASS_A;
								}
							}
						}
					}

					if (ExtDbRecordFM.FM_CLASS_OTHER == stationClass) {
						for (j = 1; j < ExtDbRecordFM.FM_CLASS_CODES.length; j++) {
							if (classStr.equals(ExtDbRecordFM.FM_CLASS_CODES[j])) {
								stationClass = j;
								break;
							}
						}
					}

					sourceFM = SourceEditDataFM.createExtSource(extDb, facilityID.intValue(), service, stationClass,
						country, errors);
					if (null == sourceFM) {
						return null;
					}
					source = sourceFM;

					sourceFM.appARN = "CANADA" + facilityID;
					sourceFM.fileNumber = "BLANK" + sourceFM.appARN;

					sourceFM.channel = channel;

					sourceFM.status = status;
					sourceFM.statusType = statusType;

					if (values[35].length() > 0) {
						sourceFM.heightAMSL = Double.parseDouble(values[35]);
					} else {
						sourceFM.heightAMSL = Source.HEIGHT_DERIVE;
					}
					if (values[28].length() > 0) {
						sourceFM.overallHAAT = Double.parseDouble(values[28]);
						if (sourceFM.overallHAAT < Source.HEIGHT_MIN) {
							sourceFM.overallHAAT = Source.HEIGHT_DERIVE;
						}
					} else {
						sourceFM.heightAMSL = Source.HEIGHT_DERIVE;
					}

					sourceFM.peakERP = Source.ERP_DEF;
					for (j = 29; j <= 32; j++) {
						if (values[j].length() > 0) {
							erp = Double.parseDouble(values[j]) / 1000.;
							if (erp > sourceFM.peakERP) {
								sourceFM.peakERP = erp;
							}
						}
					}

					if (values[27].length() > 0) {
						sourceFM.verticalPatternElectricalTilt = Double.parseDouble(values[27]);
					}
				}

				source.callSign = callSign;

				source.state = state;
				source.city = values[1];

				ll = Integer.parseInt(values[5]);
				ld = ll / 10000;
				lm = (ll % 10000) / 100;
				ls = ll % 100;
				lat = (double)ld + ((double)lm / 60.) + ((double)ls / 3600.);
				ll = Integer.parseInt(values[6]);
				ld = ll / 10000;
				lm = (ll % 10000) / 100;
				ls = ll % 100;
				lon = (double)ld + ((double)lm / 60.) + ((double)ls / 3600.);
				source.location.setLatLon(lat, lon);

				pat = hPats.get(stationKey);
				if (null != pat) {
					source.hasHorizontalPattern = true;
					source.horizontalPattern = pat;
					source.horizontalPatternChanged = true;
				}

				pat = vPats.get(stationKey);
				if (null != pat) {
					source.hasVerticalPattern = true;
					source.verticalPattern = pat;
					source.verticalPatternChanged = true;
				}

				source.setAttribute(Source.ATTR_SEQUENCE_DATE, importDate);

				if (source.isDataValid()) {
					result.add(source);
				} else {
					badRecCount++;
				}

			} catch (NumberFormatException ne) {
				badRecCount++;
			}
		}

		if ((null != errors) && ((badPatCount > 0) || (badRecCount > 0))) {
			errors.reportMessage("");
			if (badPatCount > 0) {
				errors.reportWarning(String.valueOf(badPatCount) + " antenna patterns were ignored due to bad data");
			}
			if (badRecCount > 0) {
				errors.reportWarning(String.valueOf(badRecCount) + " station records were ignored due to bad data");
			}
		}

		return result;
	}


	//=================================================================================================================
	// Stripped-down legacy ExtDbRecord subclass for wireless data sets, used for upgrade conversion only, see below.

	private static class ExtDbRecordWL extends ExtDbRecord {

		private static final String WIRELESS_BASE_TABLE = "base_station";
		private static final String WIRELESS_INDEX_TABLE = "antenna_index";
		private static final String WIRELESS_PATTERN_TABLE = "antenna_pattern";

		private String cellSiteID;
		private String sectorID;
		private String referenceNumber;


		//-------------------------------------------------------------------------------------------------------------
		// Get all records in the data set.  No errors other than SQL exception.

		private static ArrayList<ExtDbRecordWL> getAllRecords(DbConnection db, Service theService, Country defCountry)
				throws SQLException {

			ArrayList<ExtDbRecordWL> result = new ArrayList<ExtDbRecordWL>();

			db.query(
			"SELECT " +
				"cell_key," +
				"cell_site_id," +
				"sector_id," +
				"cell_lat," +
				"cell_lon," +
				"rc_amsl," +
				"haat," +
				"erp," +
				"az_ant_id," +
				"orientation," +
				"el_ant_id," +
				"e_tilt," +
				"m_tilt, " +
				"m_tilt_orientation, " +
				"reference_number, " +
				"city, " +
				"state, " +
				"country " +
			"FROM " +
				WIRELESS_BASE_TABLE);

			ExtDbRecordWL theRecord;

			int cellKey, antID;
			String recID, str;
			double lat, lon;

			while (db.next()) {

				cellKey = db.getInt(1);
				if (cellKey <= 0) {
					continue;
				}
				recID = String.valueOf(cellKey);

				lat = db.getDouble(4);
				lon = db.getDouble(5);

				theRecord = new ExtDbRecordWL();

				theRecord.extRecordID = recID;

				theRecord.service = theService;

				str = db.getString(2);
				if (null == str) {
					str = "";
				} else {
					if (str.length() > Source.MAX_CALL_SIGN_LENGTH) {
						str = str.substring(0, Source.MAX_CALL_SIGN_LENGTH);
					}
				}
				theRecord.cellSiteID = str;

				str = db.getString(3);
				if (null == str) {
					str = "";
				} else {
					if (str.length() > Source.MAX_SECTOR_ID_LENGTH) {
						str = str.substring(0, Source.MAX_SECTOR_ID_LENGTH);
					}
				}
				theRecord.sectorID = str;

				theRecord.location.setLatLon(lat, lon);

				theRecord.heightAMSL = db.getDouble(6);
				theRecord.overallHAAT = db.getDouble(7);

				theRecord.peakERP = db.getDouble(8);

				antID = db.getInt(9);
				if (antID > 0) {
					theRecord.antennaRecordID = String.valueOf(antID);
					theRecord.antennaID = theRecord.antennaRecordID;
				}
				theRecord.horizontalPatternOrientation = Math.IEEEremainder(db.getDouble(10), 360.);
				if (theRecord.horizontalPatternOrientation < 0.) theRecord.horizontalPatternOrientation += 360.;

				antID = db.getInt(11);
				if (antID > 0) {
					theRecord.elevationAntennaRecordID = String.valueOf(antID);
				}
				theRecord.verticalPatternElectricalTilt = db.getDouble(12);
				theRecord.verticalPatternMechanicalTilt = db.getDouble(13);
				theRecord.verticalPatternMechanicalTiltOrientation = Math.IEEEremainder(db.getDouble(14), 360.);
				if (theRecord.verticalPatternMechanicalTiltOrientation < 0.)
					theRecord.verticalPatternMechanicalTiltOrientation += 360.;

				theRecord.referenceNumber = db.getString(15);

				theRecord.city = db.getString(16);
				theRecord.state = db.getString(17);

				theRecord.country = Country.getCountry(db.getString(18));
				if (null == theRecord.country) {
					theRecord.country = defCountry;
				}

				result.add(theRecord);
			}

			return result;
		}


		//-------------------------------------------------------------------------------------------------------------
		// Simplified versions of pattern retrieval methods from superclass.  Errors ignored.

		private static ArrayList<AntPattern.AntPoint> getAntennaPatternWL(DbConnection db, String theAntRecordID)
				throws SQLException {

			ArrayList<AntPattern.AntPoint> result = new ArrayList<AntPattern.AntPoint>();

			db.query(
			"SELECT " +
				"degree, " +
				"relative_field " +
			"FROM " +
				WIRELESS_PATTERN_TABLE + " " +
			"WHERE " +
				"ant_id = " + theAntRecordID + " " +
			"ORDER BY 1");

			double az, pat, lastAz = AntPattern.AZIMUTH_MIN - 1., patMax = AntPattern.FIELD_MIN;
			boolean badData = false;

			while (db.next()) {

				az = Math.rint(db.getDouble(1) * AntPattern.AZIMUTH_ROUND) / AntPattern.AZIMUTH_ROUND;
				if ((az < AntPattern.AZIMUTH_MIN) || (az > AntPattern.AZIMUTH_MAX)) {
					badData = true;
					break;
				}
				if (az <= lastAz) {
					badData = true;
					break;
				}
				lastAz = az;

				pat = Math.rint(db.getDouble(2) * AntPattern.FIELD_ROUND) / AntPattern.FIELD_ROUND;
				if (pat < AntPattern.FIELD_MIN) {
					pat = AntPattern.FIELD_MIN;
				}
				if (pat > AntPattern.FIELD_MAX) {
					badData = true;
					break;
				}
				if (pat > patMax) {
					patMax = pat;
				}

				result.add(new AntPattern.AntPoint(az, pat));
			}

			if (badData || (result.size() < AntPattern.PATTERN_REQUIRED_POINTS) || (patMax < 0.5)) {
				result = null;
			}

			return result;
		}


		//-------------------------------------------------------------------------------------------------------------

		private static ArrayList<AntPattern.AntPoint> getElevationPatternWL(DbConnection db, String theAntRecordID)
				throws SQLException {

			ArrayList<AntPattern.AntPoint> result = new ArrayList<AntPattern.AntPoint>();

			db.query(
			"SELECT " +
				"degree, " +
				"relative_field " +
			"FROM " +
				WIRELESS_PATTERN_TABLE + " " +
			"WHERE " +
				"ant_id = " + theAntRecordID + " " +
			"ORDER BY 1");

			double dep, pat, lastDep = AntPattern.DEPRESSION_MIN - 1., patMax = AntPattern.FIELD_MIN;
			boolean badData = false, allZero = true;

			while (db.next()) {

				dep = Math.rint(db.getDouble(1) * AntPattern.DEPRESSION_ROUND) / AntPattern.DEPRESSION_ROUND;
				if ((dep < AntPattern.DEPRESSION_MIN) || (dep > AntPattern.DEPRESSION_MAX)) {
					badData = true;
					break;
				}
				if (dep <= lastDep) {
					badData = true;
					break;
				}
				lastDep = dep;

				pat = Math.rint(db.getDouble(2) * AntPattern.FIELD_ROUND) / AntPattern.FIELD_ROUND;
				if (pat < AntPattern.FIELD_MIN) {
					pat = AntPattern.FIELD_MIN;
				} else {
					allZero = false;
				}
				if (pat > AntPattern.FIELD_MAX) {
					badData = true;
					break;
				}
				if (pat > patMax) {
					patMax = pat;
				}

				result.add(new AntPattern.AntPoint(dep, pat));
			}

			if (badData || (result.size() < AntPattern.PATTERN_REQUIRED_POINTS) || (patMax < 0.5) || allZero) {
				result = null;
			}

			return result;
		}


		//-------------------------------------------------------------------------------------------------------------

		private ExtDbRecordWL() {

			super(null, Source.RECORD_TYPE_WL);
		}


		//-------------------------------------------------------------------------------------------------------------
		// Update a source object from this object's properties.  No errors possible other than SQL exception.

		private void updateSource(DbConnection db, SourceEditDataWL theSource) throws SQLException {

			theSource.callSign = cellSiteID;
			theSource.sectorID = sectorID;
			theSource.city = city;
			theSource.state = state;
			theSource.fileNumber = referenceNumber;
			theSource.location.setLatLon(location);
			theSource.heightAMSL = heightAMSL;
			theSource.overallHAAT = overallHAAT;
			theSource.peakERP = peakERP;

			if (null != antennaRecordID) {

				String theName = null;

				db.query("SELECT name FROM " + WIRELESS_INDEX_TABLE + " WHERE ant_id = " + antennaRecordID);
				if (db.next()) {
					theName = db.getString(1);
					if (theName.length() > Source.MAX_PATTERN_NAME_LENGTH) {
						theName = theName.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
					}
				}

				if (null != theName) {
					ArrayList<AntPattern.AntPoint> thePoints = getAntennaPatternWL(db, antennaRecordID);
					if (null != thePoints) {
						theSource.antennaID = antennaID;
						theSource.hasHorizontalPattern = true;
						theSource.horizontalPattern = new AntPattern(theSource.dbID, AntPattern.PATTERN_TYPE_HORIZONTAL,
							theName, thePoints);
						theSource.horizontalPatternChanged = true;
					}
				}
			}

			theSource.horizontalPatternOrientation = horizontalPatternOrientation;

			if (null != elevationAntennaRecordID) {

				String theName = null;

				db.query("SELECT name FROM " + WIRELESS_INDEX_TABLE + " WHERE ant_id = " +
					elevationAntennaRecordID);
				if (db.next()) {
					theName = db.getString(1);
					if (theName.length() > Source.MAX_PATTERN_NAME_LENGTH) {
						theName = theName.substring(0, Source.MAX_PATTERN_NAME_LENGTH);
					}
				}

				if (null != theName) {
					ArrayList<AntPattern.AntPoint> thePoints = getElevationPatternWL(db, elevationAntennaRecordID);
					if (null != thePoints) {
						theSource.hasVerticalPattern = true;
						theSource.verticalPattern = new AntPattern(theSource.dbID, AntPattern.PATTERN_TYPE_VERTICAL,
							theName, thePoints);
						theSource.verticalPatternChanged = true;
					}
				}
			}

			theSource.verticalPatternElectricalTilt = verticalPatternElectricalTilt;
			theSource.verticalPatternMechanicalTilt = verticalPatternMechanicalTilt;
			theSource.verticalPatternMechanicalTiltOrientation = verticalPatternMechanicalTiltOrientation;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Convert all existing wireless data sets in the old ExtDbRecord format to generic wireless data sets.  The old
	// format is no longer supported.  This is called if needed when a database is registered, see DbCore.registerDb().
	// The legacy ExtDbRecordWL code has been modified to work from this context (see above).  This is an in-place
	// conversion; a new generic database is created using the same key as the old database, once all records have been
	// converted and saved the type is updated in the index and the old database is dropped.

	public static boolean convertWirelessExtDbs(String dbID, DbConnection db, ErrorLogger errors) throws SQLException {

		String rootName = db.getDatabase();

		// Update the type key for deleted sets in the index.

		db.update("UPDATE ext_db SET db_type=" + String.valueOf(DB_TYPE_GENERIC_WL) + " WHERE db_type=" +
			String.valueOf(DB_TYPE_WIRELESS) + " AND deleted");

		// Get list of non-deleted data sets needing conversion, if any.

		ArrayList<Integer> keys = new ArrayList<Integer>();
		db.query("SELECT ext_db_key FROM ext_db WHERE db_type=" + String.valueOf(DB_TYPE_WIRELESS));
		while (db.next()) {
			keys.add(Integer.valueOf(db.getInt(1)));
		}
		if (keys.isEmpty()) {
			return true;
		}

		Service theService = Service.getService(Service.WL_CODE);
		if (null == theService) {
			if (null != errors) {
				errors.reportError("Wireless station data conversion failed, missing service");
			}
			return false;
		}

		Country defCountry = Country.getCountry(Country.US);
		if (null == defCountry) {
			if (null != errors) {
				errors.reportError("Wireless station data conversion failed, missing country");
			}
			return false;
		}

		String oldDbName, newDbName = null;
		ArrayList<ExtDbRecordWL> records;
		ExtDb extDbNew = null;
		DbConnection dbNew = null;
		SourceEditDataWL theSource;
		ArrayList<SourceEditDataWL> sources = new ArrayList<SourceEditDataWL>();
		boolean result = true;

		// Loop over data sets to convert.

		try {

			for (Integer theKey : keys) {

				oldDbName = rootName + "_wireless_" + String.valueOf(theKey);

				// First load all records from the old set.

				db.setDatabase(oldDbName);
				records = ExtDbRecordWL.getAllRecords(db, theService, defCountry);

				// Create the new set.

				newDbName = makeDbName(rootName, DB_TYPE_GENERIC_WL, theKey);
				db.update("CREATE DATABASE " + newDbName + " CHARACTER SET latin1");
				Source.createTables(db, newDbName);

				db.setDatabase(oldDbName);

				// Create a temporary ExtDb object for the new database, open a connection.

				extDbNew = new ExtDb(dbID, theKey, newDbName, new java.util.Date(), DB_TYPE_GENERIC_WL, 0, 0, false);
				extDbNew.id = "";
				extDbNew.name = "";
				extDbNew.description = "";

				dbNew = extDbNew.connectAndLock(errors);
				if (null == dbNew) {
					result = false;
					break;
				}

				// Loop over records, convert to SourceEditDataWL, save in new database.

				for (ExtDbRecordWL theRecord : records) {

					theSource = SourceEditDataWL.createExtSource(extDbNew, theRecord.service, theRecord.country);
					theRecord.updateSource(db, theSource);

					theSource.isDataChanged();
					theSource.save(dbNew);
				}

				extDbNew.releaseDb(dbNew);
				dbNew = null;

				// Update the data set type and drop the old database.

				db.setDatabase(rootName);

				db.update("UPDATE ext_db SET db_type=" + String.valueOf(DB_TYPE_GENERIC_WL) + " WHERE ext_db_key=" +
					String.valueOf(theKey));
				newDbName = null;

				db.update("DROP DATABASE " + oldDbName);
			}

		} catch (SQLException se) {
			db.reportError(errors, se);
			result = false;

		} catch (Throwable t) {
			if (null != errors) {
				errors.reportError("An unexpected error occurred:\n" + t);
			}
			result = false;
		}

		// If an error aborted a conversion before it completed, close connection and drop the new database.

		if (null != dbNew) {
			extDbNew.releaseDb(dbNew);
		}

		if (null != newDbName) {
			db.update("DROP DATABASE IF EXISTS " + newDbName);
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Wrapper around createNewDatabase() to first download a ZIP file directly from a web server, store it in a
	// temporary location, then import data from it.  Data sets imported this way are flagged as downloads.  Once a
	// new download and import succeeds, all other flagged data sets of the same type are deleted.  Data sets created
	// by manual import are not flagged.  Also when the name is set in the UI the download flag is cleared, presumably
	// if the user goes to the trouble of giving a downloaded set a name, they want to keep it.  Only one download can
	// be in progress at a time.

	private static boolean downloadInProgress = false;

	public static boolean isDownloadInProgress() {
		return downloadInProgress;
	}

	private static synchronized boolean checkDownloadInProgress() {
		if (downloadInProgress) {
			return true;
		}
		downloadInProgress = true;
		return false;
	}

	public static Integer downloadDatabase(String theDbID, int dataType, String theName) {
		return downloadDatabase(theDbID, dataType, theName, null, null);
	}

	public static Integer downloadDatabase(String theDbID, int dataType, String theName, ErrorLogger errors) {
		return downloadDatabase(theDbID, dataType, theName, null, errors);
	}

	public static Integer downloadDatabase(String theDbID, int dataType, String theName, StatusLogger status) {
		return downloadDatabase(theDbID, dataType, theName, status, null);
	}

	public static Integer downloadDatabase(String theDbID, int dataType, String theName, StatusLogger status,
			ErrorLogger errors) {

		if (checkDownloadInProgress()) {
			if (null != errors) {
				errors.reportError("Download is already in progress");
			}
			return null;
		}

		File tempFile = null;
		Integer result = null;

		try {

			String str = null;

			switch (dataType) {

				case DB_TYPE_LMS: {
					str = AppCore.getPreference(AppCore.CONFIG_LMS_DOWNLOAD_URL);
					break;
				}

				default: {
					if (null != errors) {
						errors.reportError("Unknown or unsupported station data type");
					}
					downloadInProgress = false;
					return null;
				}
			}

			if (null == str) {
				if (null != errors) {
					errors.reportError("Configuration error, no URL for station data download");
				}
				downloadInProgress = false;
				return null;
			}

			URL url = new URL(str);
			URLConnection theConn = url.openConnection();

			theConn.setConnectTimeout(DOWNLOAD_TIMEOUT);
			theConn.setReadTimeout(DOWNLOAD_TIMEOUT);

			theConn.connect();
			InputStream theInput = theConn.getInputStream();

			tempFile = File.createTempFile("dbdata", ".zip");
			BufferedOutputStream theOutput = new BufferedOutputStream(new FileOutputStream(tempFile));

			byte[] buffer = new byte[65536];
			int count = 0, bar, lastBar = 0, i;
			long length = theConn.getContentLengthLong(), done = 0;
			double percent;
			boolean error = false;

			if (null != status) {
				status.reportStatus("Downloading, 0% done");
				status.showMessage("Downloading [------------------------------] 0%");
			}

			while (true) {

				count = theInput.read(buffer);
				if (count < 0) {
					break;
				}

				if (count > 0) {

					theOutput.write(buffer, 0, count);
					done += count;

					if (null != status) {

						if (status.isCanceled()) {
							error = true;
							status.logMessage("Download canceled");
							break;
						}

						bar = (int)(((double)done / (double)length) * 30.);
						if (bar > lastBar) {
							String pct = String.format(Locale.US, "%d%%",
								(int)Math.rint(((double)done / (double)length) * 100.));
							status.reportStatus("Downloading, " + pct + " done");
							StringBuilder b = new StringBuilder("Downloading [");
							for (i = 0; i < bar; i++) {
								b.append('#');
							}
							for (; i < 30; i++) {
								b.append('-');
							}
							b.append("] ");
							b.append(pct);
							status.showMessage(b.toString());
							lastBar = bar;
						}
					}
				}
			}

			theInput.close();
			theOutput.close();

			if (!error) {
				if (done < length) {
					error = true;
					if (null != status) {
						status.logMessage("Download incomplete, import canceled");
					}
				}
			}

			if (!error) {

				if (null != status) {
					status.reportStatus("Importing");
					status.logMessage("Download complete, importing data files...");
				}

				result = createNewDatabase(theDbID, dataType, tempFile, theName, status, true, errors);
			}

		} catch (Throwable t) {
			if (null != errors) {
				errors.reportError(t.toString());
			}
		}

		if (null != tempFile) {
			tempFile.delete();
		}

		downloadInProgress = false;
		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Create a new database, importing data from files.  Multiple files required for the import always have fixed
	// names.  The files may be in a directory, or in a ZIP file.  See TableFile and the open*TableFiles() methods for
	// details.  The new data set key is returned, or null on error.  See comments at downloadDatabase() for purpose
	// of the isDownload flag.  Note the name is assumed to have been checked for length and reserved characters.  A
	// concurrent-safe uniqueness check will be performed and a suffix added to make the name unique if needed, but if
	// it is too long this will fail.

	public static Integer createNewDatabase(String theDbID, int dataType, File fileSource, String theName) {
		return createNewDatabase(theDbID, dataType, fileSource, theName, null, false, null);
	}

	public static Integer createNewDatabase(String theDbID, int dataType, File fileSource, String theName,
			ErrorLogger errors) {
		return createNewDatabase(theDbID, dataType, fileSource, theName, null, false, errors);
	}

	public static Integer createNewDatabase(String theDbID, int dataType, File fileSource, String theName,
			StatusLogger status) {
		return createNewDatabase(theDbID, dataType, fileSource, theName, status, false, null);
	}

	public static Integer createNewDatabase(String theDbID, int dataType, File fileSource, String theName,
			StatusLogger status, ErrorLogger errors) {
		return createNewDatabase(theDbID, dataType, fileSource, theName, status, false, errors);
	}

	private static Integer createNewDatabase(String theDbID, int dataType, File fileSource, String theName,
			StatusLogger status, boolean isDownload, ErrorLogger errors) {

		File fileDirectory = null;
		ZipFile zipFile = null;

		if (fileSource.isDirectory()) {
			fileDirectory = fileSource;
		} else {
			try {
				zipFile = new ZipFile(fileSource);
			} catch (IOException ie) {
				if (null != errors) {
					errors.reportError(ie.toString());
				}
				return null;
			}
		}

		ArrayList<TableFile> tableFiles = null;

		switch (dataType) {

			case DB_TYPE_CDBS: {
				tableFiles = openCDBSTableFiles(fileDirectory, zipFile, errors);
				break;
			}
				
			case DB_TYPE_LMS: {
				tableFiles = openLMSTableFiles(fileDirectory, zipFile, errors);
				break;
			}

			case DB_TYPE_CDBS_FM: {
				tableFiles = openCDBSFMTableFiles(fileDirectory, zipFile, errors);
				break;
			}

			case DB_TYPE_LMS_LIVE:
			default: {
				if (null != errors) {
					errors.reportError("Unknown or unsupported station data type");
				}
				break;
			}
		}

		if (null == tableFiles) {
			if (null != zipFile) {
				try {zipFile.close();} catch (IOException e) {};
			}
			return null;
		}

		// Open database connection, lock tables, get a new key for the database.  The LOCK TABLES is released as soon
		// as possible, the database creation can't occur with that in effect and will also take a significant amount
		// of time.  See further comments in Study.createNewStudy().

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			if (null != zipFile) {
				try {zipFile.close();} catch (IOException e) {};
			}
			return null;
		}

		String errmsg = null, theDbName = null, rootName = DbCore.getDbName(theDbID);
		boolean error = false;
		int newKey = 0;

		try {

			db.update("LOCK TABLES ext_db_key_sequence WRITE");

			db.update("UPDATE ext_db_key_sequence SET ext_db_key = ext_db_key + 1");
			db.query("SELECT ext_db_key FROM ext_db_key_sequence");
			db.next();
			newKey = db.getInt(1);

			// A range of keys is reserved for internally-composed objects, e.g. see getLMSLiveExtDb().  The range
			// start is not likely to ever be reached but if it is advance past the range.

			if (RESERVED_KEY_RANGE_START == newKey) {
				newKey = RESERVED_KEY_RANGE_END + 1;
				db.update("UPDATE ext_db_key_sequence SET ext_db_key = " + newKey);
			}

			db.update("UNLOCK TABLES");

			// Create the database, copy all the tables.

			theDbName = makeDbName(rootName, dataType, Integer.valueOf(newKey));

			DateCounter theDate = null;
			int theVersion = 0;

			switch (dataType) {

				case DB_TYPE_CDBS: {
					theDate = new DateCounter("MM/dd/yyyy");
					theVersion = CDBS_VERSION;
					break;
				}

				case DB_TYPE_LMS: {
					theDate = new DateCounter("yyyy-MM-dd");
					theVersion = LMS_VERSION;
					break;
				}

				case DB_TYPE_CDBS_FM: {
					theDate = new DateCounter("MM/dd/yyyy");
					theVersion = CDBS_FM_VERSION;
					break;
				}
			}

			db.update("CREATE DATABASE " + theDbName + " CHARACTER SET latin1");
			db.setDatabase(theDbName);

			// This now supports a limited detection of past versions, in cases where the only change was adding some
			// additional fields and the query code still has fallback support.  If a field is flagged with a version
			// number and is not found in the file, no error occurs during import.  Check for that here and adjust the
			// version to one less than the version set on the field.  Entire files can now also be flagged with a
			// version, those will have the required flag false so the failure to open earlier did not cause an error,
			// but check for that here and adjust the version as needed.

			if (null != status) {
				status.reportStatus("Importing");
			}

			for (TableFile theFile : tableFiles) {
				if (null != theFile.reader) {

					if (null != status) {
						if (status.isCanceled()) {
							error = true;
							status.logMessage("Import canceled");
							break;
						}
						status.logMessage("Importing data file " + theFile.fileName + "...");
					}

					errmsg = createAndCopyTable(db, theFile, theDate, status);
					if (null != errmsg) {
						error = true;
						break;
					}

					theFile.createAllIndex(db);

					for (TableField theField : theFile.requiredFields) {
						if ((theField.version > 0) && (theField.index < 0)) {
							if (theField.version <= theVersion) {
								theVersion = theField.version - 1;
							}
						}
					}

				} else {
					if (theFile.version > 0) {
						if (theFile.version <= theVersion) {
							theVersion = theFile.version - 1;
						}
					}
				}
			}

			// Build search index information for LMS.

			int indexVersion = 0;

			if (!error && (DB_TYPE_LMS == dataType)) {

				if ((null != status) && status.isCanceled()) {
					error = true;
					status.logMessage("Import canceled");
				} else {

					if (null != status) {
						status.logMessage("Building search index...");
					}

					updateSearchIndex(db, 0);

					indexVersion = LMS_INDEX_VERSION;
				}
			}

			// If success generate ID string, typically based on a date field scan, see DateCounter.  The ID is always
			// unique, however the data set date alone may not be unique so the key may be appended to the date, see
			// below.  If a name is provided that will also be checked below for uniqueness, again if not unique the
			// key will be appended.  A reserved character is prefixed to the key to guarantee uniqueness.

			if (!error) {

				String theID = "", theDbDate = "NOW()";

				switch (dataType) {

					case DB_TYPE_CDBS: {
						theID = theDate.getDate();
						theDbDate = "'" + AppCore.formatDateTime(theDate.getDbDate()) + "'";
						break;
					}

					case DB_TYPE_LMS: {
						theID = theDate.getDate();
						theDbDate = "'" + AppCore.formatDateTime(theDate.getDbDate()) + "'";
						break;
					}

					case DB_TYPE_CDBS_FM: {
						theID = theDate.getDate();
						theDbDate = "'" + AppCore.formatDateTime(theDate.getDbDate()) + "'";
						break;
					}
				}

				if (null == theName) {
					theName = "";
				}

				db.setDatabase(rootName);

				// Check the ID and name for uniqueness, save the new index entry in the ext_db table.

				db.update("LOCK TABLES ext_db WRITE");

				db.query("SELECT ext_db_key FROM ext_db WHERE (id = '" + db.clean(theID) + "')");
				if (db.next()) {
					theID = theID + " " + String.valueOf(DbCore.NAME_UNIQUE_CHAR) + String.valueOf(newKey);
				}

				if (theName.length() > 0) {
					db.query("SELECT ext_db_key FROM ext_db WHERE (UPPER(name) = '" +
						db.clean(theName.toUpperCase()) + "')");
					if (db.next()) {
						theName = theName + " " + String.valueOf(DbCore.NAME_UNIQUE_CHAR) + String.valueOf(newKey);
					}
				}

				db.update(
				"INSERT INTO ext_db (" +
					"ext_db_key, " +
					"db_type, " +
					"db_date, " +
					"version, " +
					"index_version, " +
					"id, " +
					"name, " +
					"deleted, " +
					"locked, " +
					"is_download, " +
					"bad_data) " +
				"VALUES (" +
					newKey + ", "  +
					dataType + ", " +
					theDbDate + ", " +
					theVersion + ", " +
					indexVersion + ", " +
					"'" + db.clean(theID) + "', " +
					"'" + db.clean(theName) + "', " +
					"false, " +
					"false, " +
					isDownload + ", " +
					"false)");

				theDbName = null;

				db.update("UNLOCK TABLES");

				// Run extra operations after successful import.

				errmsg = checkImport(db, rootName, dataType, isDownload, newKey, theDbDate, tableFiles);
			}

		} catch (SQLException se) {
			errmsg = "A database error occurred:\n" + se;
			error = true;
			db.reportError(se);
		}

		// Make sure table locks are released, if an error occurred also drop the database.

		try {
			db.update("UNLOCK TABLES");
			if (error && (null != theDbName)) {
				db.update("DROP DATABASE IF EXISTS " + theDbName);
			}
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		for (TableFile theFile : tableFiles) {
			theFile.closeFile();
		}

		if (null != zipFile) {
			try {zipFile.close();} catch (IOException e) {};
		}

		if ((null != errmsg) && (null != errors)) {
			if (error) {
				errors.reportError(errmsg);
			} else {
				errors.reportWarning(errmsg);
			}
		}

		reloadCache(theDbID);

		if (error) {
			return null;
		}

		status.logMessage("Import complete");

		return Integer.valueOf(newKey);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Update search index information for LMS data sets, this is called from DbCore.registerDb() if there are any
	// existing LMS sets that need update.

	public static boolean updateAllSearchIndex(String dbID, DbConnection db, ErrorLogger errors) throws SQLException {

		String rootName = db.getDatabase();

		// Update the version for deleted sets.

		db.update("UPDATE ext_db SET index_version=" + String.valueOf(LMS_INDEX_VERSION) + " WHERE db_type=" +
			String.valueOf(DB_TYPE_LMS) + " AND deleted");

		// Get list of non-deleted data sets needing update, if any.

		ArrayList<Integer> keys = new ArrayList<Integer>();
		ArrayList<Integer> versions = new ArrayList<Integer>();
		db.query("SELECT ext_db_key, index_version FROM ext_db WHERE db_type=" + String.valueOf(DB_TYPE_LMS) +
			" AND index_version <> " + String.valueOf(LMS_INDEX_VERSION));
		while (db.next()) {
			keys.add(Integer.valueOf(db.getInt(1)));
			versions.add(Integer.valueOf(db.getInt(2)));
		}
		if (keys.isEmpty()) {
			return true;
		}

		boolean result = true;

		try {

			Integer theKey, theVersion;

			for (int i = 0; i < keys.size(); i++) {

				theKey = keys.get(i);
				theVersion = versions.get(i);

				db.setDatabase(makeDbName(rootName, DB_TYPE_LMS, theKey));

				updateSearchIndex(db, theVersion.intValue());

				db.setDatabase(rootName);

				db.update("UPDATE ext_db SET index_version=" + String.valueOf(LMS_INDEX_VERSION) +
					" WHERE ext_db_key=" + String.valueOf(theKey));
			}

		} catch (SQLException se) {
			db.reportError(errors, se);
			result = false;

		} catch (Throwable t) {
			if (null != errors) {
				errors.reportError("An unexpected error occurred:\n" + t);
			}
			result = false;
		}

		return result;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Create additional filtered and indexed columns in LMS import tables to speed up searching.

	private static void updateSearchIndex(DbConnection db, int currentVersion) throws SQLException {

		switch (currentVersion) {

			// Add an indexed column for call sign search in the facility table; trim whitespace, up-case, and remove
			// any leading 'D' characters.

			case 0: {

				HashMap<Integer, String> calls = new HashMap<Integer, String>();

				db.query("SELECT facility_id, callsign FROM facility");

				Integer id;
				String call;
				int l, i;

				while (db.next()) {

					id = Integer.valueOf(db.getInt(1));

					call = db.getString(2);
					if (null == call) {
						continue;
					}

					call = call.trim().toUpperCase();

					l = call.length();
					if (0 == l) {
						continue;
					}

					i = 0;
					while ((call.charAt(i) == 'D') && (i < l)) {
						i++;
					}
					if (i >= l) {
						continue;
					}
					if (i > 0) {
						call = call.substring(i);
					}
						
					calls.put(id, call);
				}

				StringBuilder query;
				int startLength;
				String sep;

				db.update("CREATE TEMPORARY TABLE temp_callsign (facility_id INT, callsign CHAR(12))");

				query = new StringBuilder("INSERT INTO temp_callsign VALUES");
				startLength = query.length();
				sep = " (";
				for (Map.Entry<Integer, String> e : calls.entrySet()) {
					query.append(sep);
					query.append(String.valueOf(e.getKey()));
					query.append(",'");
					query.append(db.clean(e.getValue()));
					if (query.length() > DbCore.MAX_QUERY_LENGTH) {
						query.append("')");
						db.update(query.toString());
						query.setLength(startLength);
						sep = " (";
					} else {
						sep = "'),(";
					}
				}
				if (query.length() > startLength) {
					query.append("')");
					db.update(query.toString());
				}

				db.update("ALTER TABLE facility ADD COLUMN _search_callsign CHAR(12)");
				db.update("UPDATE facility JOIN temp_callsign USING (facility_id) SET facility._search_callsign = " +
					"temp_callsign.callsign");
				db.update("CREATE INDEX search_callsign_index ON facility (_search_callsign)");
			}

			// File number and city name.  Filtering is just up-case, but city name may come from one of two tables.

			case 1: {

				db.update("ALTER TABLE application ADD COLUMN _search_filenum CHAR(20)");
				db.update("UPDATE application SET _search_filenum = UPPER(aapp_file_num)");
				db.update("CREATE INDEX search_filenum_index ON application (_search_filenum)");

				db.update("ALTER TABLE facility ADD COLUMN _search_city VARCHAR(255)");
				db.update("UPDATE facility JOIN application_facility ON " +
					"(facility.facility_id = application_facility.afac_facility_id) " +
					"SET facility._search_city = (UPPER(CASE WHEN (facility.community_served_city <> '') " +
					"THEN facility.community_served_city ELSE application_facility.afac_community_city END))");
				db.update("CREATE INDEX search_city_index ON facility (_search_city)");
			}

			// Split out ARN from file number.

			case 2: {

				db.update("ALTER TABLE application ADD COLUMN _search_arn CHAR(20)");
				db.update("UPDATE application SET _search_arn = UPPER(CASE WHEN (LOCATE('-', aapp_file_num) > 0) " +
					"THEN SUBSTRING_INDEX(aapp_file_num, '-', -1) ELSE aapp_file_num END)");
				db.update("CREATE INDEX search_arn_index ON application (_search_arn)");
			}
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Special actions after a new set is imported.  Check all imported tables for any that are empty, that should not
	// occur and indicates data may be incomplete/corrupted.  Also compare key table row counts to a previous set of
	// the same type if possible, if counts decrease or increase excessively, the data may be corrupted in some way.
	// In either case of bad data show a warning and set the bad-data flag on the new set which will prevent it from
	// being used automatically by the "most recent" selection, and also prevent it from being used for comparison to
	// a later import.  If the bad-data result is a false positive, the flag can be cleared manually in the data set
	// manager.  Also for a download if the data tests pass and a preference indicates previous downloads of the same
	// type should be deleted, do that.  Returns null on success, a warning message if the data test fails.

	private static String checkImport(DbConnection db, String rootName, int dataType, boolean isDownload, int newKey,
			String newDate, ArrayList<TableFile> tableFiles) throws SQLException {

		boolean badData = false;
		String errmsg = null;

		// Files that are not required and did not exist have rowCount -1 so they will be ignored by both tests here.

		for (TableFile theFile : tableFiles) {
			if (0 == theFile.rowCount) {
				badData = true;
				errmsg =
					"Some imported tables are empty, the data set may be damaged.\n" +
					"The data set will be flagged and will not be used automatically.  If\n" +
					"data is valid, the flag can be cleared in the data manager.";
				break;
			}
		}

		// For the row-count check, look for the most-recent previous set of the same type that does not also have the
		// bad-data flag and is not deleted.  If there isn't one the test is skipped.

		if (!badData) {

			db.query("SELECT ext_db_key FROM ext_db WHERE db_type = " + dataType + " AND ext_db_key <> " + newKey +
				" AND db_date < " + newDate + " AND NOT bad_data AND NOT deleted ORDER BY db_date DESC LIMIT 1");

			if (db.next()) {

				db.setDatabase(makeDbName(rootName, dataType, Integer.valueOf(db.getInt(1))));

				int minCount, maxCount;
				for (TableFile theFile : tableFiles) {
					if ((null != theFile.keyField) && (theFile.rowCount > 0)) {
						db.query("SELECT COUNT(*) FROM " + theFile.tableName);
						if (db.next()) {
							minCount = db.getInt(1);
							maxCount = minCount + (minCount / 5);
							if ((theFile.rowCount < minCount) || (theFile.rowCount > maxCount)) {
								badData = true;
								errmsg =
									"Imported table row counts are inconsistent with older imports, the\n" +
									"data set may be damaged.  The data set will be flagged and will not be used\n" +
									"automatically.  If data is valid, the flag can be cleared in the data manager.";
								break;
							}
						}
					}
				}

				db.setDatabase(rootName);
			}
		}

		// If bad data, set the flag and return warning message.

		if (badData) {
			db.update("UPDATE ext_db SET bad_data = true WHERE ext_db_key = " + newKey);
			return errmsg;
		}

		// Data looks OK, if this was a download check the preference for deleting previous downloads and do that if
		// requested.  Sets are just marked deleted, the database drops will be done in closeDb(), see comments in
		// deleteDatabase().  Also respect the locked flag, that shouldn't occur along with download but check anyway.

		if (isDownload) {
			String str = AppCore.getPreference(AppCore.CONFIG_AUTO_DELETE_PREVIOUS_DOWNLOAD);
			if ((null != str) && Boolean.valueOf(str).booleanValue()) {
				db.update("UPDATE ext_db SET is_download = false, deleted = true WHERE db_type = " +
					dataType + " AND ext_db_key <> " + newKey + " AND is_download AND NOT locked");
			}
		}

		return null;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Compose the name of the database used for an imported data set.

	public static String makeDbName(String rootName, int theType, Integer theKey) {

		switch (theType) {
			case DB_TYPE_CDBS:
				return rootName + "_cdbs_" + theKey;
			case DB_TYPE_LMS:
				return rootName + "_lms_" + theKey;
			case DB_TYPE_CDBS_FM:
				return rootName + "_cdbs_fm_" + theKey;
			case DB_TYPE_GENERIC_TV:
				return rootName + "_import_tv_" + theKey;
			case DB_TYPE_GENERIC_WL:
				return rootName + "_import_wl_" + theKey;
			case DB_TYPE_GENERIC_FM:
				return rootName + "_import_fm_" + theKey;
		}

		return "";
	}


	//-----------------------------------------------------------------------------------------------------------------
	// A list of all the unique base names needed for SHOW DATABASES LIKE '[rootname]_[basename]%' expressions to
	// identify all existing databases for imported data sets; see closeDb().

	private static ArrayList<String> getDbBaseNames() {

		ArrayList<String> result = new ArrayList<String>();
		result.add("cdbs");
		result.add("lms");
		result.add("import");

		return result;
	}


	//=================================================================================================================
	// Class to define structure of a table and associated SQL dump file, and manage a file reader during import, see
	// createAndCopyTable().  The reader may be reading directly from a file, or from an entry in a ZIP file.  If the
	// fieldNames property is null the list of field names will be read from the first line.  The file name is derived
	// from the table name.  Field separator character is defined here, lines have a separator-terminator-separator
	// termination sequence, the terminator character is also defined here.  The required flag may be false allowing
	// the table file to be missing, code using tables that may not always be present must check for table existence.
	// This now also has a versioning ability like TableField, if the file is missing and version is >0 the import
	// version is adjusted to no greater than one less than version.  This also supports deferring the creation of
	// indexes on tables until after import, which can speed things up considerably; use addRequiredFieldWithIndex()
	// then call createAllIndex() just after import is complete.

	private static class TableFile {

		private static final char SEPARATOR = '|';
		private static final char TERMINATOR = '^';

		private static final int MAX_ROWS_IGNORED = 50;
		private static final int MAX_DUPLICATES_LOGGED = 10;

		private String tableName;
		private String[] fieldNames;
		private String extraDefinitions;

		private TableField dateField;
		private TableField keyField;

		private boolean required;
		private int version;

		private ArrayList<TableField> requiredFields;
		private ArrayList<TableField> indexFields;

		private String fileName;
		private File dataFile;
		private InputStream zipStream;

		private BufferedReader reader;

		private int rowCount;


		//-------------------------------------------------------------------------------------------------------------

		private TableFile(String theTableName, String[] theFieldNames, File fileDirectory, ZipFile zipFile,
				boolean theRequiredFlag) {

			required = theRequiredFlag;

			doInit(theTableName, theFieldNames, fileDirectory, zipFile);
		}


		//-------------------------------------------------------------------------------------------------------------

		private TableFile(String theTableName, String[] theFieldNames, File fileDirectory, ZipFile zipFile,
				int theVersion) {

			version = theVersion;

			doInit(theTableName, theFieldNames, fileDirectory, zipFile);
		}


		//-------------------------------------------------------------------------------------------------------------

		private void doInit(String theTableName, String[] theFieldNames, File fileDirectory, ZipFile zipFile) {

			tableName = theTableName;
			fieldNames = theFieldNames;

			requiredFields = new ArrayList<TableField>();
			indexFields = new ArrayList<TableField>();

			fileName = theTableName + ".dat";
			if (null != fileDirectory) {
				dataFile = new File(fileDirectory, fileName);
			} else {
				if (null != zipFile) {
					try {
						ZipEntry theEntry = zipFile.getEntry(fileName);
						if (null != theEntry) {
							zipStream = zipFile.getInputStream(theEntry);
						}
					} catch (IOException ie) {
					}
				}
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		private void setExtraDefinitions(String theExtraDefinitions) {

			extraDefinitions = theExtraDefinitions;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void setDateField(TableField theField) {

			dateField = theField;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void setKeyField(TableField theField) {

			keyField = theField;
		}


		//-------------------------------------------------------------------------------------------------------------

		private TableField addRequiredField(String name, String type, boolean isText) {

			TableField newField = new TableField(name, type, isText, 0);
			requiredFields.add(newField);
			return newField;
		}


		//-------------------------------------------------------------------------------------------------------------

		private TableField addRequiredField(String name, String type, boolean isText, int version) {

			TableField newField = new TableField(name, type, isText, version);
			requiredFields.add(newField);
			return newField;
		}


		//-------------------------------------------------------------------------------------------------------------

		private TableField addRequiredFieldWithIndex(String name, String type, boolean isText) {

			TableField newField = new TableField(name, type, isText, 0);
			requiredFields.add(newField);
			indexFields.add(newField);
			return newField;
		}


		//-------------------------------------------------------------------------------------------------------------

		private boolean openFile() {

			closeFile();

			try {
				if (null != dataFile) {
					reader = new BufferedReader(new FileReader(dataFile));
				} else {
					if (null != zipStream) {
						reader = new BufferedReader(new InputStreamReader(zipStream));
					} else {
						return false;
					}
				}
			} catch (IOException ie) {
				return false;
			}

			return true;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void closeFile() {

			if (null != reader) {
				try {
					reader.close();
				} catch (IOException ie) {
				}
				reader = null;
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		private void createAllIndex(DbConnection db) throws SQLException {

			for (TableField field : indexFields) {
				db.update("CREATE INDEX " + tableName + "_" + field.name + " ON " + tableName + " (" + field.name +
					")");
			}
		}
	}


	//=================================================================================================================
	// Class used to manage required field definitions in a data file being imported.  If version is >0 the field may
	// be missing and the import version number will be adjusted accordingly (to at least one less than version).

	private static class TableField {

		private String name;
		private String type;
		private boolean isText;

		private int version;

		private int index;


		//-------------------------------------------------------------------------------------------------------------

		private TableField(String theName, String theType, boolean theIsText, int theVersion) {

			name = theName;
			type = theType;
			isText = theIsText;

			version = theVersion;

			index = -1;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// For CDBS SQL dump files, the field name lists defining the table structures and so dump file formats are stored
	// in a separate data file installed with the application.  Changes to the dump file structure that don't affect
	// queries can be applied by just updating that file.  The first time it is needed, the file is parsed and cached.
	// Errors are ignored, if it fails to load the imports will fail.

	private static HashMap<String, String[]> CDBSFieldNamesMap = null;

	private static void loadCDBSFieldNames() {

		if (null != CDBSFieldNamesMap) {
			return;
		}

		CDBSFieldNamesMap = new HashMap<String, String[]>();

		String tableName;
		String[] fieldNames;

		try {

			BufferedReader reader = Files.newBufferedReader(AppCore.libDirectoryPath.resolve(CDBS_TABLE_DEFS_FILE));

			do {
				tableName = AppCore.readLineSkipComments(reader);
				if (null != tableName) {
					fieldNames = AppCore.readAndParseLine(reader);
					if ((null != fieldNames) && (tableName.length() > 3) && (fieldNames.length > 2)) {
						CDBSFieldNamesMap.put(tableName, fieldNames);
					}
				}
			} while (null != tableName);

			reader.close();

		} catch (IOException ie) {
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Open table files for a CDBS TV database, return open file list, or null on error.  This opens the TV files and
	// one AM file (AM is used only in support of TV interference-check studies).

	private static ArrayList<TableFile> openCDBSTableFiles(File fileDirectory, ZipFile zipFile, ErrorLogger errors) {

		loadCDBSFieldNames();

		ArrayList<TableFile> tableFiles = new ArrayList<TableFile>();

		String tableName;
		TableFile tableFile;

		tableName = "application";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("application_id", "INT", false));
		tableFile.addRequiredField("fac_callsign", "CHAR(12)", true);
		tableFile.addRequiredField("comm_city", "CHAR(20)", true);
		tableFile.addRequiredField("comm_state", "CHAR(2)", true);
		tableFile.addRequiredField("file_prefix", "CHAR(10)", true);
		tableFile.addRequiredField("app_arn", "CHAR(12)", true);
		tableFile.setDateField(tableFile.addRequiredField("last_change_date", "CHAR(20)", true));
		tableFiles.add(tableFile);

		tableName = "app_tracking";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("application_id", "INT", false);
		tableFile.addRequiredField("accepted_date", "CHAR(20)", true);
		tableFile.setDateField(tableFile.addRequiredField("last_change_date", "CHAR(20)", true));
		tableFiles.add(tableFile);

		tableName = "facility";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("facility_id", "INT", false));
		tableFile.addRequiredField("fac_callsign", "CHAR(12)", true);
		tableFile.addRequiredField("comm_city", "CHAR(20)", true);
		tableFile.addRequiredField("comm_state", "CHAR(2)", true);
		tableFile.addRequiredField("fac_service", "CHAR(2)", true);
		tableFile.addRequiredField("fac_country", "CHAR(2)", true);
		tableFile.setDateField(tableFile.addRequiredField("last_change_date", "CHAR(20)", true));
		tableFiles.add(tableFile);

		tableName = "tv_eng_data";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredField("application_id", "INT", false);
		tableFile.addRequiredField("site_number", "TINYINT", false);
		tableFile.setExtraDefinitions("UNIQUE (application_id,site_number)");
		tableFile.addRequiredFieldWithIndex("facility_id", "INT", false);
		tableFile.addRequiredField("eng_record_type", "CHAR(1)", true);
		tableFile.addRequiredField("vsd_service", "CHAR(2)", true);
		tableFile.addRequiredField("station_channel", "INT", false);
		tableFile.addRequiredField("tv_dom_status", "CHAR(6)", true);
		tableFile.addRequiredField("fac_zone", "CHAR(3)", true);
		tableFile.addRequiredField("freq_offset", "CHAR(1)", true);
		tableFile.addRequiredField("dt_emission_mask", "CHAR(1)", true);
		tableFile.addRequiredField("lat_dir", "CHAR(1)", true);
		tableFile.addRequiredField("lat_deg", "INT", false);
		tableFile.addRequiredField("lat_min", "INT", false);
		tableFile.addRequiredField("lat_sec", "FLOAT", false);
		tableFile.addRequiredField("lon_dir", "CHAR(1)", true);
		tableFile.addRequiredField("lon_deg", "INT", false);
		tableFile.addRequiredField("lon_min", "INT", false);
		tableFile.addRequiredField("lon_sec", "FLOAT", false);
		tableFile.addRequiredField("rcamsl_horiz_mtr", "FLOAT", false);
		tableFile.addRequiredField("haat_rc_mtr", "FLOAT", false);
		tableFile.addRequiredField("effective_erp", "FLOAT", false);
		tableFile.addRequiredField("max_erp_any_angle", "FLOAT", false);
		tableFile.addRequiredField("antenna_id", "INT", false);
		tableFile.addRequiredField("ant_rotation", "FLOAT", false);
		tableFile.addRequiredField("elevation_antenna_id", "INT", false);
		tableFile.addRequiredField("electrical_deg", "FLOAT", false);
		tableFile.addRequiredField("mechanical_deg", "FLOAT", false);
		tableFile.addRequiredField("true_deg", "FLOAT", false);
		tableFile.addRequiredField("predict_coverage_area", "FLOAT", false);
		tableFile.setDateField(tableFile.addRequiredField("last_change_date", "CHAR(20)", true));
		tableFiles.add(tableFile);

		tableName = "tv_app_indicators";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredField("application_id", "INT", false);
		tableFile.addRequiredField("site_number", "TINYINT", false);
		tableFile.setExtraDefinitions("UNIQUE (application_id,site_number)");
		tableFile.addRequiredField("da_ind", "CHAR(1)", true);
		tableFiles.add(tableFile);

		tableName = "dtv_channel_assignments";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("facility_id", "INT", false));
		tableFile.addRequiredField("callsign", "CHAR(12)", true);
		tableFile.addRequiredField("city", "CHAR(20)", true);
		tableFile.addRequiredField("state", "CHAR(2)", true);
		tableFile.addRequiredField("post_dtv_channel", "INT", false);
		tableFile.addRequiredField("latitude", "CHAR(10)", true);
		tableFile.addRequiredField("longitude", "CHAR(11)", true);
		tableFile.addRequiredField("rcamsl", "INT", false);
		tableFile.addRequiredField("haat", "FLOAT", false);
		tableFile.addRequiredField("erp", "FLOAT", false);
		tableFile.addRequiredField("da_ind", "CHAR(1)", true);
		tableFile.addRequiredField("antenna_id", "INT", false);
		tableFile.addRequiredField("ref_azimuth", "INT", false);
		tableFiles.add(tableFile);

		tableName = "ant_make";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("antenna_id", "INT", false));
		tableFile.addRequiredField("ant_make", "CHAR(3)", true);
		tableFile.addRequiredField("ant_model_num", "CHAR(60)", true);
		tableFiles.add(tableFile);

		tableName = "ant_pattern";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("antenna_id", "INT", false);
		tableFile.addRequiredField("azimuth", "FLOAT", false);
		tableFile.addRequiredField("field_value", "FLOAT", false);
		tableFiles.add(tableFile);

		tableName = "elevation_ant_make";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("elevation_antenna_id", "INT", false));
		tableFile.addRequiredField("ant_make", "CHAR(3)", true);
		tableFile.addRequiredField("ant_model_num", "CHAR(60)", true);
		tableFiles.add(tableFile);

		tableName = "elevation_pattern";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("elevation_antenna_id", "INT", false);
		tableFile.addRequiredField("depression_angle", "FLOAT", false);
		tableFile.addRequiredField("field_value", "FLOAT", false);
		tableFile.addRequiredField("field_value0", "FLOAT", false);
		tableFile.addRequiredField("field_value10", "FLOAT", false);
		tableFile.addRequiredField("field_value20", "FLOAT", false);
		tableFile.addRequiredField("field_value30", "FLOAT", false);
		tableFile.addRequiredField("field_value40", "FLOAT", false);
		tableFile.addRequiredField("field_value50", "FLOAT", false);
		tableFile.addRequiredField("field_value60", "FLOAT", false);
		tableFile.addRequiredField("field_value70", "FLOAT", false);
		tableFile.addRequiredField("field_value80", "FLOAT", false);
		tableFile.addRequiredField("field_value90", "FLOAT", false);
		tableFile.addRequiredField("field_value100", "FLOAT", false);
		tableFile.addRequiredField("field_value110", "FLOAT", false);
		tableFile.addRequiredField("field_value120", "FLOAT", false);
		tableFile.addRequiredField("field_value130", "FLOAT", false);
		tableFile.addRequiredField("field_value140", "FLOAT", false);
		tableFile.addRequiredField("field_value150", "FLOAT", false);
		tableFile.addRequiredField("field_value160", "FLOAT", false);
		tableFile.addRequiredField("field_value170", "FLOAT", false);
		tableFile.addRequiredField("field_value180", "FLOAT", false);
		tableFile.addRequiredField("field_value190", "FLOAT", false);
		tableFile.addRequiredField("field_value200", "FLOAT", false);
		tableFile.addRequiredField("field_value210", "FLOAT", false);
		tableFile.addRequiredField("field_value220", "FLOAT", false);
		tableFile.addRequiredField("field_value230", "FLOAT", false);
		tableFile.addRequiredField("field_value240", "FLOAT", false);
		tableFile.addRequiredField("field_value250", "FLOAT", false);
		tableFile.addRequiredField("field_value260", "FLOAT", false);
		tableFile.addRequiredField("field_value270", "FLOAT", false);
		tableFile.addRequiredField("field_value280", "FLOAT", false);
		tableFile.addRequiredField("field_value290", "FLOAT", false);
		tableFile.addRequiredField("field_value300", "FLOAT", false);
		tableFile.addRequiredField("field_value310", "FLOAT", false);
		tableFile.addRequiredField("field_value320", "FLOAT", false);
		tableFile.addRequiredField("field_value330", "FLOAT", false);
		tableFile.addRequiredField("field_value340", "FLOAT", false);
		tableFile.addRequiredField("field_value350", "FLOAT", false);
		tableFiles.add(tableFile);

		tableName = "elevation_pattern_addl";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("elevation_antenna_id", "INT", false);
		tableFile.addRequiredField("azimuth", "FLOAT", false);
		tableFile.addRequiredField("depression_angle", "FLOAT", false);
		tableFile.addRequiredField("field_value", "FLOAT", false);
		tableFiles.add(tableFile);

		tableName = "am_ant_sys";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, false);
		tableFile.addRequiredField("application_id", "INT", false);
		tableFile.addRequiredField("eng_record_type", "CHAR(1)", true);
		tableFile.addRequiredField("am_dom_status", "CHAR(1)", true);
		tableFile.addRequiredField("ant_mode", "CHAR(3)", true);
		tableFile.addRequiredField("lat_dir", "CHAR(1)", true);
		tableFile.addRequiredField("lat_deg", "INT", false);
		tableFile.addRequiredField("lat_min", "INT", false);
		tableFile.addRequiredField("lat_sec", "FLOAT", false);
		tableFile.addRequiredField("lon_dir", "CHAR(1)", true);
		tableFile.addRequiredField("lon_deg", "INT", false);
		tableFile.addRequiredField("lon_min", "INT", false);
		tableFile.addRequiredField("lon_sec", "FLOAT", false);
		tableFiles.add(tableFile);

		// Make sure all field name lists were found, and open all the files.

		String errmsg = null;
		for (TableFile theFile : tableFiles) {
			if (null == theFile.fieldNames) {
				errmsg = "Field name list not found for table '" + theFile.tableName + "'";
				break;
			}
			if (!theFile.openFile()) {
				theFile.rowCount = -1;
				if (theFile.required) {
					errmsg = "Data file '" + theFile.fileName + "' could not be opened";
					break;
				}
			}
		}

		if (null != errmsg) {
			for (TableFile theFile : tableFiles) {
				theFile.closeFile();
			}
			if (null != errors) {
				errors.reportError(errmsg);
			}
			return null;
		}

		return tableFiles;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Open files for an LMS database.  LMS dump files have field names defining the table and file structure provided
	// in a header line in each file, so this will adapt to changes automatically.  See createAndCopyTable().

	private static ArrayList<TableFile> openLMSTableFiles(File fileDirectory, ZipFile zipFile, ErrorLogger errors) {

		ArrayList<TableFile> tableFiles = new ArrayList<TableFile>();
		TableFile tableFile;

		tableFile = new TableFile("application", null, fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("aapp_application_id", "CHAR(36)", true));
		tableFile.addRequiredField("aapp_callsign", "CHAR(12)", true);
		tableFile.addRequiredField("aapp_receipt_date", "CHAR(20)", true);
		tableFile.addRequiredField("aapp_file_num", "CHAR(20)", true);
		tableFile.addRequiredField("dts_reference_ind", "CHAR(1)", true);
		tableFile.addRequiredField("dts_waiver_distance", "VARCHAR(255)", true);
		tableFile.setDateField(tableFile.addRequiredField("last_update_ts", "CHAR(30)", true));
		tableFile.addRequiredField("channel_sharing_ind", "CHAR(1)", true);
		tableFiles.add(tableFile);

		tableFile = new TableFile("license_filing_version", null, fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("filing_version_id", "CHAR(36)", true));
		tableFile.addRequiredField("active_ind", "CHAR(1)", true);
		tableFile.addRequiredFieldWithIndex("purpose_code", "CHAR(6)", true);
		tableFile.addRequiredField("original_purpose_code", "CHAR(6)", true);
		tableFile.addRequiredFieldWithIndex("service_code", "CHAR(6)", true);
		tableFile.addRequiredField("auth_type_code", "CHAR(6)", true);
		tableFile.addRequiredField("current_status_code", "CHAR(6)", true);
		tableFile.setDateField(tableFile.addRequiredField("last_update_ts", "CHAR(30)", true));
		tableFiles.add(tableFile);

		tableFile = new TableFile("application_facility", null, fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("afac_application_id", "CHAR(36)", true));
		tableFile.addRequiredFieldWithIndex("afac_facility_id", "INT", false);
		tableFile.addRequiredField("afac_channel", "INT", false);
		tableFile.addRequiredField("afac_community_city", "VARCHAR(255)", true);
		tableFile.addRequiredField("afac_community_state_code", "VARCHAR(255)", true);
		tableFile.addRequiredFieldWithIndex("country_code", "CHAR(3)", true);
		tableFile.setDateField(tableFile.addRequiredField("last_update_ts", "CHAR(30)", true));
		tableFile.addRequiredField("licensee_name", "VARCHAR(255)", true);
		tableFile.addRequiredField("station_class_code", "CHAR(6)", true, 7);
		tableFile.addRequiredField("afac_facility_type", "VARCHAR(255)", true);
		tableFile.addRequiredField("am_frequency", "VARCHAR(255)", true, 10);
		tableFiles.add(tableFile);

		tableFile = new TableFile("facility", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("facility_id", "INT", false);
		tableFile.addRequiredField("callsign", "CHAR(12)", true);
		tableFile.addRequiredField("community_served_city", "VARCHAR(255)", true);
		tableFile.addRequiredField("community_served_state", "VARCHAR(255)", true);
		tableFile.addRequiredField("facility_status", "CHAR(6)", true);
		tableFile.addRequiredFieldWithIndex("facility_uuid", "CHAR(36)", true);
		tableFiles.add(tableFile);

		tableFile = new TableFile("facility_applicant", null, fileDirectory, zipFile, 8);
		tableFile.addRequiredFieldWithIndex("facility_uuid", "CHAR(36)", true);
		tableFile.addRequiredField("active_ind", "CHAR(1)", true);
		tableFile.addRequiredFieldWithIndex("contact_ind", "CHAR(1)", true);
		tableFile.addRequiredField("legal_name", "VARCHAR(255)", true);
		tableFiles.add(tableFile);

		tableFile = new TableFile("app_mm_application", null, fileDirectory, zipFile, 9);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("aapp_application_id", "CHAR(36)", true));
		tableFile.addRequiredField("contour_215_protection_ind", "CHAR(1)", true);
		tableFiles.add(tableFile);

		tableFile = new TableFile("app_location", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("aloc_aapp_application_id", "CHAR(36)", true);
		tableFile.addRequiredFieldWithIndex("aloc_loc_record_id", "CHAR(36)", true);
		tableFile.addRequiredField("aloc_loc_seq_id", "INT", false);
		tableFile.addRequiredField("aloc_dts_reference_location_ind", "CHAR(1)", true);
		tableFile.addRequiredField("aloc_lat_dir", "CHAR(1)", true);
		tableFile.addRequiredField("aloc_lat_deg", "INT", false);
		tableFile.addRequiredField("aloc_lat_mm", "INT", false);
		tableFile.addRequiredField("aloc_lat_ss", "FLOAT", false);
		tableFile.addRequiredField("aloc_long_dir", "CHAR(1)", true);
		tableFile.addRequiredField("aloc_long_deg", "INT", false);
		tableFile.addRequiredField("aloc_long_mm", "INT", false);
		tableFile.addRequiredField("aloc_long_ss", "FLOAT", false);
		tableFiles.add(tableFile);

		tableFile = new TableFile("app_antenna", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("aant_aloc_loc_record_id", "CHAR(36)", true);
		tableFile.addRequiredFieldWithIndex("aant_antenna_record_id", "CHAR(36)", true);
		tableFile.addRequiredField("aant_antenna_type_code", "VARCHAR(255)", true);
		tableFile.addRequiredField("aant_rc_amsl", "FLOAT", false);
		tableFile.addRequiredField("aant_rc_haat", "FLOAT", false);
		tableFile.addRequiredField("aant_rotation_deg", "FLOAT", false);
		tableFile.addRequiredField("aant_electrical_deg", "FLOAT", false);
		tableFile.addRequiredField("aant_mechanical_deg", "FLOAT", false);
		tableFile.addRequiredField("aant_true_deg", "FLOAT", false);
		tableFile.addRequiredField("emission_mask_code", "CHAR(6)", true);
		tableFile.addRequiredField("aant_antenna_id", "VARCHAR(255)", true);
		tableFile.addRequiredField("aant_make", "VARCHAR(255)", true);
		tableFile.addRequiredField("aant_model", "VARCHAR(255)", true);
		tableFile.addRequiredField("foreign_station_beam_tilt", "FLOAT", false, 5);
		tableFile.addRequiredField("aant_horiz_rc_amsl", "FLOAT", false);
		tableFile.addRequiredField("aant_vert_rc_amsl", "FLOAT", false);
		tableFile.addRequiredField("aant_horiz_rc_haat", "FLOAT", false);
		tableFile.addRequiredField("aant_vert_rc_haat", "FLOAT", false);
		tableFiles.add(tableFile);

		tableFile = new TableFile("app_antenna_frequency", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("aafq_aant_antenna_record_id", "CHAR(36)", true);
		tableFile.addRequiredField("aafq_power_erp_kw", "FLOAT", false);
		tableFile.addRequiredField("aafq_offset", "VARCHAR(255)", true);
		tableFile.addRequiredField("aafq_horiz_max_erp_kw", "FLOAT", false);
		tableFile.addRequiredField("aafq_horiz_erp_kw", "FLOAT", false);
		tableFile.addRequiredField("aafq_vert_max_erp_kw", "FLOAT", false);
		tableFile.addRequiredField("aafq_vert_erp_kw", "FLOAT", false);
		tableFiles.add(tableFile);

		tableFile = new TableFile("app_antenna_field_value", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("aafv_aant_antenna_record_id", "CHAR(36)", true);
		tableFile.addRequiredField("aafv_azimuth", "FLOAT", false);
		tableFile.addRequiredField("aafv_field_value", "FLOAT", false);
		tableFiles.add(tableFile);

		tableFile = new TableFile("app_antenna_elevation_pattern", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("aaep_antenna_record_id", "CHAR(36)", true);
		tableFile.addRequiredField("aaep_azimuth", "FLOAT", false);
		tableFile.addRequiredField("aaep_depression_angle", "FLOAT", false);
		tableFile.addRequiredField("aaep_field_value", "FLOAT", false);
		tableFiles.add(tableFile);

		tableFile = new TableFile("app_dtv_channel_assignment", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("adca_facility_record_id", "INT", false);
		tableFile.addRequiredField("dtv_allotment_id", "CHAR(36)", true);
		tableFile.addRequiredField("callsign", "CHAR(12)", true);
		tableFile.addRequiredField("rcamsl", "FLOAT", false);
		tableFile.addRequiredField("directional_antenna_ind", "CHAR(1)", true);
		tableFile.addRequiredField("antenna_id", "CHAR(36)", true);
		tableFile.addRequiredField("antenna_rotation", "FLOAT", false);
		tableFile.addRequiredField("emission_mask_code", "VARCHAR(255)", true, 4);
		tableFile.addRequiredField("electrical_deg", "FLOAT", false, 5);
		tableFile.addRequiredField("pre_auction_channel", "INT", false);
		tableFiles.add(tableFile);

		tableFile = new TableFile("lkp_dtv_allotment", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("rdta_dtv_allotment_id", "CHAR(36)", true);
		tableFile.addRequiredField("rdta_service_code", "CHAR(6)", true, 3);
		tableFile.addRequiredField("rdta_city", "VARCHAR(255)", true);
		tableFile.addRequiredField("rdta_state", "CHAR(2)", true);
		tableFile.addRequiredField("rdta_country_code", "CHAR(3)", true, 3);
		tableFile.addRequiredField("rdta_digital_channel", "INT", false);
		tableFile.addRequiredField("rdta_erp", "FLOAT", false);
		tableFile.addRequiredField("rdta_haat", "FLOAT", false);
		tableFile.addRequiredField("rdta_lat_dir", "CHAR(1)", true);
		tableFile.addRequiredField("rdta_lat_deg", "INT", false);
		tableFile.addRequiredField("rdta_lat_min", "INT", false);
		tableFile.addRequiredField("rdta_lat_sec", "FLOAT", false);
		tableFile.addRequiredField("rdta_lon_dir", "CHAR(1)", true);
		tableFile.addRequiredField("rdta_lon_deg", "INT", false);
		tableFile.addRequiredField("rdta_lon_min", "INT", false);
		tableFile.addRequiredField("rdta_lon_sec", "FLOAT", false);
		tableFile.addRequiredField("dts_ref_application_id", "CHAR(36)", true);
		tableFiles.add(tableFile);

		tableFile = new TableFile("lkp_antenna", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("rant_antenna_id", "CHAR(36)", true);
		tableFile.addRequiredField("rant_antenna_record_id", "CHAR(36)", true);
		tableFile.addRequiredField("rant_make", "VARCHAR(255)", true);
		tableFile.addRequiredField("rant_model", "VARCHAR(255)", true);
		tableFiles.add(tableFile);

		tableFile = new TableFile("lkp_antenna_field_value", null, fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("rafv_antenna_record_id", "CHAR(36)", true);
		tableFile.addRequiredField("rafv_azimuth", "FLOAT", false);
		tableFile.addRequiredField("rafv_field_value", "FLOAT", false);
		tableFiles.add(tableFile);

		tableFile = new TableFile("shared_channel", null, fileDirectory, zipFile, 6);
		tableFile.addRequiredFieldWithIndex("application_id", "CHAR(36)", true);
		tableFile.addRequiredFieldWithIndex("facility_id", "INT", false);
		tableFile.addRequiredField("host_ind", "CHAR(1)", true);
		tableFiles.add(tableFile);

		tableFile = new TableFile("app_am_antenna", null, fileDirectory, zipFile, 10);
		tableFile.addRequiredFieldWithIndex("aapp_application_id", "CHAR(36)", true);
		tableFile.addRequiredField("am_ant_mode_code", "CHAR(6)", true);
		tableFile.addRequiredField("dir_ind", "CHAR(1)", true);
		tableFile.addRequiredField("lat_dir", "CHAR(1)", true);
		tableFile.addRequiredField("lat_deg", "INT", false);
		tableFile.addRequiredField("lat_min", "INT", false);
		tableFile.addRequiredField("lat_sec", "FLOAT", false);
		tableFile.addRequiredField("long_dir", "CHAR(1)", true);
		tableFile.addRequiredField("long_deg", "INT", false);
		tableFile.addRequiredField("long_min", "INT", false);
		tableFile.addRequiredField("long_sec", "FLOAT", false);
		tableFiles.add(tableFile);

		tableFile = new TableFile("digital_notification", null, fileDirectory, zipFile, 11);
		tableFile.addRequiredFieldWithIndex("application_id", "VARCHAR(36)", true);
		tableFile.addRequiredField("analog_erp_kw", "VARCHAR(10)", true);
		tableFile.addRequiredField("digital_erp_kw", "VARCHAR(10)", true);
		tableFiles.add(tableFile);

		// Open all the files.

		String errmsg = null;
		for (TableFile theFile : tableFiles) {
			if (!theFile.openFile()) {
				theFile.rowCount = -1;
				if (theFile.required) {
					errmsg = "Data file '" + theFile.fileName + "' could not be opened";
					break;
				}
			}
		}

		if (null != errmsg) {
			for (TableFile theFile : tableFiles) {
				theFile.closeFile();
			}
			if (null != errors) {
				errors.reportError(errmsg);
			}
			return null;
		}

		return tableFiles;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Open table files for a CDBS FM database.  This includes many of the same files as for a CDBS TV data set e.g.
	// application, facility, and pattern tables, making it a little inefficient since a combined data set with both
	// TV and FM tables would be possible.  But it simplifies the rest of the code to have a given data set only able
	// to produce a single record type.  However the CDBS field names data is shared between the two types.

	private static ArrayList<TableFile> openCDBSFMTableFiles(File fileDirectory, ZipFile zipFile, ErrorLogger errors) {

		loadCDBSFieldNames();

		ArrayList<TableFile> tableFiles = new ArrayList<TableFile>();

		String tableName;
		TableFile tableFile;

		tableName = "application";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("application_id", "INT", false));
		tableFile.addRequiredField("file_prefix", "CHAR(10)", true);
		tableFile.addRequiredField("app_arn", "CHAR(12)", true);
		tableFile.setDateField(tableFile.addRequiredField("last_change_date", "CHAR(20)", true));
		tableFiles.add(tableFile);

		tableName = "app_tracking";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("application_id", "INT", false);
		tableFile.addRequiredField("accepted_date", "CHAR(20)", true);
		tableFile.setDateField(tableFile.addRequiredField("last_change_date", "CHAR(20)", true));
		tableFiles.add(tableFile);

		tableName = "facility";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("facility_id", "INT", false));
		tableFile.addRequiredField("fac_callsign", "CHAR(12)", true);
		tableFile.addRequiredField("fac_service", "CHAR(2)", true);
		tableFile.addRequiredField("comm_city", "CHAR(20)", true);
		tableFile.addRequiredField("comm_state", "CHAR(2)", true);
		tableFile.addRequiredField("fac_country", "CHAR(2)", true);
		tableFile.addRequiredField("digital_status", "CHAR(1)", true);
		tableFile.setDateField(tableFile.addRequiredField("last_change_date", "CHAR(20)", true));
		tableFiles.add(tableFile);

		tableName = "fm_eng_data";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("application_id", "INT", false);
		tableFile.addRequiredFieldWithIndex("facility_id", "INT", false);
		tableFile.addRequiredField("eng_record_type", "CHAR(1)", true);
		tableFile.addRequiredField("asd_service", "CHAR(2)", true);
		tableFile.addRequiredField("station_channel", "INT", false);
		tableFile.addRequiredField("fm_dom_status", "CHAR(6)", true);
		tableFile.addRequiredField("lat_dir", "CHAR(1)", true);
		tableFile.addRequiredField("lat_deg", "INT", false);
		tableFile.addRequiredField("lat_min", "INT", false);
		tableFile.addRequiredField("lat_sec", "FLOAT", false);
		tableFile.addRequiredField("lon_dir", "CHAR(1)", true);
		tableFile.addRequiredField("lon_deg", "INT", false);
		tableFile.addRequiredField("lon_min", "INT", false);
		tableFile.addRequiredField("lon_sec", "FLOAT", false);
		tableFile.addRequiredField("rcamsl_horiz_mtr", "FLOAT", false);
		tableFile.addRequiredField("rcamsl_vert_mtr", "FLOAT", false);
		tableFile.addRequiredField("haat_horiz_rc_mtr", "FLOAT", false);
		tableFile.addRequiredField("haat_vert_rc_mtr", "FLOAT", false);
		tableFile.addRequiredField("max_horiz_erp", "FLOAT", false);
		tableFile.addRequiredField("horiz_erp", "FLOAT", false);
		tableFile.addRequiredField("max_vert_erp", "FLOAT", false);
		tableFile.addRequiredField("vert_erp", "FLOAT", false);
		tableFile.addRequiredField("antenna_id", "INT", false);
		tableFile.addRequiredField("ant_rotation", "FLOAT", false);
		tableFile.setDateField(tableFile.addRequiredField("last_change_date", "CHAR(20)", true));
		tableFiles.add(tableFile);

		tableName = "fm_app_indicators";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("application_id", "INT", false));
		tableFile.addRequiredField("da_ind", "CHAR(1)", true);
		tableFiles.add(tableFile);

		tableName = "if_notification";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("application_id", "INT", false));
		tableFile.addRequiredField("analog_erp", "FLOAT", false);
		tableFile.addRequiredField("digital_erp", "FLOAT", false);
		tableFiles.add(tableFile);

		tableName = "ant_make";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("antenna_id", "INT", false));
		tableFile.addRequiredField("ant_make", "CHAR(3)", true);
		tableFile.addRequiredField("ant_model_num", "CHAR(60)", true);
		tableFiles.add(tableFile);

		tableName = "ant_pattern";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("antenna_id", "INT", false);
		tableFile.addRequiredField("azimuth", "FLOAT", false);
		tableFile.addRequiredField("field_value", "FLOAT", false);
		tableFiles.add(tableFile);

		tableName = "elevation_ant_make";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.setKeyField(tableFile.addRequiredFieldWithIndex("elevation_antenna_id", "INT", false));
		tableFile.addRequiredField("ant_make", "CHAR(3)", true);
		tableFile.addRequiredField("ant_model_num", "CHAR(60)", true);
		tableFiles.add(tableFile);

		tableName = "elevation_pattern";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("elevation_antenna_id", "INT", false);
		tableFile.addRequiredField("depression_angle", "FLOAT", false);
		tableFile.addRequiredField("field_value", "FLOAT", false);
		tableFile.addRequiredField("field_value0", "FLOAT", false);
		tableFile.addRequiredField("field_value10", "FLOAT", false);
		tableFile.addRequiredField("field_value20", "FLOAT", false);
		tableFile.addRequiredField("field_value30", "FLOAT", false);
		tableFile.addRequiredField("field_value40", "FLOAT", false);
		tableFile.addRequiredField("field_value50", "FLOAT", false);
		tableFile.addRequiredField("field_value60", "FLOAT", false);
		tableFile.addRequiredField("field_value70", "FLOAT", false);
		tableFile.addRequiredField("field_value80", "FLOAT", false);
		tableFile.addRequiredField("field_value90", "FLOAT", false);
		tableFile.addRequiredField("field_value100", "FLOAT", false);
		tableFile.addRequiredField("field_value110", "FLOAT", false);
		tableFile.addRequiredField("field_value120", "FLOAT", false);
		tableFile.addRequiredField("field_value130", "FLOAT", false);
		tableFile.addRequiredField("field_value140", "FLOAT", false);
		tableFile.addRequiredField("field_value150", "FLOAT", false);
		tableFile.addRequiredField("field_value160", "FLOAT", false);
		tableFile.addRequiredField("field_value170", "FLOAT", false);
		tableFile.addRequiredField("field_value180", "FLOAT", false);
		tableFile.addRequiredField("field_value190", "FLOAT", false);
		tableFile.addRequiredField("field_value200", "FLOAT", false);
		tableFile.addRequiredField("field_value210", "FLOAT", false);
		tableFile.addRequiredField("field_value220", "FLOAT", false);
		tableFile.addRequiredField("field_value230", "FLOAT", false);
		tableFile.addRequiredField("field_value240", "FLOAT", false);
		tableFile.addRequiredField("field_value250", "FLOAT", false);
		tableFile.addRequiredField("field_value260", "FLOAT", false);
		tableFile.addRequiredField("field_value270", "FLOAT", false);
		tableFile.addRequiredField("field_value280", "FLOAT", false);
		tableFile.addRequiredField("field_value290", "FLOAT", false);
		tableFile.addRequiredField("field_value300", "FLOAT", false);
		tableFile.addRequiredField("field_value310", "FLOAT", false);
		tableFile.addRequiredField("field_value320", "FLOAT", false);
		tableFile.addRequiredField("field_value330", "FLOAT", false);
		tableFile.addRequiredField("field_value340", "FLOAT", false);
		tableFile.addRequiredField("field_value350", "FLOAT", false);
		tableFiles.add(tableFile);

		tableName = "elevation_pattern_addl";
		tableFile = new TableFile(tableName, CDBSFieldNamesMap.get(tableName), fileDirectory, zipFile, true);
		tableFile.addRequiredFieldWithIndex("elevation_antenna_id", "INT", false);
		tableFile.addRequiredField("azimuth", "FLOAT", false);
		tableFile.addRequiredField("depression_angle", "FLOAT", false);
		tableFile.addRequiredField("field_value", "FLOAT", false);
		tableFiles.add(tableFile);

		// Make sure all field name lists were found, and open all the files.

		String errmsg = null;
		for (TableFile theFile : tableFiles) {
			if (null == theFile.fieldNames) {
				errmsg = "Field name list not found for table '" + theFile.tableName + "'";
				break;
			}
			if (!theFile.openFile()) {
				theFile.rowCount = -1;
				if (theFile.required) {
					errmsg = "Data file '" + theFile.fileName + "' could not be opened";
					break;
				}
			}
		}

		if (null != errmsg) {
			for (TableFile theFile : tableFiles) {
				theFile.closeFile();
			}
			if (null != errors) {
				errors.reportError(errmsg);
			}
			return null;
		}

		return tableFiles;
	}


	//=================================================================================================================
	// A class used to accumulate a most-recent date and a count of occurrences of that date from a series of dates
	// expressed as strings; the date format string is given to the constructor.

	private static class DateCounter {

		private SimpleDateFormat dateFormat;

		private java.util.Date latestDate;
		private int latestDateCount;
		

		//-------------------------------------------------------------------------------------------------------------

		private DateCounter(String theDateFormat) {

			dateFormat = new SimpleDateFormat(theDateFormat, Locale.US);

			latestDate = new java.util.Date(0);
			latestDateCount = 0;
		}


		//-------------------------------------------------------------------------------------------------------------

		private void add(String theDateStr) {

			java.util.Date theDate = null;
			try {
				theDate = dateFormat.parse(theDateStr);
			} catch (ParseException pe) {
			}

			if ((null != theDate) && !theDate.before(latestDate)) {
				if (theDate.after(latestDate)) {
					latestDate.setTime(theDate.getTime());
					latestDateCount = 1;
				} else {
					latestDateCount++;
				}
			}
		}


		//-------------------------------------------------------------------------------------------------------------

		private String getDate() {

			if (latestDateCount > 0) {
				return dateFormat.format(latestDate);
			}
			return "(unknown)";
		}


		//-----------------------------------------------------------------------------------------------------------------

		private String getCount() {

			return String.valueOf(latestDateCount);
		}


		//-----------------------------------------------------------------------------------------------------------------

		private java.util.Date getDbDate() {

			return latestDate;
		}
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Create the table and copy data from an open file.  If the fieldNames list is non-null that provides a list of
	// field names for the table, otherwise the field names are read from the first line of the dump file (currently
	// LMS files have the field names, CDBS files do not).  In any case, the required fields are those that must appear
	// and/or require specific typing, all other fields are typed generically.  If extraDefinitions is non-null that is
	// appended to the end of the table definition, typically to add indices.  If a date counter is provided and a date
	// field index is set, contents of that field are passed to the counter.  Note the index is to the requiredFields
	// list, not the fieldNames array.  Returns null on success else an error message.

	private static String createAndCopyTable(DbConnection db, TableFile tableFile, DateCounter dateCounter,
			StatusLogger status) {

		String errmsg = null;
		int i, j, lineCount = 1, firstLine = 0;
		boolean didCreate = false;

		try {

			// Read the names list from the file if needed.  The names list from the file will have an extra element
			// due to the row terminator sequence.

			String[] fieldNames = tableFile.fieldNames;
			int fieldCount = 0;
			if (null == fieldNames) {
				fieldNames = tableFile.reader.readLine().split("\\" + TableFile.SEPARATOR);
				lineCount++;
				fieldCount = fieldNames.length - 1;
			} else {
				fieldCount = fieldNames.length;
			}

			// Check the names list, names must be at least three characters long, contain only letters, digits, or
			// the character '_', and there must be at least two names in the list.  This is mainly to confirm a names
			// list just read from file wasn't actually a line of data due to a missing header.  Check for duplicate
			// names, make them unique by adding index as suffix.  Duplicate names appear in some LMS tables for
			// unknown reasons, currently none are required fields so this is safe for now.  Also MySQL is insensitive
			// to case in field names so lowercase all first to check if that creates a duplicate.

			HashSet<String> dupNames = new HashSet<String>();
			char c;
			for (i = 0; i < fieldCount; i++) {
				if (fieldNames[i].length() < 3) {
					fieldCount = -1;
				} else {
					fieldNames[i] = fieldNames[i].toLowerCase();
					for (j = 0; j < fieldNames[i].length(); j++) {
						c = fieldNames[i].charAt(j);
						if (!Character.isLetterOrDigit(c) && ('_' != c)) {
							fieldCount = -1;
							break;
						}
					}
				}
				if (fieldCount < 0) {
					break;
				}
				if (!dupNames.add(fieldNames[i])) {
					fieldNames[i] = fieldNames[i] + String.valueOf(i);
				}
			}
			if (fieldCount < 2) {
				return "Missing or bad field names for data file '" + tableFile.fileName + "'";
			}

			// Compose the table definition.  Along the way build an array of flags indicating fields that contain
			// text, those will have content quoted during the copy, see below.

			for (TableField field : tableFile.requiredFields) {
				field.index = -1;
			}

			boolean[] textFlags = new boolean[fieldCount];

			StringBuilder query = new StringBuilder("CREATE TABLE ");
			query.append(tableFile.tableName);
			query.append(' ');

			String type;
			char sep = '(';

			for (i = 0; i < fieldCount; i++) {

				type = "VARCHAR(255)";
				textFlags[i] = true;

				for (TableField field : tableFile.requiredFields) {
					if ((field.index < 0) && field.name.equals(fieldNames[i])) {
						type = field.type;
						textFlags[i] = field.isText;
						field.index = i;
						break;
					}
				}

				query.append(sep);
				query.append(fieldNames[i]);
				query.append(' ');
				query.append(type);
				sep = ',';
			}

			// Check for missing fields.

			for (TableField field : tableFile.requiredFields) {
				if ((0 == field.version) && (field.index < 0)) {
					return "Required field '" + field.name + "' not found in data file '" + tableFile.fileName + "'";
				}
			}

			// Create the table.

			if ((null != tableFile.extraDefinitions) && (tableFile.extraDefinitions.length() > 0)) {
				query.append(sep);
				query.append(tableFile.extraDefinitions);
			}
			query.append(')');

			db.update(query.toString());
			didCreate = true;

			// Copy file contents into the table.  The rows have an explicit row termination sequence of separator-
			// terminator-separator characters.  Newline and carriage return characters are ignored regardless of
			// context.  No nulls are inserted; blank text fields get empty strings, non-text get 0.

			query.setLength(0);
			query.append("INSERT INTO " + tableFile.tableName + " VALUES (");
			int startLength = query.length();
			firstLine = lineCount;

			int fieldIndex = 0, ci = -1, termstate = 0, dateFieldIndex = -1, keyFieldIndex = -1, duplicatesIgnored = 0,
				badRowsIgnored = 0;
			char cc = '\0';
			boolean firstChar = true, isDuplicate = false;
			StringBuilder values = new StringBuilder(), dateStr = null, keyStr = null;
			HashSet<String> keys = null;

			if ((null != dateCounter) && (null != tableFile.dateField)) {
				dateFieldIndex = tableFile.dateField.index;
				dateStr = new StringBuilder();
			}

			if (null != tableFile.keyField) {
				keyFieldIndex = tableFile.keyField.index;
				keyStr = new StringBuilder();
				keys = new HashSet<String>();
			}

			// Character-by-character read loop.  EOF does not imply a row terminator, that must be explicit.  If the
			// last row is incomplete log that as a warning.

			while (true) {

				ci = tableFile.reader.read();
				if (ci < 0) {
					if (((fieldIndex > 0) || !firstChar) && (null != status)) {
						status.logMessage("Partial row ignored at end of file");
					}
					if (query.length() > startLength) {
						db.update(query.toString());
					}
					break;
				}

				cc = (char)ci;

				// Termstate follows 1-2-3 through the expected separator-terminator-separator characters of the row
				// termination sequence.  The first separator also closes the last field.  If the terminator character
				// is seen after a separator do a short loop expecting the next character to be a separator, if it is
				// not that means the terminator character was actually data, set termstate to -1 and the character
				// will be added back into the field value below.

				if (TableFile.SEPARATOR == cc) {
					if (2 == termstate) {
						termstate = 3;
					} else {
						termstate = 1;
					}
				} else {
					if (1 == termstate) {
						if (TableFile.TERMINATOR == cc) {
							termstate = 2;
							continue;
						} else {
							termstate = 0;
						}
					} else {
						if (2 == termstate) {
							termstate = -1;
						} else {
							termstate = 0;
						}
					}
				}

				// If row termination seen, first check for a completely blank line and skip.  If the field count is
				// incorrect log a message and ignore the row, but also keep count of those and abort if too many.  A
				// chronic issue with LMS data is the presence of un-escaped separators in field data but that should
				// be intermittent.  Otherwise append the values list to the query, if the query string gets too long
				// send it and start a new one.  Then reset state for the next row.

				if (3 == termstate) {

					if ((null != status) && status.isCanceled()) {
						errmsg = "Import canceled";
						break;
					}

					if ((0 == fieldIndex) && firstChar) {
						continue;
					}

					if (fieldIndex != fieldCount) {

						if (null != status) {
							status.logMessage("Row ignored at line " + lineCount + ", bad field count");
						}
						if (++badRowsIgnored > TableFile.MAX_ROWS_IGNORED) {
							errmsg = "Too many bad rows in data file '" + tableFile.fileName + "' at line " +
								lineCount;
							break;
						}

					} else {

						if (isDuplicate) {

							if ((duplicatesIgnored < TableFile.MAX_DUPLICATES_LOGGED) && (null != status)) {
								status.logMessage("Row ignored at line " + lineCount + ", duplicate key");
							}
							duplicatesIgnored++;

						} else {

							if (query.length() > startLength) {
								query.append(",(");
							}
							query.append(values);
							query.append(')');

							tableFile.rowCount++;

							if (query.length() > DbCore.MAX_QUERY_LENGTH) {
								db.update(query.toString());
								query.setLength(startLength);
								firstLine = lineCount;
							}
						}
					}

					fieldIndex = 0;
					firstChar = true;
					values.setLength(0);
					termstate = 0;

				// On a separator, if not past the maximum field count add to the values list as needed.  If the field
				// was empty add a blank string or 0 value, if the field was text and has data add a closing quote.  If
				// this is the field being used for date counting it was also accumulated separately for that, send it
				// to the date counter.

				} else {

					if (1 == termstate) {

						if (fieldIndex < fieldCount) {

							if (firstChar) {
								if (textFlags[fieldIndex]) {
									values.append("''");
								} else {
									values.append('0');
								}
							} else {
								if (textFlags[fieldIndex]) {
									values.append('\'');
								}
							}
							if (fieldIndex < (fieldCount - 1)) {
								values.append(',');
							}

							if (dateFieldIndex == fieldIndex) {
								dateCounter.add(dateStr.toString());
								dateStr.setLength(0);
							}

							if (keyFieldIndex == fieldIndex) {
								isDuplicate = !keys.add(keyStr.toString());
								keyStr.setLength(0);
							}
						}

						fieldIndex++;
						firstChar = true;

					// Process a field character.  Any newline or carriage return here is ignored, also if past the
					// max field count everything is being ignored.  Otherwise if this is the first character of a text
					// field start with a quote.  See above regarding termstate = -1.  Escape quotes and backslashes as
					// needed, else just add the character.  If this field is being used for date counting also add the
					// character to the date string, no escaping there since it will only be sent to the date counter.

					} else {

						if (('\n' != cc) && ('\r' != cc) && (fieldIndex < fieldCount)) {

							if (firstChar && textFlags[fieldIndex]) {
								values.append('\'');
							}

							if (-1 == termstate) {
								values.append(TableFile.TERMINATOR);
								termstate = 0;
							}

							switch (cc) {
								case '\'': {
									values.append("''");
									break;
								}
								case '\\': {
									values.append("\\\\");
									break;
								}
								default: {
									values.append(cc);
									break;
								}
							}

							if (dateFieldIndex == fieldIndex) {
								dateStr.append(cc);
							}

							if (keyFieldIndex == fieldIndex) {
								keyStr.append(cc);
							}

							firstChar = false;

						// Count newlines for error messages.

						} else {
							if ('\n' == cc) {
								lineCount++;
							}
						}
					}
				}
			}

			// Report rows ignored due to duplicate keys, if more than were individually reported.

			if ((duplicatesIgnored > TableFile.MAX_DUPLICATES_LOGGED) && (null != status)) {
				status.logMessage(String.valueOf(duplicatesIgnored) + " rows total ignored with duplicate keys");
			}

		} catch (IOException ie) {
			errmsg = "An I/O error occurred on data file '" + tableFile.fileName + "' at line " + lineCount +
				":\n" + ie;

		} catch (SQLException se) {
			if (firstLine > 0) {
				errmsg = "A database error occurred on data file '" + tableFile.fileName + "' between lines " +
					firstLine + " and " + lineCount + ":\n" + se;
			} else {
				errmsg = "A database error occurred on data file '" + tableFile.fileName + "' at line " + lineCount +
					":\n" + se;
			}
			db.reportError(se);

		} catch (Throwable t) {
			errmsg = "An unexpected error occurred on data file '" + tableFile.fileName + "' at line " + lineCount +
				":\n" + t;
			AppCore.log(AppCore.ERROR_MESSAGE, "Unexpected error", t);
		}

		// If an error occurred after the table was created, drop it again.

		if ((null != errmsg) && didCreate) {
			try {
				db.update("DROP TABLE " + tableFile.tableName);
			} catch (SQLException se) {
				db.reportError(se);
			}
		}

		return errmsg;
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Rename a database.  The new name is assumed to have been checked for length and reserved characters, uniqueness
	// is checked here and enforced by appending the key suffix if needed.  Note when the name is set the is_download
	// flag is cleared, see downloadDatabase().  Also when a data set is deleted the name is always cleared, so only
	// non-deleted sets need to be checked for name uniqueness.

	public static void renameDatabase(String theDbID, Integer theKey, String newName) {
		renameDatabase(theDbID, theKey, newName, null);
	}

	public static void renameDatabase(String theDbID, Integer theKey, String newName, ErrorLogger errors) {

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			return;
		}

		boolean error = false;
		String errmsg = null;

		try {

			db.update("LOCK TABLES ext_db WRITE");

			if (newName.length() > 0) {
				db.query("SELECT ext_db_key FROM ext_db WHERE NOT deleted AND UPPER(name) = '" +
					db.clean(newName.toUpperCase()) + "' AND ext_db_key <> " + theKey);
				if (db.next()) {
					newName = newName + " " + String.valueOf(DbCore.NAME_UNIQUE_CHAR) + String.valueOf(theKey);
				}
				db.update("UPDATE ext_db SET name = '" + db.clean(newName) +
					"', is_download = false WHERE ext_db_key = " + theKey);
			} else {
				db.update("UPDATE ext_db SET name = '', is_download = false WHERE ext_db_key = " + theKey);
			}

		} catch (SQLException se) {
			error = true;
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		try {
			db.update("UNLOCK TABLES");
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (error && (null != errors)) {
			errors.reportError(errmsg);
		}

		reloadCache(theDbID);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Clear the bad-data flag on a database.  See checkImport() for details on how this flag is set.

	public static void clearBadDataFlag(String theDbID, Integer theKey) {
		renameDatabase(theDbID, theKey, null);
	}

	public static void clearBadDataFlag(String theDbID, Integer theKey, ErrorLogger errors) {

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			return;
		}

		boolean error = false;
		String errmsg = null;

		try {

			db.update("LOCK TABLES ext_db WRITE");

			db.update("UPDATE ext_db SET bad_data = false WHERE ext_db_key = " + theKey);

		} catch (SQLException se) {
			error = true;
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		try {
			db.update("UNLOCK TABLES");
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (error && (null != errors)) {
			errors.reportError(errmsg);
		}

		reloadCache(theDbID);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Delete a database.  Keys in the reserved range cannot be deleted, they don't represent actual import databases.
	// When a data set is deleted the name is also cleared, names only apply to current sets.  This will not delete
	// the data set if it is locked.  Also this does not actually drop the database, just updates the content of the
	// index table.  The drops occur in closeDb() which syncs the actual database vs. index state.  That ensures
	// databases remain available if existing objects try to use them in this openDb()/closeDb() context.

	public static void deleteDatabase(String theDbID, Integer theKey) {
		deleteDatabase(theDbID, theKey, null);
	}

	public static void deleteDatabase(String theDbID, Integer theKey, ErrorLogger errors) {

		if ((theKey.intValue() >= RESERVED_KEY_RANGE_START) && (theKey.intValue() <= RESERVED_KEY_RANGE_END)) {
			return;
		}

		ExtDb theDb = getExtDb(theDbID, theKey, errors);
		if ((null == theDb) || theDb.deleted) {
			return;
		}

		DbConnection db = DbCore.connectDb(theDbID, errors);
		if (null == db) {
			return;
		}

		boolean error = false;
		String errmsg = "";

		try {

			db.update("LOCK TABLES ext_db WRITE");

			db.query("SELECT locked, deleted FROM ext_db WHERE ext_db_key = " + theKey);
			if (db.next()) {
				if (db.getBoolean(1)) {
					errmsg = "The station data is in use and cannot be deleted";
					error = true;
				} else {
					if (!db.getBoolean(2)) {
						db.update("UPDATE ext_db SET deleted = true, is_download = false, name = '' " +
							"WHERE ext_db_key = " + theKey);
					}
				}
			}

		} catch (SQLException se) {
			error = true;
			errmsg = DbConnection.ERROR_TEXT_PREFIX + se;
			db.reportError(se);
		}

		try {
			db.update("UNLOCK TABLES");
		} catch (SQLException se) {
			db.reportError(se);
		}

		DbCore.releaseDb(db);

		if (error && (null != errors)) {
			errors.reportError(errmsg);
		}

		reloadCache(theDbID);
	}


	//-----------------------------------------------------------------------------------------------------------------
	// Called by DbCore when a database is being closed.  First scan the actual database list and identify all data
	// set databases by name, then compare that to the content of the ext_db table.  If a database does not exist in
	// the table or is marked deleted, drop the database (see deleteDatabase()).  If there is an entry in the table
	// for which the database does not exist and the entry is not marked deleted, mark it deleted.

	public static synchronized void closeDb(String theDbID) {

		DbConnection db = DbCore.connectDb(theDbID);
		if (null != db) {
			try {

				String rootName = DbCore.getDbName(theDbID);

				HashSet<String> dbNames = new HashSet<String>();
				for (String baseName : getDbBaseNames()) {
					db.query("SHOW DATABASES LIKE '" + rootName + "\\_" + baseName + "%'");
					while (db.next()) {
						dbNames.add(db.getString(1));
					}
				}

				db.update("LOCK TABLES ext_db WRITE");

				HashSet<Integer> markDel = new HashSet<Integer>();
				int theKey;

				db.query("SELECT ext_db_key, db_type, locked FROM ext_db WHERE NOT deleted");

				while (db.next()) {
					theKey = db.getInt(1);
					if (!dbNames.remove(makeDbName(rootName, db.getInt(2), theKey))) {
						markDel.add(Integer.valueOf(theKey));
					}
				}

				if (!markDel.isEmpty()) {
					db.update("UPDATE ext_db SET deleted = true, is_download = false, name = '' WHERE ext_db_key IN " +
						DbConnection.makeKeyList(markDel));
				}

				db.update("UNLOCK TABLES");

				for (String dbName : dbNames) {
					db.update("DROP DATABASE IF EXISTS " + dbName);
				}

			} catch (SQLException se) {
				db.reportError(se);
			}

			try {
				db.update("UNLOCK TABLES");
			} catch (SQLException se) {
				db.reportError(se);
			}

			DbCore.releaseDb(db);
		}
	
		dbCache.remove(theDbID);
		mrCache.remove(theDbID);
		lmsLiveDbCache.remove(theDbID);
	}
}
