//
//  files.c
//  TVStudy
//
//  Copyright (c) 2012-2020 Hammett & Edison, Inc.  All rights reserved.


// Functions related to opening and writing to study and scenario output files, including map files.


#include "tvstudy.h"

#include <sys/file.h>
#include <sys/stat.h>
#include <sys/errno.h>


//---------------------------------------------------------------------------------------------------------------------

static MAPFILE *do_open_mapfile(int fileFormat, char *baseName, char *subDir, int sumFile, int nullFile, int shapeType,
	int nAttr, SHAPEATTR *attrs, char *infoName);
static void write_short_el(short val, FILE *out);
static void write_int_el(int val, FILE *out);
static void write_double_el(double val, FILE *out);
static void write_int_eb(int val, FILE *out);

// Public globals.

int UseDbIDOutPath = 0;   // Set true to use database ID instead of hostname in output file paths.

int UsePendingFiles = 0;   // Set true to use pending-files list, see open_out_file() and clear_pending_files().

// Private globals.

static char *OutPath = NULL;   // Path to output directory, see set_out_path().

// State for the pending-files feature, used to "clean up" unwanted output files post-run.

typedef struct pfl {
	char *fileName;
	char *filePath;
	struct pfl *next;
} PENDING_FILE;

static PENDING_FILE *PendingFiles = NULL;
static int PendingFilesScenarioKey = 0;

// Parameters for KML feature tiling behavior.

#define KML_TILE_SIZE 0.2
#define KML_TILE_LOD 512


//---------------------------------------------------------------------------------------------------------------------
// Set path to output file directory.  This is optional, if not called directly the first file open will set this to
// the path OUT_DIRECTORY_NAME.

void set_out_path(char *thePath) {

	int l = strlen(thePath) + 2;
	if (OutPath) {
		if (strlen(OutPath) < l) {
			OutPath = (char *)mem_realloc(OutPath, l);
		}
	} else {
		OutPath = (char *)mem_alloc(l);
	}
	lcpystr(OutPath, thePath, l);
}


//---------------------------------------------------------------------------------------------------------------------
// Open an output file using the host, study, and scenario names as the directory path, creating directories as needed.
// Optionally the database ID string may be used in place of the hostname string, that is used in situations where it
// is more important for the directory name to be unique and unambiguous than human-readable.  Alternately an explicit
// path for the current study may be set, overriding the default out/host/study behavior; see get_file_path().

// If the UsePendingFiles global is set, the paths and names generated here are accumulated in the PendingFiles list,
// and file names are not reported with STATUS_KEY_OUTFILE messages here.  Later, clear_pending_files() is used to
// either report or delete all pending files and clear the list.

// The subDir argument can be used to build a subdirectory hierarchy.  If subDir is non-NULL it is used as a path
// extension off the scenario directory within which to create the new file.  All directories in the path will be
// created as needed.

// Arguments:

//   fileName  File name, must not include any directory path!
//   subDir    Name of subdirectory or relative path to subdiretory to contain file.

// Return the open file, or NULL on error.

FILE *open_out_file(char *fileName) {
	return open_out_file_dir(fileName, NULL);
}

FILE *open_out_file_dir(char *fileName, char *subDir) {

	static int filePathMax = 0;
	static char *filePath = NULL;

	char *path = get_file_path();
	int pathLen = strlen(path);

	int len = pathLen + strlen(fileName) + 10;
	if (subDir) {
		len = len + strlen(subDir);
	}
	if (len > filePathMax) {
		filePathMax = len;
		filePath = (char *)mem_realloc(filePath, filePathMax);
	}

	// If the pending files list is being used, the list may accumulate across several scenarios.  When the scenario
	// key changes add an entry to the list for the scenario directory itself.  Entries are head-linked so when
	// processing the list a scenario entry will be encountered after all of the files for that scenario; these are
	// ignored when commiting, otherwise deleted.

	PENDING_FILE *pending = NULL;

	if (UsePendingFiles && (ScenarioKey != PendingFilesScenarioKey)) {
		pending = (PENDING_FILE *)mem_zalloc(sizeof(PENDING_FILE));
		pending->next = PendingFiles;
		PendingFiles = pending;
		len = pathLen + 1;
		pending->filePath = (char *)mem_alloc(len);
		lcpystr(pending->filePath, path, len);
		PendingFilesScenarioKey = ScenarioKey;
	}

	// Construct full file path.  If a subdirectory argument is provided create directory(ies) along the way and if
	// needed add to the pending list when a directory is created.

	char *i = path, *o = filePath;
	len = 0;
	while ('\0' != *i) {
		*(o++) = *(i++);
		len++;
	}
	*(o++) = '/';
	len++;
	if (subDir) {
		i = subDir;
		while (1) {
			if (('/' == *i) || ('\0' == *i)) {
				*o = '\0';
				if (mkdir(filePath, 0750)) {
					if (errno != EEXIST) {
						log_error("Cannot create directory '%s'", filePath);
						return NULL;
					}
				} else {
					if (UsePendingFiles) {
						pending = (PENDING_FILE *)mem_zalloc(sizeof(PENDING_FILE));
						pending->next = PendingFiles;
						PendingFiles = pending;
						pending->filePath = (char *)mem_alloc(len + 1);
						lcpystr(pending->filePath, filePath, (len + 1));
					}
				}
				if ('\0' == *i) {
					break;
				}
			}
			*(o++) = *(i++);
			len++;
		}
		*(o++) = '/';
		len++;
	}
	i = fileName;
	while ('\0' != *i) {
		*(o++) = *(i++);
		len++;
	}
	*o = '\0';
	len++;

	// Create the file.

	FILE *out = file_open(filePath, "w");

	// Add a pending files entry for the file itself, that holds both the complete path to the file for deleting, and
	// the relative name for the output file report message later.  The file name as reported in status messages is a
	// relative path from the study root.  If not using pending files, send the message immediately.  This is done
	// even if the open failed since it's possible the file already existed and the open failed for some reason other
	// than access, in which case the file may still need to be processed later.

	if (UsePendingFiles) {
		pending = (PENDING_FILE *)mem_zalloc(sizeof(PENDING_FILE));
		pending->next = PendingFiles;
		PendingFiles = pending;
		pending->filePath = (char *)mem_alloc(len);
		lcpystr(pending->filePath, filePath, len);
	}
	if (subDir) {
		len = snprintf(filePath, filePathMax, "%s/%s/%s", ScenarioName, subDir, fileName) + 1;
	} else {
		len = snprintf(filePath, filePathMax, "%s/%s", ScenarioName, fileName) + 1;
	}
	if (UsePendingFiles) {
		pending->fileName = (char *)mem_alloc(len);
		lcpystr(pending->fileName, filePath, len);
	} else {
		status_message(STATUS_KEY_OUTFILE, filePath);
	}

	return out;
}


//---------------------------------------------------------------------------------------------------------------------
// Construct the path for a file, not including the file name.  See comments for open_out_file() above.

char *get_file_path() {

	if (!ScenarioKey) {
		log_error("get_file_path() called with no scenario loaded");
		return "";
	}

	static int pathMax = 0, initForStudyKey = 0, initForScenarioKey = 0;
	static char *path = NULL;

	if ((StudyKey != initForStudyKey) || (ScenarioKey != initForScenarioKey)) {

		if (!OutPath) {
			set_out_path(OUT_DIRECTORY_NAME);
		}

		char *hostDir = NULL;
		if (UseDbIDOutPath) {
			hostDir = DatabaseID;
		} else {
			if (HostDbName[0]) {
				hostDir = HostDbName;
			}
		}

		int len = strlen(OutPath) + strlen(StudyName) + strlen(ScenarioName) + 10;
		if (hostDir) {
			len += strlen(hostDir);
		}
		if (len > pathMax) {
			pathMax = len;
			path = (char *)mem_realloc(path, pathMax);
		}

		if (mkdir(OutPath, 0750)) {
			if (errno != EEXIST) {
				log_error("Path creation failed: mkdir %s returned errno=%d", OutPath, errno);
				return NULL;
			}
		}

		if (hostDir) {

			snprintf(path, pathMax, "%s/%s", OutPath, hostDir);
			if (mkdir(path, 0750)) {
				if (errno != EEXIST) {
					log_error("Path creation failed: mkdir %s returned errno=%d", path, errno);
					return NULL;
				}
			}

			snprintf(path, pathMax, "%s/%s/%s", OutPath, hostDir, StudyName);
			if (mkdir(path, 0750)) {
				if (errno != EEXIST) {
					log_error("Path creation failed: mkdir %s returned errno=%d", path, errno);
					return NULL;
				}
			}

			snprintf(path, pathMax, "%s/%s/%s/%s", OutPath, hostDir, StudyName, ScenarioName);
			if (mkdir(path, 0750)) {
				if (errno != EEXIST) {
					log_error("Path creation failed: mkdir %s returned errno=%d", path, errno);
					return NULL;
				}
			}

		} else {

			snprintf(path, pathMax, "%s/%s", OutPath, StudyName);
			if (mkdir(path, 0750)) {
				if (errno != EEXIST) {
					log_error("Path creation failed: mkdir %s returned errno=%d", path, errno);
					return NULL;
				}
			}

			snprintf(path, pathMax, "%s/%s/%s", OutPath, StudyName, ScenarioName);
			if (mkdir(path, 0750)) {
				if (errno != EEXIST) {
					log_error("Path creation failed: mkdir %s returned errno=%d", path, errno);
					return NULL;
				}
			}
		}

		initForStudyKey = StudyKey;
		initForScenarioKey = ScenarioKey;
	}

	return path;
}


//---------------------------------------------------------------------------------------------------------------------
// Clear the pending files list and either commit or delete the files, commit is sending STATUS_KEY_OUTFILE messages
// to the front-end app, delete is of course deleting all files, and the enclosing directories.  Errors are ignored
// here, if files fail to delete they are orphaned, the front-end has a clean-up procedure to deal with that.  The
// file list is freed along the way.  Note this also clears the UsePendingFiles flag so that must be re-set if needed
// before the next open_out_file() occurs.

// Arguments:

//   commit  True to commit, false to delete.

void clear_pending_files(int commit) {

	PENDING_FILE *pending;

	while (PendingFiles) {

		pending = PendingFiles;
		PendingFiles = pending->next;
		pending->next = NULL;

		if (!commit) {
			if (pending->fileName) {
				unlink(pending->filePath);
			} else {
				rmdir(pending->filePath);
			}
		}
		mem_free(pending->filePath);
		pending->filePath = NULL;

		if (pending->fileName) {
			if (commit) {
				status_message(STATUS_KEY_OUTFILE, pending->fileName);
			}
			mem_free(pending->fileName);
			pending->fileName = NULL;
		}

		mem_free(pending);
	}

	PendingFilesScenarioKey = 0;
	UsePendingFiles = 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Open a summary-format output file, these combine output from multiple scenario runs so the scenario directory is
// not part of the path and a scenario does not even need to be loaded.

// Arguments:

//   fileName  File name, must not include any directory path!

// Return the open file, or NULL on error.

FILE *open_sum_file(char *fileName) {

	char *path = get_sum_file_path();
	if (!path) {
		return NULL;
	}
	int pathLen = strlen(path);

	static int filePathMax = 0;
	static char *filePath = NULL;

	int len = pathLen + strlen(fileName) + 10;
	if (len > filePathMax) {
		filePathMax = len;
		filePath = (char *)mem_realloc(filePath, filePathMax);
	}

	snprintf(filePath, filePathMax, "%s/%s", path, fileName);
	FILE *out = file_open(filePath, "w");

	status_message(STATUS_KEY_OUTFILE, fileName);

	return out;
}


//---------------------------------------------------------------------------------------------------------------------
// Get path for summary-level files, see comments above.

char *get_sum_file_path() {

	if (!StudyKey) {
		log_error("get_sum_file_path() called with no study open");
		return NULL;
	}

	static int pathMax = 0, initForStudyKey = 0;
	static char *path = NULL;

	if (StudyKey != initForStudyKey) {

		if (!OutPath) {
			set_out_path(OUT_DIRECTORY_NAME);
		}

		char *hostDir = NULL;
		if (UseDbIDOutPath) {
			hostDir = DatabaseID;
		} else {
			if (HostDbName[0]) {
				hostDir = HostDbName;
			}
		}

		int len = strlen(OutPath) + strlen(StudyName) + 10;
		if (hostDir) {
			len += strlen(hostDir);
		}
		if (len > pathMax) {
			pathMax = len;
			path = (char *)mem_realloc(path, pathMax);
		}

		if (mkdir(OutPath, 0750)) {
			if (errno != EEXIST) {
				log_error("Path creation failed: mkdir %s returned errno=%d", OutPath, errno);
				return NULL;
			}
		}

		if (hostDir) {

			snprintf(path, pathMax, "%s/%s", OutPath, hostDir);
			if (mkdir(path, 0750)) {
				if (errno != EEXIST) {
					log_error("Path creation failed: mkdir %s returned errno=%d", path, errno);
					return NULL;
				}
			}

			snprintf(path, pathMax, "%s/%s/%s", OutPath, hostDir, StudyName);
			if (mkdir(path, 0750)) {
				if (errno != EEXIST) {
					log_error("Path creation failed: mkdir %s returned errno=%d", path, errno);
					return NULL;
				}
			}

		} else {

			snprintf(path, pathMax, "%s/%s", OutPath, StudyName);
			if (mkdir(path, 0750)) {
				if (errno != EEXIST) {
					log_error("Path creation failed: mkdir %s returned errno=%d", path, errno);
					return NULL;
				}
			}
		}

		initForStudyKey = StudyKey;
	}

	return path;
}


//---------------------------------------------------------------------------------------------------------------------
// Open a temporary output file, the file is stored at the study level (summary-file level).  The global RunNumber is
// used in the name, also a sequential file count number.  This is not a temporary file at the OS level, it is being
// created to pass data to a post-processing tool.

FILE *open_temp_file() {

	static int fileNumber = 0;

	char tempName[MAX_STRING];

	snprintf(tempName, MAX_STRING, "outtemp_%d_%d", RunNumber, fileNumber++);
	return open_sum_file(tempName);
}


//---------------------------------------------------------------------------------------------------------------------
// Open an OS-level temporary file, file name is written into the argument string which must be MAX_STRING long.

FILE *make_tempfile(char *fileName) {

	lcpystr(fileName, "/tmp/tvstudyXXXXXXXX", MAX_STRING);
	return file_dopen(mkstemp(fileName), "w");
}


//---------------------------------------------------------------------------------------------------------------------
// API for creating output of geographic data in either ESRI shapefile or KML format.  First function is to open a new
// file, or in the case of shapefile format, a set of related files with a common base name.  Originally this only
// supported shapefiles so the API is structured around that format.

// This now has an option to create a "null" file, which means allocate and intialize a MAPFILE structure but don't
// actually open a file or write data.  Only allowed for KML format, such a MAPFILE can be used along with direct calls
// to kml_write_placemark() to write the data to different actual files with the same attribute definitions.  This is
// sort of a hack but it works well enough for now.

// Arguments:

//  fileFormat       File format, MAP_FILE_*
//  baseName         Base name for the file(s), extensions are added.  Must not include any directory path!
//  subDir           Optional sub-directory name, see open_out_file_dir() for details, may be NULL.
//  shapeType        Shape type in the file, SHP_TYPE_*.
//  nAttr            Number of shape attributes (database fields).
//  attrs            Array of attribute descriptions.
//  infoName         Descriptive name, only used for KML in the <NAME> element.  If NULL, baseName is used.

// Return is a MAPFILE structure, or NULL on error.

MAPFILE *open_mapfile(int fileFormat, char *baseName, char *subDir, int shapeType, int nAttr, SHAPEATTR *attrs,
		char *infoName) {
	return do_open_mapfile(fileFormat, baseName, subDir, 0, 0, shapeType, nAttr, attrs, infoName);
}

MAPFILE *open_sum_mapfile(int fileFormat, char *baseName, int shapeType, int nAttr, SHAPEATTR *attrs, char *infoName) {
	return do_open_mapfile(fileFormat, baseName, NULL, 1, 0, shapeType, nAttr, attrs, infoName);
}

MAPFILE *create_kml_mapfile(char *baseName, int shapeType, int nAttr, SHAPEATTR *attrs) {
	return do_open_mapfile(MAP_FILE_KML, baseName, NULL, 0, 1, shapeType, nAttr, attrs, NULL);
}

static MAPFILE *do_open_mapfile(int fileFormat, char *baseName, char *subDir, int sumFile, int nullFile, int shapeType,
		int nAttr, SHAPEATTR *attrs, char *infoName) {

	static int nameLen = 0;
	static char *fileName = NULL;

	int baseLen = strlen(baseName) + 1;
	if ((baseLen + 10) > nameLen) {
		nameLen = baseLen + 10;
		fileName = (char *)mem_realloc(fileName, nameLen);
	}

	// Argument sanity checks.

	if ((MAP_FILE_SHAPE != fileFormat) && (MAP_FILE_KML != fileFormat)) {
		log_error("In open_mapfile: Unknown file format");
		return NULL;
	}

	if ((SHP_TYPE_NULL != shapeType) && (SHP_TYPE_POINT != shapeType) &&
			(SHP_TYPE_POLYLINE != shapeType) && (SHP_TYPE_POLYGON != shapeType)) {
		log_error("In open_mapfile: Unsupported shape type");
		return NULL;
	}

	if (nAttr < 1) {
		log_error("In open_mapfile: Bad attribute count");
		return NULL;
	}

	// Allocate and initialize the map file structure.

	MAPFILE *map = (MAPFILE *)mem_zalloc(sizeof(MAPFILE));
	map->fileFormat = fileFormat;
	map->baseName = mem_alloc(baseLen);
	lcpystr(map->baseName, baseName, baseLen);
	map->shapeType = shapeType;

	map->numAttr = nAttr;
	size_t attrSize = nAttr * sizeof(SHAPEATTR);
	map->attrs = (SHAPEATTR *)mem_alloc(attrSize);
	memcpy(map->attrs, attrs, attrSize);

	// Setup for shapefile format, error check attribute lengths.

	if (MAP_FILE_SHAPE == fileFormat) {

		map->totalSize = 100;

		int i, j, max, reclen = 0;

		for (i = 0; i < nAttr; i++) {

			switch (attrs[i].type) {

				default: {
					close_mapfile(map);
					log_error("In open_mapfile: Unknown attribute type");
					return NULL;
					break;
				}

				case SHP_ATTR_CHAR: {
					max = 254;
					break;
				}

				case SHP_ATTR_NUM: {
					max = 18;
					break;
				}

				case SHP_ATTR_BOOL: {
					max = 1;
					break;
				}
			}

			if ((attrs[i].length < 1) || (attrs[i].length > max)) {
				close_mapfile(map);
				log_error("In open_mapfile: Bad attribute length");
				return NULL;
			}

			if (attrs[i].precision > (attrs[i].length - 1)) {
				close_mapfile(map);
				log_error("In open_mapfile: Bad attribute precision");
				return NULL;
			}

			reclen += attrs[i].length;
		}

		// Create the coordinate system definition file, this is currently hard-coded for NAD83.

		snprintf(fileName, nameLen, "%s.prj", baseName);
		FILE *prj;
		if (sumFile) {
			prj = open_sum_file(fileName);
		} else {
			prj = open_out_file_dir(fileName, subDir);
		}
		if (!prj) {
			close_mapfile(map);
			return NULL;
		}
		fputs("GEOGCS[\"GCS_North_American_1983\",", prj);
		fputs("DATUM[\"D_North_American_1983\",SPHEROID[\"GRS_1980\",6378137,298.257222101]],", prj);
		fputs("PRIMEM[\"Greenwich\",0],", prj);
		fputs("UNIT[\"Degree\",0.017453292519943295]]", prj);
		file_close(prj);

		// Open the shape, index, and DBF files.

		snprintf(fileName, nameLen, "%s.shp", baseName);
		if (sumFile) {
			map->mapFile = open_sum_file(fileName);
		} else {
			map->mapFile = open_out_file_dir(fileName, subDir);
		}
		if (!(map->mapFile)) {
			close_mapfile(map);
			return NULL;
		}

		snprintf(fileName, nameLen, "%s.shx", baseName);
		if (sumFile) {
			map->indexFile = open_sum_file(fileName);
		} else {
			map->indexFile = open_out_file_dir(fileName, subDir);
		}
		if (!(map->indexFile)) {
			close_mapfile(map);
			return NULL;
		}

		snprintf(fileName, nameLen, "%s.dbf", baseName);
		if (sumFile) {
			map->dbFile = open_sum_file(fileName);
		} else {
			map->dbFile = open_out_file_dir(fileName, subDir);
		}
		if (!(map->dbFile)) {
			close_mapfile(map);
			return NULL;
		}

		// Write empty headers to the shape and index files, real data will be over-written on close.

		for (i = 0; i < 25; i++) {
			write_int_el(0, map->mapFile);
			write_int_el(0, map->indexFile);
		}

		// Write header to the DBF file.  This is not documented by ESRI and there is some difference of opinion about
		// the format in other documentation sources; but the following seems to work.  The record count will be
		// written back to the header on close.

		time_t theTime;
		time(&theTime);
		struct tm *tim = localtime(&theTime);

		fputc(3, map->dbFile);
		fputc(tim->tm_mday, map->dbFile);
		fputc((tim->tm_mon + 1), map->dbFile);
		fputc(tim->tm_year, map->dbFile);

		write_int_el(0, map->dbFile);

		short s = 33 + (nAttr * 32);
		write_short_el(s, map->dbFile);

		s = reclen + 1;
		write_short_el(s, map->dbFile);

		for (i = 0; i < 5; i++) {
			write_int_el(0, map->dbFile);
		}

		char tmp[11];

		for (i = 0; i < nAttr; i++) {

			memset(tmp, 0, 11);
			lcpystr(tmp, attrs[i].name, 11);
			fwrite(tmp, 1, 11, map->dbFile);

			fputc(attrs[i].type, map->dbFile);

			write_int_el(0, map->dbFile);

			fputc(attrs[i].length, map->dbFile);
			fputc(attrs[i].precision, map->dbFile);

			fputc(0, map->dbFile);
			fputc(0, map->dbFile);

			fputc(1, map->dbFile);

			for (j = 0; j < 11; j++) {
				fputc(0, map->dbFile);
			}
		}

		fputc('\r', map->dbFile);

	// For KML format, if this is a "null" file that's all; otherwise open file, write header.

	} else {

		if (!nullFile) {

			snprintf(fileName, nameLen, "%s.kml", baseName);
			if (sumFile) {
				map->mapFile = open_sum_file(fileName);
			} else {
				map->mapFile = open_out_file_dir(fileName, subDir);
			}
			if (!(map->mapFile)) {
				close_mapfile(map);
				return NULL;
			}

			if (!infoName) {
				infoName = baseName;
			}
			kml_start(map->mapFile, infoName);
		}
	}

	// All done.

	return map;
}


//---------------------------------------------------------------------------------------------------------------------
// Write a shape to a map file.

// Arguments:

//   map            The map file.
//   ptLat, ptLon   Shape coordinates for points, otherwise not used.
//   pts            Shape data points for poly-line or polygon, otherwise NULL.
//   nParts         Number of parts in a poly-line or polygon shape.
//   iParts         Array of starting index values for parts.  Ignored and may be NULL if nParts is 1.
//   attrData       List of attribute values (always as strings, numbers must be formatted by caller).
//                    The list must match the count and lengths specified when the file was opened.
//   infoName       Optional value for the name property in KML, may be NULL.  Ignored for shapefile.
//   folderName     Optional folder name for KML, if provided, shapes with the same folder name are written to a
//                    temporary file which is merged into the main file on close.  Ignored for shapefile.
//   folderTiling   True to use KML folder structure to group point features into grid tiles with <Region>; ignored for
//                    shapefile, ignored if shape type is not point, ignored if folderName is non-NULL.
//   kmlVis         Visbility flag for KML placemarks.

// Return is 0 for success, -1 for any error.

int write_shape(MAPFILE *map, double ptLat, double ptLon, GEOPOINTS *pts, int nParts, int *iParts, char **attrData,
		char *infoName, char *folderName, int folderTiling, int kmlVis) {

	// Sanity checks on arguments.

	int i, j, j0, j1;

	if ((SHP_TYPE_POLYLINE == map->shapeType) || (SHP_TYPE_POLYGON == map->shapeType)) {

		if (!pts) {
			log_error("In write_shape: Null points argument");
			return -1;
		}

		int minpts;
		if (SHP_TYPE_POLYLINE == map->shapeType) {
			minpts = 2;
		} else {
			minpts = 4;
		}
		if (pts->nPts < minpts) {
			log_error("In write_shape: Bad point count");
			return -1;
		}

		if (nParts < 1) {
			log_error("In write_shape: Bad part count");
			return -1;
    	}

		if (1 == nParts) {

			if (SHP_TYPE_POLYGON == map->shapeType) {
				j1 = pts->nPts - 1;
				if ((pts->ptLat[0] != pts->ptLat[j1]) || (pts->ptLon[0] != pts->ptLon[j1])) {
					log_error("In write_shape: Polygon not closed");
					return -1;
				}
			}

		} else {

			for (i = 0; i < nParts; i++) {

				j0 = iParts[i];
				if (i < (nParts - 1)) {
					j1 = iParts[i + 1] - 1;
				} else {
					j1 = pts->nPts - 1;
				}

				if (((j1 - j0) + 1) < minpts) {
					log_error("In write_shape: Bad part point count");
					return -1;
				}

				if (SHP_TYPE_POLYGON == map->shapeType) {
					if ((pts->ptLat[j0] != pts->ptLat[j1]) || (pts->ptLon[j0] != pts->ptLon[j1])) {
						log_error("In write_shape: Polygon part not closed");
						return -1;
					}
				}
			}
		}
	}

	// Determine bounding box.  This is done even for points because it must update the bounds for the file.

	double wlon, slat, elon, nlat, lon, lat;

	switch (map->shapeType) {

		case SHP_TYPE_NULL:
		default: {
			wlon = 0.;
			slat = 0.;
			elon = 0.;
			nlat = 0.;
			break;
		}

		case SHP_TYPE_POINT: {
			wlon = -ptLon;
			slat = ptLat;
			elon = -ptLon;
			nlat = ptLat;
			break;
		}

		case SHP_TYPE_POLYLINE:
		case SHP_TYPE_POLYGON: {
			wlon = -pts->ptLon[0];
			slat = pts->ptLat[0];
			elon = -pts->ptLon[0];
			nlat = pts->ptLat[0];
			for (i = 1; i < pts->nPts; i++) {
				lon = -pts->ptLon[i];
				lat = pts->ptLat[i];
				if (lon < wlon) {
					wlon = lon;
				}
				if (lat < slat) {
					slat = lat;
				}
				if (lon > elon) {
					elon = lon;
				}
				if (lat > nlat) {
					nlat = lat;
				}
			}
			break;
		}
	}

	if (0 == map->recordNum) {
		map->wlon = wlon;
		map->slat = slat;
		map->elon = elon;
		map->nlat = nlat;
	} else {
		if (wlon < map->wlon) {
			map->wlon = wlon;
		}
		if (slat < map->slat) {
			map->slat = slat;
		}
		if (elon > map->elon) {
			map->elon = elon;
		}
		if (nlat > map->nlat) {
			map->nlat = nlat;
		}
	}

	map->recordNum++;

	// Shapefile output, write the shape and index data.

	if (MAP_FILE_SHAPE == map->fileFormat) {

		int siz;

		switch (map->shapeType) {

			case SHP_TYPE_NULL:
			default: {
				siz = 4;
				break;
			}

			case SHP_TYPE_POINT: {
				siz = 20;
				break;
			}

			case SHP_TYPE_POLYLINE:
			case SHP_TYPE_POLYGON: {
				siz = 44 + (nParts * 4) + (pts->nPts * 16);
				break;
			}
		}

		map->totalSize += siz + 8;

		write_int_eb(((int)ftell(map->mapFile) / 2), map->indexFile);
		write_int_eb((siz / 2), map->indexFile);

		write_int_eb(map->recordNum, map->mapFile);
		write_int_eb((siz / 2), map->mapFile);

		write_int_el(map->shapeType, map->mapFile);

		switch (map->shapeType) {

			case SHP_TYPE_NULL:
			default: {
				break;
			}

			case SHP_TYPE_POINT: {
				lon = -ptLon;
				lat = ptLat;
				write_double_el(lon, map->mapFile);
				write_double_el(lat, map->mapFile);
				break;
			}

			case SHP_TYPE_POLYLINE:
			case SHP_TYPE_POLYGON: {

				write_double_el(wlon, map->mapFile);
				write_double_el(slat, map->mapFile);
				write_double_el(elon, map->mapFile);
				write_double_el(nlat, map->mapFile);

				write_int_el(nParts, map->mapFile);
				write_int_el(pts->nPts, map->mapFile);
				if (1 == nParts) {
					write_int_el(0, map->mapFile);
				} else {
					for (i = 0; i < nParts; i++) {
   						write_int_el(iParts[i], map->mapFile);
					}
				}

				for (i = 0; i < pts->nPts; i++) {
					lon = -pts->ptLon[i];
					lat = pts->ptLat[i];
					write_double_el(lon, map->mapFile);
					write_double_el(lat, map->mapFile);
				}

				break;
			}
		}

		// Write attributes.

		fputc(' ', map->dbFile);

		int len, flen, padleft, padright;

		for (i = 0; i < map->numAttr; i++) {

			len = strlen(attrData[i]);
			flen = map->attrs[i].length;

			if (SHP_ATTR_NUM == map->attrs[i].type) {
				padright = 0;
				if (len > flen) {
					len = flen;
					padleft = 0;
				} else {
					padleft = flen - len;
				}
			} else {
				padleft = 0;
				if (len > flen) {
					len = flen;
					padright = 0;
				} else {
					padright = flen - len;
				}
			}

			for (j = 0; j < padleft; j++) {
				fputc(' ', map->dbFile);
			}
			fwrite(attrData[i], 1, len, map->dbFile);
			for (j = 0; j < padright; j++) {
				fputc(' ', map->dbFile);
			}
		}

	// KML format.  If a folder name is provided, search for an existing folder, create one as needed.  Output for each
	// folder goes to a temporary file.  If the folder name is an empty string or anything goes wrong setting up the
	// folder, output goes directly to the main file.  Creating a folder temporary file will only be attempted once,
	// to avoid churning the filesystem if there are issues like a full temporary file partition.  Attempting to use
	// this call to write to a "null" mapfile does nothing.

	} else {

		FILE *outFile = map->mapFile;
		if (!outFile) {
			return 0;
		}

		// If using folders for tiling, generate the folder name from tile coordinates.  Note the same file can have a
		// combination of tiling folders and named folders; if folderName is non-NULL the tiling flag is ignored.

		char tileName[MAX_STRING];
		if (folderTiling && !folderName && (SHP_TYPE_POINT == map->shapeType)) {
			snprintf(tileName, MAX_STRING, "%d-%d", (int)((ptLat + 90.) / KML_TILE_SIZE),
				(int)((ptLon + 180.) / KML_TILE_SIZE));
			folderName = tileName;
		}

		if (folderName) {

			int len = strlen(folderName);
			if ((len > 0) && (len < MAP_FOLDER_NAME_LEN)) {

				MAP_FOLDER *folder = map->folders, **folderLink = &(map->folders);

				while (folder) {
					if (0 == strcmp(folder->name, folderName)) {
						break;
					}
					folderLink = &(folder->next);
					folder = folder->next;
				}

				if (!folder) {
					folder = (MAP_FOLDER *)mem_zalloc(sizeof(MAP_FOLDER));
					*folderLink = folder;
					lcpystr(folder->name, folderName, MAP_FOLDER_NAME_LEN);
					folder->tempFile = make_tempfile(folder->tempFileName);
					if (folderTiling) {
						folder->isTile = 1;
						folder->nlat = ceil(ptLat / KML_TILE_SIZE) * KML_TILE_SIZE;
						folder->slat = floor(ptLat / KML_TILE_SIZE) * KML_TILE_SIZE;
						folder->elon = ceil(-ptLon / KML_TILE_SIZE) * KML_TILE_SIZE;
						folder->wlon = floor(-ptLon / KML_TILE_SIZE) * KML_TILE_SIZE;
					}
				}

				if (folder->tempFile) {
					outFile = folder->tempFile;
				}
			}
		}

		kml_write_placemark(map, outFile, ptLat, ptLon, pts, nParts, iParts, attrData, infoName, kmlVis);
	}

	// Done.

	return 0;
}


//---------------------------------------------------------------------------------------------------------------------
// Close a map file.  Do final output, write back headers, close all files, and free the structure memory.

// Arguments:

//   map  The map file to close.

#define COPYBUFLEN 1048576

void close_mapfile(MAPFILE *map) {

	int i;

	// Shapefile format, write headers to shape and index files.

	if (MAP_FILE_SHAPE == map->fileFormat) {

		if (map->mapFile && map->indexFile && map->dbFile) {

			fseek(map->mapFile, 0, SEEK_SET);
			fseek(map->indexFile, 0, SEEK_SET);

			write_int_eb(9994, map->mapFile);
			write_int_eb(9994, map->indexFile);

			for (i = 0; i < 5; i++) {
				write_int_eb(0, map->mapFile);
				write_int_eb(0, map->indexFile);
			}

			write_int_eb((map->totalSize / 2), map->mapFile);
			write_int_eb((50 + (4 * map->recordNum)), map->indexFile);

			write_int_el(1000, map->mapFile);
			write_int_el(1000, map->indexFile);

			write_int_el(map->shapeType, map->mapFile);
			write_int_el(map->shapeType, map->indexFile);

			write_double_el(map->wlon, map->mapFile);
			write_double_el(map->wlon, map->indexFile);
			write_double_el(map->slat, map->mapFile);
			write_double_el(map->slat, map->indexFile);
			write_double_el(map->elon, map->mapFile);
			write_double_el(map->elon, map->indexFile);
			write_double_el(map->nlat, map->mapFile);
			write_double_el(map->nlat, map->indexFile);

			for (i = 0; i < 4; i++) {
				write_double_el(0., map->mapFile);
				write_double_el(0., map->indexFile);
			}

			// EOF and record count to DBF file.

			fputc(0x1a, map->dbFile);

			fseek(map->dbFile, 4, SEEK_SET);
			write_int_el(map->recordNum, map->dbFile);
		}

		// Close files.

		if (map->mapFile) {
			file_close(map->mapFile);
			map->mapFile = NULL;
		}

		if (map->indexFile) {
			file_close(map->indexFile);
			map->indexFile = NULL;
		}

		if (map->dbFile) {
			file_close(map->dbFile);
			map->dbFile = NULL;
		}

	// KML format.  If output was sorted into folders, close the temporary files and copy all to the main file.

	} else {

		if (map->mapFile) {

			MAP_FOLDER *folder;
			FILE *tempFile;
			size_t nb;
			unsigned char buf[COPYBUFLEN];

			while (map->folders) {

				hb_log();

				folder = map->folders;
				map->folders = folder->next;
				folder->next = NULL;

				if (folder->tempFile) {

					file_close(folder->tempFile);
					folder->tempFile = NULL;

					tempFile = file_open(folder->tempFileName, "r");
					if (tempFile) {

						fprintf(map->mapFile, "<Folder>\n<name>%s</name>\n", kmlclean(folder->name));

						// If the folder is being used to group features into tiles, write a <Region> element.

						if (folder->isTile) {
							fprintf(map->mapFile, "<Region>\n<LatLonAltBox><north>%.8f</north><south>%.8f</south>",
								folder->nlat, folder->slat);
							fprintf(map->mapFile, "<east>%.8f</east><west>%.8f</west></LatLonAltBox>\n", folder->elon,
								folder->wlon);
							fprintf(map->mapFile, "<Lod><minLodPixels>%d</minLodPixels></Lod>\n</Region>\n",
								KML_TILE_LOD);
						}

						while ((nb = fread(buf, 1, COPYBUFLEN, tempFile))) {
							fwrite(buf, 1, nb, map->mapFile);
						}

						fputs("</Folder>\n", map->mapFile);

						file_close(tempFile);
					}

					unlink(folder->tempFileName);
				}

				mem_free(folder);
			}

			// Close root elements, close files.

			kml_close(map->mapFile);
			map->mapFile = NULL;
		}
	}

	// Free memory.

	if (map->attrs) {
		mem_free(map->attrs);
		map->attrs = NULL;
	}

	if (map->baseName) {
		mem_free(map->baseName);
		map->baseName = NULL;
	}

	mem_free(map);
}


//---------------------------------------------------------------------------------------------------------------------
// Functions for low-level writes to shape files, write 16- and 32-bit integers and 64-bit floating point in proper
// byte order, most writes are little-endian, but 32-bit integers need to be written in big-endian for some data.

static void write_short_el(short val, FILE *out) {

#ifdef __BIG_ENDIAN__
	union {
		short v;
		unsigned char c[2];
	} b;

	b.v = val;
	fputc(b.c[1], out);
	fputc(b.c[0], out);
#else
	fwrite(&val, 1, 2, out);
#endif
}


//---------------------------------------------------------------------------------------------------------------------

static void write_int_el(int val, FILE *out) {

#ifdef __BIG_ENDIAN__
	union {
		int v;
		unsigned char c[4];
	} b;

	b.v = val;
	fputc(b.c[3], out);
	fputc(b.c[2], out);
	fputc(b.c[1], out);
	fputc(b.c[0], out);
#else
	fwrite(&val, 1, 4, out);
#endif
}


//---------------------------------------------------------------------------------------------------------------------

static void write_double_el(double val, FILE *out) {

#ifdef __BIG_ENDIAN__
	union {
		double v;
		unsigned char c[8];
	} b;

	b.v = val;
	fputc(b.c[7], out);
	fputc(b.c[6], out);
	fputc(b.c[5], out);
	fputc(b.c[4], out);
	fputc(b.c[3], out);
	fputc(b.c[2], out);
	fputc(b.c[1], out);
	fputc(b.c[0], out);
#else
	fwrite(&val, 1, 8, out);
#endif
}


//---------------------------------------------------------------------------------------------------------------------

static void write_int_eb(int val, FILE *out) {

#ifdef __BIG_ENDIAN__
	fwrite(&val, 1, 4, out);
#else
	union {
		int v;
		unsigned char c[4];
	} b;

	b.v = val;
	fputc(b.c[3], out);
	fputc(b.c[2], out);
	fputc(b.c[1], out);
	fputc(b.c[0], out);
#endif
}


//---------------------------------------------------------------------------------------------------------------------
// Common header for all KML files.

void kml_start(FILE *kmlFile, char *docName) {

	fputs("<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n", kmlFile);
	fputs("<kml xmlns=\"http://www.opengis.net/kml/2.2\">\n", kmlFile);
	fprintf(kmlFile, "<Document>\n<name>%s</name>\n", kmlclean(docName));

	fputs("<Style id=\"styles\">\n", kmlFile);
	fputs("<IconStyle><color>ffff0000</color><scale>0.5</scale></IconStyle>\n", kmlFile);
	fputs("<LineStyle><color>ff00ff00</color><width>3</width></LineStyle>\n", kmlFile);
	fputs("<PolyStyle><outline>1</outline><fill>0</fill></PolyStyle>\n", kmlFile);
	fputs("</Style>\n", kmlFile);
}


//---------------------------------------------------------------------------------------------------------------------
// Write placemark element to a KML file, see write_shape() for argument details.  This is primarily called from that
// function but may also be called directly to write mixed data to other files.

// Arguments:

//   map       Map file, this provides attribute information only thus it may be a "null" file.
//   kmlFile   Destination for output, this may be map->mapFile or some other file, but is never NULL.
//   ptLat     Feature geography, see write_shape().
//   ptLon
//   pts
//   nParts
//   iParts
//   attrData  Attribute data.
//   infoName  Name for the feature, or NULL.
//   kmlVis    Visbility flag.

void kml_write_placemark(MAPFILE *map, FILE *kmlFile, double ptLat, double ptLon, GEOPOINTS *pts, int nParts,
		int *iParts, char **attrData, char *infoName, int kmlVis) {

	if (MAP_FILE_KML != map->fileFormat) return;

	int i, j, j0, j1;
	double lat, lon;

	// Start place element, write attributes in extended data element.  Include optional name if provided.  Assume
	// default visibility is always 1 so don't write the tag unless it is being set to 0.  This reduces file size a
	// little on potentially large feature sets like the coverage point symbols.

	fputs("<Placemark>\n", kmlFile);
	if (!kmlVis) {
		fputs("<visibility>0</visibility>\n", kmlFile);
	}
	if (infoName) {
		fprintf(kmlFile, "<name>%s</name>\n", kmlclean(infoName));
	}

	fputs("<styleUrl>#styles</styleUrl>\n", kmlFile);

	fputs("<ExtendedData>\n", kmlFile);
	for (i = 0; i < map->numAttr; i++) {
		fprintf(kmlFile, "<Data name=\"%s\"><value>%s</value></Data>\n", map->attrs[i].name, kmlclean(attrData[i]));
	}
	fputs("</ExtendedData>\n", kmlFile);

	switch (map->shapeType) {

		case SHP_TYPE_NULL:
		default: {
			break;
		}

		case SHP_TYPE_POINT: {
			lon = -ptLon;
			lat = ptLat;
			fprintf(kmlFile, "<Point><coordinates>%f,%f</coordinates></Point>\n", lon, lat);
			break;
		}

		case SHP_TYPE_POLYLINE:
		case SHP_TYPE_POLYGON: {

			if (1 == nParts) {

				fputs("<LineString><coordinates>\n", kmlFile);
				for (i = 0; i < pts->nPts; i++) {
					lon = -pts->ptLon[i];
					lat = pts->ptLat[i];
					fprintf(kmlFile, "%f,%f\n", lon, lat);
				}
				fputs("</coordinates></LineString>\n", kmlFile);

			} else {

				fputs("<MultiGeometry>\n", kmlFile);

				for (i = 0; i < nParts; i++) {

					j0 = iParts[i];
					if (i < (nParts - 1)) {
						j1 = iParts[i + 1] - 1;
					} else {
						j1 = pts->nPts - 1;
					}

					fputs("<LineString><coordinates>\n", kmlFile);
					for (j = j0; j <= j1; j++) {
						lon = -pts->ptLon[j];
						lat = pts->ptLat[j];
						fprintf(kmlFile, "%f,%f\n", lon, lat);
					}
					fputs("</coordinates></LineString>\n", kmlFile);
				}

				fputs("</MultiGeometry>\n", kmlFile);
			}

			break;
		}
	}

	fputs("</Placemark>\n", kmlFile);
}


//---------------------------------------------------------------------------------------------------------------------
// Close KML file.

void kml_close(FILE *kmlFile) {

	fputs("</Document>\n</kml>\n", kmlFile);
	file_close(kmlFile);
}


//---------------------------------------------------------------------------------------------------------------------
// Escape special characters in a string for KML output.

char *kmlclean(char *in) {

	static char *out = NULL;
	static int outMaxLen = 0;

	if (!out) {
		outMaxLen = MAX_STRING;
		out = (char *)mem_alloc(outMaxLen + 1);
	}

	int i, len = strlen(in), o = 0, lencpy, c;
	char *cpy;

	for (i = 0; i < len; i++) {
		switch (in[i]) {
			case '&':
				cpy = "&amp;";
				lencpy = 5;
				break;
			case '<':
				cpy = "&lt;";
				lencpy = 4;
				break;
			case '>':
				cpy = "&gt;";
				lencpy = 4;
				break;
			case '"':
				cpy = "&quot;";
				lencpy = 6;
				break;
			default:
				cpy = NULL;
				lencpy = 1;
				break;
		}
		if ((o + lencpy) >= outMaxLen) {
			outMaxLen += MAX_STRING;
			out = (char *)mem_realloc(out, (outMaxLen + 1));
		}
		if (cpy) {
			for (c = 0; c < lencpy; c++) {
				out[o++] = cpy[c];
			}
		} else {
			out[o++] = in[i];
		}
	}
	out[o] = '\0';

	return out;
}


//---------------------------------------------------------------------------------------------------------------------
// Estimate the number of tiles (temporary output files) that would be used for KML folder-based geographic tiling,
// see write_shape() for details.  This is used while grouping sources to study together on one grid during a global-
// grid study.  In that case each source may have individual output files open for KML features and those may also use
// temporary files for tiling, so the open-file count can get quite high and the number of sources studied on one grid
// may have to be limited.  This is only an estimate so just assume the entire grid will need open files.

int get_kml_tile_count(INDEX_BOUNDS grid) {

	return (((int)(((double)(grid.northLatIndex - grid.southLatIndex) / 3600.) / KML_TILE_SIZE) + 1) *
		((int)(((double)(grid.westLonIndex - grid.eastLonIndex) / 3600.) / KML_TILE_SIZE) + 1));
}


//---------------------------------------------------------------------------------------------------------------------
// Wrappers around fopen(), fdopen(), and fclose() to track file use stats.

static int openFileCount = 0;

FILE *file_open(char *filename, char *mode) {
	FILE *f = fopen(filename, mode);
	if (f) {
		openFileCount++;
	} else {
		log_error("Cannot open file '%s' (%d %d)", filename, errno, openFileCount);
	}
	return f;
}

FILE *file_dopen(int fd, char *mode) {
	if (fd < 0) {
		log_error("Cannot open file (%d %d)", errno, openFileCount);
		return NULL;
	}
	openFileCount++;
	return fdopen(fd, mode);
}

void file_close(FILE *stream) {
	fclose(stream);
	openFileCount--;
}
