#!/usr/bin/perl -w
#
# $Header: /home/bnelson/splat/unused/RCS/try_coordinate.pl,v 1.4 2010-01-30 19:55:06-06 bnelson Exp $
#
# This unpublished proprietary work is Copyright (C) 2010 by
# Bob Nelson of Frisco, Texas. All rights reserved worldwide.
#
# MODULE NAME: try_coordinate.pl
#
# AUTHOR: Bob Nelson <bnelson@nelsonbe.com>
#
# DATE: 30 January 2010
#
# DESCRIPTION: Attempts to use ``Coordinate.pm'' to convert from NAD27 to
# NAB83 as required for proper QTH files.
#
# USAGE:
#
#   ./unused/try_coordinate.pl lat_degrees long_degrees
#
# USAGE EXAMPLE (for KCBS-TV):
#
#   ./unused/try_coordinate.pl 34.227222222 -118.066666667
#
# a). Go to the KCBS page on the FCC TV query.
#
# b). Make a note of the transmitter coordinates (shown as NAD27).
#
# c). Convert from DMS to degrees for both latitude and longitude.
#
# d). Provide those two values to this script.
#
# e). Look at the results for the WGS84 data and compare it to the ASRN
#     values on the FCC page.

BEGIN {
    # Adjust the Perl search path to find ``Coordinate''.

    unshift @INC, "$ENV{'HOME'}/splat";
}

use Coordinate;
use strict;

die "usage: $0 lat_degrees long_degrees\n" unless @ARGV == 2;

my $nad27_lat = $ARGV[0];
my $nad27_long = $ARGV[1];

# 1). Load the NAD27 coordindates into the ``$position'' object;

my $position = Coordinate->new();

$position->latitude($nad27_lat);
$position->longitude($nad27_long);

$position->datum("NAD27 CONUS MEAN:W of Mississippi/Except Louisiana/Minnesota/Missouri");

# 2). Obtain the NAD27 values in DMS notation:

$position->degrees_minutes_seconds();
my $dms_nad27_lat = $position->formatted_latitude();
my $dms_nad27_long = $position->formatted_longitude();

# 3). Convert from NAD27 to WGS84 with a new object:

my $wgs84 = $position->datum_shift_to_wgs84();

# 4). Obtain the WGS84 values in DMS notation:

$wgs84->degrees_minutes_seconds();
my $dms_wgs84_lat = $wgs84->formatted_latitude();
my $dms_wgs84_long = $wgs84->formatted_longitude();

# 5). Show the results:

printf("NAD27(%.8f/%.8f [%s/%s])\nWGS84(%.8f/%.8f [%s/%s])\n",
    $nad27_lat, $nad27_long,
    $dms_nad27_lat, $dms_nad27_long,
    $wgs84->latitude(), $wgs84->longitude(),
    $dms_wgs84_lat, $dms_wgs84_long);
