#!/bin/bash
#
# $Header: /home/bnelson/splat/unused/RCS/get_qth_files.sh,v 1.10 2010-01-29 15:22:46-06 bnelson Exp $
#
# This unpublished proprietary work is Copyright (C) 2010 by
# Bob Nelson of Frisco, Texas. All rights reserved worldwide.
#
# MODULE NAME: get_qth_files.sh
#
# AUTHOR: Bob Nelson <bnelson@nelsonbe.com>
#
# DATE: 28 January 2010
#
# DESCRIPTION: Gets the QTH, LRP and AZ files (if available) for
# use with ``splat_runner.sh''.
#
# USAGE:
#
#   ./get_qth_files.sh calls app_id
#
# USAGE INSTRUCTIONS AND EXAMPLE:
#
# 1). Go the FCC's TV query website and get the CDBS application ID *NOT*
#     the facility ID for the station:
#
#       http://www.fcc.gov/fcc-bin/tvq?call=kpfw
#
#     Or go to RabbitEars, click on the channel and get the application
#     ID (the first line) from the ``QTH'' file link.
#
#     WARNING! You must first visit the RabbitEars page in order for
#              the QTH and other files to get created. They are not
#              present until created on-the-fly when the channel page
#              is visited!
#
#              That's why Bob Nelson is exploring an alternate method
#              of creating these files from the FCC site.
#
# 2). In this case, channel 61's desired app ID is 1267717.
#
# 3). Then execute either of these commands, the 's' prefix will be
#     added if not already present:
#
#       ./get_qth_files.sh kpfw 1267717
#       ./get_qth_files.sh kpfw s1267717

if [ "$#" -ne 2 ]; then
    echo "usage: $0 calls app_id" >&2
    exit 1
fi

CALLS="$(echo "$1" | tr '[A-Z]' '[a-z]')"           # Lowercase calls
APP_ID="$(echo "$2" | sed 's/^\([0-9].*\)/s\1/')"   # Add 's' prefix

# Make sure that the downloaded file is correct, otherwise remove it. Note
# that a clean slate is how this process starts by removing prior instances
# of possibly faulty fetches.

rm -f "$APP_ID".{qth,lrp,az}{,.[0-9]}

if ! wget -q http://rabbitears.info/splat/temp/"$APP_ID".{qth,lrp,az}; then
    echo "$0 error: can't fetch $APP_ID from RabbitEars, exiting..." >&2
    exit 1
fi

for ext in qth lrp az; do
    if grep -isq '<html' "$APP_ID.$ext"; then
        rm -f "$CALLS.$ext" "$APP_ID.$ext"
    fi

    if [ -f "$APP_ID.$ext" ]; then
        if mv "$APP_ID.$ext" "$HOME/splat/input/$CALLS.$ext"; then
            echo "Successfully fetched: $CALLS.$ext, moved to ./input directory"
        else
            echo "$0 error: failed to move $APP_ID.$ext to ./input/$CALLS.$ext" >&2
            exit 1
        fi
    else
        # Failure to get an ``.az'' file is not an error, since the
        # station may be non-DA:

        if [ "$ext" != 'az' ]; then
            echo "$0 error: failed to obtain $APP_ID.$ext, exiting..." >&2
            exit 1
        fi
    fi
done
