#!/bin/bash
#
# $Header: /home/bnelson/splat/RCS/splat_runner.sh,v 1.51 2010-03-07 22:24:37-06 bnelson Exp $
# vim: ts=4 sw=4 tw=90:
#
# MODULE NAME: splat_runner.sh
#
# ORIGINAL AUTHOR: Trip Ericson <webmaster@rabbitears.info>
#
# MODIFIED BY: Bob Nelson <bnelson@nelsonbe.com>
#
# This is free software. You may use it as you wish and freely distribute it.
#
# There are only these two requests:
#
# 1). Please keep this notice and the Copyright intact.
#
# 2). If you make improvements, please send them to the current author:
#
#     Bob Nelson <bnelson@nelsonbe.com>
#
# THIS SOFTWARE AND ITS SUB-PACKAGES ARE PROVIDED AS IS AND WITHOUT ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE OR
# EVEN NO PURPOSE AT ALL.
#
# DATE: 26 January 2010
#
# DESCRIPTION: Wrapper script for ``splat'' and Doug Lung's Perl
# post-processing Perl scripts.
#
# USAGE:
#
#       ./splat_runner.sh [-hmtuw -l rx_height -r range] callsign
#
#       OPTIONS:
#
#       -h: Shows usage message on stdout and exits successfully.
#
#       -l (height in m): Specify a receiving antenna height other
#                         than the 4 meter default.
#
#       -m: Send mail to Trip Ericson <webmaster@rabbitears.info>
#           when upload completes. This requires the ``-u'' option.
#
#       -r (range in km): Specify a range other than the default
#                         of 160. This also overrides what's in
#                         the ``input/$CALLS.r'' file.
#
#       -t: Test the options. The actual lengthy ``SPLAT!'' process
#           is skipped (and all subprocesses). This is useful for
#           testing the e-mail, uploading an HTML file generation.
#           This presumes the output files are already present.
#
#       -u: Upload completed files to RabbitEars. This requires
#           ``ncftpput'' to be present.
#
#       -w: Generate an HTML file for local web browsing.
#
# USAGE EXAMPLES:
#
#       ./splat_runner.sh -h
#       ./splat_runner.sh kcen
#       ./splat_runner.sh -r 128 -w k31gl
#       ./splat_runner.sh -m -u -w kten
#       ./splat_runner.sh kbvo-cd
#
# NOTES, CAVEATS AND CHANGES:
#
# 0). The script name was changed to not conflict with Trip.
# 1). Error checking was added for all of Doug Lung's Perl scripts.
# 2). The ``/tmp'' is used for generated files to save space in the CWD.
# 3). The application_id is now deduced from the top line of the QTH file.
#     a). LOOK! WE'RE USING THE APPLICATION ID AND *NOT* THE FACILITY ID!
#     b). This script drops the leading 's' from Trip's QTH files to get
#         a numeric application ID.
# 4). The completed files now are written to the ``output'' directory.
# 5). The input files are now in the ``input'' directory.
# 6). The Perl scripts are called via the interpreter line.
# 7). Option added for range (``-r'') as well as ``$input/CALLS.r'' file
#     to set a different range.
# 8). A receiving antenna height other than 4 meters can be specified with
#     the ``-l'' flag, that's a lowercase ell.
# 9). 07 MAR 2010: [REN] - Added support for ``-s <NUM>'' site, ID override.

# ---------------------------------------------------------------------------- #
# -------- Adjust these variables as required for mail and uploading --------- #
# ---------------------------------------------------------------------------- #

MAIL_RCPT="webmaster+map@rabbitears.info bnelson@nelsonbe.com"
UPLOAD_HOST="www.rabbitears.info"
UPLOAD_USER="u35536755-coverage"
UPLOAD_PASS="uploader"

# ---------------------------------------------------------------------------- #
# -------- IN MOST CASES, NOTHING BELOW THIS LINE NEEDS ADJUSTMENT ----------- #
# ---------------------------------------------------------------------------- #

readonly TER_DIR="$HOME/splat/terraindata"
readonly OUT_DIR="./output"

readonly SPLAT="splat"
readonly CONVERT="convert"
readonly PERL="perl"
readonly NCFTPPUT="ncftpput"
readonly MAILBIN="mail"
readonly SPLAT2HTML="./splat2html.sh"
readonly USAGE="usage: $0 [-hmtuw -l rx_height -r range -s site] callsign"

for exe in $SPLAT $CONVERT $PERL $NCFTPPUT $MAILBIN; do
    if ! type "$exe" >/dev/null 2>&1; then
        echo "error $0: $exe utility not found in path, exiting..." >&2
        exit 1
    fi
done

if [ ! -d "$TER_DIR" ]; then
    echo "error $0: directory not found $TER_DIR, exiting..." >&2
    exit 1
fi

# Set optional flags to default (off) state before processing command-line.

do_mail=0
do_upload=0
do_web=0
test_only=0
range_override=0
rx_agl_override=''
site_id=0

while getopts "hmtuwl:r:s:" option_letter; do
    case $option_letter in
        h)  echo "$USAGE"
            exit 0
            ;;
        l)  rx_agl_override="$OPTARG"
            if ! echo "$OPTARG" | egrep -sq '^[.0-9]+$'; then
                echo "$0 error: value for -r must be a numeric height in meters, exiting..." >&2
                exit 1
            fi
            ;;
        m)  do_mail=1
            ;;
        r)  range_override="$OPTARG"
            if ! echo "$OPTARG" | egrep -sq '^[0-9]+$'; then
                echo "$0 error: value for -r must be a numeric range in km, exiting..." >&2
                exit 1
            fi
            ;;
        s)  site_id="$OPTARG"
            if ! echo "$OPTARG" | egrep -sq '^[0-9]+$'; then
                echo "$0 error: value for -s must be a numeric site ID, exiting..." >&2
                exit 1
            fi
            ;;
        t)  test_only=1
            ;;
        u)  do_upload=1
            ;;
        w)  do_web=1
            ;;
        *)  echo "$USAGE" >&2
            exit 1
            ;;
    esac
done

shift $((--OPTIND))

# Check ``-m''' option for sanity, since uploading (``-u'') is prerequisite.

if [ "$do_mail" -eq 1 -a "$do_upload" -eq 0 ]; then
    echo "$0 warning: -m flag specified without -u, will not send mail, continuing..." >&2
fi

if [ "$#" -ne 1 ]; then
    echo "$USAGE" >&2
    exit 1
fi

readonly CALLS="$(basename "$1" .qth)"
readonly QTH="input/$CALLS.qth"
readonly RANGE_FILE="input/$CALLS.r"
readonly DATA='/tmp/splat_data.txt'

# 29 JAN 2010: [REN] - Handle the various ways to set the RANGE to
# something other than the default of 160 (km), with the ``-r'' flag
# always trumping anything else, and then consulting an ``.r'' file
# in the input directory matching the call letters.

RANGE="160"

if [ "$range_override" -gt 0 ]; then
    RANGE="$range_override"
elif [ -f "$RANGE_FILE" ]; then
    RANGE="$(head -n 1 "$RANGE_FILE" | egrep '^[0-9]+$')"
fi

if [ ! -r "$QTH" ]; then
    echo "$0 error: unable to read $QTH, exiting..." >&2
    exit 1
fi

readonly APP_ID="$(head -n1 "$QTH" | sed 's/^s//')"
readonly INFO="$OUT_DIR/$CALLS-$APP_ID-$site_id.info"
readonly OUT="$OUT_DIR/$APP_ID-$site_id"

echo "Starting $SPLAT for $CALLS, application ID $APP_ID, patience..."

# 01 FEB 2010: [REN] - BUG FIXED by creating the ``output'' directory.

mkdir -p "$OUT_DIR"

# 29 JAN 2010: [REN] - Modified to suppress the generation of the large PPM
# file since none of Doug Lung's scripts use it for input. This requires
# a patched version of ``splat'' to work, which is checked in the next
# condition.

NO_PPM_MAP=''
if "$SPLAT" | grep -sq 'use /dev/null for no map'; then
    NO_PPM_MAP='-o /dev/null'
fi

# 29 JAN 2010: [REN] - Optional override for default receiver antenna height
# in meters AGL.

RX_AGL="4"
if [ "$rx_agl_override" ]; then
    RX_AGL="$rx_agl_override"
fi

if [ "$test_only" -eq 0 ]; then
    "$SPLAT" -metric -t "$QTH" -L "$RX_AGL" -d "$TER_DIR" -R "$RANGE" -ano "$DATA" $NO_PPM_MAP

    if [ "$?" -ne 0 ]; then
        echo "$0 error: $SPLAT failed, exiting..." >&2
        exit 1
    fi
else
    echo "$SPLAT would be run with these flags: -metric -t "$QTH" -L "$RX_AGL" -d "$TER_DIR" -R "$RANGE" -ano "$DATA" $NO_PPM_MAP"
fi

echo "Now running Doug Lung's Perl scripts on $CALLS, application ID $APP_ID:"
if [ "$test_only" -eq 0 ]; then
    ./sp3gridproc.pl || { echo "sp3gridproc.pl failed, exiting..." >&2; exit 1; }
    ./3xy2png-rx-colors.pl || { echo "3xy2png-rx-colors.pl failed, exiting..." >&2; exit 1; }
fi

echo "Reducing the opacity for overlay:"
if [ "$test_only" -eq 0 ]; then
    $CONVERT pngplot.png -channel RGBA -transparent "rgb(128,128,128)" /tmp/pngplot-out.png
    rm -f pngplot.png
fi

echo "Making background transparent:"
if [ "$test_only" -eq 0 ]; then
    $CONVERT /tmp/pngplot-out.png -channel alpha -evaluate Divide 2 /tmp/pngplot-out.png
fi

echo "Renaming files for RabbitEars and performing cleanup:"
if [ "$test_only" -eq 0 ]; then
    mv -f /tmp/pngplot-out.png "$OUT.png" && chmod 0600 "$OUT.png"
    mv -f /tmp/pngplot.txt "$OUT.txt" && chmod 0600 "$OUT.txt"
fi

# Extract the transmitter coordinates to the ``.info'' file, which
# allows an HTML file to be created using ``splat2html.sh''.

if [ "$test_only" -eq 0 ]; then
    head -n3 "$QTH" | tail -n 2 > "$INFO" && chmod 0600 "$INFO"
    echo "$RX_AGL" >> "$INFO"

    rm -f "$DATA" /tmp/{clean_xy.txt,sorted_splat_xy.txt,splat_xy.txt}
    rm -f {.,input}/{pngplot.kml,*.ppm,*.scf,*site_report.txt}
fi

echo "Completed, $CALLS, application ID $APP_ID"

# If flow gets this far, then the map was successfully generated, so
# process the command-line flags, first the upload and mail and then
# the HTML page generation.

if [ "$do_upload" -eq 1 ]; then
    echo "Uploading files to $UPLOAD_HOST:"

    # 03 FEB 2010: [REN] - BUG FIXED by trying up to 3 times to upload
    # the files (in the event of a socket error), with a wait after the
    # first error.

    upload_result='-1'

    for try in 1 2 3; do
        $NCFTPPUT -u "$UPLOAD_USER" -p "$UPLOAD_PASS" "$UPLOAD_HOST" \
            . "$OUT.png" "$OUT.txt"

        upload_result="$?"

        if [ "$upload_result" -eq 0 ]; then
            break
        fi

        sleep 30        # Rest for a half-minute before trying upload again
    done

    upload_message="was successful"
    result_message="OK"

    if [ "$upload_result" -ne 0 ]; then
        upload_message="failed"
        result_message="ERROR"
    fi

    if [ "$do_mail" -eq 1 ]; then
        echo "Mailing notification to $MAIL_RCPT, upload $upload_message"

        # 02 FEB 2010: [REN] - Add the hostname and current external IP
        # address for each upload attempt.

        send_host="$(hostname)"
        external_ip_addr="$(wget -qO - 'http://www.smartsbroadcast.com/ip/' |\
                            egrep '^[[:space:]]*[0-9]+\.[0-9]+\.')"

        cat <<_EOF_ | "$MAILBIN" -s "$CALLS L-R map upload $result_message" $MAIL_RCPT
The upload of maps for $CALLS (application ID: $APP_ID) $upload_message.

The files were: $OUT.png and $OUT.txt.
The range value used for this map was: $RANGE km.
The receiver antenna height was: $RX_AGL meters (AGL).
This message came from host: $send_host, external IP address: $external_ip_addr.

This is an automated message, please do not reply to it. Instead contact
the sender by some other means.
_EOF_
    fi
fi

if [ "$do_web" -eq 1 -a -x "$SPLAT2HTML" ]; then
    "$SPLAT2HTML" "$INFO"
fi
