#!/bin/bash
#
# $Header: /home/bnelson/splat/RCS/splat_iterator.sh,v 1.20 2010-03-19 22:52:55-05 bnelson Exp bnelson $
# vim: ts=4 sw=4 tw=99:
#
# This unpublished proprietary work is Copyright (C) 2010 by
# Bob Nelson of Frisco, Texas. All rights reserved worldwide.
#
# This is free software. You may use it as you wish and freely distribute it.
#
# There are only these two requests:
#
# 1). Please keep this notice and the Copyright intact.
#
# 2). If you make improvements, please send them to the current author:
#
#     Bob Nelson <bnelson@nelsonbe.com>
#
# THIS SOFTWARE AND ITS SUB-PACKAGES ARE PROVIDED AS IS AND WITHOUT ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE OR
# EVEN NO PURPOSE AT ALL.
#
# MODULE NAME: splat_iterator.sh
#
# AUTHOR: Bob Nelson <bnelson@nelsonbe.com>
#
# DATE: 31 January 2010
#
# DESCRIPTION: An ``uber processor'' to quickly batch up the operation of
# ``splat'' DTV coverage maps by executing these two commands with the
# apropriate argument list for each:
#
#       ./fcc2qth.pl
#       ./splat_runner.sh
#
# The input file consists of rows and columns. Each row is specifies a
# map to create with these fields:
#
#    1). Filename - an arbitrary but easily recognized name such as:
#
#                    k31gl_31
#                    k31gl_33
#                    ktvt_23kw
#                    ktvt_750kw
#                    kdtn
#
#    2). Call letters - the FCC callsign such as:
#
#                    kdtn
#                    wls-tv
#                    wfaa
#
#        * The filename and call letters need not match. But the call letters
#          used should those that will be found by the FCC TV query.
#
#    3). FCC File - the FCC file identifier such as these, all of which
#                   equate to the same facility:
#
#                    BPCDT20080317AGO
#                    bpcdt20080317ago
#                    BPCDT-20080317AGO
#                    Bpcdt-20080317AGO
#
#    4). OPTIONAL range - a 2 or 3 digit range to override the default of
#                         160 km. If this field isn't present, then 160 is
#                         used.
#
#    5). OPTIONAL NAD27 > WGS84 supression - The characters ``-n'' as the
#                         last field will use the coordindates in the FCC
#                         database ``as is'' without application of the
#                         conversion from NAD27 to WGS84.
#
# USAGE:
#
#       ./splat_iterator.sh [-p -t] input_file...
#
# USAGE EXAMPLES:
#
#       ./splat_iterator.sh ./input/beaumont-tx
#       ./splat_iterator.sh ./input/beaumont-tx ./input/amarillo-tx ./input/corpus_christi-tx
#       ./splat_iterator.sh ./input/*-tx
#
#       ./splat_iterator.sh -p ./input/*-tx     <- Protects facilities already done
#
#       ./splat_iterator.sh -t ./input/*-tx     <- Test only mode, no commands executed
#
# NOTES AND CAVEATS:
#
# 1). Error handling is currently minimal. That will improve in subsequent
#     versions. It presumes you know what you're doing.
#
# 2). Comments in the code are very sparse. The code should be simple
#     enough to follow but as this evolves, comments will be added.
#
# 3). EXAMINE THE LOGFILE! The ``$HOME/splat/log'' directory is comtains
#     a  blow-by-blow account of what transpires during script execution.
#
# 4). In addition to the logfile. all messages also appear on the terminal.
#
# 5). 07 FEB 2010: [REN] - The ``-p'' (protective mode) flag allows an update
#     to be done against a market directive file, If the facility already has
#     a matching CDBS application ID in a ``QTH'' file, it gets skipped. See
#     more in ``README.splat''.
#
# 6). 07 MAR 2010: [REN] - Allows for either FCC query or ``splatinfo.php''
#     lookup from Trip Eriscon's RabbitEars site. See ``README.splat'' for
#     the changes made both to this file and to ``fcc2qth.pl''.

# ---------------------------------------------------------------------------- #
# ---- Adjust this variable to the preferred default range in km units ------- #
# ---------------------------------------------------------------------------- #

DEFAULT_RANGE='160'

# ---------------------------------------------------------------------------- #
# ---- Adjust this variable to the preferred options to splat_runner.sh -------#
# ---------------------------------------------------------------------------- #

SPLAT_RUNNER_OPTS="-u -m -w"

# ---------------------------------------------------------------------------- #
# -------- IN MOST CASES, NOTHING BELOW THIS LINE NEEDS ADJUSTMENT ----------- #
# ---------------------------------------------------------------------------- #

LOGFILE=''                                  # Defined on each pass through outer loop
TEST_ONLY='0'                               # Gets set to ``1'' with ``-t'' flag.
PROTECTIVE_MODE='0'                         # Skips records already processed

readonly USAGE="usage: $0 [-p -t] input_file..."    # Help message

readonly FCC2QTH="./fcc2qth.pl"             # FCC TVQ lookup => QTH, LRP, AZ, R files
readonly SPLAT_RUNNER="./splat_runner.sh"   # Runs ``splat'' and Doug Lung's scripts
readonly GRID_PROC="./sp3gridproc.pl"       # Doug Lung script called by $SPLAT_RUNNER
readonly PNG_RX="./3xy2png-rx-colors.pl"    # Doug Lung script called by $SPLAT_RUNNER
readonly SPLAT_HTML="./splat2html.sh"

run_commands()
{
    local infile="$1"

    while read line; do
        if echo "$line" | egrep -sq '^[[:space:]]*(#|$)'; then
            continue
        fi

        set -- $line

        local num_fields="$#"

        # Canonicalize to proper case for all fields and convert dots to
        # underscores in filename due to a defect in ``splat'' binary:

        local filename="$(echo "$1" | tr '[A-Z]' '[a-z]' | tr '.' '_')"
        local calls="$(echo "$2" | tr '[A-Z]' '[a-z]')"
        local fcc_id="$(echo "$3" | tr '[a-z]' '[A-Z]')"
        local range="$4"

        # 19 FEB 2010: [REN] - If the last field is ``-n'', pass along the
        # ``-n'' flag to ``fcc2qth.pl'' to suppress the default of
        # converting from NAD27 to WGS84.

        local no_nad27_wgs84=''

        if [ "$num_fields" = 4 -a X"$range" = X'-n' ]; then
            $range = 160
            no_nad27_wgs84='-n'
        elif [ "$num_fields" = 5 -a X"$5" = X'-n' ]; then
            no_nad27_wgs84='-n'
        fi

        local fcc_tvq='-F'
        local site_opt=''

        if [ "$num_fields" -ge 3 ]; then
            if [ "$num_fields" -eq 3 ]; then
                range='160'
            fi

            log_info "Evaluating record: $filename $calls $fcc_id $range"

            if ! echo "$calls" | grep -sq '^[kw0-9]'; then
                log_info "WARNING: Skipping malformed record with calls: $calls"
                continue
            fi

            if ! echo "$fcc_id" | egrep -sq '^[B0-9]'; then
                log_info "WARNING: Skipping malformed record with FCC ID: $fcc_id"
                continue
            fi

            # 07 MAR 2010: [REN] - Things get very different with ``splatinfo.php''
            # now on RabbitEars. The decision is made if BOTH the ``calls'' and
            # ``fcc_id'' are numeric. If so, we're using ``splatinfo'' otherwise,
            # fallback to the traditional FCC TV query.

            if echo "$calls" | egrep -sq '^[[:digit:]]+'; then
                if echo "$fcc_id" | egrep -sq '^[[:digit:]]+'; then
                    site_opt="-s $fcc_id"
                    fcc_tvq=''
                fi
            fi

            # Expect the range to be reasonable - 2 or 3 digits:

            if ! echo "$range" | egrep -sq '^[0-9]{2,3}$'; then
                log_info "WARNING: Skipping malformed record with range: $range"
                continue
            fi

            # If the code reaches here, the data has been vetted, so run the
            # $FCC2QTH program. Check its return value.

            # 07 FEB 2010: [REN] - Added code to use ``protective_mode'', to skip
            # records already processed as discerned by finding the CBDS application
            # ID already present in a QTH file.

            if [ "$PROTECTIVE_MODE" -eq 1 ]; then
                cmd="$FCC2QTH $fcc_tvq -p -f $filename -r $range $calls $fcc_id $no_nad27_wgs84"
            else
                cmd="$FCC2QTH $fcc_tvq -f $filename -r $range $calls $fcc_id $no_nad27_wgs84"
            fi

            if [ "$TEST_ONLY" -eq 1 ]; then
                log_info "Executing $cmd in TEST_ONLY mode"
                retval='0'
            else
                log_info "Executing $cmd"
                output="$($cmd 2>&1)"
                retval="$?"
            fi

            if [ "$retval" -eq 100 ]; then
                log_info "NOTE: $FCC2QTH already processed record: $output"
                log_info "Skipping to next record"
            elif [ "$retval" -ne 0 ]; then
                log_info "ERROR: $FCC2QTH failed, exit code: $retval: $output"
                log_info "Skipping to next record due to ERROR"
            else
                cmd="$SPLAT_RUNNER $SPLAT_RUNNER_OPTS $site_opt $filename"
                log_info "OK FCC data obtained for: $calls/$fcc_id"

                if [ "$TEST_ONLY" -eq 1 ]; then
                    log_info "Executing $cmd in TEST_ONLY mode"
                    retval='0'
                else
                    log_info "Executing $cmd"
                    $cmd
                    retval="$?"
                fi

                if [ "$retval" -ne 0 ]; then
                    log_info "ERROR: $SPLAT_RUNNER failed, exit code: $retval"
                else
                    log_info "OK maps completed for: $filename"
                fi
            fi
        fi
    done < "$infile"
}

log_info()
{
    local msg="$1"
    local date_fmt='+%Y-%m-%d %H:%M:%S'

    echo "$(date "$date_fmt"): $msg" >> "$LOGFILE"
    echo "$(date "$date_fmt"): $msg"
}

# ----------------- MAIN() body of execution begins here ------------------

if [ "$#" -eq 0 -o "$1" = '-h' ]; then
    echo "$USAGE" >&2
    exit 1
fi

# Check for programs to be executed prior to running:

for exe in "convert" "perl" "splat" "$FCC2QTH" "$SPLAT_RUNNER" \
           "$PNG_RX" "$GRID_PROC" "$SPLAT_HTML"; do
    if ! type -p "$exe" > /dev/null 2>&1; then
        echo "$0 error: can't find executable file $exe, exiting..." >&2
        exit 1
    fi
done

# Discern if this is a test run only:

if [ "$1" = '-t' -a "$#" -gt 1 ]; then
    TEST_ONLY='1'
    shift
fi

if [ "$1" = '-p' -a "$#" -gt 1 ]; then
    PROTECTIVE_MODE='1'
    shift
fi

for infile in "$@"; do
    logdir="$HOME/splat/log"

    mkdir -p "$logdir"
    chmod 0700 "$logdir"

    LOGFILE="$logdir/$(basename "$infile").log"

    touch "$LOGFILE"
    chmod 0600 "$LOGFILE"

    log_info "Started processing of input file: \"$infile\""

    run_commands "$infile"

    log_info "Ended processing of input file: \"$infile\""
done
