#!/bin/bash
#
# $Header: /home/bnelson/splat/RCS/splat2html.sh,v 1.29 2010-03-19 22:52:55-05 bnelson Exp bnelson $
#
# This unpublished proprietary work is Copyright (C) 2010 by
# Bob Nelson of Frisco, Texas. All rights reserved worldwide.
#
# This is free software. You may use it as you wish and freely distribute it.
#
# There are only these two requests:
#
# 1). Please keep this notice and the Copyright intact.
#
# 2). If you make improvements, please send them to the current author:
#
#     Bob Nelson <bnelson@nelsonbe.com>
#
# THIS SOFTWARE AND ITS SUB-PACKAGES ARE PROVIDED AS IS AND WITHOUT ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE OR
# EVEN NO PURPOSE AT ALL.
#
# MODULE NAME: splat2html.sh
#
# AUTHOR: Bob Nelson <bnelson@nelsonbe.com>
#
# DATE: 27 January 2010
#
# DESCRIPTION: Creates a local HTML file with the aid of Google maps for
# the ``SPLAT!'' television coverage maps. The output files are placed
# into the ``output/html'' directory.
#
# USAGE:
#
#   ./splat2html.sh calls app_id
#   ./splat2html.sh infofile_spec
#   ./splat2htm.sh --index
#
# USAGE EXAMPLES:
#
#   ./splat2html.sh k31gl 52923
#   ./splat2html.sh ./output/wfaa-1309817.info
#   ./splat2htm.sh --index
#
# NOTES AND CAVEATS:
#
# 0). Invocation allows three different methods:
#
#     a). Call_letters and application_id
#
#     b). Pathname of the ``*.info'' file.
#
#     c). ``-index'' flag creates ``output/html/index.html'', permitting
#         an easy way to browse the local maps.
#
# 1). This presumes that an ``info'' file exists in the ``output''
#     directory, which is a key/value pair formed as ``calls-app_id.info''.
#
# 2). It presumes that the ``info'' file has the transmitter latitude and
#     longitude and fails if it doesn't.
#
# 3). The RF-channel is displayed in the HTML by looking it up
#
# 4). The HTML output file is ``./output/html/CALLS-APP_ID.html''.

readonly OUT_DIR="output/html"
mkdir -p "$OUT_DIR" && chmod 0700 "$OUT_DIR"

if [ "$#" -eq 1 -a "$1" = '--index' ]; then
    # 02 FEB 2010: [REN] - Expedient hack to create an ``index.html'' file
    # in "$OUT_DIR" using existing HTML files.

    readonly IDX="$OUT_DIR/index.html"

    cat <<_EOF_ > "$IDX"
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">
<html>
    <head>
    <title>Available DTV Coverage Maps</title>
    </head>
    <body style="margin-left: 2em">
    <h4>DTV Coverage Maps - prepared $(date '+%Y-%m-%d @%H:%M:%S')</h4>
    <div style="font-size: 50%">
_EOF_

    for h in "$OUT_DIR/"[kw]*-*".html"; do
        title="$(echo "$(basename "$h")" | tr '[a-z]' '[A-Z]' |\
                    sed 's/\(.*\)-[0-9].*/\1/')"
        link="$(basename "$h")"
        details="$(grep 'RF Channel:' "$h" |\
            sed -e 's/.*RF \(Channel.*\), Application.*/\1/' -e 's/://')"

        echo "    <a href=\"$link\">$title</a>&nbsp;$details<br />" >> "$IDX"

        echo "Added $title to index"
    done

    cat <<_EOF_ >> "$IDX"
    </div>
    </body>
</html>
_EOF_

    chmod 0600 "$IDX"

    exit 0
fi

using_info_file=0

if [ "$#" -eq 1 ] && echo "$1" | grep -sq '\.info$'; then
    INFO_FILE="$1"
    using_info_file=1

    if [ ! -f "$INFO_FILE" ]; then
        echo "$0 error: can't find info file: $INFO_FILE, exiting..." >&2
        exit 1
    fi
elif [ "$#" -ne 2 ]; then
    echo "usage: $0 calls app_id" >&2
    exit 1
fi

TEMPLATE="template.html"
CHANNEL_TAB="channel-freq.tab"

for f in $TEMPLATE $CHANNEL_TAB; do
    if [ ! -f "$f" ]; then
        echo "$0 error: can't find file: $f, exiting..." >&2
        exit 1
    fi
done

if [ ! -d "terraindata" ]; then
    echo "$0 error: can't find terraindata directory in $PWD, exiting..." >&2
    exit 1
fi

SITE_ID='0'

if [ "$using_info_file" -eq 1 ]; then
    NAME="$(basename "$INFO_FILE" .info)"
    CALLS="$(echo "$NAME" | sed 's/\([^-]*\)-.*/\1/')"
    APP_ID="$(echo "$NAME" | sed 's/[^-]*-\([^-]*\)/\1/')"

    if echo "$APP_ID" | grep -sq '-'; then
        SITE_ID="$(echo "$APP_ID" | cut -d- -f2)"
        APP_ID="$(echo "$APP_ID" | cut -d- -f1)"
    fi

	INFO="$INFO_FILE"
else
    CALLS="$(echo "$1" | tr '[A-Z]' '[a-z]')"
    APP_ID="$2"

	INFO="output/$CALLS-$APP_ID.info"
fi
	
LRP="input/$CALLS.lrp"

for f in $INFO $LRP; do
    if [ ! -f "$f" ]; then
        echo "$0 error: can't find file: $f, exiting..." >&2
        exit 1
    fi
done

OUT_HTML="$OUT_DIR/$CALLS-$APP_ID-$SITE_ID.html"

# Discern the frequency from line #4 of the LRP file, then convert to
# the human readable channel number for display. And get the power
# from that file.

# 01 FEB 2010: [REN] - BUG FIXED! Discard anything after the first
# field when extracting the frequency from the LRP file, thus writing
# the correct channel number to the local HTML file.

FREQ="$(head -n 4 "$LRP" | tail -n 1 | awk '{ print $1 }')"
RF="$(awk -v freq="$FREQ" '$1 ~ /^[0-9]+$/ && freq >= $2 && freq <= $6 {
                            print $1
                            exit
                        }' "$CHANNEL_TAB")"

WATTS="$(tail -n 1 "$LRP" | sed 's/^\([0-9][0-9]*\).*/\1/')"

# By the time flow has arrived here, we have an info file, which can be
# used to discern the ``*.png'' and ``*.txt'' files in ``output'', using
# the naming convention established by Trip Ericson's RabbitEars. The
# info file has the transmitter lat/long values, if not fail.

TXT="output/$APP_ID-$SITE_ID.txt"
PNG="output/$APP_ID-$SITE_ID.png"

for f in "$TXT" "$PNG"; do
    if [ ! -f "$f" ]; then
        echo "$0 error: can't find file: $f, exiting..." >&2
        exit 1
    fi
done

XMTR_LAT="$(head -n 1 "$INFO")"
XMTR_LONG="$(head -n 2 "$INFO" | tail -n 1)"

# 29 JAN 2010: [REN] - Report the receiving antenna height which is in the
# 3rd line of the ``*info'' file. Since this is a new feature and older
# files may not have it -- make sure there are 3 lines in the file before
# grabbing this value.

RX_AGL="4"

if [ $(wc -l < "$INFO") -eq 3 ]; then
    RX_AGL="$(tail -n 1 "$INFO")"
fi

if [ ! "$XMTR_LAT" ]; then
    echo "$0 error: can't discern transmitter latitude, exiting..." >&2
    exit 1
fi

if [ ! "$XMTR_LONG" ]; then
    echo "$0 error: can't discern transmitter longitude, exiting..." >&2
    exit 1
fi

# Set the negative longitude:

XMTR_LONG="-$XMTR_LONG"

# Now get the boundaries for the Google map, which get inserted into the
# template file.

BOUNDS="$(< "$TXT")"

if [ ! "$BOUNDS" ]; then
    echo "$0 error: can't discern Google map boundaries, exiting..." >&2
    exit 1
fi

# Translate the call letters to uppercase and derive the basename for
# the PNG file;

UPPER_CALLS="$(echo "$CALLS" | tr '[a-z]' '[A-Z]')"
PNG_FILE="../$(basename "$PNG")"

sed -e "s/@@UPPER_CALLS@@/$UPPER_CALLS/g" \
    -e "s/@@RX_AGL@@/$RX_AGL/g" \
    -e "s/@@RF@@/$RF/g" \
    -e "s/@@WATTS@@/$WATTS/g" \
    -e "s/@@APP_ID@@/$APP_ID/g" \
    -e "s/@@BOUNDS@@/$BOUNDS/g" \
    -e "s/@@XMTR_LAT@@/$XMTR_LAT/g" \
    -e "s/@@XMTR_LONG@@/$XMTR_LONG/g" \
    -e "s|@@PNG_FILE@@|$PNG_FILE|g" \
        "$TEMPLATE" > "$OUT_HTML"

chmod -f 0600 "$OUT_HTML"

echo "$0 conversion to HTML succeeded, load this URL into your browser"
echo "$PWD/$OUT_HTML to view the map"
