#!/usr/bin/perl -w
#
# $Header: /home/bnelson/splat/RCS/nad2wgs.pl,v 1.3 2010-02-11 20:52:53-06 bnelson Exp $
# vim: ts=4 sw=4 tw=90:
#
# This unpublished proprietary work is Copyright (C) 2010 by
# Bob Nelson of Frisco, Texas. All rights reserved worldwide.
#
# This is free software. You may use it as you wish and freely distribute it.
#
# There are only these two requests:
#
# 1). Please keep this notice and the Copyright intact.
#
# 2). If you make improvements, please send them to the current author:
#
#     Bob Nelson <bnelson@nelsonbe.com>
#
# THIS SOFTWARE AND ITS SUB-PACKAGES ARE PROVIDED AS IS AND WITHOUT ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE OR
# EVEN NO PURPOSE AT ALL.
#
# MODULE NAME: nad2wgs.pl
#
# AUTHOR: Bob Nelson <bnelson@nelsonbe.com>
#
# DATE: 10 February 2010
#
# DESCRIPTION: Converts NAD27 lat,long coordinates to WGS84.
#
# USAGE:
#
#   ./nad2wgs.pl [-h -s]
#
#   * Shows usage message on stdout and exits successfully.
#
#   ./nad2wgs.pl ST lat,long
#
#   * Where ST is the case-insensitive 2-character state abbreviation
#     and lat,long is the coordinate pair expressed in degrees or DMS.
#
# USAGE EXAMPLES (all of which are for KGET's transmitter site)
#
#   ./nad2wgs.pl ca 35.438055556,118.739444444
#   ./nad2wgs.pl ca 35.438055556,118.739444444
#   ./nad2wgs.pl CA 35-26-17.0,118-44-22.0
#
# NOTES AND CAVEATS:
#
#   1). The state name can be uppercase, lowercase or CamelCase. All
#       of these are equivalent: CA, ca, Ca, cA.
#
#   2). The lat,long coordinate pair is presumed to be North and West.
#       A leading minus sign in the argument is silently removed. And
#       the converted value omits the minus sign unless the ``-s'' flag
#       is given to retain the sign.
#
#   3). Using DMS notation requires a comma-separated pair of dash
#       delimited coordinates.
#
#   4). A mixture of degress and DMS is disallowed:
#
#       ./nad2wgs.pl ca 35.43805556,118-44-22.0
#
#   5). The lat,long pair is expected to be NAD27 and the result is
#       written to stdout using WGS84 degree notation (not DMS).

# -------------------- preamble and file-scope globals --------------------

BEGIN {
    # Adjust the Perl search path to find the ``Coordinate'' module downloaded
    # from this URL:
    #
    #   ftp://ftp.eskimo.com/u/a/archer/aprs/xastir/
    #
    # The ``Coordinate'' module is cited here:
    #
    #   http://www.perlmonks.org/?node_id=65592

    unshift @INC, "$ENV{'HOME'}/splat";
}

use Getopt::Std;
use Coordinate;
use strict;

use vars qw($opt_h $opt_s);

my $DEBUG = 1;
my $sign_retention;

# -------------------- subroutine declarations ----------------------------

sub usage($);
sub dms2degrees($);
sub nad27towgs84($$$);
sub err($);

# -------------------- MAIN body of script starts here --------------------

{
    getopts('hs');

    usage 0 if $opt_h;
    usage 1 unless @ARGV == 2;

	$sign_retention = defined $opt_s ? 1 : 0;

    my $state_abbrev = uc $ARGV[0];

    if($state_abbrev !~ /^[A-Z][A-Z]$/) {
        print STDERR "error: ST must be two-character state abbreviation\n";
        usage 1;
    }

    my $lat_long = $ARGV[1];

    my ($lat, $long);

    if($lat_long =~ /^-?(\d+)\.([\d\.]+),-?(\d+)\.([\d\.]+)$/) {
        if(defined $1 && defined $2 && defined $3 && defined $4) {
            $lat = "$1.$2";
            $long = "$3.$4";
        }
    }
    elsif($lat_long =~ /^(\d+-\d+-[\d\.]+),(\d+-\d+-[\d\.]+)$/) {
        if(defined $1 && defined $2) {
            $lat = $1;
            $long = $2;

            $lat = dms2degrees $lat;
            $long = dms2degrees $long;
        }
    }
    else {
        print STDERR "error: malformed lat,long pair\n";
        usage 1;
    }

    # If flow reaches here, arguments have been vetted as well-formed.
    # They still may be invalid but at least are parsable.

    my ($wgs84_lat, $wgs_long) = nad27towgs84 $state_abbrev, $lat, $long;
    printf STDERR "%s,%s => %s,%s\n",
        $lat, $long, $wgs84_lat, $wgs_long if $DEBUG > 1;

    printf "%s,%s\n", $wgs84_lat, $wgs_long;
}

# -------------------- subroutine definitions -----------------------------

sub dms2degrees($)
{
    my $dms = shift;
    my $retval = undef;

    # Converts a degree, minute, second string to decimal.

    if($dms =~ /^(\d+)-(\d+)-([\d\.]+)/ && defined $1 && defined $2 && defined $3) {
        my $degrees = $1;
        my $minutes = $2;
        my $seconds = $3;

        $retval = sprintf "%.9f",
                $degrees + ($minutes / 60.0) + ($seconds / 3600.0);
    }

    printf STDERR "dms2degrees: %s => %s\n", $dms, $retval if $DEBUG > 1;

    return $retval;
}

sub nad27towgs84($$$)
{
    my ($state, $nad27_lat, $nad27_long) = (shift, shift, shift);
    my ($abbrev, $datum_val);
    my $state_nad27_tab = "$ENV{'HOME'}/splat/state-nad27.tab";
    my $fh;
    my ($wgs84lat, $wgs84long);
    my $found_datum = 0;

    # Converts a NAD27 degress to WGS84 using a lookup table keyes to the
    # state by finding the datam for the state to perform the late/long
    # correction, which must be trimmed of leading and trailing space.

    open $fh, "$state_nad27_tab" or err "can't open $state_nad27_tab, $!";

    while(<$fh>) {
        chomp;
        s/\r$//;

        next if /^$/ || /#/;

        ($abbrev, $datum_val) = split / /, $_, 2;

        if($state eq $abbrev && $datum_val) {
            $datum_val =~ s/^\s+//;
            $datum_val =~ s/\s+$//;

            printf STDERR "matched state to abbrev: %s, datum_val: |%s|\n",
                                $abbrev, $datum_val if $DEBUG > 2;
            $found_datum = 1;
            last;
        }
    }

    close $fh;

    if($found_datum && $datum_val) {
        # If the lookup was successful, create the ``Coordinate'' object
        # instance and convert to WGS84 (which is close enough to NAD83
        # per Trip Ericson for our use). Because North America is presumed,
        # the latitude remains positive but the longtitude is negated through
        # hard-coding.

        my $coord = Coordinate->new();

        $coord->latitude($nad27_lat);           # North latitude presumed
        $coord->longitude(-$nad27_long);        # West longitude presumed

        $coord->datum($datum_val);

        my $wgs84 = $coord->datum_shift_to_wgs84();

        # 30 JAN 2010: [REN] - BUG FIXED! Convert the negative longtitude
        # back to a positive number for ``SPLAT!'':

        $wgs84lat = $wgs84->latitude();

		if($sign_retention) {
	        $wgs84long = $wgs84->longitude();
		}
		else {
	        $wgs84long = -$wgs84->longitude();
		}
    }
    else {
        err "unable to find the WGS84 datum code for state: $state";
    }

    return ($wgs84lat, $wgs84long);
}

sub usage($)
{
    my $ex = shift;
    my $fh = $ex ? *STDERR : *STDOUT;

    print $fh "usage: $0 [-s] ST lat,long\n";
    print $fh "       where ST is 2-character state and lat,long is NAD27 coordinate pair\n";
    print $fh "       use -s flag to retain sign for negative latitudes\n";

    exit $ex;
}

sub err($)
{
    my $msg = shift;

    die "$0 error: $msg\n";
}
