#!/bin/bash
#
# $Header: /home/bnelson/splat/RCS/hgtstuff.sh,v 1.22 2010-02-04 21:52:34-06 bnelson Exp $
# vim: ts=4 sw=4 tw=90:
#
# MODULE NAME: hgtstuff.sh
#
# ORIGINAL AUTHOR: Trip Ericson <webmaster@rabbitears.info>
#
# MODIFIED BY: Bob Nelson <bnelson@nelsonbe.com>
#
# This is free software. You may use it as you wish and freely distribute it.
#
# There are only these two requests:
#
# 1). Please keep this notice and the Copyright intact.
#
# 2). If you make improvements, please send them to the current author:
#
#     Bob Nelson <bnelson@nelsonbe.com>
#
# THIS SOFTWARE AND ITS SUB-PACKAGES ARE PROVIDED AS IS AND WITHOUT ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE OR
# EVEN NO PURPOSE AT ALL.
#
# DATE: 26 January 2010
#
# DESCRIPTION: Automated Terrain Handling - extracts ZIP archives from
# USGS and then converts to elevation data files using ``srtm2sdf''.
#
# USAGE:
#
#       ./hgtstuff.sh [-bhu] [match_pattern...]
#
# USAGE EXAMPLES:
#
#       1). Convert all SRTM3 data in the ``North_America'' directory,
#           using bzip compression to save space with the SDF files.
#
#           ./hgtstuff.sh -b
#
#       2). Convert only SRTM3 ``North_America'' data matching single
#           QUOTED ``N3*'' pattern:
#
#           ./hgtstuff.sh 'N3*'
#
#       3). Converts only SRTM3 data matching multiple QUOTED patterns:
#
#          ./hgtstuff.sh 'N31*' 'N32*' 'N33*'
#
#       4). Presuming the ``usgs'' directory exists and is populated with
#           files from:
#
#               http://edcftp.cr.usgs.gov/pub/data/DEM/250/, use this
#
#           ...to regenerate terrain maps for only those regions that were
#           downloaded to that directory:
#
#           ./hgtstuff.sh -u
#
# NOTES AND CAVEATS: All of this presumes that data from the following site
# has been downloaded into ``$HOME/splat/North_America'' with files named
# ``*.hgt.zip'':
#
#       http://dds.cr.usgs.gov/srtm/version2_1/SRTM3/North_America/
#
# Create a symlink to ``North_America'' if ``wget -r'' was used to recursively
# fetch the directory as shown here:
#
#       cd $HOME/splat
#       ln -sf dds.cr.usgs.gov/srtm/version2_1/SRTM3/North_America .

readonly SPLAT_DIR="$HOME/splat"
readonly NA_DIR="$SPLAT_DIR/North_America"
readonly USGS_DIR="$SPLAT_DIR/usgs"
readonly TER_DIR="$SPLAT_DIR/terraindata"
readonly TMP_DIR="/tmp"
readonly USAGE="usage $0: [-bhu] [match_pattern...]"

readonly SRTM2SDF="srtm2sdf"
readonly USGS2SDF="usgs2sdf"
readonly BZIP2="bzip2"
readonly GZIP="gzip"
readonly UNZIP="unzip"

for exe in "$SRTM2SDF" "$USGS2SDF" "$BZIP2" "$GZIP" "$UNZIP"; do
    if ! type "$exe" >/dev/null 2>&1; then
        echo "error $0: $exe utility not found in path, exiting..." >&2
        exit 1
    fi
done

mkdir -p "$TER_DIR"

if [ ! -d "$NA_DIR" ]; then
    echo "error $0: directory not found $NA_DIR, exiting..." >&2
    exit 1
fi

if [ ! -d "$TER_DIR" ]; then
    echo "error $0: directory not found $TER_DIR, exiting..." >&2
    exit 1
fi

if ! cd "$TER_DIR"; then
    echo "error $0: can't change to $TER_DIR, exiting..." >&2
    exit 1
fi

bzip_sdf=0
ugsg_data=0

process_ugsg_data()
{
    # 03 FEB 2010: [REN] - Find all files named ``[a-z]*-?'' or ``[a-z]*-?.gz''
    # in $USGS_DIR, gunzip them (if necessary) and create USGS SDF files
    # to fill in the voids of SRTM terrain data.

    if ! cd "$USGS_DIR"; then
        echo "error $0: can't change to $USGS_DIR, exiting..." >&2
        exit 1
    fi

    for f in [a-z]*-?.gz; do
        if [ -f "$f" ]; then
            "$GZIP" -d "$f"
        fi
    done

    first_pass=1

    # 03 FEB 2010: [REN] - After converting the USGS file to SDF form, extract
    # the probable SRTM ``hgt'' filename and then call this script recursively
    # to regenerate the SDF files in ``terraindata''.

    for f in [a-z]*-?; do
        if [ "$first_pass" -eq 1 ]; then
            echo "Creating USGS files for filling of voids in SRTM data, wait..."
            first_pass=0
        fi

		dd if="$f" of="$f".dlm ibs=4096 cbs=1024 conv=unblock 2>/dev/null
		mv -f "$f".dlm "$f"

        output="$("$USGS2SDF" "$f" 2>&1)"
        echo "$output"

        # Now extract the names of the likely SRTM files that must be redone:

        coord="$(echo "$output" | grep 'Writing' | sed 's/.*"\([^"]*\).sdf".*/\1/')"

        # Redo the SDF files using the merged data:

        for lat in 1 2; do
            for long in 3 4; do
                fn="$(echo "$coord" | awk -F: '{ printf "N%.2dW%.3d", $'$lat', $'$long' }')"

                if [ -f "$NA_DIR/$fn.hgt.zip" ]; then
                    echo "Regenerating data for coordinates: $coord using SRTM file: $fn"
                    (cd .. && $0 -b "$fn")
                fi
            done
        done

        if [ -f "$f" ]; then
            "$GZIP" -f "$f"
        fi
    done
}

while getopts "bhu" option_letter; do
    case $option_letter in
        h)  echo "$USAGE"
            exit 0
            ;;
        b)  bzip_sdf=1
            ;;
        u)  ugsg_data=1
            if  [ ! -d "$USGS_DIR" ]; then
                echo "$0 error: directory $USGS_DIR must exist for -u option, exiting..." >&2
                exit 1
            else
                process_ugsg_data
                exit 0
            fi
            ;;
        *)  echo "$USAGE" >&2
            exit 1
            ;;
    esac
done

shift $((--OPTIND))

if [ "$#" -ge 1 ]; then
    for p in "$@"; do
        PATTERN="$PATTERN $NA_DIR/$p.hgt.zip"
    done
else
    PATTERN="$NA_DIR/*.hgt.zip"
fi

# 03 FEB 2010: [REN] - Check for existence of USGS directory and at
# least one file within matching a coordinate SDF before adding the
# the ``-d'' option to ``SRTM2SDF''.

USGS_OPT=''

if [ -d "$USGS_DIR" ]; then
    if ls "$USGS_DIR"/[0-9][0-9]:*.sdf > /dev/null 2>&1; then
        USGS_OPT="-d $USGS_DIR"
    fi
fi

for file in $PATTERN; do
    if [ ! -f "$file" ]; then
        continue
    fi

    "$UNZIP" -q -o "$file" -d "$TMP_DIR"

    if [ "$?" -eq 0 ]; then
        hgt_file="$(basename "$file" .zip)"

        output="$("$SRTM2SDF" $USGS_OPT "$TMP_DIR/$hgt_file")"
        result="$?"

        if [ "$result" -eq 0 ]; then
            outfile="$(echo "$output" | grep '^Writing' | sed 's/Writing \([0-9:]*\).*/\1/').sdf"

            rm -f "$TMP_DIR/$hgt_file"

            if [ "$bzip_sdf" -eq 1 ]; then
                if ! bzip2 -f "$TER_DIR/$outfile"; then
                    echo "error $0: failed to bzip2 $TER_DIR/$outfile, exiting..." >&2
                    exit 1
                fi

                echo "created SDF compressed file: $TER_DIR/$outfile.bz2"
            else
                echo "created SDF file: $TER_DIR/$outfile"
            fi
        else
            echo "error $0: $SRTM2SDF failed to convert $hgt_file, exiting..." >&2
            exit 1
        fi
    fi
done
