#!/usr/bin/perl -w
#
# $Header: /home/bnelson/splat/RCS/fcc2qth.pl,v 1.64 2010-03-07 20:28:42-06 bnelson Exp $
#
# This unpublished proprietary work is Copyright (C) 2010 by
# Bob Nelson of Frisco, Texas. All rights reserved worldwide.
#
# This is free software. You may use it as you wish and freely distribute it.
#
# There are only these two requests:
#
# 1). Please keep this notice and the Copyright intact.
#
# 2). If you make improvements, please send them to the current author:
#
#     Bob Nelson <bnelson@nelsonbe.com>
#
# THIS SOFTWARE AND ITS SUB-PACKAGES ARE PROVIDED AS IS AND WITHOUT ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE OR
# EVEN NO PURPOSE AT ALL.
#
# MODULE NAME: fcc2qth.pl
#
# AUTHOR: Bob Nelson <bnelson@nelsonbe.com>
#
# DATE: 28 January 2010
#
# DESCRIPTION: Creates QTH, LRP and AZ (if needed) files from FCC's TV query.
#
# * This obviates the need to visit the channels portion of the RabbitEars
#   website to obtain these input files requied for ``splat''.
#
# The files are named with the call letters, a dash, the specified app_id
# and the suffix of ``.qth', ``.lrp'' and ``.az''. The filename is written
# with lowercase characters to be easier on the eyes and to adhere to a
# consistent naming convention.
#
# - Each file is written into the ``input'' directory by default unless
#   the ``-s'' option is used for safe behavior to keep the file in the
#   CWD where it can be inspected for correctness. Once verified, it can
#   then manually be to the ``input'' directory.
#
# - The top line of the QTH file ALWAYS contains the fully-numeric CDBS
#   application number in keeping with Trip Ericson's standard.
#
# USAGE:
#
#   ./fcc2qth.pl [-h -f filename_prefix -n -p -r range -s] calls cdbs_app_id|fcc_id
#
# USAGE EXAMPLES:
#
# Visit the FCC's TV query webpage to find the CDBS Application ID Number
# or the FCC File Number if you don't already know it.
#
# Alternately, visit RabbitEars to get the FCC File Number for the desired
# facility.
#
#   Using CDBS Application ID
#   -------------------------
#   ./fcc2qth.pl wfaa 1348515
#   ./fcc2qth.pl wls-tv 1339344
#
#   Using FCC File Number
#   ---------------------
#   ./fcc2qth.pl ksfw BPDVL-20091224ACA
#   ./fcc2qth.pl ksfw bpdvl-20091224aca
#
#   Using FCC File Number with options
#   ----------------------------------
#   ./fcc2qth.pl -r 128 -f ksfw_app ksfw bpdvl-20091224aca
#   ./fcc2qth.pl -r 128 -f ksfw_app ksfw BPDVL20091224ACA
#
#   * The above will create these files in the CWD which can then be
#     moved into the ``input'' subdirectory.
#
#     ksfw_app.qth
#     ksfw_app.lrp
#     ksfw_app.r
#
# NOTES AND CAVEATS:
#
# 1). The ``LWP::Simple'' Perl module is required in order to perform the
#     data retrieval from this URL:
#
#           http://www.fcc.gov/fcc-bin/tvq?call=$CALLS
#
# 2). It may take a few seconds to get the data depending upon the server
#     load at the FCC site.
#
# 3). Although the error trapping is quite extensive, BE SURE to check the
#     files for correctness.
#
# 4). Call letters are expected to start with a 'K' or 'W'. The call letters
#     may be specified witout regard to case. The output files are written
#     with lowercase names.
#
# 5a). The application_id is expected to be a positive, non-zero integer.
#
# 5b). The fcc_file_number is expected to start with 'B' (case-insensitive)
#      and contain a sequence of at least 8 digits. A hypen is inserted
#      before the first digit if not already present in the argument.
#
# 6). If ``Additional Azimuths'' is found in the record, then a message is
#     written to stdout to indicate that additional azimuths are in use when
#     the AZ file is written. This was subsequently revoked as unnecessary.
#
# 7). 07 FEB 2010: [REN] - The ``-p'' flag was added to ``protect'' those
#     facilities already done. If a match is found in the QTH file, a message
#     is written to stdout and the program exits with a value of 100. Read more
#     about this in ``README.splat''.
#
# 8). 19 FEB 2010: [REN] - Added ``-n'' flag to NOT correct the FCC
#     coordinates from NAD27 to WGS84. Use this if you have reason to think
#     that the values on the FCC database are already adjusted for WGS84.

# ------------------ global variable (adjust as desired) ----------

my $DEBUG = 1;

# ------------------------------------------------------------------
# ------------------ nothing below this point is adjustable --------
# ------------------------------------------------------------------

# ------------------ perl modules and preamble ---------------------

BEGIN {
    # Adjust the Perl search path to find the ``Coordinate'' module downloaded
    # from this URL:
    #
    #   ftp://ftp.eskimo.com/u/a/archer/aprs/xastir/
    #
    # The ``Coordinate'' module is cited here:
    #
    #   http://www.perlmonks.org/?node_id=65592

    unshift @INC, "$ENV{'HOME'}/splat";
}

use LWP::Simple;
use Getopt::Std;
use Coordinate;
use strict;

use vars qw($opt_h $opt_F $opt_f $opt_r $opt_p $opt_s $opt_n);

my $filename_prefix = '';       # User defined basename
my $safe_behavior = 0;          # Keep file in CWD, else mv to ``input''
my $protective_mode = 0;        # Protects already fetched files
my $range = -1;                 # Create an ``r'' (range) file.
my $no_wgs84_correction = 0;    # Invoked with ``-n'' flag
my $use_fcc = 0;                # 07 MAR 2010: [REN] - Use FCC TVQ instead of RE

# ------------------ subroutine declarations -----------------------

sub extract_station_data($$$$);
sub channel2freq($);
sub dms2degrees($);
sub nad27towgs84($$$);
sub meters2feet($);
sub write_qth_file($$$$$);
sub write_lrp_file($$$);
sub write_az_file($$\@);
sub write_r_file($$);
sub does_app_id_exist($);
sub file_wrapup($$\@);
sub err($);

# ------------------ MAIN() body of execution ----------------------

{
    my $usage = <<_EOF_;
usage $0: [-h -F -f filename_prefix -r range -n -p -s] app_id|calls site|cdbs_app_id|fcc_id
      -h: show help on stdout and exit successfully
      -F: use FCC TV Query, default is splatinfo.php from RabbitEars
      -f filename_prefix: set basename of files to filename_prefix
      -r range: specify range in kilometers, default is 160 km
      -n: don't apply FCC NAD27 coordinate correction to WGS84
      -s: safe behavior, write files in CWD, not ./input directory
      -p: protective mode, exits with 100 if QTH file with app_id exists
_EOF_

    # Parse command line arguments, which allow a range to specified as well
    # as altering the final filename by using a different prefix for the
    # QTH, LRP, AZ and R files.

    getopts('hnpsf:r:F');

    if($opt_h) {
        print "$usage\n";
        exit 0;
    }
    elsif(@ARGV != 2) {
        print STDERR "$usage\n";
        exit 1;
    }

    # 07 MAR 2010: [REN] - Default is now to lookup against ``splatinfo.php''
    # unless ``-F'' flag is used. Note that it's uppercase!

    if($opt_F) {
        $use_fcc = 1;
    }

    if($opt_r) {
        unless($opt_r =~ /^\d+$/) {
            err "value for range must be a positive integer\n$usage";
        }

        $range = $opt_r;
    }

    if($opt_f) {
        unless($opt_f =~ /^[a-z\d\-\_]+$/) {
            err "value for filename must consist of lowercase alphanumerics (and _-)\n$usage";
        }

        $filename_prefix = $opt_f;
    }

    if($opt_n) {
        $no_wgs84_correction = 1;
    }

    if($opt_p) {
        $protective_mode = 1;
    }

    if($opt_s) {
        $safe_behavior = 1;
    }

    my $calls;
    my $app_id;
    my $site;

    if($use_fcc) {
        $calls = uc shift;
        $app_id = uc shift;

        unless($calls =~ /^[KW]/) {
            err "call letters must begin with 'K' or 'W'";
        }
    }
    else {
        $app_id = shift;
        $site = shift;
    }

    my $id_type = 0;

    if($use_fcc) {
        if($app_id =~ /^\d+$/) {
            $id_type = 1;       # CDBS application ID, numeric only
        }
        elsif($app_id =~ /^B[^-]+-\d{8,}/) {
            $id_type = 2;       # FCC file number, starts with 'B' has a dash a digit
        }
        elsif($app_id =~ /^(B.*?)(\d{8,}.*)$/ && defined $1 && defined $2) {
            $app_id = "$1-$2";
            $id_type = 2;       # FCC file number, starts with 'B' with dash provided
        }
    }
    else {                      # RabbitEars lookup, not FCC TV Query
        $id_type = 4;

        unless($app_id =~ /^\d+$/) {
            err "applicatiion ID must be numeric";
        }
        unless($site =~ /^\d+$/) {
            err "site must be numeric";
        }
    }

    if($id_type < 1) {
        err "specified application_id or file_number ($app_id) is malformed";
    }

    printf STDERR "looking up by type %d, |%s|\n", $id_type, $app_id if $DEBUG > 2;

    my @station_data = ();
    my $fcc_data;
    my $idx = 0;
    my $records_retrieved = -1;
    my $url;

    if($use_fcc) {
        $url = sprintf 'http://www.fcc.gov/fcc-bin/tvq?call=%s', $calls;
    }
    else {
        $url = sprintf 'http://www.rabbitears.info/splatinfo.php?appid=%d&site=%d', $app_id, $site;
    }

    # The LWP::Simple::get() function returns the entire page as one long
    # string. Make sure the string isn't empty and that a non-zero count
    # of FCC records were retrieved.

    $fcc_data = get($url);

    if(!$fcc_data) {
        err "no data at all retrieved from URL: $url";
    }

    if($use_fcc) {
        if($fcc_data =~ />\s*Page\s+Not\s+Found\s*</) {
            err "no page found on the FCC site for URL: $url";
        }
        elsif($fcc_data =~ /\*\s+(\d+)\s+Records\s+Retrieved\s+\*/ && defined $1) {
            $records_retrieved = $1;
        }

        if($records_retrieved < 1) {
            err "no records retrieved from the FCC site for: $calls";
        }

        printf "Now scanning %d FCC record(s) for %s's application id: %s, wait...\n",
                    $records_retrieved,
                    $calls,
                    $app_id;

        # By the time flow reaches here, remove needless header and footer data
        # from the ``$fcc_data'' string to make the parse faster. This may be
        # brittle because it presumes certain strings will always be present.

        $fcc_data =~ s/^.*\bBREADCRUMB\s+TEXT\b//;
        $fcc_data =~ s/\bULS\s+Search\b.*//;
    }

    foreach(split /\n/, $fcc_data) {
        chomp;

        s/\r$//;
        s/^\s+//;
        s/\s+$//;

        if(!$use_fcc) {
            # 07 MAR 2010: [REN] - For ``splatinfo.php'' just vet that every
            # single line starts with a digit (except the first, which is the
            # state code). The data is presumed to be well-formed since Trip
            # Ericson maintains the page.

            push @station_data, $_;

            if($idx == 0) {
                unless(/^[A-Z][A-Z]$/) {
                    err "malformed first line found (\"$_\"), expecting state code in URL: $url";
                }
            }
            else {
                unless(/^\d/) {
                    err "malformed data line found (\"$_\") in URL: $url";
                }
            }

            ++$idx;
        }
        else {
            last if /\*\s+\d+\s+Records Retrieved\s+\*/;

            if(/span.*>\s*$calls/) {
                $station_data[$idx] = $_;
                printf STDERR "station_data[%d]: %s\n", $idx, $_ if $DEBUG > 2;
            }
            elsif(/(Previous|First)\s+Record</) {
                ++$idx;
            }
            elsif (length $_) {
                $station_data[$idx] .= " $_";
                printf STDERR "station_data[%d]: %s\n", $idx, $_ if $DEBUG > 2;
            }
        }
    }

    if(!$use_fcc) {
        # 07 MAR 2010: [REN] - By the time flow reaches here, ``splatinfo.php''
        # data has been deemed well-formed and all of the lines read from the
        # page are in the ``@station_data'' array, ready for massaging before
        # writing the ``qth'', ``lrp'' and optional ``az'' files.

        my @files = ();

        my $qth = lc "$app_id.qth";
        my $lrp = lc "$app_id.lrp";
        my $az = lc "$app_id.az";
        my $r = lc "$app_id.r";

        if($filename_prefix) {
            $qth = lc "$filename_prefix.qth";
            $lrp = lc "$filename_prefix.lrp";
            $az = lc "$filename_prefix.az";
            $r = lc "$filename_prefix.r";
        }

        if($protective_mode) {
            if(does_app_id_exist $app_id) {
                exit 100;
            }
        }

        my $state = $station_data[0];

        my $lat = dms2degrees $station_data[1];
        my $long = dms2degrees $station_data[2];

        if(!defined $lat) {
            err "can't discern latititude for app_id: $app_id";
        }
        if(!defined $long) {
            err "can't discern longtitude for app_id: $app_id";
        }

        my $agl = $station_data[3];
        my $freq = $station_data[4];
        my $power = $station_data[5];
        my $antenna_id = $station_data[6];      # Not currently used

        if(!$no_wgs84_correction) {
            ($lat, $long) = nad27towgs84 $state, $lat, $long;
        }

        if(!defined $lat) {
            err "can't discern WG84 corrected latititude for app_id: $app_id";
        }
        if(!defined $long) {
            err "can't discern WG84 corrected longtitude for app_id: $app_id";
        }
        if(!defined $agl) {
            err "can't discern AGL for app_id: $app_id";
        }
        if(!defined $freq) {
            err "can't discern frequency for app_id: $app_id";
        }
        if(!defined $power) {
            err "can't discern power for app_id: $app_id";
        }

        write_qth_file $qth, $app_id, $lat, $long, $agl;
        printf "Wrote QTH file: %s\n", $qth if $DEBUG > 0;
        push @files, $qth;

        write_lrp_file $lrp, $freq, $power * 1000;
        printf "Wrote LRP file: %s\n", $lrp if $DEBUG > 0;
        push @files, $lrp;

        # 07 MAR 2010: [REN] - Only write an AZ file if directional, based
        # upon the amount of data plucked from ``splatinfo.php'':

        if(scalar @station_data > 8) {
            my $rotation = $station_data[7];
            my @dir_values= ();

            for(my $i = 8; $i < scalar @station_data; ++$i) {
                my $plot = $station_data[$i];

                if($plot =~ /^\d+\s+([\d\.]+)/ && defined $1) {
                    push @dir_values, $1;
                }
            }

            write_az_file $az, $rotation, @dir_values;
            printf "Wrote AZ file: %s\n", $az if $DEBUG > 0;
            push @files, $az;
        }
        else {
            printf "(facility is non-directional, no AZ file is needed)\n" if $DEBUG > 0;
        }

        file_wrapup $r, $range, @files;
    }
    else {                  # FCC Data lookup
        printf STDERR "station_data elems: %d\n", scalar @station_data if $DEBUG > 2;

        my $match_idx = -1;

        for(my $i = 0; $i < @station_data; ++$i) {
            my $elem = $station_data[$i];

            printf STDERR "station_data[%d] length: %d\n", $i, length $elem if $DEBUG > 2;

            # Attempt to find a match based upon the numeric CDBS application when using
            # ``$id_type'' of 1.
            #
            # Alternaately if ``$id_type'' is 2, find a lookup on the FCC file identifer.

            if($id_type == 1 && $elem =~ /\bCDBS\s+Application\s+ID\s+No\.\s*:\s+$app_id</) {
                $match_idx = $i;

                printf STDERR "matched cdbs_app_id %d - station_data[%d] length: %d\n",
                                    $app_id, $i, length $elem if $DEBUG > 2;

                last;
            }
            elsif($id_type == 2 && $elem =~ /\bFile\s+No\.\s*:\s+$app_id/) {
                $match_idx = $i;

                printf STDERR "matched fcc_file_no %s - station_data[%d] length: %d\n",
                                    $app_id, $i, length $elem if $DEBUG > 2;

                last;
            }
        }

        if($match_idx < 0) {
            # 03 FEB 2010: [REN] - BUG FIXED by treating app_id as string not
            # a digit. This makes the error message correct:

            err sprintf "no match found in FCC data for %s, app_id: %s", $calls, $app_id;
        }
        else {
            extract_station_data $id_type, $calls, $app_id, $station_data[$match_idx];
        }
    }
}

# ------------------ subroutine definitions ------------------------

sub extract_station_data($$$$)
{
    my ($id_type, $calls, $app_id, $data) = (shift, shift, shift, shift);

    my $qth = lc "$calls-$app_id.qth";
    my $lrp = lc "$calls-$app_id.lrp";
    my $az = lc "$calls-$app_id.az";
    my $r = lc "$calls-$app_id.r";

    # The user may have specified an alternate filename prefix. By the time
    # flow gets here, the validity of that prefix has been vetted so apply
    # that desired user-defined name to the 4 possible files.

    if($filename_prefix) {
        $qth = lc "$filename_prefix.qth";
        $lrp = lc "$filename_prefix.lrp";
        $az = lc "$filename_prefix.az";
        $r = lc "$filename_prefix.r";
    }

    my ($freq, $lat, $long, $agl, $watts, $numeric_cdbs_app_id, $state);
    my @files = ();

    # If the user specified the fully-numeric CDBS application ID, there's no
    # need to dig through the data to find it. If an FCC alphanumeric file number
    # is used, find the numeric CDBS application ID in this record since that
    # number is written to the top of the QTH file.

    if($id_type == 1) {
        $numeric_cdbs_app_id = $app_id;
    }
    elsif($id_type == 2) {
        if($data =~ /\bCDBS\s+Application\s+ID\s+No\.\s*:\s+(\d+)</ && defined $1) {
            $numeric_cdbs_app_id = $1;
        }
        else {
            err "can't find CDBS Application ID for $app_id";
        }
    }
    else {
        err "impossible condition reached in script at line: " . __LINE__;
    }

    if($protective_mode) {
        if(does_app_id_exist $numeric_cdbs_app_id) {
            exit 100;
        }
    }

    printf STDERR "extract_station_data() - %s, app_id: %s, numeric_cdbs_app_id:%d\n%s\n",
                    $calls, $app_id, $numeric_cdbs_app_id, $data if $DEBUG > 2;

    # The state must be obtained in order to peform a lookup top convert from
    # NAD27 to WGS84 coordinates.

    if($data =~ /\b$calls\S*?\s+([A-Z][A-Z])\s+/ && defined $1) {
        $state = $1;

        printf STDERR "state: |%s|\n", $state if $DEBUG > 2;
    }
    else {
        err "unable to discern the 2-letter state abbreviation for $calls";
    }

    if($data =~ /\bChannel\s*:\D+(\d+)\D+/ && defined $1) {
        my $channel = $1;

        $freq = channel2freq $channel;

        printf STDERR "channel: %d, freq: %s\n", $channel, $freq if $DEBUG > 2;
    }
    else {
        err "unable to discern the RF channel number for $calls";
    }

    if($state && $data =~ />.*?(\d+\s*&#176;.*?")\s*\bN\s+Latitude(.*?)\bW\s+Longitude\b/ && defined $1 && defined $2) {
        my $first = $1;
        my $second = $2;

        printf STDERR "first: %s, second: %s\n", $first, $second if $DEBUG > 2;

        if($second =~ /(\d+\s*&#176;.*?")/ && defined $1) {
            $lat = $first;
            $long = $1;

            $lat = dms2degrees $lat;
            $long = dms2degrees $long;

            printf STDERR "(NAD27) lat: |%s|, long: |%s|\n", $lat, $long if $DEBUG > 2;

            if(!$no_wgs84_correction) {
                ($lat, $long) = nad27towgs84 $state, $lat, $long;

                printf STDERR "(WGS84) lat: |%s|, long: |%s|\n", $lat, $long if $DEBUG > 2;
            }
        }
    }
    unless($lat && $long) {
        err "unable to discern latitude and longitude for $calls";
    }

    if($data =~ /\bAntenna\s+Height\s+Above\s+Ground\s+Level:.*?>\s*([\d\.]+)\s*</ && defined $1) {
        my $tx_agl = $1;

        if($tx_agl == 0) {
            # 31 JAN 2010: [REN] - Hack for a 0.0 TX antenna, just mirandize
            # it to 5 meters, found with KAMR's 8.5 KW map.

            $tx_agl = 5;
        }

        $agl = sprintf "%.1f", meters2feet $tx_agl;

        printf STDERR "agl: %s\n", $agl if $DEBUG > 2;
    }
    else {
        err "unable to discern antenna height for $calls";
    }

    if($data =~ /Effective\s+Radiated\s+Power\s+\(ERP\)\s*:.*?>\s*([\d\.]+)\s*</ && defined $1) {
        $watts = sprintf "%0.f", $1 * 1000.0;

        printf STDERR "watts: %s\n", $watts if $DEBUG > 2;
    }
    else {
        err "unable to discern wattage for $calls";
    }

    # By the time flow arrives here, there's enough to write the QTH file if all
    # if these values have been defined:

    if(defined $lat && defined $long && defined $agl) {
        write_qth_file $qth, $numeric_cdbs_app_id, $lat, $long,  $agl;

        printf "Wrote QTH file: %s (WGS84 coordinates)\n", $qth if $DEBUG > 0;
        push @files, $qth;
    }
    else {
        err "unable to write QTH file: $qth";
    }

    # By the time flow arrives here, there's enough to write the LRP file if all
    # if these values have been defined:

    if(defined $freq && defined $watts) {
        write_lrp_file $lrp, $freq, $watts;

        printf "Wrote LRP file: %s\n", $lrp if $DEBUG > 0;
        push @files, $lrp;
    }
    else {
        err "unable to write LRP file: $lrp";
    }

    # Now examine the data for any possible instances of directional antenna
    # data. This is a file that may not get written for a non-DA facility.
    #
    # While in this block of code -- check for ``Additional azimuths'' in
    # the record to write a warning -- later revoked per Trip Ericson.

    if($data =~ /\&(p0.*?\&p360.*?)>/ && defined $1) {
        my $dir_plots = $1;
        my @dir_values = ();

        printf STDERR "directional data found: |%s|\n", $dir_plots if $DEBUG > 2;

        my $additional_azimuths_msg = '';

        if(0) {
            if($data =~ /\bAdditional\s+azimuths?\s*:/) {
                $additional_azimuths_msg = ', WARNING! has additional azimuths';
            }
        }

        for(my $i = 0; $i <= 350; $i += 10) {
            if($dir_plots =~ /p$i\s*=([\d\.]+)\&/ && defined $1) {
                push @dir_values, $1;
            }
        }

        printf STDERR "\@dir_values array elems: %d\n", scalar @dir_values if $DEBUG > 2;

        # Check that the array of values is fully populated, 0-350 degrees in
        # increments of 10 is 36 elements.

        if(scalar @dir_values == 36) {
            my $rotation = 0;

            # 30 JAN 2010: [REN] - BUG FIXED! - now uses proper regex scan for
            # pattern rotation value:

            if($data =~ /\bPattern\s+Rotation\s*:\s*([\d\.]+)/ && defined $1) {
                $rotation = $1;
            }

            write_az_file $az, $rotation, @dir_values;

            printf "Wrote AZ file: %s%s\n", $az, $additional_azimuths_msg if $DEBUG > 0;
            push @files, $az;
        }
        else {
            err "unable to write AZ file: $az";
        }
    }
    else {
        printf "(%s is non-directional, no AZ file is needed)\n", $calls if $DEBUG > 0;
    }

    file_wrapup $r, $range, @files;
}

sub channel2freq($)
{
    my $channel = shift;
    my $channel_freq_tab = "channel-freq.tab";
    my $fh;
    my $freq = -1;

    open $fh, "$channel_freq_tab" or err "can't open $channel_freq_tab, $!";

    while(<$fh>) {
        chomp;
        s/\r$//;

        next unless /^\d+/;

        # Use the video frequency when the channel matches from the lookup
        # table, ``channel-freq.tab'':

        my ($ch, $lo, $vid, $atsc, $audio, $hi) = split '\s+', $_;

        if($ch == $channel) {
            $freq = sprintf "%.2f", $vid;
            last;
        }
    }

    close $fh;

    return $freq;
}

sub dms2degrees($)
{
    my $dms = shift;
    my $retval = undef;

    # Converts a degree, minute, second string to decimal.

    if(($dms =~ /^\s*(\d+)&#176;\s*(\d+)\s*'.\s*([\d\.]+)\s*"/ ||
       $dms =~ /^(\d+)-(\d+)-([\.\d]+)$/) && defined $1 && defined $2 && defined $3) {
        my $degrees = $1;
        my $minutes = $2;
        my $seconds = $3;

        $retval = sprintf "%.9f", $degrees + ($minutes / 60.0) + ($seconds / 3600.0);
    }

    return $retval;
}

sub nad27towgs84($$$)
{
    my ($state, $nad27_lat, $nad27_long) = (shift, shift, shift);
    my ($abbrev, $datum_val);
    my $state_nad27_tab = "state-nad27.tab";
    my $fh;
    my ($wgs84lat, $wgs84long);
    my $found_datum = 0;

    # Converts a NAD27 degress to WGS84 using a lookup table keyes to the
    # state by finding the datam for the state to perform the late/long
    # correction, which must be trimmed of leading and trailing space.

    open $fh, "$state_nad27_tab" or err "can't open $state_nad27_tab, $!";

    while(<$fh>) {
        chomp;
        s/\r$//;

        next if /^$/ || /#/;

        ($abbrev, $datum_val) = split / /, $_, 2;

        if($state eq $abbrev && $datum_val) {
            $datum_val =~ s/^\s+//;
            $datum_val =~ s/\s+$//;

            printf STDERR "matched state to abbrev: %s, datum_val: |%s|\n",
                                $abbrev, $datum_val if $DEBUG > 2;

            $found_datum = 1;
            last;
        }
    }

    close $fh;

    if($found_datum && $datum_val) {
        # If the lookup was successful, create the ``Coordinate'' object
        # instance and convert to WGS84 (which is close enough to NAD83
        # per Trip Ericson for our use). Because North America is presumed,
        # the latitude remains positive but the longtitude is negated through
        # hard-coding.

        my $coord = Coordinate->new();

        $coord->latitude($nad27_lat);           # North latitude presumed
        $coord->longitude(-$nad27_long);        # West longitude presumed

        $coord->datum($datum_val);

        my $wgs84 = $coord->datum_shift_to_wgs84();

        # 30 JAN 2010: [REN] - BUG FIXED! Convert the negative longtitude
        # back to a positive number for ``SPLAT!'':

        $wgs84lat = $wgs84->latitude();
        $wgs84long = -$wgs84->longitude();
    }
    else {
        err "unable to find the WGS84 datum code for state: $state";
    }

    return ($wgs84lat, $wgs84long);
}

sub meters2feet($)
{
    my $meters = shift;

    return 3.2808399 * $meters;
}

sub write_qth_file($$$$$)
{
    my ($qth, $app_id, $lat, $long,  $agl) = (shift, shift, shift, shift, shift);
    my $fh;

    open $fh, "> $qth" or err "can't open file: $qth, $!";

    printf $fh <<_EOF_;
$app_id
$lat
$long
$agl
_EOF_

    close $fh;

    chmod 0600, $qth;
}

sub write_lrp_file($$$)
{
    my ($lrp, $freq, $watts) = (shift, shift, shift);
    my $fh;

    # Much of what's in the LRP file are hard-coded values, only the
    # frequency and power are variable:

    open $fh, "> $lrp" or err "can't open file: $lrp, $!";

    printf $fh <<_EOF_;
15.000      ; Earth Dielectric Constant (Relative permittivity)
0.005       ; Earth Conductivity (Siemens per meter)
301.000     ; Atmospheric Bending Constant (N-Units)
$freq       ; Frequency in MHz (20 MHz to 20 GHz)
5           ; Radio Climate
0           ; Polarization (0 = Horizontal, 1 = Vertical)
0.50        ; Fraction of situations
0.90        ; Fraction of time
$watts      ; ERP in watts
_EOF_

    close $fh;

    chmod 0600, $lrp;
}

sub write_az_file($$\@)
{
    my ($az, $rotation, $dir_values_ref) = (shift, shift, shift);
    my @dir_values = @$dir_values_ref;
    my $fh;

    # By the time this function is called, an array has been populated of
    # exactly 37 elements, each of which is a value for the polar plot of
    # directional elements.

    open $fh, "> $az" or err "can't open file: $az, $!";

    printf $fh "%.3f\n", $rotation;

    for(my $i = 0; $i <= 35; ++$i) {
        printf $fh "%d\t%.3f\n", $i * 10, $dir_values[$i];
    }

    close $fh;

    chmod 0600, $az;
}

sub write_r_file($$)
{
    my ($r, $range) = (shift, shift);
    my $fh;

    # By the time this function is called, an optional range was specified,
    # just write that numeric value as the sole contents of the ``r'' file.

    open $fh, "> $r" or err "can't open file: $r, $!";

    printf $fh "%d\n", $range;

    close $fh;

    chmod 0600, $r;
}

sub file_wrapup($$\@)
{
    my ($r, $range, $files_ref) = (shift, shift, shift);
    my @files = @$files_ref;

    if($range > 0) {
        write_r_file $r, $range;

        printf "Wrote optional range ($range km) to file: %s\n", $r if $DEBUG > 0;
        push @files, $r;
    }

    if($safe_behavior) {
        printf "* Carefully examine the %d files before moving them into the \"./input\" directory.\n",
                    scalar @files;
    }
    else {
        # 30 JAN 2010: [REN] - Default behavior (sans ``-s'') is to move the files
        # into the ``./input'' directory, which is checked for existence.

        my $location;

        if(! -d "input") {
            print "$0 warning: \"./input\" directory doesn't exist, keeping files in CWD\n";

            $location = 'current';
        }
        else {
            foreach my $file(@files) {
                system qq|mv $file ./input|;
            }

            $location = '"./input"';
        }

        printf "* Carefully examine those %d files in the %s directory before creating maps.\n",
                    scalar @files, $location;
    }
}

sub does_app_id_exist($)
{
    my $app_id = shift;
    my @qth_files = glob './input/*.qth';
    my $fh;
    my $retval = 0;

    # 07 FEB 2010: [REN] - To support the ``-p'' flag, check all of the
    # ``input/*qth'' files to find the CDBS application ID on the top line.
    # If there an exact match or a match with an 's' prefix, write a helpful
    # message and return a true value, which will then cause the program to
    # exit with a status code of 100.

    foreach my $fn(@qth_files) {
        if(open $fh, $fn) {
            my $qth_app_id = scalar <$fh>;

            chomp $qth_app_id;
            $retval = $qth_app_id =~ /^s?$app_id/i;

            close $fh;

            if($retval) {
                print "Already found CDBS application ID $app_id in $fn, " .
                      "exiting due to protective mode\n";
                last;
            }
        }
    }

    return $retval;
}

sub err($)
{
    my $msg = shift;

    die "$0 error: $msg\n";
}
